/*
 This file is part of GNU Taler
 (C) 2020-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AmountString,
  j2s,
  Logger,
  NotificationType,
  TalerCorebankApiClient,
  TransactionMajorState,
  TransactionMinorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  ExchangeServiceInterface,
  GlobalTestState,
  WalletClient,
} from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  WithdrawViaBankResult,
} from "../harness/environments.js";

const logger = new Logger("test-account-restrictions.ts");

/**
 * Test for credit/debit account restrictions.
 */
export async function runAccountRestrictionsTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t, undefined, {
      accountRestrictions: [
        [
          "debit-restriction",
          "regex",
          "payto://x-taler-bank/.*/foo-.*",
          "bla",
          "{}",
        ],
        [
          "credit-restriction",
          "regex",
          "payto://x-taler-bank/.*/foo-.*",
          "bla",
          "{}",
        ],
      ],
    });

  // Withdraw digital cash into the wallet.

  const withdrawalResult = await myWithdrawViaBank(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
    acctname: "foo-123",
  });

  await withdrawalResult.withdrawalFinishedCond;

  // When withdrawing from an account that doesn't begin with "foo-",
  // it fails.
  await t.assertThrowsAsync(async () => {
    await myWithdrawViaBank(t, {
      walletClient,
      bankClient,
      exchange,
      amount: "TESTKUDOS:20",
      acctname: "bar-123",
    });
  });

  // Invalid account, does not start with "foo-"
  const err = await t.assertThrowsTalerErrorAsyncLegacy(
    walletClient.call(WalletApiOperation.CheckDeposit, {
      amount: "TESTKUDOS:5",
      depositPaytoUri: "payto://x-taler-bank/localhost/bar-42",
    })
  );

  logger.info(`checkResp ${j2s(err)}`);

  // Valid account
  await walletClient.call(WalletApiOperation.CheckDeposit, {
    amount: "TESTKUDOS:5",
    depositPaytoUri: "payto://x-taler-bank/localhost/foo-42",
  });
}

export async function myWithdrawViaBank(
  t: GlobalTestState,
  p: {
    walletClient: WalletClient;
    bankClient: TalerCorebankApiClient;
    exchange: ExchangeServiceInterface;
    amount: AmountString | string;
    restrictAge?: number;
    acctname: string;
  },
): Promise<WithdrawViaBankResult> {
  const { walletClient: wallet, bankClient, exchange, amount } = p;
  await bankClient.registerAccountExtended({
    name: p.acctname,
    password: "password",
    username: p.acctname,
  });
  const user = {
    password: "password",
    username: p.acctname,
  };
  const accountPaytoUri = `payto://x-taler-bank/localhost/${p.acctname}?receiver-name=${p.acctname}`;
  const bankClient2 = new TalerCorebankApiClient(bankClient.baseUrl);
  bankClient2.setAuth({
    username: user.username,
    password: user.password,
  });

  const wop = await bankClient2.createWithdrawalOperation(
    user.username,
    amount,
  );

  // Hand it to the wallet

  await wallet.client.call(WalletApiOperation.GetWithdrawalDetailsForUri, {
    talerWithdrawUri: wop.taler_withdraw_uri,
    restrictAge: p.restrictAge,
  });

  // Withdraw (AKA select)

  const acceptRes = await wallet.client.call(
    WalletApiOperation.AcceptBankIntegratedWithdrawal,
    {
      exchangeBaseUrl: exchange.baseUrl,
      talerWithdrawUri: wop.taler_withdraw_uri,
      restrictAge: p.restrictAge,
    },
  );

  const withdrawalFinishedCond = wallet.waitForNotificationCond(
    (x) =>
      x.type === NotificationType.TransactionStateTransition &&
      x.newTxState.major === TransactionMajorState.Done &&
      x.transactionId === acceptRes.transactionId,
  );

  await wallet.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: acceptRes.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.BankConfirmTransfer,
    },
  });

  // Confirm it

  await bankClient2.confirmWithdrawalOperation(user.username, {
    withdrawalOperationId: wop.withdrawal_id,
  });

  return {
    accountPaytoUri,
    withdrawalFinishedCond,
    transactionId: acceptRes.transactionId,
  };
}

runAccountRestrictionsTest.suites = ["wallet"];
