/*
 * Copyright (C) 1999, 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>

#include "genfun.h"

#include "messages.h"
#include "generators.h"
#include "tags.h"
#include "keys.h"
#include "textgen.h"
#include "decorators.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

extern int yylex() ;

extern int tabSpaces; // FIXME

static char *get_input_file_name(char *file_name);

void
processFile( char *progname, char *version,
             char *inputFileName, char *outputFileName, char *docTitle, 
             const char *docHeader, const char *docFooter,
             short entire_doc, char *cssUrl )
{
  FILE *in = 0;
  short deleteOStream = 1 ;

  if ( outputFileName ) {
    sout = new ofstream(outputFileName) ;
    if ( ! sout ) {
      cerr << "Error in creating " << outputFileName << " for output" << endl ;
      exit(1) ;
    }
  }

  if (inputFileName)
    {
      char *input_file_name = get_input_file_name (inputFileName);

      in = freopen (input_file_name, "r", stdin);
      if (!in) 
        {
          cerr << "Error in opening " << inputFileName
               << " for input" << endl ;
          exit(1) ;
        }

      delete input_file_name;
    }

  /*
   * Use default values for any options not provided
   */
  if (sout == 0) {
    sout = &cout;
    deleteOStream = 0 ; // we can't delete cout !!!
  }
  if (in == 0) {
    ; /* Well stdin already points to stdin so, .... */
  }
  if (docTitle == 0) {
    docTitle = inputFileName; /* inputFileName may also be 0,
                                 this is OK. */
  }
  
  if ( entire_doc ) {
    print_top( progname, version, docTitle, cssUrl, docHeader, inputFileName );
  }

  printMessage( "translating source code... ", cerr ) ;

  generateln( "<pre>" ) ;
  generateln( "<tt>" ) ;
  yylex() ;
  generateln( "</tt>" ) ;
  generateln( "</pre>" ) ;

  printMessage( "done !", cerr ) ;
  
  if ( entire_doc )
    print_bottom( docFooter ) ;

  if ( deleteOStream )
    delete sout ;
}

char *
get_input_file_name(char *file_name)
{
  unsigned int length = strlen (CHROOT_INPUT_DIR);

  if (length != 0)
    {
      char *new_name = new char [strlen (file_name) + length + 1];
      strcpy (new_name, strdup (CHROOT_INPUT_DIR));
      return strcat (new_name, file_name);
    }
  else
    return strdup (file_name);
}

void
print_cgi_header()
{
  printf( "Content-type: text/html\n" ) ;
  printf( "\n" ) ;
}

void
print_top( char *progname, char *version, char *docTitle, char *cssUrl,
           const char *docHeader, char *inputFileName)
{
  if( cssUrl == 0 ) {
    generateln( "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">" ) ;
  }
  else {
    generateln( "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\"");
    generateln( "    \"http://www.w3.org/TR/REC-html40/strict.dtd\">");
  }
  generateln( "<html>" ) ;
  generateln( "<head>" ) ;
  generateln( "<meta http-equiv=\"Content-Type\"" ) ;
  generateln( "content=\"text/html; charset=iso-8859-1\">" ) ;
  generate( "<meta name=\"GENERATOR\" content=\"GNU " ) ;
  generate( progname ) ;
  generate( " " ) ;
  generate( version ) ;
  generate ("\npart of GNU ");
  generate (PACKAGE);
  generate (" ");
  generate (VERSION);
  generate( "\nby Lorenzo Bettini, bettini@gnu.org" ) ;
  generate( "\nhttp://w3.newnet.it/bettini" ) ;
  generate( "\nhttp://www.gnu.org/software/" ) ;
  generate( PACKAGE ) ;
  generateln( "\">" ) ;
  generate( "<title>" ) ;
  generate( ( docTitle ? docTitle : 
              ( inputFileName ? inputFileName : "source file" ) ) ) ;
  generateln( "</title>" ) ;
  if( cssUrl != 0 ) {
    generate( "<link rel=\"stylesheet\" href=\"" );
    generate( cssUrl );
    generateln( "\" type=\"text/css\">");
  }
  generateln( "</head>" ) ;
  if( cssUrl == 0 && docHeader == 0) {
    generate ("<body bgcolor=\"#FFFFFF\" text=\"#000000\" link=\"#0000EE\" ");
    generateln ( "vlink=\"#551A8B\" alink=\"#FF0000\">" );
  }
  else {
    generateln( "<body>" ) ;
  }
  if (docHeader)
    generateln (docHeader) ;
}

void print_bottom( const char *docFooter) {
  if ( docFooter ) generateln( docFooter ) ;
  generateln( "</body>" ) ;
  generateln( "</html>" ) ;
}

void
generate( const char *s )
{
  GlobalGenerator->generate(s) ;
}

void
generate( const char *s, int start, int end )
{
  GlobalGenerator->generate(s, start, end) ;
}

void
generateln( const char *s )
{
  GlobalGenerator->generateln(s) ;
}

void
generate_preproc(const char *s)
{
  GlobalGenerator->generate_preproc (s);
}

void generateNewLine() {
  generateln( "" ) ;
}

void generateTab() {
  if ( tabSpaces )
    for ( register int i = 0 ; i < tabSpaces ; ++i )
      generate( SPACE_CHAR ) ;
  else
    generate( "\t" ) ;
}

void startComment( const char *s )
{
  CommentGenerator->beginText(s) ;
}

void endComment( const char *s )
{
  CommentGenerator->endText(s) ;
}

void generateComment( const char *s ) {
  CommentGenerator->generateEntire(s) ;
}

void startString( const char *s )
{
  StringGenerator->beginText(s) ;
}

void endString( const char *s )
{
  StringGenerator->endText(s) ;
}

void generateString( const char *s ) {
  StringGenerator->generateEntire(s) ;
}

void generateKeyWord( const char *s ) {
  KeywordGenerator->generateEntire(s) ;
}

void generateBaseType( const char *s ) {
  TypeGenerator->generateEntire(s) ;
}

void
generatePreProc( const char *s )
{
  PreProcGenerator->generateEntire(s) ;
}

void generateNumber( const char *s ) {
  NumberGenerator->generateEntire(s) ;
}

void
generateSymbol( const char *s )
{
  SymbolGenerator->generateEntire(s);
}

void
generateFunction( const char *s )
{
  FunctionGenerator->generateEntire(s) ;
}

void
generateCBracket( const char *s )
{
  CBracketGenerator->generateEntire(s) ;
}

void startTAG( const char *tag, const char *attr, const char *val ) {
  (*sout) << "<" << tag ;
  if ( attr && val )
    (*sout) << " " << attr << "=" << val ;
  (*sout) << ">" ;
}

void endTAG( const char *tag ) {
  (*sout) << "</" << tag << ">" ;
}

void startColor( const char *color ) {
  startTAG( FONT_TAG, COLOR_TAG, color ) ;
}

void endColor() {
  endTAG( FONT_TAG ) ;
}
