/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright (C) 2000 Edgar Denny (e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "load_data.h"
#include "recipe_win.h"
#include "recipe_file_dlg.h"
#include "messages.h"

static GladeXML *xml = NULL;
static void load_xml( void);

/* load the glade xml if not already loaded. */
static void
load_xml()
{
	static gboolean loaded_xml = FALSE;

	/* load the glade interface. */
	if ( !loaded_xml)
	{
		xml = glade_xml_new( GNUTRITION_GLADEDIR "/recipe_file_dlg.glade", 
				NULL);
		loaded_xml = TRUE;

		if (  xml != NULL)
		{
			glade_xml_signal_autoconnect( xml);
		}
		else
		{
			g_log( "Gnutrition", G_LOG_LEVEL_ERROR,
					"cannot load glade file: recipe_file_dlg.glade\n");
			return;
		}
	}
}

void
gnutr_show_file_open_dlg()
{
	if ( xml == NULL) load_xml();
	gtk_widget_show( glade_xml_get_widget( xml, "file_open_dlg"));
}

/* show the file save dialog. */
void
gnutr_show_file_save_dlg()
{
	if ( xml == NULL) load_xml();
	gtk_widget_show( glade_xml_get_widget( xml, "file_save_dlg"));
}

/* show the file exists dialog. */
void
gnutr_show_file_exists_dlg()
{
	GtkWidget *widget;

	if ( xml == NULL) load_xml();
	widget = glade_xml_get_widget( xml, "file_exists_msgbox");
	gnome_dialog_close_hides( GNOME_DIALOG( widget), TRUE);
	gtk_widget_show( widget);
}

/* show the save current recipe dialog. */
void
gnutr_show_save_cur_recipe_dlg()
{
	GtkWidget *widget;

	if ( xml == NULL) load_xml();
	widget = glade_xml_get_widget( xml, "save_cur_recipe_dlg");
	gnome_dialog_close_hides( GNOME_DIALOG( widget), TRUE);
	gtk_widget_show( widget);
}

/* Called when a button is clicked on the file exists message box. */
void
on_file_exists_msgbox_clicked( GnomeDialog *gnomedialog,
                               gint         button,
                               gpointer     user_data)
{
	switch( button)
	{
		/* cancel button pressed. */
		case 0:
		break;

		/* ok button pressed. */
		case 1:
		{
			Recipe_t *recipe = gnutr_ret_recipe();
			GtkWidget *dlg = glade_xml_get_widget( xml, "file_save_dlg");

			if ( GTK_WIDGET_VISIBLE( dlg))
			{
				gchar *fn = gtk_file_selection_get_filename( 
						GTK_FILE_SELECTION( dlg));

				if ( recipe->file_name != NULL) g_free( recipe->file_name);
				recipe->file_name = g_strdup( fn);
				gnutr_save_recipe_to_file( fn);
				gtk_widget_hide( dlg);
			}
			else if ( recipe->file_name != NULL) /* extra check. */
			{
				gnutr_save_recipe_to_file( recipe->file_name);
			} 
		}
		break;
	}
	gtk_widget_hide( GTK_WIDGET( gnomedialog));
}

/* The "OK" button is released on the file open dialog. */
void
on_file_open_dlg_ok_button_released( GtkButton *button,
                                     gpointer   user_data)
{
	GtkWidget *fdlg = glade_xml_get_widget( xml, "file_open_dlg");
	gchar *fn = gtk_file_selection_get_filename( GTK_FILE_SELECTION( fdlg));
	gchar *home = g_strconcat( g_get_home_dir(), "/", NULL);

	/* check that a file name has been given. */
	if ( strcmp( fn, "") == 0 || strcmp( fn, home) == 0)
	{
		gnutr_show_msg_no_filename_given();
	}
	else if( !g_file_exists( fn))
	{
		/* open file not found message box. */
		gnutr_show_msg_file_not_found();
	}
	else
	{
		gnutr_open_recipe_file( fn);
		gtk_widget_hide( fdlg);
	}
	g_free( home);
}

/* The "Cancel" button is released on the file open dialog. */
void
on_file_open_dlg_cancel_button_released( GtkButton *button,
                                         gpointer   user_data)
{
	gtk_widget_hide( glade_xml_get_widget( xml, "file_open_dlg"));
}

/* The "OK" button is released on the file save dialog. */
void
on_file_save_dlg_ok_button_released( GtkButton *button,
                                     gpointer   user_data)
{
	Recipe_t *recipe = gnutr_ret_recipe();
	GtkWidget *fdlg = glade_xml_get_widget( xml, "file_save_dlg");
	gchar *fn = gtk_file_selection_get_filename( GTK_FILE_SELECTION( fdlg));
	gchar *home = g_strconcat( g_get_home_dir(), "/", NULL);

	/* check that a file name has been given. */
	if ( strcmp( fn, "") == 0 || strcmp( fn, home) == 0)
	{
		gnutr_show_msg_no_filename_given();
	}
	else if( g_file_exists( fn))
	{
		/* show the file exists message box. */
		gnutr_show_file_exists_dlg();
	}
	else
	{
		if ( recipe->file_name != NULL) g_free( recipe->file_name);
		recipe->file_name = g_strdup( fn);
		gnutr_save_recipe_to_file( fn);
		gtk_widget_hide( fdlg);
	}
}

/* The "Cancel" button is released in the file save dialog. */
void
on_file_save_dlg_cancel_button_released( GtkButton *button,
                                         gpointer   user_data)
{
	gtk_widget_hide( glade_xml_get_widget( xml, "file_save_dlg"));
}

/* on "save current recipe?" dialog clicked. */
void
on_save_cur_recipe_dlg_clicked( GnomeDialog *gnomedialog,
                                gint         button,
                                gpointer     user_data)
{	
	Recipe_t *recipe = gnutr_ret_recipe();
	switch( button)
	{
		/* no button pressed. */
		case 0:
			gnutr_free_recipe();
		break;

		/* yes button pressed. */
		case 1:
			if ( recipe->file_name == NULL)
			{
				GtkWidget *fdlg = glade_xml_get_widget( xml, "file_save_dlg");
				gtk_widget_show( fdlg);
				/* FIXME: How do I call gnutr_free_recipe() after
				 * the file_save_dlg is closed? At present when I
				 * save the recipe it is not cleared. */
			}
			else
			{
				gnutr_save_recipe_to_file( recipe->file_name);
				gnutr_free_recipe();
			}
		break;
	}
	gtk_widget_hide( GTK_WIDGET( gnomedialog));
}
