/* -*- Mode: c++ -*-
*******************************************************************************
*
* File:         exp2.cc
* Description:  convert raw ATSC data collected at 21.52 MHz to baseband
*
*******************************************************************************
*/

/*
 * Copyright 2001,2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <make_GrMC4020Source.h>
#include <GrFFTSink.h>
#include <VrFixOffset.h>
#include <VrQuadratureDemod.h>
#include <GrFIRfilterFFF.h>
#include <VrAudioSink.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <VrGUI.h>
#include <gr_firdes.h>
#include <gr_fir_builder.h>
#include <VrNullSink.h>
#include <VrFileSource.h>
#include <VrFileSink.h>
#include <GrReal.h>
#include <GrAtscFPLL.h>
#include <getopt.h>
#include <VrSigSource.h>
#include <atsc_root_raised_cosine_bandpass.h>
#include <atsc_consts.h>
#include <GrAtscBitTimingLoop.h>
#include <atsc_exp2_lp.h>

typedef GrAtscBitTimingLoop BitTimingLoop;


double input_rate = 2 * ATSC_SYMBOL_RATE;
static const double our_IF_freq = 5.75e6;	// IF freq of tuner module
static const double vanu_IF_freq = 5.9815e6;	// IF freq of Vanu test data



static void 
usage (const char *name)
{
  cerr << "usage: " << name
       << " [-g] [-r] [-I] [-D] [-V] [-f <filename>] [-F <filename> ] [-o <filename>]\n";

  cerr << "  -g : no gui\n";
  cerr << "  -r : continuously repeat the file contents (loop)\n";
  cerr << "  -I : sampling rate is 20 MHz not 2*atsc_symbol_rate\n";
  cerr << "  -D : input from ch3, clocked with daughtercard\n";
  cerr << "  -V : IF freq is that of vanu test data\n";
  cerr << "  -P <degrees>  : set initial phase of FPLL and don't update freq\n";
  cerr << "  -f <filename> : take input from file of shorts, else VrSigSource\n";
  cerr << "  -F <filename> : take input from file of floats, else VrSigSource\n";
  cerr << "  -o <filename> : put output in filename, else VrNullSink\n";


  exit (1);
}

int main (int argc, char **argv)
{
  static const double DONT_SET_INITIAL_PHASE = -1;
  
  try {
    
    bool use_gui_p = true;
    bool repeat_p = false;		// continuous loop through file
    bool clk_from_daughter_card_p = false;
    bool input_is_float_p = false;
    char *input_filename = 0;
    char *output_filename = 0;
    double IF_freq = our_IF_freq;
    double initial_phase = DONT_SET_INITIAL_PHASE;

    float mu = -1;		// valid if >= 0
    bool  no_update = false;

    int c;
    while ((c = getopt (argc, argv, "grNDIVf:F:o:M:NP:")) != EOF){
      switch (c){
      case 'g':	use_gui_p = false;    		break;
      case 'r': repeat_p = true;		break;
      case 'I': input_rate = 20e6;		break;
      case 'D': clk_from_daughter_card_p = true;break;
      case 'V': IF_freq = vanu_IF_freq;		break;

      case 'P':
	initial_phase = strtod (optarg, 0);
	break;

      case 'f':
	input_filename = optarg;
	input_is_float_p = false;
	break;

      case 'F':
	input_filename = optarg;
	input_is_float_p = true;
	break;

      case 'o': output_filename = optarg;	break;
      case 'M': mu = strtod (optarg, 0);	break;
      case 'N': no_update = true;		break;

      default:
	usage (argv[0]);
      }
    }

    if (optind != argc)
      usage (argv[0]);

    // freq of hdtv suppressed carrier pilot tone.  
    //
    // This assumes that the tuner is actually tuned to the middle of the 6
    // MHz channel.  The pilot is 0.31 MHz up from the bottom edge of the channel
    
    double pilot_freq = IF_freq - 3e6 + 0.31e6;


    VrGUI *guimain = 0;
    VrGUILayout *horiz = 0;
    VrGUILayout *vert = 0;
    VrGUILayout *vert2 = 0;

    if (use_gui_p){
      guimain = new VrGUI(argc, argv);
      horiz = guimain->top->horizontal();
      vert = horiz->vertical();
      vert2 = horiz->horizontal()->vertical();
    }

    cerr << "Input Sampling Rate: " << input_rate << endl;
    cerr << "Pilot Freq: " << pilot_freq << endl;

    VrSigProc		*vsrc;		// virtual source
    VrSource<short>	*source = 0;
    VrSink<float>	*final_sink = 0;

    VrSink<float>	*input_fft = 0;
    VrSink<float>	*matched_fft = 0;
    VrSink<float>	*lowpass_fft = 0;
    VrSink<float>	*bt_fft = 0;
    

    //
    // ================================================================
    //   Get data either from the
    //		SigSource or
    //		an external file or
    //		the ADC ch3 clocked with the daughter card
    // ================================================================
    //

    if (input_filename && clk_from_daughter_card_p){
      cerr << "-f <filename> and -D are mutually exclusive\n";
      exit (1);
    }

    if (input_filename){
      if (!input_is_float_p){	// shorts
	
	// --> float
	source = new VrFileSource<short>(input_rate, input_filename, repeat_p);
	VrFixOffset<short,float> *offset_fixer = new VrFixOffset<short,float>();

	NWO_CONNECT (source, offset_fixer);
	vsrc = offset_fixer;
      }
      else {
	vsrc = new VrFileSource<float>(input_rate, input_filename, repeat_p);
      }
    }
    else if (clk_from_daughter_card_p){
      source = make_GrMC4020SourceS(2 * ATSC_SYMBOL_RATE,
					 MCC_CH3_EN
					 | MCC_ALL_1V
					 | MCC_CLK_AD_START_TRIG_IN);

      VrFixOffset<short,float> *offset_fixer = new VrFixOffset<short,float>();

      NWO_CONNECT (source, offset_fixer);
      vsrc = offset_fixer;
    }
    else {
      // --> float
      vsrc = new VrSigSource<float>(input_rate, VR_SIN_WAVE, 3.06e6, 1);
    }

    if (use_gui_p){
      // sink0 is raw data
      input_fft = new GrFFTSink<float>(vert, 0, 80, 512);
    }

    // ================================================================
    //  apply the band pass matched filter (root raised cosine)
    //  (selects only the data of interest also...)
    // ================================================================

    GrFIRfilterFFF *matched_filter =
      new GrFIRfilterFFF (1, new atsc_root_raised_cosine_bandpass (IF_freq));


    if (use_gui_p){
      // sink1 is the band passed data
      matched_fft = new GrFFTSink<float>(vert, 0, 80, 512);
    }
    

    // ================================================================
    // Build the FPLL to track the carrier and down convert
    // ================================================================

    // float --> float
    GrAtscFPLL *fpll = new GrAtscFPLL (pilot_freq + 5e3);

    if (initial_phase != DONT_SET_INITIAL_PHASE){
      fpll->set_initial_phase (initial_phase * M_PI/180);
      fpll->set_no_update (true);
    }

    // ================================================================
    // low pass to kill the 2x carrier term here...
    // ================================================================

    // build low pass filter.

#if 0

    double transition_width = 0;

    // The lower edge of the transition band is 6 MHz (the channel width)
    // minus the offset to the pilot .31 MHz.  This is because we've
    // translated the pilot to DC in the FPLL.

    double lower_edge = 6e6 - 0.31e6;

    // The upper edge of the transition band is at twice the pilot.
    // We want the image of the pilot killed.  We could actually
    // set the upper edge at 2.75 (original band edge) + 3.06 (pilot_freq)
    // but that gives an even narrower transition band (more expensive).
    
    double upper_edge = IF_freq - 3e6 + pilot_freq; 	// tight spec
    // double upper_edge = 2 * pilot_freq;		// loose spec
    transition_width = upper_edge - lower_edge;

    vector<float> lp_coeffs =
      gr_firdes::low_pass (1.0,
			   input_rate,
			   (lower_edge + upper_edge) * 0.5,
			   transition_width,
			   gr_firdes::WIN_HAMMING);

    cerr << "Number of lp_coeffs: " << lp_coeffs.size () << endl;

    // float --> float
    GrFIRfilterFFF* lp_filter =
      new  GrFIRfilterFFF(1, lp_coeffs);

#else

    GrFIRfilterFFF* lp_filter = new GrFIRfilterFFF (1, new atsc_exp2_lp ());

#endif

    if (use_gui_p){
      lowpass_fft = new GrFFTSink<float>(vert2, -20, 60, 512);
    }
    
    // ================================================================
    // how add in bit timing loop...
    // ================================================================

    BitTimingLoop *bt_loop = 0;
    // float --> float
    bt_loop = new BitTimingLoop ();

    if (mu >= 0)
      bt_loop->set_mu (mu);

    cerr << "mu: " << mu << endl;

    bt_loop->set_no_update (no_update);
    cerr << "no_update: "<< no_update << endl;

    if (use_gui_p){
      bt_fft = new GrFFTSink<float>(vert2, -20, 60, 512);
    }

    // ================================================================
    // final sink is either a file sink or a null sink
    // ================================================================

    if (output_filename)
      final_sink = new VrFileSink<float>(output_filename);

    else
      final_sink = new VrNullSink<float>();


    // connect the modules together

    VrMultiTask *m = new VrMultiTask ();
    m->add (final_sink);

    NWO_CONNECT (vsrc, matched_filter);
    NWO_CONNECT (matched_filter, fpll);
    NWO_CONNECT (fpll, lp_filter);
    NWO_CONNECT (lp_filter, bt_loop);
    NWO_CONNECT (bt_loop, final_sink);

    
    if (use_gui_p){
      NWO_CONNECT (vsrc, input_fft);
      NWO_CONNECT (matched_filter, matched_fft);
      NWO_CONNECT (lp_filter, lowpass_fft);
      NWO_CONNECT (bt_loop, bt_fft);

      m->add (input_fft);
      m->add (matched_fft);
      m->add (lowpass_fft);
      m->add (bt_fft);
    }

    m->start ();
    if (use_gui_p)
      guimain->start ();

    while (1){
      if (use_gui_p)
	guimain->processEvents(10 /*ms*/);
      m->process();
    }	

  } // end try

  catch (std::exception &e){
    cerr << "std library exception: " << e.what () << endl;
    exit (1);
  }
  catch (...) {
    cerr << "unknown exception thrown" << endl;
    exit (1);
  }
}
