// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "server.h"

#ifndef	WIN32
#include <sys/utsname.h>
#endif

namespace server {
using namespace ost;
using namespace std;

#ifdef	WIN32
#define	PROMPT_FILES	"C:\\Program Files\\GNU Telephony\\Phrasebook Audio"
#define	SCRIPT_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Scripts"
#define	MACRO_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Macros"
#define	VAR_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Files"
#define	LOG_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Logs"
#define	CONFIG_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Config"
#define	RUN_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Logs"
#define	LIBEXEC_FILES	"C:\\Program Files\\GNU Telephony\\Bayonne Libxec"

#ifdef	_DEBUG
#define	LIBDIR_FILES	"C:\\Program Files\\GNU Telephony\\CAPE Framework\\debug"
#define	MODULE_FILES	LIBDIR_FILES
#define	DRIVER_FILES	LIBDIR_FILES
#define	PHRASE_FILES	LIBDIR_FILES	
#else	
#define	LIBDIR_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Runtime"
#define	MODULE_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Plugins"
#define	DRIVER_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Drivers"
#define	PHRASE_FILES	"C:\\Program Files\\Common Files\\GNU Telephony\\Bayonne Languages"
#endif

#else
#define	MODULE_FILES	LIBDIR_FILES
#define	DRIVER_FILES	LIBDIR_FILES
#define	PHRASE_FILES	LIBDIR_FILES
#endif

static Keydata::Define paths[] = {
	{"libexec", LIBEXEC_FILES},
	{"datafiles", VAR_FILES},
	{"logfiles", LOG_FILES},
	{"runfiles", RUN_FILES},
	{"config", CONFIG_FILES},
	{"scripts", SCRIPT_FILES},
	{"macros", MACRO_FILES},
	{"prompts", PROMPT_FILES},
#ifndef	WIN32
	{"btsexec", BIN_FILES "/btsexec"},
#endif
	{NULL, NULL}};

static Keydata::Define server[] = {
	{"user", "bayonne"},
	{"group", "bayonne"},
	{"language", "none"},
	{"location", "us"},
	{"voice", "none/prompts"},
	{"state", "init"},
	{"logging", "warn"},
	{"monitoring", "tcp"},
	{"binding", "default"},
	{"definitions", "url"},
	{NULL, NULL}};

static Keydata::Define engine[] = {
	{"priority", "0"},
	{"scheduler", "default"},
	{"events", "32"},
	{"server", "localhost"},
	{"userid", ""},
	{"secret", ""},
	{"timeslots", "32"},
	{"driver", "autodetect"},
	{"protocol", "none"},
	{NULL, NULL}};

static Keydata::Define libkeys[] = {
	{"filesystem", VAR_FILES},
	{"interface", "eth0"},
	{"limit", "0"},
	{NULL, NULL}};

static Keydata::Define smtpkeys[] = {
	{"limit", "0"},
	{"sendmail", "/usr/sbin/sendmail"},
	{"sender", "bayonne@localhost.localdomain"},
	{"from", "Bayonne Server <bayonne@localhost.localdomain>"},
	{"errors", "postmaster@localhost.localdomain"}, 
	{NULL, NULL}};

static Keydata::Define urlkeys[] = {
	{"limit", "0"},
	{"prefix", "http://localhost"},
	{"agent", "bayonne/" VERSION},
	{NULL, NULL}};

#ifdef	WIN32
Keydata keypaths(paths, "/bayonne/paths");
Keydata keyserver(server, "/bayonne/server");
Keydata keyengine(engine, "/bayonne/engine");
BayonneConfig keylibexec("libexec", libkeys, "/bayonne/libexec");
BayonneConfig keysmtp("smtp", smtpkeys, "/bayonne/smtp");
BayonneConfig keyurl("url", urlkeys, "/bayonne/url");
#else
Keydata keypaths(paths, "/bayonne/server/paths");
Keydata keyserver(server, "/bayonne/server/server");
Keydata keyengine(engine, "/bayonne/server/engine");
BayonneConfig keylibexec("libexec", libkeys, "/bayonne/server/libexec");
BayonneConfig keysmtp("smtp", smtpkeys, "/bayonne/server/smtp");
BayonneConfig keyurl("url", urlkeys, "/bayonne/server/url");

#endif
Keydata keyoptions;

void liveConfig(void)
{
	char path[256];
	char lang[32];
	const char *env;
	char *p, *sp, *tok;
	const char *cp;
	unsigned timeslots = atoi(keyengine.getLast("timeslots"));
	Bayonne::timeslot_t overdraft = 1;

#ifdef	WIN32
        char nodename[MAX_COMPUTERNAME_LENGTH + 1];
        DWORD namelen = MAX_COMPUTERNAME_LENGTH + 1;
        SYSTEM_INFO sysinfo;
        OSVERSIONINFO osver;
	const char *cpu = "x86";
	const char *tmp = Process::getEnv("TEMP");
	if(!tmp)
		tmp = Process::getEnv("TMP");
	if(!tmp)
		tmp = "C:\\TEMP";
	if(!keypaths.getLast("tmp"))
		keypaths.setValue("tmp", tmp);
	if(!keypaths.getLast("tmpfs"))
		keypaths.setValue("tmpfs", tmp);
	Dir::create(tmp);

        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        GetComputerName(nodename, &namelen);
        keyserver.setValue("node", nodename);
        GetSystemInfo(&sysinfo);
        switch(sysinfo.wProcessorArchitecture)
        {
        case PROCESSOR_ARCHITECTURE_INTEL:
		cpu="x86";
                break;
        case PROCESSOR_ARCHITECTURE_PPC:
                cpu = "ppc";
                break;
        case PROCESSOR_ARCHITECTURE_MIPS:
                cpu = "mips";
                break;
        case PROCESSOR_ARCHITECTURE_ALPHA:
                cpu = "alpha";
                break;
        }
	keyserver.setValue("node", nodename);
	keyserver.setValue("platform", "w32");
	keyserver.setValue("cpu", cpu);

#else
	struct utsname uts;
	char hostname[128];
	char *userid;
	uname(&uts);
	memset(hostname, 0, sizeof(hostname));
	gethostname(hostname, sizeof(hostname) - 1);
	hostname[sizeof(hostname) - 1] = 0;
	char *hp = strchr(hostname, '.');
	if(hp)
		*hp = 0;

	if(!keyserver.getLast("node"))
		keyserver.setValue("node", hostname);
	
	keyserver.setValue("platform", uts.sysname);
	keyserver.setValue("cpu", uts.machine);
	userid = getenv("USER");

	if(!userid || !stricmp(userid, "root"))
		userid="bayonne";	

	snprintf(path, sizeof(path), "/tmp/bayonne-%s", userid);
	keypaths.setValue("tmp", path);
	purgedir(path);
	Dir::create(path);

	if(canModify("/dev/shm"))
	{
		snprintf(path, sizeof(path), "/dev/shm/bayonne-%s", userid);
		keypaths.setValue("tmpfs", path);
		purgedir(path);
		Dir::create(path);
	}
	else
		keypaths.setValue("tmpfs", path);

#endif
        sp = (char *)keyserver.getLast("definitions");

	if(sp && !stricmp(sp, "default"))
	{
		sp = NULL;
		cp = Process::getEnv("DEFINES");
		if(cp)
		{
			setString(path, sizeof(path), cp);
			sp = path;
		}
	}

        if(sp && !stricmp(sp, "none"))
                sp = NULL;

	if(sp && !stricmp(sp, "off"))
	{
		sp = NULL;
		Bayonne::provision_defines = false;
	}

        if(sp && !*sp)
                sp = NULL;
        if(sp)
                cp = strtok_r(sp, " ,;:\t", &tok);
        else
                cp = NULL;

        while(cp)
        {
                if(strrchr(cp, '.'))
                        setString(path, sizeof(path), cp);
                else
                        snprintf(path, sizeof(path), "%s.def", cp);
                runtime.setValue("definitions", path);
                cp = strtok_r(NULL, " ,;:\t", &tok);
        }

	cp = keyoptions.getLast("prefix");
#ifdef	WIN32
	if(Bayonne::provision_test)
		cp = NULL;
#endif
	if(cp && *cp)
		chdir(cp);

	env = Process::getEnv("LANG");
	if(env)
	{
		snprintf(lang, sizeof(lang), "%s", env);
		p = strchr(lang, '.');
		if(p)
			*p = 0;
		p = strchr(lang, '_');
		if(p)
		{
			p[1] = tolower(p[1]);
			p[2] = tolower(p[2]);
			if(isalpha(p[1]) && isalpha(p[2]) && !p[3])
				keyserver.setValue("location", ++p);
		}
		lang[0] = tolower(lang[0]);
		lang[1] = tolower(lang[1]);
		if(isalpha(lang[0]) && isalpha(lang[1]))
			keyserver.setValue("language", lang);
	}
#ifdef	WIN32
	snprintf(path, sizeof(path), "%s/tones.ini", keypaths.getLast("config"));
#else
	snprintf(path, sizeof(path), "%s/tones.conf", keypaths.getLast("config"));
#endif
	TelTone::load(path);
	keypaths.setValue("tones", path);

#ifdef	WIN32
#define	EVENTS	"%s/%s.evt"
#define	ERRORS	"%s/%s.err"
#define	OUTPUT	"%s/%s.out"
#define	CALLER	"%s/%s.cdr"
#define	STATS	"%s/%s.sta"
#else
#define	EVENTS	"%s/%s.events"
#define	ERRORS	"%s/%s.errors"
#define	OUTPUT	"%s/%s.output"
#define	CALLER	"%s/%s.calls"
#define	STATS	"%s/%s.stats"
#endif

	snprintf(path, sizeof(path), ERRORS,
		keypaths.getLast("logfiles"), keyserver.getLast("node"));
	runtime.setValue("errlog", path);

        snprintf(path, sizeof(path), EVENTS,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
        runtime.setValue("evtlog", path); 

        snprintf(path, sizeof(path), OUTPUT,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
       	runtime.setValue("output", path);

        snprintf(path, sizeof(path), CALLER,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
        runtime.setValue("calls", path);

        snprintf(path, sizeof(path), STATS,
                keypaths.getLast("logfiles"), keyserver.getLast("node"));
        runtime.setValue("stats", path);

	runtime.setValue("runlevel.conf", CONFIG_FILES "/runlevel.conf");
	runtime.setValue("include", keypaths.getLast("macros"));
	keypaths.setValue("include", keypaths.getLast("macros"));
	runtime.setValue("tmp", keypaths.getLast("tmp"));
	runtime.setValue("tmpfs", keypaths.getLast("tmpfs"));
	runtime.setValue("prompts", keypaths.getLast("prompts"));
	runtime.setValue("prefix", keypaths.getLast("datafiles"));
	runtime.setValue("config", keypaths.getLast("config"));
	runtime.setValue("voice", keyserver.getLast("voice"));
	runtime.setValue("logfiles", keypaths.getLast("logfiles"));
	runtime.setValue("runfiles", keypaths.getLast("runfiles"));
	runtime.setValue("servername", "bayonne2");
	runtime.setValue("serverversion", VERSION);
	
	cp = keyengine.getLast("server");
	if(cp)
		runtime.setValue("server", cp);
	cp = keyengine.getLast("proxy");
	if(cp)
		runtime.setValue("proxy", cp);

	runtime.setValue("userid", keyengine.getLast("userid"));
	runtime.setValue("secret", keyengine.getLast("secret"));
	runtime.setValue("location", keyserver.getLast("location"));

	cp = keyengine.getLast("autostack");
	if(cp)
		Thread::setStack(atoi(cp) * 1024);

	cp = keyengine.getLast("stepping");
	if(cp)
		Bayonne::step_timer = atoi(cp);

	cp = keyengine.getLast("reset");
	if(cp)
		Bayonne::reset_timer = atoi(cp);

	cp = keyengine.getLast("exec");
	if(cp)
		Bayonne::exec_timer = atoi(cp) * 1000;

	cp = keyengine.getLast("autostep");
	if(cp)
		Script::autoStepping = atoi(cp);

	cp = keyengine.getLast("faststep");
	if(cp)
		Script::fastStepping = atoi(cp);

	cp = keyengine.getLast("paging");
	if(!cp)
		cp = keyengine.getLast("pagesize");
	if(cp)
	{
		Script::pagesize = atoi(cp);
		Script::symlimit = Script::pagesize - sizeof(Script::Symbol) - 32;
	}

	cp = keyengine.getLast("symsize");
	if(cp)
		Script::symsize = atoi(cp);	

	Script::fastStart = false;

#ifdef	WIN32
	Script::exec_extensions = SCRIPT_EXTENSIONS;
#else
	Script::exec_extensions = SCRIPT_EXTENSIONS;
#endif
	Script::exec_token = "bayonne:";

	cp = keyengine.getLast("virtual");
	if(cp)
		overdraft = atoi(cp);

        Bayonne::allocate(timeslots, dynamic_cast<ScriptCommand *>(&runtime), overdraft); 
}

#ifdef	HAVE_TESTING

void testConfig(char *argv0)
{
#ifdef	WIN32
	keypaths.setValue("prompts", "../audio");
	keypaths.setValue("scripts", "../scripts");
	keypaths.setValue("macros", "../scripts");
	keypaths.setValue("config", "../config");
	keypaths.setValue("datafiles", "tempfiles");
	keypaths.setValue("logfiles", "tempfiles");
	keypaths.setValue("runfiles", "tempfiles");
	Process::setEnv("DRIVERCONFIG", "../config/driver.ini", true);
	Process::setEnv("MODULECONFIG", "../config/module.ini", true);
	Process::setEnv("SPANCONFIG", "../config/spans.ini", true);
#else
	char pbuf[256];
	getcwd(pbuf, sizeof(pbuf));
	keyoptions.setValue("prefix", "");
	keypaths.setValue("libexec", pbuf);
	keypaths.setValue("btsexec", SOURCE_FILES "/utils/btsexec");
	keypaths.setValue("prompts", SOURCE_FILES "/audio");
	keypaths.setValue("scripts", SOURCE_FILES "/scripts");
	keypaths.setValue("macros", SOURCE_FILES "/scripts");
	keypaths.setValue("config", SOURCE_FILES "/config");
	keypaths.setValue("datafiles", SOURCE_FILES "/server/tempfiles");
	keypaths.setValue("logfiles", SOURCE_FILES "/server/tempfiles");
	keypaths.setValue("runfiles", SOURCE_FILES "/server/tempfiles");
#endif
	Bayonne::provision_test = true;
	Bayonne::provision_daemon = false;
	runtime.setValue("virtual.tts", "theta");
	keypaths.setValue("modexec", "../scripts");
	Dir::create("tempfiles");
	keyserver.setValue("state", "test");
	keyserver.setValue("logging", "debug");
	keyserver.setValue("monitoring", "no");
}	

#endif

void loadConfig(void)
{
#ifdef	WIN32
	char pbuf[256];
	const char *config = keypaths.getLast("config");

	snprintf(pbuf, sizeof(pbuf), "%s/server.ini", config); 
	keypaths.loadFile(pbuf, "paths");
	keyserver.loadFile(pbuf, "server");
	keyengine.loadFile(pbuf, "engine");
	keylibexec.loadFile(pbuf, "libexec");
	keysmtp.loadFile(pbuf, "smtp");
	keyurl.loadFile(pbuf, "url");
	runtime.loadFile(pbuf, "virtual", "virtual");

	snprintf(pbuf, sizeof(pbuf), "%s/driver.ini", config);
	Process::setEnv("DRIVERCONFIG", pbuf, true); 
	snprintf(pbuf, sizeof(pbuf), "%s/module.ini", config);
	Process::setEnv("MODULECONFIG", pbuf, true);

//	Process::setGroup(keyserver.getLast("group"));
        Dir::create(keypaths.getLast("datafiles"));
        Dir::create(keypaths.getLast("runfiles"));
        Dir::create(keypaths.getLast("logfiles"));
#else
	char homepath[256];
	const char *home = NULL;
	bool homed = false;

	if(getuid())
		home = Process::getEnv("HOME");

	if(!isDir(keypaths.getLast("config")))
		keypaths.setValue("config", "/bayonne");

	if(home)
		snprintf(homepath, sizeof(homepath), "%s/.bayonne", home);
	else
		strcpy(homepath, "tempfiles");

	runtime.loadPrefix("virtual", "/bayonne/server/virtual");
	if(Bayonne::provision_user)
	{
		keypaths.load("~bayonne/paths");
		keyserver.load("~bayonne/server");
		keyengine.load("~bayonne/engine");
		keysmtp.load("~bayonne/smtp");
		keyurl.load("~bayonne/url");
		runtime.loadPrefix("virtual", "~bayonne/virtual");
	}

	if(Bayonne::provision_system)
	{
		keypaths.load("/bayonne/provision/paths");
		keyserver.load("/bayonne/provision/server");
		keyengine.load("/bayonne/provision/engine");
		keylibexec.load("/bayonne/provision/libexec");
		keysmtp.load("/bayonne/provision/smtp");
		keyurl.load("/bayonne/provision/url");
		runtime.loadPrefix("virtual", "/bayonne/provision/virtual");
	}

	umask(007);
	Process::setGroup(keyserver.getLast("group"));
	Dir::create(keypaths.getLast("datafiles"));
	Dir::create(keypaths.getLast("runfiles"));
	Dir::create(keypaths.getLast("logfiles"));

	keypaths.setValue("home", keypaths.getLast("datafiles"));

	if(!getuid())
		goto live;

	if(!canModify(keypaths.getLast("datafiles")))
	{
		homed = true;
		keypaths.setValue("datafiles", homepath);
	}

	keypaths.setValue("control", keypaths.getLast("runfiles"));
	if(!canModify(keypaths.getLast("runfiles")))
	{
		homed = true;
		keypaths.setValue("runfiles", homepath);
	}

	if(!canModify(keypaths.getLast("logfiles")))
	{
		homed = true;
		keypaths.setValue("logfiles", homepath);
	}

	if(homed)
	{
		umask(077);
	        Dir::create(homepath, File::attrPrivate);
	}
live:
#endif
	keyoptions.setValue("prefix", keypaths.getLast("datafiles"));
	keyserver.setValue("state", "live");

#ifdef	SCRIPT_SERVER_PREFIX
	Script::var_prefix = keypaths.getLast("datafiles");
	Script::etc_prefix = keypaths.getLast("config");
	Script::log_prefix = keypaths.getLast("logfiles");
#endif

	// only for live server...
}

bool parseConfig(char **argv)
{
	char buffer[64];
	char path[256];
	const char *prefix;
	char *option;
	char *script = NULL;
	char *argv0 = *(argv++);
	char *ext;
	unsigned scrcount = 0;
	unsigned minslots = 0;
	bool live = true;

#ifndef	WIN32
	const char *cp;
	cp = keyengine.getLast("driver");
	if(cp && !stricmp(cp, "autodetect"))
		cp = Process::getEnv("TIMESLOTS");
	else if(cp && !stricmp(cp, "default"))
		cp = Process::getEnv("TIMESLOTS");

	if(cp && atoi(cp) > 0)
		keyengine.setValue("timeslots", cp);
#endif

retry:
	if(!*argv)
		goto skip;

	option = *argv;
        if(!strcmp("--", option))
        {
                ++argv;
                goto skip;
        }

        if(!strnicmp("--", option, 2))
                ++option;

	if(!strnicmp(option, "-driver=", 8))
	{
		keyengine.setValue("driver", option + 8);
		++argv;
		goto retry;
	} 

	if(!strnicmp(option, "-protocol=", 10))
	{
		keyengine.setValue("protocol", option + 10);
		++argv;
		goto retry;
	}

        if(!strnicmp(option, "-bind=", 6))
        {
                keyserver.setValue("binding", option + 6);    
                ++argv;
                goto retry;
        }  

#ifdef	HAVE_TESTING
        if(!strnicmp(option, "-test=", 6))
        {
		testConfig(argv0);
                keyengine.setValue("driver", option + 6);
                ++argv;
                goto retry;
        }  
	
	if(!stricmp(option, "-test"))
	{
		++argv;
		if(!*argv)
		{
			cerr << "bayonne: -test: missing driver" << endl;
			exit(-1);
		}
		testConfig(argv0);
		keyengine.setValue("driver", *(argv++));
		goto retry;
	}

	if(!strnicmp(option, "-logging=", 9))
        {
                keyserver.setValue("logging", option + 9);
                ++argv;
                goto retry;
        }

	if(!strnicmp(option, "-scripts=", 9))
	{
		keypaths.setValue("scripts", option + 9);
		++argv;
		goto retry;
	}

	if(!strnicmp(option, "-macros=", 8))
	{
		keypaths.setValue("macros", option + 8);
		++argv;
		goto retry;
	} 

        if(!strnicmp(option, "-libexec=", 9))
        {
                keypaths.setValue("libexec", option + 9);
                ++argv;
                goto retry;
        }
#endif            

	if(!stricmp(option, "-check"))
	{
		keyserver.setValue("logging", "debug");
		Bayonne::provision_check = true;
		++argv;
		goto retry;
	}

	if(!strnicmp(option, "-vvv", 4))
	{
		Bayonne::provision_daemon = false;
		keyserver.setValue("logging", "debug");
		++argv;
		goto retry;
	}

        if(!strnicmp(option, "-vv", 4))
        {
		Bayonne::provision_daemon = false;
                keyserver.setValue("logging", "notice");
                ++argv;
                goto retry;
        }

        if(!stricmp(option, "-v"))
        {
		Bayonne::provision_daemon = false;
                keyserver.setValue("logging", "error");
                ++argv;
                goto retry;
        }

	if(!stricmp(option, "-p"))
	{
		keyengine.setValue("priority", "2");
		keyengine.setValue("scheduler", "fifo");
		++argv;
		goto retry;
	}

	if(!strnicmp(option, "-voice=", 7))
	{
		keyserver.setValue("voice", option + 7);
		++argv;
		goto retry;
	} 

	if(!strnicmp(option, "-location=", 10))
	{ 
		Process::setEnv("LANG", option + 10, true);
		keyserver.setValue("location", option + 10);
		++argv;
		goto retry;
	}

        if(!strnicmp(option, "-timeslots=", 11))
        {
		keyengine.setValue("timeslots", option + 11);
                ++argv;
                goto retry;
        }

#ifdef	HAVE_TESTING
        if(!stricmp(option, "-libexec"))
        {
                ++argv;
                if(!*argv)
                {
                        cerr << "bayonne: -libexec: missing argument" << endl;
                        exit(-1);
                }
                keypaths.setValue("libexec", *(argv++));
                goto retry;
        }

        if(!stricmp(option, "-scripts"))     
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: -scripts: missing argument" << endl;
                        exit(-1);
                }
                keypaths.setValue("scripts", *(argv++));
                goto retry;
        }
           

        if(!stricmp(option, "-macros"))     
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: -macros: missing argument" << endl;
                        exit(-1);
                }
                keypaths.setValue("macros", *(argv++));
                goto retry;
        } 

	if(!stricmp(option, "-logging"))
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: -logging: missing argument" << endl;
                        exit(-1);
                }
                keyserver.setValue("logging", *(argv++));
                goto retry;
        }
#endif

        if(!stricmp(option, "-location"))  
        {
                ++argv;
                if(!argv) 
                {
        		cerr << "bayonne: -location: missing argument" << endl;
                        exit(-1);
                }      
		Process::setEnv("LANG", *argv, true);
                keyserver.setValue("location", *(argv++));
                goto retry;
        }


	if(!stricmp(option, "-voice"))
	{
		++argv;
		if(!argv)
		{
			cerr << "bayonne: -voice: missing argument" << endl;
			exit(-1);
		}
		keyserver.setValue("voice", *(argv++));
		goto retry;
	}

        if(!stricmp(option, "-driver"))
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: -driver: missing argument" << endl;
                        exit(-1);
                }
                keyengine.setValue("driver", *(argv++));
                goto retry;
        }

        if(!stricmp(option, "-protocol"))
        {
                ++argv;
                if(!*argv)
                {
                        cerr << "bayonne: -protocol: missing argument" << endl;
                        exit(-1);
                }
                keyengine.setValue("protocol", *(argv++));
                goto retry;
        }

        if(!stricmp(option, "-bind"))
        {
                ++argv;    
                if(!*argv)
                {
                        cerr << "bayonne: -binding: missing argument" << endl;
                        exit(-1);
                }
                keyserver.setValue("binding", *argv);
		++argv;
                goto retry;
        } 

        if(!stricmp(option, "-timeslots"))
        {
                ++argv;
                if(!*argv)
                {
			cerr << "bayonne: -timeslots: missing argument" << endl;
                        exit(-1);
                }
		keyengine.setValue("timeslots", *argv);
		++argv;
                goto retry;
	}

	if(**argv == '-')
	{
		cerr << "bayonne: " << *argv << ": unknown option" << endl;
		exit(-1);
	}

skip:
	while(*argv)
	{
		ext = strrchr(*argv, '.');
		if(!ext)
			keyoptions.setValue("modules", *argv);
		else if(!stricmp(ext, ".conf") || !stricmp(ext, ".ini"))
			slog.error("bayonne: %s: invalid script file", *argv);
		else if(!stricmp(ext, ".mac") || !stricmp(ext, ".scr") || strstr(Script::exec_extensions, ext) || !stricmp(ext, ".def"))
		{
			prefix = strrchr(*argv, '/');
			if(!prefix)
				prefix = strrchr(*argv, '\\');
			if(!prefix)
				prefix = strrchr(*argv, ':');

			if(prefix)
				++prefix;

			if(!prefix)
			{
				getcwd(path, sizeof(path));
				addString(path, sizeof(path), "/");
				addString(path, sizeof(path), *argv);
				prefix = path;
			}

			if(!canAccess(*argv))
				cerr << "bayonne: " << *argv << ": cannot access" << endl;
			else
			{
				if(!scrcount++)
					script = *argv;
				else
					script = NULL;

				if(!stricmp(ext, ".def"))
					runtime.setValue("definitions", prefix);
				else
					runtime.setValue(*argv, prefix);
			}
		}
		else
		{
			if(canAccess(*argv))
				keyoptions.setValue("configs", *argv);
			else
				cerr << "bayonne: " << *argv << ": cannot access" << endl;
		}
		++argv;
	}

	if(!script) 
	{
		if(!runtime.getLast("scripts"))
			runtime.setValue("scripts", keypaths.getLast("scripts"));
		if(keypaths.getLast("addons"))
			runtime.setValue("addons", keypaths.getLast("addons"));
		return false;
	}

	snprintf(path, sizeof(path), "%s", script);
	ext = strrchr(path, '/');
	if(!ext)
		ext = strrchr(path, '\\');
	if(ext)
		*ext = 0;
	else
		getcwd(path, sizeof(path));
	Process::setEnv("SERVER_SYSEXEC", path, true);
	option = strrchr(script, '/');
	if(!option)
		option = strrchr(script, '\\');
	if(!option)
		option = strrchr(script, ':');
	if(option)
		script = ++option;
	ext = strrchr(script, '.');
	if(ext && strstr(Script::exec_extensions, ext))
		snprintf(buffer, sizeof(buffer), "exec::%s", script);
	else
		setString(buffer, sizeof(buffer), script);
	script = buffer;
	ext = strrchr(script, '.');
	if(ext && !strstr(Script::exec_extensions, ext))
		*ext = 0;

	runtime.setValue("startup", script);
	return true;
}		

#ifdef	HAVE_TESTING
void dumpConfig(Keydata &keydef)
{
	unsigned count = keydef.getCount();
	char **keys = new char *[count + 1];
	count = keydef.getIndex(keys, count + 1);
	while(count--)
	{
		cout << *keys << " = " << keydef.getLast(*keys) << endl;
		++keys;
	}
}
#endif

void purgedir(const char *dir)
{
	const char *cp;
	if(isDir(dir) && canAccess(dir))
	{
		DirTree dt(dir, 16);
		while(NULL != (cp = dt.getPath()))
			remove(cp);
		remove(dir);
	}
}

} // namespace
