// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#define	BUILD_LIBEXEC
#include "engine.h"

#ifdef	HAVE_LIBEXEC
#include <cc++/process.h>
#include <cc++/slog.h>
#include <cc++/export.h>
#include "libexec.h"

using namespace ost;
using namespace std;

Keydata *Libexec::head = NULL;
Keydata *Libexec::args = NULL;
const char *Libexec::tsid = NULL;
bool Libexec::eof = false;
char Libexec::digits[64] = "";
char Libexec::position[16] = "00:00:00.000";
int Libexec::result = -1;
unsigned Libexec::exitcode = 3;	// server never started

void Libexec::init(void)
{
	const char *cp;
	char *p;
	char lbuf[256];

	if(tsid)
		return;

#ifndef	WIN32
	Process::setPosixSignal(SIGPIPE, SIG_IGN);
#endif

	tsid = Process::getEnv("PORT_TSESSION");
	if(!tsid)
		return;

	head = new Keydata();
	args = new Keydata();	 

	cout << tsid << " head" << endl;
	for(;;)
	{
		cin.getline(lbuf, sizeof(lbuf));
		if(cin.eof())
		{
			eof = true;
			break;
		}
		cp = lbuf;
		p = strchr(lbuf, '\n');
		if(p)
			*p = 0;
		p = strchr(lbuf, '\r');
		if(p)
			*p = 0;
		if(!*cp)
			break;
		if(atoi(cp) >= 900)
		{
			exitcode = atoi(cp) - 900;
			eof = true;
			break;
		}
		if(!isalpha(*cp))
			continue;
		p = strchr(lbuf, ':');
		if(!p)
			continue;
		*p = 0;
		head->setValue(lbuf, p + 2);		
	}

	cout << tsid << " args" << endl;
	while(!eof)
	{
		cin.getline(lbuf, sizeof(lbuf));
		if(cin.eof())
		{
			eof = true;
			break;
		}
		cp = lbuf;
		p = strchr(lbuf, '\n');
		if(p)
			*p = 0;
		p = strchr(lbuf, '\r');
		if(p)
			*p = 0;
		if(!*cp)
			break;
		if(atoi(cp) >= 900)
		{
			exitcode = atoi(cp) - 900;
			eof = true;
			break;
		}
		if(!isalpha(*cp))
			continue;
		p = strchr(lbuf, ':');
		if(!p)
			continue;
		*p = 0;
		args->setValue(lbuf, p + 2);		
	}
	if(isLive())
		exitcode = 0;
}

bool Libexec::isLive(void)
{
	init();

	if(eof)
		return false;

	if(!tsid)
		return false;

	return true;
}

int Libexec::getResult(const char *id, char *buffer, unsigned size)
{
	char *p;
	char lbuf[512];
	char tid[64];
	unsigned len = 0;

	result = -1;

	if(id && buffer)
	{
		len = strlen(id);
		*buffer = 0;
		setString(tid, sizeof(tid), id);
		while(NULL != (p = strchr(tid, '.')))
			*p = '_';
		id = tid;
	}

	while(isLive())
	{
		cin.getline(lbuf, sizeof(lbuf));
		if(cin.eof())
		{
			eof = true;
			break;
		}
		p = strchr(lbuf, '\n');
		if(p)
			*p = 0;
		p = strchr(lbuf, '\r');
		if(p)
			*p = 0;

		if(!lbuf[0])
			break;

		if(atoi(lbuf) >= 900)
		{
			exitcode = atoi(lbuf) - 900;
			eof = true;
			break;
		}

		if(len && !strnicmp(lbuf, id, len))
		{
			if(lbuf[len] == ':')
				setString(buffer, size, lbuf + len + 2);
		}

		if(!strnicmp(lbuf, "RESULT: ", 8))
			result = atoi(lbuf + 8);
		else if(!strnicmp(lbuf, "DIGITS: ", 8))
			setString(digits, sizeof(digits), lbuf + 8);
		else if(!strnicmp(lbuf, "POSITION: ", 10))
			setString(position, sizeof(position), lbuf + 10);  
	}

	if(!isLive() || result == -1)
	{
		eof = true;
		result = -1;
	}
	return result;
}

const char *Libexec::getArg(const char *id)
{
	init();

	if(args)
		return args->getLast(id);
	return NULL;
}

const char *Libexec::getEnv(const char *id)
{
	const char *cp;
	char bid[48];
	char *up;

	init();

	if(!head)
		return NULL;

	cp = head->getLast(id);
	if(cp)
		return cp;

	snprintf(bid, sizeof(bid), "%s", id);
	up = bid;
	while(*up)
	{
		*up = toupper(*up);
		++up;
	}
	return Process::getEnv(bid);
}

const char *Libexec::getDigits(char *buf, unsigned size)
{
	setString(buf, size, digits);
	digits[0] = 0;
	return buf;
}

bool Libexec::clrVar(const char *id)
{
	if(!isLive())
		return false;

	cout << tsid << " clr " << id << endl;
	if(getResult())
		return false;
	return true;
}

bool Libexec::newVar(const char *id, unsigned size)
{
	if(!isLive())
		return false;

	cout << tsid << " new " << id << " " << size << endl;
	if(getResult())
		return false;
	return true;
}

bool Libexec::setResult(const char *value)
{
	if(!isLive())
		return false;

	cout << tsid << " result " << value << endl;
	if(getResult())
		return false;

	return true;
}

bool Libexec::setVar(const char *id, const char *value)
{
	if(!isLive())
		return false;

	cout << tsid << " set " << id << " " << value << endl;
	if(getResult())
		return false;
	return true;
}

bool Libexec::getVar(const char *id, char *save, unsigned size)
{
	if(!isLive())
		return false;

	cout << tsid << " get " << id << endl;
	if(getResult(id, save, size))
		return false;

	return true;
}

bool Libexec::replay(const char *file, const char *offset)
{
	if(!isLive())
		return false;

	if(offset)
		cout << tsid << " replay " << file << " " << offset << endl;
	else
		cout << tsid << " replay " << file << endl;

	switch(getResult())
	{
	case Bayonne::RESULT_SUCCESS:
	case Bayonne::RESULT_PENDING:
	case Bayonne::RESULT_TIMEOUT:
	case Bayonne::RESULT_COMPLETE:
		return true;
	}
	return false;
}

bool Libexec::record(const char *file, timeout_t total, timeout_t silence, const char *offset)
{
	if(!isLive())
		return false;

	if(offset)
		cout << tsid << " record " << file << " " << total << " " << silence << " " << offset << endl;
	else
		cout << tsid << " record " << file << " " << total << " " << silence << endl;	

	switch(getResult())
	{
	case Bayonne::RESULT_SUCCESS:
	case Bayonne::RESULT_PENDING:
	case Bayonne::RESULT_TIMEOUT:
	case Bayonne::RESULT_COMPLETE:
		return true;
	}
	return false;
}

bool Libexec::prompt(const char *text, const char *voice)
{
	if(!voice)
		voice = "prompt";

	if(!isLive())
		return false;

	cout << tsid << " " << voice << " " << text << endl;
	switch(getResult())
	{
	case Bayonne::RESULT_SUCCESS:
	case Bayonne::RESULT_TIMEOUT:
	case Bayonne::RESULT_PENDING:
	case Bayonne::RESULT_COMPLETE:
		return true;
	}
	return false;
}

bool Libexec::clear(void)
{
	digits[0] = 0;
	if(!isLive())
		return false;

	cout << tsid << " flush" << endl;
	if(getResult())
		return false;

	return true;
}	

bool Libexec::input(char *buffer, unsigned size, timeout_t timeout)
{
	bool rtn = true;
	unsigned count = size - 1;

	if(!isLive())
		return false;

	cout << tsid << " read " << timeout << " " << count << endl;
	if(getResult())
		rtn = false;

	getDigits(buffer, size);
	return rtn;
}

bool Libexec::wait(timeout_t timeout)
{
	if(!isLive())
		return false;
	
	cout << tsid << " wait " << timeout << endl;
	if(getResult() != Bayonne::RESULT_PENDING)
		return false;

	return true;
}

char Libexec::inkey(timeout_t timeout)
{
	char buf;

	if(!isLive())
		return 0;

	cout << tsid << " read " << timeout << endl;
	getResult();
	buf = digits[0];
	digits[0] = 0;
	return buf;
}	

void Libexec::resume(unsigned code)
{
	if(!isLive())
		return;

	if(code > 255)
		code = 255;

	cout << tsid << " exit " << code << endl;
	eof = true;
	return;
}

void Libexec::hangup(void)
{
	if(!isLive())
		return;

	cout << tsid << " hangup" << endl;
	eof = true;
	return;
}

#endif

