/*
 * Copyright (C) 1999, 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>

#include "genfun.h"

#include "messages.h"
#include "generators.h"
#include "tags.h"
#include "keys.h"
#include "textgen.h"
#include "decorators.h"
#include "fileutil.h"

// globals
#include "linenumdigit.h"
#include "cmdlineargs.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

extern int java_scanner_lex() ;
extern int cpp_scanner_lex() ;
extern int prolog_scanner_lex() ;
extern int perl_scanner_lex() ;

extern int tabSpaces; // FIXME

static void _generate( const char *s );
static void _generateln( const char *s );

static char *get_input_file_name(char *file_name);

void
processFile( char *source_language,
             char *inputFileName, char *outputFileName, char *docTitle, 
             const char *docHeader, const char *docFooter,
             short entire_doc, char *cssUrl )
{
  FILE *in = 0;
  short deleteOStream = 1 ;

  if ( outputFileName ) {
    sout = new ofstream(outputFileName) ;
    if ( ! (*sout) ) {
      cerr << "Error in creating " << outputFileName << " for output" << endl ;
      exit(1) ;
    }
  }

  if (inputFileName)
    {
      unsigned int lines = get_line_count (inputFileName);

      line_num_digit = 0;
      while (lines)
        {
          ++line_num_digit;
          lines /= 10;
        }
    }

  if (inputFileName)
    {
      char *input_file_name = get_input_file_name (inputFileName);

      in = freopen (input_file_name, "r", stdin);
      if (!in) 
        {
          cerr << "Error in opening " << inputFileName
               << " for input" << endl ;
          exit(1) ;
        }

      delete input_file_name;
    }

  /*
   * Use default values for any options not provided
   */
  if (sout == 0) {
    sout = &cout;
    deleteOStream = 0 ; // we can't delete cout !!!
  }
  if (in == 0) {
    ; /* Well stdin already points to stdin so, .... */
  }
  if (docTitle == 0) {
    docTitle = inputFileName; /* inputFileName may also be 0,
                                 this is OK. */
  }
  
  if ( entire_doc ) {
    print_top( docTitle, cssUrl, docHeader, inputFileName );
  }

  printMessage( "translating source code... ", cerr ) ;

  _generateln( "<pre>" ) ;
  _generateln( "<tt>" ) ;

  // decide which lexer to call
  if (strcmp (source_language, "java") == 0)
    java_scanner_lex () ;
  else if (strcmp (source_language, "cpp") == 0)
    cpp_scanner_lex ();
  else if (strcmp (source_language, "prolog") == 0)
    prolog_scanner_lex ();
  else if (strcmp (source_language, "perl") == 0)
    perl_scanner_lex ();
  else
    {
      cerr << PACKAGE << ": ";
      cerr << "source language " << source_language << " not handled" << endl;
      exit (1);
    }
  // FIXME: a more object-oriented approach?

  _generateln( "</tt>" ) ;
  _generateln( "</pre>" ) ;

  printMessage( "done !", cerr ) ;
  
  if ( entire_doc )
    print_bottom( docFooter ) ;

  if ( deleteOStream )
    delete sout ;
}

char *
get_input_file_name(char *file_name)
{
  unsigned int length = strlen (CHROOT_INPUT_DIR);

  if (length != 0)
    {
      char *new_name = new char [strlen (file_name) + length + 1];
      strcpy (new_name, strdup (CHROOT_INPUT_DIR));
      return strcat (new_name, file_name);
    }
  else
    return strdup (file_name);
}

void
print_cgi_header()
{
  printf( "Content-type: text/html\n" ) ;
  printf( "\n" ) ;
}

void
print_top( char *docTitle, char *cssUrl,
           const char *docHeader, char *inputFileName)
{
  if( cssUrl == 0 ) {
    _generateln( "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\">" ) ;
  }
  else {
    _generateln( "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\"");
    _generateln( "    \"http://www.w3.org/TR/REC-html40/strict.dtd\">");
  }
  _generateln( "<html>" ) ;
  _generateln( "<head>" ) ;
  _generateln( "<meta http-equiv=\"Content-Type\"" ) ;
  _generateln( "content=\"text/html; charset=iso-8859-1\">" ) ;
  _generate( "<meta name=\"GENERATOR\" content=\"GNU " ) ;
  _generate (PACKAGE);
  _generate (" ");
  _generate (VERSION);
  _generate( "\nby Lorenzo Bettini, bettini@gnu.org" ) ;
  _generate( "\nhttp://w3.newnet.it/bettini" ) ;
  _generate( "\nhttp://www.gnu.org/software/src-highlite" ) ;
  _generateln( "\">" ) ;
  _generate( "<title>" ) ;
  _generate( ( docTitle ? docTitle : 
              ( inputFileName ? inputFileName : "source file" ) ) ) ;
  _generateln( "</title>" ) ;
  if( cssUrl != 0 ) {
    _generate( "<link rel=\"stylesheet\" href=\"" );
    _generate( cssUrl );
    _generateln( "\" type=\"text/css\">");
  }
  _generateln( "</head>" ) ;
  if( cssUrl == 0 && docHeader == 0) {
    _generate ("<body bgcolor=\"#FFFFFF\" text=\"#000000\" link=\"#0000EE\" ");
    _generateln ( "vlink=\"#551A8B\" alink=\"#FF0000\">" );
  }
  else {
    _generateln( "<body>" ) ;
  }
  if (docHeader)
    _generateln (docHeader) ;
}

void print_bottom( const char *docFooter) {
  if ( docFooter ) _generateln( docFooter ) ;
  _generateln( "</body>" ) ;
  _generateln( "</html>" ) ;
}

// for generating header and footer
void
_generate( const char *s )
{
  *sout << s ;
}

// for generating header and footer
void
_generateln( const char *s )
{
  *sout << s << endl;
}

void
generate( const char *s )
{
  GlobalGenerator->generate(s) ;
}

void
generate( const char *s, int start, int end )
{
  GlobalGenerator->generate(s, start, end) ;
}

void
generateln( const char *s )
{
  GlobalGenerator->generateln(s) ;
}

void
generate_preproc(const char *s)
{
  GlobalGenerator->generate_preproc (s);
}

void
generate_normal(const char *s)
{
  NormalGenerator->generateEntire (s);
}

void 
generateNewLine() 
{
  generateln( "" ) ;
}

void 
generateTab() 
{
  // if --line-number option is given we have to translate tabs
  // otherwise it results non-aligned, because of inserted chars for
  // line numbers

  if ( tabSpaces )
    {
      for ( int i = 0 ; i < tabSpaces ; ++i )
        generate( SPACE_CHAR ) ;
    }
  else
    {
      if (args_info.line_number_given)
        {
          for ( unsigned int i = 0 ; i < 8 ; ++i )
            generate( SPACE_CHAR ) ;
        }
      else
        generate( "\t" ) ;
    }
}

void startComment( const char *s )
{
  CommentGenerator->beginText(s) ;
}

void endComment( const char *s )
{
  CommentGenerator->endText(s) ;
}

void generateComment( const char *s ) {
  CommentGenerator->generateEntire(s) ;
}

void startString( const char *s )
{
  StringGenerator->beginText(s) ;
}

void endString( const char *s )
{
  StringGenerator->endText(s) ;
}

void generateString( const char *s ) {
  StringGenerator->generateEntire(s) ;
}

void generateKeyWord( const char *s ) {
  KeywordGenerator->generateEntire(s) ;
}

void generateBaseType( const char *s ) {
  TypeGenerator->generateEntire(s) ;
}

void
generatePreProc( const char *s )
{
  PreProcGenerator->generateEntire(s) ;
}

void generateNumber( const char *s ) {
  NumberGenerator->generateEntire(s) ;
}

void
generateSymbol( const char *s )
{
  SymbolGenerator->generateEntire(s);
}

void
generateFunction( const char *s )
{
  FunctionGenerator->generateEntire(s) ;
}

void
generateCBracket( const char *s )
{
  CBracketGenerator->generateEntire(s) ;
}

void startTAG( const char *tag, const char *attr, const char *val ) {
  (*sout) << "<" << tag ;
  if ( attr && val )
    (*sout) << " " << attr << "=" << val ;
  (*sout) << ">" ;
}

void endTAG( const char *tag ) {
  (*sout) << "</" << tag << ">" ;
}

void startColor( const char *color ) {
  startTAG( FONT_TAG, COLOR_TAG, color ) ;
}

void endColor() {
  endTAG( FONT_TAG ) ;
}
