/******************************** -*- C -*- ****************************
 *
 *	OOP printing and debugging module
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"
#include "snprintfv/mem.h"


/* Show information about the contents of the object OBJ. */
static void display_object (mst_Object obj);

/* Show information about the contents of the indirect object pointer OOP. */
static void display_oop (OOP oop);

/* Print a String OOP to a snprintfv stream, STREAM.  */
static void print_string_to_stream (STREAM *stream,
				    OOP string);

/* Print an Association OOP's key to a snprintfv stream, STREAM.  */
static void print_association_key_to_stream (STREAM *stream,
					     OOP associationOOP);

/* Print a Class OOP's name to a snprintfv stream, STREAM.  */
static void print_class_name_to_stream (STREAM *stream,
					OOP class_oop);

/* Print a brief description of an OOP to a snprintfv stream, STREAM.  */
static void print_oop_constructor_to_stream (STREAM *stream,
					     OOP oop);

/* The main routine to handle the %O modifier to printf.  %#O prints
   Strings and Symbols without the leading # or the enclosing single
   quotes, while %+O expects that an Association is passed and prints
   its key.  */
static void printf_oop (STREAM *stream,
                        struct printf_info *info,
                        const void *const *args);

static int printf_oop_arginfo (struct printf_info *info,
                               size_t n,
                               int *argtypes);


void
_gst_print_object (OOP oop)
{
  printf ("%O", oop);
  fflush (stdout);
}

void
print_string_to_stream (STREAM *stream, OOP string)
{
  int len;

  len = _gst_string_oop_len (string);
  if (!len)
    return;

  stream_printf (stream, "%.*s", len, (char *) (OOP_TO_OBJ (string)->data));
}

void
print_association_key_to_stream (STREAM *stream, OOP associationOOP)
{
  gst_association association;

  if (!IS_OOP (associationOOP)
      || !is_a_kind_of (OOP_CLASS(associationOOP), _gst_association_class))
    {
      stream_printf (stream, "<non-association %O in association context>",
                     associationOOP);
      return;
    }

  association = (gst_association) OOP_TO_OBJ (associationOOP);
  if (OOP_CLASS (association->key) != _gst_symbol_class)
    stream_printf (stream, "<unprintable key type>");
  else
    stream_printf (stream, "%#O", association->key);
}

void
print_class_name_to_stream (STREAM *stream, OOP class_oop)
{
  gst_class class;
  class = (gst_class) OOP_TO_OBJ (class_oop);
  if (IS_A_CLASS (class_oop) && !IS_NIL (class->name))
    print_string_to_stream (stream, class->name);
  else if (IS_A_CLASS (OOP_CLASS (class_oop)))
    {
      stream_printf (stream, "<unnamed ");
      print_class_name_to_stream (stream, OOP_CLASS (class_oop));
      stream_printf (stream, ">");
    }
  else
    stream_printf (stream, "<unnamed class>");
}

void
print_oop_constructor_to_stream (STREAM *stream, OOP oop)
{
  long instanceSpec;
  OOP class_oop;

  class_oop = OOP_CLASS (oop);
  print_class_name_to_stream (stream, class_oop);

  instanceSpec = CLASS_INSTANCE_SPEC (class_oop);
  if (instanceSpec & ISP_ISINDEXABLE)
    stream_printf (stream, " new: %ld ", NUM_INDEXABLE_FIELDS (oop));

  else
    stream_printf (stream, " new ");

  if (_gst_regression_testing)
    stream_printf (stream, "\"<0>\"");
  else
    stream_printf (stream, "\"<%#lx>\"", (unsigned long) oop);
}

void
printf_oop (STREAM *stream,
            struct printf_info *info,
            const void *const *args)
{
  OOP oop = (OOP) (args[0]);

  if (info->showsign)
    {
      print_association_key_to_stream (stream, oop);
      return;
    }

  if (IS_INT (oop))
    stream_printf (stream, "%ld", TO_INT (oop));

  else if (IS_NIL (oop))
    stream_printf (stream, "nil");

  else if (oop == _gst_true_oop)
    stream_printf (stream, "true");

  else if (oop == _gst_false_oop)
    stream_printf (stream, "false");

  else if (OOP_CLASS (oop) == _gst_char_class)
    stream_printf (stream, "$%c", CHAR_OOP_VALUE (oop));

  else if (OOP_CLASS (oop) == _gst_floatd_class)
    {
      double f = FLOATD_OOP_VALUE (oop);
      stream_printf (stream, "%#.6g", f);
    }

  else if (OOP_CLASS (oop) == _gst_floate_class)
    {
      double f = FLOATE_OOP_VALUE (oop);
      stream_printf (stream, "%#.6g", f);
    }

  else if (OOP_CLASS (oop) == _gst_floatq_class)
    {
      long double f = FLOATQ_OOP_VALUE (oop);
      stream_printf (stream, "%#.6Lg", f);
    }

  else if (OOP_CLASS (oop) == _gst_symbol_class)
    {
      if (!info->alt)
        stream_printf (stream, "#");
      print_string_to_stream (stream, oop);
    }

  else if (OOP_CLASS (oop) == _gst_string_class)
    {
      /* ### have to quote embedded quote chars */
      if (!info->alt)
        stream_printf (stream, "'");
      print_string_to_stream (stream, oop);
      if (!info->alt)
        stream_printf (stream, "'");
    }

  else if (IS_A_METACLASS (oop))
    {
      OOP class_oop = _gst_find_an_instance (oop);
      if (IS_NIL (class_oop))
        print_oop_constructor_to_stream (stream, oop);
      else
        {
          print_class_name_to_stream (stream, class_oop);
          stream_printf (stream, " class");
        }
    }

  else if (IS_A_CLASS (oop))
    print_class_name_to_stream (stream, oop);

  else
    print_oop_constructor_to_stream (stream, oop);

  fflush (stdout);
}

int
printf_oop_arginfo (struct printf_info *info,
                    size_t n,
                    int *argtypes)
{
  /* We always take exactly one argument and this is a pointer to the
     structure.. */
  if (n > 0)
    argtypes[0] = PA_POINTER;
  return 1;
}



void
_gst_classify_addr (void *addr)
{
  if (IS_OOP_ADDR (addr))
    display_oop (addr);

  else if (IS_OBJ_ADDR (addr))
    display_object (addr);

  else if (IS_INT (addr))
    printf ("Smalltalk SmallInteger %ld\n", TO_INT (addr));

  else
    printf ("Address %p is not a Smalltalk entity\n", addr);

  fflush (stdout);
}

void
display_oop (OOP oop)
{
  mst_Boolean isBuiltin;

  if (!IS_OOP_ADDR (oop))
    {
      printf ("Parameter %p does not appear to be an OOP!\n", oop);
      return;
    }

  isBuiltin = (oop < _gst_oop_table);

  if (!isBuiltin)
    printf ("OOP %p [%ld]\n", oop,
	    (unsigned long) (oop - _gst_oop_table));

  if (oop->flags & F_FREE)
    printf ("Free ");

  if (oop->flags & F_REACHABLE)
    printf ("Reachable ");

  printf ("   Empty bytes = %ld\n", (oop->flags & EMPTY_BYTES));
  if (!(oop->flags & F_FREE))
    _gst_print_object (oop);

  printf ("\n");
}


void
display_object (mst_Object obj)
{
  if (!IS_OBJ_ADDR (obj))
    {
      printf ("Parameter %p does not appear to be an object!\n", obj);
      return;
    }

  printf ("Object at %p, ", obj);
  printf ("Size %ld\n", NUM_OOPS (obj));
  printf ("class %O", obj->objClass);
  printf ("\n");
}



void _gst_init_snprintfv ()
{
  spec_entry *spec;

  snv_malloc = xmalloc;
  snv_realloc = xrealloc;
  snv_free = xfree;
  spec = register_printf_function ('O', printf_generic,
                                   printf_oop_arginfo);

  spec->user = printf_oop;
}

