/* engine.c: -*- C -*-  Mini-server runs as CGI. */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Sat Jul 20 17:48:27 1996.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#if defined (sgi)
#include <bstring.h>
#endif
#include <errno.h>
#include <sys/types.h>
#if defined (Solaris)
#  include <ucbinclude/sys/fcntl.h>
#  include <ucbinclude/sys/file.h>
#else
#  include <sys/file.h>
#  if defined (sgi)
#    include <fcntl.h>
#  endif
#endif /* !Solaris */
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <bprintf/bprintf.h>
#include <xmalloc/xmalloc.h>
#include "pages.h"
#include "symbols.h"
#include "parser.h"
#include "session.h"
#include "streamfuncs.h"
#include "http.h"
#include "globals.h"
#include "logging.h"

/* The name of this program, as taken from argv[0]. */
static char *rawprogname = (char *)NULL;

/* The last componenent of rawprogname. */
static char *progname = (char *)NULL;

/* The port number, as taken from SERVER_PORT. */
static int http_port = 80;

/* The host name as taken from SERVER_HOST. */
static char *http_host = (char *)NULL;

/* The full pathname to the configuration file. */
static char *engine_config_path = (char *)NULL;

/* Non-zero means this engine is named "nph-" something. */
static int nph = 0;

/* Forward declarations. */
static void parse_program_args (int argc, char *argv[]);
static void initialize_engine (void);
static void fatal (char *format, ...);

#if defined (CHECK_TIMESTAMP)
static void check_timestamp (char *filename);
#endif

static void process_request (void);
static void find_config_file (void);

extern MIME_HEADER **mime_headers_from_string (char *string, int *last_char);

int
main (int argc, char *argv[])
{
  char *temp;

  rawprogname = strdup (argv[0]);
  temp = strrchr (rawprogname, '/');
  if (temp != (char *)NULL)
    progname = strdup (temp + 1);
  else
    progname = strdup (rawprogname);

  nph = (strncasecmp (progname, "nph-", 4) == 0);
  parse_program_args (argc, argv);

#if defined (CHECK_TIMESTAMP)
  check_timestamp (argv[0]);
#endif
  initialize_engine ();
  process_request ();
  exit (0);
}

#if defined (CHECK_TIMESTAMP)
#include "../libtimestamp/timestamp.h"

static void
check_timestamp (char *filename)
{
  if (check_stamp_expired (filename))
    {
      fprintf (stderr, "Sorry, but your free trial has ended.\n");
      fprintf (stderr, "You may obtain another copy of the Engine from\n");
      fprintf (stderr, "\n\thttp://www.metahtml.com\n\n");
      fprintf (stderr, "We hope that you have enjoyed the use of our\n");
      fprintf (stderr, "engine, and will continue to watch our site for\n");
      fprintf (stderr, "new additions to our product suite.\n");
      exit (2);
    }
}
#endif /* CHECK_TIMESTAMP */

/* How to initialize the Engine. */
static void
initialize_engine (void)
{
  char *temp;
  PAGE *page;
  char *include_prefix = (char *)NULL;

  pagefunc_set_variable ("mhttpd::copyright-string",metahtml_copyright_string);

  /* Set up the input and output file descriptors. */
  mhtml_stdout_fileno = fileno (stdout);
  mhtml_stdin_fileno = fileno (stdin);
  mhtml_stderr_fileno = fileno (stderr);

  /* Quickly make a package containing the minimum mime-types. */
  pagefunc_set_variable ("mime-type::.mhtml", "metahtml/interpreted");
  pagefunc_set_variable ("mime-type::.jpeg", "image/jpeg");
  pagefunc_set_variable ("mime-type::.jpg", "image/jpeg");
  pagefunc_set_variable ("mime-type::.gif", "image/gif");
  pagefunc_set_variable ("mime-type::.html", "text/html");
  pagefunc_set_variable ("mime-type::.txt", "text/plain");
  pagefunc_set_variable ("mime-type::.mov", "video/quicktime");
  pagefunc_set_variable ("mime-type::.default", "text/plain");

  /* The minimum startup documents. */
  pagefunc_set_variable ("mhtml::default-filenames[]",
"Welcome.mhtml\nwelcome.mhtml\n\
Index.mhtml\nindex.mhtml\n\
Home.mhtml\nhome.mhtml\n\
Directory.mhtml\ndirectory.mhtml\n\
index.html\nhome.html");

  /* The default extensions for running files through the engine. */
  pagefunc_set_variable ("mhtml::metahtml-extensions[]", ".mhtml");

  /* Default the value of mhtml::include-prefix to the directory above
     the location of this program.  This empirically seems to be the
     right thing -- if the program resides in /www/foo/docs/cgi-bin/nph-engine,
     then the right include-prefix is /www/foo/docs.  So we try. */
  include_prefix = strdup (rawprogname);
  temp = strrchr (include_prefix, '/');
  if (temp != (char *)NULL)
    {
      *temp = '\0';
      temp = strrchr (include_prefix, '/');

      if (temp != (char *)NULL)
	*temp = '\0';
    }

  pagefunc_set_variable ("mhtml::include-prefix", include_prefix);
  free (include_prefix);

  /* Try hard to find a configuration file. */
  find_config_file ();

  if (engine_config_path == (char *)NULL)
    {
      engine_config_path = (char *)getenv ("WEBBASEDIR");
      if (engine_config_path == (char *)NULL)
	engine_config_path = "/www/bin/engine.conf";
    }

  page = page_read_template (engine_config_path);

#if defined (FORCE_ENGINE_CONFIG)
  if (!page)
    fatal ("Cannot read configuration file `%s'!", engine_config_path);
#endif

  if (page && page->buffer)
    page_process_page (page);

  if (page)
    page_free_page (page);

  /* If the user didn't set mhtml::require-directories[], give a reasonable
     value here. */
  temp = pagefunc_get_variable ("mhtml::require-directories");
  if (temp == (char *)NULL)
    pagefunc_set_variable ("mhtml::require-directories[]",
".\ntagsets\nmacros\nincludes\n\
..\n../tagsets\n../macros\n../includes\n\
../..\n../../tagsets\n../../macros\n../../includes");

  /* Now set our local variables. */
  temp = (char *)getenv ("SERVER_NAME");

  if (temp != (char *)NULL)
    {
      http_host = strdup (temp);
    }
  else
    {
      char buffer[100];

      strcpy (buffer, "www.nohost.net");
      gethostname (buffer, 100);
      http_host = strdup (buffer);
    }
  pagefunc_set_variable ("mhtml::server-name", http_host);

  temp = (char *)getenv ("SERVER_PORT");
  if (temp == (char *)NULL)
    temp = "80";

  http_port = atoi (temp);
  pagefunc_set_variable ("mhtml::server-port", temp);

  temp = pagefunc_get_variable ("mhtml::document-root");
  if (temp == (char *)NULL)
    temp = pagefunc_get_variable ("mhtml::include-prefix");

  if (temp == (char *)NULL)
    temp = (char *)getenv ("WEBBASEDIR");

  if (temp != (char *)NULL)
    {
      sv_DocumentRoot = strdup (temp);
      pagefunc_set_variable ("mhtml::include-prefix", temp);

      chdir (sv_DocumentRoot);
    }

  mhttpd_per_request_function =
    pagefunc_get_variable ("mhttpd::per-request-function");

  set_session_database_location
    (pagefunc_get_variable ("mhttpd::session-database-file"));

  /* Create a reasonable default PATH variable if the user didn't do so. */
  if (pagefunc_get_variable ("mhtml::exec-path") == (char *)NULL)
    {
      page = page_create_page ();
      bprintf (page, "<set-var mhtml::exec-path =");
      bprintf (page, "<get-var mhtml::server-root>/bin:/bin:");
      bprintf (page, "<get-var mhtml::document-root>/cgi-bin:/usr/bin:");
      bprintf (page, "/usr/local/bin:/usr/ucb");
      bprintf (page, ">");
      page_process_page (page);

      if (page)
	page_free_page (page);
    }

  /* Finally, the logfiles. */
  mhttpd_set_logfile (log_ACCESS,  "");
  mhttpd_set_logfile (log_ERROR,   "");
  mhttpd_set_logfile (log_DEBUG,   "");
  mhttpd_set_logfile (log_REFERER, "");
  mhttpd_set_logfile (log_AGENT, "");

  mhttpd_debugging = 0;
  temp = pagefunc_get_variable ("mhtml::debug-log");
  if (temp)
    {
      mhttpd_set_logfile (log_DEBUG, temp);
      mhttpd_debugging = 1;
    }
}

int
mhttpd_check_access (HTTP_RESULT *result)
{
  return (1);
}

static int
file_exists_p (char *path)
{
  struct stat finfo;

  return (stat (path, &finfo) != -1);
}

static char *
find_config_locally (char *dir)
{
  char *result = (char *)NULL;
  char path[1024 + 256];
  char *temp;

  /* Try ./engine.conf */
  sprintf (path, "%s/engine.conf", dir);

  if (file_exists_p (path))
    result = strdup (path);
  else
    {
      /* Try ./conf/engine.conf */
      sprintf (path, "%s/conf/engine.conf", dir);
      if (file_exists_p (path))
	result = strdup (path);
      else
	{
	  /* Try ../engine.conf */
	  strcpy (path, dir);
	  temp = strrchr (path, '/');
	  if (temp)
	    {
	      *temp = '\0';
	      strcat (path, "/engine.conf");
	      if (file_exists_p (path))
		result = strdup (path);
	      else
		{
		  /* Try ../conf/engine.conf */
		  *temp = '\0';
		  strcat (path, "/conf/engine.conf");
		  if (file_exists_p (path))
		    result = strdup (path);
		}
	    }
	}
    }

  return (result);
}

static void
find_config_file (void)
{
  char dir[1024];
  char *temp;

  if (engine_config_path)
    return;

  /* Try starting with the directory that this program is in. */
  strcpy (dir, rawprogname);
  temp = strrchr (dir, '/');
  if (temp != (char *)NULL)
    {
      *temp = 0;
      engine_config_path = find_config_locally (dir);
    }

  if (engine_config_path == (char *)NULL)
    {
      /* Okay, search in the CWD. */
      temp = getcwd (dir, sizeof (dir));

      if (!temp)
	fatal ("Can't get working directory!");

      engine_config_path = find_config_locally (dir);
    }
}

static void
usage (void)
{
  fprintf (stderr, "Usage: %s --config config-path\n", progname);
  exit (1);
}

static void
parse_program_args (int argc, char *argv[])
{
  int arg_index = 1;

  while (arg_index < argc)
    {
      char *arg = argv[arg_index++];

      if ((strcasecmp (arg, "--config") == 0) ||
	  (strcasecmp (arg, "-config") == 0) ||
	  (strcasecmp (arg, "-f") == 0))
	{
	  engine_config_path = strdup (argv[arg_index++]);
	}
      else
	usage ();
    }
}

static void
fatal (char *format, ...)
{
  int actual_error = errno;

  va_list args;
  va_start (args, format);

  fprintf (stderr, "%s: ", rawprogname);
  vfprintf (stderr, format, args);
  fprintf (stderr, "\n");

  if (actual_error)
    fprintf (stderr, "Error returned in errno: %d: %s\n", actual_error,
	     strerror (actual_error));
  exit (2);
}

static HTTP_REQUEST *
engine_make_request (int fd)
{
  HTTP_REQUEST *req = (HTTP_REQUEST *)xmalloc (sizeof (HTTP_REQUEST));
  char *temp;

  memset (req, 0, sizeof (HTTP_REQUEST));

  /* Try to get the client's request method. */
  temp = (char *)getenv ("HTTP_REQUEST_METHOD");
  if (temp == (char *)NULL)
    {
      if (((char *)getenv ("CONTENT_LENGTH")) != (char *)NULL)
	temp = "POST";
      else
	temp = "GET";
    }
  req->method = strdup (temp);

  /* Get the relative URL. */
  {
    char *query_string, *path_info, *url, *prefix;

    path_info = (char *)getenv ("PATH_INFO");
    query_string = (char *)getenv ("QUERY_STRING");

    if (path_info == (char *)NULL)
      path_info = "/";

    prefix = pagefunc_get_variable ("mhtml::include-prefix");

    if ((prefix != (char *)NULL) &&
	(strncmp (path_info, prefix, strlen (prefix)) == 0))
      path_info += strlen (prefix);

    if ((query_string != (char *)NULL) && (*query_string))
      {
	url = (char *)xmalloc (3 + strlen (path_info) + strlen (query_string));
	sprintf (url, "%s?%s", path_info, query_string);
      }
    else
      url = strdup (path_info);

    req->location = url;
  }

  /* Get the protocol and version used by this client. */
  req->protocol = strdup ("HTTP");
  req->protocol_version = "1.0";

  /* Get the client's address. */
  temp = (char *)getenv ("REMOTE_ADDR");
  if (temp != (char *)NULL)
    req->requester_addr = strdup (temp);
  else
    req->requester_addr = strdup ("127.0.0.1");

  /* Get the client's hostname. */
  temp = (char *)getenv ("REMOTE_HOST");
  if (temp != (char *)NULL)
    req->requester = strdup (temp);
  else
    req->requester = strdup (req->requester_addr);

  /* Build a reasonable copy of the Mime headers that we might expect to be
     present. */
  {
    BPRINTF_BUFFER *headers = bprintf_create_buffer ();
    int ignore = 0;

    /* User-Agent */
    temp = (char *)getenv ("HTTP_USER_AGENT");
    if (temp == (char *)NULL) temp = "Unknown";
    bprintf (headers, "User-Agent: %s\n", temp);

    /* Referer */
    temp = (char *)getenv ("HTTP_REFERER");
    if (temp != (char *)NULL)
      bprintf (headers, "Referer: %s\n", temp);

    /* Cookie! */
    temp = (char *)getenv ("HTTP_COOKIE");
    if (temp != (char *)NULL)
      bprintf (headers, "Cookie: %s\n", temp);

    /* Content-Length */
    temp = (char *)getenv ("CONTENT_LENGTH");
    if (temp != (char *)NULL)
      bprintf (headers, "Content-length: %s\n", temp);

    bprintf (headers, "\n");

    req->headers = mime_headers_from_string (headers->buffer, &ignore);
    bprintf_free_buffer (headers);
  }

  return (req);
}

static void
process_request (void)
{
  int connection = fileno (stdin);

  /* In the child, handle a single request. */
  HTTP_REQUEST *req = (HTTP_REQUEST *)NULL;
  HTTP_RESULT *result = (HTTP_RESULT *)NULL;

  /* Okay, read the HTTP request and handle it. */
  req = engine_make_request (connection);

  if (req && mhttpd_per_request_function)
    {
      PAGE *perfun = page_create_page ();
      mhttpd_mime_headers_to_package
	(req->headers, "mhttpd-received-headers");
      pagefunc_set_variable ("mhttpd::method", req->method);
      pagefunc_set_variable ("mhttpd::protocol", req->protocol);
      pagefunc_set_variable ("mhttpd::protocol-version",
			     req->protocol_version);
      pagefunc_set_variable ("mhttpd::location", req->location);
      pagefunc_set_variable ("mhttpd::requester", req->requester);
      pagefunc_set_variable ("mhttpd::requester-addr",
			     req->requester_addr);
      bprintf (perfun, "<%s>\n", mhttpd_per_request_function);
      page_process_page (perfun);

      if (mhttpd_page_redirect_p (perfun))
	{
	  if (!strncasecmp (perfun->buffer, "HTTP/", 5) == 0)
	    bprintf_insert (perfun, 0, "HTTP/1.0 302 Found\n");

	  mhttpd_write (mhtml_stdout_fileno, perfun->buffer, perfun->bindex);
	  return;
	}

      if (perfun)
	page_free_page (perfun);
    }

  if (req != (HTTP_REQUEST *)NULL)
    result = http_handle_request (req, connection);

  if (result && result->page && result->page->buffer)
    {
      register int i = 0;

      /* If we are running as parsed header CGI, then remove the
	 HTTP result code line. */
      if (!nph)
	{
	  for (i = 0; i < result->page->bindex; i++)
	    if (result->page->buffer[i] == '\n')
	      {
		i++;
		break;
	      }
	}

      mhttpd_write (mhtml_stdout_fileno, result->page->buffer + i,
		    result->page->bindex - i);
    }
}
