/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * $Id: CharStream.java,v 1.1 2001/08/19 16:44:54 metlov Exp $
 *
 * This file is part of the Java Expressions Library (JEL).
 *   For more information about JEL visit :
 *    http://galaxy.fzu.cz/JEL/
 *
 * (c) 1998 -- 2001 by Konstantin Metlov(metlov@fzu.cz);
 *
 * JEL is Distributed under the terms of GNU General Public License.
 *    This code comes with ABSOLUTELY NO WARRANTY.
 *  For license details see COPYING file in this directory.
 */

package gnu.jel.generated;

import gnu.jel.debug.Debug;
import java.io.EOFException;

/**
 * This interface describes a character stream that maintains line and
 * column number positions of the characters.  It also has the capability
 * to backup the stream to some extent.  An implementation of this
 * interface is used in the TokenManager implementation generated by
 * JavaCCParser.
 *
 * All the methods except backup can be implemented in any fashion. backup
 * needs to be implemented correctly for the correct operation of the lexer.
 * Rest of the methods are all used to get information like line number,
 * column number and the String that constitutes a token and are not used
 * by the lexer. Hence their implementation won't affect the generated lexer's
 * operation.
 */

public class CharStream {

  private String expr;     // string to read
  private int bufline[];   // line numbers for each character in expr
  private int bufcolumn[]; // column numbers for each character in expr
  private int pos=-1;      // current position
  private int tokenpos=-1; // starting position of the current token

  public CharStream(String expr) {
    this.expr=expr;
    bufline=new int[expr.length()+1];
    bufcolumn=new int[expr.length()+1];

    // pre-calculate line and column numbers at each position
    {
      int column = 0;
      int line = 1;

      boolean prevCharIsCR = false;
      boolean prevCharIsLF = false;

      for (int i=0;i<expr.length();i++) {
        char c=expr.charAt(i);
        
        column++;
        
        if (prevCharIsLF)
          {
            prevCharIsLF = false;
            line += (column = 1);
          }
        else if (prevCharIsCR)
          {
            prevCharIsCR = false;
            if (c == '\n')
              {
                prevCharIsLF = true;
              }
            else
              line += (column = 1);
          }
        
        switch (c)
          {
          case '\r' :
            prevCharIsCR = true;
            break;
          case '\n' :
            prevCharIsLF = true;
            break;
          case '\t' :
            column--;
            column += (8 - (column & 07));
            break;
          default :
            break;
          }
        
        bufline[i] = line;
        bufcolumn[i] = column;
      };
    }

  };
    

  /**
   * Returns the next character from the selected input.  The method
   * of selecting the input is the responsibility of the class
   * implementing this interface.  Can throw any java.io.IOException.
   */
  public char readChar() throws java.io.IOException {
    try {
      return expr.charAt(++pos);
    } catch (IndexOutOfBoundsException e) {
      if (pos>0) --pos;
      else tokenpos=pos; // fix for an empty stream
      throw new EOFException();
    };
  };

  /**
   * Returns the column number of the last character for current token (being
   * matched after the last call to BeginTOken).
   */
  public int getEndColumn() {
    return bufcolumn[pos];
  };

  /**
   * Returns the line number of the last character for current token (being
   * matched after the last call to BeginTOken).
   */
  public int getEndLine() {
    return bufline[pos];
  };

  /**
   * Returns the column number of the first character for current token (being
   * matched after the last call to BeginTOken).
   */
  public int getBeginColumn() {
    return bufcolumn[tokenpos];
  };

  /**
   * Returns the line number of the first character for current token (being
   * matched after the last call to BeginTOken).
   */
  public int getBeginLine() {
    return bufline[tokenpos];
  };

  /**
   * Backs up the input stream by amount steps. Lexer calls this method if it
   * had already read some characters, but could not use them to match a
   * (longer) token. So, they will be used again as the prefix of the next
   * token and it is the implemetation's responsibility to do this right.
   */
  public void backup(int amount) {
    if (Debug.enabled)
        Debug.assert((pos-tokenpos)+1>=amount);
    pos-=amount;
  };

  /**
   * Returns the next character that marks the beginning of the next token.
   * All characters must remain in the buffer between two successive calls
   * to this method to implement backup correctly.
   */
  public char BeginToken() throws java.io.IOException {
    char c=readChar();
    tokenpos=pos;
    return c;
  };

  /**
   * Returns a string made up of characters from the marked token beginning 
   * to the current buffer position. Implementations have the choice of returning
   * anything that they want to. For example, for efficiency, one might decide
   * to just return null, which is a valid implementation.
   */
  public String GetImage() {
    return expr.substring(tokenpos,pos+1);
  };

  /**
   * Returns an array of characters that make up the suffix of length 'len' for
   * the currently matched token. This is used to build up the matched string
   * for use in actions in the case of MORE. A simple and inefficient
   * implementation of this is as follows :
   *
   *   {
   *      String t = GetImage();
   *      return t.substring(t.length() - len, t.length()).toCharArray();
   *   }
   */
  public char[] GetSuffix(int len) {
    return expr.substring(tokenpos,tokenpos+len).toCharArray();
  };

  /**
   * The lexer calls this function to indicate that it is done with the stream
   * and hence implementations can free any resources held by this class.
   * Again, the body of this function can be just empty and it will not
   * affect the lexer's operation.
   */
  public void Done() {
  };

}
