/* Entry for the GNATS system.
   Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.
   Originally written by Tim Wicinski (wicinski@barn.com).
   Further work by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "db_conf.h"
#include "query.h"

static void version		PARAMS((void));
static void usage		PARAMS((void));

/* Name of this program.  */
char *program_name;

/* Whether we should ignore the index and drop the PR on stdin into the
   database as a new one.  */
int force = 0;

/* If 1, we're running the daemon.  */
int is_daemon = 0;

/* If 1, we're running a network client.  */
int is_client = 0;

enum {
  NONE,
  LOCK,
  UNLOCK,
  LOCKDB,
  UNLOCKDB,
  CHECK,
  MODIFY
} edit_options;

struct option long_options[] =
{
  {"lock", 1, NULL, 'l'},
  {"unlock", 0, NULL, 'u'},
  {"lockdb", 0, NULL, 'L'},
  {"unlockdb", 0, NULL, 'U'},
  {"check", 0, NULL, 'c'},
  {"process", 1, NULL, 'p'},
  {"directory", 1, NULL, 'd'},
  {"filename", 1, NULL, 'f'},
  {"version", 0, NULL, 'V'},
  {"help", 0, NULL, 'h'},
  {NULL, 0, NULL, 0}
};

int
main (argc, argv)
     int argc;
     char **argv;
{

  FILE *fp = stdin;
  int option;
  char *fname = (char *)NULL, *path;
  char *username = (char *)NULL, *processid = (char *)NULL;
  int result = 0;

  outfile = stdout;
  
  program_name = basename (argv[0]);
  edit_options = MODIFY;

  while ((option = getopt_long (argc, argv, "l:uLUhcp:d:f:VF",
				long_options, (int *)0)) != EOF)
    {
      switch (option)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'f':
	  fp = fopen (optarg, "r");
	  /* If they gave a bogus argument, then exit right away; we don't
	     want to send mail to gnats-admin every time someone tries to
	     edit a bogus PR number.  */
	  if (fp == (FILE *)NULL)
            {
              fprintf (stderr, "%s: Bad PR file %s.\n", program_name, optarg);
              exit (2);
            }
	  break;

	case 'F':
	  force = 1;
	  break;

	case 'p':
	  processid = optarg;
	  break;

	case 'l': 
	  edit_options = LOCK;
	  username = optarg;
	  break;

	case 'u':
	  edit_options = UNLOCK;
	  break;

	case 'L':
	  edit_options = LOCKDB;
	  break;

	case 'U':
	  edit_options = UNLOCKDB;
	  break;

	case 'c':
	  edit_options = CHECK;
	  break; 

	case 'V':
	  version ();
	  exit (0);
	  break;
	  
	case 'h':
	  usage ();
	  exit (0);
	  break;
	  
	default:
	  usage ();
	  exit (3);
	  break;
	}
    }

  /* if there is another arg, then we take it to be a file name. */
  if (optind < argc)
    fname = argv[optind];
 
  if ((edit_options == LOCK) && (! fname || ! username))
    {
      fprintf (stderr, "%s: Must lock with PR number, username, and process locking it.\n",
               program_name);
      exit (2);
    }
     
  if ((edit_options == UNLOCK) && (! fname))
    {
      fprintf (stderr, "%s: Must unlock with PR number.\n",
               program_name);
      exit (2);
    }
        
  init_gnats (program_name);
  
  umask (022);
  
  block_signals ();

  if (edit_options == LOCKDB || edit_options == MODIFY)
    lock_gnats ();
  
  switch (edit_options) {
    case LOCK:
      path   = lookup_pr_path (fname);
      result = lock_pr (path, username, processid);
      xfree (path);
      break;
    case UNLOCK:
      path   = lookup_pr_path (fname);
      result = unlock_pr (path);
      xfree (path);
      break;
    case CHECK:
      result = check_pr (fp);
      break;
    case LOCKDB:
    case UNLOCKDB:
      result = 1; /* so that the program exits with 0 status */
      break;
    default: 
      result = modify_pr (fp);
      break;
  }

  fclose (fp);

  if (edit_options == UNLOCKDB || edit_options == MODIFY)
    unlock_gnats ();

  unblock_signals ();

  free_db_conf();
  exit (!result);
}

static void
usage ()
{
  fprintf (stderr, "\
Usage: %s [-cuLUVFh] [-d directory] [-f filename] [-l username]\n\
          [--directory=directory] [--filename=filename] [--lock=user]\n\
          [--unlock] [--lockdb] [--unlockdb] [--help] [--version]\n\
          [--check] [PR]\n", program_name);
}

static void
version ()
{
  printf ("pr-edit %s\n", version_string);
}
