/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  TransactionMajorState,
  TransactionMinorState,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  WalletCli,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

const coinRsaCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  rsaKeySize: 1024,
};

const coin_u1 = (curr: string): CoinConfig => ({
  ...coinRsaCommon,
  name: `${curr}_u1`,
  value: `${curr}:1`,
  feeDeposit: `${curr}:0`,
  feeRefresh: `${curr}:0`,
  feeRefund: `${curr}:0`,
  feeWithdraw: `${curr}:1`,
});

const coin_u5 = (curr: string): CoinConfig => ({
  ...coinRsaCommon,
  name: `${curr}_u5`,
  value: `${curr}:5`,
  feeDeposit: `${curr}:0`,
  feeRefresh: `${curr}:0`,
  feeRefund: `${curr}:0`,
  feeWithdraw: `${curr}:1`,
});

export const weirdCoinConfig = [coin_u1, coin_u5];

/**
 * Test withdrawal with a weird denomination structure to
 * make sure fees are computed as expected.
 */
export async function runWithdrawalFeesTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw-password";
  let exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);

  await exchange.addBankAccount("1", {
    wireGatewayAuth: {
      username: exchangeBankUsername,
      password: exchangeBankPassword,
    },
    wireGatewayApiBaseUrl: new URL(
      "accounts/exchange/taler-wire-gateway/",
      bank.baseUrl,
    ).href,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  const coinConfig: CoinConfig[] = weirdCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  console.log("setup done!");

  const wallet = new WalletCli(t);

  await wallet.client.call(WalletApiOperation.AddExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  const amount = "TESTKUDOS:7.5";

  const user = await bankClient.createRandomBankUser();
  bankClient.setAuth(user);
  const wop = await bankClient.createWithdrawalOperation(user.username, amount);

  t.logStep("Hand it to the wallet");

  const details = await wallet.client.call(
    WalletApiOperation.GetWithdrawalDetailsForUri,
    {
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  console.log(j2s(details));

  const myAmount = details.amount;
  t.assertTrue(!!myAmount);

  const amountDetails = await wallet.client.call(
    WalletApiOperation.GetWithdrawalDetailsForAmount,
    {
      amount: myAmount,
      exchangeBaseUrl: details.possibleExchanges[0].exchangeBaseUrl,
    },
  );

  console.log(j2s(amountDetails));

  t.assertAmountEquals(amountDetails.amountEffective, "TESTKUDOS:5");
  t.assertAmountEquals(amountDetails.amountRaw, "TESTKUDOS:7.5");

  t.logStep("Complete all pending operations");

  await wallet.runPending();

  t.logStep("Withdraw (AKA select)");

  const acceptResp = await wallet.client.call(
    WalletApiOperation.AcceptBankIntegratedWithdrawal,
    {
      exchangeBaseUrl: exchange.baseUrl,
      talerWithdrawUri: wop.taler_withdraw_uri,
    },
  );

  t.logStep("Confirm it");

  await wallet.client.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: acceptResp.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.BankConfirmTransfer,
    },
  });

  await bankClient.confirmWithdrawalOperation(user.username, {
    withdrawalOperationId: wop.withdrawal_id,
  });
  await wallet.runUntilDone();

  t.logStep("Check balance");

  const balResp = await wallet.client.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(balResp));

  t.assertAmountEquals(balResp.balances[0].available, "TESTKUDOS:5");

  const txns = await wallet.client.call(WalletApiOperation.GetTransactions, {});
  console.log(j2s(txns));
}

runWithdrawalFeesTest.suites = ["wallet"];
