/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TalerCorebankApiClient,
  TalerMerchantApi,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  SimpleTestEnvironmentNg3,
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  HarnessExchangeBankAccount,
  MerchantService,
  WalletCli,
  WalletClient,
  delayMs,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

async function revokeAllWalletCoins(req: {
  walletClient: WalletClient;
  exchange: ExchangeService;
  merchant: MerchantService;
}): Promise<void> {
  const { walletClient, exchange, merchant } = req;
  const coinDump = await walletClient.call(WalletApiOperation.DumpCoins, {});
  console.log(coinDump);
  const usedDenomHashes = new Set<string>();
  for (const coin of coinDump.coins) {
    usedDenomHashes.add(coin.denomPubHash);
  }
  for (const x of usedDenomHashes.values()) {
    await exchange.revokeDenomination(x);
  }
  await delayMs(1000);
  await exchange.keyup();
  await delayMs(1000);
  await merchant.stop();
  await merchant.start();
  await merchant.pingUntilAvailable();
}

async function createTestEnvironment(
  t: GlobalTestState,
): Promise<SimpleTestEnvironmentNg3> {
  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw-password";
  let exchangePaytoUri = getTestHarnessPaytoForLabel(exchangeBankUsername);

  const exchangeBankAccount: HarnessExchangeBankAccount = {
    wireGatewayAuth: {
      username: exchangeBankUsername,
      password: exchangeBankPassword,
    },
    wireGatewayApiBaseUrl: new URL(
      "accounts/exchange/taler-wire-gateway/",
      bank.baseUrl,
    ).href,
    accountPaytoUri: exchangePaytoUri,
  };

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  const coin_u1: CoinConfig = {
    cipher: "RSA" as const,
    durationLegal: "3 years",
    durationSpend: "2 years",
    durationWithdraw: "7 days",
    rsaKeySize: 1024,
    name: `TESTKUDOS_u1`,
    value: `TESTKUDOS:1`,
    feeDeposit: `TESTKUDOS:0`,
    feeRefresh: `TESTKUDOS:0`,
    feeRefund: `TESTKUDOS:0`,
    feeWithdraw: `TESTKUDOS:0`,
  };

  exchange.addCoinConfigList([coin_u1]);

  await exchange.start();
  await exchange.pingUntilAvailable();

  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const { accessToken: adminAccessToken } = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
  });

  await merchant.addInstanceWithWireAccount({
    id: "minst1",
    name: "minst1",
    paytoUris: [getTestHarnessPaytoForLabel("minst1")],
  }, {adminAccessToken});

  console.log("setup done!");

  const wallet = new WalletCli(t);

  const { walletService, walletClient } = await createWalletDaemonWithClient(
    t,
    {
      name: "default",
    },
  );

  return {
    commonDb: db,
    exchange,
    merchant,
    walletClient,
    merchantAdminAccessToken: adminAccessToken,
    walletService,
    bank,
    bankClient,
    exchangeBankAccount,
  };
}

/**
 * Basic time travel test.
 */
export async function runRevocationTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createTestEnvironment(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:15",
  });
  await wres.withdrawalFinishedCond;

  console.log("revoking first time");
  await revokeAllWalletCoins({ walletClient, exchange, merchant });

  // FIXME: this shouldn't be necessary once https://bugs.taler.net/n/6565
  // is implemented.
  await walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
    exchangeBaseUrl: exchange.baseUrl,
  });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log("wallet balance", bal);

  const order = {
    summary: "Buy me!",
    amount: "TESTKUDOS:10",
    fulfillment_url: "taler://fulfillment-success/thx",
  } satisfies TalerMerchantApi.Order;

  await makeTestPaymentV2(t, { walletClient, merchant, order, merchantAdminAccessToken });

  await walletClient.call(WalletApiOperation.ClearDb, {});

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:15",
  });

  const coinDump = await walletClient.call(WalletApiOperation.DumpCoins, {});
  console.log(coinDump);
  const coinPubList = coinDump.coins.map((x) => x.coinPub);
  await walletClient.call(WalletApiOperation.ForceRefresh, {
    refreshCoinSpecs: coinPubList.map((x) => ({ coinPub: x })),
  });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  console.log("revoking second time");
  await revokeAllWalletCoins({ walletClient, exchange, merchant });

  // FIXME: this shouldn't be necessary once https://bugs.taler.net/n/6565
  // is implemented.
  await walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
    exchangeBaseUrl: exchange.baseUrl,
  });
  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
  {
    const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
    console.log("wallet balance", bal);
  }

  await makeTestPaymentV2(t, { walletClient, merchant, order, merchantAdminAccessToken });
}

runRevocationTest.timeoutMs = 120000;
runRevocationTest.suites = ["wallet"];
runRevocationTest.experimental = true;
