/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Amounts,
  Duration,
  succeedOrThrow,
  TalerMerchantInstanceHttpClient,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { delayMs, GlobalTestState } from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runRefundIncrementalTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createSimpleTestkudosEnvironmentV3(t);

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Set up order.

  const orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:10",
        fulfillment_url: "taler://fulfillment-success/thx",
      },
      refund_delay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ minutes: 5 }),
      ),
    }),
  );

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  // Make wallet pay for the order

  const r1 = await walletClient.call(WalletApiOperation.PreparePayForUri, {
    talerPayUri: orderStatus.taler_pay_uri,
  });

  await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: r1.transactionId,
  });

  // Check if payment was successful.

  orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "paid");

  let ref = succeedOrThrow(
    await merchantClient.addRefund(merchantAdminAccessToken, orderResp.order_id, {
      refund: "TESTKUDOS:2.5",
      reason: "foo",
    }),
  );

  console.log("first refund increase response", ref);

  {
    let wr = await walletClient.call(WalletApiOperation.StartRefundQuery, {
      transactionId: r1.transactionId,
    });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
    console.log(wr);
    const txs = await walletClient.call(WalletApiOperation.GetTransactions, {});
    console.log(
      "transactions after applying first refund:",
      JSON.stringify(txs, undefined, 2),
    );
  }

  // Wait at least a second, because otherwise the increased
  // refund will be grouped with the previous one.
  await delayMs(1200);

  ref = succeedOrThrow(
    await merchantClient.addRefund(merchantAdminAccessToken, orderResp.order_id, {
      refund: "TESTKUDOS:5",
      reason: "bar",
    }),
  );

  console.log("second refund increase response", ref);

  // Wait at least a second, because otherwise the increased
  // refund will be grouped with the previous one.
  await delayMs(1200);

  ref = succeedOrThrow(
    await merchantClient.addRefund(merchantAdminAccessToken, orderResp.order_id, {
      refund: "TESTKUDOS:10",
      reason: "bar",
    }),
  );

  console.log("third refund increase response", ref);

  {
    let wr = await walletClient.call(WalletApiOperation.StartRefundQuery, {
      transactionId: r1.transactionId,
    });
    await walletClient.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );
    console.log(wr);
  }

  orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "paid");

  t.assertAmountEquals(orderStatus.refund_amount, "TESTKUDOS:10");

  console.log(JSON.stringify(orderStatus, undefined, 2));

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(JSON.stringify(bal, undefined, 2));

  {
    const txs = await walletClient.call(WalletApiOperation.GetTransactions, {});
    console.log(JSON.stringify(txs, undefined, 2));

    const txTypes = txs.transactions.map((x) => x.type);
    t.assertDeepEqual(txTypes, ["withdrawal", "payment", "refund", "refund"]);

    for (const tx of txs.transactions) {
      if (tx.type !== TransactionType.Refund) {
        continue;
      }
      t.assertAmountLeq(tx.amountEffective, tx.amountRaw);
    }

    const raw = Amounts.sum(
      txs.transactions
        .filter((x) => x.type === TransactionType.Refund)
        .map((x) => x.amountRaw),
    ).amount;

    t.assertAmountEquals("TESTKUDOS:10", raw);

    const effective = Amounts.sum(
      txs.transactions
        .filter((x) => x.type === TransactionType.Refund)
        .map((x) => x.amountEffective),
    ).amount;

    t.assertAmountEquals("TESTKUDOS:8.59", effective);
  }

  await t.shutdown();
}

runRefundIncrementalTest.suites = ["wallet"];
