/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import test from "ava";
import * as ts from "typescript";
import { processFileForTesting } from "./potextract.js";

function wrapIntoFunction(src: string): string {
  return `
function testing():VNode {
const TALER_SCREEN_ID = 5;
return ${src}
}
`;
}

function process(src: string): string {
  const source = ts.createSourceFile(
    "test.tsx",
    wrapIntoFunction(src),
    ts.ScriptTarget.ES2020,
  );
  return processFileForTesting(source).trim();
}

test("should extract the key from inner body", (t) => {
  t.deepEqual(
    process(`<i18n.Translate>something</i18n.Translate>`),
    `#. screenid: 5
#: test.tsx:4
#, c-format
msgid "something"
msgstr ""`,
  );
});

test("should support context on tags", (t) => {
  t.deepEqual(
    process(
      `return <div>
      <i18n.Translate context="some_context" anotherkey="not the context" asd={"asd"} zxc={asd()}>something</i18n.Translate>
      </div>`,
    ),
    `#. screenid: 5
#: test.tsx:5
#, c-format
msgctxt "some_context"
msgid "something"
msgstr ""`,
  );
});

test("should support context on string template", (t) => {
  t.deepEqual(
    process(`return i18n.context("wire transfer")\`send\`;`),
    `#. screenid: 5
#: test.tsx:4
#, c-format
msgctxt "wire transfer"
msgid "send"
msgstr ""`,
  );
});

test("should support same message id with different context", (t) => {
  t.deepEqual(
    process(
      `return i18n.context("wire transfer")\`send\` + i18n.context("gift")\`send\`;`,
    ),
    `#. screenid: 5
#: test.tsx:4
#, c-format
msgctxt "wire transfer"
msgid "send"
msgstr ""

#. screenid: 5
#: test.tsx:4
#, c-format
msgctxt "gift"
msgid "send"
msgstr ""`,
  );
});

test("should support on string template", (t) => {
  t.deepEqual(
    process(`
      // comment of the translation
      return i18n.str\`another key\`;`),
    `#. screenid: 5
#. comment of the translation
#: test.tsx:6
#, c-format
msgid "another key"
msgstr ""`,
  );
});

test("should override screen id", (t) => {
  t.deepEqual(
    process(`
      {
      const TALER_SCREEN_ID = 6;
      // comment of the translation
      return i18n.str\`another key\`;
      }
      `),
    `#. screenid: 6
#. comment of the translation
#: test.tsx:8
#, c-format
msgid "another key"
msgstr ""`,
  );
});


test("should support nested tags", (t) => {
  t.deepEqual(
    process(`
      <p class="warning">
        <i18n.Translate>
          Purging an instance{" "}
          <b>
            <i18n.Translate>This cannot be undone!</i18n.Translate>
          </b>
        </i18n.Translate>
        `),
    `#. screenid: 5
#: test.tsx:6
#, c-format
msgid "Purging an instance %1$s"
msgstr ""

#. screenid: 5
#: test.tsx:9
#, c-format
msgid "This cannot be undone!"
msgstr ""`,
  );
});
