/*
  This file is part of TALER
  (C) 2020-2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU Affero General Public License as
  published by the Free Software Foundation; either version 3,
  or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not,
  see <http://www.gnu.org/licenses/>
*/

/**
 * @file taler-merchant-httpd_private-post-instances.c
 * @brief implementing POST /instances request handling
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-post-instances.h"
#include "taler-merchant-httpd_helper.h"
#include "taler-merchant-httpd.h"
#include "taler-merchant-httpd_auth.h"
#include "taler-merchant-httpd_mfa.h"
#include "taler_merchant_bank_lib.h"
#include <taler/taler_dbevents.h>
#include <taler/taler_json_lib.h>
#include <regex.h>

/**
 * How often do we retry the simple INSERT database transaction?
 */
#define MAX_RETRIES 3


/**
 * Generate an instance, given its configuration.
 *
 * @param rh context of the handler
 * @param connection the MHD connection to handle
 * @param[in,out] hc context with further information about the request
 * @param login_token_expiration set to how long a login token validity
 *   should be, use zero if no login token should be created
 * @param validation_needed true if self-provisioned and
 *   email/phone registration is required before the
 *   instance can become fully active
 * @return MHD result code
 */
static MHD_RESULT
post_instances (const struct TMH_RequestHandler *rh,
                struct MHD_Connection *connection,
                struct TMH_HandlerContext *hc,
                struct GNUNET_TIME_Relative login_token_expiration,
                bool validation_needed)
{
  struct TALER_MERCHANTDB_InstanceSettings is = { 0 };
  struct TALER_MERCHANTDB_InstanceAuthSettings ias;
  const char *auth_password = NULL;
  struct TMH_WireMethod *wm_head = NULL;
  struct TMH_WireMethod *wm_tail = NULL;
  const json_t *jauth;
  const char *iphone = NULL;
  bool no_pay_delay;
  bool no_refund_delay;
  bool no_transfer_delay;
  bool no_rounding_interval;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_string ("id",
                             (const char **) &is.id),
    GNUNET_JSON_spec_string ("name",
                             (const char **) &is.name),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("email",
                               (const char **) &is.email),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("phone_number",
                               &iphone),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("website",
                               (const char **) &is.website),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("logo",
                               (const char **) &is.logo),
      NULL),
    GNUNET_JSON_spec_object_const ("auth",
                                   &jauth),
    GNUNET_JSON_spec_json ("address",
                           &is.address),
    GNUNET_JSON_spec_json ("jurisdiction",
                           &is.jurisdiction),
    GNUNET_JSON_spec_bool ("use_stefan",
                           &is.use_stefan),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_relative_time ("default_pay_delay",
                                      &is.default_pay_delay),
      &no_pay_delay),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_relative_time ("default_refund_delay",
                                      &is.default_refund_delay),
      &no_refund_delay),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_relative_time ("default_wire_transfer_delay",
                                      &is.default_wire_transfer_delay),
      &no_transfer_delay),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_time_rounder_interval (
        "default_wire_transfer_rounding_interval",
        &is.default_wire_transfer_rounding_interval),
      &no_rounding_interval),
    GNUNET_JSON_spec_end ()
  };

  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
  }
  if (no_pay_delay)
    is.default_pay_delay = TMH_default_pay_delay;
  if (no_refund_delay)
    is.default_refund_delay = TMH_default_refund_delay;
  if (no_transfer_delay)
    is.default_wire_transfer_delay = TMH_default_wire_transfer_delay;
  if (no_rounding_interval)
    is.default_wire_transfer_rounding_interval
      = TMH_default_wire_transfer_rounding_interval;
  if (GNUNET_TIME_relative_is_forever (is.default_pay_delay))
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "default_pay_delay");
  }
  if (GNUNET_TIME_relative_is_forever (is.default_refund_delay))
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "default_refund_delay");
  }
  if (GNUNET_TIME_relative_is_forever (is.default_wire_transfer_delay))
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "default_wire_transfer_delay");
  }
  if (NULL != iphone)
  {
    is.phone = TALER_MERCHANT_phone_validate_normalize (iphone,
                                                        false);
    if (NULL == is.phone)
    {
      GNUNET_break_op (0);
      GNUNET_JSON_parse_free (spec);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "phone_number");
    }
  }
  if ( (NULL != is.email) &&
       (! TALER_MERCHANT_email_valid (is.email)) )
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    GNUNET_free (is.phone);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "email");
  }

  {
    enum GNUNET_GenericReturnValue ret;

    ret = TMH_check_auth_config (connection,
                                 jauth,
                                 &auth_password);
    if (GNUNET_OK != ret)
    {
      GNUNET_free (is.phone);
      GNUNET_JSON_parse_free (spec);
      return (GNUNET_NO == ret) ? MHD_YES : MHD_NO;
    }
  }

  /* check 'id' well-formed */
  {
    static bool once;
    static regex_t reg;
    bool id_wellformed = true;

    if (! once)
    {
      once = true;
      GNUNET_assert (0 ==
                     regcomp (&reg,
                              "^[A-Za-z0-9][A-Za-z0-9_.@-]+$",
                              REG_EXTENDED));
    }

    if (0 != regexec (&reg,
                      is.id,
                      0, NULL, 0))
      id_wellformed = false;
    if (! id_wellformed)
    {
      GNUNET_JSON_parse_free (spec);
      GNUNET_free (is.phone);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                         "id");
    }
  }

  if (! TMH_location_object_valid (is.address))
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    GNUNET_free (is.phone);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "address");
  }

  if (! TMH_location_object_valid (is.jurisdiction))
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    GNUNET_free (is.phone);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "jurisdiction");
  }

  if ( (NULL != is.logo) &&
       (! TALER_MERCHANT_image_data_url_valid (is.logo)) )
  {
    GNUNET_break_op (0);
    GNUNET_JSON_parse_free (spec);
    GNUNET_free (is.phone);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_BAD_REQUEST,
                                       TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                       "logo");
  }

  {
    /* Test if an instance of this id is known */
    struct TMH_MerchantInstance *mi;

    mi = TMH_lookup_instance (is.id);
    if (NULL != mi)
    {
      if (mi->deleted)
      {
        GNUNET_JSON_parse_free (spec);
        GNUNET_free (is.phone);
        return TALER_MHD_reply_with_error (connection,
                                           MHD_HTTP_CONFLICT,
                                           TALER_EC_MERCHANT_PRIVATE_POST_INSTANCES_PURGE_REQUIRED,
                                           is.id);
      }
      /* Check for idempotency */
      if ( (0 == strcmp (mi->settings.id,
                         is.id)) &&
           (0 == strcmp (mi->settings.name,
                         is.name)) &&
           ((mi->settings.email == is.email) ||
            (NULL != is.email && NULL != mi->settings.email &&
             0 == strcmp (mi->settings.email,
                          is.email))) &&
           ((mi->settings.website == is.website) ||
            (NULL != is.website && NULL != mi->settings.website &&
             0 == strcmp (mi->settings.website,
                          is.website))) &&
           ((mi->settings.logo == is.logo) ||
            (NULL != is.logo && NULL != mi->settings.logo &&
             0 == strcmp (mi->settings.logo,
                          is.logo))) &&
           ( ( (NULL != auth_password) &&
               (GNUNET_OK ==
                TMH_check_auth (auth_password,
                                &mi->auth.auth_salt,
                                &mi->auth.auth_hash)) ) ||
             ( (NULL == auth_password) &&
               (GNUNET_YES ==
                GNUNET_is_zero (&mi->auth.auth_hash))) ) &&
           (1 == json_equal (mi->settings.address,
                             is.address)) &&
           (1 == json_equal (mi->settings.jurisdiction,
                             is.jurisdiction)) &&
           (mi->settings.use_stefan == is.use_stefan) &&
           (GNUNET_TIME_relative_cmp (mi->settings.default_wire_transfer_delay,
                                      ==,
                                      is.default_wire_transfer_delay)) &&
           (GNUNET_TIME_relative_cmp (mi->settings.default_pay_delay,
                                      ==,
                                      is.default_pay_delay)) &&
           (GNUNET_TIME_relative_cmp (mi->settings.default_refund_delay,
                                      ==,
                                      is.default_refund_delay)) )
      {
        GNUNET_JSON_parse_free (spec);
        GNUNET_free (is.phone);
        return TALER_MHD_reply_static (connection,
                                       MHD_HTTP_NO_CONTENT,
                                       NULL,
                                       NULL,
                                       0);
      }
      GNUNET_JSON_parse_free (spec);
      GNUNET_free (is.phone);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_CONFLICT,
                                         TALER_EC_MERCHANT_PRIVATE_POST_INSTANCES_ALREADY_EXISTS,
                                         is.id);
    }
  }

  /* Check MFA is satisfied */
  if (validation_needed)
  {
    enum GNUNET_GenericReturnValue ret = GNUNET_SYSERR;

    if ( (0 != (TEH_TCS_SMS & TEH_mandatory_tan_channels)) &&
         (NULL == is.phone) )
    {
      GNUNET_break_op (0);
      GNUNET_JSON_parse_free (spec);
      GNUNET_free (is.phone); /* does nothing... */
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MISSING,
                                         "phone_number");

    }
    if ( (0 != (TEH_TCS_EMAIL & TEH_mandatory_tan_channels)) &&
         (NULL == is.email) )
    {
      GNUNET_break_op (0);
      GNUNET_JSON_parse_free (spec);
      GNUNET_free (is.phone);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_BAD_REQUEST,
                                         TALER_EC_GENERIC_PARAMETER_MISSING,
                                         "email");
    }
    switch (TEH_mandatory_tan_channels)
    {
    case TEH_TCS_NONE:
      GNUNET_assert (0);
      ret = GNUNET_OK;
      break;
    case TEH_TCS_SMS:
      is.phone_validated = true;
      ret = TMH_mfa_challenges_do (hc,
                                   TALER_MERCHANT_MFA_CO_INSTANCE_PROVISION,
                                   true,
                                   TALER_MERCHANT_MFA_CHANNEL_SMS,
                                   is.phone,
                                   TALER_MERCHANT_MFA_CHANNEL_NONE);
      break;
    case TEH_TCS_EMAIL:
      is.email_validated = true;
      ret = TMH_mfa_challenges_do (hc,
                                   TALER_MERCHANT_MFA_CO_INSTANCE_PROVISION,
                                   true,
                                   TALER_MERCHANT_MFA_CHANNEL_EMAIL,
                                   is.email,
                                   TALER_MERCHANT_MFA_CHANNEL_NONE);
      break;
    case TEH_TCS_EMAIL_AND_SMS:
      is.phone_validated = true;
      is.email_validated = true;
      ret = TMH_mfa_challenges_do (hc,
                                   TALER_MERCHANT_MFA_CO_INSTANCE_PROVISION,
                                   true,
                                   TALER_MERCHANT_MFA_CHANNEL_SMS,
                                   is.phone,
                                   TALER_MERCHANT_MFA_CHANNEL_EMAIL,
                                   is.email,
                                   TALER_MERCHANT_MFA_CHANNEL_NONE);
      break;
    }
    if (GNUNET_OK != ret)
    {
      GNUNET_JSON_parse_free (spec);
      GNUNET_free (is.phone);
      return (GNUNET_NO == ret)
        ? MHD_YES
        : MHD_NO;
    }
  }

  /* handle authentication token setup */
  if (NULL == auth_password)
  {
    memset (&ias.auth_salt,
            0,
            sizeof (ias.auth_salt));
    memset (&ias.auth_hash,
            0,
            sizeof (ias.auth_hash));
  }
  else
  {
    /* Sets 'auth_salt' and 'auth_hash' */
    TMH_compute_auth (auth_password,
                      &ias.auth_salt,
                      &ias.auth_hash);
  }

  /* create in-memory data structure */
  {
    struct TMH_MerchantInstance *mi;
    enum GNUNET_DB_QueryStatus qs;

    mi = GNUNET_new (struct TMH_MerchantInstance);
    mi->wm_head = wm_head;
    mi->wm_tail = wm_tail;
    mi->settings = is;
    mi->settings.address = json_incref (mi->settings.address);
    mi->settings.jurisdiction = json_incref (mi->settings.jurisdiction);
    mi->settings.id = GNUNET_strdup (is.id);
    mi->settings.name = GNUNET_strdup (is.name);
    if (NULL != is.email)
      mi->settings.email = GNUNET_strdup (is.email);
    mi->settings.phone = is.phone;
    is.phone = NULL;
    if (NULL != is.website)
      mi->settings.website = GNUNET_strdup (is.website);
    if (NULL != is.logo)
      mi->settings.logo = GNUNET_strdup (is.logo);
    mi->auth = ias;
    mi->validation_needed = validation_needed;
    GNUNET_CRYPTO_eddsa_key_create (&mi->merchant_priv.eddsa_priv);
    GNUNET_CRYPTO_eddsa_key_get_public (&mi->merchant_priv.eddsa_priv,
                                        &mi->merchant_pub.eddsa_pub);

    for (unsigned int i = 0; i<MAX_RETRIES; i++)
    {
      if (GNUNET_OK !=
          TMH_db->start (TMH_db->cls,
                         "post /instances"))
      {
        mi->rc = 1;
        TMH_instance_decref (mi);
        GNUNET_JSON_parse_free (spec);
        return TALER_MHD_reply_with_error (connection,
                                           MHD_HTTP_INTERNAL_SERVER_ERROR,
                                           TALER_EC_GENERIC_DB_START_FAILED,
                                           NULL);
      }
      qs = TMH_db->insert_instance (TMH_db->cls,
                                    &mi->merchant_pub,
                                    &mi->merchant_priv,
                                    &mi->settings,
                                    &mi->auth,
                                    validation_needed);
      switch (qs)
      {
      case GNUNET_DB_STATUS_HARD_ERROR:
        {
          MHD_RESULT ret;

          TMH_db->rollback (TMH_db->cls);
          GNUNET_break (0);
          ret = TALER_MHD_reply_with_error (connection,
                                            MHD_HTTP_INTERNAL_SERVER_ERROR,
                                            TALER_EC_GENERIC_DB_STORE_FAILED,
                                            is.id);
          mi->rc = 1;
          TMH_instance_decref (mi);
          GNUNET_JSON_parse_free (spec);
          return ret;
        }
      case GNUNET_DB_STATUS_SOFT_ERROR:
        goto retry;
      case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
        {
          MHD_RESULT ret;

          TMH_db->rollback (TMH_db->cls);
          GNUNET_break (0);
          ret = TALER_MHD_reply_with_error (connection,
                                            MHD_HTTP_CONFLICT,
                                            TALER_EC_MERCHANT_PRIVATE_POST_INSTANCES_ALREADY_EXISTS,
                                            is.id);
          mi->rc = 1;
          TMH_instance_decref (mi);
          GNUNET_JSON_parse_free (spec);
          return ret;
        }
      case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
        /* handled below */
        break;
      }
      qs = TMH_db->commit (TMH_db->cls);
      if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
        qs = GNUNET_DB_STATUS_SUCCESS_ONE_RESULT;
retry:
      if (GNUNET_DB_STATUS_SOFT_ERROR != qs)
        break; /* success! -- or hard failure */
    } /* for .. MAX_RETRIES */
    if (GNUNET_DB_STATUS_SUCCESS_ONE_RESULT != qs)
    {
      mi->rc = 1;
      TMH_instance_decref (mi);
      GNUNET_JSON_parse_free (spec);
      return TALER_MHD_reply_with_error (connection,
                                         MHD_HTTP_INTERNAL_SERVER_ERROR,
                                         TALER_EC_GENERIC_DB_COMMIT_FAILED,
                                         NULL);
    }
    /* Finally, also update our running process */
    GNUNET_assert (GNUNET_OK ==
                   TMH_add_instance (mi));
    TMH_reload_instances (mi->settings.id);
  }
  GNUNET_JSON_parse_free (spec);
  if (GNUNET_TIME_relative_is_zero (login_token_expiration))
  {
    return TALER_MHD_reply_static (connection,
                                   MHD_HTTP_NO_CONTENT,
                                   NULL,
                                   NULL,
                                   0);
  }

  {
    struct TALER_MERCHANTDB_LoginTokenP btoken;
    enum TMH_AuthScope iscope = TMH_AS_REFRESHABLE | TMH_AS_SPA;
    enum GNUNET_DB_QueryStatus qs;
    struct GNUNET_TIME_Timestamp expiration_time;
    bool refreshable = true;

    GNUNET_CRYPTO_random_block (GNUNET_CRYPTO_QUALITY_NONCE,
                                &btoken,
                                sizeof (btoken));
    expiration_time
      = GNUNET_TIME_relative_to_timestamp (login_token_expiration);
    qs = TMH_db->insert_login_token (TMH_db->cls,
                                     is.id,
                                     &btoken,
                                     GNUNET_TIME_timestamp_get (),
                                     expiration_time,
                                     iscope,
                                     "login token from instance creation");
    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
    case GNUNET_DB_STATUS_SOFT_ERROR:
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      GNUNET_break (0);
      return TALER_MHD_reply_with_ec (connection,
                                      TALER_EC_GENERIC_DB_STORE_FAILED,
                                      "insert_login_token");
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      break;
    }

    {
      char *tok;
      MHD_RESULT ret;
      char *val;

      val = GNUNET_STRINGS_data_to_string_alloc (&btoken,
                                                 sizeof (btoken));
      GNUNET_asprintf (&tok,
                       RFC_8959_PREFIX "%s",
                       val);
      GNUNET_free (val);
      ret = TALER_MHD_REPLY_JSON_PACK (
        connection,
        MHD_HTTP_OK,
        GNUNET_JSON_pack_string ("access_token",
                                 tok),
        GNUNET_JSON_pack_string ("token",
                                 tok),
        GNUNET_JSON_pack_string ("scope",
                                 TMH_get_name_by_scope (iscope,
                                                        &refreshable)),
        GNUNET_JSON_pack_bool ("refreshable",
                               refreshable),
        GNUNET_JSON_pack_timestamp ("expiration",
                                    expiration_time));
      GNUNET_free (tok);
      return ret;
    }
  }
}


/**
 * Generate an instance, given its configuration.
 *
 * @param rh context of the handler
 * @param connection the MHD connection to handle
 * @param[in,out] hc context with further information about the request
 * @return MHD result code
 */
MHD_RESULT
TMH_private_post_instances (const struct TMH_RequestHandler *rh,
                            struct MHD_Connection *connection,
                            struct TMH_HandlerContext *hc)
{
  return post_instances (rh,
                         connection,
                         hc,
                         GNUNET_TIME_UNIT_ZERO,
                         false);
}


/**
 * Generate an instance, given its configuration.
 * Public handler to be used when self-provisioning.
 *
 * @param rh context of the handler
 * @param connection the MHD connection to handle
 * @param[in,out] hc context with further information about the request
 * @return MHD result code
 */
MHD_RESULT
TMH_public_post_instances (const struct TMH_RequestHandler *rh,
                           struct MHD_Connection *connection,
                           struct TMH_HandlerContext *hc)
{
  struct GNUNET_TIME_Relative expiration;

  TALER_MHD_parse_request_rel_time (connection,
                                    "token_validity_ms",
                                    &expiration);
  if (GNUNET_YES !=
      TMH_have_self_provisioning)
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (connection,
                                       MHD_HTTP_FORBIDDEN,
                                       TALER_EC_MERCHANT_GENERIC_UNAUTHORIZED,
                                       "Self-provisioning is not enabled");
  }

  return post_instances (rh,
                         connection,
                         hc,
                         expiration,
                         TEH_TCS_NONE !=
                         TEH_mandatory_tan_channels);
}


/* end of taler-merchant-httpd_private-post-instances.c */
