/*
   This file is part of TALER
   Copyright (C) 2022, 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_product.c
 * @brief Implementation of the lookup_product function for Postgres
 * @author Iván Ávalos
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_lookup_product.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_product (void *cls,
                       const char *instance_id,
                       const char *product_id,
                       struct TALER_MERCHANTDB_ProductDetails *pd,
                       size_t *num_categories,
                       uint64_t **categories)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_string (product_id),
    GNUNET_PQ_query_param_end
  };

  PREPARE (pg,
           "lookup_product",
           "SELECT"
           " mi.description"
           ",mi.description_i18n::TEXT"
           ",mi.product_name"
           ",mi.unit"
           ",mi.price_array"
           ",mi.taxes::TEXT"
           ",mi.total_stock"
           ",mi.total_stock_frac"
           ",mi.allow_fractional_quantity"
           ",mi.fractional_precision_level"
           ",mi.total_sold"
           ",mi.total_sold_frac"
           ",mi.total_lost"
           ",mi.total_lost_frac"
           ",mi.image"
           ",mi.address::TEXT"
           ",mi.next_restock"
           ",mi.minimum_age"
           ",mi.product_group_serial"
           ",mi.money_pot_serial"
           ",mi.price_is_net"
           ",t.category_array AS categories"
           " FROM merchant_inventory mi"
           " JOIN merchant_instances inst"
           "   USING (merchant_serial)"
           ",LATERAL ("
           "   SELECT ARRAY ("
           "     SELECT mpc.category_serial"
           "       FROM merchant_product_categories mpc"
           "      WHERE mpc.product_serial = mi.product_serial"
           "   ) AS category_array"
           " ) t"
           " WHERE inst.merchant_id=$1"
           "   AND mi.product_id=$2"
           );
  if (NULL == pd)
  {
    struct GNUNET_PQ_ResultSpec rs_null[] = {
      GNUNET_PQ_result_spec_end
    };

    check_connection (pg);
    return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                     "lookup_product",
                                                     params,
                                                     rs_null);
  }
  else
  {
    char *my_name = NULL;
    char *my_description = NULL;
    json_t *my_description_i18n = NULL;
    char *my_unit = NULL;
    char *my_image = NULL;
    json_t *my_address = NULL;
    json_t *my_taxes = NULL;
    uint64_t *my_categories = NULL;
    struct TALER_Amount *my_price_array = NULL;
    size_t my_price_array_length = 0;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_string ("description",
                                    &my_description),
      TALER_PQ_result_spec_json ("description_i18n",
                                 &my_description_i18n),
      GNUNET_PQ_result_spec_string ("product_name",
                                    &my_name),
      GNUNET_PQ_result_spec_string ("unit",
                                    &my_unit),
      TALER_PQ_result_spec_array_amount_with_currency (pg->conn,
                                                       "price_array",
                                                       &my_price_array_length,
                                                       &my_price_array),
      TALER_PQ_result_spec_json ("taxes",
                                 &my_taxes),
      GNUNET_PQ_result_spec_uint64 ("total_stock",
                                    &pd->total_stock),
      GNUNET_PQ_result_spec_uint32 ("total_stock_frac",
                                    &pd->total_stock_frac),
      GNUNET_PQ_result_spec_bool ("allow_fractional_quantity",
                                  &pd->allow_fractional_quantity),
      GNUNET_PQ_result_spec_uint32 ("fractional_precision_level",
                                    &pd->fractional_precision_level),
      GNUNET_PQ_result_spec_uint64 ("total_sold",
                                    &pd->total_sold),
      GNUNET_PQ_result_spec_uint32 ("total_sold_frac",
                                    &pd->total_sold_frac),
      GNUNET_PQ_result_spec_uint64 ("total_lost",
                                    &pd->total_lost),
      GNUNET_PQ_result_spec_uint32 ("total_lost_frac",
                                    &pd->total_lost_frac),
      GNUNET_PQ_result_spec_string ("image",
                                    &my_image),
      TALER_PQ_result_spec_json ("address",
                                 &my_address),
      GNUNET_PQ_result_spec_timestamp ("next_restock",
                                       &pd->next_restock),
      GNUNET_PQ_result_spec_uint32 ("minimum_age",
                                    &pd->minimum_age),
      GNUNET_PQ_result_spec_array_uint64 (pg->conn,
                                          "categories",
                                          num_categories,
                                          &my_categories),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_uint64 ("product_group_serial",
                                      &pd->product_group_id),
        NULL),
      GNUNET_PQ_result_spec_allow_null (
        GNUNET_PQ_result_spec_uint64 ("money_pot_serial",
                                      &pd->money_pot_id),
        NULL),
      GNUNET_PQ_result_spec_bool ("price_is_net",
                                  &pd->price_is_net),
      GNUNET_PQ_result_spec_end
    };
    enum GNUNET_DB_QueryStatus qs;

    check_connection (pg);
    pd->product_group_id = 0;
    pd->money_pot_id = 0;
    qs = GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "lookup_product",
                                                   params,
                                                   rs);
    pd->product_name = my_name;
    pd->description = my_description;
    pd->description_i18n = my_description_i18n;
    pd->unit = my_unit;
    pd->taxes = my_taxes;
    pd->image = my_image;
    pd->image_hash = NULL;
    pd->address = my_address;
    pd->price_array = my_price_array;
    pd->price_array_length = my_price_array_length;
    *categories = my_categories;
    /* Clear original pointers to that cleanup_result doesn't squash them */
    my_name = NULL;
    my_description = NULL;
    my_description_i18n = NULL;
    my_unit = NULL;
    my_taxes = NULL;
    my_image = NULL;
    my_address = NULL;
    my_price_array = NULL;
    my_price_array_length = 0;
    my_categories = NULL;
    GNUNET_PQ_cleanup_result (rs);
    return qs;
  }
}
