/*
   This file is part of TALER
   Copyright (C) 2026 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_account_kyc_get_outdated.c
 * @brief Implementation of the account_kyc_get_outdated function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_account_kyc_get_outdated.h"
#include "pg_helper.h"

/**
 * Closure for kyc_status_cb().
 */
struct KycStatusContext
{
  /**
   * Function to call with results.
   */
  TALER_MERCHANTDB_KycOutdatedCallback kyc_cb;

  /**
   * Closure for @e kyc_cb.
   */
  void *kyc_cb_cls;

  /**
   * Number of results found.
   */
  unsigned int count;

  /**
   * Set to true on failure(s).
   */
  bool failure;
};


/**
 * Function to be called with the results of a SELECT statement
 * that has returned @a num_results results about accounts.
 *
 * @param[in,out] cls of type `struct KycStatusContext *`
 * @param result the postgres result
 * @param num_results the number of results in @a result
 */
static void
kyc_status_cb (void *cls,
               PGresult *result,
               unsigned int num_results)
{
  struct KycStatusContext *ksc = cls;

  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "Got %u outdated KYC records\n",
              num_results);
  for (unsigned int i = 0; i < num_results; i++)
  {
    struct TALER_MerchantWireHashP h_wire;
    char *exchange_url;
    char *instance_id;
    struct GNUNET_PQ_ResultSpec rs[] = {
      GNUNET_PQ_result_spec_auto_from_type ("h_wire",
                                            &h_wire),
      GNUNET_PQ_result_spec_string ("exchange_url",
                                    &exchange_url),
      GNUNET_PQ_result_spec_string ("instance_id",
                                    &instance_id),
      GNUNET_PQ_result_spec_end
    };

    if (GNUNET_OK !=
        GNUNET_PQ_extract_result (result,
                                  rs,
                                  i))
    {
      GNUNET_break (0);
      ksc->failure = true;
      return;
    }
    ksc->kyc_cb (ksc->kyc_cb_cls,
                 instance_id,
                 exchange_url,
                 &h_wire);
    GNUNET_PQ_cleanup_result (rs);
  }
}


enum GNUNET_DB_QueryStatus
TMH_PG_account_kyc_get_outdated (
  void *cls,
  TALER_MERCHANTDB_KycOutdatedCallback kyc_cb,
  void *kyc_cb_cls)
{
  struct PostgresClosure *pg = cls;
  struct KycStatusContext ksc = {
    .kyc_cb = kyc_cb,
    .kyc_cb_cls = kyc_cb_cls
  };
  struct GNUNET_TIME_Absolute now
    = GNUNET_TIME_absolute_get ();
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_absolute_time (&now),
    GNUNET_PQ_query_param_end
  };
  enum GNUNET_DB_QueryStatus qs;

  check_connection (pg);
  PREPARE (pg,
           "account_kyc_get_outdated",
           "SELECT "
           "  mi.merchant_id"
           " ,ma.h_wire"
           " ,kyc.exchange_url"
           " FROM merchant_kyc kyc"
           " JOIN merchant_accounts ma"
           "   USING (account_serial)"
           " JOIN merchant_instances mi"
           "   USING (merchant_serial)"
           " WHERE kyc.next_kyc_poll < $1"
           " ORDER BY kyc.next_kyc_poll ASC;");
  qs = GNUNET_PQ_eval_prepared_multi_select (
    pg->conn,
    "account_kyc_get_outdated",
    params,
    &kyc_status_cb,
    &ksc);
  if (ksc.failure)
  {
    GNUNET_break (0);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (0 > qs)
    return qs;
  return ksc.count;
}
