/*
   This file is part of TALER
   Copyright (C) 2024 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_upsert_donau_keys.c
 * @brief Implementation of the upsert_donau_keys function for Postgres
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "donau/donau_service.h"
#include "pg_upsert_donau_keys.h"
#include "pg_helper.h"

enum GNUNET_DB_QueryStatus
TMH_PG_upsert_donau_keys (
  void *cls,
  const struct DONAU_Keys *keys,
  struct GNUNET_TIME_Absolute first_retry)
{
  enum GNUNET_DB_QueryStatus qs;
  struct PostgresClosure *pg = cls;
  json_t *jkeys = DONAU_keys_to_json (keys);

  if (NULL == jkeys)
    return GNUNET_DB_STATUS_HARD_ERROR;

  {
    struct GNUNET_PQ_QueryParam params[] = {
      TALER_PQ_query_param_json (jkeys),
      GNUNET_PQ_query_param_absolute_time (&first_retry),
      GNUNET_PQ_query_param_string (keys->donau_url),
      GNUNET_PQ_query_param_end
    };

    check_connection (pg);
    PREPARE (pg,
             "insert_donau_keys",
             "INSERT INTO merchant_donau_keys"
             "(keys_json"
             ",first_retry"
             ",donau_url"
             ") VALUES ($1, $2, $3);");
    PREPARE (pg,
             "update_donau_keys",
             "UPDATE merchant_donau_keys SET"
             " keys_json=$1,"
             " first_retry=$2"
             " WHERE"
             " donau_url=$3;");
    qs = GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                             "update_donau_keys",
                                             params);
    if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
      qs = GNUNET_PQ_eval_prepared_non_select (pg->conn,
                                               "insert_donau_keys",
                                               params);
  }

  json_decref (jkeys);
  return qs;
}