/*
 * This file is part of LibEuFin.
 * Copyright (C) 2024 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

package tech.libeufin.nexus.cli

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.core.subcommands
import com.github.ajalt.clikt.parameters.arguments.argument
import com.github.ajalt.clikt.parameters.arguments.convert
import com.github.ajalt.clikt.parameters.groups.provideDelegate
import com.github.ajalt.clikt.parameters.options.convert
import com.github.ajalt.clikt.parameters.options.default
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.types.enum
import kotlinx.coroutines.delay
import tech.libeufin.common.*
import tech.libeufin.nexus.*
import tech.libeufin.nexus.ebics.*
import java.time.Instant

class Wss: CliktCommand("Listen to EBICS instant notification over websocket") {
    private val common by CommonOption()
    private val ebicsLog by ebicsLogOption()

    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, nexusCgf ->
            val cfg = nexusCgf.ebics
            val (clientKeys, bankKeys) = expectFullKeys(cfg)
            val httpClient = httpClient()
            val client = EbicsClient(
                cfg,
                httpClient,
                db,
                EbicsLogger(ebicsLog),
                clientKeys,
                bankKeys
            )
            val wssNotifications = listenForNotification(client)
            if (wssNotifications != null) {
                while (true) {
                    val notifications = wssNotifications.receive()
                    logger.debug("{}", wssNotifications)
                }
            }
        }
    }
}

class FakeIncoming: CliktCommand("Genere a fake incoming payment") {
    private val common by CommonOption()
    private val amount by option(
        "--amount",
        help = "The amount to transfer, payto 'amount' parameter takes the precedence"
    ).convert { TalerAmount(it) }
    private val subject by option(
        "--subject",
        help = "The payment subject, payto 'message' parameter takes the precedence"
    )
    private val payto by argument(
        help = "The debited account IBAN payto URI"
    ).convert { Payto.parse(it).expectIban() }

    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, cfg ->
            val subject = requireNotNull(payto.message ?: subject) { "Missing subject" }
            val amount = requireNotNull(payto.amount ?: amount) { "Missing amount" }

            require(amount.currency == cfg.currency) {
                "Wrong currency: expected ${cfg.currency} got ${amount.currency}"
            }

            val bankId = run {
                val bytes = ByteArray(16).rand()
                Base32Crockford.encode(bytes)
            }
            
            ingestIncomingPayment(db, 
                IncomingPayment(
                    amount = amount,
                    debitPaytoUri = payto.toString(),
                    wireTransferSubject = subject,
                    executionTime = Instant.now(),
                    bankId = bankId
                ),
                cfg.accountType
            )
        }
    }
}

class TxCheck: CliktCommand("Check transaction semantic") {
    private val common by CommonOption()

    override fun run() = cliCmd(logger, common.log) {
        val nexusCgf = nexusConfig(common.config)
        val cfg = nexusCgf.ebics
        val (clientKeys, bankKeys) = expectFullKeys(cfg)
        val order = cfg.dialect.downloadDoc(OrderDoc.acknowledgement, false)
        val client = httpClient()
        val result = tech.libeufin.nexus.test.txCheck(client, cfg, clientKeys, bankKeys, order, cfg.dialect.directDebit())
        println("$result")
    }
}

enum class ListKind {
    incoming,
    outgoing,
    initiated;

    fun description(): String = when (this) {
        incoming -> "Incoming transactions"
        outgoing -> "Outgoing transactions"
        initiated -> "Initiated transactions"
    }
}

class EbicsDownload: CliktCommand("Perform EBICS requests", name = "ebics-btd") {
    private val common by CommonOption()
    private val type by option().default("BTD")
    private val name by option()
    private val scope by option()
    private val messageName by option()
    private val messageVersion by option()
    private val container by option()
    private val option by option()
    private val ebicsLog by ebicsLogOption()
    private val pinnedStart by option(
        help = "Constant YYYY-MM-DD date for the earliest document" +
                " to download (only consumed in --transient mode).  The" +
                " latest document is always until the current time."
    )
    private val dryRun by option().flag()

    class DryRun: Exception()

    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, nexusCgf ->
            val cfg = nexusCgf.ebics
            val (clientKeys, bankKeys) = expectFullKeys(cfg)
            val pinnedStartVal = pinnedStart
            val pinnedStartArg = if (pinnedStartVal != null) {
                logger.debug("Pinning start date to: $pinnedStartVal")
                dateToInstant(pinnedStartVal)
            } else null
            val client = EbicsClient(
                cfg,
                httpClient(),
                db,
                EbicsLogger(ebicsLog),
                clientKeys,
                bankKeys
            )
            try {
                client.download(
                    EbicsOrder.V3(type, name, scope, messageName, messageVersion, container, option),
                    pinnedStartArg,
                    null
                ) { stream ->
                    if (container == "ZIP") {
                        stream.unzipEach { fileName, xmlContent ->
                            println(fileName)
                            println(xmlContent.readBytes().toString(Charsets.UTF_8))
                        }
                    } else {
                        println(stream.readBytes().toString(Charsets.UTF_8))
                    }
                    if (dryRun) throw DryRun()
                }
            } catch (e: DryRun) {
                // We throw DryRun to not consume files while testing
            }
        }
    }
}

class ListCmd: CliktCommand("List nexus transactions", name = "list") {
    private val common by CommonOption()
    private val kind: ListKind by argument(
        help = "Which list to print",
        helpTags = ListKind.entries.associate { Pair(it.name, it.description()) }
    ).enum<ListKind>()

    override fun run() = cliCmd(logger, common.log) {
        nexusConfig(common.config).withDb { db, cfg ->
            fun fmtPayto(payto: String?): String {
                if (payto == null) return ""
                try {
                    val parsed = Payto.parse(payto).expectIban()
                    return buildString {
                        append(parsed.iban.toString())
                        if (parsed.bic != null) append(" ${parsed.bic}")
                        if (parsed.receiverName != null) append(" ${parsed.receiverName}")
                    }
                } catch (e: Exception) {
                    return payto
                }
            }
            val (columnNames, rows) = when (kind) {
                ListKind.incoming -> {
                    val txs = db.payment.metadataIncoming()
                    Pair(
                        listOf(
                            "transaction", "id", "talerable", "debtor", "subject"
                        ),
                        txs.map {
                            listOf(
                                "${it.date} ${it.amount}",
                                it.id.toString(),
                                it.talerable?.toString() ?: "",
                                fmtPayto(it.debtor),
                                it.subject
                            )
                        }
                    )
                }
                ListKind.outgoing -> {
                    val txs = db.payment.metadataOutgoing()
                    Pair(
                        listOf(
                            "transaction", "id", "creditor", "wtid", "exchange URL", "subject"
                        ),
                        txs.map {
                            listOf(
                                "${it.date} ${it.amount}",
                                it.id,
                                fmtPayto(it.creditor),
                                it.wtid?.toString() ?: "",
                                it.exchangeBaseUrl ?: "",
                                it.subject ?: "",
                            )
                        }
                    )
                }
                ListKind.initiated -> {
                    val txs = db.payment.metadataInitiated()
                    Pair(
                        listOf(
                            "transaction", "id", "submission", "creditor", "status", "subject"
                        ),
                        txs.map {
                            listOf(
                                "${it.date} ${it.amount}",
                                it.id,
                                "${it.submissionTime} ${it.submissionCounter}",
                                fmtPayto(it.creditor),
                                "${it.status} ${it.msg ?: ""}".trim(),
                                it.subject
                            )
                        }
                    )
                }
            }
            printTable(columnNames, rows)
        }
    } 
}

class TestingCmd : CliktCommand("Testing helper commands", name = "testing") {
    init {
        subcommands(FakeIncoming(), ListCmd(), EbicsDownload(), TxCheck(), Wss())
    }

    override fun run() = Unit
}