#!/usr/bin/python

import pprint
import sys
import os.path
import re
import optparse
import glob

solfege_modules = ('mpd', 'soundcard', 'src')
deps = {}

class ModuleInfo(object):
    def __init__(self, fn):
        assert os.path.isfile(fn)
        head, tail = os.path.split(fn)
        m, e = os.path.splitext(tail)
        assert os.path.isdir(head)
        assert e == '.py'
        self.m_modulename = "%s.%s" % (head, m)
        self.m_location = head
        assert "/" not in self.m_location
        self.m_filename = fn
        self.m_usage = []

re2 = re.compile("^(?P<imp>import)\s+(?P<modulelist>((\w[\.\w]+,\s*)*)\w[\.\w]+)$")
re3 = re.compile("^(?P<imp>import)\s+(?P<module>\w[\.\w]+)\s+as\s+(?P<asmodule>\w+)")
re4 = re.compile("^from\s+(?P<module>\w[\.\w]+)\s+import\s+((?P<modulelist>((\w[\.\w]+,\s*)*)\w[\.\w]+)|\*)$")
re5 = re.compile("^from\s+(?P<module>\w[\.\w]+)\s+import\s+(?P<imodule>\w[\.\w]+)\s+as\s+(?P<asmodule>\w[\.\w]+)$")

def test_re():
    m = re2.match("import app")
    assert m.group('imp') == 'import'
    assert m.group('modulelist') == 'app'
    m = re2.match("import app, abstract")
    assert m.group('modulelist') == "app, abstract"
    m = re2.match("import app, abstract \\")
    assert m is None
    m = re2.match("import mpd.interval")
    assert m.group('modulelist') == "mpd.interval"
    m = re2.match("import app, mpd.interval")
    assert m.group('modulelist') == "app, mpd.interval"
    m = re2.match("import app, .interval")
    assert m is None
    m = re3.match("import abc as DE")
    assert m.group('module') == 'abc'
    assert m.group('asmodule') == 'DE'
    m = re4.match("from gtk import Label, Widget")
    assert m.group('module') == 'gtk'
    m = re4.match("from os.path import stat")
    assert m.group('module') == 'os.path'
    m = re4.match("from os.path import *")
    assert m.group('module') == 'os.path'
    m = re5.match("from os.path import isfile as asfile")
    assert m.group('module') == 'os.path'
    assert m.group('imodule') == 'isfile'
    assert m.group('asmodule') == 'asfile'


def name_of_imported(info, importname):
    """
    """
    if "." not in importname:
        return "%s.%s" % (info.m_location, importname)
    elif importname.count(".") == 1:
        package, modulename = importname.split(".")
        assert package in solfege_modules
        return importname
    else:
        raise Exception("Only one level of module names are allowed.")


def test_name_of_imported():
    info = ModuleInfo("src/app.py")
    assert name_of_imported(info, "abstract") == "src.abstract"

def usage_of_module(info):
    def is_local(modulename, info):
        """
        Return True if the module exist in the directory of the module
        importing it.
        modulename - the name of the module, as imported. Example: mpd.interval
                     or abstract
        info -       the ModuleInfo object of the importing module.
        """
        if "." not in modulename:
            if os.path.isfile(os.path.join(info.m_location, modulename) + ".py"):
                return True
            return False
        v = modulename.split(".")
        if v[0] in solfege_modules:
            return True
        return False

    f = open(info.m_filename, 'rU')
    for line in f.readlines():
        line = line.strip("\n")
        if line.startswith('import '):
            m = re2.match(line)
            if m:
                for module in re.split(",\s*", m.group('modulelist')):
                    module = module.strip()
                    if is_local(module, info):
                        deps[info.m_modulename].m_usage.append(name_of_imported(info, module))
                    continue
                continue
            m = re3.match(line)
            if m:
                if is_local(module, info):
                    deps[info.m_modulename].m_usage.append(name_of_imported(info, m.group('module')))
                continue
            else:
                print "** ", info.m_modulename, "line:'%s'" % line
                raise "Shit"
        elif line.startswith('from '):
            m = re4.match(line)
            if m:
                if is_local(m.group('module'), info):
                    deps[info.m_modulename].m_usage.append(name_of_imported(info, m.group('module')))
                continue
            m = re5.match(line)
            if m:
                if is_local(m.group('module'), info):
                    deps[info.m_modulename].m_usage.append(name_of_imported(info, m.group('module')))
                continue
            else:
                print "** ", info.m_modulename, "'%s'" % line
                raise "Shit3"

def do_file(fn):
    global deps
    info = ModuleInfo(fn)
    assert info.m_modulename not in deps
    deps[info.m_modulename] = info
    usage_of_module(info)


opt_parser = optparse.OptionParser(description="""
We use this to create a graph showing how the modules in Solfege
import each other. Typical usage is:
./tools/create_depgraph.py -a | ./tools/depgraph2dot.py | dot -T png -o classhier.png
""")
opt_parser.add_option('-t', action='store_true', dest='run_testsuite',
                      help="Run small test suite and exit.")
opt_parser.add_option('-o', dest='outfile',
                      help="Save to OUTFILE instead of STDOUT")
opt_parser.add_option('-a', action='store_true', dest='all_files',
                      help="Scan all source files")
opt_parser.add_option('-s', action='store_true', dest='simplify')

options, args = opt_parser.parse_args()
if options.run_testsuite:
    test_re()
    test_name_of_imported()
    sys.exit()

if options.all_files:
    v = glob.glob("mpd/*.py") + glob.glob("soundcard/*.py") + glob.glob("src/*.py")
else:
    v = []
for fn in args + v:
    do_file(fn)

exmods = (
    'chordvoicing',
    'chord',
    'compareintervals',
    'dictation',
    'example',
    'elembuilder',
    'harmonicinterval',
    'harmonicprogressiondictation',
    'idbyname',
    'idproperty',
    'idtone',
    'identifybpm',
    'melodicinterval',
    'nameinterval',
    'singinterval',
    'twelvetone',
    'singchord',
    'rhythmtapping2',
    'rhythmtapping',
    'rhythm',
    'singanswer',
    'tuner',
    )

def replace_modules(to_module, remove_modules):
    """
    to_module is the name of the a module
    remove_modules is a list of names
    """
    for k in deps.keys():
        for ex in remove_modules:
            if k == ex:
                for dep in deps[k].m_usage:
                    if dep not in deps[to_module].m_usage:
                        deps[to_module].m_usage.append(dep)
    for ex in remove_modules:
        del deps[ex]
    for k in deps.keys():
        for ex in remove_modules:
            if ex in deps[k].m_usage:
                deps[k].m_usage.remove(ex)
                if to_module not in deps[k].m_usage:
                    deps[k].m_usage.append(to_module)


def remove_module(modulename):
    for k in deps.keys():
        if k in deps[k].m_usage:
            deps[k].m_usage.remove(k)
    if modulename in deps:
        del deps[modulename]


if options.simplify:
    replace_modules("src.abstract", ["src.%s" % x for x in exmods])
    replace_modules("src.gu", ["src.specialwidgets",
                               "src.multipleintervalconfigwidget",
                               "src.instrumentselector",
                               "src.notenamespinbutton",
                               "src.statisticsviewer",
    ])
    remove_module("src.lessonfile_editor_main")

d = {'depgraph': {}, 'types': {}}
for info in deps.values():
    nd = {}
    for x in info.m_usage:
        nd[x] = 1
    d['depgraph'][info.m_modulename] = nd
    d['types'][info.m_modulename] = 1

if options.outfile:
    f = open(options.outfile, "w")
    pprint.pprint(d, f)
    f.close()
else:
    pprint.pprint(d)
