/* shake128-test.c

   Copyright (C) 2017 Daiki Ueno
   Copyright (C) 2017 Red Hat, Inc.

   This file is part of GNU Nettle.

   GNU Nettle is free software: you can redistribute it and/or
   modify it under the terms of either:

     * the GNU Lesser General Public License as published by the Free
       Software Foundation; either version 3 of the License, or (at your
       option) any later version.

   or

     * the GNU General Public License as published by the Free
       Software Foundation; either version 2 of the License, or (at your
       option) any later version.

   or both in parallel, as here.

   GNU Nettle is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received copies of the GNU General Public License and
   the GNU Lesser General Public License along with this program.  If
   not, see http://www.gnu.org/licenses/.
*/

#include "testutils.h"

#include "sha3.h"

const struct nettle_xof nettle_shake128 =
  {
   "shake128",
   sizeof(struct sha3_ctx),
   SHA3_128_BLOCK_SIZE,
   (nettle_hash_init_func *) sha3_init,
   (nettle_hash_update_func *) sha3_128_update,
   (nettle_output_func *) sha3_128_shake,
   (nettle_output_func *) sha3_128_shake_output,
  };

void
test_main(void)
{
  /* Extracted from ShortMsgKAT_SHAKE128.txt. */
  test_xof (&nettle_shake128, /* 0 octets */
             SHEX(""),
             SHEX("7F9C2BA4E88F827D616045507605853ED73B8093F6EFBC88EB1A6EACFA66EF263CB1EEA988004B93103CFB0AEEFD2A686E01FA4A58E8A3639CA8A1E3F9AE57E235B8CC873C23DC62B8D260169AFA2F75AB916A58D974918835D25E6A435085B2BADFD6DFAAC359A5EFBB7BCC4B59D538DF9A04302E10C8BC1CBF1A0B3A5120EA17CDA7CFAD765F5623474D368CCCA8AF0007CD9F5E4C849F167A580B14AABDEFAEE7EEF47CB0FCA9767BE1FDA69419DFB927E9DF07348B196691ABAEB580B32DEF58538B8D23F87732EA63B02B4FA0F4873360E2841928CD60DD4CEE8CC0D4C922A96188D032675C8AC850933C7AFF1533B94C834ADBB69C6115BAD4692D8619F90B0CDF8A7B9C264029AC185B70B83F2801F2F4B3F70C593EA3AEEB613A7F1B1DE33FD75081F592305F2E4526EDC09631B10958F464D889F31BA010250FDA7F1368EC2967FC84EF2AE9AFF268E0B1700AFFC6820B523A3D917135F2DFF2EE06BFE72B3124721D4A26C04E53A75E30E73A7A9C4A95D91C55D495E9F51DD0B5E9D83C6D5E8CE803AA62B8D654DB53D09B8DCFF273CDFEB573FAD8BCD45578BEC2E770D01EFDE86E721A3F7C6CCE275DABE6E2143F1AF18DA7EFDDC4C7B70B5E345DB93CC936BEA323491CCB38A388F546A9FF00DD4E1300B9B2153D2041D205B443E41B45A653F2A5C4492C1ADD544512DDA2529833462B71A41A45BE97290B6F"));
  test_xof (&nettle_shake128, /* 1 octets */
             SHEX("CC"),
             SHEX("4DD4B0004A7D9E613A0F488B4846F804015F0F8CCDBA5F7C16810BBC5A1C6FB254EFC81969C5EB49E682BABAE02238A31FD2708E418D7B754E21E4B75B65E7D39B5B42D739066E7C63595DAF26C3A6A2F7001EE636C7CB2A6C69B1EC7314A21FF24833EAB61258327517B684928C7444380A6EACD60A6E9400DA37A61050E4CD1FBDD05DDE0901EA2F3F67567F7C9BF7AA53590F29C94CB4226E77C68E1600E4765BEA40B3644B4D1E93EDA6FB0380377C12D5BB9DF4728099E88B55D820C7F827034D809E756831A334C078FC28ACB76B5ADB3BFF6DD659CAA49CC34F726880F293BD3FC132027AE7602242064EFEC4D9D656E069D4DFAE0B40C1D6B6CDB21D89D00E168B0B74D72EBB3B672B57AF3E99C85DA2F41CE70672CD0E0521678FC56EAB6314A0B3AF8B724376C01433D84943A73AF703D293634BC24322992756EE261FFF0D71BFFB8AEBF1026A6A345F2EAED505BC7E02498A3225FC91499DD5F5E30E386557C5FE0A88BC2337C80D7EA42B60622960230577CE800CB63594F619B7DE31E026429B7648C5835AFC00559FA4C7695D6DD9F7B2537A265E9AF7A2C986F8B60E7DC6EB3C4D805A6EEFB6FBB5BFDE21ED7E41CFDBEB02B0BAB76F9998BA1E52815A246B084EFAE7960AFFC2BA5C647E7CC05EF8120568432DFDE1D7246473304808985600A1AFC20B99185AF25E89DC2EC6F4880DC79BAD50DFFCC9EA"));
  test_xof (&nettle_shake128, /* 2 octets */
             SHEX("41FB"),
             SHEX("09C9652BB968996A35E4814E27587131F53FD01AB9FE83758ACEB8134FCECA24C84F592CEE43A4476E8853FCAB7DAFEF7B60ECFEBFD70DFCF587B3AF358A286FE3713BF4735A84975BB65E3586C81EA716BFB999626DC973A495A6E0024061387D628E9E59DFD2B39C68C8CEAD665AB43F6D2625A10630761DFB60276EA97B280442462246C6D74A1960A8419A76A37B68449A9E427D6A7EC1FBDF4760847AD6F6F5A08CEFB767CAEB6C2382F4F3D0E49DE4428CD4240635C9136911A82FF0B9C74569A1B7C8AF72AB1EA5F2F6F6A45E3BB08229ADDFA916B18A74F7939C5130152AC8343A10694154FDC6E1570EC7ECABBB01EDDC92EF0BB1B3DB914C74CCE399ACC9B766FD7494B2EF27AC57B80D52535942D55E2DBFAA61CDF3F48759AA612DED11421855AD15FFAB91462A56F873BBAF4FE88457A47B6C0594818D0A9189895239C1429ED8754EEE5498F4D0FB6C9D0DF0EB5316289E72C6AAEB8C61317B409156D4221CE6CFC7C5F39272D87C2D884F88F1B8B3C05CA9E235ED92C7DD7806CDADA7166CC1B9107DA5E6536D4FF111BF9199D6B72AC17D874323D68D76AEC4650F1A4B067C50215362201A7F71116BF6633AF08D712804B83F08A5DC7CCD4315963106D50453D44EFF59C9C652F4A924BE93C0B958EA286B0A4B597899A28C9BD5419C042668AA7B0CFCAC4CDF9260F2824ABF3EE79FEF53EBE3C36DF831"));
  test_xof (&nettle_shake128, /* 3 octets */
             SHEX("1F877C"),
             SHEX("E2D3144669AB578347FCCA0B572783A269A8CF9ADDA4D87782053D80D5F0FDD27835CF883036E536CE76FEF689A5E7BD646A7FB7D74F090193B2390E614759B7EB7DE915A38328745890B1EF1E7AED78168E996D7AC774D47F8F118B3E00A7BD151131BA3705AE81B57FB7CBFFE114E2F4C3CA152B8874FB906E862840624E02BBF9502E46D8888433A38E82E04CAACB60019222D433E8F2E758BD41AAB395BF83611FD0C3F7FD5173306182449B9A22C4013F2263B41EAC4D0EDA16854961FBAA6AD04A89E72A602AC59659EC2A60C1D020BACC74A711D4254A2ECC5F8F0627B4F72AE130C50590F8B91C52957B795D12DA09BDD40D41E3CD48E30E37FE5FD0B86EB38AB4661085EF515E70231A5F11F9DBF7433763B17D84F36BF42FB6D57789C6DA2BE8D205EAD87BDCC3FA6AC85FCC3E70571EBB2EE4BBBFC04E71B3A88E000464608475C94C7D7EC1BBFCEC980B166096E9BDDE9192C53E350D3B92F3AB0FCEF4E49E05BBBC18B11ECA942C8A0776ED4F293B7FC9B8E7C0303A29A41DE64BFEBD2967875290D471640A914B0775CDFF140B2B66CB099A05F5357C9D5E7F9F1D78B828CCD23FC7CD58329613DE4BE9869B4F2DB18F5792795C34C43D23E2CC5525B4A363A9619AD57E42C36EC459FFAE56B61717EF36AFC9FBE5A72A7D23D435B2EF38A2B0E22AB5DFC8BB8747B6EBF17A63F6368795C6A16DEE05362422"));
  test_xof (&nettle_shake128, /* 4 octets */
             SHEX("C1ECFDFC"),
             SHEX("B5EB987E7CBFC7C9D140AFD20B500E30F2F71188BCE885951F22FBC35DE40E7411311BC8479F895A069600483751295F5C3855474D65436B347608583125A6BD41CA30DC99CB342B72A96F8A2213E98A16BBB9E38A141B4FBA68C1A723D1D578A91B4A1BAFD03B25BD3CFB6D4E1E044637889C6D7AF0209DBB5E10837D5C5991D2766012E7E86A82838B455B96D882B7E25C9072A66DA22B0ACB992FD3CCF5FBC28625B896BDF8D4B7358901D12698FD823FE09AFB4F238631EE1277752F2900E38C7FC163381A01601DBDE8CD30A9A8EE8353DEF6C610B50962D0EC1F4F3EEC2AFD7FCD5296E299C23005960C1AABF3408EDE96DE18E10FDE99774B5BD33092639D8BE9D872130C96311EB6DCA82CC28A62C01E3F9C454251D87FA890284C06187F904CF23CCC62690628C590BC8B84EEE68371D3412DCB5648A634F2359951CD011028E61C28A33EC4E31550C1C0E4123D1091E22D56BD587E73B7DD43C1B431D9547398C5E7EB6C682EA2FD758C86C1222C0D6AB236B8106DAC19338E86CB8F6EC0FB607D70C938CAC172C80079B018BB62939546505B8B6E146A3AD7A35F20D7F9A353F1E6535C23AE93CE8F78C045402E707A664F5918E6836D343240E6112EFA29DFC4D18778C91E2B8530E4FF6E4947623D31DAFAC9B0D194E36C44E10B1A846935684F89ADB44782E8FFF2E3E62B65D18140F10D36CD817F835EB0C0"));
  test_xof (&nettle_shake128, /* 5 octets */
             SHEX("21F134AC57"),
             SHEX("7BFBB40DA3704A558291B39E1E56ED9F6F56AE783270AB02A202060C9173FBB0B45575B32348A6ED2C927A39A30DA0A2BBC1807497AD50F27A107754AF62762C0B9E8E62438101DECA378899C521CB76C75D299EDE19BA70E2AB91DA9E39EBDC917355B6FE28EE254C0CD48F3A28FB51766FD58154C98A5C650A4E4841D14E2C8F5BAEBF5AAF2A2120279D2D33214A51CA30C4E05209BBD60106CC343EC867EFFC826CB32A2E736C4AA561FFAECB46FFB74D1B3FD36242244312067C7CEA2EBBCF988E17742C3C80E8DEE6D509997866A308C101214F860512D38E3B6C307C1CE779FD42FDCB9D909A2D6F9DD5A3687708905390479950D55E8890D10D7057863CF006034A6D01D77FB33C0E28B093FE3337BE8416202DC866796739271C9E326199092FA27556C9F40D7E9BAC869B6D7E8018409EA92AF2464C58245E89556DA3E80B6B4200B9E53FC80C93149E3DC4921A5FBE7DF67F1D0B753CBA41B75EE5CBA78A89A2A85A46BBFAA625A40EE62616E68B07C06339717C81B4343760E97E2676C347D9625ECE383FBB8448E7837DC704E2956B18CC506E3BA1416C936BFD66E9AED3A2C7B0A8DEDADB04AB711D147F1C1C83322EC1E6F1468AD770663C7938D4032758462513652498CF124B6357B4F7C7E4DB0D0C0C1B7247875CB6685E0567A8AB8CFBE9383DAEAA27D224CA9C69C07AFA929FFC09A03E165D72153AD2"));
  test_xof (&nettle_shake128, /* 6 octets */
             SHEX("C6F50BB74E29"),
             SHEX("8D876C67A5CC15A544EB81E93B6FC15713FD1C55D778204EB6AF7086AF3FE97D12B4DD22C7AAD54EB9C9E391B90A7C55ABACC3C81DD1788895F17033D561060E2B9CFFD47974F196C6907907DFDC65CDF33BAAFA7049635A70353D92A56A59ECF0CD291AB9B509D296356E8B0C91983D158DB1568681DE035E56B86EAE32AD3E0193CB816BD2D111C266632DDF907BB276E14880571506E2C328A3EEAA9744210EBFEAD123E241A8C49C92ADA79B718D970F0E8525A60FFE9074E73491E6F0EED1E047EB95CB52C47F6A1BAB77017D020F016B64798E5D7FC360D5FA213D44E5B75C918C9ECBB134CE1DBE13C419DA62326A550AD4A22E171CE75A369BB820CD75FDE01993CA131F302BE7941F7950A9969D78148FE7C45BBD22FE83AD0354194859F2F0DDA1C92F2C8EDFD3F7A9D8F618D0D02222442DF41D858CBB7A46E5B12A0923DCB53587957EE07ED3E1C4A46EB1141CEE394972726DD003239ECFE8FDDC4E58C50359AF3F89FE58F5CD5AEBC0DC6B9D600F3ED08506256F18695DDC1E3B459AD75090F0764FA89E5CCEC98EE0B387B5FB778620BDA72CB3E2AE50DE8B0D6B43F26E13A16FDF08BEA752B457BF486CE427FD17BD4C5890A89AF2C85B292C51CEC75490C1C6DE9D60DC49D3A77EEB7E19002150794234194115BC685E9981F88B0E1D9C6585F2C5F9DC805FF6C7636C1E06D9F5C018F74A2CE92EFF6B26"));
  test_xof (&nettle_shake128, /* 7 octets */
             SHEX("119713CC83EEEF"),
             SHEX("E2A9537BAC3C4DFC9008C1A7ABA653883D7A1DF35685DBF49ABE5A7E93BF044BC3312A5E4D9743D72BD28ACC16F64AC5090A71761D936FB9DA7C782AF9BC1F636D0E17CB41C7E0E9DFBDB2017ECABA6DBECDCE2AECCE3ED4F59324E74D58D434096356E567B35AC85F7CA9AB80B1C987CE70F998ABE6536FE485A866A22CDCC37DB08CC742B4612121CF34C2D404B37E8EA8D90CA9CFD0C8C6ECB6B44BF73F4D048A0FD85591D8726BE6246DF406472CA05D18DFFEBEAB7ED991BE6CD2940EC9B26F9C9219DFFFCD721F5DD93CF5342A89A6C69168871B14934D4F1794E14B92EA991A78292423F5CD0A5D3454E55DEE27077D6AF78D87BB3FB9E139331ABF326FDE2FE1D3B1533AA038BF59BDC2D6333F968611A4F9BA66EFE92AE8C099ECFC1A36BE726816FAFCC0B016C58B1E42D2CC7065E71ABBD7CEC1ED547AEBF46F5DD72BD472CD764CCB3C4AFACA540BDC27F52F63151E9FD90AD5BBA7F84C9EC7E996692B842FB1A54D2860A301A9DF6F357C2E05C14154F5FB77A86E4DA9E9A039CDDEB10D6E5DE2E0A585B2701D48E8298BF71F5F72351F8EB3FBA5426286A82ABB75EA46BFC0F65A2E49567CFC970C36D63B2CC73AD7F9791179B1A0C4D84651DCE57B966410CA23CFCB9BF5BD70D56F5BF1A7C939DE74924EDBFCC614113C83803B123F3EEFCE2C809A7A9FB01C709BC09B5F7632D6B59E31A6469D90185477"));
  test_xof (&nettle_shake128, /* 8 octets */
             SHEX("4A4F202484512526"),
             SHEX("977735A853D872FF7C17C4A825AFD9DF886AC43387DF245C37A608AC7F4E0ED015811167222000AEE1968960174EE5A39369A23C5DFFB4991AD247B8801DE71BEA97069EA77202999F8E8CF3829816EE598B00C4049265DFBB2B138B13EC3194B988242BC099248BAF9997AED80E95B5F859D42B12DBD578FDEAE47CCC2F8D3E90BF6E8D98AFBE2F4813F68B6FBC4C18C9B8A557A0D87D744A4238D89260941030945538B2DF07E05FD45B50BD790A0510ED4430ADB3B7768DF8C4914E0914FB00DA09331F11371052D301031853C7F3C32528C0622A77C48463CD1CD96ECF7490F70F16E941CBFADE71DD8477ECFF1177B7193E456ECC42BEFBE070667D6F39ABEE5CBA354B33A83602C1C17280D2DBD0CE597ECF1FEC3B27810DF38C2805D8B85D60994EDD2C83F5898CC6623241F16C4C92444FB6A0714D8BF189AA5AEC9D5BF1448805764EA0CCC8B6E399961DFA7CB9D8DE8D800080EEB5D010BCACAC6728E8DE482C37270459DCBC80F496267377D97817149372A053B2D53209C2DD61216CC3AAD29C7238D6B142D71A92CEEE4710476C2A48FADB683B9423727CE772FCE2BDBCF781C119FB43526B8EAAF1D10F21E586952227E29BAE61FA2C7EDC6260F76AB543244E538180CD90C207330EF29CEA987F7ACDFA028A78D3E93F11EA159B21BF3F50FAEB7961874E816162D42735C9D3567AFA45D1D8B66CEFB58678"));
  test_xof (&nettle_shake128, /* 9 octets */
             SHEX("1F66AB4185ED9B6375"),
             SHEX("1D96E0454BE037D890FBEB776CE5C922BDFB827AF1D1ABEE1BC8B3AE95453532D40C0B3F9C93E400FF70F6FBD4DD0470DF1F06E04AB4A596B6F92BBCAAD3650D60B940096260226913435C9BBF5F29DE8BCC9721748E3F1DEB7FAE88741084507E1BA72A267C2C2276D8BD8ED284578B46AE81230F383214137CD7C190524818576F77B0A53C3421E0476D91E11BAE79ACC4995A24FD1398F6F67E22033BC381ABE6DE4E0563238FDC39A1B305FA9403EE35800432999EB3421C8CC17F77F2A544957AFC2CE9496D2B739511B3CC52DB118EE874E43E95FA69D7EF024931377AF86A9448F3899DD0A87EDCFECFC1BCF7C17D470C71CC488744F2B627626112AEE465ADDE9C6719AC0F6E928C77B0BA0AE8A5CE20005C46768021438B611826675E0B94FDC7484A3854936216E2239650C794EF001EF8F9F4CB440664472445875D06C519E14C957D6FCFB73FB72B46CFAC4548BA94B95979FE5FEAC07035E0FDD21AF7E3B69228AFB2D97F2A333A5D40A84399C6D4F5A7EBEF9ACD08D66C2781EB9085A42E38C434230816B90DD838A2F10E727EE5F58D7469B47FD5324EF53A88208C44943009AE7A2803D55760A6AD4BC6E961C8BCC7F7F5BD07DA63709D85832E26772E01886B2D5B73CA488B8402C6A15D2AEC605D154F9A41EFA93601D26374D3FE8A5337E98BB7A1B5F0071D3C8BCB69CBD0B3683E7A494808C9BC4490"));
  test_xof (&nettle_shake128, /* 10 octets */
             SHEX("EED7422227613B6F53C9"),
             SHEX("0C5DB5EB4C5C83FDCEFE9EF623596F96D2078C6DCC5D9B5E4F1EB5ABBEA23ED4478895D9D9C8FDE4733933986C6CB4D03AAC65DC7BD628169785CFC5D9D4D33ECC7C6AC98312F020B117C831B2ACE85E1ADBE0A5349BBE9FECBC6E998C6DA1E2F333C105362506766D105F707D905CB616D8BD4B23ED5072E3F4D42412E3ED6C894CD37D8F9A46BE6EF9A53D60254F4E9627625A3671B234664740C24BDA608A44EDDA0CC652B843DF722AD38B9FD7F122C89D8DF0A68E03FBC0320BBA016A0F3C36FBA6278F9C2AF9BCFC9F312CE55F011255332505FCBAF0ADC512864D633CC0FB4D81BAE980E9522AEBB2A07199B238592E4379537E045CCDE8B0FC44B2781421DC657938851324F5FB6E1B7D9548A50CB6C02A71B16DACE90B84EF033594B2F0A688F6F9719A1008C1451353F8895AE1B7B16B27AC94B67F390D9586B7EE31AF7B6E7D110813A6A4E01D4C94F17A1EDB0633A1C4F69DE184BF4315DC98BD453FC22EA11F8EA5F5548C5DC09D3F186B40C4783AAA4BDE1A8F2AEED007A0BE440BE16649111921F779C7E284B8326785D87C3DF06A05B3373E88526D9E8AFDAFD8E673D1876711FF2F68ABADB7CAD4545F0DFF64BBA98C86D7422C4DFCFA8F2BF7D72FF0617BD076B4DE0B457F5FA2B881FCA5722864EBCA11602923BD0D6B21A9D7101ED1F34AD480F8F73E2ADF821B2B259D6DC535AA94D3CAD7158DF14E"));
  test_xof (&nettle_shake128, /* 11 octets */
             SHEX("EAEED5CDFFD89DECE455F1"),
             SHEX("055C0A829125F20A0DBAA6106DE95579BE4188363C7D7E3F3EFC40AFF360298897968B250B8410829CDDCE5EDC80B764A5AFF16FFF1C3D29614C6BC4C80D095F0C73443E8FE596C86C83AC1D9BA69847F6E64CA643EF3D7010C5F9944A660B14559FF27D4A5AA9862CFFB67B47810D04C736BF842450A7E5239C5D2C61087888EB892B9BA495EF7031FFE2A56BC4E8FE4CA92D908666091D40F8DBADB0186DB30AF31A4BCD3FAF962B56271A4C5C4A02C75F11D13F3DEDB0F57EE554B9AA47B19109D3CC2AC58D9C55653B42BF5CC3C8143481146BB54B6284077E94A6B2DAADCCDB3E85FDEBD9003C12FC62DFD07DCD9CE747893DC611EB71580F2AFFC624958E1C78C1554706A1582359F6962ED70FCF82FCCDDF9FBFAF0023FA1FE771D358A5D69E6F8917EC3425543557ABA96D40303C5034FC7A873CA12D485998A856DB5724009B884590C284B685DE5677C76944D07DB1FF855743587374D9F0AB70136AFEE1D1EDF962D1AE9D9DB036F28499ADE8386E78D2D54F8F54A85EF0D771750A5D51511439ECFAB2A486FA1CF77E8973E977BEC186ED45F6C49845E9E19502CCB3CFBC4F2576D9D30A003F8C00DC209EE68F147CF17BEB5E5E410388A0CEC4643A3BA24B6532CD73DB049B74E4EC5DB415586095378297A60C2EFB70674C473488385B47EA9E9D39952AE9A515E139B20D76A30E65F09B7D701B3172899643"));
  test_xof (&nettle_shake128, /* 12 octets */
             SHEX("5BE43C90F22902E4FE8ED2D3"),
             SHEX("9F8F1774FBEA8EC70D0661F8DCCA2953EBD20FD73BFD74E6BF037B4FBFEF17177F1A02E919FC2B40BEDCF640BE5F191D6572BC57C0F5EA29CDD413E8A97BC24BD490B221FA452594F78136C68B21984DAEA6B61899C386963EA6F078A9A903F05506D0C5592E341A798DC46C0198C039FA2913B7F7560CA6A703A6A86B18CD67ADB53A13295ADE0BA89522CF98263094B362C0635890CF4FD14D0CE5DEAE75049C5AF37EDBA5C0803B26EBE951996018D66390FBD37FF9F24FC5A7435513D3173E26E1850EF3AE729EFF8C38AA18A41B97E11A7AC452445D7EC617B8E0894153B067EB380D83209D05BC1B18B41272ECBEF29472EC90686256F0CBBF9F5089A78D853B50772E494BF27FC0DBD3492FB16B8AAD6D7AE027D41AEB633CFA3CA931CDACD968C1F24684267ABF45534F89A362C1195F8CB31F1026D3D554AF70DECA0C5FD3973F55F20218056FE70B74A3311AC394EFE4FC0130B5BB092B33BA3B91E2288815BAC931C5625297C2F3180DAE4B824409B435842AA12F7F01A185F699FDB2C6A9D42CB475B0493EB6759FD5809781560626FEF4A4522EE06A2D3FEE7CA2203EB84B717A1A825A2965D2C1C1CC0402658EC5D2448961F521AC318FA4A672F0F8051AE59C05394F84D25A1F9E27B21550A71EC285C8B2F86BBD8FC0DA09614BBB635F905AAE3D401E7969DDC9FE0837959A6749AA76FCD3A70920766662"));
  test_xof (&nettle_shake128, /* 13 octets */
             SHEX("A746273228122F381C3B46E4F1"),
             SHEX("E73D11EF660C0F9B35CE16C12C58051953C229AC29229FD38E00ABEA6FC7917427D30952DC52FF435A928F34BF14EA4EDF960C78A5008F4678727BDB4C591D47C0D3A1ABC314CC475E47CF68B243A0F193B5A0FE06613C2715829E672EAE605905E6F52624F971731AED8D1CC022CF166C7391E249626FFC578F2CC48F89784D7F38DD14012151834DDFDAA245922A27FDDA1465A8056C511A72A784095496FA0258007394C193ED705588607DF19A6452F4E5C890FE648339A597418BA2E7E706C36D3DC493A3A6842F39409CE186222AFB81EB044081C6145CCF8B0E852DC602A2B7031355911A7C1B16ED74E8BD6C4EC8AC56511768C008BA25343FE5CC7AC262C4662AC803EC1082A52A0530F0C8431F68EF9F7C4C7DFC9C05228E3638B7A26DCD9FD28331F7B04BEF3956D6C49FC946DC97E935A76B3B2B4DECDA6113A9DCAB916538520B28DEFB05DB55F2D168FD6FAF678C6ED335A33C5DA6B575FCCFB64FF2EFBC5BCF2500F006E2F3DFB4C60A2BABEC8240143C779A3D9753F107BCF9981206922757E04FC93083ED10A77D5C603EFFF8592AC3902E2D07B72A5C75C71B631DFDBF3997744FD63250D2EBBEC2DD3BB10E05FBB3A993831D62A6C03F403E30EA3B80F2C8A3B80473C2D2F2C51B8B5563E46A0A57A02C4E73DA0491312865E15C4251D3DB6516A51DCFEFB1167CEF7021E570FDCBF862BB5B81B461E2"));
  test_xof (&nettle_shake128, /* 14 octets */
             SHEX("3C5871CD619C69A63B540EB5A625"),
             SHEX("B28D5EF1E7CB85B4406FD3349EC1D5F3727A9FC9C7132EF6593779B0D3677512D5F10A6022DF96AEFFB4252159CCE5562008E3E6BF02DC91A1DA479B5B091C88B65C80F4E8D83279DFECCAE72DA723842EDCB2BA02C4E3E9325804A04366E7FADCEE8BB2D678E31928D59F2F0A546C742D8120694F5F02810D5BCDC320855164F60226865BDCEC1437F262D6D222D1CBA8577910847B75D5C4549BDDDCA32C3768F1B789CEF62F56993CB72338274E5740062BE3EC7AFFF53215E894600B35DDEEB1AA1E9962B22329FC41A5696580DCCE450DE6C11F92332A3FD959CA5B367B9EC156FAD0318191314E98C1E2F82FCEBAFCB6523693A0C633B82251AA5AB4801B1AFB004FB28DE7B790E65F50E0CC85181FD914C76D915C1E220A53070D73BDC1C1BF2D0D9A3EA45697C3538AFAB5F577E09483195EC8B7737236491143D7AB0F81228031CDCA4EBB2442AD5AF92CAAEC92CC1F72E32E573D57BB48EB58453B0E0D680B7A3D46B4A5716B7756F8F73BC876DE181C443C082022C7A3322059F83AA16FCDFF5E078AF2F85562F0D60632F4A76B29F27BD91301B276CC414AF7C7B0A20FE0D0540DBC2B7A4F6918B37BAC345411247F2F7ED5AEDD12E72ADE756DA4962C2E19B998AE1A29E0DAF6379A45ADFEF26AECC80D9ED1714582F7D40DE0C3BFADF4A35DA62ED4357B88C45227810D63F33BE90529F426B6AF513C1B43BA"));
  test_xof (&nettle_shake128, /* 15 octets */
             SHEX("FA22874BCC068879E8EF11A69F0722"),
             SHEX("D7DA38E9032FFCADF7CB93C9969171A0493525F8EEE3073DCC9AFC83C704843A8173EF5374C4D564BDF246D41AF6EDD0872E5EE43F5D2C5C85134166F9B2AB9203CFCA96A405F934F3B834299C954A3D30A2CC6737935DB8C094636D659D8A3192AC8925809567719FAF13AD40CCBE7FB1099BF4FCF415817B32CE923B1AD764B88E6FA24A9F0876C84D8B6FBB0346A87248A99E15C528D8EFB167F6F1F00099D67F5B879FE8B7943726F648158C0356985CC48D92424A44B906B14EB8217B5C75D88E50311007C1CE7BADBD9A48ECCD4F8936B81D0C2A76918A06E3D2B2936C5A157D9375058162259D7D0AD25DBA2490AE1BABF69176B394FE2F00DC000EDA5A0DA049153C077D51A89100DD5B8EF4EC3868F5C9D8E4C1875F292C4FD22C51ECFFE30AA40EB4FF50FA00A61F6116A337BED4D13B52E12F06BB4F8A88060B77B9F1300B5D895F7314AE728421CA951FE8DA4BE55723AE5533E59E06E0BC8D828603CEDDDFDFB9D82506141E1D90FFCD65403A94586C9259A6CB18550BD22B241112DD9D866F12ADF2B009740AD71630C35B16E062460C708831DF28918A5DAE14BAB1CA781EFFAFBD267081A7FCDB418417717756F34BD98A799B02FF2E275193B0C1B721C2696AD732AC77AF212DA45B22F27D54D1559A1CD7B39A49A6286D305DD4A257452E930D5F2C3CF16C6674ADF81BE1D8231DB6C1B8A9CCAFA6E98B"));
  test_xof (&nettle_shake128, /* 16 octets */
             SHEX("52A608AB21CCDD8A4457A57EDE782176"),
             SHEX("3A0FACA70C9D2B81D1064D429EA3B05AD27366F64985379DDD75BC73D6A8381045C2AE2E9C723462EE09EFBB1C2A8ED7A0729D0D9B20F03BBCF55A86859ECBE80C8CAB60BAB4C5D063DEA224E825E38642124EA705327E075B61D08E0B49DC184C5194292BB4A797CD24D924CC64816BF911FBF4985130A59D68FF0673CC8C4A390AD593BEBF16419FF464ECB3FC78C160B6DB9F755A5FAA7A9365B58CE7F90465AF960C48B771699E0EB227F5370387E6248E17EE192007128EE7AD3D94BB9A2193BBD4618AFB3A399CB2016ECD5F9E41AF10701FF1915A6E091F44F193B0F729CC4AF5FECF683B1C7DD2644D7458C45FFD635EEB85C79E241C1F4869CDA9E77E80F7B878C24E9AF77D22D8C7C0C406C8AAF50F57BAB68FC6C3A20274B6BC353E6D60DA40E8369139B4508DAE96DBA12DCA9D80A19041A3798B252FD24BF2BE64035CDA6D95D6E570EA868EB8808193B3792897A2147396A47D27C81D40FF4BF9212AB239D7A789D8CDD545A98B447F6ABBFF4BF6FE631CF2483881E933C1E62A21BEC503A6EA60F3B179F9CA6852DAABA4CED7ADE5E35E960463FA4C3A32F4C580F03CD2E45F10E32507FB2880827F56BFC5030A4CA94635EDB134580715A23C87D755FD91B1566D1A471F310EDB2C12AA11F2D280683F43155D67E2EC04EC2CB2CEE53A4D00F77AA73DCD6CAD61D1AB7C30A627CC75F83D48BDF9A76AB456"));
  test_xof (&nettle_shake128, /* 17 octets */
             SHEX("82E192E4043DDCD12ECF52969D0F807EED"),
             SHEX("3D62E50DE92BF40637D481CEDBD22F2DBABAEEC0E94D1F99F3EF2E68B1F55D97ADA13D2372104D63946E3EB7C50D7E58F5AA0D5730E4366BBCB6CD332AEF45DA6C4E5EFFFDA948E4E7E6FB9C49B46AC954CBD53A925C8EBF73BC4FD53ECF34F0D67026136E8CFF3C8B8661C9D5D19420509B47A19D4C848C83B791D1F843F8DF7F69F736A414BB9075B4BB6070FA094A0095F101FE569E45E4E7FDD2B3622B5B912C21BE28208CE90C7C342E6DE485D7344DCF3E89D9C71F320001C52054EAD0EC37B4CB356CCF80088DFD2EB63823210CA3767ABB13623C9BAEA9BE5ECB8DBB8D3DC97C88F6BA9C4A9C443A00E9A476FDAD93ADA9F64D12803DC7FE1C7126D23430CF680F94F0B218A06D5FB4F7698DCB5C896070A03D5DAEAD54E3624EA55E966EA64D0714AAF9E17FEF5DA0998F6A301B20E988579FD98831649B5F71B67CE86F65003EF5F160B18CA218D446C6A18F341282CC89E585894A2CDB24E107BCEBE66D3265895A622E51B617A57020B29E236DF78271B1B87C5FFEEEFBDF3F36EB3152687209929C800B9CB4A29CFE7AFF4302C990E60CF61ACDBABC3CD7F98BFAA55BD61C7D97B3B0A935E254D1D46801FB6A02567DE9F75343B612F95DC555340BEE70E61BA8B3E1D7478D77D52C5F395301178D42FC3DDDD1322CD31ECB8E236F1748260F45AF38A57AA8DD1796BDBC99AC14D1801BE95A08C1C6C61B70AE"));
  test_xof (&nettle_shake128, /* 18 octets */
             SHEX("75683DCB556140C522543BB6E9098B21A21E"),
             SHEX("D062C8F1218F1DC5F3CA5C2FF3F33847F3F74C6A40A3C2C6C15E316B6C9FB36A7C1DA9C25833D25C8DC7FCF4F852E3955B4C9246F4CCC3FB9BEC64990E5D84730D56E5E4839EA23B258BE3660014C49645A268EB03BFD932D879B2260BF4C6CC4EE97C8BDC9C02E609372D1FC24029193475992A27C52C20F1F821F285BFD1F4C6ED944EE89A96860364E44971E0B6AAECD446D868C9F7175C7C23E980ADAFA860E690EBA14B71F7EEF16DA6017BA3FAFCE89628F64662AF12BEA0DC93222FEFA733434A4778F0F2F5C0DA2439C5C577345B59771A4991A56EBCAC98961E7EF8AE3F1F8062C1268DE1E6CA5291D4D7F1C1C4652A49AEA180A5685F9E6C8BA01F8E3BE7C1E5D40A9A0CB8661FC77F624F373C90DA1876218A78BE6507C2E999A25FE87E5F334ED83689936AD06A7F031590AD13105DCDFDE7554D42F85BBA80A03D3C1459A25F07E93B77293B48E8D831EC371E268E361454CA01539357D4C10255E3DB0576CA7F9E6F4AE766822274A6669AB218474F9D0CB5B96FD0A55D4F454C6EE31E3A040568CF77F97D5B8C4FA783AA5625BD8EA951DEB4BEF186F37663AE83B2CD27276A390D730EFBDBB798049B151E4962B5B17C4739F9127B87CD5ACB00C4CF04295ACBBF53324C3C539BDED5E64FB4A3BC08076F52EB0A4BAB60138E50DCFBE48765047FF4468404820088485974F4C4FD8546538698EA3904795D"));
  test_xof (&nettle_shake128, /* 19 octets */
             SHEX("06E4EFE45035E61FAAF4287B4D8D1F12CA97E5"),
             SHEX("ABC182A9D28B3E043F887C072ACEA1E0E6C26A90BF6FE9E1BF925D20100C3F8E46DD5634CE077F6768450626105210BBD36789FEB64E9E0A0EB5007EAE54D2431156A6BAA49D85B31A716B69336DBCAD7BD43F0C9AFD4D62BBE69D5EA276F01B1B39C637BA13F9D5CFD637FEFA3D80706896F47A58D602166645AA5CAFBF5C280EAF765A47F4981A9673EDB27089F31314A6367BFB333E4A937CCC3E704218696F998A3269518965BEA095A934C171F78979D6CED938F8FB3D6CC8F61EDEABFDEA038DBF7159917E3F941C2D17E351390AB77678477B11D2DFE2351180E305025403FB826255DEF38AAF0ED06BC2AF707337DCDC1E5A08FE03E9CD043D5A152EF37FD4893266ABC1875DD28E2C7160700899D5F39DF5DEAC2E91131360922F2ABB6BFC49A8FC0B638CD13B6037F99BD0D72A56346804E146E8B2889AD843813DACB56FD8D9909A989C82064E4838D4F01C19A8E8D1B189EB060F642CA5CEF14A2F361C1F34D968FDCF37D7837E3AA9B866935095966109C763C8DD1CC70DF91FBE7B5AC7FC6433F5DB4B780BC9BE4EF862B596E08AD32C201A07F04E267EE2705245C48918B71C84D78BA48AF70A4776523D37A8B90D53AD45465DABA2B48CE0C09978FAB5F4A0CFA6174D55612B7102CEDB7A591EB6A778BD41CF3C289D06E86B4D329F71DC6C3BE75D7E661E7E9B5F4CD92225FB5991F73237E2520B9AA604"));
  test_xof (&nettle_shake128, /* 20 octets */
             SHEX("E26193989D06568FE688E75540AEA06747D9F851"),
             SHEX("BFCBFB531C1689126C2323F9294C1BEA280013A2315A596ADA34F022BB6D3D8A699BA8E2C54AE3BB7B32C422C727B41678F781D652DCBF1A40B8627B56E004AA5F1753B1152920121B1A2D8CD5E9CA02929BE7F2EB433BAAE2FA41830F146C72A12AAA97E8F8B863749B3026E99E0BB7E0A937F003604B470C913546E73413A86EE2E9892FCA2A7839B20309AFC1FC9A0EF14478CE0CCB5CCBE37469B7DAC3A7B74B253CCF2ED085F9A0BEF3759A4002E2E67F81D953E9BD53A4D39A267CC0D262E5D80F50F8B87D816C91A2C64D798784A5677744EE02B5DA2CF07D79282BF8C791433E0B3E296C38AD9E0162BAEDC89F1B368537103EF797AE54E68AABD23DE81B760689839341155930796AD7A49702113F737D0812E20B850C38C3767421177B39FB60F3FCD0B81A50A8624E57B676203BE7C8BDC66BE9ABB5A1AB175705DBF65BDF5D119F331205AA74E6E04C766E408D0B2651511CDC513C75C6E326220DD3DA4A12C82317455D450F44F5F1E55BB7A91BAC1C34C1E405BAF53FC8BCBFF7EA119CA53F6DDFCB5E4F2892B837846CEAFD8ABF5BBACF09A3127685B8B40A2452E52EA6C006F22EC7FD9DF925B939570096E5E0AE74D67E46E5402DC94D4A47E460D718C98A4426F3E68DEAE1A4E509707A20B136FCCF3C26C8956C14312BD855F0B3FC6A97CF1C7E9ADE4A4907A7CCFE6EB5830D17451378ACFEE3884E8A"));
  test_xof (&nettle_shake128, /* 21 octets */
             SHEX("D8DC8FDEFBDCE9D44E4CBAFE78447BAE3B5436102A"),
             SHEX("E3A05F6157BA90878E8C394E09753A69D271C77B233799E0D68A32209F0CDFCD5E3F41DC9109F836A4C45043E7EABF41946FD05922723B0E6526E5B99B10E89B3C4CA6CE5A2C6C431FB14E6CBC7986C4CA691037B1F306F8ECAA146D3E0B5736A53A2BE74E1704E65864C3ACF042459BBFBB440EC73EC884D89AEE7525CECBA651429F2CD092E4B3BF24AA144873EB870749564A52AE62FC86674B0D9FCEE67DE8910DFD812CA75A47F8FD0B2393E616FE8250899C36C27DBC30716C9F5FA181AC6A6065B73E3DDB94447E22EB858E8D1E12C24B80D99D4E6291A0D588508319C5B54E616270FCC504D5925B0ECB12F211C3B63286383498FA1A7B4B1BD399934DC6DF4B7A6BE0F42934052AAE7624EA4B6837E07EBA9EF2D066A911CF71776A2EEA41E04122E5AFA6F91865DAC64686212A93CCE21305D925606FAAFABF7C6D27A8A91C5BD24730E2EF0F11A9467F7661736123E7B8EC29B4ACABF9AED239EBF3C79F4C37F54DB912916EBA3A91AD6362DC616D7339AB457629BDD0FB97B8AB605B9FEFB65E2E70FEE3F2A29A3D6D487682BEE87D5C615870CB615ADF197AEC1E5EE35C242149DB2D5DCDD598DB5981A5EC57D31C9644CFBDDC03FD2F5F8881D2244A094FF4140D252E2505089337B925D84E3EF632C96888E0FE5C23B1D67A64FF3269A430DB1927CECE798ED159DAAA3B69819FED596EABA5B6BF4FE3E688"));
  test_xof (&nettle_shake128, /* 22 octets */
             SHEX("57085FD7E14216AB102D8317B0CB338A786D5FC32D8F"),
             SHEX("2D81F6711464AD6B3DE6F985E02D52C30DDB350BCAD92FC15DF3A67290E4D74E2EAC52EBF9479DC223770D6481B1A640784B6E4A31EA840692E625BB2D400A5C0C7966E5C4AE2A4CD0A3204D7DB266745D64122F83141FDDC676B57713E45CA04139D49BF42EDCDF443B5600668C1534953AA95FCF4BF9A8EE8B52680C9463E5CB5AAF09F18EEB2885A061DAEE4C3EE5DD6E824BDEF31A93E092236F91B8E2ADE094E718F8E203E43EF00A843A2548457D0EF2315FF91E5221011B874CD798254FB4A2E4C65405644E0DEAC6529EF59EE52F2496D9B19485E8128A76548C212D667637213B637919186AFAF827A1A0CB5F584F475723D7D2EEF4162E05F9ACD221C3416E7E88D22D4C471A05EB0DC70822E912942261E9369C0D35A672ACC5A48C86D6835ED99BC201A02CE40E86E0C096B2C34E5BCA207E5C5C90CF86BB3CCD7FB89613E53EC90245F9D8315D1102462CE10AAEE4608101AEDB37857DB22ACCD5B024ED323151D75223E6CB0EF402F12EE2905869396EF6AD09F6FCFC57310BBE46CA7817021DF43FBDCCD9D62158AFA8F20300F8FA17B202733438C1F6B0F1D1FB1FC8AD8414E864DEBBEBB5F96367135224581B9B8E54E64937F708A2CF07573A42C45444C37DDB1287D4D27CD3A9DBE476EAB7C39A37722BC4A1345A34C168DC93D48BF719E79F25FF112FD687B86C20C98533CA4B0FBC1F3D4A6CA469E4"));
  test_xof (&nettle_shake128, /* 23 octets */
             SHEX("A05404DF5DBB57697E2C16FA29DEFAC8AB3560D6126FA0"),
             SHEX("CFDDA60353118CB298A271246886E4C7F976465C1B65FB51E87F7E5A9BE050B38BA3EB23F52116F0A13AD802E12AF81A0626A51C17E9EDEF5CFD1937BD774B4E63B19BF1BC61A4ED23A2ACF45869E4EFE9CFCC8C89AF6861E4122515E2B5EA8DBF024F36EFEDB7E6D6DDA829A5E45E21466F4BFC55596748A9FE0A61B3B9B9AB2906008DC3B8BFB71761EEACCE68FED2F0F0B380661831E4754175AE79E9C093DB7F2219C39CC36EC72EA39C65128D835127C3808DE4F3113E9CE42BF1F86250919C8907E21B0905D960480D9786ED5BE6FEA4CA1017708E1A153846AE80F8C81DF6118DE9C25DE8B72473E5A2F40980417259CA2D3FAC3AFCC42E1D5DDEF03673CC2D9A7E6170357085A135B31B6E8334D331E222055FE01AC3DDA56472572CFCD908797BBD7B8DA672E2F9A10497E423AC9E84C17C81F81A944952C7511FC34C5626471B85D80CBC618A68DCC343575219999F28D01F732E76DCD6B056F60D0A9CEB822C1BB772F55BC85FD122A0972B0C0FE8B88D731887656036A4A4278D0F46B27B93530D3DDA7A689DBBB73D9DE39A6F96082C8D9B9AEC1FCEF64C7434C38B132A66B3E2DD51367EB83FEC5E6AF8B18A151058D092ED381AB4A86CB542F54EBF5B0824DF726B91DE3BD088CF8D1E55EB1F2C834AB5AB9E25BDD0A4445C692D568B6458BC29CEDB6CB5C5E7C29F16E1B95F22C2752FE1175FFD41F112EE"));
  test_xof (&nettle_shake128, /* 24 octets */
             SHEX("AECBB02759F7433D6FCB06963C74061CD83B5B3FFA6F13C6"),
             SHEX("8FDB0BFC41E23944858401D837455D17966DB41F4B9D2BAB52E208B2E32B22E1FBF3CD83F13B6B99F14BCDE767B286F7D16331FD56DB42A4ACD9C370BAFA8C99F91A5CEB24D52B52CC657686163057471F2D8C982AF779B05BA1E49114BD55BD59E1DFDC69F49EA96F771FB4F6B67D1606E6D652368A31CC3F7418427A8DF6FFF8E0B5E43EFB6D9795D766017284879AA199B6726C84623ED69288633A34D3742D20F1249E1C70370BB19320627598D208A7531EA233E772CF7F3994C06FFDA7E63B85BEF873CB89A83499174B461B36A375FAE6D5779E23EED9D6624DE2EF309721C428D8A64A2FA7ABDE1BB56F66F04225E30501AAD9B9D9078CCEFC8F721252A232B82D984DC5D1A03E9F94630B4023CCC4BA6A47FFBBCABEDD754BB403C25EBD27D445743E17DEF49EB16A75B25921E49373A4A669A3B3C6BF5ACF72FAAD2D9E0AEC3723FC07E541B72EA98DC3E7903CC89DF00F6B20F147302D6AF18FC94FBA446D37F5EA3A49A9B2942DC4926E06F6D385409753E3ED1C9E1672B783E491635B2701FCEB0F133C1CDF9364B7BD55CF1CA89DE5ACD13FD7B94B3EE849E789454EC39474636522FE41EA3504CE134610D675D5CF299F3E94686F1AD9BE58AF227928B9033823CA7195E6856F72008B31F4AD7095F98B4A5256D2A676817E02E7839B2B875B59AD020B97BCAEFD515B061F49526821043B3B1A1F29B61F5B"));
  test_xof (&nettle_shake128, /* 25 octets */
             SHEX("AAFDC9243D3D4A096558A360CC27C8D862F0BE73DB5E88AA55"),
             SHEX("64C80B3CDEBA491185A9ED7E0B5EFC031E36D4F22F9B99A30237C5459C7DA37B16296A7D2E221B314B47F524A01D94845F7B1F38D7D52DA83860E576E8910365A217EBBA342C7E26A350FFCDA54FAD792756EB1D87096F0625D6B237E77060E350B9CB9179DB967BFD8C63E15AAA24336B8314B692F027C99EDBB757E572B0DA00B45D90200BD778B396D18375D77E0CF32E7EAC5003794A010B3A1DCC5968F59A0CEDD4B2EBF4F0F151E5D8F88C297506295080C91CA11DA2CF32447245E0667D1D4FB619C514014ADFB7733D972198BA4846D57EE8D9FF73D543CB32CB6822996D04B08AF13DF3A696DF38C29B677BA6FA481A5A22D2611B8D721EF11C4361A3205F75C90B7B439B47893CD5C38462646355800EE91DD9366D8CA3C2EF754B8DDB4FA83397FAAEE9BB24EE4FDA89532CD7E1F05CC4D52754ECB59DC6BCA363D9E804B32C47664BE92A23023BDEBABC70DC45F15A708D5A0CA7E6F461C825D58784748284722DF182296103D27A2868C06DEFCE8F0669A173A9EA7C9D7BA1AD721A48A7076CA7E005D0BE0C6F5DF889747A581B54674DDD18A3002662A8D1F1373CCF9C0760B008A805E737660CE28BE2DB4FDFCA40BFDE0357615B8B09B264B9A9C3EDFB70F967BB520B09A76654ECA9FDF6AF19B298DC3348E7A5771F3C3E2ABEE576819A1C59881E6B8FB1DB1B114D746E90CE6F9F6C09ACB5C46B03653E"));
  test_xof (&nettle_shake128, /* 26 octets */
             SHEX("7BC84867F6F9E9FDC3E1046CAE3A52C77ED485860EE260E30B15"),
             SHEX("50102CB39FFBD98F3499FF53914870CBE4440CA8B6D61B79B7E6481B943D8E532947D06A5E04A6EC8E595958245088FA807600B1300ED4DE47A2E800D812BADA538E351DAC64DD3043E7F655B411241C9031ED2765264E372C6F59435988963EF83098452029761D43E4D329CBBD04825719ADAB1920DBED999C1C82BC9A362AB6C0DE15A65EA6E7EAF9B75FA15543910E7CFC214F25CAFBC771D9E641FA65A6F52FA97CE428910972EFCE8C23FFB56AA5ECCA3EFE08E12125E3C88C652C89765AB1B72C9D16AE3CF4C8A413ACA5675C18439F3909BC8DA92ED71C7219A89DCDBB3E07A3A8D4145B2D65E9525D3E583C70589D79519793F463DE6799D682E24E83E91A96E866BBF0BD72C041DFD44FF345FA83C24C13AE0689DE442DD01C85179E69FD9E07FE2E2AF1F5D1B5365FE16C3D9DA5F1D83A6BD71923252371D230D20DE8A76A810EF29469A370CC552BC0FE8329CDF85905134BA4F7C4F77FC83BEF84F7560ADCBBF8D384267AE24EFCBD78B8DEDC130BA739C5CE96D8CD05299C50037B94F9B48012BF346E079C4599C2DFBBB1F3980A1800BD593ACDF5C7E355D3CC44CA412A3565FBD8B2A9E8A9E44667979C9B261AAD8EC2B201259A2D7E4D5DA6D5DE05AE61760EE8DEA70CE5F380BF3F1620B3C67D8EC5DCD47EB14C078EFF113A952B7C77B8E832DFCBB5E7492F090D8D4A9893D222AB77BEE17501B8CBBE"));
  test_xof (&nettle_shake128, /* 27 octets */
             SHEX("FAC523575A99EC48279A7A459E98FF901918A475034327EFB55843"),
             SHEX("EA2A6C14ACAA13B8CE15F4363206525A44AE834D82B000EFE0438A4321EC1EB6DE31390BB495EA7F067D4A3FD9BE173981076BBE4928F2E9173F4A3158DC4DE9F940BD4C8E76C2F886C5D4C07A28E6CA1B35A4C31C153323508DB12D63908ED758195CBA55B1B8C9687D219E4B053849D5134D3BDD8D8B0ADB2C02793677853D6FB496813D72DD415F9EE59B6C30AEBE1BE2CC5FC3D9B4D4456AC65BE1970E5853F87C57BEC289E995FAA9FFF16AAB0ECA5EEE8319897D24E6D68F29B9A8937A0D98717AD61370E25C0F0B68974D41ABF5DBC3C05A26C4B5BD4C60F5D2F2F0126675B785CC2A0D0BD1B2EFC2BFD14790AEB55DB281535950891BB90A6A02279852BC07D6F03F7C42B4E2B4DA0008E6CCC0A8FDDE198159F6A2BCD516EA80D48F8EE63580ABF97980530AB589FC2158705EB0B5015D6B92D41D4D985661BED07CD1E43595C1A9C902F08976F22E01825CD2C9970A2D09FA0AC68C3428FE2B4957E528DEAB2D1324DB176761ED9A3CEFD129EDAA882C848E994B45D7DF77A489569B1720C183902A5A7E8116223DC3CD0FE84826A28D2266834D0B09AFF134611203D8704FEB7F1BE1736EC9868F4F638598BD530F6D0DAA795EA85C2954DC188A27A14AACBE2EA7CDDF85B3DC9F2DF61175530127AF3594EA39186F9A18348F9D60DD2B32DB207E5398F59DBA345511D9798613D66318C7F3504A431A0CB8E9D9"));
  test_xof (&nettle_shake128, /* 28 octets */
             SHEX("0F8B2D8FCFD9D68CFFC17CCFB117709B53D26462A3F346FB7C79B85E"),
             SHEX("3325661A828829064E99ADEF00E99E8B9B08A53A4C6B008FAE8DB1A833CF26D83F3F8C613DEFA9885593996ADAE75995DD196893DE7A89718296F12611B3315E22218B8E26E8124BA7604B4CFE5F427DAFA90E78D0B5E14351D52F9B8EEE659B54E07F1A8AA08281B2D8672CD405359A681C74E84B1F6F6553ED9B5184D10EB608B37807E5A824C3F0E1126824ACF192C9767919F1070E77E2F81A1B5E5DDAFDCE09DE562CAA1470BC1023E0EDB8901EB9CCFB813366EF60DECF20D0BEB552E7883EA4063383B123D94C35F5D00D57827D60DFB3550485B75D2235400A696684289962C2F71227C847548961B4F3631C656ACCA90F9AB21547A99197C32FBDD410FB7DEF62D1B293BB0FA36921BEE609A6D7C2C9D6ABB287EB2920003C1E965BDDE75A90A41518855039E3EDFEF1A3DDDAA192FC6198ABA7C0631949F140291FCDFB8D387AD4C19F71B6EC4A6C3F5B22A0F8BBC0F852E03953057CEAC0183569815625F566F050AC6B14BD87D3056A8FA5D99E621632AAABE723189B0E87DCD3CC3D29A4E3ACF2EA915A8A7E0ED73AF0BB905A4E8B0599D20D699D2C971BBE338C964BB735F72540D8DFDC069F950AA13C814F46866A32C3EFD0EC5D28482298BEE0CFED7B8725E64C75ED0A35D8DB157278D169849B56AFF015603E1006C168276D96AAA91EBCFC6C6890D5382CDF1C6C5A1BB6A3C090F37ACA605E9E7459EE"));
  test_xof (&nettle_shake128, /* 29 octets */
             SHEX("A963C3E895FF5A0BE4824400518D81412F875FA50521E26E85EAC90C04"),
             SHEX("1ACB385112F369E6D00420FAF1291B1BAEB428C09017B647D7343DD2FF50870CBD47601EFE679AC0724DC63CBF543C87DF5060CB0F11C786E97844A02BDAB528D3FA5890DBCADD9EDDF2DE00A639B0FFB8FB2CDE673F66C54BD554E737B5F370F53823B135B3257AF3394408A853666BB00139F147C6EC198A5829E8A5F85279B306A428DC2579E66E1E18EB3C4971644DBEDF4F99BDDC9E4096D0BC915AD1F72D522395C6306D047C399BB1C4877BD8278FD29926C3B74AAC5414E9E0936E01DAEE1061C5CD243C380AD802459972D3C2DF479719A554CD9318A8FAA3016AA6EE057A4E57CCCB698A4B0EC411F31D11F0642053880662EB3E098F88F74C65225669A4CC8F3A04E23F6E5E40F9A3C7FC2B4993782D58B27F80040DD94475A85A14BC792C6B86525D359CAB32744E2BDB4890A392EE847C065364BFA18C85D7E328E5ACD9F04E6A6542C8B734A3E3795898D7E48FFA8686CC12497883092395239A55E56AC18C4D63A9281FB8AECA2C099BDCCC65CD239CBE597CF5D577585FDDE3F4B573A666C2D1E3C611DF94C9C519DCE3110C33EA697FC1955127C91BDD420DE5EB667933A49E15959EB3049C7D265CB638BDFE3D6425E20DCE650892E0FC346922A4277C1CB1ADE59886F8006F0D0074937E852A0542B5A111CC4ACD65F3B2580670DFD41C0C9EE3CA5FCCDD7743D131EEFBA0543646B31A9EAAC4CD30DE"));
  test_xof (&nettle_shake128, /* 30 octets */
             SHEX("03A18688B10CC0EDF83ADF0A84808A9718383C4070C6C4F295098699AC2C"),
             SHEX("3C7002A2ECED36B19A3D71464F86A526D0F2A92CA76FE392810616E75A25C223F06BB1EC3BB5BA0D00A51958789DE67724E1D57C4806F6752E610327B4C84E9C5E815F8765E6B452376CB1EA5E3151C21E789FCBA16ED05306C53F2F089B21F2B5590CD4B48647CAAD444450A1A60E1CD0BD87465C2A3C2D25B15146767B6DCEA1A8AEB8B717410EEBAB139B041FBBA1F654A7136B7071573316B3A7E5BCF8929CC1465FFDE55DF7449C7068EB1D74C0B12418B50046FA2197A47F4CD15E7D3702E260C188CFDFBD55CF4332BE6CBB157FA8E62D003D902433719B516C59D31F8F935CED0F010EE66E8E48448FB9BF3A855BED9CC604DF713A689009223F858DE44137CFDAF2B194131B84559A98AFD4F86AB13D7A1DF01E414A10768F6D48E426EA64A0602BDACA1CA56C903940A35F28842922C114F25F7000D720EDD40AAE30F300ABCE4FA5A3B6088B27377D7767762B220DA383308D00A43E3B9EAA1899C3D4E261EB132C41029AC024D781001B56A1127226C3517A015782097D754233D83D3237A04754F3F22547C4321FC175F9E83E679B0EC7564D942D77F64EA480780242CCA0BA64A8C41C7E94E575CD9E60C229E4765C8BC993048A81EB4D9A7F1F1FE449ABCC1CC5B5CF500D922A3DD26D6ABC62085249AF7A60271177B19F2D150495FF3B4E8AA0414C58FC1A1979EB9938FCFC6AB3AF97B9A7C8C94D12C821"));
  test_xof (&nettle_shake128, /* 31 octets */
             SHEX("84FB51B517DF6C5ACCB5D022F8F28DA09B10232D42320FFC32DBECC3835B29"),
             SHEX("E22AC7FE9DB19147B2DBD586D6C5837A5DD3DF2346EA61DAC753B0371274DC110612AE3DB350EAFEEB89BB1179EB9D84A0590B243D0DD9BAA00796030D2782F0163E85328A0AA06974A7321E66649281DB8C506400310AB3E21243F4C2CC5CD8B36AC7C35C235305E6B1585B33784897D82A2D31F664D963ADA323A9C922A61D9AA5BEF0B90C6B7183F1FD0ED4128B2FE0E12EB6B461176C52AEBFD608C00C7D79799071AB30DA33CA9AA26932AEEE0D585905BBC853E80AA746706FAF7BE50C90C1FBC18E290505277E9BB9BFA9C767E952A68C4F93A044F62E066E61A0AD301BBFB921C818690BF6D116C6CBE7DF174A7E57E22294303820494757B3254AC40404A7F4A6D1F9624E5CF3A770392B2DF9FD1FFEF01AC9AFDC74442C0EB6F11E1EABC59270B4DA6F2D6356E79607D6462C08A6F29154BCADF4FFD6E20ECB700E188E523B3930E35C8990AFAD2141FF71912ADB07DC00D5BB78D7FC5590467815BA9F46CE4F5CAD34910A574687D8F7FAC2F60B34D4C3BA6D25D3E5118B851BCB73C1B1004A623F8DDC8A0D07AD21B45F543CA8E705B3864D1C4FE024A19ED5FB0542DBA0C39FE0A82D83266D9C124E61DDB107D8E0AB57C970CFE5879DAAA7170022408F7A9A228196C5C7AC614CB98CC276D1F5ECD79347A41D97360A19E65681A5B75E78C7F79ADDCD401DA6DE7DED3B1DFF1F746806AE03F496CA701C8448"));
  test_xof (&nettle_shake128, /* 32 octets */
             SHEX("9F2FCC7C90DE090D6B87CD7E9718C1EA6CB21118FC2D5DE9F97E5DB6AC1E9C10"),
             SHEX("FCDEAD82F39CDFCEF99C1BABB674A9A8E24AC594646C31D020A4CD2BC2554AFD78C4E413F7804FA1708B9F4000FD860E3070F2E1BA9EEE3805352AAD655B4B0A728F2D5FCC43243825BC0DCE33CA716626DC76E920D72575E26DDD7110D0F991A91200B513AEE23AC9BC7043A152ACE0CD0B49181D2BB6BD36E93C0B627ACA9C6AB6C85ED70CE762429C8F262708103284C0A792138F10E8568EFB2399B38A31055C1188BA59344E6A2B73D5C04AA524056649844D1DADCD07D35DF5D851EBAFFCA5703B80153EA627B1BADFB2885F70F786D34F5650FE73E3690A8A96610059253DD3ABB5FA7C54CF6E77695D24A6594077EE4D3673F9FC56C62FC7F710CF872014C0A7DE8B1CA6AE8CEFAADEAF5F4D055FF765AD418713F2DD08EAFB5E16EED9FE344EE8D4388FDC22351F63834017B539E3FF143F394B5B74D06F65E96A7A3D028FD14F6C7001EB7AD2DCFCF4B2447AA173A2AE8EDB581B5BBD89E8A468FE0A38507530B9795DA3BCEC6DDEBCE9EB3132EF18C9C2A8B936A431C8B121FA996FF9BA5CE522986B678A5EC99A103A91CF33196E08C82DC65E68AED238A9316A73E71CF5A67CE440B93BDB845B3A60539ECCDCE41BC723EC9A14EE4E082F60C0BE3D5E50DFC8BE1E86A97ECEE9D88E2BB2A3AFF47FBD6D6675D81EFE0708926B81AB314A524FC745090162D2AC723C4326E0F9E16FBDBA2B1E9914BBEEDFF96B"));
  test_xof (&nettle_shake128, /* 33 octets */
             SHEX("DE8F1B3FAA4B7040ED4563C3B8E598253178E87E4D0DF75E4FF2F2DEDD5A0BE046"),
             SHEX("F9A4D63D82C6EFA5B1FF90B26510ECF06C53BB1C3B7427B64C41DE1FA03BC0728085C972E94A7FBFE5F3150A7B9FD8477C6CC1C250DC6FE017B0D0668ADCA12E3C1C9BDC78247529DD1F69198BB0A371072F0225420AFDDFD64AC1F7A8F06FEFDE803579491D8BF61D6CA84E985B1EBC7F0B48822B8BDA9CF9AAF75B7C33C5D360547A34C68114E8A894D658E4F27E3C137E1B2A203293C015975AEB49AB315D142024767836EFE41DDCF42458C5B5238ADE07302A0EDC28A5E4F686D1F98372A179DCDD2EFF78914AEFA492C05AFF03A5CA3E188F611AC24922CC28AC8010572F406C0ED1BBA849664952AE23842454F60D7F4091D0A4BC2CA6C7B3E1C7DC72659DF6A709587BDB30FFC3BFE6391FFE3D94084D93F2186024B9AA7EB52D5CBE760E5E3A49496A2239C9C71FD2C18C199441F7509108B8E32BCE086A110507038069F99477DD9F75EFDAD8ED861C38C61E2888F7BA3E1AC237A2703BCD64F19BEFE921259F88D225B9B5461F070297A28908A35D78BD66165C8CA532EC58C64DA1988E39D1237588661933320C80B30C183126A10222D6C2DEBA60A55EB6574AF1050AB4A6253817EF90A1DA8B42633B97F792950364D1E7C490A5A2BBB8C3412289329D2B0A644F8A447C5CE8A420C402429B94F4FE11BCC71FA2314D4692DB47EC2EB58C32513B07B7CF7DB276D1AB7190232E7025F805CA2FF9F27AFEC20E"));
  test_xof (&nettle_shake128, /* 34 octets */
             SHEX("62F154EC394D0BC757D045C798C8B87A00E0655D0481A7D2D9FB58D93AEDC676B5A0"),
             SHEX("CB65EDDD9ED9FBB19E03DDBEE2D0EB5F0C2610F0AE9F89C50A463A303C3CEBD64345F39F1CE7E04BD5AFF16B15D5DED79E717CE07ECAEE9B7BF1841B17A1266B5F92F2A92D6BF964E53C3727C50FD9418C1CDBD06F0F0510FEF3ABED7B4CC31534E885EDD00FB1316617D6433B249956E1565BE72E5483EDC6E5E90B470CB2E089150D3B3B70FFF3227131E0DFA12A2248C064DFC83702FC462FB2165CAFDBEF309EDF1BB785ADDBE17C6E6DE702BA4BE1229A7AB7BE122306E306E92D0DE7020FC1CB568F5EAB5F0A2140B010587DEBDD8404DDA2C4D6E728A4F1A55182E018789CFE03600301641D5DE2DC58FECADA9A9E1425DABF1F00EA013C05EF3D64113FF351F10916438C6B92B6B6BC5FD3D6CAB43669A1C60ECD39829C327899885EDBA19B8A9B4992805916C460E5B5F72F53242D206D2B3DA3A976784FC40B6A4F0FCDE7422453565398A040F9E5545C482EFBF9D3110100DFB04DBFB24A94FCF3FA05406EE9B7E677BA0D3A52032AEBC2CBA54DDDD8CC0900C90F48CB9AE107DE01C3218F0EF5BD9DB9D7DF8E926E60A384DCA5299610413316BCCAF2821B926174EFBB791D465F887CE3020B908B07E545A386283E17F980639E81E71B0203A3D22EF233B4B2E593FADB04503F0CFC5EAD74075767228458B1EF4C5DA77D126D6DB7EAAD6A2BF5FE9C18E562AFE5671DCCB60F8DDEF168375BE45EF2A391556F"));
  test_xof (&nettle_shake128, /* 35 octets */
             SHEX("B2DCFE9FF19E2B23CE7DA2A4207D3E5EC7C6112A8A22AEC9675A886378E14E5BFBAD4E"),
             SHEX("7A3E2CF760D677AB42125BC78324A8E42866E2C148087F92FF27E149DCB2ECC3F3209D3000583D96C60ACCBB637714C4175B0DB1F982A99A7D121774AFA37ADAC741AD9CF72BFE7175F591E18256A95A0CBEEA06B5CACE02C7903B1448DED23F3A7AD61F9C5CE0F61235520706667E94062CBA04F735CFF813A5B7501975BB7DE70747E619EFAD4D548CBA8FA818A0F4B59F8974429075DC492E014436CCEE77FF6947FE3702659A14BAF89FDEE45DEDCFE4CFB1D45D59EE5D79DDE87C187AD1E04F06217C41C35F1185E63B625E30E67C5377A11909612DAC6539F72A6BF46EFA1674D41F00FC2DF3A603A480517894B88556971B793FE3D14E7699762641913ACE9460A4791487937FC6C94B6A0A091C465A3EF24EC55ACC8BE486255848DC7BACE4A3F860C7DF0307B70FB33D8164FE3FEA9ACD2BF3484BACEEFBB4D24D77BDE756CF7614BC2F4D037E16943E08437695DAE204CA08972BE843F59D608325C7D6D64B136148887B49FC71ADE52B11E80576FACBA740D88993DC77DF9B7ADEC54A6DAF3149B21B9D0D910F2EEF2C39AB3304F2E5656DF646E551BAD1E4B2BAA3C7BA7857CA2387F516A0DE8600F14B653055DB1221A57CEAE2E4AE4D3102B60BEE67A5456E0E2C0E5A0A2EC249A3AD11B533005946BC6038A98D129FC696F48F05F0914913D45BD7626046ADDB5AE3BB4C4E1BEF5332767B49D8B576953832"));
  test_xof (&nettle_shake128, /* 36 octets */
             SHEX("47F5697AC8C31409C0868827347A613A3562041C633CF1F1F86865A576E02835ED2C2492"),
             SHEX("0622ECABCDD9A8C2BC20FE0B9B39CE1A6A835B5181327794DB2E1B40070FD9D89A498D242AAAA8FB5063D5E2AF45A1CD04CB28B96EDF844AA37C7CF6EFE39363AAE56E1E6C1E537CF8E58C20D5AF57AA0687978E8EA46936DEFD6A6F3486ABB9E1F04B6128710F75314DF64706190F4AACCA8FA9F4E1008385E641F40FABD228B9DD421E30184995720D49112FC429165AAB6DADB0CFB6F986E11866D138DDA8A5B4656B062D73CFCBC8B8873509FEA2820E150672AE7C01EACF5D9575EC6B1DCFF1F55E6C2DCA4ECE5FB4A6F1EC7BD089C24149F1136AF9018920977519F390FB7929FE84228D5ECBF8FCD5291791CF7FB6CA7208350CFD5D7A761CB8B6524DE5A85467DEE4202232218CE6B39E5505EF02452B4FC0A87C32C9B9C9B8BF5FBD5755313F551891C5D11E26FD57B65A42DEF3BEB07C88D117CD168335A7749B97B1FDB04C7ED3D2AA1A2DB4E70226F3D8B87ACC19A1F34F60BFC1CD0EA1864C5A991CA21C72C163D0B6BA5833D8EFC7AAADE5BC6D335E62852ADDAB38831AA04DC4247DC7E1CA5E0B2DA9ECC5358A889EAD2F2CDFB538D900FACE60AA53228145644FC4AA2BE363655ED6CCC47D23CCAA0EEA0048887E2C670AB4AC85BC71D1CAF7A1C02EB3C784FB189AE756EE9133115232A92105B033148C1C9806821BEB545035A1359CB4290B736D3D14C5F990AD2E4FB02A1780E34DB86FEE3BD2A3A9FD"));
  test_xof (&nettle_shake128, /* 37 octets */
             SHEX("512A6D292E67ECB2FE486BFE92660953A75484FF4C4F2ECA2B0AF0EDCDD4339C6B2EE4E542"),
             SHEX("0D126808ED2AC9458F8002CF172966C8DAF02045518EDBCF87FD315F31EA781C73579DD9BE6A3000F7D6E5C400970D4B6731F90D75CA01E55EDE430D8A459A98C71D9CE16BCFC334BFA1CE8E18A70A50E3B795B810FE3273B9EB7DE8D9B7175F97C32FB057678D819B1134AC30797F6FF7B622CE48616436FF7C38453029DD7B35E648C14AC725EB49067DE2F0DFCED2CB89AC6988CA178E72D0D19ECC481748BE17E8BAAABDBADBB84D0B2F2A49C4E60C00018ECB61D2CD25D8841157D9B6ED74C70C7791F6F88CE61B5FE0F28DDD2B4EAE5E2A71DFF495907FD883CCD522EA1DD55291A5B274B4333EB62650D55B0CE61EC1761CB2D5EA39C712365309537918EBAC6EB8A0BE3D19CDD0F32FCA2C32FDA5DEB68EA99FF5F7B66EC4FDBC2F8A829A4B3512A52044A7F7C567532376D971AE504601841C8091695533266D14A14AC46A7370372195CBB4BC212EF72A18F7962EEE6738EF1A669B663B336CB324ED808C3AFFE2F39514BECD5D654715C85F3E5CDBD9029C1F589CD0D563AA98C6FA99769815922CA7A30F13F55843DD952F5724CCDBB58A269DA7D066A548540C23F4A740D82F90246D193C22A0CD0601C2030E08E5F0A9AB7B468DE0530F50B156C38DBF08217B63B726B85227A5FAD8BF9165E76B02CB00CF7DDFB03D9D38C882E81E657FE5596B66E777E672FF416A3155D4E2EBEEAC5B49A8AE3851AB13BA"));
  test_xof (&nettle_shake128, /* 38 octets */
             SHEX("973CF2B4DCF0BFA872B41194CB05BB4E16760A1840D8343301802576197EC19E2A1493D8F4FB"),
             SHEX("863DD45342E1AF41F682C138EE8DAF96308B1973B3740CEBE9C2999D4190B05C7630EDC87F9942F31A089712BF2B705F1F3D835E25E51E8BC6D5FFD017A5FC2F30ECFA82C6B9033AA336ACAEC58FA5E0D4F4B40BE4F60B0AB303AD007336B5D08989E635E92C7C0F22CBEB049BB4761E786FDE0E7AA378F6C492B3357916DE530E7FDFAB45D2116B3137B0BA6BBC5A04313757B26CB3467031D82A98025E628F0A3BCDB234D4E7FAAEBD7F7F4E94A2C4DDBD947A3AD2E66F7D45E66C89AA037B956AF67D7A2DD696D231D31159B5E217EF4A911A24B4C012AD67152C9A1FBF85BFFDEF6523629BAF766E22F009B6371A4C9473A42AC84E2BE89D2698E668373A01A2E8EDF60F9C7D83BA415EBF8588C0812A4409A5EE205D794EC5E3B5A3688D8520C98D516C877F5448B6CDFC1A1DF445008CEE775160096237EFC27FD32E1629835B6663601647AADE495A117C025E03C80D8DC2F31BBB7CDD4D2C50C7987E8003AC04226E5E051E8D2930F93C143F2804DDDB343591CE11FE55BED40A5DEB0728EB575694D4F62B7E7901E3A005A306961051A0219FA49E263E85E87B2880CE9C9CAD37F5DCF001DF265FB448093CF08EE592C66D2BDD37907D6492B05EF24E96B08DA4E55A289E6A2CCF65624257A754C133B54DB543FFCE3D11093576357C75D09E0E5D18D95D2FD5206A1F5750030FA789B83A81B6E81D98B1D360C179"));
  test_xof (&nettle_shake128, /* 39 octets */
             SHEX("80BEEBCD2E3F8A9451D4499961C9731AE667CDC24EA020CE3B9AA4BBC0A7F79E30A934467DA4B0"),
             SHEX("015BB0DE6A28737D365353E9F0052EDB5272A67F83863341A7B798CA9197A051039987E22E6FF95F31EF2BE455956F7C5704FA4108F600DB247802712ED268F35B27AB89AD0116245783DC7BE9B511EE4A0388A7EF9A234E1161F69910D3C0C76738DB4BE4480EA0E8BBABB081428410C34CB54968F7DBB8CDE3336317B6D813A9A859E4AE69BFCD855D261EC89A54C510C4DD2F68CD5D607B299204654F01DE5A8158F2E2BBFCB20857F029A579CB769B3F232A17F546A653D04FEFFBEDF3C6875DB3B1905E73A4C7BB383A6B825C240F047A2B6F6B944B3C51362DD2DCA46D23CB502A7267ACDE99A8E76E63086461EEC706612452E240F97873CA213F9FE6A72E66D91BD3F60D04F7DA7CB059258875F0D458120F0800BDFCEBDFFE2ABE784189406C9BB63F3E807BE88C72CEEA54A55244B481875DC2C736D899D9A928CF02C2B5176D8AC5B64BBA7D949C974E613F41B5CE1B687B91DBEECEB66AEBA8CE8799B3484D9255D54559ADB01D960710D198FA8C5A18EFF47E3399A8E2C386D846BDA56E5D9E7C9478DD563C50D840DC664D105EA92B62C6656E2CCD31955EE7C438256B2275036DEB2380C08D26D26C1E638F7EAE12F5E8539F74516F1340DA7B16A4EBF8ADC93F690FF39B39C612D42F8D4CBD764B2F7CECD12250527810C490B0FF8A3991E3924EAEC03F67BA99759232D00CD58130BB3BE9235E14772462"));
  test_xof (&nettle_shake128, /* 40 octets */
             SHEX("7ABAA12EC2A7347674E444140AE0FB659D08E1C66DECD8D6EAE925FA451D65F3C0308E29446B8ED3"),
             SHEX("9CE519A1F37B2C70C6F641C076F787E8AB26186EFABEB6CE18EB8E75B0A8EEDE06FD78D326EBB3B049AE484D6AF7B1DAEAD737653717796A34DE1DBCF900FC59B299756398EDCE063C5654205C5FD268AA71A639C35E35D4DEEBFB8DCA17686B58A639F0DA50E1A7938B47A466D03DE7041B5192FDAF036547B7F22D5A23F8B6AC48E7C67B7372AF0E358CF73D2FA922AB3CE823A68B95BC3A543BA236E413A3CC2B3181DF4D3649474213A0F8148836230BAB4AD722B2A3FC446C395977CB3875C06C0031E807C24BE7EA9B025248F1A99C494472CF417A803C69B3FF8880C0D364512166F7E34F9837B5C66931017A4DA3E63998D0F402D5A75212C8042712AE4AF4B4900EFE6C9E1EC5ACA07FC230E6E7834A5A865F2DA71EEC9054427945A913C5345506EE32E6AEFE4EA3E488BA07455D98C94B5078E75E2304F2E85DD9DD4BE18F023D1AC6ABE5924C8E7B8219E3248F14EB4B78B6C16FE2C3AA40B97566AD3BB89154926BAF8820D7E8E4557F8138EDAB423CC0C800DBAEC892482B6155422D2AE3C1E90DFC96AB800FD35DF97092AB0A83A40C7B925514CCA3833BC1FF6B7E25469D6744A117699C481E4EDDE082211E9D94A73A0C29FECDB271E75AD038EED0C7D5B856C1768C4B9749B25811F16B1C408C1F2E3F2BFDCFC8E5A329B675418ABC4809DCDF2A3C4F867AB9E0C0AE28924FCE90802715FD5424E783C4"));
  test_xof (&nettle_shake128, /* 41 octets */
             SHEX("C88DEE9927679B8AF422ABCBACF283B904FF31E1CAC58C7819809F65D5807D46723B20F67BA610C2B7"),
             SHEX("648167774B09A9FBFBFD351051D18F9112D63B066BEAF5070D3780189658FEBEA0A4581DAFC6948EC1FA96F0364EF3D1F2323EFE5294BD76C28632FDDEDA51079E9BCA5C1EFE95F76BCD4593836B9C64D82DD6ED02AD07622BF2B1F7F0B0738E0E3929F49EDAC321268C78D47D853390FDD8F1BB29C6DF469360E75E96493DD4FE5BB2D0816938D5642B87F0344FE4186DDFCCA06643A691BD9272F407D6DF7A1633550F076042277C325152C8673F3FD6C980D6BDE4EAB43018E322DD3647DA183910CF4BCD3464CDFE0D0283E79201553BF03B028FE6F4862953F865C6108BCE672C4DB5503C1D969F87BDC310F7B78D9DF08ADE46F54EF718083F90E60599E41CCE34110BEFBE315E8A0BB91D428DF35AF59CD4F0C8059B6ED68339307280B05C3AA490B7900C13C3E12F1A4A3983DB5F0A620179AF9EF900535E32146C8801C750080E363A6A67E4A9037FC4301FB28B006822598B38DC38DEE16AF9A1127518684BFD4A0920C7267AD52A447CBD6178A0329D4DA146657549CDCED7ECF9C2053541943AFCD508F677549F364D7F793A7B7BB8FF12B7594FF768FAF6DD77B3CEFE97715FE70D8CE8468AEB8266CB09F21285A001AE13D0FC8F29CF8380685034A886DDFE8D3AD57329E6FB92D2A079E5AF01F77AEFD78A29D294A6A8E37E20159408BBA4F10F6FBA5D54D4C4D93B1ACD65D84088ABA61EC29E122EFDCB6E"));
  test_xof (&nettle_shake128, /* 42 octets */
             SHEX("01E43FE350FCEC450EC9B102053E6B5D56E09896E0DDD9074FE138E6038210270C834CE6EADC2BB86BF6"),
             SHEX("BE4DDBAD9BE9ABC823F65A647910AA06265B547B0234C1AF6A12987000F710016295EA57CC5F30656C56A66F28E85C24567ADF2224011191A045C23D4491E444114392B74BFBC60858E719A67BFDEBC135725DE961EF7A81AB90CE4192107A5CA84AABBFD919BCDB8270E8079E29EDA4C22E130A1F046E5A87A4F2CDFBB4DAFAE337C1FDB5BDD9DEB321704C585D632CBB7A0438427A86CCF7575856B535B2E92A3872F2B5D3C945C16F3547B4779E5798BC82883E850585A7FAE200A47B68FFBAE54B17238D7CB3FA2C9374942523584D85F1485969BF61AF5E117275FAD27577DAAD38EA3702B5DADE3D27B4A2E2A66F88CCDA568D5D4A11E1DDA5D8BA7ED9CBDEC91487C19E1F9A57D59AFE374741D8FC9DF32B3BB60F8FABEA1EAA95289B43D1FE712A50B73BF68CAD16AA25A1DD95D80A0F174061A01D9F18509589188D186BDB991AC9A41423BAF9154210ED6CB098D6844699F07A388EEE76E8DBA801276F6D2AA7965065F3C923AB64620DA1E17A444790F9B7B326A3579B9A5EA5FEC313D294F97FC846D721945B6ADDE91B66454FF2F2A81CFA19F6B45C51D07BE8216FE7C1E052A24904B2EC2E9929314CD9DA48450BEB8FFED8C499816C078AEB2A1B52F07B5917DEC0A7BEEC2B83F778BFB15CE413C9A4307CCB7C83C879824D0CFCCC4B3AECC519326FFAB353E37C59BB99FFBEB44A95BEB1DAD25843BECC51"));
  test_xof (&nettle_shake128, /* 43 octets */
             SHEX("337023370A48B62EE43546F17C4EF2BF8D7ECD1D49F90BAB604B839C2E6E5BD21540D29BA27AB8E309A4B7"),
             SHEX("2378C76E867947E94C0383861E60E659401B1177BDC1133FBDFBA3552B458DCF1BA41C825ED8F41F7F8C8986B0A47B25BFD5F0294FDC707C8389560BE932C582E2B5C63CD85754E9D96A9725C12AD234B887FD5E96CCD52507BC564F59E2E0A8FE89A444AA3542C2B302D1B9F8F3D41F1C960E1867A4FE64D5F6216E64199F3D66FE4BD22ABE7B23EF97328F5FDB5CD502A118000E0606D31560C2CBD4CB7EBF90A0AA1826CB32AC85156B678198D225F4D58D3076FFF881D9E6815DAE811E7A04191602E4287F72C95A497C2DC47C925B7B1B78BEF334B153566A8B46AC8F6CC0765300E970B12ADF8E0DFACCA6A59092891D73ED55DD624A2464509FB5610BE2AE0696097D5645CBC9E992B7F1F39005EEA0AE0A8B55ADA1F28EC2A544266462E6A3BA97A000EA722701690007EA7FD41FB3F83BF85D5E19AB140F2FFE6DD70F199B7995997EEC43B98ABCA0E7902442D0F14E7E4B288A83F8E4F5287F78F57A2DF951D62BDB9B9CFDF3D28F0374BBD3B4AB955244AE444E23988721E5B3C4F536A80FBB7114DE769D587600C92539D92C3B2621C679064693BD9893BED3C353AF5347707B6586912D6CED5903ABB41DBCD5B4BB67EF2A315F8BFD23073D76715E869F474DF6F5C98E3C1E65117B1844B03835187A6B2AE054EAE51797DE704C7D4C88A258EFB12D1FD618D9E95C4AE37EA0F54EFFE216C71A0F0514BD569B"));
  test_xof (&nettle_shake128, /* 44 octets */
             SHEX("6892540F964C8C74BD2DB02C0AD884510CB38AFD4438AF31FC912756F3EFEC6B32B58EBC38FC2A6B913596A8"),
             SHEX("66C5253CB16C479B2D7B63E7DB76CA275F14319DACC1D00C08A7E170D97366638949FE0FE252448D9749EC105105E8BB483224076FA9204F9A4585F150EAADD931B5D895EDD8F2AE37E30FD991174064F7CD93B03434C22A8E5CD27CFE19B5D8F4BD405A9D182E4D7F9A2B009C68EF4798A469662E40467AD626FDD0553B0EE86320B3DB087C096B54FD9F367199AAE4490F796BFE72DF10756BB10E336DBA6305B686A291D1097FD9DCB1A629F38D2ED30B87A8F5C1EA1A2AE384BD5F316BF112F6BED8DDC50B4E17509C3D194750D02AE76FC1105753B07130861E866E89B2DF9C498C87BA2B9161753303E0DB34FA6A523A6BBCB262D3C644D1EC88F9FE1710A65290DB69FF847D850107E028EE6CF444E5AA986C98D3D5585D66B4FAFA1331FF4A6F54EACD712F39FCB234F4D4CB8C992D079471A269CD3D3F733F0806A78BFC30F8FB3AA97F2F2AD36E8AB019D7E833855299EBC91CA512FB226D797226486153133858649C31CE43308E6E0499EDE16AF5FCC5CC9593A71BEED6206063D471A79F65E640B8C20A131FB15439B9F35D6053DBA46B2F2682281E9572720BD3945CAC603E61E82BCE2144A19DC1F254179E011D096BD9FB2CAF77BDA529BF7F654118D5DB1CD2F973F55EC7B11AB616E731955752FD8347718CFDB2CF6EB6551C449CA601299166DD7E10EBA447DF8C29E96436B6834F162D7987D9F55B12"));
  test_xof (&nettle_shake128, /* 45 octets */
             SHEX("F5961DFD2B1FFFFDA4FFBF30560C165BFEDAB8CE0BE525845DEB8DC61004B7DB38467205F5DCFB34A2ACFE96C0"),
             SHEX("0128B09A01815373846EB62037245B4BF522303FFF74A76B0AC17482AD79E0BB2686871B1916DF154A7506985F2A3A6D86074C988C2FA255DDC266CB8EFFDE2D0275BD3360C1E920595E185EDA95B54A0B02D6BAD88F63509C5B816CD375C0281D1FE29F11318BD35E4AFE0CDF51B9C7CCEA285787FA74A72878DD3878565606F4889D3454913B0DCB7ABC4B2F08FA3F2AAEB6856A25185A00B1BE8A8BC7A9A34670793923647AE426BB98C75E45896D4DB8A990D2712401694E202EF2E3F33DF1120EA9E821A874E67F37648A898CCD759A1DB6FA6FD5C14756538F725C468BD3C0B6973A7592C8CF0091D347174E8D954E5795D360C4DE1D0785FE45B71DA884659C98BAC3005E5BD88817450359B51510AD95F16F08ECC64CB9842F0F8EA005315F6CB17C1CF6E01BA64F6847EBD4472DB1AFFCBC0E7B7F5EE8DC5DD18B148BC542CF0987E294AFF2EDB3EE60C8AA4B8BFBD42433243D55B4B9E00CCDED77251F789AAD74681880AC765B21E01D499553ACEFA48456A7E70A9A39163C2E089F26C94733DD63EAC1D2BCA7082145402FD762665C3F72760DCFF13CA4E80F504148144054B714573381C7770C33E07D3A787ACC072F3C35B9573FCDC352CD87BE8653618D159F78BAF16ECF6F8E316A97ACDF4BCE2C65CA1BC98F257914FCE9BE62E899CA3BA40A98A57AFC3D63B0B0B92B69FD0DF0451D7F34297523A99D58"));
  test_xof (&nettle_shake128, /* 46 octets */
             SHEX("CA061A2EB6CEED8881CE2057172D869D73A1951E63D57261384B80CEB5451E77B06CF0F5A0EA15CA907EE1C27EBA"),
             SHEX("FB206D80AA1D2AF8CE64890D3FA84600F80423AA98447792F732FCE4716E559F7FD4338BAFA60937781B68736603B7A86F1688A2EC60880F6A90781DC8BA3D9B329D662DE842961AC84D8FC876449DD8C393441767206188D561AF59BEFE6B91E365413A5F357622A670E22E730183358F9689E70F1B0D09CA9924903379394F516003E37B800602EDB84912492288A2E09B46BACFA3F4677D2B24605A584B3CE3174FCD1AE4DEBDC99C9A75EA7F4E770E2EF184F801E4D111EE5B11950B29376FF19B30A50C4DF93A82EC891A321F9A6EA1E0F96EDA2FA0C2FC176D5151A6DAE7C9536AB17BA47AE31AD69B2AF9248E5236AA58D7B864202512E5356DA226D5750CE12C062733BD73B7163812EFD452FFC4CDD8E5611996E23CC0A5824C5AE4ABB59A5DAE40563965483B66437B1E75369FF6AB31ABFB34581CCD00CA76CF72AF7ECA65824A46A48FB88C8ACA06CBC5D3785DB3AB78BB1174354C7AFFA85086444D3EC92538CAB178023E46C7E5E5EA2FEE1A970F41D374A73FA6D477D224F095A829AE8E0835D197DC66D1DEDAB9427DC085A6A95C4F065CF656AD1146ED0C45EE7BCF9F6185358702685E9539C921501E3338C2A6DF7B5F25BB248E567F2173164CDCBD8DAF0130879CA8362CC6CA28F531D81D607258B66D589EAA9CD5A22FD7490B9A01CB6C0958C2820FF832EB94F0E0CA959B93DCAE56A4FB52981F09"));
  test_xof (&nettle_shake128, /* 47 octets */
             SHEX("1743A77251D69242750C4F1140532CD3C33F9B5CCDF7514E8584D4A5F9FBD730BCF84D0D4726364B9BF95AB251D9BB"),
             SHEX("EB74359AE344B26E3533947E26CEFBE809935CE7CE0D597D1C4125542FFBF1783052E0F90A402D7888CCA5F96CE9D874C600B2E026BA4E79ED427F3DA96448BEEF4701C27CA31860790156E93B12A9F42929A8F8D9771CA3BB59ACE00C4732983997DCAC1046E574EB9E5C912AD4E3813001C86C91B6385E92DFE92FF5CF90652680392A33F6A1937A77DF9B9EF2556643308029DC5B5FE23AC9B567E6C81D0E909A279E58215D22132ED83DCF743065C0CB16EB7B991F842125254BAC71D80B64F205D799F698F0FDF7EA19C9234357C5838A0BF6ADD176416DB3324CB07B543156C38C2FFB0DF104FE3884A6B8F03B0E751B3824FD097FBD3557C5B9C962E9FB29175D1F20799FF6AD9843E1A5B64B2E69FE39F77C2D1A3FD3ADB8672D1771FD6769C54775DF97B0964E14B2B326362D9792C0C4CDC37AFF30DC3D007AC62141D370F0984F4B3D16B87FEF8028792CA92CE6620952D766474566C1E96DF672F3C335BF2949228006BE2B1912C0DDFE66BA6648EB6EA8624B5DBBDEC70BD0A745434DC5404C7AC7DCC94197F8C48FA381A5E8E7FFD1C35442C04EE2378D2761B3647A7637423300CC09910B9A4EF90D4992A8FF8FAA3B6138AD01E7C3C89660E963BB2422649589DF53D5181479CD55A55B1B2807ED120FA9777ACFF785225129AB1802EC247C48DB4BBC284768BDC155D77B072D686462D831491C80752E13"));
  test_xof (&nettle_shake128, /* 48 octets */
             SHEX("D8FABA1F5194C4DB5F176FABFFF856924EF627A37CD08CF55608BBA8F1E324D7C7F157298EABC4DCE7D89CE5162499F9"),
             SHEX("691033AB5C34D52ACBB6EAE6A669CFB53C39A8020A6EBE9600ACA2D9A2ACFC898A0DAC7796D46E6F304550A8894A9E1C516A1A127287D4EF74E9A403D326DABE1FC548D1AA1323C2AC40938708D7872960E11248235D2AA39CF3ED88A18F8A2FC2794174DBD4C1FB8C686B37641C7CDDC565683F0D173952D2C1DC26ED5F5B806C127114910892B1BA0B0EB07C20AEF016AC83A78C152B4C4EC41CCD8974D93E686F52CA8656DA6D85BEBDE10C4447005278630595379F57C529C4EE94B8E4100AB930206D9CCEE8811FAC2F1B5425B6F0740FBAF2F8470A9985B77865750326CD60F855F4427F6EBBAA27CDF0A0444FF278BC22A55BCA5F90A58B27D792EE6E8998E94819B673B725079C95F0EAE473F62C09D9BB1060EE9F6263950150461A13D758B8DA498284F8F355259B2B332AE9117D2A35E0E09BC8CA406C14E8799CCBF0592BFBFB40C1DA7B5A88609D71093B46A91FBE94C78A6C9CE9C81A2E189BD832DD746CE19CA3AC9F56572FA691CB506CAE417F405088BBC6425118BDBC0C99F97702029C177D8EC398DFC19A98AE3A3F86AE1A8122ADC5952B1891269C46821C1561B21E4B47F36904FF7C6814C0A08B6045400BE72EC1E647887E27AF46EA49C9EC4FF621E544390BF60429CDF1307C93A3A1C8FAE2EFB6062B9314A06FAAE8F84481A369B64234ECEA944EC6FB3CFA31258017DDB373C55A64C0F9E74F"));
  test_xof (&nettle_shake128, /* 49 octets */
             SHEX("BE9684BE70340860373C9C482BA517E899FC81BAAA12E5C6D7727975D1D41BA8BEF788CDB5CF4606C9C1C7F61AED59F97D"),
             SHEX("68E791106DA01933EC70EB650AC2F03EEF1685213522F2E88EA7F8DFAF92AABE75EA0D169D9BC11D13E294338A57295279B28B41D8A4B738CFFD629FD4D7D5D8AC7B4304E384DB31B0328391C19CA9D381E66D41816E647ACB66F312A418795C0D65FDAA6E25AA24811022C2BE529C83BE47274E8C01D4A2710D6AB8BDF095566F1B3DBDA7D4F1BB5BBD3EAFD4486AFE67B682066B8CECD447B9C2E9E9416F2CE5CA8CA1282B4AE9BBD7C842F776BFF7372591CB3EED785C82A27076470ECA15A33D5B38AE02FCCCE396326F1BBE28995C0F8DDE42C09182995F143A0F2D4427E8F2497F0B257CCBBA286C2FB7885B642494F933B584B0CA83B545FF1521F67EC8E5AE4D93FC82544FC20778CE197F3A2F64EF3D9CA583A0D821A5151F462CD70C83724F9976457EE49D3CAEDF9B6CF0AB72FE7DC0C9F41F9C5ECD12D5F2D8BD6EDB2CBE11F0462D36CF248A9552B0265E386DD44B9853F0AC63FECE4F860107903F8DB2E1F6F4017F05338FEE8467C0C40720663B54F556464F4E87F20192299EFC178BC73FBBF350782520900644265863223BB42A5ED2E416B9D31F3021EBDBA693CD56C4B17EAE16D1A10A381713AE39D5AD6A220CDA8A91ADFCFAC51C65D03910DF4AAB7F7BC1CA49CA6AD08EDFCC976DE0229AE9D379E067AA9B3FC3A6CD0DBD7D1FC63C4AC2116F9BD3CD70B94A53B71987C17F65D5E77AD5674CC16F"));
  test_xof (&nettle_shake128, /* 50 octets */
             SHEX("7E15D2B9EA74CA60F66C8DFAB377D9198B7B16DEB6A1BA0EA3C7EE2042F89D3786E779CF053C77785AA9E692F821F14A7F51"),
             SHEX("3AD90F57FCFFBB40076CB98869510BE446F8161AE4F37F57939F8084CDCDB4E64DC7F380FEF8B68CF274780A6A5FF31B1D573942E717470742E891B084E24F9441D27F7FAE86A8278E84601B9D4830AD9FAEC738FBBF86AA299653C2ECA3B233EA2D1350DE9FEAFA76FB8986010A26EDDFCF33FE976071B3C719EA15A2298D060F5AF88E3F06FFB4C70AB13886686B8643843C65BD882C924E760B9D78D705B78B2ED14980E9E89D6E99E8945B3A92B9AC282642651D94600D8E5E583A59BAE4CF6C4A38353B90894A7AFF2881BBE06464FECD3A54B2726A55FEC0FE5BB75F403DEFAB989F75DC04123F5563999D1503413412E16D671969AD0D97A6808B1957A7481D32BCCBDEEC7809A90343EE59C84CD6AD134529CEC5E42A31843BCCA3449DD3A2E075998788811A601A3661E241A88406071B90C88799F370DC83B440C9247DFFAB2955DC5EF8BAE47C1AD9157CD0F5DB61FA245D7179A8768BB3B2DA7AE28129258FB8A22F9FD4A7906432EAE33158D4AC1550E379EFEB091C883A0102F7DAEBEC2AA4EACD277A7E45DC03CC85FE952D326C3521BFF980E16BB00426D2138A9F32593CD1355EB7CCDE7FAFA16AD8638A9C16CE34920879E753C723C494E4F8B18251D20EC437FD42FB178017E7569647720D309097C982AFF3D6C54AB23DFC665023A01B3990E6A8999E7037C0405A4CAB91876BB660FAEE8AC30FDB9B"));
  test_xof (&nettle_shake128, /* 51 octets */
             SHEX("9A219BE43713BD578015E9FDA66C0F2D83CAC563B776AB9F38F3E4F7EF229CB443304FBA401EFB2BDBD7ECE939102298651C86"),
             SHEX("58B040B46002BC13D05C76E840037B3EB08DD51881CA6281595D39B8B12D1C15F9CD40D94DAD00469BF5EB73205FC649C501B57A33F4EB39A7FD76769A6EB0F84BCF985C6ACB281E4989F5EDA5E823C31E94F41E577B0069809E1C3B09F4480677D56C487CA148DD5260027BD9EF399A3F08F5F5BE869AC7A403B53532B9775DDEF6B80918BF367AF8700D6B9E75FDCC45EB405D37E9A9D7F9BFE2517FB5BA5225C72AA292AC7835A5BF4FA245E782EA67081448F31854F91F21DF99BC13C615A67ADAD644FA6336B2288F1E29C649341C6A4959E5060A99F69981A10FECC30E29CC7784153816BD228FD7859186375FD66DCC78EE4AA5E9DC0B190FB8A6CF2CE15C7F3E0B54597A559BDCE7BCFAB8FCD25EB0E0C7DF02882CA67E2F04E6266558C7F36FD0A2BED459B50FA2D883C9B86AC8E40A8E92F4099CABBEE9E7FD1282F06B3D590897BDAFD8643729B09B47544BA3F3AE28DEFC3DEE3CD6586B340835F3AD34D6E16053E2AC94B5CBD4FCBBCE2D0245421AF021E6E0C55C53584CC917D99818D4F0585AD66500AFB13C7CF90B233D01DFDDC738C325E20B63CF82A307F44C543BE58974110BDAECF43468C902429DBFB0458936F67CB59C4E4415B5D03109E88451989CCED136DBA9002752CF608423CA52483B4253D24E5B45443016DEE2D75E7EB9DE415A148C439052621AC1DF79E60CB585A4B08CA70286C77949"));
  test_xof (&nettle_shake128, /* 52 octets */
             SHEX("C8F2B693BD0D75EF99CAEBDC22ADF4088A95A3542F637203E283BBC3268780E787D68D28CC3897452F6A22AA8573CCEBF245972A"),
             SHEX("D0E4650C52BD1FF64C2F5B7853E8A906D7457B67CBEA12322B5D1D5BCB8C3EBF49CC6DFE8FB41895131956661C772D2B90DF4BD6B379715278C9E72B6FD93C3968FBFD3194C2C527D15C6311F1990DC2ABEE5B6645711251B3D39202C9D3FB147B39D1D3DA108254FF8DEADDAE6599E17C301BC18DEEF2A9BA8FD035A084D5917FB9E4E56FA6EE7731332B50CA3D2BD069F793AC6867448697395CC5147481F9F6499433D6242502EA1E6226A88068032F656FC8D509D041F1D0978EBD4F2EB347E75758A29287F5BE37F9D6C14E4934E5E16806D6DE41FF8268BCAEC27E999A7BA73B2131147FE05F9EE8A491E47A3A7C8CC93E6D2DB374BD5E091810F0D7103215726A6F80D62C19D9BEB92DDAE8C459002376F58511792CFBE7A2158EBBA922656ED88163072BF6667F121747F05B84D470193ECC8BBFAB750D162CEE7139EFECF889ACCEFB6CEC328A184F5CBAECEE51119A87CD21110FC20EE85560CC979362C12F1C7665ABE62D5ADA62DA0FF3682B32C6DA7150841205BFC43735741E07363B9CB15DAEDB03352BDFF62CF00399CC429A8A25CA1CC765AED117B72917B25CD2729CF46F92D8D3F1131AF9B6BF16226140CB57BE5C96CB2E4A369D226FA82BF0DF3FE2A4EE373434C5CA68682C0C594B783A0B2FA6604EF3828510E1C360A617588AC81BA8D1C18C23A30D245DADCD1C6893D3809D6EB305E538F39DCA"));
  test_xof (&nettle_shake128, /* 53 octets */
             SHEX("EC0F99711016C6A2A07AD80D16427506CE6F441059FD269442BAAA28C6CA037B22EEAC49D5D894C0BF66219F2C08E9D0E8AB21DE52"),
             SHEX("1EAE5DF54490647FF03985658785B5BC1480D5DDF3EB3A1FE649520E07C0174FDF1DDE9A8518879833D64A99F3255568223E2DD0C9A67BAABDF222C2A314936CD57578395288D40DD047D67ADF8289207C6C672023E25A5F66D2BAADBE2972EA32516AB753CA37AD5F436A94CFD62182B7D68EE0E7A016C8540840D1EADB3A158D88A9E7C90F551FE857C17265AC59BBA6947F24582319490817B99A591978CA945D4D0F23499DD3AA73216B7C28EEDB5C8F3C0851859D6BA9B04DC67FDB4BAE4D73BA540E66536BEFE2FFBB5DAAF91CEB627CF4869327C5D3C2CE79700795E7975A8C6F9C07B95265614DF45E50EA89DDF9FF5F2AB150629937B9197E543CA30086E829AFBDE5524DC1064AD96FFB4D83EA91CE7CA8843C4B1A9722872057B0AC9CE57E7C97D22CC06FF1F74658AA28BF398EDE9F186AF6E485D6408CA4A4564689183AD52EC0F38250CDD837D684A1E4FAE12306D066F2F1ACD56797F1DE2A9DE8095DD464BA58D5B2D58F9312F1779EB2B2ADF1773F82FBAC43B18EA5B0CC9FE2FFBD9D2AFDF61026363817B795A2590336EE1855384E2A26C7E15E8706AEF41C671F2D8E2A0EA2B0A558CB35A80952825C845875B2A907E84397E2A3D125F64C313C15DCF0CDC6637176A58FF6C0BE30315863C64A21A1FA147AF0ECEDC22FDBFA9CB892FB17D26C460B7ECEC9713E90A46499BC30CD2735FE98B13E5FDF"));
  test_xof (&nettle_shake128, /* 54 octets */
             SHEX("0DC45181337CA32A8222FE7A3BF42FC9F89744259CFF653504D6051FE84B1A7FFD20CB47D4696CE212A686BB9BE9A8AB1C697B6D6A33"),
             SHEX("A8A37D45AACB607C3A2DD97A43E22E226F11EAA21DAFFB8ECE97E42DF057DDC9459EAE2AD1C04ABD300833E6D63600320EE9E368CD2DEB5E38ED7726C583FC9EB411C193C42DD8A1D5577ACA20B889A5382C0F1A59256BEE99AA09A6B2CC7296A8DE514CE14173E683268746738C2F8EB29D6F51AC434BF51872BE626241E43183F6C7621479B62739B975B6F25E63D9668D3198E82E66E9E5FB871DF0AD3A06DA385C4A7FC914CB371DC1EAA929583F14DB910C8883C7B90635E5D3FB5A3652E98AB62AC884C885C4606E8F453B89BADB15F7B9DE49EFBDBD210859AFC079FD2A1FD82ED87949AF0906B5E75E561102739D4CC58AC9420694F5C043DDC463E0438E3851AB841CE31D994E1FE378EF75E7F979B7801A284AA8BE26222B2BAF06857E8ED6765C9C040798B2A6520E56877C1E4297AA43A860F734A6667C1613C0EA76EB70435BD64D34B11728113BC5CA24734A41F3203641D931CFC74E46F3D23037E3878740A6B152A77BB10791E09420B288D89E06DBD6B882913589D82DCD808918AC853ADE79B7B8104637F8C73940890AB44E31DEA7830F2BC08C26EBA38E3B1B198650A9191F366D90BFE1A4CAA7E03DB8357BD8258F13E5BE051E638689E99BFA6DBB5E2A623BA8B2107AB7A4C7551918A1C023F1FE7DD5564861FAFF0060E30C14941379117A3D1845842D935C4072B28913A3C89D1A63D4658048B8"));
  test_xof (&nettle_shake128, /* 55 octets */
             SHEX("DE286BA4206E8B005714F80FB1CDFAEBDE91D29F84603E4A3EBC04686F99A46C9E880B96C574825582E8812A26E5A857FFC6579F63742F"),
             SHEX("140DFA98962AB4D69B6063B0252DE5E4D65E4C00A9BE1F8EDE781A54011D1DB01366193A4235F6E7D684B847B2DC4A0A5239589B4C0266817DC37D850E5B23A40B50A6D9D1BAE6E64FDA20A3D27F933AE866F9C8F00D455D97BC71D5F5A7C9778CC8FE1596191C2B1882DCEBCD37A9A8C3605359E886942B04EBBCAC83FE730B84BA3B4B3022E0B1A1110EB4D23D2E2D05D77C230E36A1ACCB885C0FAAA312B7F975CF8DA5D6B118ED773C5F1C75366E16A1AC75BE309EB328DA88C9FEDD1A13C84E2F856EB0DBB249B3C166F9CFE917A93A3EF624DD4782B1F84B3E2ED70E493615D6BA9928886A59617A35AEFED5B46D773FD5B22CBA302205CCD97426B7095872BBC7D1932F6157BD3FC9D4A2182FBBCF4BEAB71A026AA212BD8CCC81EEE265E4F53390740E3E9B1E37CA3A7C3A4FA824D0AFAAAB7CC89C35756D8F80896E8686260CCD94857870490AA8AEE2FD3842C62A256107A7A06F87894EE70D218B6166EEB67B6F711D421C45FF2CCC903DD1683D12EB5EBB9464EC0B3C92CB62EA6CBF37591B61E6BE15FCEB0C1731B42D8D3587762167C3A43F62FF57D58F2A2247CDF43CB756F6A25FCE0084D74BAA73A8CE12E41953C03C2A9FF94468A69C03C2B5F6B85A2F16397326C7DD562E90F8439E720C1AC0B4273A7DDCB71581DB1D6A57BA4E028BB96462DC54CEC9EBE36C704C44E5F60D04B993454DAF405795E2"));
  test_xof (&nettle_shake128, /* 56 octets */
             SHEX("EEBCC18057252CBF3F9C070F1A73213356D5D4BC19AC2A411EC8CDEEE7A571E2E20EAF61FD0C33A0FFEB297DDB77A97F0A415347DB66BCAF"),
             SHEX("F0FC5662D4D28B3235E49CFF708D90BCAB50764C4081E2F30F0479573ED4FCADA1A85443D1369E09CAAD970570A4CE3D8EAB46CCCE3E48DF5F26C68C7485911286DDB1B369D13079498653B21E3BA7FBA9007500BE927490336082C1D7FA5B78BB741C459177C05E6294DEE054FBE526D983B496E731B3379DE525493733305DA1E1E09612CD179EFAB1E6E67AFCCF60776E319651CE1A1FC66AC945BEC6C45A99A5BD7763ABA920314F04E67EE193484E54461C68E67A922DAF9205FAF10F4653541F2A5CAAE6D28449CC919341CE63939A0433A9DBD307020E80B65AB7B2AB39A9C29D167DCFC05624860B50B0C3FC3986B020B48F5FCF00C3A6BD8FFBEC6A34ADD7E97B106B4C1DAA9E5EF91B797C55DE36B88E9AC19B301979BEBF36FB86F4F72A6560A4125BC1F657769F89CFB1F3A29B2A0287072740E52A60C1C898EA42CD9FCFCB3987A015DEDD297F8382D639CD7118CA85255D1F98ABF1F5D94649A161082D7D5944B2B4EFE4975D5EA38138B91AB5B61A790349D5463D2DD278196364115C18B95F916EEEBD71A21DE18C9AEB22566D3AEF009A84AB5479222EF8F37DD29C7B22D0F1627C25EE467607F86DEBD8267A493CE5C361A6F6126DBC8263B6FA5D5C9B9F8AC9FDDA42FAD433BC9A76A218C1134C1CB5F03F3F7387E245089A5B18C96DF31211B5983214B64FF3E1D90BC234A2F5EF7E993C8F4E5FA9BE"));
  test_xof (&nettle_shake128, /* 57 octets */
             SHEX("416B5CDC9FE951BD361BD7ABFC120A5054758EBA88FDD68FD84E39D3B09AC25497D36B43CBE7B85A6A3CEBDA8DB4E5549C3EE51BB6FCB6AC1E"),
             SHEX("AAC37FF9FA6F64049B4D72457C87BEE3FFC23044ABE0F1D121392401B60E3FEC089D4AE3340961915BBC79DCA9337C4A02EE99AB932B346375169057E8BC731FAA49422CF5296122471580AE4263B6C01D8BF9C7F736A4CB652AA5E3521BDFCE2CE11332E53236EE4F613B2800C28E5907D2C3214F6071611496F80594193DCC09AFCE3EAB1769A5FA2DF9ABE16AB019759A07154D21C3365DC20BDD7C9ECA82045092EA5405B56861BBA693F322871BDD64E2E41985775ED7A3F1095C74EBDCA15B9EB9F2B86234967DB868A082A146A439746901EBA5DABDE0F133D99C89A4D6F769D1C6FB1758B3CB1E8986A025F5AC2A8D6F1A9AEB66DD3C41D41221706A4824CFAC6A6DB69E3D157244294B9D25B59274B1FDF992988F3F18B91FD78767ECCD352B7CCD335F96D3B3913902C885C470C2F7D2B00A92269709863E77D3FB4EA1C80CB58C9CFD996E0C6FB85F3BB5308B95E8B43555CCD4B589829DA442BBCBF0A5C7B9B24E3DE846E9E8433F1A4A326ABC7AF83A5BBCFE9B4BD848414B2560C58B3E9A3CAD48F860BA46A5111D0AB629B91369A964BF22B5C2C6DC5C3A521DE18DFCC43BB2E641AFC466B75FC3529BC4B99B843A809773DD2130452E61EC162FCCCB0A94B3AB0848EBD59C9FC25E84349CD2154E5E326CF2B97071E7C47F0F574D11729EEDADB3D0DD0FC69A7F7224891EFABE76942808D6459217F06A7D"));
  test_xof (&nettle_shake128, /* 58 octets */
             SHEX("5C5FAF66F32E0F8311C32E8DA8284A4ED60891A5A7E50FB2956B3CBAA79FC66CA376460E100415401FC2B8518C64502F187EA14BFC9503759705"),
             SHEX("3930D75F8AEC436928F62AC548FD6B215C2E6FC22E918690F0CD699A0CB1AC218BE324232295546CB3C421E939222827890A8FC563F6DC455B7D0CAF24C7B1BE44256DE645EBFD281F263C9006F78A69E8939809538A74E646E75C4D23CD06EF68B617E6C5D36C85348373FCFCCF19AA3BD69E18CA670091D2B50B0C2A3AAAE21FBEFF894A189512F4EEF17258AF048D073179F450D366295F609567EEA83CB6A2E2B8170F69E739DB47874E1F249AC9595442571D100CED32C151ED4399F5AEC97875142EC1E8B748A9FA5C3EDE5548B875E60A224D9830419E233FC9A88CF0F937566EB289591916C1D159BD68B380F784BD3E6A00D4C785694F38FE3D8FBFDE5063B91FA397AFCA0731B97294269EF8157B531B08E15CC52AC8E5E768E6DD3905BB9E1B462481AFA989441CF0780CE6B1926D71940D4EC82521A4347992B37AF9970F47658B05AB54A617275F51FD723A35E72F9F965BA4330569573A58A0021E1DAB0AF22EA03E1BFE1E88EB2D2B97E5857793FE7B63371F3569C77A77E331924DCFA9FB8072F4EAF3A78FB6462A5C68F26D034BFC0B3BC8ADA4992C66B4C8B2F0489A446B26333146B4420CF2B5AEB41934B3CC05B7ACE62AF74997B797C82845F4C4D7FED571879D255EB34482B86383E924D45B4C9EF52E76FF236A0DAA1C9F6D342BFF82305FCC88F8B13FBB724A9FE732CB33A6D53D95753FB350C4"));
  test_xof (&nettle_shake128, /* 59 octets */
             SHEX("7167E1E02BE1A7CA69D788666F823AE4EEF39271F3C26A5CF7CEE05BCA83161066DC2E217B330DF821103799DF6D74810EED363ADC4AB99F36046A"),
             SHEX("1C8C8643B1FA6B0D217D21E542850127895DD4E455DFD136C4E221787A8F3CAF76478BDE517C7DC3EDB97C8AD824292EDF671BC4874044B96A109BB5821858E544090E892B3EEACB739AF35EB05DCE51B131394C9972247F1C0D7D335D18B9355509BBE94C4DEBB0C1B18614CF3FA8119B70FCF5B59E37673B470CAB14302D805BC32352B340FB71F7E824EFDF2732EF3D74998A5AD70898115C9CBFF3D5067EF05D859FF932304A5AC5F6AFC46C879F0BEF369ECD7B5065ACB19C655F7FC2DEA4DD7AD38A2AE841A67EC1EB4869EC285B5A517C930350D7CBB9C61CDBA188553B9D4EF140DD1A8ED312822E79EA68662BBB1AC3643AA56DE5991F5F128F036B32023CE34B7A7AF7CC43D6B67DEADA4849BFFEC361699AC7175311195822382051BBE20565552014D6A4343CD94F7F4AA705E1D5388AFE2D96BBE8010FA0E82305B94B70AD954BD5F6FBDEAF090AC510B75CC91DB450EB522183371D6B9D09D3FCA171FD10174ACF3FC47D915619DC0D06D8567C4E8875D0216DE9C7AC1E7D1758DD59573273886897382CE8E77552961D51ED0B87FFCA38BDB6734A23FE0C9A403ABB4FED73DCD504289E7E14906A10B121A57B94C2C37542EA833C55C1130AD70F753C9A368F7C7D1B51450BBBE794F2824713153F83FE6AF09545ED91F0E39FA732205025A49E1538FF7DAAED5B72EEC88862DC0213A012239DEBD24FCB07"));
  test_xof (&nettle_shake128, /* 60 octets */
             SHEX("2FDA311DBBA27321C5329510FAE6948F03210B76D43E7448D1689A063877B6D14C4F6D0EAA96C150051371F7DD8A4119F7DA5C483CC3E6723C01FB7D"),
             SHEX("1DA38DF421AADB3DA692F4987E171113E15D3685DB1017EBBFD07BB68DB236903F3CD3EC679B6BC9004072E90ECBD0FE0F2FC4F5531D24FA6F25BC6EAF4B545A2149517CE19971E3AFB866DAA702DCB94982446E7AF77418164C5E92EDB14D98EA0DC32A3E0BFEA4E71E9C8F58A751578200772A1E0460083021552FEDA426AEC69346E0FD31FAE18B894B67DC38D7E54B662CE051A8453C1DB20F55095C177F5883625D40BC4B49965C48C4A1188954C50DFAE423A12315DEEE0FEB3548AA9752432CD6857AAF42BF80472178D0714D777135898F71D838058F9F31E6AC92D819C44775C2EDE28038E565DC33F1A157D8715AE67AB993179469D6C98E8574A7D8FF2A5F12353F18544E2D623AA42AF85159438F247AA947922A5AAF98BD562AE54599783853CBFA5CB67EB1FC36CFBE5AF1C18C99D867CB02966D90EA8ED648831A2228657972642A17DD34FD43C11CD7D407CAC86E20D8816E69763B60294C19497C151620A2B93B3B3E87639C233C17B9B865E920F6EAB6C72218A182DD46177A7AD18E3775FC0A5A0EC956B5E92D8B48A22A6364006BFD2843B3EA4CB63B8C0AC10970853D196918DFD513D4E1A1558EBD3D4C60BA253078417304A50851BCB6E41B5758A08BE338D62B0021851A42962313C11BAAA95E434F6329FD5D6C85D67A665A675CA527C555C77C7435786BDB8D9642050931C57CB79AEE5973AA"));
  test_xof (&nettle_shake128, /* 61 octets */
             SHEX("95D1474A5AAB5D2422ACA6E481187833A6212BD2D0F91451A67DD786DFC91DFED51B35F47E1DEB8A8AB4B9CB67B70179CC26F553AE7B569969CE151B8D"),
             SHEX("660BA471FA9414C985A47B1941804295ADAE88BC1BA145DEFB407AD66F193BE41EEE4E0F71D175F6D1D65C4B227A899E0D282AC714E50E3002AAF8696B00A736A60C9CCF94D15E46DB68228BA174DAE1575138D3F9EB76CB16FB09A50A6F7024206D4A02277D2D0B53F189CD8E45E85F4C0FADFA235C963A9A63743D88FADFA6CB99BF47F32488B4E009B47F7BF59A01BD92C0B6E771E8784B67A0AEED96B29E6A1B110B868B144105B85C11CC01C589A89342DD456E154A0D2A638FEFEB03058CCB301FD551294251D32B3F03BD62AC5ECF5A249940C2401F751A9FCF036A384ED26F569D227235CB47673F7003C4EFB36676044D09CD586A54D744AB4922BB337D4436C4BC76CF4A478D1CFF7F7D97EE2DAE912C5BEC910FEF7BAFF0BC9E32DF5ACE8DAB30840A31CA192E258A4141275534361977D2FC22EB1C1BB62A21B893A65B16C45ADDD3032F2CCAE38BEB6DE32E1B5FC70214E2BE1B7BCDECAFD744F82EDCEC99F92D46AA66044EFFF26EF3BCE081A19F1B7392C68D2414D92D426D78B4BBED4AA32D05BEA547A8DCCA688CEF6B0BC754FAE7C13915B97478BFA72EBD409FBA5AE48CC89B37918F04C068AA1883F2A950675761120989A9F7821799F5EB64C07299ADE85E1DC1BB488BCA7B96BA7F04D49CAB2B41803F17BB94CBF04B641E140D8686EEDCE954ED1A712BDE789C31F02F721F467E167C3E9B035687"));
  test_xof (&nettle_shake128, /* 62 octets */
             SHEX("C71BD7941F41DF044A2927A8FF55B4B467C33D089F0988AA253D294ADDBDB32530C0D4208B10D9959823F0C0F0734684006DF79F7099870F6BF53211A88D"),
             SHEX("9D027F5434C6F6EC15B26BF3D61F31D81D46C0FF426D4209E0FD3F9A3E01D0AF5BAFEA98F9E3FB66E8390AA2ED009C0D6DBA82B3E36CED600FE79C0F50EB1C1F7530F20D51782C13A79AB1D0BC655112D745043911B5A9524E6EC88122CDDAF702C34FA8840E8785A7046B6794E623526826B471CD9F62CF9AF02E398C077C7F8608730578AD22A44CDA39DF8F101CCF21052C52FC12967DDFC77C153D8068A2B0B783376FE56B1DACD9A0B3E87645132BEC0F3C2C9A521DEAC286750C2C42B4A29B29144FB1B0F5CC997AD57D04BEA965343843788DABA5600A7AC5403D649097EB402205CE4BB12EF836FD4B7CEFF704F8A2BACA5D6381504793F1C6B47D15DFA45F749A28C7FC16C695A0BEF69437AD3F8C8D3DA5386A6D9752529EBCD84CF92D641A4CA1755177BF7E16CA144E270E4191EA26BFB144528521D32DFA8319FE37CD026E5F707F81DEE84327CD0B74C05CF69543E94848CE616FC118F73206B132F0257925B8D93DB60FA6E034D27A65EA0B2EB18056CC4690BE78D9BDCB4DEB61FE5089C58AD8CD08DB2AC9A2C2B0811EF71309DB1831D9ED1032A1267D3152E4ADBF2042EA48069D0519FC5B03A5375EB355BCC327766B4B3559737DD4EACDBDF5B28F121E7C13EEE9A8C10E66F22CC80EC40AE72CEB0B2821BB7215CC9E645B65E766076AB78D04771DA21B501987F8BAD685C6EEE40C9F60605DA631CD"));
  test_xof (&nettle_shake128, /* 63 octets */
             SHEX("F57C64006D9EA761892E145C99DF1B24640883DA79D9ED5262859DCDA8C3C32E05B03D984F1AB4A230242AB6B78D368DC5AAA1E6D3498D53371E84B0C1D4BA"),
             SHEX("3951B52498F3C2E30411A6D89C55340A73066B36B1AE55D9C95142FA1EE13EE8B7296074D82E44D752EE29705606D823809786DF5A2312F8922E4C50FF0D7297B864FCC3BD81311D69D2657169773C41A8F7F0E62E076F5C0055DA042DDE07D82172BFFEFBBB031FB715546CA47903895DCB4807B9C435D16F90C41AB7836A53454E089A25951545F0B39A849F329AE9568A3640F08329B0422807AB15461F6C67E1A1E5B8439F4D0613C5B4A8CE10AC1DB59DA2CFDC1F8FA78C60E88A7E6C0F2488AFEBE16BBCA680EE65BB53421E29C8238B7F0D1387C60EAC50ECB43AF4FF075520199BEE31641CA433F35D62B96FE477C304265002440018E4A4C656679189ECA635FE4090724784F56DB38563EF05A8EBDA4271ADCE14B8644BAF09B83574DDB6803997AC473BF537F0F93A2FBBD0E0C75C387F21EDB2EEBE0088CE8886E6FA9952DB209BC16E624A72DA33878538549AA8F955CABA17947FBEB6A304EB9B1B494A197814EE4281A74334F71780455A352EE1586BF298361C40140C3B6A384167632D7EFD7F3FD327E18769DD481444400B669BE07EDDFD2783962751896B703B3E196EDC111411DB17531B96D0557066DD6D2FA3EABE0E4735837D1DA7BEA7543D6B591E55B2D95D46F822CFAB994770F8F51A58BCFC10B4A7DC9C8A645937F5AFF66A59687457CA35C42DC2C9D75EE0AAFD9BEF334D12DC33BC0881A9"));
  test_xof (&nettle_shake128, /* 64 octets */
             SHEX("E926AE8B0AF6E53176DBFFCC2A6B88C6BD765F939D3D178A9BDE9EF3AA131C61E31C1E42CDFAF4B4DCDE579A37E150EFBEF5555B4C1CB40439D835A724E2FAE7"),
             SHEX("B21B2ED68FA53BF4EDFB8EE5282964C8FC370A702A229D3C4BF87DFA5279B1AE2FBC85A00067F8407F586B52B5F1AAF97716025EC2E292206E18570AD50CB51E072A7A06781EB9E1B1B23C576C689DEEC4DA40513361B413F513A0EC21491F0E6EFB792688519C1E6B0760B3E1DE34890C2BBB5C3A44DAC9B0B0EF2B1B4CC4E30DA7F93F52D656712EB12DFCF30C80273FE5A8F42B909AA2C9E1E9C82199418AB858E1B8897AD632C0C030C9F56776378A86DF63A5CC284ADAFCEF50DF97847DE927F296E63933FC174EB3009C5580CFE39B77BF359A865DED988D3BB631256DE6EB736C637693DC5798429498609A6201C6E9CACD7782DEE77B3BD4728BA32A0DA88A14212BD204292B6EEFD0C7EC1814B7002D8FCB08975D4A9C62100F94404CFD32EBF49407BA4C6697CD65D7C40951CE7774234C4A793265CAE9037020E35D38C9C016B15D214550A94F6B245DC4F5B9B1C8298D683AC59ACBF177F266F1818628B4174D66129C057B1336F718F67D5CE7A3627E31D8E64CAB4BAA5E89B7B2971B4C82D23C36305B01D30F830BAEB2CACA3BAEDA18F912E4DBCB1F5CDCFA9F562E3BE2B01826EEF5CF0D80DA116765E0D467352264C8A9C1242F74D98D35F77472CA4DF8234E76F7600B6052E7516C514695BBCFC1793A9C9AF512E1D2EDD7490D8BE500710B305F9F54FFE4D87B50776A29C222B09F1913E71F63AEDB32"));
  test_xof (&nettle_shake128, /* 65 octets */
             SHEX("16E8B3D8F988E9BB04DE9C96F2627811C973CE4A5296B4772CA3EEFEB80A652BDF21F50DF79F32DB23F9F73D393B2D57D9A0297F7A2F2E79CFDA39FA393DF1AC00"),
             SHEX("0A7A5CBFC9ED4A55B8C8CFE99E9D71D4EC34AA1C4FE8A69A39DDD2C0E1923A3D4D7EEB8769709172866F0F93D96504307700D75F4922EB0CC199ACACF31E8BDBFA166342CD7294A723608528732BC85D7AC57307C37D07AC03D99009C2052DE7536B9049B0750A5186855BF41FC58560EEBB743F6572C102F21D3A92F8908434519D8F797DB47CFB93BBE9A567ABA819FA035F64E2A718497FD815968F70869066AE8DB9314F006466FBC53660BCA5CC65B6586CD75E58427DC7B3F7D1A7F0A5804775234C8D0907C5906AEB4EC4DF01E081695897B324A97485C22F694E48EEDDC010D198E5DC4C65A24F62D6FE7EE0E60B2E4F79A6BF349F9612303CB272C0F1D0A952312F151EFBDE16D17D3B2F2BB9729F444DB0B1B993A6CD2EA973D430291D276CB479D8F07B9FE022F384743EB3C1593F062994D2853AEC6267E68C724F959ED98D9DDC0919F53DBD54E28BBB28AE5F0EABBC36A81E25A02174C14387E8C1695F97DE66C74BA20CC8EC1EB5D3DAFD147A894AAAD4B20DF7A06EB9B676846133868F02EAB7102E6D7EA8BBA678B0DBBF1F6D6D6AD0FA43412A8A53C6D814BA26388BEF08F89042E573FACAC5588E4B5C4E2806542F2E68148F4B7CFF1EC3DA21F633372B567005F2CBE81DDDD450E597FAB577A1935281F758193CEF43961C380B62F0CFD084FB4FF19045CEB3422C01E5CCF9EAE9C3B300A117F1D81D"));
  test_xof (&nettle_shake128, /* 66 octets */
             SHEX("FC424EEB27C18A11C01F39C555D8B78A805B88DBA1DC2A42ED5E2C0EC737FF68B2456D80EB85E11714FA3F8EABFB906D3C17964CB4F5E76B29C1765DB03D91BE37FC"),
             SHEX("0F001A78A98382078267D5DE881F8717B55A31C1781473BA089C72CDB201CE93B943687E3863BC24BE5D255A4C02E43EB498B9EE4D8A0CB4778A742051DE21C4FC7745525ABF188F38B7B46DB5DD7AB30906F7A31E0B030DDA800C7766EB48E0F4B12C474C3535441985F40376EDB99FC2A0BB2BAA9ADAEC49C1F5FD3A604EE8DB5D0C51AF7EDD29306686B3795653E2EBABD2E9118B789868A7F821FB5BC5E4CF732456448417F43F6027337B4E1D481D65D239DD15EB66F5C6C289EE85E065FCA3DAB1126E4522DFE9A1B428310C7DF543BED5D4439E87549AB0BE6ACA17180714D2636FBAD2F7BAFDBB1961764F48518EA1ECDBA3328DE8A6684730E747174345B77052E8CCE9765ABAD9F7B3AC4E8C2155FA5AF14FBEC002B20BD2EE417F6CA5EB8B95087C34EA52C8D1DC95453F6744272BBD1F14A1AFFEBE475EB2D20CC26C5893F71995B63E725E7B02FA02B3145944D9D8A32F898DFE1F7FD7A73ED1D67DDE341C4F95B603597F74269835762A1F348E10BF5FF15B7A325077AA6DA99A8B92885A344AF35D9FE34A5B5549D37EC1C536D7B71962E1A54E2601E0F0EF19A2F81151A477D35F36374E7F44E00B25C2F7C4FD804490BBB68BC755F19EB25F04FF370107237153E50BB1F8867917CBB70332B41441B6434458C848970BBE860A07DCCB6E6F9C076998B54AD88D500634948A73A3478AFB1CBA9FBB399D39"));
  test_xof (&nettle_shake128, /* 67 octets */
             SHEX("ABE3472B54E72734BDBA7D9158736464251C4F21B33FBBC92D7FAC9A35C4E3322FF01D2380CBAA4EF8FB07D21A2128B7B9F5B6D9F34E13F39C7FFC2E72E47888599BA5"),
             SHEX("D442B526EE0DE0A7977AC04F9C822EF3369CD2CEDACC05ABEBC71128F0C0A2F8D5C2566CCDFCCEE773AFDD2C0F8AAE2E59B56995F219A9B06B638DD91D0C2AC8D11ADA3CC491A5985255716C75B474FBFE4DF204B75176468511807478FF873F17225D022A78141B962E1A4EDC7F93BDE19DC7D804C11C49942A54CE606322693845C11EB0E20E60AC7A15287848C37EACEED986F98476AC62A23E366804EBD5690856D69E82CAB976A79931FD5058F0562B006E822087DD8D277338A6E555C54A536D29831D6587B4796179A6F45C89DC9E85DD3067ADF043E3CC90E96391D60145DB31BE7778DF6402D7808F46E42C8BBE987591EDF593CE50515D4DFD647AFE7A65579E894EC6951C2F03CCB4FB4F4F212F3D87E953D4BF6093E4D390260791924EBC438B56693ECB11BEB8B10B85CD56D39A95BFA7924B9F78E181337E550AFFE40CFE99C709994EB9F81BA4E14C6663733477320EE3A62955975E5F3F25EB3922DDB67F0A312D916B8FF2C14493981C7A6FAE0E14F222F194B674173C051EDB795DED1041073BFAF62B76D3623D065F797A82BA490CECEE1136C08193D46FF6BB06B414C8A861A4604A4CBC87CC652EAAE3D648E4D2CEDEA13344AC22CBD4DAC4AC9D6662D3694B38BF12169320B5249021C85A7477E2587DF9617D7D44E09BF7FDAED71E3D5818DA093E6D019243A17D9F32A5046FC6BA19EDB0AE6349"));
  test_xof (&nettle_shake128, /* 68 octets */
             SHEX("36F9F0A65F2CA498D739B944D6EFF3DA5EBBA57E7D9C41598A2B0E4380F3CF4B479EC2348D015FFE6256273511154AFCF3B4B4BF09D6C4744FDD0F62D75079D440706B05"),
             SHEX("6EC28683A74038F869A23B21669F27B746A1AC2BD41E3BBAED1CFDCF79D7D1A9499F2AD4701F64830F14C27FEC0CF645E553B3D76D8512B879D2D19E37D93B0E9357160FFC00F20CD96FC1AA520B0870F853B3241B687549DC2D1F119198CFFAE38402CE4CC77A973CF49F19A52EF7232E39B5638C680D44334762657397CEE9DC65FA37CF9FA1F902AB290F8CE4CEC82ABA455A794B576BE404ED24BD50B9A4923D04C3F19CB812E00D6D9DCA00A906EC0F3460F14D3CCE6CFEA1C78E8400C0FC1361A4A60810BF51AFA3CA21B18C7DF5FDFBBDD8FC1671D21CEB404F9DD37008419CDFEFBF1D57F7AB7665BAEFD500C38A5AB04ED47E4B3256B0B015D0511657290418D5DFECCED4AA47FA14A9A0E9766F3123C80B65FDC07CBED2C8B0753F2B20EE271D11D28FEE068BDD3C4F449973463C17F21F60A5E59BE4FB3A3490ED87AA1709A62643E65FF82F2980BF2460FABCA0719E575614A1211D256C59F86ABBCA382B36F970C489660C484CEAB850A13011FAA07C93D2FC2E8E447A997B0BFC286769017189A68D7DE05CDDF696EB93FB8792F25B2979AEDDAEA7A935589BE4825E83E9BB302C3A6E88D83BBAD06BA6BA274A672CED23276A499DAD01555D85C620B6FC64C62282C545195EEE61C8A127EA8994AD5A706A2DC17AE5C6F165B477655D067267D88374F674BE12F58A35DA878346BD48351175EA72A0CB5E40"));
  test_xof (&nettle_shake128, /* 69 octets */
             SHEX("ABC87763CAE1CA98BD8C5B82CABA54AC83286F87E9610128AE4DE68AC95DF5E329C360717BD349F26B872528492CA7C94C2C1E1EF56B74DBB65C2AC351981FDB31D06C77A4"),
             SHEX("FF0DB45E2E8959375D3915DBB45FC32DC6CFE7D2BB227647F6877619BB6884D8BBF9668F407784739D34975576B99AB1325716C2B83212DBE9E5E33B5A6EC68FD75015124F1DA731F519F983CC783F176286228E59068AEC6241B24808406CD60F67D5A1AB7083D7854984A1F7030B7D700D5D48EEFC6BE0AEA46C8022256866CECDA897DC22D4A18B28126444D956B93190D7A060CEBC2E8BE65D90CF7837A90DFC9ACF4BE4193A892DBA8B3A068E7BD20D12C47A95E0482A2CD4252A34E1F13BCBCEB9C0A12E719A27E2759FDE1B2F6E75A0A38CB80EB44C023FB0EBC33BC8290513D95C431D72450951EC1A86A89D59D435CD3D4B4E069556A00A443F045224D46A8DBA8CD7E8BBC89A21D3CA024C4EC96BC7169E6AFFBB05DEF58A0F2535997F345C8660451E7CBCD83C622EDD0D8A8F30D8377BB31BEDEFB546C989231C53447B7817AC3F822D596B5C746D5839D6CC8F6D45C48281357E87312C33F62EC568DB4AEF7F194DE67981D67B2B6659268556EE38DF95FEC4BC340A3BB9539EB6143BFDDC12BD233BFFF5FE0743DA3DC89F57DD159A475D6D8882BADE66E4C93D6950ED026C747968F84DED68284F5F02A9F7549B88F244FDC261B34BD66CF2FF5A6925565A08BC8A5F8828775A5A35147C57CF327C49CFEA4797CC130D1E8AA278256FCBE416420FB7683E6D0A5B1466A586E98C144F4CD5FB277C1E59B190"));
  test_xof (&nettle_shake128, /* 70 octets */
             SHEX("94F7CA8E1A54234C6D53CC734BB3D3150C8BA8C5F880EAB8D25FED13793A9701EBE320509286FD8E422E931D99C98DA4DF7E70AE447BAB8CFFD92382D8A77760A259FC4FBD72"),
             SHEX("5C00245176070CD2F23F1C5E15B21A9161BEBA24BA71D989BED890782ED39EB857E325AFA269ADB03FF1AEF7D26EE524D6CE821D2C842C43D0FAF47B602E55B7960FB0A9638E6DA8272DAB5B8DB784DAAC730BF0394D50A59C87B422080191769C21BD08A32AF4C263B09D61A13D0EC26B6D1D63A2EFA5B4FB9CD4214EA76B2C915F20E242D104745EA9AF73251F2CCC9E78AC8FD3C5EAE80CA665A87C517EA620CE5217C215AE38CC43F2756D9331B0359FDEA9CEF8FBBFD3606AA5FBE2D5BD31411A857B3C8E79A2FF1B00E3EF56BC25D0D83958B64DCDC095241837E333E58E4DA64F3B5B57308B165924B4C5F70567779248EF4400BE3E80F699FB3B6E40C36BADC2BFB504BC777733BA9B59659F8661D57B3543BD26101FC3DFD6CCF430036776DF0F2979B03DFF7E6B2E98F161E050A38F71EE438088D953EF72D5651B5674EA6FFD3440AF8AC94D5A808EE426959131E460C858A77C4690801B5B3A28C6BF555F02B5C60DE30C4301363BD6322A289F36B30E00E152627FF7485F032B61041D4C1EB872365B8BD22F1A6308F5CC56D4707752DC3E32315C8559A3B8083D2F743779A3EAB10394F8DF71A878CCE871E0835B57DE865CF16AD2D2BD1E5E5705F68C6BAEE103C785E43F3582F591DA55077C6B9EFA8F93300822C6EAAC9B5A71FB676C0E73928ADE3102B61284C676C140F33A17FAB48002394EB36265D1"));
  test_xof (&nettle_shake128, /* 71 octets */
             SHEX("13BD2811F6ED2B6F04FF3895ACEED7BEF8DCD45EB121791BC194A0F806206BFFC3B9281C2B308B1A729CE008119DD3066E9378ACDCC50A98A82E20738800B6CDDBE5FE9694AD6D"),
             SHEX("812C14B78AEEE6820EA1CF8E0F3C61AE7CBD2AE493017ADE0756FBC57545830910DDC27F3183DF48C1E1D6D1A9F3DE220F17402830AC73F77AD2977BDEF013C739CA0E9EC11AC4FBEEF624F0FBBC534F0467F09E775BDBB8ACBCAC8B3DC6ECFA92F35EBD729A1BC90236F57AD32FFB46C055C93412D33CF612059C4D5BEDB6E45807A7CE2E5029C0594CBFAECEC3F9178EB01D60ACB5A72D5DBABD6F95691A1F949E4DA7B48AA78C1BEE810B175218C5902753B88253FEF57681E60FDA1851A18EC1F3953CDAA6B9C32D085C7EF9BEED9598354CEA388F3D8EEE372312168B590ACD0C330816949211D8CEAEA737EB87BF9E6D4929D83B87E287C887E32B66FAF1540B64C53593BCE073AAE4BBF711FDC7AF6E7CB0FE938D2F87643FC4DAB4A7D4154015E82422C14600E44E41D005DB80CBB4C297335EE0E616E32D7E9BA5BB0D88FBB05B4D5799DF5462E4544D3883CC2C4850FBE5B43613841B0D515093D246B714E96656E1F6227341E76051D5C9C52730D66D075DF0A9EDEED5D8E878A1FDE3AD2C962608CE17655AE769C484E676904A29F22A3B4524DE2894AA633DF9F65CFFA329AF056E9BE4E67E7EBB786AC3541C00A792B4217E5EA2E51FBD979CC8C6E40F59B6A665F1BE6E6F2FF4D1D48250BE44572FC5B731707B8611E8CD7679037D565BA7B066781437FADBB47F68AE752E879286A25020F8E00FD7E62AF6"));
  test_xof (&nettle_shake128, /* 72 octets */
             SHEX("1EED9CBA179A009EC2EC5508773DD305477CA117E6D569E66B5F64C6BC64801CE25A8424CE4A26D575B8A6FB10EAD3FD1992EDDDEEC2EBE7150DC98F63ADC3237EF57B91397AA8A7"),
             SHEX("CAFA188EBD8CDF392EB8E67776D6F574F72615ABBB5DA35999507460FB1D357938F521BB18699510D353D3B42EF72E704EE769C02A1EE7E366871F28B25F798725CA3FD147F7DA2B49329FC5A00E486A19E9465B782B1176B84465A96F083957F3D55512F6CEF4AF5E891108B468326C1051EE1519C21051353F1EAA2AD39A32E5ADFD30026D4B56A2498309C27B03A195FF5A7CA27E5184E322B9B783C44C4E0842F6695BD89D5648BE09B927990DC3980E7F818392380840E1C4310EA67ABF7285CFE20C50933A92E74FEF83AFA6B6CD6A372AAFF36DC582040B34E8521285EE9A67235D016CAFC9478B80D7B9778CAF5DA4E7D1865BDFB903DEDD7FB8477F007E60AF960BC04A81E77A74AD25641FDB9D81AEC87634D692E445807C0ACE094F99DB07FFA01C2DA7517D12C830973D425839722EC0C5A2DF093F01EB9F5A6D2E075CAEA8C7AA69BC899463D0E55C81596907F1B0F81BAEAE734B24FAE3D99984D5E39EB4822C4ADEEA640228D72387656938CFE5DB953C1ED993FD9705CA9C9C70E6DF0075392F965682BC163BD85911C99511896BAFC35D9378B12E00628EEF3EA11E908F92498C90314C55698A6B9C6D8FD88AB4D7250166FE57E9C106FA674619C1D53CB3DE576806742FA538B5653BE7D0C08282A0159E7B4F3273C4CBE1735F3B07AA0004CD5C7F763983CDE554AD9926FCFF1D1AC48B9777C7266C8D"));
  test_xof (&nettle_shake128, /* 73 octets */
             SHEX("BA5B67B5EC3A3FFAE2C19DD8176A2EF75C0CD903725D45C9CB7009A900C0B0CA7A2967A95AE68269A6DBF8466C7B6844A1D608AC661F7EFF00538E323DB5F2C644B78B2D48DE1A08AA"),
             SHEX("105ADA0018016331E288A355DACA7426116C7C0AF86060A1A652AF34C6D6BA4504E5C681F493AA44BB869997CF024403FE9516112C732C26DEE54865ADF6594BF7A546B5998EA1026FEB209E30D3BAA289B81651B29916EECB48E6B4D6F8CEE317F46732B2BC865D78FF0237DCA71EEC837215D67F9B7C244232529FB1801282F77CB08DD3B4992A0B4A551175173BD50FE8798836C60BB748E6D3112D7EAB27F0AC52F48FB776F92D89C895D7B1F148F27EF57B47DFEA43AAEE0C8507654DCA384FC7C64B39227B6E750B431588D56567EAE250ED6AED8F987D5A6140F0A7C9A4D47DE2B2036038D43D9630E849431B7A3F7C853399871BA360107009905164D9707C0A81D00A276699249ACCEFCDAF118A8D28FE791FA0FC4D4A2CA862913400FD550513D0CCEC4A487521EC53B33D4FBD2D5F6713F0F1F3C1E6A2895C4E2B56D86904CADE3EF04AF4CD1F3E8AB7876DE83D133170E3918639F508858A335D4244C8DA8893D51A7D6CAD5A2411DBA3B95EE3B95F3B2EC6DB652868A8158FFAD739199C47C32D9A4917AD06CDD7149738E8568B0EB6B7A0252E151538286D122AB6A6A68A6BBABA4958C0572625E7F4837BDF970DB69A943E5C80E5F98DD0FCC590B782308650B86E0C12A0CF2FEB059A2857118507639C4B56C749F3840AD13325C5E843ED1895F6CCD47BF2FD68D708C1B2001A083777896BA92BAC91E280"));
  test_xof (&nettle_shake128, /* 74 octets */
             SHEX("0EFA26AC5673167DCACAB860932ED612F65FF49B80FA9AE65465E5542CB62075DF1C5AE54FBA4DB807BE25B070033EFA223BDD5B1D3C94C6E1909C02B620D4B1B3A6C9FED24D70749604"),
             SHEX("B37F11D6837064D393422AA32CA502E6FC0DE85697E9DDF15B45648D99CD2B0E35653FD071546D893ACB91EA8438E768BF446781A2BAE49C6CF7E589FE80B2C6F35628A9577CC1F902F061E425A84D224D2D8C5FD4C06F1C7CEA72BBF53F982626DD1208CE375EE4F593EEDF93F72011CAB8281049A76EC1F90528D53AA2E0589BF90440907B0F02C51A1A1D9A4C4B31FDF378A548C4D65CE4EBBE77B4F95E9AE29A3CE0172A4D80C370D6967BF27DDDB29065566B2C3205FEA1E2871F6726DFD8FB5BE9199C31E77615BF060A33F3ADF9A1C86D205C2496DFA694D343C058F7F212D9E056D6525479DF0C4C184D2D8191A9FA849FC0DBB2BC27741CD5B43C65DB771DD99872862546B5D1C313456423E649E9869A43B6ECAFB030897D049907F307C767885014F7DCB21FC4C7E52E1F6E95A017F9AAD349BFE109AD8BCCCDA1E4FA761317BD7B7036A09095164A79A7A4952FA2512A5B826009414BE0C504A27CC3AF069718DAECDBE4F43D93A1AE4E1A9B28A48F69533E6EFCEB17520D78EA41500797B18E193C98C574A5339D2879DA6340D8EBE07723549E51F71BF956BD90E74325BD912FAB8C63150BBDEDD455973C084A6E96BE7AA3B8BA503D5215B77151007A7D32ECED117503F1F509A2F6750B242AC63D2A53873D8CBD3A30F105078F0ADCA870DC906C0CF7CBE1197C13C0C65FD8471BC52ED07D8090C83CFFB4"));
  test_xof (&nettle_shake128, /* 75 octets */
             SHEX("BBFD933D1FD7BF594AC7F435277DC17D8D5A5B8E4D13D96D2F64E771ABBD51A5A8AEA741BECCBDDB177BCEA05243EBD003CFDEAE877CCA4DA94605B67691919D8B033F77D384CA01593C1B"),
             SHEX("A0042E6235AFF4430623E540E15655139EF9FB84BF8199702EC9A68D4802E7F37328A5CD7AF4153CD203938DCD947F0C7CB7F398A4867FCB38694BC3CFF4C10971C44EC166F225CD4B0F1C264743815CF7BF37933EDC38C3A1453852E3B220C5D5263D67C82A7E62FC19D24C9B32C10A8A6B2DBFA059197C97DEC62A05456A26CB555546A4219A905BDD2A0FFEFD1D63A56258F5A9FBD535276A163745E8911D55A5E47A4C3411A128F461149B3FF20B9EB422661A67B4785F38CCC43F7B87F5844D8E4665B0F010AFC1270B14176913DD56B76BFDDECAD17374553FBC5C31A7A28E98F58036171BBFCF79CE10861CC0CF343822399546E100C532BA2E5013126976E02C7783193EC43C7E35FC2DFF6AB1483F7B5E5955C91F7F7ABCB52A132A038FCE9387C4635E1A3394BFE589D6C9561D05CB210B25D575BFC0251374E541D72DC2C43E8FC51F98DBA46BDBD7A4C6EFD775E1E5CAB94C8E03540FDF4D1EA7AEFE8298958B618A288A31BBC9FE7B2381200652D5B2EAFC6A89F806102F753824ECEF4CDCC3E7329DC8F7C93D94F98BA5CB51DB8F615D59476B32BBE20111DDEB7AC6B9AC8CF66CF7B338E225BDEDAD4A2C5F0C2941808EE7BD2D2E59DA7010CE6656F6B11FE5925508E5A266E0C6F079533C97FC0FB5559E06793AE9BFBFEC8BF4D3114783ACF6D0E709AA1144326D08D415E7AFCA80725DEC248313E982DE"));
  test_xof (&nettle_shake128, /* 76 octets */
             SHEX("90078999FD3C35B8AFBF4066CBDE335891365F0FC75C1286CDD88FA51FAB94F9B8DEF7C9AC582A5DBCD95817AFB7D1B48F63704E19C2BAA4DF347F48D4A6D603013C23F1E9611D595EBAC37C"),
             SHEX("439E31ECAFD806B8E573AF44AAD0B7143D82FBC725C963633D67665A2116949EED1BD5338839CB0F41FF79E4B5A7F0B3A2FDADCADF52D7379A935AD0357E7658FF2538B99A993EA6E9AFDC9A2B1DA9B68F9C9377DDC1CAABF7334099043E0997F446DBBAEAB14A507220CE36EF110128E0ABDD3DA6F98D2413E4742E82D749433C545EABA95151695B38D1E485C49CA747A11B83CAB42896CF35B275490CBCC856B227C0BBB11EE110E37054B9D8E64E645E1BCE138EAE18118357C362AE2A729407EFEFF38AD2ACFA4D17203D17B933479946E39C4E4BBC467C7B623042C4B770C179991BF159F8A1DF1D3A13218A441C7A3F296C34B1736A62F1C09005CBF972A66BB46D7DC124BB43CEDCCD07F9C7993C9E365A13ABAF64859CD4244223D074005906558147EE618F8C68403B3733AA633DF77F4F743652EE53C26FA64D251C09C8ADC99AEE3D6CA13CAC5C6A4B95B389E2FE51E11D85E060CB76E331628D0494216FB2378917D09A7558262DE00A345D7AE026BCC7938BC3BC039E9C50A62F8FA7AAB5AE44E8B9A1FCA0B65DE335F5D72FCBEC9010BA19040C38B5B37E1F03CED896106C3254663A5AC8A547E5E94738EB86787CB237A2AE65A0D7DC3EF44711325BD1BD4DB4ABB1C17DB9FAE2C7DEFC2CE08AA1E7B68654F936355EC50ABBA8A1AC45C9D95F1C0669E8812F313D63DBC7F7DC40D2EA71DC8F9F1581F43A"));
  test_xof (&nettle_shake128, /* 77 octets */
             SHEX("64105ECA863515C20E7CFBAA0A0B8809046164F374D691CDBD6508AAABC1819F9AC84B52BAFC1B0FE7CDDBC554B608C01C8904C669D8DB316A0953A4C68ECE324EC5A49FFDB59A1BD6A292AA0E"),
             SHEX("91349C4ACA2EE1A2646A6A92E4335E7C1DA9F8482C2791FB96734E3301123E4059F603BB4C42B4FF5584CCE75960B5AD5F667FAA54DA85ACA5FEBD5908FFE7A1EBD05AD9E4DFA8C8D8D92B7F94666C119DE471326E1F673EA67EDA5D66C8035E047AB08157E1228B4B09309F19A0208EFE7BE1788C2F77347C644B4394933F6AA3E10C1B9A984D12A6D0C68E280785EFA11F1731580D6C71C98D6E6FCBCDF56DD55D87865ADB395E7913DAD6E9A4EC938A1D7E2391FB11CF6007FC9C02EA70F44B67EDB1206DC86C00D1DF6906C878FFD93B034F830C05F8571A6068D87A2AC42692DEC62F4318F8FE6DF3690C480E48FE406B85C9FCDAF558AC040FF15011B6C1AC183AEDA7B00BD48E16B7977098BA9412B470B7E7F4016209A02F0F585C29C554B5CF8ABA78E7E5E71447929E598046BAE2BC32958E65A7F7AA29A29C2EC8B34E521BD3601F755BD42EDF60B0D04539D4BF63ECC09943E0C12860F0C5665797A840E85496C4E3DF04B36CE8FBA42DDA559D630BC2CEC7459020FC32F1545B520B0806BAC854E6974A28FB95B41EEAF1CF01DFB3E89B8861F205C100DFC063CD8393330F6A00FB379901D0C47EC2E4EA5D96722EF13FE456E03107AD1179625F968864FB40D7B64879535CCA63CFDD660651FAC389CE4864BC3B382073D74D5346E6499801FBBDB590C1D4D983BEA74FA6A26FAFA4E57B65BE886D55A54622"));
  test_xof (&nettle_shake128, /* 78 octets */
             SHEX("D4654BE288B9F3B711C2D02015978A8CC57471D5680A092AA534F7372C71CEAAB725A383C4FCF4D8DEAA57FCA3CE056F312961ECCF9B86F14981BA5BED6AB5B4498E1F6C82C6CAE6FC14845B3C8A"),
             SHEX("5F8ECC71A5D264ABDDAF1B4895CB11EC9DD50F3DCCB1E1562F1840A0C196A85181BE2B69011DE4A3C75087A3A303555C074A281BEE10E66AA2F15E6A531E5133B6075F84A82BACAC7D771C3EBC74DA2A1CC1921B3C20E99E2866031BF725D08B4400F38BFEFA0F810F82FD1E4BC12DC9E20B3295745B7F132530583C44A18806F537D96B3F352E61ADF285810D32758284D29AE60FF4845893ED07EAF2720282016CF95E28C9BD49F0AD80B87A4C71179ED9D5744843E811D9C162CF57013A7FA9AA9D0A313D44C6BE7FF069506067817BFEB731EAFF820C812765614F3C1A33ABE050DCDE56FD5C6FAC1A43515E2F5CAFBB00A75DB2FA51145A994045B8F71A6998E6FE47CD467549E548EC4FB43C23E7759445BB36A444FB0C91DB3F9A81F1048E21CF32BC3E427A6109DE84FD0EA8B006CED78D08BB7C5678EE401EB2B73F0CD2638B2C7134EDF7F40166EB23D56F7E4BED122B70981F8133B594121ABEAD3E6C96E0C2D76D93B4323DC8CBD869A1811CAB6285E0CAF2C021792A759F8FB907FA45E2B276383F6A385EB299FA5C070AC7B1B713685D6454AFB9AE4850AEEBA6A042F25B1902D18CF5069EDEE770CA2B342B3D3208BDEBFD50822B1CA0C9526306C061B2B4EC6724A35D86B352504D79F44EA7B1A8926C392001EAF2018D4840500E5500472AB05593F3AEE0250C27EBDB3EEB5955DD7D73DDA812D988FF45"));
  test_xof (&nettle_shake128, /* 79 octets */
             SHEX("12D9394888305AC96E65F2BF0E1B18C29C90FE9D714DD59F651F52B88B3008C588435548066EA2FC4C101118C91F32556224A540DE6EFDDBCA296EF1FB00341F5B01FECFC146BDB251B3BDAD556CD2"),
             SHEX("FF65B09E0A806202E10997EC4F42A049E0A97654626056EBF7B92A44E06AA3D11769160A5EBF6357DD6591B373F9943110F609BA15B5893E7869B17C381B549267130047C1286ED9C56F83776A0F99124F8E8939C8EAD777DA433F8C786BEB7B16B3CA4E4F13CEF93B4236D82E870CD3C6945FD7B8F0CA527AEE5BA66B753D0E6DC84E4ACC7D51686C2D9E7E3A31BB0731FD0EF488F1CAE2A8CA588B428068E09E63696F05B71BFDBD89F4E7DCAC2BB804939868DFE7849FC68B095C8C6E7BFDDB3F35E9A8C1B353158C1F3533A0A327789CA271F6B5EE1711ADBE112166ECF76BD0BC91BF957D4B60D84440BEACCACBCE2C2775AD1A2437FFA3B7DCFE2B4A0CBDE28B09E4C306D8133ECA952810D22C132BDC680CFC7E74BF5949E7D532DA24AA71B22B4576CA48EC1FB264D157A0CEC81377ADC125F29B27CFF0640E0CDE89D3875B749E412A50A919A08586A62B8788C4ACEAACC2842C7799D125E99C6A2C1F6150CC503FD420F5900DC509B4815A54B893B8F36B1FE27052A5C66324D7EAE83957DB1FF627508B77FA072F2A01C4F406B6FEFC5F552CB38EA06FC2DBA6A9428F4980D880751E459A1D5B9B3E637917F177920314C7882AFC3298B3E21E18CC126E690A594FEFA0AB722E2B1EDD1C24C0EB118F9C16E4964D637C4BFC6865B9E290C4DB71C510A1CE623132FDC742CCB5C5BBBBAEFCFE6F95F40C861964F5"));
  test_xof (&nettle_shake128, /* 80 octets */
             SHEX("871A0D7A5F36C3DA1DFCE57ACD8AB8487C274FAD336BC137EBD6FF4658B547C1DCFAB65F037AA58F35EF16AFF4ABE77BA61F65826F7BE681B5B6D5A1EA8085E2AE9CD5CF0991878A311B549A6D6AF230"),
             SHEX("0C3F7284A63D46A75707ECE96D5DD51409A0E601765E8760F696175433BF34A781B7DCF131E425F25C5CCE52FB457CB5745E62E7A8F9BCBCB9113FBA1299D82E403E34A6E7438660D0C7930B8A5CF3F127084885447A91D1F5D5F117E3EE8E2C529ABE188745609E9948E02D82F1F87855BBF46E8E4FD39402F749AD00FE886759956FAF77E0689485C08022343167E23180F5F1B1D33B8BEB5F73EC818B92426CE60F5016B8E644DBACDBA9900CAA61D75985B5D063A3BBDD5B95937483A970BAC40A0664E95C7C4F26A2791754D77A6FE5C8A80DC3D7321BADCEE567B7950A686B1490FB0CA411F08FDAA06753B67477F67A1CAAA38F09530FE58184CA552DECC0E54B4B0073E77FF4351FDFC5F7C6E6F153935C1E5288ECB57C3DD6022DD0854B2F986E292FCAFAD0421AAEB47A7BA53C981DEFE9CFB53C95EAA02AC7F1DD155E476B6746DD2F1692A7E2D6EDF79E795924A0B919BEB278A2E65AF71FA1CAC81C5D5EFB138B1DD374F98B98CB5A14D96D9249F0F9CD42856A275F92075330B1926DDE0BDBF123A8BECBB6B6B3797DD89BA4EAC728E6B56696A752A07EEB1A09810D450B2682F2699DC6117BFB45FE306ABB889976838DD1BF31791402CE24F2895F70434FAD1AD4DC86F0F309EE90FAFACC7E72C0B717289D695E4000FADC058A119D9C4BDE9AC0F8B45CAD48C78BD2686105D9A27680CB82800509914206"));
  test_xof (&nettle_shake128, /* 81 octets */
             SHEX("E90B4FFEF4D457BC7711FF4AA72231CA25AF6B2E206F8BF859D8758B89A7CD36105DB2538D06DA83BAD5F663BA11A5F6F61F236FD5F8D53C5E89F183A3CEC615B50C7C681E773D109FF7491B5CC22296C5"),
             SHEX("88C391F4788DBBA363859497F71794349F3A81CC9D1746CA273208680A0CE4ACCC12021E1EFA31AE40EA75A009E95DA25F9F0CF451F658739E2453280EC6253BFC1D8DD72E91B4ABDCFCB240C6EFB4E829934BEF46069274A60B268DF86A7F265B23089DCC09D77B8BF9A02A84ECFBCA193208AED46101F48C10778619C154F880893FC0C440662AB04106B272A9FD18B8723EACDB8AD190DD58E9514BCD43A9299FB361DA76F7E079A0E83615B6CCE6E69CE3E724C36561C728010134ADE4C44B18D1FCE86289310D30195A936BB1E23C0CF98DFAD41E6007E56C905F5DB8DC13B56E16C3454146C31C9866628736B5F537268E0FFE463DBEB2F3DAF103D24F95CF5BA0A4D94C68D61B6319389761EF1A0F22B3D416E67AF916AA4772277F4E51D3A8DF919300805A0CAF8C36A43215E7307B49AA9697B6FEBEEEC4F4F5F7F664C120FF27C8DDCA2224CEA4733DB2D1A1C28F8C030A23039BD0514AD6808805FB5111F514A10EFEC4F425E1C91B0EE3DBFF70CCF6346E6F4640DB8184DA28FD8C63156B28226491C1922B862203F23C870A2D88EB5FF5A6930820A4D81A62DD8C4F381F35C18012336011DCF71D58D167F8C5BBD3E5F0F278507D085F01B0DB2EA2860490AABE253DF4694288EA862F9AB78FC60827241768DFE617E725482DF62275031474592A57FD33FFFDD837654C22C96424A09A62A5F146FCBCE1F2AD"));
  test_xof (&nettle_shake128, /* 82 octets */
             SHEX("E728DE62D75856500C4C77A428612CD804F30C3F10D36FB219C5CA0AA30726AB190E5F3F279E0733D77E7267C17BE27D21650A9A4D1E32F649627638DBADA9702C7CA303269ED14014B2F3CF8B894EAC8554"),
             SHEX("20F8FCA8BC4274E6CE95C558FCF5EFE7F6A43FB47BA3A4AEB4AFBCB53DBF5BF224654A242DFC1F318A4A26924E5F2759ABC96D8B9C50C7185B5A7C3ACE2BC0ADAF1E39C6EEFD79CF89D2F3A8829F180699DF802662F285960EB6676560655FACD507528D862473623AAE970E559117687F87AF4A981C7F2F1E40A1E01F2A3AC0E89D1C62ED751B9B178155F2A72F215E51E49A4E585F4422C757F891AD4004FA618DA654BA3CE4C6A7614924B9C0ABB8B2C6CDE9B8CCF22A1E30A21A33723992ADE39424252850535DA5FEF4F047CD998EC7AD87DC784DA7D755BFB9D5F6CDC9EE52BA15CFB7F1429A15F8F0C67F98D6E44CBC8BA16D0DFDA6D72FBC1A44932AEFC2D8A618814957D8B748E0CA0D3B5095BC8785B6C20053E4A204343316D74FC96E16C775AB43CC33FF2B4F4CE85086D47BCD132C0B0E196344B01CF6E8BDA3E06E2618D5604BB5040474413FB54669BB744704D89C5B06DEEA58D4E29F6DF9C926A6CEA1297967DC66583654FDD3F6449033A496199E4B6F37E0FB31BE50747376122951575095467010006D5F9EA709D0F15927A060D828AF4180393EAAFA27701B919DAF424BFDB3F59AF0B40D0A4890006C98894BDF402861C76655414AE631A50FD80F0E5C63002B604AD1AB9684C094BCDB82C0540FD6E5924AC51F3FC6DA074AC7DD7DA1A87A633C16ECDE716641975182803039231320E9FA16430B"));
  test_xof (&nettle_shake128, /* 83 octets */
             SHEX("6348F229E7B1DF3B770C77544E5166E081850FA1C6C88169DB74C76E42EB983FACB276AD6A0D1FA7B50D3E3B6FCD799EC97470920A7ABED47D288FF883E24CA21C7F8016B93BB9B9E078BDB9703D2B781B616E"),
             SHEX("F0E4F184D33E8B5F8AABC173B04E61BFB420E34AD22E3E8AC318AD06D92E8B5DFDF4B0F28157FC9E7D64545F527C805D9F953626BC35923C922FC09E7D2947D342BF91E54EF34FB5987704F7D0C36C9A9347F3A99027E192907342B2531BD1DA066D34ECC1FB50C85C261DA65F4AF9C2C9E93775F39D632E3DFC1216ABBBCC8DD2680F8081A2816BBDC1646540CEFE4881FCE2ED57B4822E11461B5E6443AE2DE5CCA426258D68B01520594B9547A714B0A377AD398F42599358F7F33855E64F12F88D9E2909E6397ACF34ACF79632AFDE5711858814B84F472EBCD03EBC9671F3985C99DCAB92FED40DE97A3C425D38DBC48ED34C2023771EDFA4F988931B7BAB6FDD612D4AB6DC39D43E66D74307EA8A9359AD73E0965E885AD919F12E6C645C0DC9925CE65F5F3EB132C7090BDFF4230CB16B6C9567AA62DD92949AF10A6FE66CC50B29CF189DBB521477A99DC45C937856DE855C303A5EEC572DA0CB0F584961971A15D8B0907151A046CC1B193E55F7E8419601C1699DB52BBFF3142270F22D628F53A18E64736C1DCC09F7497CB56ABD9AF5917EC07C35BFE8D041D00B379067F2D85E05732D5270C61C592F8088DEDA0CB76BB2826492A61BE9270A8CD7F9D26F51DB2207763BF41A465ECC8B877FC53139925D45541EC090DEB2622864D9E3529F0F23B090912E04BB95F56B1BD33381611D973AE347FCC232CBFF00"));
  test_xof (&nettle_shake128, /* 84 octets */
             SHEX("4B127FDE5DE733A1680C2790363627E63AC8A3F1B4707D982CAEA258655D9BF18F89AFE54127482BA01E08845594B671306A025C9A5C5B6F93B0A39522DC877437BE5C2436CBF300CE7AB6747934FCFC30AEAAF6"),
             SHEX("1893FF8490BAEF747390ACAC15D42CEDB909C2FF3D30B2250A3F9DE11E79943B9AAA5C5747736E29E559F93115B0C029AD6AD8779804083C3F211C8D6CEE56AE65F0B650CC45B9DF5436456879F62648EC81FBF968583582CF4063B5F007E1B04A9BCB2249833170C1968E3040C808F2D9EBA238E74931CBAFAFD94690A275D1A1C1C29F71B8DCCCACC4E9543FF05900417D7E46C64FB4454B3DDB7A0BC01B85A0AECBDF9272EEFBDA881D2ABE77ADC9F17A26C50DE64DEF1AD2A09A79EC9F9C64895540FD56E93423DD7DF3E566D168E2560D8336177937F46DF87AC29F9FBDEE919DD8D2462FC1DB70E27B992E35F1376459FC15B76FDC3C6D9C3F6D994B300423F154B6FB28F845AA3DC8537DE62D81C78326BBF8A89E39C865F395BE31B38BAD26CFD4CD866B872D427ECE798968E66658839121301BC52E07CAE272B6C373B01C4ECD42F75D3CD45B2CB1131179A9F21F8CC84C08368547DACCCD56985477B4067B865B0268C1366D35C59D7C4EBE54E0C7EB383860EECFFE308A2E71858ACE60EE4A2898FFFFC507F0011F47B16BC9D696955C8E2C7FB64F4EE9DC9F9CF9987EE2CCB0FB780BFF780C4253E398763AC6668FF7082A848BD5E1E31E6B5E1E771F7AE21918C1ADA32E788D89822ECD3029AC5604132A681AED4348B631A2C9CCF19D14CB10F570E3B7F9F301B041E6B901CC007CDB125AD9CFDF7A889AEB"));
  test_xof (&nettle_shake128, /* 85 octets */
             SHEX("08461F006CFF4CC64B752C957287E5A0FAABC05C9BFF89D23FD902D324C79903B48FCB8F8F4B01F3E4DDB483593D25F000386698F5ADE7FAADE9615FDC50D32785EA51D49894E45BAA3DC707E224688C6408B68B11"),
             SHEX("56758CE6042783CF0D18A2C44FBF953065571BCFC3D19DEEC28F0D6564B18B227B6DAEBAE658DD065F414E4FFAF756D10820FE38C61A767BF2FD0C242EA3E994CCC6516FCF68BD7514833E061A65392D175CB6EB87374DE426730E4E771C7905706BB5D7E3364F946B8A061462519D0BE6F72B0220451F2C6C6E5CE217262323E4FC81DBFF169DCA043487D88F02BB5C22434FC31043603A5140364B7C2CF179FF13241CED0911ADCE30C4D7B7EEA34E094C86B522EA88F0126943FC2E3F8C1C81753C4A10B844ADD7868457DFC91094458806B2C115A052BD616161AED4BC1690E5DAA09BADFFB3BE59318D7ABEC04E7B5B439B92E63BA3EBDE71AAE8C5BEC81C9073072EA49E85B53539C2462A446448F4D362136638EEC10F227A0A977B3460499A77A9E951124B55340E49F9D2B0A60808153D357B585F2C940398B12BB807E730DA556C3D156D84E5C0BE2C625E3FA725AC5659E07348894842711A704A15201C72F4FD2E54CAE47531C07C7D1FDBC61446E35666175B2CB7B25A5FEEDC1E5FEFD2547F408F0E91D65C7AD0E636FC7D8BAFBA355275AACD0E2BFAAE9FF7B750AE20C828D66793A7D275DE882DEB8A97002C9B7B90F5ABA4D1E7B9D5E4BAB59DCC44E487BE10854C76D2EE6EB5628F1E791B731E5B1CF339270E90B3F25AA78B28EFDB906AB2D5560D976AA8FD4A560A29DE12C47A360131BB5F1F8FD71A"));
  test_xof (&nettle_shake128, /* 86 octets */
             SHEX("68C8F8849B120E6E0C9969A5866AF591A829B92F33CD9A4A3196957A148C49138E1E2F5C7619A6D5EDEBE995ACD81EC8BB9C7B9CFCA678D081EA9E25A75D39DB04E18D475920CE828B94E72241F24DB72546B352A0E4"),
             SHEX("D5076FE06DC454041F41D77DF1680958CB4EB8311940D58CBA2D6CCC1B57EF7D66D0D947B068928E0284A292D6F3610BE7CCDCD2117D59EDF468F4CEF33DE2BE52ADFEA056F586047F8FBD3BDFDED58237C4CF541200488A3DE301D5F36C735C387DD852959609A9609D370F63EC8E1649119F6966887C38C99F94B2293CE52EE234AA5AF55B0EB3AFB8B5A1D283BDE8F3580F5AD1951D8BEDB834EECBAC86CA72435B52519091D94647ACBBCB1B55DFD9A531C6C966B481FEFA1396CF42FBE362ECF403FB046AFF1B080D47BC282F4D053BA30C2A00364E11EF82FEFFC4096DF1F8AB109B45331611A8AC84B8DF0B7B94952A5AC80B075BFFBF16065D8E5A47F771C0F983E045D4104B7E8E5226EBD9D7ADD387DE852DF26EE9DA3EE70C7E20B877A45797DE836548664026A380F306705232ED37F4104114D58065B1C74441EB9CBF00B8073325B40851B44033FA4E65148161B0DE0C1517B6830CF5417EA3C004B7465C48F9B2766C43CCE5738E9EEF6D9D0F296DCBDB41DB35924EC1B746225142775268908A63EBE1240CEC4F02972523C8C7DE4B44530D1821BD43694647F355FC0473D7BD4D3E6D30EA453EFDEAC396E0DC03F796F70709F118920F8A76D4D85DA2B5DC2964A8824BB96AA892E43FCAD95470CDDE85433D5CAB38424609A9682E1FEDF7776F2D77F2FFD9707FF110387DE49F9422F11933D6CCE6DDF1"));
  test_xof (&nettle_shake128, /* 87 octets */
             SHEX("B8D56472954E31FB54E28FCA743F84D8DC34891CB564C64B08F7B71636DEBD64CA1EDBDBA7FC5C3E40049CE982BBA8C7E0703034E331384695E9DE76B5104F2FBC4535ECBEEBC33BC27F29F18F6F27E8023B0FBB6F563C"),
             SHEX("904786DCB5A8F3C4BE4DC96CF168C65EFD3DE65FBE92DFD96B1CB99F33581F561E02A9D85740FC7295726F36F966392956B2C61122481D7DA372662D0F2B42BEB4228877BB85A3AFB52C689B3DA75637B52B81B9182EA56980E2C0AFD0D27644D0CD86B770FB3F4FF2618FF000FF7846055449E1ACB3721F03A0E65FC377F189824E06B13EA6A797DC7E8A8AF064075A9424A709456CBA62C43B51F4145D6F5456159678B09FF1A97BB19F4A41A836B0E6090617A2BB9B9ABB7D4B6777489FF1CC36E6FF828BB27B26DD61245A3F583839F4EFA1C453D6D5E9B2B3947EEE25D006DE476BB37D505FC5F0AC0AB97F8B56B8587A8FF2EB61C3F83598120B469E9CBC5FC03AFE2F72EE0B117F818EB8B33CD538AFAD801CB75D4B2F51578D628A1701DF56B9B9BBF10A30DE0893B6E3CAD4E264C5829B1C5C35B3A1B859BA7E49C29C93F98E69EBEBF8909F535956DCF767C51E2AFC49103C052AAC1A39D02909383AC02AB8220E4985872916DD6DEAA9E3F57475A33B0DEC69F266F1B990DF7F9F85EAA82FA226DBDB94292B91C3F124B8BA13C3ABE0908E52B67EA7131E61AFBF20293A61CB3659415EE890C151A5628225D61DED612DAD545B62D819A4F3D1ADE9CAE43B9CFDFA4C32A8D16116A58EE3743483663E9401CF1280DA465025EFB103139E4E8850D1329F1B9BF5E1A8C05D20E963E824EAE0AF586635D19D13244D"));
  test_xof (&nettle_shake128, /* 88 octets */
             SHEX("0D58AC665FA84342E60CEFEE31B1A4EACDB092F122DFC68309077AED1F3E528F578859EE9E4CEFB4A728E946324927B675CD4F4AC84F64DB3DACFE850C1DD18744C74CECCD9FE4DC214085108F404EAB6D8F452B5442A47D"),
             SHEX("9FB52B8120553CA1F4B801FCB753DBB0E45AF3F160749A05FC2ADF8FF1F507DC304214D785C5E0797C2F6F767D76A3EE22CBE0D114F5F8C32BBF8DA8505090381E16064C001F1807289D0DB586C09A85EF727F393BD2EF1F95CDC1777E137B43DE60D425C5F5D685D43D5F6912458FDA3F4ACECDC2024CB7E6F4340FCF6333CB29BA1801AA80C0F8A248A21305BA62D62E689D55D905EB03368783CB32F765A55D3061774C9344E85C35F8B944420F47070B34C9E8996C281D890E591D5B5CEF03463D86E88211917E7DB7F5BD4F36565BD855EFDDB859C97C206D5F3836840FF3863B93E17AA19281202AF6EC0B1B5DCC7C00D9DC08C2DCCCAE41EE2FB8E591A4AADADF2B28FC2133067D45F98B4ECBBB234593159888F832CDF1BD4864B8B720E42219A220C7FEAF6C0D6A1C14AF2F1020218C65A475386A71788F7BFDB172B12819F2D78D6D200382992D9F39500C4F15E44E1E48FD5734D110FFD1967E098F673492630074DF33434C36EC9FE067E8E794794191F355C2D564FA4BB691702DC55F4E6C83D1C487ABB281F0246814A1EBE626F05F06D09A371AFB4E746653872F83B42509504B057DB7BF06AFDC913F679EAE1FA4D945B93C4BB07E70F48F147BD668B121B05A777A5637CFE24A2BC155C97B9330B3E3B848F689A00AB1F916F321FEC05763DDC6D6563E7AAA3CA309ACECBB0884DEC5815558D8FF72E820"));
  test_xof (&nettle_shake128, /* 89 octets */
             SHEX("1755E2D2E5D1C1B0156456B539753FF416651D44698E87002DCF61DCFA2B4E72F264D9AD591DF1FDEE7B41B2EB00283C5AEBB3411323B672EAA145C5125185104F20F335804B02325B6DEA65603F349F4D5D8B782DD3469CCD"),
             SHEX("906BFC0B13DBD86ACC32B93C56C7187ACC1E5984C1A7AB8CB4EC39A00355837F70B6790360AC9C447E28E93CDFB58CFAB3DC8EE4C2378E68298ECA240869FB5652803F590245F8F40810FEBD33FF25794E39758E9984E47626EEA8AB2145CD2E260873D4524759A855B2DA0C8A80C6C365E76E8B37AB4B9A6D9882870DC22F61A341FC932152BE5C84AF645EB71E6713B1D9B582491653A8365BE531D565B27159F9C5C7FAECA04FE1E626C1C9A7E7FF3882662B084B827BA96E865B753FE73E3936E47EF9F1EA151878F671AF51722A4755041960EC2A7C2C4CD364C4914F64E90BCAFFBEF147E935A6B03166065506FA7F20A707AD4771FF026D7BB0DB7E744FECD63782DCBD9DA7573D77F82E6A5446B1595F7EF81084CAEAE91E522FA604A370612A6BF464D5872ECDDDF46285A407D7835E22438E82E62CA0F1FDF7B420E15F9DC061A3519B8F389022BD06E9B31A34F854F0F75FB9CC538ADC0AE1D981DB68C17E6F3A91D6C60EFBAA357A49FBD148AB7EE2D96EAA8909782E74A9895890D02FE8C8B09B8A9F2A8E18200825C51896C42C59AF9AD93D847F4A4FCFF008D1BB8D6BB2E7C75912A086A23646939DE69D7BC42F568FCB076EF9C37912BCBC0259FFE04CA17F799FF7ACA1D34FBEA76EBEFDCF8D22BDA634DE9B6852A5DA831EDD0B92AC09D9D8235A0AF2BD5FA6B0157722115561EFE7C14C22145137D729"));
  test_xof (&nettle_shake128, /* 90 octets */
             SHEX("B180DE1A611111EE7584BA2C4B020598CD574AC77E404E853D15A101C6F5A2E5C801D7D85DC95286A1804C870BB9F00FD4DCB03AA8328275158819DCAD7253F3E3D237AEAA7979268A5DB1C6CE08A9EC7C2579783C8AFC1F91A7"),
             SHEX("E2746CD339332913E436FDC5B21215B39D0326E95882F291849BEF33AFC6C7B9A02F438C2E6F4175DE5FF04B79256B845213B917B34A0111814AF58B06C8B5AF9D2478AD18F012373B9F030ABB79BD7BF380C2C2EF07061E75D7BA113845C80E43BB718377732064B80339E8316068BCFD37ABE6AEA6EDA3C58BBF95A77799155B30FD6776012F098EF9F29DB911E11BAA67937A52EE25C429836FA45A2ADCFBC704A404AF8421FB3B75DFEA979F65415CE0E9B6D17B2E5825929115DAFA8E9A51E16F8E2563876702E04854D48AC2159BB782AA6B1B764A8359F7EC423CB00E9915BC63B27F6E758402B56799D7C3093B1A222FFB8C0F08273A6D5115D2CC7429B9382763807FCF97BA5EE82F78F835BCFA65645AD36F56D595ED06924574B581D6C460BB0209CD69581732EF9D1DE78F0B8109A6D4C271F2AC07179A84BF36AE328B1D0862F65B4ABFFFC0E4B6A17FDE288304234CC2447ABAAA223D16DB5AAE8E026D17A9ADDBA9C51F3B63446DEFBB9CC80323E6B22DFEE9D9E438BCF5E14EDDC490D9DCEC2F5C635D32A1B6C64D1CFD07DC72E954C97DB2A6E82DB75575172F6AB0D4519158F84C836F3B7B4830CE0856C37839DD7D7FAB6A9AB00F9DBC020AE45C6450C257B7C0661CD75F515EE380C2864F0610A682D49CB4D19409A317AA0D6667E3E61FBC2D9027F4A5BF4FB1A0D142183F84FFEFEB92EAD6ECB2A9"));
  test_xof (&nettle_shake128, /* 91 octets */
             SHEX("CF3583CBDFD4CBC17063B1E7D90B02F0E6E2EE05F99D77E24E560392535E47E05077157F96813544A17046914F9EFB64762A23CF7A49FE52A0A4C01C630CFE8727B81FB99A89FF7CC11DCA5173057E0417B8FE7A9EFBA6D95C555F"),
             SHEX("0B82455171770082FF4A4FAD8A5DE99DF716033D27071278E9C1256D63C34778980889FDB7661153982EEB0FF904E24D31C8D48D89F412178E527D054F822DD2DF6C4FD555E55CCF3452FEC05F9CB3BD4F0B4D2EE851EE3E22ED180F0C02E618563BA89E42C4C6A7DA1A9256029B6449A9C63625CEFCD003F936613092CF59F676AC91298C820436F58BCBB2F61F3945E67353C854CC384D607DC11B906A1C1A5143D9AAA294CBE03F946CF2703C0983727FAACAFAB85C6725E8A9CAA7AA69891799C560BF6B885255674A2940DEAE220BBDA252E636C4526449BA895DE17312EE0BD0520FA68DB28BF122219E571C081DF18C2437E4174D1D6B529E394C4090E5FA957A996564D11A3B4652522FE2999903F6095422B4AEE2D47F2CDD4301BBE42AF671F58228D76D7CBFDCE870242872EF85D0D1E18CEA60C87565E48A1463030566C9E34DF8829A58FA36770E861D6543B50AC6A439ABC062236A065BBDD866A237C09897F3CE2EA5B0B1D8B576048685A53DA7A409FC6F121DA36AC23B9419F254C76E7024FAFF9D45BEE4392CE4C968E737E8F9EECD4800F7C928E30EF664FD4D2A7150954E0599CE5C4F07B9351A408CE35166DA4240F1E751CA4A7CC4946F5451A0BD137857DC003B61424A6682D942F4DA9644672ECE7A1EEE295EEF46B5AABA3F46099E93E01BE7AD9A5E27D378701CF9514B374204D387145349F5"));
  test_xof (&nettle_shake128, /* 92 octets */
             SHEX("072FC02340EF99115BAD72F92C01E4C093B9599F6CFC45CB380EE686CB5EB019E806AB9BD55E634AB10AA62A9510CC0672CD3EDDB589C7DF2B67FCD3329F61B1A4441ECA87A33C8F55DA4FBBAD5CF2B2527B8E983BB31A2FADEC7523"),
             SHEX("4167BCE24865262ADADC78E611A960807758EC6D0D46E3CAC221B4FE9818937092BD3AC5219F0FD765B613F2EB7E36A5F48FCBE3CECC67540F9282DB1842F95CECF0648C12E52E7EA0C96FF7249A8446D0DDD7625BC55568642DA2F2CC897B14D2995DC382293CF2DBA9A63EAB9F6940AC9CA87FE5CFFA196DC930FF3353E34313894A3D2A0A86B0FCF7631735A2C663BCB71D5947C4B2E97441982B791A4F20BB86F7FB033210E4FE58DCE7256309FDBECD7D7F3FEF2D1DF7E0247FF981D417C56EB7B14B7C9453786C7B30F173C372275324E834D94DBC6D10714407EA7886D92F810F2DBAFD57484A09D70951CF85C0B7820643D0E7EF706934CEA0CCCF94B1D1F75F1E05B5D033A5CE1F804C3BC7A9CCF00C8CFEBDB62F8C654805C5ACD4FBEECB6745803721D17F2B8303A98ABB9B8618107B4493735B4001757890D772F78093AAECA244E5297CFBF0B9442370C9FE0EFEC199201E8667D61FE4B9CE20EDE827433C91F521EF2B296B15CFF658E18226F17DF114675F57E988A14B9F104F933BB685A3A9DBDA96A9B87F2EAFE028A7C656300814692C3DCF00D9F07CFF361F601BF2A1FAC9EA58D7543604382ED7CF35DE7AD7355247318796466081FBAA05FDAA65665448DE1FE67AB560250A213FE60F506E9EEE720FA762AF7D2A23492597BF37E0067997095FFD63480BFAFF67C084EDC5BE6C0BFFF969BF06E25E"));
  test_xof (&nettle_shake128, /* 93 octets */
             SHEX("76EECF956A52649F877528146DE33DF249CD800E21830F65E90F0F25CA9D6540FDE40603230ECA6760F1139C7F268DEBA2060631EEA92B1FFF05F93FD5572FBE29579ECD48BC3A8D6C2EB4A6B26E38D6C5FBF2C08044AEEA470A8F2F26"),
             SHEX("4EC44E5E1F62935ECA450494ADD6AEF7DB1C1C8353F1354C4C9366E9E56A8D5BDD02383B9A04BA4927959531B9169606CD982B6B72927D2E433D900F3B402D7F5BCA979B3742E2E2C1DF8D4E163628357AEB37C94161F8A9269832DC28A4E4983B45140A5A7C9A4313FE39BB2C117453C2E5CEC3C0D4D5C1169B042E319C77712C0171E3487C590FE9B223B7F681F6C4B46ADC912BABA3F1E190CB66DA0B38B3389ABCACA3D4EACC26046CC71C4B3EE1520EA84A1B8897A7E39B9EFD65130DE8F694A0A81F4CE994A0F4FE1CD94439D6551CA8B695A5F22EC51022ADA24C3EBF42840C42B2E345F8AC07BD5561DB40D66DA87FFF248909EAFA8D35C380133AC87303C83A8368492406B45424CD0C7143BCA5BF6DDB6B2FA45FDF45B96BD1CB33F16C1843985C5A8B1B902361D10B235F3965457CCBF2ADE247CD371B9AD808FECFD503272D4918BC3CA6EDF09BDECD56DFF372DE8E2E1D35CCF55E7E81AC5DD717C49F492BD1B242331EB66287F10E15D7BC69DEE9C3F1032C162B0845083ED653D38F270747510F82E43F0EAE6EBDFCAF296D360998A5AE8E735FCE78AEE2332F7766FA8B289B46CD1F129BEC02DD6C3D7C63D544C22687AD7E4802E0363D05B8B7965C1D9C9EE8D08B0B54A8D55D2A52E0A053D31857EF350C713CAF7D7B00FA5513CBBD6FCCCAAC97501E3964DDB3012D2E101C59CB8FD8B28AA567292E27"));
  test_xof (&nettle_shake128, /* 94 octets */
             SHEX("7ADC0B6693E61C269F278E6944A5A2D8300981E40022F839AC644387BFAC9086650085C2CDC585FEA47B9D2E52D65A2B29A7DC370401EF5D60DD0D21F9E2B90FAE919319B14B8C5565B0423CEFB827D5F1203302A9D01523498A4DB10374"),
             SHEX("4D2597545337647A12662A7A13065021A6D206EBD13459DB348C1F5361C5990DAFF469677E61DA9A4ADD24A1B4ADC67A092BDCF80D98C0A858EA96B0BE188619A82A1F5055B461660CF805AE75E4F869CB830E153E840D6F164DEABE9065695F6C8F982BBB7D2ACBFCD7DE93E8028AC978E0D78CE878D3F813ABF1E1B51806099DA8D76E16C47D0C1D6CB70E766F5F3F2336642B9BDC2958DEEA3D6EFC30FC4E54A90FDCA6BFA5E8D4410D38CECE522B3A32F11D4599D464E9CF16949385110D71D86A2943C03E67B19DA04ECDF7D30A3B9E16D256D5CB1545B1312B6E62AE028D61317E689CC3EAC351B034E83CBE91140E20E292A3AF44C27A026DCA5674B9497A10876CA0F21D79184922CE71FC11AC98072C2368F70D113A413E2B9FAF616E3729E515FE22146A35CB93D0178C1C07FF9F403BA947759CA0FBB09FD66C7EB562307F46C074501EA7418FC64911AC940C28E75E2146231E0B94893EFDB0DED97478796E87DD906FFFE9232368CD0673F9966E7B312C44F26649355448FE2FAEEA198A5F6EB45040D56068E5B98C02DD948277E1FE5061EFFCB028B61748E2D10128E0E0F7B70EC8CC6FFD071C65B5C1B9CED47C9B0DF76D3B0A5255E27391409B07B0C112CA444E2E926BFCFFB1AE7B00252F6831FFF4992D63BA2114502DF0A3489AC58F463C8F9A1B3CBD3C8836F4CAA301B7336D8DCD255C0762C15E24"));
  test_xof (&nettle_shake128, /* 95 octets */
             SHEX("E1FFFA9826CCE8B86BCCEFB8794E48C46CDF372013F782ECED1E378269B7BE2B7BF51374092261AE120E822BE685F2E7A83664BCFBE38FE8633F24E633FFE1988E1BC5ACF59A587079A57A910BDA60060E85B5F5B6F776F0529639D9CCE4BD"),
             SHEX("05A819096B5B6835090CAD5109CD34F40A234F1569B226C42F0207AE06E6EB37A50AD688C81A98C9E747BEB1BB2AD741454430C0E36A8B4DE03A1DE7C3D1FBD836F90DDBF10B10AD9DC427F0039DE7C0A8336B10A5F2A24EF2A0DE7D5765ABE3ECCAE743AD3339521D78006F5859339F15B5A4C82FE758294CDCAD306D30E57EBFCA0C22042FD0CA678BF2E5D59CC39CF1174B1629243115B9247B620BB14217C7EDEA7CBFACD209FE77206275CA39D15F6995E27108C9740A27E98BC205606D68F4D5592756390DCF9C3B5C58585142BB6E31928AB39843538001D792DB744145E5E148F15336C1D6F334608D514C8A0D28515C17CD88C47AAFC95A7D42785D8425AAE5D2D8D74A90914AA30BD428F0282E51445DEEF80B7C1A4429E4E810C753DE4AB21D2E6E895FA4797F3B4C267CFB123C337CBE0ED977A9C3D4147780DD20B2E0C1781E3A8EC690F3D32F38D55D8AE641F61537E0649FD681857156837DD46D1E6005F5747882F895684A6817D40EA1DF7B6B049DB621CA257A3705896DA9B5E38A6D5EFD1D2FB4906285B1AF522D34A5F141C1B72E74FBF02C9BFC38D744C4D6B77C32317774F4F3743B7E684A72AB54BEAE64994F20DC8AB350FACAB35726B55667ABCDC7CDC18576076B5FCAAB6264687D497D32EC0D0A5C42FEF8899441F36F7B72925326885E1B67D59D7BC7DE352F8844F44C3B634753812844E5"));
  test_xof (&nettle_shake128, /* 96 octets */
             SHEX("69F9ABBA65592EE01DB4DCE52DBAB90B08FC04193602792EE4DAA263033D59081587B09BBE49D0B49C9825D22840B2FF5D9C5155F975F8F2C2E7A90C75D2E4A8040FE39F63BBAFB403D9E28CC3B86E04E394A9C9E8065BD3C85FA9F0C7891600"),
             SHEX("B83D4F95FEAFC7E5AA60BB1035F69C8074B1A25FF516AD6A18DE1808AE64AE212B63FD05F983A65BAF28B228EA3AB587B163E334A70ECEA2691A77AC940259827355C5459D074532A3EFD811F419E29FD67D8CE82BB4F79451F6AB0A8EAC6BC48713A04F318AA45DA6246FB8CD9C866FAFC9A48C5E6C6D05D501D00E08F7D4320E5E294FCA2B6439DD7EA3BEF65CFCF256E9C4E6F902BD9F6AEFA2AF303E0028D0DD0B7E266F31E21A23F546FC4CEB9BEA5A4034D3DDF7BF63B546F8A1D73F9F56E1C70B6B5E1ECBC244651322025086A933D94D1F21C9BA03493E38D28CDDF351F4DEC99C734589CC45684DEE9AAC79F367B940D6E1AB57AFDAFF422F9D60E83FD6A080D60267D19DE6626FCCDE0759697DC7CCEC6384B7AC7E936A7F12E134C053B80CFDC38609149B277B9F6881495562EB466B99D4D4A1B80C3751F51B3D53261DD5BBDAF1B5401373140C35D31907EA197397DFEA7C420A80C2151E73E0EE8E27495B0C9EDA13B93E8DDF9426C26995317CD273960301973FAA9AAFEBDEC14D92466B059D3FA7CDD533075F5861C538D69DE08F490616B650AB5A0FD177B500F05E011AE3AC06C2480816770D0EEF7908761AF08302DFCAC984BE0B3E60AFFCF5281FC6E28AEE30421E1DBEF6775CA9BCB2803954C0BE501D55D5D0A1CC7882AB5F1287B5A3EF564B91735B9CA85D5E7103D0175B2CA00EAC7B627BAE5E"));
  test_xof (&nettle_shake128, /* 97 octets */
             SHEX("38A10A352CA5AEDFA8E19C64787D8E9C3A75DBF3B8674BFAB29B5DBFC15A63D10FAE66CD1A6E6D2452D557967EAAD89A4C98449787B0B3164CA5B717A93F24EB0B506CEB70CBBCB8D72B2A72993F909AAD92F044E0B5A2C9AC9CB16A0CA2F81F49"),
             SHEX("5A81FB81F8CB6473C04930BDC5C0A9CA96D4A500A7BA5A9167491573AA7178101D64F46A93486406614AB9F397E78B0198FC988CCA849EF70C39B0B3A844493439CECA2B7CFF8A32D18285BBD32B1E5EE26AB0F82626F4130C1B40CD6E2C9DBD6677E6644A1ED6A016DBDA139B868FC2F7C9263BF4535E8ECC24DC626B1F3ABF4B08E229CCD076DFCCD89C8EDDC77E9FF0252BDEC35DBF712CF0A5BBBA7B4C7B702CB9353EA4BA3D0E0DF48D6C56D0B51CA269C9E428B22951555C8CE65B5704CC3F4ED38649474C4C049A63894405186E116E616D686C7C033CFDB75B8202FC6B6EEAEF697A4F5277DA6C5E7EB9814A6798262535292AB6791633CD687E68EF223464493AE8D1E27F8B2F483F5B137E48F4390CC27D96A694971B28C9CDB1A24648A0E0573141DA6A1AB54789199E8E909F6FDB0766DEA735E2D87CCBF975FE8CDD6C728DF888FED6F049008D349A476EB63A87E0CDA622F9B611278868E8396BF4E878BFB58AA847F04C2E3A90DBF386954874BB7B1E8CEC42F1AC16C1824F328AF9CB154CAFD4C9E2348A02B0F9638C8562655CD5E3FC2BDED644DB61EA0C4DD27E8B79C7BE52F55C321619B5C8BEDA60D00DD3DA3C19D4C226ADF2D5C4A66518940F2441BCAFFA704C80900512516BE2710301579F023393892FB27EE9DBC71AD34EF26651E133B3C21D8914BB6E9446F11288E1A76431C3999686881D62"));
  test_xof (&nettle_shake128, /* 98 octets */
             SHEX("6D8C6E449BC13634F115749C248C17CD148B72157A2C37BF8969EA83B4D6BA8C0EE2711C28EE11495F43049596520CE436004B026B6C1F7292B9C436B055CBB72D530D860D1276A1502A5140E3C3F54A93663E4D20EDEC32D284E25564F624955B52"),
             SHEX("E41027E76A055F0A826481C650FC24F7816460ED519AF8802AB191E411ECA719CA1696EA14CD62306071D4E1DD65333619D0193D9830B6E8796234F892010681923BF30A1A0AB89D82E36DCEF3ACC00DD286B84240D0282F5AE182FFB84BDA59531C519374ED5A2E80DB2137BEE790CF49B9FAA524F75ABCDC5AC3C3027F6DCA443B2CB5976B9B8D66E9EAA1F9F62A23D5C9A9D5FEC16A5925ECCF1021D3253BD67F471BA1530C0601DD96A4D7ED6E37DEA3EFB24284B61002158AF78E35F8BD617C9E7D33FA4128C31980A6D8631A009FEB0039B635D1030C200AB651C3701CEBE8F2CAABA57E452C0E8FE41FACFB398A2769AABCCB21AD2AA39DDB53EC706E796E7327E30FB4A631E6F2A35A8DCA86769BF6697F8490EC5980F3143EA2FB79FAA9A5F2AC7B5030170A4548AF817CD80C26E1B013BDECE9CBFC553464F1597077E3ED435804DCFF5FE05BD39E87B9B35D9F2810C32EEF8E30459566EA3CF380A7F9E1E2BD3E4843D06B018B55006FDB22EE1CCC0AE615917E8D17FB019F7419FEDB242F0676FF4815989821135073E5CF91DF42CC6C6F532D3BD8BE9118EA6E583D360F1F8C20600A740D9310A984AF3C82A48FBA84A84AFF42F2F1EC8C4115B5704704AC2073F2014BBEF1913FEF7862F514B198983BA63F4D38E81D822CFB548AF76F561EBAB987B208B11824481309B7AEDF0EF44376847162326193DCAA"));
  test_xof (&nettle_shake128, /* 99 octets */
             SHEX("6EFCBCAF451C129DBE00B9CEF0C3749D3EE9D41C7BD500ADE40CDC65DEDBBBADB885A5B14B32A0C0D087825201E303288A733842FA7E599C0C514E078F05C821C7A4498B01C40032E9F1872A1C925FA17CE253E8935E4C3C71282242CB716B2089CCC1"),
             SHEX("28E204A3450796BA182A339CEF54A3B224626681C51297A71C2BC0ADD56BF57C7653BADCC1438E3CF7156EF7ED4EA9A084F3041A8157E1FD2AEAD827A5E79257C1BF7E37D4FBC089422ACFBE116A2E3889614B8B01BDDCF63B66589061A690790357BC31C68014CE422060CAB07F2A254C83E1A7A9AC1A0807462E6DE48B194AE5FA0D6BBF39AD1835C86BB25F5C273CB266DD5167350904EA22F912343E261E38B298495B38019724A1EEC356ED016BDE8A81B3764A39D153A5691C196557E6FD50DC5E42B6E6376E388D9C14970B67E9910499D94F5839ED933246DAFB712A122043D7D5047F1B53923C14979244DFED3CB742C1BD30176985AE2F76B912A6A3AB059F2BDC49DA4EB6A9D3FC9CBA1DFB1ED0A4ACF46829D51FDA861B8DCD62424A6E4272A1BCF28AB878FA9709DB43D800E9E6E33F6D644B62EA30BF758E496BD152E41715E6AADD766F1FC6BB95DBF19F497729BB30B29FDC4BC4F5502396969CF6957BE8FA7F17A4F04A6FDE71264B0F1B40EFFDFD560868056DAA10A6CCE695AA8A815E43121C971E4C4A3FC5E6EB27170D4539F3A37A4665F179CC6C804DBFB199814FB2D5121FBF93A472DE6F4D1DB176CED29F254F228538B065F8494ADCDC3D1117B33A1A51EB1AC3F6D7C8315E5A674703827C0C506977D1CD68873B4F18D9A03CA54F74D7F82900A27E8863CE13E92E0EAA86C2CD5BC99121F1D0"));
  test_xof (&nettle_shake128, /* 100 octets */
             SHEX("433C5303131624C0021D868A30825475E8D0BD3052A022180398F4CA4423B98214B6BEAAC21C8807A2C33F8C93BD42B092CC1B06CEDF3224D5ED1EC29784444F22E08A55AA58542B524B02CD3D5D5F6907AFE71C5D7462224A3F9D9E53E7E0846DCBB4CE"),
             SHEX("FE3145A2E9E54B8888C1FB87A70402EB4600B0A2D636DFB23F869583900544F602B9DC325B6A9E8C0A9192602EB5EE581E3B0BE65E918DBBB49EF017F174C3AEB12C20D13D20C3370E1CFC19634899833A91375631713DB2954A8E33CA839EE5F12703C814A63496BB438195D89AF5CDE1F22ED8AF340B72B6FD95660BB304568AF54F162D47CC3783806274F967835038858719A4E3115E76690B4E848FD6ACEC695B3174C7BBED140DB8FCC2187FC0585F4E975FA5EF3907588E415638F1487E6244F3A773F8E2B3D9C23B4154509F87564BC76FD437C718BFDEDBDAD24E3393A851EB30A82A641D3B144E3D9D223B1E92813DE477DAA28BC9CB621F8B3D06B4A3E3F9A52EFD9095365976144325269160F7B859EC570A14DD43057BF4AB3CD2ECF13E921FB8212F4379C59B332FA1817360747F58A63E0F635FA52C59E31962951AAA64264DD3F26ECA0AF7C423C0D3DCA624A4D33C383BA0F4E04630D74A40F760349116D6CC1A8BA17F228D7BF24AF707D130D13AC825DB8E6ED4B97B4AA589230CE9A471DECBB2DBAC9F68269030ADD24C712F1C2CE7E6DBD330DEB3249F7AF42E31E717666236DE7299283063D3DD6CC88D0096359A1E83795844714BCC9D88DA6FBA06D1156114C62FFB5D9A58274AE471D544D7E9701ECDAB9B63B2EC4FEEB07126F8C1CF8BCAE823E7BB18A9566627CF95E6D8AF7BE94275932629"));
  test_xof (&nettle_shake128, /* 101 octets */
             SHEX("A873E0C67CA639026B6683008F7AA6324D4979550E9BCE064CA1E1FB97A30B147A24F3F666C0A72D71348EDE701CF2D17E2253C34D1EC3B647DBCEF2F879F4EB881C4830B791378C901EB725EA5C172316C6D606E0AF7DF4DF7F76E490CD30B2BADF45685F"),
             SHEX("4A021C69D45ECF926FE7592AC05951D86FACB97509B459D0459CDA7DCF2B084B91C8550A9F2A92C490C270CA90C41018C4F205E5A33B1584FB54BB1F163A3BE83DF0C721235E24D4410F73B2680C0835B840DB4C4244C1F85598C2B3A50A896F40C8FD249E99588A8AE1FD6D706A394541B942EC543343FB377854141A35BD35F10983543B17FC5C30B31F86C48F601506C21EA462420C5992F5B5DD12AE33CDD99BDD92667D4BC87410B1456929350D06C91A558A2A17964C3EAF7BDF67F743AF6ADAAA92766F36655A9496263B38F26A7F1DADB95B6A5B63F3FF9E0081F61D0351EE557199851596E5D221EE891CBA27349B4BA671EAE59473FF0C30F0DAA2F9706977C79746C8C1B70AA1397DF9F692DC4D1182EB479C98343DE7BC11D940DBB778AE8A75FF09D3BA328A032122318C021BF07AC179BEF0EE42FDE3D8A64FA0453057B1B2FF6804AC6719F6823270647E142BE5D338B1458F8CC7AD7637942AD4F0FCAB36EA6FF768273362B8EF001D50C4E5BDBE74F48DD91854A92FF02872C0963A2C2C52D0156E58BA1DF0B7FD9A178DE79CBCDEDC20CA2A226551C86835AE9ECAE09FFF19AEB6B114F7CC602065CBF6A6A740874FBD6662A3EE1D407337B935C4975D269DB7C54C453AC4674B9CFF128DC20C62CC86B9D14259E06315EAC81923D415E1F35BCCF48F48BF655E6B87F8A0B55A21AC760BDB828BA351DF"));
  test_xof (&nettle_shake128, /* 102 octets */
             SHEX("006917B64F9DCDF1D2D87C8A6173B64F6587168E80FAA80F82D84F60301E561E312D9FBCE62F39A6FB476E01E925F26BCC91DE621449BE6504C504830AAE394096C8FC7694651051365D4EE9070101EC9B68086F2EA8F8AB7B811EA8AD934D5C9B62C60A4771"),
             SHEX("2D008F8ED51C6A86910C30ECAF2B6441262D7CC66E54B39B9AABCEECAF2E96A0B5F1CBA234F45242BD8F96A57BAA98CAD302814C95E067CF8F47B2A4E6FB6BCC5C70A7A750F754824E9F970E4C54D43CF8F4E0D433B931DF2B6BEB10FE5B28F2A6B5CCE24D0C6B414DC8E3C064BD3B26BA82E4464B34C3735E935EC3B7AA36E6AF002A0AAADE986B0A15AF0E9F1AC5B8CDF3E9022CB6D2BE9CF5B30AC8E2BB5E766F528802FCAB3C3C3405F25367FEEFC12F1C7D99717F2EA62D7001B4845793B288DD9E777270608FE40586605AF8C9FF52D0FE4111E8DA8C315958694ABE386C85A86AAF2038DF6281D8CD73E201973704D314DFF7A5C3B4D6543A34C6FFC561195CB3B782C1ACC2793BA6A75974126BD3E4188FAA0E69B255336366DED52D809D6F70DAF11BA57B604DDC8359C1E5D9E8F2863EDFE71935BA18A4D373E4ACB73CCA7887B0F3A84B2BFBB5340AFB87BEAB7E3708484BB4EA2E17CFF49DAB7236BD83DD11B2761AF3E7E58452E9BAFE5C665ABC543F41D98C6478465BB67C6D9ADFFDEFECB1ADC2E22B97504590D07FC112E03D1E221CD860928D13995FF1DB025927B2B7EDB434DF01F1692AB329D07A1C6947E64258E53A8082EA6945E1FE21ACBCC1475054A0A9493ACF37E5B607C79A54C985E70A99041F3EB3D5511390A1F81FC845C9E29FC39F12D019A0DE262FA9426C123EC8C2C733ADB2217DEAAB"));
  test_xof (&nettle_shake128, /* 103 octets */
             SHEX("F13C972C52CB3CC4A4DF28C97F2DF11CE089B815466BE88863243EB318C2ADB1A417CB1041308598541720197B9B1CB5BA2318BD5574D1DF2174AF14884149BA9B2F446D609DF240CE335599957B8EC80876D9A085AE084907BC5961B20BF5F6CA58D5DAB38ADB"),
             SHEX("7A46B367BAA023EE1922A3408B145DABC97A55EBC78CFFF95FD8D6206E5C10EFC9CBE3644A49C200097BA79F68B08193C6914E016DAB98361705D27B8B046F087DF7DD998C02BC80BB173772843645C4F93080B0EA657EB0902AFDDB28963BFC845ADAFF3A431705C74D92346BD678EE699FC12E9B06E0C0E08785A419CBC944E770CCBF89203FAF5D3B629D541BD51EAFA5857F585537CA60ECE26E09D6B9D05166360417B5259DFCBDDC5B6FA5B653423FE052E98387C97AED7DFAC41274D9BD912BDD661A02CB80CAAE1BD1704320BA54ACCA5EC6854D4F83E938F8535EF287AD26F9D47D047C74FF6E4EC338222BC7A1E346ECB50A72572D3F7A9621DD0574A9A109979992F18690750D035A65B29F0387FE78E91948B8EA8EA31A074FA405AC30D44FA3CF16BEC6D44B5C5791DC2376BD1C192175058C4CA5B1E0D989D866D9E076E8252AA53C5B89D7AAFE5B753705020DD7F79C93D31F768ADE1F54496B62F6D7B126A2558E4B2A3B7DFF1F046F9FDC7BB2902389E1C0FA49E00F4E892D75FBB5A18B22F863B7D57A8F1895B13AC99042B8BE114308B730ADCC2B8E5C2E6DB1C7DF35CC9E8160DF55BED2C9083A2BF79C8F141E44D48AB49DA6A77D7AB22F4E8A6401C9A600657C4E3D66700F452219BE1B5008344FD3B6E76D10445853C2814D7EAF753F57A70E0644C94A2E3D21478ED44D6C5D568E68231AA180BB"));
  test_xof (&nettle_shake128, /* 104 octets */
             SHEX("E35780EB9799AD4C77535D4DDB683CF33EF367715327CF4C4A58ED9CBDCDD486F669F80189D549A9364FA82A51A52654EC721BB3AAB95DCEB4A86A6AFA93826DB923517E928F33E3FBA850D45660EF83B9876ACCAFA2A9987A254B137C6E140A21691E1069413848"),
             SHEX("224E3495D98A8729F88069BC2578F98F76EE3D73B1309F75357EF595D217E4D851BEEF015DC7BB03F9B8C80AD1B132AF1B99F42459527FB67ABE4B5EB3DBECB90E679454903BFB51E9ACE93A4784722B2D0453F3400632A86870A427DF2AAF93081AF99CFB012807766C440A010E6EA068C2BFE388F26D95BDB1947BDF1E364D252C81835AB7083AE5CDFC12EEE193016E5BC5A90C2529419DF2C0E14F8FD900D4B9519705221C70DB7178B81817B0BECACCEE975B396E63383F03FC8097CE6EE9A612D53EBC572738B17442C8C5F2D8E7C67073B6FE241E6DD00D356641552EED0D1C32A09A34223F1D3E0305A38675EE066B63155BC2B7851E0F51967DF90AAC3ED7F88B1EDBDF123D02C2F4BA23FD2F1DA2CFC2410CAD5188975BB5465F502B33A4F41D7B2DBF63B71085DDEEEEB6B9BD0C9F89F4424F55DDA4804E86E2B983130B46403FDE7B70C5006FEA04CF3350FD6124C735613DCEC7696FC3F75CEB8238D09D1B440E759AFA129788A91C43FF7423F05C6E9132F644458CE8369DD714CA9E4841DFBB29B942475058837FFBF1F59706915EBC473868A718F2887AAC888D2C7FECFC1F8E5CFDB5D3B2C71CF1CE400A0CAB18982BAA41B84415A0A49A25E18A8F4575B46A544F1079431E5F07915B92D8BA512340F2E43B689DEE6118786A2410D58CE5EF6E8446EE49CCD2A50557D32FD1E9E0954A4373545DA250CC"));
  test_xof (&nettle_shake128, /* 105 octets */
             SHEX("64EC021C9585E01FFE6D31BB50D44C79B6993D72678163DB474947A053674619D158016ADB243F5C8D50AA92F50AB36E579FF2DABB780A2B529370DAA299207CFBCDD3A9A25006D19C4F1FE33E4B1EAEC315D8C6EE1E730623FD1941875B924EB57D6D0C2EDC4E78D6"),
             SHEX("CB85425A1CC6A4FA6D8593BD03206867737F8D749D2BD7135ADE0D7A1F216FB92AC5C4897CE355835490699A0F715A2357100CB8932BEFADAD55C0B5D90C909F0272F16BEEE79CA7F7F1002F711BFA3BDE142FA6D5C908E8F5923291E0D48495E50DE065CFF609A24AAF17491D228E5AB226E21E790C13F8917C1284BE0434C8B2B2A3CC672D484A25A47820390FC50F2736FAF26E65E09DB68F52A9A65280789017F1336250D41C0DF4C93AF9E94036E598E6F06494068F4A59813D1396B00E65E179572FBEC6D30DDAFB3A898CC244E1B0E82043C1470B7602D0A65E455D297770265F72D82D584657316E52CAF4C8BA8CB4E3ECD82433DBDCE526EBC60217529CF03548CB114D7208B93726B5556E4DCBF2764712144024510AF30265FD4FB13E38BA677B34D69409F91F6F44DF602C9F84362767D1E1B587054FF7CAB98A3CD4BCD2319278F0144DA884306660DADE52096A10928C5AF800819E3D890DBCDBA7CDF1BFA4D992074D60CD6BA33C98CE2D9D849D60EDA8307B0F509AE78215BA900D5A571B3EAB6DF1575D7D2B8CAC2F03C4B552E5247AAFE2349F5EEB0AAA56627EB4D5C33C92D8B1FE5D591FD25828A5114C0F8B3B6E799BB69ADCFD03566C7959D1DB21B6FEB79E02006D1920CA35CEADA75B22A0B8235E9C6182872377B56A762AC54A719D8AED8DBA18AD925FD21C77E84D92A75F3FEB8692144F7D28"));
  test_xof (&nettle_shake128, /* 106 octets */
             SHEX("5954BAB512CF327D66B5D9F296180080402624AD7628506B555EEA8382562324CF452FBA4A2130DE3E165D11831A270D9CB97CE8C2D32A96F50D71600BB4CA268CF98E90D6496B0A6619A5A8C63DB6D8A0634DFC6C7EC8EA9C006B6C456F1B20CD19E781AF20454AC880"),
             SHEX("93BF13DCDDEF177EECFAC1D8731237CBF8CE4459ADFFC34289016A689E88E5B000553F08989EF2B007ACB4AC69EDEB6CE2E5A7F85C9D75F7656858A7B824472BB89C2D25701CCEF18E8707D2C3444A28AE245D337CB3DBD4E2000D7C44D474CCB34033AB203D9B1CA69816ACE4569771E1F7F66C7E482F81A332D8D16D0A4AB5142DF5742570ACD048C52E8BB777D68041A67E870946AEC83DB5464AE2863020A4126EC32E1D64B5B1A21425C66471C7FB6C5CACA4091A96ACE9BB25DFE5627562892149EF463ED60D03549CB9643E9BC7403B244A0E9A5A8A38FD785984661C926FD959117C229EBD357847F853768F632A0A2A0F1703AB5E130524A33F825731740AD0B08DF92F8FF6EE73C1E60A35DED2882675C6A7941DEFF0D934BC2BCE9005DFD49A9CEB0F08A27CC972A66232A068A93C2B4A143C57387C257218E850CDAEB93C474C8B5AEC355BBA3A4D0B09FA9F7F32534CAAA727120C2D937CEE1039DE63AA5AF5803F7A1BB2F203C87DAB85ACA2E68A88C9C7584C1444024109F9E02CB994CD7C89E87B976A6BF7D75E570D4A11D15640A4235667669BAFAD64CF940158BBA56184ABA70E6097EE4F2379727F7BFC5FF67595B3DBBB7EFBC13EA4AA9C1093EE2174B5F0304B10A089AD1980C8A64A9B4E29F68F2BCE9AB7D31AC30B9ECB5E009169413B64D661E7BC6D909396C25EE3E64DE744FF937F2498A361"));
  test_xof (&nettle_shake128, /* 107 octets */
             SHEX("03D9F92B2C565709A568724A0AFF90F8F347F43B02338F94A03ED32E6F33666FF5802DA4C81BDCE0D0E86C04AFD4EDC2FC8B4141C2975B6F07639B1994C973D9A9AFCE3D9D365862003498513BFA166D2629E314D97441667B007414E739D7FEBF0FE3C32C17AA188A8683"),
             SHEX("0DA9BF94215FDA7F85A36B7D31080F48FE09C92514A844E7F8E5A54E422997BE17E0EAD0C8CAF75A27B4D721F63DE3F4A9326E74081EC77A71DE52AA4E7DF346C7206C8ACECE21F739352B2EFD6446878075B585735471A189B32F41CF0C48C0E92BF492752792D05E324FDDBEAA786DECD3C0E24BE3CBFDF160F45CD87CF73E61C643E46772EF8C62DAAE119E0010FD84530025ECCE198B2B86FB6E8EE88F6C47F6B4710E5764F405E7856688EF030E5FB84B24B48292DF29260B918DD4FC54C325708B83A9E44AB92F67217B9EA0A488C9B45920B76A7644C38D36F9AE1A48F56284CC354BF35CB53F0E7A5745955E0E9A77400ECF97B0DEE9F4B32E8E7DBB100A1749762821AC0746CCAFEB49162B0115B4E3D26EB3B9FBF662CDCA1CEDAF5AD8933C7EC1CE6EBF154474FDECACE144F8B86E24B85C4FF0EC29736BB0B778AC071D2286EA20302C98B6883520D5B4BA22AFBEC4B6188FEFEEFBC487D147622B3E97DAB9319DCF6C7EFD3374FEC69D575BDAC70D400923234CC41435A803DDA6F1C1E91802F56D048B5147A1B9778FDCE25AFA4B0846D7D5FBD04614EA62263370B4CAA6E87E0B6D2BEFCA78E6A0B7E5A6B5E891F7BF4ACCB221359C98B3091C75FFD25805EC452F7251D8720AE51AC49620C6AD376F13A16A269942842FACD25C2F68A4B93A5A7CE9ED38C7285B2777AF32307D84A84CC5F937B7950689CF"));
  test_xof (&nettle_shake128, /* 108 octets */
             SHEX("F31E8B4F9E0621D531D22A380BE5D9ABD56FAEC53CBD39B1FAB230EA67184440E5B1D15457BD25F56204FA917FA48E669016CB48C1FFC1E1E45274B3B47379E00A43843CF8601A5551411EC12503E5AAC43D8676A1B2297EC7A0800DBFEE04292E937F21C005F17411473041"),
             SHEX("5BC15E410825D711C7219161491C5F068DC27D0B66B5A7F0AF4F86DBA7392037E886D236FFC9EA19511AA43FA27DD2C4B8026D074B3033D7184C95858DB5F237A84E93065F9AA4B40123AAF318356CABA01C20C97DF8ECF528F56B22A927C705F6DB6C2E64B60D36FADB2D867C0F5A515A2DC7DF26B151D700E43540B82A0CC2E105269EB6CABD3148871CCD3B2F36B101D051D60E28FA838C67E8F655B1532976E512DB752A2941DBF6FFBCFC0AFED99D379D4E86B8C32327F6105BB75D7669A2D69BD7A2C40DF3B13899377A29AD7D724574F545CD91A67659B3781B788B4E9EFB137BE1B3B8D93C491BC70581C241B184C915B651737A0B2A99B50E8786DF067CFE443879D4BF2EFA9970A2C214E538C2086AC7BC47DA706513BFCC4CBC6D5F387A9E3BAC080138F2D713C6C9AAD2E0CE7A938ADF4BFA3CF688CBE7E22FD5707EEF2D707E5004CF90DC880E3AF23A435A2682F9C117BEAEB8F5A57F521708EF7233BC6AD5750DF2272EC379142283BF5EF492240335740F43DD5042124C9B37E726438357472013ABE81ABE2C32E8E859E4D064C3C4220E5F0B5DC984EB1A8BBFD6DB41470174BFB8A1ED5DF559A65B1FF02CCFEF42DD546CFF79D34FEE13317BE00EABE8C6FF6D722D048EB757E7A38C65CA1A7B2418994D53CE8090122C94A934554D0BC3856E041D0A29F8539246924F7634B5265708208B4C59DCC084"));
  test_xof (&nettle_shake128, /* 109 octets */
             SHEX("758EA3FEA738973DB0B8BE7E599BBEF4519373D6E6DCD7195EA885FC991D896762992759C2A09002912FB08E0CB5B76F49162AEB8CF87B172CF3AD190253DF612F77B1F0C532E3B5FC99C2D31F8F65011695A087A35EE4EEE5E334C369D8EE5D29F695815D866DA99DF3F79403"),
             SHEX("D289FC6490F9DFE3335FE9F43131ADE538280B78DC2E16822D5FCE70E0BB307FC497E1390E04108C66C682FD6EDC0B47450AC2FCA43C151982F2EDE9F82196CAB4381F873AC6E21085B2B64C21A1A5DC59C440ADF566159798D6773A09FC737B6CF1BFE0CD0C9B90346F1290DE5E6AECC1F464E1B3E522DF1A986D8AD7C2A42B1A6067A3448FCF9E4BBB2335A3E4276B330C421894BACC448DB09F72751D2EA50873A9DF900C775D5E36579DC26830C37851BF456085925185B91CCF9A35B36E5358C4CC4F1C807D285421192BE028CB7D5D3DB67AA11AB666382641ABDE3F1690490DE0D556B0A25016D5B7171AE26592B5C460E41296C01222D074B8C53A9F73AEC29C63E54EF0D68FA1E57C604A2F4302B7AF05B20D28E5722B0229EAB76FDCDB1D2F7E58EEBE4BFCC82B4E4B6D5E6FAEF0AC442B78B483622EBCD98EFA538527D4C0E72526D9CFD1F13E24613BE03A79CE2612B19D7C9B639D3C2BAD28E968E7859D224958C017637458526521C770DCAB61A84181CD0E92AA485AC2D3883070E8DB9FB5F06EB5E71D10C175E0D7EC69947D1FBFF8A857362383368F52C193EFC3DF83F5A9B6DDD53D736A3543D46388B2845632C00F0676715CE8C1AA3329515495EEAF860A9AFDC9630B06E9B5CEDCC7CB571179F35735D02D185D7B03EDA48D30FC86C239AA3580816F3AF360FE78E84364872C2CD9AC5EBA876AF94F"));
  test_xof (&nettle_shake128, /* 110 octets */
             SHEX("47C6E0C2B74948465921868804F0F7BD50DD323583DC784F998A93CD1CA4C6EF84D41DC81C2C40F34B5BEE6A93867B3BDBA0052C5F59E6F3657918C382E771D33109122CC8BB0E1E53C4E3D13B43CE44970F5E0C079D2AD7D7A3549CD75760C21BB15B447589E86E8D76B1E9CED2"),
             SHEX("B8C21DE1F120C5744352EB71EAEE81E9DE8A8CF6EE3E2394438013D9B1D7780803C28186E334A34B4CF17550AEFB58DE785E358D2AA6A61BD59BEB1B2F494CC9976D67085210D64FD1C8FFB0FE69814C2048D0937B17E79AC9CE6303562065D4C79438E87C7687B8B10AD1E8F9364EB9EF997B09B963C7A57FA9EAFED8571F5FF9B7981418AA8904FCC2267F3723FD67A427E7CD3D5EE6620BE53367ABA1BF9D2EE74C33BE1049356CD8A3B4AE481748872717394D6793A25CF02F95B6EE62EFD8E269841234F07041D5C3940C7BDFF37686B8E72C23342578FA0DB4DC21F4DC03D3539B851F5441B54BCD991322EF0FCC85C14DEF9BE72B8B4294EF34A6620854C291F91CA67152B48560683DF13F4DE00216180391AEA39BDD22D563E48CC622BDC72C990F795CC6CDC280BF917743A04854BBCAA2CF64128DE9C42E38B40107A97F71CD1B89DFEAF194674CDB35E036B45CAC35C45EB2ED6BD9F8ACB9402BE9E57ACC00E0AB6C5B32EEBB9867878C23F4275DC2A8754328DED15E206AE31B60BEEAF27F7E2A571C42994414355C28715621D39566FCAEBE6251EB5955FB8A9B1BF4D897A4931C5E7C29F27F28AB4D2B9950A81DABBA7EE63B679DF6E5C1DA4E23BB9529946E4F467A73EF02D1FF59D7141E6D59E14450B7A35BB4795EAEEE51545563BDF99BDB5DC0B17CC60FA04DFB3070930EE46577250FFA863528A498"));
  test_xof (&nettle_shake128, /* 111 octets */
             SHEX("F690A132AB46B28EDFA6479283D6444E371C6459108AFD9C35DBD235E0B6B6FF4C4EA58E7554BD002460433B2164CA51E868F7947D7D7A0D792E4ABF0BE5F450853CC40D85485B2B8857EA31B5EA6E4CCFA2F3A7EF3380066D7D8979FDAC618AAD3D7E886DEA4F005AE4AD05E5065F"),
             SHEX("F1E1E702011F4A39197D769A6AF63160BED980DB70888266AFCB744A8FE41F1BD9A1B668B4A96A6C3C2FEA3EC1564592134854B26E31102D1B95F112118C959475AA6F2DFC300E389FD061FD6667DB36C32055901609EF59B3A8DE4F9BC05D52745C78E8D155738F30EB3A8E3B61B36AD34A6BF7C59B59C1F973B9F94C9ED61A746F85610B060E8BD04CC9691E7979EBDE5E9833AEEF21F9DCE96F61A21D2B8864F07FDCD6D3F2BB3DB29CECB6670E859A4962A5FE0179236BE7426E9C9B2A0B97D11B6C9E049ABAAB494513128471F88914E506C6A5006ED295F436410CD24B4CD1EC5AA0CA7A66E29DF6C362389C7A9B731E0720400C0923BD8C4A98A651B62AA8AADAA5FAD481E08A17A2B3F86AD23FF6303463DA6300B47483F7696962D33AF1DE2739C30346665F60C90FA35B74CC240D3831C01ED8CB91FA34A584DEA37846CF4A0B43B5D1C26597B2AA1C973A070F120F08C78A9FC8D524FEC1F10274F03A4A7E8E8C4C1FDB415A1AB18B4C1F07E1936F81FFE7F2A0A1A815E2F29334F1B1E8029E7E12A798187CCB20F6C1EAF91B14B61B2FE4C3852BC670295BCDAD418CA7FCF39C69FA9F9CE54AF17E695F1D614BCBD49FEE3935238CE59D9E378F3E596B34149149E14D9B81F0BB09C2DF8E6E632436EBD525BBDB657822940D25713B737A2DFDCDF0B4F6DD7920A8D1E69EAF7B764D0A469CC1CE6DEB70DDD69E"));
  test_xof (&nettle_shake128, /* 112 octets */
             SHEX("58D6A99BC6458824B256916770A8417040721CCCFD4B79EACD8B65A3767CE5BA7E74104C985AC56B8CC9AEBD16FEBD4CDA5ADB130B0FF2329CC8D611EB14DAC268A2F9E633C99DE33997FEA41C52A7C5E1317D5B5DAED35EBA7D5A60E45D1FA7EAABC35F5C2B0A0F2379231953322C4E"),
             SHEX("98326BE2F713456EC96C941FA0B58E311C226474487F78734C8D0514AFDE43293116D6C33FA3EC687D1E9087572C68479AE1080FF8B2238C11AEA13D9A020A7B3FEEE56FC2FA69914489EE11B17CB9D3AF448525C3992232B58F2182AD27D93AB88B22DC94368D10B112AAAC096DCD8AB76250984EE4E4D3CCAB7FA3124F7B02D05111D97EF0CC52C5B8221383A6256C42762E546ACD6691DE9810FC2271D92834DA1992CF4031B98CD0919E01D35DDDC738D6A305C6E400049A97E14AFBF40822558832D23165CACAF16A514EFD6D207A95F97C330CA333C810C19AEFAE12F3AA267CC9FCAB0B73D2B7E5F3B007CD8055D0C4665C9C53FC818A53ECD9C0DA776EEB3904409ED63B337F755CC7783DA8CB713B51C720C2826711FF36D98723C8B00C4853EED1B3E2F8B8D74ECFAD079A49F37665DE5031F2FBF54DA6F1FB796913B9DCE2BA1206E4255F39C314A34AEEA01F1965C36CAF89A2F7FB9EAF7E9C97E2B70636ABF0ED00F59D1DDAD409AC33055E6F7517E962BC535EEDFCBAEC203F91FBD2B13C26A20F9C2236C55F4B506C7AEA579ED3A0204BC67002D158D80139379DC23DA06B0DEFEA079E8FC1D51976757E9A455719C45B8965959A2958E75B7E6DA966B86029D22FA50841494303AB6B16600324EF918AFF9DF161224F1A7F26B7DF364772724D4B1AFBA11B5C114563F9DA2604A2D1B60BCC3C116A5E9C58"));
  test_xof (&nettle_shake128, /* 113 octets */
             SHEX("BEFAB574396D7F8B6705E2D5B58B2C1C820BB24E3F4BAE3E8FBCD36DBF734EE14E5D6AB972AEDD3540235466E825850EE4C512EA9795ABFD33F330D9FD7F79E62BBB63A6EA85DE15BEAEEA6F8D204A28956059E2632D11861DFB0E65BC07AC8A159388D5C3277E227286F65FF5E5B5AEC1"),
             SHEX("D539B406312A6A14A2CB8C2B2BCCB28B7C164DCA4BC34564E65B807F59809E0844FC897BBDA2FD7A0082BE317B8CD94C2B2DC8A625691D56AB21A2C854CC6952A4374CC34374AAAD9090326D493EAECC945848FC4653A15E2566A960CDEFF139508882B766A1ED60CD292C696CC75136A6DBC0BDA628ED1E16129A9025D715B7F83CE3A35F4EFC7178BDAD06FE6030E086D416775BE9DF9DB66A365F56129F7B25A8701502BD36CD439A4CC122DA4D78B1D832F5033123D9A9D72EDE2ED7D7815EB7A3EABBD21E2E507DE4EC4AEA0B40792337953DCD9D7BB952F710F8BBCD7E8CC1ACF3B7AB4EEA9C066D9D0D27AAD600405AB5DAAB1FE4551CDDEA659307B7ED1913BD8635AA409754A1860377E0453A63EE7C2C6CD1A33099A81240B92382572CAF3698C809237548002EAFCEA5BD042DB3A7506189E3FBB4DDA6DFB1C95BE136011FA127B0FA3A4A386DA3C327B072417BBC5346A5060D770933142B6CEEEE1021BC0DE076C2FC8ECF7A16A6CB752F39B8FA1EAD88A7D4364204BF613F5B108E2FB60854F801210180B9DC4757D145E715D4E2D79A7D4E8AA891977DFE08A92AA423A05D22B06C6A3D1EA5953214855BE7CBE9C2E62F2A7171BF014F3F077AD244FABF9C1584EE12DFCBEA4E4816C470AED4FCB900A5175C35189380829E92805E47969FF5A1C3B52FD850FB404BFE189866844E7B586DCDCBB236910011"));
  test_xof (&nettle_shake128, /* 114 octets */
             SHEX("8E58144FA9179D686478622CE450C748260C95D1BA43B8F9B59ABECA8D93488DA73463EF40198B4D16FB0B0707201347E0506FF19D01BEA0F42B8AF9E71A1F1BD168781069D4D338FDEF00BF419FBB003031DF671F4A37979564F69282DE9C65407847DD0DA505AB1641C02DEA4F0D834986"),
             SHEX("60565A24B70C76463CDC5397BD0C077B77CC70CF33128A0366A7C30161E21C32F2F024511A9605040D699E1410311EC3E8952938C91D73AD71803BE3F7AB60E2AAC7A91D62FC0729294BCBB031DDD5926F775FA85D25C85E1CBC349237F9D1E400E64C15A797C1651FBE5B8079D586BB751E1B765C36A40008C9C612A8A67C49A2D4C15001562D7F7197CE107AA7601E490FD3CEE738D397A5D9E1A9568385A243975B9AEF5E24C4235CB3612527B7ABB7DCFD22A124F9BF5E1E169F2F58512B8AB04F4953D40E6B1D4FFB920186E4C0751F5D8516F142C426F3893B6B4F45567461059968369B3D333B74F9A596E7585C6646AF458173500B052213EC6074890AF149A6554A92106868A64C3957F43C8730DFB3136040EC688ADFD0E79EA74E07F6A6418656343D24E446013ADD76058241F19F5CD42F256F8E98C7A5F7B9C4CE3F5D19CFAE69AF8DE868FFB5807DDE0C4A5DAC1409297C5B8AD27C1E1B8FFD570BC41F71F21AF7C9639BC6502511FF9D6CF63BFE2E8AE791B50527D79334CB87D9BCA919833D7123B8D77F96D07106BDB50A3C4B9860E5E1CDBFE4B49CD4063D0C4796BFF2FE1507F43CEE6CA6BD19038CD6B7118120C25EA0DD1BAD05018DF08DB67AF168880C284DAF7DD33A7BF18248BA7E79BF18B68C3E14664D0E1349D458CE7E128999CDA5EE706FD48FAD8D0489CA8C5718CF3D2C16563178AE3309"));
  test_xof (&nettle_shake128, /* 115 octets */
             SHEX("B55C10EAE0EC684C16D13463F29291BF26C82E2FA0422A99C71DB4AF14DD9C7F33EDA52FD73D017CC0F2DBE734D831F0D820D06D5F89DACC485739144F8CFD4799223B1AFF9031A105CB6A029BA71E6E5867D85A554991C38DF3C9EF8C1E1E9A7630BE61CAABCA69280C399C1FB7A12D12AEFC"),
             SHEX("7FD57AC85310E74667C025609A2CD37D12EB3FA49F9D9B9AA4568C1300C4B5FA69B6591F82D4C7C5B25EE6F4A6E2521090ACA1588D75B4F6BC1D58B106E825BA2409F84BE64AB520CF4513A7BFF52FA7A67A761B86E9D749FFC71532F202A2B619FF772EC25C00C269E7FF10EC38B93999DD8BC7B3DB40CBE870CBCEE6C0A6842A72F978BA9AF677CF4F055762ABB1C8DA60CF57E9352C48B0BDC0031FDE8A146B8D881DD4D0D05BD5D14B52C89ADB6A27CFDA2B3D7650EA650BB75D86748A722D7E3003E10E43D89AE21E9155231556B7335F576A922248EA2040DE648A0D49D8EB88DA240E586DEAAF9895CB2F35CDFE60E61D6643E5B38FBE40E82BCC8462FAC98F078DFEF5D42E1D562997EBBEAEF637FA8F65510E604E99933D92798CA05DFEE0E935910BE5086DBC8BA9493CE52CC49B85B062164F6BE8EBFB4F0C27641B65C397B92182A47D4E926C6D6947AAB598EA0D012EFC47556EC7F6407A2DFB3CBAE71237D968EFCF9192379384FD2DDA510BAD5B30F01852064938A17C38B6166170FE15F09F645CC740383715DC35C98F6E2738473D324C7C6F63E76124824554B0EA34120FACD2826C7F2A717D2817D871BABACF1F061CAC045EA61BED8D4AAAA1D41A277E50C09606B1786BE65E1ABEF10F6DC6D28EB8206D3E3B8A56FC83728291420ACBF606C34B2B3E852381EF31556FDAA341E05BEE90DC9AB7173D"));
  test_xof (&nettle_shake128, /* 116 octets */
             SHEX("2EEEA693F585F4ED6F6F8865BBAE47A6908AECD7C429E4BEC4F0DE1D0CA0183FA201A0CB14A529B7D7AC0E6FF6607A3243EE9FB11BCF3E2304FE75FFCDDD6C5C2E2A4CD45F63C962D010645058D36571404A6D2B4F44755434D76998E83409C3205AA1615DB44057DB991231D2CB42624574F545"),
             SHEX("CC03A5BACC26E1580F37813E60005111261AE521159086C7A6C3087E238D2731A02BE233200FA54BDED93D6959DD03B71A1A4C7CCA20E9685406F70D722DFA1718B3A169BE6A8C5FA275EAC81667046A605AE8514BAC24F216358A27F036BB618951A1FF5AECA2F64FE1571CD01EC7139F86D370BCBA6A721520BC115330398F55691D7A212BE824F27AA10305523F7C11262D2EA2520D72AF1E34A887BBB4A8DF9966FDEA757351EB67DF1DF9F9DFD7D7892D3CFD1F0B8A3A16AD5055D5904DDD900E4A5F94941FA796E8F32A04E807EDE1958191D58551378A57087A62F48E3AE2852EC03654CB142D2B851C5995324D15927F3DAAAAB188DDB6665E08596B71D9C2B740A15F8E78F02C6A820C79C3223A98D79D225375F6375EC21A5216A9D01D0238FB456857E7D6611CADC2DF844D99C6380CF59E7BDC268CE91D733FE00EE99A50ACFBED98E6AC17DD8A377EE4AF9F318BAF14143AD4873422DC56AA6E4DFB6B0C312AB14F16105D8CBEB7F57DA85644005CA4294FC8667C1CC1FC78256562CD7AEA9779899F431FB581AD6CE91A46F003248A42BD286E9F4A0909DDE99E2FF674D4D5E71EDA126995DA15F9C00F11BA6691E3614CE6FFF05AAD6B7C63A72DA235E900698D09BA3787742EF61701EB25C3D70927272441709D2625B23FC97FBFEB1FC06D3693509424FF5DBF005023CE06D80D7956D1DC0030BAE8D0BC"));
  test_xof (&nettle_shake128, /* 117 octets */
             SHEX("DAB11DC0B047DB0420A585F56C42D93175562852428499F66A0DB811FCDDDAB2F7CDFFED1543E5FB72110B64686BC7B6887A538AD44C050F1E42631BC4EC8A9F2A047163D822A38989EE4AAB01B4C1F161B062D873B1CFA388FD301514F62224157B9BEF423C7783B7AAC8D30D65CD1BBA8D689C2D"),
             SHEX("FDAED26927DC53F12F002A9C47467A4EB53D941B08106D6CD0BDFF6021B6AE116562D2255562F70BE884427A8B6D593E0CB7040ADDB87663E5F03981992435C2BAACE87C49CDABDA178D0663CF81E96C2EDFCB6D50D48ED128128F6F3B56E33BF33187BCCB5C2BCEB418DACD4C820EAF880B80E32F00B7105615CE305065716316221019A84CA2D29FA57565CEA5239484B8E9740AE07048BD0F75B50827C138979FC9671B4E684952DF9A681E0C7ED8101DF50620DEFD82132EC81F5073991E65E1DD72A52B596CDAFE6C5B6D9205533EF15D137A1FEEA7564C03E3A19E2B9F40600FF1A66A241737EF7FAA29EB83793BC40D1E48CDACF3936C499B572AEBF2DEF5086354DF7DFE682B9C281F4CF68405D0268168F5495913F930B6009357867398ACC5F951D55D854B0C42D2E2127CAC73AD0B75139111414F6D6A68E01CDB794EA6E524E45A62A82970D22168F37ED72575A513751368718AF490E28ACC47D2BB7D59F567DAAED80093929FAFC0EBA01F66D67DAC97A99886D0B11EC0C4C23A593BBF05F3F80A047A07C883063A9FB967CB9A796D18168D0A7F731132AADB3D105B1475C583BCD4B939015F460FEE9B6F23B0D3BB3E2E7B7C5615AC8EF7305567397EC4C1DE8EF5A05C2A14196AD00660DD0FD11838A707C0FAD38D343AD9D65056A4C3701490B3CA235063D7D21BF02FFB2E0DA3D9B9092E20DA716BF450"));
  test_xof (&nettle_shake128, /* 118 octets */
             SHEX("42E99A2F80AEE0E001279A2434F731E01D34A44B1A8101726921C0590C30F3120EB83059F325E894A5AC959DCA71CE2214799916424E859D27D789437B9D27240BF8C35ADBAFCECC322B48AA205B293962D858652ABACBD588BCF6CBC388D0993BD622F96ED54614C25B6A9AA527589EAAFFCF17DDF7"),
             SHEX("F375FB504784706DCD5BC3A500BAE8DE6630009D865739A0B30C81C74E0379A59996410854FDFF996B46F9F30BEDD7DBA1C5801A68D1D3792119D7FEB29A648E08502CA04637AFB9427027D59E9DA235BAA4D602AB2FF257BB5FBA4AA79C9D285E99FC4314685181020C2E9C1CFB22D1E7B1ADD9C037A7F81B7E0135F9628CD67B8591271BC12D025322BC21E7BF4E91B049E291DA2828DE012384AA65E3CE26F98186828A3C144A095D82BBF5A325E41DC28A33828D9717BAED4BB8BC50116CF39C8EB6DF037E29ED6A32FC5039EC1704E199AFF83768E58F1858ED741DDDCF75E8B2621BBBE14420217CDAE7561C24213F76530204141EF5C7CE0A76C587C2223181F5F2DF58BA5D89EBD209C004F5A2155CBD896A17701B19D5CD5D555F2A964BFBEF27D22FB428DB161C0AB99D040AF196EFB7640EFC55B7CA8E1BAC86F4368F612EDB120E2A7CA084B9F5EF8DA0A11136C760DCF4426CA98F20514049BEE2F9E6F66AAFC4225CFBD4397BBBFBA872A47BE903EE0DBEBC7EF85A1395B14504C18A8B5BC5416739DAA543DFAD6B4B8A88BB87B388A6974AB90756E836DE528869EA2734E75D8AAD84EF1A60BD371190808436D5C7632A2F027C3CCFE8F3D94B0C262FB7AC99103B0688238BE68F26B8CF35876B6E214ED1973FB44B9C3F3DC7D7296BE8E150F82E80594E34A3E008EF40D9647EE26E1E4CFDD8DC673F2AC5"));
  test_xof (&nettle_shake128, /* 119 octets */
             SHEX("3C9B46450C0F2CAE8E3823F8BDB4277F31B744CE2EB17054BDDC6DFF36AF7F49FB8A2320CC3BDF8E0A2EA29AD3A55DE1165D219ADEDDB5175253E2D1489E9B6FDD02E2C3D3A4B54D60E3A47334C37913C5695378A669E9B72DEC32AF5434F93F46176EBF044C4784467C700470D0C0B40C8A088C815816"),
             SHEX("98D27AC62145261BF2CDF99F8F77AC04697D2FE6F1531D3B1E113A7BEA4B9BE780394CDE5EC43EF820B0D39E20A287B29390A68091A8938EF5640437A0F09F5262BFB4C55F703D44E7E21EB69C27566B66BDC95A525B988BE281243DAC0CEF5E04195B318CDC96035205CCC556AD42B220FCB0E01516E66D8CA2D045AE0AF7FFF91119CD4C92E664FB339203E8C0B3766C78874DC85D48FE668BACE760A96F5D562FE4ACE97FB33C93F160057C0475EC440111C46777B0E2C95DF666EC753057596713FF452EE59DD7E1D4DC278238518E23C78E8882393B86427E8F6188D7A08D770B2D19E08CA672C89582431F30CF4A0CE89E87A635EC2E05765DA600115B704E35B787FA90AC545877EBC8A5616EB998380D4EA1C6B127844A422789B9B3A41DAEE940F75CF975B464736616D80D89387C171D919874F38453AF44ED16888212AC279D4890CBB5672254D76D5FB922412A2EEDA598087558B1E89528FB1EA285303ECF4843B18EDAF6130D1374273DEDE9EAAFD6E6F448547E96AA385235919600E4BFD04CC2BDD89E6860ADAC61E73E37C563709F3187439A305B9EA8DE8EB552A615712451D1DA495A8E02B9733AA1F7B78DE6C0F0CF6A379959EFBFA8A78CB6E93A758889D826D7E3DC741A10BFB55544993683B2383419D3F5BFBAB13EB1EA0575E528FCDD10FEDB0C0133C8E78FA0DDE413FD951262E0CC958D2850"));
  test_xof (&nettle_shake128, /* 120 octets */
             SHEX("D1E654B77CB155F5C77971A64DF9E5D34C26A3CAD6C7F6B300D39DEB1910094691ADAA095BE4BA5D86690A976428635D5526F3E946F7DC3BD4DBC78999E653441187A81F9ADCD5A3C5F254BC8256B0158F54673DCC1232F6E918EBFC6C51CE67EAEB042D9F57EEC4BFE910E169AF78B3DE48D137DF4F2840"),
             SHEX("6095C8CAF02D995DAAC5F6CD94A33FE0CFC2743574B352FC35A056135D20BE12FA7826DD41BACCA18C90060C08C5BAC8B3E706622A2BC8AAD1FB6EF93259A5B8092540F68497F09E47476F9D94111B2FAFE27CB04DEEAFF43A435E2A5C0E44AAC6DC3104453896C9C702E8F59BB61A62ACF889174955BEAC0319FC897BF989C97307A997F4570F3544E5A003F22A7C5295705201F025EE7D6B5B278DEBA559FFB82F22AFF5DB6D534E6C50FC4E55FDAA0DCDF977178DB7AC2D98416A6E0F5AB5A49C29E414BC0AC0122D608A592C605E418EDCD083B26429BA0A1A53584EBFEB4FC005F144F04CEACC971C9EDD23950A1B7E6A4A9308C517FF80EBB1610F8E78D963E222BF148659BEF6B0C8BE1F466FE786128431A9950D3C70493D1CD72731D0A3C956E34F35339AEA578F614806DBAD3C02389643340DB162661003D2CF42F1C913A7FAB77EC3B9E8E3177DC0394305C21D6E9713F34516CEE321992E4BFCA4DB5DD03E240A7FDBC83826EFFF82E2B36BDA326D8FD0C858B00E02A83D0DF7216A475CDF0A37B0389095F501CF47C497E23F199D711991FE71DFB56D789F30C86D3BD17C8BDEF8A797BCFD3EBCD5E62CF99BA0403B79BBC8A0FCF4FA7C4E65B674A796AF85E7D09134FE214CE8355EE337FAF6F04660753C1A96D95964E587F72F0FA5BFF435778084EADB1807DC66772850152AEA381866202C8A38723121"));
  test_xof (&nettle_shake128, /* 121 octets */
             SHEX("626F68C18A69A6590159A9C46BE03D5965698F2DAC3DE779B878B3D9C421E0F21B955A16C715C1EC1E22CE3EB645B8B4F263F60660EA3028981EEBD6C8C3A367285B691C8EE56944A7CD1217997E1D9C21620B536BDBD5DE8925FF71DEC6FBC06624AB6B21E329813DE90D1E572DFB89A18120C3F606355D25"),
             SHEX("765A48E25EE43AC1630C65BF5138F5F792095E182483DE8887F0F56ABDFC0526A5B9EE792A2533409F3FCAD2B9CD43D7CFB5454194344169DEA3FC67CCB5B9969449454E18869E7CFD821ED0F3D655F5B33C528BED1778BD504B2C84BD38DBCBFBF5CAD7E46F37394622C4B065686773AC0171EF70717E8C6A887AD952343D02EE96B4129B3F189C631A719910F543540DE4AD65993B634AFCF313100771468C81B0E270769ABE2BD6603E525682AAF172CEDEADD12D9389B86DD6459DC490CB89A637F6E24C5819DFB51C817A3AC9837092D408C6816B57A4B0DCD0FD1C7AC49FC012BC9C22D16C779066A670252515938764F4E05FC57908FB67A3C6927B4639014AEA06EF91F8D43E507C48A567F36B94591EA1319D7F266722BC9AD6F5CB8196994F7DDEA570720EDD94E6C56D215D8290CF0227DE007B71E80B65301D53363D51C73303FDFE76A064BB1169862A02F541D782FC64C3DA45FDFE2FEFBEF8A9383690F1D6C695855387CC206B9E7266D6E6C9A2160074A962DE01FEBFABACC6E88CF7F724CB4B2F09862D63CFCCF1D880D956E7E5381F44568DE6BADA3923F2F09CAE80D25661B9431201C1918DF1E442F26710314D8B3F88E94502E6E9514C2EFEEC7A816E82517767BF7972D6329BB54AD1BF67842C449F54C8CFE2FFA312129995F17059F59E7AE05D21EA17F8E93C1FF4D9E2E71C9ECC9DD5512FD374"));
  test_xof (&nettle_shake128, /* 122 octets */
             SHEX("651A6FB3C4B80C7C68C6011675E6094EB56ABF5FC3057324EBC6477825061F9F27E7A94633ABD1FA598A746E4A577CAF524C52EC1788471F92B8C37F23795CA19D559D446CAB16CBCDCE90B79FA1026CEE77BF4AB1B503C5B94C2256AD75B3EAC6FD5DCB96ACA4B03A834BFB4E9AF988CECBF2AE597CB9097940"),
             SHEX("270EE64E434D9CFA8A81B256AA86959D6D9E92B10325A1816776A309002E7C9346F21BFCBEFFB2FB31BF7214572A7F437E3450AF26743B418CDEA45AE28557078D8275E406864BCA916F178181CAA3396B8582B04148249102D761111507C48722506D0314A05767E89B3D9D5125A43A2B13EA8827CCECAECA6CBC864B8B7591B15BBE0BF1093CD15C5C14B4B3705E37CD97752CFCCA1C1A4340C02298B028A633CAB0E218525E94C3E150EEB8BA36C3AC4E8EAFA5ED05D9A79519CEE2A8B8EEA9D8EBAF54624ACB2BD593CBB353BD2F577CAB60508DAB081BD29200E9A9E9357978CC15F4B98A383C4A46594303143AD019D5BCB052504EFD2ECDABD1731DB02D6FA4AC9E1D3037B75B7A5CFF8702F8EDA7F2CE848A0B96EC2F7C62BDB1E23DEC39AF110A4E6C0597D61C8AEE900276FC7B1AB2F63BC9874A00C79705D87DE2135218EF233C8939671399A32C02DB4A6D779F7F933E7F7A467D236178E2137F1E5FF53A3E8FBBDD8D01B737EA95C762B4D4D65BDDF50C527E01DA1A322151757E0F10525196D3293E1443BC7282496F3DA971869DED47DDFB7BD99E799EE60FD3CC31BD4B3956DAF6D70F79D4A5E4348004F526D90CA47D7C3A9EB09074EEB00D799F14DE7AE5A4615D87496FAF2C60D1074C194E7CDDDC0765C7FE7C6FD419335189C2D584438D3879C50D43E6CB25D5776B4802819C5DEC7CB2197B334615"));
  test_xof (&nettle_shake128, /* 123 octets */
             SHEX("8AAF072FCE8A2D96BC10B3C91C809EE93072FB205CA7F10ABD82ECD82CF040B1BC49EA13D1857815C0E99781DE3ADBB5443CE1C897E55188CEAF221AA9681638DE05AE1B322938F46BCE51543B57ECDB4C266272259D1798DE13BE90E10EFEC2D07484D9B21A3870E2AA9E06C21AA2D0C9CF420080A80A91DEE16F"),
             SHEX("652A6B87D0E770909BBAA2172F47CD05B7A07B57D9EC47396E9CCA40EDB76B99BAE0A8FAFA517D12ED62B5E95AD89560F3EACD241A2AF24BB1F02BDA6F0A5ACFA85F4F86CFF75BE9E7A57E0F182890C39ACB6DE889D24EA360DD5A4C59E54DCA7306AAE46768205C3440AD777A853DC8FE2D2EF2E6A5FDEF23BF514472B89405FE1B7342AA4FB448E0EED8A46B6F259073057571B03A4DF2EAF696838A51449A5F7DBE9644A9D092AC68B9A30034460C1891A389D06A2AC3165ECA91F203DED47B5192C8F6AA2E5FABDE178A012D94A8A35A1D446D08EDF55F4E4163C2BB9C46025D8253B3F062926D17F853728EFEFB10D69E6846BCE7E7A037996654E1C36C3A55404A1FA579A1D55FB95B30EB61EB14A29BFB54878C0D59A4209E1C746F7B2A2F2A28BB19A5F0342026BD807BB366363623BD68D4C9D9BB42D2AC67DF49DD3727F4D2AAB52A678F6F98816FC86BA0073F4AFA336A32CD55DAAFFF5D993926522D770AA44D62F6AFF7E758AD33DAE3C642318894C2DA99F43ADF2F97E53BA48657604259A8BAC4268933F1B97BD47FB30F95AD284F8345D6E013AEB9DC2478CE49CEB054555D25E36B13B322E51BF1C1E7C062E328AA440F87EE883C7760FE2F86B487BBA73163A13755E6CBA804F0EA6399E875BDB2966D427933D66D82C4D17FAF6CDFBCDA36F09621A3923B4EB93A27E44796CD1C1770835B83E89D28ED"));
  test_xof (&nettle_shake128, /* 124 octets */
             SHEX("53F918FD00B1701BD504F8CDEA803ACCA21AC18C564AB90C2A17DA592C7D69688F6580575395551E8CD33E0FEF08CA6ED4588D4D140B3E44C032355DF1C531564D7F4835753344345A6781E11CD5E095B73DF5F82C8AE3AD00877936896671E947CC52E2B29DCD463D90A0C9929128DA222B5A211450BBC0E02448E2"),
             SHEX("2716A3A219BD690B9B0450CC0BF4F0713C46C4C02744F8F71DE261B94159278566583D4DD04CD573E79C576C1F25834C91D3AF69F3A301AEFA516082869F3CEB0F916A903D902175EF29A0B29AEE92CAD8555A26E7CD0395BB937750246A8D5A722DCADE4F44459A15ECC29B207FD8FFB8510722D925627372D86528B9B513967B5BCBBB233B459AFB5AEACE48E5355C3E7A387F9DCC9A55773276D14C85B8EEBE3D211FA08CBDD8AAC4656F028FC16A6D6083A969BD6EB5247695D8F6DE51C036A743D2DB7480DEA41C228FFDFAF129BCF0D0D5F7644405FF8AF778AD9A63EB54F62E1517EA4CFAA92F2F25E5F001A562B6B4D28BFB398C26DF8E88C288AED5B7AFE8686AA4AAC77C032D21203FB7BA79707A7E00371F56FDFEB0E607252448CDA319C39447ECB7C69F36424D74E33D723AE53996B35E5554182015493AE03DAEC5C31B00D481CD430F4723677C7B33F6E325A855300E66DB559DDBCCD550DE02D797D6B28A1EE01FA13A62F721BCC577A6C3363A03ED976391E969379BA3BB6BBF99271F3E5DFADDDBF0BB32DB9ECD658F0DA2863FE8F48CE2A13539D8E8276C083C4CAB1AA9D92914ACB238C3EF0A5952E5686E39D2804DED740E0457F69FEE9489B7E548514FEB18FCE21F265DCD56F0081528A24FCA7595E96D31CAEF0969F56B122749C74FA108C41D34902828B22F814BF352A1B5B52E9C65F75F232E"));
  test_xof (&nettle_shake128, /* 125 octets */
             SHEX("A64599B8A61B5CCEC9E67AED69447459C8DA3D1EC6C7C7C82A7428B9B584FA67E90F68E2C00FBBED4613666E5168DA4A16F395F7A3C3832B3B134BFC9CBAA95D2A0FE252F44AC6681EB6D40AB91C1D0282FED6701C57463D3C5F2BB8C6A7301FB4576AA3B5F15510DB8956FF77478C26A7C09BEA7B398CFC83503F538E"),
             SHEX("6995A90158EA5FD54F6D4DA0C129B92EA3E716E196DE4AC014CAF106EF4CABDFB9ED799545A0E53E32ABA2EA78A6C7CB936947F79817D200F32053E81D10131278CBCB903980E13125C99B1696F26D355882A107FA65FA43766C929485696D56D49D371643E9D360E25668B5B7BC7CDE13B678723FDC4598B60B6B19B47791BA99F4258ACD94604F26D5587EE8DFF613DD53E083CB2B0A202930D61435601A1D3CA8D97C198CC278967BA493B675D2F42CF4F41072B7A9403DF36E0E15B8811E37CE1119542C5168C95396653F72FC9E59727DE36B693CA58FE374AEE8DEFC657035FC874B7851EC0DEDE274BE064D258C0B39A238D2863BF8B7EBD0F7E2C3FB2A7EDA6800DAB2DEF61FA8F5E6891652796C2E9CF586BB0549F6F73B3374A4D7C6345FF73FBF99FB25CA47823A4B0470738AA67CD26D2EB954EF40A4121DFED58A1741EF335E8ADB32A588452774FF26B04EAE29EC1AC3BD83D2F3A27AF2A464D29589AA837890D7711CC0AF76ECD6142EEEF1C904C0CFBB287BABF951B8A10281A7CD1B49D766704350C7612A74606D2BB80DE11A37CA9AAA1CCB91F4AAC8D382A5A16E0CC7B426065F3AF566592C51B52BBBF007F4F4B7910DD0C7042406F61E5EFB6E04EAAC76E07A7E86C6E3A5706B833C52BDDF814E1261724F97DA9B7887399B7FDCFB14321D8A16683C1DC615EA8C738F0BF6076E05855F80D8DA43F0"));
  test_xof (&nettle_shake128, /* 126 octets */
             SHEX("0E3AB0E054739B00CDB6A87BD12CAE024B54CB5E550E6C425360C2E87E59401F5EC24EF0314855F0F56C47695D56A7FB1417693AF2A1ED5291F2FEE95F75EED54A1B1C2E81226FBFF6F63ADE584911C71967A8EB70933BC3F5D15BC91B5C2644D9516D3C3A8C154EE48E118BD1442C043C7A0DBA5AC5B1D5360AAE5B9065"),
             SHEX("791C8B8021BEE5393FFAF003B9C429A5ABD2E35C5C7F556E843D735265C5B7807343EE946BFF31F45B5626555CFD792F55465AD9FF95647A41D9759DE484A0564492236FD770ED3F78C92514B90636CEEC76D3FC44A6CCA482BCF45D3B5BBCDBB1EEF6D5D6EABFD0FD10B0B4AEA88C5FA68182A8A9805B5E90B1767C9E79500237D07BE59C0F09A9AF2236AA954F9DDAADE492D691E36F3F4043F28BF0BA6349591700C4211185D99CB25553E2F6F405F9D8E00FB658BDF4A091CBC7092934A6DC6D65E42CC9042F90D169662F80D66386489D2E7F750251130E1529CD8BA70929252551B0617C960203EDE536CCA5EB0C5217E9E4756D7B57CCBE5087A4FB9710EE89FF62355203DF415975AD106D84DD5526B3EF3432770B163856B4D46F7315F1ED70055CBCC4C78C5C0CD4B19FCED545EA00F0E2E9752A227445DEC6D50B3D73ABD2D81C9AAC6B53E108DA04C42D491254AF39584068E34CE2F4026FB63C29A52B913E5D2FA42FF7ADF413227B784711A3513BEA6AAA97DC019A25A0F09EE27A46FEAC4EC13B43E341C505C308ADA1DB5D72E0609A0A68698C809067F2EF17434F3205E03BE67433C1D67072E8CAD0E2276723AE7D9F552DF19C142A68A7B875CD7A73FB586E128856A9259F7373242C498E621E7F16E7AF57F36953001925115806972ED2795B9A2F398676BF37F2EF8FD4325863D599E10FD84D8D77D1"));
  test_xof (&nettle_shake128, /* 127 octets */
             SHEX("A62FC595B4096E6336E53FCDFC8D1CC175D71DAC9D750A6133D23199EAAC288207944CEA6B16D27631915B4619F743DA2E30A0C00BBDB1BBB35AB852EF3B9AEC6B0A8DCC6E9E1ABAA3AD62AC0A6C5DE765DE2C3711B769E3FDE44A74016FFF82AC46FA8F1797D3B2A726B696E3DEA5530439ACEE3A45C2A51BC32DD055650B"),
             SHEX("42AEF4301642ED54CB1FF9B446BFAB73CDFAC0320167CA3460A9AF1995B5DD50C1ADB402A55A3E5C4F7BE66BB2041F3E4C9EDDF7F08826F8B9C1163B0CE8B31769AA72AB4C598B3BB1CD5791A6BC30D11F0591D0EAAB34D80F134734D792BCC6EA61AA535F4F28331AA6E9A572D43F776341C971829C67C0E7D9321B09807965E1E9C70E611A4BBC3C42743A2D04F50804730C857F02E631C1B81DBD33D3EA2539F9AEECA9842B3C95FA1B8B1C330926FE5C6A396214537BD06B49B7951AF0F9AE04D158F364757A52AB1F21B4155895929480D87AB365A7950B321D5FD0C18FE3FF63B10D3CFD74AF45774500AD3FAD98DA5D240D613B386E677D57211B8781AEA7EF4052A25E122064BAE6DD59B00D533E6B041130B89778DB87F88DD37DEB78136B55A958744B9769612A0DC6D29B6C0EF863934302029C800B50AB385DE742D436809C7665B868C83DC450E438E94EA3124DC080355AE5931797E75DFACD09A7C97345608E021BD2034F7660ECD194F628A9868BC0DEF0231DA66AB89E2DF1EDA2C403203F971C1E61E4098FD306202B9F96C78CCC37EFAA7983C5D7712034328D1E65FBE96DB94B3BD88732E60ACE5EF946E896ABD4E1F6978700C12C4A8A372CA54B6F8EFB3402B898838A26D8900F69393E56D2ADDD2A0EA26E7F630BA6D19C46DE493C516277FC491F1FE273D013F4A32BDE43994B031EA51C2CC8F5"));
  test_xof (&nettle_shake128, /* 128 octets */
             SHEX("2B6DB7CED8665EBE9DEB080295218426BDAA7C6DA9ADD2088932CDFFBAA1C14129BCCDD70F369EFB149285858D2B1D155D14DE2FDB680A8B027284055182A0CAE275234CC9C92863C1B4AB66F304CF0621CD54565F5BFF461D3B461BD40DF28198E3732501B4860EADD503D26D6E69338F4E0456E9E9BAF3D827AE685FB1D817"),
             SHEX("5E0693E64DBE678055A31D80DA85A81A1BA5EB1CB6394B7B1AA2ADC2B8C1FEE4F41DFF3623502FCC349D4B2BEBBC1311EF291EF47F490C6FC1323075B519C87DBD77CB94C213EB548278A6C11B0C321C941FBEB9742B690A9777875D8A1FD0627B5CE3679A6DBCC8B53D775E73AC9F951749FC353EEE7220CAB40F4F7C2EEA311CD0B50974AF188EA9742509C4121FC07A70FC28F0633B56BABB451554BDF86396594B57B1F88DFA07DD4F8C47AB5FFBAD03DA1DA642EF4A6E75A952E8ACB3989136B8A792E6F33B24DA2CC4E422192EEBAF121561BD33A6D79DF6517B3F001E674F39BB98D9AA0E15226389B17655E7B2796D75EDE0430A0E7D07039B8E35642570F6457F8344A59DBDD5128F74F9AB83CD152DB17987019AA0B27B6E7951CEF67B574D05535CEB5EB023A39B5731B6AFE6CBB8BFEE0F71819D791C288BFE9EBE6706778FE62A80AEF22CB9C0920938502F30A20FB4A4822D0C0D42D50D67FA6A76EF919C985C6306FEA73A3BF41006F0FE60C0D92B80106D8C715B97E35E39FA8ECA59220B86C2C46B9C61EEBCE07DA4B64CAD247A67112F6548AA693BFC4BFC9AE78144571CBE7B81B83C8832938A71FD3AC6B8E4AECF17FD2848FD9558DC5DDB261CCAAF083AFFA341EA295C7264F35734F5179038AB70C90E03BF53E3FF438A9C0655EC38F5B9DAB7A4C3BA852A2A0593C80999818B460015C79093B1D6"));
  test_xof (&nettle_shake128, /* 129 octets */
             SHEX("10DB509B2CDCABA6C062AE33BE48116A29EB18E390E1BBADA5CA0A2718AFBCD23431440106594893043CC7F2625281BF7DE2655880966A23705F0C5155C2F5CCA9F2C2142E96D0A2E763B70686CD421B5DB812DACED0C6D65035FDE558E94F26B3E6DDE5BD13980CC80292B723013BD033284584BFF27657871B0CF07A849F4AE2"),
             SHEX("B620CEAA437E16A85BE761A8563AC80D9C49608DD0EBF9A4F049B9541623F4A3783C9E67B85820BA44A80F1AFE3FD835791D9F3C785BD8A77C9623EAF11F695675E3A2FB951D54050D0500688860BD270D3B723BE51579B81CF4ED94CEBC294A97BEEBB5A851160012514D3BAFEE6E275ADC7BCC64EB7879C32D8E0605241786DC3CC1BD44DB64BBB2E9C6E1FC581A5124A734625CD811B6BBE00B203319A736EEA07AF977190D777C12465E1614FCA56DB7154F9C456F59989062D60899D7CD7925DF1E1AC2EECD9B070F7E380B0775089F303B864D5CE049C55177E3421443B508EDEEDAC6307967A4321121EE2227C47E357FA92B741FAB390957CCD9337B42BC18E6981FD9EFC9241FDD24BAD0C6772216BCCCB92C2EACAB3A312CE020216DDC9175F3F86343E204E1A76573294DE6EC6BB75679379645FB973C37354323E15202415172D7F3D6A1060FCD35DC011B3B2FB4FAE21C69AD0119191BA4277ED6F71045FCD69F267E667DC4D219FA256BF35ABFA06A05738D212C42EAB91C2C993A5B6ADF9ADC0F276E79EC9EED6F115AF4BCAFBA9CEC43B0532038E863B2686996E997807ED67DE7F48C14B17D40CFCCEFD9A943CFD7C96FC665EABC70E606213DA37CD7EB6F942A6A5608305B8CD8B9BD4D625DE5D7DCCB554D3CE466D6EA8C1D9F111FA4E96B253EA11B4DEE3BE218C880844006D6401659C35C2A293DA2"));
  test_xof (&nettle_shake128, /* 130 octets */
             SHEX("9334DE60C997BDA6086101A6314F64E4458F5FF9450C509DF006E8C547983C651CA97879175AABA0C539E82D05C1E02C480975CBB30118121061B1EBAC4F8D9A3781E2DB6B18042E01ECF9017A64A0E57447EC7FCBE6A7F82585F7403EE2223D52D37B4BF426428613D6B4257980972A0ACAB508A7620C1CB28EB4E9D30FC41361EC"),
             SHEX("224D0DE7B21690B72A375907A10A4284E9E730A20CB3B4FE396CC6078D9DDA5D1A8E81934745361D8EED0B31B6276DB8F280104CC0E0FF4B36723CB302381A9BE703B0DE4C624A27996155599204DD2E219FF77008463BEB4461DE2DE3010A002FDB300C24B1A80C8DF311C80D174590043EC365FAC147A671DAAE41F38914E1E6614CB62C4DB154735363D7C7B7135096D7EDC6DBFF1DB407C0061C3135BA4577D5682E57007AD6E5B9ADC87DC27EAC3D03B3FECE4D9B4C2B2667E6D9311E342D037D58BF67CF38A4AB448AA4DB2C847E678F66AB45F0BD1B581AF6E881E63FC170F44FD0F8D87D6FB160A78097A4896BE055D0167007D9CBA610DA08C8C342E4BCDACEC6F0BBB30108F29A21947A9D842510E68FB175078B9064364046E544023CCAE9448C0727E8FD26330F26FA60EF891489B8CFAA623086372C09C0373C9FFD7E3648EA0F8DA7FA99D9F2D6617C0467F2F069F85ADEE5535100C519A528C690EE9192A888ACBC11BEFEA2B81E8FD380CB8679B6D80E37CBE3271AD1EF633204A8B97DA2E05565F8500DB9B1FD8032C69F57409251932CBDF046E0AB99896109F2EB1217351B242EAF3236139F20EEAC594750231186628453970F3C8452E454BE3D4D80E99503F350BE1EFF8778ED13F3D8828768E2F66E03C246B4FB59CC2CED5CDD266D320195A601C4E4928D84C62CDD8AE2EA357E4A8F34223FB18A"));
  test_xof (&nettle_shake128, /* 131 octets */
             SHEX("E88AB086891693AA535CEB20E64C7AB97C7DD3548F3786339897A5F0C39031549CA870166E477743CCFBE016B4428D89738E426F5FFE81626137F17AECFF61B72DBEE2DC20961880CFE281DFAB5EE38B1921881450E16032DE5E4D55AD8D4FCA609721B0692BAC79BE5A06E177FE8C80C0C83519FB3347DE9F43D5561CB8107B9B5EDC"),
             SHEX("78023C45DC98541F14C21DF4148A80ECD01F52EBEAECF7C28E9FE2B2C69B4B882805515064800089E6F18223A624155DE37EC8EC07E68E282204F3AC9BC33F8ED6DD98ED2FFD9EBBD0627B0EFC88F7BBCB7815795928766698DD530BD0AA461AD9499971FE935BFE38B6A84ED2D842572EEB97D7290ACE13C41899656BC30317A95EFAFE163494EB7E358FC31A39A20012A82F564813234D598EFDBA11EDD2B0F3A2554BF4FB216084772DF94E3D43E55F188BD8B8B5B297E406F2B8407FFDEDE7EDE9227ED6AD78E098147A658269C26CCC0F2988F06DE00651A7156407CAD75884EDDF56A755EDDD45030A4C43A82D55600D8FDAD155D13014CD27F827A30F6031E0173A2E8DA9A8FD9F7590ECA5F37747E5ABCC9C0A0EA589422BF9647F3FA9893BA3616C38C15ED6CF1F9EC2A64B616A3C5E6C5862FDEF36E2A44A911269F4ADE8F7B90272EAB51FDA36D9CD0B88DC8951088EB0D7D68BEBB332D22C4240F5DD354CC67650CFE4659D354742ADA6828A7CDFA82F77F1C846361087D14173775F44C8601A56774280D941F256A89EEB850056B38480972A6BC0D893A2105C584BCC3E963DF2DCA0B39B82823F5A4261FB9D4412ABABA7785FCA635A84BB757E1380531F5EF986F3A771E4948653178B240A32F8672B8B655F0BC832F853C206D2F6E73DBBDC2669DA67666947EE50CA82CCA023B5A3C485A9D21623D8A3D8"));
  test_xof (&nettle_shake128, /* 132 octets */
             SHEX("FD19E01A83EB6EC810B94582CB8FBFA2FCB992B53684FB748D2264F020D3B960CB1D6B8C348C2B54A9FCEA72330C2AAA9A24ECDB00C436ABC702361A82BB8828B85369B8C72ECE0082FE06557163899C2A0EFA466C33C04343A839417057399A63A3929BE1EE4805D6CE3E5D0D0967FE9004696A5663F4CAC9179006A2CEB75542D75D68"),
             SHEX("31EDB63FA182A6DAADAE56C08BC794E74677457C69CB23EC82D2B8F74B8649F79754A8D9BD42BA3BDABE0C4A161F7464D2CD80972824E6A04BD9CECCA798C2E26937AFA93194246468D66BC8769A26088513962435D3DAD9B1DB5544E6A2050238A242397FF6E7419569AB3CE2EC14C54481D634F56D1F32D453AAEB23B2078502190C8D1CF54E467CFD51DCBC6E82F95197AD1F6CD301849B76CCDE540CFF8403D19ECB88DB81A45FC13629800D000FC5CFDF508014D4324FF6D2414535E5EF3418555B81AA3CCC74FFC80C6F4F170A3DDD304D1D5EFACF554164E0EA798572DBB3393F63A7B9C066AD68CF25AF0D36400E427A5DB9B3CE639BE2DB8EDAC6AB1EB2E362682D6B0ED0D2B84CB28B56F1AF4F4A9DF0FAF3447DE9A2841B49CB2AFF2423A2CF58930D0BE6331349511C26D6F07A0ABD32A54F70BFF5F43C599AAD00931EA7F28A85010ABE8C93F941B4B4931813796419FB41F3BD0A6203B14BBB288116736B3A31CB53FF26EBA377624C1A0A280B768DADAAC7AC69835D8C324234EB907B64053678F629784C6CF34E0AB5522D6E9326998AAF24D1C1C62B93652896CC6A9BD7DE4F4CE7F6D309AC957C2BB446E6BEB8B026537F7FCB7E45BA1083F1D690BCDA15B1487DEFBF59DCC1A5A6EDB3DFC783B4224702A7F6E466B4E48DA6C1ABDD918512783B6A7FB6446D32FB580FD292F2EBF8E3D0302545B8C163"));
  test_xof (&nettle_shake128, /* 133 octets */
             SHEX("59AE20B6F7E0B3C7A989AFB28324A40FCA25D8651CF1F46AE383EF6D8441587AA1C04C3E3BF88E8131CE6145CFB8973D961E8432B202FA5AF3E09D625FAAD825BC19DA9B5C6C20D02ABDA2FCC58B5BD3FE507BF201263F30543819510C12BC23E2DDB4F711D087A86EDB1B355313363A2DE996B891025E147036087401CCF3CA7815BF3C49"),
             SHEX("2611992C1226B5B402F19662FDF3D9C44D118B949C8AA1073B8BF8A5DAF30D9802DF7B1ECFE6A9A1E9DB9AADE3B308867397BD48B89EEAF36C3EBDC97B35EECC2C241941CE702A0AE96F02E881F82AE2842E1A7BA19D3390B2D4B5E2BF3883ECADC81AFEB6543DCE3857FAB5F926B1621D32AAA0B1BE10452A4E4D2229B0E50B478E28595E564122318ECA4DE680E354A745501A8A11AD263E22B5A69E2F70A3F53594540B2B9160C3057A32E633C616F6DBDF8D6EE19936CE8EBE8BB7BBC118D720EC88F3536FAD10FB623801C720D07365148672384BDC9039F0118C356CE20EEB8CE68E7C85A0E4BA3AC5BC7B21E6B567F13FA1D5A4DB953589913CF2424B7B3DEF3B234388B7327A4E737B590B9AB1EE126D949C630420A04C32EDCE22DA326236980734249CFAE18AC0FC2A253CE77D2813694FFC66B0E60D3AD245AB4ADD0BDB1CCA7E4228B73C18335558959A06BBDF24DA8095184B31CDE2F4DEAD983C9E29DB2D4A3C5C40C7775EA30A80CDCA44B4DB2783379074017FFB6F27E45B86F26CA42742DB1B11E11BC5B99D5642E482409AAFAB2BC833B31774A8AC578BFF32131D714B2F8DFE91F4DB4448C34E341D26FB64AE00E5E985F16A35857E44B9A846402858A192EDC0CEB7A6FBFA90F0EB6D743BDA4BB9C9E621D7433AE693CFCC991B71EEC3B03DF1FB03BDBA05A63B9BF17C3CAABD40808CBEA2DF0697C8"));
  test_xof (&nettle_shake128, /* 134 octets */
             SHEX("77EE804B9F3295AB2362798B72B0A1B2D3291DCEB8139896355830F34B3B328561531F8079B79A6E9980705150866402FDC176C05897E359A6CB1A7AB067383EB497182A7E5AEF7038E4C96D133B2782917417E391535B5E1B51F47D8ED7E4D4025FE98DC87B9C1622614BFF3D1029E68E372DE719803857CA52067CDDAAD958951CB2068CC6"),
             SHEX("6E722A099B2B47E38C69A0B967420FC5C1CA2926568BCE435E5F97D945B261C61483B45F517827DCA03FF3AE670883077D5625112ACA377E731364932767E5B3AE406ADC06283549D64FBF3EEAFBD79064420EDD544914FA479355F6BC6CD0A88269EB1B4C5E6E2F43F380BA5C1D343338FE73DA047E9AA9A06C4A14AA7096B5AE025759244A99A8FBE61B7E326841EDD3779A84591E9E44E810CC24870B42BCE705F06039FE93627B51DBB8163CF8D4AC0AD3983ECCD3A90A85C58AB222463323D5F786292DFFE07E607C100D2C830436AA1E4741ADC7A7439053045EE3538822CADFE37413EF7948B67F858951DF901A7E3D39EBAC040EAAF4C6277656EFF711034EE862DCC247672DD0BFF9B577286750083FEF332F630348119461CADC2089C332F9A0078D074CEBEE2E8B5B8A1A5B4818A53E9C99C330DA1682073A7631ACF1BF621C59AA45935FB82E634D2549FF405D36B23327C66F9B4C7E092B409228F8B0C95539C3A9D89FD7FF92D2F811055F6ED23CFD984445F7BE0B906F6A920EE0AF326F4F60D8B5987C736710796B33E58D2661AF11F8DDDDB12A12B196308F6493567904D49124DA69D291D73B104B5C99C002A6606718EAC2CA92F8A6BE6328A5DF6968DB53CA0DF2990B68F0A15A7F8B006BAFF2483A0F6D93281C62FE689F5856909CF3C56C5B16169482CD779A3137B6C702C7641B2EF2E9987F5497"));
  test_xof (&nettle_shake128, /* 135 octets */
             SHEX("B771D5CEF5D1A41A93D15643D7181D2A2EF0A8E84D91812F20ED21F147BEF732BF3A60EF4067C3734B85BC8CD471780F10DC9E8291B58339A677B960218F71E793F2797AEA349406512829065D37BB55EA796FA4F56FD8896B49B2CD19B43215AD967C712B24E5032D065232E02C127409D2ED4146B9D75D763D52DB98D949D3B0FED6A8052FBB"),
             SHEX("7B35855B74D677AB6DBDBBBEED8F8BED3B08DF15055606A7E7FE9F5E319BC70BD9DF00A8571ABB6E6C202461EA00A2D701DA5C3942975C48D78CE565689E36379EA603FB1C6C53B508E0952B892A5ABA6660980A38FBF7C91D43BDAD1F3EB20910BEB91502404B21D3C5283A0A88403C6AA47D9DD0A04A23DCEFCD3527E4E961782A0E585AAE6BC9B7E6ACDBBBFF9CC76F757887174CEE9CFA531C380467388FD2153BA8ACC2AF706F35667253216D3A1E623A747748FAD46AD65265E4530AB84D39D9988EB876853E976156900D9DE464CC2C57385F55242608C9E9C26B67683293B1F83181C60BD77549C576F2BEB8E54B3EC5BE2FEAC8CD340CF104B0A49D9FB484B538296B31E30E55F7A8421E5E16B91869214D2F7F4DC68736E09FF2981518FF83F24D51E5F9D9300938600EA6C007C9C98A392B2E63216DC1F07EF4D602F47C1F234B857D8650D23EDD344152EBB5DAA47BFF8E394931397AB2231DF1E22C566384112E8560E69EE422758E8A76E03D16770E5D417B6FB5C68D9CC93E16759987FE02BB8C99B8D368248A6893ADBEA888315D4AD251FC57FB1EABB2AB314B24ECD374F4566989038134788F344453714F05A81CCAEBE92C75200FD8A019E33A362722F222708C4F3894307D09A4B4D3666002A8EAF2D6CD19DB119A957D2B66B0C788241CB7702DDECB63142AA7CA5B028327CB51D72904F2D819B707"));
  test_xof (&nettle_shake128, /* 136 octets */
             SHEX("B32D95B0B9AAD2A8816DE6D06D1F86008505BD8C14124F6E9A163B5A2ADE55F835D0EC3880EF50700D3B25E42CC0AF050CCD1BE5E555B23087E04D7BF9813622780C7313A1954F8740B6EE2D3F71F768DD417F520482BD3A08D4F222B4EE9DBD015447B33507DD50F3AB4247C5DE9A8ABD62A8DECEA01E3B87C8B927F5B08BEB37674C6F8E380C04"),
             SHEX("5B31C72A95D9EA4AB6EBFB2B2501F14EC41223A010E123B68865DFB7061F8CC00F69BE86E56A376A377E9544860F44B543B48C8A56C6A25546AA7573C29DABA9B746A94EE82A33825047C019EEAD9E7A79D50362F649ACD1EF27DF8BDEAB02965AA0C0CE67312433E0511B19A5D213EF9720E688A80CBDC7BA1C50032701676BD4AAB79675F85F76B7458F3E6D723C62802B1892F0C2B45138142AD9EF329D771E6873BE098CB8C501C3732836EED9165B69AD40AE5A642D980C7BC843FA76B605FB68872DE523C591CE09F21976AB47FF64F773B5D7EB54366BC003FD6F75745C4B14EFBD1ACDFEC8439550681208543E469274A1FDDB16D4125A01E9107855149AEDDB7E21298DD8C309AB1C1BD506F2D377042F27FC275017B642456C93FC882E3CF44C1978307E7B9BBA16F612E28E99D2E18D2D85DDAA235FE65EE40D905A845768372DD621E6394AFA4C59378DB5FEEADEE6E342D004598779F08EB9A231BA0CCDA0A1C7D19AC741505F0D931C73F241D1999EEDE40AA53A8A47F629937D29D1A6D2AEF1807BE7C333DAA332B4C5551A7F37177F484740BE772EB03D04E82B2487AA65A2E4AD00F553060C6A9577D21A6C3B731D9D1659E54BF14CE4E4B7932250A8F3711FEB7D9997A5C554DBA364DD1F16AB4F34B37FCF5AFBA5B06BED58E6ACD232D69963AAED993F51A1C4A2347D8119D381686AB0942190680CCA"));
  test_xof (&nettle_shake128, /* 137 octets */
             SHEX("04410E31082A47584B406F051398A6ABE74E4DA59BB6F85E6B49E8A1F7F2CA00DFBA5462C2CD2BFDE8B64FB21D70C083F11318B56A52D03B81CAC5EEC29EB31BD0078B6156786DA3D6D8C33098C5C47BB67AC64DB14165AF65B44544D806DDE5F487D5373C7F9792C299E9686B7E5821E7C8E2458315B996B5677D926DAC57B3F22DA873C601016A0D"),
             SHEX("D4D8F990ACBC762BF3BF4382EE78466D36AEE8E7D3597C8ACDAE9A6245020BA5D936E82C2C8D7F861186D428F3ABAC5E1F5F4FA25EA3FEB4FCB850ABB7F654167118574A730CEB1EA5D4A3EF579AC58169CD02DEFB896FB0E8C0797D9A326C1AA696EB1AE150EB2ECE9B49725FC6C285AF8DA2908EED93B6B01512B80E03BDC74E9A528843ACD2B9FDEB352230BC1BD9F0F94EB5C9356259530E8E0060E0EB0751EE2F5CFCC2890E0FB8E5512530EA0D6A7475AFD535C34BFD14EBC48A0268F3FA671E87EFFE404D63A03EE92759C6490ADDA245BC924D582209E7406905678CDDE5529F68DB56602D546D2E36F5417DB7D3ACBFDAAC622A65560611FF98A262558B304728CBD7BA748174C0EB2FAA8907E1056B4AF0F820B849BD9411129F4FCF0CCBAA3400C2B7B16A2126C53EF6CA35122F8A8B92549E6AB330C4436268080898AC9C03C27CEE9164EBB9C4B1E54D211B65284047F2C535D3F871FF50B81C437DAEBC09507A318C296F1FF48572E4BABEAF96082B177B4B80BDFC2F99FC5B5BD6983E0DEFB28C98F5EB2ED1D4B36641D3DDDED2D8268C10900284B8146359694F4D09F176EEB550B7629E6A936E0BCE2877C8525CF4983327227E13B119791D465F9FEBAAD25943ADE5631F1DAB35AEEB7312EA7B908D19F257015C1CF6BFAA29728437F7620CC9C0B00E79F684E4F4313375845B9B524B3842FA0DD1560B"));
  test_xof (&nettle_shake128, /* 138 octets */
             SHEX("8B81E9BADDE026F14D95C019977024C9E13DB7A5CD21F9E9FC491D716164BBACDC7060D882615D411438AEA056C340CDF977788F6E17D118DE55026855F93270472D1FD18B9E7E812BAE107E0DFDE7063301B71F6CFE4E225CAB3B232905A56E994F08EE2891BA922D49C3DAFEB75F7C69750CB67D822C96176C46BD8A29F1701373FB09A1A6E3C7158F"),
             SHEX("77A1157DFD30A79C34D64CC29B9712C9E1CA76CA5565D0861D337627C20A25D85AA061B876A850B43C5EA74946D03E35B6669336AA47C5859EBFF82333C03741BDB0930375C34275F2B2CAEB83CFD28965DA12F60A646EE52372773DE04613A3DDEE266E20E0AC047B09260E0B69F4F134B5AB8EEEE481F887087C0F8639CD32C19FA514A3114E938EC70E8E7E6CF40D11342775C0AEAAFD8F8B2DCC884CAA603BC8FFB7A282FA6291BB3AA37F37CF9707632879532CC93C6F5C46FA6D6235D7FAC64DD9104FAF484275CC635C3F732C30B155E22552D2BEC2967AF002683DF1286F98F25D916EA9972DC6D2C5AF1A1572BC5363023621A33C0FE41285148CA9BFEC731ACF75743C9C64B48BF98FA32B6D92BEFC777F6B29996319BDD632D56F1BCEA5C52DB9E72BC71D6CC4D1CE7876B02FD48E4EB1110D727AFBA35AFBF8C43E3445B650F06C91B90B12C3036EBBDAEB3D2193129B22E188469A3C2D469F24B530200A08E931B2E95891D966C7A50C2365A73D169D010FDCCB7F3A739D4A589A84780ECDD84BB34D54EE5CED2C729A04B27F9C82DEB329E491C7B462308C397BEEC649E4D76239A75BC8BF28B1A8CBB81F25F73E53E19B882F0855AB1E93D45F9293508C66C2211E978267686F5F54AE987D59D73DB8144834FDB2DF2D52631C7671ACA994A99B8009C98A4F847B71A29B3A0636670721AA58BAA5A11B6FE8"));
  test_xof (&nettle_shake128, /* 139 octets */
             SHEX("FA6EED24DA6666A22208146B19A532C2EC9BA94F09F1DEF1E7FC13C399A48E41ACC2A589D099276296348F396253B57CB0E40291BD282773656B6E0D8BEA1CDA084A3738816A840485FCF3FB307F777FA5FEAC48695C2AF4769720258C77943FB4556C362D9CBA8BF103AEB9034BAA8EA8BFB9C4F8E6742CE0D52C49EA8E974F339612E830E9E7A9C29065"),
             SHEX("4031B77C2A88AB9A31A357652B5EC18239EA42FEF5B390156D7A50A1C1D826A52297B169BC808725B00A671F83F856FC37AB8C48E5D12E277558896F19FDE5038C03A95A3762FF48DFAE4D644B904845DCBA1145084E6B7316FCB5742F31325308942FC31D7F465F6C4D686BE9788DE37FB57A6CBC67CCC39F09831ACB1B1D8010254E131FF7A05031F031E6375CFBED440BF0D66F7021A927AD249BD3C78E2596DCC4E4FFC58D0F7331BED5BCF249C8092832C17C701C65BDA1DC9C50894726B04661F5FB4FEB570D5D3773E33D2DCD957C171ED631E4AC5A8F42B3828F78BA032630596DD8B8462A2F25E502B25DA1747D80F96DAFFF366E787A340A0AF43AEB8D95D565B1BD8D7F8DB4FA4C76F5F8422E6C240339C3DFE8D4D13E1DB745D8212BCFE232AA5C519BECCF64B38085109753E2FE5014F9EAE6761D6D600EA05CF0AE6D4300C7670FA1849333BB739B43A7CE58998E4F834309E9505856ED020EB8205571D3FCBF3A9D316A4A8B6E3A397C0200977CA95E854AAE35A5581C88448966112FFC0128F6A7B5DE197019D830C619D70803B72D318D78FA067DBCAD646857FD8CA5B6FFBCD7E0E3D02B6D13EA02B9C346E0000C2488FCA6E8E868FDEE7891FA12B095860B11B7C5459465D09B6A7B388D9BF6EDB0B99486A8C949DCB2431CC8EF13870BB15956B17405DD1047B6D54CC74D0D4EDC3DF689E09D7381C2"));
  test_xof (&nettle_shake128, /* 140 octets */
             SHEX("9BB4AF1B4F09C071CE3CAFA92E4EB73CE8A6F5D82A85733440368DEE4EB1CBC7B55AC150773B6FE47DBE036C45582ED67E23F4C74585DAB509DF1B83610564545642B2B1EC463E18048FC23477C6B2AA035594ECD33791AF6AF4CBC2A1166ABA8D628C57E707F0B0E8707CAF91CD44BDB915E0296E0190D56D33D8DDE10B5B60377838973C1D943C22ED335E"),
             SHEX("31B51A74B7392B14E2FD94E248E7FB7740536FB6E88CA508D019066C6CDF83BAF2162F4DD175BD7E78326F1A5D6BD3A9D9B5874C474C08612EDC3935EE24BD643A0A3ABEE5626C43E99BFC73EA434872AA22BEA87311AB71880096CE23085DCD489FBB2CEFA4CFBB06D12CEABD4B1D28611606CAE04EF71E7B37E881C2C1089E08268CE45E0D25AAF6843B45AA50F2BFCBF8FE291FB2745B770CBD758CB0A86C8FA439893D93B5667011A8EBF345FA5059F062DA72D0E5C3417BBD3C6ABFF924EB6BC74F6DD0B30BF48848117E7820AC9892EC18DFB1B567612627F9C2585342D165418D211CA7FA4A1AD6511D3B68539CF2DB2F6C3A4C1055785517E0528A6CFB006B54808D7F897B0386F7514897874C5A4A3A30D8D8A277A6A59710D0BA870DF52F2A0916AEDB0870963C205BF49C6A2251EA92E2F9B4A9B9DF98BBE752A5DC258E55A8B3854BF19873F1908E95B537DE470319CE3648BBA5A038D11C87B4FB9DFC9ECCA1BED391C0D48825B022132DB26FA3D17D60985D5D0611353F96B03C4B5831FB9BC0C411D1909393B54734B59B15EE26A1952438C886D61C5FAAD119EBE8576AA904F18FBE19C549ABD70F02B1BE98D343CE0082CB6005DB51A6A10AB5E50A89EE44F3502AB19D62C04637ACD5C49DE78F5A509674AE8EAE906693FD8110E0D750F61562D088E0B87DDAF96FC775CEB0F13800669BF2A00424E6ED"));
  test_xof (&nettle_shake128, /* 141 octets */
             SHEX("2167F02118CC62043E9091A647CADBED95611A521FE0D64E8518F16C808AB297725598AE296880A773607A798F7C3CFCE80D251EBEC6885015F9ABF7EAABAE46798F82CB5926DE5C23F44A3F9F9534B3C6F405B5364C2F8A8BDC5CA49C749BED8CE4BA48897062AE8424CA6DDE5F55C0E42A95D1E292CA54FB46A84FBC9CD87F2D0C9E7448DE3043AE22FDD229"),
             SHEX("3CC7FD2AA14A9DBA70ADA029992842231C54ECD7D14FA4687424FE1B54A58C7DF8A1E9C87569B695CDA9B2A27548398440C77A5A66BA94217DC24854462FAAC1CAA8D171E610B6D280668BFBC87C8EE2160E148A6EEA29DC04A57FFC139089AD5EA85F89EED89212EC7439530767563137A4C028A5FB28C1AC53B3696282560E15C1585E51B799273709CC5A1C1A93C774BDB19C4E336CDFDB696A39EF6A8ED8EEAE92C5660718A7E90C691C84A56BFFF85DA418C9DF7A913352FA53D8635F9FF4644B8EF03892FFFFC937BCEE4B9D66154964623DB90BAB15EB783EB14888411CFE2EA417A373BEB4BE6CA5771F6FC0EA8266D56275F139EB3A90C3915AD6355287E06FFCE64D84FD8B0E06BAD9DFAF7F7EBC17CD7ED316977AB481AC47253E2508D349993FED324FBF97067AD19C255275D7EE2E119FC0F4490C62EDDA700C27B5088D57A7223729A64AC1A418071F4365C61F898A350CB9B82F59562847016A40E72F862E10398FDCCA0B86FA1F92D17F35A07C46B3ED570738D231DF3B80B0C2261ED932822D811970509D7C664D20BFD597543B3FF61D6CE3B6CCF924ECAB5FDD9D0B0DEB0444E62D950CB6A688BF470C53930B44FF4FC9B4DC7903E97BF29C5373582DDD79C115F66D31F839D7BBD34808539025AE3DA3FFD56778160E18F8FCDBF6744431D42CAF155028E96CE1096F856E805D7435AB1DA522708411"));
  test_xof (&nettle_shake128, /* 142 octets */
             SHEX("94B7FA0BC1C44E949B1D7617D31B4720CBE7CA57C6FA4F4094D4761567E389ECC64F6968E4064DF70DF836A47D0C713336B5028B35930D29EB7A7F9A5AF9AD5CF441745BAEC9BB014CEEFF5A41BA5C1CE085FEB980BAB9CF79F2158E03EF7E63E29C38D7816A84D4F71E0F548B7FC316085AE38A060FF9B8DEC36F91AD9EBC0A5B6C338CBB8F6659D342A24368CF"),
             SHEX("4144315DAE1FA8FA09E41ED69B72FB8FC64A62F8133EB9C4BD53D6CA7C3475048AFE1D0BC778672D715ED01C5532B393888BDDBDC4D43DFD82FBDE560A80329D03AB5F4D62358E65D36EAE4094880E810D022529A8945AC9BE8255CAC708FC435310A7741825257BABD70E84BE8A0099CA66754C135476340F1438237D7C227C31353D3878E67A1856868C665CF049E16FAB2FA6A2316053712C3EE808762920FC814B3F336FF8C9846CB43422458F743ACC398F922EEA476823CCE9AABA48D159C55BDCE636D231F18F49C9B70794FBF1DBC4F71D87B917EC9D1AD5545A4471592883D63F91CFAF7E64805F82411E715BB61CD96EAF2AAF6A22C14790FC76310FACF3C2FB78B34D3CFF50BB3F3A19227A18A91E7A6D3CD38D5281CDA2CEBE333C7EFF4D357C55889109D5E0B762471C565E1AAD50839D0298EAB2723EB4E9D1BE73FBFE507189E5F20C54F256084E5EBC04CC622B080EF52C38329F301F8F97FFF2A7D8249E2EFCA30BF77A8A3B9630CB5AF5E9A34318B94E17BD8248EDABDEBFDC37E3476155434D32B7A7905786487BA232786C290BDD790AC255BD100EFAE4BE59FA6B1674C651E8EB9FD2ADC4BF5BE6A763408DC558825B3A22BF141FA45E6DF862959F9F47A91BDE715AFA65099573F1062333FFED43870CECFAA423FD2211F82389BD80F0F033B4FA12964448846E55AE2BA46AC9E7422C228D75A1BD"));
  test_xof (&nettle_shake128, /* 143 octets */
             SHEX("EA40E83CB18B3A242C1ECC6CCD0B7853A439DAB2C569CFC6DC38A19F5C90ACBF76AEF9EA3742FF3B54EF7D36EB7CE4FF1C9AB3BC119CFF6BE93C03E208783335C0AB8137BE5B10CDC66FF3F89A1BDDC6A1EED74F504CBE7290690BB295A872B9E3FE2CEE9E6C67C41DB8EFD7D863CF10F840FE618E7936DA3DCA5CA6DF933F24F6954BA0801A1294CD8D7E66DFAFEC"),
             SHEX("95597492CD331A89C013D6179031DDAA08A90FA825C5A77BB5E27E32EE7EB1F02EA0D348B5372EEF60A45A84B84D45C18695270FCFC07F3EC95E00A2FFE901E5C9EE3DAD600273AF0D73FFECFAF141C3B63F3FAEA90633917E260857CD1C4ED198FBD4753E82A42128F2525764B5A40A66E1623B81804BD3F17622598BF39FCA57D8C8D0E2E2EC45A4EF13A2B6B0ACD6E6878BA5BAEED796E3B3498385AD7E88576EA08E148698BE845B3D7483371B43FA9B35D785A0028BA84B0C8FD81122935DC09869E065D47D95287D3CB4346D8BCA8BF5D4DCA3BEDB1AAD70EA3B5ACD48C5C02FED30B4706C104DC159EA96971E8F44248501BB782DDC201569622C1F98E893F820D149164F9C465C41A998FFB561CFCD7B3E327F78961636192D6158DA80A7987533C01C2DD380912C1ACF14330B85E93CE05F2238AD0C1A05995B631ABC9CFBE70ACA565E9A1CDD424D3F8DDFEDDC0D3D3360CA917B30D7216A85A14BB13E8DA29228E57AECF1906B084C72236B91D669A312F01A4B7787C4DECEE4F63CA276F63766EDD2E46555C328D538581E86F2E3D74F5592C1016396082ED06ADF8DF9EA53B5E0C020AC75B636A828CCAFFC064657A657F24E62107D772AFB24A5EDDFDAA4EAF2B99341AFFAA1651010E15F6F4E41A67C1EA2101BF5A61C491D5A88577C1FA3BE0A0E0781C53BA9F9BB85DA59C5D0F3B2AAC31F45C9CA283928"));
  test_xof (&nettle_shake128, /* 144 octets */
             SHEX("157D5B7E4507F66D9A267476D33831E7BB768D4D04CC3438DA12F9010263EA5FCAFBDE2579DB2F6B58F911D593D5F79FB05FE3596E3FA80FF2F761D1B0E57080055C118C53E53CDB63055261D7C9B2B39BD90ACC32520CBBDBDA2C4FD8856DBCEE173132A2679198DAF83007A9B5C51511AE49766C792A29520388444EBEFE28256FB33D4260439CBA73A9479EE00C63"),
             SHEX("E7814E78CF280CD8D0B27B311476F21D6185ABF37FDFE2851143EC3425F12DCCCC0FAA95B2E6F8ECE8C4156124C9DFBFFA77624E5A4A9553F11675844C3AF0E816CEFCC3EEA430D7C730CF135FF6A34E542E50F25FCBDBF3A051F39CB028F20B8D6F85221B7DCDE00D7E45B78529638508AB2E6BB8D885752CA6C97455E61564C2FFAC68E5892BDEDFBADFCC88018A92B599807583E9B8BAD62306948FE52B397CF88D20D440898075E3674FD09469D1FDBBB2350751F21E48BE74F2B04E7A148286FD25A90995B5A95716D67AB566B78394746DA84326B83CFF482F82DF9B8CBDF3D92992D0F05275D37AF3FCD4CE211A9480DAD745B6CBC4789C9015AE23D1C3A25D067C9A9F7E611AD05B8D44940AD622C48730282789A839AF93D0735725CBEE9DC4231E8773B6DAD31178053014F11421AD0ABCBF32B1E6386691759D58EF7E2A6E2BBCC11B5E62CED89C4BE975FDA74ABE9395DFFD9E8C102A5142933252D94B0266FF776E9224985F3F2C96F7BEED03BA881B99479862F8B7DFA96AE0B2D2A97F773164E1B473A0C671ECCCEA64DC67F3A53A21E51B794AFC4531AE4F41898E250C80622E123B27FA28EB7F07D4B4FE63D75866FBDD730B54A9297697F541FB429084AD43DC69CCE73C6C17253C481A002AD7EE11FE077554A4BB45ACA4E9D8B8C8E8407866C011B749C3DBA372C2966AA0C20F43C86C01EA74BE5BFF"));
  test_xof (&nettle_shake128, /* 145 octets */
             SHEX("836B34B515476F613FE447A4E0C3F3B8F20910AC89A3977055C960D2D5D2B72BD8ACC715A9035321B86703A411DDE0466D58A59769672AA60AD587B8481DE4BBA552A1645779789501EC53D540B904821F32B0BD1855B04E4848F9F8CFE9EBD8911BE95781A759D7AD9724A7102DBE576776B7C632BC39B9B5E19057E226552A5994C1DBB3B5C7871A11F5537011044C53"),
             SHEX("50F678CE2970C84EA11DBDAF399535A692DF95267340A394616760BA82BAD865F920932DCE65D0F10B5AF1B378F76BBE05CD8C8E16E5720E527326FBF375A5F08AFF54C4554282058636C69B6010330C1DEFBA22F175BBFE434B6430C24C06F034F6BAC59D1063A4DB1FB4243162F368385C5083A93A7E693009BAB0F3D8F24AB45F3D20E9582D1E9D0AF578BFC8C9A0F09A92E4D76643EBA46E71AB177307EDBB2DC46139DED52FAD1459A41114EE88F439DDD064600402E237835797B4BE67EFCECFAC28D9F3F8B8CC773147400526202984C0CC2CC6616AD2EF5E0FC16509372F9EFE63D66CD41C61AE88928411054953BD95813E520E02ABBB52BC49CEF99CF09BC3C83DCFECF2F6F4768ECCCC8639752A28531C0A4038284C174B34978850B02E45D7810C4FEB224E3DE0C89CE69B06DFEE9CE2ABE8AFB209E29194ACB6C526EAC490D5A9518D5A7E1C1A7CF734C92579D162F2862EBE0722E2DCE53DFFE84A52A459B9C1D0A677A41B0F1A945AB597EBD8544AFDB1AF97CC1E3027EC287DF62EA1E81E91CF4325D202DDAB62E09E6094FAB327CDCFE2BA6F760C2FF7D7992A3FB675C07F77DB5BAD69056BBD3A50A0A24DF64A73ABC3CE3C1F131E7E6C76F0FB207F6D6DB54C8D128F2C8B670350602D7B0A52153452CC92EE4F185F4570405E54CB95647758A31B8434EBCEFDCBF00A43857A58F6506CF8B7AAFF3B4B"));
  test_xof (&nettle_shake128, /* 146 octets */
             SHEX("CC7784A4912A7AB5AD3620AAB29BA87077CD3CB83636ADC9F3DC94F51EDF521B2161EF108F21A0A298557981C0E53CE6CED45BDF782C1EF200D29BAB81DD6460586964EDAB7CEBDBBEC75FD7925060F7DA2B853B2B089588FA0F8C16EC6498B14C55DCEE335CB3A91D698E4D393AB8E8EAC0825F8ADEBEEE196DF41205C011674E53426CAA453F8DE1CBB57932B0B741D4C6"),
             SHEX("C860CD3E9223E1CB941E130F9F71C376EE3A9B89DA339C61D90FF2B48D1F285FC48A4B6D87C199989D3CFFCE4FD84699FE385C7CA018D4F31F00E41216412932A833794AF207132EA430B07201EED784C2FBAF9506957FB9F7E18EB9FDB06F051419BCC7D67ACDB93A9F78529D9B5344AE3C4FCCFFC406350847E228BE20C93B22760BA43977451B3740BE9E3B283C86959CF062B69D5BCCA35BA5362EF0B2ECFDAE225CFEEAC473E420EA1AB7CA03D4F4A82F6C32FFF843712B78476ACC46E4B88F280E3612D5AE2FDB94FC26D5C40812B0E9577DD995FD4F5282E180C37A02D6140FC88EBF5972739D591F13AEAB894F240ED4BE309DAA4F3B0E2D536F3DEF87BA53A881794B80EFEEF855BD35026EE9D8492C90208C313288CDE719A111BBBDC3BB56AA218ED1B466E6E01728DA23950F821016F313A69665D43BF2C619884D20E83D6202ACBF85BF86D271A1633F32372B067BE55570CCC2C7600B0CEA6133FFC787B743F3D2346564A2D598049D92F9C4160873AC7E03143BC3AC5366EE6C4621C25AAC1E0AE4EFF6A35ADD223A61DD044593629C082603847BF2B7F07A4115DC1C87AF0F7854563EB18FFA8E24EE09E299738FFD4F1E79019FCE4E4582B063EC1F36442B085F82FFB99D409B4F692A8EBE9AACF2589306204FE0F168373476393783874525FB808C2B90F1FC5D7A368C90AAB7AE46E21FEAB48D667A5A"));
  test_xof (&nettle_shake128, /* 147 octets */
             SHEX("7639B461FFF270B2455AC1D1AFCE782944AEA5E9087EB4A39EB96BB5C3BAAF0E868C8526D3404F9405E79E77BFAC5FFB89BF1957B523E17D341D7323C302EA7083872DD5E8705694ACDDA36D5A1B895AAA16ECA6104C82688532C8BFE1790B5DC9F4EC5FE95BAED37E1D287BE710431F1E5E8EE105BC42ED37D74B1E55984BF1C09FE6A1FA13EF3B96FAEAED6A2A1950A12153"),
             SHEX("97AFED51EA7CE6D0082A0EF242C37FD626DB617DAEC0C967076D8E63E4861DB3AE1308F19EF4B9F0B7246AA0E255E7535B5C70B341BE15EA4537CD1F0235C0F9E60C2168C2B04FDB3DD871460E0BBF1E3193D04E171E0A24BECE6517CB04A0B020AD042F28680ED42EE01D4A05C3892B0654C961743591722054CCB0756341B54643817802B3E05D70D71E339E487206EC34765F5C0D2346135070F50D28B8AFD91EA2E88FB9DBC1A932FEEB781F13FE5E047F7F43F51464B9D28CEB8B19BF34F03C1C9D1DC77A8FE475B349251E7AE0AD171C1FC81B862F7F73302F7B48337EDC72D39C59785C5572929A6E4F947632B41ECB4EE9BC5E69CBB746E2512986D3A23C7F455FBE5F1891C7101BFD3AA738C5C84BC194753D9671B8F3EEE5A35399615537121713DC85D23E0A7B47B55DE82C8109DB1E0CA96C471F366318E5FD9671D6468A36BE694635F65249D65B9AB9A6E6A5361E35350CD258B8DD145A3B68DCC900673BD1A13369163E5B5D3BC524849F73C68F01C7C8B88C9643341D225E13DACA45C62D9B5693E2E04163F96454A52B4BAF64DFE2293E438E511854A99682059BAD034712FA3E99384244F83AEF74B96848D338CCF439BB073552137E903767117959A2D841A157F468E597522515BC43B121B257B698234D19280C4F41EB865882E4ABFB4571DFCB2DEC5A30CF7D6E867F8B5E0880DADFF8DA337230EB"));
  test_xof (&nettle_shake128, /* 148 octets */
             SHEX("EB6513FC61B30CFBA58D4D7E80F94D14589090CF1D80B1DF2E68088DC6104959BA0D583D585E9578AB0AEC0CF36C48435EB52ED9AB4BBCE7A5ABE679C97AE2DBE35E8CC1D45B06DDA3CF418665C57CBEE4BBB47FA4CAF78F4EE656FEC237FE4EEBBAFA206E1EF2BD0EE4AE71BD0E9B2F54F91DAADF1FEBFD7032381D636B733DCB3BF76FB14E23AFF1F68ED3DBCF75C9B99C6F26"),
             SHEX("1A895365874A70F2304878B1334BAA09EF898A0607F3243410EF04B14A8490C193A18FC2FC0AF0655E219956A8DD3C7055DA00345CB411AF9167CB9BDAB8939D1565647EB251B7459045184BB050F3700CEB6285114509C8167DFC122B33CD3DF761F37A3573A75F5C16AA2376B3270A26522869157A3BF26C4D30A9C1D7B4E7490FC9FC5A4EEAC331996B16574A27C32659D0AF52E0E50F0C221342BBFE2AC04455B659B814CB888DBBC90DC3A899FDC538AE8E8FEE2FB085FBDB79830392084FA1761A49619018F5EBFEC31BE185F0945F4CF9F5A21AAC1DD3F1F3C7B9FA05069264F6368EE4D3BE24ADD6CE5B70E328DFFE08043587C39213DD66A6890FFAD586F9B13CF08DE44834278E04DA2F91DE05257E0AFC44B5BA9BFF15CC6F7EE306482704DE0FB80B93ADE6537358B252D050BEF48B9A246CA75D62CEE920272B9C4B06FD39FDA8DAD4D1C3B01ACC5D507299D44DDD034D21802E1FAC9E02A7C7CDB00F64FC7794A253B1BDC042C35416EC64A3392FDB954D63BDE0C1BD7E08CFC655BCD02E27537EE61FAE64C0187CCADC045F91F3BFB1758F224AA8768981617F20C889A390B59588E366648BB3A8680B3431A4D0596FC5B5F95C4184BF361B7A2CBEEB424FD08A98B322C37F703A73FA874946CF430A4F068BB1D16898DA8F9BB04C03AB3F068A36EACE3E2227E7E92BB189B1A4A14C9430EF5833214A62EA"));
  test_xof (&nettle_shake128, /* 149 octets */
             SHEX("1594D74BF5DDE444265D4C04DAD9721FF3E34CBF622DAF341FE16B96431F6C4DF1F760D34F296EB97D98D560AD5286FEC4DCE1724F20B54FD7DF51D4BF137ADD656C80546FB1BF516D62EE82BAA992910EF4CC18B70F3F8698276FCFB44E0EC546C2C39CFD8EE91034FF9303058B4252462F86C823EB15BF481E6B79CC3A02218595B3658E8B37382BD5048EAED5FD02C37944E73B"),
             SHEX("97FC612935AA839179BD26673524AEABC59FA66AD4F719106826D62D56EE0201965D4659877928087D079948D4B3FFE6CB55BB212E802807A0B9DCE7C6CA2CA07C357943891A5ABE2DDA248B8F703236FCE8EDE40EC6609DAF6E012791C53E10B0000EF3B8DF19082A3EAF5C0C0B524816B66BB7B08CB6B65207D0D7C84432AF10A552DF425FCB2FA965728EF2B974FB59065EEBB105719D2534A58435651D743C91FA1953255F777CF6F92D3345AB9AF11BEF3ABC00FC46DC4EEA0557DF7B683FE58CA3887311224567880C7FB5FFDD5B24B3B8EE66E63FA09202D3133768AC2A4933C923159D651D616791A1A2BDE91EA53736EA6B4A097BEE70F0EF943DD246184EFAEA018B9DEE2C4FADB93E7A93E40273B5938647A6A10D31A0354DEC08A03937C9E5AC71F52DB0F50F6D066169C03E883476EF512ECBE2D109D5015A1BF660EA842436060ACCA32E846612E8FD799EEF914E1EE2093BB2E30257BF432D5DAA8AF87625B07C3C6327768F0867AF0B80DCE96888FE27203BA076E5B8B90ACB51D2AE442B424D6651E8BB7B36BD770C3059CAF189AA6CC8575F16B58241FB0B7454E089949FD3AA8B347E0A76D2C5BE54A60D9DA4BE789656FCD7BD5ABB9851EC7B72AD4BE53D9D2D771EA11644A9AEB6061D74293866CB112D89E93BA0CDB02390AA66F5CEC4F9651E306876FB5100F8EAE6ADCBFB87BCE433983768D2CB"));
  test_xof (&nettle_shake128, /* 150 octets */
             SHEX("4CFA1278903026F66FEDD41374558BE1B585D03C5C55DAC94361DF286D4BD39C7CB8037ED3B267B07C346626449D0CC5B0DD2CF221F7E4C3449A4BE99985D2D5E67BFF2923357DDEAB5ABCB4619F3A3A57B2CF928A022EB27676C6CF805689004FCA4D41EA6C2D0A4789C7605F7BB838DD883B3AD3E6027E775BCF262881428099C7FFF95B14C095EA130E0B9938A5E22FC52650F591"),
             SHEX("531D6B06BEFF0AF847CBF3484D1DC52B2E7799D3C5C384B25858603BE71B0C57AC073A330B816BBEE8164116424AC5AF8E5B44E875E9A6FF54C54E4030CF51215552D53FDF24FA63E95590D4B11A86E305AAB52A62F7F8B62FE3365D239B53B540ADC34303D5FFDE35B702844D05B3CC804365BC38C76F92F1EF81660FC8E8BA535EC58D2CBC4B0EFA6D840F537740675A1EC1619A8956BA91D3B91D2417C4D151AFF2CF54880B0F7B8CE36A432FC44B30312FE9817BE795F07C8E521A183930EC5422079B915FE6ACE193B7F59928A2F85605D257FFC3FAB1762CBAACE675FE130CD4FC90138AE5303332DB73C0649459C4F8E0C44291AA61EB3D5982E77E1DD1EA3646C2557E62DC13938E805DA7AE0CD2B3EFCFF6E6E61A8F2793E94665AA13456453282C7DB7ECCF7C52BD5BCD1A7A8D0FEC0A5D69DC71FB78E8FF08C7AD7420C0CA52C80E274D9BC623531E375FE41436B4062EFC72154657536391B91D43CEC7EF9B19B901183B70866EA2C4416021A339B827D23626380D72AA8A3D66BD4E119B2D254EA426F1B40CC564DB0605E015773907DD964471710E7EB3435C6CC7951BE6AD4207F40485693F57ECB41D6774A840E495CA1DCDBE1F112F61F8E135F1245FB09059DDAE40250862C79BEFC5432B50B8B8A4CE0804A3127697BDF3331D3CADEA1D459F565E0F2AC133234160F8111766D31A89C64351817AE4A3"));
  test_xof (&nettle_shake128, /* 151 octets */
             SHEX("D3E65CB92CFA79662F6AF493D696A07CCF32AAADCCEFF06E73E8D9F6F909209E66715D6E978788C49EFB9087B170ECF3AA86D2D4D1A065AE0EFC8924F365D676B3CB9E2BEC918FD96D0B43DEE83727C9A93BF56CA2B2E59ADBA85696546A815067FC7A78039629D4948D157E7B0D826D1BF8E81237BAB7321312FDAA4D521744F988DB6FDF04549D0FDCA393D639C729AF716E9C8BBA48"),
             SHEX("4F56F2BAADD2B624F6643C1DBB5ED95F05FBD0550C3E5829472734BE3B4064529CC3DCB3F886C65F074539D3F1B13215256AF4952784FADDF00E184211DB2EC238A5EBDE55DAFA606B0AF6D108791F63121865B6D064CE12E446C0B719A856B1647DC4BA13614505D9B3A91417537E0B110D086750325AC8A1964B8A36BAB91BAA1141DCE862167A1EE4DAA01DA6B3EC875AF9636268CD995DD980DFB6476C0BCBDD8430462AA56BFBEA69130DD43640E23DDC4D9B469A521A533DCA9E8A197130C3A9503393E958A77CDBF9F93FCA6921283C96EE38C7EE92C7F0781AE962C68A07014D6C623A3338A1A19216925A432A9971E7F6366ADB701EFC4D4404E02DF452F9A6B9AECE18AA61C56CAB2582409E4E741B0CE17C6FF881E3F509F536F8F681268E71AEA6376767F5D49DF3239C65775199B68C84389D1D4A426AC10D42046FD6AEE02D4E0F48BA2EFCECF83130792996801E64B81797214A2EDA5B2296550A13E38E7DAF36F9F2A7B89C1DD0A95590E2B77415C3A35B9EEB89BE993333551D52835FE36D91D2F205F626FC03CC4F9386E4D815F6B8B128A9C13AF40386538AFD481650EF886D3DDB7B0772252B73E9F44D7822CC99455FE118C6D1003FE24F304B83E6C76419527F1CEC5E5CA57F7911554D1515A026E573D9D4BF5FF7536475AD14BDB8490CE353E1EDFCB038DBC2819D4EDF325D6C3479F2C3B02E67"));
  test_xof (&nettle_shake128, /* 152 octets */
             SHEX("842CC583504539622D7F71E7E31863A2B885C56A0BA62DB4C2A3F2FD12E79660DC7205CA29A0DC0A87DB4DC62EE47A41DB36B9DDB3293B9AC4BAAE7DF5C6E7201E17F717AB56E12CAD476BE49608AD2D50309E7D48D2D8DE4FA58AC3CFEAFEEE48C0A9EEC88498E3EFC51F54D300D828DDDCCB9D0B06DD021A29CF5CB5B2506915BEB8A11998B8B886E0F9B7A80E97D91A7D01270F9A7717"),
             SHEX("F153AFF7498C363D04FF8BEB191E394852B79F54681E8D03F04B31BC037102BFCD4E7D709A387A89C7DD124FC0192B586F7AD7755C36A8D5DFCB6DD855671298006EF379D2CBBD12CC5C05186717A3985594F19FB4AC62C673CEC6D374114270EC32E8C0946C3034AEACCFC38B745D758791842995BEDB09A4778BCD504A961CFDE3208728B57D246ABBC99DC632A2396AA5B3497D4F1F94FB2C90A5A1F51DFCFFAABA610C747CE10E5EF2ACC9AB03383C8B8DC73517E4D32FA7118BFF599A83F937358F7407B4EA4771D5C504311DCCEBC1262B464415ECA2E3662CA7DFCCFE38157F15B36682221F65B6F65877726072B9A30C0028DBDFD14C7842D99657778C339A5248AEDA1DFA8B9ED56B79F107CB3E690D56A3DCD5DB03C47E46D29C433BEA894D1386133A78E8EB83C2E6BC2E7593CC3AE4D2C0BDA573BCB16E7C44732A370FDE3AF4DC452D029F44DBE0F539811DDEE2CAFA9DAF752169FEC82EEC4F35093237F0ED1CA6CE62A1221AFC870020B0CA219D03E5ACFEF2592DF5916FEA97154E62485A38EFA4DF203EE808B12692064C609F29EA0C6960569E102122AC74E99A992F37C1A090A9927DEAB998303E47487E2910760A6D604A73C78D66213A5D2C28C91528340B2D6371A391C3330FDFAA70C5D109A209152D21B63CF616C9C78621EFBCF52064DA08D90510EBFBD957E28053EFD0B78F28DAD025A72127"));
  test_xof (&nettle_shake128, /* 153 octets */
             SHEX("6C4B0A0719573E57248661E98FEBE326571F9A1CA813D3638531AE28B4860F23C3A3A8AC1C250034A660E2D71E16D3ACC4BF9CE215C6F15B1C0FC7E77D3D27157E66DA9CEEC9258F8F2BF9E02B4AC93793DD6E29E307EDE3695A0DF63CBDC0FC66FB770813EB149CA2A916911BEE4902C47C7802E69E405FE3C04CEB5522792A5503FA829F707272226621F7C488A7698C0D69AA561BE9F378"),
             SHEX("D44138D25419FDCEBB743314D267D3AB2839CD302EDB06DE0EFEB5B837F19D7CCBE8A1F22922437FC4EB591BB23EE25AF00BC1BB3FC6F43D524D5B8447AFA47D966440F620D45C3CE7B482FCCD768E436AEEE5C25E9CF4C3FD814949B61B40A54CA867825BA2A55E6C3E778FF5CDED4346530BBB8D1685A3300E995291B06AE05AEBAE336C37ABDE854E57180A79BF5BE46F6C879B11FFA6573717AC168AC16CBFBBF5C46F81A4EBE14F0A0FAEE8D09D8664D26C327721D25E551676A00492442E3F73E4B748943B188ED3E10A22C54C16BDC78AB1FA8A50D6D46EB057DC0FB048EA4FBEE4E5A1A653131855A03DB2FCC39F698D14D28FD9C361818267E222ABCA88EB0326F4672DC1305AF072E0B75674596A616401DF9A53D44D53FA529A7C28CE7D5A71C97880725D51A58A065136034DC3D706F13648A68AB4665AF00D5D0FA0FF44D1D1A4EBC44B3A13D273848A3BB558BBCB599300728B7FCC6FE16C5A6948BB09FD1B8185BC6863C1D9F5FFF7F0A82EDFF0CE04ECF004587AD52AFDED64E4E094B3BC15A3F7511620958FCFAA69E869E85DAF7DDCFF6CEA52C1631ED6418C822380FEFDBEC1255911829E780F99DAA9C6F4D729A91F28EF1D3DFC505971593663710F82BB95FCFD1BFD5C3CBF916AFDBD7CB89FBFE586982D0BB7E78D031EBE310606557F83A87FB1070B6839499E4D360E2D36607450C720AB8A72F5"));
  test_xof (&nettle_shake128, /* 154 octets */
             SHEX("51B7DBB7CE2FFEB427A91CCFE5218FD40F9E0B7E24756D4C47CD55606008BDC27D16400933906FD9F30EFFDD4880022D081155342AF3FB6CD53672AB7FB5B3A3BCBE47BE1FD3A2278CAE8A5FD61C1433F7D350675DD21803746CADCA574130F01200024C6340AB0CC2CF74F2234669F34E9009EF2EB94823D62B31407F4BA46F1A1EEC41641E84D77727B59E746B8A671BEF936F05BE820759FA"),
             SHEX("D03ED7399E6FC424AF6C4F08715B409BA54237CFBCC4B27A2D97811F1CFAF8FE8A1E72B72C9C0C04587D89D9D3B46487EDA3AC473DC325E2477FFD14F37DA4AE638A2B14E103E2AC9E2F94A8BE49B28DB4144E2EF8A673412B214EE502FB517C480123289DE425543A51910BCB3B52E9B0A5F25BAA700A3DE0782D4BB409938E7BDA9B9A31D1ADE69571E4F1835E7A89C41F35606B9FD7DD513200C3CDAAA0B25B6B2BAEA99B79380D33A21678BF7FD80B2B714E894E61321E559645047F68597244AB611FBA697A079634412B64E2AC53EC230859EC86300B4544239BBF88558CDD48BF183FF1026636A1A28B512704C60E805CAB7E1F8534BCEBB1D40EABB8B98877BCA3A98375F2C305803637B75E77EFFDDA79991F059F697910453D2E986C9E6450A6115BA82506376222FE56DA466FA8740707116F50F27911BA028A633E30A4F83558EF664B63B6890B766E46875A6AB2FEAF64C3587A3F3872D179E826D2E8C6079F159C8A54F36CB770025E7628E936B2BCED01B339E13EEDF21CF01B783731875BBF199227B9A87E5DD4EFA361B86BEE1198B54626D105FF05B293308A7DA89398F50B736D04113F38E3EB803BAF06F049D207E1B6E3C919139BA6A94EF97649151327B2649F7AE1CE88D42CCEE07D826B34943CC5D240C20D084391EA0F3AC7DFA5772BFC3DD37443C14682ACB0B127C66C284F341D5BADCFC381"));
  test_xof (&nettle_shake128, /* 155 octets */
             SHEX("83599D93F5561E821BD01A472386BC2FF4EFBD4AED60D5821E84AAE74D8071029810F5E286F8F17651CD27DA07B1EB4382F754CD1C95268783AD09220F5502840370D494BEB17124220F6AFCE91EC8A0F55231F9652433E5CE3489B727716CF4AEBA7DCDA20CD29AA9A859201253F948DD94395ABA9E3852BD1D60DDA7AE5DC045B283DA006E1CBAD83CC13292A315DB5553305C628DD091146597"),
             SHEX("5C28103BE52C28589EC6B8EA53EB4769AA91CFB8A8F5875A6201BE4F03FC9A1CAA8E582A11FFC6C9FD4CBA3F5E3F21549514CBF8470568DFA2ECE224181B865E224F8F07CAA4A0891DCA60EEED7CCFD23A62731892A1EF32D515165BCC4885997ADD88F59D0124B53E14677F8C4FFF979E9998A76E7A3041D3FB0EE9FE528D08C80481B8E3EB77189AA29E64497269F88894DBF7141F21175132ECCC4E56BFE70CD2AAC208A6A0898A474470A5961324E16821A78AFC25FDB7CF1684295C5036C95DD52478BD2FDA24A733FC7E08BFFCA82387327E69BB63F8E769B423CDAFA965734B666C489044C23617594B309AFB2F262BEBF447A87F41E3824972FFFD9F4A0D06B4BDCE742123C6FC7B85939DD8F837ABF2990C6DEF398DF2B44E40E3C25A6D9E4987B1BD636FC276CB0000095CF441BA3FA274F1F95AB321DE9A5A9BCF219449EAB4E91EDFD058674A4E8CC5F8C0831969DE0C17D0EEC98DF883619FA1DF794F2D09372A7EF20BB06427125B6C0849EB80238855CDD3CAC4C4968601D053D7E972785AFF96D5483ABA9D55E3695655FFE879D27FF6CFEF0328764D87F786F6BC74CD40B48C50BD1428BCE5AA96A477EA02B70D17333FF176F2E70B6EB7BDEBDBD0750414E5F32BC521F381F1827214F4D5746A40558064D49E5484D54442F7414BF25DD469181D524462AB73B9D31A729DBFB06DA0615FBFE49869D98D"));
  test_xof (&nettle_shake128, /* 156 octets */
             SHEX("2BE9BF526C9D5A75D565DD11EF63B979D068659C7F026C08BEA4AF161D85A462D80E45040E91F4165C074C43AC661380311A8CBED59CC8E4C4518E80CD2C78AB1CABF66BFF83EAB3A80148550307310950D034A6286C93A1ECE8929E6385C5E3BB6EA8A7C0FB6D6332E320E71CC4EB462A2A62E2BFE08F0CCAD93E61BEDB5DD0B786A728AB666F07E0576D189C92BF9FB20DCA49AC2D3956D47385E2"),
             SHEX("0AA13616E9EE0C4C35FE2B921118C67145B8458E6179431E8145DA829A2EFD5B8831B067409487FF6C865B8CD69695FBB2B1C4AE1FCB014540B842F5FB3EA40DCF7C16FF295B2271215F5367FDEA93C17B530C472B6200362ABF8D81F3840738EEA11961F7262C163F5BAE5FED2ED42A936CE1A1917DF2360DDF4DECDAE45E1522400F37D55687CD334B47FB8B55464AB7CFF75E9DC0CE51D60B43BC2D468F1DD5BF489B690C7DCFCF6E75301C24BA534047739FE592E8637107CCCE1B3D2E4BC037AAB54B3F8DFE57B9580CAB38A983EC5DE9184AF589D939FB7F6EDC886790D1CA4C780F82437EEFE261EE4737545EB7379FF09B644936F8EC9775E9F118A1958E7E4687C205424E4A31C08BA819F6827654EED2451F6A2242F81AAB3A9AEDFA71437E5AC288BBAC9505B07E302BCDB109FDC9E2B753432AC6502A3FAF9EC54BFDA1CE399561410B62C4FF8FD922A1BFBA071348ED9F1B242DF772BBA66A76C0C1CB209CCF7828051F0A85BA54D10B4DB0D71E37A77ABDD510EA1FF02EC72BA6BEB54523DE49167BF01594D0C49020CEA7809BCC89A6BECF9A6C655D9D1F06C2CB1D2D15128FBEC2FC388DC39AEDE65029954735285463E2684226A10770358156C98300FA70FBE2C200DBC0D60F3194BCE21BCA23AA6554D56503C56E8D4A4B246348A8DAE6FD1150CB6663C113B3F15FE8DF8066C9BF3979CBB3DBCA43A4"));
  test_xof (&nettle_shake128, /* 157 octets */
             SHEX("CA76D3A12595A817682617006848675547D3E8F50C2210F9AF906C0E7CE50B4460186FE70457A9E879E79FD4D1A688C70A347361C847BA0DD6AA52936EAF8E58A1BE2F5C1C704E20146D366AEB3853BED9DE9BEFE9569AC8AAEA37A9FB7139A1A1A7D5C748605A8DEFB297869EBEDD71D615A5DA23496D11E11ABBB126B206FA0A7797EE7DE117986012D0362DCEF775C2FE145ADA6BDA1CCB326BF644"),
             SHEX("E27B1C3BA27DC48F20B9605C4FFFE0F610FF909B482D213C34B49F0FCB04A927F0B1C4D2C6AB6C9BBC6E7DB696F7E41FA1875D08A65E21D0459170E4BC955E7E596AA01C8F52127E858687E53F17691AD1B1AEF7BBE75D09613281CF96E86EF93850AB4277A54B45DE3663A906B55BCEB2528A55CA6E17094E5D0D63F3294D8BF4DF2DA524296CFCEAEC6F1063A7BFA92B300A1C4BA7E2CEA76EC4C7FE9531E7D9205BEB4E2B5FEE32CC0881401C573A88BA9D29B5F5A5CAF044B0A7931BD605073B19D50F82D822ACDF1A6674CBBA6C88362BAF769F4220D731F3E8DBE707B12D44208E646A441FE742290E9FF783744B0F5627CEE1B91DE431DFDCD7AEDCCDBE04EEAD6DC6EF9E9C50131FE37B6707EDF8A021B3D9BCB0F62AB331F43C80B49F3DD5AD47CD3956A2B45A510FEC358BD211B2FBF06B6DCEB82A3190BD0AB28C7CAFCFBA4432102CA3067201533D64D44DCE8A632C204CCAC68B248F0FDCD54BEB300E32CA76F03413B4401BA4721863577860C892252DB31774EBED6BF8418EF0A12508BBB15B41DC0F2EE5A890915ADB2D0F6E28ED207EE8B6536015B763A60A1A96C9945AEB4616613A0EBA3B87C02FB1DA1F9AA169EF4A5CCD967E811BF83D2728B88BB5CF2A5FEC6020137C668EE4F21EAE6D2759F054AB1559F9241EE2B65CF36A907720A7DCCE655FAFB5E534EAD32A4745A947BA6C4B20240DD1FC40"));
  test_xof (&nettle_shake128, /* 158 octets */
             SHEX("F76B85DC67421025D64E93096D1D712B7BAF7FB001716F02D33B2160C2C882C310EF13A576B1C2D30EF8F78EF8D2F465007109AAD93F74CB9E7D7BEF7C9590E8AF3B267C89C15DB238138C45833C98CC4A471A7802723EF4C744A853CF80A0C2568DD4ED58A2C9644806F42104CEE53628E5BDF7B63B0B338E931E31B87C24B146C6D040605567CEEF5960DF9E022CB469D4C787F4CBA3C544A1AC91F95F"),
             SHEX("B0FC5EAC13AD8A3CC49B96E1D9FA96CE4658EFACC7394DE19AA7A28CCBDE37CF1201DEA78B4C2F127D397AB5EDB08A505F72C70B61678AE081E139F761DBD5741DF0CDD04D5A9015C0A65975E7DB09D9999FB77324366EF22BB63F2776F9A783A263CE31583D0860F3B91A6620312ACA03EAC2C38BA62C9C9FA0B060A4F8CC0FCDCAE1D414323F2D649A8E9F808FB02DBF926DA2FD36C64ECCC04909E2308D2C0367A069DD64E2DFDFED7A5A941D2B84C5B6A69956D43843C9B7F14A0636036444870B1B9329B38285A1054A1019B6E01DB1C66A39EF9B20EC4E51580EF546E885B99C0E3EA1CD724F8BCF56D1D00D1AA3B8D9B487964E773330B3F84577DA7CE9385D37A8F503456897F8BE8DFA9A60D129E538AE0CA2F6BBF851F1E56CD3F14864EE0B119A1BAC4181C667EBB6C464AB76A6B72681758C70422207E6F62EE438ED6FA467408CA04E48C53350FB11282E89D41A35EFB31C80B18BBF4E28CEE577798B9726A270245F35FAA53DA25DCB9E445A842D843DA21F197F36D6B43E83C3617B2AC92AA937697D5589564D0FB4C225857B6CEF6EAD6F193AAD457A5F5F7EFD7FD88B1AC34B00F4788C4A549474A852403EFCFB151D941C949BE41151A7130D54730F615E638DAE243EAAB7648A028CCD05A542054E66B3D5507D78117734180D7656D77677A2C61040DC7AE56819489C0D106B798B9C549046251163A4"));
  test_xof (&nettle_shake128, /* 159 octets */
             SHEX("25B8C9C032EA6BCD733FFC8718FBB2A503A4EA8F71DEA1176189F694304F0FF68E862A8197B839957549EF243A5279FC2646BD4C009B6D1EDEBF24738197ABB4C992F6B1DC9BA891F570879ACCD5A6B18691A93C7D0A8D38F95B639C1DAEB48C4C2F15CCF5B9D508F8333C32DE78781B41850F261B855C4BEBCC125A380C54D501C5D3BD07E6B52102116088E53D76583B0161E2A58D0778F091206AABD5A1"),
             SHEX("7B6BFD2981CC29669A120D28930CC39BA577D26BE4A822499FD5D36C7BE1528DBA41F2A60C0F1832AAE0EDCC384B5552013BE17E1C81C326CCE1FFF5E32B24985A2E2121B072902BD2FC6CE05476ACBF2D2719128AB1C35F93EF82290A56542A044B295A26C85E4BAF8B8544816F0BAC4753D405C9CE02AFEE0B312B5EC8AD12A07BD3D177F4BC7AFEF6E092D23C7D445F8B21DB59C9A6E819AF9AFDE919AB550E1A6640967D296D3DF3D987B6E62CF781A65BA41177ADB60B7CA83F57C5FFEA1D1D96BF9EB2914CCD73162A8B61B85211D010AFB8AFEDE8EA96473B080E98EB7A2F000485BFE6E1E7E6E2DDC1901E77C60DB3A65C2F5E57AE72B0CA4BAB6E2EC40F62F33E865F04598F66FE14039FE03C5DF379319413440A31782C2FE4092B0B701E4B792EE449FA452C023FA79EAC7A11A458B497213E7FACAD5D5ED475E71FABB4D9F6A2C2613A8D4DF3F574B5C2A884645B6D5FEFCB355FF18DB8ABD8A85F2BC531EFBEB96F24E204733E9EE3F0ED2ACDAF99713ADF32DCA518D5D86659F25EFB8EC5E5D82E64460094B88DB621962F76C9F12E739A5061ADB812043B9BD01FE9B4E549BEE9E304940DC7F05D45ACEDD417B9B37F214411F75B427E2E9CE14B6DCA4FAB0BC893001CB7A0529C1421E5B721A9BF9031D164C6C6192FB38C229C852474CBADD6924CC8A401662DB173D7FC9FC0028C905574802EA77820CD"));
  test_xof (&nettle_shake128, /* 160 octets */
             SHEX("21CFDC2A7CCB7F331B3D2EEFFF37E48AD9FA9C788C3F3C200E0173D99963E1CBCA93623B264E920394AE48BB4C3A5BB96FFBC8F0E53F30E22956ADABC2765F57FB761E147ECBF8567533DB6E50C8A1F894310A94EDF806DD8CA6A0E141C0FA7C9FAE6C6AE65F18C93A8529E6E5B553BF55F25BE2E80A9882BD37F145FECBEB3D447A3C4E46C21524CC55CDD62F521AB92A8BA72B897996C49BB273198B7B1C9E"),
             SHEX("6D3173F315362F961F2728C60EFFD5EC721D68EB7D2E11C1973A84A19060EB5E64DD8FAC20C76A89D811CE260D7020EA9F7ECC78EB3DC91171CBBD0818B31C4D717274DC164EC5AF48BD1916B0440B4BEA6E4D482329C17AA557DCB4DBAD69AD74BE1560EFB043B78301B3B1D42E9C2B096CF5D8138811DEE6FF3753055C77FB2C490BF79E2F7C48D492E59B5D74798A1095D9BD03EF74CD171E615DD3845499A546BF69AD263C1B845B05B5B3CB32351CE01EB10B918D192D1605802A6607EC5DE09824790B5750685A8C679B54789BEC0D6FABF80E5438F9BED0DA8DC5773FB21C4CF231E1163B30D676A68BA04F7C6BFE5AAA2807A52A84E1D12855EBFB2BDF44C34AD846293D71E3B05D17C1D4102D5C7775F30D13F8290C89DFF886B1D126834516FD3ACC848B988ED4C4F046B39CA0BF2F17A0876B5EA782160DEBF5A2709BD38FC0F6BF3D1E286B7A34A59AFDF4B7559AEB2330D94576BC10BF5D1247EEB12B45516C2287D6738080569FE9179B83A27F007073428C7316AC974DBD98E137E771FC7F4ED1B779C3C6774A2A36A5CABAF10CCC742D9190A3278A33D6AF490278796EA76D8706B99561390E06E90396C4CAF101AF1D86748171D07B5794FA831ACC94A2452769C83203B2F845AF987C81BBD02AA91A1CA481ABB0B25991C0E61836ADE076F25DD1852A032A223B3D8D12CD544ED0A2F97D735A0E1AA9AC"));
  test_xof (&nettle_shake128, /* 161 octets */
             SHEX("4E452BA42127DCC956EF4F8F35DD68CB225FB73B5BC7E1EC5A898BBA2931563E74FAFF3B67314F241EC49F4A7061E3BD0213AE826BAB380F1F14FAAB8B0EFDDD5FD1BB49373853A08F30553D5A55CCBBB8153DE4704F29CA2BDEEF0419468E05DD51557CCC80C0A96190BBCC4D77ECFF21C66BDF486459D427F986410F883A80A5BCC32C20F0478BB9A97A126FC5F95451E40F292A4614930D054C851ACD019CCF"),
             SHEX("07CD8E0F1916D265A748A8EACBC81B045D004FA104FAA1E3752B9AF131367FF0CA7F63DB46480B67C405EC16EB2922E60667018B9ABD91FCE022AE9B0FAA249444B12223CB43750A42B1F9166F54836C7866BEE31ACC681B521D61EEA57C2E84A8F8CB9826E20AD2B14377B313225B50A1B49AFF085F89AAE22D3BEAED142C585316615281D88C5482E9F7E23770DA82F27D2511E30DD1958C903EF7AF6B2077252C7701D474F8DA4149AB45D80B6CE9EC1CF08877B6AB13CED7FDE405A1DED59F856068D3619AEA92789E8E31BE67DBC2F402FDDEC3E3E505FBAA396BCB6DFC5861E497EF576FE4015EF743BC4D5936F8A5247B7667A321614539BD4DFBFD79F3BC90E6CCE084E8AA6C9ADC17C2E1704E4E2EB879DB4D6A7A6621C972EE54CE582F3403C76BF3FE60ADF14D21D6438146EC92673B4D277825F1BA515C86345CC280FD66A006A387488BCD537C5B0325B21BB69E629763259884E9C3202A16ACB5234CAF9882ABC9B0935466F0C7104A04981F75488EEDCACA9EF55F03E3323FD4086E19043DB1F7ED7F9AFAFC968B7313FE6705673CC31387935F92C26AF140D2AABC7BB76F976ECB2A6F6DD1D0E65DD997F8648B7BB9F1C9ACEBFE1988E9514F8770DA4DF0B5EF397E58470C656898667A2E0218C70CEABCCE94025C3914C91BC1BC76E82B1C17A7B938E92A57E953CB80AB15726776382C74DBDB5FAEEA56"));
  test_xof (&nettle_shake128, /* 162 octets */
             SHEX("FA85671DF7DADF99A6FFEE97A3AB9991671F5629195049880497487867A6C446B60087FAC9A0F2FCC8E3B24E97E42345B93B5F7D3691829D3F8CCD4BB36411B85FC2328EB0C51CB3151F70860AD3246CE0623A8DC8B3C49F958F8690F8E3860E71EB2B1479A5CEA0B3F8BEFD87ACAF5362435EAECCB52F38617BC6C5C2C6E269EAD1FBD69E941D4AD2012DA2C5B21BCFBF98E4A77AB2AF1F3FDA3233F046D38F1DC8"),
             SHEX("16CAA902C04424D5FBA44B3ADDA34326EEFF62D6354E3EF5200DC07A0325B2004CFBC3C972A453FA32BBD68AF381F78FD768158194ADAF9B10A2A95D9EA9E8D0266C5D3271BB7B59226C8EAA9F15F9ED324AC91FD88B9A23F50EAF0F17D8582E8476E8806F31B58ACC36EBAB6B856CE1738704753A7ED31F822BE83E4E08F2A3F3DD52D4BDF8090DAD90A015E9184A3202974897D525F23A4EAFD1A565B76411D8B78DF548AD2BEE3830FAAA3E8ADD2DBDE479236AD2DF12841622877EB4EE895A0AF9799DD727ED96ACEAD478C8F4B9AEDDFE7FA3879529C60026BA0A7966799B372B922B4CFF3F6213C5D73C0D778CC4B9CE67013D905870B12CB3E2326223F57FDDA8DA32F3700E855FDF3DA47BCA440B219A925258D6B38625AD1408AB303855E3734067ED188FE122C5C279710E8EDF954CE4F43F692DA7EDDD3B84164C567B2852EFA262AFB09F49E50B8DB278ED215EFE1F0F43EB8BED5C3F1ECAB591E191CD36B6EA75260285F667C204E159E72072E165457A170F32A7FE29664051396BA1BC202663F1CA8F17A99598DF58A512B6685798E5E93FAF9B49C9AF0CA8FE4B6544492027CB5599448643E1EFC642C7C71821BC06D5C600A80CF43FA64342CFACE0C57071627903DEF82A0AB3A3089C06F89F99A5B3E11524ED2FAD86B73F3F48BC5B912CF31286C498305B77CC8F7D54F343993888CED370DC93E0CAC9"));
  test_xof (&nettle_shake128, /* 163 octets */
             SHEX("E90847AE6797FBC0B6B36D6E588C0A743D725788CA50B6D792352EA8294F5BA654A15366B8E1B288D84F5178240827975A763BC45C7B0430E8A559DF4488505E009C63DA994F1403F407958203CEBB6E37D89C94A5EACF6039A327F6C4DBBC7A2A307D976AA39E41AF6537243FC218DFA6AB4DD817B6A397DF5CA69107A9198799ED248641B63B42CB4C29BFDD7975AC96EDFC274AC562D0474C60347A078CE4C25E88"),
             SHEX("1E8FD76A766A100A4434AD568DA7E753C08E76E515DC7A4D42A9CBE5369E3CFAAB7A09799B9CB66940DE779266051581E6B039E76F446327188A6B34565D738F1A926DA3BE16CF6EBD6FF7F788FDBABC0961158321E34C26207ECE1A122E73D5F9675A2FCB2E4871FC04875FBDB9B6411BEBBFDBE71DD643BA6BB2E2CCF46D1C1DF9DC6C0777E7A19823C1D55A648B847E398F52D02D5632A69323E588ACB113289260752BE85A42746E62087B5A32DF812817D0A2409C36F6C80912D4AB80D47C7DE7FFE26E10B1C850B7339274DFCA594D418C3C4948BB3528F4EA463CFB54F7944B32C050C1C3651168BDDCC873ED8057372743A6CC8365763B523FB3D9752A4FDB167A768E96C7D4F277BEA031B941BB7002DB130B76123A0EDC4544DA03D9255BDCD896789123D9999AA6BDECCA01E4FBFE33860C74ED2FDB54412DB927DF2D896D289A87591377923EDBCF2D2AE204AE6EA85B1A9AD2C2A2729DBA3D6E6FAB6269288416D569532793E49F61E7002871B8F4A68DE84DDC42D081D1C4CBE332AC58C3D81896A438D1861E594DB79390AC2BC9398DC21B27E6F7677A94F91EBED0103B2E8CA4068D6489BD0AE5D6B77053616335ECC7D7218B1B2DF70A279123268A06EA959965CA1AA26E1DC273142CFCAD570330AADAA62E04C383A938CE349C997BDC41E9BC38C29767B964FE2D8F6DF96EF1A5BD0A38F06734327315"));
  test_xof (&nettle_shake128, /* 164 octets */
             SHEX("F6D5C2B6C93954FC627602C00C4CA9A7D3ED12B27173F0B2C9B0E4A5939398A665E67E69D0B12FB7E4CEB253E8083D1CEB724AC07F009F094E42F2D6F2129489E846EAFF0700A8D4453EF453A3EDDC18F408C77A83275617FABC4EA3A2833AA73406C0E966276079D38E8E38539A70E194CC5513AAA457C699383FD1900B1E72BDFB835D1FD321B37BA80549B078A49EA08152869A918CA57F5B54ED71E4FD3AC5C06729"),
             SHEX("58407675270C110F4E998D9EB9F9BCAB175E4D17F2C3A88896F31CC97A389728B3A93729ACA3AAA426E2CAF11AC9D121B3239915C857FD377FA2871F76E533A954E6790847FF03531111A89B35192BD624A5FE0823992F9A7C25721E10897FE397C2DE7828C258EF6083544AAB1C7FA5AFCB7B27FC37E64DDFE827883FA0389B8E785676C92396C3E1C1E161E2776538CFE4B6E20653F5706F034929EA002926C0BFB6D9C31857BBBEF51CA094FAA6A7C9C6B34DD92182E21AA47923C908FA00617F9E910EEBFF2BE3696B6BBD489D72510840F075E24C81F63A52772B8B5F3296C7F7095B4ED732F5DCCEF632A37F0419378EDF651DF2870119A4F1EB0C961AE792D7DB1FEC88BA12BE14520648AC3C78D7304305B75D3B1AA631591C1B8128DBAE0EBEA7AE1DD44ED1950A87BBA1BB72FD0BCF158F7CFA4881DEF36B30A36425C429E3D4BAE81EC821B58B6EE1C5FB9C9D93993CC1C341921797797D39CC523D32987EB0D75C0E502F52EF1FFB3DC10A19766C2117122010AA729326AC5273BED622CD04B924C60CD405A14ED06ADFCAF35B1AD5EA469CDEA963D7E97B2058D2588BD866A17D701761439C491A0AF937F7B8B5E8FC781737986505CDCC593DC0627C0FFABC0A1737AFAE194E24888377AF99AD33AA8A0455CBA1411A40D1D66ABD03F2DA49C34C74C65C830BC68C36F87F9E578072AE67A60C23A6AB0F02E0"));
  test_xof (&nettle_shake128, /* 165 octets */
             SHEX("CF8562B1BED89892D67DDAAF3DEEB28246456E972326DBCDB5CF3FB289ACA01E68DA5D59896E3A6165358B071B304D6AB3D018944BE5049D5E0E2BB819ACF67A6006111089E6767132D72DD85BEDDCBB2D64496DB0CC92955AB4C6234F1EEA24F2D51483F2E209E4589BF9519FAC51B4D061E801125E605F8093BB6997BC163D551596FE4AB7CFAE8FB9A90F6980480CE0C229FD1675409BD788354DAF316240CFE0AF93EB"),
             SHEX("D102BF50A47C12EA4C8D5169BFD14CC92EFBDC10B042631B1E0CFEDB32DE59947EB35402FBEAB157062BC6E89E0CBAE88BC68190B51FF1F45EE278A486E8535CD9D272A4439EC3F7E65F1C11209B77028B78EAE7A19CAC0BE054FE0811ED8A1F230954FD78130D27D909555EC58DAD0A7494BAC5323FC56E616795DE5D16C1E6427A02E7C33FA8C74459A24AC3A377A7D5D9036AFE7F23830FA3ED8F4762E86B3ABE0F3159377CEBB1E6844835374534F86C04B5215345947ED2062872F3C8FCC82826B0AB63A9A5EF699687C44602F61BF1E9AF303F977BFC3232235D95B5B6AEB5241010F01DCA2807FBD43D84675C5839A6E2D73B60C037A9E2D98F3AF199CB28E4EA6B3BEB9DE80A702C8B70C039B32B7D63C892A6DB1B6FE5A9E2638E8344E4EC2E77394FA03735C7201A656FF0A95E19A6F8D5A8C41605A2731235BA721A4A74F7BCE9CC2B1FED1B92F42C2EB665A41EF09B83E75D1B3A438DEFB704BB360404BA7D67BC00DDB8541949633A106A2BA75930F30F6079BDB01568E46435CA6BF80CEAF3F0D878A560C969263202A8EA9500B71369C958737D85BC3BFB4C3452A47CC9ABA4D69570AE9A2449C1CB84CA47F226471423F0AB54516458C23E31DF0F80F9A3EB37B0E74DABD4B27BD1DDB50650C92FD01ADADE439445A054D463F3B068E0F890C33D9F9BB4C31A15945080717CCC0CDB71199B7CF607AFDF5E"));
  test_xof (&nettle_shake128, /* 166 octets */
             SHEX("2ACE31ABB0A2E3267944D2F75E1559985DB7354C6E605F18DC8470423FCA30B7331D9B33C4A4326783D1CAAE1B4F07060EFF978E4746BF0C7E30CD61040BD5EC2746B29863EB7F103EBDA614C4291A805B6A4C8214230564A0557BC7102E0BD3ED23719252F7435D64D210EE2AAFC585BE903FA41E1968C50FD5D5367926DF7A05E3A42CF07E656FF92DE73B036CF8B19898C0CB34557C0C12C2D8B84E91181AF467BC75A9D1"),
             SHEX("29F1559D1AC3E5F5D6F36852E6FA06AE720B939F3EE14FF2B354C9B941A4F608313FD16527486E4DC5C032D0278297797CB591B80F3190EC0F1FA08F87A529EB22A2247E71F05FB4B1ED4127BE219E73B8572F9AE21356D8C245FB3AD4DE33A743536DD852F2772595D2CAD5E85928B6199830EA78D6D0226E7D2CFC35AF41EBB590215E416201A58D990A848125247A1A8319630DB6FD7F5FF4CF2499F94F1C0041AFC9764C2BDF2B0FACA97B3A9F8F508162077FB21549AFD5095AC0F2D744065B90B9CCA781640213B3920B284C6AE6E9877366AC4695B19BC7DE88F3E6113FE6491E7830B4CDF74D3E170D7390D9B0403923C8B6471EB5D639E2704D3C360F4E38F6774C6A440D68C2898D11F17601F33EA893C5E4B42D2A7BE5D07492D20D1550BA335E3D105B4898A2707E5C97B61011707375D5AF0FEF1477FEA62D383C227E2DB4CD59B85E45A81A4562E7541DC2814F20C969028CA8A0E9C14386A648346898DBDE372ED9C09A40A2AF6E0AC541BE0ABBAB84B2DD6A2FDAC9EABBB8C87F58DD95BBA7DB96403B9BD2274367439775119DA7EBB8BC46FBA2A50C75454E386C3749D03691F6705E70AD716095CB30326B1E628B0E29D7AF932F554A6FC0D0769C4D0A56779A1878DE7C3671B68A4964C5A9FCD86DAA7BC5B95C6044AC825BAEB4F073AFA2502CF3388290F5658094DFF31F9C68EFA7AC1D74881CE092"));
  test_xof (&nettle_shake128, /* 167 octets */
             SHEX("0D8D09AED19F1013969CE5E7EB92F83A209AE76BE31C754844EA9116CEB39A22EBB6003017BBCF26555FA6624185187DB8F0CB3564B8B1C06BF685D47F3286EDA20B83358F599D2044BBF0583FAB8D78F854FE0A596183230C5EF8E54426750EAF2CC4E29D3BDD037E734D863C2BD9789B4C243096138F7672C232314EFFDFC6513427E2DA76916B5248933BE312EB5DDE4CF70804FB258AC5FB82D58D08177AC6F4756017FFF5"),
             SHEX("C73D8FAAB5D0B4D660BD5082E44C3CAC97E61648BE0A04B116724E6F6B657684674B4B0E90D0AE96C0853EBD837BD8249ADBD3B60A1AD1FCF8A6AB8E2F5AA7FF197A3D7DBEDEFB433B613536AEC4D655B7BCD778526BE667847ACD2E0564D96CE5140C91357FADE000EFCB40457E1B6CED41FA102E36E799792DB03E9A40C799BCA91262948E17605065FBF638FB40A157B45CF7911A753D0D205DF84716A57112BEAB44F6201FF75AADE0BAFBA504745CFE23E4E60E67E3993622AED73A1DD6A465BD453DD3C5BA7D2CDF3F1D39376A67C23E555F5ACF25BCE1E55F307252B9AAC2C0A39C885C7E44F204CB821C0D37A22DE3A71F3A1909B11B7181C42BE9B78AA0D0A15444F330007554BCFCC0D8FD87D6431FB93C7CC38767A55D30D3545560BD380DB8C4C0EDA9399F68F854644266C1B7958B270E75B79134AA01E7DCF1E6FDB6D9AE5D02CCE8CE8E480475E9617CC42A91C08D9AF6E5101B8AC5834ADB2C66987F42A580BB503A4B34A9F15ADCD0E23D0D4029479764831D06B5CAF3F14B91449F15A291F4AC250B270B6CB3C304725C99E32645E1FC02A0CDDD9E717911F2342D9482F8E0979985A0170D725DAB4EA66D44F626BA475925FA39FC9DEE929C06D009416C0ADC1D987CD625A20ACBA4CC87F72F610867C3A7A928A3A0379676E8FE257107AB2F5C030BD2480E3D1863562E1FD0790280333ED9D5DD5A5C"));
  test_xof (&nettle_shake128, /* 168 octets */
             SHEX("C3236B73DEB7662BF3F3DAA58F137B358BA610560EF7455785A9BEFDB035A066E90704F929BD9689CEF0CE3BDA5ACF4480BCEB8D09D10B098AD8500D9B6071DFC3A14AF6C77511D81E3AA8844986C3BEA6F469F9E02194C92868CD5F51646256798FF0424954C1434BDFED9FACB390B07D342E992936E0F88BFD0E884A0DDB679D0547CCDEC6384285A45429D115AC7D235A717242021D1DC35641F5F0A48E8445DBA58E6CB2C8EA"),
             SHEX("4A05F2EF1AAD5FF4306429EC0F19044077FB64BFE1DCC50F74C3F045E9A9C3DE4A3B5963AEF771B049111B7B4640E20B1BA84ED7AFEE325571ACF347E311F33C1D421F21D663065C4DADDBD1785C5DAC0D554CEDB1A45A32E28145E98F49DEE285B33DE14C336D10950ECC30966B79E8613FFEBB702FCC00A1C4250DD385ABB537A284E9108D16B6F08F4E103F2C5E9E5C879CB5095534151E3C9A316D06DCE53B7F01B424D375B564FE6839D1D1F00A2E62604060A9748BCDC8143737959FABBCAE1851213E6DC28BEFDA48149DE6AAF4A60D4615BED67D11796F6173C3DCF139037B31EEC9A8404DF07597BC266D3C7D9EB9A7CABF749FB44E40D746D0E9DFB5C8BBEB25E3F1612D03D3EB0C154DE4B2708C4F8A89762E171F744518AEC134A02EEAF49DB2E2C6C9914711288D6B0CE877861D9B10ACFCC19643738287DA005282F3FC82F9F50AA681F2F55FE1809C9E23A3A59E51C2E894F718372F9FA1564B47AB3F43F0747A17839E933369B6778053E1764F52C5F319E33C8B25678F72332E33CCA97C68F19E058E70C31410DF4DE7E08169D6096B7B4EA48271EB684FEE9FC8B561C3FEE2DCE83D092B142BEC1478D26B48C3C6E597A7B2E44027E1ECA23178D3AFCC67BB530A529C7E1336E1ADAE74EF0BE9CD61E91C6AEA57F7CCB23B64B2F84861CE159209FEF7A897A16A871AA99E63A5126DF2B03387E42C3D18"));
  test_xof (&nettle_shake128, /* 169 octets */
             SHEX("B39FEB8283EADC63E8184B51DF5AE3FD41AAC8A963BB0BE1CD08AA5867D8D910C669221E73243360646F6553D1CA05A84E8DC0DE05B6419EC349CA994480193D01C92525F3FB3DCEFB08AFC6D26947BDBBFD85193F53B50609C6140905C53A6686B58E53A319A57B962331EDE98149AF3DE3118A819DA4D76706A0424B4E1D2910B0ED26AF61D150EBCB46595D4266A0BD7F651BA47D0C7F179CA28545007D92E8419D48FDFBD744CE"),
             SHEX("D99E2166E72399F4B775C8C446CB809F149095DF22378DDAB711B867E3CAD17830EFC3B9F7D90FBA1327E103D3191595AD415EF71604C7A95DFC417F14F5B5AC1B6905396ED94D56ED993E240586E95D98F3D34630C32FC1719E7754BF4F12A321691EFDCD42DCA695BDA56F975EBB08D59F76911ECEC5B503E7030D1E626EC89CFDE51042F1B75063AFB50FF7B43563A09E209B7842507B67E85E87A99FFFA72DA7229CE5C9F5303EDA061A209F46C76BE4114BBF5EBC5AEDE7E22F5921DA87265C19F87F1E37CCC218ACB903BFB4D617CB95DF94255F980924A045B959AAE9C58DBFDDAFB47C9AD78324D27495EADFE665E8B7154EE52AD0219421014FE945AA8C2F3B6223F99E170D2D169C13C0D1CD481B6901A1770DFCB39ECCBB40FC6790E69C61C9EC6E99AF6E557EEA2AAD2E73D8A82FFDD2FD32C63DBE4858D97A5955C6B3420FA12AF5CDD10DD8C9D6D3C230272187E855B9F405853E6B8E199F071F0EC781BE8DFC29E93288F22F600302475CE1651D3CB671A0635BAA3DAEFB73487A813A0448EA5F2EFC915C847795BB0F4F5879DF7B5C95A2DD3A5CE79DF85BCFC1D99AE98748052A27B775B690FF9B0240E0BC0B79935AF998BBCDFD37A2829482A9F51FFB253152ECCD35588D2CD96BB19441A14D7CB3B7EB2F47CFFF6BD8FF73BDF9D9C2A613EBFC5D69A3D7912C4CF2B41DB244267AF889D47A037EB961"));
  test_xof (&nettle_shake128, /* 170 octets */
             SHEX("A983D54F503803E8C7999F4EDBBE82E9084F422143A932DDDDC47A17B0B7564A7F37A99D0786E99476428D29E29D3C197A72BFAB1342C12A0FC4787FD7017D7A6174049EA43B5779169EF7472BDBBD941DCB82FC73AAC45A8A94C9F2BD3477F61FD3B796F02A1B8264A214C6FEA74B7051B226C722099EC7883A462B83B6AFDD4009248B8A237F605FE5A08FE7D8B45321421EBBA67BD70A0B00DDBF94BAAB7F359D5D1EEA105F28DCFB"),
             SHEX("CB7A1F8F3B4A92508EEDA0287EF3407C5F1A51F2600D3F13C214A36F54F5FA63878DBF6E76E570B72ED63159E01BB948D0ED7602E9D649AFEA2BDD4C97C7B22AC32485816D9F0BD8FE1443C7709884334E4AEB5104FAD05B6C62FBBA4F5C5BD2CC9CF576991DA24AE01827F3EA70509E45E1BB0742F3F21D673A34E9AE992ABFF4A33EFFE61C2C7ACDD0B653757C987DC1F575B7181E79702C82A389B0DDE618DDE5EE97A1A515365D04F3413BFF31C3F9F222BFBA2061993C7562AA7761175FD51D48BEAFE94E16AC34179CFE68791FFAD63358885C289C0D1D5C99DC42B7DEE58D9ED595CF5158D4D5055CB249025F3FA9690A18C9459D7E8111B88AA37D26B384A7BD02E923C2ACEDDBE31A6F3237CCAA778D5F7C71CA96E72E11920377C785AEE9DA9D8F11A4D4CAAC6CB66AE4F251422104E5330BF06C004791ADA67D687A0462070B51C0860F1AF7AF4968FEAEB157A4A4448B69AE1166578804D384720A2BAA55F06AB17FE0B4A5E4EF058A14D17F99BCA359CD1E95DE00B5F309052FF4231ACE9E1BCB6684355EDF7924BCE95264C3FFC641588FECC3775AF5E5361B5EE995F85F9A612A0055B22F2D440DFD80E91329A2FA02E20476CF84120634F9C320E6803992817947EFD9BCF620F4B038CDFDA1CD559801CE76FFCF4AF63A4AF6284293789B548314B2B88C2DDD31A6C92E4E62B705E196CFB2D978D6F32F3F"));
  test_xof (&nettle_shake128, /* 171 octets */
             SHEX("E4D1C1897A0A866CE564635B74222F9696BF2C7F640DD78D7E2ACA66E1B61C642BB03EA7536AAE597811E9BF4A7B453EDE31F97B46A5F0EF51A071A2B3918DF16B152519AE3776F9F1EDAB4C2A377C3292E96408359D3613844D5EB393000283D5AD3401A318B12FD1474B8612F2BB50FB6A8B9E023A54D7DDE28C43D6D8854C8D9D1155935C199811DBFC87E9E0072E90EB88681CC7529714F8FB8A2C9D88567ADFB974EE205A9BF7B848"),
             SHEX("908FC63FEE9E11D44E7BCEACBC161269A96993D101265F825B86C107C61CFD11EBA3CFDB0983BC9BA3A81DDBD8517D2E313997ACBE48A47AEF48B3BC1A56EEC2FAC280C96D9ADB557B14334C6C555F374AAF9F9560815EB7278DBC81A78E3DC4144B879119C310A2283862574D055D2B0D8D53219C479E8C2F6DAC3186C1EA36A6517D26E14F2230F00A1B30B44BF987B3A3F47240498734E6858444A82ED1098442688034D38C578E995424E1FC2386FB10E682EBA287F3757B8A4CEF19544CDA032C3592E13E07241AE38F0BFB77B030273B28D09637B4AD359C7798CF6A76980A3EA331597DAB4E18ADD97B8404D4AE4E5EC874D0F31A60636B2BDE4FCFCAB9569205E7261FC728A701F19E3791B538BB1AE8E1435A85F7E8F7B95AA45D275AF770FD9FF856F6666C8598626CB54404FBA35B523F55EC971F70E10E09FAB0A88E39CA1E7180B482985E82EACD81CBD2C8675F8BD41691CFBA39228D4FE2561C2E8BA1DA3124AE3204517F09CB4ABD54698F1010E1836E141C706B9FF0916BACF70010A0F75110E79E95A48FCF4724BC0674C9AFCF12DE59D31C64E532572DA4586BC73A6BEBED22752A05BD1ADD51464180DE96CDB44D11611FA661D0EAC1080DBD25DF51A2C7519489E05E1BE5B05C60070EB8075A9F0B3FDF6C14D6C07076578ED3A16D8E048350CE9B2956ED00AB61B024AEA9E07C87665B7F33E2F5BA"));
  test_xof (&nettle_shake128, /* 172 octets */
             SHEX("B10C59723E3DCADD6D75DF87D0A1580E73133A9B7D00CB95EC19F5547027323BE75158B11F80B6E142C6A78531886D9047B08E551E75E6261E79785366D7024BD7CD9CF322D9BE7D57FB661069F2481C7BB759CD71B4B36CA2BC2DF6D3A328FAEBDB995A9794A8D72155ED551A1F87C80BF6059B43FC764900B18A1C2441F7487743CF84E565F61F8DD2ECE6B6CCC9444049197AAAF53E926FBEE3BFCA8BE588EC77F29D211BE89DE18B15F6"),
             SHEX("080C392EE04A0D4EFE020FA6C20236E7C1290088BF6FFC2CFCF72D1B16A21ED000FC7A064491899BDA1AD4856449D89AC7BC54E3896534837EC4BF950A18A3ECD5FF286D1D0A02E90531298F2430484A667DE0121E0D0B392679AD9168BFB4F0D020E7EC1AEC006BD0E8A551BB7EF0B751D0075F615C02775C983F45212E4F394E7680D9F81254431BFB9E772EF2CA54600D2F1F4DB12D6DB49BB9ADF03BA6C912E0A8BFAB20C0F4D3DA675A908277522C8964ACE05F138D192B7A2EFD8FE663FB4B2486339555AA1C35DA18D8899149339F1AC47555080627C3AEF75DA932BB063FD3DACB6E0019C7DDC5627776250033E6F47EB5930907FCC1F1B627B520AA18B22B12103F3E361801C6EB8B72E18CA9C43EFD1E8A04948A9A7AE1B92E177AC6D3CFFA217E13CCE790EFC636153FE24821A7FB500E61F3098711F60FEFD6EA90F6F63D6822331D8CEACE7C192E3482192B0A408E5C5A36A7C7A5490756FE4022C7D16EB28232A70EC1BD675DEFE9F87A52CB633DC5BD99CA49D2BEE54870FB0919217810AC08312AA2DD16556089FC8706D0293EBFEBBDF11204793901190BFC7AFF9C87C9B4FE9116DD1E1789E08776EFEB71F619827A89287E183DD77C42D791E7CB4ED2E051F2CEC42D2AD83B50EF3C0FB6ADFCBAED484886A4DCFF41844ABBFA46016A0FBFDF2D891B982356B315DC17D9FB62517E3163A93AAF05A12A"));
  test_xof (&nettle_shake128, /* 173 octets */
             SHEX("DB11F609BABA7B0CA634926B1DD539C8CBADA24967D7ADD4D9876F77C2D80C0F4DCEFBD7121548373582705CCA2495BD2A43716FE64ED26D059CFB566B3364BD49EE0717BDD9810DD14D8FAD80DBBDC4CAFB37CC60FB0FE2A80FB4541B8CA9D59DCE457738A9D3D8F641AF8C3FD6DA162DC16FC01AAC527A4A0255B4D231C0BE50F44F0DB0B713AF03D968FE7F0F61ED0824C55C4B5265548FEBD6AAD5C5EEDF63EFE793489C39B8FD29D104CE"),
             SHEX("85F82CAF1A3D6DB8E7C6CD49C697DC314AEAA79309FECAAC21FF46639EE5CDCB719A6379D871138041F2908D4DD605AC31FDE9F89A0136018682FF1CC0E36A3019A260FDA0FE430B082445B7F10F56A50E69A8260954F72B356B710A35A5A2A22FAFAA5B3E4E3F0919A1FD44C24D33EFE5790F9FA707DF0E06CB4F195F1AF7B2578EEC38B8565D34DE9665987F3710D8388176AAB6F3D7CCAB61713B175D5343958419F7C2D65775B7BECADC1AB8DFE486EFB98D336AA17EC4083DEE1253A7E2B32437D75FCBD338F77BB1548C704090C3417DB225534F85C92AE1E9B16F66A624F5309297F2FA3DF12FAC8AF36A1F2A11093D4544E0A67C826D7CDA89730031D4A35C64BFF9CB9512C09FE45F0D628C075737819252B38724BB5A341D3045B4FA4B8BA742B947EB66A9A9121A48B8CFB83B41C90A7A672A4B9BCF2D1A32266C8F13EC46DF4D80384C117A56180398D2268C8009CF586ABBB7452911FD73C9C3449004A11AED41B557A33D7825C42575C75C0641393346C753B1415D4D4AC224C831F185C2B635C4B6B1C388358C232DC629A090FC0D472A4CCEED43FD9657EB33850171F9D6D2DEA433874B47F77DCA16CDCD782EC1A335D2DA3C235914315A243EAC2AB81E06D97DBAA25EC029A62F58A1DC78D586C226046886C23A3D9681D68503893F8A9ED9A5B53A22A1345DB167BE59A6AB8295B41FC464329A6CB739"));
  test_xof (&nettle_shake128, /* 174 octets */
             SHEX("BEBD4F1A84FC8B15E4452A54BD02D69E304B7F32616AADD90537937106AE4E28DE9D8AAB02D19BC3E2FDE1D651559E296453E4DBA94370A14DBBB2D1D4E2022302EE90E208321EFCD8528AD89E46DC839EA9DF618EA8394A6BFF308E7726BAE0C19BCD4BE52DA6258E2EF4E96AA21244429F49EF5CB486D7FF35CAC1BACB7E95711944BCCB2AB34700D42D1EB38B5D536B947348A458EDE3DC6BD6EC547B1B0CAE5B257BE36A7124E1060C170FFA"),
             SHEX("E07CAEB114B9180ED922574BADB071B5962ECAB18474D1A82216B152DD0DAE5CCE14001FBE59E25F55604CC3C26E5D88A25393303FF47C1F14C87EA4D16B0A435A43A3DDD22AABC7ADB4680C8FFA05BAF28F45F95ABF93F8453E267B5E584C7AAB8E2CF2723ECF211B958DD8277B89D3FF2470CDE45077DE7535840B0BFEF8D2AF5FCA3CD91FA6EA83C5F7B58C75DA302A67154474B563AE9B59B1A3BE93BA2BC2A5484BA24CC409896109CD57AB6357376FC0B65A46506D34DCCA170DA71CC52DFF4FA34BC4B979A130EF5663CCC28FBD3E36B6CD103212A96B8710B49EBDACF4A8C41A5860175B694F75F077718C93170A98EE9757237EE90E8692E4AD6C1AF6979D567C104FB3F30EBB5BC52B99A934D527FD44704DE65431D8063DF25E4B31A89629E1F7B7AED0E49C2AB6D3055675C7C63AA4E7686C880F3AF3F6CAF7251BD8B9EBC0A3C46C72B05AC4DB44E85068811EF050F042BE1E7EC841DC8F7EB2A7D9D2C10EB3603F952E307289C1634C7D9E648C36B75F31EA91C7125DEBC5BD3F70F7CCA9330484ABB5A73523013DC507E1FC5689848D6985F73866ED146506DF21BC98D35BC24256C4F911C3D4CE437798431F68E7EA9F608E7BD98D7DF892581B2756106A5C7590F6C33116B7C269D4A3010F121AEB0B10666AFDDEBD18AF9FF571241EE39E4F2E7802CC75DEC1ED6E7F19985774F0733C833CCB9D82CA1A"));
  test_xof (&nettle_shake128, /* 175 octets */
             SHEX("5ACA56A03A13784BDC3289D9364F79E2A85C12276B49B92DB0ADAA4F206D5028F213F678C3510E111F9DC4C1C1F8B6ACB17A6413AA227607C515C62A733817BA5E762CC6748E7E0D6872C984D723C9BB3B117EB8963185300A80BFA65CDE495D70A46C44858605FCCBED086C2B45CEF963D33294DBE9706B13AF22F1B7C4CD5A001CFEC251FBA18E722C6E1C4B1166918B4F6F48A98B64B3C07FC86A6B17A6D0480AB79D4E6415B520F1C484D675B1"),
             SHEX("1BA3E5682B5210C2503367CF92FC02BA6FEFA6C43CF725D9E0A5A796F949BF605C9401D6235C09265C000AFCCA9FACF32B1C6CDE831D051DBC604B72B90689CFF3E9179BB0A8D151C2FCA7DB9147A8A95AE8831C475B16B2478FD794087D0D979FA66DA5240AC77BE4CECB3DBDE50FE372C5045BE23FDE7CC0B6B86A0007D47DE80D59A2DBC7883BBB2F76B74E7EB53E8C929AB73819BB09CD61010689DACEF17BEEADA0311F2F6594922EE835DD09A61B41AAF5792C63CEC79C40EC126A68DEB00EECA6AAEAD43072AC65E7101B4C7A3FB42101BFA922E43329E736D1DBC90EF45032C82E26CD102176CF8490B554AC9C6FFFE81D281CBF29BF5056C06CE2CDA918BB3C9F8F49A045DD2CC4D680BA8A22562F1DC53838665720EC60E4E3BCB76014B619EE24FF41B2860B3BA4438527D975A1302E8FF1923211F36673B0BE8B58BF0C7FEFB0CD50A28A386CE6DAD7C30B8F6A3A437E864143143E638FC6474A2A8A6640862FF4C4491FDD52ECE0834128137011D46EE7B0CF611B237AD515BCC11FB7B0A4650FD6E60BA8B6BC601DE5CF75A66C3F16AEAD1EF19C47420A9840CA4B1352AF4C5C217350840AD035621EABF2EADA2A517250D3A6AD834CAF2A472B2D5DCA3829C372EDC64703226EBBCCAFD316E4EA3A9BC9AA8DD42A9715B29702BBA9AE232C7806E5C0C02AEA68602F4B580396B27B94385A3DCEBECE345828"));
  test_xof (&nettle_shake128, /* 176 octets */
             SHEX("A5AAD0E4646A32C85CFCAC73F02FC5300F1982FABB2F2179E28303E447854094CDFC854310E5C0F60993CEFF54D84D6B46323D930ADB07C17599B35B505F09E784BCA5985E0172257797FB53649E2E9723EFD16865C31B5C3D5113B58BB0BFC8920FABDDA086D7537E66D709D050BD14D0C960873F156FAD5B3D3840CDFCDC9BE6AF519DB262A27F40896AB25CC39F96984D650611C0D5A3080D5B3A1BF186ABD42956588B3B58CD948970D298776060"),
             SHEX("7F8D0AB3060B5A6ACF3522A83AC1A432DD9EBDD62BEE5CA14CC54FEC0D3AB6D1EACC33603F214C35E8C256BBE7F0A1C110A65A6E7F5EBD6733350DCBBA851271A0EA684A0AFA032EF530511B5FD923399DF93567D9BB1EEC74D6FA4AC5451D81FEEFCE00A7074A1F32498AE68380E2D9800AD3DA75E0BFACDF66B9F34CE4C60615AB46A8310F85C2A7F9A737E8E19E0BCB540FA9F621378C44E2DBBF57C059D52F79FDA1DB8A3D5B44C558C8CC70901219AE76CF5F6DB962ADBB6515B25B5FC81CDCA21B7A51911C5F5C52E8F2A863E9F09E927D8F66B063AFC73CFA8592B710063544143A8B5D06CC78A604E9B72771DE91F48EE55C9FDD831BF91171532AE5EDCACA1BDE2B96362D4E30115E12ADA76ACA05DD128FCB524CA8831CF2459A53D97EBC3AC5A25517F254A14316A402A509FF5868CD290C88F43229BE54A3443BC547A093687B3C3D97A4D1A09004CBAA6939E86CD75F5135D9FBFDDE568E651806AF401095C3602BA958FB7A0F4276791B28A6BB75A6944EEC6EB8CB9A7A9E288916A15A496DF80777FEEA0F42695AC62E29F137E355A13B630F30E5D4E4435C3180EE86173F59BC19B79065CF3F33F8BBEFEFDE35A8AB09C13DD9D326E98C9730BD033CB2DAE20BF076EE9CE764D48BF8A0F7BB9181E60889293866E1C601AB38122220A2CCAAC996FB1770CFDBC4E628BBA2B3122E0CA351FADA84F8183FF4"));
  test_xof (&nettle_shake128, /* 177 octets */
             SHEX("06CBBE67E94A978203EAD6C057A1A5B098478B4B4CBEF5A97E93C8E42F5572713575FC2A884531D7622F8F879387A859A80F10EF02708CD8F7413AB385AFC357678B9578C0EBF641EF076A1A30F1F75379E9DCB2A885BDD295905EE80C0168A62A9597D10CF12DD2D8CEE46645C7E5A141F6E0E23AA482ABE5661C16E69EF1E28371E2E236C359BA4E92C25626A7B7FF13F6EA4AE906E1CFE163E91719B1F750A96CBDE5FBC953D9E576CD216AFC90323A"),
             SHEX("C625EC73E6D951EB899E865B5012ED72229958F2850A79CB8221AD2248A8D670A8072519AE6DFF8CFBF7A20863FD01183D55093C80CCD14106426579ED2FD99578532F9B4D8E11BE19A0F0A8336FCBF1BB1D6D91247DC87B563C1399F50144F563F8583DBC208165C106567EC51FFD321D9F2ABA7AD2E7F2C1950D04E9027D7364ED831C2155E17C0B083B2EE2C4E720A2638CBBB971CCCBA1DAC6860917D28D341A49CDE2B8727179C8B596C045382E6E7C23A35D47880B4A47D3DC79687919194397C3EC00D9AA232E666BA24A1A39CBE847B2E864478AF2F59F73B2ABF2A98481430943980BA197DB6E67A87F3782B7BD99A6E27F82F133DBC6D722B5CDE09277F2407671FF44C1117E1265F5EC9FABA0A946B67DD268C3B97BE198BD2B5185EB07378FBDACE2B4B97C8DE2C05176CD186B0AD3F7D54DA7701654EB1D1FA200191CEA96D0F02C86CAA277FEC59E1C93097FB5D0D43600E3403C53798A33EF5CCAB3D5405E5DB570A26C3A0BDB304328FCD36E327B5D957913AACDC4D938BD00577963D5FA343109F4719D8192B12272DDFA338612AAACDBB4CB91E129E4401E16A673D6FEB16294345FCD0E9BAC8BDC30EBECC3BC6DCFDF25ADBCB42777A6FF4C05B5981571AF8A33D9E7D34534F8F89F555A1A55761FBE2DD66B93330820D3EB032D7A56DB79AA7CC372340B953097453509F240BF9AC6DCD0DF08D21E10"));
  test_xof (&nettle_shake128, /* 178 octets */
             SHEX("F1C528CF7739874707D4D8AD5B98F7C77169DE0B57188DF233B2DC8A5B31EDA5DB4291DD9F68E6BAD37B8D7F6C9C0044B3BF74BBC3D7D1798E138709B0D75E7C593D3CCCDC1B20C7174B4E692ADD820ACE262D45CCFAE2077E878796347168060A162ECCA8C38C1A88350BD63BB539134F700FD4ADDD5959E255337DAA06BC86358FABCBEFDFB5BC889783D843C08AADC6C4F6C36F65F156E851C9A0F917E4A367B5AD93D874812A1DE6A7B93CD53AD97232"),
             SHEX("5FCBA2303DA2ED78B7A3B3DBD3D7D57BBF37B25B75BF8330D39576EE26D830002F5FD36102BD09A6025532371A6246417F94A527220194F287BAB34A936836AE84D87C251674F1BD361FAF5ED3C023A96AC1E089050C7975D4FB6DDA11305D73F2D9BF03FCDB4ADC31C5356E231580AF9E23B8AC79B28409DC4EDE2CDC5D98C67508ED4D688377583E06FAE8F195B10C60FA61BA10A7A57A8A52029371CA12C9FBA5EDD2378FE78E5CC119695FA5F6ECB4F7C27B0552B1231D4A1BA31196CFC7D30E92603CF2BF14E2D4CF8860A3B4B543E48401D9FDC6B8D0E1AAA559E91ED8FE3D6DDE2F8A16A59AACA1FEFD304D0C42B78441D4AFE148B1ABF86C92706C5C1528D54566EBF0F4F7F6BA0ADAF6ABAF5BF8DEA607B6C86C789E7EA3229031BB69E60BA745F561208101DBE4774AAE3CCD7DA3E6C4625B0744B33E6697C31650FF94095C5CFB87024FAC512D81CD395D8A30FC562E8E817D5DE70E991E4B3ABF091591769EA3B42197A4F4DEC475F3CFFB47700E832FB072B4783DCF44014642D9930F09A2E3C4984A20BED71E4D2ED1C5FE750B0F803D4891A128731F48DF37D7E30EA34F7ED816C51F3DED629C7F430CB3F1ED74610E4D6535DE05EB6E7520B1CF6653DA236DC9E4F8CCA8BEB696AF2A3C15C42E6F87C2EE1E9013E7FE31E891F6F9658F2E270A423154824CCD6BE7498A6FFB8EA6C4E5B8A6CF1C33314C03"));
  test_xof (&nettle_shake128, /* 179 octets */
             SHEX("9D9F3A7ECD51B41F6572FD0D0881E30390DFB780991DAE7DB3B47619134718E6F987810E542619DFAA7B505C76B7350C6432D8BF1CFEBDF1069B90A35F0D04CBDF130B0DFC7875F4A4E62CDB8E525AADD7CE842520A482AC18F09442D78305FE85A74E39E760A4837482ED2F437DD13B2EC1042AFCF9DECDC3E877E50FF4106AD10A525230D11920324A81094DA31DEAB6476AA42F20C84843CFC1C58545EE80352BDD3740DD6A16792AE2D86F11641BB717C2"),
             SHEX("07F4933F2A7C2988B9D753240C1EBBEA38E95AA1029508296BB49BBDF3BC164839C76D905596EA153C4746B4FEA069D2A784D9CACB598A24BAAD6D215F83F72D4B52604EEC0066344FD062B1835570028C1FD61D5B4785FF5B904F5DB03C4EFB61B424226578392B9E6259FD86DC979526C560057B9395E32116371B3737825A9579B92E1AB5FF9006AFDE8A618687205438D99B7E1BA038922B140BE0DAFB7423092F362F537CAC8272AA5DF14B2DFC2B73F5F4104BA1FE603E00AF8E47898C0E955D57DED792C3C93518CD84FAB72A2022F189773FAE574D5342EEBB23F7D7497F301023C1143CB3481ECB0EE732D1477E6BF872EA0D0DC0623A5C0AE526D8885DB1D3C1CA0A513D2E4D53E13BFD82129DBAA7A2BF6004D2091E627928203D05B11D9F6335DFC73CB9A7F5C3CF43990E8DF0269FA9BBB1FCE7646C6BA85BEFD3F9ECE88A93CBECC3F8F68CA4D0AE8197CA479EFE0327463AB5B78685BA4000482558CEC55D8B934726B820EA4FA9471AEF2962A7E1CB4B2891C0B545547A146CA28AD3E9CC6ABF69D7DFB87B298CA252387888E20CCABC9E9ED484262A3B6A1F857E6900330ABE74454A62E2DEDFEC3AFF7BE28E4351C4AC0738032B81C0FF12AFB4CB1B94F7119A6F2AB4157CB7C2FAB41879943A075233CEF8A64523F70E3C6F66C5EE3D5FDB99222B2A3EB09FFABCAC867E0B2D06955CB80E7EAE176788"));
  test_xof (&nettle_shake128, /* 180 octets */
             SHEX("5179888724819FBAD3AFA927D3577796660E6A81C52D98E9303261D5A4A83232F6F758934D50AA83FF9E20A5926DFEBAAC49529D006EB923C5AE5048ED544EC471ED7191EDF46363383824F915769B3E688094C682B02151E5EE01E510B431C8865AFF8B6B6F2F59CB6D129DA79E97C6D2B8FA6C6DA3F603199D2D1BCAB547682A81CD6CF65F6551121391D78BCC23B5BD0E922EC6D8BF97C952E84DD28AEF909ABA31EDB903B28FBFC33B7703CD996215A11238"),
             SHEX("7E3FBAD2449B2F16426651EA24F642FB8CBB40B1969FBF10C5A6C3E7B1AD2D413C1F8A78CCD5F4AB0BA0A04529EEB3F72B656EA1030CD94DA5CE6B6D7C44728C0CEFEBD7FA597073F8BEFD6BAA5325918C1BAAC3B8DE7D1E4B723C0518936CFDE3F039941FFEC3979AED13C16E5F24A8C01F4D87E07F797E2DB4D5CE17E4AC40565D32896E3B6DBDB50FC2BBC4D13C2E375F171D8EC62C423A126B45D3C13B6E3EF9CFB9B8FD25BFA74A1793EBBE1168705B7A26EF12C2424E9689D993BA8C02790E83CE166DDB7C838DE164744BAB16743231569645B144A34BDE3A40712F01DC02E476140D07BA3DC4F51DA894727443892E7B12555932AAA5755FD9887A9EAC8C949E71EE35414150B63B00247CADCDDA9C2FB245748DC519C39EC8877C860729D90BBEFB89C5D914F99AF63725E91401ECAE6F69C67CAE0C94BACF3D6B13C273019CC3B5D416D6A24D314EF01C6348F2E539578F9368C234EFB559352E5ACEDD53C14A35F680428BBA25A252F056C4E452A921E4F83D648838ECB076F4764256912D41D504044BA68F2734D7236FCCD924D22735F6CFADA2349421BFD51341EABA98DA57B3F1A1B9B7B4588CF3A0D1A7E116700FB90C3DE339DFB9EA3662364807B91F7D2105D59034A6F5816070B5C4B24ADAE525C1BF2D267EA4B03079405A436DA4264CC50B2B30A0881E4D22A1904DC7311CA397AF7399734A0CDB79"));
  test_xof (&nettle_shake128, /* 181 octets */
             SHEX("576EF3520D30B7A4899B8C0D5E359E45C5189ADD100E43BE429A02FB3DE5FF4F8FD0E79D9663ACCA72CD29C94582B19292A557C5B1315297D168FBB54E9E2ECD13809C2B5FCE998EDC6570545E1499DBE7FB74D47CD7F35823B212B05BF3F5A79CAA34224FDD670D335FCB106F5D92C3946F44D3AFCBAE2E41AC554D8E6759F332B76BE89A0324AA12C5482D1EA3EE89DED4936F3E3C080436F539FA137E74C6D3389BDF5A45074C47BC7B20B0948407A66D855E2F"),
             SHEX("B6869F1189F52EAFCC0353F7AFDB8076C5CE063E2DF404F3D1314A4DE573541927E9F4C67A554BC82E97C3A2F6C679C812C9DA6542681461BAF009F3ED10D7D2C571288A41EFF6FC73DFF8AA3CB40288C84C9ED5CFA70F59395E09CF3CDD58B4D2A65187CF38828B981EB5BA8F7E591070F97B3A567188AD4C455FEC91ECC2C9829E00639BF6CE1DDAEE45CE19D9A183F0E93630886F511125A69218D86C5008760E8EE0CD299062CD1A06D0D2A5E07A0A30DA7DE30531136B4BAE8546290469FA1B18557E3B1316ED8FF1BC94131E9797EFC0A2D997F8DA22CE6DE0358128AFAFF496DB6466210A8682F9D286AB66304E79CE92174ABCDD0C4DB508F03C2BEE0C87F6C39D0FA2EF7F3DBD4201B4A46F418CA1DE312A2FC9F82E6008BF06042050CA302C9A386AAB9A3A8FB558EABEA6771C6FC301F7DEA5C13DA704E4F41DFC2EA508F1EFAF743FD4A3EBD9370CC3C872A41C4BD40EDDF311C2F5B7C208BBDC7BC4A93804D10B32769A333E8BC2507143912CE4D918D0DCB4D6462F0790F10BFDDCD7FD711E14F411A6CE58D50E70E1EA23645E97B6C154E3BBA3AB35411537BC8830ECBF953BB8214DF95E6FB5AABBCE08F8D017958848916FE04B3D2D5F69DB9234220DA3388E3385CCB90194959027093A93F7048F35E3A5B0AEF90A792820B242C34700FBE7F7E00C00D6ABDFAE26DDC95CA47A72133F97924D67291C6F"));
  test_xof (&nettle_shake128, /* 182 octets */
             SHEX("0DF2152FA4F4357C8741529DD77E783925D3D76E95BAFA2B542A2C33F3D1D117D159CF473F82310356FEE4C90A9E505E70F8F24859656368BA09381FA245EB6C3D763F3093F0C89B972E66B53D59406D9F01AEA07F8B3B615CAC4EE4D05F542E7D0DAB45D67CCCCD3A606CCBEB31EA1FA7005BA07176E60DAB7D78F6810EF086F42F08E595F0EC217372B98970CC6321576D92CE38F7C397A403BADA1548D205C343AC09DECA86325373C3B76D9F32028FEA8EB32515"),
             SHEX("EEDF8414ACBA3931CCB5C41B045E708926CC68C513372B2A5C9F8BEB76C060461CDBDE486F81952169B465061541E397192CAB92B64CD5AE0072811017140D53861F56A96A66211FB30BD1E911D31FDD29CE66882529874724A64216C13AEE8110B963ECC99CF46D2C1384CB39B27F2EB85B517893AC45A126F57E93DEDEC78CD2708E2C718D783D847E2FBC8B3FDC67260B33852576910D2E43923BCB8882D318AC5421B99BE5E532101833585E744B65EEA3104337FFAFFF4266B6ADD5F081E42A46C5365CC2F03DF43C3AA3D0F9A9C477D65E5EBE634C806246544B97EAC179C932E37948BB5E5C84DB32450F3B9FA54C89FD03FE58861B8E8A6C8C3D7205B7A65A02B90E0412C493ACBC3385E1A197DF785B02EF42B0BEBBD5431EE59DF31520B88E30BD376BC4E720D71F1C0C9AE4C2FBE126DADE6F667E2CE825CA1118F5B82F2A86FD5BD80621FA6FEA14768EFD08D75E4C416FD1245B0C07568DF42244DBF76614E0F3010684210E49D0868C3ABF1C89084510662BDCCF932176DAF9393B00A4FDF911EC8BA14E81B0648E5126B7B6CC1DFC902A53D0BA7204C256DE67FBB525F6C2D4C4DB08484A14431B350B76B0DDEE9924A8E0DDC4E6B9DFA90FD9BE0BE70BFFBF6F9ABFE0EC489822B94DFEDA911FAA7626271264100E2A40A2AB932CBE92BF2DBF57D273B0B056A6DDF29BA36A7F8138390865AD9ECC76BBC6"));
  test_xof (&nettle_shake128, /* 183 octets */
             SHEX("3E15350D87D6EBB5C8AD99D42515CFE17980933C7A8F6B8BBBF0A63728CEFAAD2052623C0BD5931839112A48633FB3C2004E0749C87A41B26A8B48945539D1FF41A4B269462FD199BFECD45374756F55A9116E92093AC99451AEFB2AF9FD32D6D7F5FBC7F7A540D5097C096EBC3B3A721541DE073A1CC02F7FB0FB1B9327FB0B1218CA49C9487AB5396622A13AE546C97ABDEF6B56380DDA7012A8384091B6656D0AB272D363CEA78163FF765CDD13AB1738B940D16CAE"),
             SHEX("61B6E091F3A8BEF3886484730629A32553B1EF7EFA8689AE6C01062F505ECF48E8460B66448D355046FA728AF5778FEB8B827BA30211E5B1BB82F6BEB7F2ED7450E9EE0A19A47AF63BB687A0A4E611B3762B97BFDC6E212EDE1F874C33636AB4EA613BA991E8B9558F24D1C02C5731B155DC1EDD6E984C5CF51BC59BE5E084AE89C87AC2AF5792C6DE0B76448AEF8C5F248CA6FD1525770C8B4434A76552834CF1481843A9E3A2051108D3DB6B8265F736395848DA2C5755B504F4B958755BA1820DEEE9EAD314C9515B996AB96C8F349BC12618FA4E8A026EFC5508707F642F5CAFE1EF135764515F4CE2DE357D93A0F3BDFE8E3A7E01140212A735FAECB49659039AA05F0C5EA0690E304C155A64804954E24039D001A1F7C2F1691E81120FD275A6FB292F3FC1BB16763887E3BF327FD36FF14D8B89D02990AE9D4B2B4F5492934079D17945613AA03771605B259446664FADE2292036ECA5894C4EC3D6673FB95617770AF54CA581BDC6B52097938A98BEDE6CA627A9322D4FF7390DF9A3E46FC5C0663A88A16A7D4EE708D92EB5EA6DE0527FA5BC3454455B678791E07586123CEC8864396B4B972BA29C935E6D3BA2F7B2FD315131C0F008D047E6CC234535D947F068548287140D4303365990E62B4845A3E7CD90667039C205BD1B836C4240B2120B54DF12C462D2B10FFE2D61B456F090EFA26C53E1F51D2292D31F"));
  test_xof (&nettle_shake128, /* 184 octets */
             SHEX("C38D6B0B757CB552BE40940ECE0009EF3B0B59307C1451686F1A22702922800D58BCE7A636C1727EE547C01B214779E898FC0E560F8AE7F61BEF4D75EAA696B921FD6B735D171535E9EDD267C192B99880C87997711002009095D8A7A437E258104A41A505E5EF71E5613DDD2008195F0C574E6BA3FE40099CFA116E5F1A2FA8A6DA04BADCB4E2D5D0DE31FDC4800891C45781A0AAC7C907B56D631FCA5CE8B2CDE620D11D1777ED9FA603541DE794DDC5758FCD5FAD78C0"),
             SHEX("E4E5622998509E275AED8D103E2581877B09D570EFE558CF03626FDF3A2C04044C1531D3F92A025CA693FCAB59D90ED2A2DD33F4768F56B972B2E87D34E8F293862C22602DE928714C34FFE22C118C474467005C250F77971D59BD93581E5EDC4F2BE3C61F5414DA40DDFD3053595E616AFEB18D3CCE87691AF4C228DA5595E2C4498FDFDE8360DBCF014B368A88C5ED7A71EEEB003E8839A371B8D86F7B58DA353EDFB2891BF5A5CCBEAC1E5F2EDEC1442FD79B6A31B85E4702DF68ED3A0504E26489B15C0C1A52B443581BD671FD207302B4BADBA9F41B974BA6179EA1671E4660CC8F5A35F71A8B9BAAEDC457FF13263B062DB6059FB4275D2831218B04B2E24F1E3B3DC4CEE9261A1919343D5274D64CE11E0C27866ECA3C91BE98762834142C730F4C3354F0866D9F0CBDE0E54BF747E0D8E50C8654A1C87FDF7EB6D13C06A8D0A02FC6EA72C77D1D957B960D6726EB5F1D4B3C056B2B67D9C23DD393A25D03129FB4D894DCEE892D4D7CBBEAF404FDDEF8707C5850B319E04DF5D64BB9F256A6B2A7BFABE6487B16DBA0CF415F8DE98AC8D63498DF68DD9E209B23297C79D108BEE3756989D751D91FD0954A7F5B166109FA11C7AF4BB18B695FF6110659E37683F92BA03C24DB9276D88E6FA587543E785A4D49F37240CBD826162E0C2AB12DEEB71552FB1AD45E7E5780F832D831D2AAC556A3A0005C37C148C41AE1"));
  test_xof (&nettle_shake128, /* 185 octets */
             SHEX("8D2DE3F0B37A6385C90739805B170057F091CD0C7A0BC951540F26A5A75B3E694631BB64C7635EED316F51318E9D8DE13C70A2ABA04A14836855F35E480528B776D0A1E8A23B547C8B8D6A0D09B241D3BE9377160CCA4E6793D00A515DC2992CB7FC741DACA171431DA99CCE6F7789F129E2AC5CF65B40D703035CD2185BB936C82002DAF8CBC27A7A9E554B06196630446A6F0A14BA155ED26D95BD627B7205C072D02B60DB0FD7E49EA058C2E0BA202DAFF0DE91E845CF79"),
             SHEX("F3ADDAA4382A61CF38C2963346547922642B064CE36B4D424F41ACB02AFE7AEBBCA83DF589208FC828EF58300E96A4D20592BFD58640CDA4C970E1AD58BA410ACCC75CF6D2566B6A10D7FDF02D169E6BBE8953E5812CEBF84E6E79B952680B4CB59373D8A6741C3483389E611A290D1CA14178556D0CFE750C787C4DB1558A1F39A2F39F9021F1E420F57CA91CD1EFD2D61D0D8CD34AD1FF7E607096B9787A51CB7684D8040FC66710C63D85A510DDFDD3F60056051406E7F217F12867312197690EE8EA33E6935306B76AD816043FCD52255976F9691C31AFB361612886758BE8AE9112412456B4395017AE96A3D118E351EE2D17FBA96B3AF726AB1B2EFF01FC3B8B4DCE461A452F0E6885EE4937B34BBDD357A83A68A95D12BF9CDE1C5ABAD70CE7A30B76BDB601B8EB533D57E4DAB508956440FFA0EA3EF2773A446E2CB11F08226385A71D93CA67CF6CCC8EB8E48444D508EA035AB59AFF79E076FFF850CD497798F49BC633EF41B0A4239681DA28A3BCFB2FFA3A992909E0A0148781A1BC03FE3FDF9B7B9971213F9BA331E5B1235314B3F173D48BC3249E946627F84070AC2D8B57E184B0E8EAF0321B38857383BC1C27CF9390E79A2E09A3B5552A7AA9CC55AAB534BF7D7571D3ED97C258EFC9E2D3C1D34DC84EAAF55D1725AB388C0A5FA4DF270630E468980FEF3A79A67CBB2289BB03CD239C14F15A50403787A3"));
  test_xof (&nettle_shake128, /* 186 octets */
             SHEX("C464BBDAD275C50DCD983B65AD1019B9FF85A1E71C807F3204BB2C921DC31FBCD8C5FC45868AE9EF85B6C9B83BBA2A5A822201ED68586EC5EC27FB2857A5D1A2D09D09115F22DCC39FE61F5E1BA0FF6E8B4ACB4C6DA748BE7F3F0839739394FF7FA8E39F7F7E84A33C3866875C01BCB1263C9405D91908E9E0B50E7459FABB63D8C6BBB73D8E3483C099B55BC30FF092FF68B6ADEDFD477D63570C9F5515847F36E24BA0B705557130CEC57EBAD1D0B31A378E91894EE26E3A04"),
             SHEX("EB6CCAAD9EF05B657B6835655748051E33250C4FA600B3431E96053B0AD6ADEF7D8E5CCEDA256D96CBCF713E1998B19F8033BF032C9B2A56309B31755DB88A5CB11669A49AF4999C551ECE42E69DBC4D53B0AB2705295A649364E8AE495C793B73AC4F5AAA01A7A66F4542A4A5B29AA266158D3379ABB4C26596CC3E7195EA9112507ABA0C1BBD8EFA5785254681E11ACBF65A977199D0D489CD4357CDF61F0DA3B640958FF3EC7CAB5ED0C3E725EC64D18CBC8BDB097967FCE3AF9A717F591DD49AFC38FBB4437915D7B161E6800C3D8C0EAAB61C5ED05D8711B00E75892AAC5169E42B95AC207A27981684A7A26B9C7E00F7AB9A2DD84437940A7169BC998A56E0A31693345AC53F45B9A7D0532ACC80E093624C5341312E7F2E7FFF5B0712E2B119F4C6F5CA1E55899F394C850EB038BEF7EB1DF1303B7C97292732F96F4F6B8CDFBAD6C8D6FB5721A5441B4897FCFCB2959026DEACF17A99A27EBDC108EB1D77518C342B8E9892CE44241F94B88803D7EF72E26E8209BC251FAF58B8E0E2CDACA5B4509174B07C98AEDBB2C3871FE972A4C31DFAFCA4109985904DAACAB1C124E659562B77287B6D9DB73A38A26AD0E746B71FA086F6AA4E39207093A04CDBC69993C2205CFEDC4D321694D3FAB832CD729EC2501EF2D386EB170A011BAAE3E83A7EFD110FD36C9A7FBDDB835FA033891FF21C4B634368FBE8AD991649A0"));
  test_xof (&nettle_shake128, /* 187 octets */
             SHEX("8B8D68BB8A75732FE272815A68A1C9C5AA31B41DEDC8493E76525D1D013D33CEBD9E21A5BB95DB2616976A8C07FCF411F5F6BC6F7E0B57ACA78CC2790A6F9B898858AC9C79B165FF24E66677531E39F572BE5D81EB3264524181115F32780257BFB9AEEC6AF12AF28E587CAC068A1A2953B59AD680F4C245B2E3EC36F59940D37E1D3DB38E13EDB29B5C0F404F6FF87F80FC8BE7A225FF22FBB9C8B6B1D7330C57840D24BC75B06B80D30DAD6806544D510AF6C4785E823AC3E0B8"),
             SHEX("28721A5CA8A0B2F0EA39D1E51F18FF57F838B0F7F95106E13950D717E3BE00F9CDE1FB3EAA37E3B0DFB080B1A08EB6D07B4C36FBE73C248FB86E993A7A44A42501593B8721082AF6836EFB165084AB5DD13C26413AA4303D19183CBF09526E9D6F590990C7A61103F82C3E3E58A673F8EA1D00C85EFD867D4D7F9A0E295636933238C39A721ED301D4168B2346B115998279BA080D3D63A7F8A5BEE1DB83F415F9FA26CB0C6511A3AB2CA7CEE4B30188570FEDCA952B91A299DB985E0EA5D884646DE594F5A3ACD3CF975CB8987018EF6D5CB6C7044D946111D250B0EE5F40329D6EC2423D085DC8A8F8BEEA38EE4F3E4D7E206B6D1B988AABE52B074673C1D74C6DB329CC978B88B0C88A41BB91D541BA4D3ABF38F892E90639ECB37C79172352FA2273ADDF88BD6CD6A76B305E001F78E9D6C29C46477DE5ED69FD0D398C8008C839C87FD95C51D35AF3AC874F9E33E60813A984FFEE299547BDF305756A9F7560411B7BB475870D8ACE7671C093E210D3BDB10D31F48E0A78D1D9E3E44199CC49B7D2A9E2CA0EA8EBED0D265FBE461A12883EE515FD738BAC8299309C86B77ADBEA857C4C92ED8E0F380733B47853CE530091A6D700B4D1EBC9830C2443C7D82C3E0E446CC72D4FE75A5C3AB4E38971EA41FE993BF270C477FE90E2E207DF59D011E23777FBA3B454138B31F1E055818CE61649F0D4D06765247A9B8FEB8D"));
  test_xof (&nettle_shake128, /* 188 octets */
             SHEX("6B018710446F368E7421F1BC0CCF562D9C1843846BC8D98D1C9BF7D9D6FCB48BFC3BF83B36D44C4FA93430AF75CD190BDE36A7F92F867F58A803900DF8018150384D85D82132F123006AC2AEBA58E02A037FE6AFBD65ECA7C44977DD3DC74F48B6E7A1BFD5CC4DCF24E4D52E92BD4455848E4928B0EAC8B7476FE3CC03E862AA4DFF4470DBFED6DE48E410F25096487ECFC32A27277F3F5023B2725ADE461B1355889554A8836C9CF53BD767F5737D55184EEA1AB3F53EDD0976C485"),
             SHEX("1E98D0535E8D9201A8E74F5D60FE3151AB5C246993D2C39AE2C906720F898DA70500D14E94ADD0842DDD949FF614C1DF73D57879B3649CCFA780D22131DCB51E14CAF79948D497D0A4B8C3BE023551C68D88DC3E36AB8D73820FD880F3964964FABF60BF9009927F25D3EBCEE347C54B3E7B9A957F0CD10AD15D0A9A2DAACE6715633A9400DA882337E8505A431251BFA1B31096004E6E0F40F00F6C389A9FD5DDBBF7E7F26F1259D25F2C176B4B71F0C00DEB9185FF0E4508FDD0E057AF4F4D2DF3A15DFA5416ABCAF086D0118BE1AA70C7A14DA41B9612814C39B6E7DBEB9A415FD4E7AC8F3E046F0BE25BC0D05A183A4FFFC498100C21D013D712D7FB48667454A4A54BC6CE355107267E245675002AC83E00A26A76EE6081F8C621620C37A03FBFBDC6F7BD88A5D996BEA811F9F753A2486262E6452F896AB8F40B2BD97F95038DB68FED4A52194B9E85815E67DACA1EEE6B0788343999453E8758AFD73DC3E1B9088561305E929A0128C6D229743567C7885EAABFD9BEA8644FE8DB4D19187FF46388DB5C1583DFFE70DA6349ECA26E839DB999F75C1C72455643A902466C24717069EFADED38C138D6A499FCA999AB653F5B5DEF624EE990F46177FEC751F55833E721DF57F6B813A55C2D0C463D25E96A53DB14E6CB4340F6059C924CB7C4EDF94812959251282B5F46ADD9A97DC8951B6AE72F7FAF6EFE2BD11D9B34"));
  test_xof (&nettle_shake128, /* 189 octets */
             SHEX("C9534A24714BD4BE37C88A3DA1082EDA7CABD154C309D7BD670DCCD95AA535594463058A29F79031D6ECAA9F675D1211E9359BE82669A79C855EA8D89DD38C2C761DDD0EC0CE9E97597432E9A1BEAE062CDD71EDFDFD464119BE9E69D18A7A7FD7CE0E2106F0C8B0ABF4715E2CA48EF9F454DC203C96656653B727083513F8EFB86E49C513BB758B3B052FE21F1C05BB33C37129D6CC81F1AEF6ADC45B0E8827A830FE545CF57D0955802C117D23CCB55EA28F95C0D8C2F9C5A242B33F"),
             SHEX("518802FDC9FA52A2A70FDBF2AF1B43EDE346E59B2709319CB57FE6480C46C7513868686CD374AA43D656C6BA98A2A879A1A3D973C46A10E95BD0FE28029E5BC8AF03029D744F4B2D9BC9D83CE895618B9E21E6C4C2D295A285F251C26D22CB1662A2AA4E8609C8503CA9C908EF853BA212A87411059118596C88795C972F8FF55607BF82B8C128AB5D041E86D5784B35EDEE160FFCD06314510DC4AF9B9FC51999D70C9A10449C5B62B584614742032BE710E50B42ACA942C7C73776EAB51B0FEF37F5447E8CE2198266E7FD1E378D5CB2F296AFB77600B64677FA13BBB19DA426288F423EC6617BD19A77833F01CA5B684BDC5BAE939A79B6203B22F4FB27F4BE92EBF337F9153CD3D78DA095C0E76C9877A1A80D2AE4DF7CE56A8F876F32E1134E0C51E52798389C35646D3165DC164C5F77ED519345D21D055A332F87CCBA7D3EF19BE950977710E9706AD07E30928639148B1250026FF23B04A06E62C2799072A319E8346EE80F9A0A2389C4B80B9A397C45BC1F0E92620455A1A04B6BED80B1BB1B78370992C5E41FB56E984421FB1E8465132A33D6D227C85970DF92D7159E6F59EB766C3511B84C85865DE1FB3FB3405D2102F1CA0AA4C1EB6A0EC90F6B89D783423D848BD70C8220C7F024A43888B9947D1E9E52A6204CA17B04765F503A887EA929AC4866890110FED0713A0479C6FB619C08BF2CB24E026EBC6151"));
  test_xof (&nettle_shake128, /* 190 octets */
             SHEX("07906C87297B867ABF4576E9F3CC7F82F22B154AFCBF293B9319F1B0584DA6A40C27B32E0B1B7F412C4F1B82480E70A9235B12EC27090A5A33175A2BB28D8ADC475CEFE33F7803F8CE27967217381F02E67A3B4F84A71F1C5228E0C2AD971373F6F672624FCEA8D1A9F85170FAD30FA0BBD25035C3B41A6175D467998BD1215F6F3866F53847F9CF68EF3E2FBB54BC994DE2302B829C5EEA68EC441FCBAFD7D16AE4FE9FFF98BF00E5BC2AD54DD91FF9FDA4DD77B6C754A91955D1FBAAD0"),
             SHEX("DDBB83438554C66AA9D9569AF53EB13DA65440E0B1AFD88B80DF59E767E069AE4C70741F68ADC2A674D66E1829AB7DF21BAD9665E5ECD655F756A78EB9F78D838461A03C1A6D1836B27DDA6D269ED65B7E352030401ACC8266F41FBA4E234D225EE32BFF12406F7F620CB3484D6757EF3AB69ECD675A9250200735826B630B72E64F21E3D84B135A35E1F4AE48AB2E80424C10BE6BB7AA78F8DF9C91537FABB91DB4D58428F6DE62364292141E9A7FFED93F16D1D36C3CEB4960E7CB8FCD8B91EFF75B027159586DC304051556E1401A6C555A804616AB2757D699BFB1E37426983C39F34DF6C1FA6C084FDC9C633B6AA331C5E5028BD185A4EBF3EB0CA4CC397D481F404440F4AB2AB7DFE0FFC935D91D417D94D9858F89D28B4900E634A5CAAADE9F149C6CA1E011EC684CFA3B3042F9A320501522B6211DA5F5555C87102DF37F17E4AD3BA55A37381D204FBA5711ABED49EE51F3985ECEA7E239FBA0E8A060A4990590795EDD1A936C185BF37AF9B95FA3E6294A780DC9FFFAD7D9319002D18794FF0DA59CC6AD9E3AD9D74BDBCA343EF694CFD333F87278E59CC445FBE0E622857F3745FFD8C90A161F7F49EA875BB1CB234C63AA0C55E530A72B19CB770FE91872A91420FD1CAB520EE922C2B6FB59CB8E516C303F4C74852769EF1787FBD7429D33B2FC0ADC18B23034736FAC59F926E88DF27D811591F03E7009193D"));
  test_xof (&nettle_shake128, /* 191 octets */
             SHEX("588E94B9054ABC2189DF69B8BA34341B77CDD528E7860E5DEFCAA79B0C9A452AD4B82AA306BE84536EB7CEDCBE058D7B84A6AEF826B028B8A0271B69AC3605A9635EA9F5EA0AA700F3EB7835BC54611B922964300C953EFE7491E3677C2CEBE0822E956CD16433B02C68C4A23252C3F9E151A416B4963257B783E038F6B4D5C9F110F871652C7A649A7BCEDCBCCC6F2D0725BB903CC196BA76C76AA9F10A190B1D1168993BAA9FFC96A1655216773458BEC72B0E39C9F2C121378FEAB4E76A"),
             SHEX("3A0476C8E248067F0F96F01561CD0F6E55BB31FED70D2AD5F282F030044A331C6A20F14F254D1CA11EEE4226323874A1035C0F522284627B51F29EE43B291D24001C4A44C638AAE5A89045C141E16B562FC1893F39CF49BBA6D17EFBCFA4919A1C60D60CDB132CEF068A05A81FA5C374273FE366F4CAA16887C5AD1288BD066B741D234FF63AAE976E742F5DCFF610CAA33826A7856E5E5265FDBFA04007FF3F0F701850DE3E911FD71E00012DE9823D78E381CCBAF09976FD686811255423A7CDCADBD854BED848EE0ED2977FC192573F2F36C62A5539CE807C2B01AE7220C6139D1ACCEC6CA37F2C964BE922547BC692274D557ABC1E7D521D1215CC56260D0C7EF4C3EEFA6D54182FE1928D8179FD4508B8A4C7A3F78D2D0CDF76ABF581F0689D1BBAE1F7ED3A8AF23D687C369398C715F71377DA92A518EAE81C7587876691AF81B982340D71F3FF3836AE9E87EEF7DB14AFECE5F57E45746173E7834132F922AB5AFDE73076B67E446A0F096ED2BC87E12216BACCEAF1E912CBE89483B5E0BC35DF7C38F79F31C1567D16A321920B2D0E6F3637B2C65AC762239522433AA51EB34A2A174C93A693C187E97B3BFA4C8C9C7DD512A74F26EF74EEE24D3F425F0CC05F0A3A630806FB960437F6EDB2C31B0103B3933986BA9E0B2D26B554FB9AF7DB5454A18A8D52EA8445F6844BA7371511CD14FBE171BB4045C7377553E7"));
  test_xof (&nettle_shake128, /* 192 octets */
             SHEX("08959A7E4BAAE874928813364071194E2939772F20DB7C3157078987C557C2A6D5ABE68D520EEF3DC491692E1E21BCD880ADEBF63BB4213B50897FA005256ED41B5690F78F52855C8D9168A4B666FCE2DA2B456D7A7E7C17AB5F2FB1EE90B79E698712E963715983FD07641AE4B4E9DC73203FAC1AE11FA1F8C7941FCC82EAB247ADDB56E2638447E9D609E610B60CE086656AAEBF1DA3C8A231D7D94E2FD0AFE46B391FF14A72EAEB3F44AD4DF85866DEF43D4781A0B3578BC996C87970B132"),
             SHEX("868C7566EDF719D04D7457BCA5CA9B4B62F309FBE1046A7351F24DCBE2E2224E34CC23E2C859E83B8B09E15FD575FE5225A03D1869CD1879728825A67008CBD1C7F0535086DE803A3E55DBAD1BD9998EBEACE9CBE63C4500B5ED3380153A41C33948E82E2D375E70E2A8C3F1C1AEAE72D000EFCBD7F9DA3FE0422AEB9A10B81B07ECCEE0C227947E7DD5E2B277A80718015BF5CA4C9B3EA5EC7CAC5CBDC79642CC0B547729A0956C739C956D09FAF013F402D764EF87A5027049C4FA0BD8EE8D77D99F3124710DB8B274B1F8878AD74CF88F21ABC520153B2668B79F824D74C4F113A66A275FF882B64794E307182C5EA6C96A719FEC5FCDDB619579246EFACF142F976DE21ACEB21136E9AAABF231706C767A664602C676EA46C70F0BE2A7F9907A484BD6064E1BF132FEE115362DC13566F763F4CB29A3CCB68594D9E7427F2D29B1A3D02520A81AF9D41990E04C9766260841E86965DB59B764E1867F9F2B4BFC6C13BBC08C4DB843982E262F75CE6F0E034B58DE6ECDB71A5E32E9F91CA8398EC286277B0DD6A518031C16AB864971C243CEBCF3755A72AFCC186C5D200873EB27B070603A6FE494ADCAC893162CFFB544C2B69083693A407DD92CB4F6B1B341EBE491738258EC5D109856BF106B56E6B96481B933093DC4DBE2CD075C5E3B760D36F3B92A286F91AF57604B70206C8922ED21006E05A271F1415F84F6BA"));
  test_xof (&nettle_shake128, /* 193 octets */
             SHEX("CB2A234F45E2ECD5863895A451D389A369AAB99CFEF0D5C9FFCA1E6E63F763B5C14FB9B478313C8E8C0EFEB3AC9500CF5FD93791B789E67EAC12FD038E2547CC8E0FC9DB591F33A1E4907C64A922DDA23EC9827310B306098554A4A78F050262DB5B545B159E1FF1DCA6EB734B872343B842C57EAFCFDA8405EEDBB48EF32E99696D135979235C3A05364E371C2D76F1902F1D83146DF9495C0A6C57D7BF9EE77E80F9787AEE27BE1FE126CDC9EF893A4A7DCBBC367E40FE4E1EE90B42EA25AF01"),
             SHEX("49D7FA3A63EE1611E9ECEFA88F80DFCAD8D7AD69E6A899AD04D102885AE1419F8E9E6897F1D6937177E1BD8CA104A7C227B8DCDAD473D0A2BC12C1A2287B5DC66A0F808D02C452BDFBF5AC4F43B38105B5A9F2E67A4C6C81FE5DDCC1AD3EAD625B29031FD5F1EF18F4AAE15EECC2B6F5A4336E5BE3FD7D3970D50261FF6F992E5A0F818BE342B910B267C8740B7F331E0623B5793D4146DBD8BA3636D12914DF35CCFBEEE62BF3E0033DDD26114EE709A641ED54C21A130C8B3E1F9A49965E48B8AB914AA2B2A377DB1873A1882625EBCFDD1C366360B8DBC631DB94BFF615BCAB90D8BAB3C14F2CF4A6E4C4130AF4255ACF14B2B3699EDF8753C745F9D8DC23AFBF449A05FE05FCC83FB609856A5BE79B06E57D3E42531EDCC13B1AEB76BE54CEDF8665439731CEE4144CEDC0E8ACF9EE2AF245769D24DA6261E368CF0D599225C397C61A87EA8C0EF648A74CB1E3C5ED2C6CDB766E83BC68299B1AD882CEF5B436608847548141FE1DB303BDB13771EF19674F196EFE3502B14A7A47283EE6BBBA86354B88A8FB15187E0AC84CE0E9CDF737F62C15ED3A4A6FA0708F91DA2190268C0923D1E696B5E98251A6783479E6BA28336C39E27680BB86BEC2991D82EFB40CFE6985944ED3B18E2ACC7E51BB24D0F2C19BDE84CB271FDA6BBB5DE6B40203AB54CEEAAB0D84A339DD84108929663A50EF23BCF1E79909E2275FDDBE57"));
  test_xof (&nettle_shake128, /* 194 octets */
             SHEX("D16BEADF02AB1D4DC6F88B8C4554C51E866DF830B89C06E786A5F8757E8909310AF51C840EFE8D20B35331F4355D80F73295974653DDD620CDDE4730FB6C8D0D2DCB2B45D92D4FBDB567C0A3E86BD1A8A795AF26FBF29FC6C65941CDDB090FF7CD230AC5268AB4606FCCBA9EDED0A2B5D014EE0C34F0B2881AC036E24E151BE89EEB6CD9A7A790AFCCFF234D7CB11B99EBF58CD0C589F20BDAC4F9F0E28F75E3E04E5B3DEBCE607A496D848D67FA7B49132C71B878FD5557E082A18ECA1FBDA94D4B"),
             SHEX("815BB24A227FF0B94478EC10D6B4FAC313D56FD9DED13E94A57869D8BC5E542E47CD97D42F1387757539A452C5C40492DA8A4D65190AE36BF2E169446BBE5CE0742A1A9865126EA07AD3875D47C5ED5F85C1A7A57CC350E05A4DCB1F052D9FF4AED38C5E1D65C05C3C7DB8A9A6C00AE2B3B45FEDCA8C309B2C369A7B9608AA365015AF193F33092D084B2A7AC5CFCC018A045A1D89989FAA5FB16F854453D9A6336B900ECC4D9EAEF9DE4C6A287C44D0E4866CC8E4F8CDE5317EEE01113DA6A64B88A1469835BBA057760EC1D03D63D9CA40AACD7CFD8619E92739CDD72C9A11B0705A4E50DC386D4D378E624818B2D36B4FFB6DD1AD2C9F9D2A8BC4A7FB73827930E98CFA03C221B7C9D648463B8C2AF1DD195A817C0E986E8DE6C11371AB830EF74CC5E734B56C6C013350C1212901322BFCB0CDDDF13344473E3950C393466DFDB59FFF9E582E9B79F5938B0476DE125FC36FF74C03CFB685CB09BFF4C234F272D9B1BA6A2560308464CE36DDC0EB6863AD42B7D0490091C05A745CA5F288B303A0FD08BFF7BA2265B6C12B25840357DAC735CEAD9E6087D38A97421CD4E54518D7C0CF0968860B6348D1531C56C1B4F6B5C7771377194EF24E91F9D9DF96253B80009341E10122C14D919A037C820822A23E47620B35D0DA729A1AE329F99EBCC26FCD8CB9C67334F8B04714D2B76D5D0EA6156187C9017E764AEA66E88B"));
  test_xof (&nettle_shake128, /* 195 octets */
             SHEX("8F65F6BC59A85705016E2BAE7FE57980DE3127E5AB275F573D334F73F8603106EC3553016608EF2DD6E69B24BE0B7113BF6A760BA6E9CE1C48F9E186012CF96A1D4849D75DF5BB8315387FD78E9E153E76F8BA7EC6C8849810F59FB4BB9B004318210B37F1299526866F44059E017E22E96CBE418699D014C6EA01C9F0038B10299884DBEC3199BB05ADC94E955A1533219C1115FED0E5F21228B071F40DD57C4240D98D37B73E412FE0FA4703120D7C0C67972ED233E5DEB300A22605472FA3A3BA86"),
             SHEX("47A1ACF4036C3A62A24F7A2E1147F5729DB886339C7CE7A9A6E7613BB60C475724D573E64E4027F296023E762CA20F880AB6C73142D3D1D36C19905BBA147317880DD9210A75FD7C552076F2419432E46C858E69B7F7C72372BE510F07F21977DEA627514F6ECBB34DDCD6CF5D86B01BE3DA48481B25D56071CEEE2142A9956AAB4760666DE359A36B8808D1DF3F3C15F22163CEC7C1181445DF7D06E79BDA799A6C4F278968B677FB78C8157CDA2FB7447ACC2D485E6922D82E1AF596C760FCC2D7D83A6A4B52400E3FDF64A2A7E4D9D9C59930579D4516618FE9979B10562B35F26CEED8D53DE5B322B3BB4F1B989EAF31517AD896CE3E57F879D6DE9D4A847E4F7C9EE093D7AD482D84D947CAB0B6619D8895FACC2DA8BAC086E054B3ECEB72689DF5730FA6E51F98205B1FA5AC9DAF82A78867B655B39792C686518BE3024A5F975E97A021F64FC75014635371C9DCC8952F2B5404582855FFB049561F3E20013E74C8A05FD434516218CC6E463F62515B454B358C611F2902B9D11BAD43862497532DF84DE73A02054459B79CB956E6DFF229E8FBC8CC558D666E10660B87B9B0831DF729CD87A22FA33891D9CEB2CC7CEEB1F316600B9432346D1FC211CE1946946F33BA59D6B53F9208F8F1FA8362524448CF87A851CAE07D0CCE86E594B5EACCF0B4F10F8C4D414AC194909CFE526CC87FCCE1386C7F5537D10352F5"));
  test_xof (&nettle_shake128, /* 196 octets */
             SHEX("84891E52E0D451813210C3FD635B39A03A6B7A7317B221A7ABC270DFA946C42669AACBBBDF801E1584F330E28C729847EA14152BD637B3D0F2B38B4BD5BF9C791C58806281103A3EABBAEDE5E711E539E6A8B2CF297CF351C078B4FA8F7F35CF61BEBF8814BF248A01D41E86C5715EA40C63F7375379A7EB1D78F27622FB468AB784AAABA4E534A6DFD1DF6FA15511341E725ED2E87F98737CCB7B6A6DFAE416477472B046BF1811187D151BFA9F7B2BF9ACDB23A3BE507CDF14CFDF517D2CB5FB9E4AB6"),
             SHEX("2F06C83D5CBC776C9E2E490482D92BBD8D32B07B1028E046582DD17951C13CDFA27C065A2821D997F37CED1D3F69ED8A8EEC35A55C1419B28770800A66A8CCC05332C64B0A310A1F4A99EE9A18ED342CC03A89193BB63849A6F0288B29B0801000EC0E3961FD85E6CF488031F028232B8FEDE6827C7C24FBC33A26A87A9B6210004038C67EECF0EF9D19C3DC8FE2C7DB8A449F168A98CFC190043FD6E2FF366CC35556192E39D1664194D6943F9EA5B4675D5051100E1497C1970C2927506B5B717D2DBFFCC9F62A0AC338340C8C473B397B98812BC77202A67426ACB83F4A103687425D9F793312570FF6CBD897733CE7399A3853887918A9EF8D0BFAC5A303B9E66CEB5BA52B826EADA8E5E158422A6D42F89E986C30F9D8E14C8B3FC7DAA5B012452666612ACEE1884C069F98E476F89E74B35EF94002744108A0EEFB9A9992104973E1176F2AE0F560791B9486E7A27C75BAD42054047E87FB95F43AE211FED7E3948745624F8EAE4A81CFFCB4BA389E169C9B55FDE64DBC31E6287144D0803EC3F0E5A86DE5FCD0CBEBAAC40D7103B3C0BCC8AFDE491B25AA472BEEF91D2AFA59D980EF1A833002838A2159C0F9FDAE39489B6E865ADDA3EA6DC8E675E45A45857111A2EB49FE3ADCD52EFAE414B6EE423440419B310E3CF751F497ADED9BD3CEC9172A23FFC878DD9FF6033EAC9C4CE7697BA9EEF60B67665CEBABC43B"));
  test_xof (&nettle_shake128, /* 197 octets */
             SHEX("FDD7A9433A3B4AFABD7A3A5E3457E56DEBF78E84B7A0B0CA0E8C6D53BD0C2DAE31B2700C6128334F43981BE3B213B1D7A118D59C7E6B6493A86F866A1635C12859CFB9AD17460A77B4522A5C1883C3D6ACC86E6162667EC414E9A104AA892053A2B1D72165A855BACD8FAF8034A5DD9B716F47A0818C09BB6BAF22AA503C06B4CA261F557761989D2AFBD88B6A678AD128AF68672107D0F1FC73C5CA740459297B3292B281E93BCEB761BDE7221C3A55708E5EC84472CDDCAA84ECF23723CC0991355C6280"),
             SHEX("ACB6638C5302BFD244F9C4F1ADEA0461562CDBC92670F2C0C9135BA8754BFFEDF6621FB48B702792F71AC3B800BACD3A18791642865247B75D086AE816CE17F6D0CDD63D0A2FDBA9C743FC9E3273E6834D21F12554B977A146906682B5DFD85F41BEBD1ED03CA3B67DE188A0547493753465D76145F5F860EDD2F8F3C09ADEA139C3E057AC64487962E3EE38E1FB9F29BB5FCF2D70993AEF47E56A26C70D200D357758A4B34BD9327DBACF987754D805CCB5D2F4CAB0E238DE726E6FF78239C0F2E020A1FE4D4F6CC27C0747ECA37B4F72A110B3B8715E3DB07DBAABB0E580C4F20DDCB7ECE60B295211AB7CEF858153DBF3F8D0E27F621551B317534E9680576D629C5928F1A03C9928A93B67B4FFA019DBADF9AA71D37E79D40051F6BC6DA2D567E3E48706346275DAA4506A0CE668C2D4768FFF917A11E4C1726842D315DB4D189FCF686C8BC1061A63FB05A73720088C9FDCA783EAD1FB390ABBBF0B6BDCC775CA9AC32A1CBBD66BD80DA59152C97E295AB35DDEB7048B975C9E2A794993852F31A18840528D2F297454598A3F31BF9973B9CE54D5E2A8012EB2CEE9D49FD8299DAD5BB566629F6EE4EDDD71E6D08C223D0A1F48BB804096B24BC6DA27B80AC2EADF7B7B39C29F3DBE556AF66463785707A23495E2FFAA815640BC925230DDE6E5E545A7C414543D55CFA23330BE5CC1720A816E4064FDC0BB45C0D9A426"));
  test_xof (&nettle_shake128, /* 198 octets */
             SHEX("70A40BFBEF92277A1AAD72F6B79D0177197C4EBD432668CFEC05D099ACCB651062B5DFF156C0B27336687A94B26679CFDD9DAF7AD204338DD9C4D14114033A5C225BD11F217B5F4732DA167EE3F939262D4043FC9CBA92303B7B5E96AEA12ADDA64859DF4B86E9EE0B58E39091E6B188B408AC94E1294A8911245EE361E60E601EFF58D1D37639F3753BEC80EBB4EFDE25817436076623FC65415FE51D1B0280366D12C554D86743F3C3B6572E400361A60726131441BA493A83FBE9AFDA90F7AF1AE717238D"),
             SHEX("2856032A7399BC2D4A4D43D7DC024AA3040ED171982042E0D1A4063B6603526EB4DEA45FDB007AC0E61043A11A502D4C49440C174BB183979A5D3A35C593E1A01886C60472D40D161A0512E3372BD5F8D35A217391D5AD14DB17EB23BCB74D7701286A158B03A927CFD03C4062EC0461F8C9C3D4B8DE0D70CEC5EB018375E138278ED9ED366D546D896DD885DA0C3513C5AB7EFE2ECA848B270CDAC3D2FE7E14F3DE6848FE01D64D841F198C85CE57E16B06FF41FFD454B6C61569B71D538CE5729DC5A5AA1D4DFD5F1D3F4CCA0F424701E8857EC66C8DC3D44DAACAB9D05626751240E1E8F1ADC1E064AAF47C66D9EDB0EE114A6155E28BF4769F88C053313792CD9BB2D9687E595D321E152CFB42F67EC9DA6E373E7621E4379A3C22B4828412223AED28946784CF67B6CE99918E0C1217501624599E4E79E90016ABD265E21E91205824B7FBB95E9ED999C50F90C5748F83E71EABB66DC2C16CAB63C064BB88C865FD30EAB8FAFE0529312BE74B55BADC4C944C69B89D98E7E07054A0AAC8AAD91F53582B91CB22A56741C41998EC878B11671AFC9DF26B5261BE5EEAD565BE962E87A40A5200A0856123ED7D248431EFEC79D7932407A7D762137EEFABBBE0C6C482CBCE115FA9CB7254614DE54E361C5CBC4EAC03757B83F4E4A96BD47D8347ED5E2C9E96DF3F03A9762988D671AA75EC9F3B5E833570F13CEA7C323522"));
  test_xof (&nettle_shake128, /* 199 octets */
             SHEX("74356E449F4BF8644F77B14F4D67CB6BD9C1F5AE357621D5B8147E562B65C66585CAF2E491B48529A01A34D226D436959153815380D5689E30B35357CDAC6E08D3F2B0E88E200600D62BD9F5EAF488DF86A4470EA227006182E44809009868C4C280C43D7D64A5268FA719074960087B3A6ABC837882F882C837834535929389A12B2C78187E2EA07EF8B8EEF27DC85002C3AE35F1A50BEE6A1C48BA7E175F3316670B27983472AA6A61EED0A683A39EE323080620EA44A9F74411AE5CE99030528F9AB49C79F2"),
             SHEX("6AB8F0F45B860126332E32AD1819FD5A23DFEEE2C7FE47081AC855901D2DA3F590FB64F989EA616FAA0533E49A3B08589FF2DF8B8A27BAE25A23FA93710032F2DDEC4E3F905CA8AC37CBA1D9D225B2F62343DD5418283DB7D6E427EF537650944723DDD7DFED6394BA1540B2FE72944C330CFAEF76A9CC3998C1B6C38A83EA71B075F1E65C9209BEDD565149137A4BE1C640F4EC86A556B196BDF14F5C336418DCA1330CBE1781DFCBDFC26B562025C9DB90461658C4F1947A1B1374D8120168A1AB8D0CEA7527F0503F19C4605463CE134CABE1C72866F37EA2F105D44A3F2080DD42DA509ECDB1EA8C0AD439820D4BCFFCC5C34BF78CD8BEED8E394567A78864517CD183EC4366249FA9340FE4A112DBB418B32AEF24426CC95E12BD243F3BD4E744407F3B461B7E8C7FF096F2279962D60E778E1C362A5F47E4ACAE0561944BE126C8ED384BA5E71F8C23FD914D525D3AFFA915CEBA52CB3B6265E003D6B2483C7C331DB0F56EB5F4D0B4A7DB558A154E34F539136C6869126FE3A7FBBC07A92CA251A6ED301CB1A8415FF16CC28D1482FBB7A70DF1EF2D5AA55A084116577301EAACD88BFF5139D091D0BBAE55E31E50FBF28952E7071594FD81B626FA4194D97572058F217ED070B97ED42D8C7D9DCE730D270A01BF847929FF0E2190446D82F93D9C2BF309F836811DE1E4700D9BA736672B407E539219634F32956ECC"));
  test_xof (&nettle_shake128, /* 200 octets */
             SHEX("8C3798E51BC68482D7337D3ABB75DC9FFE860714A9AD73551E120059860DDE24AB87327222B64CF774415A70F724CDF270DE3FE47DDA07B61C9EF2A3551F45A5584860248FABDE676E1CD75F6355AA3EAEABE3B51DC813D9FB2EAA4F0F1D9F834D7CAD9C7C695AE84B329385BC0BEF895B9F1EDF44A03D4B410CC23A79A6B62E4F346A5E8DD851C2857995DDBF5B2D717AEB847310E1F6A46AC3D26A7F9B44985AF656D2B7C9406E8A9E8F47DCB4EF6B83CAACF9AEFB6118BFCFF7E44BEF6937EBDDC89186839B77"),
             SHEX("6E9D7E55E8B258208FE6C1C50CE490BF81748DEB2B4401800C2423E9479AF3BC1B4C37249A2DC00F120DC5E2114ADE14C830437D6B0190FEC5AAAF57D4E9C5078B867B2905D48E6748E85CDBB3F1995389AB373014E6237948B8343AB418716F0E9971DB6A179B11277ACB2DF6B6B5903EBF311F2D94B10C3AD0BEDF2ADC7486929C8C43CF0ED6673C6468CF69B493C9EAC2ADE0AB3987179570886A28C08D0DCC83B0162A8D459680FDD2422C2C40A44E3FE55D0AB24CEADE4063C995915794B0C842573E2E7C4BFF8E201D3F9B3E8F49D90186AABC9DDC6110E98E3410617DA24FFBA5A7E5C3193C16B70CD6CF919DD6F15DA3627F42B6225EAF4BF1851A570E099FE3B8E7D746C34573A74744D42135332DAAC9A9341E598C714FAFBE052F7E745BA1D424CBE0CB1932A9E497D2111AC597F7E5010A86567C51218451EC3D1461D1D2D54F5E8754C854CD4D60972BC09482084AB865DFDA01D1C7AE4C749BFBDDD19BBCD95E8A53009529468BC4C47D9015A119B9C37DD2C149C65E6C99699C69C3CFA405C65E0E51A3585D35EAD701FEB58F1AC72D74E87D2C65FB072C11B235FFCDE39559A45F881DCB292CAED95B3AB6E0E468F86A235E2D83708044D75E768A2F3EB13523338761DBC38A8E014052DBD46A0064AE0255BAFBA6A0C8FBFB40CB41DCDACBC5466787638B901AD452D7D50A0C610001FBB6F126902D2FD5"));
  test_xof (&nettle_shake128, /* 201 octets */
             SHEX("FA56BF730C4F8395875189C10C4FB251605757A8FECC31F9737E3C2503B02608E6731E85D7A38393C67DE516B85304824BFB135E33BF22B3A23B913BF6ACD2B7AB85198B8187B2BCD454D5E3318CACB32FD6261C31AE7F6C54EF6A7A2A4C9F3ECB81CE3555D4F0AD466DD4C108A90399D70041997C3B25345A9653F3C9A6711AB1B91D6A9D2216442DA2C973CBD685EE7643BFD77327A2F7AE9CB283620A08716DFB462E5C1D65432CA9D56A90E811443CD1ECB8F0DE179C9CB48BA4F6FEC360C66F252F6E64EDC96B"),
             SHEX("BF52ACA76D241CB569D39EB2D8669D1B2642C0CDB8B141E87C3B1BDB1F202641A1600AD388806BA9F5DB50A7CFFF97CB23EE977C0D3899F414FF5C47D689840F59E7E7DB2EDD92AFF3FB36FD198E7BF5D81FA279F743FDB6B9C00E974065802903DC40CC3E216FF727A72F267624894F114C62E7B101D9DA391E3A205C7B492B7EA28D836AC6657960CBBE4C38A73F87EA1F8A155A62C0A5D5BA821D9ADFD42BCF42CB3150267AEAAE507A4009BA8DCF70A13AFEDCB2121503CFFE786304E3AA980D0FCD7C908406FD2CD9CAE45DE97414647DE04D03C1214C967A14756830AFA6F83AD91CA66D9DE50B8F0483F99572A284BF9468A6ABEB45F335EAF0782C04563DFBF231956BA4575DD9BDFA10E2F6DF8878415B357867E5C22B9CF349B480DD0CA1F7CD432FCDA057A5F9AE580814A7CFE843FA831A5CDB87646BCBE70229A3EE7CBB3094E591FACC8680F298E10E1383D3A6387A503FD537A6ED924903CC9149879A95A164A8B3BDD385A69A93C58AACC066FBE4E59C7F6E16C00E45BDC19EC267F7684705CEF34D830ACCC03822EFE9C1B3903C47FCFA91FC7A7B58240765EEF217341FF96F0D0CDF5062D8B1939185C47FABE5498A784622D1B2DA0DBF06DAE6F661206F98EBDB257BB2FC4C86EF6E7C6E31E7756C8CFE7C4842CDA9878D6257EDD7F204A009A9E1A492F5ABFD42B48593231324728F8A510C47F516E2"));
  test_xof (&nettle_shake128, /* 202 octets */
             SHEX("B6134F9C3E91DD8000740D009DD806240811D51AB1546A974BCB18D344642BAA5CD5903AF84D58EC5BA17301D5EC0F10CCD0509CBB3FD3FFF9172D193AF0F782252FD1338C7244D40E0E42362275B22D01C4C3389F19DD69BDF958EBE28E31A4FFE2B5F18A87831CFB7095F58A87C9FA21DB72BA269379B2DC2384B3DA953C7925761FED324620ACEA435E52B424A7723F6A2357374157A34CD8252351C25A1B232826CEFE1BD3E70FFC15A31E7C0598219D7F00436294D11891B82497BC78AA5363892A2495DF8C1EEF"),
             SHEX("08E74B227985F88B7C1884A179D0B9C9F58BD2078F8938519F460511B00003E51A8EF4C93F69B716DF11DF2839688108ADF72476D1675E4DF3399C05257B1DB542D92F0E27FC59A927F00C4758ADA0FD95DB9D3251C01CB9167B2D6310E051407A2EAF415A2E478BE53C3945A702F04E25713EC2E2E1D51D636870642F6023A43FD0859A3B468A9502A8EFECF19BF97906E199C095464600B0B7A1B180B841158C6C98D13C371F7AB56496EA14CFF3110AA9D33869069A1F9215A6FD99CE9226EEF5A272B004827DDE0E1B1BF4B0C0CBB670CAF16B1840C6E88E577ACC2ED4493C598B938AECE1824717964048786D9CAB9ED14DAD9F87DF1E9C374010D89D4A8616CCDBF687E12D7CDFB388CBBB88B56FEE3AEDBC5360088A49FC1B2E1DD5518E0E00F50E3C922C8CA600A41F72B985F02E32A1F9A1FEE1E1929F5746FFEB87EC41F08861EA3463ABB9803CCE8C8257853FDC0B0C077D4255DAD1FA9E5D865C981BA15236828E7E425F5DACC4B738DE18E6C68A6C925B72C14E3042916117E91CB69311081E4E845B09679CA0607BBAFCC1341B283BAD5465633073E5DAEED6541B1FD09537EDA2F0D2A51CFD8715D3064FB074B52D48D4E7175B4AD0A0A6084128424958275068BAD56B5AD2769918B6AABA7AB1E8C9F88859B3DAB310FBA2FF67BDD3FC9E0B2EDDE0B716897220AB75611975E137DB525DE4465ECBB2D9D1"));
  test_xof (&nettle_shake128, /* 203 octets */
             SHEX("C941CDB9C28AB0A791F2E5C8E8BB52850626AA89205BEC3A7E22682313D198B1FA33FC7295381354858758AE6C8EC6FAC3245C6E454D16FA2F51C4166FAB51DF272858F2D603770C40987F64442D487AF49CD5C3991CE858EA2A60DAB6A65A34414965933973AC2457089E359160B7CDEDC42F29E10A91921785F6B7224EE0B349393CDCFF6151B50B377D609559923D0984CDA6000829B916AB6896693EF6A2199B3C22F7DC5500A15B8258420E314C222BC000BC4E5413E6DD82C993F8330F5C6D1BE4BC79F08A1A0A46"),
             SHEX("3B3E210E0FB73556801A2250F1E76C45DAD31E12F9EE3448640B73D5AFD6D2D6FACDC1B0CFA53F46A56923CF16B6BDCAB566F3A0105E442ED7E88C09D705249BE34463D435C14D5118A7E028DC35FAA98112F80D816ABE04F7345EA55E7F7652995FC047AFF6B8E4D05C24A0C78FD9FD40835E523BE7867312630993FC9E5CBFB47048453C205880EEEEA2ED9A198AC145537BEA2C216A60774C400444C07CF6546159EAE47FD6CC79CA43D0D26C79DD13F39748A696F181984B5B8EAED82976C9B0C63C915BC2A2DF91139618AC82E4478EA26F9731D422C692AD1BCF483936D99F65CC8E5CE20CD94ABC2C4176B668D1813AA3ED84CBE68B8C982A8F884671CCD29B70EFBA765CB4BE7D0D139798FBF344D7D1829F3AFA946C3558B4C958A9235E4E96B28BE25641120E00EFE178304804D90946CDD1DB6229C0147339A1F75FBEC606A3B3FDCDDD9235EA6BD3F166AD132F8C222E45A805D166F8FEC56D488A2AB66737E3238E67419E194665DDAD4D8214394A4DA1AE09E0EC4EC5B209C18004F4008ABFE997E456BBF99F1FD9B31E69B934BA694ACADDD534E4BB4472179403BF9014A9EC311B16CA2900C669A8981BE3B5A8A9D3BF4D05028FE6D3D97D54B9BE443F9B0F73E59780D09AFC937D1F36C4CD8CAA1EB2A8522AAA175FF2E4A3285849C059991174BEC1BA6E42E399C4AB0A566BFBC09CF1B836BF42138026"));
  test_xof (&nettle_shake128, /* 204 octets */
             SHEX("4499EFFFAC4BCEA52747EFD1E4F20B73E48758BE915C88A1FFE5299B0B005837A46B2F20A9CB3C6E64A9E3C564A27C0F1C6AD1960373036EC5BFE1A8FC6A435C2185ED0F114C50E8B3E4C7ED96B06A036819C9463E864A58D6286F785E32A804443A56AF0B4DF6ABC57ED5C2B185DDEE8489EA080DEEEE66AA33C2E6DAB36251C402682B6824821F998C32163164298E1FAFD31BABBCFFB594C91888C6219079D907FDB438ED89529D6D96212FD55ABE20399DBEFD342248507436931CDEAD496EB6E4A80358ACC78647D043"),
             SHEX("615FBF41D7184EF0EDF7AB0B079B76A4A7BF2B10A79A2FFE8D22E2F0A8B6280559BFD74E4F68E7691264A5B4651AE0EDC504F6AC0677E14389D92913660E75626615622DD458B7CAE4EBF1DA4F7ADB0771A809BDB44E918F7D681711EAE7ED84745A1E7F3AFF0931162BBFBE975C394AE6843CEAD454A00EA54DB5A55AA845413EC88BDF9119E238E6C84BCB76342FA8BA1A448AF7CBA6D0F1C964EE981AEF20C5268BC1C3AE3F723E48D1932CDB14B95629DCC5269F2EA77F277354BF910B05BDED92D6B8BAF9601080156EF0D1600E80D8050B99243C64221E9B8AE82773C6BE95829203FBEE6CAEDF299509D6A2DA570320CA6410E9853575A342F9E9AE76E27E1B0C892B2C932D80AD2B91A8D62D805FFDE196DC1F169D2B9D51F28FAE6546887182B91298B7B2FDE472BFFD86384EE1B78B4B5654F3DC38FE4400CEF50338F1D79B478C6781ECA2130CF3A86D9124C31754989A22F022658CFA62833CD5F25A021B03A84A7AA3F8890922AD179D589E46B535B0E6033C8383CD6B80324EF8EFB3BC6CB9CF5BA0BF5F6B185B83EDBE3F0BF27A9942FA87D172A5C9294BD63396EC8E1A5C7EE0BD7DA1109A69E2ED928025B8383C55C6EDED0BB9E23E15F50388209B3508FBACB7DAC520E0327F2148FE27450D0C5C06712A45011C95D7B02C0E0F15BF8C222527451C1DA4AB70CF649A025F8C95CD919E937B079F364C97"));
  test_xof (&nettle_shake128, /* 205 octets */
             SHEX("EECBB8FDFA4DA62170FD06727F697D81F83F601FF61E478105D3CB7502F2C89BF3E8F56EDD469D049807A38882A7EEFBC85FC9A950952E9FA84B8AFEBD3CE782D4DA598002827B1EB98882EA1F0A8F7AA9CE013A6E9BC462FB66C8D4A18DA21401E1B93356EB12F3725B6DB1684F2300A98B9A119E5D27FF704AFFB618E12708E77E6E5F34139A5A41131FD1D6336C272A8FC37080F041C71341BEE6AB550CB4A20A6DDB6A8E0299F2B14BC730C54B8B1C1C487B494BDCCFD3A53535AB2F231590BF2C4062FD2AD58F906A2D0D"),
             SHEX("8043574AA03D7C6409677E5E579FCBCE191118E2161C214640D725D384A3D941857498CC75480574F9BF4F7613241183B7B1354D4D08A9AD1AF03230636A14D40A30A1CE8D1DD9CBFFE41011E40D3FC3DC1ECCB83374F14209EB83BD16861E161882CFC86A5332BF0357DE84D38CB6FE65FF8EC3C8A01EBADAC09FBE960E5D55F752878A608B3E67EA840503D59B6625B42570F74671B4267246995B5575294D20B6E693D02395AC5329451F905B14D7FFF86A4401F78E6CC7B83806C19F30AF6C75787F56222B9FF673F5631165ED8F20DA13167F4DFA4C53476D7E2D7F76F4EA6DD77072FEF26AC3AC7F65EA2C4F113897D4F00ED38CBA8AB162B1BC3151D769D24A42DDF88F9515A04E3FE57FB4ECAAE26F608FA6A0E30166F5A12223C10ADB56803343E938EB88C839EF9F91FC7A0B9DF899AD3DC328166C8AF33DE8560EDE98EC371A22938252432FA57716C4058B7CDFB1DD7F5087CD45EAE5BF1A4C3AE03C19CCB580EC122DFD54DD2106A4E5A647558B24D15D30CE2FCA05673F8991003EC61270F279B7C2DE67D1930A5C948C9259D4E9DBDAB2D94DC1E1403697499544A1FA1109956AA22DA488B61A97E933FECAC59711E1BD5967DF18130CA394FDA025BA334B2857A1EDECB002BA56DCCECF9044E579143DC47BE03072D935607DE1E674C955A63508CB5D42AF7FDB80459F4A239DC78C31097E71C06924092A"));
  test_xof (&nettle_shake128, /* 206 octets */
             SHEX("E64F3E4ACE5C8418D65FEC2BC5D2A303DD458034736E3B0DF719098BE7A206DEAF52D6BA82316CAF330EF852375188CDE2B39CC94AA449578A7E2A8E3F5A9D68E816B8D16889FBC0EBF0939D04F63033AE9AE2BDAB73B88C26D6BD25EE460EE1EF58FB0AFA92CC539F8C76D3D097E7A6A63EBB9B5887EDF3CF076028C5BBD5B9DB3211371AD3FE121D4E9BF44229F4E1ECF5A0F9F0EBA4D5CEB72878AB22C3F0EB5A625323AC66F7061F4A81FAC834471E0C59553F108475FE290D43E6A055AE3EE46FB67422F814A68C4BE3E8C9"),
             SHEX("2ABA810B7512B1B3B179995F76F843BF39FEDB5B969267D5F545A605ABBB3F627B59550678B4E3C3B8E0959D5907F88B814413A5D94B0011FC8BC445096CF1B49D11FB2A4EA1EF1F219F7DDEC03B3678F2D95DD8FBDB7A1604AF0D8A5F3DC33E257A54CA3EB50C49C03BC55545A36F807ACE5A69F2D7069D3530F304F419B8D5DCB80890036299465C3C9C45FC49A73F9BE6C869113AC03DC5DAA116BC60F687DF7DB354201DAD9CD841467C27E6F0ABDF5EDA1F8B7646365A93BA71E1FF6C28D83AB5EA309E2C902A6A1A6396525A97EABF4F69C56B3527811046674F99EDD2779AE353654982A10C22BAF7FDDE401CE67E8044539995F528B398DDDE5132FC156E5587A9A3B3D5329F8DF4C55E86F3FEC30788D8E00DF696CCC46A564F2371932E21A45541315D8C31DED3A161E0F595EF47F8FB1DD7F1B57537FA36D105F1AD0E60C917F01D9C88FA0020349CCA8B01D77AFFE23CA4B575ABF1E42726B271FE77FF0FE81A8A20C06AC8DB699AA2E11F05CC5E9BB794B7E1F847F411FC4621281D257219709D38BA231E7060F745FEFF6B4BEF5DF477A24ED763E2BD49929E6C8C83EC776BE988A8528B1FB360BF5D7BCD654FF9A1CFE1EBD5F6F21954F155B1F6ECFAA7E6CD4E52D9B08C68943B29B54C5628F3EF3A966A94FA964A5874BD2FDAF65A663743F07EB0D6B3B3427DD419C8EF627675593728885A3DDB1E3C4D"));
  test_xof (&nettle_shake128, /* 207 octets */
             SHEX("D2CB2D733033F9E91395312808383CC4F0CA974E87EC68400D52E96B3FA6984AC58D9AD0938DDE5A973008D818C49607D9DE2284E7618F1B8AED8372FBD52ED54557AF4220FAC09DFA8443011699B97D743F8F2B1AEF3537EBB45DCC9E13DFB438428EE190A4EFDB3CAEB7F3933117BF63ABDC7E57BEB4171C7E1AD260AB0587806C4D137B6316B50ABC9CCE0DFF3ACADA47BBB86BE777E617BBE578FF4519844DB360E0A96C6701290E76BB95D26F0F804C8A4F2717EAC4E7DE9F2CFF3BBC55A17E776C0D02856032A6CD10AD2838"),
             SHEX("3582DEF5C0EC3C751727CBDFF5D9561E3A0F9EFFEF28C063461605140A2126265FA88C15B9EA3AD5CFD82378F78AC98BAA5ED67969473ED9177EF14F44BA210BAFEBFCA9C8DDA7E1AB5C6881AAE8EE0C824B47493C38EA7B5180323C7E962ABA1DE928A0ACBA596B2F8B7AF188265EB92C5BEBC9EF0A41F692E4B762990A40866AEC96CEECB20EF9F7CCBFB83A85485163789811FED7B8E0A43FD15CFA4BB13099F3CAB95791F8B05F5027672EAC2D397FDE1CD7E06FAD506CA1EFC22AD9CBB354D98BE5790B4021C7AFE6EF25D223510664DEC63D35158999C9CDCC513C8C7E93175A677A30259CB280ACA38FC7A8997B663A29EBD98D1C35F7DC1DAABD6770BFD3B84F1C0314E67F14E7FA9A881C21BC0E5C2AFD5F1B14EF4EDF9AD382F7D7E008934EF762B41782DCAE9B64448ABAE73D5A3CC80094B4F9983CB92D61C3555E58C0D24726E8073C39C3480FB2BA9F8679B9E8618CFA04921741B3EE72A276568068334E61089CEA277CF0D34EA33E02095BE33202EC69D2314128DDAD63F001DF3809BAA75E3A5720B34E8C7A63E06DA419A4818E48B30D8B0AE2E2405053C9CE76AC45C52EDEAF6237260F18940F0300125CE674BE744DC2A07262A146CB3BF595B59EB499439D94F8135DE7E58C0FD09744EBBB97BD4C10C178904F52C7481B4D808E663DDDBBF3235844CEA3DB323BBCEC1756BC4480C4FED0C14E14D3"));
  test_xof (&nettle_shake128, /* 208 octets */
             SHEX("F2998955613DD414CC111DF5CE30A995BB792E260B0E37A5B1D942FE90171A4AC2F66D4928D7AD377F4D0554CBF4C523D21F6E5F379D6F4B028CDCB9B1758D3B39663242FF3CB6EDE6A36A6F05DB3BC41E0D861B384B6DEC58BB096D0A422FD542DF175E1BE1571FB52AE66F2D86A2F6824A8CFAACBAC4A7492AD0433EEB15454AF8F312B3B2A577750E3EFBD370E8A8CAC1582581971FBA3BA4BD0D76E718DACF8433D33A59D287F8CC92234E7A271041B526E389EFB0E40B6A18B3AAF658E82ED1C78631FD23B4C3EB27C3FAEC8685"),
             SHEX("8C05C9635942C0661C29CDA0375395B595A82E4B1EF87D767E5A820CC6975F2B1AF87B76F69DE5BB9B77E14753C2EFEFCB735601DF5CB4DDAF5810BB89AC8863379192C578A3F2FEA30994EDF58F5A995857AF481FC4FBAA552E6564AE091C3EC2C5FB5144798405972C6947D45696F7E918D6820274BAC4ED1E721E85EE688E7EF7E6B4C87CA94CB6AD3221B5B0189D2F39A5482FFAB8A928A8FE4C117827E33E9C8B024BC35E30A9475D54293F196C83EED3B458D7E676A48D2BC2E09AD96167E556F00799E63C8C0668AA3710769C6533AF4F04816355BF4E88A2EBBA1B442BEF0BC9F7DD85A1FA42745893CD1F87614356E42E897821BBDE09CBA135ACE56612AD79A99A3397D2BBB6FCBDC505A0372DF7ACE31B53231887A618E4E846E3E458078B4AB9FA4D6D4D118F16EC8AA376B40CDD1594C0C41FEE2DFCC866FFB1F9B8AEFA04435F895B6D73E2FAA89BFF85FF14C133729B3789251E2A2CB38B8D5F9E6F8F0EEF26FA1A17A8255AE9EF58FA4BA9BD8A8FDE46ED5B0CFE9BFCBC20BDDAB37812FA44B0A1A99A6F10CF0E6BD931B6AFE22AF298483FB4D45BCCE9FD79C5A9CE2EB3359CBBB4673E0FEB2D1677F06A15F0938086DAA9CDF2B08FEF130BC793134369D4344D474F93C3C9EB877D0DE0CFEF351FB2589436DE6247B51352CA913C3E774B70655AD48BEFB6E4D494A6E610CD968E8CC613339D789C16DF"));
  test_xof (&nettle_shake128, /* 209 octets */
             SHEX("447797E2899B72A356BA55BF4DF3ACCA6CDB1041EB477BD1834A9F9ACBC340A294D729F2F97DF3A610BE0FF15EDB9C6D5DB41644B9874360140FC64F52AA03F0286C8A640670067A84E017926A70438DB1BB361DEFEE7317021425F8821DEF26D1EFD77FC853B818545D055ADC9284796E583C76E6FE74C9AC2587AA46AA8F8804F2FEB5836CC4B3ABABAB8429A5783E17D5999F32242EB59EF30CD7ADABC16D72DBDB097623047C98989F88D14EAF02A7212BE16EC2D07981AAA99949DDF89ECD90333A77BC4E1988A82ABF7C7CAF3291"),
             SHEX("84E7658F462F68915CCCC917B3F7A01F18E9E883CDA1FA6C856E2D3B77C53B9E9C3841EF72295CE3418319E1DAFF2F58AA0CCF89C1107C459B505A46DF314E670B51753CCE3701822C4FAE1F08EA93789FAE1A8A61755FCC17F57C8C8E344208A299995A3C3ED6266756D17DFB70E67F27621A6AF9EF926ED36F3CB169C9BFBB84DF0B2C51602902C12F519EA23C3B4B8BB2B1241798F72A748C6AE4E439CF4FCF0FBA17407D1366047B4A8DED133DA17A5CC2BCD4E9803DF6E5FC76A732EC64CD4C556DC97C8E8BE5BAD0FE5FE440E462DA16FB1C7B18A97F58DA4B5EA9AE8700A3E2D1325EBDE9DBAD97FAAD13DB86D8F438588AD1E966E4A7DABD0BA4A3907DBEBD2B2599718B885730BC1D7491A0E2FA468F29843EBE5E551E0CEFD21F85568D6A77B8515EF42908360B57CAD79B0004724DD117028BBCEAADB514D45D76FCE77FABFA64C11B9AC2BCB830F77C79FCB67A71D669BF7DF2F3B7F5F9FAFAA9571C1F2A03892E6BC576699806539AF01C1A840D6B168686C5F0874BBA3EDE49B161719927B5E32E4F19982D6D32A0453877F08C592EBCF7AA323F78A66EE81CE5297E3B81A01EF64FC9454FBEDC15C13A39F6AC8BB9A0CEB14E879181F72719F325B8EE60B3CF56CCFB42FFF90CF1FC987F4ACD7F052F137572809BDDFC6D6D0A7F082F9868F3492924F8B9BA5D924C961A4B9282F312916100AED0966523C6"));
  test_xof (&nettle_shake128, /* 210 octets */
             SHEX("9F2C18ADE9B380C784E170FB763E9AA205F64303067EB1BCEA93DF5DAC4BF5A2E00B78195F808DF24FC76E26CB7BE31DC35F0844CDED1567BBA29858CFFC97FB29010331B01D6A3FB3159CC1B973D255DA9843E34A0A4061CABDB9ED37F241BFABB3C20D32743F4026B59A4CCC385A2301F83C0B0A190B0F2D01ACB8F0D41111E10F2F4E149379275599A52DC089B35FDD5234B0CFB7B6D8AEBD563CA1FA653C5C021DFD6F5920E6F18BFAFDBECBF0AB00281333ED50B9A999549C1C8F8C63D7626C48322E9791D5FF72294049BDE91E73F8"),
             SHEX("83E17CB699843EAA6308F12943513AB4E4912203026F4653EF6711E5968CFD2CC135DA5B4E87707CD732F538B904148B0A50BFDD3F72155D85A22936C2BD38BDBD2B13145ED531E85F2E0319B6022E8F2115E6FA99E8D92E8AB418A84AED36D574B82A282099DB5FBBCA069589A3EC308E76B03C8F6C3C5CE11DF0D5635740384A22F9047C52F8C561A9848F0104167D8BE0A4BC47BB8A6474180BBCE3EE691147B74DC57FDF56420848DECDF3FC5A06648C1C42A0172FA02DF905F8318482F8189D5351751DE1ED5A63512AE6A4C456FB7D55F83142CAC479452879E71FC774C9BD9FEA4560ACB7786CF56FABBBD3881BEA980061623645BCF1D20448ACBF69FAD1030008BFA7298FDE63963386AF5C2B230E9E367B78C4B172B142C7EF43022C866E975F71128719DCBC2663D892C6C4D888E66C67D9E767AF52231B93DA9ED7895B4B015D9D84623C3008871B95382EC788FB58222C4DD8834054C694722248F4A7D0A77E4BD2A5C418BB5229C0E2EC9ED3E03E79E3F824C2C6E1EECC72A13E341F46F3E30085B3A3FA0C55F59C56CE150537E305C37263AED88D8EF9BF0CDF9978636E03E7DEB701345D9442725664FCB38EDEF923CC11CFB94AE152D0BAE72E09A813A3C25002FABA1DBCC0BF5102BC466D348E6027F5AD72FF9DC35D9C27C4C1E5DD39A647CF24269E6437FA114C74B6C1604B906C8F5ABA1AE06B1BD9"));
  test_xof (&nettle_shake128, /* 211 octets */
             SHEX("AE159F3FA33619002AE6BCCE8CBBDD7D28E5ED9D61534595C4C9F43C402A9BB31F3B301CBFD4A43CE4C24CD5C9849CC6259ECA90E2A79E01FFBAC07BA0E147FA42676A1D668570E0396387B5BCD599E8E66AAED1B8A191C5A47547F61373021FA6DEADCB55363D233C24440F2C73DBB519F7C9FA5A8962EFD5F6252C0407F190DFEFAD707F3C7007D69FF36B8489A5B6B7C557E79DD4F50C06511F599F56C896B35C917B63BA35C6FF8092BAF7D1658E77FC95D8A6A43EEB4C01F33F03877F92774BE89C1114DD531C011E53A34DC248A2F0E6"),
             SHEX("2BEF02AAAB7478A7F4B6E3C22AA5A979BDBA90CE91A42782E785AD6D06F82EF0017B54C408CD7027FB3057BEAB1DA1B150E54194BDEEB87F65EAEF6754AD8E00EC4B3F98ED2E64B85A3907B353D68614515A2A2D110A1E3249286377446C5339284B3C3EBBF8D841D49A34F7C79C935A69CE342569E173D87DF08B332F45C42C9B318C6B02F4550058E2E0697F1E1E76DC0DC78F1DEAF7A44BBB58152964DA263F633F09FEFE2C62F4A482209FC9715B0C424221F7BC81C67F438D920E7D9CEF3D98BAB3A27AB5E136439DA3C5B8774C01E6906416F0D48675751174F09DCD2185AD246CDD351414B9B5591E500B29AABC046F048BBD5776DA155B13A5970F26B393EACB372BD28B4934A6F252BC45F7217B29FD77A1DECCB20B060C9858A98485671E02ABC45DCF956255E52B5D21A59902A33425BD6853D56C18000F5D637CDE2F109EC6D9FD3C5427E6BC2468A71CFF6C57476FB1A850DF304645C0E64F969A541611298E1B19F95EA628937C2ADFD563FA309838EE470FA4810317177D61ED849B9FEA54410C877B1C5C59625AE5314E823CBC1D3DD79BC1B6498E22F7D15328EB55D0823AA19F0922BE4D6F545B8A2256EE141EEBD18FD1644368AE4DDE10482E8758E4865FC1845CB50155F574E177A3B0A446361801C2847FEDB035EBDBC0C7A67D216A4EAA7E04862D6C09D3D04693DF25BD2E3FE0A43AD80A9212DA"));
  test_xof (&nettle_shake128, /* 212 octets */
             SHEX("3B8E97C5FFC2D6A40FA7DE7FCEFC90F3B12C940E7AB415321E29EE692DFAC799B009C99DCDDB708FCE5A178C5C35EE2B8617143EDC4C40B4D313661F49ABDD93CEA79D117518805496FE6ACF292C4C2A1F76B403A97D7C399DAF85B46AD84E16246C67D6836757BDE336C290D5D401E6C1386AB32797AF6BB251E9B2D8FE754C47482B72E0B394EAB76916126FD68EA7D65EB93D59F5B4C5AC40F7C3B37E7F3694F29424C24AF8C8F0EF59CD9DBF1D28E0E10F799A6F78CAD1D45B9DB3D7DEE4A7059ABE99182714983B9C9D44D7F5643596D4F3"),
             SHEX("824271A836FE959710453742E9D957981C2CFA672894C8A886F57F79608DD62FA4310FC0582528E15CA09C7A92D1084D45ED2A3020A9A2A10C2355FDF052EBAA1007B0C01E1CA19D95B1B05C03167923927AC26E33ED6A823C72D105A2B35085926BDED8E59A49CBD7C606116BF560FAC1364C7E5860A919167E844086D648BCBDB5F148CADEEBE2CA2D2C34E9BDE155DCC9CF99DB22785D8F5327AF41BA93AA1DD59A25F26615FD6EB6D57FF6475FE10BEDDECBD16CE84096CDC6AE5A6B7FAD0BFDF6E8CFC974B082A3F0C6D8C5B25295A81BD2650446E7447951F7002FD2F417F4F553F2733D294FF90C629EEF9FAF0B39DC9CC4EA6B50D438369FEC506BBC185D37DDA2EF1C6E7079B09BE893E6C1C62C24E6296C8CAD9921DC182F23C027CB753212E3EE41C401396322A43F4800C48ACDA4A6198E40B76F0F6CE78354BCBCC33017FB30C33D7A90AF566F4BD3BC2CBB08E800BBC7F309D749F9D2A352D9B284C358FF2109BD2C2CEEC6E85BC16B83A7D666E582C3E3D39A6A92C196A11A397203F9C5613E2ECC8B2B91C229FE785040A17485B3DBAD2B155C270AB2A510CDA80E99E1A79BE01C7262C217DDE69D068753F9E44A3B035E10BB7AA53F397D7A1EC95342DDC973CDDE9583B8348D832B976B255362240C5927D24A00050F970334AA404C531D93621516DAB81E0FD89611C238D85454C6D77F1AB5D7B2C5BB"));
  test_xof (&nettle_shake128, /* 213 octets */
             SHEX("3434EC31B10FAFDBFEEC0DD6BD94E80F7BA9DCA19EF075F7EB017512AF66D6A4BCF7D16BA0819A1892A6372F9B35BCC7CA8155EE19E8428BC22D214856ED5FA9374C3C09BDE169602CC219679F65A1566FC7316F4CC3B631A18FB4449FA6AFA16A3DB2BC4212EFF539C67CF184680826535589C7111D73BFFCE431B4C40492E763D9279560AAA38EB2DC14A212D723F994A1FE656FF4DD14551CE4E7C621B2AA5604A10001B2878A897A28A08095C325E10A26D2FB1A75BFD64C250309BB55A44F23BBAC0D5516A1C687D3B41EF2FBBF9CC56D4739"),
             SHEX("3768C48BF02D0DB9AADAFFFBD228A4119AC6B7A815D9C74C714EAC4B4BA4A062F7E9962B98C361A6B596838C671C95892FA4E82A95C60B9E8BA787158808C2C00BEA3BED0AB4846FD57E638DD29B2ED47DF5CD19699329626F6BDDD14BB709417A972AE8020D05AADFE12034A98F9121A191926566C499C536B2505AEBB97C9C49780E5E0007EE5596717E7BE42D40DB322BDE5B7FD227B15987B599C71F1F20EB8BE722DDBF3C76F9EA58B490EB11395BEFA02C259505F038B40E312B5AF1519BB6DA9E99E0214C2249A1E8C534D768A7DB9EF46970A64C6E5F746B7C50670CC8E7765C4B55B26F3A53E4C3178F9599435AAC9B96A203CD04988D1D95E1B5C29F632ECD66204B7BB2B7F776F2DE67DCBF7F0A81FFA1CD18506B8FD9DBBBD354E3CD388F21EDEC9A4EB319C766EBA3E1E8565B0CCEF3698592E19081472004AFE38FD8B17E6D1A5AFED52EE44990598CE17E09DED65965A380E3452CCE6D053F22F1E8CD24A76C74F6A69550C36B43890C7E239658B7DA6DA768869D3D96333BE7422AB7C2B5D90B2C3973D666B4E399A2C5F014566E3BA919AD1964A148DA2D35476B17799B82EA54AB6CA0BAA098D0734081C7D52B7DA8150AB3C1B8DD1174418700C46036E3918E5A21E70679D72952025FB52E9348509B4F10F412A8C8C11EF59D57A3A77C5462253C87475F38911AD2B08732DBDDCC6D6A1B438896327D"));
  test_xof (&nettle_shake128, /* 214 octets */
             SHEX("7C7953D81C8D208FD1C97681D48F49DD003456DE60475B84070EF4847C333B74575B1FC8D2A186964485A3B8634FEAA3595AAA1A2F4595A7D6B6153563DEE31BBAC443C8A33EED6D5D956A980A68366C2527B550EE950250DFB691EACBD5D56AE14B970668BE174C89DF2FEA43AE52F13142639C884FD62A3683C0C3792F0F24AB1318BCB27E21F4737FAB62C77EA38BC8FD1CF41F7DAB64C13FEBE7152BF5BB7AB5A78F5346D43CC741CB6F72B7B8980F268B68BF62ABDFB1577A52438FE14B591498CC95F071228460C7C5D5CEB4A7BDE588E7F21C"),
             SHEX("EA94DA7FEDF4B2296823EB57F0FD81468F4476E6F9035043F84BA952015AF88E68233219E4D31D809A9B13A703EE6E99C230E1E037D777FDC6B4BCE3F81FF2E7A15B9DAD86F4816D1ADF0F5C34E6EE2FEC26353444EFC417BABE16C45BDE0CB3B06ABD19F354F5BA5298CD486DDA4C6216D4826C3EB1B221C24A63BE7DC751F8AB5458000C28937809359A4C49200949048156D48D537466EFFD085610F221454ADDD95519071C8A671EAAFBA7B6F894A3274599C76BF0964801FA38B601869D4C04EE9FC6AC7E54B4A6D8522912F5F3F2A7C6CBEC20FAF1E164388D932C5260763457D27525288BE09ECDFAA1A892B99AD0DA9FC1CB3D60B1B85DC953CE9B37711AE6BB82A1F96CF2479155BC3B328803D1EAF56EE0F02223BF167FB33E6E7190F41A121DE59B9FE9790C8FBBC2B1B77ADD9C316D75A7F4DBDB52DA2EDD9D2356C6AA77A1CECB5330ABC382A748C8972FBD78315DD2A4ADDF335D18D5C97547641F6B5ABC4C5F166265D6934C77879A5B4CADD27A8A7F9F817ACB1307A88970D29DD929D5B03A71D5B863A997C84D01B58DBA539CCF6693B6004812727B254D22480BD50C5E07F162DB59EC112E1DCFF813BC266F704357202AD0723B373BDF49B1EE3D4E24D942D2E8F95E41FF496B9F4F535601980366FA79B662507CAF88AACE17ED6498E6DDB65C799EA698C7B8A49BF48021CF3B41E96225C43481B7C1"));
  test_xof (&nettle_shake128, /* 215 octets */
             SHEX("7A6A4F4FDC59A1D223381AE5AF498D74B7252ECF59E389E49130C7EAEE626E7BD9897EFFD92017F4CCDE66B0440462CDEDFD352D8153E6A4C8D7A0812F701CC737B5178C2556F07111200EB627DBC299CAA792DFA58F35935299FA3A3519E9B03166DFFA159103FFA35E8577F7C0A86C6B46FE13DB8E2CDD9DCFBA85BDDDCCE0A7A8E155F81F712D8E9FE646153D3D22C811BD39F830433B2213DD46301941B59293FD0A33E2B63ADBD95239BC01315C46FDB678875B3C81E053A40F581CFBEC24A1404B1671A1B88A6D06120229518FB13A74CA0AC5AE"),
             SHEX("73E672E46B274D309899FF2C8186AF4A9F42872135309C85621347C55EDF8D3B5A7DB51B0242618AB6D750EF75CF816C23A0DDFA21ADC9976CF47A6957E914BAAB02E4C2DC809C0D2F67FE759E92D7755CE965EA4B4045D6172E1A0C749B4498795FC375AA6894118A1BE2821A7878D25F597633CEE5576B3D15E23CA7D664709ADA20E3CC182D3004B102BDCF5566AF522C5511049FECCF80D4A6C21D7CCAD1B24DDDDDDA0DAB0C2FF40E056D0A715A31DD35679FD4A4CA8BDD17293F7FD63319BB5B4ABD3698D574BA5D36445E7FBDA9EAB049F6A8B277C0593949DA84805036B776368566F622DF37FD3D42EFD06EB1D5C1C2847717F51D236D4317498514D95E1F4587669CE819DE249EBC8D05D55E2C35DBBFF3C017C7A8D55B1B184ECF72F288948027E2AC2CBE6589B351437E185A888EA5AE7219B9121E438EC3CB397133B75330B1418AF085E7B42FB4C3B0F41A95BC6583CA7DED8A470E2CEC237013DF9CBB866454F276648B1DC0BFDC6BC5346348735764F19F08905F1D17FDDD454BE1B0E8DF34879B4E7709BCA044AD49A14F9EE971283F2A20CF6377C648E9D85599FCE482697B8FEB218418B1EF0979935E0F88EA5A2D38506955532A0B9DA8AC1E7641622DCA9F2E31642AA41DB7FC73328590B69459741998AB90D796293999BA8A2111DF20256D2FE50AC6B0BCFD5EDD61C260512EC357E844596EFF10"));
  test_xof (&nettle_shake128, /* 216 octets */
             SHEX("D9FAA14CEBE9B7DE551B6C0765409A33938562013B5E8E0E1E0A6418DF7399D0A6A771FB81C3CA9BD3BB8E2951B0BC792525A294EBD1083688806FE5E7F1E17FD4E3A41D00C89E8FCF4A363CAEDB1ACB558E3D562F1302B3D83BB886ED27B76033798131DAB05B4217381EAAA7BA15EC820BB5C13B516DD640EAEC5A27D05FDFCA0F35B3A5312146806B4C0275BCD0AAA3B2017F346975DB566F9B4D137F4EE10644C2A2DA66DEECA5342E236495C3C6280528BFD32E90AF4CD9BB908F34012B52B4BC56D48CC8A6B59BAB014988EABD12E1A0A1C2E170E7"),
             SHEX("BDF536D8B28E91F00B710DECD109897B4E96882AEF06337FE83B053EAC20CC54523D246C41C9DE3115870B82A1BB2EB2B1E02CCF7A4E607BCEA3DCAFCA38F237DFC72D447CF8C544F541C1B53E7769B0CD6F2EC6D756891DFC818323CC1E1AD565F9130F59889798E418FE5FEB176E77F5FF8F10C6ACDDBC59C8D084AB19DA2EE5B737E537609BFA6F81DF1846B033663D586CED1F4E86D99131D37EA5807881D2DE749803E333AF2C360F8EE9FD99A166F3FB8F0C50BEF02DF50EE615C08E88299A1F3CDDF290A3BC9E4357F3D93D76EC5DF462175EDC5DE421A6BA456EA7C42ECEA24EEC74011132C27FC63F1E969B474438222C7DE33762F5377C197DDE9AFA9912900FF25373EFC3A3D023E091350D0D269BCBFF64EF35E2E1E1C300214C3671932CCCAD7C871FEF4C969B34CF6E8B0881C2F7704B455F118E3BD37A74BE89354D8C8DFCF649FA03701AE17B705A865C7CE9ED81E4FB374EB643294356EED48B3579315BCB32CE6C7F302ED5BCD68F2183110EF7FF60AEFAC6BFB8252BC66C0F4026DC37EBE4D9783CA7459394E929B8A83FD8378D9E097AFCB0A7163F16E5579E8F5546898C4B521264CAA006B3FCCDF46FFADC1925752EF47DA30F354D66F763AAE46A0BFB385FFBE153641569E5E222F374F8E21EE3EF8D42663A4C424803E7966F2F0A1EA723A06C92AF2393C8FA6711D89514C20B6170DCA448FD27"));
  test_xof (&nettle_shake128, /* 217 octets */
             SHEX("2D8427433D0C61F2D96CFE80CF1E932265A191365C3B61AAA3D6DCC039F6BA2AD52A6A8CC30FC10F705E6B7705105977FA496C1C708A277A124304F1FC40911E7441D1B5E77B951AAD7B01FD5DB1B377D165B05BBF898042E39660CAF8B279FE5229D1A8DB86C0999ED65E53D01CCBC4B43173CCF992B3A14586F6BA42F5FE30AFA8AE40C5DF29966F9346DA5F8B35F16A1DE3AB6DE0F477D8D8660918060E88B9B9E9CA6A4207033B87A812DBF5544D39E4882010F82B6CE005F8E8FF6FE3C3806BC2B73C2B83AFB704345629304F9F86358712E9FAE3CA3E"),
             SHEX("B5BECCA38D903F2D87CFBA9A80C51D4A0733674C78EF85BB236E8FA64BDED9BDD01DB1905E46F4210BB65BFD26965AA3ADDDC893087AFB56089083C72F7649CD5987F449F9668CE66ED21514C05E7F09F52D84FDE30E96353D48DF0208415DB51EA5D120A7182DC9458C471127253ACA05D52FCC18B73CA1C3A5BCFF4D3B6A41C450826397D9B38AC5C8E301C7A4077EDCDB6A46B3979128F20B1080F7768EDC1F651D1D122F97620BF2B4E9FD20189BB331FF9026EA58D7203FFC49E901FDB9233255F349A19AAEDA9EA1E4F3C45EA9C54F41CE1625C0EF02DB370909949BBDFBAEA49FEE3E3CBB74E1807F553A2000A3C753BC5D529B4E25154E2C86F755645304F2D18ED0E62B62C87931DBDAD5D1CC9CBB64430D56AFBE7D4C708EC992EAA8E13C5B3CE0343C30AE3F09EA3ED5C0C933A0C76F0FC18FFD4BAF93BF95B2C1023DC87D4D641EBC6E1CEA6E756F45FD2E58E0562F43389A1048A8BC12C8B3B19E05180597B5CA79B7531C6B8FCAB88AC1AEA686494B98E211C975B48B7157F985F25B168B93FEA82218A227D0B89BEF905A5F13EB37438831C2CD871FD8A36A4EA22879570B22C63B67BB60EC1E415F4654F6A3EA5ACF74DF41FE09DE398351FDEEE499CD98DB89B320A58EE62F8569E10DA4C68B9B61620E9D51A343D06CEB484CA532AFA8FF141A4ED8EAC22870D72F5E1085107781D2E422B3EECABF88FC"));
  test_xof (&nettle_shake128, /* 218 octets */
             SHEX("5E19D97887FCAAC0387E22C6F803C34A3DACD2604172433F7A8A7A526CA4A2A1271ECFC5D5D7BE5AC0D85D921095350DFC65997D443C21C8094E0A3FEFD2961BCB94AED03291AE310CCDA75D8ACE4BC7D89E7D3E5D1650BDA5D668B8B50BFC8E608E184F4D3A9A2BADC4FF5F07E0C0BC8A9F2E0B2A26FD6D8C550008FAAAB75FD71AF2A424BEC9A7CD9D83FAD4C8E9319115656A8717D3B523A68FF8004258B9990ED362308461804BA3E3A7E92D8F2FFAE5C2FBA55BA5A3C27C0A2F71BD711D2FE1799C2ADB31B200035481E9EE5C4ADF2AB9C0FA50B23975CF"),
             SHEX("EC7A09A04EC20DE5EE683681D42177715F85562DF50177CBE122DDFB9C5E81EE349D3E2A18726BE2E46287759D6D28398851ED22AFA78BB4B3A41C76F7006F785887D2D030428ED6ECF7DECBE20BF09B1A321B96616F9D1F4ECC90BC2FED1C30C18BAA23455303074F10ABA5BDB596F3BB5BC1FAA89589A92AA95DB198915C1FEC3420D6A4A5E79808012DE71B41CC7A77AB96821B5BD1E822A06E698101E7E4DE1E5E7CCF9093992C0B62419A660E1AC4FEE0B1D0C4735416C18570697220ADBBCA56289C2475E17AC6FA89E96AAAA8D5F174D6D9A3EDB4A6BEE4B606D0B789F942A1F8D5BAA758AF6F6DFBBE59686FF6A8D625302931C34C8D908BBB0C5269ADC95715F90259384A0F88B6BAA1FDAA5BD57F5FEB2FE0B968120AA0206F911D211C2D77CC465BB6E1839C0271CF55F12658A1FD1F2F4538BF7E9FD784E4C12AE95A5A2967D2847EABE150F2E113D8542B9A7CAD092B3D86AD4219632C5F3740EEE45AACA529D83F0BF3CD656384848AD577CE603D31B70C40A55DB571868C1F5B7ABA3CF853CB0ED682BDE964C3051E7E81D4454906DE1966491E1CAF7073035268311BC1CD3590656B1321D8D6885D15C0BF840BB40C0FA4D127878E422DFCA7C3BBE8CA4447D5228F83E453C94002C43F5D87E989627F89CF5B6057DE6D86D2A82ED982035519F47807CE6C61DDCD9160FF0DDB3BFE08ED966C08317E4EB5"));
  test_xof (&nettle_shake128, /* 219 octets */
             SHEX("C8E976AB4638909387CE3B8D4E510C3230E5690E02C45093B1D297910ABC481E56EEA0F296F98379DFC9080AF69E73B2399D1C143BEE80AE1328162CE1BA7F6A8374679B20AACD380EB4E61382C99998704D62701AFA914F9A2705CDB065885F50D086C3EB5753700C387118BB142F3E6DA1E988DFB31AC75D7368931E45D1391A274B22F83CEB072F9BCABC0B216685BFD789F5023971024B1878A205442522F9EA7D8797A4102A3DF41703768251FD5E017C85D1200A464118AA35654E7CA39F3C375B8EF8CBE7534DBC64BC20BEFB417CF60EC92F63D9EE7397"),
             SHEX("E65D36FD64B13DB62CDF8640E6A217CD4952BE909BE1119B04CEB6C8A71550E953BE0ACE373231964D61B89A575AF582A66D7BB1441EA399D59A7C585D762433D64F44C53DD9B8FE8F357A26DC66E04B15B9FFD0FC7C1B4ED07DB3E4C80A3527602DD16BAB1F8ED85A821056F97FE2912411F7CF6BDB9070997CA4D5BC16B0B26159F7F3CFAC7252241C079633CD8287CC41FB3F7CB6185FCCEE1E3467E90788B9E943C936C1C82115EB5B5A1AF3FC6B0104959E98E0EBB0F72FA02FA5745380952A6C2EB0AB84CEE35661F34CF0DE4E6985BAEFFAF545D48ADF065F13B9272098D5906DA5C8B688A3C922AA74D840DD9C1EE31BECABC417EA07E2C45A90E1AFE588E65109C8DA674533D506E17EBA1ABBC25DD84CC79A4CA0E15013D9D014DACC5F696A7BECAC187F0A7B233D59A7BA6821B3EDC71C8073F2EE0CAA8C9F96032AEE6D9FBC15BED15E879D0F497BB8DE468B4050FDDD15C97AF4413C4CF60859EA04103A13704CC9B7076359E28A1B90949166F5B11055B415EC8B9A552AC94E4D1F45403AF5EBAC0762D1ED0967AE032DB30DFFC8132942F45626C7E50BBF80DDFAB80103E423BF00AC575194489BE02624BED16C4D1FDCD0011D0C64FAF6BB6FEB21E0F0653F05D8FB69CF891347DFF0A39136BB5D4D20101CB3EE900B508A136185FDD0FF54FEFBCC6C069E1A433742D51129A1B556CBCDB8A7E76DC43B5A"));
  test_xof (&nettle_shake128, /* 220 octets */
             SHEX("7145FA124B7429A1FC2231237A949BA7201BCC1822D3272DE005B682398196C25F7E5CC2F289FBF44415F699CB7FE6757791B1443410234AE061EDF623359E2B4E32C19BF88450432DD01CAA5EB16A1DC378F391CA5E3C4E5F356728BDDD4975DB7C890DA8BBC84CC73FF244394D0D48954978765E4A00B593F70F2CA082673A261ED88DBCEF1127728D8CD89BC2C597E9102CED6010F65FA75A14EBE467FA57CE3BD4948B6867D74A9DF5C0EC6F530CBF2EE61CE6F06BC8F2864DFF5583776B31DF8C7FFCB61428A56BF7BD37188B4A5123BBF338393AF46EDA85E6"),
             SHEX("2B6D5B37DA462A050946D447375EA4DEC9309EC30FB6C7FA68F67AEB7B6E1921A74FF35ED31560E695499F2A5AFCE3E900E1C6F199373951A406A8F57DEDDA8C9B9D2038E787CC54B62CBD97CD4247ED933DE1BD68FA670158701F088C6E785E8DDE870306B80DECAD2DBC40A8C2F02630499292A19339E4D63351EE5A44E36FEA062D2253674F5531B1C51FAF694A0E963859B26A11CE890A3E5C03815BAC2CCC43AAC806941BEFA00CC6BCE72FF59F60C85B5096BD7E9D4B60D23A050D3B4333A26B0CB6B69E62F33E87E65FC1562E5D1384F439D717A940E2A291897A4EB3F941397EAFFCF4EC738DC643E72721B5662F698BDD502ED1A82D924BC0D780A049F7D965BB585D0F674A06FE03F0B18AE56B7AEA705328072E4A374B7B9715882BD73C34E99656EC073B653C2608A04D84ADFB4607B62B4B504E43F07F9D4555935E3D432DDFF23D9AAFC87C743C1BF8A91EA4671A03E5461CF13E76E3C1779CD97DA4055EAB8E705556EB7538539FC6FFC1638E702CA2A60F0506693D54A35F7AF6FE0773A67767865B3F1397674EFF365B3849D261E29B160CDC91368899867689E599A85572464FD895CBA2BFB4ABA496F395E7950E064CA509AB8D8491BB193661E250F35D5E2AF12E1F981970390F456953A8FFDF72BF998D0FD7393C33FCBACE063C0D63F91E741F95FD542C6C509036824B2784F394E27F6BCE88B1FC"));
  test_xof (&nettle_shake128, /* 221 octets */
             SHEX("7FDFADCC9D29BAD23AE038C6C65CDA1AEF757221B8872ED3D75FF8DF7DA0627D266E224E812C39F7983E4558BFD0A1F2BEF3FEB56BA09120EF762917B9C093867948547AEE98600D10D87B20106878A8D22C64378BF634F7F75900C03986B077B0BF8B740A82447B61B99FEE5376C5EB6680EC9E3088F0BDD0C56883413D60C1357D3C811950E5890E7600103C916341B80C743C6A852B7B4FB60C3BA21F3BC15B8382437A68454779CF3CD7F9F90CCC8EF28D0B706535B1E4108EB5627BB45D719CB046839AEE311CA1ABDC8319E050D67972CB35A6B1601B25DBF487"),
             SHEX("E8B51670E0D27D9F7989447A97C07D8D142804F876E701A49E6C4A0EE149938881900637AC7329EEAB1F1E032DDC2194B2417856401060EAF019AD4D9DA3BC702BD70BE5FD9D9C9B86F4044AC441560E887AE21E3C3EF3F05E38C7F8B078D3A66C6FC8B98E33EA8AD1D029433626CEAE2468E3034CB44326C89D7A7188A5272C116D3783FA16AF98BA3070AC96C07E459F1127304483A2394837EE1A10020A79F8519795895F4786F5A15134DD44FCBD9BD1EFD0EBEE9CD991001FCCB6A92E490793B6F16471670E1D69F9CE49020C1E8E3B453CDC1A8498D8C510B8D21FA00239088E065B742C51257766F19E17B397362DEC9453F50328C5E14BF800396A0A3A1A957B29CB1C8FA059BDE39B995D45900B56E2C8CD2F4E8BAC756350DE38293D1FE4436489D9B23860E1AD35E2411D0210DDD1C3D8994635F29674A64AA5F92B2088F3E7B4FEC26485D93483AF96FC4C5FC9F34254E014DDBAB361DC55E0F5019BA9FF8ED7B104E09122A80BE3BD29A3E4C4C26A7D2B08F8915087D9A6A2E7888710A5B2B87448B81F889830E26E6813385B761FE97E3729AD04BCF08F536A152039F04C283DD36B9544C4D6CBA97E26B61E034259CAAAADCE508343DAFBD4DBCBCE25957406B22E13DBBBB186C18DCBD5B8AA18C86D8660698AF26997726FB7AF26C397DF8608708AD5D6524001634575A2F1070A5EF72D327E91C7184B4A"));
  test_xof (&nettle_shake128, /* 222 octets */
             SHEX("988638219FD3095421F826F56E4F09E356296B628C3CE6930C9F2E758FD1A80C8273F2F61E4DAAE65C4F110D3E7CA0965AC7D24E34C0DC4BA2D6FF0BF5BBE93B3585F354D7543CB542A1AA54674D375077F2D360A8F4D42F3DB131C3B7AB7306267BA107659864A90C8C909460A73621D1F5D9D3FD95BEB19B23DB1CB6C0D0FBA91D36891529B8BD8263CAA1BAB56A4AFFAED44962DF096D8D5B1EB845EF31188B3E10F1AF811A13F156BEB7A288AAE593EBD1471B624AA1A7C6ADF01E2200B3D72D88A3AED3100C88231E41EFC376906F0B580DC895F080FDA5741DB1CB"),
             SHEX("120CA7E361A65A0985563012CB908028105ABDBCB51CD8EACA141662B0C184E9B2D6DFCDADED7D7CBD42455B3F5A9F15596D2BACE19510256275209CC9899A87C6224DF8E919458EA46A35CBC8D66A488069703CBBDEEA6D767CC315FBF518F7B0A0154448F8D7C56DA1B116730A76954C288B4891B56FED41B2EFDE6C27DD4B3DE9BCC25A2A901D4C87975E82372AF173982601B605D845DEE7BAC94EB410B369BBBDE500D458F9F8A3933D3BCADBCC85EA05ACEFC9A7A77E26E28E82DB157EC1CB2C99C272F33B1B2685533CD988DBDB6E850BD3C71E0589A165FC236A65B1F1DB750A3FDC1392A76E7714469B47E20D8946AF7D68EF9B16109178F662320D60A0D843EFCC5B1E56CF6D8FA8B649DDC28E58983ABAC7B2CE019AA0E07B387D8BA46CD3FAF47B522D766971BA603C84684BB094E38D82EBCB108CE0353E38FA0A4D7294F7B7BAF745F3E036EC0231F6A194585FF7CE1C792E4879BB44E4AD65717952B080BEAAE1FD0CB8AE87E6F130F91BB06F95E54FA924E2D0C42921D7CA9965AE70A3587F12017CF1E4CB70D1704A51EFE27DFDD919C7FE70F967CF7044C62D6FF232B6680278E9498F247AD92D5E9491DB0F00D583C019142A8EB63678231E1273F0A263C57B98E0AF650FA207D4B21650417DDDB40830F89E15C85E934FC745AAD87BA35DC16B7A82CC97926494BCC1FEB4ED43D75F518ACC22D208D3"));
  test_xof (&nettle_shake128, /* 223 octets */
             SHEX("5AAB62756D307A669D146ABA988D9074C5A159B3DE85151A819B117CA1FF6597F6156E80FDD28C9C3176835164D37DA7DA11D94E09ADD770B68A6E081CD22CA0C004BFE7CD283BF43A588DA91F509B27A6584C474A4A2F3EE0F1F56447379240A5AB1FB77FDCA49B305F07BA86B62756FB9EFB4FC225C86845F026EA542076B91A0BC2CDD136E122C659BE259D98E5841DF4C2F60330D4D8CDEE7BF1A0A244524EECC68FF2AEF5BF0069C9E87A11C6E519DE1A4062A10C83837388F7EF58598A3846F49D499682B683C4A062B421594FAFBC1383C943BA83BDEF515EFCF10D"),
             SHEX("F0715DE35692FD70123DC68368D0FEEC06A0C74CF8ADB05DDC255487B1A8D4D1213E9EABAF41F1161719D065D794B750F84BE32A3234B4D536460D5520688A5A79A17A4BA8987FCB61BF7DAA8B547BF5C1CE36B56A73257DBBF1BABB64F249BDCEB67BA1C888370A963DFD6B6A2ADE2CEFD14C3252CB3758520F0C65F4524682772499463AE1A341800183AA60EFA05118A28201744F7BA0B0A3928DD7C0263FD264B7CD7B2E2E09B322BFCEA8EED04275795BE7C0F00E113827370D051D50269580300005AC1288FEA6CD9AE9F4F37CE0F8ACE8BF3EBE1D7056255954C761931D3C42ED62F7F1CE1B945CDECC0A74322D7F64D6004FF21684149307288B448E45433475B1EA1314B00F1FC450089A9D1F7710C6D7652ECF654F3B487D0283D4D8A28EFB5066C4250D5AD698E15DBA88E925E4DE99B69BC383AC8045B7F1022ADD39D443546AE0924F13F4896096DFDF37CA72207987C4A7705A7ABE724B7FA10C909F3925449F010D61E207ADD95219071ACEEDB9B9DCED32A9E123561D6082D46AEFAE07EE1BD132765E3E513C66501B387AB2EE09A04AE63E25808517AFEA3E051169CFD2FFF8C5858E2D9623897C9E85175AC5A86394CD0A32A0A62A8F5D6CCCBF493DAA43F78362BBCA40ADF733F871E0C00998D9BFD6880656666CD7BE4FE9892C61DCD5CD23A5E4277EEE8B4AFD29B69BBA55660A217112FF6E3456B1"));
  test_xof (&nettle_shake128, /* 224 octets */
             SHEX("47B8216AA0FBB5D67966F2E82C17C07AA2D6327E96FCD83E3DE7333689F3EE79994A1BF45082C4D725ED8D41205CB5BCDF5C341F77FACB1DA46A5B9B2CBC49EADF786BCD881F371A95FA17DF73F606519AEA0FF79D5A11427B98EE7F13A5C00637E2854134691059839121FEA9ABE2CD1BCBBBF27C74CAF3678E05BFB1C949897EA01F56FFA4DAFBE8644611685C617A3206C7A7036E4AC816799F693DAFE7F19F303CE4EBA09D21E03610201BFC665B72400A547A1E00FA9B7AD8D84F84B34AEF118515E74DEF11B9188BD1E1F97D9A12C30132EC2806339BDADACDA2FD8B78"),
             SHEX("2E27811D2E132C58A1D053CCFAAA2813B620554C49C31CF208964CFA18A3EB5724D4EBB37E37398CE9201CA33C5F6A9409F262AFC5EBF0970BE0618EC7E56C15C13B2F0270AB5A9536FADEB84EB2D934B989354D47FCD1DBFB40FD96485FAD8DF7C6E62A60866F154901E4427C2B5E6059C76224589A6BE352673430740903E4225C1379561AE5D361B67F3ACACA9C7633CCCD2601ED278B5C7D6A378D713BC549E2797CA34FCB0A00E649AC3A4B59001E5D8E1E2785433ED65C76F4FEC4B0579059DD4DA49BBE678279F534B84A280F89DFBDA09E09C073EA11AFCAFD5685A714EC31E8F86B49D2C97E6A2A66B4F2AA76D9DF2D27CFACC2423AA18492E2F4EF6A9F8971627839375307F45A8B26133FF1D02F100EF05DF168A2BE02F0C17CCB90F23A0FBC5F536C410923B154E81A6557935E2572165A7E39F64F57E2B8CED8D2CD1025EDD5DEA6DBDACC8860BD6A0C598EEF14619D4BC11B6AEAEB5D10BFF3F9D5B20B7F2AB8254DB0C1765A10A125E98D762851F05261DAE06B224E2076112A62B4C3F3A56074E01CB3DD264CB1F4B58EFF2A1C9ACC4764A68680A40FCACC719286684FE0A742F13B5BDAC8AABFAC68174C7E153796C240279B0ACC719158D1FE4F78EC3D47EA8B8D3A1B3A6842970FD439C09096E8B032E551DBDFD0EF86DA0537F4E72F78368B319B6DB735AFFEA80633F2F8EB9D26C8CFC321C7C346F7"));
  test_xof (&nettle_shake128, /* 225 octets */
             SHEX("8CFF1F67FE53C098896D9136389BD8881816CCAB34862BB67A656E3D98896F3CE6FFD4DA73975809FCDF9666760D6E561C55238B205D8049C1CEDEEF374D1735DAA533147BFA960B2CCE4A4F254176BB4D1BD1E89654432B8DBE1A135C42115B394B024856A2A83DC85D6782BE4B444239567CCEC4B184D4548EAE3FF6A192F343292BA2E32A0F267F31CC26719EB85245D415FB897AC2DA433EE91A99424C9D7F1766A44171D1651001C38FC79294ACCC68CEB5665D36218454D3BA169AE058A831338C17743603F81EE173BFC0927464F9BD728DEE94C6AEAB7AAE6EE3A627E8"),
             SHEX("76A498F8112B364E0DB20CB4E7AEAA1C96400A3E1EB1A24ED865203C4A583CE5CA10B47E725C362696F0ADE5FD75058D6882B435849030868486318E566BE51B6D985A76B36D98D893A04CC715CD26C412BB4CCDC0DB6D1B13CC071FBFC83F114C72D3B54CBCFE57243189251A7C0608BCAA1409B243F2652CF5CC4386CF4551D2955F371591C0A04CBFDA2736A514E773B5EE8C5CFD820A7542CBA65245F729E650D085B3F60EBCE2CEC0D8C952ADD51A664788B3197952E7D0D9FDCCA35EE118BC0144688621DFB373C2BC1DE7CA7BC751D539276A119ECAE8AE2987E018E9FD70C632F2A54701302C12B971E0EE5308378669EA10D0E9A7B955BE1C1BB9E7FF6857C39EB6B074C061F28D1979B33AB8D2F81078325C9F7966806D2CB62903FBAF4C12AC4AA15F010034080571BEE9E863C1CB89CB01B86643DB1D1A6C224934A79D969616CC3F9C1382B81EB83A8CFA2CDF20BB03CB67DDAFCF9CD827D76A0661B85E82664BDCED0EF34D2F807AD956724EA2CBC511632D2C9E9F8E4FD71E127D581CBBB39782BE58902927B0C0D922493255F805C5CC78A96AEE84ED086AA93F8FBF7282D78DAA0B9C126216BD929EA4A4E53F78088CA154E9A63B3227A25705532F699606EE73A5FCE6E6C94DEF1324FF2E0EDF6807700DED2F088E2D77B7F690E646E7A73CA05FEBB2634D862D7B21AE2ACFE3A5B3DC6ED1071B0F3E5E"));
  test_xof (&nettle_shake128, /* 226 octets */
             SHEX("EACD07971CFF9B9939903F8C1D8CBB5D4DB1B548A85D04E037514A583604E787F32992BF2111B97AC5E8A938233552731321522AB5E8583561260B7D13EBEEF785B23A41FD8576A6DA764A8ED6D822D4957A545D5244756C18AA80E1AAD4D1F9C20D259DEE1711E2CC8FD013169FB7CC4CE38B362F8E0936AE9198B7E838DCEA4F7A5B9429BB3F6BBCF2DC92565E3676C1C5E6EB3DD2A0F86AA23EDD3D0891F197447692794B3DFA269611AD97F72B795602B4FDB198F3FD3EB41B415064256E345E8D8C51C555DC8A21904A9B0F1AD0EFFAB7786AAC2DA3B196507E9F33CA356427"),
             SHEX("760DD76E50C13B411D04ED9D81078F0E5B9FC5DF9523F7C2AEEB418CA328E047071852FC5EC2F96C2133C4AA7C68306FFD70A782490F92812E04A3343B9502C14B781A4800A8928956EA457A3CFB94A63BB4407524DCCDB6136C5228FC1CD426F75E1701F0AB7D7ADA8079E4761FCC4DCC72155B4B27370A7EB6A5E3995A729AD8D8B5356A084B0678B114078B0D43C657A23F09D0862D606A8A485FED68EBBBC63A126FBABDF36151D5A39BF17632B3D967983FD62E19C944400F7A68393603243CE499947F3EDCE7F30206BCBFF83615DBAB7642C5D82DE050D67C93B836B554949B96947B7D1534E0755859ADB7BA5AC9F519B18C3AA66300ACF9A26849665810250004529334169211624C6A63350B392F6DC6753D59A03997180980EE0E742388A2BDFCA619524DD16250800918E95CAB914EBBFB12232F1687BE0B46426E126787E394A91F775FA91EE1C5D57AB1FFA151D554B79DE8EE9A937AAF5C98497AAA726CF01A9DBCE6E57A07F8D098B2844ECEC9856D3D8B189D016B34C4AB61AA3191B3E4C2B44F6BCC9444E6093A56545230FCFAD19A2CB124ABF524689F4F665F2E84D10A2C93B8CA412B6D280C2680E946A6C9225DB6F97A86FA3415A84D52FF3A0A15DC7CE97738B0B2DC1430BD56BB71AB89EE108B5BDB706C7919D5EC67C487D9C8DFA6D2333C59248104F33F4C162A5156A3A666948E3A6DA13D0D"));
  test_xof (&nettle_shake128, /* 227 octets */
             SHEX("23AC4E9A42C6EF45C3336CE6DFC2FF7DE8884CD23DC912FEF0F7756C09D335C189F3AD3A23697ABDA851A81881A0C8CCAFC980AB2C702564C2BE15FE4C4B9F10DFB2248D0D0CB2E2887FD4598A1D4ACDA897944A2FFC580FF92719C95CF2AA42DC584674CB5A9BC5765B9D6DDF5789791D15F8DD925AA12BFFAFBCE60827B490BB7DF3DDA6F2A143C8BF96ABC903D83D59A791E2D62814A89B8080A28060568CF24A80AE61179FE84E0FFAD00388178CB6A617D37EFD54CC01970A4A41D1A8D3DDCE46EDBBA4AB7C90AD565398D376F431189CE8C1C33E132FEAE6A8CD17A61C630012"),
             SHEX("76BBAC5E72BFA9B5952D4FA8313C2C8995F1B762B0855EA28417A04CDF1BC9A72038014F4D1AF37D6338E217DAAB8993CFC222510A869403DC46B5D798646BB76ABF4006F31BE334DF8C6DA07AB8C6E6523DCAB9367E6B747600E785EA0BD3480A07D7AD1304019C0F00F90B2A650599B0B416655AE1B1D8FACBCE38D0DAB5A6375144790B0F7E8C61B4A98297C78EE6FCA19051DCF60B4EB1F6FC59DFBC5BD1B2ED6B9C72D7B5DC33E8D13566BC67013CC7114EAD63D25ECCAD196A5C258DE23D82E8CB2BA6753C9BC598A54F921CA924750A45353BD44593415AADD251B604457A3CAF3DA681C0FBF43E4F3814AEAD086A72A4787914FD08BC727A69CA5646AC709B9E0DF5B56C8FF82EDBE3C23218A1D5382AED38B3BF54EBFA6015C388C8E92D57CCB97FE8171E1D24360311D5ED53545B541E7F24CCCA9C5B520FECA2547ED6E1889A05EBA1C0831D8BA9162B38491287B46BBCDE9BD2198EEDA5FD0869A83543713BD294D84371F1932E3EB370E30210FDBC2853F27372DF095F71A34FA4BB0E1FDDA682170A29F9E29C2BCF946CD2865865A70DAC22A81BC387B6A06840230FF8790B6BF1953DB882DACB6A9208709C121AF24A6436B1A5312B3E6D77892FFF968138DEC8B933E3022B4141A4B4605524C95468CD89AA56BF577E16A365531434FB98FD8399DDDB58C0E5CE66C8A8955AE1ED8EAE5251026861C825A6"));
  test_xof (&nettle_shake128, /* 228 octets */
             SHEX("0172DF732282C9D488669C358E3492260CBE91C95CFBC1E3FEA6C4B0EC129B45F242ACE09F152FC6234E1BEE8AAB8CD56E8B486E1DCBA9C05407C2F95DA8D8F1C0AF78EE2ED82A3A79EC0CB0709396EE62AADB84F8A4EE8A7CCCA3C1EE84E302A09EA802204AFECF04097E67D0F8E8A9D2651126C0A598A37081E42D168B0AE8A71951C524259E4E2054E535B779679BDADE566FE55700858618E626B4A0FAF895BCCE9011504A49E05FD56127EAE3D1F8917AFB548ECADABDA1020111FEC9314C413498A360B08640549A22CB23C731ACE743252A8227A0D2689D4C6001606678DFB921"),
             SHEX("1E7CDE630A92BD14100AE80073CCE0301AA6566DFEC50400147A9DFBE921BF35B110BC9C0B7AD6CFB50785C63881BE1569495092866BCE9AE303B3719B14A47DEA8FA19CAD3015D320C678CBD2B7488E71180D967BBACFCA686EFF014B7EED274F678CC43825E79C81269F99D1B824D913F8ACBA4DABF5441A29C2E769B44BBE9954A7444BFA438EF1BA2B6A3DB11E11151A1736BB8CA22494D4A3D126F100A41D49E6114CD66CB44B80C2E3308A0B551B692BC5B693BA9634E219E0EE8E7F28DF196D8C517F9D71A58F1D945E03FC25426E13612EF4158C17590922CD6A8DE677398690A8A1816C6A01B162FE6E5F0F502314C542FB568F12CA86F5CA6DA7B36255451360CA352C05D10C293A8A35417991EFE965BA6F5BB845BE9E8C021B2B4AF2AE50200C60B7BA42FB5D3147615339FD71343662090F2E7FAD4CD0AC0ED49C3FC8685116A8D51D4183CCC9F39B401D47CDE7EF7EEA63FC218C96EDEF7190A2529126633A7F3EEE421478D66CE367FDFDA6A9A60ACA3B288C6462BDFF20DBF78A3DD29E73D1A3E8F3FD824A3616722EE0B87A9E393CA89159EA8169DCA108F8E2FEB35C9372EC3B9112E534EDF733397B7585051257A564B844C07DB5DCF56403E59EAB6F56759D50C47779DD900849C94DCDAAC1AF51048098BF95A0B9EDA23DF0520A3DBD86A0B59CEE20822EFDB9DBA6D272A7CD5CC95498E340062539"));
  test_xof (&nettle_shake128, /* 229 octets */
             SHEX("3875B9240CF3E0A8B59C658540F26A701CF188496E2C2174788B126FD29402D6A75453BA0635284D08835F40051A2A9683DC92AFB9383719191231170379BA6F4ADC816FECBB0F9C446B785BF520796841E58878B73C58D3EBB097CE4761FDEABE15DE2F319DFBAF1742CDEB389559C788131A6793E193856661376C81CE9568DA19AA6925B47FFD77A43C7A0E758C37D69254909FF0FBD415EF8EB937BCD49F91468B49974C07DC819ABD67395DB0E05874FF83DDDAB895344ABD0E7111B2DF9E58D76D85AD98106B36295826BE04D435615595605E4B4BB824B33C4AFEB5E7BB0D19F909"),
             SHEX("9576636B4F4D13A576B2DD4FBB3092F358E7CCE04CDDA4F83A10A8B5D35B5EE034543098D9A063FAC9CD26CA46837C5CB6E96F7596D4F07EA6FDD8AF8B59F9E86611F1514D5DC0B7A4216C69B9C85366059BED3EA5CED38A16E4D6945C7C9539E062DE51FA2DEE1C666DB1916369FA6BC0D7A2811ED1BEEAB61397B8A9FAF224FB4ACD7F8D9C7A8513C8A93D924FB4EE027821AB6DE0415AF9BAC5CF3D5CB7075BE5A6D410F12A9BBC96BF3FF254812C48423F383E4D74FFB5EDD7AD34C6E0D64BBE7F45288A49F5DE9AC14F317BD4862E1F25E338A0309739C8DDD27839B8AB3B04B9FBCE65CEAAB460DAB0B969222250FCAAD072C445D472BF251C4911C7D5D2B7D41C9D2840826E0BA6786309DB6E8EC0FD94031E2DDFBB0D880CB780772A93C0B4C32377C3C8CE97C73412F5C3E1CC72AEBC050737495C703459B9231C05477CD9A5DF1662F5FC883A91B42ED7ADABE6CB1EE654FE9E616B40CE9035CD3B91A05E050CB2558645A0939C536A2EF3B5C8353729C3EE238C5C88DB75C05DB6B2FEBE0A598F937CA7AD4FAF495A10CB3C3B9E754A570BFA5E723B8C0CC107F0EDFF76CE07A2CF3ACEC4BB44E11C633BDBF89E5501D6001206D457962AA2F57C9C2F3FCDD1B76ACE292728F21A6ADDAD4551970B1532BD3E46D9A5160B1A55CF4BE18FA15ADF0DB055BBB21ABD9A5B0FBC474CED2B8787732FCBBB231475F29F"));
  test_xof (&nettle_shake128, /* 230 octets */
             SHEX("747CC1A59FEFBA94A9C75BA866C30DC5C1CB0C0F8E9361D98484956DD5D1A40F6184AFBE3DAC9F76028D1CAECCFBF69199C6CE2B4C092A3F4D2A56FE5A33A00757F4D7DEE5DFB0524311A97AE0668A47971B95766E2F6DD48C3F57841F91F04A00AD5EA70F2D479A2620DC5CD78EAAB3A3B011719B7E78D19DDF70D9423798AF77517EBC55392FCD01FC600D8D466B9E7A7A85BF33F9CC5419E9BD874DDFD60981150DDAF8D7FEBAA4374F0872A5628D318000311E2F5655365AD4D407C20E5C04DF17A222E7DEEC79C5AB1116D8572F91CD06E1CCC7CED53736FC867FD49ECEBE6BF8082E8A"),
             SHEX("68618F4B2868634D8FFD3483FA4AA9C200B43FE87D59107BAC648665613805242B1CBB4DB2761D4CBEA5337E5E07005B5B4E8C80038019DDEC76395BA981D36823522698C98B80D313A7B6110A0BDE7ED5E9268FA655B8DCC2C762510FF03506DC1CA8769B372F07D0E325786FB1C58206578E600E9093AA84468F01A2DF258E670959966B419BA5D4B6063C997822E073717565C6F9B4777E186A0BD6D72F65BB3CCA7E82601F25667FA0186208F7988611A76BC4D116AB3944D6797828FC5FCF17ADFA99D959F5013FB2516A2D3DC05508E9C28AB5F9E0770D0DB2FBF9CD3C8AB90F255B465A9B724D90C25E1EEF81B839A971DD84C4D610E00ED858C78B2FC6147CB6B1266ADFAC5FECDA9A6F78B31AE35BE6369C3E8290E6F297B7FDB35BBD59F0BFE16146CF3A5A75297FFD5D02685C39ED9394ADFD3C7C814AE86EB652D152B7AE980B328F3E16F883C7C6DF6672EA1EA6E08C61505E65852A1BE11DECB86D8386079E17E72EF8FA38B69EC192C0B90622150A5C43584C3159CF61189138089653AE6453D24F464D96A9BD0CFDEF7C1C9D92D46E854D644982BDBF0E5BE7C38EDF469AD3C0EBAE549322BF271590E38AE16A1FABDA49AD6BF8B1218FD9072C25027AB31787B9F38529156113A82A8DD27BD22CFA719B3AB0D31297A0DE9AF901A443057D34CDC11747D6576B7602D7D6DF58C292A69B4549ECB1518A02"));
  test_xof (&nettle_shake128, /* 231 octets */
             SHEX("57AF971FCCAEC97435DC2EC9EF0429BCEDC6B647729EA168858A6E49AC1071E706F4A5A645CA14E8C7746D65511620682C906C8B86EC901F3DDED4167B3F00B06CBFAC6AEE3728051B3E5FF10B4F9ED8BD0B8DA94303C833755B3CA3AEDDF0B54BC8D6632138B5D25BAB03D17B3458A9D782108006F5BB7DE75B5C0BA854B423D8BB801E701E99DC4FEAAD59BC1C7112453B04D33EA3635639FB802C73C2B71D58A56BBD671B18FE34ED2E3DCA38827D63FDB1D4FB3285405004B2B3E26081A8FF08CD6D2B08F8E7B7E90A2AB1ED7A41B1D0128522C2F8BFF56A7FE67969422CE839A9D4608F03"),
             SHEX("9A3CDA57019548E6A68D89CDEBDF96557DAACAC29C5EF8532DCA123231D0226D0D7F62042D70BC225CBB1ED8902D8E6BE75B2B37504990471FD01F4C656E6F6B77961C753A780D91FE03598FA2D5F60B772B34918DB836D2CD278FB41B223A6D1F25C91A4B98B64BA3EB32DFA5DC5E1CC5BA1467F287D56CE98478BEEE91087DACA1E2FE0ADCA38883C9A73699AC98AADB5C4B087719EBA775B0ABB1B0E5ACB4F79F0F75DCB2E7E25CA22C06B1629EC1CA989AD1F1F12C4580E43DA76B2B58345B61085553A827BA0D3954B0BEFCA8394B9D19022D33DF6978148234CD16C7491075517081261E9AF8FE42C66A7580132EA3307AF3372489D8EFE01C5AE579A92399B844266D13804B887538D70D2A4EAE3C65E09F6BF877113992C42D490456CEC22FF8C60B0C7D2517A49D2822CB2B2AB6BEF74F87CE134F78141157D0498F267C94123E1606D4CE1AE8D7847CF9FB23D8A11901292CF36FB975FDF34AA0CE10B851E116EE7129B06D3EF10CC1ABA6F761EE2F1E4B497DA11FCA3E844A8DD1A6129DB5B3F863378D3488CE7E208241946BAC2E4AAFC912CC46E5A6451832A2A43176E06DBD49E9D379A69E6F08081F609B7207C5CD391445E10D10FF2B99E0D672C19CBD9F5FFC6CA46AD9536973368A747A7A2F23AADE81A43E49D2E9CCED3F64CA24C012CF3A306C40ECF36843317BE8D29803A40F6F71E7E9229154B0C4"));
  test_xof (&nettle_shake128, /* 232 octets */
             SHEX("04E16DEDC1227902BAAF332D3D08923601BDD64F573FAA1BB7201918CFE16B1E10151DAE875DA0C0D63C59C3DD050C4C6A874011B018421AFC4623AB0381831B2DA2A8BA42C96E4F70864AC44E106F94311051E74C77C1291BF5DB9539E69567BF6A11CF6932BBBAD33F8946BF5814C066D851633D1A513510039B349939BFD42B858C21827C8FF05F1D09B1B0765DC78A135B5CA4DFBA0801BCADDFA175623C8B647EACFB4444B85A44F73890607D06D507A4F8393658788669F6EF4DEB58D08C50CA0756D5E2F49D1A7AD73E0F0B3D3B5F090ACF622B1878C59133E4A848E05153592EA81C6FBF"),
             SHEX("2DEC1B61A2977A54FF13CEFC08F9EC6F11BE80E7F5B77C4CCF2692450CE86C9D57EA58CB34AB2995A6193935DDE84BFD62283A881CEDBF685CA26A8A727252223CD6E7BB8A3C7751B8721ED16D1209611803778DF77C6A005E1A05AAB97F6205E0C1C6B339F8D8346A84C6A6D86EFEB17B6F8C8E30960DADBA3DDDB860F46D2718C7307C55B65AEF1082B98E407F56B633A16578528B8211254FA6AF384AF0EE36BA49638386494E347E961D6D65CCDF2D5D221AC54F6A2F6C4EAB62B7B55E26192A2E223BB053558A6C3646355F3686DD1D3712406677F183E94CF822D0AFA9416815C14BAB7F5C5B4E44F364446E08464370E60A436653B2DD29E60562712136940DF4AE19617774776F0784F7E8AC600BCF8359E02B070C0480F372DFD69DF9C33ECD3484678087087C7DAD64E160B32A0372318AFF1668F5A0B03C940C384239994F546FEE94E3ED0FB8AC259002D09C3110806FFB56419CBDC76F28C8F828DB0AEC39A573A903A49C0C9F77F0504B9C8CBECA83B6D3A7AF24631F7E4B99E9B42EE201CDA0AC1FFB79AC24F3516D766B0F71E1EFEF8E6CBE38C24D86B83388D6D3C2F9BD6AF94B8A56B6B5CD53AF40837FECC5481F9AF1FE216852C281E068F0891908B583845B58A4CB4169A0F2AAB2C36FD184B15C3E4008E38E00C0B6DB256B89C4C373E8328D13E840D65605C44AE017B0876CB5ABE1AC4D9CA14A3B"));
  test_xof (&nettle_shake128, /* 233 octets */
             SHEX("7C815C384EEE0F288ECE27CCED52A01603127B079C007378BC5D1E6C5E9E6D1C735723ACBBD5801AC49854B2B569D4472D33F40BBB8882956245C366DC3582D71696A97A4E19557E41E54DEE482A14229005F93AFD2C4A7D8614D10A97A9DFA07F7CD946FA45263063DDD29DB8F9E34DB60DAA32684F0072EA2A9426ECEBFA5239FB67F29C18CBAA2AF6ED4BF4283936823AC1790164FEC5457A9CBA7C767CA59392D94CAB7448F50EB34E9A93A80027471CE59736F099C886DEA1AB4CBA4D89F5FC7AE2F21CCD27F611ECA4626B2D08DC22382E92C1EFB2F6AFDC8FDC3D2172604F5035C46B8197D3"),
             SHEX("DC2038C613A5F836BD3D7A4881B5B3BFF3023DA72D253E1B520BCAD5162E181685662D40252BEE982EB3214AA70DDF0A95C5D1031DE9781266B1E0972FC9777D4A74164DA68A5D4585F7A8E7438FE28D8AF577306B8E2CBF6863C83431CC4C898DDA50C94EFD4925432FCA36A6304790FBF4FEFAEEE279C01B8B6A8D1C275E3CB4E8BF17D880903FBAF27BFA65A2E3DB8E285878A94955F6FC14F05A0FA2556994B8612BB7A494B4DD8B3CF1BC9E4BF833D4BFBF878C4D3BDC8FC70D26D7B7EDAF0AFE2F963DC6884C871C1475F4B92378B9824970E40DA0A59780E84AC5138AA1EFA46C1B50C3B045BE59037C6A0C89E1D3CF246F1362794E8107B7CBA74888F0BF4B905CFB9C33517F472BAC16259809797F2FC883FFBDD7CEDE9518F891B9117DE5DDC6D3E29FA56EB617F25E9EB1B66F7E46ED54C1D43AC07471D35C57B8C73BC68F5612ED042BFF5E68634A4FB81E2EF0D92FFF1E11E43FD6D9A935678D2FDD04E06061DA3BA7DE415B93C5A8DB1653CF08DE1866F5C3D33BE32A3B8D2B7BB39E9745C6E88C782F220C367F945828B9B9250DE71E8A14EC847BBEEC2B1A486CE61731CEF21B4A3A6353C2C705759FAFA50AD33FB6ABC23B45F28EE7736DF6F59AAF38D59881547274CF9AF2CFC8FC1ECADF81AB72E38ABCCD281DF956F279BACC1796AD1F90D6930A5829BB95E94A8682A51A6743AE91B6C12C08E1465A"));
  test_xof (&nettle_shake128, /* 234 octets */
             SHEX("E29D505158DBDD937D9E3D2145658EE6F5992A2FC790F4F608D9CDB44A091D5B94B88E81FAC4FDF5C49442F13B911C55886469629551189EAFF62488F1A479B7DB11A1560E198DDCCCCF50159093425FF7F1CB8D1D1246D0978764087D6BAC257026B090EFAE8CEC5F22B6F21C59ACE1AC7386F5B8837CA6A12B6FBF5534DD0560EF05CA78104D3B943DDB220FEAEC89AA5E692A00F822A2AB9A2FE60350D75E7BE16FF2526DC643872502D01F42F188ABED0A6E9A6F5FD0D1CE7D5755C9FFA66B0AF0B20BD806F08E06156690D81AC811778CA3DAC2C249B96002017FCE93E507E3B953ACF99964B847"),
             SHEX("91ECB5133525E291736CF8F4793BDB00A4EA32F48C3F24CBD3238E6408FEDFA6E7A750409F197041001C1A090659F8879CB26912B46A0E7E4E791FBF5692FD36EC84A55462B7876724EA944CE7246261F26038B166668E6B7EAB0B5119B44E4581F3A3280508CD8B3789AC37F992EB9777B9E11C0F19A3DA5F4B8D304AC4DE68B3C2CD6659F33FBE20B47E1F6B0CE06EC0963F8E7412FCD84E3DA3867CFF08F82AA17C43C19BDF3BC0D242FDA4874DDEC389564345AD11A267D7A466925D1E9BEEC667B029DA0982B37B51413B4601CAC17A02CBC681F927DEFAEDDB0F6A69FDFC39AB735470E86A00F82EADC94018A6E2A29E7F393F30D954C82936E158A364FB139FF29E42F30C0B95BD5714EB7A535A5B1D1B3B36A862E35D10D1F8A5AFBEEDECC51A855724271DA9FFB8EFC6615C42B03A57A2D762B46D5D94E097DAFE5D7A2EF522DDF2DDADC1F83EADB72510E770735463495B0217B8BC91DB597683254F6E0AC0E9B1E56ED0216EC9618173B163CB3F393152469268C056B1A204567EBE7367F863BA145C3BB367FA3C01E556D62FBC8381CB3C8A9BED907D481C093F14B70D1BD31AAA41FA753CE31FC59740A47BF5980F8A0E32EACA8A488572A3A91DB07A25C34330CB7DA7E0FB7D1541C3F3C377963DE4560A8D584E11E9C6A1B917F907C34F211CD78904471199FCF30D4180F88E7BDDC0392197444ADB1968D2"));
  test_xof (&nettle_shake128, /* 235 octets */
             SHEX("D85588696F576E65ECA0155F395F0CFACD83F36A99111ED5768DF2D116D2121E32357BA4F54EDE927F189F297D3A97FAD4E9A0F5B41D8D89DD7FE20156799C2B7B6BF9C957BA0D6763F5C3BC5129747BBB53652B49290CFF1C87E2CDF2C4B95D8AAEE09BC8FBFA6883E62D237885810491BFC101F1D8C636E3D0EDE838AD05C207A3DF4FAD76452979EB99F29AFAECEDD1C63B8D36CF378454A1BB67A741C77AC6B6B3F95F4F02B64DABC15438613EA49750DF42EE90101F115AA9ABB9FF64324DDE9DABBB01054E1BD6B4BCDC7930A44C2300D87CA78C06924D0323AD7887E46C90E8C4D100ACD9EED21E"),
             SHEX("54A3C3F41FFA1A12309CCC1D6B7915E126CB13371A2953F33D6512241A5B835005A7FC8844E4E5BC097B9C5D8E388DDB5A55F43905E11C380B8EF1DC661B77593DBFAFB6B2A4A28039ABF57706ED1193E860CEC1F5D9C522839607A02F3AE0D057100CDDFE48BD84F60DB063AA3011EF8A6AAD3947D8F86E2A6A28E258A99D72E998E4148DC877945844094125DB2BE06B98508704E3901AD2D8D2CADB19291AF8436DC4CD89D97060DF9F7CE0738ED5C1456CFEF18862DABE84A08A22BF310FE273496401619F156EDE9311697566B11C38BB10D4AC5787D4D6491CB43B4D79C5F062645FACDD48EDF609737C7FFC226E36587B39759F23DA6CF94349CFA50B9E18F1C6CD42DD22EA04A4A7A73ACCE723E4E5948FCCC9857EEBF22C9761AB8AFF73386D470A34254750BCE7EDC88A46F54AB64F562BEA863488F846671503E5E305F3CC61212E95D055851A4D4A9A145DF52DFBA84D27D084952F7E12AF3C4E20860A479156B48CD65AB2BEE5407AA200CE205E3E46C35DC4B50205C11187EC81ECA5BE3B2E90A170D53E6637947470220B2A6EDBA8C0D0E3525BE456417E6EA0FD98BF54A5804AAB6BFB728B22A5D01BAF0CEC173821F5E075A0E4B7649E303D17296981EFB88327D5B1A3A8CA72129B979DFBCFDC45EC3D1BFD01D10814F225C6BF09962F0491756CF9607D3974C24FDA6DDF6E87013EC5CB2B883BD0A50D"));
  test_xof (&nettle_shake128, /* 236 octets */
             SHEX("3A12F8508B40C32C74492B66323375DCFE49184C78F73179F3314B79E63376B8AC683F5A51F1534BD729B02B04D002F55CBD8E8FC9B5EC1EA6BBE6A0D0E7431518E6BA45D124035F9D3DCE0A8BB7BF1430A9F657E0B4EA9F20EB20C786A58181A1E20A96F1628F8728A13BDF7A4B4B32FC8AA7054CC4881AE7FA19AFA65C6C3EE1B3ADE3192AF42054A8A911B8EC1826865D46D93F1E7C5E2B7813C92A506E53886F3D4701BB93D2A681AD109C845904BB861AF8AF0646B6E399B38B614051D34F6842563A0F37EC00CB3D865FC5D746C4987DE2A65071100883A2A9C7A2BFE1E2DD603D9EA24DC7C5FD06BE"),
             SHEX("264A7D71BA8E67773AC5AB0CE3616F1147341EDEEBA1A177E8529B0DEF09C3A99D91029D77C3F9F85508AE74DE172F358CDD2BC4EFC8B27BF280E162B03A08272D0165D9012099B8EC0274801AF9A0D672FDA4510C09347ADD664505BE434469CE78DF597A7890381FBEEBCEE79FDE94EE240A2B3DFD60B23F4126A5FC397B4B4E5E21893CAFBABEC64BF53F603434F168FCB667828E79F44B1526E9875F9EC9982648B9C6FC1AD6527D7FD2AAD064227D9A85EF9733B2B484202E00F8929E18899BFA6B92C3AA8B05F9EAD25DF02E02C7E8C264B67EC07AB47A7D95905F660CC6722ACA28215AEE543FB7950F0204784AFF77374B803E5EA679F22A0F5659A363EE122AA76BDC881EB28D5E347842B2CC81FF0447B68989E744E187808D7AF0011A32B66F9050161168D6F246A5C7B212838B831E71D71606F21A40D8A9AEF42B21465689B7CE967DC2E53F869F0584E19DDC629068C1ACD7E6B986C3D1F1E24B1A907F7ED4E5B83EB7B0A5C6A4508E0DC21BE2C1BB5506BB2B97D9721B57DEB9D4D39D58B77DFE6C869F5FBFD74559082B5EFC3692E72A9B961B6A669ED700CC41FFEAF370F88373D5BDA8DB6DD2C3BCBF1D833F4B9CBD3B4134502931178C58FCFFA80CB496FD850352634F73D002FC4A6725848CE2541862B3C5566A915878E60B3B4C1EF7F014F0F32446298891A383F63568C849DF2A208E75EAE37ED8"));
  test_xof (&nettle_shake128, /* 237 octets */
             SHEX("1861EDCE46FA5AD17E1FF1DEAE084DEC580F97D0A67885DFE834B9DFAC1AE076742CE9E267512CA51F6DF5A455AF0C5FD6ABF94ACEA103A3370C354485A7846FB84F3AC7C2904B5B2FBF227002CE512133BB7E1C4E50057BFD1E44DB33C7CDB969A99E284B184F50A14B068A1FC5009D9B298DBE92239572A7627AAC02ABE8F3E3B473417F36D4D2505D16B7577F4526C9D94A270A2DFE450D06DA8F6FA956879A0A55CFE99E742EA555EA477BA3E9B44CCD508C375423611AF92E55345DC215779B2D5119EBA49C71D49B9FE3F1569FA24E5CA3E332D042422A8B8158D3EC66A80012976F31FFDF305F0C9C5E"),
             SHEX("C87CC74FE777B2DC9A33473F88383DFECE2244A807FF92BABEB7BF08F37EABD9305BE93213A63A4851BCDC648C2EB31BD509844A56B7B890772E040E67D32A4098EFBCCD0ED6D25B2AB9EEFD0B65EBB76D540D0FECEC277AC98139AE8F67025DEC7AE76850E047098557D2EF7D9319BC153641B1D2A376F23E4A1F2FDF1168EB5901B3E11A0410C8563DC4BF1FB0F616CFEAB2B086384207CF1A10D51634874493DB0533787C0564C1400BC56CDB7C32A6FA4800F0E685156F2BB63A63B67F6D40AB574CE0EC54A4A533958B3A48266EE038E5D85E67F3BA429EA1671D89A083B406A568A0795C56EF042881AB67665258EDEDB97AD04B5A30388D7EF0DEA494BA645F8FE47E81A2CC4A9C6885ED80C6F6D37BB0E81A60575201C0EC3E9FF30F70AE746B86AA3EFB309F13FAD5CA2C824C2FCD34AF1DEA5E60E4A8CD92E7B653301721937E03ECC2AD5642A3922D4D666448759961C6D830CCB45D666B675C920AC2486B100B5A51A7659F24735C173F9FDF02B1401A03ED5BC45EF098E799E7A3C29EB262D408CA9A6D2C18C5636F3B378E63F8C71C70058976E880D8A0F23E0F2F1CAE00F4E06FD60673CF0968A2E3C5D97A79659BC81F6639FE69B18C6E4D1F3BF535814A4D6187F3F9BD7F7509B2EB899A6ED48526C665B706A526F0E5EE433DCB3B5C7BFC99078342B78EDCA9502D6F73566BAB6AFEAF63EC9A020A7EF5"));
  test_xof (&nettle_shake128, /* 238 octets */
             SHEX("08D0FFDE3A6E4EF65608EA672E4830C12943D7187CCFF08F4941CFC13E545F3B9C7AD5EEBBE2B01642B486CAF855C2C73F58C1E4E3391DA8E2D63D96E15FD84953AE5C231911B00AD6050CD7AAFDAAC9B0F663AE6AAB45519D0F5391A541707D479034E73A6AD805AE3598096AF078F1393301493D663DD71F83869CA27BA508B7E91E81E128C1716DC3ACFE3084B2201E04CF8006617EECF1B640474A5D45CFDE9F4D3EF92D6D055B909892194D8A8218DB6D8203A84261D200D71473D7488F3427416B6896C137D455F231071CACBC86E0415AB88AEC841D96B7B8AF41E05BB461A40645BF176601F1E760DE5F"),
             SHEX("61390F3D091CFA220147B1C85C59F1D6B7D9F87DB3A8910884AEDD2DB4EF65BD77F1D091D8375002713322F3AC618126D47DC4EA0404402FFFE139C8163CF8746030B2E57BB2A74965159F4484E3A4C41E7F849BFE872DB3B47EB91E4C9E67DC09AB0FB88FDE8A9815BAADCA11238B97DF04A0F5E5570DDC87D18E6E33339DD481D20CB746478162CD1464929DCE1FB906B6545D846D891CE91358B9EA029F281ECDC557CB769EC8C1FAAAD10FFE13B8076072455BDE3A3276CA458E8FFC4E377ACA54833C31E906866E9DEC7EAEAF74AE07339D70AD34278B076CA541CEF4DD10F68FB29C17F030052702B7B87D857F462414BFCB9FBA427BEEF0B930DF09C7A6BB35620EF843C9866E242EB2B30E30C4427C1E5671F2390F6F9F42B77305CB9969224355FB7A751E2066FB8C492D6070049C768B5263642802D21632DCF50EDC2C9357A4A3CA20C8446CD322CB5D8EDF82D31871424575F022D8B7A6B7BAA4F6326CE757AD523B8BD1B4A77C5D96E7EEF9E2963B02B107F7F74C9341D964D70DAA7C9E1ECBCC354935EE07192F22C15D693E5E10DF094FA8B74CB2CC87B967F11237B3AB5D3F53EEE9BB9E4CA16B7DF02E46EA9F141DCDA939B2D7C964574A0E11E3505706C0E9FE33BE262A7E6589D7BD42276CC3146F0A728131E9D346B91DA1EEB6B3E34EA5AE180396E219B5FFB6F87148ED8D4BAC21E760861B49F248"));
  test_xof (&nettle_shake128, /* 239 octets */
             SHEX("D782ABB72A5BE3392757BE02D3E45BE6E2099D6F000D042C8A543F50ED6EBC055A7F133B0DD8E9BC348536EDCAAE2E12EC18E8837DF7A1B3C87EC46D50C241DEE820FD586197552DC20BEEA50F445A07A38F1768A39E2B2FF05DDDEDF751F1DEF612D2E4D810DAA3A0CC904516F9A43AF660315385178A529E51F8AAE141808C8BC5D7B60CAC26BB984AC1890D0436EF780426C547E94A7B08F01ACBFC4A3825EAE04F520A9016F2FB8BF5165ED12736FC71E36A49A73614739EAA3EC834069B1B40F1350C2B3AB885C02C640B9F7686ED5F99527E41CFCD796FE4C256C9173186C226169FF257954EBDA81C0E5F99"),
             SHEX("39EC35D31923584FBD73A715015CEF05B945A5AFD2D109E72E356BDBE0F73CDF666AA7B01E77BD3BC0C409991F1129961F6BDD772ADE7EA97AF0B88A8875190B453664FFB64FC4FAC3C1108A3C222F23B12B7613941B137449471BA4847EAA4A16BEB50960FFF5DFBB2E756938B38A93509BCE5F900A59E69FB0305A19E460FFAE88D74A727C82FC7138B88DB07EDB172B39566DD22F0E852F715AF9D3C3F0089E9EBA72FF6063357DE9A744F20E1DA367F27C6AB6FC9D231A606A79A621F8150773D1DFD52868820EE5BC28D286ED746BEAF8F13213548DF0A56BB0E01E5BF6BDD1C901EA8C791E7916E5679C88D4D85CB1ACCA63C211E7218FE6EB2ABFD2BADBB31147177F123FD111BAC4880F1224ED01F3259E4EB91DEB7562CB9C3009B3676998AFAB8D5D24BFE7D27BE7F3A57A932C895B60A9FA74504472808479F332FD9915B8ED0C11604E80457C0DE5F7015C19A607AAFDE448FC08BB185517E6AD7DD330A465F9043A0478E1CB4BE9185187604F30A1680C67D3A0DB1BACCC05D401AAA8FC66D9189089EB7DDBABA05FAA8DD83FA544FDE68A793A36D854752F22FA13333C6152F7F18ED789848A13327071D85D5E36C4FC291CF5B4B9391F8E2513D4B660B1290790AB50D12B73FCA21A93F8998D5706C037C387D662FC20BE35228D85E5BD007A7173BD06E5CBB7E41063AE4843BF3F5A77C44E1A36FD2E1C31"));
  test_xof (&nettle_shake128, /* 240 octets */
             SHEX("5FCE8109A358570E40983E1184E541833BB9091E280F258CFB144387B05D190E431CB19BAA67273BA0C58ABE91308E1844DCD0B3678BAA42F335F2FA05267A0240B3C718A5942B3B3E3BFA98A55C25A1466E8D7A603722CB2BBF03AFA54CD769A99F310735EE5A05DAE2C22D397BD95635F58C48A67F90E1B73AAFCD3F82117F0166657838691005B18DA6F341D6E90FC1CDB352B30FAE45D348294E501B63252DE14740F2B85AE5299DDEC3172DE8B6D0BA219A20A23BB5E10FF434D39DB3F583305E9F5C039D98569E377B75A70AB837D1DF269B8A4B566F40BB91B577455FD3C356C914FA06B9A7CE24C7317A172D"),
             SHEX("9BCBAF72085F4EE5B2B237F520977C7CFE8780BD62B0ABA13C161D0E7EB036AB27E9B08407BD501B2482C280997EAE9C4B571ED6E39A13D74D7A1E2258CBF3583B28E738FB14AB2B60E6F1E188953D96A9361D282B85DDC243FE6C7CC644BA203B328FBC4E0088E40283B6A62122065A11A164DD1EA05F421B4ED60F44EC982636F0253AE4E6EA320EDA2351DFDE0285D1ED860A51699E702A0519B0C3119462BC9CB3DECD66647823EE139DBFE3EE125A6F74AEDF5044238389E0B0388698FD8D7D882D010C194F4478D6DA8FC195529A475932E4FB178CF318AD8F24427BBB90C0B08ADAB27BE5A48C9804ACD9E56E1CB2BFEB236388917458408C446C6B5F1D3BD55AC0F6F1AD06AB419460F525B564DC67BC6BB2CFD08CDB510161F71A29AEC95C67323259C214CC39401E483E982EE6576A7F1068B088348916A5C47D1378236DDF475733F394E14216D2AC112457295100EB70381E632876E700D355973944ED80AB91AC39F2D9DF4E338C02082505C028EDDF867E60ACEFFBA59480E6953BAEA105652BAC6D29DEF70962B6BB93E3D9424A77D31D3020FBB3BE37E7905A6360C1B624327FD68ED02639A2B54FBAFDD9804FCFD60AEC2B044C921A77B25B7032CF685475F503D00D9A8DB733CB6115C1BAD1C847C1BBBD342E62470D6DF7C21119A35179AD3C7F68CD246B8F3F51E2635DAC460DB83FCC5C0DD049A2B7"));
  test_xof (&nettle_shake128, /* 241 octets */
             SHEX("6172F1971A6E1E4E6170AFBAD95D5FEC99BF69B24B674BC17DD78011615E502DE6F56B86B1A71D3F4348087218AC7B7D09302993BE272E4A591968AEF18A1262D665610D1070EE91CC8DA36E1F841A69A7A682C580E836941D21D909A3AFC1F0B963E1CA5AB193E124A1A53DF1C587470E5881FB54DAE1B0D840F0C8F9D1B04C645BA1041C7D8DBF22030A623AA15638B3D99A2C400FF76F3252079AF88D2B37F35EE66C1AD7801A28D3D388AC450B97D5F0F79E4541755356B3B1A5696B023F39AB7AB5F28DF4202936BC97393B93BC915CB159EA1BD7A0A414CB4B7A1AC3AF68F50D79F0C9C7314E750F7D02FAA58BFA"),
             SHEX("8536A6C25934A44FF97C0A09E026EB12B9C81CC8357859BA736B29BB329B182B5DA8FFD7B855200CB9D64FE463317C213997EBDF1810BF37E00D56F75791375C5FA84DCFA37EC196BB2DD3C66C1D33034AC48BC7FEAF15F9702D7D11E4712AC87934E2064283D04B56BDDBFB3B56C3CDE69B7D736196B9DC5623A3132BE33AF12A386A3004EEF98D4487B614547BA59EDABB0B83B6CA39A5955F925F35A5209C0CD567369751312A6A63D0F4821D442C14CC26FC147E63A4656745663CD96B01064F60FAA4BA98258B09C09833046412221A59E23D5266FAD8C068CC949D172862EC30EBDDEA21E1CFDD6B5FE4AF566A19D0D41FC9328EA915D8B97CB1D903BC40A4083F6997439055B44A695450039CC00931CF3BC9FC67BC417A8AA270B2B4A410BCE4A2131523749A957D293C1438A993432941A9FF4210B33A5CF8B4434553B17CFE869B940669CA9F18B128BA354855F08A692FC6D23BF0FA12D9F4DF0A571E44D4221E8018C4282559E6FF2EB02CCAC7A7B83E34ACA7D554C33F05CBAEEC49D0516B10DE06BCDEDC09D0C4A9810B0797B62CFFA75FC0D3BDD796327BB1FA5DCB546630E9BB0457D024E982D6D7930391066C58286BEAC627D6ECA09B4F6DFCCC71CE589B1DA1E890AE0C5A0CF531D9C1D2330BC8397F09F248F9919995326FB92147959C3E6C569ED662F6968F4E9EF42B6C2FB954D79DA22117373E27"));
  test_xof (&nettle_shake128, /* 242 octets */
             SHEX("5668ECD99DFBE215C4118398AC9C9EAF1A1433FAB4CCDD3968064752B625EA944731F75D48A27D047D67547F14DD0FFAA55FA5E29F7AF0D161D85EAFC4F2029B717C918EAB9D304543290BDBA7158B68020C0BA4E079BC95B5BC0FC044A992B94B4CCD3BD66D0EABB5DBBAB904D62E00752C4E3B0091D773BCF4C14B4377DA3EFFF824B1CB2FA01B32D1E46C909E626ED2DAE920F4C7DBEB635BC754FACBD8D49BEBA3F23C1C41CCBFCD0EE0C114E69737F5597C0BF1D859F0C767E18002AE8E39C26261FFDE2920D3D0BAF0E906138696CFE5B7E32B600F45DF3AAA39932F3A7DF95B60FA8712A2271FCAF3911CE7B511B1"),
             SHEX("7201FC2AF67D095A5E317269E750F965B8BEC311EDAD18DA2435D2872C0056617F0FCCC80552010F996E5B34084B09F31F35870486A8C53D220D7D009FFC2C58F26AC41BAE05EB486ED4A18E74FD1C31CEA640E8D47D88C5D506C5719A3EAA4716AD4FB906A5085AFD46570F2B048264D4BD3554F5AB0A8271D6529B9E2F1634B5C78ABAB6BBACF4F01A4E33377F9E1FB12FC20435DE0AFD62F90E2CA73BD5260285BF5CB9B4302A312BBE48C3B68A4DA6E842B7D87B3BE0BA82C9B7953CB63B7E7E86296B88557BEE2EC9576A468D556B77F872BF728FE3051E4932988029D88320088B70C8B19B59FC03F2F32BF34F42746A6E9672C21EA85087A46D5BAE484F460AF64F8F62BC615E79ADF67EFA9941DEB5B358A5833CCE4A92DA0F237A506D530D64435689E1DB79F7AFDA051CFC065397C9393B3A87D6DE9C08522BEBAEF1B33D2EAA746A5AE7FAD267F168ECE1DB4BDF453A100FA3743D9DCA3BD0C00DD7033AE1EB573BB79C6CF53BB38130AF279C0A0F8198C5145A5DFA32B322716EF6115BB5466CE54EEF73E68C4C2B1D0C321A03B6C7BD1BE98DFCB50C08DF2005DAD35C430004DE8E6C531368B5F37E53DF3D376CAAF854298657573BE7038052C2169BC907341CEB02925385A9FC03400F53CCCD6D3BC039BC49F7E5CADBB68FD5E1589CF1B97C9C6360A4DDE6474FF39B96CC2F4C698300DB2AD5668BC19F45"));
  test_xof (&nettle_shake128, /* 243 octets */
             SHEX("03D625488354DF30E3F875A68EDFCF340E8366A8E1AB67F9D5C5486A96829DFAC0578289082B2A62117E1CF418B43B90E0ADC881FC6AE8105C888E9ECD21AEA1C9AE1A4038DFD17378FED71D02AE492087D7CDCD98F746855227967CB1AB4714261EE3BEAD3F4DB118329D3EBEF4BC48A875C19BA763966DA0EBEA800E01B2F50B00E9DD4CACA6DCB314D00184EF71EA2391D760C950710DB4A70F9212FFC54861F9DC752CE18867B8AD0C48DF8466EF7231E7AC567F0EB55099E622EBB86CB237520190A61C66AD34F1F4E289CB3282AE3EAAC6152ED24D2C92BAE5A7658252A53C49B7B02DFE54FDB2E90074B6CF310AC661"),
             SHEX("C7B0EA1011BBE2D11DF8353D06A6AD5176DAC3E33039C621A635E1124EDF2425AD88BA552D6B6034AE3A5FBDC735E4381064925BFD6CB5E3E753D27C7A77F4BFD3D33652461F5409CA29C060AC991F3CA90A29B9F67502A524795F1306430FE7961430B431E9CB7C21EABE1EC3D1D6440EEA64496F30A3A44172B2C5670DA2F4DEA42F783F5138C966A642488231DCA509C144B59E0F4A40C191C035E4A464ABB6F4B9E0F9ACD86B7BDA0485238A56B2934C7AACB484287EE013C46E48FB648E73ADD7E8ECA6B2A8FC34EA0CEB46B0C360A6004B34403AE5FD345A78B55634B242CBE07A6FB13FAB721EF976B3E1F82FF4DB06F86DCA50348398F1DC831C70618FB387F0B13D84BC8A606E72D6B85991A4D7151E2D9B4B5E07D303E2D639D2A243E316B6C328EB01F52C524FA7D76763EDE5F0FEC6824D73EFE546547798D9C78B22765E69EE0F72B538C816B230481214099ACF758193AAE0688CB5A9E11B323704AB7724E5F1C4355C11EEB595312A25277794B996C6BC19C77F0A048CD31FC8AB70C7B6B6B0DFEF5F632879DCCFCF462908F04AFC8B440E9C4FBEB428B70A56511EB7B16053BE225B72CEE491366742A77152A9BB0FAC5D261C01ABEA1B632289F26FCD628598C1F9B6AB1A5C36F926A2B96BAF71644C0723A6A773DD14C66A770F11BD82EA85C6608C821139601F9B98232BFC21C7A3B54E0F7A2A6E4CE5"));
  test_xof (&nettle_shake128, /* 244 octets */
             SHEX("2EDC282FFB90B97118DD03AAA03B145F363905E3CBD2D50ECD692B37BF000185C651D3E9726C690D3773EC1E48510E42B17742B0B0377E7DE6B8F55E00A8A4DB4740CEE6DB0830529DD19617501DC1E9359AA3BCF147E0A76B3AB70C4984C13E339E6806BB35E683AF8527093670859F3D8A0FC7D493BCBA6BB12B5F65E71E705CA5D6C948D66ED3D730B26DB395B3447737C26FAD089AA0AD0E306CB28BF0ACF106F89AF3745F0EC72D534968CCA543CD2CA50C94B1456743254E358C1317C07A07BF2B0ECA438A709367FAFC89A57239028FC5FECFD53B8EF958EF10EE0608B7F5CB9923AD97058EC067700CC746C127A61EE3"),
             SHEX("FD6ABC78D5FF667AEE2029F1683E1EC57E82E69E6F3963E38486590650A18EB3A6B611D5ECDD6F8E556D2F8BED3726DC51B2086275923E2BAC576625FCA0A10CE1CAB318AE80806929EC5EF10F3262F460DB6D39D6DB17FE364792AC2CF5B067F260366FDA234EE09761858A3C5EDFA7D8C8BB7E968BF78922CAD2A276096E06164D15782B74E75ABCB32D1D87CC5BA1B188AE777B5FA570DAFC393C7FA0CA76623543BA75FF214A0F01AE3455A9C454BB6773721DF3376BF92B4C6DB9FD4EDCD802E30CDD698A865C29912517E0D195B1D05C391BE04710121C764DF515FD0ECDEF0D91603BC2A3B9F905B5D7C25133657FB6798ABA7119FFBBB836028764298EC082FD36CBD7D32A81C549B32E6E82C8C897318EEED0C3720AE2B620A40ABC87C28613AA2322EDD83414B8A532D48FA5769E4E8A46E159D5BF9926C6647836527BF5AB23846EA38F321EE7303A1DAFAF61C97D6059B10314F09F3D5DFF337802EDF70F27E0224D1B110453902D3F4D5A074F2FA532856EAE37307507C2E9BE3952EBCCDE1599A3A9B331FE760D29720AFD9134B6F5C33CDFA60E9B700E0FF15E13D197830BD93E785482BBA5DCECBE0977433AD8625280F0DAC5C6A4963F23844C7EE7AC6C88D098B8189B274D49194727118D6E73898584BA8FEAC1947E0B0ACB81DC7B580F06F52A5743F8E7E8066F6C76B5013DA694072434508C2802E0"));
  test_xof (&nettle_shake128, /* 245 octets */
             SHEX("90B28A6AA1FE533915BCB8E81ED6CACDC10962B7FF82474F845EEB86977600CF70B07BA8E3796141EE340E3FCE842A38A50AFBE90301A3BDCC591F2E7D9DE53E495525560B908C892439990A2CA2679C5539FFDF636777AD9C1CDEF809CDA9E8DCDB451ABB9E9C17EFA4379ABD24B182BD981CAFC792640A183B61694301D04C5B3EAAD694A6BD4CC06EF5DA8FA23B4FA2A64559C5A68397930079D250C51BCF00E2B16A6C49171433B0AADFD80231276560B80458DD77089B7A1BBCC9E7E4B9F881EACD6C92C4318348A13F4914EB27115A1CFC5D16D7FD94954C3532EFACA2CAB025103B2D02C6FD71DA3A77F417D7932685888A"),
             SHEX("AEDA7E54B44C4EC1DC28A305AB30CDC9FBB1BD923F41DA60558ACC8C3FB34D85EF153DECD3090860E383151EBD008044B20B20B0C6216D16EA221ADDF00B99DFA692CE79E69AC853FD70C6D077860C54FF77AC55962CF0D06888CE0A95DEC7CE760E8E0364AE935CC8CE962BAFB617DBE45A4669F0F5C6A7767FB0C9BDC853C84CDEF5F722DA5C31D15181725158F946C8D810AF7296EC91F3EC782D38B00563E454AC9299B8DA2C4D2FE89C0C2C6E094A14DC3CDE4B4B5A868CBE8C6032812CCF17A1E511E7E59C0239E6696C2CE093CBF79F641CF1B59D97B88A16359DAE8366E23E13C7B8F9CCCC099F9B6CC0A228223FBDD5242139F3BE2C19C06A15563E404428052E32ED5F6EBC61AFE642340625C29174ED962D6EA5D8D6B06DAA63CDBE674CA6ABF8AACDF727BCF6279D4AB7B2511D782C21DB7BA2543C2273970CC18C136D74D6583782FD44422A2DD01BAB97AF2A420F8F1DE60AC0F4727B91972289943B5DCB3D22030D93A6ED0863E1920DC8A6688DFCFABFC3ECEE59AFC7B349F5B6B061E599145E2242C05D63566906A40B98F8D280F31D25880D0DB0A90AF20108715EC9FEE5FFE151BBA41E594254B7EAF61C5B444094B307511A1799D2E983500FE9643DDC2652B7B77FA27DB41B50F788224BE4B4F7DED43B56A3BD7EB52620689B4D22859ED3E0771431168D12404B5C56347FAFFB9E133FC5730EE76A"));
  test_xof (&nettle_shake128, /* 246 octets */
             SHEX("2969447D175490F2AA9BB055014DBEF2E6854C95F8D60950BFE8C0BE8DE254C26B2D31B9E4DE9C68C9ADF49E4EE9B1C2850967F29F5D08738483B417BB96B2A56F0C8ACA632B552059C59AAC3F61F7B45C966B75F1D9931FF4E596406378CEE91AAA726A3A84C33F37E9CDBE626B5745A0B06064A8A8D56E53AAF102D23DD9DF0A3FDF7A638509A6761A33FA42FA8DDBD8E16159C93008B53765019C3F0E9F10B144CE2AC57F5D7297F9C9949E4FF68B70D339F87501CE8550B772F32C6DA8AD2CE2100A895D8B08FA1EEAD7C376B407709703C510B50F87E73E43F8E7348F87C3832A547EF2BBE5799ABEDCF5E1F372EA809233F006"),
             SHEX("9FEF3F4FEFEF93B5124DC0C0D3FCD29FC5D6F680F511C682BD41E397EEA9147441E698E4B420CF2A165B15AA36895BE8AE914ED53FC9950F16D540FF5054C0D731789D126091A75FC5C7DE3C28478E86E05C7A604CE904DA8F7EF031DC9D957401B155D110745F3145C34754A45B7FEF94D0253DE536BFA0ED5E655243E689FF909DED44948A339CD55B897D015BF49E890192D051A13C26BFD5E8E71AE287A6E2C18017FC997BBC41A7D39A18DE8B30F5F313D449C5095314E740501E2101807EAF33F596D0AD639C83BF7037E27C1D8597B4BC9F335433137EF4081E381B831F17AB09BC704BF1C581F4EDAF9502290A92501CD7A0E5F0AA10674BFC9B2C9C2C812F4C3725D6E7975A1F181BA63349D18EFB824DBC4D7D6267E7C4855AFB5F4DA4B26BF74785AC1E97D3E78DE97179739481A47686545A7B81F9D6BCA18CC3008BE300C34134EDCB9C87870D8BB4DA18CAD8C29CA349392FD82E0BBCA5A06A043826194A52AD80E1C3BAFA28751338BDAC266D8245E055546611CBF929CC8CE6DA35667A66AB87EC22CF3B54D8EC1D7CDEC5C75E0EC3B53ED0C030AE30913C12AAAA71567C731AC8C9CA1279C7A8FACCA20554A506661630F2B6B9792DBF5C8FD545C283DFC65E9D758B4B86079F569AF818138304E617D74081D80CCFF0179542F6FFF41151B85F84B2DD18FCD8C2B7790B645552C94C0348315BD91E1BEA"));
  test_xof (&nettle_shake128, /* 247 octets */
             SHEX("721645633A44A2C78B19024EAECF58575AB23C27190833C26875DC0F0D50B46AEA9C343D82EA7D5B3E50EC700545C615DAEAEA64726A0F05607576DCD396D812B03FB6551C641087856D050B10E6A4D5577B82A98AFB89CEE8594C9DC19E79FEFF0382FCFD127F1B803A4B9946F4AC9A4378E1E6E041B1389A53E3450CD32D9D2941B0CBABDB50DA8EA2513145164C3AB6BCBD251C448D2D4B087AC57A59C2285D564F16DA4ED5E607ED979592146FFB0EF3F3DB308FB342DF5EB5924A48256FC763141A278814C82D6D6348577545870AE3A83C7230AC02A1540FE1798F7EF09E335A865A2AE0949B21E4F748FB8A51F44750E213A8FB"),
             SHEX("E62CFE44B169A061FAEDFE6CE667056BC0255D1F22A56D3DF255FFBB145433FBAC3DBC0B998A40001B805927BDC1EA47762980586CA088D64BABC0ACBD488C5F08B8C220FD5A15EC679448532BB992360C8B166BD8EFB59F2B5E091F06CB1DF8D79FF9B69C4A49A9ECECFD6403D6F984A987C0FCE8E6ED4C0E9CAEDE1C522E4F71495F8599F2864C42809C931771A32FCA62B0CCEB02E5D7FFA15754E6AFA27E568D84FB041CD35140A48F4B23204CAB939B5C48217711A5426F194CCBF07491C0DC2F885156B40C9F782F4A17BB7108E465D9AE1F52B494C74611A1D9D379EA8835E0F98AC6880C339D70B65436A07F056E9C118A89AA41A0E4463482B5B037E96736D265CAECF4764E6F206AB47EB83689E3380DB572BF815F7E2ACC67BBA3F4D074DF16D3B95AED1C4718EA77D1E38608400EA88C6F5FFA084EAEE206EEC8D2560A601587DDC5804D86912E4B844D87F2CBF10634CC32F492BBFC32B4E746EE3FE9E9C9B03DA7DE441A53C0E3B648366A4A997A28DF2A10628406C530175191F230A97E2248226ECF6293D39809B0DC0493CEEC48337468831A5EEF0038AFFC160732C097CF96BBB90F5A302406FFE8C0F1FBD7554C1C0679D2AF199E565D96AC5697F2FF748CF0BDD87A0DA1A9DE71020DA59BF69ADC9E7FB7D6B8A38113C3370A120815DDAAC8CE49ED4738C3A2805100DE7C6B2279AAAC367A930CADCC"));
  test_xof (&nettle_shake128, /* 248 octets */
             SHEX("6B860D39725A14B498BB714574B4D37CA787404768F64C648B1751B353AC92BAC2C3A28EA909FDF0423336401A02E63EC24325300D823B6864BB701F9D7C7A1F8EC9D0AE3584AA6DD62EA1997CD831B4BABD9A4DA50932D4EFDA745C61E4130890E156AEE6113716DAF95764222A91187DB2EFFEA49D5D0596102D619BD26A616BBFDA8335505FBB0D90B4C180D1A2335B91538E1668F9F9642790B4E55F9CAB0FE2BDD2935D001EE6419ABAB5457880D0DBFF20ED8758F4C20FE759EFB33141CF0E892587FE8187E5FBC57786B7E8B089612C936DFC03D27EFBBE7C8673F1606BD51D5FF386F4A7AB68EDF59F385EB1291F117BFE717399"),
             SHEX("E55556489CFF55AE9DF1AB48F860E81C3EB95D979929BED9C747FD1467D201FCF041D0B3D72EF227C82C93E6346CB02A5C78AFCF0B9CB6AF551E9645C09244EA67E02E17BF1941F3910B069C04DCD5FF7B4B9C69BB973EFA516B4AFF0F8947DA042E21BF838A94BC8D98E6A30FA830FB9774D14CED360125DA945F44528B99DF2095C5186776D434A4AB2202A8CD5443FF2636F6111228BF1F5C5BDFBCF6278598733776BB589230D74314083B983CBBC476E4C9F0996A01DDE09C8341DA4C2382923FBF32A1AB8E37E9A18DA38F54154C44FAB88AD8857EC5FF82CBE1C9D24D6AA676978FE7F18F1C2503AF6981551710690DB902F46AA5AB3856EBBA8EBC644C7339D6E28EEB2AEB6D5F168E0447CBEA439ECC89743BFDFA88F963A6B57F28157F55CF46B57C2396A357A8E44EC964CA32C51FA5DC17C1E0F6F80FA2C8D26E04F03C32CD763F6CCF0F16F915D72F6E9159174B69535D2229C8A543E992AB0E86BB51DA378EB378A0F28ED49A8F57F224E03C9C1248F0133D70AF45AC7F0D27C5804CC99E982A2A583AE566262F3CFBDB91BB0C2F2561651A812CBE22F1380929F85A94FA721AFE6BA1EE27430F2CFE89966A9919AB21036E81962AFDA1026B3B4A95FDF0BA51940053F01EEFCAC57CD11C4C60D66DE753F41A178CC56A938A72A0791874EAB6DAD49AF265CFC6A03016F0A1A490ECECD045B9F7509F1193F9"));
  test_xof (&nettle_shake128, /* 249 octets */
             SHEX("6A01830AF3889A25183244DECB508BD01253D5B508AB490D3124AFBF42626B2E70894E9B562B288D0A2450CFACF14A0DDAE5C04716E5A0082C33981F6037D23D5E045EE1EF2283FB8B6378A914C5D9441627A722C282FF452E25A7EA608D69CEE4393A0725D17963D0342684F255496D8A18C2961145315130549311FC07F0312FB78E6077334F87EAA873BEE8AA95698996EB21375EB2B4EF53C14401207DEB4568398E5DD9A7CF97E8C9663E23334B46912F8344C19EFCF8C2BA6F04325F1A27E062B62A58D0766FC6DB4D2C6A1928604B0175D872D16B7908EBC041761187CC785526C2A3873FEAC3A642BB39F5351550AF9770C328AF7B"),
             SHEX("5C3CD7AEFDDD08E09F343A8226155BC3BA959998299AED4CE4E85C6758833685A58E5CB8002FA4788ECB673E810C98BDF71A18F7A2DDFC351056D0639BC600FA42BD03B1315CC4C9640A8D61D481BC7F9077C9EFDEA6FC3993993CEC88481FCBA0E4C519CFBB57B465FD1E3D33557D8DA77321CB21E584E677F3A66D38C9E49D9A9545D6B84AFA37547F1CEB22C2ED9D8D97693A69EAA2D15F376E6DE77B0EE6C7DC28FB1C1BCB42BF8751400EA2470B87F19830A6EA33DA0912DC894892C6C242E90D24238439C6794243A5A6512E3486680F4F9166150502A1BC049F48EC80D2694FF8334515EB4A97D483E7E401BF6E446035FDDB8B600CA7478B8C40AEC176B79CC671E90CAAE5D1EC05E51D5E7EF9D8B4FF3B3B319D80C093A83A3D21F22EB4DD3B88FAB1345B29AB527F8D69CABE8FECE7E26D40A63B3FA08340501A40D433D77DD212FFB48265901D9A05E17E8F24786C7C23E50CFA11D5AD497D98579B02B642D48FD74CA608DC32E76C3961E07064168B898FDA5B4FEC10554F32426DD818B7E207F3FACAD3525577B5FBC19F1CFD39FE6453B274362C456B8FDB3AA26306E306EF1BFF87056AE99FD0CC9BBE36EC508532E8239B3C33B1705D6DA65813621E8D319460CB6BBE94C9CC0A5AB5C9CF538775D42068D9665239E1FAD0EE8FCA7EA4829C8BE493AC2DE44B340028DB1F27F9E4470D3EE6AB48B416CE36"));
  test_xof (&nettle_shake128, /* 250 octets */
             SHEX("B3C5E74B69933C2533106C563B4CA20238F2B6E675E8681E34A389894785BDADE59652D4A73D80A5C85BD454FD1E9FFDAD1C3815F5038E9EF432AAC5C3C4FE840CC370CF86580A6011778BBEDAF511A51B56D1A2EB68394AA299E26DA9ADA6A2F39B9FAFF7FBA457689B9C1A577B2A1E505FDF75C7A0A64B1DF81B3A356001BF0DF4E02A1FC59F651C9D585EC6224BB279C6BEBA2966E8882D68376081B987468E7AED1EF90EBD090AE825795CDCA1B4F09A979C8DFC21A48D8A53CDBB26C4DB547FC06EFE2F9850EDD2685A4661CB4911F165D4B63EF25B87D0A96D3DFF6AB0758999AAD214D07BD4F133A6734FDE445FE474711B69A98F7E2B"),
             SHEX("83D754C973E2F1CEA0A8EEF4CB2340F2BF78CB88CB3AA458E7C915B5C8412CEE403CB18E062D2D896D5BFFE7DD1C3CA2E5E37E193554A176BEB2C636ABB687E867AF6B9EB5FBA57F371C167CF210956C5855F27FBE890708A28321BBF2D1D44DC7A911906CC4054D6310CDDC4573EBFC9F7227031EE0EAED3A810AA18367D2F6F56B478881F628897E2678FAC39F76A745460DA7913374C9AF81415592891035F93286BFF02DC737EB087DB8FE7E77A188C206CAEC84832A120547D967107A49AF42D422EE7364DB5B6592185C07DC7FBC2D0ABC2A02C3CCE0070C23F2DAFF0C8FC61563CB7CE0E930B75C706AB8180BCECD4E2C47859217B42719EF341765FF7C59C239C1752A3CA73E04DA7918B61A4D98DF5481F3B2A23A4797D66786CCF940ED96D6817D6172F748262448A69844E59CE9673CD9AA3F5AEE057608D4DF64042D78A03920FA23DE7BC65881465B31C4204F68D69CDDCFAEB9A5A6F4AEAC6572DF206802403D169EB5B2B69CFE3FDFCDE19B198385605C864B5F02BDCABBF646C5945899F52830C3CDA909443C96DF6DC1417677BD80675621755D47D076DF56DAEDA124A3D6B1C7B6E050F6D3024C5627A62C57C94438A8A3C14C5F572210893BAFF4B7E4E84C99C6CA09EB3629087F1C700513F2447525EE23622099E8D8B5D2CA8918CC5775FEB5F34FE5141CCC93EC172168638A736CEB0541A1B18369"));
  test_xof (&nettle_shake128, /* 251 octets */
             SHEX("83AF34279CCB5430FEBEC07A81950D30F4B66F484826AFEE7456F0071A51E1BBC55570B5CC7EC6F9309C17BF5BEFDD7C6BA6E968CF218A2B34BD5CF927AB846E38A40BBD81759E9E33381016A755F699DF35D660007B5EADF292FEEFB735207EBF70B5BD17834F7BFA0E16CB219AD4AF524AB1EA37334AA66435E5D397FC0A065C411EBBCE32C240B90476D307CE802EC82C1C49BC1BEC48C0675EC2A6C6F3ED3E5B741D13437095707C565E10D8A20B8C20468FF9514FCF31B4249CD82DCEE58C0A2AF538B291A87E3390D737191A07484A5D3F3FB8C8F15CE056E5E5F8FEBE5E1FB59D6740980AA06CA8A0C20F5712B4CDE5D032E92AB89F0AE1"),
             SHEX("C6D3296ECCE00F96E8E0224FC094279AFDC51F4D65790B7DE6612A536D16DCA57EDF5AF53B7DCFF98910F8C41AF597692C5B9590B7EBD712076A74BA9923C364E042286F754143B2F1E00C47D4D01ED6A08FC3B62918D4C7FC8A5AFEADF7B83A02ACBFDAD3032D72C60E5FF342AE9376C60CAC094780D9F0A6FCFE63496A1FC660A33E1111FF441BDB600B37A33256A35E86CB245CE8DD2951E6B4C7C96C8561207D402D143902F085191084EE39E8FB472965F51C6E556CF4EAE55C540ADCEDEB9E77699C161A88DD07093251DBF403E7A26EA6FF93B2E5C61E5C0538CC29D69DE806D995C9BB59B52915A61B9DAAA3B21FC325AE7E1D5923D7E2CDB4F71E9C1E9DEB331916F09B22A34CA70FD20410EEDB22118D606870188BBB9800445B136FFEF3D7539B7104EED36E3E663B5167A5649B0FD2013424153B92BF520845972C146F8E15670BE40CF2EF1E73E23E404F1753833ABEB111327E862904C7CA969F5820B41C6484AFCE6374C72620C1B0DF72C043C1BEF6B33EB17A64B0FBE48D68E60F90CFA669214F96709777B9A278B65F197956345395008272E6CC3DDF43DF36B39E491897323CEA03B88D2FB8FB9F9F1219E951F4CD652AA28C1CD03A369D85C7ECEE4C3016C89885373ACABC20E40DEA4BCFA7B499941D8BD06738A71F3B40ED89FFE85C7C84ECF5F644CF1F3A434D2B12EA3339303353890B3C4C5534"));
  test_xof (&nettle_shake128, /* 252 octets */
             SHEX("A7ED84749CCC56BB1DFBA57119D279D412B8A986886D810F067AF349E8749E9EA746A60B03742636C464FC1EE233ACC52C1983914692B64309EDFDF29F1AB912EC3E8DA074D3F1D231511F5756F0B6EEAD3E89A6A88FE330A10FACE267BFFBFC3E3090C7FD9A850561F363AD75EA881E7244F80FF55802D5EF7A1A4E7B89FCFA80F16DF54D1B056EE637E6964B9E0FFD15B6196BDD7DB270C56B47251485348E49813B4EB9ED122A01B3EA45AD5E1A929DF61D5C0F3E77E1FDC356B63883A60E9CBB9FC3E00C2F32DBD469659883F690C6772E335F617BC33F161D6F6984252EE12E62B6000AC5231E0C9BC65BE223D8DFD94C5004A101AF9FD6C0FB"),
             SHEX("E7B210D8AC4395B2DE4A9066A345B1592A5C0649D47EC8392816A65C03B5717FFED2427F513EC34D3DB5AD59AF61E0EC5DF9664A1F24E50A31C6C146799745465E3A130C84B52EDC04140AAD1A8AFDBC22D816339D5AD126C616B55302CEA748E744C6F2F538E2285538697ED1B2E169D9B335B6194B59775CB19B14E2A5BB0754705E1342E25EE1824C75479D317CE844D477599CC6A637604DFDEAAA02A3CD2C1B707661DBD2B1FDD77D2187C1E2295D92D98D4E5AD4208DB1771713BBAF33E63F16163F63EC956C7DB48D265E53058363A923BD51BC3602B184EAF2B8855DFB15AB9015D5FA72999CECFDDE888F95252885988052C9F1666D4E90F82C1BA1C27E4B0FF07F99BC4A54B9FDE4BE5C3882C395F69AE3754576F65CA8CB03EB2890447CCFB37A475748742AC932B615537B7FFDBD9E73FEB1BD29A8039B1321F8B7F6BD80BCE4522174DDBC4D6E95F18DA7EA0D4DAE4D14AFDE65F8BA726662563C225DE20A218927E2CEEF7573AF8313DB530D5FEB42BB2FB7B4DD616D25DBCC6F443A9BC91BA616B858AD827B049A65916CFD0A823A2726243D8393E248A8E1A66A057350107D01B65F4BFA03E6F22A4C26645B7C2CC63767E89F2A8860632226CD6FA6A68DB79949AFFD3EF91758168A295A5342E10C402F2B2076ADC203BA107A177D6FA79F12DE20D084CB463BF0C74B9D3FF6ED9455301A850008499CEE"));
  test_xof (&nettle_shake128, /* 253 octets */
             SHEX("A6FE30DCFCDA1A329E82AB50E32B5F50EB25C873C5D2305860A835AECEE6264AA36A47429922C4B8B3AFD00DA16035830EDB897831C4E7B00F2C23FC0B15FDC30D85FB70C30C431C638E1A25B51CAF1D7E8B050B7F89BFB30F59F0F20FECFF3D639ABC4255B3868FC45DD81E47EB12AB40F2AAC735DF5D1DC1AD997CEFC4D836B854CEE9AC02900036F3867FE0D84AFFF37BDE3308C2206C62C4743375094108877C73B87B2546FE05EA137BEDFC06A2796274099A0D554DA8F7D7223A48CBF31B7DECAA1EBC8B145763E3673168C1B1B715C1CD99ECD3DDB238B06049885ECAD9347C2436DFF32C771F34A38587A44A82C5D3D137A03CAA27E66C8FF6"),
             SHEX("8DD1F85C100C7F05B59A9A0A78581B5541ECE6A40672D9AF23E9F80A9D3FC96346E1B0F3DC309BB110E3F9F465E08371D93616258FC5CF5F3264B6F20CD795FACECABAD03BF361324E242F098FE715E14D0D207B5122B01E47238E8AD8CD9C2E874E816C970F8CABA5C148B9D2638ADC5D13384DD5867679B5821A37B77B5C8AE773F982D07F037E319B0C2A88B8716D8642EDB7C51AFDB44297AA2AD22ED1DEFFEBBB8A89A601B3C8ED2BC1894AD8EE27D85E4B88547D64394709532582EB95A682ACEB7C205F4A3CC9693E443D7CBB1E1DEF42905622AC0E12D0543BE1786C13A913D45C63CD7F06E058440B34DD44007CD2B7BA489D89358276A3D597E19A0B973F558B4BEE867712543C1BDC817495D3222D4472B00784D7CFD83BBCBCE7E2FCD0F9C1EC0FD003B3D8EC74233A3BF7E95430189EAAB5CE341739C8F198052A783B9EC0331CDC1E2BD8B25FBE9B471A768CC6B8F2C90298196BC65861678C7632B7914D365DE4BB2CF9DBC4A0A06034711086A0D09A9C139132556EFD7CEDA2933EB9B86AD5BA22196DF404684993A8E7FECFE6E582E5A3A4717753E5F3BEF462DD8820BF3857881D54AF1D0F2ACE27AB1C208C3D5087DB201F155164C398F2ADE81CE38CE0D741920364C57580522FE1F0299A6263C3F4A78E03300372D379A4A9A550FFAEEDA27B30B329435A8B2E3C127568B48B1A75462F576DAFDDF6"));
  test_xof (&nettle_shake128, /* 254 octets */
             SHEX("83167FF53704C3AA19E9FB3303539759C46DD4091A52DDAE9AD86408B69335989E61414BC20AB4D01220E35241EFF5C9522B079FBA597674C8D716FE441E566110B6211531CECCF8FD06BC8E511D00785E57788ED9A1C5C73524F01830D2E1148C92D0EDC97113E3B7B5CD3049627ABDB8B39DD4D6890E0EE91993F92B03354A88F52251C546E64434D9C3D74544F23FB93E5A2D2F1FB15545B4E1367C97335B0291944C8B730AD3D4789273FA44FB98D78A36C3C3764ABEEAC7C569C1E43A352E5B770C3504F87090DEE075A1C4C85C0C39CF421BDCC615F9EFF6CB4FE6468004AECE5F30E1ECC6DB22AD9939BB2B0CCC96521DFBF4AE008B5B46BC006E"),
             SHEX("16322D3005BEF6A6CD077C13F13A0DBA11CAC621605D78E2CA3EBB07386C0ACC23D0B21573123E1BD43ACCC1F12DBCC34874B5C18B6D037B339E958972902BA3174A9831EAAB3543B901B6CF4EF5CBD1CE1B60AC90E94F6E15C7DAD13AC4B8AFFAB3141326AE057852DAB28FB1564A7375204805747A45C0E96E2E046611FFB1BDDB6E6EF5C0BD75E4D654701BCD7F903DAA5B2D32E017C5A81A907FA908C8523D821A9947E647767F04FE6A8D0A9297F65610B1ECB332D3DD37EC4501400C3C1E0035A48652DE4F0BC038C90C5B4FC2D969B1951DE5400184E6C761C7129711ECF3F4146C2B8D662C96A30491771DAD21E0D9667F7312147D53B1377999F0274D239ADA0E2E2843676ECEC0D528C89A5E6C4EA111ACCFDEA9D10BD3F79326283A609F1E2C9FDD4120367DC5403D53A6D0B0DFC3A3726096EC7F6CE189A29F23053F79E179F275E32DB13567E108A63C7033286C4F6636AFA91C37E32A5B2BF01995133FAD11A96BA922CE4BF5192A0536F4F238F9CF0AEAD2265AE73FF3923682644CB414B8E92E9C5EB8D46E39E46A5A394D75BA2D01A18B2309DEC99931FD9C29DD61073B23A19EDCBACCA8D9227E8FE23D313947AB6D9C807C8C3BC8E189158A1815410A94CAC21D931560BAE3AFE409D4904C32D4AA5C7FF6463EC8C432F97EFFD812DDBBDE53EA65956C9C2AF72ADB2C7B5BC1EFCFC8EF4FF8A152AEB8"));
  test_xof (&nettle_shake128, /* 255 octets */
             SHEX("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
             SHEX("14236E75B9784DF4F57935F945356CBE383FE513ED30286F91060759BCB0EF4BAAC858ECAE7C6E7EDD498F01A082B63FA57D22540231E2E25C83EFB3B3F2953A5F674502AB635226446B84937643DCD5789EE73F1D734BC8FE5F7F0883AB10961B9A31FF60DEE16159BC6982EFB08545984BF71FED1C4CD81C0914B4C19FCFEEF54AF4BBE372F18CFCD3A18657F5B9450F99A78F0FA2C3CDCA7461C4ED7569536883B66CD87E9C200962902EAA16A54DB6A0A5CC26D889038C0760810B5BB4F33F1E5D639B6F9BC7CA62BA6F8C9F8DE770260AFE47F4E0F82F102198EBA27F543252AC8DDD83E1B8DB0A91AC65633FD12A550EBE96F93AA6704ED5905C234FA6D9203910CBD02DE166C4C3348FB81EF7B84AE1455FE318B5FD170883F49BA2F24289C479A2C7531406BA989BEAEF3A79F659028642E9B033F7DEB9ECEC3A7A9F1DBD2451FCB47C81E21E91D20B924C6BD04C1F0B2710D2E570CD24BAD5B5DE4E49AA80B6ADD5507B4D2E510370C7AFA814D7E1A7E278E53D7CCF49A0A866CA3A7B5BB71EF3425E460FEEB29149F217066613695F85506A0946CF68979F04AE073AF8028976BF0C5BDC2212E8C364583DE9FBD03B34DDEE5EC4CFA8ED8CE592971D0108FAF76C8940E25E6C5F865584C34A233C14F00532673FDBE388CC7E98A5B867B1C591307A9015112B567FF6B4F318114111FC95E5BD7C9C60B74C1F8725"));
}
