/* packfuncs.c: -*- C -*-  Functions which manipulate packages. */

/*  Copyright (c) 1997 Brian J. Fox
    Author: Brian J. Fox (bfox@ai.mit.edu) Tue Jul 22 22:01:08 1997.

    This file is part of <Meta-HTML>(tm), a system for the rapid deployment
    of Internet and Intranet applications via the use of the Meta-HTML
    language.

    Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
    Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

    Meta-HTML is free software; you can redistribute it and/or modify
    it under the terms of the UAI Free Software License as published
    by Universal Access Inc.; either version 1, or (at your option) any
    later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    UAI Free Software License for more details.

    You should have received a copy of the UAI Free Software License
    along with this program; if you have not, you may obtain one by
    writing to:

    Universal Access Inc.
    129 El Paseo Court
    Santa Barbara, CA
    93101  */

#include "language.h"

static void pf_package_names (PFunArgs);
static void pf_package_vars (PFunArgs);
static void pf_package_delete (PFunArgs);
static void pf_in_package (PFunArgs);
static void pf_with_local_package (PFunArgs);
static void pf_package_to_alist (PFunArgs);
static void pf_alist_to_package (PFunArgs);

/************************************************************/
/*							    */
/*		  Package Manipulation Functions	    */
/*							    */
/************************************************************/

static PFunDesc func_table[] =
{
  { "PACKAGE-NAMES",		0, 0, pf_package_names },
  { "PACKAGE-VARS",		0, 0, pf_package_vars },
  { "PACKAGE-DELETE",		0, 0, pf_package_delete },
  { "IN-PACKAGE",		1, 0, pf_in_package },
  { "WITH-LOCAL-PACKAGE",	1, 0, pf_with_local_package },
  { "PACKAGE-TO-ALIST",		0, 0, pf_package_to_alist },
  { "ALIST-TO-PACKAGE",		0, 0, pf_alist_to_package },

  { (char *)NULL,		0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_package_functions)
DOC_SECTION (PACKAGES)

DEFUN (pf_package_names, ,
"Returns a newline separated list of the all of the named packages
which are currently defined.  Because the list is newline separated,
the result can easily be assigned to an array variable:

<example>
<set-var all-packages[]=<package-names>>
</example>")
{
  if (AllPackages)
    {
      register int i;
      Package *pack;

      for (i = 0; (pack = AllPackages[i]) != (Package *)NULL; i++)
	if (pack->name != (char *)NULL)
	  {
	    bprintf_insert (page, start, "%s\n", pack->name);
	    start += strlen (pack->name) + 1;
	  }

      *newstart = start;
    }
}

DEFUN (pf_package_vars, &optional package-name &key strip=true,
"Returns a newline separated list of the fully qualified variable
names found in the package named by <var package-name>, or in the
current package if <var package-name> is not given.  When <var
strip=true> is supplied, the returned variable names have the package
prefix stripped off, making them <i>not</i> fully qualified.  The
names are not returned in any significant order.  Because the list is
newline separated, the results can easily be assigned to an array
variable:

<complete-example>
<set-var foo::bar=baz>
<set-var foo::baz=bar>
<set-var names[]=<package-vars foo>>
<get-var names[1]>
</complete-example>")
{
  register int pos = 0;
  char *strip = get_value (vars, "STRIP");
  char *name;

  if ((CurrentPackage != (Package *)NULL) &&
      (get_positional_arg (vars, 0) == (char *)NULL))
    {
      Symbol **symbols = symbols_of_package (CurrentPackage);

      if (symbols != (Symbol **)NULL)
	{
	  register int i;

	  for (i = 0; symbols[i] != (Symbol *)NULL; i++)
	    {
	      bprintf_insert (page, start, "%s\n", symbols[i]->name);
	      start += symbols[i]->name_len + 1;
	    }

	  free (symbols);
	}
    }

  while ((name = get_positional_arg (vars, pos)) != (char *)NULL)
    {
      Package *pack = (Package *)NULL;

      pos++;

      name = mhtml_evaluate_string (name);

      if (!empty_string_p (name))
	pack = symbol_lookup_package (name);

      if (pack)
	{
	  Symbol **symbols = symbols_of_package (pack);

	  if (symbols != (Symbol **)NULL)
	    {
	      register int i;

	      for (i = 0; symbols[i] != (Symbol *)NULL; i++)
		{
		  if ((pack->name[0] != '\0') && (strip == (char *)NULL))
		    {
		      bprintf_insert (page, start, "%s::%s\n",
				      pack->name, symbols[i]->name);
		      start += pack->name_len + 3 + symbols[i]->name_len;
		    }
		  else
		    {
		      bprintf_insert (page, start, "%s\n", symbols[i]->name);
		      start += symbols[i]->name_len + 1;
		    }
		}

	      free (symbols);
	    }
	}

      if (name) free (name);
    }

  *newstart = start;
}

DEFUN (pf_package_delete, package-name...,
"Remove the definition of the packages named by <var package-name>s,
and all of the variables defined within them.")
{
  char **names = get_vars_names (vars);

  if (names != (char **)NULL)
    {
      register int i;

      for (i = 0; names[i] != (char *)NULL; i++)
	{
	  char *name = names[i];

	  name = mhtml_evaluate_string (name);

	  if (name)
	    {
	      pagefunc_destroy_package (name);
	      free (name);
	    }
	}
    }
}

DEFMACRO (pf_in_package, package-name,
"Evaluate <var body> in an environment where variables which are not
specifically prefixed with a package name are looked up and stored
within <var package-name>.

The special package name <code>\"local\"</code> creates an anonymous
package within which to work.  The contents of local packages are only
accessible within the expressions surrounded by the
<code>in-package</code> operator.")
{
  int jump_again = 0;
  char *packname = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *result = (char *)NULL;

  if (empty_string_p (packname))
    {
      if (packname) free (packname);
      packname = strdup ("DEFAULT");
    }

  if (strcasecmp (packname, "local") == 0)
    {
      free ((char *)packname);
      packname = (char *)NULL;
    }

  {
    PageEnv *page_environ = pagefunc_save_environment ();

    symbol_push_package (symbol_get_package (packname));

    if ((jump_again = setjmp (page_jmp_buffer)) == 0)
      result = mhtml_evaluate_string (body->buffer);

    symbol_pop_package ();

    pagefunc_restore_environment (page_environ);
  }

  if (result != (char *)NULL)
    {
      if (jump_again == 0)
	bprintf_insert (page, start, "%s", (char *)result);
      free ((char *)result);
    }

  if (packname) free (packname);
  if (jump_again) longjmp (page_jmp_buffer, 1);
}

DEFMACRO (pf_with_local_package, ,
"Shorthand for <example code><in-package local> <i>body</i>
</in-package></example>")
{
  int jump_again = 0;
  char *result = (char *)NULL;

  {
    PageEnv *page_environ = pagefunc_save_environment ();

    symbol_push_package (symbol_get_package ((char *)NULL));

    if ((jump_again = setjmp (page_jmp_buffer)) == 0)
      result = mhtml_evaluate_string (body->buffer);

    symbol_pop_package ();
    pagefunc_restore_environment (page_environ);
  }

  if (result != (char *)NULL)
    {
      if (jump_again == 0)
	bprintf_insert (page, start, "%s", (char *)result);
      free ((char *)result);
    }
  if (jump_again) longjmp (page_jmp_buffer, 1);
}

DEFUN (pf_package_to_alist, &optional package,
"Returns a Lisp readable string containing the names and values of the variables in <var package>.  If <var strip=true> is supplied, the package name is removed from the variables before placing them in the list.  See the following code sequence:

<complete-example>
<set-var
  foo::bar=baz
  foo::array[0]=Elt-0
  foo::array[1]=Elt-1>

  The contents of Foo: <package-to-alist foo>
The stripped contents: <package-to-alist foo strip=true>
</complete-example>")
{
  char *packname = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *strip = get_value (vars, "STRIP");
  char *result = (char *)NULL;
  Package *package = (Package *)NULL;

  if (!empty_string_p (packname))
    package = symbol_lookup_package (packname);
  else
    package = CurrentPackage;

  if (packname != (char *)NULL) free (packname);

  if (package != (Package *)NULL)
    result = package_to_alist (package, (strip != (char *)NULL));

  if (result)
    {
      bprintf_insert (page, start, "%s", result);
      *newstart += strlen (result);
      free (result);
    }
}

DEFUN (pf_alist_to_package, alist &optional package,
"Takes the textual list representation of a package, and creates (or
modifies) the package named by <var package-name>.

<code>alist-to-package</code> is the inverse of the <funref packages
package-to-alist> function -- given an \"alist\" (short for
`association list') you can create a package, and vice-versa.  The
following expression is one way to copy all of the variables from the
package <code>FOO</code> into the package <code>BAR</code>:

<example>
<alist-to-package <package-to-alist foo> bar>
</example>")
{
  char *alist = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *packname = mhtml_evaluate_string (get_positional_arg (vars, 1));

  if (!empty_string_p (alist))
    {
      Package *from = (Package *)NULL;
      Package *to = (Package *)NULL;

      from = alist_to_package (alist);

      if (!empty_string_p (packname))
	to = symbol_get_package (packname);
      else
	to = CurrentPackage;

      if (from && to)
	{
	  Symbol **symbols = symbols_of_package (from);

	  if (symbols != (Symbol **)NULL)
	    {
	      register int i;
	      Symbol *sym, *copy;

	      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
		{
		  char *sym_name = sym->name;
		  char *temp = strstr (sym_name, "::");

		  if (temp)
		    sym_name = temp + 2;

		  copy = symbol_copy (sym, to);
		  if (temp)
		    symbol_rename (copy, sym_name);
		}
	      free (symbols);
	    }

	  symbol_destroy_package (from);
	}
    }

  xfree (alist);
  xfree (packname);
}
