/*
 * Copyright (C) 2017 Red Hat, Inc.
 *
 * Author: Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * GnuTLS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuTLS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GnuTLS.  If not, see <https://www.gnu.org/licenses/>.
 */

/* This checks the old low level DN encoding and decoding routines */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include <string.h>
#include <gnutls/gnutls.h>

#include "utils.h"

static char buf[32 * 1024];

static void decode(const char *test_name, const gnutls_datum_t * raw,
		   const char *expected, const char *expected_compat)
{
	int ret;
	gnutls_datum_t out;
	size_t bsize;

	ret = gnutls_x509_rdn_get2(raw, &out, 0);
	if (ret < 0) {
		test_fail("%s\n", gnutls_strerror(ret));
	}

	if (out.size != strlen(expected)) {
		test_fail
		    ("The length of the output (%d) doesn't match the expected (%d)\n",
		     (int)out.size, (int)strlen(expected));
	}

	if (memcmp(out.data, expected, out.size) != 0) {
		test_fail
		    ("The string output (%s) doesn't match the expected (%s)\n",
		     (char *)out.data, expected);
	}

	if (out.data[out.size] != 0) {
		test_fail("The string output isn't null terminated\n");
	}

	gnutls_free(out.data);

	/* compat mode */
	ret = gnutls_x509_rdn_get2(raw, &out, GNUTLS_X509_DN_FLAG_COMPAT);
	if (ret < 0) {
		test_fail("%s\n", gnutls_strerror(ret));
	}

	if (out.size != strlen(expected_compat)) {
		test_fail
		    ("The length of the output (%d) doesn't match the expected (%d)\n",
		     (int)out.size, (int)strlen(expected_compat));
	}

	if (memcmp(out.data, expected_compat, out.size) != 0) {
		test_fail
		    ("The string output (%s) doesn't match the expected (%s)\n",
		     (char *)out.data, expected_compat);
	}

	if (out.data[out.size] != 0) {
		test_fail("The string output isn't null terminated\n");
	}

	gnutls_free(out.data);

	/* compat mode 2 */
	bsize = sizeof(buf);
	ret = gnutls_x509_rdn_get(raw, buf, &bsize);
	if (ret < 0) {
		test_fail("%s\n", gnutls_strerror(ret));
	}
	out.data = (void *)buf;
	out.size = bsize;

	if (out.size != strlen(expected_compat)) {
		test_fail
		    ("The length of the output (%d) doesn't match the expected (%d)\n",
		     (int)out.size, (int)strlen(expected_compat));
	}

	if (memcmp(out.data, expected_compat, out.size) != 0) {
		test_fail
		    ("The string output (%s) doesn't match the expected (%s)\n",
		     (char *)out.data, expected_compat);
	}

	if (out.data[out.size] != 0) {
		test_fail("The string output isn't null terminated\n");
	}

	return;
}

struct tests_st {
	const char *name;
	gnutls_datum_t raw;
	const char *str;
	const char *compat_str;	/* GNUTLS_X509_DN_FLAG_COMPAT */
	unsigned can_encode;
};

struct tests_st tests[] = {
	{
	 .name = "simple DN",
	 .str =
	 "C=GR,ST=Attiki,O=Koko inc.,OU=sleeping dept.,UID=clauper,CN=Cindy Lauper",
	 .compat_str =
	 "CN=Cindy Lauper,UID=clauper,OU=sleeping dept.,O=Koko inc.,ST=Attiki,C=GR",
	 .raw = {(void *)
		 "\x30\x7b\x31\x15\x30\x13\x06\x03\x55\x04\x03\x13\x0c\x43\x69\x6e\x64\x79\x20\x4c\x61\x75\x70\x65\x72\x31\x17\x30\x15\x06\x0a\x09\x92\x26\x89\x93\xf2\x2c\x64\x01\x01\x13\x07\x63\x6c\x61\x75\x70\x65\x72\x31\x17\x30\x15\x06\x03\x55\x04\x0b\x13\x0e\x73\x6c\x65\x65\x70\x69\x6e\x67\x20\x64\x65\x70\x74\x2e\x31\x12\x30\x10\x06\x03\x55\x04\x0a\x13\x09\x4b\x6f\x6b\x6f\x20\x69\x6e\x63\x2e\x31\x0f\x30\x0d\x06\x03\x55\x04\x08\x13\x06\x41\x74\x74\x69\x6b\x69\x31\x0b\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52",
		 125},
	 },
	{
	 .name = "UTF8 DN",
	 .str = "C=GR,ST=Αττική,O=Μεγάλη εταιρία,CN=🐨",
	 .compat_str =
	 "CN=🐨,O=Μεγάλη εταιρία,ST=Αττική,C=GR",
	 .raw = {(void *)
		 "\x30\x59\x31\x0d\x30\x0b\x06\x03\x55\x04\x03\x0c\x04\xf0\x9f\x90\xa8\x31\x24\x30\x22\x06\x03\x55\x04\x0a\x0c\x1b\xce\x9c\xce\xb5\xce\xb3\xce\xac\xce\xbb\xce\xb7\x20\xce\xb5\xcf\x84\xce\xb1\xce\xb9\xcf\x81\xce\xaf\xce\xb1\x31\x15\x30\x13\x06\x03\x55\x04\x08\x0c\x0c\xce\x91\xcf\x84\xcf\x84\xce\xb9\xce\xba\xce\xae\x31\x0b\x30\x09\x06\x03\x55\x04\x06\x13\x02\x47\x52",
		 91},
	 },
	{
	 .name = "combo DN",
	 .compat_str = "C=\\,\\ ,OU=\\  X\\ ,CN=\\#XXX",
	 .str = "CN=\\#XXX,OU=\\  X\\ ,C=\\,\\ ",
	 .raw = {(void *)
		 "\x30\x2b\x31\x0b\x30\x09\x06\x03\x55\x04\x06\x13\x02\x2c\x20\x31\x0d\x30\x0b\x06\x03\x55\x04\x0b\x13\x04\x20\x20\x58\x20\x31\x0d\x30\x0b\x06\x03\x55\x04\x03\x0c\x04\x23\x58\x58\x58",
		 45},
	 },
	{
	 .name = "very long DN",
	 .compat_str =
	 "C=ES,ST=CACERES,L=CACERES,O=DIPUTACION PROVINCIAL DE CACERES,OU=DIPUTACION PROVINCIAL DE CACERES,CN=www.dip-caceres.es,EMAIL=webmaster@dip-caceres.es,2.5.29.17=#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",
	 .str =
	 "2.5.29.17=#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,EMAIL=webmaster@dip-caceres.es,CN=www.dip-caceres.es,OU=DIPUTACION PROVINCIAL DE CACERES,O=DIPUTACION PROVINCIAL DE CACERES,L=CACERES,ST=CACERES,C=ES",
	 .raw = {(void *)
		 "\x30\x82\x31\x29\x31\x0b\x30\x09\x06\x03\x55\x04\x06\x13\x02\x45\x53\x31\x10\x30\x0e\x06\x03\x55\x04\x08\x13\x07\x43\x41\x43\x45\x52\x45\x53\x31\x10\x30\x0e\x06\x03\x55\x04\x07\x13\x07\x43\x41\x43\x45\x52\x45\x53\x31\x29\x30\x27\x06\x03\x55\x04\x0a\x13\x20\x44\x49\x50\x55\x54\x41\x43\x49\x4f\x4e\x20\x50\x52\x4f\x56\x49\x4e\x43\x49\x41\x4c\x20\x44\x45\x20\x43\x41\x43\x45\x52\x45\x53\x31\x29\x30\x27\x06\x03\x55\x04\x0b\x13\x20\x44\x49\x50\x55\x54\x41\x43\x49\x4f\x4e\x20\x50\x52\x4f\x56\x49\x4e\x43\x49\x41\x4c\x20\x44\x45\x20\x43\x41\x43\x45\x52\x45\x53\x31\x1b\x30\x19\x06\x03\x55\x04\x03\x13\x12\x77\x77\x77\x2e\x64\x69\x70\x2d\x63\x61\x63\x65\x72\x65\x73\x2e\x65\x73\x31\x27\x30\x25\x06\x09\x2a\x86\x48\x86\xf7\x0d\x01\x09\x01\x16\x18\x77\x65\x62\x6d\x61\x73\x74\x65\x72\x40\x64\x69\x70\x2d\x63\x61\x63\x65\x72\x65\x73\x2e\x65\x73\x31\x82\x30\x58\x30\x82\x30\x54\x06\x03\x55\x1d\x11\x13\x82\x30\x4b\x44\x4e\x53\x2e\x31\x3d\x61\x62\x61\x64\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x3d\x61\x62\x65\x72\x74\x75\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x3d\x61\x63\x65\x62\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x3d\x61\x63\x65\x68\x75\x63\x68\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x3d\x61\x63\x65\x69\x74\x75\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x3d\x61\x68\x69\x67\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x3d\x61\x6c\x61\x67\x6f\x6e\x64\x65\x6c\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x3d\x61\x6c\x63\x6f\x6c\x6c\x61\x72\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x3d\x61\x79\x74\x6f\x61\x6c\x62\x61\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x3d\x61\x79\x74\x6f\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x3d\x61\x6c\x63\x75\x65\x73\x63\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x3d\x61\x6c\x64\x65\x61\x63\x65\x6e\x74\x65\x6e\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x3d\x61\x6c\x64\x65\x61\x64\x65\x6c\x63\x61\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x3d\x6c\x61\x61\x6c\x64\x65\x61\x64\x65\x6c\x6f\x62\x69\x73\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x3d\x61\x6c\x64\x65\x61\x6e\x75\x65\x76\x61\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x3d\x61\x6c\x64\x65\x61\x6e\x75\x65\x76\x61\x64\x65\x6c\x63\x61\x6d\x69\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x3d\x61\x6c\x64\x65\x68\x75\x65\x6c\x61\x64\x65\x6c\x6a\x65\x72\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x3d\x61\x79\x74\x6f\x61\x6c\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x3d\x61\x6c\x69\x73\x65\x64\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x3d\x61\x6c\x6d\x61\x72\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x3d\x61\x6c\x6d\x6f\x68\x61\x72\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x3d\x61\x79\x74\x6f\x61\x72\x72\x6f\x79\x6f\x64\x65\x6c\x61\x6c\x75\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x3d\x61\x72\x72\x6f\x79\x6f\x6d\x6f\x6c\x69\x6e\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x3d\x61\x72\x72\x6f\x79\x6f\x6d\x6f\x6c\x69\x6e\x6f\x73\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x3d\x62\x61\x6e\x6f\x73\x64\x65\x6d\x6f\x6e\x74\x65\x6d\x61\x79\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x3d\x62\x61\x72\x72\x61\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x3d\x62\x65\x6c\x76\x69\x73\x64\x65\x6d\x6f\x6e\x72\x6f\x79\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x3d\x62\x65\x6e\x71\x75\x65\x72\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x3d\x62\x65\x72\x72\x6f\x63\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x3d\x62\x65\x72\x7a\x6f\x63\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x3d\x62\x6f\x68\x6f\x6e\x61\x6c\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x3d\x62\x6f\x74\x69\x6a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x3d\x62\x72\x6f\x7a\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x3d\x63\x61\x62\x61\x6e\x61\x73\x64\x65\x6c\x63\x61\x73\x74\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x3d\x63\x61\x62\x65\x7a\x61\x62\x65\x6c\x6c\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x3d\x63\x61\x62\x65\x7a\x75\x65\x6c\x61\x64\x65\x6c\x76\x61\x6c\x6c\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x3d\x63\x61\x62\x72\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x3d\x63\x61\x63\x68\x6f\x72\x72\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x3d\x63\x61\x64\x61\x6c\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x3d\x63\x61\x6c\x7a\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x3d\x63\x61\x6d\x69\x6e\x6f\x6d\x6f\x72\x69\x73\x63\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x3d\x63\x61\x6d\x70\x69\x6c\x6c\x6f\x64\x65\x64\x65\x6c\x65\x69\x74\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x3d\x63\x61\x6d\x70\x6f\x6c\x75\x67\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x3d\x63\x61\x6e\x61\x6d\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x3d\x63\x61\x6e\x61\x76\x65\x72\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x3d\x63\x61\x72\x62\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x37\x3d\x63\x61\x72\x63\x61\x62\x6f\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x38\x3d\x63\x61\x72\x72\x61\x73\x63\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x39\x3d\x63\x61\x73\x61\x72\x64\x65\x63\x61\x63\x65\x72\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x30\x3d\x63\x61\x73\x61\x72\x64\x65\x70\x61\x6c\x6f\x6d\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x31\x3d\x63\x61\x73\x61\x72\x65\x73\x64\x65\x6c\x61\x73\x68\x75\x72\x64\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x32\x3d\x63\x61\x73\x61\x73\x64\x65\x64\x6f\x6e\x61\x6e\x74\x6f\x6e\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x33\x3d\x63\x61\x73\x61\x73\x64\x65\x64\x6f\x6e\x67\x6f\x6d\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x34\x3d\x63\x61\x73\x61\x73\x64\x65\x6c\x63\x61\x73\x74\x61\x6e\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x35\x3d\x63\x61\x73\x61\x73\x64\x65\x6c\x6d\x6f\x6e\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x36\x3d\x63\x61\x73\x61\x73\x64\x65\x6d\x69\x6c\x6c\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x37\x3d\x63\x61\x73\x61\x73\x64\x65\x6d\x69\x72\x61\x76\x65\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x38\x3d\x63\x61\x73\x61\x74\x65\x6a\x61\x64\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x35\x39\x3d\x63\x61\x73\x69\x6c\x6c\x61\x73\x64\x65\x63\x6f\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x30\x3d\x63\x61\x73\x74\x61\x6e\x61\x72\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x31\x3d\x63\x65\x63\x6c\x61\x76\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x32\x3d\x63\x65\x64\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x33\x3d\x63\x65\x72\x65\x7a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x34\x3d\x63\x69\x6c\x6c\x65\x72\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x35\x3d\x63\x6f\x6c\x6c\x61\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x36\x3d\x63\x6f\x6e\x71\x75\x69\x73\x74\x61\x64\x65\x6c\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x37\x3d\x63\x6f\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x38\x3d\x63\x75\x61\x63\x6f\x73\x64\x65\x79\x75\x73\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x36\x39\x3d\x6c\x61\x63\x75\x6d\x62\x72\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x30\x3d\x64\x65\x6c\x65\x69\x74\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x31\x3d\x64\x65\x73\x63\x61\x72\x67\x61\x6d\x61\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x32\x3d\x65\x6c\x6a\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x33\x3d\x65\x73\x63\x75\x72\x69\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x34\x3d\x66\x72\x65\x73\x6e\x65\x64\x6f\x73\x6f\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x35\x3d\x67\x61\x6c\x69\x73\x74\x65\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x36\x3d\x67\x61\x72\x63\x69\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x37\x3d\x6c\x61\x67\x61\x72\x67\x61\x6e\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x38\x3d\x67\x61\x72\x67\x61\x6e\x74\x61\x6c\x61\x6f\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x37\x39\x3d\x67\x61\x72\x67\x61\x6e\x74\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x30\x3d\x67\x61\x72\x67\x75\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x31\x3d\x67\x61\x72\x72\x6f\x76\x69\x6c\x6c\x61\x73\x64\x65\x61\x6c\x63\x6f\x6e\x65\x74\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x32\x3d\x67\x61\x72\x76\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x33\x3d\x67\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x34\x3d\x61\x79\x74\x6f\x65\x6c\x67\x6f\x72\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x35\x3d\x6c\x61\x67\x72\x61\x6e\x6a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x36\x3d\x6c\x61\x67\x72\x61\x6e\x6a\x61\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x37\x3d\x61\x79\x75\x6e\x74\x61\x6d\x69\x65\x6e\x74\x6f\x64\x65\x67\x75\x61\x64\x61\x6c\x75\x70\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x38\x3d\x67\x75\x69\x6a\x6f\x64\x65\x63\x6f\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x38\x39\x3d\x67\x75\x69\x6a\x6f\x64\x65\x67\x61\x6c\x69\x73\x74\x65\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x30\x3d\x67\x75\x69\x6a\x6f\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x31\x3d\x67\x75\x69\x6a\x6f\x64\x65\x73\x61\x6e\x74\x61\x62\x61\x72\x62\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x32\x3d\x68\x65\x72\x67\x75\x69\x6a\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x33\x3d\x68\x65\x72\x6e\x61\x6e\x70\x65\x72\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x34\x3d\x68\x65\x72\x72\x65\x72\x61\x64\x65\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x35\x3d\x68\x65\x72\x72\x65\x72\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x36\x3d\x68\x65\x72\x76\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x37\x3d\x68\x69\x67\x75\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x38\x3d\x68\x69\x6e\x6f\x6a\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x39\x39\x3d\x68\x6f\x6c\x67\x75\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x30\x3d\x68\x6f\x79\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x31\x3d\x68\x75\x65\x6c\x61\x67\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x32\x3d\x69\x62\x61\x68\x65\x72\x6e\x61\x6e\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x33\x3d\x6a\x61\x72\x61\x69\x63\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x34\x3d\x6a\x61\x72\x61\x69\x7a\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x35\x3d\x6a\x61\x72\x61\x6e\x64\x69\x6c\x6c\x61\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x36\x3d\x6a\x61\x72\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x37\x3d\x6a\x65\x72\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x38\x3d\x6c\x61\x64\x72\x69\x6c\x6c\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x30\x39\x3d\x6c\x6f\x67\x72\x6f\x73\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x30\x3d\x6c\x6f\x73\x61\x72\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x31\x3d\x6d\x61\x64\x72\x69\x67\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x32\x3d\x6d\x61\x64\x72\x69\x67\x61\x6c\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x33\x3d\x6d\x61\x64\x72\x6f\x6e\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x34\x3d\x6d\x61\x6a\x61\x64\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x35\x3d\x6d\x61\x6c\x70\x61\x72\x74\x69\x64\x61\x64\x65\x63\x61\x63\x65\x72\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x36\x3d\x6d\x61\x6c\x70\x61\x72\x74\x69\x64\x61\x64\x65\x70\x6c\x61\x73\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x37\x3d\x6d\x61\x72\x63\x68\x61\x67\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x38\x3d\x6d\x61\x74\x61\x64\x65\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x31\x39\x3d\x6d\x65\x6d\x62\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x30\x3d\x6d\x65\x73\x61\x73\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x31\x3d\x6d\x69\x61\x6a\x61\x64\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x32\x3d\x6d\x69\x6c\x6c\x61\x6e\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x33\x3d\x6d\x69\x72\x61\x62\x65\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x34\x3d\x6d\x6f\x68\x65\x64\x61\x73\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x35\x3d\x6d\x6f\x6e\x72\x6f\x79\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x36\x3d\x6d\x6f\x6e\x74\x61\x6e\x63\x68\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x37\x3d\x6d\x6f\x6e\x74\x65\x68\x65\x72\x6d\x6f\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x38\x3d\x6d\x6f\x72\x61\x6c\x65\x6a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x32\x39\x3d\x6d\x6f\x72\x63\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x30\x3d\x6e\x61\x76\x61\x63\x6f\x6e\x63\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x31\x3d\x6e\x61\x76\x61\x6c\x76\x69\x6c\x6c\x61\x72\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x32\x3d\x6e\x61\x76\x61\x6c\x6d\x6f\x72\x61\x6c\x64\x65\x6c\x61\x6d\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x33\x3d\x6e\x61\x76\x61\x73\x64\x65\x6c\x6d\x61\x64\x72\x6f\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x34\x3d\x6e\x61\x76\x61\x74\x72\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x35\x3d\x6e\x61\x76\x65\x7a\x75\x65\x6c\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x36\x3d\x6e\x75\x6e\x6f\x6d\x6f\x72\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x37\x3d\x6f\x6c\x69\x76\x61\x64\x65\x70\x6c\x61\x73\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x38\x3d\x70\x61\x6c\x6f\x6d\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x33\x39\x3d\x70\x61\x73\x61\x72\x6f\x6e\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x30\x3d\x70\x65\x64\x72\x6f\x73\x6f\x64\x65\x61\x63\x69\x6d\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x31\x3d\x70\x65\x72\x61\x6c\x65\x64\x61\x64\x65\x6c\x61\x6d\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x32\x3d\x70\x65\x72\x61\x6c\x65\x64\x61\x64\x65\x73\x61\x6e\x72\x6f\x6d\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x33\x3d\x70\x65\x72\x61\x6c\x65\x73\x64\x65\x6c\x70\x75\x65\x72\x74\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x34\x3d\x70\x65\x73\x63\x75\x65\x7a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x35\x3d\x6c\x61\x70\x65\x73\x67\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x36\x3d\x70\x69\x65\x64\x72\x61\x73\x61\x6c\x62\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x37\x3d\x70\x69\x6e\x6f\x66\x72\x61\x6e\x71\x75\x65\x61\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x38\x3d\x70\x69\x6f\x72\x6e\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x34\x39\x3d\x70\x6c\x61\x73\x65\x6e\x7a\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x30\x3d\x70\x6f\x72\x74\x61\x6a\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x31\x3d\x70\x6f\x72\x74\x65\x7a\x75\x65\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x32\x3d\x70\x6f\x7a\x75\x65\x6c\x6f\x64\x65\x7a\x61\x72\x7a\x6f\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x33\x3d\x70\x75\x65\x62\x6c\x6f\x6e\x75\x65\x76\x6f\x64\x65\x6d\x69\x72\x61\x6d\x6f\x6e\x74\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x34\x3d\x70\x75\x65\x72\x74\x6f\x64\x65\x73\x61\x6e\x74\x61\x63\x72\x75\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x35\x3d\x72\x65\x62\x6f\x6c\x6c\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x36\x3d\x72\x69\x6f\x6c\x6f\x62\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x37\x3d\x72\x6f\x62\x6c\x65\x64\x69\x6c\x6c\x6f\x64\x65\x67\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x38\x3d\x72\x6f\x62\x6c\x65\x64\x69\x6c\x6c\x6f\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x35\x39\x3d\x72\x6f\x62\x6c\x65\x64\x69\x6c\x6c\x6f\x64\x65\x74\x72\x75\x6a\x69\x6c\x6c\x6f\x2c\x44\x4e\x53\x2e\x31\x36\x30\x3d\x72\x6f\x62\x6c\x65\x64\x6f\x6c\x6c\x61\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x31\x3d\x72\x6f\x6d\x61\x6e\x67\x6f\x72\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x32\x3d\x72\x75\x61\x6e\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x33\x3d\x73\x61\x6c\x6f\x72\x69\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x34\x3d\x73\x61\x6c\x76\x61\x74\x69\x65\x72\x72\x61\x64\x65\x73\x61\x6e\x74\x69\x61\x67\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x35\x3d\x73\x61\x6e\x6d\x61\x72\x74\x69\x6e\x64\x65\x74\x72\x65\x76\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x36\x3d\x61\x79\x74\x6f\x73\x61\x6e\x74\x61\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x37\x3d\x73\x61\x6e\x74\x61\x63\x72\x75\x7a\x64\x65\x6c\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x38\x3d\x73\x61\x6e\x74\x61\x63\x72\x75\x7a\x64\x65\x70\x61\x6e\x69\x61\x67\x75\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x36\x39\x3d\x73\x61\x6e\x74\x61\x6d\x61\x72\x74\x61\x64\x65\x6d\x61\x67\x61\x73\x63\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x30\x3d\x73\x61\x6e\x74\x69\x61\x67\x6f\x64\x65\x6c\x63\x61\x6d\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x31\x3d\x73\x61\x6e\x74\x69\x62\x61\x6e\x65\x7a\x65\x6c\x61\x6c\x74\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x32\x3d\x73\x61\x6e\x74\x69\x62\x61\x6e\x65\x7a\x65\x6c\x62\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x33\x3d\x73\x61\x75\x63\x65\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x34\x3d\x73\x65\x67\x75\x72\x61\x64\x65\x74\x6f\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x35\x3d\x73\x65\x72\x72\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x36\x3d\x73\x65\x72\x72\x65\x6a\x6f\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x37\x3d\x73\x69\x65\x72\x72\x61\x64\x65\x66\x75\x65\x6e\x74\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x38\x3d\x74\x61\x6c\x61\x76\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x37\x39\x3d\x74\x61\x6c\x61\x76\x65\x72\x75\x65\x6c\x61\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x30\x3d\x74\x61\x6c\x61\x79\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x31\x3d\x74\x65\x6a\x65\x64\x61\x64\x65\x74\x69\x65\x74\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x32\x3d\x74\x6f\x72\x69\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x33\x3d\x74\x6f\x72\x6e\x61\x76\x61\x63\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x34\x3d\x61\x79\x74\x6f\x65\x6c\x74\x6f\x72\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x35\x3d\x74\x6f\x72\x72\x65\x63\x69\x6c\x6c\x61\x64\x65\x6c\x6f\x73\x61\x6e\x67\x65\x6c\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x36\x3d\x74\x6f\x72\x72\x65\x63\x69\x6c\x6c\x61\x73\x64\x65\x6c\x61\x74\x69\x65\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x37\x3d\x74\x6f\x72\x72\x65\x64\x65\x64\x6f\x6e\x6d\x69\x67\x75\x65\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x38\x3d\x74\x6f\x72\x72\x65\x64\x65\x73\x61\x6e\x74\x61\x6d\x61\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x38\x39\x3d\x74\x6f\x72\x72\x65\x6a\x6f\x6e\x65\x6c\x72\x75\x62\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x30\x3d\x74\x6f\x72\x72\x65\x6a\x6f\x6e\x63\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x31\x3d\x74\x6f\x72\x72\x65\x6d\x65\x6e\x67\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x32\x3d\x74\x6f\x72\x72\x65\x6d\x6f\x63\x68\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x33\x3d\x74\x6f\x72\x72\x65\x6f\x72\x67\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x34\x3d\x74\x6f\x72\x72\x65\x71\x75\x65\x6d\x61\x64\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x35\x3d\x76\x61\x6c\x64\x61\x73\x74\x69\x6c\x6c\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x36\x3d\x76\x61\x6c\x64\x65\x63\x61\x6e\x61\x73\x64\x65\x74\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x37\x3d\x76\x61\x6c\x64\x65\x66\x75\x65\x6e\x74\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x38\x3d\x76\x61\x6c\x64\x65\x68\x75\x6e\x63\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x31\x39\x39\x3d\x76\x61\x6c\x64\x65\x69\x6e\x69\x67\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x30\x3d\x76\x61\x6c\x64\x65\x6c\x61\x63\x61\x73\x61\x64\x65\x74\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x31\x3d\x76\x61\x6c\x64\x65\x6d\x6f\x72\x61\x6c\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x32\x3d\x76\x61\x6c\x64\x65\x6f\x62\x69\x73\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x33\x3d\x76\x61\x6c\x64\x65\x73\x61\x6c\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x34\x3d\x76\x61\x6c\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x35\x3d\x76\x61\x6c\x65\x6e\x63\x69\x61\x64\x65\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x36\x3d\x76\x61\x6c\x76\x65\x72\x64\x65\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x37\x3d\x76\x61\x6c\x76\x65\x72\x64\x65\x64\x65\x6c\x66\x72\x65\x73\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x38\x3d\x76\x65\x67\x61\x76\x69\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x30\x39\x3d\x76\x69\x61\x6e\x64\x61\x72\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x30\x3d\x76\x69\x6c\x6c\x61\x64\x65\x6c\x63\x61\x6d\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x31\x3d\x76\x69\x6c\x6c\x61\x64\x65\x6c\x72\x65\x79\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x32\x3d\x76\x69\x6c\x6c\x61\x6d\x65\x73\x69\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x33\x3d\x76\x69\x6c\x6c\x61\x6d\x69\x65\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x34\x3d\x76\x69\x6c\x6c\x61\x6e\x75\x65\x76\x61\x64\x65\x6c\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x35\x3d\x76\x69\x6c\x6c\x61\x72\x64\x65\x6c\x70\x65\x64\x72\x6f\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x36\x3d\x76\x69\x6c\x6c\x61\x72\x64\x65\x70\x6c\x61\x73\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x37\x3d\x76\x69\x6c\x6c\x61\x73\x62\x75\x65\x6e\x61\x73\x64\x65\x67\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x38\x3d\x7a\x61\x72\x7a\x61\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x31\x39\x3d\x7a\x61\x72\x7a\x61\x64\x65\x6d\x6f\x6e\x74\x61\x6e\x63\x68\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x30\x3d\x7a\x61\x72\x7a\x61\x6c\x61\x6d\x61\x79\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x31\x3d\x7a\x6f\x72\x69\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x32\x3d\x72\x6f\x73\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x33\x3d\x76\x65\x67\x61\x76\x69\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x34\x3d\x61\x6c\x61\x67\x6f\x6e\x64\x65\x6c\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x35\x3d\x74\x69\x65\x74\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x36\x3d\x76\x61\x6c\x64\x65\x73\x61\x6c\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x37\x3d\x6e\x61\x76\x61\x74\x72\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x38\x3d\x72\x69\x76\x65\x72\x61\x64\x65\x66\x72\x65\x73\x6e\x65\x64\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x32\x39\x3d\x65\x6c\x6d\x73\x61\x6e\x67\x69\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x30\x3d\x74\x61\x6a\x6f\x73\x61\x6c\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x31\x3d\x76\x61\x6c\x6c\x65\x61\x6d\x62\x72\x6f\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x32\x3d\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x76\x61\x6c\x6c\x65\x64\x65\x6c\x61\x6c\x61\x67\x6f\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x33\x3d\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x76\x61\x6c\x6c\x65\x64\x65\x6c\x6a\x65\x72\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x34\x3d\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x76\x65\x67\x61\x73\x61\x6c\x74\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x35\x3d\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x36\x3d\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x7a\x6f\x6e\x61\x63\x65\x6e\x74\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x37\x3d\x76\x69\x6c\x6c\x75\x65\x72\x63\x61\x73\x2d\x69\x62\x6f\x72\x65\x73\x2d\x6a\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x38\x3d\x77\x77\x77\x2e\x61\x62\x61\x64\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x33\x39\x3d\x77\x77\x77\x2e\x61\x62\x65\x72\x74\x75\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x30\x3d\x77\x77\x77\x2e\x61\x63\x65\x62\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x31\x3d\x77\x77\x77\x2e\x61\x63\x65\x68\x75\x63\x68\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x32\x3d\x77\x77\x77\x2e\x61\x63\x65\x69\x74\x75\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x33\x3d\x77\x77\x77\x2e\x61\x68\x69\x67\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x34\x3d\x77\x77\x77\x2e\x61\x6c\x61\x67\x6f\x6e\x64\x65\x6c\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x35\x3d\x77\x77\x77\x2e\x61\x6c\x63\x6f\x6c\x6c\x61\x72\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x36\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x61\x6c\x62\x61\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x37\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x38\x3d\x77\x77\x77\x2e\x61\x6c\x63\x75\x65\x73\x63\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x34\x39\x3d\x77\x77\x77\x2e\x61\x6c\x64\x65\x61\x63\x65\x6e\x74\x65\x6e\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x30\x3d\x77\x77\x77\x2e\x61\x6c\x64\x65\x61\x64\x65\x6c\x63\x61\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x31\x3d\x77\x77\x77\x2e\x6c\x61\x61\x6c\x64\x65\x61\x64\x65\x6c\x6f\x62\x69\x73\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x32\x3d\x77\x77\x77\x2e\x61\x6c\x64\x65\x61\x6e\x75\x65\x76\x61\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x33\x3d\x77\x77\x77\x2e\x61\x6c\x64\x65\x61\x6e\x75\x65\x76\x61\x64\x65\x6c\x63\x61\x6d\x69\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x34\x3d\x77\x77\x77\x2e\x61\x6c\x64\x65\x68\x75\x65\x6c\x61\x64\x65\x6c\x6a\x65\x72\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x35\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x61\x6c\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x36\x3d\x77\x77\x77\x2e\x61\x6c\x69\x73\x65\x64\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x37\x3d\x77\x77\x77\x2e\x61\x6c\x6d\x61\x72\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x38\x3d\x77\x77\x77\x2e\x61\x6c\x6d\x6f\x68\x61\x72\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x35\x39\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x61\x72\x72\x6f\x79\x6f\x64\x65\x6c\x61\x6c\x75\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x30\x3d\x77\x77\x77\x2e\x61\x72\x72\x6f\x79\x6f\x6d\x6f\x6c\x69\x6e\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x31\x3d\x77\x77\x77\x2e\x61\x72\x72\x6f\x79\x6f\x6d\x6f\x6c\x69\x6e\x6f\x73\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x32\x3d\x77\x77\x77\x2e\x62\x61\x6e\x6f\x73\x64\x65\x6d\x6f\x6e\x74\x65\x6d\x61\x79\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x33\x3d\x77\x77\x77\x2e\x62\x61\x72\x72\x61\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x34\x3d\x77\x77\x77\x2e\x62\x65\x6c\x76\x69\x73\x64\x65\x6d\x6f\x6e\x72\x6f\x79\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x35\x3d\x77\x77\x77\x2e\x62\x65\x6e\x71\x75\x65\x72\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x36\x3d\x77\x77\x77\x2e\x62\x65\x72\x72\x6f\x63\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x37\x3d\x77\x77\x77\x2e\x62\x65\x72\x7a\x6f\x63\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x38\x3d\x77\x77\x77\x2e\x62\x6f\x68\x6f\x6e\x61\x6c\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x36\x39\x3d\x77\x77\x77\x2e\x62\x6f\x74\x69\x6a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x30\x3d\x77\x77\x77\x2e\x62\x72\x6f\x7a\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x31\x3d\x77\x77\x77\x2e\x63\x61\x62\x61\x6e\x61\x73\x64\x65\x6c\x63\x61\x73\x74\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x32\x3d\x77\x77\x77\x2e\x63\x61\x62\x65\x7a\x61\x62\x65\x6c\x6c\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x33\x3d\x77\x77\x77\x2e\x63\x61\x62\x65\x7a\x75\x65\x6c\x61\x64\x65\x6c\x76\x61\x6c\x6c\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x34\x3d\x77\x77\x77\x2e\x63\x61\x62\x72\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x35\x3d\x77\x77\x77\x2e\x63\x61\x63\x68\x6f\x72\x72\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x36\x3d\x77\x77\x77\x2e\x63\x61\x64\x61\x6c\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x37\x3d\x77\x77\x77\x2e\x63\x61\x6c\x7a\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x38\x3d\x77\x77\x77\x2e\x63\x61\x6d\x69\x6e\x6f\x6d\x6f\x72\x69\x73\x63\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x37\x39\x3d\x77\x77\x77\x2e\x63\x61\x6d\x70\x69\x6c\x6c\x6f\x64\x65\x64\x65\x6c\x65\x69\x74\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x30\x3d\x77\x77\x77\x2e\x63\x61\x6d\x70\x6f\x6c\x75\x67\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x31\x3d\x77\x77\x77\x2e\x63\x61\x6e\x61\x6d\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x32\x3d\x77\x77\x77\x2e\x63\x61\x6e\x61\x76\x65\x72\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x33\x3d\x77\x77\x77\x2e\x63\x61\x72\x62\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x34\x3d\x77\x77\x77\x2e\x63\x61\x72\x63\x61\x62\x6f\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x35\x3d\x77\x77\x77\x2e\x63\x61\x72\x72\x61\x73\x63\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x36\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x72\x64\x65\x63\x61\x63\x65\x72\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x37\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x72\x64\x65\x70\x61\x6c\x6f\x6d\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x38\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x72\x65\x73\x64\x65\x6c\x61\x73\x68\x75\x72\x64\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x38\x39\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x73\x64\x65\x64\x6f\x6e\x61\x6e\x74\x6f\x6e\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x30\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x73\x64\x65\x64\x6f\x6e\x67\x6f\x6d\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x31\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x73\x64\x65\x6c\x63\x61\x73\x74\x61\x6e\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x32\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x73\x64\x65\x6c\x6d\x6f\x6e\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x33\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x73\x64\x65\x6d\x69\x6c\x6c\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x34\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x73\x64\x65\x6d\x69\x72\x61\x76\x65\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x35\x3d\x77\x77\x77\x2e\x63\x61\x73\x61\x74\x65\x6a\x61\x64\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x36\x3d\x77\x77\x77\x2e\x63\x61\x73\x69\x6c\x6c\x61\x73\x64\x65\x63\x6f\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x37\x3d\x77\x77\x77\x2e\x63\x61\x73\x74\x61\x6e\x61\x72\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x38\x3d\x77\x77\x77\x2e\x63\x65\x63\x6c\x61\x76\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x32\x39\x39\x3d\x77\x77\x77\x2e\x63\x65\x64\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x30\x3d\x77\x77\x77\x2e\x63\x65\x72\x65\x7a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x31\x3d\x77\x77\x77\x2e\x63\x69\x6c\x6c\x65\x72\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x32\x3d\x77\x77\x77\x2e\x63\x6f\x6c\x6c\x61\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x33\x3d\x77\x77\x77\x2e\x63\x6f\x6e\x71\x75\x69\x73\x74\x61\x64\x65\x6c\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x34\x3d\x77\x77\x77\x2e\x63\x6f\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x35\x3d\x77\x77\x77\x2e\x63\x75\x61\x63\x6f\x73\x64\x65\x79\x75\x73\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x36\x3d\x77\x77\x77\x2e\x6c\x61\x63\x75\x6d\x62\x72\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x37\x3d\x77\x77\x77\x2e\x64\x65\x6c\x65\x69\x74\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x38\x3d\x77\x77\x77\x2e\x64\x65\x73\x63\x61\x72\x67\x61\x6d\x61\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x30\x39\x3d\x77\x77\x77\x2e\x65\x6c\x6a\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x30\x3d\x77\x77\x77\x2e\x65\x73\x63\x75\x72\x69\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x31\x3d\x77\x77\x77\x2e\x66\x72\x65\x73\x6e\x65\x64\x6f\x73\x6f\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x32\x3d\x77\x77\x77\x2e\x67\x61\x6c\x69\x73\x74\x65\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x33\x3d\x77\x77\x77\x2e\x67\x61\x72\x63\x69\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x34\x3d\x77\x77\x77\x2e\x6c\x61\x67\x61\x72\x67\x61\x6e\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x35\x3d\x77\x77\x77\x2e\x67\x61\x72\x67\x61\x6e\x74\x61\x6c\x61\x6f\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x36\x3d\x77\x77\x77\x2e\x67\x61\x72\x67\x61\x6e\x74\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x37\x3d\x77\x77\x77\x2e\x67\x61\x72\x67\x75\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x38\x3d\x77\x77\x77\x2e\x67\x61\x72\x72\x6f\x76\x69\x6c\x6c\x61\x73\x64\x65\x61\x6c\x63\x6f\x6e\x65\x74\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x31\x39\x3d\x77\x77\x77\x2e\x67\x61\x72\x76\x69\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x30\x3d\x77\x77\x77\x2e\x67\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x31\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x65\x6c\x67\x6f\x72\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x32\x3d\x77\x77\x77\x2e\x6c\x61\x67\x72\x61\x6e\x6a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x33\x3d\x77\x77\x77\x2e\x6c\x61\x67\x72\x61\x6e\x6a\x61\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x34\x3d\x77\x77\x77\x2e\x61\x79\x75\x6e\x74\x61\x6d\x69\x65\x6e\x74\x6f\x64\x65\x67\x75\x61\x64\x61\x6c\x75\x70\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x35\x3d\x77\x77\x77\x2e\x67\x75\x69\x6a\x6f\x64\x65\x63\x6f\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x36\x3d\x77\x77\x77\x2e\x67\x75\x69\x6a\x6f\x64\x65\x67\x61\x6c\x69\x73\x74\x65\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x37\x3d\x77\x77\x77\x2e\x67\x75\x69\x6a\x6f\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x38\x3d\x77\x77\x77\x2e\x67\x75\x69\x6a\x6f\x64\x65\x73\x61\x6e\x74\x61\x62\x61\x72\x62\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x32\x39\x3d\x77\x77\x77\x2e\x68\x65\x72\x67\x75\x69\x6a\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x30\x3d\x77\x77\x77\x2e\x68\x65\x72\x6e\x61\x6e\x70\x65\x72\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x31\x3d\x77\x77\x77\x2e\x68\x65\x72\x72\x65\x72\x61\x64\x65\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x32\x3d\x77\x77\x77\x2e\x68\x65\x72\x72\x65\x72\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x33\x3d\x77\x77\x77\x2e\x68\x65\x72\x76\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x34\x3d\x77\x77\x77\x2e\x68\x69\x67\x75\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x35\x3d\x77\x77\x77\x2e\x68\x69\x6e\x6f\x6a\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x36\x3d\x77\x77\x77\x2e\x68\x6f\x6c\x67\x75\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x37\x3d\x77\x77\x77\x2e\x68\x6f\x79\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x38\x3d\x77\x77\x77\x2e\x68\x75\x65\x6c\x61\x67\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x33\x39\x3d\x77\x77\x77\x2e\x69\x62\x61\x68\x65\x72\x6e\x61\x6e\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x30\x3d\x77\x77\x77\x2e\x6a\x61\x72\x61\x69\x63\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x31\x3d\x77\x77\x77\x2e\x6a\x61\x72\x61\x69\x7a\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x32\x3d\x77\x77\x77\x2e\x6a\x61\x72\x61\x6e\x64\x69\x6c\x6c\x61\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x33\x3d\x77\x77\x77\x2e\x6a\x61\x72\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x34\x3d\x77\x77\x77\x2e\x6a\x65\x72\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x35\x3d\x77\x77\x77\x2e\x6c\x61\x64\x72\x69\x6c\x6c\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x36\x3d\x77\x77\x77\x2e\x6c\x6f\x67\x72\x6f\x73\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x37\x3d\x77\x77\x77\x2e\x6c\x6f\x73\x61\x72\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x38\x3d\x77\x77\x77\x2e\x6d\x61\x64\x72\x69\x67\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x34\x39\x3d\x77\x77\x77\x2e\x6d\x61\x64\x72\x69\x67\x61\x6c\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x30\x3d\x77\x77\x77\x2e\x6d\x61\x64\x72\x6f\x6e\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x31\x3d\x77\x77\x77\x2e\x6d\x61\x6a\x61\x64\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x32\x3d\x77\x77\x77\x2e\x6d\x61\x6c\x70\x61\x72\x74\x69\x64\x61\x64\x65\x63\x61\x63\x65\x72\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x33\x3d\x77\x77\x77\x2e\x6d\x61\x6c\x70\x61\x72\x74\x69\x64\x61\x64\x65\x70\x6c\x61\x73\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x34\x3d\x77\x77\x77\x2e\x6d\x61\x72\x63\x68\x61\x67\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x35\x3d\x77\x77\x77\x2e\x6d\x61\x74\x61\x64\x65\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x36\x3d\x77\x77\x77\x2e\x6d\x65\x6d\x62\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x37\x3d\x77\x77\x77\x2e\x6d\x65\x73\x61\x73\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x38\x3d\x77\x77\x77\x2e\x6d\x69\x61\x6a\x61\x64\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x35\x39\x3d\x77\x77\x77\x2e\x6d\x69\x6c\x6c\x61\x6e\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x30\x3d\x77\x77\x77\x2e\x6d\x69\x72\x61\x62\x65\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x31\x3d\x77\x77\x77\x2e\x6d\x6f\x68\x65\x64\x61\x73\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x32\x3d\x77\x77\x77\x2e\x6d\x6f\x6e\x72\x6f\x79\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x33\x3d\x77\x77\x77\x2e\x6d\x6f\x6e\x74\x61\x6e\x63\x68\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x34\x3d\x77\x77\x77\x2e\x6d\x6f\x6e\x74\x65\x68\x65\x72\x6d\x6f\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x35\x3d\x77\x77\x77\x2e\x6d\x6f\x72\x61\x6c\x65\x6a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x36\x3d\x77\x77\x77\x2e\x6d\x6f\x72\x63\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x37\x3d\x77\x77\x77\x2e\x6e\x61\x76\x61\x63\x6f\x6e\x63\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x38\x3d\x77\x77\x77\x2e\x6e\x61\x76\x61\x6c\x76\x69\x6c\x6c\x61\x72\x64\x65\x69\x62\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x36\x39\x3d\x77\x77\x77\x2e\x6e\x61\x76\x61\x6c\x6d\x6f\x72\x61\x6c\x64\x65\x6c\x61\x6d\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x30\x3d\x77\x77\x77\x2e\x6e\x61\x76\x61\x73\x64\x65\x6c\x6d\x61\x64\x72\x6f\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x31\x3d\x77\x77\x77\x2e\x6e\x61\x76\x61\x74\x72\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x32\x3d\x77\x77\x77\x2e\x6e\x61\x76\x65\x7a\x75\x65\x6c\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x33\x3d\x77\x77\x77\x2e\x6e\x75\x6e\x6f\x6d\x6f\x72\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x34\x3d\x77\x77\x77\x2e\x6f\x6c\x69\x76\x61\x64\x65\x70\x6c\x61\x73\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x35\x3d\x77\x77\x77\x2e\x70\x61\x6c\x6f\x6d\x65\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x36\x3d\x77\x77\x77\x2e\x70\x61\x73\x61\x72\x6f\x6e\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x37\x3d\x77\x77\x77\x2e\x70\x65\x64\x72\x6f\x73\x6f\x64\x65\x61\x63\x69\x6d\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x38\x3d\x77\x77\x77\x2e\x70\x65\x72\x61\x6c\x65\x64\x61\x64\x65\x6c\x61\x6d\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x37\x39\x3d\x77\x77\x77\x2e\x70\x65\x72\x61\x6c\x65\x64\x61\x64\x65\x73\x61\x6e\x72\x6f\x6d\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x30\x3d\x77\x77\x77\x2e\x70\x65\x72\x61\x6c\x65\x73\x64\x65\x6c\x70\x75\x65\x72\x74\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x31\x3d\x77\x77\x77\x2e\x70\x65\x73\x63\x75\x65\x7a\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x32\x3d\x77\x77\x77\x2e\x6c\x61\x70\x65\x73\x67\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x33\x3d\x77\x77\x77\x2e\x70\x69\x65\x64\x72\x61\x73\x61\x6c\x62\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x34\x3d\x77\x77\x77\x2e\x70\x69\x6e\x6f\x66\x72\x61\x6e\x71\x75\x65\x61\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x35\x3d\x77\x77\x77\x2e\x70\x69\x6f\x72\x6e\x61\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x36\x3d\x77\x77\x77\x2e\x70\x6c\x61\x73\x65\x6e\x7a\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x37\x3d\x77\x77\x77\x2e\x70\x6f\x72\x74\x61\x6a\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x38\x3d\x77\x77\x77\x2e\x70\x6f\x72\x74\x65\x7a\x75\x65\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x38\x39\x3d\x77\x77\x77\x2e\x70\x6f\x7a\x75\x65\x6c\x6f\x64\x65\x7a\x61\x72\x7a\x6f\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x30\x3d\x77\x77\x77\x2e\x70\x75\x65\x62\x6c\x6f\x6e\x75\x65\x76\x6f\x64\x65\x6d\x69\x72\x61\x6d\x6f\x6e\x74\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x31\x3d\x77\x77\x77\x2e\x70\x75\x65\x72\x74\x6f\x64\x65\x73\x61\x6e\x74\x61\x63\x72\x75\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x32\x3d\x77\x77\x77\x2e\x72\x65\x62\x6f\x6c\x6c\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x33\x3d\x77\x77\x77\x2e\x72\x69\x6f\x6c\x6f\x62\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x34\x3d\x77\x77\x77\x2e\x72\x6f\x62\x6c\x65\x64\x69\x6c\x6c\x6f\x64\x65\x67\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x35\x3d\x77\x77\x77\x2e\x72\x6f\x62\x6c\x65\x64\x69\x6c\x6c\x6f\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x36\x3d\x77\x77\x77\x2e\x72\x6f\x62\x6c\x65\x64\x69\x6c\x6c\x6f\x64\x65\x74\x72\x75\x6a\x69\x6c\x6c\x6f\x2c\x44\x4e\x53\x2e\x33\x39\x37\x3d\x77\x77\x77\x2e\x72\x6f\x62\x6c\x65\x64\x6f\x6c\x6c\x61\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x38\x3d\x77\x77\x77\x2e\x72\x6f\x6d\x61\x6e\x67\x6f\x72\x64\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x33\x39\x39\x3d\x77\x77\x77\x2e\x72\x75\x61\x6e\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x30\x3d\x77\x77\x77\x2e\x73\x61\x6c\x6f\x72\x69\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x31\x3d\x77\x77\x77\x2e\x73\x61\x6c\x76\x61\x74\x69\x65\x72\x72\x61\x64\x65\x73\x61\x6e\x74\x69\x61\x67\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x32\x3d\x77\x77\x77\x2e\x73\x61\x6e\x6d\x61\x72\x74\x69\x6e\x64\x65\x74\x72\x65\x76\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x33\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x73\x61\x6e\x74\x61\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x34\x3d\x77\x77\x77\x2e\x73\x61\x6e\x74\x61\x63\x72\x75\x7a\x64\x65\x6c\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x35\x3d\x77\x77\x77\x2e\x73\x61\x6e\x74\x61\x63\x72\x75\x7a\x64\x65\x70\x61\x6e\x69\x61\x67\x75\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x36\x3d\x77\x77\x77\x2e\x73\x61\x6e\x74\x61\x6d\x61\x72\x74\x61\x64\x65\x6d\x61\x67\x61\x73\x63\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x37\x3d\x77\x77\x77\x2e\x73\x61\x6e\x74\x69\x61\x67\x6f\x64\x65\x6c\x63\x61\x6d\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x38\x3d\x77\x77\x77\x2e\x73\x61\x6e\x74\x69\x62\x61\x6e\x65\x7a\x65\x6c\x61\x6c\x74\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x30\x39\x3d\x77\x77\x77\x2e\x73\x61\x6e\x74\x69\x62\x61\x6e\x65\x7a\x65\x6c\x62\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x30\x3d\x77\x77\x77\x2e\x73\x61\x75\x63\x65\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x31\x3d\x77\x77\x77\x2e\x73\x65\x67\x75\x72\x61\x64\x65\x74\x6f\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x32\x3d\x77\x77\x77\x2e\x73\x65\x72\x72\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x33\x3d\x77\x77\x77\x2e\x73\x65\x72\x72\x65\x6a\x6f\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x34\x3d\x77\x77\x77\x2e\x73\x69\x65\x72\x72\x61\x64\x65\x66\x75\x65\x6e\x74\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x35\x3d\x77\x77\x77\x2e\x74\x61\x6c\x61\x76\x61\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x36\x3d\x77\x77\x77\x2e\x74\x61\x6c\x61\x76\x65\x72\x75\x65\x6c\x61\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x37\x3d\x77\x77\x77\x2e\x74\x61\x6c\x61\x79\x75\x65\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x38\x3d\x77\x77\x77\x2e\x74\x65\x6a\x65\x64\x61\x64\x65\x74\x69\x65\x74\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x31\x39\x3d\x77\x77\x77\x2e\x74\x6f\x72\x69\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x30\x3d\x77\x77\x77\x2e\x74\x6f\x72\x6e\x61\x76\x61\x63\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x31\x3d\x77\x77\x77\x2e\x61\x79\x74\x6f\x65\x6c\x74\x6f\x72\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x32\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x63\x69\x6c\x6c\x61\x64\x65\x6c\x6f\x73\x61\x6e\x67\x65\x6c\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x33\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x63\x69\x6c\x6c\x61\x73\x64\x65\x6c\x61\x74\x69\x65\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x34\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x64\x65\x64\x6f\x6e\x6d\x69\x67\x75\x65\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x35\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x64\x65\x73\x61\x6e\x74\x61\x6d\x61\x72\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x36\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x6a\x6f\x6e\x65\x6c\x72\x75\x62\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x37\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x6a\x6f\x6e\x63\x69\x6c\x6c\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x38\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x6d\x65\x6e\x67\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x32\x39\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x6d\x6f\x63\x68\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x30\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x6f\x72\x67\x61\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x31\x3d\x77\x77\x77\x2e\x74\x6f\x72\x72\x65\x71\x75\x65\x6d\x61\x64\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x32\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x61\x73\x74\x69\x6c\x6c\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x33\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x63\x61\x6e\x61\x73\x64\x65\x74\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x34\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x66\x75\x65\x6e\x74\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x35\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x68\x75\x6e\x63\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x36\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x69\x6e\x69\x67\x6f\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x37\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x6c\x61\x63\x61\x73\x61\x64\x65\x74\x61\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x38\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x6d\x6f\x72\x61\x6c\x65\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x33\x39\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x6f\x62\x69\x73\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x30\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x73\x61\x6c\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x31\x3d\x77\x77\x77\x2e\x76\x61\x6c\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x32\x3d\x77\x77\x77\x2e\x76\x61\x6c\x65\x6e\x63\x69\x61\x64\x65\x61\x6c\x63\x61\x6e\x74\x61\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x33\x3d\x77\x77\x77\x2e\x76\x61\x6c\x76\x65\x72\x64\x65\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x34\x3d\x77\x77\x77\x2e\x76\x61\x6c\x76\x65\x72\x64\x65\x64\x65\x6c\x66\x72\x65\x73\x6e\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x35\x3d\x77\x77\x77\x2e\x76\x65\x67\x61\x76\x69\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x36\x3d\x77\x77\x77\x2e\x76\x69\x61\x6e\x64\x61\x72\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x37\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x64\x65\x6c\x63\x61\x6d\x70\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x38\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x64\x65\x6c\x72\x65\x79\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x34\x39\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x6d\x65\x73\x69\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x30\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x6d\x69\x65\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x31\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x6e\x75\x65\x76\x61\x64\x65\x6c\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x32\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x72\x64\x65\x6c\x70\x65\x64\x72\x6f\x73\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x33\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x72\x64\x65\x70\x6c\x61\x73\x65\x6e\x63\x69\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x34\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x61\x73\x62\x75\x65\x6e\x61\x73\x64\x65\x67\x61\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x35\x3d\x77\x77\x77\x2e\x7a\x61\x72\x7a\x61\x64\x65\x67\x72\x61\x6e\x61\x64\x69\x6c\x6c\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x36\x3d\x77\x77\x77\x2e\x7a\x61\x72\x7a\x61\x64\x65\x6d\x6f\x6e\x74\x61\x6e\x63\x68\x65\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x37\x3d\x77\x77\x77\x2e\x7a\x61\x72\x7a\x61\x6c\x61\x6d\x61\x79\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x38\x3d\x77\x77\x77\x2e\x7a\x6f\x72\x69\x74\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x35\x39\x3d\x77\x77\x77\x2e\x72\x6f\x73\x61\x6c\x65\x6a\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x30\x3d\x77\x77\x77\x2e\x76\x65\x67\x61\x76\x69\x61\x6e\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x31\x3d\x77\x77\x77\x2e\x61\x6c\x61\x67\x6f\x6e\x64\x65\x6c\x72\x69\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x32\x3d\x77\x77\x77\x2e\x74\x69\x65\x74\x61\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x33\x3d\x77\x77\x77\x2e\x76\x61\x6c\x64\x65\x73\x61\x6c\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x34\x3d\x77\x77\x77\x2e\x6e\x61\x76\x61\x74\x72\x61\x73\x69\x65\x72\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x35\x3d\x77\x77\x77\x2e\x72\x69\x76\x65\x72\x61\x64\x65\x66\x72\x65\x73\x6e\x65\x64\x6f\x73\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x36\x3d\x77\x77\x77\x2e\x65\x6c\x6d\x73\x61\x6e\x67\x69\x6c\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x37\x3d\x77\x77\x77\x2e\x74\x61\x6a\x6f\x73\x61\x6c\x6f\x72\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x38\x3d\x77\x77\x77\x2e\x76\x61\x6c\x6c\x65\x61\x6d\x62\x72\x6f\x7a\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x36\x39\x3d\x77\x77\x77\x2e\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x76\x61\x6c\x6c\x65\x64\x65\x6c\x61\x6c\x61\x67\x6f\x6e\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x37\x30\x3d\x77\x77\x77\x2e\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x76\x61\x6c\x6c\x65\x64\x65\x6c\x6a\x65\x72\x74\x65\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x37\x31\x3d\x77\x77\x77\x2e\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x76\x65\x67\x61\x73\x61\x6c\x74\x61\x73\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x37\x32\x3d\x77\x77\x77\x2e\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x64\x65\x6c\x61\x76\x65\x72\x61\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x37\x33\x3d\x77\x77\x77\x2e\x6d\x61\x6e\x63\x6f\x6d\x75\x6e\x69\x64\x61\x64\x7a\x6f\x6e\x61\x63\x65\x6e\x74\x72\x6f\x2e\x65\x73\x2c\x44\x4e\x53\x2e\x34\x37\x34\x3d\x77\x77\x77\x2e\x76\x69\x6c\x6c\x75\x65\x72\x63\x61\x73\x2d\x69\x62\x6f\x72\x65\x73\x2d\x6a\x61\x72\x61\x2e\x65\x73",
		 12589},
	 }
};

void doit(void)
{
	unsigned i;

	for (i = 0; i < sizeof(tests) / sizeof(tests[0]); i++) {
		decode(tests[i].name, &tests[i].raw, tests[i].str,
		       tests[i].compat_str);
	}
}
