/*
 *  Copyright (c) 2013 The WebRTC project authors. All Rights Reserved.
 *
 *  Use of this source code is governed by a BSD-style license
 *  that can be found in the LICENSE file in the root of the source
 *  tree. An additional intellectual property rights grant can be found
 *  in the file PATENTS.  All contributing project authors may
 *  be found in the AUTHORS file in the root of the source tree.
 *
 */

/*
 * Description:
 * Twiddle table for Forward FFT in float format.
 * It contains complex pairs [-cos (W * i), -sin (W * i)] where W = -2*PI/N
 * and 0<= i<= N/8.  N is the max size of the FFT. Here N = 2^12
 * Values for N/8 < i < N are generated in the FFTInit function using the
 * symmetries of cos and sine.
 *
 * We use the negative to be consistent with the S32 version.
**/

#include "dl/api/omxtypes.h"

#define TWIDDLE_TABLE_ENTRIES ((1 << TWIDDLE_TABLE_ORDER) / 8)

const OMX_F32 armSP_FFT_F32TwiddleTable[2 * TWIDDLE_TABLE_ENTRIES + 2] = {
#if TWIDDLE_TABLE_ORDER == 15
  -1.000000000000000e+00,   0.000000000000000e+00,
  -9.999999816164293e-01,   1.917475973107033e-04,
  -9.999999264657179e-01,   3.834951875713956e-04,
  -9.999998345478677e-01,   5.752427637320661e-04,
  -9.999997058628822e-01,   7.669903187427045e-04,
  -9.999995404107661e-01,   9.587378455533014e-04,
  -9.999993381915255e-01,   1.150485337113848e-03,
  -9.999990992051678e-01,   1.342232786374338e-03,
  -9.999988234517019e-01,   1.533980186284766e-03,
  -9.999985109311378e-01,   1.725727529795126e-03,
  -9.999981616434870e-01,   1.917474809855419e-03,
  -9.999977755887623e-01,   2.109222019415644e-03,
  -9.999973527669782e-01,   2.300969151425805e-03,
  -9.999968931781499e-01,   2.492716198835908e-03,
  -9.999963968222944e-01,   2.684463154595962e-03,
  -9.999958636994299e-01,   2.876210011655979e-03,
  -9.999952938095762e-01,   3.067956762965976e-03,
  -9.999946871527541e-01,   3.259703401475973e-03,
  -9.999940437289858e-01,   3.451449920135994e-03,
  -9.999933635382952e-01,   3.643196311896068e-03,
  -9.999926465807072e-01,   3.834942569706228e-03,
  -9.999918928562480e-01,   4.026688686516512e-03,
  -9.999911023649456e-01,   4.218434655276963e-03,
  -9.999902751068289e-01,   4.410180468937631e-03,
  -9.999894110819284e-01,   4.601926120448570e-03,
  -9.999885102902757e-01,   4.793671602759841e-03,
  -9.999875727319041e-01,   4.985416908821511e-03,
  -9.999865984068480e-01,   5.177162031583651e-03,
  -9.999855873151432e-01,   5.368906963996342e-03,
  -9.999845394568270e-01,   5.560651699009674e-03,
  -9.999834548319377e-01,   5.752396229573736e-03,
  -9.999823334405153e-01,   5.944140548638633e-03,
  -9.999811752826011e-01,   6.135884649154475e-03,
  -9.999799803582377e-01,   6.327628524071378e-03,
  -9.999787486674688e-01,   6.519372166339468e-03,
  -9.999774802103399e-01,   6.711115568908879e-03,
  -9.999761749868976e-01,   6.902858724729756e-03,
  -9.999748329971898e-01,   7.094601626752250e-03,
  -9.999734542412659e-01,   7.286344267926521e-03,
  -9.999720387191767e-01,   7.478086641202744e-03,
  -9.999705864309741e-01,   7.669828739531097e-03,
  -9.999690973767116e-01,   7.861570555861772e-03,
  -9.999675715564438e-01,   8.053312083144972e-03,
  -9.999660089702269e-01,   8.245053314330906e-03,
  -9.999644096181183e-01,   8.436794242369799e-03,
  -9.999627735001769e-01,   8.628534860211886e-03,
  -9.999611006164628e-01,   8.820275160807411e-03,
  -9.999593909670375e-01,   9.012015137106633e-03,
  -9.999576445519639e-01,   9.203754782059820e-03,
  -9.999558613713061e-01,   9.395494088617251e-03,
  -9.999540414251298e-01,   9.587233049729225e-03,
  -9.999521847135018e-01,   9.778971658346043e-03,
  -9.999502912364905e-01,   9.970709907418030e-03,
  -9.999483609941654e-01,   1.016244778989551e-02,
  -9.999463939865975e-01,   1.035418529872884e-02,
  -9.999443902138591e-01,   1.054592242686838e-02,
  -9.999423496760239e-01,   1.073765916726449e-02,
  -9.999402723731670e-01,   1.092939551286757e-02,
  -9.999381583053646e-01,   1.112113145662802e-02,
  -9.999360074726946e-01,   1.131286699149626e-02,
  -9.999338198752360e-01,   1.150460211042271e-02,
  -9.999315955130692e-01,   1.169633680635784e-02,
  -9.999293343862761e-01,   1.188807107225209e-02,
  -9.999270364949396e-01,   1.207980490105596e-02,
  -9.999247018391445e-01,   1.227153828571993e-02,
  -9.999223304189765e-01,   1.246327121919451e-02,
  -9.999199222345228e-01,   1.265500369443024e-02,
  -9.999174772858718e-01,   1.284673570437766e-02,
  -9.999149955731135e-01,   1.303846724198733e-02,
  -9.999124770963392e-01,   1.323019830020983e-02,
  -9.999099218556415e-01,   1.342192887199577e-02,
  -9.999073298511143e-01,   1.361365895029574e-02,
  -9.999047010828529e-01,   1.380538852806039e-02,
  -9.999020355509539e-01,   1.399711759824037e-02,
  -9.998993332555154e-01,   1.418884615378634e-02,
  -9.998965941966367e-01,   1.438057418764901e-02,
  -9.998938183744185e-01,   1.457230169277906e-02,
  -9.998910057889630e-01,   1.476402866212725e-02,
  -9.998881564403733e-01,   1.495575508864430e-02,
  -9.998852703287545e-01,   1.514748096528099e-02,
  -9.998823474542126e-01,   1.533920628498810e-02,
  -9.998793878168549e-01,   1.553093104071645e-02,
  -9.998763914167904e-01,   1.572265522541686e-02,
  -9.998733582541293e-01,   1.591437883204018e-02,
  -9.998702883289830e-01,   1.610610185353729e-02,
  -9.998671816414644e-01,   1.629782428285906e-02,
  -9.998640381916877e-01,   1.648954611295644e-02,
  -9.998608579797685e-01,   1.668126733678033e-02,
  -9.998576410058239e-01,   1.687298794728171e-02,
  -9.998543872699719e-01,   1.706470793741156e-02,
  -9.998510967723322e-01,   1.725642730012088e-02,
  -9.998477695130259e-01,   1.744814602836069e-02,
  -9.998444054921752e-01,   1.763986411508205e-02,
  -9.998410047099040e-01,   1.783158155323604e-02,
  -9.998375671663371e-01,   1.802329833577375e-02,
  -9.998340928616010e-01,   1.821501445564629e-02,
  -9.998305817958234e-01,   1.840672990580482e-02,
  -9.998270339691334e-01,   1.859844467920051e-02,
  -9.998234493816616e-01,   1.879015876878455e-02,
  -9.998198280335394e-01,   1.898187216750818e-02,
  -9.998161699249004e-01,   1.917358486832262e-02,
  -9.998124750558788e-01,   1.936529686417916e-02,
  -9.998087434266105e-01,   1.955700814802908e-02,
  -9.998049750372329e-01,   1.974871871282373e-02,
  -9.998011698878843e-01,   1.994042855151444e-02,
  -9.997973279787047e-01,   2.013213765705259e-02,
  -9.997934493098353e-01,   2.032384602238959e-02,
  -9.997895338814188e-01,   2.051555364047688e-02,
  -9.997855816935992e-01,   2.070726050426589e-02,
  -9.997815927465217e-01,   2.089896660670814e-02,
  -9.997775670403329e-01,   2.109067194075512e-02,
  -9.997735045751810e-01,   2.128237649935839e-02,
  -9.997694053512153e-01,   2.147408027546951e-02,
  -9.997652693685865e-01,   2.166578326204008e-02,
  -9.997610966274466e-01,   2.185748545202174e-02,
  -9.997568871279491e-01,   2.204918683836614e-02,
  -9.997526408702488e-01,   2.224088741402496e-02,
  -9.997483578545018e-01,   2.243258717194993e-02,
  -9.997440380808654e-01,   2.262428610509280e-02,
  -9.997396815494987e-01,   2.281598420640535e-02,
  -9.997352882605617e-01,   2.300768146883937e-02,
  -9.997308582142160e-01,   2.319937788534672e-02,
  -9.997263914106245e-01,   2.339107344887926e-02,
  -9.997218878499513e-01,   2.358276815238889e-02,
  -9.997173475323622e-01,   2.377446198882755e-02,
  -9.997127704580239e-01,   2.396615495114721e-02,
  -9.997081566271049e-01,   2.415784703229986e-02,
  -9.997035060397746e-01,   2.434953822523753e-02,
  -9.996988186962042e-01,   2.454122852291229e-02,
  -9.996940945965660e-01,   2.473291791827622e-02,
  -9.996893337410336e-01,   2.492460640428147e-02,
  -9.996845361297821e-01,   2.511629397388019e-02,
  -9.996797017629879e-01,   2.530798062002457e-02,
  -9.996748306408287e-01,   2.549966633566685e-02,
  -9.996699227634838e-01,   2.569135111375929e-02,
  -9.996649781311333e-01,   2.588303494725420e-02,
  -9.996599967439592e-01,   2.607471782910390e-02,
  -9.996549786021447e-01,   2.626639975226076e-02,
  -9.996499237058742e-01,   2.645808070967719e-02,
  -9.996448320553336e-01,   2.664976069430562e-02,
  -9.996397036507102e-01,   2.684143969909853e-02,
  -9.996345384921923e-01,   2.703311771700843e-02,
  -9.996293365799701e-01,   2.722479474098788e-02,
  -9.996240979142346e-01,   2.741647076398944e-02,
  -9.996188224951786e-01,   2.760814577896574e-02,
  -9.996135103229960e-01,   2.779981977886944e-02,
  -9.996081613978821e-01,   2.799149275665324e-02,
  -9.996027757200335e-01,   2.818316470526987e-02,
  -9.995973532896484e-01,   2.837483561767210e-02,
  -9.995918941069260e-01,   2.856650548681273e-02,
  -9.995863981720671e-01,   2.875817430564461e-02,
  -9.995808654852737e-01,   2.894984206712064e-02,
  -9.995752960467492e-01,   2.914150876419372e-02,
  -9.995696898566986e-01,   2.933317438981684e-02,
  -9.995640469153277e-01,   2.952483893694298e-02,
  -9.995583672228443e-01,   2.971650239852519e-02,
  -9.995526507794570e-01,   2.990816476751655e-02,
  -9.995468975853760e-01,   3.009982603687020e-02,
  -9.995411076408129e-01,   3.029148619953928e-02,
  -9.995352809459805e-01,   3.048314524847701e-02,
  -9.995294175010931e-01,   3.067480317663663e-02,
  -9.995235173063663e-01,   3.086645997697141e-02,
  -9.995175803620170e-01,   3.105811564243470e-02,
  -9.995116066682634e-01,   3.124977016597986e-02,
  -9.995055962253253e-01,   3.144142354056030e-02,
  -9.994995490334236e-01,   3.163307575912948e-02,
  -9.994934650927806e-01,   3.182472681464089e-02,
  -9.994873444036201e-01,   3.201637670004806e-02,
  -9.994811869661670e-01,   3.220802540830459e-02,
  -9.994749927806478e-01,   3.239967293236409e-02,
  -9.994687618472900e-01,   3.259131926518023e-02,
  -9.994624941663232e-01,   3.278296439970672e-02,
  -9.994561897379773e-01,   3.297460832889734e-02,
  -9.994498485624845e-01,   3.316625104570586e-02,
  -9.994434706400778e-01,   3.335789254308614e-02,
  -9.994370559709915e-01,   3.354953281399207e-02,
  -9.994306045554617e-01,   3.374117185137758e-02,
  -9.994241163937256e-01,   3.393280964819666e-02,
  -9.994175914860217e-01,   3.412444619740333e-02,
  -9.994110298325898e-01,   3.431608149195165e-02,
  -9.994044314336713e-01,   3.450771552479575e-02,
  -9.993977962895086e-01,   3.469934828888980e-02,
  -9.993911244003460e-01,   3.489097977718800e-02,
  -9.993844157664286e-01,   3.508260998264462e-02,
  -9.993776703880028e-01,   3.527423889821395e-02,
  -9.993708882653172e-01,   3.546586651685035e-02,
  -9.993640693986205e-01,   3.565749283150822e-02,
  -9.993572137881640e-01,   3.584911783514202e-02,
  -9.993503214341994e-01,   3.604074152070623e-02,
  -9.993433923369802e-01,   3.623236388115540e-02,
  -9.993364264967612e-01,   3.642398490944411e-02,
  -9.993294239137984e-01,   3.661560459852703e-02,
  -9.993223845883495e-01,   3.680722294135883e-02,
  -9.993153085206731e-01,   3.699883993089426e-02,
  -9.993081957110295e-01,   3.719045556008812e-02,
  -9.993010461596801e-01,   3.738206982189523e-02,
  -9.992938598668878e-01,   3.757368270927049e-02,
  -9.992866368329167e-01,   3.776529421516886e-02,
  -9.992793770580327e-01,   3.795690433254531e-02,
  -9.992720805425026e-01,   3.814851305435489e-02,
  -9.992647472865944e-01,   3.834012037355269e-02,
  -9.992573772905781e-01,   3.853172628309387e-02,
  -9.992499705547244e-01,   3.872333077593362e-02,
  -9.992425270793058e-01,   3.891493384502719e-02,
  -9.992350468645959e-01,   3.910653548332989e-02,
  -9.992275299108696e-01,   3.929813568379706e-02,
  -9.992199762184035e-01,   3.948973443938412e-02,
  -9.992123857874753e-01,   3.968133174304653e-02,
  -9.992047586183639e-01,   3.987292758773981e-02,
  -9.991970947113499e-01,   4.006452196641952e-02,
  -9.991893940667149e-01,   4.025611487204128e-02,
  -9.991816566847423e-01,   4.044770629756078e-02,
  -9.991738825657164e-01,   4.063929623593374e-02,
  -9.991660717099230e-01,   4.083088468011595e-02,
  -9.991582241176494e-01,   4.102247162306324e-02,
  -9.991503397891841e-01,   4.121405705773152e-02,
  -9.991424187248169e-01,   4.140564097707674e-02,
  -9.991344609248392e-01,   4.159722337405489e-02,
  -9.991264663895434e-01,   4.178880424162206e-02,
  -9.991184351192235e-01,   4.198038357273436e-02,
  -9.991103671141749e-01,   4.217196136034795e-02,
  -9.991022623746941e-01,   4.236353759741907e-02,
  -9.990941209010791e-01,   4.255511227690402e-02,
  -9.990859426936293e-01,   4.274668539175913e-02,
  -9.990777277526454e-01,   4.293825693494082e-02,
  -9.990694760784293e-01,   4.312982689940555e-02,
  -9.990611876712846e-01,   4.332139527810983e-02,
  -9.990528625315159e-01,   4.351296206401024e-02,
  -9.990445006594293e-01,   4.370452725006342e-02,
  -9.990361020553323e-01,   4.389609082922607e-02,
  -9.990276667195337e-01,   4.408765279445494e-02,
  -9.990191946523435e-01,   4.427921313870685e-02,
  -9.990106858540734e-01,   4.447077185493867e-02,
  -9.990021403250360e-01,   4.466232893610732e-02,
  -9.989935580655457e-01,   4.485388437516982e-02,
  -9.989849390759180e-01,   4.504543816508320e-02,
  -9.989762833564698e-01,   4.523699029880459e-02,
  -9.989675909075193e-01,   4.542854076929115e-02,
  -9.989588617293861e-01,   4.562008956950014e-02,
  -9.989500958223912e-01,   4.581163669238884e-02,
  -9.989412931868569e-01,   4.600318213091462e-02,
  -9.989324538231067e-01,   4.619472587803491e-02,
  -9.989235777314658e-01,   4.638626792670716e-02,
  -9.989146649122604e-01,   4.657780826988894e-02,
  -9.989057153658183e-01,   4.676934690053786e-02,
  -9.988967290924684e-01,   4.696088381161159e-02,
  -9.988877060925413e-01,   4.715241899606787e-02,
  -9.988786463663687e-01,   4.734395244686448e-02,
  -9.988695499142836e-01,   4.753548415695930e-02,
  -9.988604167366205e-01,   4.772701411931025e-02,
  -9.988512468337152e-01,   4.791854232687533e-02,
  -9.988420402059048e-01,   4.811006877261259e-02,
  -9.988327968535280e-01,   4.830159344948014e-02,
  -9.988235167769245e-01,   4.849311635043618e-02,
  -9.988141999764354e-01,   4.868463746843894e-02,
  -9.988048464524034e-01,   4.887615679644676e-02,
  -9.987954562051724e-01,   4.906767432741801e-02,
  -9.987860292350876e-01,   4.925919005431114e-02,
  -9.987765655424956e-01,   4.945070397008466e-02,
  -9.987670651277444e-01,   4.964221606769716e-02,
  -9.987575279911833e-01,   4.983372634010728e-02,
  -9.987479541331629e-01,   5.002523478027373e-02,
  -9.987383435540352e-01,   5.021674138115531e-02,
  -9.987286962541537e-01,   5.040824613571086e-02,
  -9.987190122338729e-01,   5.059974903689928e-02,
  -9.987092914935490e-01,   5.079125007767958e-02,
  -9.986995340335393e-01,   5.098274925101080e-02,
  -9.986897398542026e-01,   5.117424654985208e-02,
  -9.986799089558991e-01,   5.136574196716259e-02,
  -9.986700413389901e-01,   5.155723549590161e-02,
  -9.986601370038385e-01,   5.174872712902846e-02,
  -9.986501959508083e-01,   5.194021685950254e-02,
  -9.986402181802653e-01,   5.213170468028332e-02,
  -9.986302036925760e-01,   5.232319058433035e-02,
  -9.986201524881089e-01,   5.251467456460322e-02,
  -9.986100645672333e-01,   5.270615661406163e-02,
  -9.985999399303204e-01,   5.289763672566532e-02,
  -9.985897785777422e-01,   5.308911489237413e-02,
  -9.985795805098725e-01,   5.328059110714795e-02,
  -9.985693457270861e-01,   5.347206536294673e-02,
  -9.985590742297593e-01,   5.366353765273052e-02,
  -9.985487660182699e-01,   5.385500796945944e-02,
  -9.985384210929967e-01,   5.404647630609366e-02,
  -9.985280394543202e-01,   5.423794265559345e-02,
  -9.985176211026222e-01,   5.442940701091913e-02,
  -9.985071660382855e-01,   5.462086936503111e-02,
  -9.984966742616946e-01,   5.481232971088985e-02,
  -9.984861457732354e-01,   5.500378804145592e-02,
  -9.984755805732948e-01,   5.519524434968993e-02,
  -9.984649786622612e-01,   5.538669862855260e-02,
  -9.984543400405248e-01,   5.557815087100468e-02,
  -9.984436647084763e-01,   5.576960107000703e-02,
  -9.984329526665084e-01,   5.596104921852057e-02,
  -9.984222039150150e-01,   5.615249530950629e-02,
  -9.984114184543913e-01,   5.634393933592529e-02,
  -9.984005962850336e-01,   5.653538129073870e-02,
  -9.983897374073402e-01,   5.672682116690775e-02,
  -9.983788418217100e-01,   5.691825895739374e-02,
  -9.983679095285438e-01,   5.710969465515806e-02,
  -9.983569405282434e-01,   5.730112825316216e-02,
  -9.983459348212124e-01,   5.749255974436757e-02,
  -9.983348924078550e-01,   5.768398912173590e-02,
  -9.983238132885776e-01,   5.787541637822886e-02,
  -9.983126974637873e-01,   5.806684150680819e-02,
  -9.983015449338929e-01,   5.825826450043575e-02,
  -9.982903556993044e-01,   5.844968535207348e-02,
  -9.982791297604332e-01,   5.864110405468334e-02,
  -9.982678671176921e-01,   5.883252060122744e-02,
  -9.982565677714952e-01,   5.902393498466793e-02,
  -9.982452317222579e-01,   5.921534719796706e-02,
  -9.982338589703968e-01,   5.940675723408715e-02,
  -9.982224495163305e-01,   5.959816508599059e-02,
  -9.982110033604782e-01,   5.978957074663987e-02,
  -9.981995205032607e-01,   5.998097420899755e-02,
  -9.981880009451003e-01,   6.017237546602626e-02,
  -9.981764446864205e-01,   6.036377451068874e-02,
  -9.981648517276462e-01,   6.055517133594779e-02,
  -9.981532220692038e-01,   6.074656593476629e-02,
  -9.981415557115205e-01,   6.093795830010720e-02,
  -9.981298526550256e-01,   6.112934842493359e-02,
  -9.981181129001492e-01,   6.132073630220858e-02,
  -9.981063364473230e-01,   6.151212192489538e-02,
  -9.980945232969800e-01,   6.170350528595730e-02,
  -9.980826734495546e-01,   6.189488637835772e-02,
  -9.980707869054823e-01,   6.208626519506009e-02,
  -9.980588636652002e-01,   6.227764172902797e-02,
  -9.980469037291468e-01,   6.246901597322500e-02,
  -9.980349070977618e-01,   6.266038792061487e-02,
  -9.980228737714862e-01,   6.285175756416140e-02,
  -9.980108037507625e-01,   6.304312489682849e-02,
  -9.979986970360344e-01,   6.323448991158007e-02,
  -9.979865536277470e-01,   6.342585260138023e-02,
  -9.979743735263470e-01,   6.361721295919310e-02,
  -9.979621567322820e-01,   6.380857097798290e-02,
  -9.979499032460012e-01,   6.399992665071394e-02,
  -9.979376130679553e-01,   6.419127997035064e-02,
  -9.979252861985960e-01,   6.438263092985747e-02,
  -9.979129226383766e-01,   6.457397952219898e-02,
  -9.979005223877516e-01,   6.476532574033989e-02,
  -9.978880854471771e-01,   6.495666957724487e-02,
  -9.978756118171102e-01,   6.514801102587883e-02,
  -9.978631014980095e-01,   6.533935007920663e-02,
  -9.978505544903351e-01,   6.553068673019333e-02,
  -9.978379707945483e-01,   6.572202097180399e-02,
  -9.978253504111116e-01,   6.591335279700380e-02,
  -9.978126933404893e-01,   6.610468219875808e-02,
  -9.977999995831465e-01,   6.629600917003213e-02,
  -9.977872691395500e-01,   6.648733370379145e-02,
  -9.977745020101678e-01,   6.667865579300156e-02,
  -9.977616981954696e-01,   6.686997543062811e-02,
  -9.977488576959257e-01,   6.706129260963682e-02,
  -9.977359805120086e-01,   6.725260732299350e-02,
  -9.977230666441916e-01,   6.744391956366405e-02,
  -9.977101160929496e-01,   6.763522932461448e-02,
  -9.976971288587585e-01,   6.782653659881087e-02,
  -9.976841049420960e-01,   6.801784137921939e-02,
  -9.976710443434410e-01,   6.820914365880633e-02,
  -9.976579470632737e-01,   6.840044343053801e-02,
  -9.976448131020754e-01,   6.859174068738094e-02,
  -9.976316424603293e-01,   6.878303542230163e-02,
  -9.976184351385196e-01,   6.897432762826675e-02,
  -9.976051911371316e-01,   6.916561729824298e-02,
  -9.975919104566526e-01,   6.935690442519721e-02,
  -9.975785930975708e-01,   6.954818900209630e-02,
  -9.975652390603758e-01,   6.973947102190730e-02,
  -9.975518483455584e-01,   6.993075047759731e-02,
  -9.975384209536113e-01,   7.012202736213352e-02,
  -9.975249568850280e-01,   7.031330166848325e-02,
  -9.975114561403035e-01,   7.050457338961386e-02,
  -9.974979187199342e-01,   7.069584251849285e-02,
  -9.974843446244179e-01,   7.088710904808780e-02,
  -9.974707338542537e-01,   7.107837297136640e-02,
  -9.974570864099419e-01,   7.126963428129640e-02,
  -9.974434022919844e-01,   7.146089297084568e-02,
  -9.974296815008842e-01,   7.165214903298221e-02,
  -9.974159240371460e-01,   7.184340246067403e-02,
  -9.974021299012753e-01,   7.203465324688933e-02,
  -9.973882990937795e-01,   7.222590138459632e-02,
  -9.973744316151671e-01,   7.241714686676341e-02,
  -9.973605274659479e-01,   7.260838968635899e-02,
  -9.973465866466332e-01,   7.279962983635167e-02,
  -9.973326091577355e-01,   7.299086730971004e-02,
  -9.973185949997686e-01,   7.318210209940289e-02,
  -9.973045441732480e-01,   7.337333419839903e-02,
  -9.972904566786902e-01,   7.356456359966743e-02,
  -9.972763325166132e-01,   7.375579029617710e-02,
  -9.972621716875362e-01,   7.394701428089720e-02,
  -9.972479741919799e-01,   7.413823554679698e-02,
  -9.972337400304663e-01,   7.432945408684576e-02,
  -9.972194692035187e-01,   7.452066989401300e-02,
  -9.972051617116618e-01,   7.471188296126821e-02,
  -9.971908175554219e-01,   7.490309328158108e-02,
  -9.971764367353262e-01,   7.509430084792130e-02,
  -9.971620192519033e-01,   7.528550565325877e-02,
  -9.971475651056835e-01,   7.547670769056339e-02,
  -9.971330742971981e-01,   7.566790695280523e-02,
  -9.971185468269800e-01,   7.585910343295445e-02,
  -9.971039826955633e-01,   7.605029712398126e-02,
  -9.970893819034834e-01,   7.624148801885607e-02,
  -9.970747444512773e-01,   7.643267611054928e-02,
  -9.970600703394830e-01,   7.662386139203149e-02,
  -9.970453595686400e-01,   7.681504385627334e-02,
  -9.970306121392895e-01,   7.700622349624564e-02,
  -9.970158280519733e-01,   7.719740030491920e-02,
  -9.970010073072353e-01,   7.738857427526505e-02,
  -9.969861499056202e-01,   7.757974540025422e-02,
  -9.969712558476743e-01,   7.777091367285795e-02,
  -9.969563251339453e-01,   7.796207908604749e-02,
  -9.969413577649822e-01,   7.815324163279423e-02,
  -9.969263537413351e-01,   7.834440130606970e-02,
  -9.969113130635557e-01,   7.853555809884548e-02,
  -9.968962357321972e-01,   7.872671200409330e-02,
  -9.968811217478138e-01,   7.891786301478494e-02,
  -9.968659711109613e-01,   7.910901112389238e-02,
  -9.968507838221966e-01,   7.930015632438760e-02,
  -9.968355598820782e-01,   7.949129860924277e-02,
  -9.968202992911657e-01,   7.968243797143013e-02,
  -9.968050020500204e-01,   7.987357440392200e-02,
  -9.967896681592046e-01,   8.006470789969089e-02,
  -9.967742976192820e-01,   8.025583845170932e-02,
  -9.967588904308180e-01,   8.044696605295001e-02,
  -9.967434465943789e-01,   8.063809069638571e-02,
  -9.967279661105325e-01,   8.082921237498933e-02,
  -9.967124489798480e-01,   8.102033108173386e-02,
  -9.966968952028961e-01,   8.121144680959244e-02,
  -9.966813047802483e-01,   8.140255955153824e-02,
  -9.966656777124782e-01,   8.159366930054465e-02,
  -9.966500140001601e-01,   8.178477604958508e-02,
  -9.966343136438699e-01,   8.197587979163307e-02,
  -9.966185766441851e-01,   8.216698051966231e-02,
  -9.966028030016841e-01,   8.235807822664654e-02,
  -9.965869927169470e-01,   8.254917290555967e-02,
  -9.965711457905548e-01,   8.274026454937569e-02,
  -9.965552622230905e-01,   8.293135315106870e-02,
  -9.965393420151379e-01,   8.312243870361291e-02,
  -9.965233851672824e-01,   8.331352119998268e-02,
  -9.965073916801108e-01,   8.350460063315243e-02,
  -9.964913615542109e-01,   8.369567699609672e-02,
  -9.964752947901722e-01,   8.388675028179023e-02,
  -9.964591913885854e-01,   8.407782048320770e-02,
  -9.964430513500426e-01,   8.426888759332407e-02,
  -9.964268746751372e-01,   8.445995160511433e-02,
  -9.964106613644641e-01,   8.465101251155362e-02,
  -9.963944114186193e-01,   8.484207030561713e-02,
  -9.963781248382002e-01,   8.503312498028028e-02,
  -9.963618016238057e-01,   8.522417652851848e-02,
  -9.963454417760359e-01,   8.541522494330733e-02,
  -9.963290452954924e-01,   8.560627021762253e-02,
  -9.963126121827780e-01,   8.579731234443990e-02,
  -9.962961424384968e-01,   8.598835131673534e-02,
  -9.962796360632546e-01,   8.617938712748490e-02,
  -9.962630930576581e-01,   8.637041976966475e-02,
  -9.962465134223155e-01,   8.656144923625117e-02,
  -9.962298971578365e-01,   8.675247552022054e-02,
  -9.962132442648320e-01,   8.694349861454938e-02,
  -9.961965547439142e-01,   8.713451851221430e-02,
  -9.961798285956970e-01,   8.732553520619206e-02,
  -9.961630658207950e-01,   8.751654868945953e-02,
  -9.961462664198246e-01,   8.770755895499366e-02,
  -9.961294303934037e-01,   8.789856599577159e-02,
  -9.961125577421511e-01,   8.808956980477050e-02,
  -9.960956484666873e-01,   8.828057037496774e-02,
  -9.960787025676340e-01,   8.847156769934077e-02,
  -9.960617200456140e-01,   8.866256177086715e-02,
  -9.960447009012520e-01,   8.885355258252460e-02,
  -9.960276451351736e-01,   8.904454012729089e-02,
  -9.960105527480059e-01,   8.923552439814401e-02,
  -9.959934237403774e-01,   8.942650538806196e-02,
  -9.959762581129178e-01,   8.961748309002296e-02,
  -9.959590558662583e-01,   8.980845749700528e-02,
  -9.959418170010313e-01,   8.999942860198734e-02,
  -9.959245415178708e-01,   9.019039639794770e-02,
  -9.959072294174117e-01,   9.038136087786498e-02,
  -9.958898807002907e-01,   9.057232203471799e-02,
  -9.958724953671457e-01,   9.076327986148562e-02,
  -9.958550734186158e-01,   9.095423435114693e-02,
  -9.958376148553416e-01,   9.114518549668100e-02,
  -9.958201196779649e-01,   9.133613329106718e-02,
  -9.958025878871292e-01,   9.152707772728483e-02,
  -9.957850194834788e-01,   9.171801879831346e-02,
  -9.957674144676598e-01,   9.190895649713272e-02,
  -9.957497728403195e-01,   9.209989081672239e-02,
  -9.957320946021064e-01,   9.229082175006235e-02,
  -9.957143797536706e-01,   9.248174929013260e-02,
  -9.956966282956635e-01,   9.267267342991331e-02,
  -9.956788402287375e-01,   9.286359416238472e-02,
  -9.956610155535469e-01,   9.305451148052725e-02,
  -9.956431542707469e-01,   9.324542537732138e-02,
  -9.956252563809943e-01,   9.343633584574779e-02,
  -9.956073218849470e-01,   9.362724287878720e-02,
  -9.955893507832646e-01,   9.381814646942055e-02,
  -9.955713430766078e-01,   9.400904661062884e-02,
  -9.955532987656385e-01,   9.419994329539320e-02,
  -9.955352178510204e-01,   9.439083651669494e-02,
  -9.955171003334181e-01,   9.458172626751545e-02,
  -9.954989462134978e-01,   9.477261254083624e-02,
  -9.954807554919269e-01,   9.496349532963899e-02,
  -9.954625281693744e-01,   9.515437462690549e-02,
  -9.954442642465103e-01,   9.534525042561762e-02,
  -9.954259637240062e-01,   9.553612271875747e-02,
  -9.954076266025349e-01,   9.572699149930716e-02,
  -9.953892528827707e-01,   9.591785676024904e-02,
  -9.953708425653890e-01,   9.610871849456551e-02,
  -9.953523956510668e-01,   9.629957669523913e-02,
  -9.953339121404823e-01,   9.649043135525259e-02,
  -9.953153920343151e-01,   9.668128246758873e-02,
  -9.952968353332461e-01,   9.687213002523047e-02,
  -9.952782420379577e-01,   9.706297402116092e-02,
  -9.952596121491334e-01,   9.725381444836327e-02,
  -9.952409456674581e-01,   9.744465129982087e-02,
  -9.952222425936184e-01,   9.763548456851720e-02,
  -9.952035029283015e-01,   9.782631424743586e-02,
  -9.951847266721969e-01,   9.801714032956060e-02,
  -9.951659138259946e-01,   9.820796280787528e-02,
  -9.951470643903865e-01,   9.839878167536388e-02,
  -9.951281783660655e-01,   9.858959692501058e-02,
  -9.951092557537261e-01,   9.878040854979962e-02,
  -9.950902965540640e-01,   9.897121654271543e-02,
  -9.950713007677762e-01,   9.916202089674250e-02,
  -9.950522683955610e-01,   9.935282160486554e-02,
  -9.950331994381186e-01,   9.954361866006932e-02,
  -9.950140938961497e-01,   9.973441205533883e-02,
  -9.949949517703570e-01,   9.992520178365907e-02,
  -9.949757730614441e-01,   1.001159878380153e-01,
  -9.949565577701164e-01,   1.003067702113929e-01,
  -9.949373058970801e-01,   1.004975488967772e-01,
  -9.949180174430432e-01,   1.006883238871540e-01,
  -9.948986924087149e-01,   1.008790951755089e-01,
  -9.948793307948056e-01,   1.010698627548278e-01,
  -9.948599326020273e-01,   1.012606266180968e-01,
  -9.948404978310932e-01,   1.014513867583021e-01,
  -9.948210264827179e-01,   1.016421431684298e-01,
  -9.948015185576171e-01,   1.018328958414665e-01,
  -9.947819740565083e-01,   1.020236447703987e-01,
  -9.947623929801099e-01,   1.022143899482132e-01,
  -9.947427753291420e-01,   1.024051313678967e-01,
  -9.947231211043257e-01,   1.025958690224363e-01,
  -9.947034303063839e-01,   1.027866029048190e-01,
  -9.946837029360402e-01,   1.029773330080322e-01,
  -9.946639389940204e-01,   1.031680593250632e-01,
  -9.946441384810507e-01,   1.033587818488996e-01,
  -9.946243013978594e-01,   1.035495005725291e-01,
  -9.946044277451757e-01,   1.037402154889394e-01,
  -9.945845175237303e-01,   1.039309265911185e-01,
  -9.945645707342554e-01,   1.041216338720546e-01,
  -9.945445873774843e-01,   1.043123373247358e-01,
  -9.945245674541517e-01,   1.045030369421506e-01,
  -9.945045109649937e-01,   1.046937327172874e-01,
  -9.944844179107476e-01,   1.048844246431350e-01,
  -9.944642882921524e-01,   1.050751127126820e-01,
  -9.944441221099480e-01,   1.052657969189176e-01,
  -9.944239193648760e-01,   1.054564772548307e-01,
  -9.944036800576791e-01,   1.056471537134106e-01,
  -9.943834041891014e-01,   1.058378262876467e-01,
  -9.943630917598886e-01,   1.060284949705284e-01,
  -9.943427427707873e-01,   1.062191597550455e-01,
  -9.943223572225458e-01,   1.064098206341877e-01,
  -9.943019351159136e-01,   1.066004776009450e-01,
  -9.942814764516416e-01,   1.067911306483074e-01,
  -9.942609812304818e-01,   1.069817797692652e-01,
  -9.942404494531879e-01,   1.071724249568088e-01,
  -9.942198811205150e-01,   1.073630662039288e-01,
  -9.941992762332189e-01,   1.075537035036156e-01,
  -9.941786347920576e-01,   1.077443368488603e-01,
  -9.941579567977897e-01,   1.079349662326537e-01,
  -9.941372422511757e-01,   1.081255916479869e-01,
  -9.941164911529771e-01,   1.083162130878512e-01,
  -9.940957035039569e-01,   1.085068305452379e-01,
  -9.940748793048794e-01,   1.086974440131387e-01,
  -9.940540185565102e-01,   1.088880534845452e-01,
  -9.940331212596164e-01,   1.090786589524492e-01,
  -9.940121874149662e-01,   1.092692604098428e-01,
  -9.939912170233294e-01,   1.094598578497180e-01,
  -9.939702100854769e-01,   1.096504512650671e-01,
  -9.939491666021811e-01,   1.098410406488826e-01,
  -9.939280865742158e-01,   1.100316259941570e-01,
  -9.939069700023561e-01,   1.102222072938831e-01,
  -9.938858168873781e-01,   1.104127845410536e-01,
  -9.938646272300597e-01,   1.106033577286617e-01,
  -9.938434010311802e-01,   1.107939268497006e-01,
  -9.938221382915197e-01,   1.109844918971634e-01,
  -9.938008390118601e-01,   1.111750528640437e-01,
  -9.937795031929846e-01,   1.113656097433352e-01,
  -9.937581308356774e-01,   1.115561625280315e-01,
  -9.937367219407246e-01,   1.117467112111266e-01,
  -9.937152765089132e-01,   1.119372557856146e-01,
  -9.936937945410318e-01,   1.121277962444896e-01,
  -9.936722760378700e-01,   1.123183325807462e-01,
  -9.936507210002191e-01,   1.125088647873787e-01,
  -9.936291294288717e-01,   1.126993928573819e-01,
  -9.936075013246216e-01,   1.128899167837505e-01,
  -9.935858366882640e-01,   1.130804365594796e-01,
  -9.935641355205953e-01,   1.132709521775643e-01,
  -9.935423978224136e-01,   1.134614636309999e-01,
  -9.935206235945181e-01,   1.136519709127819e-01,
  -9.934988128377094e-01,   1.138424740159057e-01,
  -9.934769655527892e-01,   1.140329729333672e-01,
  -9.934550817405610e-01,   1.142234676581623e-01,
  -9.934331614018294e-01,   1.144139581832869e-01,
  -9.934112045374001e-01,   1.146044445017374e-01,
  -9.933892111480807e-01,   1.147949266065101e-01,
  -9.933671812346796e-01,   1.149854044906015e-01,
  -9.933451147980069e-01,   1.151758781470082e-01,
  -9.933230118388740e-01,   1.153663475687271e-01,
  -9.933008723580933e-01,   1.155568127487553e-01,
  -9.932786963564790e-01,   1.157472736800897e-01,
  -9.932564838348464e-01,   1.159377303557278e-01,
  -9.932342347940123e-01,   1.161281827686669e-01,
  -9.932119492347945e-01,   1.163186309119048e-01,
  -9.931896271580126e-01,   1.165090747784390e-01,
  -9.931672685644872e-01,   1.166995143612677e-01,
  -9.931448734550404e-01,   1.168899496533888e-01,
  -9.931224418304956e-01,   1.170803806478006e-01,
  -9.930999736916776e-01,   1.172708073375015e-01,
  -9.930774690394123e-01,   1.174612297154900e-01,
  -9.930549278745273e-01,   1.176516477747649e-01,
  -9.930323501978514e-01,   1.178420615083250e-01,
  -9.930097360102146e-01,   1.180324709091693e-01,
  -9.929870853124484e-01,   1.182228759702972e-01,
  -9.929643981053856e-01,   1.184132766847078e-01,
  -9.929416743898605e-01,   1.186036730454007e-01,
  -9.929189141667083e-01,   1.187940650453756e-01,
  -9.928961174367660e-01,   1.189844526776323e-01,
  -9.928732842008717e-01,   1.191748359351709e-01,
  -9.928504144598651e-01,   1.193652148109914e-01,
  -9.928275082145868e-01,   1.195555892980941e-01,
  -9.928045654658791e-01,   1.197459593894796e-01,
  -9.927815862145856e-01,   1.199363250781485e-01,
  -9.927585704615511e-01,   1.201266863571015e-01,
  -9.927355182076218e-01,   1.203170432193397e-01,
  -9.927124294536455e-01,   1.205073956578641e-01,
  -9.926893042004707e-01,   1.206977436656761e-01,
  -9.926661424489480e-01,   1.208880872357771e-01,
  -9.926429441999288e-01,   1.210784263611686e-01,
  -9.926197094542661e-01,   1.212687610348526e-01,
  -9.925964382128143e-01,   1.214590912498308e-01,
  -9.925731304764288e-01,   1.216494169991055e-01,
  -9.925497862459667e-01,   1.218397382756789e-01,
  -9.925264055222861e-01,   1.220300550725534e-01,
  -9.925029883062470e-01,   1.222203673827315e-01,
  -9.924795345987100e-01,   1.224106751992162e-01,
  -9.924560444005377e-01,   1.226009785150102e-01,
  -9.924325177125937e-01,   1.227912773231168e-01,
  -9.924089545357428e-01,   1.229815716165391e-01,
  -9.923853548708517e-01,   1.231718613882805e-01,
  -9.923617187187879e-01,   1.233621466313447e-01,
  -9.923380460804204e-01,   1.235524273387354e-01,
  -9.923143369566196e-01,   1.237427035034565e-01,
  -9.922905913482574e-01,   1.239329751185122e-01,
  -9.922668092562066e-01,   1.241232421769066e-01,
  -9.922429906813417e-01,   1.243135046716442e-01,
  -9.922191356245385e-01,   1.245037625957297e-01,
  -9.921952440866739e-01,   1.246940159421676e-01,
  -9.921713160686265e-01,   1.248842647039631e-01,
  -9.921473515712761e-01,   1.250745088741212e-01,
  -9.921233505955037e-01,   1.252647484456471e-01,
  -9.920993131421918e-01,   1.254549834115462e-01,
  -9.920752392122241e-01,   1.256452137648243e-01,
  -9.920511288064857e-01,   1.258354394984870e-01,
  -9.920269819258634e-01,   1.260256606055403e-01,
  -9.920027985712445e-01,   1.262158770789903e-01,
  -9.919785787435186e-01,   1.264060889118434e-01,
  -9.919543224435760e-01,   1.265962960971058e-01,
  -9.919300296723085e-01,   1.267864986277844e-01,
  -9.919057004306093e-01,   1.269766964968859e-01,
  -9.918813347193730e-01,   1.271668896974172e-01,
  -9.918569325394955e-01,   1.273570782223854e-01,
  -9.918324938918738e-01,   1.275472620647980e-01,
  -9.918080187774064e-01,   1.277374412176623e-01,
  -9.917835071969935e-01,   1.279276156739861e-01,
  -9.917589591515361e-01,   1.281177854267771e-01,
  -9.917343746419368e-01,   1.283079504690434e-01,
  -9.917097536690995e-01,   1.284981107937932e-01,
  -9.916850962339294e-01,   1.286882663940347e-01,
  -9.916604023373332e-01,   1.288784172627765e-01,
  -9.916356719802187e-01,   1.290685633930274e-01,
  -9.916109051634954e-01,   1.292587047777961e-01,
  -9.915861018880735e-01,   1.294488414100918e-01,
  -9.915612621548653e-01,   1.296389732829236e-01,
  -9.915363859647839e-01,   1.298291003893009e-01,
  -9.915114733187439e-01,   1.300192227222333e-01,
  -9.914865242176615e-01,   1.302093402747306e-01,
  -9.914615386624538e-01,   1.303994530398027e-01,
  -9.914365166540394e-01,   1.305895610104597e-01,
  -9.914114581933385e-01,   1.307796641797117e-01,
  -9.913863632812723e-01,   1.309697625405694e-01,
  -9.913612319187635e-01,   1.311598560860433e-01,
  -9.913360641067361e-01,   1.313499448091442e-01,
  -9.913108598461154e-01,   1.315400287028831e-01,
  -9.912856191378282e-01,   1.317301077602712e-01,
  -9.912603419828024e-01,   1.319201819743198e-01,
  -9.912350283819674e-01,   1.321102513380404e-01,
  -9.912096783362541e-01,   1.323003158444447e-01,
  -9.911842918465942e-01,   1.324903754865445e-01,
  -9.911588689139214e-01,   1.326804302573521e-01,
  -9.911334095391702e-01,   1.328704801498794e-01,
  -9.911079137232769e-01,   1.330605251571391e-01,
  -9.910823814671786e-01,   1.332505652721436e-01,
  -9.910568127718143e-01,   1.334406004879057e-01,
  -9.910312076381241e-01,   1.336306307974383e-01,
  -9.910055660670494e-01,   1.338206561937547e-01,
  -9.909798880595327e-01,   1.340106766698681e-01,
  -9.909541736165185e-01,   1.342006922187920e-01,
  -9.909284227389520e-01,   1.343907028335401e-01,
  -9.909026354277800e-01,   1.345807085071262e-01,
  -9.908768116839507e-01,   1.347707092325643e-01,
  -9.908509515084136e-01,   1.349607050028687e-01,
  -9.908250549021195e-01,   1.351506958110539e-01,
  -9.907991218660204e-01,   1.353406816501342e-01,
  -9.907731524010698e-01,   1.355306625131246e-01,
  -9.907471465082227e-01,   1.357206383930399e-01,
  -9.907211041884352e-01,   1.359106092828953e-01,
  -9.906950254426646e-01,   1.361005751757062e-01,
  -9.906689102718701e-01,   1.362905360644880e-01,
  -9.906427586770116e-01,   1.364804919422563e-01,
  -9.906165706590506e-01,   1.366704428020271e-01,
  -9.905903462189501e-01,   1.368603886368164e-01,
  -9.905640853576744e-01,   1.370503294396404e-01,
  -9.905377880761888e-01,   1.372402652035156e-01,
  -9.905114543754603e-01,   1.374301959214586e-01,
  -9.904850842564571e-01,   1.376201215864860e-01,
  -9.904586777201486e-01,   1.378100421916151e-01,
  -9.904322347675060e-01,   1.379999577298628e-01,
  -9.904057553995013e-01,   1.381898681942466e-01,
  -9.903792396171082e-01,   1.383797735777839e-01,
  -9.903526874213014e-01,   1.385696738734925e-01,
  -9.903260988130573e-01,   1.387595690743904e-01,
  -9.902994737933536e-01,   1.389494591734955e-01,
  -9.902728123631691e-01,   1.391393441638262e-01,
  -9.902461145234840e-01,   1.393292240384010e-01,
  -9.902193802752800e-01,   1.395190987902385e-01,
  -9.901926096195400e-01,   1.397089684123576e-01,
  -9.901658025572484e-01,   1.398988328977772e-01,
  -9.901389590893906e-01,   1.400886922395167e-01,
  -9.901120792169538e-01,   1.402785464305954e-01,
  -9.900851629409260e-01,   1.404683954640330e-01,
  -9.900582102622971e-01,   1.406582393328492e-01,
  -9.900312211820580e-01,   1.408480780300641e-01,
  -9.900041957012009e-01,   1.410379115486977e-01,
  -9.899771338207196e-01,   1.412277398817705e-01,
  -9.899500355416090e-01,   1.414175630223030e-01,
  -9.899229008648655e-01,   1.416073809633160e-01,
  -9.898957297914867e-01,   1.417971936978304e-01,
  -9.898685223224716e-01,   1.419870012188673e-01,
  -9.898412784588205e-01,   1.421768035194480e-01,
  -9.898139982015353e-01,   1.423666005925942e-01,
  -9.897866815516186e-01,   1.425563924313273e-01,
  -9.897593285100752e-01,   1.427461790286695e-01,
  -9.897319390779106e-01,   1.429359603776427e-01,
  -9.897045132561318e-01,   1.431257364712692e-01,
  -9.896770510457472e-01,   1.433155073025715e-01,
  -9.896495524477665e-01,   1.435052728645723e-01,
  -9.896220174632009e-01,   1.436950331502945e-01,
  -9.895944460930625e-01,   1.438847881527610e-01,
  -9.895668383383651e-01,   1.440745378649952e-01,
  -9.895391942001239e-01,   1.442642822800204e-01,
  -9.895115136793552e-01,   1.444540213908605e-01,
  -9.894837967770768e-01,   1.446437551905390e-01,
  -9.894560434943077e-01,   1.448334836720802e-01,
  -9.894282538320682e-01,   1.450232068285082e-01,
  -9.894004277913804e-01,   1.452129246528475e-01,
  -9.893725653732670e-01,   1.454026371381226e-01,
  -9.893446665787526e-01,   1.455923442773583e-01,
  -9.893167314088630e-01,   1.457820460635799e-01,
  -9.892887598646252e-01,   1.459717424898122e-01,
  -9.892607519470676e-01,   1.461614335490809e-01,
  -9.892327076572200e-01,   1.463511192344115e-01,
  -9.892046269961138e-01,   1.465407995388298e-01,
  -9.891765099647810e-01,   1.467304744553617e-01,
  -9.891483565642556e-01,   1.469201439770336e-01,
  -9.891201667955727e-01,   1.471098080968718e-01,
  -9.890919406597688e-01,   1.472994668079028e-01,
  -9.890636781578815e-01,   1.474891201031536e-01,
  -9.890353792909503e-01,   1.476787679756510e-01,
  -9.890070440600153e-01,   1.478684104184222e-01,
  -9.889786724661185e-01,   1.480580474244947e-01,
  -9.889502645103030e-01,   1.482476789868960e-01,
  -9.889218201936132e-01,   1.484373050986540e-01,
  -9.888933395170951e-01,   1.486269257527965e-01,
  -9.888648224817956e-01,   1.488165409423519e-01,
  -9.888362690887635e-01,   1.490061506603484e-01,
  -9.888076793390485e-01,   1.491957548998148e-01,
  -9.887790532337015e-01,   1.493853536537797e-01,
  -9.887503907737754e-01,   1.495749469152722e-01,
  -9.887216919603238e-01,   1.497645346773215e-01,
  -9.886929567944020e-01,   1.499541169329570e-01,
  -9.886641852770662e-01,   1.501436936752082e-01,
  -9.886353774093748e-01,   1.503332648971050e-01,
  -9.886065331923864e-01,   1.505228305916774e-01,
  -9.885776526271620e-01,   1.507123907519556e-01,
  -9.885487357147632e-01,   1.509019453709700e-01,
  -9.885197824562533e-01,   1.510914944417513e-01,
  -9.884907928526966e-01,   1.512810379573302e-01,
  -9.884617669051593e-01,   1.514705759107378e-01,
  -9.884327046147083e-01,   1.516601082950053e-01,
  -9.884036059824124e-01,   1.518496351031642e-01,
  -9.883744710093413e-01,   1.520391563282461e-01,
  -9.883452996965661e-01,   1.522286719632827e-01,
  -9.883160920451597e-01,   1.524181820013063e-01,
  -9.882868480561958e-01,   1.526076864353491e-01,
  -9.882575677307495e-01,   1.527971852584434e-01,
  -9.882282510698974e-01,   1.529866784636220e-01,
  -9.881988980747176e-01,   1.531761660439178e-01,
  -9.881695087462891e-01,   1.533656479923639e-01,
  -9.881400830856926e-01,   1.535551243019934e-01,
  -9.881106210940098e-01,   1.537445949658400e-01,
  -9.880811227723241e-01,   1.539340599769373e-01,
  -9.880515881217201e-01,   1.541235193283194e-01,
  -9.880220171432835e-01,   1.543129730130201e-01,
  -9.879924098381019e-01,   1.545024210240739e-01,
  -9.879627662072634e-01,   1.546918633545154e-01,
  -9.879330862518584e-01,   1.548812999973793e-01,
  -9.879033699729778e-01,   1.550707309457005e-01,
  -9.878736173717142e-01,   1.552601561925142e-01,
  -9.878438284491617e-01,   1.554495757308559e-01,
  -9.878140032064155e-01,   1.556389895537609e-01,
  -9.877841416445722e-01,   1.558283976542652e-01,
  -9.877542437647295e-01,   1.560178000254048e-01,
  -9.877243095679870e-01,   1.562071966602159e-01,
  -9.876943390554451e-01,   1.563965875517349e-01,
  -9.876643322282057e-01,   1.565859726929984e-01,
  -9.876342890873722e-01,   1.567753520770434e-01,
  -9.876042096340492e-01,   1.569647256969068e-01,
  -9.875740938693424e-01,   1.571540935456259e-01,
  -9.875439417943592e-01,   1.573434556162382e-01,
  -9.875137534102084e-01,   1.575328119017815e-01,
  -9.874835287179997e-01,   1.577221623952936e-01,
  -9.874532677188446e-01,   1.579115070898127e-01,
  -9.874229704138554e-01,   1.581008459783770e-01,
  -9.873926368041462e-01,   1.582901790540252e-01,
  -9.873622668908324e-01,   1.584795063097960e-01,
  -9.873318606750304e-01,   1.586688277387283e-01,
  -9.873014181578584e-01,   1.588581433338614e-01,
  -9.872709393404354e-01,   1.590474530882348e-01,
  -9.872404242238823e-01,   1.592367569948878e-01,
  -9.872098728093208e-01,   1.594260550468606e-01,
  -9.871792850978743e-01,   1.596153472371931e-01,
  -9.871486610906676e-01,   1.598046335589254e-01,
  -9.871180007888263e-01,   1.599939140050983e-01,
  -9.870873041934779e-01,   1.601831885687522e-01,
  -9.870565713057510e-01,   1.603724572429283e-01,
  -9.870258021267756e-01,   1.605617200206675e-01,
  -9.869949966576830e-01,   1.607509768950112e-01,
  -9.869641548996057e-01,   1.609402278590011e-01,
  -9.869332768536777e-01,   1.611294729056788e-01,
  -9.869023625210345e-01,   1.613187120280864e-01,
  -9.868714119028125e-01,   1.615079452192661e-01,
  -9.868404250001497e-01,   1.616971724722604e-01,
  -9.868094018141855e-01,   1.618863937801118e-01,
  -9.867783423460604e-01,   1.620756091358633e-01,
  -9.867472465969166e-01,   1.622648185325580e-01,
  -9.867161145678971e-01,   1.624540219632392e-01,
  -9.866849462601467e-01,   1.626432194209503e-01,
  -9.866537416748113e-01,   1.628324108987352e-01,
  -9.866225008130385e-01,   1.630215963896378e-01,
  -9.865912236759764e-01,   1.632107758867024e-01,
  -9.865599102647754e-01,   1.633999493829732e-01,
  -9.865285605805867e-01,   1.635891168714950e-01,
  -9.864971746245629e-01,   1.637782783453127e-01,
  -9.864657523978579e-01,   1.639674337974712e-01,
  -9.864342939016272e-01,   1.641565832210158e-01,
  -9.864027991370272e-01,   1.643457266089922e-01,
  -9.863712681052160e-01,   1.645348639544460e-01,
  -9.863397008073530e-01,   1.647239952504232e-01,
  -9.863080972445987e-01,   1.649131204899699e-01,
  -9.862764574181151e-01,   1.651022396661327e-01,
  -9.862447813290655e-01,   1.652913527719580e-01,
  -9.862130689786145e-01,   1.654804598004928e-01,
  -9.861813203679283e-01,   1.656695607447841e-01,
  -9.861495354981739e-01,   1.658586555978793e-01,
  -9.861177143705201e-01,   1.660477443528258e-01,
  -9.860858569861368e-01,   1.662368270026714e-01,
  -9.860539633461954e-01,   1.664259035404641e-01,
  -9.860220334518686e-01,   1.666149739592521e-01,
  -9.859900673043301e-01,   1.668040382520837e-01,
  -9.859580649047555e-01,   1.669930964120077e-01,
  -9.859260262543211e-01,   1.671821484320729e-01,
  -9.858939513542052e-01,   1.673711943053284e-01,
  -9.858618402055870e-01,   1.675602340248236e-01,
  -9.858296928096470e-01,   1.677492675836079e-01,
  -9.857975091675675e-01,   1.679382949747312e-01,
  -9.857652892805313e-01,   1.681273161912434e-01,
  -9.857330331497235e-01,   1.683163312261948e-01,
  -9.857007407763299e-01,   1.685053400726359e-01,
  -9.856684121615376e-01,   1.686943427236173e-01,
  -9.856360473065354e-01,   1.688833391721900e-01,
  -9.856036462125134e-01,   1.690723294114050e-01,
  -9.855712088806627e-01,   1.692613134343138e-01,
  -9.855387353121761e-01,   1.694502912339680e-01,
  -9.855062255082473e-01,   1.696392628034193e-01,
  -9.854736794700718e-01,   1.698282281357198e-01,
  -9.854410971988462e-01,   1.700171872239220e-01,
  -9.854084786957684e-01,   1.702061400610781e-01,
  -9.853758239620377e-01,   1.703950866402409e-01,
  -9.853431329988548e-01,   1.705840269544636e-01,
  -9.853104058074216e-01,   1.707729609967992e-01,
  -9.852776423889412e-01,   1.709618887603012e-01,
  -9.852448427446185e-01,   1.711508102380233e-01,
  -9.852120068756594e-01,   1.713397254230193e-01,
  -9.851791347832711e-01,   1.715286343083434e-01,
  -9.851462264686622e-01,   1.717175368870500e-01,
  -9.851132819330427e-01,   1.719064331521935e-01,
  -9.850803011776238e-01,   1.720953230968290e-01,
  -9.850472842036182e-01,   1.722842067140114e-01,
  -9.850142310122398e-01,   1.724730839967960e-01,
  -9.849811416047040e-01,   1.726619549382383e-01,
  -9.849480159822270e-01,   1.728508195313941e-01,
  -9.849148541460272e-01,   1.730396777693194e-01,
  -9.848816560973237e-01,   1.732285296450703e-01,
  -9.848484218373370e-01,   1.734173751517035e-01,
  -9.848151513672891e-01,   1.736062142822754e-01,
  -9.847818446884034e-01,   1.737950470298432e-01,
  -9.847485018019042e-01,   1.739838733874638e-01,
  -9.847151227090176e-01,   1.741726933481948e-01,
  -9.846817074109709e-01,   1.743615069050937e-01,
  -9.846482559089926e-01,   1.745503140512185e-01,
  -9.846147682043126e-01,   1.747391147796272e-01,
  -9.845812442981622e-01,   1.749279090833782e-01,
  -9.845476841917740e-01,   1.751166969555299e-01,
  -9.845140878863818e-01,   1.753054783891413e-01,
  -9.844804553832209e-01,   1.754942533772714e-01,
  -9.844467866835279e-01,   1.756830219129795e-01,
  -9.844130817885407e-01,   1.758717839893250e-01,
  -9.843793406994985e-01,   1.760605395993678e-01,
  -9.843455634176419e-01,   1.762492887361679e-01,
  -9.843117499442128e-01,   1.764380313927854e-01,
  -9.842779002804544e-01,   1.766267675622809e-01,
  -9.842440144276111e-01,   1.768154972377150e-01,
  -9.842100923869290e-01,   1.770042204121487e-01,
  -9.841761341596553e-01,   1.771929370786433e-01,
  -9.841421397470386e-01,   1.773816472302600e-01,
  -9.841081091503285e-01,   1.775703508600607e-01,
  -9.840740423707764e-01,   1.777590479611072e-01,
  -9.840399394096350e-01,   1.779477385264616e-01,
  -9.840058002681579e-01,   1.781364225491863e-01,
  -9.839716249476003e-01,   1.783251000223440e-01,
  -9.839374134492189e-01,   1.785137709389975e-01,
  -9.839031657742715e-01,   1.787024352922100e-01,
  -9.838688819240172e-01,   1.788910930750447e-01,
  -9.838345618997166e-01,   1.790797442805654e-01,
  -9.838002057026316e-01,   1.792683889018357e-01,
  -9.837658133340252e-01,   1.794570269319199e-01,
  -9.837313847951621e-01,   1.796456583638822e-01,
  -9.836969200873081e-01,   1.798342831907871e-01,
  -9.836624192117303e-01,   1.800229014056995e-01,
  -9.836278821696972e-01,   1.802115130016844e-01,
  -9.835933089624787e-01,   1.804001179718072e-01,
  -9.835586995913459e-01,   1.805887163091333e-01,
  -9.835240540575713e-01,   1.807773080067286e-01,
  -9.834893723624287e-01,   1.809658930576590e-01,
  -9.834546545071933e-01,   1.811544714549908e-01,
  -9.834199004931415e-01,   1.813430431917905e-01,
  -9.833851103215512e-01,   1.815316082611250e-01,
  -9.833502839937015e-01,   1.817201666560611e-01,
  -9.833154215108728e-01,   1.819087183696662e-01,
  -9.832805228743470e-01,   1.820972633950076e-01,
  -9.832455880854071e-01,   1.822858017251533e-01,
  -9.832106171453376e-01,   1.824743333531711e-01,
  -9.831756100554244e-01,   1.826628582721293e-01,
  -9.831405668169545e-01,   1.828513764750963e-01,
  -9.831054874312163e-01,   1.830398879551410e-01,
  -9.830703718994996e-01,   1.832283927053321e-01,
  -9.830352202230956e-01,   1.834168907187391e-01,
  -9.830000324032966e-01,   1.836053819884313e-01,
  -9.829648084413964e-01,   1.837938665074784e-01,
  -9.829295483386902e-01,   1.839823442689505e-01,
  -9.828942520964741e-01,   1.841708152659177e-01,
  -9.828589197160461e-01,   1.843592794914505e-01,
  -9.828235511987052e-01,   1.845477369386196e-01,
  -9.827881465457520e-01,   1.847361876004960e-01,
  -9.827527057584878e-01,   1.849246314701508e-01,
  -9.827172288382160e-01,   1.851130685406555e-01,
  -9.826817157862409e-01,   1.853014988050819e-01,
  -9.826461666038680e-01,   1.854899222565019e-01,
  -9.826105812924048e-01,   1.856783388879876e-01,
  -9.825749598531592e-01,   1.858667486926117e-01,
  -9.825393022874412e-01,   1.860551516634466e-01,
  -9.825036085965618e-01,   1.862435477935656e-01,
  -9.824678787818332e-01,   1.864319370760416e-01,
  -9.824321128445691e-01,   1.866203195039483e-01,
  -9.823963107860847e-01,   1.868086950703593e-01,
  -9.823604726076962e-01,   1.869970637683485e-01,
  -9.823245983107213e-01,   1.871854255909903e-01,
  -9.822886878964788e-01,   1.873737805313591e-01,
  -9.822527413662894e-01,   1.875621285825296e-01,
  -9.822167587214745e-01,   1.877504697375768e-01,
  -9.821807399633571e-01,   1.879388039895759e-01,
  -9.821446850932616e-01,   1.881271313316024e-01,
  -9.821085941125136e-01,   1.883154517567321e-01,
  -9.820724670224400e-01,   1.885037652580409e-01,
  -9.820363038243690e-01,   1.886920718286052e-01,
  -9.820001045196305e-01,   1.888803714615014e-01,
  -9.819638691095552e-01,   1.890686641498062e-01,
  -9.819275975954755e-01,   1.892569498865967e-01,
  -9.818912899787251e-01,   1.894452286649502e-01,
  -9.818549462606386e-01,   1.896335004779442e-01,
  -9.818185664425525e-01,   1.898217653186564e-01,
  -9.817821505258043e-01,   1.900100231801650e-01,
  -9.817456985117330e-01,   1.901982740555481e-01,
  -9.817092104016788e-01,   1.903865179378845e-01,
  -9.816726861969831e-01,   1.905747548202527e-01,
  -9.816361258989891e-01,   1.907629846957321e-01,
  -9.815995295090407e-01,   1.909512075574018e-01,
  -9.815628970284836e-01,   1.911394233983414e-01,
  -9.815262284586648e-01,   1.913276322116309e-01,
  -9.814895238009321e-01,   1.915158339903502e-01,
  -9.814527830566355e-01,   1.917040287275798e-01,
  -9.814160062271255e-01,   1.918922164164002e-01,
  -9.813791933137546e-01,   1.920803970498924e-01,
  -9.813423443178760e-01,   1.922685706211375e-01,
  -9.813054592408447e-01,   1.924567371232168e-01,
  -9.812685380840167e-01,   1.926448965492121e-01,
  -9.812315808487497e-01,   1.928330488922052e-01,
  -9.811945875364023e-01,   1.930211941452784e-01,
  -9.811575581483348e-01,   1.932093323015140e-01,
  -9.811204926859087e-01,   1.933974633539947e-01,
  -9.810833911504867e-01,   1.935855872958036e-01,
  -9.810462535434328e-01,   1.937737041200238e-01,
  -9.810090798661126e-01,   1.939618138197388e-01,
  -9.809718701198928e-01,   1.941499163880324e-01,
  -9.809346243061416e-01,   1.943380118179886e-01,
  -9.808973424262284e-01,   1.945261001026916e-01,
  -9.808600244815239e-01,   1.947141812352260e-01,
  -9.808226704734001e-01,   1.949022552086765e-01,
  -9.807852804032304e-01,   1.950903220161282e-01,
  -9.807478542723898e-01,   1.952783816506665e-01,
  -9.807103920822540e-01,   1.954664341053770e-01,
  -9.806728938342005e-01,   1.956544793733454e-01,
  -9.806353595296081e-01,   1.958425174476578e-01,
  -9.805977891698568e-01,   1.960305483214008e-01,
  -9.805601827563278e-01,   1.962185719876609e-01,
  -9.805225402904041e-01,   1.964065884395250e-01,
  -9.804848617734694e-01,   1.965945976700802e-01,
  -9.804471472069091e-01,   1.967825996724141e-01,
  -9.804093965921099e-01,   1.969705944396143e-01,
  -9.803716099304598e-01,   1.971585819647689e-01,
  -9.803337872233480e-01,   1.973465622409659e-01,
  -9.802959284721653e-01,   1.975345352612940e-01,
  -9.802580336783036e-01,   1.977225010188419e-01,
  -9.802201028431561e-01,   1.979104595066987e-01,
  -9.801821359681174e-01,   1.980984107179536e-01,
  -9.801441330545836e-01,   1.982863546456962e-01,
  -9.801060941039518e-01,   1.984742912830164e-01,
  -9.800680191176206e-01,   1.986622206230042e-01,
  -9.800299080969901e-01,   1.988501426587501e-01,
  -9.799917610434612e-01,   1.990380573833447e-01,
  -9.799535779584367e-01,   1.992259647898788e-01,
  -9.799153588433205e-01,   1.994138648714438e-01,
  -9.798771036995176e-01,   1.996017576211310e-01,
  -9.798388125284347e-01,   1.997896430320321e-01,
  -9.798004853314798e-01,   1.999775210972392e-01,
  -9.797621221100618e-01,   2.001653918098444e-01,
  -9.797237228655912e-01,   2.003532551629404e-01,
  -9.796852875994799e-01,   2.005411111496200e-01,
  -9.796468163131412e-01,   2.007289597629761e-01,
  -9.796083090079895e-01,   2.009168009961022e-01,
  -9.795697656854405e-01,   2.011046348420919e-01,
  -9.795311863469115e-01,   2.012924612940390e-01,
  -9.794925709938208e-01,   2.014802803450377e-01,
  -9.794539196275882e-01,   2.016680919881825e-01,
  -9.794152322496348e-01,   2.018558962165680e-01,
  -9.793765088613832e-01,   2.020436930232893e-01,
  -9.793377494642568e-01,   2.022314824014415e-01,
  -9.792989540596810e-01,   2.024192643441202e-01,
  -9.792601226490820e-01,   2.026070388444211e-01,
  -9.792212552338877e-01,   2.027948058954404e-01,
  -9.791823518155269e-01,   2.029825654902744e-01,
  -9.791434123954302e-01,   2.031703176220198e-01,
  -9.791044369750292e-01,   2.033580622837733e-01,
  -9.790654255557569e-01,   2.035457994686322e-01,
  -9.790263781390476e-01,   2.037335291696939e-01,
  -9.789872947263370e-01,   2.039212513800561e-01,
  -9.789481753190622e-01,   2.041089660928169e-01,
  -9.789090199186613e-01,   2.042966733010744e-01,
  -9.788698285265741e-01,   2.044843729979272e-01,
  -9.788306011442415e-01,   2.046720651764742e-01,
  -9.787913377731057e-01,   2.048597498298144e-01,
  -9.787520384146103e-01,   2.050474269510472e-01,
  -9.787127030702004e-01,   2.052350965332723e-01,
  -9.786733317413222e-01,   2.054227585695896e-01,
  -9.786339244294232e-01,   2.056104130530992e-01,
  -9.785944811359523e-01,   2.057980599769018e-01,
  -9.785550018623596e-01,   2.059856993340979e-01,
  -9.785154866100969e-01,   2.061733311177887e-01,
  -9.784759353806168e-01,   2.063609553210755e-01,
  -9.784363481753737e-01,   2.065485719370599e-01,
  -9.783967249958231e-01,   2.067361809588437e-01,
  -9.783570658434216e-01,   2.069237823795291e-01,
  -9.783173707196277e-01,   2.071113761922186e-01,
  -9.782776396259005e-01,   2.072989623900147e-01,
  -9.782378725637011e-01,   2.074865409660206e-01,
  -9.781980695344914e-01,   2.076741119133396e-01,
  -9.781582305397350e-01,   2.078616752250751e-01,
  -9.781183555808967e-01,   2.080492308943309e-01,
  -9.780784446594424e-01,   2.082367789142113e-01,
  -9.780384977768396e-01,   2.084243192778206e-01,
  -9.779985149345571e-01,   2.086118519782635e-01,
  -9.779584961340648e-01,   2.087993770086449e-01,
  -9.779184413768344e-01,   2.089868943620701e-01,
  -9.778783506643381e-01,   2.091744040316446e-01,
  -9.778382239980504e-01,   2.093619060104742e-01,
  -9.777980613794464e-01,   2.095494002916649e-01,
  -9.777578628100028e-01,   2.097368868683233e-01,
  -9.777176282911975e-01,   2.099243657335559e-01,
  -9.776773578245099e-01,   2.101118368804696e-01,
  -9.776370514114208e-01,   2.102993003021717e-01,
  -9.775967090534119e-01,   2.104867559917697e-01,
  -9.775563307519665e-01,   2.106742039423714e-01,
  -9.775159165085693e-01,   2.108616441470849e-01,
  -9.774754663247062e-01,   2.110490765990184e-01,
  -9.774349802018643e-01,   2.112365012912807e-01,
  -9.773944581415323e-01,   2.114239182169807e-01,
  -9.773539001452000e-01,   2.116113273692276e-01,
  -9.773133062143587e-01,   2.117987287411308e-01,
  -9.772726763505009e-01,   2.119861223258003e-01,
  -9.772320105551203e-01,   2.121735081163461e-01,
  -9.771913088297123e-01,   2.123608861058784e-01,
  -9.771505711757732e-01,   2.125482562875081e-01,
  -9.771097975948009e-01,   2.127356186543459e-01,
  -9.770689880882945e-01,   2.129229731995032e-01,
  -9.770281426577544e-01,   2.131103199160914e-01,
  -9.769872613046824e-01,   2.132976587972223e-01,
  -9.769463440305817e-01,   2.134849898360081e-01,
  -9.769053908369565e-01,   2.136723130255610e-01,
  -9.768644017253126e-01,   2.138596283589937e-01,
  -9.768233766971572e-01,   2.140469358294194e-01,
  -9.767823157539987e-01,   2.142342354299510e-01,
  -9.767412188973466e-01,   2.144215271537022e-01,
  -9.767000861287118e-01,   2.146088109937868e-01,
  -9.766589174496070e-01,   2.147960869433189e-01,
  -9.766177128615456e-01,   2.149833549954128e-01,
  -9.765764723660426e-01,   2.151706151431834e-01,
  -9.765351959646145e-01,   2.153578673797455e-01,
  -9.764938836587786e-01,   2.155451116982145e-01,
  -9.764525354500541e-01,   2.157323480917059e-01,
  -9.764111513399610e-01,   2.159195765533355e-01,
  -9.763697313300211e-01,   2.161067970762195e-01,
  -9.763282754217573e-01,   2.162940096534743e-01,
  -9.762867836166936e-01,   2.164812142782167e-01,
  -9.762452559163558e-01,   2.166684109435637e-01,
  -9.762036923222706e-01,   2.168555996426326e-01,
  -9.761620928359661e-01,   2.170427803685410e-01,
  -9.761204574589719e-01,   2.172299531144068e-01,
  -9.760787861928188e-01,   2.174171178733482e-01,
  -9.760370790390390e-01,   2.176042746384836e-01,
  -9.759953359991660e-01,   2.177914234029319e-01,
  -9.759535570747343e-01,   2.179785641598122e-01,
  -9.759117422672802e-01,   2.181656969022438e-01,
  -9.758698915783410e-01,   2.183528216233463e-01,
  -9.758280050094557e-01,   2.185399383162398e-01,
  -9.757860825621639e-01,   2.187270469740444e-01,
  -9.757441242380073e-01,   2.189141475898808e-01,
  -9.757021300385286e-01,   2.191012401568698e-01,
  -9.756600999652716e-01,   2.192883246681325e-01,
  -9.756180340197818e-01,   2.194754011167903e-01,
  -9.755759322036057e-01,   2.196624694959650e-01,
  -9.755337945182914e-01,   2.198495297987787e-01,
  -9.754916209653881e-01,   2.200365820183536e-01,
  -9.754494115464464e-01,   2.202236261478124e-01,
  -9.754071662630183e-01,   2.204106621802779e-01,
  -9.753648851166570e-01,   2.205976901088735e-01,
  -9.753225681089169e-01,   2.207847099267226e-01,
  -9.752802152413542e-01,   2.209717216269491e-01,
  -9.752378265155258e-01,   2.211587252026770e-01,
  -9.751954019329904e-01,   2.213457206470308e-01,
  -9.751529414953076e-01,   2.215327079531352e-01,
  -9.751104452040389e-01,   2.217196871141152e-01,
  -9.750679130607465e-01,   2.219066581230961e-01,
  -9.750253450669941e-01,   2.220936209732035e-01,
  -9.749827412243471e-01,   2.222805756575634e-01,
  -9.749401015343718e-01,   2.224675221693019e-01,
  -9.748974259986358e-01,   2.226544605015455e-01,
  -9.748547146187084e-01,   2.228413906474211e-01,
  -9.748119673961598e-01,   2.230283126000558e-01,
  -9.747691843325618e-01,   2.232152263525770e-01,
  -9.747263654294873e-01,   2.234021318981124e-01,
  -9.746835106885107e-01,   2.235890292297900e-01,
  -9.746406201112076e-01,   2.237759183407382e-01,
  -9.745976936991550e-01,   2.239627992240855e-01,
  -9.745547314539312e-01,   2.241496718729609e-01,
  -9.745117333771157e-01,   2.243365362804936e-01,
  -9.744686994702896e-01,   2.245233924398132e-01,
  -9.744256297350350e-01,   2.247102403440494e-01,
  -9.743825241729355e-01,   2.248970799863325e-01,
  -9.743393827855759e-01,   2.250839113597928e-01,
  -9.742962055745424e-01,   2.252707344575612e-01,
  -9.742529925414225e-01,   2.254575492727685e-01,
  -9.742097436878052e-01,   2.256443557985463e-01,
  -9.741664590152803e-01,   2.258311540280262e-01,
  -9.741231385254396e-01,   2.260179439543400e-01,
  -9.740797822198757e-01,   2.262047255706202e-01,
  -9.740363901001826e-01,   2.263914988699992e-01,
  -9.739929621679558e-01,   2.265782638456100e-01,
  -9.739494984247922e-01,   2.267650204905857e-01,
  -9.739059988722896e-01,   2.269517687980598e-01,
  -9.738624635120473e-01,   2.271385087611662e-01,
  -9.738188923456661e-01,   2.273252403730389e-01,
  -9.737752853747481e-01,   2.275119636268123e-01,
  -9.737316426008964e-01,   2.276986785156212e-01,
  -9.736879640257157e-01,   2.278853850326005e-01,
  -9.736442496508120e-01,   2.280720831708857e-01,
  -9.736004994777924e-01,   2.282587729236124e-01,
  -9.735567135082656e-01,   2.284454542839165e-01,
  -9.735128917438414e-01,   2.286321272449342e-01,
  -9.734690341861311e-01,   2.288187917998022e-01,
  -9.734251408367470e-01,   2.290054479416573e-01,
  -9.733812116973033e-01,   2.291920956636368e-01,
  -9.733372467694149e-01,   2.293787349588780e-01,
  -9.732932460546982e-01,   2.295653658205189e-01,
  -9.732492095547712e-01,   2.297519882416975e-01,
  -9.732051372712528e-01,   2.299386022155522e-01,
  -9.731610292057635e-01,   2.301252077352219e-01,
  -9.731168853599251e-01,   2.303118047938454e-01,
  -9.730727057353605e-01,   2.304983933845624e-01,
  -9.730284903336942e-01,   2.306849735005122e-01,
  -9.729842391565517e-01,   2.308715451348350e-01,
  -9.729399522055602e-01,   2.310581082806711e-01,
  -9.728956294823478e-01,   2.312446629311611e-01,
  -9.728512709885442e-01,   2.314312090794458e-01,
  -9.728068767257804e-01,   2.316177467186665e-01,
  -9.727624466956886e-01,   2.318042758419648e-01,
  -9.727179808999022e-01,   2.319907964424824e-01,
  -9.726734793400564e-01,   2.321773085133617e-01,
  -9.726289420177873e-01,   2.323638120477450e-01,
  -9.725843689347322e-01,   2.325503070387752e-01,
  -9.725397600925302e-01,   2.327367934795954e-01,
  -9.724951154928212e-01,   2.329232713633490e-01,
  -9.724504351372468e-01,   2.331097406831797e-01,
  -9.724057190274498e-01,   2.332962014322316e-01,
  -9.723609671650741e-01,   2.334826536036491e-01,
  -9.723161795517653e-01,   2.336690971905768e-01,
  -9.722713561891700e-01,   2.338555321861598e-01,
  -9.722264970789363e-01,   2.340419585835434e-01,
  -9.721816022227134e-01,   2.342283763758732e-01,
  -9.721366716221522e-01,   2.344147855562952e-01,
  -9.720917052789044e-01,   2.346011861179556e-01,
  -9.720467031946235e-01,   2.347875780540010e-01,
  -9.720016653709639e-01,   2.349739613575783e-01,
  -9.719565918095817e-01,   2.351603360218347e-01,
  -9.719114825121340e-01,   2.353467020399178e-01,
  -9.718663374802794e-01,   2.355330594049755e-01,
  -9.718211567156777e-01,   2.357194081101558e-01,
  -9.717759402199901e-01,   2.359057481486074e-01,
  -9.717306879948792e-01,   2.360920795134789e-01,
  -9.716854000420085e-01,   2.362784021979196e-01,
  -9.716400763630434e-01,   2.364647161950788e-01,
  -9.715947169596502e-01,   2.366510214981064e-01,
  -9.715493218334966e-01,   2.368373181001524e-01,
  -9.715038909862518e-01,   2.370236059943672e-01,
  -9.714584244195860e-01,   2.372098851739016e-01,
  -9.714129221351709e-01,   2.373961556319066e-01,
  -9.713673841346795e-01,   2.375824173615336e-01,
  -9.713218104197862e-01,   2.377686703559342e-01,
  -9.712762009921665e-01,   2.379549146082605e-01,
  -9.712305558534974e-01,   2.381411501116648e-01,
  -9.711848750054570e-01,   2.383273768592998e-01,
  -9.711391584497251e-01,   2.385135948443184e-01,
  -9.710934061879825e-01,   2.386998040598740e-01,
  -9.710476182219111e-01,   2.388860044991200e-01,
  -9.710017945531947e-01,   2.390721961552106e-01,
  -9.709559351835180e-01,   2.392583790213000e-01,
  -9.709100401145670e-01,   2.394445530905426e-01,
  -9.708641093480295e-01,   2.396307183560936e-01,
  -9.708181428855939e-01,   2.398168748111080e-01,
  -9.707721407289504e-01,   2.400030224487415e-01,
  -9.707261028797901e-01,   2.401891612621499e-01,
  -9.706800293398061e-01,   2.403752912444894e-01,
  -9.706339201106922e-01,   2.405614123889167e-01,
  -9.705877751941436e-01,   2.407475246885884e-01,
  -9.705415945918571e-01,   2.409336281366619e-01,
  -9.704953783055306e-01,   2.411197227262946e-01,
  -9.704491263368631e-01,   2.413058084506444e-01,
  -9.704028386875555e-01,   2.414918853028693e-01,
  -9.703565153593094e-01,   2.416779532761280e-01,
  -9.703101563538281e-01,   2.418640123635792e-01,
  -9.702637616728161e-01,   2.420500625583821e-01,
  -9.702173313179792e-01,   2.422361038536960e-01,
  -9.701708652910245e-01,   2.424221362426809e-01,
  -9.701243635936603e-01,   2.426081597184968e-01,
  -9.700778262275964e-01,   2.427941742743042e-01,
  -9.700312531945440e-01,   2.429801799032639e-01,
  -9.699846444962152e-01,   2.431661765985369e-01,
  -9.699380001343240e-01,   2.433521643532847e-01,
  -9.698913201105851e-01,   2.435381431606691e-01,
  -9.698446044267148e-01,   2.437241130138522e-01,
  -9.697978530844309e-01,   2.439100739059963e-01,
  -9.697510660854521e-01,   2.440960258302642e-01,
  -9.697042434314989e-01,   2.442819687798190e-01,
  -9.696573851242924e-01,   2.444679027478242e-01,
  -9.696104911655559e-01,   2.446538277274433e-01,
  -9.695635615570132e-01,   2.448397437118407e-01,
  -9.695165963003900e-01,   2.450256506941805e-01,
  -9.694695953974131e-01,   2.452115486676275e-01,
  -9.694225588498103e-01,   2.453974376253470e-01,
  -9.693754866593113e-01,   2.455833175605041e-01,
  -9.693283788276467e-01,   2.457691884662646e-01,
  -9.692812353565485e-01,   2.459550503357946e-01,
  -9.692340562477500e-01,   2.461409031622605e-01,
  -9.691868415029860e-01,   2.463267469388290e-01,
  -9.691395911239923e-01,   2.465125816586672e-01,
  -9.690923051125062e-01,   2.466984073149424e-01,
  -9.690449834702662e-01,   2.468842239008224e-01,
  -9.689976261990124e-01,   2.470700314094753e-01,
  -9.689502333004858e-01,   2.472558298340693e-01,
  -9.689028047764289e-01,   2.474416191677733e-01,
  -9.688553406285856e-01,   2.476273994037563e-01,
  -9.688078408587010e-01,   2.478131705351877e-01,
  -9.687603054685214e-01,   2.479989325552371e-01,
  -9.687127344597948e-01,   2.481846854570748e-01,
  -9.686651278342701e-01,   2.483704292338710e-01,
  -9.686174855936975e-01,   2.485561638787966e-01,
  -9.685698077398289e-01,   2.487418893850225e-01,
  -9.685220942744174e-01,   2.489276057457201e-01,
  -9.684743451992168e-01,   2.491133129540614e-01,
  -9.684265605159832e-01,   2.492990110032182e-01,
  -9.683787402264733e-01,   2.494846998863630e-01,
  -9.683308843324452e-01,   2.496703795966686e-01,
  -9.682829928356587e-01,   2.498560501273080e-01,
  -9.682350657378743e-01,   2.500417114714547e-01,
  -9.681871030408544e-01,   2.502273636222824e-01,
  -9.681391047463624e-01,   2.504130065729652e-01,
  -9.680910708561630e-01,   2.505986403166777e-01,
  -9.680430013720223e-01,   2.507842648465945e-01,
  -9.679948962957077e-01,   2.509698801558907e-01,
  -9.679467556289878e-01,   2.511554862377419e-01,
  -9.678985793736327e-01,   2.513410830853239e-01,
  -9.678503675314136e-01,   2.515266706918126e-01,
  -9.678021201041033e-01,   2.517122490503847e-01,
  -9.677538370934755e-01,   2.518978181542170e-01,
  -9.677055185013055e-01,   2.520833779964864e-01,
  -9.676571643293699e-01,   2.522689285703708e-01,
  -9.676087745794465e-01,   2.524544698690477e-01,
  -9.675603492533144e-01,   2.526400018856955e-01,
  -9.675118883527541e-01,   2.528255246134926e-01,
  -9.674633918795476e-01,   2.530110380456179e-01,
  -9.674148598354775e-01,   2.531965421752506e-01,
  -9.673662922223285e-01,   2.533820369955702e-01,
  -9.673176890418863e-01,   2.535675224997566e-01,
  -9.672690502959378e-01,   2.537529986809900e-01,
  -9.672203759862714e-01,   2.539384655324511e-01,
  -9.671716661146766e-01,   2.541239230473206e-01,
  -9.671229206829444e-01,   2.543093712187800e-01,
  -9.670741396928670e-01,   2.544948100400107e-01,
  -9.670253231462380e-01,   2.546802395041948e-01,
  -9.669764710448521e-01,   2.548656596045146e-01,
  -9.669275833905057e-01,   2.550510703341525e-01,
  -9.668786601849959e-01,   2.552364716862917e-01,
  -9.668297014301218e-01,   2.554218636541155e-01,
  -9.667807071276833e-01,   2.556072462308074e-01,
  -9.667316772794818e-01,   2.557926194095516e-01,
  -9.666826118873201e-01,   2.559779831835324e-01,
  -9.666335109530021e-01,   2.561633375459345e-01,
  -9.665843744783331e-01,   2.563486824899429e-01,
  -9.665352024651197e-01,   2.565340180087430e-01,
  -9.664859949151698e-01,   2.567193440955207e-01,
  -9.664367518302927e-01,   2.569046607434619e-01,
  -9.663874732122989e-01,   2.570899679457531e-01,
  -9.663381590630001e-01,   2.572752656955811e-01,
  -9.662888093842097e-01,   2.574605539861331e-01,
  -9.662394241777419e-01,   2.576458328105964e-01,
  -9.661900034454125e-01,   2.578311021621590e-01,
  -9.661405471890387e-01,   2.580163620340090e-01,
  -9.660910554104388e-01,   2.582016124193349e-01,
  -9.660415281114324e-01,   2.583868533113256e-01,
  -9.659919652938406e-01,   2.585720847031703e-01,
  -9.659423669594855e-01,   2.587573065880587e-01,
  -9.658927331101909e-01,   2.589425189591805e-01,
  -9.658430637477815e-01,   2.591277218097262e-01,
  -9.657933588740837e-01,   2.593129151328862e-01,
  -9.657436184909248e-01,   2.594980989218517e-01,
  -9.656938426001337e-01,   2.596832731698138e-01,
  -9.656440312035406e-01,   2.598684378699643e-01,
  -9.655941843029768e-01,   2.600535930154952e-01,
  -9.655443019002752e-01,   2.602387385995988e-01,
  -9.654943839972695e-01,   2.604238746154680e-01,
  -9.654444305957954e-01,   2.606090010562958e-01,
  -9.653944416976894e-01,   2.607941179152755e-01,
  -9.653444173047894e-01,   2.609792251856011e-01,
  -9.652943574189347e-01,   2.611643228604665e-01,
  -9.652442620419658e-01,   2.613494109330664e-01,
  -9.651941311757247e-01,   2.615344893965955e-01,
  -9.651439648220544e-01,   2.617195582442490e-01,
  -9.650937629827996e-01,   2.619046174692226e-01,
  -9.650435256598059e-01,   2.620896670647120e-01,
  -9.649932528549203e-01,   2.622747070239136e-01,
  -9.649429445699914e-01,   2.624597373400240e-01,
  -9.648926008068689e-01,   2.626447580062400e-01,
  -9.648422215674036e-01,   2.628297690157592e-01,
  -9.647918068534479e-01,   2.630147703617790e-01,
  -9.647413566668553e-01,   2.631997620374976e-01,
  -9.646908710094810e-01,   2.633847440361133e-01,
  -9.646403498831809e-01,   2.635697163508249e-01,
  -9.645897932898128e-01,   2.637546789748313e-01,
  -9.645392012312352e-01,   2.639396319013323e-01,
  -9.644885737093084e-01,   2.641245751235275e-01,
  -9.644379107258939e-01,   2.643095086346171e-01,
  -9.643872122828543e-01,   2.644944324278016e-01,
  -9.643364783820537e-01,   2.646793464962819e-01,
  -9.642857090253575e-01,   2.648642508332593e-01,
  -9.642349042146322e-01,   2.650491454319353e-01,
  -9.641840639517458e-01,   2.652340302855118e-01,
  -9.641331882385676e-01,   2.654189053871913e-01,
  -9.640822770769681e-01,   2.656037707301763e-01,
  -9.640313304688193e-01,   2.657886263076699e-01,
  -9.639803484159941e-01,   2.659734721128756e-01,
  -9.639293309203671e-01,   2.661583081389970e-01,
  -9.638782779838142e-01,   2.663431343792382e-01,
  -9.638271896082123e-01,   2.665279508268037e-01,
  -9.637760657954398e-01,   2.667127574748984e-01,
  -9.637249065473765e-01,   2.668975543167273e-01,
  -9.636737118659032e-01,   2.670823413454962e-01,
  -9.636224817529022e-01,   2.672671185544109e-01,
  -9.635712162102573e-01,   2.674518859366776e-01,
  -9.635199152398531e-01,   2.676366434855031e-01,
  -9.634685788435760e-01,   2.678213911940941e-01,
  -9.634172070233133e-01,   2.680061290556583e-01,
  -9.633657997809540e-01,   2.681908570634032e-01,
  -9.633143571183882e-01,   2.683755752105369e-01,
  -9.632628790375071e-01,   2.685602834902679e-01,
  -9.632113655402035e-01,   2.687449818958050e-01,
  -9.631598166283714e-01,   2.689296704203573e-01,
  -9.631082323039062e-01,   2.691143490571344e-01,
  -9.630566125687043e-01,   2.692990177993461e-01,
  -9.630049574246639e-01,   2.694836766402028e-01,
  -9.629532668736839e-01,   2.696683255729151e-01,
  -9.629015409176650e-01,   2.698529645906939e-01,
  -9.628497795585090e-01,   2.700375936867506e-01,
  -9.627979827981190e-01,   2.702222128542969e-01,
  -9.627461506383994e-01,   2.704068220865448e-01,
  -9.626942830812559e-01,   2.705914213767069e-01,
  -9.626423801285957e-01,   2.707760107179960e-01,
  -9.625904417823269e-01,   2.709605901036252e-01,
  -9.625384680443592e-01,   2.711451595268080e-01,
  -9.624864589166034e-01,   2.713297189807584e-01,
  -9.624344144009721e-01,   2.715142684586907e-01,
  -9.623823344993784e-01,   2.716988079538195e-01,
  -9.623302192137374e-01,   2.718833374593597e-01,
  -9.622780685459651e-01,   2.720678569685269e-01,
  -9.622258824979790e-01,   2.722523664745367e-01,
  -9.621736610716979e-01,   2.724368659706052e-01,
  -9.621214042690416e-01,   2.726213554499490e-01,
  -9.620691120919316e-01,   2.728058349057848e-01,
  -9.620167845422906e-01,   2.729903043313299e-01,
  -9.619644216220423e-01,   2.731747637198019e-01,
  -9.619120233331122e-01,   2.733592130644187e-01,
  -9.618595896774266e-01,   2.735436523583987e-01,
  -9.618071206569135e-01,   2.737280815949605e-01,
  -9.617546162735020e-01,   2.739125007673233e-01,
  -9.617020765291225e-01,   2.740969098687064e-01,
  -9.616495014257068e-01,   2.742813088923297e-01,
  -9.615968909651879e-01,   2.744656978314132e-01,
  -9.615442451495000e-01,   2.746500766791777e-01,
  -9.614915639805790e-01,   2.748344454288439e-01,
  -9.614388474603617e-01,   2.750188040736332e-01,
  -9.613860955907862e-01,   2.752031526067673e-01,
  -9.613333083737923e-01,   2.753874910214681e-01,
  -9.612804858113206e-01,   2.755718193109581e-01,
  -9.612276279053135e-01,   2.757561374684601e-01,
  -9.611747346577141e-01,   2.759404454871972e-01,
  -9.611218060704674e-01,   2.761247433603928e-01,
  -9.610688421455194e-01,   2.763090310812711e-01,
  -9.610158428848172e-01,   2.764933086430560e-01,
  -9.609628082903098e-01,   2.766775760389724e-01,
  -9.609097383639468e-01,   2.768618332622453e-01,
  -9.608566331076797e-01,   2.770460803060999e-01,
  -9.608034925234608e-01,   2.772303171637622e-01,
  -9.607503166132440e-01,   2.774145438284581e-01,
  -9.606971053789845e-01,   2.775987602934143e-01,
  -9.606438588226386e-01,   2.777829665518577e-01,
  -9.605905769461641e-01,   2.779671625970154e-01,
  -9.605372597515200e-01,   2.781513484221151e-01,
  -9.604839072406668e-01,   2.783355240203849e-01,
  -9.604305194155658e-01,   2.785196893850531e-01,
  -9.603770962781801e-01,   2.787038445093485e-01,
  -9.603236378304739e-01,   2.788879893865003e-01,
  -9.602701440744128e-01,   2.790721240097378e-01,
  -9.602166150119634e-01,   2.792562483722912e-01,
  -9.601630506450940e-01,   2.794403624673905e-01,
  -9.601094509757739e-01,   2.796244662882666e-01,
  -9.600558160059739e-01,   2.798085598281504e-01,
  -9.600021457376660e-01,   2.799926430802732e-01,
  -9.599484401728232e-01,   2.801767160378670e-01,
  -9.598946993134205e-01,   2.803607786941638e-01,
  -9.598409231614338e-01,   2.805448310423962e-01,
  -9.597871117188399e-01,   2.807288730757972e-01,
  -9.597332649876177e-01,   2.809129047876000e-01,
  -9.596793829697468e-01,   2.810969261710383e-01,
  -9.596254656672082e-01,   2.812809372193461e-01,
  -9.595715130819845e-01,   2.814649379257579e-01,
  -9.595175252160593e-01,   2.816489282835086e-01,
  -9.594635020714175e-01,   2.818329082858334e-01,
  -9.594094436500455e-01,   2.820168779259676e-01,
  -9.593553499539308e-01,   2.822008371971476e-01,
  -9.593012209850622e-01,   2.823847860926094e-01,
  -9.592470567454301e-01,   2.825687246055897e-01,
  -9.591928572370257e-01,   2.827526527293259e-01,
  -9.591386224618419e-01,   2.829365704570554e-01,
  -9.590843524218727e-01,   2.831204777820158e-01,
  -9.590300471191137e-01,   2.833043746974457e-01,
  -9.589757065555611e-01,   2.834882611965835e-01,
  -9.589213307332132e-01,   2.836721372726684e-01,
  -9.588669196540690e-01,   2.838560029189398e-01,
  -9.588124733201293e-01,   2.840398581286372e-01,
  -9.587579917333957e-01,   2.842237028950010e-01,
  -9.587034748958716e-01,   2.844075372112719e-01,
  -9.586489228095612e-01,   2.845913610706904e-01,
  -9.585943354764702e-01,   2.847751744664983e-01,
  -9.585397128986057e-01,   2.849589773919370e-01,
  -9.584850550779761e-01,   2.851427698402487e-01,
  -9.584303620165909e-01,   2.853265518046759e-01,
  -9.583756337164612e-01,   2.855103232784613e-01,
  -9.583208701795989e-01,   2.856940842548483e-01,
  -9.582660714080177e-01,   2.858778347270806e-01,
  -9.582112374037323e-01,   2.860615746884020e-01,
  -9.581563681687588e-01,   2.862453041320571e-01,
  -9.581014637051147e-01,   2.864290230512907e-01,
  -9.580465240148186e-01,   2.866127314393478e-01,
  -9.579915490998904e-01,   2.867964292894741e-01,
  -9.579365389623514e-01,   2.869801165949156e-01,
  -9.578814936042244e-01,   2.871637933489184e-01,
  -9.578264130275329e-01,   2.873474595447295e-01,
  -9.577712972343023e-01,   2.875311151755959e-01,
  -9.577161462265589e-01,   2.877147602347652e-01,
  -9.576609600063306e-01,   2.878983947154852e-01,
  -9.576057385756463e-01,   2.880820186110041e-01,
  -9.575504819365365e-01,   2.882656319145708e-01,
  -9.574951900910326e-01,   2.884492346194342e-01,
  -9.574398630411677e-01,   2.886328267188438e-01,
  -9.573845007889759e-01,   2.888164082060495e-01,
  -9.573291033364928e-01,   2.889999790743014e-01,
  -9.572736706857552e-01,   2.891835393168502e-01,
  -9.572182028388012e-01,   2.893670889269470e-01,
  -9.571626997976702e-01,   2.895506278978430e-01,
  -9.571071615644028e-01,   2.897341562227903e-01,
  -9.570515881410410e-01,   2.899176738950408e-01,
  -9.569959795296282e-01,   2.901011809078471e-01,
  -9.569403357322088e-01,   2.902846772544623e-01,
  -9.568846567508289e-01,   2.904681629281398e-01,
  -9.568289425875354e-01,   2.906516379221332e-01,
  -9.567731932443769e-01,   2.908351022296968e-01,
  -9.567174087234030e-01,   2.910185558440851e-01,
  -9.566615890266651e-01,   2.912019987585529e-01,
  -9.566057341562151e-01,   2.913854309663557e-01,
  -9.565498441141068e-01,   2.915688524607490e-01,
  -9.564939189023951e-01,   2.917522632349893e-01,
  -9.564379585231362e-01,   2.919356632823328e-01,
  -9.563819629783877e-01,   2.921190525960364e-01,
  -9.563259322702082e-01,   2.923024311693576e-01,
  -9.562698664006580e-01,   2.924857989955539e-01,
  -9.562137653717985e-01,   2.926691560678835e-01,
  -9.561576291856921e-01,   2.928525023796048e-01,
  -9.561014578444030e-01,   2.930358379239768e-01,
  -9.560452513499964e-01,   2.932191626942586e-01,
  -9.559890097045389e-01,   2.934024766837101e-01,
  -9.559327329100983e-01,   2.935857798855912e-01,
  -9.558764209687436e-01,   2.937690722931624e-01,
  -9.558200738825454e-01,   2.939523538996847e-01,
  -9.557636916535754e-01,   2.941356246984190e-01,
  -9.557072742839066e-01,   2.943188846826274e-01,
  -9.556508217756133e-01,   2.945021338455717e-01,
  -9.555943341307711e-01,   2.946853721805143e-01,
  -9.555378113514569e-01,   2.948685996807183e-01,
  -9.554812534397488e-01,   2.950518163394467e-01,
  -9.554246603977263e-01,   2.952350221499632e-01,
  -9.553680322274704e-01,   2.954182171055320e-01,
  -9.553113689310627e-01,   2.956014011994174e-01,
  -9.552546705105870e-01,   2.957845744248843e-01,
  -9.551979369681277e-01,   2.959677367751979e-01,
  -9.551411683057708e-01,   2.961508882436238e-01,
  -9.550843645256034e-01,   2.963340288234282e-01,
  -9.550275256297142e-01,   2.965171585078775e-01,
  -9.549706516201928e-01,   2.967002772902383e-01,
  -9.549137424991305e-01,   2.968833851637783e-01,
  -9.548567982686196e-01,   2.970664821217647e-01,
  -9.547998189307537e-01,   2.972495681574658e-01,
  -9.547428044876279e-01,   2.974326432641500e-01,
  -9.546857549413383e-01,   2.976157074350862e-01,
  -9.546286702939827e-01,   2.977987606635435e-01,
  -9.545715505476596e-01,   2.979818029427918e-01,
  -9.545143957044695e-01,   2.981648342661009e-01,
  -9.544572057665135e-01,   2.983478546267414e-01,
  -9.543999807358945e-01,   2.985308640179841e-01,
  -9.543427206147165e-01,   2.987138624331003e-01,
  -9.542854254050847e-01,   2.988968498653618e-01,
  -9.542280951091057e-01,   2.990798263080405e-01,
  -9.541707297288873e-01,   2.992627917544088e-01,
  -9.541133292665388e-01,   2.994457461977399e-01,
  -9.540558937241707e-01,   2.996286896313068e-01,
  -9.539984231038945e-01,   2.998116220483834e-01,
  -9.539409174078235e-01,   2.999945434422436e-01,
  -9.538833766380718e-01,   3.001774538061620e-01,
  -9.538258007967550e-01,   3.003603531334135e-01,
  -9.537681898859903e-01,   3.005432414172735e-01,
  -9.537105439078957e-01,   3.007261186510175e-01,
  -9.536528628645905e-01,   3.009089848279219e-01,
  -9.535951467581957e-01,   3.010918399412631e-01,
  -9.535373955908333e-01,   3.012746839843179e-01,
  -9.534796093646266e-01,   3.014575169503639e-01,
  -9.534217880817003e-01,   3.016403388326788e-01,
  -9.533639317441803e-01,   3.018231496245407e-01,
  -9.533060403541939e-01,   3.020059493192281e-01,
  -9.532481139138693e-01,   3.021887379100200e-01,
  -9.531901524253367e-01,   3.023715153901960e-01,
  -9.531321558907268e-01,   3.025542817530356e-01,
  -9.530741243121722e-01,   3.027370369918191e-01,
  -9.530160576918065e-01,   3.029197810998273e-01,
  -9.529579560317647e-01,   3.031025140703411e-01,
  -9.528998193341829e-01,   3.032852358966417e-01,
  -9.528416476011987e-01,   3.034679465720113e-01,
  -9.527834408349509e-01,   3.036506460897319e-01,
  -9.527251990375796e-01,   3.038333344430864e-01,
  -9.526669222112262e-01,   3.040160116253576e-01,
  -9.526086103580333e-01,   3.041986776298291e-01,
  -9.525502634801449e-01,   3.043813324497849e-01,
  -9.524918815797063e-01,   3.045639760785091e-01,
  -9.524334646588640e-01,   3.047466085092865e-01,
  -9.523750127197659e-01,   3.049292297354024e-01,
  -9.523165257645609e-01,   3.051118397501421e-01,
  -9.522580037953996e-01,   3.052944385467917e-01,
  -9.521994468144336e-01,   3.054770261186374e-01,
  -9.521408548238158e-01,   3.056596024589661e-01,
  -9.520822278257006e-01,   3.058421675610651e-01,
  -9.520235658222436e-01,   3.060247214182218e-01,
  -9.519648688156014e-01,   3.062072640237242e-01,
  -9.519061368079323e-01,   3.063897953708609e-01,
  -9.518473698013956e-01,   3.065723154529207e-01,
  -9.517885677981521e-01,   3.067548242631928e-01,
  -9.517297308003638e-01,   3.069373217949669e-01,
  -9.516708588101939e-01,   3.071198080415331e-01,
  -9.516119518298068e-01,   3.073022829961818e-01,
  -9.515530098613686e-01,   3.074847466522041e-01,
  -9.514940329070464e-01,   3.076671990028912e-01,
  -9.514350209690083e-01,   3.078496400415349e-01,
  -9.513759740494244e-01,   3.080320697614273e-01,
  -9.513168921504656e-01,   3.082144881558611e-01,
  -9.512577752743040e-01,   3.083968952181292e-01,
  -9.511986234231132e-01,   3.085792909415251e-01,
  -9.511394365990682e-01,   3.087616753193425e-01,
  -9.510802148043450e-01,   3.089440483448757e-01,
  -9.510209580411211e-01,   3.091264100114194e-01,
  -9.509616663115751e-01,   3.093087603122687e-01,
  -9.509023396178871e-01,   3.094910992407191e-01,
  -9.508429779622382e-01,   3.096734267900664e-01,
  -9.507835813468111e-01,   3.098557429536071e-01,
  -9.507241497737896e-01,   3.100380477246379e-01,
  -9.506646832453589e-01,   3.102203410964559e-01,
  -9.506051817637053e-01,   3.104026230623587e-01,
  -9.505456453310166e-01,   3.105848936156445e-01,
  -9.504860739494817e-01,   3.107671527496115e-01,
  -9.504264676212909e-01,   3.109494004575586e-01,
  -9.503668263486358e-01,   3.111316367327853e-01,
  -9.503071501337093e-01,   3.113138615685909e-01,
  -9.502474389787052e-01,   3.114960749582759e-01,
  -9.501876928858193e-01,   3.116782768951405e-01,
  -9.501279118572481e-01,   3.118604673724860e-01,
  -9.500680958951896e-01,   3.120426463836135e-01,
  -9.500082450018430e-01,   3.122248139218249e-01,
  -9.499483591794090e-01,   3.124069699804224e-01,
  -9.498884384300893e-01,   3.125891145527087e-01,
  -9.498284827560871e-01,   3.127712476319868e-01,
  -9.497684921596067e-01,   3.129533692115602e-01,
  -9.497084666428538e-01,   3.131354792847328e-01,
  -9.496484062080355e-01,   3.133175778448090e-01,
  -9.495883108573600e-01,   3.134996648850935e-01,
  -9.495281805930367e-01,   3.136817403988915e-01,
  -9.494680154172765e-01,   3.138638043795085e-01,
  -9.494078153322916e-01,   3.140458568202507e-01,
  -9.493475803402952e-01,   3.142278977144244e-01,
  -9.492873104435021e-01,   3.144099270553367e-01,
  -9.492270056441282e-01,   3.145919448362947e-01,
  -9.491666659443907e-01,   3.147739510506061e-01,
  -9.491062913465083e-01,   3.149559456915791e-01,
  -9.490458818527006e-01,   3.151379287525224e-01,
  -9.489854374651887e-01,   3.153199002267449e-01,
  -9.489249581861952e-01,   3.155018601075560e-01,
  -9.488644440179433e-01,   3.156838083882657e-01,
  -9.488038949626585e-01,   3.158657450621840e-01,
  -9.487433110225665e-01,   3.160476701226219e-01,
  -9.486826921998951e-01,   3.162295835628903e-01,
  -9.486220384968730e-01,   3.164114853763010e-01,
  -9.485613499157303e-01,   3.165933755561658e-01,
  -9.485006264586983e-01,   3.167752540957973e-01,
  -9.484398681280096e-01,   3.169571209885081e-01,
  -9.483790749258981e-01,   3.171389762276118e-01,
  -9.483182468545991e-01,   3.173208198064217e-01,
  -9.482573839163491e-01,   3.175026517182523e-01,
  -9.481964861133856e-01,   3.176844719564180e-01,
  -9.481355534479480e-01,   3.178662805142337e-01,
  -9.480745859222762e-01,   3.180480773850149e-01,
  -9.480135835386122e-01,   3.182298625620775e-01,
  -9.479525462991987e-01,   3.184116360387378e-01,
  -9.478914742062798e-01,   3.185933978083124e-01,
  -9.478303672621010e-01,   3.187751478641185e-01,
  -9.477692254689092e-01,   3.189568861994737e-01,
  -9.477080488289521e-01,   3.191386128076959e-01,
  -9.476468373444793e-01,   3.193203276821036e-01,
  -9.475855910177411e-01,   3.195020308160157e-01,
  -9.475243098509896e-01,   3.196837222027514e-01,
  -9.474629938464777e-01,   3.198654018356305e-01,
  -9.474016430064599e-01,   3.200470697079731e-01,
  -9.473402573331920e-01,   3.202287258130999e-01,
  -9.472788368289309e-01,   3.204103701443318e-01,
  -9.472173814959348e-01,   3.205920026949903e-01,
  -9.471558913364633e-01,   3.207736234583973e-01,
  -9.470943663527772e-01,   3.209552324278752e-01,
  -9.470328065471386e-01,   3.211368295967467e-01,
  -9.469712119218109e-01,   3.213184149583349e-01,
  -9.469095824790588e-01,   3.214999885059635e-01,
  -9.468479182211480e-01,   3.216815502329566e-01,
  -9.467862191503460e-01,   3.218631001326386e-01,
  -9.467244852689212e-01,   3.220446381983345e-01,
  -9.466627165791434e-01,   3.222261644233696e-01,
  -9.466009130832835e-01,   3.224076788010699e-01,
  -9.465390747836141e-01,   3.225891813247613e-01,
  -9.464772016824087e-01,   3.227706719877707e-01,
  -9.464152937819421e-01,   3.229521507834253e-01,
  -9.463533510844906e-01,   3.231336177050523e-01,
  -9.462913735923316e-01,   3.233150727459800e-01,
  -9.462293613077438e-01,   3.234965158995367e-01,
  -9.461673142330074e-01,   3.236779471590512e-01,
  -9.461052323704034e-01,   3.238593665178529e-01,
  -9.460431157222146e-01,   3.240407739692714e-01,
  -9.459809642907248e-01,   3.242221695066370e-01,
  -9.459187780782191e-01,   3.244035531232802e-01,
  -9.458565570869839e-01,   3.245849248125321e-01,
  -9.457943013193070e-01,   3.247662845677242e-01,
  -9.457320107774772e-01,   3.249476323821884e-01,
  -9.456696854637847e-01,   3.251289682492571e-01,
  -9.456073253805213e-01,   3.253102921622629e-01,
  -9.455449305299797e-01,   3.254916041145393e-01,
  -9.454825009144537e-01,   3.256729040994198e-01,
  -9.454200365362391e-01,   3.258541921102386e-01,
  -9.453575373976323e-01,   3.260354681403302e-01,
  -9.452950035009312e-01,   3.262167321830297e-01,
  -9.452324348484350e-01,   3.263979842316725e-01,
  -9.451698314424442e-01,   3.265792242795944e-01,
  -9.451071932852606e-01,   3.267604523201317e-01,
  -9.450445203791871e-01,   3.269416683466214e-01,
  -9.449818127265281e-01,   3.271228723524005e-01,
  -9.449190703295892e-01,   3.273040643308067e-01,
  -9.448562931906772e-01,   3.274852442751780e-01,
  -9.447934813121003e-01,   3.276664121788531e-01,
  -9.447306346961678e-01,   3.278475680351708e-01,
  -9.446677533451905e-01,   3.280287118374707e-01,
  -9.446048372614803e-01,   3.282098435790925e-01,
  -9.445418864473505e-01,   3.283909632533766e-01,
  -9.444789009051155e-01,   3.285720708536637e-01,
  -9.444158806370913e-01,   3.287531663732950e-01,
  -9.443528256455948e-01,   3.289342498056122e-01,
  -9.442897359329444e-01,   3.291153211439573e-01,
  -9.442266115014598e-01,   3.292963803816727e-01,
  -9.441634523534618e-01,   3.294774275121017e-01,
  -9.441002584912727e-01,   3.296584625285875e-01,
  -9.440370299172158e-01,   3.298394854244739e-01,
  -9.439737666336160e-01,   3.300204961931054e-01,
  -9.439104686427991e-01,   3.302014948278266e-01,
  -9.438471359470927e-01,   3.303824813219828e-01,
  -9.437837685488251e-01,   3.305634556689195e-01,
  -9.437203664503262e-01,   3.307444178619829e-01,
  -9.436569296539272e-01,   3.309253678945195e-01,
  -9.435934581619604e-01,   3.311063057598764e-01,
  -9.435299519767595e-01,   3.312872314514008e-01,
  -9.434664111006593e-01,   3.314681449624409e-01,
  -9.434028355359962e-01,   3.316490462863447e-01,
  -9.433392252851077e-01,   3.318299354164611e-01,
  -9.432755803503325e-01,   3.320108123461394e-01,
  -9.432119007340106e-01,   3.321916770687292e-01,
  -9.431481864384834e-01,   3.323725295775806e-01,
  -9.430844374660935e-01,   3.325533698660442e-01,
  -9.430206538191847e-01,   3.327341979274711e-01,
  -9.429568355001021e-01,   3.329150137552127e-01,
  -9.428929825111922e-01,   3.330958173426208e-01,
  -9.428290948548027e-01,   3.332766086830479e-01,
  -9.427651725332825e-01,   3.334573877698468e-01,
  -9.427012155489819e-01,   3.336381545963709e-01,
  -9.426372239042525e-01,   3.338189091559736e-01,
  -9.425731976014469e-01,   3.339996514420094e-01,
  -9.425091366429192e-01,   3.341803814478327e-01,
  -9.424450410310249e-01,   3.343610991667987e-01,
  -9.423809107681205e-01,   3.345418045922629e-01,
  -9.423167458565638e-01,   3.347224977175812e-01,
  -9.422525462987140e-01,   3.349031785361102e-01,
  -9.421883120969318e-01,   3.350838470412066e-01,
  -9.421240432535786e-01,   3.352645032262278e-01,
  -9.420597397710173e-01,   3.354451470845316e-01,
  -9.419954016516126e-01,   3.356257786094763e-01,
  -9.419310288977296e-01,   3.358063977944205e-01,
  -9.418666215117353e-01,   3.359870046327234e-01,
  -9.418021794959976e-01,   3.361675991177445e-01,
  -9.417377028528862e-01,   3.363481812428440e-01,
  -9.416731915847714e-01,   3.365287510013824e-01,
  -9.416086456940252e-01,   3.367093083867206e-01,
  -9.415440651830208e-01,   3.368898533922201e-01,
  -9.414794500541326e-01,   3.370703860112426e-01,
  -9.414148003097363e-01,   3.372509062371506e-01,
  -9.413501159522090e-01,   3.374314140633068e-01,
  -9.412853969839287e-01,   3.376119094830746e-01,
  -9.412206434072752e-01,   3.377923924898175e-01,
  -9.411558552246292e-01,   3.379728630768997e-01,
  -9.410910324383728e-01,   3.381533212376859e-01,
  -9.410261750508893e-01,   3.383337669655411e-01,
  -9.409612830645633e-01,   3.385142002538309e-01,
  -9.408963564817808e-01,   3.386946210959212e-01,
  -9.408313953049289e-01,   3.388750294851784e-01,
  -9.407663995363961e-01,   3.390554254149696e-01,
  -9.407013691785719e-01,   3.392358088786619e-01,
  -9.406363042338476e-01,   3.394161798696234e-01,
  -9.405712047046152e-01,   3.395965383812221e-01,
  -9.405060705932683e-01,   3.397768844068269e-01,
  -9.404409019022018e-01,   3.399572179398069e-01,
  -9.403756986338115e-01,   3.401375389735317e-01,
  -9.403104607904951e-01,   3.403178475013717e-01,
  -9.402451883746509e-01,   3.404981435166972e-01,
  -9.401798813886789e-01,   3.406784270128792e-01,
  -9.401145398349803e-01,   3.408586979832894e-01,
  -9.400491637159574e-01,   3.410389564212997e-01,
  -9.399837530340140e-01,   3.412192023202824e-01,
  -9.399183077915550e-01,   3.413994356736104e-01,
  -9.398528279909867e-01,   3.415796564746572e-01,
  -9.397873136347166e-01,   3.417598647167963e-01,
  -9.397217647251533e-01,   3.419400603934022e-01,
  -9.396561812647072e-01,   3.421202434978495e-01,
  -9.395905632557893e-01,   3.423004140235135e-01,
  -9.395249107008122e-01,   3.424805719637698e-01,
  -9.394592236021899e-01,   3.426607173119944e-01,
  -9.393935019623375e-01,   3.428408500615640e-01,
  -9.393277457836714e-01,   3.430209702058555e-01,
  -9.392619550686092e-01,   3.432010777382465e-01,
  -9.391961298195699e-01,   3.433811726521150e-01,
  -9.391302700389736e-01,   3.435612549408394e-01,
  -9.390643757292420e-01,   3.437413245977985e-01,
  -9.389984468927975e-01,   3.439213816163717e-01,
  -9.389324835320646e-01,   3.441014259899388e-01,
  -9.388664856494681e-01,   3.442814577118802e-01,
  -9.388004532474348e-01,   3.444614767755765e-01,
  -9.387343863283925e-01,   3.446414831744090e-01,
  -9.386682848947702e-01,   3.448214769017593e-01,
  -9.386021489489984e-01,   3.450014579510097e-01,
  -9.385359784935086e-01,   3.451814263155425e-01,
  -9.384697735307338e-01,   3.453613819887412e-01,
  -9.384035340631081e-01,   3.455413249639891e-01,
  -9.383372600930670e-01,   3.457212552346701e-01,
  -9.382709516230472e-01,   3.459011727941690e-01,
  -9.382046086554865e-01,   3.460810776358704e-01,
  -9.381382311928244e-01,   3.462609697531600e-01,
  -9.380718192375013e-01,   3.464408491394235e-01,
  -9.380053727919588e-01,   3.466207157880473e-01,
  -9.379388918586403e-01,   3.468005696924183e-01,
  -9.378723764399899e-01,   3.469804108459237e-01,
  -9.378058265384531e-01,   3.471602392419512e-01,
  -9.377392421564770e-01,   3.473400548738891e-01,
  -9.376726232965095e-01,   3.475198577351261e-01,
  -9.376059699610000e-01,   3.476996478190514e-01,
  -9.375392821523992e-01,   3.478794251190545e-01,
  -9.374725598731593e-01,   3.480591896285256e-01,
  -9.374058031257330e-01,   3.482389413408553e-01,
  -9.373390119125750e-01,   3.484186802494346e-01,
  -9.372721862361410e-01,   3.485984063476549e-01,
  -9.372053260988880e-01,   3.487781196289084e-01,
  -9.371384315032741e-01,   3.489578200865875e-01,
  -9.370715024517592e-01,   3.491375077140850e-01,
  -9.370045389468037e-01,   3.493171825047944e-01,
  -9.369375409908699e-01,   3.494968444521095e-01,
  -9.368705085864210e-01,   3.496764935494248e-01,
  -9.368034417359216e-01,   3.498561297901349e-01,
  -9.367363404418376e-01,   3.500357531676352e-01,
  -9.366692047066362e-01,   3.502153636753216e-01,
  -9.366020345327856e-01,   3.503949613065901e-01,
  -9.365348299227555e-01,   3.505745460548375e-01,
  -9.364675908790170e-01,   3.507541179134611e-01,
  -9.364003174040421e-01,   3.509336768758584e-01,
  -9.363330095003042e-01,   3.511132229354275e-01,
  -9.362656671702783e-01,   3.512927560855671e-01,
  -9.361982904164401e-01,   3.514722763196763e-01,
  -9.361308792412670e-01,   3.516517836311546e-01,
  -9.360634336472375e-01,   3.518312780134020e-01,
  -9.359959536368314e-01,   3.520107594598191e-01,
  -9.359284392125297e-01,   3.521902279638068e-01,
  -9.358608903768146e-01,   3.523696835187666e-01,
  -9.357933071321699e-01,   3.525491261181005e-01,
  -9.357256894810804e-01,   3.527285557552107e-01,
  -9.356580374260320e-01,   3.529079724235002e-01,
  -9.355903509695124e-01,   3.530873761163725e-01,
  -9.355226301140099e-01,   3.532667668272312e-01,
  -9.354548748620146e-01,   3.534461445494808e-01,
  -9.353870852160178e-01,   3.536255092765260e-01,
  -9.353192611785116e-01,   3.538048610017721e-01,
  -9.352514027519899e-01,   3.539841997186248e-01,
  -9.351835099389476e-01,   3.541635254204903e-01,
  -9.351155827418809e-01,   3.543428381007755e-01,
  -9.350476211632874e-01,   3.545221377528874e-01,
  -9.349796252056658e-01,   3.547014243702338e-01,
  -9.349115948715161e-01,   3.548806979462228e-01,
  -9.348435301633395e-01,   3.550599584742629e-01,
  -9.347754310836387e-01,   3.552392059477633e-01,
  -9.347072976349174e-01,   3.554184403601336e-01,
  -9.346391298196808e-01,   3.555976617047839e-01,
  -9.345709276404350e-01,   3.557768699751246e-01,
  -9.345026910996879e-01,   3.559560651645668e-01,
  -9.344344201999480e-01,   3.561352472665221e-01,
  -9.343661149437258e-01,   3.563144162744024e-01,
  -9.342977753335325e-01,   3.564935721816201e-01,
  -9.342294013718808e-01,   3.566727149815883e-01,
  -9.341609930612845e-01,   3.568518446677203e-01,
  -9.340925504042590e-01,   3.570309612334300e-01,
  -9.340240734033204e-01,   3.572100646721320e-01,
  -9.339555620609867e-01,   3.573891549772409e-01,
  -9.338870163797769e-01,   3.575682321421723e-01,
  -9.338184363622110e-01,   3.577472961603419e-01,
  -9.337498220108106e-01,   3.579263470251660e-01,
  -9.336811733280984e-01,   3.581053847300616e-01,
  -9.336124903165985e-01,   3.582844092684458e-01,
  -9.335437729788362e-01,   3.584634206337365e-01,
  -9.334750213173380e-01,   3.586424188193520e-01,
  -9.334062353346315e-01,   3.588214038187109e-01,
  -9.333374150332462e-01,   3.590003756252325e-01,
  -9.332685604157120e-01,   3.591793342323365e-01,
  -9.331996714845607e-01,   3.593582796334431e-01,
  -9.331307482423252e-01,   3.595372118219731e-01,
  -9.330617906915394e-01,   3.597161307913476e-01,
  -9.329927988347390e-01,   3.598950365349881e-01,
  -9.329237726744601e-01,   3.600739290463170e-01,
  -9.328547122132411e-01,   3.602528083187569e-01,
  -9.327856174536211e-01,   3.604316743457307e-01,
  -9.327164883981403e-01,   3.606105271206623e-01,
  -9.326473250493404e-01,   3.607893666369756e-01,
  -9.325781274097644e-01,   3.609681928880952e-01,
  -9.325088954819566e-01,   3.611470058674462e-01,
  -9.324396292684624e-01,   3.613258055684543e-01,
  -9.323703287718285e-01,   3.615045919845453e-01,
  -9.323009939946028e-01,   3.616833651091458e-01,
  -9.322316249393345e-01,   3.618621249356830e-01,
  -9.321622216085744e-01,   3.620408714575842e-01,
  -9.320927840048740e-01,   3.622196046682775e-01,
  -9.320233121307865e-01,   3.623983245611913e-01,
  -9.319538059888660e-01,   3.625770311297548e-01,
  -9.318842655816681e-01,   3.627557243673972e-01,
  -9.318146909117497e-01,   3.629344042675486e-01,
  -9.317450819816687e-01,   3.631130708236395e-01,
  -9.316754387939846e-01,   3.632917240291008e-01,
  -9.316057613512578e-01,   3.634703638773638e-01,
  -9.315360496560503e-01,   3.636489903618605e-01,
  -9.314663037109251e-01,   3.638276034760235e-01,
  -9.313965235184466e-01,   3.640062032132855e-01,
  -9.313267090811804e-01,   3.641847895670799e-01,
  -9.312568604016934e-01,   3.643633625308406e-01,
  -9.311869774825537e-01,   3.645419220980021e-01,
  -9.311170603263308e-01,   3.647204682619993e-01,
  -9.310471089355953e-01,   3.648990010162673e-01,
  -9.309771233129189e-01,   3.650775203542422e-01,
  -9.309071034608751e-01,   3.652560262693603e-01,
  -9.308370493820382e-01,   3.654345187550584e-01,
  -9.307669610789837e-01,   3.656129978047739e-01,
  -9.306968385542889e-01,   3.657914634119446e-01,
  -9.306266818105318e-01,   3.659699155700087e-01,
  -9.305564908502918e-01,   3.661483542724053e-01,
  -9.304862656761498e-01,   3.663267795125736e-01,
  -9.304160062906875e-01,   3.665051912839534e-01,
  -9.303457126964885e-01,   3.666835895799849e-01,
  -9.302753848961371e-01,   3.668619743941091e-01,
  -9.302050228922191e-01,   3.670403457197672e-01,
  -9.301346266873214e-01,   3.672187035504010e-01,
  -9.300641962840324e-01,   3.673970478794527e-01,
  -9.299937316849415e-01,   3.675753787003653e-01,
  -9.299232328926397e-01,   3.677536960065820e-01,
  -9.298526999097187e-01,   3.679319997915464e-01,
  -9.297821327387722e-01,   3.681102900487030e-01,
  -9.297115313823944e-01,   3.682885667714966e-01,
  -9.296408958431813e-01,   3.684668299533723e-01,
  -9.295702261237299e-01,   3.686450795877760e-01,
  -9.294995222266386e-01,   3.688233156681539e-01,
  -9.294287841545068e-01,   3.690015381879528e-01,
  -9.293580119099355e-01,   3.691797471406200e-01,
  -9.292872054955268e-01,   3.693579425196031e-01,
  -9.292163649138840e-01,   3.695361243183506e-01,
  -9.291454901676117e-01,   3.697142925303112e-01,
  -9.290745812593159e-01,   3.698924471489341e-01,
  -9.290036381916034e-01,   3.700705881676691e-01,
  -9.289326609670828e-01,   3.702487155799664e-01,
  -9.288616495883637e-01,   3.704268293792768e-01,
  -9.287906040580570e-01,   3.706049295590517e-01,
  -9.287195243787748e-01,   3.707830161127426e-01,
  -9.286484105531305e-01,   3.709610890338020e-01,
  -9.285772625837388e-01,   3.711391483156826e-01,
  -9.285060804732156e-01,   3.713171939518375e-01,
  -9.284348642241780e-01,   3.714952259357208e-01,
  -9.283636138392444e-01,   3.716732442607865e-01,
  -9.282923293210347e-01,   3.718512489204895e-01,
  -9.282210106721694e-01,   3.720292399082850e-01,
  -9.281496578952712e-01,   3.722072172176288e-01,
  -9.280782709929631e-01,   3.723851808419774e-01,
  -9.280068499678700e-01,   3.725631307747873e-01,
  -9.279353948226179e-01,   3.727410670095158e-01,
  -9.278639055598338e-01,   3.729189895396208e-01,
  -9.277923821821463e-01,   3.730968983585606e-01,
  -9.277208246921852e-01,   3.732747934597940e-01,
  -9.276492330925812e-01,   3.734526748367803e-01,
  -9.275776073859667e-01,   3.736305424829793e-01,
  -9.275059475749752e-01,   3.738083963918512e-01,
  -9.274342536622413e-01,   3.739862365568570e-01,
  -9.273625256504011e-01,   3.741640629714579e-01,
  -9.272907635420917e-01,   3.743418756291159e-01,
  -9.272189673399518e-01,   3.745196745232932e-01,
  -9.271471370466209e-01,   3.746974596474526e-01,
  -9.270752726647401e-01,   3.748752309950575e-01,
  -9.270033741969517e-01,   3.750529885595719e-01,
  -9.269314416458991e-01,   3.752307323344599e-01,
  -9.268594750142272e-01,   3.754084623131866e-01,
  -9.267874743045817e-01,   3.755861784892172e-01,
  -9.267154395196103e-01,   3.757638808560177e-01,
  -9.266433706619612e-01,   3.759415694070544e-01,
  -9.265712677342843e-01,   3.761192441357943e-01,
  -9.264991307392305e-01,   3.762969050357048e-01,
  -9.264269596794522e-01,   3.764745521002538e-01,
  -9.263547545576029e-01,   3.766521853229096e-01,
  -9.262825153763372e-01,   3.768298046971413e-01,
  -9.262102421383114e-01,   3.770074102164183e-01,
  -9.261379348461826e-01,   3.771850018742104e-01,
  -9.260655935026093e-01,   3.773625796639883e-01,
  -9.259932181102515e-01,   3.775401435792229e-01,
  -9.259208086717701e-01,   3.777176936133856e-01,
  -9.258483651898273e-01,   3.778952297599485e-01,
  -9.257758876670867e-01,   3.780727520123840e-01,
  -9.257033761062132e-01,   3.782502603641652e-01,
  -9.256308305098727e-01,   3.784277548087656e-01,
  -9.255582508807327e-01,   3.786052353396591e-01,
  -9.254856372214615e-01,   3.787827019503205e-01,
  -9.254129895347291e-01,   3.789601546342247e-01,
  -9.253403078232063e-01,   3.791375933848473e-01,
  -9.252675920895657e-01,   3.793150181956644e-01,
  -9.251948423364805e-01,   3.794924290601526e-01,
  -9.251220585666259e-01,   3.796698259717889e-01,
  -9.250492407826776e-01,   3.798472089240512e-01,
  -9.249763889873132e-01,   3.800245779104173e-01,
  -9.249035031832109e-01,   3.802019329243660e-01,
  -9.248305833730508e-01,   3.803792739593766e-01,
  -9.247576295595139e-01,   3.805566010089285e-01,
  -9.246846417452824e-01,   3.807339140665021e-01,
  -9.246116199330400e-01,   3.809112131255781e-01,
  -9.245385641254714e-01,   3.810884981796375e-01,
  -9.244654743252626e-01,   3.812657692221624e-01,
  -9.243923505351010e-01,   3.814430262466347e-01,
  -9.243191927576752e-01,   3.816202692465374e-01,
  -9.242460009956749e-01,   3.817974982153536e-01,
  -9.241727752517912e-01,   3.819747131465672e-01,
  -9.240995155287163e-01,   3.821519140336626e-01,
  -9.240262218291438e-01,   3.823291008701245e-01,
  -9.239528941557686e-01,   3.825062736494382e-01,
  -9.238795325112867e-01,   3.826834323650898e-01,
  -9.238061368983954e-01,   3.828605770105654e-01,
  -9.237327073197933e-01,   3.830377075793520e-01,
  -9.236592437781800e-01,   3.832148240649372e-01,
  -9.235857462762567e-01,   3.833919264608087e-01,
  -9.235122148167256e-01,   3.835690147604549e-01,
  -9.234386494022904e-01,   3.837460889573650e-01,
  -9.233650500356557e-01,   3.839231490450284e-01,
  -9.232914167195276e-01,   3.841001950169350e-01,
  -9.232177494566135e-01,   3.842772268665755e-01,
  -9.231440482496219e-01,   3.844542445874408e-01,
  -9.230703131012624e-01,   3.846312481730226e-01,
  -9.229965440142462e-01,   3.848082376168129e-01,
  -9.229227409912857e-01,   3.849852129123042e-01,
  -9.228489040350941e-01,   3.851621740529899e-01,
  -9.227750331483864e-01,   3.853391210323633e-01,
  -9.227011283338786e-01,   3.855160538439188e-01,
  -9.226271895942879e-01,   3.856929724811511e-01,
  -9.225532169323328e-01,   3.858698769375553e-01,
  -9.224792103507332e-01,   3.860467672066272e-01,
  -9.224051698522099e-01,   3.862236432818630e-01,
  -9.223310954394854e-01,   3.864005051567594e-01,
  -9.222569871152830e-01,   3.865773528248139e-01,
  -9.221828448823276e-01,   3.867541862795242e-01,
  -9.221086687433452e-01,   3.869310055143886e-01,
  -9.220344587010628e-01,   3.871078105229060e-01,
  -9.219602147582092e-01,   3.872846012985758e-01,
  -9.218859369175140e-01,   3.874613778348979e-01,
  -9.218116251817081e-01,   3.876381401253727e-01,
  -9.217372795535239e-01,   3.878148881635012e-01,
  -9.216629000356947e-01,   3.879916219427849e-01,
  -9.215884866309555e-01,   3.881683414567257e-01,
  -9.215140393420420e-01,   3.883450466988262e-01,
  -9.214395581716914e-01,   3.885217376625896e-01,
  -9.213650431226423e-01,   3.886984143415192e-01,
  -9.212904941976345e-01,   3.888750767291193e-01,
  -9.212159113994087e-01,   3.890517248188944e-01,
  -9.211412947307073e-01,   3.892283586043497e-01,
  -9.210666441942736e-01,   3.894049780789909e-01,
  -9.209919597928523e-01,   3.895815832363243e-01,
  -9.209172415291895e-01,   3.897581740698564e-01,
  -9.208424894060322e-01,   3.899347505730947e-01,
  -9.207677034261288e-01,   3.901113127395469e-01,
  -9.206928835922291e-01,   3.902878605627212e-01,
  -9.206180299070840e-01,   3.904643940361266e-01,
  -9.205431423734455e-01,   3.906409131532724e-01,
  -9.204682209940671e-01,   3.908174179076685e-01,
  -9.203932657717036e-01,   3.909939082928254e-01,
  -9.203182767091106e-01,   3.911703843022539e-01,
  -9.202432538090454e-01,   3.913468459294656e-01,
  -9.201681970742663e-01,   3.915232931679724e-01,
  -9.200931065075332e-01,   3.916997260112869e-01,
  -9.200179821116066e-01,   3.918761444529223e-01,
  -9.199428238892486e-01,   3.920525484863921e-01,
  -9.198676318432230e-01,   3.922289381052103e-01,
  -9.197924059762939e-01,   3.924053133028917e-01,
  -9.197171462912274e-01,   3.925816740729515e-01,
  -9.196418527907905e-01,   3.927580204089053e-01,
  -9.195665254777515e-01,   3.929343523042695e-01,
  -9.194911643548801e-01,   3.931106697525608e-01,
  -9.194157694249471e-01,   3.932869727472964e-01,
  -9.193403406907243e-01,   3.934632612819943e-01,
  -9.192648781549854e-01,   3.936395353501729e-01,
  -9.191893818205045e-01,   3.938157949453510e-01,
  -9.191138516900578e-01,   3.939920400610481e-01,
  -9.190382877664220e-01,   3.941682706907841e-01,
  -9.189626900523756e-01,   3.943444868280796e-01,
  -9.188870585506980e-01,   3.945206884664556e-01,
  -9.188113932641700e-01,   3.946968755994336e-01,
  -9.187356941955735e-01,   3.948730482205358e-01,
  -9.186599613476919e-01,   3.950492063232848e-01,
  -9.185841947233095e-01,   3.952253499012037e-01,
  -9.185083943252123e-01,   3.954014789478164e-01,
  -9.184325601561869e-01,   3.955775934566468e-01,
  -9.183566922190217e-01,   3.957536934212201e-01,
  -9.182807905165061e-01,   3.959297788350613e-01,
  -9.182048550514309e-01,   3.961058496916963e-01,
  -9.181288858265880e-01,   3.962819059846515e-01,
  -9.180528828447704e-01,   3.964579477074539e-01,
  -9.179768461087727e-01,   3.966339748536308e-01,
  -9.179007756213905e-01,   3.968099874167103e-01,
  -9.178246713854206e-01,   3.969859853902209e-01,
  -9.177485334036612e-01,   3.971619687676916e-01,
  -9.176723616789119e-01,   3.973379375426521e-01,
  -9.175961562139730e-01,   3.975138917086323e-01,
  -9.175199170116463e-01,   3.976898312591632e-01,
  -9.174436440747352e-01,   3.978657561877758e-01,
  -9.173673374060439e-01,   3.980416664880018e-01,
  -9.172909970083779e-01,   3.982175621533736e-01,
  -9.172146228845443e-01,   3.983934431774240e-01,
  -9.171382150373507e-01,   3.985693095536863e-01,
  -9.170617734696068e-01,   3.987451612756944e-01,
  -9.169852981841230e-01,   3.989209983369829e-01,
  -9.169087891837110e-01,   3.990968207310865e-01,
  -9.168322464711839e-01,   3.992726284515410e-01,
  -9.167556700493560e-01,   3.994484214918821e-01,
  -9.166790599210427e-01,   3.996241998456468e-01,
  -9.166024160890608e-01,   3.997999635063720e-01,
  -9.165257385562282e-01,   3.999757124675953e-01,
  -9.164490273253642e-01,   4.001514467228551e-01,
  -9.163722823992891e-01,   4.003271662656901e-01,
  -9.162955037808248e-01,   4.005028710896395e-01,
  -9.162186914727942e-01,   4.006785611882432e-01,
  -9.161418454780214e-01,   4.008542365550417e-01,
  -9.160649657993317e-01,   4.010298971835756e-01,
  -9.159880524395520e-01,   4.012055430673867e-01,
  -9.159111054015099e-01,   4.013811742000168e-01,
  -9.158341246880347e-01,   4.015567905750085e-01,
  -9.157571103019567e-01,   4.017323921859050e-01,
  -9.156800622461077e-01,   4.019079790262497e-01,
  -9.156029805233202e-01,   4.020835510895870e-01,
  -9.155258651364285e-01,   4.022591083694615e-01,
  -9.154487160882678e-01,   4.024346508594184e-01,
  -9.153715333816748e-01,   4.026101785530037e-01,
  -9.152943170194870e-01,   4.027856914437635e-01,
  -9.152170670045439e-01,   4.029611895252449e-01,
  -9.151397833396853e-01,   4.031366727909953e-01,
  -9.150624660277529e-01,   4.033121412345625e-01,
  -9.149851150715893e-01,   4.034875948494953e-01,
  -9.149077304740387e-01,   4.036630336293426e-01,
  -9.148303122379462e-01,   4.038384575676541e-01,
  -9.147528603661582e-01,   4.040138666579799e-01,
  -9.146753748615224e-01,   4.041892608938707e-01,
  -9.145978557268878e-01,   4.043646402688778e-01,
  -9.145203029651044e-01,   4.045400047765530e-01,
  -9.144427165790239e-01,   4.047153544104486e-01,
  -9.143650965714986e-01,   4.048906891641176e-01,
  -9.142874429453824e-01,   4.050660090311133e-01,
  -9.142097557035307e-01,   4.052413140049899e-01,
  -9.141320348487995e-01,   4.054166040793016e-01,
  -9.140542803840466e-01,   4.055918792476039e-01,
  -9.139764923121306e-01,   4.057671395034521e-01,
  -9.138986706359117e-01,   4.059423848404025e-01,
  -9.138208153582511e-01,   4.061176152520118e-01,
  -9.137429264820114e-01,   4.062928307318374e-01,
  -9.136650040100563e-01,   4.064680312734370e-01,
  -9.135870479452508e-01,   4.066432168703690e-01,
  -9.135090582904611e-01,   4.068183875161923e-01,
  -9.134310350485547e-01,   4.069935432044665e-01,
  -9.133529782224002e-01,   4.071686839287516e-01,
  -9.132748878148678e-01,   4.073438096826080e-01,
  -9.131967638288282e-01,   4.075189204595969e-01,
  -9.131186062671542e-01,   4.076940162532801e-01,
  -9.130404151327192e-01,   4.078690970572198e-01,
  -9.129621904283982e-01,   4.080441628649787e-01,
  -9.128839321570672e-01,   4.082192136701201e-01,
  -9.128056403216035e-01,   4.083942494662080e-01,
  -9.127273149248859e-01,   4.085692702468068e-01,
  -9.126489559697939e-01,   4.087442760054814e-01,
  -9.125705634592087e-01,   4.089192667357974e-01,
  -9.124921373960126e-01,   4.090942424313210e-01,
  -9.124136777830890e-01,   4.092692030856186e-01,
  -9.123351846233227e-01,   4.094441486922576e-01,
  -9.122566579195998e-01,   4.096190792448057e-01,
  -9.121780976748072e-01,   4.097939947368311e-01,
  -9.120995038918335e-01,   4.099688951619029e-01,
  -9.120208765735683e-01,   4.101437805135902e-01,
  -9.119422157229026e-01,   4.103186507854633e-01,
  -9.118635213427285e-01,   4.104935059710924e-01,
  -9.117847934359394e-01,   4.106683460640487e-01,
  -9.117060320054299e-01,   4.108431710579039e-01,
  -9.116272370540957e-01,   4.110179809462302e-01,
  -9.115484085848340e-01,   4.111927757226002e-01,
  -9.114695466005430e-01,   4.113675553805872e-01,
  -9.113906511041224e-01,   4.115423199137652e-01,
  -9.113117220984728e-01,   4.117170693157086e-01,
  -9.112327595864962e-01,   4.118918035799922e-01,
  -9.111537635710959e-01,   4.120665227001916e-01,
  -9.110747340551764e-01,   4.122412266698829e-01,
  -9.109956710416431e-01,   4.124159154826427e-01,
  -9.109165745334034e-01,   4.125905891320482e-01,
  -9.108374445333650e-01,   4.127652476116773e-01,
  -9.107582810444376e-01,   4.129398909151081e-01,
  -9.106790840695316e-01,   4.131145190359194e-01,
  -9.105998536115589e-01,   4.132891319676910e-01,
  -9.105205896734327e-01,   4.134637297040024e-01,
  -9.104412922580672e-01,   4.136383122384345e-01,
  -9.103619613683780e-01,   4.138128795645683e-01,
  -9.102825970072818e-01,   4.139874316759854e-01,
  -9.102031991776965e-01,   4.141619685662681e-01,
  -9.101237678825417e-01,   4.143364902289991e-01,
  -9.100443031247375e-01,   4.145109966577618e-01,
  -9.099648049072057e-01,   4.146854878461400e-01,
  -9.098852732328692e-01,   4.148599637877183e-01,
  -9.098057081046522e-01,   4.150344244760816e-01,
  -9.097261095254802e-01,   4.152088699048156e-01,
  -9.096464774982795e-01,   4.153833000675062e-01,
  -9.095668120259783e-01,   4.155577149577404e-01,
  -9.094871131115054e-01,   4.157321145691054e-01,
  -9.094073807577913e-01,   4.159064988951888e-01,
  -9.093276149677673e-01,   4.160808679295792e-01,
  -9.092478157443663e-01,   4.162552216658655e-01,
  -9.091679830905224e-01,   4.164295600976372e-01,
  -9.090881170091706e-01,   4.166038832184843e-01,
  -9.090082175032475e-01,   4.167781910219976e-01,
  -9.089282845756906e-01,   4.169524835017682e-01,
  -9.088483182294391e-01,   4.171267606513879e-01,
  -9.087683184674329e-01,   4.173010224644489e-01,
  -9.086882852926134e-01,   4.174752689345443e-01,
  -9.086082187079232e-01,   4.176495000552674e-01,
  -9.085281187163061e-01,   4.178237158202123e-01,
  -9.084479853207073e-01,   4.179979162229736e-01,
  -9.083678185240729e-01,   4.181721012571463e-01,
  -9.082876183293505e-01,   4.183462709163263e-01,
  -9.082073847394887e-01,   4.185204251941097e-01,
  -9.081271177574376e-01,   4.186945640840936e-01,
  -9.080468173861483e-01,   4.188686875798751e-01,
  -9.079664836285734e-01,   4.190427956750524e-01,
  -9.078861164876663e-01,   4.192168883632239e-01,
  -9.078057159663819e-01,   4.193909656379889e-01,
  -9.077252820676764e-01,   4.195650274929469e-01,
  -9.076448147945072e-01,   4.197390739216982e-01,
  -9.075643141498326e-01,   4.199131049178436e-01,
  -9.074837801366126e-01,   4.200871204749845e-01,
  -9.074032127578081e-01,   4.202611205867229e-01,
  -9.073226120163814e-01,   4.204351052466612e-01,
  -9.072419779152958e-01,   4.206090744484025e-01,
  -9.071613104575162e-01,   4.207830281855505e-01,
  -9.070806096460085e-01,   4.209569664517094e-01,
  -9.069998754837396e-01,   4.211308892404840e-01,
  -9.069191079736781e-01,   4.213047965454796e-01,
  -9.068383071187934e-01,   4.214786883603023e-01,
  -9.067574729220566e-01,   4.216525646785583e-01,
  -9.066766053864395e-01,   4.218264254938549e-01,
  -9.065957045149153e-01,   4.220002707997997e-01,
  -9.065147703104588e-01,   4.221741005900008e-01,
  -9.064338027760455e-01,   4.223479148580671e-01,
  -9.063528019146524e-01,   4.225217135976078e-01,
  -9.062717677292577e-01,   4.226954968022330e-01,
  -9.061907002228407e-01,   4.228692644655531e-01,
  -9.061095993983820e-01,   4.230430165811790e-01,
  -9.060284652588636e-01,   4.232167531427226e-01,
  -9.059472978072685e-01,   4.233904741437960e-01,
  -9.058660970465809e-01,   4.235641795780120e-01,
  -9.057848629797866e-01,   4.237378694389838e-01,
  -9.057035956098720e-01,   4.239115437203256e-01,
  -9.056222949398253e-01,   4.240852024156516e-01,
  -9.055409609726356e-01,   4.242588455185770e-01,
  -9.054595937112933e-01,   4.244324730227174e-01,
  -9.053781931587901e-01,   4.246060849216891e-01,
  -9.052967593181188e-01,   4.247796812091088e-01,
  -9.052152921922736e-01,   4.249532618785939e-01,
  -9.051337917842497e-01,   4.251268269237624e-01,
  -9.050522580970436e-01,   4.253003763382326e-01,
  -9.049706911336532e-01,   4.254739101156238e-01,
  -9.048890908970775e-01,   4.256474282495556e-01,
  -9.048074573903165e-01,   4.258209307336482e-01,
  -9.047257906163719e-01,   4.259944175615224e-01,
  -9.046440905782462e-01,   4.261678887267996e-01,
  -9.045623572789433e-01,   4.263413442231018e-01,
  -9.044805907214682e-01,   4.265147840440515e-01,
  -9.043987909088274e-01,   4.266882081832719e-01,
  -9.043169578440283e-01,   4.268616166343864e-01,
  -9.042350915300797e-01,   4.270350093910197e-01,
  -9.041531919699918e-01,   4.272083864467963e-01,
  -9.040712591667754e-01,   4.273817477953418e-01,
  -9.039892931234433e-01,   4.275550934302821e-01,
  -9.039072938430090e-01,   4.277284233452438e-01,
  -9.038252613284875e-01,   4.279017375338541e-01,
  -9.037431955828946e-01,   4.280750359897407e-01,
  -9.036610966092480e-01,   4.282483187065320e-01,
  -9.035789644105661e-01,   4.284215856778567e-01,
  -9.034967989898685e-01,   4.285948368973444e-01,
  -9.034146003501763e-01,   4.287680723586251e-01,
  -9.033323684945118e-01,   4.289412920553295e-01,
  -9.032501034258984e-01,   4.291144959810888e-01,
  -9.031678051473607e-01,   4.292876841295346e-01,
  -9.030854736619246e-01,   4.294608564942995e-01,
  -9.030031089726172e-01,   4.296340130690164e-01,
  -9.029207110824667e-01,   4.298071538473187e-01,
  -9.028382799945028e-01,   4.299802788228406e-01,
  -9.027558157117561e-01,   4.301533879892169e-01,
  -9.026733182372588e-01,   4.303264813400826e-01,
  -9.025907875740439e-01,   4.304995588690738e-01,
  -9.025082237251459e-01,   4.306726205698268e-01,
  -9.024256266936004e-01,   4.308456664359787e-01,
  -9.023429964824442e-01,   4.310186964611670e-01,
  -9.022603330947155e-01,   4.311917106390299e-01,
  -9.021776365334536e-01,   4.313647089632063e-01,
  -9.020949068016989e-01,   4.315376914273355e-01,
  -9.020121439024932e-01,   4.317106580250573e-01,
  -9.019293478388795e-01,   4.318836087500122e-01,
  -9.018465186139017e-01,   4.320565435958415e-01,
  -9.017636562306057e-01,   4.322294625561867e-01,
  -9.016807606920377e-01,   4.324023656246901e-01,
  -9.015978320012457e-01,   4.325752527949946e-01,
  -9.015148701612787e-01,   4.327481240607437e-01,
  -9.014318751751870e-01,   4.329209794155813e-01,
  -9.013488470460220e-01,   4.330938188531520e-01,
  -9.012657857768366e-01,   4.332666423671009e-01,
  -9.011826913706845e-01,   4.334394499510741e-01,
  -9.010995638306210e-01,   4.336122415987176e-01,
  -9.010164031597023e-01,   4.337850173036785e-01,
  -9.009332093609862e-01,   4.339577770596044e-01,
  -9.008499824375314e-01,   4.341305208601433e-01,
  -9.007667223923979e-01,   4.343032486989440e-01,
  -9.006834292286470e-01,   4.344759605696557e-01,
  -9.006001029493409e-01,   4.346486564659283e-01,
  -9.005167435575435e-01,   4.348213363814123e-01,
  -9.004333510563198e-01,   4.349940003097587e-01,
  -9.003499254487356e-01,   4.351666482446193e-01,
  -9.002664667378585e-01,   4.353392801796461e-01,
  -9.001829749267568e-01,   4.355118961084920e-01,
  -9.000994500185004e-01,   4.356844960248105e-01,
  -9.000158920161603e-01,   4.358570799222555e-01,
  -8.999323009228085e-01,   4.360296477944816e-01,
  -8.998486767415186e-01,   4.362021996351440e-01,
  -8.997650194753651e-01,   4.363747354378983e-01,
  -8.996813291274239e-01,   4.365472551964012e-01,
  -8.995976057007722e-01,   4.367197589043094e-01,
  -8.995138491984880e-01,   4.368922465552804e-01,
  -8.994300596236509e-01,   4.370647181429724e-01,
  -8.993462369793416e-01,   4.372371736610441e-01,
  -8.992623812686420e-01,   4.374096131031548e-01,
  -8.991784924946353e-01,   4.375820364629644e-01,
  -8.990945706604058e-01,   4.377544437341334e-01,
  -8.990106157690391e-01,   4.379268349103229e-01,
  -8.989266278236219e-01,   4.380992099851945e-01,
  -8.988426068272424e-01,   4.382715689524104e-01,
  -8.987585527829894e-01,   4.384439118056337e-01,
  -8.986744656939538e-01,   4.386162385385277e-01,
  -8.985903455632270e-01,   4.387885491447563e-01,
  -8.985061923939020e-01,   4.389608436179843e-01,
  -8.984220061890725e-01,   4.391331219518769e-01,
  -8.983377869518343e-01,   4.393053841401000e-01,
  -8.982535346852836e-01,   4.394776301763198e-01,
  -8.981692493925181e-01,   4.396498600542035e-01,
  -8.980849310766368e-01,   4.398220737674185e-01,
  -8.980005797407399e-01,   4.399942713096333e-01,
  -8.979161953879287e-01,   4.401664526745163e-01,
  -8.978317780213056e-01,   4.403386178557372e-01,
  -8.977473276439747e-01,   4.405107668469659e-01,
  -8.976628442590409e-01,   4.406828996418729e-01,
  -8.975783278696102e-01,   4.408550162341294e-01,
  -8.974937784787903e-01,   4.410271166174072e-01,
  -8.974091960896897e-01,   4.411992007853787e-01,
  -8.973245807054183e-01,   4.413712687317167e-01,
  -8.972399323290872e-01,   4.415433204500949e-01,
  -8.971552509638085e-01,   4.417153559341873e-01,
  -8.970705366126959e-01,   4.418873751776689e-01,
  -8.969857892788640e-01,   4.420593781742147e-01,
  -8.969010089654288e-01,   4.422313649175010e-01,
  -8.968161956755073e-01,   4.424033354012041e-01,
  -8.967313494122179e-01,   4.425752896190012e-01,
  -8.966464701786802e-01,   4.427472275645700e-01,
  -8.965615579780150e-01,   4.429191492315890e-01,
  -8.964766128133441e-01,   4.430910546137369e-01,
  -8.963916346877908e-01,   4.432629437046933e-01,
  -8.963066236044795e-01,   4.434348164981385e-01,
  -8.962215795665360e-01,   4.436066729877530e-01,
  -8.961365025770868e-01,   4.437785131672182e-01,
  -8.960513926392601e-01,   4.439503370302161e-01,
  -8.959662497561852e-01,   4.441221445704292e-01,
  -8.958810739309924e-01,   4.442939357815406e-01,
  -8.957958651668135e-01,   4.444657106572340e-01,
  -8.957106234667813e-01,   4.446374691911938e-01,
  -8.956253488340301e-01,   4.448092113771049e-01,
  -8.955400412716950e-01,   4.449809372086527e-01,
  -8.954547007829124e-01,   4.451526466795236e-01,
  -8.953693273708203e-01,   4.453243397834042e-01,
  -8.952839210385576e-01,   4.454960165139817e-01,
  -8.951984817892642e-01,   4.456676768649443e-01,
  -8.951130096260818e-01,   4.458393208299803e-01,
  -8.950275045521526e-01,   4.460109484027789e-01,
  -8.949419665706208e-01,   4.461825595770301e-01,
  -8.948563956846310e-01,   4.463541543464238e-01,
  -8.947707918973296e-01,   4.465257327046513e-01,
  -8.946851552118640e-01,   4.466972946454041e-01,
  -8.945994856313827e-01,   4.468688401623742e-01,
  -8.945137831590356e-01,   4.470403692492544e-01,
  -8.944280477979738e-01,   4.472118818997383e-01,
  -8.943422795513495e-01,   4.473833781075196e-01,
  -8.942564784223160e-01,   4.475548578662930e-01,
  -8.941706444140283e-01,   4.477263211697536e-01,
  -8.940847775296420e-01,   4.478977680115973e-01,
  -8.939988777723142e-01,   4.480691983855204e-01,
  -8.939129451452033e-01,   4.482406122852199e-01,
  -8.938269796514686e-01,   4.484120097043934e-01,
  -8.937409812942710e-01,   4.485833906367392e-01,
  -8.936549500767725e-01,   4.487547550759560e-01,
  -8.935688860021359e-01,   4.489261030157433e-01,
  -8.934827890735259e-01,   4.490974344498010e-01,
  -8.933966592941077e-01,   4.492687493718299e-01,
  -8.933104966670482e-01,   4.494400477755311e-01,
  -8.932243011955153e-01,   4.496113296546065e-01,
  -8.931380728826783e-01,   4.497825950027587e-01,
  -8.930518117317074e-01,   4.499538438136905e-01,
  -8.929655177457744e-01,   4.501250760811057e-01,
  -8.928791909280517e-01,   4.502962917987086e-01,
  -8.927928312817136e-01,   4.504674909602041e-01,
  -8.927064388099354e-01,   4.506386735592976e-01,
  -8.926200135158932e-01,   4.508098395896953e-01,
  -8.925335554027646e-01,   4.509809890451039e-01,
  -8.924470644737287e-01,   4.511521219192306e-01,
  -8.923605407319654e-01,   4.513232382057835e-01,
  -8.922739841806558e-01,   4.514943378984711e-01,
  -8.921873948229825e-01,   4.516654209910025e-01,
  -8.921007726621291e-01,   4.518364874770875e-01,
  -8.920141177012805e-01,   4.520075373504364e-01,
  -8.919274299436225e-01,   4.521785706047604e-01,
  -8.918407093923427e-01,   4.523495872337709e-01,
  -8.917539560506295e-01,   4.525205872311800e-01,
  -8.916671699216723e-01,   4.526915705907009e-01,
  -8.915803510086623e-01,   4.528625373060468e-01,
  -8.914934993147914e-01,   4.530334873709316e-01,
  -8.914066148432529e-01,   4.532044207790702e-01,
  -8.913196975972414e-01,   4.533753375241777e-01,
  -8.912327475799525e-01,   4.535462375999701e-01,
  -8.911457647945832e-01,   4.537171210001639e-01,
  -8.910587492443316e-01,   4.538879877184760e-01,
  -8.909717009323969e-01,   4.540588377486244e-01,
  -8.908846198619795e-01,   4.542296710843273e-01,
  -8.907975060362815e-01,   4.544004877193036e-01,
  -8.907103594585056e-01,   4.545712876472729e-01,
  -8.906231801318559e-01,   4.547420708619554e-01,
  -8.905359680595378e-01,   4.549128373570719e-01,
  -8.904487232447579e-01,   4.550835871263438e-01,
  -8.903614456907238e-01,   4.552543201634931e-01,
  -8.902741354006446e-01,   4.554250364622424e-01,
  -8.901867923777302e-01,   4.555957360163150e-01,
  -8.900994166251923e-01,   4.557664188194346e-01,
  -8.900120081462433e-01,   4.559370848653260e-01,
  -8.899245669440967e-01,   4.561077341477141e-01,
  -8.898370930219679e-01,   4.562783666603246e-01,
  -8.897495863830728e-01,   4.564489823968839e-01,
  -8.896620470306289e-01,   4.566195813511189e-01,
  -8.895744749678546e-01,   4.567901635167572e-01,
  -8.894868701979699e-01,   4.569607288875270e-01,
  -8.893992327241955e-01,   4.571312774571570e-01,
  -8.893115625497539e-01,   4.573018092193766e-01,
  -8.892238596778682e-01,   4.574723241679161e-01,
  -8.891361241117632e-01,   4.576428222965058e-01,
  -8.890483558546646e-01,   4.578133035988772e-01,
  -8.889605549097993e-01,   4.579837680687621e-01,
  -8.888727212803956e-01,   4.581542156998931e-01,
  -8.887848549696828e-01,   4.583246464860032e-01,
  -8.886969559808916e-01,   4.584950604208263e-01,
  -8.886090243172539e-01,   4.586654574980966e-01,
  -8.885210599820023e-01,   4.588358377115491e-01,
  -8.884330629783713e-01,   4.590062010549196e-01,
  -8.883450333095964e-01,   4.591765475219441e-01,
  -8.882569709789139e-01,   4.593468771063596e-01,
  -8.881688759895617e-01,   4.595171898019035e-01,
  -8.880807483447789e-01,   4.596874856023139e-01,
  -8.879925880478056e-01,   4.598577645013295e-01,
  -8.879043951018832e-01,   4.600280264926896e-01,
  -8.878161695102544e-01,   4.601982715701343e-01,
  -8.877279112761630e-01,   4.603684997274040e-01,
  -8.876396204028539e-01,   4.605387109582400e-01,
  -8.875512968935734e-01,   4.607089052563841e-01,
  -8.874629407515688e-01,   4.608790826155787e-01,
  -8.873745519800889e-01,   4.610492430295669e-01,
  -8.872861305823831e-01,   4.612193864920924e-01,
  -8.871976765617029e-01,   4.613895129968995e-01,
  -8.871091899213002e-01,   4.615596225377331e-01,
  -8.870206706644284e-01,   4.617297151083388e-01,
  -8.869321187943422e-01,   4.618997907024627e-01,
  -8.868435343142974e-01,   4.620698493138518e-01,
  -8.867549172275508e-01,   4.622398909362533e-01,
  -8.866662675373610e-01,   4.624099155634154e-01,
  -8.865775852469870e-01,   4.625799231890868e-01,
  -8.864888703596896e-01,   4.627499138070167e-01,
  -8.864001228787306e-01,   4.629198874109551e-01,
  -8.863113428073728e-01,   4.630898439946525e-01,
  -8.862225301488806e-01,   4.632597835518601e-01,
  -8.861336849065193e-01,   4.634297060763298e-01,
  -8.860448070835556e-01,   4.635996115618140e-01,
  -8.859558966832570e-01,   4.637695000020656e-01,
  -8.858669537088928e-01,   4.639393713908385e-01,
  -8.857779781637329e-01,   4.641092257218870e-01,
  -8.856889700510490e-01,   4.642790629889658e-01,
  -8.855999293741134e-01,   4.644488831858307e-01,
  -8.855108561362000e-01,   4.646186863062378e-01,
  -8.854217503405837e-01,   4.647884723439440e-01,
  -8.853326119905406e-01,   4.649582412927067e-01,
  -8.852434410893483e-01,   4.651279931462839e-01,
  -8.851542376402851e-01,   4.652977278984346e-01,
  -8.850650016466309e-01,   4.654674455429178e-01,
  -8.849757331116667e-01,   4.656371460734937e-01,
  -8.848864320386746e-01,   4.658068294839227e-01,
  -8.847970984309378e-01,   4.659764957679662e-01,
  -8.847077322917410e-01,   4.661461449193859e-01,
  -8.846183336243699e-01,   4.663157769319444e-01,
  -8.845289024321115e-01,   4.664853917994049e-01,
  -8.844394387182538e-01,   4.666549895155309e-01,
  -8.843499424860861e-01,   4.668245700740870e-01,
  -8.842604137388992e-01,   4.669941334688380e-01,
  -8.841708524799845e-01,   4.671636796935498e-01,
  -8.840812587126350e-01,   4.673332087419884e-01,
  -8.839916324401449e-01,   4.675027206079209e-01,
  -8.839019736658095e-01,   4.676722152851148e-01,
  -8.838122823929251e-01,   4.678416927673382e-01,
  -8.837225586247897e-01,   4.680111530483598e-01,
  -8.836328023647019e-01,   4.681805961219493e-01,
  -8.835430136159619e-01,   4.683500219818765e-01,
  -8.834531923818709e-01,   4.685194306219123e-01,
  -8.833633386657316e-01,   4.686888220358279e-01,
  -8.832734524708474e-01,   4.688581962173953e-01,
  -8.831835338005234e-01,   4.690275531603871e-01,
  -8.830935826580654e-01,   4.691968928585766e-01,
  -8.830035990467808e-01,   4.693662153057375e-01,
  -8.829135829699780e-01,   4.695355204956445e-01,
  -8.828235344309666e-01,   4.697048084220725e-01,
  -8.827334534330576e-01,   4.698740790787974e-01,
  -8.826433399795628e-01,   4.700433324595956e-01,
  -8.825531940737955e-01,   4.702125685582442e-01,
  -8.824630157190702e-01,   4.703817873685207e-01,
  -8.823728049187023e-01,   4.705509888842035e-01,
  -8.822825616760087e-01,   4.707201730990716e-01,
  -8.821922859943074e-01,   4.708893400069045e-01,
  -8.821019778769176e-01,   4.710584896014825e-01,
  -8.820116373271596e-01,   4.712276218765863e-01,
  -8.819212643483550e-01,   4.713967368259976e-01,
  -8.818308589438266e-01,   4.715658344434984e-01,
  -8.817404211168983e-01,   4.717349147228714e-01,
  -8.816499508708953e-01,   4.719039776579002e-01,
  -8.815594482091438e-01,   4.720730232423687e-01,
  -8.814689131349714e-01,   4.722420514700615e-01,
  -8.813783456517069e-01,   4.724110623347640e-01,
  -8.812877457626801e-01,   4.725800558302622e-01,
  -8.811971134712221e-01,   4.727490319503428e-01,
  -8.811064487806651e-01,   4.729179906887928e-01,
  -8.810157516943429e-01,   4.730869320394001e-01,
  -8.809250222155899e-01,   4.732558559959533e-01,
  -8.808342603477420e-01,   4.734247625522415e-01,
  -8.807434660941363e-01,   4.735936517020545e-01,
  -8.806526394581110e-01,   4.737625234391828e-01,
  -8.805617804430057e-01,   4.739313777574174e-01,
  -8.804708890521608e-01,   4.741002146505500e-01,
  -8.803799652889182e-01,   4.742690341123730e-01,
  -8.802890091566210e-01,   4.744378361366792e-01,
  -8.801980206586132e-01,   4.746066207172626e-01,
  -8.801069997982404e-01,   4.747753878479171e-01,
  -8.800159465788491e-01,   4.749441375224378e-01,
  -8.799248610037869e-01,   4.751128697346203e-01,
  -8.798337430764029e-01,   4.752815844782607e-01,
  -8.797425928000474e-01,   4.754502817471559e-01,
  -8.796514101780716e-01,   4.756189615351033e-01,
  -8.795601952138279e-01,   4.757876238359011e-01,
  -8.794689479106702e-01,   4.759562686433481e-01,
  -8.793776682719533e-01,   4.761248959512436e-01,
  -8.792863563010332e-01,   4.762935057533877e-01,
  -8.791950120012675e-01,   4.764620980435812e-01,
  -8.791036353760143e-01,   4.766306728156253e-01,
  -8.790122264286335e-01,   4.767992300633221e-01,
  -8.789207851624858e-01,   4.769677697804742e-01,
  -8.788293115809334e-01,   4.771362919608848e-01,
  -8.787378056873394e-01,   4.773047965983579e-01,
  -8.786462674850681e-01,   4.774732836866981e-01,
  -8.785546969774854e-01,   4.776417532197105e-01,
  -8.784630941679579e-01,   4.778102051912010e-01,
  -8.783714590598535e-01,   4.779786395949762e-01,
  -8.782797916565416e-01,   4.781470564248430e-01,
  -8.781880919613922e-01,   4.783154556746095e-01,
  -8.780963599777771e-01,   4.784838373380840e-01,
  -8.780045957090691e-01,   4.786522014090755e-01,
  -8.779127991586418e-01,   4.788205478813939e-01,
  -8.778209703298705e-01,   4.789888767488495e-01,
  -8.777291092261316e-01,   4.791571880052533e-01,
  -8.776372158508022e-01,   4.793254816444171e-01,
  -8.775452902072614e-01,   4.794937576601530e-01,
  -8.774533322988886e-01,   4.796620160462742e-01,
  -8.773613421290651e-01,   4.798302567965942e-01,
  -8.772693197011732e-01,   4.799984799049273e-01,
  -8.771772650185959e-01,   4.801666853650884e-01,
  -8.770851780847184e-01,   4.803348731708930e-01,
  -8.769930589029259e-01,   4.805030433161575e-01,
  -8.769009074766057e-01,   4.806711957946986e-01,
  -8.768087238091457e-01,   4.808393306003340e-01,
  -8.767165079039354e-01,   4.810074477268816e-01,
  -8.766242597643653e-01,   4.811755471681603e-01,
  -8.765319793938271e-01,   4.813436289179897e-01,
  -8.764396667957136e-01,   4.815116929701899e-01,
  -8.763473219734190e-01,   4.816797393185815e-01,
  -8.762549449303385e-01,   4.818477679569860e-01,
  -8.761625356698685e-01,   4.820157788792256e-01,
  -8.760700941954066e-01,   4.821837720791227e-01,
  -8.759776205103518e-01,   4.823517475505010e-01,
  -8.758851146181038e-01,   4.825197052871844e-01,
  -8.757925765220639e-01,   4.826876452829975e-01,
  -8.757000062256346e-01,   4.828555675317657e-01,
  -8.756074037322193e-01,   4.830234720273149e-01,
  -8.755147690452229e-01,   4.831913587634719e-01,
  -8.754221021680509e-01,   4.833592277340638e-01,
  -8.753294031041109e-01,   4.835270789329187e-01,
  -8.752366718568109e-01,   4.836949123538651e-01,
  -8.751439084295604e-01,   4.838627279907323e-01,
  -8.750511128257700e-01,   4.840305258373500e-01,
  -8.749582850488516e-01,   4.841983058875490e-01,
  -8.748654251022183e-01,   4.843660681351604e-01,
  -8.747725329892841e-01,   4.845338125740162e-01,
  -8.746796087134645e-01,   4.847015391979487e-01,
  -8.745866522781761e-01,   4.848692480007911e-01,
  -8.744936636868366e-01,   4.850369389763773e-01,
  -8.744006429428648e-01,   4.852046121185418e-01,
  -8.743075900496810e-01,   4.853722674211198e-01,
  -8.742145050107063e-01,   4.855399048779470e-01,
  -8.741213878293633e-01,   4.857075244828598e-01,
  -8.740282385090757e-01,   4.858751262296953e-01,
  -8.739350570532682e-01,   4.860427101122913e-01,
  -8.738418434653669e-01,   4.862102761244864e-01,
  -8.737485977487989e-01,   4.863778242601194e-01,
  -8.736553199069926e-01,   4.865453545130303e-01,
  -8.735620099433778e-01,   4.867128668770592e-01,
  -8.734686678613849e-01,   4.868803613460473e-01,
  -8.733752936644460e-01,   4.870478379138364e-01,
  -8.732818873559942e-01,   4.872152965742688e-01,
  -8.731884489394638e-01,   4.873827373211874e-01,
  -8.730949784182901e-01,   4.875501601484360e-01,
  -8.730014757959099e-01,   4.877175650498588e-01,
  -8.729079410757611e-01,   4.878849520193010e-01,
  -8.728143742612824e-01,   4.880523210506083e-01,
  -8.727207753559143e-01,   4.882196721376268e-01,
  -8.726271443630980e-01,   4.883870052742035e-01,
  -8.725334812862762e-01,   4.885543204541862e-01,
  -8.724397861288923e-01,   4.887216176714231e-01,
  -8.723460588943915e-01,   4.888888969197632e-01,
  -8.722522995862199e-01,   4.890561581930560e-01,
  -8.721585082078245e-01,   4.892234014851520e-01,
  -8.720646847626539e-01,   4.893906267899019e-01,
  -8.719708292541578e-01,   4.895578341011574e-01,
  -8.718769416857869e-01,   4.897250234127709e-01,
  -8.717830220609931e-01,   4.898921947185952e-01,
  -8.716890703832297e-01,   4.900593480124839e-01,
  -8.715950866559510e-01,   4.902264832882912e-01,
  -8.715010708826125e-01,   4.903936005398720e-01,
  -8.714070230666710e-01,   4.905606997610820e-01,
  -8.713129432115840e-01,   4.907277809457774e-01,
  -8.712188313208110e-01,   4.908948440878151e-01,
  -8.711246873978119e-01,   4.910618891810526e-01,
  -8.710305114460483e-01,   4.912289162193483e-01,
  -8.709363034689828e-01,   4.913959251965608e-01,
  -8.708420634700790e-01,   4.915629161065499e-01,
  -8.707477914528018e-01,   4.917298889431758e-01,
  -8.706534874206174e-01,   4.918968437002993e-01,
  -8.705591513769932e-01,   4.920637803717820e-01,
  -8.704647833253977e-01,   4.922306989514860e-01,
  -8.703703832693003e-01,   4.923975994332744e-01,
  -8.702759512121719e-01,   4.925644818110106e-01,
  -8.701814871574846e-01,   4.927313460785588e-01,
  -8.700869911087115e-01,   4.928981922297840e-01,
  -8.699924630693269e-01,   4.930650202585517e-01,
  -8.698979030428063e-01,   4.932318301587279e-01,
  -8.698033110326266e-01,   4.933986219241798e-01,
  -8.697086870422657e-01,   4.935653955487748e-01,
  -8.696140310752023e-01,   4.937321510263810e-01,
  -8.695193431349169e-01,   4.938988883508675e-01,
  -8.694246232248909e-01,   4.940656075161036e-01,
  -8.693298713486068e-01,   4.942323085159597e-01,
  -8.692350875095484e-01,   4.943989913443066e-01,
  -8.691402717112006e-01,   4.945656559950160e-01,
  -8.690454239570495e-01,   4.947323024619599e-01,
  -8.689505442505824e-01,   4.948989307390113e-01,
  -8.688556325952879e-01,   4.950655408200436e-01,
  -8.687606889946553e-01,   4.952321326989312e-01,
  -8.686657134521757e-01,   4.953987063695490e-01,
  -8.685707059713409e-01,   4.955652618257725e-01,
  -8.684756665556441e-01,   4.957317990614780e-01,
  -8.683805952085798e-01,   4.958983180705422e-01,
  -8.682854919336433e-01,   4.960648188468429e-01,
  -8.681903567343313e-01,   4.962313013842582e-01,
  -8.680951896141417e-01,   4.963977656766672e-01,
  -8.679999905765735e-01,   4.965642117179493e-01,
  -8.679047596251269e-01,   4.967306395019848e-01,
  -8.678094967633033e-01,   4.968970490226545e-01,
  -8.677142019946051e-01,   4.970634402738403e-01,
  -8.676188753225362e-01,   4.972298132494242e-01,
  -8.675235167506015e-01,   4.973961679432893e-01,
  -8.674281262823069e-01,   4.975625043493191e-01,
  -8.673327039211598e-01,   4.977288224613979e-01,
  -8.672372496706684e-01,   4.978951222734109e-01,
  -8.671417635343425e-01,   4.980614037792434e-01,
  -8.670462455156926e-01,   4.982276669727819e-01,
  -8.669506956182309e-01,   4.983939118479132e-01,
  -8.668551138454704e-01,   4.985601383985251e-01,
  -8.667595002009254e-01,   4.987263466185059e-01,
  -8.666638546881111e-01,   4.988925365017446e-01,
  -8.665681773105445e-01,   4.990587080421309e-01,
  -8.664724680717430e-01,   4.992248612335551e-01,
  -8.663767269752258e-01,   4.993909960699082e-01,
  -8.662809540245130e-01,   4.995571125450818e-01,
  -8.661851492231258e-01,   4.997232106529685e-01,
  -8.660893125745868e-01,   4.998892903874613e-01,
  -8.659934440824195e-01,   5.000553517424539e-01,
  -8.658975437501488e-01,   5.002213947118407e-01,
  -8.658016115813008e-01,   5.003874192895166e-01,
  -8.657056475794024e-01,   5.005534254693774e-01,
  -8.656096517479820e-01,   5.007194132453199e-01,
  -8.655136240905691e-01,   5.008853826112407e-01,
  -8.654175646106944e-01,   5.010513335610380e-01,
  -8.653214733118898e-01,   5.012172660886100e-01,
  -8.652253501976882e-01,   5.013831801878558e-01,
  -8.651291952716238e-01,   5.015490758526754e-01,
  -8.650330085372319e-01,   5.017149530769691e-01,
  -8.649367899980490e-01,   5.018808118546383e-01,
  -8.648405396576129e-01,   5.020466521795847e-01,
  -8.647442575194624e-01,   5.022124740457108e-01,
  -8.646479435871375e-01,   5.023782774469198e-01,
  -8.645515978641793e-01,   5.025440623771157e-01,
  -8.644552203541304e-01,   5.027098288302030e-01,
  -8.643588110605340e-01,   5.028755768000870e-01,
  -8.642623699869350e-01,   5.030413062806735e-01,
  -8.641658971368793e-01,   5.032070172658689e-01,
  -8.640693925139138e-01,   5.033727097495810e-01,
  -8.639728561215868e-01,   5.035383837257176e-01,
  -8.638762879634475e-01,   5.037040391881871e-01,
  -8.637796880430467e-01,   5.038696761308990e-01,
  -8.636830563639358e-01,   5.040352945477632e-01,
  -8.635863929296681e-01,   5.042008944326903e-01,
  -8.634896977437971e-01,   5.043664757795920e-01,
  -8.633929708098784e-01,   5.045320385823803e-01,
  -8.632962121314682e-01,   5.046975828349676e-01,
  -8.631994217121242e-01,   5.048631085312676e-01,
  -8.631025995554049e-01,   5.050286156651941e-01,
  -8.630057456648703e-01,   5.051941042306622e-01,
  -8.629088600440814e-01,   5.053595742215873e-01,
  -8.628119426966003e-01,   5.055250256318854e-01,
  -8.627149936259907e-01,   5.056904584554734e-01,
  -8.626180128358167e-01,   5.058558726862689e-01,
  -8.625210003296445e-01,   5.060212683181897e-01,
  -8.624239561110406e-01,   5.061866453451552e-01,
  -8.623268801835731e-01,   5.063520037610848e-01,
  -8.622297725508112e-01,   5.065173435598985e-01,
  -8.621326332163254e-01,   5.066826647355176e-01,
  -8.620354621836872e-01,   5.068479672818632e-01,
  -8.619382594564693e-01,   5.070132511928582e-01,
  -8.618410250382453e-01,   5.071785164624252e-01,
  -8.617437589325907e-01,   5.073437630844879e-01,
  -8.616464611430813e-01,   5.075089910529709e-01,
  -8.615491316732947e-01,   5.076742003617989e-01,
  -8.614517705268093e-01,   5.078393910048977e-01,
  -8.613543777072049e-01,   5.080045629761940e-01,
  -8.612569532180622e-01,   5.081697162696146e-01,
  -8.611594970629634e-01,   5.083348508790874e-01,
  -8.610620092454915e-01,   5.084999667985409e-01,
  -8.609644897692310e-01,   5.086650640219040e-01,
  -8.608669386377673e-01,   5.088301425431070e-01,
  -8.607693558546872e-01,   5.089952023560801e-01,
  -8.606717414235784e-01,   5.091602434547546e-01,
  -8.605740953480300e-01,   5.093252658330625e-01,
  -8.604764176316321e-01,   5.094902694849364e-01,
  -8.603787082779761e-01,   5.096552544043093e-01,
  -8.602809672906545e-01,   5.098202205851154e-01,
  -8.601831946732610e-01,   5.099851680212895e-01,
  -8.600853904293901e-01,   5.101500967067668e-01,
  -8.599875545626382e-01,   5.103150066354832e-01,
  -8.598896870766023e-01,   5.104798978013757e-01,
  -8.597917879748807e-01,   5.106447701983816e-01,
  -8.596938572610726e-01,   5.108096238204390e-01,
  -8.595958949387791e-01,   5.109744586614868e-01,
  -8.594979010116017e-01,   5.111392747154644e-01,
  -8.593998754831434e-01,   5.113040719763120e-01,
  -8.593018183570085e-01,   5.114688504379703e-01,
  -8.592037296368019e-01,   5.116336100943812e-01,
  -8.591056093261304e-01,   5.117983509394869e-01,
  -8.590074574286015e-01,   5.119630729672302e-01,
  -8.589092739478239e-01,   5.121277761715547e-01,
  -8.588110588874076e-01,   5.122924605464049e-01,
  -8.587128122509635e-01,   5.124571260857257e-01,
  -8.586145340421042e-01,   5.126217727834630e-01,
  -8.585162242644427e-01,   5.127864006335630e-01,
  -8.584178829215939e-01,   5.129510096299730e-01,
  -8.583195100171734e-01,   5.131155997666406e-01,
  -8.582211055547982e-01,   5.132801710375142e-01,
  -8.581226695380861e-01,   5.134447234365435e-01,
  -8.580242019706565e-01,   5.136092569576778e-01,
  -8.579257028561298e-01,   5.137737715948680e-01,
  -8.578271721981274e-01,   5.139382673420654e-01,
  -8.577286100002721e-01,   5.141027441932217e-01,
  -8.576300162661876e-01,   5.142672021422897e-01,
  -8.575313909994992e-01,   5.144316411832228e-01,
  -8.574327342038327e-01,   5.145960613099750e-01,
  -8.573340458828156e-01,   5.147604625165012e-01,
  -8.572353260400765e-01,   5.149248447967565e-01,
  -8.571365746792450e-01,   5.150892081446972e-01,
  -8.570377918039517e-01,   5.152535525542802e-01,
  -8.569389774178288e-01,   5.154178780194629e-01,
  -8.568401315245092e-01,   5.155821845342038e-01,
  -8.567412541276275e-01,   5.157464720924614e-01,
  -8.566423452308188e-01,   5.159107406881956e-01,
  -8.565434048377200e-01,   5.160749903153666e-01,
  -8.564444329519686e-01,   5.162392209679355e-01,
  -8.563454295772036e-01,   5.164034326398640e-01,
  -8.562463947170652e-01,   5.165676253251144e-01,
  -8.561473283751945e-01,   5.167317990176499e-01,
  -8.560482305552339e-01,   5.168959537114342e-01,
  -8.559491012608269e-01,   5.170600894004319e-01,
  -8.558499404956182e-01,   5.172242060786083e-01,
  -8.557507482632539e-01,   5.173883037399291e-01,
  -8.556515245673807e-01,   5.175523823783609e-01,
  -8.555522694116469e-01,   5.177164419878711e-01,
  -8.554529827997018e-01,   5.178804825624277e-01,
  -8.553536647351960e-01,   5.180445040959993e-01,
  -8.552543152217810e-01,   5.182085065825555e-01,
  -8.551549342631096e-01,   5.183724900160661e-01,
  -8.550555218628360e-01,   5.185364543905022e-01,
  -8.549560780246149e-01,   5.187003996998350e-01,
  -8.548566027521028e-01,   5.188643259380369e-01,
  -8.547570960489572e-01,   5.190282330990809e-01,
  -8.546575579188365e-01,   5.191921211769402e-01,
  -8.545579883654005e-01,   5.193559901655896e-01,
  -8.544583873923102e-01,   5.195198400590038e-01,
  -8.543587550032274e-01,   5.196836708511584e-01,
  -8.542590912018155e-01,   5.198474825360302e-01,
  -8.541593959917388e-01,   5.200112751075960e-01,
  -8.540596693766628e-01,   5.201750485598338e-01,
  -8.539599113602542e-01,   5.203388028867220e-01,
  -8.538601219461808e-01,   5.205025380822397e-01,
  -8.537603011381114e-01,   5.206662541403672e-01,
  -8.536604489397164e-01,   5.208299510550847e-01,
  -8.535605653546668e-01,   5.209936288203739e-01,
  -8.534606503866353e-01,   5.211572874302166e-01,
  -8.533607040392954e-01,   5.213209268785957e-01,
  -8.532607263163219e-01,   5.214845471594943e-01,
  -8.531607172213904e-01,   5.216481482668971e-01,
  -8.530606767581783e-01,   5.218117301947885e-01,
  -8.529606049303636e-01,   5.219752929371544e-01,
  -8.528605017416258e-01,   5.221388364879808e-01,
  -8.527603671956453e-01,   5.223023608412546e-01,
  -8.526602012961038e-01,   5.224658659909638e-01,
  -8.525600040466841e-01,   5.226293519310966e-01,
  -8.524597754510701e-01,   5.227928186556421e-01,
  -8.523595155129471e-01,   5.229562661585901e-01,
  -8.522592242360011e-01,   5.231196944339312e-01,
  -8.521589016239198e-01,   5.232831034756564e-01,
  -8.520585476803917e-01,   5.234464932777578e-01,
  -8.519581624091064e-01,   5.236098638342279e-01,
  -8.518577458137548e-01,   5.237732151390602e-01,
  -8.517572978980291e-01,   5.239365471862486e-01,
  -8.516568186656224e-01,   5.240998599697877e-01,
  -8.515563081202290e-01,   5.242631534836734e-01,
  -8.514557662655443e-01,   5.244264277219014e-01,
  -8.513551931052652e-01,   5.245896826784690e-01,
  -8.512545886430891e-01,   5.247529183473734e-01,
  -8.511539528827153e-01,   5.249161347226130e-01,
  -8.510532858278438e-01,   5.250793317981868e-01,
  -8.509525874821757e-01,   5.252425095680947e-01,
  -8.508518578494135e-01,   5.254056680263369e-01,
  -8.507510969332608e-01,   5.255688071669147e-01,
  -8.506503047374221e-01,   5.257319269838298e-01,
  -8.505494812656035e-01,   5.258950274710846e-01,
  -8.504486265215118e-01,   5.260581086226828e-01,
  -8.503477405088550e-01,   5.262211704326281e-01,
  -8.502468232313427e-01,   5.263842128949251e-01,
  -8.501458746926852e-01,   5.265472360035794e-01,
  -8.500448948965942e-01,   5.267102397525970e-01,
  -8.499438838467822e-01,   5.268732241359846e-01,
  -8.498428415469633e-01,   5.270361891477501e-01,
  -8.497417680008525e-01,   5.271991347819013e-01,
  -8.496406632121659e-01,   5.273620610324475e-01,
  -8.495395271846209e-01,   5.275249678933982e-01,
  -8.494383599219361e-01,   5.276878553587637e-01,
  -8.493371614278308e-01,   5.278507234225552e-01,
  -8.492359317060260e-01,   5.280135720787846e-01,
  -8.491346707602436e-01,   5.281764013214644e-01,
  -8.490333785942068e-01,   5.283392111446077e-01,
  -8.489320552116396e-01,   5.285020015422285e-01,
  -8.488307006162675e-01,   5.286647725083413e-01,
  -8.487293148118171e-01,   5.288275240369619e-01,
  -8.486278978020159e-01,   5.289902561221060e-01,
  -8.485264495905926e-01,   5.291529687577906e-01,
  -8.484249701812776e-01,   5.293156619380333e-01,
  -8.483234595778016e-01,   5.294783356568520e-01,
  -8.482219177838970e-01,   5.296409899082659e-01,
  -8.481203448032972e-01,   5.298036246862946e-01,
  -8.480187406397368e-01,   5.299662399849586e-01,
  -8.479171052969514e-01,   5.301288357982790e-01,
  -8.478154387786779e-01,   5.302914121202773e-01,
  -8.477137410886544e-01,   5.304539689449763e-01,
  -8.476120122306197e-01,   5.306165062663993e-01,
  -8.475102522083143e-01,   5.307790240785701e-01,
  -8.474084610254797e-01,   5.309415223755136e-01,
  -8.473066386858583e-01,   5.311040011512550e-01,
  -8.472047851931941e-01,   5.312664603998204e-01,
  -8.471029005512315e-01,   5.314289001152368e-01,
  -8.470009847637169e-01,   5.315913202915317e-01,
  -8.468990378343972e-01,   5.317537209227333e-01,
  -8.467970597670209e-01,   5.319161020028706e-01,
  -8.466950505653374e-01,   5.320784635259735e-01,
  -8.465930102330972e-01,   5.322408054860722e-01,
  -8.464909387740521e-01,   5.324031278771979e-01,
  -8.463888361919549e-01,   5.325654306933826e-01,
  -8.462867024905597e-01,   5.327277139286588e-01,
  -8.461845376736216e-01,   5.328899775770598e-01,
  -8.460823417448970e-01,   5.330522216326194e-01,
  -8.459801147081433e-01,   5.332144460893730e-01,
  -8.458778565671190e-01,   5.333766509413553e-01,
  -8.457755673255840e-01,   5.335388361826031e-01,
  -8.456732469872991e-01,   5.337010018071530e-01,
  -8.455708955560263e-01,   5.338631478090426e-01,
  -8.454685130355288e-01,   5.340252741823104e-01,
  -8.453660994295710e-01,   5.341873809209954e-01,
  -8.452636547419182e-01,   5.343494680191375e-01,
  -8.451611789763371e-01,   5.345115354707771e-01,
  -8.450586721365955e-01,   5.346735832699555e-01,
  -8.449561342264622e-01,   5.348356114107146e-01,
  -8.448535652497071e-01,   5.349976198870972e-01,
  -8.447509652101015e-01,   5.351596086931466e-01,
  -8.446483341114178e-01,   5.353215778229071e-01,
  -8.445456719574292e-01,   5.354835272704234e-01,
  -8.444429787519107e-01,   5.356454570297411e-01,
  -8.443402544986376e-01,   5.358073670949064e-01,
  -8.442374992013870e-01,   5.359692574599667e-01,
  -8.441347128639369e-01,   5.361311281189695e-01,
  -8.440318954900664e-01,   5.362929790659632e-01,
  -8.439290470835559e-01,   5.364548102949971e-01,
  -8.438261676481867e-01,   5.366166218001210e-01,
  -8.437232571877417e-01,   5.367784135753859e-01,
  -8.436203157060042e-01,   5.369401856148429e-01,
  -8.435173432067592e-01,   5.371019379125441e-01,
  -8.434143396937928e-01,   5.372636704625425e-01,
  -8.433113051708921e-01,   5.374253832588916e-01,
  -8.432082396418454e-01,   5.375870762956454e-01,
  -8.431051431104422e-01,   5.377487495668594e-01,
  -8.430020155804729e-01,   5.379104030665889e-01,
  -8.428988570557293e-01,   5.380720367888906e-01,
  -8.427956675400041e-01,   5.382336507278217e-01,
  -8.426924470370917e-01,   5.383952448774400e-01,
  -8.425891955507867e-01,   5.385568192318041e-01,
  -8.424859130848856e-01,   5.387183737849736e-01,
  -8.423825996431858e-01,   5.388799085310084e-01,
  -8.422792552294860e-01,   5.390414234639694e-01,
  -8.421758798475856e-01,   5.392029185779182e-01,
  -8.420724735012856e-01,   5.393643938669170e-01,
  -8.419690361943877e-01,   5.395258493250289e-01,
  -8.418655679306953e-01,   5.396872849463176e-01,
  -8.417620687140125e-01,   5.398487007248476e-01,
  -8.416585385481448e-01,   5.400100966546840e-01,
  -8.415549774368984e-01,   5.401714727298929e-01,
  -8.414513853840813e-01,   5.403328289445407e-01,
  -8.413477623935020e-01,   5.404941652926952e-01,
  -8.412441084689706e-01,   5.406554817684242e-01,
  -8.411404236142981e-01,   5.408167783657967e-01,
  -8.410367078332966e-01,   5.409780550788821e-01,
  -8.409329611297798e-01,   5.411393119017508e-01,
  -8.408291835075616e-01,   5.413005488284741e-01,
  -8.407253749704581e-01,   5.414617658531234e-01,
  -8.406215355222857e-01,   5.416229629697715e-01,
  -8.405176651668625e-01,   5.417841401724915e-01,
  -8.404137639080075e-01,   5.419452974553574e-01,
  -8.403098317495408e-01,   5.421064348124439e-01,
  -8.402058686952836e-01,   5.422675522378265e-01,
  -8.401018747490584e-01,   5.424286497255812e-01,
  -8.399978499146888e-01,   5.425897272697853e-01,
  -8.398937941959995e-01,   5.427507848645159e-01,
  -8.397897075968164e-01,   5.429118225038517e-01,
  -8.396855901209661e-01,   5.430728401818717e-01,
  -8.395814417722771e-01,   5.432338378926559e-01,
  -8.394772625545786e-01,   5.433948156302848e-01,
  -8.393730524717007e-01,   5.435557733888395e-01,
  -8.392688115274752e-01,   5.437167111624023e-01,
  -8.391645397257347e-01,   5.438776289450560e-01,
  -8.390602370703127e-01,   5.440385267308838e-01,
  -8.389559035650445e-01,   5.441994045139703e-01,
  -8.388515392137658e-01,   5.443602622884004e-01,
  -8.387471440203139e-01,   5.445211000482596e-01,
  -8.386427179885273e-01,   5.446819177876345e-01,
  -8.385382611222453e-01,   5.448427155006124e-01,
  -8.384337734253083e-01,   5.450034931812812e-01,
  -8.383292549015583e-01,   5.451642508237293e-01,
  -8.382247055548381e-01,   5.453249884220465e-01,
  -8.381201253889915e-01,   5.454857059703225e-01,
  -8.380155144078638e-01,   5.456464034626486e-01,
  -8.379108726153012e-01,   5.458070808931161e-01,
  -8.378062000151509e-01,   5.459677382558176e-01,
  -8.377014966112617e-01,   5.461283755448460e-01,
  -8.375967624074830e-01,   5.462889927542952e-01,
  -8.374919974076659e-01,   5.464495898782596e-01,
  -8.373872016156619e-01,   5.466101669108349e-01,
  -8.372823750353243e-01,   5.467707238461168e-01,
  -8.371775176705073e-01,   5.469312606782022e-01,
  -8.370726295250660e-01,   5.470917774011885e-01,
  -8.369677106028570e-01,   5.472522740091741e-01,
  -8.368627609077379e-01,   5.474127504962579e-01,
  -8.367577804435672e-01,   5.475732068565398e-01,
  -8.366527692142050e-01,   5.477336430841201e-01,
  -8.365477272235120e-01,   5.478940591731002e-01,
  -8.364426544753504e-01,   5.480544551175819e-01,
  -8.363375509735835e-01,   5.482148309116678e-01,
  -8.362324167220756e-01,   5.483751865494616e-01,
  -8.361272517246923e-01,   5.485355220250674e-01,
  -8.360220559852999e-01,   5.486958373325901e-01,
  -8.359168295077664e-01,   5.488561324661353e-01,
  -8.358115722959607e-01,   5.490164074198094e-01,
  -8.357062843537526e-01,   5.491766621877197e-01,
  -8.356009656850134e-01,   5.493368967639740e-01,
  -8.354956162936154e-01,   5.494971111426810e-01,
  -8.353902361834319e-01,   5.496573053179499e-01,
  -8.352848253583374e-01,   5.498174792838909e-01,
  -8.351793838222077e-01,   5.499776330346149e-01,
  -8.350739115789194e-01,   5.501377665642336e-01,
  -8.349684086323504e-01,   5.502978798668592e-01,
  -8.348628749863800e-01,   5.504579729366048e-01,
  -8.347573106448882e-01,   5.506180457675843e-01,
  -8.346517156117564e-01,   5.507780983539121e-01,
  -8.345460898908669e-01,   5.509381306897039e-01,
  -8.344404334861032e-01,   5.510981427690754e-01,
  -8.343347464013501e-01,   5.512581345861436e-01,
  -8.342290286404934e-01,   5.514181061350261e-01,
  -8.341232802074201e-01,   5.515780574098410e-01,
  -8.340175011060181e-01,   5.517379884047073e-01,
  -8.339116913401768e-01,   5.518978991137452e-01,
  -8.338058509137863e-01,   5.520577895310750e-01,
  -8.336999798307383e-01,   5.522176596508179e-01,
  -8.335940780949251e-01,   5.523775094670961e-01,
  -8.334881457102408e-01,   5.525373389740321e-01,
  -8.333821826805797e-01,   5.526971481657498e-01,
  -8.332761890098382e-01,   5.528569370363733e-01,
  -8.331701647019132e-01,   5.530167055800275e-01,
  -8.330641097607029e-01,   5.531764537908384e-01,
  -8.329580241901067e-01,   5.533361816629323e-01,
  -8.328519079940251e-01,   5.534958891904366e-01,
  -8.327457611763595e-01,   5.536555763674793e-01,
  -8.326395837410128e-01,   5.538152431881891e-01,
  -8.325333756918887e-01,   5.539748896466955e-01,
  -8.324271370328923e-01,   5.541345157371289e-01,
  -8.323208677679297e-01,   5.542941214536200e-01,
  -8.322145679009080e-01,   5.544537067903009e-01,
  -8.321082374357356e-01,   5.546132717413040e-01,
  -8.320018763763220e-01,   5.547728163007625e-01,
  -8.318954847265776e-01,   5.549323404628104e-01,
  -8.317890624904144e-01,   5.550918442215824e-01,
  -8.316826096717451e-01,   5.552513275712140e-01,
  -8.315761262744837e-01,   5.554107905058416e-01,
  -8.314696123025452e-01,   5.555702330196022e-01,
  -8.313630677598459e-01,   5.557296551066334e-01,
  -8.312564926503032e-01,   5.558890567610738e-01,
  -8.311498869778355e-01,   5.560484379770626e-01,
  -8.310432507463623e-01,   5.562077987487399e-01,
  -8.309365839598044e-01,   5.563671390702464e-01,
  -8.308298866220836e-01,   5.565264589357236e-01,
  -8.307231587371229e-01,   5.566857583393139e-01,
  -8.306164003088463e-01,   5.568450372751601e-01,
  -8.305096113411791e-01,   5.570042957374060e-01,
  -8.304027918380475e-01,   5.571635337201962e-01,
  -8.302959418033791e-01,   5.573227512176762e-01,
  -8.301890612411024e-01,   5.574819482239916e-01,
  -8.300821501551470e-01,   5.576411247332894e-01,
  -8.299752085494440e-01,   5.578002807397170e-01,
  -8.298682364279248e-01,   5.579594162374230e-01,
  -8.297612337945230e-01,   5.581185312205561e-01,
  -8.296542006531726e-01,   5.582776256832663e-01,
  -8.295471370078089e-01,   5.584366996197041e-01,
  -8.294400428623682e-01,   5.585957530240208e-01,
  -8.293329182207883e-01,   5.587547858903683e-01,
  -8.292257630870076e-01,   5.589137982128998e-01,
  -8.291185774649660e-01,   5.590727899857685e-01,
  -8.290113613586044e-01,   5.592317612031289e-01,
  -8.289041147718649e-01,   5.593907118591361e-01,
  -8.287968377086906e-01,   5.595496419479458e-01,
  -8.286895301730258e-01,   5.597085514637147e-01,
  -8.285821921688158e-01,   5.598674404006002e-01,
  -8.284748237000071e-01,   5.600263087527604e-01,
  -8.283674247705475e-01,   5.601851565143541e-01,
  -8.282599953843857e-01,   5.603439836795409e-01,
  -8.281525355454714e-01,   5.605027902424811e-01,
  -8.280450452577558e-01,   5.606615761973360e-01,
  -8.279375245251909e-01,   5.608203415382674e-01,
  -8.278299733517299e-01,   5.609790862594382e-01,
  -8.277223917413272e-01,   5.611378103550114e-01,
  -8.276147796979384e-01,   5.612965138191515e-01,
  -8.275071372255198e-01,   5.614551966460233e-01,
  -8.273994643280295e-01,   5.616138588297924e-01,
  -8.272917610094258e-01,   5.617725003646253e-01,
  -8.271840272736691e-01,   5.619311212446895e-01,
  -8.270762631247203e-01,   5.620897214641525e-01,
  -8.269684685665416e-01,   5.622483010171831e-01,
  -8.268606436030962e-01,   5.624068598979511e-01,
  -8.267527882383485e-01,   5.625653981006266e-01,
  -8.266449024762643e-01,   5.627239156193804e-01,
  -8.265369863208100e-01,   5.628824124483844e-01,
  -8.264290397759535e-01,   5.630408885818112e-01,
  -8.263210628456635e-01,   5.631993440138341e-01,
  -8.262130555339102e-01,   5.633577787386270e-01,
  -8.261050178446646e-01,   5.635161927503648e-01,
  -8.259969497818991e-01,   5.636745860432231e-01,
  -8.258888513495868e-01,   5.638329586113782e-01,
  -8.257807225517024e-01,   5.639913104490070e-01,
  -8.256725633922214e-01,   5.641496415502877e-01,
  -8.255643738751205e-01,   5.643079519093986e-01,
  -8.254561540043776e-01,   5.644662415205195e-01,
  -8.253479037839714e-01,   5.646245103778301e-01,
  -8.252396232178822e-01,   5.647827584755114e-01,
  -8.251313123100911e-01,   5.649409858077452e-01,
  -8.250229710645802e-01,   5.650991923687140e-01,
  -8.249145994853332e-01,   5.652573781526008e-01,
  -8.248061975763343e-01,   5.654155431535897e-01,
  -8.246977653415695e-01,   5.655736873658653e-01,
  -8.245893027850253e-01,   5.657318107836131e-01,
  -8.244808099106895e-01,   5.658899134010196e-01,
  -8.243722867225513e-01,   5.660479952122714e-01,
  -8.242637332246006e-01,   5.662060562115567e-01,
  -8.241551494208286e-01,   5.663640963930638e-01,
  -8.240465353152278e-01,   5.665221157509821e-01,
  -8.239378909117914e-01,   5.666801142795016e-01,
  -8.238292162145140e-01,   5.668380919728133e-01,
  -8.237205112273914e-01,   5.669960488251087e-01,
  -8.236117759544203e-01,   5.671539848305801e-01,
  -8.235030103995985e-01,   5.673118999834208e-01,
  -8.233942145669251e-01,   5.674697942778245e-01,
  -8.232853884604001e-01,   5.676276677079862e-01,
  -8.231765320840249e-01,   5.677855202681011e-01,
  -8.230676454418017e-01,   5.679433519523656e-01,
  -8.229587285377340e-01,   5.681011627549765e-01,
  -8.228497813758264e-01,   5.682589526701315e-01,
  -8.227408039600844e-01,   5.684167216920293e-01,
  -8.226317962945150e-01,   5.685744698148691e-01,
  -8.225227583831259e-01,   5.687321970328510e-01,
  -8.224136902299264e-01,   5.688899033401759e-01,
  -8.223045918389263e-01,   5.690475887310451e-01,
  -8.221954632141372e-01,   5.692052531996612e-01,
  -8.220863043595711e-01,   5.693628967402272e-01,
  -8.219771152792416e-01,   5.695205193469471e-01,
  -8.218678959771633e-01,   5.696781210140256e-01,
  -8.217586464573517e-01,   5.698357017356680e-01,
  -8.216493667238239e-01,   5.699932615060805e-01,
  -8.215400567805976e-01,   5.701508003194703e-01,
  -8.214307166316919e-01,   5.703083181700449e-01,
  -8.213213462811267e-01,   5.704658150520130e-01,
  -8.212119457329236e-01,   5.706232909595838e-01,
  -8.211025149911046e-01,   5.707807458869673e-01,
  -8.209930540596936e-01,   5.709381798283744e-01,
  -8.208835629427146e-01,   5.710955927780167e-01,
  -8.207740416441937e-01,   5.712529847301067e-01,
  -8.206644901681575e-01,   5.714103556788572e-01,
  -8.205549085186339e-01,   5.715677056184826e-01,
  -8.204452966996520e-01,   5.717250345431971e-01,
  -8.203356547152418e-01,   5.718823424472166e-01,
  -8.202259825694347e-01,   5.720396293247570e-01,
  -8.201162802662628e-01,   5.721968951700356e-01,
  -8.200065478097597e-01,   5.723541399772699e-01,
  -8.198967852039598e-01,   5.725113637406788e-01,
  -8.197869924528990e-01,   5.726685664544812e-01,
  -8.196771695606139e-01,   5.728257481128975e-01,
  -8.195673165311422e-01,   5.729829087101486e-01,
  -8.194574333685233e-01,   5.731400482404560e-01,
  -8.193475200767969e-01,   5.732971666980422e-01,
  -8.192375766600045e-01,   5.734542640771304e-01,
  -8.191276031221882e-01,   5.736113403719446e-01,
  -8.190175994673915e-01,   5.737683955767096e-01,
  -8.189075656996590e-01,   5.739254296856507e-01,
  -8.187975018230360e-01,   5.740824426929945e-01,
  -8.186874078415697e-01,   5.742394345929679e-01,
  -8.185772837593076e-01,   5.743964053797987e-01,
  -8.184671295802987e-01,   5.745533550477158e-01,
  -8.183569453085932e-01,   5.747102835909483e-01,
  -8.182467309482421e-01,   5.748671910037267e-01,
  -8.181364865032977e-01,   5.750240772802817e-01,
  -8.180262119778134e-01,   5.751809424148451e-01,
  -8.179159073758439e-01,   5.753377864016495e-01,
  -8.178055727014443e-01,   5.754946092349281e-01,
  -8.176952079586717e-01,   5.756514109089151e-01,
  -8.175848131515837e-01,   5.758081914178453e-01,
  -8.174743882842392e-01,   5.759649507559542e-01,
  -8.173639333606985e-01,   5.761216889174783e-01,
  -8.172534483850223e-01,   5.762784058966549e-01,
  -8.171429333612730e-01,   5.764351016877218e-01,
  -8.170323882935139e-01,   5.765917762849178e-01,
  -8.169218131858095e-01,   5.767484296824824e-01,
  -8.168112080422253e-01,   5.769050618746560e-01,
  -8.167005728668278e-01,   5.770616728556794e-01,
  -8.165899076636849e-01,   5.772182626197949e-01,
  -8.164792124368654e-01,   5.773748311612449e-01,
  -8.163684871904392e-01,   5.775313784742727e-01,
  -8.162577319284774e-01,   5.776879045531228e-01,
  -8.161469466550523e-01,   5.778444093920398e-01,
  -8.160361313742368e-01,   5.780008929852699e-01,
  -8.159252860901055e-01,   5.781573553270594e-01,
  -8.158144108067338e-01,   5.783137964116556e-01,
  -8.157035055281983e-01,   5.784702162333066e-01,
  -8.155925702585768e-01,   5.786266147862614e-01,
  -8.154816050019478e-01,   5.787829920647697e-01,
  -8.153706097623913e-01,   5.789393480630818e-01,
  -8.152595845439883e-01,   5.790956827754491e-01,
  -8.151485293508208e-01,   5.792519961961236e-01,
  -8.150374441869722e-01,   5.794082883193579e-01,
  -8.149263290565266e-01,   5.795645591394056e-01,
  -8.148151839635694e-01,   5.797208086505214e-01,
  -8.147040089121871e-01,   5.798770368469603e-01,
  -8.145928039064673e-01,   5.800332437229782e-01,
  -8.144815689504986e-01,   5.801894292728317e-01,
  -8.143703040483711e-01,   5.803455934907783e-01,
  -8.142590092041753e-01,   5.805017363710765e-01,
  -8.141476844220034e-01,   5.806578579079853e-01,
  -8.140363297059484e-01,   5.808139580957645e-01,
  -8.139249450601046e-01,   5.809700369286748e-01,
  -8.138135304885672e-01,   5.811260944009776e-01,
  -8.137020859954327e-01,   5.812821305069350e-01,
  -8.135906115847985e-01,   5.814381452408102e-01,
  -8.134791072607632e-01,   5.815941385968669e-01,
  -8.133675730274266e-01,   5.817501105693696e-01,
  -8.132560088888894e-01,   5.819060611525838e-01,
  -8.131444148492536e-01,   5.820619903407754e-01,
  -8.130327909126220e-01,   5.822178981282117e-01,
  -8.129211370830988e-01,   5.823737845091601e-01,
  -8.128094533647893e-01,   5.825296494778893e-01,
  -8.126977397617995e-01,   5.826854930286685e-01,
  -8.125859962782371e-01,   5.828413151557676e-01,
  -8.124742229182105e-01,   5.829971158534577e-01,
  -8.123624196858292e-01,   5.831528951160104e-01,
  -8.122505865852039e-01,   5.833086529376983e-01,
  -8.121387236204465e-01,   5.834643893127943e-01,
  -8.120268307956697e-01,   5.836201042355728e-01,
  -8.119149081149878e-01,   5.837757977003081e-01,
  -8.118029555825155e-01,   5.839314697012762e-01,
  -8.116909732023690e-01,   5.840871202327534e-01,
  -8.115789609786659e-01,   5.842427492890170e-01,
  -8.114669189155242e-01,   5.843983568643446e-01,
  -8.113548470170637e-01,   5.845539429530153e-01,
  -8.112427452874048e-01,   5.847095075493084e-01,
  -8.111306137306692e-01,   5.848650506475045e-01,
  -8.110184523509795e-01,   5.850205722418845e-01,
  -8.109062611524597e-01,   5.851760723267304e-01,
  -8.107940401392347e-01,   5.853315508963249e-01,
  -8.106817893154308e-01,   5.854870079449513e-01,
  -8.105695086851746e-01,   5.856424434668944e-01,
  -8.104571982525948e-01,   5.857978574564389e-01,
  -8.103448580218205e-01,   5.859532499078706e-01,
  -8.102324879969823e-01,   5.861086208154764e-01,
  -8.101200881822116e-01,   5.862639701735436e-01,
  -8.100076585816411e-01,   5.864192979763605e-01,
  -8.098951991994044e-01,   5.865746042182162e-01,
  -8.097827100396365e-01,   5.867298888934004e-01,
  -8.096701911064731e-01,   5.868851519962040e-01,
  -8.095576424040513e-01,   5.870403935209180e-01,
  -8.094450639365092e-01,   5.871956134618348e-01,
  -8.093324557079860e-01,   5.873508118132477e-01,
  -8.092198177226217e-01,   5.875059885694500e-01,
  -8.091071499845582e-01,   5.876611437247367e-01,
  -8.089944524979377e-01,   5.878162772734029e-01,
  -8.088817252669036e-01,   5.879713892097450e-01,
  -8.087689682956009e-01,   5.881264795280599e-01,
  -8.086561815881750e-01,   5.882815482226452e-01,
  -8.085433651487730e-01,   5.884365952877998e-01,
  -8.084305189815427e-01,   5.885916207178229e-01,
  -8.083176430906333e-01,   5.887466245070145e-01,
  -8.082047374801947e-01,   5.889016066496757e-01,
  -8.080918021543784e-01,   5.890565671401085e-01,
  -8.079788371173363e-01,   5.892115059726150e-01,
  -8.078658423732221e-01,   5.893664231414988e-01,
  -8.077528179261904e-01,   5.895213186410639e-01,
  -8.076397637803965e-01,   5.896761924656154e-01,
  -8.075266799399972e-01,   5.898310446094588e-01,
  -8.074135664091502e-01,   5.899858750669009e-01,
  -8.073004231920144e-01,   5.901406838322488e-01,
  -8.071872502927500e-01,   5.902954708998108e-01,
  -8.070740477155176e-01,   5.904502362638958e-01,
  -8.069608154644797e-01,   5.906049799188133e-01,
  -8.068475535437993e-01,   5.907597018588742e-01,
  -8.067342619576409e-01,   5.909144020783895e-01,
  -8.066209407101697e-01,   5.910690805716714e-01,
  -8.065075898055523e-01,   5.912237373330329e-01,
  -8.063942092479562e-01,   5.913783723567876e-01,
  -8.062807990415505e-01,   5.915329856372500e-01,
  -8.061673591905044e-01,   5.916875771687354e-01,
  -8.060538896989891e-01,   5.918421469455601e-01,
  -8.059403905711763e-01,   5.919966949620410e-01,
  -8.058268618112393e-01,   5.921512212124955e-01,
  -8.057133034233522e-01,   5.923057256912423e-01,
  -8.055997154116901e-01,   5.924602083926008e-01,
  -8.054860977804292e-01,   5.926146693108911e-01,
  -8.053724505337471e-01,   5.927691084404341e-01,
  -8.052587736758222e-01,   5.929235257755513e-01,
  -8.051450672108342e-01,   5.930779213105655e-01,
  -8.050313311429637e-01,   5.932322950397998e-01,
  -8.049175654763923e-01,   5.933866469575785e-01,
  -8.048037702153029e-01,   5.935409770582264e-01,
  -8.046899453638795e-01,   5.936952853360692e-01,
  -8.045760909263071e-01,   5.938495717854336e-01,
  -8.044622069067718e-01,   5.940038364006467e-01,
  -8.043482933094608e-01,   5.941580791760368e-01,
  -8.042343501385624e-01,   5.943123001059328e-01,
  -8.041203773982658e-01,   5.944664991846644e-01,
  -8.040063750927615e-01,   5.946206764065622e-01,
  -8.038923432262413e-01,   5.947748317659576e-01,
  -8.037782818028976e-01,   5.949289652571824e-01,
  -8.036641908269241e-01,   5.950830768745700e-01,
  -8.035500703025157e-01,   5.952371666124538e-01,
  -8.034359202338681e-01,   5.953912344651687e-01,
  -8.033217406251786e-01,   5.955452804270498e-01,
  -8.032075314806449e-01,   5.956993044924334e-01,
  -8.030932928044664e-01,   5.958533066556563e-01,
  -8.029790246008432e-01,   5.960072869110565e-01,
  -8.028647268739767e-01,   5.961612452529725e-01,
  -8.027503996280692e-01,   5.963151816757437e-01,
  -8.026360428673242e-01,   5.964690961737104e-01,
  -8.025216565959464e-01,   5.966229887412132e-01,
  -8.024072408181413e-01,   5.967768593725944e-01,
  -8.022927955381157e-01,   5.969307080621965e-01,
  -8.021783207600774e-01,   5.970845348043627e-01,
  -8.020638164882354e-01,   5.972383395934374e-01,
  -8.019492827267998e-01,   5.973921224237657e-01,
  -8.018347194799813e-01,   5.975458832896932e-01,
  -8.017201267519923e-01,   5.976996221855668e-01,
  -8.016055045470462e-01,   5.978533391057339e-01,
  -8.014908528693570e-01,   5.980070340445427e-01,
  -8.013761717231402e-01,   5.981607069963423e-01,
  -8.012614611126125e-01,   5.983143579554826e-01,
  -8.011467210419914e-01,   5.984679869163143e-01,
  -8.010319515154953e-01,   5.986215938731889e-01,
  -8.009171525373443e-01,   5.987751788204587e-01,
  -8.008023241117591e-01,   5.989287417524769e-01,
  -8.006874662429616e-01,   5.990822826635973e-01,
  -8.005725789351749e-01,   5.992358015481746e-01,
  -8.004576621926228e-01,   5.993892984005645e-01,
  -8.003427160195307e-01,   5.995427732151234e-01,
  -8.002277404201248e-01,   5.996962259862083e-01,
  -8.001127353986324e-01,   5.998496567081772e-01,
  -7.999977009592819e-01,   6.000030653753889e-01,
  -7.998826371063028e-01,   6.001564519822032e-01,
  -7.997675438439257e-01,   6.003098165229804e-01,
  -7.996524211763822e-01,   6.004631589920816e-01,
  -7.995372691079050e-01,   6.006164793838690e-01,
  -7.994220876427280e-01,   6.007697776927052e-01,
  -7.993068767850862e-01,   6.009230539129541e-01,
  -7.991916365392152e-01,   6.010763080389802e-01,
  -7.990763669093524e-01,   6.012295400651485e-01,
  -7.989610678997358e-01,   6.013827499858254e-01,
  -7.988457395146046e-01,   6.015359377953777e-01,
  -7.987303817581992e-01,   6.016891034881730e-01,
  -7.986149946347608e-01,   6.018422470585800e-01,
  -7.984995781485321e-01,   6.019953685009680e-01,
  -7.983841323037564e-01,   6.021484678097072e-01,
  -7.982686571046784e-01,   6.023015449791685e-01,
  -7.981531525555438e-01,   6.024546000037238e-01,
  -7.980376186605994e-01,   6.026076328777454e-01,
  -7.979220554240930e-01,   6.027606435956072e-01,
  -7.978064628502736e-01,   6.029136321516830e-01,
  -7.976908409433912e-01,   6.030665985403482e-01,
  -7.975751897076967e-01,   6.032195427559784e-01,
  -7.974595091474425e-01,   6.033724647929503e-01,
  -7.973437992668817e-01,   6.035253646456415e-01,
  -7.972280600702688e-01,   6.036782423084304e-01,
  -7.971122915618589e-01,   6.038310977756959e-01,
  -7.969964937459088e-01,   6.039839310418180e-01,
  -7.968806666266758e-01,   6.041367421011775e-01,
  -7.967648102084188e-01,   6.042895309481560e-01,
  -7.966489244953973e-01,   6.044422975771359e-01,
  -7.965330094918720e-01,   6.045950419825004e-01,
  -7.964170652021050e-01,   6.047477641586334e-01,
  -7.963010916303591e-01,   6.049004640999198e-01,
  -7.961850887808984e-01,   6.050531418007453e-01,
  -7.960690566579880e-01,   6.052057972554965e-01,
  -7.959529952658939e-01,   6.053584304585605e-01,
  -7.958369046088836e-01,   6.055110414043255e-01,
  -7.957207846912251e-01,   6.056636300871804e-01,
  -7.956046355171881e-01,   6.058161965015150e-01,
  -7.954884570910430e-01,   6.059687406417197e-01,
  -7.953722494170613e-01,   6.061212625021861e-01,
  -7.952560124995156e-01,   6.062737620773064e-01,
  -7.951397463426796e-01,   6.064262393614736e-01,
  -7.950234509508280e-01,   6.065786943490813e-01,
  -7.949071263282370e-01,   6.067311270345245e-01,
  -7.947907724791832e-01,   6.068835374121985e-01,
  -7.946743894079445e-01,   6.070359254764996e-01,
  -7.945579771188004e-01,   6.071882912218252e-01,
  -7.944415356160306e-01,   6.073406346425729e-01,
  -7.943250649039165e-01,   6.074929557331415e-01,
  -7.942085649867406e-01,   6.076452544879308e-01,
  -7.940920358687860e-01,   6.077975309013411e-01,
  -7.939754775543372e-01,   6.079497849677736e-01,
  -7.938588900476797e-01,   6.081020166816304e-01,
  -7.937422733531002e-01,   6.082542260373145e-01,
  -7.936256274748863e-01,   6.084064130292292e-01,
  -7.935089524173267e-01,   6.085585776517795e-01,
  -7.933922481847111e-01,   6.087107198993703e-01,
  -7.932755147813306e-01,   6.088628397664082e-01,
  -7.931587522114771e-01,   6.090149372472998e-01,
  -7.930419604794436e-01,   6.091670123364532e-01,
  -7.929251395895243e-01,   6.093190650282768e-01,
  -7.928082895460141e-01,   6.094710953171802e-01,
  -7.926914103532094e-01,   6.096231031975737e-01,
  -7.925745020154077e-01,   6.097750886638684e-01,
  -7.924575645369071e-01,   6.099270517104761e-01,
  -7.923405979220072e-01,   6.100789923318096e-01,
  -7.922236021750083e-01,   6.102309105222826e-01,
  -7.921065773002124e-01,   6.103828062763095e-01,
  -7.919895233019218e-01,   6.105346795883053e-01,
  -7.918724401844405e-01,   6.106865304526863e-01,
  -7.917553279520732e-01,   6.108383588638692e-01,
  -7.916381866091259e-01,   6.109901648162717e-01,
  -7.915210161599052e-01,   6.111419483043126e-01,
  -7.914038166087195e-01,   6.112937093224109e-01,
  -7.912865879598778e-01,   6.114454478649870e-01,
  -7.911693302176902e-01,   6.115971639264619e-01,
  -7.910520433864680e-01,   6.117488575012573e-01,
  -7.909347274705233e-01,   6.119005285837961e-01,
  -7.908173824741698e-01,   6.120521771685015e-01,
  -7.907000084017216e-01,   6.122038032497980e-01,
  -7.905826052574945e-01,   6.123554068221106e-01,
  -7.904651730458049e-01,   6.125069878798656e-01,
  -7.903477117709705e-01,   6.126585464174893e-01,
  -7.902302214373100e-01,   6.128100824294097e-01,
  -7.901127020491433e-01,   6.129615959100552e-01,
  -7.899951536107911e-01,   6.131130868538549e-01,
  -7.898775761265753e-01,   6.132645552552390e-01,
  -7.897599696008191e-01,   6.134160011086386e-01,
  -7.896423340378463e-01,   6.135674244084853e-01,
  -7.895246694419822e-01,   6.137188251492117e-01,
  -7.894069758175529e-01,   6.138702033252513e-01,
  -7.892892531688857e-01,   6.140215589310384e-01,
  -7.891715015003089e-01,   6.141728919610080e-01,
  -7.890537208161519e-01,   6.143242024095960e-01,
  -7.889359111207452e-01,   6.144754902712390e-01,
  -7.888180724184203e-01,   6.146267555403750e-01,
  -7.887002047135097e-01,   6.147779982114421e-01,
  -7.885823080103471e-01,   6.149292182788796e-01,
  -7.884643823132675e-01,   6.150804157371275e-01,
  -7.883464276266063e-01,   6.152315905806268e-01,
  -7.882284439547005e-01,   6.153827428038192e-01,
  -7.881104313018881e-01,   6.155338724011473e-01,
  -7.879923896725080e-01,   6.156849793670546e-01,
  -7.878743190709002e-01,   6.158360636959851e-01,
  -7.877562195014061e-01,   6.159871253823838e-01,
  -7.876380909683675e-01,   6.161381644206969e-01,
  -7.875199334761278e-01,   6.162891808053710e-01,
  -7.874017470290314e-01,   6.164401745308536e-01,
  -7.872835316314236e-01,   6.165911455915931e-01,
  -7.871652872876510e-01,   6.167420939820387e-01,
  -7.870470140020608e-01,   6.168930196966407e-01,
  -7.869287117790018e-01,   6.170439227298498e-01,
  -7.868103806228235e-01,   6.171948030761176e-01,
  -7.866920205378768e-01,   6.173456607298968e-01,
  -7.865736315285132e-01,   6.174964956856409e-01,
  -7.864552135990858e-01,   6.176473079378039e-01,
  -7.863367667539483e-01,   6.177980974808410e-01,
  -7.862182909974557e-01,   6.179488643092081e-01,
  -7.860997863339639e-01,   6.180996084173620e-01,
  -7.859812527678302e-01,   6.182503297997602e-01,
  -7.858626903034126e-01,   6.184010284508610e-01,
  -7.857440989450704e-01,   6.185517043651237e-01,
  -7.856254786971637e-01,   6.187023575370085e-01,
  -7.855068295640539e-01,   6.188529879609763e-01,
  -7.853881515501036e-01,   6.190035956314887e-01,
  -7.852694446596759e-01,   6.191541805430084e-01,
  -7.851507088971356e-01,   6.193047426899987e-01,
  -7.850319442668481e-01,   6.194552820669240e-01,
  -7.849131507731800e-01,   6.196057986682493e-01,
  -7.847943284204992e-01,   6.197562924884407e-01,
  -7.846754772131743e-01,   6.199067635219647e-01,
  -7.845565971555752e-01,   6.200572117632891e-01,
  -7.844376882520728e-01,   6.202076372068824e-01,
  -7.843187505070389e-01,   6.203580398472137e-01,
  -7.841997839248466e-01,   6.205084196787534e-01,
  -7.840807885098700e-01,   6.206587766959721e-01,
  -7.839617642664841e-01,   6.208091108933419e-01,
  -7.838427111990652e-01,   6.209594222653352e-01,
  -7.837236293119905e-01,   6.211097108064256e-01,
  -7.836045186096382e-01,   6.212599765110876e-01,
  -7.834853790963878e-01,   6.214102193737961e-01,
  -7.833662107766197e-01,   6.215604393890273e-01,
  -7.832470136547154e-01,   6.217106365512577e-01,
  -7.831277877350573e-01,   6.218608108549654e-01,
  -7.830085330220291e-01,   6.220109622946286e-01,
  -7.828892495200155e-01,   6.221610908647268e-01,
  -7.827699372334020e-01,   6.223111965597403e-01,
  -7.826505961665757e-01,   6.224612793741500e-01,
  -7.825312263239242e-01,   6.226113393024377e-01,
  -7.824118277098365e-01,   6.227613763390863e-01,
  -7.822924003287024e-01,   6.229113904785795e-01,
  -7.821729441849130e-01,   6.230613817154013e-01,
  -7.820534592828603e-01,   6.232113500440373e-01,
  -7.819339456269376e-01,   6.233612954589732e-01,
  -7.818144032215388e-01,   6.235112179546964e-01,
  -7.816948320710594e-01,   6.236611175256945e-01,
  -7.815752321798956e-01,   6.238109941664561e-01,
  -7.814556035524446e-01,   6.239608478714707e-01,
  -7.813359461931049e-01,   6.241106786352285e-01,
  -7.812162601062761e-01,   6.242604864522207e-01,
  -7.810965452963585e-01,   6.244102713169393e-01,
  -7.809768017677537e-01,   6.245600332238772e-01,
  -7.808570295248646e-01,   6.247097721675281e-01,
  -7.807372285720945e-01,   6.248594881423863e-01,
  -7.806173989138484e-01,   6.250091811429475e-01,
  -7.804975405545319e-01,   6.251588511637076e-01,
  -7.803776534985520e-01,   6.253084981991640e-01,
  -7.802577377503166e-01,   6.254581222438144e-01,
  -7.801377933142346e-01,   6.256077232921574e-01,
  -7.800178201947160e-01,   6.257573013386929e-01,
  -7.798978183961720e-01,   6.259068563779211e-01,
  -7.797777879230146e-01,   6.260563884043435e-01,
  -7.796577287796569e-01,   6.262058974124621e-01,
  -7.795376409705133e-01,   6.263553833967800e-01,
  -7.794175244999989e-01,   6.265048463518008e-01,
  -7.792973793725303e-01,   6.266542862720293e-01,
  -7.791772055925247e-01,   6.268037031519712e-01,
  -7.790570031644006e-01,   6.269530969861327e-01,
  -7.789367720925775e-01,   6.271024677690209e-01,
  -7.788165123814760e-01,   6.272518154951441e-01,
  -7.786962240355175e-01,   6.274011401590110e-01,
  -7.785759070591250e-01,   6.275504417551315e-01,
  -7.784555614567219e-01,   6.276997202780162e-01,
  -7.783351872327332e-01,   6.278489757221765e-01,
  -7.782147843915845e-01,   6.279982080821247e-01,
  -7.780943529377028e-01,   6.281474173523740e-01,
  -7.779738928755161e-01,   6.282966035274383e-01,
  -7.778534042094531e-01,   6.284457666018327e-01,
  -7.777328869439440e-01,   6.285949065700726e-01,
  -7.776123410834200e-01,   6.287440234266747e-01,
  -7.774917666323130e-01,   6.288931171661565e-01,
  -7.773711635950563e-01,   6.290421877830360e-01,
  -7.772505319760841e-01,   6.291912352718323e-01,
  -7.771298717798316e-01,   6.293402596270656e-01,
  -7.770091830107353e-01,   6.294892608432566e-01,
  -7.768884656732324e-01,   6.296382389149270e-01,
  -7.767677197717615e-01,   6.297871938365992e-01,
  -7.766469453107621e-01,   6.299361256027964e-01,
  -7.765261422946744e-01,   6.300850342080432e-01,
  -7.764053107279404e-01,   6.302339196468644e-01,
  -7.762844506150025e-01,   6.303827819137859e-01,
  -7.761635619603043e-01,   6.305316210033346e-01,
  -7.760426447682908e-01,   6.306804369100380e-01,
  -7.759216990434077e-01,   6.308292296284245e-01,
  -7.758007247901016e-01,   6.309779991530235e-01,
  -7.756797220128206e-01,   6.311267454783653e-01,
  -7.755586907160136e-01,   6.312754685989808e-01,
  -7.754376309041305e-01,   6.314241685094019e-01,
  -7.753165425816225e-01,   6.315728452041610e-01,
  -7.751954257529414e-01,   6.317214986777923e-01,
  -7.750742804225405e-01,   6.318701289248299e-01,
  -7.749531065948739e-01,   6.320187359398091e-01,
  -7.748319042743969e-01,   6.321673197172659e-01,
  -7.747106734655655e-01,   6.323158802517376e-01,
  -7.745894141728376e-01,   6.324644175377618e-01,
  -7.744681264006709e-01,   6.326129315698774e-01,
  -7.743468101535251e-01,   6.327614223426240e-01,
  -7.742254654358607e-01,   6.329098898505418e-01,
  -7.741040922521390e-01,   6.330583340881721e-01,
  -7.739826906068229e-01,   6.332067550500572e-01,
  -7.738612605043755e-01,   6.333551527307400e-01,
  -7.737398019492618e-01,   6.335035271247643e-01,
  -7.736183149459475e-01,   6.336518782266749e-01,
  -7.734967994988990e-01,   6.338002060310173e-01,
  -7.733752556125845e-01,   6.339485105323378e-01,
  -7.732536832914726e-01,   6.340967917251837e-01,
  -7.731320825400331e-01,   6.342450496041033e-01,
  -7.730104533627370e-01,   6.343932841636455e-01,
  -7.728887957640562e-01,   6.345414953983600e-01,
  -7.727671097484639e-01,   6.346896833027977e-01,
  -7.726453953204339e-01,   6.348378478715100e-01,
  -7.725236524844413e-01,   6.349859890990495e-01,
  -7.724018812449625e-01,   6.351341069799692e-01,
  -7.722800816064743e-01,   6.352822015088234e-01,
  -7.721582535734552e-01,   6.354302726801672e-01,
  -7.720363971503845e-01,   6.355783204885561e-01,
  -7.719145123417424e-01,   6.357263449285471e-01,
  -7.717925991520102e-01,   6.358743459946977e-01,
  -7.716706575856703e-01,   6.360223236815663e-01,
  -7.715486876472063e-01,   6.361702779837122e-01,
  -7.714266893411026e-01,   6.363182088956955e-01,
  -7.713046626718448e-01,   6.364661164120772e-01,
  -7.711826076439193e-01,   6.366140005274191e-01,
  -7.710605242618138e-01,   6.367618612362842e-01,
  -7.709384125300169e-01,   6.369096985332359e-01,
  -7.708162724530185e-01,   6.370575124128386e-01,
  -7.706941040353091e-01,   6.372053028696576e-01,
  -7.705719072813808e-01,   6.373530698982591e-01,
  -7.704496821957260e-01,   6.375008134932102e-01,
  -7.703274287828389e-01,   6.376485336490788e-01,
  -7.702051470472142e-01,   6.377962303604335e-01,
  -7.700828369933479e-01,   6.379439036218440e-01,
  -7.699604986257372e-01,   6.380915534278808e-01,
  -7.698381319488798e-01,   6.382391797731153e-01,
  -7.697157369672751e-01,   6.383867826521196e-01,
  -7.695933136854229e-01,   6.385343620594668e-01,
  -7.694708621078247e-01,   6.386819179897307e-01,
  -7.693483822389823e-01,   6.388294504374863e-01,
  -7.692258740833993e-01,   6.389769593973091e-01,
  -7.691033376455797e-01,   6.391244448637757e-01,
  -7.689807729300289e-01,   6.392719068314635e-01,
  -7.688581799412533e-01,   6.394193452949507e-01,
  -7.687355586837603e-01,   6.395667602488163e-01,
  -7.686129091620584e-01,   6.397141516876405e-01,
  -7.684902313806569e-01,   6.398615196060039e-01,
  -7.683675253440663e-01,   6.400088639984884e-01,
  -7.682447910567983e-01,   6.401561848596765e-01,
  -7.681220285233654e-01,   6.403034821841517e-01,
  -7.679992377482813e-01,   6.404507559664981e-01,
  -7.678764187360606e-01,   6.405980062013010e-01,
  -7.677535714912190e-01,   6.407452328831464e-01,
  -7.676306960182734e-01,   6.408924360066214e-01,
  -7.675077923217413e-01,   6.410396155663134e-01,
  -7.673848604061417e-01,   6.411867715568113e-01,
  -7.672619002759945e-01,   6.413339039727043e-01,
  -7.671389119358204e-01,   6.414810128085832e-01,
  -7.670158953901415e-01,   6.416280980590388e-01,
  -7.668928506434807e-01,   6.417751597186635e-01,
  -7.667697777003619e-01,   6.419221977820502e-01,
  -7.666466765653104e-01,   6.420692122437925e-01,
  -7.665235472428522e-01,   6.422162030984854e-01,
  -7.664003897375142e-01,   6.423631703407243e-01,
  -7.662772040538247e-01,   6.425101139651057e-01,
  -7.661539901963129e-01,   6.426570339662269e-01,
  -7.660307481695090e-01,   6.428039303386860e-01,
  -7.659074779779443e-01,   6.429508030770821e-01,
  -7.657841796261510e-01,   6.430976521760151e-01,
  -7.656608531186625e-01,   6.432444776300859e-01,
  -7.655374984600131e-01,   6.433912794338958e-01,
  -7.654141156547383e-01,   6.435380575820477e-01,
  -7.652907047073744e-01,   6.436848120691449e-01,
  -7.651672656224590e-01,   6.438315428897914e-01,
  -7.650437984045305e-01,   6.439782500385927e-01,
  -7.649203030581284e-01,   6.441249335101545e-01,
  -7.647967795877935e-01,   6.442715932990838e-01,
  -7.646732279980671e-01,   6.444182293999884e-01,
  -7.645496482934921e-01,   6.445648418074766e-01,
  -7.644260404786121e-01,   6.447114305161583e-01,
  -7.643024045579717e-01,   6.448579955206437e-01,
  -7.641787405361167e-01,   6.450045368155439e-01,
  -7.640550484175940e-01,   6.451510543954712e-01,
  -7.639313282069511e-01,   6.452975482550384e-01,
  -7.638075799087372e-01,   6.454440183888592e-01,
  -7.636838035275019e-01,   6.455904647915487e-01,
  -7.635599990677961e-01,   6.457368874577223e-01,
  -7.634361665341720e-01,   6.458832863819963e-01,
  -7.633123059311824e-01,   6.460296615589883e-01,
  -7.631884172633813e-01,   6.461760129833163e-01,
  -7.630645005353237e-01,   6.463223406495995e-01,
  -7.629405557515657e-01,   6.464686445524578e-01,
  -7.628165829166644e-01,   6.466149246865120e-01,
  -7.626925820351780e-01,   6.467611810463839e-01,
  -7.625685531116654e-01,   6.469074136266960e-01,
  -7.624444961506872e-01,   6.470536224220715e-01,
  -7.623204111568043e-01,   6.471998074271352e-01,
  -7.621962981345789e-01,   6.473459686365121e-01,
  -7.620721570885746e-01,   6.474921060448281e-01,
  -7.619479880233554e-01,   6.476382196467103e-01,
  -7.618237909434870e-01,   6.477843094367864e-01,
  -7.616995658535354e-01,   6.479303754096853e-01,
  -7.615753127580680e-01,   6.480764175600365e-01,
  -7.614510316616536e-01,   6.482224358824704e-01,
  -7.613267225688614e-01,   6.483684303716183e-01,
  -7.612023854842618e-01,   6.485144010221124e-01,
  -7.610780204124266e-01,   6.486603478285858e-01,
  -7.609536273579282e-01,   6.488062707856725e-01,
  -7.608292063253400e-01,   6.489521698880073e-01,
  -7.607047573192369e-01,   6.490980451302260e-01,
  -7.605802803441944e-01,   6.492438965069649e-01,
  -7.604557754047893e-01,   6.493897240128617e-01,
  -7.603312425055990e-01,   6.495355276425547e-01,
  -7.602066816512024e-01,   6.496813073906832e-01,
  -7.600820928461793e-01,   6.498270632518871e-01,
  -7.599574760951103e-01,   6.499727952208075e-01,
  -7.598328314025774e-01,   6.501185032920862e-01,
  -7.597081587731634e-01,   6.502641874603659e-01,
  -7.595834582114520e-01,   6.504098477202903e-01,
  -7.594587297220282e-01,   6.505554840665039e-01,
  -7.593339733094779e-01,   6.507010964936520e-01,
  -7.592091889783881e-01,   6.508466849963809e-01,
  -7.590843767333466e-01,   6.509922495693377e-01,
  -7.589595365789424e-01,   6.511377902071703e-01,
  -7.588346685197657e-01,   6.512833069045277e-01,
  -7.587097725604074e-01,   6.514287996560598e-01,
  -7.585848487054596e-01,   6.515742684564170e-01,
  -7.584598969595154e-01,   6.517197133002509e-01,
  -7.583349173271690e-01,   6.518651341822139e-01,
  -7.582099098130153e-01,   6.520105310969595e-01,
  -7.580848744216507e-01,   6.521559040391416e-01,
  -7.579598111576723e-01,   6.523012530034155e-01,
  -7.578347200256783e-01,   6.524465779844367e-01,
  -7.577096010302681e-01,   6.525918789768624e-01,
  -7.575844541760418e-01,   6.527371559753503e-01,
  -7.574592794676007e-01,   6.528824089745588e-01,
  -7.573340769095471e-01,   6.530276379691475e-01,
  -7.572088465064846e-01,   6.531728429537768e-01,
  -7.570835882630171e-01,   6.533180239231077e-01,
  -7.569583021837505e-01,   6.534631808718023e-01,
  -7.568329882732908e-01,   6.536083137945239e-01,
  -7.567076465362457e-01,   6.537534226859361e-01,
  -7.565822769772235e-01,   6.538985075407038e-01,
  -7.564568796008337e-01,   6.540435683534926e-01,
  -7.563314544116869e-01,   6.541886051189690e-01,
  -7.562060014143945e-01,   6.543336178318004e-01,
  -7.560805206135691e-01,   6.544786064866553e-01,
  -7.559550120138244e-01,   6.546235710782027e-01,
  -7.558294756197748e-01,   6.547685116011126e-01,
  -7.557039114360359e-01,   6.549134280500560e-01,
  -7.555783194672245e-01,   6.550583204197049e-01,
  -7.554526997179583e-01,   6.552031887047318e-01,
  -7.553270521928557e-01,   6.553480328998105e-01,
  -7.552013768965365e-01,   6.554928529996153e-01,
  -7.550756738336216e-01,   6.556376489988218e-01,
  -7.549499430087326e-01,   6.557824208921060e-01,
  -7.548241844264924e-01,   6.559271686741454e-01,
  -7.546983980915245e-01,   6.560718923396176e-01,
  -7.545725840084538e-01,   6.562165918832019e-01,
  -7.544467421819064e-01,   6.563612672995780e-01,
  -7.543208726165088e-01,   6.565059185834266e-01,
  -7.541949753168892e-01,   6.566505457294289e-01,
  -7.540690502876761e-01,   6.567951487322681e-01,
  -7.539430975334996e-01,   6.569397275866271e-01,
  -7.538171170589908e-01,   6.570842822871902e-01,
  -7.536911088687812e-01,   6.572288128286425e-01,
  -7.535650729675043e-01,   6.573733192056702e-01,
  -7.534390093597936e-01,   6.575178014129601e-01,
  -7.533129180502843e-01,   6.576622594452001e-01,
  -7.531867990436125e-01,   6.578066932970786e-01,
  -7.530606523444151e-01,   6.579511029632855e-01,
  -7.529344779573302e-01,   6.580954884385112e-01,
  -7.528082758869970e-01,   6.582398497174469e-01,
  -7.526820461380553e-01,   6.583841867947850e-01,
  -7.525557887151464e-01,   6.585284996652186e-01,
  -7.524295036229124e-01,   6.586727883234419e-01,
  -7.523031908659964e-01,   6.588170527641495e-01,
  -7.521768504490428e-01,   6.589612929820373e-01,
  -7.520504823766964e-01,   6.591055089718021e-01,
  -7.519240866536036e-01,   6.592497007281415e-01,
  -7.517976632844116e-01,   6.593938682457539e-01,
  -7.516712122737684e-01,   6.595380115193387e-01,
  -7.515447336263237e-01,   6.596821305435961e-01,
  -7.514182273467275e-01,   6.598262253132273e-01,
  -7.512916934396309e-01,   6.599702958229345e-01,
  -7.511651319096865e-01,   6.601143420674205e-01,
  -7.510385427615474e-01,   6.602583640413890e-01,
  -7.509119259998679e-01,   6.604023617395450e-01,
  -7.507852816293037e-01,   6.605463351565939e-01,
  -7.506586096545107e-01,   6.606902842872423e-01,
  -7.505319100801464e-01,   6.608342091261976e-01,
  -7.504051829108693e-01,   6.609781096681681e-01,
  -7.502784281513387e-01,   6.611219859078629e-01,
  -7.501516458062151e-01,   6.612658378399923e-01,
  -7.500248358801598e-01,   6.614096654592669e-01,
  -7.498979983778353e-01,   6.615534687603989e-01,
  -7.497711333039051e-01,   6.616972477381010e-01,
  -7.496442406630335e-01,   6.618410023870869e-01,
  -7.495173204598862e-01,   6.619847327020709e-01,
  -7.493903726991296e-01,   6.621284386777687e-01,
  -7.492633973854311e-01,   6.622721203088966e-01,
  -7.491363945234594e-01,   6.624157775901718e-01,
  -7.490093641178839e-01,   6.625594105163123e-01,
  -7.488823061733751e-01,   6.627030190820374e-01,
  -7.487552206946048e-01,   6.628466032820669e-01,
  -7.486281076862454e-01,   6.629901631111215e-01,
  -7.485009671529704e-01,   6.631336985639230e-01,
  -7.483737990994546e-01,   6.632772096351941e-01,
  -7.482466035303734e-01,   6.634206963196583e-01,
  -7.481193804504036e-01,   6.635641586120398e-01,
  -7.479921298642227e-01,   6.637075965070640e-01,
  -7.478648517765094e-01,   6.638510099994573e-01,
  -7.477375461919433e-01,   6.639943990839466e-01,
  -7.476102131152051e-01,   6.641377637552600e-01,
  -7.474828525509766e-01,   6.642811040081262e-01,
  -7.473554645039402e-01,   6.644244198372752e-01,
  -7.472280489787799e-01,   6.645677112374375e-01,
  -7.471006059801801e-01,   6.647109782033448e-01,
  -7.469731355128268e-01,   6.648542207297297e-01,
  -7.468456375814065e-01,   6.649974388113253e-01,
  -7.467181121906071e-01,   6.651406324428661e-01,
  -7.465905593451173e-01,   6.652838016190872e-01,
  -7.464629790496268e-01,   6.654269463347247e-01,
  -7.463353713088263e-01,   6.655700665845155e-01,
  -7.462077361274078e-01,   6.657131623631976e-01,
  -7.460800735100638e-01,   6.658562336655097e-01,
  -7.459523834614883e-01,   6.659992804861915e-01,
  -7.458246659863761e-01,   6.661423028199835e-01,
  -7.456969210894228e-01,   6.662853006616273e-01,
  -7.455691487753254e-01,   6.664282740058652e-01,
  -7.454413490487817e-01,   6.665712228474406e-01,
  -7.453135219144905e-01,   6.667141471810977e-01,
  -7.451856673771516e-01,   6.668570470015812e-01,
  -7.450577854414661e-01,   6.669999223036375e-01,
  -7.449298761121353e-01,   6.671427730820133e-01,
  -7.448019393938626e-01,   6.672855993314564e-01,
  -7.446739752913517e-01,   6.674284010467155e-01,
  -7.445459838093074e-01,   6.675711782225403e-01,
  -7.444179649524356e-01,   6.677139308536810e-01,
  -7.442899187254433e-01,   6.678566589348893e-01,
  -7.441618451330382e-01,   6.679993624609174e-01,
  -7.440337441799293e-01,   6.681420414265185e-01,
  -7.439056158708265e-01,   6.682846958264467e-01,
  -7.437774602104409e-01,   6.684273256554568e-01,
  -7.436492772034841e-01,   6.685699309083050e-01,
  -7.435210668546691e-01,   6.687125115797480e-01,
  -7.433928291687100e-01,   6.688550676645436e-01,
  -7.432645641503216e-01,   6.689975991574503e-01,
  -7.431362718042198e-01,   6.691401060532276e-01,
  -7.430079521351217e-01,   6.692825883466360e-01,
  -7.428796051477452e-01,   6.694250460324369e-01,
  -7.427512308468090e-01,   6.695674791053925e-01,
  -7.426228292370335e-01,   6.697098875602658e-01,
  -7.424944003231392e-01,   6.698522713918210e-01,
  -7.423659441098485e-01,   6.699946305948230e-01,
  -7.422374606018840e-01,   6.701369651640376e-01,
  -7.421089498039699e-01,   6.702792750942318e-01,
  -7.419804117208311e-01,   6.704215603801731e-01,
  -7.418518463571935e-01,   6.705638210166300e-01,
  -7.417232537177841e-01,   6.707060569983722e-01,
  -7.415946338073311e-01,   6.708482683201696e-01,
  -7.414659866305633e-01,   6.709904549767942e-01,
  -7.413373121922107e-01,   6.711326169630177e-01,
  -7.412086104970043e-01,   6.712747542736135e-01,
  -7.410798815496761e-01,   6.714168669033554e-01,
  -7.409511253549591e-01,   6.715589548470183e-01,
  -7.408223419175873e-01,   6.717010180993783e-01,
  -7.406935312422958e-01,   6.718430566552119e-01,
  -7.405646933338202e-01,   6.719850705092969e-01,
  -7.404358281968980e-01,   6.721270596564117e-01,
  -7.403069358362669e-01,   6.722690240913359e-01,
  -7.401780162566662e-01,   6.724109638088498e-01,
  -7.400490694628356e-01,   6.725528788037347e-01,
  -7.399200954595162e-01,   6.726947690707729e-01,
  -7.397910942514500e-01,   6.728366346047473e-01,
  -7.396620658433800e-01,   6.729784754004421e-01,
  -7.395330102400502e-01,   6.731202914526421e-01,
  -7.394039274462058e-01,   6.732620827561330e-01,
  -7.392748174665925e-01,   6.734038493057017e-01,
  -7.391456803059575e-01,   6.735455910961361e-01,
  -7.390165159690487e-01,   6.736873081222243e-01,
  -7.388873244606151e-01,   6.738290003787560e-01,
  -7.387581057854069e-01,   6.739706678605216e-01,
  -7.386288599481748e-01,   6.741123105623124e-01,
  -7.384995869536711e-01,   6.742539284789204e-01,
  -7.383702868066486e-01,   6.743955216051390e-01,
  -7.382409595118613e-01,   6.745370899357620e-01,
  -7.381116050740643e-01,   6.746786334655845e-01,
  -7.379822234980136e-01,   6.748201521894022e-01,
  -7.378528147884660e-01,   6.749616461020119e-01,
  -7.377233789501797e-01,   6.751031151982114e-01,
  -7.375939159879136e-01,   6.752445594727993e-01,
  -7.374644259064276e-01,   6.753859789205748e-01,
  -7.373349087104829e-01,   6.755273735363385e-01,
  -7.372053644048412e-01,   6.756687433148919e-01,
  -7.370757929942657e-01,   6.758100882510369e-01,
  -7.369461944835203e-01,   6.759514083395770e-01,
  -7.368165688773699e-01,   6.760927035753159e-01,
  -7.366869161805805e-01,   6.762339739530590e-01,
  -7.365572363979191e-01,   6.763752194676116e-01,
  -7.364275295341537e-01,   6.765164401137811e-01,
  -7.362977955940532e-01,   6.766576358863750e-01,
  -7.361680345823874e-01,   6.767988067802018e-01,
  -7.360382465039274e-01,   6.769399527900711e-01,
  -7.359084313634452e-01,   6.770810739107935e-01,
  -7.357785891657136e-01,   6.772221701371803e-01,
  -7.356487199155065e-01,   6.773632414640439e-01,
  -7.355188236175989e-01,   6.775042878861974e-01,
  -7.353889002767667e-01,   6.776453093984549e-01,
  -7.352589498977868e-01,   6.777863059956315e-01,
  -7.351289724854372e-01,   6.779272776725430e-01,
  -7.349989680444967e-01,   6.780682244240066e-01,
  -7.348689365797452e-01,   6.782091462448399e-01,
  -7.347388780959635e-01,   6.783500431298615e-01,
  -7.346087925979335e-01,   6.784909150738911e-01,
  -7.344786800904384e-01,   6.786317620717494e-01,
  -7.343485405782616e-01,   6.787725841182577e-01,
  -7.342183740661883e-01,   6.789133812082384e-01,
  -7.340881805590040e-01,   6.790541533365149e-01,
  -7.339579600614959e-01,   6.791949004979112e-01,
  -7.338277125784517e-01,   6.793356226872526e-01,
  -7.336974381146604e-01,   6.794763198993650e-01,
  -7.335671366749114e-01,   6.796169921290756e-01,
  -7.334368082639957e-01,   6.797576393712120e-01,
  -7.333064528867053e-01,   6.798982616206033e-01,
  -7.331760705478327e-01,   6.800388588720789e-01,
  -7.330456612521721e-01,   6.801794311204697e-01,
  -7.329152250045178e-01,   6.803199783606072e-01,
  -7.327847618096658e-01,   6.804605005873239e-01,
  -7.326542716724128e-01,   6.806009977954530e-01,
  -7.325237545975567e-01,   6.807414699798291e-01,
  -7.323932105898960e-01,   6.808819171352872e-01,
  -7.322626396542308e-01,   6.810223392566637e-01,
  -7.321320417953613e-01,   6.811627363387954e-01,
  -7.320014170180896e-01,   6.813031083765205e-01,
  -7.318707653272183e-01,   6.814434553646779e-01,
  -7.317400867275510e-01,   6.815837772981075e-01,
  -7.316093812238926e-01,   6.817240741716497e-01,
  -7.314786488210485e-01,   6.818643459801467e-01,
  -7.313478895238256e-01,   6.820045927184408e-01,
  -7.312171033370313e-01,   6.821448143813756e-01,
  -7.310862902654743e-01,   6.822850109637956e-01,
  -7.309554503139644e-01,   6.824251824605461e-01,
  -7.308245834873122e-01,   6.825653288664733e-01,
  -7.306936897903290e-01,   6.827054501764246e-01,
  -7.305627692278276e-01,   6.828455463852481e-01,
  -7.304318218046215e-01,   6.829856174877927e-01,
  -7.303008475255255e-01,   6.831256634789087e-01,
  -7.301698463953549e-01,   6.832656843534467e-01,
  -7.300388184189263e-01,   6.834056801062587e-01,
  -7.299077636010571e-01,   6.835456507321975e-01,
  -7.297766819465661e-01,   6.836855962261166e-01,
  -7.296455734602725e-01,   6.838255165828707e-01,
  -7.295144381469970e-01,   6.839654117973154e-01,
  -7.293832760115610e-01,   6.841052818643071e-01,
  -7.292520870587870e-01,   6.842451267787031e-01,
  -7.291208712934982e-01,   6.843849465353617e-01,
  -7.289896287205194e-01,   6.845247411291423e-01,
  -7.288583593446758e-01,   6.846645105549050e-01,
  -7.287270631707938e-01,   6.848042548075106e-01,
  -7.285957402037009e-01,   6.849439738818215e-01,
  -7.284643904482252e-01,   6.850836677727004e-01,
  -7.283330139091964e-01,   6.852233364750112e-01,
  -7.282016105914446e-01,   6.853629799836187e-01,
  -7.280701804998012e-01,   6.855025982933886e-01,
  -7.279387236390986e-01,   6.856421913991875e-01,
  -7.278072400141700e-01,   6.857817592958830e-01,
  -7.276757296298496e-01,   6.859213019783436e-01,
  -7.275441924909728e-01,   6.860608194414387e-01,
  -7.274126286023758e-01,   6.862003116800386e-01,
  -7.272810379688959e-01,   6.863397786890145e-01,
  -7.271494205953710e-01,   6.864792204632390e-01,
  -7.270177764866407e-01,   6.866186369975846e-01,
  -7.268861056475450e-01,   6.867580282869259e-01,
  -7.267544080829250e-01,   6.868973943261376e-01,
  -7.266226837976228e-01,   6.870367351100957e-01,
  -7.264909327964819e-01,   6.871760506336768e-01,
  -7.263591550843460e-01,   6.873153408917590e-01,
  -7.262273506660604e-01,   6.874546058792210e-01,
  -7.260955195464710e-01,   6.875938455909422e-01,
  -7.259636617304249e-01,   6.877330600218032e-01,
  -7.258317772227704e-01,   6.878722491666855e-01,
  -7.256998660283561e-01,   6.880114130204716e-01,
  -7.255679281520323e-01,   6.881505515780448e-01,
  -7.254359635986498e-01,   6.882896648342893e-01,
  -7.253039723730608e-01,   6.884287527840904e-01,
  -7.251719544801180e-01,   6.885678154223343e-01,
  -7.250399099246754e-01,   6.887068527439077e-01,
  -7.249078387115878e-01,   6.888458647436990e-01,
  -7.247757408457113e-01,   6.889848514165970e-01,
  -7.246436163319026e-01,   6.891238127574916e-01,
  -7.245114651750196e-01,   6.892627487612735e-01,
  -7.243792873799212e-01,   6.894016594228343e-01,
  -7.242470829514670e-01,   6.895405447370668e-01,
  -7.241148518945179e-01,   6.896794046988648e-01,
  -7.239825942139355e-01,   6.898182393031225e-01,
  -7.238503099145829e-01,   6.899570485447354e-01,
  -7.237179990013235e-01,   6.900958324186000e-01,
  -7.235856614790221e-01,   6.902345909196134e-01,
  -7.234532973525444e-01,   6.903733240426740e-01,
  -7.233209066267570e-01,   6.905120317826811e-01,
  -7.231884893065275e-01,   6.906507141345346e-01,
  -7.230560453967244e-01,   6.907893710931357e-01,
  -7.229235749022177e-01,   6.909280026533862e-01,
  -7.227910778278775e-01,   6.910666088101892e-01,
  -7.226585541785756e-01,   6.912051895584485e-01,
  -7.225260039591845e-01,   6.913437448930687e-01,
  -7.223934271745776e-01,   6.914822748089559e-01,
  -7.222608238296293e-01,   6.916207793010163e-01,
  -7.221281939292153e-01,   6.917592583641577e-01,
  -7.219955374782119e-01,   6.918977119932888e-01,
  -7.218628544814963e-01,   6.920361401833187e-01,
  -7.217301449439472e-01,   6.921745429291581e-01,
  -7.215974088704438e-01,   6.923129202257182e-01,
  -7.214646462658664e-01,   6.924512720679111e-01,
  -7.213318571350963e-01,   6.925895984506504e-01,
  -7.211990414830157e-01,   6.927278993688498e-01,
  -7.210661993145081e-01,   6.928661748174246e-01,
  -7.209333306344575e-01,   6.930044247912909e-01,
  -7.208004354477492e-01,   6.931426492853654e-01,
  -7.206675137592695e-01,   6.932808482945660e-01,
  -7.205345655739053e-01,   6.934190218138118e-01,
  -7.204015908965448e-01,   6.935571698380223e-01,
  -7.202685897320772e-01,   6.936952923621182e-01,
  -7.201355620853924e-01,   6.938333893810213e-01,
  -7.200025079613817e-01,   6.939714608896540e-01,
  -7.198694273649369e-01,   6.941095068829398e-01,
  -7.197363203009510e-01,   6.942475273558033e-01,
  -7.196031867743181e-01,   6.943855223031697e-01,
  -7.194700267899330e-01,   6.945234917199655e-01,
  -7.193368403526917e-01,   6.946614356011178e-01,
  -7.192036274674912e-01,   6.947993539415549e-01,
  -7.190703881392292e-01,   6.949372467362058e-01,
  -7.189371223728045e-01,   6.950751139800009e-01,
  -7.188038301731169e-01,   6.952129556678708e-01,
  -7.186705115450672e-01,   6.953507717947477e-01,
  -7.185371664935574e-01,   6.954885623555644e-01,
  -7.184037950234898e-01,   6.956263273452549e-01,
  -7.182703971397683e-01,   6.957640667587537e-01,
  -7.181369728472975e-01,   6.959017805909968e-01,
  -7.180035221509832e-01,   6.960394688369207e-01,
  -7.178700450557317e-01,   6.961771314914630e-01,
  -7.177365415664510e-01,   6.963147685495621e-01,
  -7.176030116880491e-01,   6.964523800061578e-01,
  -7.174694554254358e-01,   6.965899658561904e-01,
  -7.173358727835217e-01,   6.967275260946012e-01,
  -7.172022637672181e-01,   6.968650607163325e-01,
  -7.170686283814375e-01,   6.970025697163275e-01,
  -7.169349666310931e-01,   6.971400530895304e-01,
  -7.168012785210995e-01,   6.972775108308865e-01,
  -7.166675640563719e-01,   6.974149429353418e-01,
  -7.165338232418267e-01,   6.975523493978432e-01,
  -7.164000560823810e-01,   6.976897302133388e-01,
  -7.162662625829531e-01,   6.978270853767773e-01,
  -7.161324427484623e-01,   6.979644148831087e-01,
  -7.159985965838287e-01,   6.981017187272838e-01,
  -7.158647240939735e-01,   6.982389969042543e-01,
  -7.157308252838186e-01,   6.983762494089729e-01,
  -7.155969001582875e-01,   6.985134762363930e-01,
  -7.154629487223038e-01,   6.986506773814695e-01,
  -7.153289709807926e-01,   6.987878528391577e-01,
  -7.151949669386801e-01,   6.989250026044141e-01,
  -7.150609366008931e-01,   6.990621266721961e-01,
  -7.149268799723595e-01,   6.991992250374621e-01,
  -7.147927970580082e-01,   6.993362976951711e-01,
  -7.146586878627691e-01,   6.994733446402838e-01,
  -7.145245523915729e-01,   6.996103658677610e-01,
  -7.143903906493514e-01,   6.997473613725650e-01,
  -7.142562026410375e-01,   6.998843311496588e-01,
  -7.141219883715648e-01,   7.000212751940063e-01,
  -7.139877478458678e-01,   7.001581935005727e-01,
  -7.138534810688825e-01,   7.002950860643238e-01,
  -7.137191880455452e-01,   7.004319528802264e-01,
  -7.135848687807936e-01,   7.005687939432483e-01,
  -7.134505232795663e-01,   7.007056092483585e-01,
  -7.133161515468026e-01,   7.008423987905261e-01,
  -7.131817535874432e-01,   7.009791625647224e-01,
  -7.130473294064293e-01,   7.011159005659187e-01,
  -7.129128790087035e-01,   7.012526127890875e-01,
  -7.127784023992090e-01,   7.013892992292022e-01,
  -7.126438995828902e-01,   7.015259598812373e-01,
  -7.125093705646923e-01,   7.016625947401685e-01,
  -7.123748153495617e-01,   7.017992038009717e-01,
  -7.122402339424455e-01,   7.019357870586244e-01,
  -7.121056263482919e-01,   7.020723445081046e-01,
  -7.119709925720501e-01,   7.022088761443919e-01,
  -7.118363326186701e-01,   7.023453819624659e-01,
  -7.117016464931030e-01,   7.024818619573080e-01,
  -7.115669342003007e-01,   7.026183161239001e-01,
  -7.114321957452164e-01,   7.027547444572253e-01,
  -7.112974311328040e-01,   7.028911469522674e-01,
  -7.111626403680184e-01,   7.030275236040112e-01,
  -7.110278234558153e-01,   7.031638744074428e-01,
  -7.108929804011517e-01,   7.033001993575487e-01,
  -7.107581112089854e-01,   7.034364984493167e-01,
  -7.106232158842750e-01,   7.035727716777356e-01,
  -7.104882944319805e-01,   7.037090190377948e-01,
  -7.103533468570624e-01,   7.038452405244849e-01,
  -7.102183731644822e-01,   7.039814361327976e-01,
  -7.100833733592028e-01,   7.041176058577253e-01,
  -7.099483474461874e-01,   7.042537496942615e-01,
  -7.098132954304008e-01,   7.043898676374004e-01,
  -7.096782173168086e-01,   7.045259596821374e-01,
  -7.095431131103768e-01,   7.046620258234688e-01,
  -7.094079828160730e-01,   7.047980660563920e-01,
  -7.092728264388657e-01,   7.049340803759049e-01,
  -7.091376439837240e-01,   7.050700687770068e-01,
  -7.090024354556183e-01,   7.052060312546978e-01,
  -7.088672008595198e-01,   7.053419678039788e-01,
  -7.087319402004006e-01,   7.054778784198521e-01,
  -7.085966534832341e-01,   7.056137630973205e-01,
  -7.084613407129942e-01,   7.057496218313878e-01,
  -7.083260018946559e-01,   7.058854546170590e-01,
  -7.081906370331954e-01,   7.060212614493397e-01,
  -7.080552461335895e-01,   7.061570423232371e-01,
  -7.079198292008163e-01,   7.062927972337585e-01,
  -7.077843862398546e-01,   7.064285261759128e-01,
  -7.076489172556844e-01,   7.065642291447095e-01,
  -7.075134222532863e-01,   7.066999061351594e-01,
  -7.073779012376421e-01,   7.068355571422738e-01,
  -7.072423542137347e-01,   7.069711821610654e-01,
  -7.071067811865476e-01,   7.071067811865475e-01,
#elif TWIDDLE_TABLE_ORDER == 12
  -1.000000000000000e+00,   0.000000000000000e+00,
  -9.999988234517019e-01,   1.533980186284766e-03,
  -9.999952938095762e-01,   3.067956762965976e-03,
  -9.999894110819284e-01,   4.601926120448570e-03,
  -9.999811752826011e-01,   6.135884649154475e-03,
  -9.999705864309741e-01,   7.669828739531097e-03,
  -9.999576445519639e-01,   9.203754782059820e-03,
  -9.999423496760239e-01,   1.073765916726449e-02,
  -9.999247018391445e-01,   1.227153828571993e-02,
  -9.999047010828529e-01,   1.380538852806039e-02,
  -9.998823474542126e-01,   1.533920628498810e-02,
  -9.998576410058239e-01,   1.687298794728171e-02,
  -9.998305817958234e-01,   1.840672990580482e-02,
  -9.998011698878843e-01,   1.994042855151444e-02,
  -9.997694053512153e-01,   2.147408027546951e-02,
  -9.997352882605617e-01,   2.300768146883937e-02,
  -9.996988186962042e-01,   2.454122852291229e-02,
  -9.996599967439592e-01,   2.607471782910390e-02,
  -9.996188224951786e-01,   2.760814577896574e-02,
  -9.995752960467492e-01,   2.914150876419372e-02,
  -9.995294175010931e-01,   3.067480317663663e-02,
  -9.994811869661670e-01,   3.220802540830459e-02,
  -9.994306045554617e-01,   3.374117185137758e-02,
  -9.993776703880028e-01,   3.527423889821395e-02,
  -9.993223845883495e-01,   3.680722294135883e-02,
  -9.992647472865944e-01,   3.834012037355269e-02,
  -9.992047586183639e-01,   3.987292758773981e-02,
  -9.991424187248169e-01,   4.140564097707674e-02,
  -9.990777277526454e-01,   4.293825693494082e-02,
  -9.990106858540734e-01,   4.447077185493867e-02,
  -9.989412931868569e-01,   4.600318213091462e-02,
  -9.988695499142836e-01,   4.753548415695930e-02,
  -9.987954562051724e-01,   4.906767432741801e-02,
  -9.987190122338729e-01,   5.059974903689928e-02,
  -9.986402181802653e-01,   5.213170468028332e-02,
  -9.985590742297593e-01,   5.366353765273052e-02,
  -9.984755805732948e-01,   5.519524434968993e-02,
  -9.983897374073402e-01,   5.672682116690775e-02,
  -9.983015449338929e-01,   5.825826450043575e-02,
  -9.982110033604782e-01,   5.978957074663987e-02,
  -9.981181129001492e-01,   6.132073630220858e-02,
  -9.980228737714862e-01,   6.285175756416140e-02,
  -9.979252861985960e-01,   6.438263092985747e-02,
  -9.978253504111116e-01,   6.591335279700380e-02,
  -9.977230666441916e-01,   6.744391956366405e-02,
  -9.976184351385196e-01,   6.897432762826675e-02,
  -9.975114561403035e-01,   7.050457338961386e-02,
  -9.974021299012753e-01,   7.203465324688933e-02,
  -9.972904566786902e-01,   7.356456359966743e-02,
  -9.971764367353262e-01,   7.509430084792130e-02,
  -9.970600703394830e-01,   7.662386139203149e-02,
  -9.969413577649822e-01,   7.815324163279423e-02,
  -9.968202992911657e-01,   7.968243797143013e-02,
  -9.966968952028961e-01,   8.121144680959244e-02,
  -9.965711457905548e-01,   8.274026454937569e-02,
  -9.964430513500426e-01,   8.426888759332407e-02,
  -9.963126121827780e-01,   8.579731234443990e-02,
  -9.961798285956970e-01,   8.732553520619206e-02,
  -9.960447009012520e-01,   8.885355258252460e-02,
  -9.959072294174117e-01,   9.038136087786498e-02,
  -9.957674144676598e-01,   9.190895649713272e-02,
  -9.956252563809943e-01,   9.343633584574779e-02,
  -9.954807554919269e-01,   9.496349532963899e-02,
  -9.953339121404823e-01,   9.649043135525259e-02,
  -9.951847266721969e-01,   9.801714032956060e-02,
  -9.950331994381186e-01,   9.954361866006932e-02,
  -9.948793307948056e-01,   1.010698627548278e-01,
  -9.947231211043257e-01,   1.025958690224363e-01,
  -9.945645707342554e-01,   1.041216338720546e-01,
  -9.944036800576791e-01,   1.056471537134106e-01,
  -9.942404494531879e-01,   1.071724249568088e-01,
  -9.940748793048794e-01,   1.086974440131387e-01,
  -9.939069700023561e-01,   1.102222072938831e-01,
  -9.937367219407246e-01,   1.117467112111266e-01,
  -9.935641355205953e-01,   1.132709521775643e-01,
  -9.933892111480807e-01,   1.147949266065101e-01,
  -9.932119492347945e-01,   1.163186309119048e-01,
  -9.930323501978514e-01,   1.178420615083250e-01,
  -9.928504144598651e-01,   1.193652148109914e-01,
  -9.926661424489480e-01,   1.208880872357771e-01,
  -9.924795345987100e-01,   1.224106751992162e-01,
  -9.922905913482574e-01,   1.239329751185122e-01,
  -9.920993131421918e-01,   1.254549834115462e-01,
  -9.919057004306093e-01,   1.269766964968859e-01,
  -9.917097536690995e-01,   1.284981107937932e-01,
  -9.915114733187439e-01,   1.300192227222333e-01,
  -9.913108598461154e-01,   1.315400287028831e-01,
  -9.911079137232769e-01,   1.330605251571391e-01,
  -9.909026354277800e-01,   1.345807085071262e-01,
  -9.906950254426646e-01,   1.361005751757062e-01,
  -9.904850842564571e-01,   1.376201215864860e-01,
  -9.902728123631691e-01,   1.391393441638262e-01,
  -9.900582102622971e-01,   1.406582393328492e-01,
  -9.898412784588205e-01,   1.421768035194480e-01,
  -9.896220174632009e-01,   1.436950331502945e-01,
  -9.894004277913804e-01,   1.452129246528475e-01,
  -9.891765099647810e-01,   1.467304744553617e-01,
  -9.889502645103030e-01,   1.482476789868960e-01,
  -9.887216919603238e-01,   1.497645346773215e-01,
  -9.884907928526966e-01,   1.512810379573302e-01,
  -9.882575677307495e-01,   1.527971852584434e-01,
  -9.880220171432835e-01,   1.543129730130201e-01,
  -9.877841416445722e-01,   1.558283976542652e-01,
  -9.875439417943592e-01,   1.573434556162382e-01,
  -9.873014181578584e-01,   1.588581433338614e-01,
  -9.870565713057510e-01,   1.603724572429283e-01,
  -9.868094018141855e-01,   1.618863937801118e-01,
  -9.865599102647754e-01,   1.633999493829732e-01,
  -9.863080972445987e-01,   1.649131204899699e-01,
  -9.860539633461954e-01,   1.664259035404641e-01,
  -9.857975091675675e-01,   1.679382949747312e-01,
  -9.855387353121761e-01,   1.694502912339680e-01,
  -9.852776423889412e-01,   1.709618887603012e-01,
  -9.850142310122398e-01,   1.724730839967960e-01,
  -9.847485018019042e-01,   1.739838733874638e-01,
  -9.844804553832209e-01,   1.754942533772714e-01,
  -9.842100923869290e-01,   1.770042204121487e-01,
  -9.839374134492189e-01,   1.785137709389975e-01,
  -9.836624192117303e-01,   1.800229014056995e-01,
  -9.833851103215512e-01,   1.815316082611250e-01,
  -9.831054874312163e-01,   1.830398879551410e-01,
  -9.828235511987052e-01,   1.845477369386196e-01,
  -9.825393022874412e-01,   1.860551516634466e-01,
  -9.822527413662894e-01,   1.875621285825296e-01,
  -9.819638691095552e-01,   1.890686641498062e-01,
  -9.816726861969831e-01,   1.905747548202527e-01,
  -9.813791933137546e-01,   1.920803970498924e-01,
  -9.810833911504867e-01,   1.935855872958036e-01,
  -9.807852804032304e-01,   1.950903220161282e-01,
  -9.804848617734694e-01,   1.965945976700802e-01,
  -9.801821359681174e-01,   1.980984107179536e-01,
  -9.798771036995176e-01,   1.996017576211310e-01,
  -9.795697656854405e-01,   2.011046348420919e-01,
  -9.792601226490820e-01,   2.026070388444211e-01,
  -9.789481753190622e-01,   2.041089660928169e-01,
  -9.786339244294232e-01,   2.056104130530992e-01,
  -9.783173707196277e-01,   2.071113761922186e-01,
  -9.779985149345571e-01,   2.086118519782635e-01,
  -9.776773578245099e-01,   2.101118368804696e-01,
  -9.773539001452000e-01,   2.116113273692276e-01,
  -9.770281426577544e-01,   2.131103199160914e-01,
  -9.767000861287118e-01,   2.146088109937868e-01,
  -9.763697313300211e-01,   2.161067970762195e-01,
  -9.760370790390390e-01,   2.176042746384836e-01,
  -9.757021300385286e-01,   2.191012401568698e-01,
  -9.753648851166570e-01,   2.205976901088735e-01,
  -9.750253450669941e-01,   2.220936209732035e-01,
  -9.746835106885107e-01,   2.235890292297900e-01,
  -9.743393827855759e-01,   2.250839113597928e-01,
  -9.739929621679558e-01,   2.265782638456100e-01,
  -9.736442496508120e-01,   2.280720831708857e-01,
  -9.732932460546982e-01,   2.295653658205189e-01,
  -9.729399522055602e-01,   2.310581082806711e-01,
  -9.725843689347322e-01,   2.325503070387752e-01,
  -9.722264970789363e-01,   2.340419585835434e-01,
  -9.718663374802794e-01,   2.355330594049755e-01,
  -9.715038909862518e-01,   2.370236059943672e-01,
  -9.711391584497251e-01,   2.385135948443184e-01,
  -9.707721407289504e-01,   2.400030224487415e-01,
  -9.704028386875555e-01,   2.414918853028693e-01,
  -9.700312531945440e-01,   2.429801799032639e-01,
  -9.696573851242924e-01,   2.444679027478242e-01,
  -9.692812353565485e-01,   2.459550503357946e-01,
  -9.689028047764289e-01,   2.474416191677733e-01,
  -9.685220942744174e-01,   2.489276057457201e-01,
  -9.681391047463624e-01,   2.504130065729652e-01,
  -9.677538370934755e-01,   2.518978181542170e-01,
  -9.673662922223285e-01,   2.533820369955702e-01,
  -9.669764710448521e-01,   2.548656596045146e-01,
  -9.665843744783331e-01,   2.563486824899429e-01,
  -9.661900034454125e-01,   2.578311021621590e-01,
  -9.657933588740837e-01,   2.593129151328862e-01,
  -9.653944416976894e-01,   2.607941179152755e-01,
  -9.649932528549203e-01,   2.622747070239136e-01,
  -9.645897932898128e-01,   2.637546789748313e-01,
  -9.641840639517458e-01,   2.652340302855118e-01,
  -9.637760657954398e-01,   2.667127574748984e-01,
  -9.633657997809540e-01,   2.681908570634032e-01,
  -9.629532668736839e-01,   2.696683255729151e-01,
  -9.625384680443592e-01,   2.711451595268080e-01,
  -9.621214042690416e-01,   2.726213554499490e-01,
  -9.617020765291225e-01,   2.740969098687064e-01,
  -9.612804858113206e-01,   2.755718193109581e-01,
  -9.608566331076797e-01,   2.770460803060999e-01,
  -9.604305194155658e-01,   2.785196893850531e-01,
  -9.600021457376660e-01,   2.799926430802732e-01,
  -9.595715130819845e-01,   2.814649379257579e-01,
  -9.591386224618419e-01,   2.829365704570554e-01,
  -9.587034748958716e-01,   2.844075372112719e-01,
  -9.582660714080177e-01,   2.858778347270806e-01,
  -9.578264130275329e-01,   2.873474595447295e-01,
  -9.573845007889759e-01,   2.888164082060495e-01,
  -9.569403357322088e-01,   2.902846772544623e-01,
  -9.564939189023951e-01,   2.917522632349893e-01,
  -9.560452513499964e-01,   2.932191626942586e-01,
  -9.555943341307711e-01,   2.946853721805143e-01,
  -9.551411683057708e-01,   2.961508882436238e-01,
  -9.546857549413383e-01,   2.976157074350862e-01,
  -9.542280951091057e-01,   2.990798263080405e-01,
  -9.537681898859903e-01,   3.005432414172735e-01,
  -9.533060403541939e-01,   3.020059493192281e-01,
  -9.528416476011987e-01,   3.034679465720113e-01,
  -9.523750127197659e-01,   3.049292297354024e-01,
  -9.519061368079323e-01,   3.063897953708609e-01,
  -9.514350209690083e-01,   3.078496400415349e-01,
  -9.509616663115751e-01,   3.093087603122687e-01,
  -9.504860739494817e-01,   3.107671527496115e-01,
  -9.500082450018430e-01,   3.122248139218249e-01,
  -9.495281805930367e-01,   3.136817403988915e-01,
  -9.490458818527006e-01,   3.151379287525224e-01,
  -9.485613499157303e-01,   3.165933755561658e-01,
  -9.480745859222762e-01,   3.180480773850149e-01,
  -9.475855910177411e-01,   3.195020308160157e-01,
  -9.470943663527772e-01,   3.209552324278752e-01,
  -9.466009130832835e-01,   3.224076788010699e-01,
  -9.461052323704034e-01,   3.238593665178529e-01,
  -9.456073253805213e-01,   3.253102921622629e-01,
  -9.451071932852606e-01,   3.267604523201317e-01,
  -9.446048372614803e-01,   3.282098435790925e-01,
  -9.441002584912727e-01,   3.296584625285875e-01,
  -9.435934581619604e-01,   3.311063057598764e-01,
  -9.430844374660935e-01,   3.325533698660442e-01,
  -9.425731976014469e-01,   3.339996514420094e-01,
  -9.420597397710173e-01,   3.354451470845316e-01,
  -9.415440651830208e-01,   3.368898533922201e-01,
  -9.410261750508893e-01,   3.383337669655411e-01,
  -9.405060705932683e-01,   3.397768844068269e-01,
  -9.399837530340140e-01,   3.412192023202824e-01,
  -9.394592236021899e-01,   3.426607173119944e-01,
  -9.389324835320646e-01,   3.441014259899388e-01,
  -9.384035340631081e-01,   3.455413249639891e-01,
  -9.378723764399899e-01,   3.469804108459237e-01,
  -9.373390119125750e-01,   3.484186802494346e-01,
  -9.368034417359216e-01,   3.498561297901349e-01,
  -9.362656671702783e-01,   3.512927560855671e-01,
  -9.357256894810804e-01,   3.527285557552107e-01,
  -9.351835099389476e-01,   3.541635254204903e-01,
  -9.346391298196808e-01,   3.555976617047839e-01,
  -9.340925504042590e-01,   3.570309612334300e-01,
  -9.335437729788362e-01,   3.584634206337365e-01,
  -9.329927988347390e-01,   3.598950365349881e-01,
  -9.324396292684624e-01,   3.613258055684543e-01,
  -9.318842655816681e-01,   3.627557243673972e-01,
  -9.313267090811804e-01,   3.641847895670799e-01,
  -9.307669610789837e-01,   3.656129978047739e-01,
  -9.302050228922191e-01,   3.670403457197672e-01,
  -9.296408958431813e-01,   3.684668299533723e-01,
  -9.290745812593159e-01,   3.698924471489341e-01,
  -9.285060804732156e-01,   3.713171939518375e-01,
  -9.279353948226179e-01,   3.727410670095158e-01,
  -9.273625256504011e-01,   3.741640629714579e-01,
  -9.267874743045817e-01,   3.755861784892172e-01,
  -9.262102421383114e-01,   3.770074102164183e-01,
  -9.256308305098727e-01,   3.784277548087656e-01,
  -9.250492407826776e-01,   3.798472089240512e-01,
  -9.244654743252626e-01,   3.812657692221624e-01,
  -9.238795325112867e-01,   3.826834323650898e-01,
  -9.232914167195276e-01,   3.841001950169350e-01,
  -9.227011283338786e-01,   3.855160538439188e-01,
  -9.221086687433452e-01,   3.869310055143886e-01,
  -9.215140393420420e-01,   3.883450466988262e-01,
  -9.209172415291895e-01,   3.897581740698564e-01,
  -9.203182767091106e-01,   3.911703843022539e-01,
  -9.197171462912274e-01,   3.925816740729515e-01,
  -9.191138516900578e-01,   3.939920400610481e-01,
  -9.185083943252123e-01,   3.954014789478164e-01,
  -9.179007756213905e-01,   3.968099874167103e-01,
  -9.172909970083779e-01,   3.982175621533736e-01,
  -9.166790599210427e-01,   3.996241998456468e-01,
  -9.160649657993317e-01,   4.010298971835756e-01,
  -9.154487160882678e-01,   4.024346508594184e-01,
  -9.148303122379462e-01,   4.038384575676541e-01,
  -9.142097557035307e-01,   4.052413140049899e-01,
  -9.135870479452508e-01,   4.066432168703690e-01,
  -9.129621904283982e-01,   4.080441628649787e-01,
  -9.123351846233227e-01,   4.094441486922576e-01,
  -9.117060320054299e-01,   4.108431710579039e-01,
  -9.110747340551764e-01,   4.122412266698829e-01,
  -9.104412922580672e-01,   4.136383122384345e-01,
  -9.098057081046522e-01,   4.150344244760816e-01,
  -9.091679830905224e-01,   4.164295600976372e-01,
  -9.085281187163061e-01,   4.178237158202123e-01,
  -9.078861164876663e-01,   4.192168883632239e-01,
  -9.072419779152958e-01,   4.206090744484025e-01,
  -9.065957045149153e-01,   4.220002707997997e-01,
  -9.059472978072685e-01,   4.233904741437960e-01,
  -9.052967593181188e-01,   4.247796812091088e-01,
  -9.046440905782462e-01,   4.261678887267996e-01,
  -9.039892931234433e-01,   4.275550934302821e-01,
  -9.033323684945118e-01,   4.289412920553295e-01,
  -9.026733182372588e-01,   4.303264813400826e-01,
  -9.020121439024932e-01,   4.317106580250573e-01,
  -9.013488470460220e-01,   4.330938188531520e-01,
  -9.006834292286470e-01,   4.344759605696557e-01,
  -9.000158920161603e-01,   4.358570799222555e-01,
  -8.993462369793416e-01,   4.372371736610441e-01,
  -8.986744656939538e-01,   4.386162385385277e-01,
  -8.980005797407399e-01,   4.399942713096333e-01,
  -8.973245807054183e-01,   4.413712687317167e-01,
  -8.966464701786802e-01,   4.427472275645700e-01,
  -8.959662497561852e-01,   4.441221445704292e-01,
  -8.952839210385576e-01,   4.454960165139817e-01,
  -8.945994856313827e-01,   4.468688401623742e-01,
  -8.939129451452033e-01,   4.482406122852199e-01,
  -8.932243011955153e-01,   4.496113296546065e-01,
  -8.925335554027646e-01,   4.509809890451039e-01,
  -8.918407093923427e-01,   4.523495872337709e-01,
  -8.911457647945832e-01,   4.537171210001639e-01,
  -8.904487232447579e-01,   4.550835871263438e-01,
  -8.897495863830728e-01,   4.564489823968839e-01,
  -8.890483558546646e-01,   4.578133035988772e-01,
  -8.883450333095964e-01,   4.591765475219441e-01,
  -8.876396204028539e-01,   4.605387109582400e-01,
  -8.869321187943422e-01,   4.618997907024627e-01,
  -8.862225301488806e-01,   4.632597835518601e-01,
  -8.855108561362000e-01,   4.646186863062378e-01,
  -8.847970984309378e-01,   4.659764957679662e-01,
  -8.840812587126350e-01,   4.673332087419884e-01,
  -8.833633386657316e-01,   4.686888220358279e-01,
  -8.826433399795628e-01,   4.700433324595956e-01,
  -8.819212643483550e-01,   4.713967368259976e-01,
  -8.811971134712221e-01,   4.727490319503428e-01,
  -8.804708890521608e-01,   4.741002146505500e-01,
  -8.797425928000474e-01,   4.754502817471559e-01,
  -8.790122264286335e-01,   4.767992300633221e-01,
  -8.782797916565416e-01,   4.781470564248430e-01,
  -8.775452902072614e-01,   4.794937576601530e-01,
  -8.768087238091457e-01,   4.808393306003340e-01,
  -8.760700941954066e-01,   4.821837720791227e-01,
  -8.753294031041109e-01,   4.835270789329187e-01,
  -8.745866522781761e-01,   4.848692480007911e-01,
  -8.738418434653669e-01,   4.862102761244864e-01,
  -8.730949784182901e-01,   4.875501601484360e-01,
  -8.723460588943915e-01,   4.888888969197632e-01,
  -8.715950866559510e-01,   4.902264832882912e-01,
  -8.708420634700790e-01,   4.915629161065499e-01,
  -8.700869911087115e-01,   4.928981922297840e-01,
  -8.693298713486068e-01,   4.942323085159597e-01,
  -8.685707059713409e-01,   4.955652618257725e-01,
  -8.678094967633033e-01,   4.968970490226545e-01,
  -8.670462455156926e-01,   4.982276669727819e-01,
  -8.662809540245130e-01,   4.995571125450818e-01,
  -8.655136240905691e-01,   5.008853826112407e-01,
  -8.647442575194624e-01,   5.022124740457108e-01,
  -8.639728561215868e-01,   5.035383837257176e-01,
  -8.631994217121242e-01,   5.048631085312676e-01,
  -8.624239561110406e-01,   5.061866453451552e-01,
  -8.616464611430813e-01,   5.075089910529709e-01,
  -8.608669386377673e-01,   5.088301425431070e-01,
  -8.600853904293901e-01,   5.101500967067668e-01,
  -8.593018183570085e-01,   5.114688504379703e-01,
  -8.585162242644427e-01,   5.127864006335630e-01,
  -8.577286100002721e-01,   5.141027441932217e-01,
  -8.569389774178288e-01,   5.154178780194629e-01,
  -8.561473283751945e-01,   5.167317990176499e-01,
  -8.553536647351960e-01,   5.180445040959993e-01,
  -8.545579883654005e-01,   5.193559901655896e-01,
  -8.537603011381114e-01,   5.206662541403672e-01,
  -8.529606049303636e-01,   5.219752929371544e-01,
  -8.521589016239198e-01,   5.232831034756564e-01,
  -8.513551931052652e-01,   5.245896826784690e-01,
  -8.505494812656035e-01,   5.258950274710846e-01,
  -8.497417680008525e-01,   5.271991347819013e-01,
  -8.489320552116396e-01,   5.285020015422285e-01,
  -8.481203448032972e-01,   5.298036246862946e-01,
  -8.473066386858583e-01,   5.311040011512550e-01,
  -8.464909387740521e-01,   5.324031278771979e-01,
  -8.456732469872991e-01,   5.337010018071530e-01,
  -8.448535652497071e-01,   5.349976198870972e-01,
  -8.440318954900664e-01,   5.362929790659632e-01,
  -8.432082396418454e-01,   5.375870762956454e-01,
  -8.423825996431858e-01,   5.388799085310084e-01,
  -8.415549774368984e-01,   5.401714727298929e-01,
  -8.407253749704581e-01,   5.414617658531234e-01,
  -8.398937941959995e-01,   5.427507848645159e-01,
  -8.390602370703127e-01,   5.440385267308838e-01,
  -8.382247055548381e-01,   5.453249884220465e-01,
  -8.373872016156619e-01,   5.466101669108349e-01,
  -8.365477272235120e-01,   5.478940591731002e-01,
  -8.357062843537526e-01,   5.491766621877197e-01,
  -8.348628749863800e-01,   5.504579729366048e-01,
  -8.340175011060181e-01,   5.517379884047073e-01,
  -8.331701647019132e-01,   5.530167055800275e-01,
  -8.323208677679297e-01,   5.542941214536200e-01,
  -8.314696123025452e-01,   5.555702330196022e-01,
  -8.306164003088463e-01,   5.568450372751601e-01,
  -8.297612337945230e-01,   5.581185312205561e-01,
  -8.289041147718649e-01,   5.593907118591361e-01,
  -8.280450452577558e-01,   5.606615761973360e-01,
  -8.271840272736691e-01,   5.619311212446895e-01,
  -8.263210628456635e-01,   5.631993440138341e-01,
  -8.254561540043776e-01,   5.644662415205195e-01,
  -8.245893027850253e-01,   5.657318107836131e-01,
  -8.237205112273914e-01,   5.669960488251087e-01,
  -8.228497813758264e-01,   5.682589526701315e-01,
  -8.219771152792416e-01,   5.695205193469471e-01,
  -8.211025149911046e-01,   5.707807458869673e-01,
  -8.202259825694347e-01,   5.720396293247570e-01,
  -8.193475200767969e-01,   5.732971666980422e-01,
  -8.184671295802987e-01,   5.745533550477158e-01,
  -8.175848131515837e-01,   5.758081914178453e-01,
  -8.167005728668278e-01,   5.770616728556794e-01,
  -8.158144108067338e-01,   5.783137964116556e-01,
  -8.149263290565266e-01,   5.795645591394056e-01,
  -8.140363297059484e-01,   5.808139580957645e-01,
  -8.131444148492536e-01,   5.820619903407754e-01,
  -8.122505865852039e-01,   5.833086529376983e-01,
  -8.113548470170637e-01,   5.845539429530153e-01,
  -8.104571982525948e-01,   5.857978574564389e-01,
  -8.095576424040513e-01,   5.870403935209180e-01,
  -8.086561815881750e-01,   5.882815482226452e-01,
  -8.077528179261904e-01,   5.895213186410639e-01,
  -8.068475535437993e-01,   5.907597018588742e-01,
  -8.059403905711763e-01,   5.919966949620410e-01,
  -8.050313311429637e-01,   5.932322950397998e-01,
  -8.041203773982658e-01,   5.944664991846644e-01,
  -8.032075314806449e-01,   5.956993044924334e-01,
  -8.022927955381157e-01,   5.969307080621965e-01,
  -8.013761717231402e-01,   5.981607069963423e-01,
  -8.004576621926228e-01,   5.993892984005645e-01,
  -7.995372691079050e-01,   6.006164793838690e-01,
  -7.986149946347608e-01,   6.018422470585800e-01,
  -7.976908409433912e-01,   6.030665985403482e-01,
  -7.967648102084188e-01,   6.042895309481560e-01,
  -7.958369046088836e-01,   6.055110414043255e-01,
  -7.949071263282370e-01,   6.067311270345245e-01,
  -7.939754775543372e-01,   6.079497849677736e-01,
  -7.930419604794436e-01,   6.091670123364532e-01,
  -7.921065773002124e-01,   6.103828062763095e-01,
  -7.911693302176902e-01,   6.115971639264619e-01,
  -7.902302214373100e-01,   6.128100824294097e-01,
  -7.892892531688857e-01,   6.140215589310384e-01,
  -7.883464276266063e-01,   6.152315905806268e-01,
  -7.874017470290314e-01,   6.164401745308536e-01,
  -7.864552135990858e-01,   6.176473079378039e-01,
  -7.855068295640539e-01,   6.188529879609763e-01,
  -7.845565971555752e-01,   6.200572117632891e-01,
  -7.836045186096382e-01,   6.212599765110876e-01,
  -7.826505961665757e-01,   6.224612793741500e-01,
  -7.816948320710594e-01,   6.236611175256945e-01,
  -7.807372285720945e-01,   6.248594881423863e-01,
  -7.797777879230146e-01,   6.260563884043435e-01,
  -7.788165123814760e-01,   6.272518154951441e-01,
  -7.778534042094531e-01,   6.284457666018327e-01,
  -7.768884656732324e-01,   6.296382389149270e-01,
  -7.759216990434077e-01,   6.308292296284245e-01,
  -7.749531065948739e-01,   6.320187359398091e-01,
  -7.739826906068229e-01,   6.332067550500572e-01,
  -7.730104533627370e-01,   6.343932841636455e-01,
  -7.720363971503845e-01,   6.355783204885561e-01,
  -7.710605242618138e-01,   6.367618612362842e-01,
  -7.700828369933479e-01,   6.379439036218440e-01,
  -7.691033376455797e-01,   6.391244448637757e-01,
  -7.681220285233654e-01,   6.403034821841517e-01,
  -7.671389119358204e-01,   6.414810128085832e-01,
  -7.661539901963129e-01,   6.426570339662269e-01,
  -7.651672656224590e-01,   6.438315428897914e-01,
  -7.641787405361167e-01,   6.450045368155439e-01,
  -7.631884172633813e-01,   6.461760129833163e-01,
  -7.621962981345789e-01,   6.473459686365121e-01,
  -7.612023854842618e-01,   6.485144010221124e-01,
  -7.602066816512024e-01,   6.496813073906832e-01,
  -7.592091889783881e-01,   6.508466849963809e-01,
  -7.582099098130153e-01,   6.520105310969595e-01,
  -7.572088465064846e-01,   6.531728429537768e-01,
  -7.562060014143945e-01,   6.543336178318004e-01,
  -7.552013768965365e-01,   6.554928529996153e-01,
  -7.541949753168892e-01,   6.566505457294289e-01,
  -7.531867990436125e-01,   6.578066932970786e-01,
  -7.521768504490428e-01,   6.589612929820373e-01,
  -7.511651319096865e-01,   6.601143420674205e-01,
  -7.501516458062151e-01,   6.612658378399923e-01,
  -7.491363945234594e-01,   6.624157775901718e-01,
  -7.481193804504036e-01,   6.635641586120398e-01,
  -7.471006059801801e-01,   6.647109782033448e-01,
  -7.460800735100638e-01,   6.658562336655097e-01,
  -7.450577854414661e-01,   6.669999223036375e-01,
  -7.440337441799293e-01,   6.681420414265185e-01,
  -7.430079521351217e-01,   6.692825883466360e-01,
  -7.419804117208311e-01,   6.704215603801731e-01,
  -7.409511253549591e-01,   6.715589548470183e-01,
  -7.399200954595162e-01,   6.726947690707729e-01,
  -7.388873244606151e-01,   6.738290003787560e-01,
  -7.378528147884660e-01,   6.749616461020119e-01,
  -7.368165688773699e-01,   6.760927035753159e-01,
  -7.357785891657136e-01,   6.772221701371803e-01,
  -7.347388780959635e-01,   6.783500431298615e-01,
  -7.336974381146604e-01,   6.794763198993650e-01,
  -7.326542716724128e-01,   6.806009977954530e-01,
  -7.316093812238926e-01,   6.817240741716497e-01,
  -7.305627692278276e-01,   6.828455463852481e-01,
  -7.295144381469970e-01,   6.839654117973154e-01,
  -7.284643904482252e-01,   6.850836677727004e-01,
  -7.274126286023758e-01,   6.862003116800386e-01,
  -7.263591550843460e-01,   6.873153408917590e-01,
  -7.253039723730608e-01,   6.884287527840904e-01,
  -7.242470829514670e-01,   6.895405447370668e-01,
  -7.231884893065275e-01,   6.906507141345346e-01,
  -7.221281939292153e-01,   6.917592583641577e-01,
  -7.210661993145081e-01,   6.928661748174246e-01,
  -7.200025079613817e-01,   6.939714608896540e-01,
  -7.189371223728045e-01,   6.950751139800009e-01,
  -7.178700450557317e-01,   6.961771314914630e-01,
  -7.168012785210995e-01,   6.972775108308865e-01,
  -7.157308252838186e-01,   6.983762494089729e-01,
  -7.146586878627691e-01,   6.994733446402838e-01,
  -7.135848687807936e-01,   7.005687939432483e-01,
  -7.125093705646923e-01,   7.016625947401685e-01,
  -7.114321957452164e-01,   7.027547444572253e-01,
  -7.103533468570624e-01,   7.038452405244849e-01,
  -7.092728264388657e-01,   7.049340803759049e-01,
  -7.081906370331954e-01,   7.060212614493397e-01,
  -7.071067811865476e-01,   7.071067811865475e-01,
#else
#error Unsupported twiddle table size
#endif
};
