#!/usr/bin/env python

from gnuradio import gr, eng_notation
from gnuradio import audio
from gnuradio import usrp
from gnuradio.eng_option import eng_option
from optparse import OptionParser
import sys
import math


#
# return a gr.flow_graph
#
def build_graph (IF_freq, freqs, mid_freq):
    adc_rate = 64e6
    decim = 250
    quad_rate = adc_rate / decim               # 256 kHz
    audio_decimation = 8
    audio_rate = quad_rate / audio_decimation  # 32 kHz

    nchan = len (freqs)
    
    fg = gr.flow_graph ()
    
    # usrp is data source
    mux = 0xf0f0f0f0
    u = usrp.source_c (0, decim, nchan, mux)
    di = gr.deinterleave (gr.sizeof_gr_complex)
    
    fg.connect (u, di)
    
    # sound card as final sink
    audio_sink = audio.sink (int (audio_rate))

    i = 0
    for f in freqs:
        x = (mid_freq - f) + IF_freq
        print x
        u.set_rx_freq (i, -x)
        (head, tail) = build_pipeline (fg, quad_rate, audio_decimation)
        fg.connect ((di, i), head)
        fg.connect (tail, (audio_sink, i))
        i += 1

    return fg

def build_pipeline (fg, quad_rate, audio_decimation):
    '''Given a flow_graph, fg, construct a pipeline
    for demodulating a broadcast FM signal.  The
    input is the downconverteed complex baseband
    signal. The output is the demodulated audio.

    build_pipeline returns a two element tuple
    containing the input and output endpoints.
    '''
    fm_demod_gain = 2200.0/32768.0
    audio_rate = quad_rate / audio_decimation
    volume = 1.0

    # input: complex; output: float
    fm_demod = gr.quadrature_demod_cf (volume*fm_demod_gain)

    # compute FIR filter taps for audio filter
    width_of_transition_band = audio_rate / 32
    audio_coeffs = gr.firdes.low_pass (1.0,            # gain
                                       quad_rate,      # sampling rate
                                       audio_rate/2 - width_of_transition_band,
                                       width_of_transition_band,
                                       gr.firdes.WIN_HAMMING)

    TAU  = 75e-6  # 75us in US, 50us in EUR
    fftaps = [ 1 - math.exp(-1/TAU/quad_rate), 0]
    fbtaps= [ 0 , math.exp(-1/TAU/quad_rate) ]
    deemph = gr.iir_filter_ffd(fftaps,fbtaps)

    # input: float; output: float
    audio_filter = gr.fir_filter_fff (audio_decimation, audio_coeffs)

    fg.connect (fm_demod, deemph)
    fg.connect (deemph, audio_filter)
    return ((fm_demod, 0), (audio_filter, 0))


def main (args):
    parser = OptionParser (option_class=eng_option)
    (options, args) = parser.parse_args ()
    nargs = len (args)
    if nargs < 1 or nargs > 4:
        sys.stderr.write ('usage: wfm_rcv_many freq1...\n')
        sys.exit (1)

    freqs = [float(x) * 1e6 for x in args]
    freqs.sort ()

    min_freq = freqs[0]
    max_freq = freqs[-1]
    mid_freq = (min_freq + max_freq) / 2
    
    print eng_notation.num_to_str (mid_freq)
    
    # connect to RF front end
    rf_front_end = gr.microtune_4937_eval_board ()
    if not rf_front_end.board_present_p ():
        raise IOError, 'RF front end not found'

    # set front end gain
    rf_front_end.set_AGC (300)
    IF_freq = rf_front_end.get_output_freq ()

    rf_front_end.set_RF_freq (mid_freq)

    fg = build_graph (IF_freq, freqs, mid_freq)

    fg.start ()        # fork thread(s) and return
    raw_input ('Press Enter to quit: ')
    fg.stop ()

if __name__ == '__main__':
    main (sys.argv[1:])


