#include "CBRadio.h"
#include "CBTuner.h"
#include <qlcdnumber.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qwt_knob.h>
#include <qtimer.h>

#include <VrAR5000Source.h>
#include <VrComplexFIRfilter.h>
#include <VrComplexCascadefilter.h>
#include <VrAmplitudeDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>

const int inputRate = 33000000;
const int inputFreq = 10700000;

const int CFIRdecimate = 275;
const int CFIRdecimate2 = 5;
const int RFIRdecimate = 1;

const int quadRate = inputRate / CFIRdecimate / CFIRdecimate2;
const int audioRate = quadRate / RFIRdecimate;

const int chanTaps = 40;
const float chanGain = 2.0;

const int cascTaps = 50;
const float cascFreq = 1.0;
const float cascCutoff = audioRate/2;
const float cascGain = 2.0;

const float demodGain = 2200;

const int ifTaps = 20;
const float ifGain = 1.0;


CBRadio::CBRadio(VrAR5000Source<char> *source, 
		 QWidget *p, const char *name) :
  QWidget(p, name),
  source(source)
{
  chanFilter = new VrComplexFIRfilter<char>(CFIRdecimate, chanTaps, 
					    inputFreq, chanGain);
  cascFilter = new VrComplexCascadefilter<complex>(CFIRdecimate2, cascTaps,
						   cascFreq, cascCutoff,
						   cascGain);
  demod = new VrAmplitudeDemod<float>(0.0, 0.05);
  ifFilter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
					      ifTaps, ifGain);
  audio = new VrAudioSink<short>();

  CONNECT(audio, ifFilter, audioRate, 16);
  CONNECT(ifFilter, demod, quadRate, 32);
  CONNECT(demod, cascFilter, quadRate, 64);
  CONNECT(cascFilter, chanFilter, inputRate/CFIRdecimate, 64);
  CONNECT(chanFilter, source, inputRate, 8);

  audio->setup();


  tuner = new CBTuner(this);
  tuner->setWrap(true);

  const QColorGroup &c = this->colorGroup();
  QColorGroup colors(green, black, c.light(), 
		     c.dark(), c.mid(), green, c.base());
  QPalette palette(colors, colors, colors);

  QBoxLayout *toplayout = new QHBoxLayout(this);
  toplayout->addSpacing(10);
  QBoxLayout *klayout = new QVBoxLayout();
  toplayout->addLayout(klayout);
  toplayout->addSpacing(10);

  vol = new QwtKnob(this);
  vol->setFixedSize(80, 80);
  vol->setRange(0, 10, 0.1);
  vol->setScaleMaxMajor(0);
  vol->setValue(0.0);
  klayout->addWidget(vol);

  QLabel *volLabel = new QLabel("Vol", vol);
  volLabel->adjustSize();
  volLabel->setAlignment(AlignCenter);
  volLabel->move((vol->width() - volLabel->width()) / 2,
		 vol->height() - volLabel->height());

  squelch = new QwtKnob(this);
  squelch->setFixedSize(80,80);
  squelch->setRange(0, 10, 0.1);
  squelch->setScaleMaxMajor(0);
  squelch->setValue(0.0);
  klayout->addWidget(squelch);
  klayout->addSpacing(10);

  QLabel *sqLabel = new QLabel("Sq", squelch);
  sqLabel->setAlignment(AlignCenter);
  sqLabel->adjustSize();
  sqLabel->move((squelch->width() - sqLabel->width()) / 2,
		 squelch->height() - sqLabel->height());
  

  QBoxLayout *layout = new QVBoxLayout();
  toplayout->addLayout(layout);
  toplayout->addSpacing(10);
  QBoxLayout *layout2 = new QHBoxLayout();
  layout->addSpacing(10);
  layout->addLayout(layout2);
  layout->addStretch(1);

  disp = new QLCDNumber(2, this);
  disp->setFixedSize(80,60);
  disp->setSegmentStyle(QLCDNumber::Flat);
  disp->setSmallDecimalPoint(true);
  disp->setPalette(palette);
  disp->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  disp->setLineWidth(3);
  layout2->addWidget(disp);
  layout2->addSpacing(10);

  QBoxLayout *clayout = new QVBoxLayout();
  layout2->addLayout(clayout);

  chanUp = new QPushButton("+", this);
  chanUp->setFixedWidth(40);
  chanUp->setAutoRepeat(true);
  clayout->addWidget(chanUp);
  
  QLabel *chanLabel = new QLabel("Ch", this);
  chanLabel->setAlignment(AlignCenter);
  clayout->addWidget(chanLabel);

  chanDown = new QPushButton("-", this);
  chanDown->setFixedWidth(40);
  chanDown->setAutoRepeat(true);
  clayout->addWidget(chanDown);

  talk = new QPushButton("Talk", this);
  talk->setFixedHeight(40);
  layout->addWidget(talk);
  layout->addSpacing(10);

  toplayout->activate();
  toplayout->freeze();

  QTimer *timer = new QTimer(this);

  connect(tuner, SIGNAL(channelChanged(int)), SLOT(setChannel(int)));
  connect(chanUp, SIGNAL(clicked()), tuner, SLOT(nextChannel()));
  connect(chanDown, SIGNAL(clicked()), tuner, SLOT(prevChannel()));
  connect(vol, SIGNAL(valueChanged(double)), SLOT(setVolume(double)));
  connect(timer, SIGNAL(timeout()), SLOT(run()));

  setChannel(tuner->channel());

  timer->start(10);
}

CBRadio::~CBRadio()
{
  delete audio;
  delete ifFilter;
  delete demod;
  delete cascFilter;
  delete chanFilter;
}

void
CBRadio::setChannel(int channel)
{
  tuner->setChannel(channel);
  source->setRxFrequency(tuner->frequency());
  disp->display(channel);
}

void
CBRadio::setVolume(double vol)
{
  this->vol->moveTo(vol);
  demod->setGain(vol * demodGain);
}

void
CBRadio::run()
{
  audio->start(1.0);
}
