/* -*- Mode: c++ -*-
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifdef PERFMON
#include <VrPerfGraph.h>
#include <VrSigProc.h>
#include <stdio.h>
#include <strings.h>

#define NUM_BARS 3

VrCycleCount *overhead = new VrCycleCount();

void VrPerfGraph::add(const char *name, VrSigProc *m) {

  if(num_modules >= MAXGRAPHMODULES) 
    return;

  VrSigProc ** ms = new (VrSigProc *[num_modules+1]);
  ms[num_modules]=m;
  labels[num_modules]=name;
  if(num_modules>0) {
    for(int i=0;i<num_modules;i++) {
      ms[i]=modules[i];
      if(modules[i]==m) {
	delete ms;
	return; //already added
      }
    }
    delete modules;
  }
  modules=ms;
  num_modules++;
}

void VrPerfGraph::outputGraph(double seconds) {
    
#ifdef PERFMON
  FILE *f=fopen("/proc/cpuinfo","r");
  float cpu_mhz=0;
  if(f!=NULL) {
    char line[80];
    while(fgets(line,80,f)!=NULL) {
      float b=0;
      if(sscanf(line,"cpu MHz\t: %f",&b)!=0) {
	if(b>10 && b<5000) {
	  cpu_mhz=b;
	  break;
	}
      }
    }
  }

  FILE *fd=popen("gnuplot -geometry 800x500","w");
  int i;

  fprintf(fd,"set xlabel \"%s\"\n",title);
  fprintf(fd,"set xtics (");
  for(i=0;i<num_modules;i++) {
    fprintf(fd,"\"%s\" %d",labels[i],i+1);
    if(i+1<num_modules)
      fprintf(fd,", ");
  }
  fprintf(fd,")\n");

  fprintf(fd,"set ylabel \"MCycles\"\n");
  if(cpu_mhz>0) {
    fprintf(fd,"set y2label \"%%CPU\"\n");
    fprintf(fd,"set ytics nomirror\n");
    fprintf(fd,"set y2tics\n");
  }

  fprintf(fd,"set title \"Performance Results on ");
  if(cpu_mhz>0) 
    fprintf(fd,"%.1f",cpu_mhz);
  else
    fprintf(fd,"???");
  fprintf(fd," MHz machine (for %f seconds)\"\n",seconds);
  //  fprintf(fd,"set nokey\n");
  //  printf("set xtics 0,history\n");
  fprintf(fd,"set time\n");
  fprintf(fd,"set grid\n");

  float bar_width = 1/(float) (NUM_BARS+2);
  fprintf(fd,"set boxwidth %.2f\n",bar_width);

  float max_cycles=1;
  for(i=0; i<num_modules; i++) {
    if(max_cycles<modules[i]->getTotalCycles())
      max_cycles=modules[i]->getTotalCycles();
  }

  float divisor1 = 1000000; //Million
  if(cpu_mhz>0) {
    //Convert to percent
    float divisor2 = 10000; //Million / 100
    divisor2 *= (seconds*cpu_mhz); // Divide by # Mcycles in seconds
    fprintf(fd,"set y2range [0:%d]\n",(int) (((max_cycles / divisor2) * 1.1)+.5));
  }

  max_cycles /= divisor1; 
  max_cycles *= 1.1;     //Add a little
  max_cycles += .5;      //Round up

  fprintf(fd,"plot [0:%d][%d:%d] \"-\" title \"cycles\" w boxes",num_modules+1, 0, (int) max_cycles);
  fprintf(fd,", \"-\" title \"%s\" w boxes",PERF1_NAME);
  fprintf(fd,", \"-\" title \"%s\" w boxes\n",PERF2_NAME);

  for(int j=0;j<NUM_BARS;j++) {
    for(i=0; i<num_modules; i++) {
      fprintf(fd,"%f %f\n",i+1+(bar_width*(j-NUM_BARS/2.0)), ((float) modules[i]->getTotalCycles(j))/divisor1);
    }
    fprintf(fd,"e\n");
  }

  fflush(fd);

  fprintf(stderr, "\nPress enter:\n");
  //  sleep(10);
  getchar();
#else

  fprintf(stderr, "You must define PERFMON before compiling to get performance results\n");

#endif
}

void VrPerfGraph::print_stats() {
    
#ifdef PERFMON
  int i;
  fprintf(stderr, "Total samples generated by module:\n");
  for(i=0; i<num_modules; i++) {
    const char *n = modules[i]->name();
    fprintf(stderr, "%20s: %15ld\n",n, modules[i]->getTotalSamples());
  }

  long long total[NUM_BARS];
  for(int j=0;j<NUM_BARS;j++) {
    fprintf(stderr, "\nTotal (Average per sample) %s generated by module:\n",VrCycleCount::measurementName(j));
    total[j]=0;
    for(i=0; i<num_modules; i++) {
      const char *n = modules[i]->name();
      fprintf(stderr, "%20s: %15lld (%7lld)\n",n, modules[i]->getTotalCycles(j),
	      modules[i]->getCyclesPerSample(j));
      total[j]+=modules[i]->getTotalCycles(j);
    }
  }

  fprintf(stderr, "\nTotal modules/overhead:\n");
  for(int j=0;j<NUM_BARS;j++) {
    char *n=VrCycleCount::measurementName(j);
    long long system = overhead->getTotalCycles(j);
    long long sysoverhead = system - total[j]; 
    fprintf(stderr, "%s:\n       %18lld / %18lld %02.2f%%\n",n, total[j], sysoverhead, (sysoverhead*100.0)/system);
  }

#else

  fprintf(stderr, "You must define PERFMON before compiling to get performance results\n");

#endif
}
#endif /* PERFMON */
