/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

////////////////////////////////////////////////////////////////////////
// GnuRadio.i
// SWIG interface definition
//

////////////////////////////////////////////////////////////////////////


// set it on the command line
// %module GnuRadio


////////////////////////////////////////////////////////////////////////
// Headers


%{

#include <VrTypes.h>
#include <VrFileSink.h>
#include <VrFileSource.h>
#include <GrMagnitude.h>
#include <GrAudioSource.h>
#include <GrAudioSink.h>
#include <VrSigSource.h>
#include <GrNoiseSource.h>
#include <VrConnect.h>
#include <VrMultiTask.h>
#include <GrSSBMod.h>
#include <VrAmp.h>
#include <GrHilbert.h>
#include <sdr_1000.h>
#include "mainloop.h"
#include "Python.h"
#include <vector>
#include <gr_FlowGraph.h>
#include <stddef.h>		// size_t

%}


////////////////////////////////////////////////////////////////////////
// Core Classes


struct VrSampleRange;
class VrMultiTask;
class VrBuffer;
class VrConnect;
class VrSigProc;
class VrSink;
class VrSource;
class VrSigSource;
class mainloop;

////////////////////////////////////////////////////////////////////////
// VrTypes.h

typedef unsigned long long VrSampleIndex;

typedef struct {
	VrSampleIndex index;
	unsigned long size;
} VrSampleRange; 


////////////////////////////////////////////////////////////////////////
// VrComplex.h

class VrComplex {
public:
  VrComplex (float r, float i);
  ~VrComplex ();
};


////////////////////////////////////////////////////////////////////////
// C++ STL vectors (probably needs typemap)

%{
typedef std::vector<short>	vector_short;
typedef std::vector<float>	vector_float;
typedef std::vector<VrComplex>	vector_VrComplex;
%}

typedef std::vector<short>	vector_short;
typedef std::vector<float>	vector_float;
typedef std::vector<VrComplex>	vector_VrComplex;

%include "PythonTypeMaps.i"


////////////////////////////////////////////////////////////////////////
// VrMultiTask.h


class VrMultiTask {
public:
  void add(VrSigProc *s);
  void start();
  void stop();
  void process();

  int isStarted();
  void startTimer();
  double elapsedTime();

  VrMultiTask();
  virtual ~VrMultiTask();
};



////////////////////////////////////////////////////////////////////////
// gr_FlowGraph.h

%newobject gr_FlowGraph::make;
%newobject gr_FlowGraph_real::make;

%name (gr_FlowGraph_real) class gr_FlowGraph {
  %rename(connectn) connect(VrSigProc*, int, VrSigProc*);

public:
  virtual ~gr_FlowGraph() = 0;
  virtual bool start () = 0;	// for thread(s) and return immediately
  virtual void stop () = 0;	// stop threads
  virtual void wait () = 0;	// wait for threads to stop (typically blocks forever)
  virtual bool isRunning () = 0;
  virtual bool connect (VrSigProc *upstream, VrSigProc *downstream);
  virtual bool connect (VrSigProc *upstream, int upstream_port, VrSigProc *downstream);

  //! class method to create concrete instance
  static gr_FlowGraph *make ();
};



////////////////////////////////////////////////////////////////////////
// mainloop.h

class mainloop {
public:	
  bool		exit_now;

  mainloop ();

  void loop (VrMultiTask *);
};


////////////////////////////////////////////////////////////////////////
// VrSigProc.h


extern unsigned int cacheSize;
extern float maxLatency;

class VrSigProc {

// Hide constructor and destructor since they're abstract.
private:
  VrSigProc(
    int number_of_outputs,
    unsigned int arg_itype_size, 
    unsigned int arg_type_size);
  virtual ~VrSigProc();

public:
  bool connect_proc (VrSigProc* proc, int n);
  virtual const char *name();

#if 0
  virtual int forecast(VrSampleRange output, VrSampleRange inputs[]); 
  virtual int work(VrSampleRange output, void *o[],
		   VrSampleRange inputs[], void *i[]); 
  virtual int setSamplingFrequency(double sf); 
  VrSampleIndex getWP();
#endif

  bool isSink ();
  double getSamplingFrequency();
};

void NWO_CONNECTN (VrSigProc *src, int n, VrSigProc *dst);
void NWO_CONNECT  (VrSigProc *src, VrSigProc *dst);



////////////////////////////////////////////////////////////////////////
// VrHistoryProc, VrDecimatingSigProc, VrInterpolatingSigProc

template<class iType, class oType> 
class VrHistoryProc : public VrSigProc {
};

%template(VrHistoryProcCC) VrHistoryProc<VrComplex,VrComplex>;
%template(VrHistoryProcCF) VrHistoryProc<VrComplex,float>;
%template(VrHistoryProcFC) VrHistoryProc<float,VrComplex>;
%template(VrHistoryProcFF) VrHistoryProc<float,float>;
%template(VrHistoryProcFS) VrHistoryProc<float,short>;
%template(VrHistoryProcSC) VrHistoryProc<short,VrComplex>;
%template(VrHistoryProcSI) VrHistoryProc<short,int>;
%template(VrHistoryProcSF) VrHistoryProc<short,float>;


template<class iType, class oType> 
class VrDecimatingSigProc : public VrHistoryProc<iType,oType> {
};

%template(VrDecimatingSigProcCC) VrDecimatingSigProc<VrComplex,VrComplex>;
%template(VrDecimatingSigProcCF) VrDecimatingSigProc<VrComplex,float>;
%template(VrDecimatingSigProcFC) VrDecimatingSigProc<float,VrComplex>;
%template(VrDecimatingSigProcFF) VrDecimatingSigProc<float,float>;
%template(VrDecimatingSigProcFS) VrDecimatingSigProc<float,short>;
%template(VrDecimatingSigProcSC) VrDecimatingSigProc<short,VrComplex>;
%template(VrDecimatingSigProcSI) VrDecimatingSigProc<short,int>;


template<class iType, class oType> 
class VrInterpolatingSigProc : public VrHistoryProc<iType,oType> {
};


////////////////////////////////////////////////////////////////////////
// Sources


template<class oType> 
class VrSource : public VrSigProc {

// Hide constructor since it's abstract.
private:
  VrSource();
  ~VrSource();

public:
  // virtual int work2(VrSampleRange output, void *o[]);
  // virtual bool isConnectedToSource();
};

%template(VrSource_char)  VrSource<char>;
%template(VrSource_short) VrSource<short>;
%template(VrSource_float) VrSource<float>;
%template(VrSource_complex)  VrSource<VrComplex>;


template<class oType>
class VrFileSource : public VrSource<oType> {
public:
  VrFileSource(double sampling_freq, const char *file, bool repeat = 0);
  virtual ~VrFileSource();
};


%template(VrFileSource_char) VrFileSource<char>;
%template(VrFileSource_short) VrFileSource<short>;
%template(VrFileSource_float) VrFileSource<float>;


enum {VR_SQUARE_WAVE, VR_TRIANGLE_WAVE, VR_SIN_WAVE, VR_DC_WAVE, VR_COS_WAVE};

template<class oType> 
class VrSigSource : public VrSource<oType> {
public:
  // %name(VrSigSource_freq_wf_f_a) VrSigSource(double sampling_freq, int wf, double f, double a);
  // %name(VrSigSource_freq) VrSigSource(double sampling_freq);
  VrSigSource (double sampling_freq, int wf, double f, double a);

  virtual ~VrSigSource();

  int getWaveform();
  void setWaveform(int w);
  double getFrequency();

  void setFrequency(double f);
  double getAmplitude();
  void setAmplitude(double a);
};


%template(VrSigSourceF) VrSigSource<float>;
%template(VrSigSourceS) VrSigSource<short>;

enum { GR_GAUSSIAN, GR_LAPLACIAN, GR_IMPULSE };

template<class oType>
class GrNoiseSource : public VrSource<oType> {
public:
  GrNoiseSource(double sampfreq, int type, float amp, int seed = 3021);
};

%template(GrNoiseSourceF) GrNoiseSource<float>;

/// --- mc4020 stuff constructor

%{
#include <mc4020_bitmask.h>
#include <make_GrMC4020Source.h>
#include <VrFixOffset.h>
%}

%include <mc4020_bitmask.h>

// produce a stream of shorts from the MC4020 A/D

%newobject make_GrMC4020Source (double, unsigned long);

VrSource<short> *
make_GrMC4020SourceS (double sampling_freq = 20e6,
		      unsigned long bitmask = MCC_CH0_EN | MCC_ALL_5V);


////////////////////////////////////////////////////////////////////////
// Sinks

template<class iType> 
class VrSink : public VrSigProc {

// Hide constructor since it's abstract.
private:
  VrSink();
  ~VrSink();

public:
  //  void setOptimalSize(unsigned int s);
  // void setup_upstream(); 
  // virtual int work3(VrSampleRange output, 
  // VrSampleRange inputs[], void *i[]);
};


%template(VrSink_char)  VrSink<char>;
%template(VrSink_short) VrSink<short>;
%template(VrSink_float) VrSink<float>;


template<class iType> 
class VrFileSink : public VrSink<iType> {
public:
  VrFileSink(char *filefile);
  virtual ~VrFileSink();
};

%template(VrFileSink_char)  VrFileSink<char>;
%template(VrFileSink_short) VrFileSink<short>;
%template(VrFileSink_float) VrFileSink<float>;



////////////////////////////////////////////////////////////////////////
// Audio Sources and Sinks


template<class oType>
class GrAudioSource : public VrSource<oType> {
public: 
  GrAudioSource(double sampling_freq, int streams,
		double output_range = 32767, const char* dev = "/dev/dsp");
  virtual ~GrAudioSource();
};

%template(GrAudioSourceS) GrAudioSource<short>;
%template(GrAudioSourceF) GrAudioSource<float>;
%template(GrAudioSourceC) GrAudioSource<VrComplex>;


template<class iType>
class GrAudioSink : public VrSink<iType> {
public:
  GrAudioSink(iType input_range = 32767, const char* dev = "/dev/dsp");
  virtual ~GrAudioSink();
};

%template(GrAudioSinkS) GrAudioSink<short>;
%template(GrAudioSinkF) GrAudioSink<float>;
// %template(GrAudioSinkC) GrAudioSink<VrComplex>;


////////////////////////////////////////////////////////////////////////
// General purpose modules


template<class iType, class oType>
class GrMagnitude : public VrSigProc {
 public:
  GrMagnitude ();
};

%template(GrMagnitudeCF) GrMagnitude<VrComplex,float>;


template<class oType> 
class GrSSBMod : public VrSigProc {
public:
  GrSSBMod(float freq, float gain);
  void set_freq (double freq);
  void set_gain (float g);
};

%template(GrSSBModCF) GrSSBMod<float>;

template<class iType,class oType>
class VrAmp : public VrSigProc {
public:
  void setGain(float gn);
  VrAmp(float gn = 1.0);
};

%template(VrAmpFF) VrAmp<float,float>;
%template(VrAmpSS) VrAmp<short,short>;
%template(VrAmpCC) VrAmp<VrComplex,VrComplex>;

template<class iType>
class GrHilbert : public VrHistoryProc<iType,VrComplex> {
public:
  GrHilbert(unsigned int taps);
};

%template(GrHilbertFC) GrHilbert<float>;
%template(GrHilbertSC) GrHilbert<short>;

template<class iType, class oType>
class VrFixOffset : public VrSigProc {
public: 
  VrFixOffset ();
  ~VrFixOffset ();
};

%template(VrFixOffsetSS) VrFixOffset<short,short>;
%template(VrFixOffsetSF) VrFixOffset<short,float>;


%{
#include <VrQuadratureDemod.h>
%}

template<class oType> 
class VrQuadratureDemod : public VrHistoryProc<VrComplex,oType> {
public:
  void setGain(float gain);
  VrQuadratureDemod(oType gain);
};

%template(VrQuadratureDemodCF) VrQuadratureDemod<float>;


%{
#include <GrAdd.h>
#include <GrMultiply.h>
%}

template<class iType,class oType> 
class GrAdd : public VrSigProc {
public:
  GrAdd ();
};

template<class iType,class oType> 
class GrMultiply : public VrSigProc {
public:
  GrMultiply ();
};

%template(GrAddFF) GrAdd<float,float>;
%template(GrMultiplyFF) GrMultiply<float,float>;



////////////////////////////////////////////////////////////////////////
/// filter design functions & filter modules

%{
#include <gr_firdes.h>
#include <generated.h>
%}

%include "gr_firdes.i"
%include "generated.i"

////////////////////////////////////////////////////////////////////////
/// SDR 1000


const int L_EXT  = 0;
const int L_BAND = 1;
const int L_DDS0 = 2;
const int L_DDS1 = 3;

class sdr_1000_base {
public:

  sdr_1000_base (int which_pp = 0);
  ~sdr_1000_base ();

  void reset ();
  void write_latch (int which, int value, int mask);
};

////////////////////////////////////////////////////////////////////////
/// ATSC / dtv

%include "atsc.i"

////////////////////////////////////////////////////////////////////////
/// Microtune eval board control

%include "microtune.i"

////////////////////////////////////////////////////////////////////////
/// New, non-template based sources and sinks

%{

#include <GrSource.h>
#include <GrSink.h>
#include <GrNullSink.h>
#include <GrNullSource.h>
#include <GrFileDescriptorSource.h>
#include <GrFileDescriptorSink.h>
#include <GrFileSource.h>
#include <GrFileSink.h>
#include <GrUsrpSink.h>
#include <GrUsrpSource.h>
#include <GrComplexNCOSource.h>
#include <GrOscopeSinkX.h>
#include <GrOscopeSinkC.h>
#include <GrOscopeSinkF.h>

%}

class GrSource : public VrSigProc {
public:
  GrSource (size_t sizeof_oType) : VrSigProc (1, 0, sizeof_oType) {};
  GrSource (size_t sizeof_oType, int outputs) : VrSigProc (outputs, 0, sizeof_oType) {};

  virtual int work2 (VrSampleRange output, void *o[]) = 0;
}; 


class GrSink : public VrSigProc {
public: 
  GrSink (size_t sizeof_iType);

  virtual int work3 (VrSampleRange output, 
		     VrSampleRange inputs[], void *i[]) = 0; 

  // FIXME does anyone actually use this?
  //  set this if you want the sink to use smaller chunks
  //  than the maxLatency would dictate
  void setOptimalSize (unsigned int s);

};

class GrFileDescriptorSource : public GrSource
{
public:
  GrFileDescriptorSource (size_t itemsize, double sampling_freq, int file_descriptor);
  ~GrFileDescriptorSource ();

  virtual int work2 (VrSampleRange output, void *o[]);
};

class GrFileDescriptorSink : public GrSink
{
public:
  GrFileDescriptorSink (size_t itemsize, int file_descriptor);
  ~GrFileDescriptorSink ();

  virtual int work3 (VrSampleRange output, 
		     VrSampleRange inputs[], void *i[]);
};

class GrFileSource : public GrSource
{
public:
  GrFileSource (size_t itemsize, double sampling_freq, const char *filename, bool repeat);
  ~GrFileSource ();

  virtual int work2 (VrSampleRange output, void *o[]);
};

class GrFileSink : public GrSink
{
public:
  GrFileSink (size_t itemsize, const char *filename);
  ~GrFileSink ();

  virtual int work3 (VrSampleRange output,
		     VrSampleRange inputs[], void *i[]);
};

class GrNullSink : public GrSink 
{
 public:
  GrNullSink (size_t sizeof_iType);

  virtual int work3 (VrSampleRange output,
		     VrSampleRange inputs[], void *i[]);

};

class GrNullSource : public GrSource
{
public:
  GrNullSource (size_t sizeof_oType, double sampling_freq);
  virtual int work2 (VrSampleRange output, void *o[]);
};

class GrUsrpSink : public GrSink
{
 public:
  GrUsrpSink (int which_board,
	      int channel_selection_bitmask,
	      unsigned int interp_rate);
  ~GrUsrpSink ();

  virtual int work3 (VrSampleRange output,
		     VrSampleRange inputs[], void *i[]);

  bool set_interp_rate (unsigned int rate);
  bool set_tx_freq (int channel, double freq);
  bool set_ext_clk_div (unsigned int div);

  bool _set_interp_reg (unsigned int reg);	// diagnostics only

  // ACCESSORS

  double  get_oscillator_freq () const; 	// freq of master oscillator on board
  unsigned int get_interp_rate () const;
  double get_tx_freq (int channel) const;
  unsigned int get_ext_clk_div () const;
};

class GrUsrpSource : public GrSource
{
 public:
  GrUsrpSource (int which_board, int channel_selection_bitmask,
		unsigned int adc_clk_div, unsigned int decim_rate);
  ~GrUsrpSource ();

  virtual int work2 (VrSampleRange output, void *o[]);

  bool set_adc_clk_div (unsigned int div);
  bool set_decim_rate (unsigned int rate);
  bool set_rx_freq (int channel, double freq);
  bool set_ext_clk_div (unsigned int div);

  bool _set_decim_reg (unsigned int regval);	// diagnostics only

  // ACCESSORS

  double  get_oscillator_freq () const; 	// freq of master oscillator on board
  unsigned int get_adc_clk_div () const;
  unsigned int get_decim_rate () const;
  double get_rx_freq (int channel) const;
  unsigned int get_ext_clk_div () const;
};

class GrComplexNCOSource : public GrSource
{
 public:
  GrComplexNCOSource (double sampling_rate, double freq, double ampl);

  virtual int work2 (VrSampleRange output, void *o[]);
};


////////////////////////////////////////////////
//  python oscilloscope building blocks


enum gr_TriggerMode {
  gr_TRIG_AUTO,			// auto trigger (on anything)
  gr_TRIG_POS_SLOPE,		// trigger on positive slope across trigger level
  gr_TRIG_NEG_SLOPE		// trigger on negative slope across trigger level
};

class GrOscopeSinkX : public GrSink
{
protected:
  gr_OscopeGuts		*d_guts;

public:
  GrOscopeSinkX (int input_itemsize);
  ~GrOscopeSinkX ();

  bool setUpdateRate (double update_rate);
  bool setDecimationCount (int decimation_count);
  bool setTriggerChannel (int channel);
  bool setTriggerMode (gr_TriggerMode mode);
  bool setTriggerLevel (double trigger_level);
  bool setTriggerLevelAuto ();				// set to 50% level


  // ACCESSORS
  int getNumChannels () const;
  double getSamplingRate () const;
  double getUpdateRate () const;
  int getDecimationCount () const;
  int getTriggerChannel () const;
  gr_TriggerMode getTriggerMode () const;
  double getTriggerLevel () const;

  // # of samples written to each output record.
  int getSamplesPerOutputRecord () const;
};

class GrOscopeSinkF : public GrOscopeSinkX
{
private:
  int			 d_fd;
  float			*d_tmp;

public:
  GrOscopeSinkF (int file_descriptor);
  ~GrOscopeSinkF ();

  virtual const char *name () { return "GrOscopeSinkF"; }

  virtual void initialize ();
  
  virtual int work3 (VrSampleRange output,
		     VrSampleRange inputs[], void *ai[]);

};

class GrOscopeSinkC : public GrOscopeSinkX
{
private:
  int			 d_fd;
  float			*d_tmp;

public:
  GrOscopeSinkC (int file_descriptor);
  ~GrOscopeSinkC ();

  virtual const char *name () { return "GrOscopeSinkC"; }

  virtual void initialize ();
  
  virtual int work3 (VrSampleRange output,
		     VrSampleRange inputs[], void *ai[]);

};



%constant int sizeof_char 	= sizeof(char);
%constant int sizeof_short	= sizeof(short);
%constant int sizeof_int	= sizeof(int);
%constant int sizeof_float	= sizeof(float);
%constant int sizeof_double	= sizeof(double);
%constant int sizeof_VrComplex 	= sizeof (VrComplex);


////////////////////////////////////////////////////////////////////////
/// Serial / Parallel conversion

%{
#include <GrSerialToParallel.h>
#include <GrParallelToSerial.h>
%}

class GrSerialToParallel : public VrSigProc
{
public:
  GrSerialToParallel (int item_size, int items_per_block);
  ~GrSerialToParallel ();
};

class GrParallelToSerial : public VrSigProc
{
public:
  GrParallelToSerial (int item_size, int items_per_block);
  ~GrParallelToSerial ();
};

////////////////////////////////////////////////////////////////////////
/// Float / Complex conversion

%{
#include <GrFloatToComplex.h>
%}

class GrFloatToComplex : public VrSigProc
{
 public:
  GrFloatToComplex ();
};



////////////////////////////////////////////////////////////////////////
/// FFT blocks

%{

#include <GrFFT_vCC.h>
#include <GrFFT_vFC.h>

%}

/*!
 * \brief compute forward or reverse FFT.  complex vector in / complex vector out
 */

class GrFFT_vCC : public VrSigProc
{
 public:
  GrFFT_vCC (int fft_size, bool forward = true, bool window = true);
  ~GrFFT_vCC ();

  virtual int work (VrSampleRange output, void *o[],
		    VrSampleRange inputs[], void *i[]);
};

/*!
 * \brief compute forward or reverse FFT.  float vector in / complex vector out
 */

class GrFFT_vFC : public VrSigProc
{
 public:
  GrFFT_vFC (int fft_size, bool forward = true, bool window = true);
  ~GrFFT_vFC ();

  virtual int work (VrSampleRange output, void *o[],
		    VrSampleRange inputs[], void *i[]);
};

////////////////////////////////////////////////////////////////////////
/// misc

%{

#include <GrConvertSF.h>
#include <GrConvertFS.h>
#include <GrKeepOneInN.h>

%}

/*!
 * \brief convert shorts to floats
 */

class GrConvertSF : public VrHistoryProc<short, float>
{
public:
  GrConvertSF ();
  ~GrConvertSF ();

  int work (VrSampleRange output, void *o[],
	    VrSampleRange inputs[], void *i[]);
};

/*!
 * \brief convert float to shorts with rounding and saturation
 */

class GrConvertFS : public VrHistoryProc<float,short>
{
public:
  GrConvertFS ();
  ~GrConvertFS ();

  int work (VrSampleRange output, void *o[],
	    VrSampleRange inputs[], void *i[]);
};

class GrKeepOneInN : public VrSigProc {
public:
  GrKeepOneInN (int item_size, int N);
  ~GrKeepOneInN ();
};


////////////////////////////////////////////////////////////////////////

#if 0	// no longer needed since we typemap to and from Python lists
%inline %{
  int vector_sizeF (vector_float *v){ return v->size (); }
  int vector_sizeS (vector_short *v){ return v->size (); }
  int vector_sizeC (vector_VrComplex *v){ return v->size (); }
%}
#endif

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
