#!/usr/bin/env python
#
# Copyright 2003 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from GnuRadio import *
from wxPyPlot import wxPyPlot
import wx
import Numeric
import os
import threading

# ========================================================================
# returns (sigproc, win).
#   sigproc requires a single input stream of float
#   win is a subclass of wxWindow

def makeFFTSinkF (fg, parent, label, fft_size, input_rate):
    (r_fd, w_fd) = os.pipe ()
    fft_rate = 20

    s2p = GrSerialToParallel (sizeof_float, fft_size)
    one_in_N = GrKeepOneInN (sizeof_float * fft_size,
                             int (input_rate / fft_size / fft_rate))
    fft = GrFFT_vFC (fft_size, True, True)
    dst = GrFileDescriptorSink (sizeof_VrComplex * fft_size, w_fd)

    fg.connect (s2p, one_in_N)
    fg.connect (one_in_N, fft)
    fg.connect (fft, dst)

    sigproc = s2p                       # head of pipeline

    win = FFT_Window (FFT_info (r_fd, fft_size, input_rate, True, label), parent)
    
    return (sigproc, win)

# ========================================================================
# returns (sigproc, win).
#   sigproc requires a single input stream of VrComplex
#   win is a subclass of wxWindow

def makeFFTSinkC (fg, parent, label, fft_size, input_rate):
    (r_fd, w_fd) = os.pipe ()
    fft_rate = 20

    s2p = GrSerialToParallel (sizeof_VrComplex, fft_size)
    one_in_N = GrKeepOneInN (sizeof_VrComplex * fft_size,
                             int (input_rate / fft_size / fft_rate))
    fft = GrFFT_vCC (fft_size, True, True)
    dst = GrFileDescriptorSink (sizeof_VrComplex * fft_size, w_fd)

    fg.connect (s2p, one_in_N)
    fg.connect (one_in_N, fft)
    fg.connect (fft, dst)

    sigproc = s2p                       # head of pipeline

    win = FFT_Window (FFT_info (r_fd, fft_size, input_rate, False, label), parent)

    return (sigproc, win)

# ------------------------------------------------------------------------

wxDATA_EVENT = wx.NewEventType()

def EVT_DATA_EVENT(win, func):
    win.Connect(-1, -1, wxDATA_EVENT, func)

class DataEvent(wx.PyEvent):
    def __init__(self, data):
        wx.PyEvent.__init__(self)
        self.SetEventType (wxDATA_EVENT)
        self.data = data

    def Clone (self): 
        self.__class__ (self.GetId())


class FFT_info:
    def __init__ (self, file_descriptor, fft_size, sample_rate, input_is_real, title = "FFT"):
        self.file_descriptor = file_descriptor
        self.fft_size = fft_size
        self.sample_rate = sample_rate
        self.input_is_real = input_is_real
        self.title = title;
        
class input_watcher (threading.Thread):
    def __init__ (self, file_descriptor, fft_size, event_receiver, **kwds):
        threading.Thread.__init__ (self, **kwds)
        self.file_descriptor = file_descriptor
        self.fft_size = fft_size
        self.event_receiver = event_receiver
        self.keep_running = True
        self.start ()

    def run (self):
        print "input_watcher: pid = ", os.getpid ()
        while (self.keep_running):
            s = os.read (self.file_descriptor, sizeof_VrComplex * self.fft_size)
            if not s:
                self.keep_running = False
                break

            complex_data = Numeric.fromstring (s, Numeric.Complex32)
            de = DataEvent (complex_data)
            wx.PostEvent (self.event_receiver, de)
            del de
    

class FFT_Window (wxPyPlot.PlotCanvas):
    def __init__ (self, info, parent, id = -1,
                  pos = wx.DefaultPosition, size = wx.DefaultSize,
                  style = wx.DEFAULT_FRAME_STYLE, name = ""):
        wxPyPlot.PlotCanvas.__init__ (self, parent, id, pos, size, style, name)

        self.SetEnableGrid (True)
        self.SetEnableZoom (True)
        # self.SetBackgroundColour ('black')
        
        self.info = info;
        self.y_range = None
        self.avg_y_min = None
        self.avg_y_max = None

        EVT_DATA_EVENT (self, self.set_data)

        self.input_watcher = input_watcher (info.file_descriptor,
                                            info.fft_size,
                                            self)


    def set_data (self, evt):
        data = evt.data
        dB = 20 * Numeric.log10 (abs(data))
        l = len (dB)

        if self.info.sample_rate >= 1e6:
            sf = 1e-6
            units = "MHz"
        else:
            sf = 1e-3
            units = "kHz"

        if self.info.input_is_real:     # only plot 1/2 the points
            x_vals = Numeric.arrayrange (l/2) * (self.info.sample_rate * sf / l)
            points = zip (x_vals, dB[0:l/2])
        else:
            # the "negative freqs" are in the second half of the array
            x_vals = Numeric.arrayrange (-l/2+1, l/2) * (self.info.sample_rate * sf / l)
            points = zip (x_vals, Numeric.concatenate ((dB[l/2:], dB[0:l/2])))


        lines = wxPyPlot.PolyLine (points, colour='LIME GREEN')

        graphics = wxPyPlot.PlotGraphics ([lines],
                                          title=self.info.title,
                                          xLabel = units, yLabel = "dB")

        self.Draw (graphics, xAxis=None, yAxis=self.y_range)
        self.update_y_range ()

    def update_y_range (self):
        alpha = 1.0/25
        graphics = self.last_draw[0]
        p1, p2 = graphics.boundingBox ()     # min, max points of graphics

        if self.avg_y_min:
            self.avg_y_min = p1[1] * alpha + self.avg_y_min * (1 - alpha)
            self.avg_y_max = p2[1] * alpha + self.avg_y_max * (1 - alpha)
        else:
            self.avg_y_min = p1[1]
            self.avg_y_max = p2[1]

        self.y_range = self._axisInterval ('auto', self.avg_y_min, self.avg_y_max)


# ----------------------------------------------------------------


if __name__ == '__main__':

    class TestPanel (wx.Panel):
        def __init__ (self, parent, frame):
            wx.Panel.__init__ (self, parent, -1)
            self.frame = frame

            input_name = "/home/eb/data/short.dat"
            input_rate = 20e6
            
            self.fg = gr_FlowGraph ()

            if 0:
                src = GrFileSource (sizeof_short, input_rate, input_name, True)
                short_to_float = GrConvertSF ()
                sigproc, fft_win = makeFFTSinkF (self.fg, self, "Secret Data", 512, input_rate)
                self.fg.connect (src, short_to_float)
                self.fg.connect (short_to_float, sigproc)
            else:
                src = GrComplexNCOSource (input_rate, 5.75e6, 1e3)
                sigproc, fft_win = makeFFTSinkC (self.fg, self, "Secret Data", 512, input_rate)
                self.fg.connect (src, sigproc)

            box = wx.BoxSizer (wx.VERTICAL)
            box.Add (fft_win, 1, wx.EXPAND)
            
            self.sizer = box
            self.SetSizer (self.sizer)
            self.SetAutoLayout (True)
            self.sizer.Fit (self)

            wx.EVT_CLOSE (self, self.OnCloseWindow)

        def OnCloseWindow (self, event):
            self.fg.stop ()
            self.Destroy ()


    class MainFrame (wx.Frame):
        def __init__ (self):
            wx.Frame.__init__(self, None, -1, "Testing...")

            self.CreateStatusBar ()
            mainmenu = wx.MenuBar ()
            menu = wx.Menu ()
            menu.Append (200, 'E&xit', 'Get outta here!')
            mainmenu.Append (menu, "&File")
            self.SetMenuBar (mainmenu)
            wx.EVT_MENU (self, 200, self.OnExit)
            self.panel = TestPanel (self, self)
            wx.EVT_CLOSE (self, self.OnCloseWindow)

        def OnCloseWindow (self, event):
            self.Destroy ()

        def OnExit (self, event):
            self.Close (True)


    class TestApp (wx.App):
        def OnInit(self):
            frame = MainFrame ()
            frame.Show (True)
            self.SetTopWindow (frame)
            print "FlowGraph: ", frame.panel.fg
            frame.panel.fg.start ()
            return True

    app = TestApp (0)
    app.MainLoop ()


# ----------------------------------------------------------------
