/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef _GRFRACTIONALINTERPOLATINGSIGPROC_H_
#define _GRFRACTIONALINTERPOLATINGSIGPROC_H_

#include <VrHistoryProc.h>
#include <stdlib.h>
#include <gr_math.h>
#include <gr_mmse_fir_interpolator.h>

/*!
 * \brief base class template for N/M interpolator.
 */

template<class iType, class oType>
class GrFractionalInterpolatingSigProc : public VrHistoryProc<iType,oType>
{
 public:
  GrFractionalInterpolatingSigProc (int outputs, int arg_N, int arg_M)
    : VrHistoryProc<iType,oType>(outputs), N(arg_N), M(arg_M) {}

  GrFractionalInterpolatingSigProc (int arg_N, int arg_M)
    : VrHistoryProc<iType,oType>(1), N(arg_N), M(arg_M) {}

  ~GrFractionalInterpolatingSigProc () {};

  void pre_initialize ();
  int forecast (VrSampleRange output, VrSampleRange inputs[]);

protected:
  int		N;	// numerator of ratio
  int		M;	// denominator of ratio
  int		iratio;	// smallest integer multiple of N/M
};

template<class iType,class oType> int
GrFractionalInterpolatingSigProc<iType,oType>::forecast(VrSampleRange output,
							VrSampleRange inputs[]) {
  /* 1:iratio ratio */
  /*
   * Note that this code only works correctly if output size of the
   * module (set using setOutputSize(n) in pre_initialize() ) is a multiple
   * of iratio.
   */
  assert ((output.size % iratio) == 0);
  
  for(unsigned int i=0;i<numberInputs;i++) {
    inputs[i].index=output.index/iratio; /* ! do not subtract history ! */
    inputs[i].size=output.size/iratio + history-1;
  }
  return 0;
}  


template<class iType,class oType> void 
GrFractionalInterpolatingSigProc<iType,oType>::pre_initialize ()
{
  if (getSamplingFrequency () == 0.0)
    setSamplingFrequency ((getInputSamplingFrequencyN (0) * N) / M);

  // reduce N/M to lowest terms and then use N*M as the size

  int d = gr_gcd (N, M);
  iratio = (N * M) / (d * d);
  
  setOutputSize(iratio);
}

#endif /* _GRFRACTIONALINTERPOLATINGSIGPROC_H_ */
