/*
    Copyright (C) 2004  Dale Mellor

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/


static const char RCSID []
   = "$Id: guile-hooks.c,v 1.5 2004/01/07 10:23:34 jmd Exp $";


#include "guile-hooks.h"


#ifdef HAVE_GUILE


#include <memory.h>
#include <dirent.h>
#include <stdio.h>
#include <pwd.h>
#include <unistd.h>

#include "cube.h"
#include "ui.h"
#include "widget-set.h"


/* The head of a singly linked list of menu items that the guile scripts request
   as they are executed. */

static Menu_Item_List *list_head = NULL;




/* This function is called from the menu when the user makes a selection. The
   data is a string which was registered with the menu system and gives the name
   of a scheme procedure to execute. */

static void run_scheme (gpointer data, guint action, GtkWidget *widget)
{
    char *buffer = (char*) malloc (strlen (data) + 3);

    sprintf (buffer, "(%s)", (char*) data);
    
    scm_eval_string (scm_makfrom0str (buffer));

    free (buffer);
}




/* Function callable from scheme (as gnubik-register-script) which allows a
   script to specify a menu entry and the name of a procedure to call when that
   menu entry is selected. Note that /Script-fu/ is always appended, so all
   scripts are forced under the Script-fu main menu item. This function prepends
   a new item to the head of the linked list, with the data necessary for the
   menu system to function. */

static SCM gnubik_register_script (SCM menu_location,
                                   SCM callback)
{
    const int string_length = SCM_STRING_LENGTH (menu_location);
    const char menu_path_prefix [] = "/Script-_fu/";
    Menu_Item_List *new_entry;
    char *const buffer
        = (char*) malloc (strlen (menu_path_prefix) + string_length + 1);

    
    strcpy (buffer, menu_path_prefix);
    strcat (buffer, SCM_STRING_CHARS (menu_location));

                 
    new_entry = (Menu_Item_List*) malloc (sizeof (Menu_Item_List));

    new_entry->entry.path = buffer;
    new_entry->entry.accelerator = 0;
    new_entry->entry.callback = run_scheme;
    new_entry->entry.callback_action = 1;
    new_entry->entry.item_type = NULL;
    new_entry->callback_data = g_strdup (SCM_STRING_CHARS (callback));
    new_entry->next = list_head;

    list_head = new_entry;

    return SCM_UNSPECIFIED;
}




/* Function callable from scheme as gnubik-cube-state which returns a structure
   reflecting the current state of the cube. */

static SCM gnubik_cube_state ()
{
    return make_scm_cube (the_cube);
}




/* Function which allows a scheme script to rotate the colours on the cube. The
   rotation is done entirely in memory, and then the cube gets rendered on
   screen when the menu pull-downs (which were activated to get the script
   running) disappear and the screen is redrawn. */

static SCM gnubik_rotate_fast (SCM list)
{
    for (; ! SCM_NULLP (list); list = SCM_CDR (list))
    {
        Slice_Blocks *const blocks
            = identify_blocks_2 (the_cube,
                                 SCM_INUM (SCM_CADAR (list)),
                                 SCM_INUM (SCM_CAAR (list)));

        rotate_slice (the_cube,
                      SCM_INUM (SCM_CADDAR (list)) == 0 ? 1 : 3,
                      blocks);

        free_slice_blocks (blocks);
    }

    return SCM_UNSPECIFIED;
}




/* Function which, when called from scheme as gnubik-rotate-animated, causes one
   side of the cube to rotate on-screen. */

static SCM gnubik_rotate_animated (SCM list)
{
    for (; ! SCM_NULLP (list); list = SCM_CDR (list))
    {
        struct move_data move;

        move.axis = SCM_INUM (SCM_CAAR (list));
        move.slice = SCM_INUM (SCM_CADAR (list));
        move.dir = SCM_INUM (SCM_CADDAR (list));
        
        request_rotation (&move);
    }

    return SCM_UNSPECIFIED;
}



/* Function to allow a guile script to display a message to the user. */

static SCM gnubik_error_dialog (SCM message)
{
#if WIDGET_SET_gtk
    error_dialog (main_application_window, SCM_STRING_CHARS (message));
#endif
    
    return SCM_UNSPECIFIED;
}




/* Function to scan the named directory for all files with a .scm extension, and
   execute the contents of each file. */

static void read_script_directory (const char *const dir_name)
{
    static char buffer [1024];
    
    DIR *const directory = opendir (dir_name);

    if (directory)
    {
        struct dirent *entry;

        for (entry = readdir (directory); entry; entry = readdir (directory))

            if (strcmp (".scm",
                        entry->d_name + strlen (entry->d_name) - 4)
                     == 0)
            {
                snprintf (buffer, 1024, "%s/%s", dir_name, entry->d_name);

                scm_primitive_load (scm_makfrom0str (buffer));
            }
    }
}

        


/* This function initializes the scheme world for us, and once the scripts have
   all been run, it returns the requested menu structure to the caller. Before
   running the scripts, however, it first makes sure all the pertinent C
   functions are registered in the guile world. */

Menu_Item_List *startup_guile_scripts ()
{
    static const char local_dir [] = "/.gnubik/scripts";
    const char *home_dir;
    char *buffer;

    
    /* Register C functions that the scheme world can access. */

    scm_c_define_gsubr ("gnubik-register-script",
                        2, 0, 0,
                        gnubik_register_script);

    scm_c_define_gsubr ("gnubik-cube-state",
                        0, 0, 0,
                        gnubik_cube_state);
    
    scm_c_define_gsubr ("gnubik-rotate-animated",
                        1, 0, 0,
                        gnubik_rotate_animated);

    scm_c_define_gsubr ("gnubik-rotate-fast",
                        1, 0, 0,
                        gnubik_rotate_fast);

    scm_c_define_gsubr ("gnubik-error-dialog",
                        1, 0, 0,
                        gnubik_error_dialog);
    

    /* Run all the initialization files in .../share/gnubik/guile, and the
       system scripts in .../share/gnubik/scripts. */
    
    read_script_directory (GUILEDIR);
    read_script_directory (SCRIPTDIR);


    /* Run all the user scripts in $(HOME)/.gnubik/scripts. */
    
    home_dir = getenv ("HOME");

#if HAVE_GETPWUID
    if (home_dir == NULL)
        home_dir = getpwuid (getuid ())->pw_dir;
#endif    

    if (home_dir) 
      {
	buffer = (char*) malloc (strlen (home_dir) + strlen (local_dir) + 1);
	strcpy (buffer, home_dir);
	strcat (buffer, local_dir);
    
	read_script_directory (buffer);

	free (buffer);
      }


    /* Hopefully the scripts that we have run will have called
       gnubik_register_script above to populate the list_head, which we now
       return to the calling application which can in turn make the menu entries
       that run the scripts. */
    
    return list_head;
}


#endif  /* defined HAVE_GUILE. */
