/*
    A system to permit user selection of a block and rotation axis
    of a magic cube.
    Copyright (C) 1998, 2003  John Darrington

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
static const char RCSID[]="$Id: select-athena.c,v 1.4 2003/10/26 04:32:51 jmd Exp $";


/* This library provides  a means of picking a block using the mouse cursor.

Two mutually co-operative mechanisms are used in this library.  There is a
timer callback, which occurs at regular intervals.  There is also the mouse
motion callback, which occurs whenever the mouse cursor is moving.  If two 
consecutive timer callbacks occur, without and intervening mouse motion callback,
then the cursor is assumed to be stationary. 

If a stationary mouse is detected, the program goes on to determine which block
in the cube (if any) the cursor is located upon.
*/

#include "select.h"
#include <float.h>
#include <stdio.h>
#include "gnubik.h"
#include "ui.h"

#include <GL/glu.h>
#include <GL/GLwDrawA.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>




static int idle_threshold;
static GLboolean motion= GL_FALSE;
static GLboolean Stop_Detected = GL_FALSE;

extern int mouse_x, mouse_y;
extern GLdouble granularity;
extern void (*action)(void);

void UnsetMotion(XtPointer xtp, XtIntervalId *id) ;


void detect_motion (Widget w,XEvent *event,String *params,Cardinal *num_params);


static void enableDisableSelection(Widget w, XtPointer clientData, XEvent * event, 
			    Boolean * cont) ;

extern Widget glxarea;
extern XtAppContext app;
extern Widget main_application_window;

/* the id of the timer */
static XtIntervalId timer ;
static Boolean timerActive;




/* Initialise the selection mechanism.  Holdoff is the time for which
the mouse must stay still, for anything to happen. Precision is the
minimum distance it must have moved. Do_this is a pointer to a function
to be called when a new block is selected. */
void
initSelection(int holdoff, GLdouble precision, void (*do_this)(void) )
{


  idle_threshold = holdoff;
  granularity = precision ;


  action = do_this;
  timer = XtAppAddTimeOut(app, idle_threshold, UnsetMotion, NULL);
  timerActive=TRUE;

  /* Add a handler to for all those occasions when we don't need the
     selection mechanism going */

  XtAddEventHandler(main_application_window, 
		    EnterWindowMask | LeaveWindowMask |
		    VisibilityChangeMask | StructureNotifyMask , False, 
		    enableDisableSelection, NULL);


}

void 
disableSelection(void)
{

  XtRemoveTimeOut(timer);
  
  timerActive=FALSE;
}

void 
enableSelection(void)
{
    if ( !timerActive) { 
      timer = XtAppAddTimeOut(app, idle_threshold, UnsetMotion, NULL);
    }
    timerActive=TRUE;
}




/* When the window is not mapped, kill the selection mechanism.  It wastes
processor time */
void
enableDisableSelection(Widget w, XtPointer clientData, XEvent * event, Boolean * cont)
{
  /* This is a kludge to work around a rather horrible bug;  for some
     reason, some  platforms emit a EnterNotify and LeaveNotify (in
     that order) when animations occur.  This workaround makes sure
     that the window is not `entered twice' */
  static int entered =0;

  switch (event->type) {
  case EnterNotify:
    entered++;
  case MapNotify:
    enableSelection();
    break;
  case LeaveNotify:
    updateSelection();
    entered--;
    if ( entered > 0 ) break ;
    /* else fallthrough */
  case UnmapNotify:
    disableSelection();
    break;
  case VisibilityNotify:
    if ( ((XVisibilityEvent *) event)->state == VisibilityFullyObscured ) {
      disableSelection();
    }
    else {
      enableSelection();
    }
    break;
  default:
    break;
  }
}


/* This callback occurs whenever the mouse is moving */
void 
detect_motion (Widget w,XEvent *event,String *params,Cardinal *num_params)
{
  XMotionEvent *xme = (XMotionEvent *) event;

  if ( event->type != MotionNotify )
    return;

  mouse_x = xme->x;
  mouse_y = xme->y;

  motion = GL_TRUE;
  Stop_Detected = GL_FALSE;
}





/* This callback occurs at regular intervals.   The period is determined by 
idle_threshold.  It checks to see if the mouse has moved, since the last
call of this function.  
Post-condition:  motion is FALSE. 
*/
void
UnsetMotion(XtPointer xtp, XtIntervalId *id)
{


  timerActive = TRUE; 

  if ( motion == GL_FALSE ) { /* if not moved since last time */

    if ( ! Stop_Detected ) {
      /* in here, things happen upon the mouse stopping */
      Stop_Detected = GL_TRUE;
      updateSelection();
    }
  }

  motion = GL_FALSE ;
	
  /* re issue the callback */
  XtAppAddTimeOut(app, idle_threshold, UnsetMotion, NULL);
  timerActive=TRUE;

}  /* end UnsetMotion */



int
get_widget_height(Widget w)
{
  Dimension height;
  Arg arg[1] ;

  /* Find out the height of this widget */
  XtSetArg(arg[0], XtNheight, &height);
  XtGetValues(w,arg,1);

  return height;
}



