/**
 * DFArc main frame

 * Copyright (C) 2005, 2006  Dan Walma
 * Copyright (C) 2008  Sylvain Beucler

 * This file is part of GNU FreeDink

 * GNU FreeDink is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.

 * GNU FreeDink is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DFArcFrame.hpp"

#include <stdio.h>
#include <fstream>

#include <wx/file.h>
#include <wx/filename.h>
#include <wx/image.h>

#include <wx/msgdlg.h>
#include <wx/filedlg.h>

#include <wx/listbox.h>
#include <wx/clntdata.h>

// wxLaunchDefaultBrowser backport
#if !wxCHECK_VERSION(2, 6, 4) && __UNIX__
#include <wx/mimetype.h>
#endif

#include "IOUtils.hpp"
#include "InstallVerifyFrame.hpp"
#include "Options.hpp"

#include "Config.hpp"
#include "RecursiveDelete.hpp"
#include "Package.hpp"
#include "DMod.hpp"
#include "icon_xpm.hpp"

#define LOGO_WIDTH 160
#define LOGO_HEIGHT 120

// FRAME EVENT TABLE
BEGIN_EVENT_TABLE(DFArcFrame, wxFrame)
EVT_SHOW(DFArcFrame::onShow)

EVT_MENU(ID_FileInstall, DFArcFrame::Install)
EVT_MENU(ID_Download, DFArcFrame::onDownload)
EVT_MENU(wxID_EXIT, DFArcFrame::OnQuit)

EVT_MENU(ID_Refresh, DFArcFrame::onRefresh)
EVT_MENU(ID_Browse, DFArcFrame::onBrowse)
EVT_MENU(ID_Uninstall, DFArcFrame::uninstall)
EVT_MENU(ID_Options, DFArcFrame::showOptions)

EVT_MENU(ID_IntroductionText, DFArcFrame::showIntroductionText)
EVT_MENU(ID_Walkthroughs, DFArcFrame::onWalkthroughs)
EVT_MENU(ID_Forums, DFArcFrame::onForums)
EVT_MENU(wxID_ABOUT, DFArcFrame::OnAbout)

EVT_BUTTON(ID_Play, DFArcFrame::OnPlay)
EVT_BUTTON(ID_Edit, DFArcFrame::onEdit)
EVT_BUTTON(ID_Package, DFArcFrame::onPackage)

EVT_LISTBOX(ID_DmodTitleList, DFArcFrame::OnEvtListBox)

EVT_CHECKBOX(ID_Truecolor, DFArcFrame::OnSetPlayOption)
EVT_CHECKBOX(ID_Windowed, DFArcFrame::OnSetPlayOption)
EVT_CHECKBOX(ID_Sound, DFArcFrame::OnSetPlayOption)
EVT_CHECKBOX(ID_Joystick, DFArcFrame::OnSetPlayOption)
EVT_CHECKBOX(ID_Debug, DFArcFrame::OnSetPlayOption)
EVT_CHECKBOX(ID_V107, DFArcFrame::OnSetPlayOption)
END_EVENT_TABLE()

// Integer wrapper class to store the DMod index associated with a
// listbox entry from mDModListBox. Previously I was abusing 'void*'
// as 'int' but g++ produces an error under amd64.  If we were using
// glib, we'd use GINT_TO_POINTER().  Sadly, we're not using glib :/
// Using wxClientData as a base class to make the wxListBox
// automatically free client data when needed.
class Integer : public wxClientData
{
 public:
  int i;
  Integer(int i) : i(i) {}
};

DFArcFrame::DFArcFrame() :
DFArcFrame_Base(NULL, wxID_ANY, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE),
  mNoDmods(false),
  mSelectedDModIndex(wxNOT_FOUND)
{
  mConfig = Config::GetConfig();

  PrepareGUI();
  refreshDmodList();
  mConfig->Update();
  return;
}

void DFArcFrame::OnQuit(wxCommandEvent& aCommandEvent)
{
    this->Close();
}

void DFArcFrame::OnAbout(wxCommandEvent& aCommandEvent)
{
  // Use a wxString for concatenation with the date.
  wxString version = wxT(PACKAGE_VERSION);
  wxString wxs = wxString::Format(_(
    "DFArc version %s\n"
    "Copyright (C) 2004  Andrew Reading (merlin)\n"
    "Copyright (C) 2005, 2006  Dan Walma (redink1)\n"
    "Copyright (C) 2008  Sylvain Beucler (Beuc)\n"
    "Build Date: %s\n"
    "Powered by bzip2 (http://www.bzip.org) and wxWidgets (http://www.wxwidgets.org)"),
				  version.c_str(), __TDATE__);
  wxMessageBox(wxs, _("About DFArc v3"), wxOK | wxICON_INFORMATION, this);
}

// Display or hide "Edit" and "Package" buttons
void DFArcFrame::showDeveloperButtons(bool visible)
{
  wxSizer *mButtonSizer = mEditButton->GetContainingSizer();
  wxSizer *mImageSizer = mLogoButton->GetContainingSizer();
  mImageSizer->Show(mButtonSizer, visible);
  // Re-layout, starting with expanding mDModListBox
  mDModListBox->GetContainingSizer()->Layout();
}

static wxBitmap CreateTextLogo(wxString text)
{
  wxBitmap lLogo = wxBitmap(LOGO_WIDTH, LOGO_HEIGHT);
  wxMemoryDC dc;
  dc.SelectObject(lLogo);
  dc.SetBackground(*wxBLACK_BRUSH);
  dc.SetTextForeground(*wxWHITE);
  dc.Clear();
  // position '?'
  int w, h;
  dc.GetTextExtent(text, &w, &h);
  dc.DrawText(text,
	      (LOGO_WIDTH - w) / 2,
	      (LOGO_HEIGHT - h) / 2);
  dc.SelectObject(wxNullBitmap);
  return lLogo;
}

// GUI for main screen (with no args).
void DFArcFrame::PrepareGUI()
{   
  // Features not supported by wxGlade:
  SetIcon(wxIcon(dink_xpm));
  mSplitter->SetSashGravity(0.5); // proportional resize
  // Avoid 'unsplit' (where one of the pane is hidden)
  mSplitter->SetMinimumPaneSize(20);
  // Manual contraints
  mDModListBox->SetMinSize(wxSize(150, 165));
  mDmodDescription->SetMinSize(wxSize(-1, 220));
  this->SetMinSize(this->GetBestSize());
  this->SetSize(this->GetMinSize());
  // user can override size restrictions:
  mDmodDescription->SetMinSize(wxSize(-1, -1));

  // Optionally show developper buttons
  showDeveloperButtons(mConfig->mShowDeveloperButtons);
  // Update the checkboxes
  mTrueColor->SetValue(mConfig->mTrueColorValue);
  mWindowed->SetValue(mConfig->mWindowedValue);
  mSound->SetValue(mConfig->mSoundValue);
  mJoystick->SetValue(mConfig->mJoystickValue);
  mDebug->SetValue(mConfig->mDebugValue);
  mV107->SetValue(mConfig->mV107Value);

  // Default logo (currently all black with a question mark)
  /* TRANSLATORS: please make this SHORT, possibly rephrasing as "<
     Choose!". This is included in the 160x120px logo box in the main
     window and it doesn't word-wrap. */
  mDefaultLogoBitmap = CreateTextLogo(_("< Pick a D-Mod"));
  mLogoButton->SetBitmap(mDefaultLogoBitmap);
}

void DFArcFrame::updateDescription()
{
  std::fstream lFileStream;
  wxString lDescription;
  wxBitmap lLogoBitmap;

  // Set the current D-Mod directory to the currently selected item's client string
  DMod cur_dmod = mAvailableDModsList.at(mSelectedDModIndex);
  mConfig->mSelectedDmod = cur_dmod.GetFullPath();
  mConfig->Update();
  mDmodDescription->SetValue(cur_dmod.GetDescription());
  mStatusBar->SetStatusText(mConfig->mSelectedDmod);
  
  wxString lDmodPreview(mConfig->mSelectedDmod + _T("/preview.bmp"));
  IOUtils::ciconvert(lDmodPreview);
  wxString lDmodTitle(mConfig->mSelectedDmod + _T("/graphics/title-01.bmp"));
  IOUtils::ciconvert(lDmodTitle);
  if (::wxFileExists(lDmodPreview) == true)
    {
      lLogoBitmap = wxBitmap(lDmodPreview, wxBITMAP_TYPE_BMP);
    }
  else if (::wxFileExists(lDmodTitle) == true)
    {
      wxImage lImage(lDmodTitle);
      lImage.Rescale(LOGO_WIDTH, LOGO_HEIGHT);
      lLogoBitmap = wxBitmap(lImage);
    }
  else
    {
      lLogoBitmap = CreateTextLogo(cur_dmod.GetBaseName());
    }
  lLogoBitmap.SetWidth(LOGO_WIDTH);
  lLogoBitmap.SetHeight(LOGO_HEIGHT);
  mLogoButton->SetBitmap(lLogoBitmap);
  return;
}

void DFArcFrame::SelectDModFromListBox()
{
  int lb_index = mDModListBox->GetSelection();
  if (lb_index != wxNOT_FOUND)
    {
      Integer *I = (Integer*)mDModListBox->GetClientObject(lb_index);
      mSelectedDModIndex = I->i;
      mConfig->mSelectedDmod = mAvailableDModsList.at(mSelectedDModIndex).GetFullPath();
      mConfig->Update();
      updateDescription();
      mPlayButton->Enable();
      mEditButton->Enable();
      mPackageButton->Enable();
    }
}

void DFArcFrame::OnEvtListBox(wxCommandEvent &Event)
{
  SelectDModFromListBox();
  return;
}

// Function for Install menu entry
void DFArcFrame::Install(wxCommandEvent& aCommandEvent)
{
  wxString description = _("D-Mod files (*.dmod)");
  wxFileDialog FileDlg(0, _("Select a .dmod file"), _T(""), _T(""), description + _T("|*.dmod"),
        wxOPEN | wxFILE_MUST_EXIST);
     
  if (FileDlg.ShowModal() == wxID_OK)
    {
      InstallVerifyFrame *lTemp = new InstallVerifyFrame(FileDlg.GetPath());
      if (lTemp->ShowModal() == wxID_OK)
	{
	  mConfig->mSelectedDmod = lTemp->getSelectedDmod();
	  
	  // Update the D-Mod list
	  refreshDmodList();
	}
      lTemp->Destroy();
    }
}

void DFArcFrame::onPackage(wxCommandEvent &Event)
{
  DMod cur_dmod = mAvailableDModsList.at(mSelectedDModIndex);
  Package* lPackage = new Package(cur_dmod);
  lPackage->ShowModal();
  lPackage->Destroy();
}

void DFArcFrame::OnSetPlayOption(wxCommandEvent &Event)
{
  // Set variables
  mConfig->mTrueColorValue = mTrueColor->IsChecked();
  mConfig->mWindowedValue = mWindowed->IsChecked();
  mConfig->mSoundValue = mSound->IsChecked();
  mConfig->mJoystickValue = mJoystick->IsChecked();
  mConfig->mDebugValue = mDebug->IsChecked();
  mConfig->mV107Value = mV107->IsChecked();
  mConfig->Update();
  return;
}

wxString DFArcFrame::BuildCommand(enum program progname)
{
  wxString lCommand;
  wxString executable;
  if (progname == GAME)
    executable = mConfig->mDinkExe;
  else
    executable = mConfig->mEditorExe;

  // Attempt to use the binary in the Dink directory (important for
  // disambiguation under woe, where the PATH includes the current
  // DFArc _binary_ directory (not cwd()) by default).
  wxString test_dinkref = mConfig->GetDinkrefDir() + wxFileName::GetPathSeparator() + executable;
  if (::wxFileExists(test_dinkref))
    lCommand = test_dinkref;
  else
    lCommand = executable;

  if (mConfig->mDebugValue == true)
    lCommand += _T(" -debug ");
  if (mConfig->mSoundValue == false)
    lCommand += _T(" -nosound ");
  if (mConfig->mTrueColorValue == true)
    lCommand += _T(" -truecolor ");
  if (mConfig->mWindowedValue == true)
    lCommand += _T(" -window ");
  if (mConfig->mWriteIniValue == false)
    lCommand += _T(" -noini ");
  if (mConfig->mJoystickValue == false)
    lCommand += _T(" -nojoy ");
  if (mConfig->mV107Value == true)
    lCommand += _T(" --v1.07 ");
  
  /* Specify the directory, as short as possible*/
  wxString dinkref = mConfig->GetDinkrefDir();
  wxFileName dmod_dir(mConfig->mSelectedDmod);
  dmod_dir.MakeRelativeTo(dinkref);
  if (dmod_dir.GetFullPath().StartsWith(_T("..")))
    dmod_dir.MakeAbsolute(dinkref);
  if (dmod_dir.GetFullPath().IsSameAs(_T("dink")))
    dmod_dir = wxEmptyString;
  if (!dmod_dir.GetFullName().IsEmpty())
    {
      wxString fullpath = dmod_dir.GetFullPath();
      if (fullpath.Find(wxT(' ')) != wxNOT_FOUND)
	// Only do that if necessary, quoting doesn't work with the
	// original engine for some reason
	fullpath = _T("\"") + dmod_dir.GetFullPath() + _T("\"");
      lCommand += _T(" -game ") + fullpath;
    }

  /* Always specify the dinkref dir, because we don't know what
     FreeDink's default refdir is. (/usr/local?  /usr?...) */
  /* Note: this is ignored by Seth and Dan's versions */
  lCommand += _T(" --refdir \"") + mConfig->GetDinkrefDir() + _T("\"");

  /* If at a point we need better escaping, it is suggested to replace
     spaces with '\ ' under GNU/Linux and '" "' under woe -
     http://trac.wxwidgets.org/ticket/4115#comment:22 . I wish the
     wxWidgets dev properly fixed it with a wxExecute(wxArrayString)
     version... */

  return lCommand;
}

void DFArcFrame::OnPlay(wxCommandEvent &Event)
{
  wxString lCommand = BuildCommand(GAME);
  
  // Start the child process.
  long code = 0;
  bool found = true;
  if (mConfig->mCloseDfarcOnPlay)
    {
      // We want to run in asynchronous mode, so we can execute just
      // after starting Dink. Unfortunately wxWidgets sucks big time
      // when it comes to wxExecute's return values, apparently
      // because it successfully starts a shell before actually
      // starting the processus. This requires a work-around to detect
      // Dink's presence:
      if ((unsigned char)(code = ::wxExecute(mConfig->mDinkExe + _T(" --version"), wxEXEC_SYNC))
	  == (unsigned char)-1) // 255 if not casted under GNU/Linux, always -1 under woe...
	found = false;

      if (found)
	{
	  if ((code = ::wxExecute(lCommand, wxEXEC_ASYNC)) < 0)
	    ::wxMessageBox(wxString::Format(_("Dink Smallwood failed! Error code %d."),
					    code),
			   _("Error"));
	  else
	    this->Close(true);
	}
    }
  else
    {
      // If not closing DFArc when playing just run Dink in
      // synchronous mode, with actual^Wbetter error reporting
      if ((unsigned char)(code = ::wxExecute(lCommand, wxEXEC_SYNC)) == (unsigned char)-1)
	{
	  found = false;
	}
      else if (code != 0)
	{
	  ::wxMessageBox(wxString::Format(_("Dink Smallwood failed! Error code %d."),
					  code),
			 _("Error"));
	}
      else
	{
	  // Dink ran fine.
	}
    }

  if (!found)
    ::wxMessageBox(wxString::Format(_("Dink Smallwood ('%s') was not found on your computer."
				      " Please configure the Dink program name in the Options menu."),
				    mConfig->mDinkExe.c_str()),
		   _("Error"));
}

void DFArcFrame::onEdit( wxCommandEvent& aEvent )
{
  if (mConfig->mWarnOnEdit == true)
    {
      if (::wxMessageBox(_("Dinkedit saves all changes automatically."
			   " Altering maps can ruin the game."
			   " Are you sure you want to continue?"),
			 _("Warning"),
			 wxOK | wxCANCEL | wxICON_INFORMATION, this ) == wxCANCEL)
        {
	  return;
        }
      else
        {
	  // Don't display the warning again
	  mConfig->mWarnOnEdit = false;
	  mConfig->Update();
        }
    }


  wxString lCommand = BuildCommand(EDITOR);
  
  // Start the child process.
  long code = 0;

  if ((unsigned char)(code = ::wxExecute(lCommand, wxEXEC_SYNC)) == (unsigned char)-1)
    ::wxMessageBox(wxString::Format(_("The editor ('%s') was not found on your computer."
				      " Please configure the editor program name in the Options menu."),
				    mConfig->mEditorExe.c_str()),
		   _("Error"));
  else if (code != 0)
    ::wxMessageBox(_("Error while running the editor"), _("Error"));
}

void DFArcFrame::onRefresh( wxCommandEvent& aEvent )
{
  refreshDmodList();
  return;
}

void DFArcFrame::onDownload( wxCommandEvent& aEvent )
{
  launchURL(_T("http://www.dinknetwork.com/"));
}

void DFArcFrame::onWalkthroughs( wxCommandEvent& aEvent )
{
  launchURL(_T("http://solutions.dinknetwork.com/"));
}

void DFArcFrame::onForums( wxCommandEvent& aEvent )
{
  launchURL(_T("http://www.dinknetwork.com/forum.cgi"));
}

void DFArcFrame::onBrowse( wxCommandEvent& aEvent )
{
  wxString cur_dmod_dir = mConfig->mSelectedDmod;
  if (mConfig->mPreferredFileBrowserExe.IsEmpty())
    {
      // Try default browser(s)
#if defined _WIN32 || defined __WIN32__ || defined __CYGWIN__
      // Use '\' to force opening the directory, not dink.exe...
      ::ShellExecute(0, _T("open"), (cur_dmod_dir + wxT("\\")).c_str(), NULL, NULL, SW_SHOW);
#else
      /* Crudely escape spaces, since wxWidgets 2.8 doesn't have a
	 proper way to separate command line arguments */
      cur_dmod_dir.Replace(wxT(" "), wxT("\\ "), true);
      if (::wxExecute(_T("xdg-open ") + cur_dmod_dir) < 0) // FreeDesktop
	if (::wxExecute(_T("nautilus ") + cur_dmod_dir) < 0) // Gnome
	  if (::wxExecute(_T("konqueror ") + cur_dmod_dir) < 0) // KDE
	    if (::wxExecute(_T("thunar ") + cur_dmod_dir) < 0) // Xfce
	      ::wxMessageBox(_("Could not find a file manager"
			       " (tried 'xdg-open', 'nautilus', 'konqueror' and 'thunar')"),
			     _("Error"), wxICON_ERROR);
#endif
    }
  else
    {
      if (::wxExecute(mConfig->mPreferredFileBrowserExe + _T(" ") + cur_dmod_dir) < 0)
	::wxMessageBox(wxString::Format(_("Cannot start '%s', please check your"
					  " configuration in the Options window."),
					mConfig->mPreferredFileBrowserExe.c_str()),
		       _("Error"), wxICON_ERROR);
    }
  return;
}

void DFArcFrame::showIntroductionText(wxCommandEvent& aEvent)
{
  ::wxMessageBox(_("Welcome to DFArc, the Dink Smallwood front end!\n"
"\n"
"You can choose to play the original game, Dink Smallwood, or"
" Dink-Modules (D-Mods) which contain new adventures.  After completing"
" the main game, give the included D-Mod (Mystery Island) a try.\n"
"\n"
"There are hundreds of free D-Mods available, just click File-Download"
" D-Mods."),
		 _("Introduction"),
		 wxOK | wxICON_INFORMATION, this);
  return;
}

void DFArcFrame::onShow(wxShowEvent& aEvent)
{
  if (mConfig->mShowIntroductionText == true)
    {
      wxCommandEvent se;
      showIntroductionText(se);
      mConfig->mShowIntroductionText = false;
      mConfig->Update();
    }
  return;
}

void DFArcFrame::refreshDmodList()
{
  mDModListBox->Clear();
  populateAvailableDModsList();
  
  if (mAvailableDModsList.empty() == false)
    {
      mNoDmods = false;
      for (unsigned int i = 0; i < mAvailableDModsList.size(); i++)
	{
	  // Using 'wxClientData*' as int for simplicity.
	  Integer *I = new Integer(i);
	  mDModListBox->Append(mAvailableDModsList.at(i).GetTitle(), I);
	}
    }
  else
    {
      mNoDmods = true;
    }
  RestoreListBoxFromConfig();
}

/**
 * Select DMod from configuration
 */
void DFArcFrame::RestoreListBoxFromConfig()
{
  int lb_index_to_select = wxNOT_FOUND;
  // Update the selection
  if (!mConfig->mSelectedDmod.IsEmpty())
    {
      for (unsigned int i = 0; i < mDModListBox->GetCount(); i++)
	{
	  Integer *I = (Integer*)mDModListBox->GetClientObject(i);
	  int cur_dmod_index = I->i;
	  DMod cur_dmod = mAvailableDModsList.at(cur_dmod_index);

	  if (cur_dmod.GetFullPath().IsSameAs(mConfig->mSelectedDmod))
	    {
	      lb_index_to_select = i;
	      break;
	    }
	}
      mDModListBox->SetSelection(lb_index_to_select);
      SelectDModFromListBox();
    }
  else
    {
      mPlayButton->Disable();
      mEditButton->Disable();
      mPackageButton->Disable();
      mLogoButton->SetBitmap(mDefaultLogoBitmap);
    }
}

void DFArcFrame::showOptions(wxCommandEvent& aEvent)
{
  Options* lOptions = new Options(mConfig);
  if (lOptions->ShowModal() == 1)
    {
      // Apply configuration changes
      showDeveloperButtons(mConfig->mShowDeveloperButtons);
      if (mConfig->mOverrideDinkrefDir)
	{
	  if (::wxDirExists(mConfig->mSpecifiedDinkrefDir))
	    {
	      // Get the full path to the directory, in case users type in a non-full path.
	      if (chdir(mConfig->mSpecifiedDinkrefDir.fn_str()) == 0)
		{
		  mConfig->mSpecifiedDinkrefDir = ::wxGetCwd();
		}
	      else
		{
		  // If there's no directory, let's not override
		  ::wxMessageBox(_("Cannot use the overriden Dink Smallwood directory"
				   " - ignoring it. (permission problem?)"),
				 _("Configuration error"));
		  mConfig->mOverrideDinkrefDir = false;
		}
	    }
	  else
	    {
	      // If there's no directory, let's not override
	      ::wxMessageBox(_("The Dink Smallwood directory you entered does not exist - ignoring it."),
			     _("Configuration error"));
	      mConfig->mOverrideDinkrefDir = false;
	    }
	}
      refreshDmodList();
    }
  lOptions->Destroy();
  return;
}

void DFArcFrame::uninstall( wxCommandEvent& aEvent )
{
  if (mConfig->mSelectedDmod == _T("dink"))
    {
      wxMessageBox(_("You must select the uninstall option from the start menu to uninstall the main game."),
		   _("Uninstall - Error"),
		   wxOK | wxICON_INFORMATION, this);
    }
  else
    {
      int lResult = wxMessageBox(_("Do you want to remove all save game files?"),
				 _("Uninstall - Save Game Files"),
				 wxYES_NO | wxCANCEL | wxICON_QUESTION, this);
      if (lResult != wxCANCEL)
        {
	  bool lRemoveSaveGames(false);
	  if (lResult == wxYES)
            {
	      lRemoveSaveGames = true;
            }
	  
	  RecursiveDelete lRecursiveDelete(lRemoveSaveGames);
	  wxDir* lDir = new wxDir(mConfig->mSelectedDmod);
	  lDir->Traverse(lRecursiveDelete);
	  delete lDir;
	  
	  if (lRecursiveDelete.getError() == false)
            {
	      bool lSuccess(true);
	      if (lRemoveSaveGames == true)
                {
		  if (::wxRmdir(mConfig->mSelectedDmod) == false)
                    {
		      ::wxLogError(_("Unable to remove D-Mod directory. All other files were removed."));
		      lSuccess = false;
                    }
                }
	      if (lSuccess == true)
                {
		  wxMessageBox(_("D-Mod successfully uninstalled"),
			       _("Uninstall - Success"),
			       wxOK | wxICON_INFORMATION, this);
                }
            }
	  mConfig->mSelectedDmod = wxEmptyString;
	  refreshDmodList();
        }
    }
  return;
}





/**
 * This finds all of the DMods.
 */
void DFArcFrame::populateAvailableDModsList()
{
  mAvailableDModsList.clear();

  wxString* folders = new wxString[2];
  folders[0] = mConfig->GetDinkrefDir();
  folders[1] = mConfig->mDModDir;

  for (int i = 0; i < 2; i++)
    {
      if (folders[i].IsEmpty() || !::wxDirExists(folders[i]))
	continue;
      wxDir lDirectory(folders[i]);
      if (lDirectory.IsOpened() == false)
	continue;

      // Grab dir names.
      wxString lCurrentFolder;
      bool cont = lDirectory.GetFirst(&lCurrentFolder, wxEmptyString, wxDIR_DIRS);
      while (cont)
	{
	  wxString full_path = folders[i] + wxFileName::GetPathSeparator() + lCurrentFolder;
	  if (DMod::IsADModDir(full_path))
	      mAvailableDModsList.push_back(DMod(full_path));
	  cont = lDirectory.GetNext(&lCurrentFolder);
	}
    }

  delete[] folders;
  return;
}



/**
 * Some packaged version of wxWidgets 2.6, such as 2.6.3.2 in Debian
 * Etch and Lenny, lack good enough ::wxLaunchDefaultBrowser
 * implementation, introduced in 2.6.4 (argh). In practice, it will
 * say it succeeded in launching the URL in the default browser, but
 * will probably just launch 'epiphany' which may not be installed. It
 * will still return true/success, denying simple work-arounds. wx2.8
 * is good. I eventually used conditional compilation to include
 * updated Unix code from wxWidgets 2.6.4.
 */
bool DFArcFrame::launchURL(wxString url)
{
#if wxCHECK_VERSION(2, 6, 4) || !__UNIX__
  return ::wxLaunchDefaultBrowser(url);
#else
/////////////////////////////////////////////////////////////////////////////
// Name:        utilscmn.cpp
// Purpose:     Miscellaneous utility functions and classes
// Author:      Julian Smart
// Modified by:
// Created:     29/01/98
// RCS-ID:      $Id: utilscmn.cpp,v 1.145.2.5 2007/03/20 12:39:53 VZ Exp $
// Copyright:   (c) 1998 Julian Smart
// Licence:     wxWindows licence
/////////////////////////////////////////////////////////////////////////////

    wxArrayString errors;
    wxArrayString output;

    // gconf will tell us the application to use as browser under GNOME
    long res = wxExecute
               (
                wxT("gconftool-2 --get /desktop/gnome/applications/browser/exec"),
                output,
                errors,
                wxEXEC_NODISABLE
               );

    if ( res != -1 && errors.empty() && !output.empty() )
    {
        wxString cmd = output[0];
        cmd << _T(' ') << url;
        if ( wxExecute(cmd) )
            return true;
    }
    else // try the KDE way if gconf didn't work
    {
        // kfmclient directly opens the given URL
        if ( wxExecute(wxT("kfmclient openURL ") + url) )
            return true;
    }

    // if we got here, the platform-specific way of opening URLs failed, try
    // the generic one
#if wxUSE_MIMETYPE
    // Non-windows way
    bool ok = false;
    wxString cmd;

    wxFileType *ft = wxTheMimeTypesManager->GetFileTypeFromExtension(_T("html"));
    if ( ft )
    {
        wxString mt;
        ft->GetMimeType(&mt);

        ok = ft->GetOpenCommand(&cmd, wxFileType::MessageParameters(url));
        delete ft;
    }

    if ( !ok || cmd.empty() )
    {
        // fallback to checking for the BROWSER environment variable
        cmd = wxGetenv(wxT("BROWSER"));
        if ( !cmd.empty() )
            cmd << _T(' ') << url;
    }

    ok = ( !cmd.empty() && wxExecute(cmd) );
    if (ok)
        return ok;

    // no file type for HTML extension
    wxLogError(_T("No default application configured for HTML files."));
#endif // !wxUSE_MIMETYPE && !__WXMSW__

    wxLogSysError(_T("Failed to open URL \"%s\" in default browser."),
                  url.c_str());

    return false;

  //::wxMessageBox(wxString::Format(_("Visit %s !"), url.c_str()),
  //		 _("Website (could not locate default browser)"));
#endif
}
