/*
 * Electric(tm) VLSI Design System
 *
 * File: simirsim.c
 * Simulation tool: IRSIM deck generator
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2001 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "config.h"
#if SIMTOOL

#include "global.h"
#include "sim.h"
#include "simirsim.h"
#include "efunction.h"
#include "egraphics.h"
#include "network.h"
#include "tecschem.h"
#include <math.h>

       IRSIMTRANSISTOR *sim_firstirsimtransistor = NOIRSIMTRANSISTOR;
static IRSIMTRANSISTOR *sim_irsimtransistorfree = NOIRSIMTRANSISTOR;

       INTBIG   sim_irsimnetnumber = 0;
       char   **sim_irsimnetnames;
       INTBIG  *sim_irsimnetsignals;
static INTBIG   sim_irsimnettotal = 0;
static INTBIG   sim_irsimlambda;

/* prototypes for local routines */
static IRSIMTRANSISTOR *sim_allocirsimtransistor(void);
static void             sim_freeirsimtransistor(IRSIMTRANSISTOR *it);
static INTBIG           sim_irsimrecurse(NODEPROTO *facet, XARRAY trans);
static void             sim_irsimgetallgeometry(INTSML layer, TECHNOLOGY *tech, INTBIG *x, INTBIG *y,
							INTSML count);
static void             sim_irsimgetareaperimeter(INTBIG *x, INTBIG *y, INTSML count,
							float *a, INTBIG *p);
static INTBIG           sim_irsimnewnetnumber(char *name);
static char            *sim_irsimbuildnetname(char *name);
static INTSML           sim_cropbox(INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy, INTBIG bx,
							INTBIG ux, INTBIG by, INTBIG uy);
static INTSML           sim_clipagainstgates(POLYGON *poly, ARCINST *ai, XARRAY trans);
static void             sim_irsimfreenetworkinfo(void);

/*
 * Routine to free all memory associated with this module.
 */
void sim_freeirsimmemory(void)
{
	REGISTER IRSIMTRANSISTOR *it;

	sim_irsimfreenetworkinfo();
	while (sim_irsimtransistorfree != NOIRSIMTRANSISTOR)
	{
		it = sim_irsimtransistorfree;
		sim_irsimtransistorfree = it->nextirsimtransistor;
		efree((char *)it);
	}
	if (sim_irsimnettotal > 0)
	{
		efree((char *)sim_irsimnetnames);
		efree((char *)sim_irsimnetsignals);
	}
}

/*
 * Routine to write a simulation deck for facet "facet".  This is called when the user
 * just requests a deck, so the file is not temporary.
 */
void sim_writeirsim(NODEPROTO *facet)
{
	REGISTER FILE *f;
	char *truename;
	REGISTER char *netfile;

	/* make sure network tool is on */
	if ((net_tool->toolstate&TOOLON) == 0)
	{
		ttyputerr(_("Network tool must be running...turning it on"));
		toolturnon(net_tool, 0);
		ttyputerr(_("...now reissue the simulation command"));
		return;
	}

	/* construct the simulation description file name from the facet */
	(void)initinfstr();
	(void)addstringtoinfstr(facet->cell->cellname);
	(void)addstringtoinfstr(".sim");
	netfile = returninfstr();

	/* create the simulation file */
	f = xcreate(netfile, sim_filetypeirsim, _("Netlist File"), &truename);
	if (f == NULL)
	{
		if (truename != 0) ttyputerr(_("Cannot write %s"), truename);
		return;
	}

	sim_irsimgeneratedeck(facet, f);

	xclose(f);
	ttyputmsg(_("%s written"), truename);
}

/*
 * Routine to clear out the IRSIM network in memory (from the last facet that was
 * simulated).
 */
void sim_irsimfreenetworkinfo(void)
{
	REGISTER IRSIMTRANSISTOR *it, *nextit;
	REGISTER INTBIG i;

	for(it = sim_firstirsimtransistor; it != NOIRSIMTRANSISTOR; it = nextit)
	{
		nextit = it->nextirsimtransistor;
		sim_freeirsimtransistor(it);
	}
	for(i=0; i<sim_irsimnetnumber; i++) efree(sim_irsimnetnames[i]);
}

/*
 * Routine to generate an IRSIM deck for facet "facet" and write it to stream "f".
 */
void sim_irsimgeneratedeck(NODEPROTO *facet, FILE *f)
{
	REGISTER INTBIG backannotate;
	REGISTER TECHNOLOGY *tech;
	char creationdate[80], revisiondate[80];
	REGISTER IRSIMTRANSISTOR *it;
	REGISTER PORTPROTO *pp;
	REGISTER NETWORK *net;
	REGISTER char *name;
	REGISTER INTBIG sa, sp, da, dp, i;

	/* get value of lambda */
	tech = facet->tech;
	sim_irsimlambda = facet->cell->lib->lambda[tech->techindex];

	/*
	 * extract the IRSIM information from facet "facet".  Creates a linked
	 * list of transistors in "sim_firstirsimtransistor" and creates
	 * "sim_irsimnetnumber" networks in "sim_irsimnetnames".
	 */
	sim_irsimfreenetworkinfo();
	sim_irsimnetnumber = 0;
	begintraversehierarchy();
	sim_firstirsimtransistor = NOIRSIMTRANSISTOR;
	backannotate = 0;
	if (asktool(net_tool, "name-nets", (INTBIG)facet) != 0) backannotate++;
	for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
		net->temp1 = -1;
	for(pp = facet->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp->network->temp1 != -1) continue;
		if (pp->network->signals > 1)
		{
			for(i=0; i<pp->network->signals; i++)
			{
				net = pp->network->networklist[i];
				if (net->namecount <= 0) continue;
				net->temp1 = sim_irsimnewnetnumber(net->netname);
			}
		} else
		{
			name = pp->protoname;
			if (portispower(pp) != 0) name = "vdd"; else
				if (portisground(pp) != 0) name = "gnd";
			pp->network->temp1 = sim_irsimnewnetnumber(name);
		}
	}
	mergeinit();
	if (sim_irsimrecurse(facet, el_matid) != 0) backannotate++;
	mergedone(sim_irsimgetallgeometry);

	/* write deck to "f" */	
	xprintf(f,"| units: %g tech: %s format: SU\n",
		scaletodispunit(sim_irsimlambda, DISPUNITCMIC), tech->techname);
	if (facet->creationdate == 0) creationdate[0] = 0; else
		sprintf(creationdate, _(", created %s"), timetostring(facet->creationdate));
	if (facet->revisiondate == 0) revisiondate[0] = 0; else
		sprintf(revisiondate, _(", modified %s"), timetostring(facet->revisiondate));
	xprintf(f, "| Facet %s%s%s\n", describenodeproto(facet), creationdate, revisiondate);

	for(it = sim_firstirsimtransistor; it != NOIRSIMTRANSISTOR; it = it->nextirsimtransistor)
	{
		if (it->transistortype == NPTRANMOS) xprintf(f, "n"); else
			xprintf(f, "p");
		xprintf(f, " %s", sim_irsimnetnames[it->gate]);
		xprintf(f, " %s", sim_irsimnetnames[it->source]);
		xprintf(f, " %s", sim_irsimnetnames[it->drain]);
		xprintf(f, " %s %s", latoa(it->length), latoa(it->width));
		xprintf(f, " %s %s", latoa(it->xpos), latoa(it->ypos));
		if (it->transistortype == NPTRANMOS) xprintf(f, " g=S_gnd"); else
			xprintf(f, " g=S_vdd");
		sa = (INTBIG)(it->sarea / sim_irsimlambda);
		if (sa == 0.0) sa = it->width * 6;
		sa = (sa + sim_irsimlambda/2) / sim_irsimlambda * sim_irsimlambda;
		sp = (it->sperimeter + sim_irsimlambda/2) / sim_irsimlambda * sim_irsimlambda;
		if (sp == 0) sp = it->width + 12*sim_irsimlambda;
		da = (INTBIG)(it->darea / sim_irsimlambda);
		if (da == 0.0) da = it->width * 6;
		da = (da + sim_irsimlambda/2) / sim_irsimlambda * sim_irsimlambda;
		dp = (it->dperimeter + sim_irsimlambda/2) / sim_irsimlambda * sim_irsimlambda;
		if (dp == 0) dp = it->width + 12*sim_irsimlambda;
		xprintf(f, " s=A_%s,P_%s", latoa(sa), latoa(sp));
		xprintf(f, " d=A_%s,P_%s\n", latoa(da), latoa(dp));
	}

	if (backannotate != 0)
		ttyputmsg(_("Back-annotation information has been added (library must be saved)"));
}

INTBIG sim_irsimrecurse(NODEPROTO *facet, XARRAY trans)
{
	REGISTER NETWORK *net, *gate, *source, *drain;
	REGISTER PORTPROTO *pp, *s, *g1, *g2, *d;
	REGISTER NODEPROTO *np, *cnp;
	REGISTER NODEINST *ni;
	REGISTER PORTARCINST *pi;
	REGISTER PORTEXPINST *pe;
	REGISTER ARCINST *ai;
	REGISTER VARIABLE *var;
	REGISTER INTBIG netnumber, i, tot, fun, backannotate, type, realtype;
	INTBIG x, y;
	REGISTER IRSIMTRANSISTOR *it;
	XARRAY singletrans, localtrans, subrot;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, sim_tool->cluster);

	backannotate = 0;
	if (asktool(net_tool, "name-nodes", (INTBIG)facet) != 0) backannotate++;

	/* create unique numbers for internal nets */
	for(net = facet->firstnetwork; net != NONETWORK; net = net->nextnetwork)
	{
		if (net->temp1 != -1) continue;
		if (net->signals > 1) continue;
		if (net->namecount > 0)
		{
			net->temp1 = sim_irsimnewnetnumber(sim_irsimbuildnetname(net->netname));
		} else
		{
			net->temp1 = sim_irsimnewnetnumber("INTERNAL");
		}
	}

	/* now examine all geometry in the facet */
	for(ni = facet->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		np = ni->proto;
		if (np == sch_pwrprim || np == sch_gndprim)
		{
			if (ni->firstportarcinst != NOPORTARCINST)
			{
				i = ni->firstportarcinst->conarcinst->network->temp1;
				if (i != -1)
				{
					if (np == sch_pwrprim)
					{
						if (namesame(sim_irsimnetnames[i], "vdd") != 0)
							(void)reallocstring(&sim_irsimnetnames[i], "vdd", sim_tool->cluster);
					} else
					{
						if (namesame(sim_irsimnetnames[i], "gnd") != 0)
							(void)reallocstring(&sim_irsimnetnames[i], "gnd", sim_tool->cluster);
					}
				}
			}
		}
		makerot(ni, singletrans);
		transmult(singletrans, trans, localtrans);
		if (np->primindex == 0)
		{
			/* ignore recursive references (showing icon in contents) */
			if (np->cell == facet->cell) continue;

			/* facet instance: recurse */
			cnp = contentsview(np);
			if (cnp == NONODEPROTO) cnp = np;

			if (asktool(net_tool, "name-nets", (INTBIG)cnp) != 0) backannotate++;
			for(net = cnp->firstnetwork; net != NONETWORK; net = net->nextnetwork)
				net->temp1 = -1;
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				pp = equivalentport(np, pi->proto, cnp);
				if (pp == NOPORTPROTO)
				{
					ttyputmsg("Cannot find icon export '%s' in facet %s",
						pi->proto->protoname, describenodeproto(cnp));
				} else
				{
					pp->network->temp1 = pi->conarcinst->network->temp1;
					if (pp->network->signals > 1 &&
						pp->network->signals == pi->conarcinst->network->signals)
					{
						for(i=0; i<pp->network->signals; i++)
						{
							net = pp->network->networklist[i];
							net->temp1 = pi->conarcinst->network->networklist[i]->temp1;
						}
					}
				}
			}

			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
			{
				pp = equivalentport(np, pe->proto, cnp);
				if (pp == NOPORTPROTO)
				{
					ttyputmsg("Cannot find icon export '%s' in facet %s",
						pe->proto->protoname, describenodeproto(cnp));
				} else
				{
					pp->network->temp1 = pe->exportproto->network->temp1;
					if (pp->network->signals > 1 &&
						pp->network->signals == pe->exportproto->network->signals)
					{
						for(i=0; i<pp->network->signals; i++)
						{
							net = pp->network->networklist[i];
							net->temp1 = pe->exportproto->network->networklist[i]->temp1;
						}
					}
				}
			}
			for(pp = cnp->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
			{
				if (pp->network->temp1 != -1) continue;
				var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
				if (pp->network->signals > 1)
				{
					for(i=0; i<pp->network->signals; i++)
					{
						net = pp->network->networklist[i];
						if (net->temp1 != -1) continue;
						if (net->namecount <= 0) continue;
						(void)initinfstr();
						if (var != NOVARIABLE)
							(void)formatinfstr("%s/", (char *)var->addr);
						(void)addstringtoinfstr(net->netname);
						net->temp1 = sim_irsimnewnetnumber(sim_irsimbuildnetname(returninfstr()));
					}
				} else
				{
					(void)initinfstr();
					if (var != NOVARIABLE)
						(void)formatinfstr("%s/", (char *)var->addr);
					(void)addstringtoinfstr(pp->protoname);
					pp->network->temp1 = sim_irsimnewnetnumber(sim_irsimbuildnetname(returninfstr()));
				}
			}
			maketrans(ni, singletrans);
			transmult(singletrans, localtrans, subrot);
			downhierarchy(ni);
			if (sim_irsimrecurse(cnp, subrot) != 0) backannotate = 1;
			uphierarchy();
		} else
		{
			/* primitive: see if it is a transistor */
			it = NOIRSIMTRANSISTOR;
			type = (np->userbits&NFUNCTION) >> NFUNCTIONSH;
			realtype = nodefunction(ni);
			if (realtype == NPTRANMOS || realtype == NPTRA4NMOS ||
				realtype == NPTRAPMOS || realtype == NPTRA4PMOS)
			{
				if (type == NPTRANS || type == NPTRANS4)
				{
					/* gate is port 0, source is port 1, drain is port 2 */
					g1 = g2 = np->firstportproto;
					s = g1->nextportproto;
					d = s->nextportproto;
				} else
				{
					/* gate is port 0 or 2, source is port 1, drain is port 3 */
					g1 = np->firstportproto;
					s = g1->nextportproto;
					g2 = s->nextportproto;
					d = g2->nextportproto;
					if (realtype == NPTRA4NMOS || realtype == NPTRA4PMOS) g2 = g1;
				}
				gate = source = drain = NONETWORK;
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					if (pi->proto == g1 || pi->proto == g2) gate = pi->conarcinst->network;
					if (pi->proto == s) source = pi->conarcinst->network;
					if (pi->proto == d) drain = pi->conarcinst->network;
				}
				for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
				{
					if (pe->proto == g1 || pe->proto == g2) gate = pe->exportproto->network;
					if (pe->proto == s) source = pe->exportproto->network;
					if (pe->proto == d) drain = pe->exportproto->network;
				}
				if (gate == NONETWORK || source == NONETWORK || drain == NONETWORK)
				{
					ttyputerr(_("Warning: ignoring transistor %s in facet %s (not fully connected)"),
						describenodeinst(ni), describenodeproto(facet));
				} else
				{
					it = sim_allocirsimtransistor();
					if (it == NOIRSIMTRANSISTOR) continue;
					it->nextirsimtransistor = sim_firstirsimtransistor;
					sim_firstirsimtransistor = it;
					it->transistortype = realtype;
					it->source = source->temp1;
					it->drain = drain->temp1;
					it->gate = gate->temp1;
					xform((ni->lowx+ni->highx)/2, (ni->lowy+ni->highy)/2,
						&it->xpos, &it->ypos, localtrans);
					portposition(ni, s, &x, &y);
					xform(x, y, &it->sourcex, &it->sourcey, trans);
					portposition(ni, d, &x, &y);
					xform(x, y, &it->drainx, &it->drainy, trans);
					transistorsize(ni, &it->length, &it->width);
					if (it->length < 0) it->length = 0;
					if (it->width < 0) it->width = 0;
					it->sarea = it->darea = (float)it->width * 6.0f;
					it->sperimeter = it->dperimeter = it->width + 12*sim_irsimlambda;
					it->sarea = it->darea = 0.0;
					it->sperimeter = it->dperimeter = 0;
				}
			}
			
			tot = nodeEpolys(ni, 0, NOWINDOWPART);
			for(i=0; i<tot; i++)
			{
				shapeEnodepoly(ni, i, poly);
				if (poly->portproto == NOPORTPROTO) continue;
				fun = layerfunction(poly->tech, poly->layer);
				if ((fun&LFTYPE) != LFDIFF) continue;
				netnumber = -1;
				for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
				{
					if (pi->proto != poly->portproto) continue;
					netnumber = pi->conarcinst->network->temp1;
					break;
				}
				if (netnumber < 0)
				{
					for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = pe->nextportexpinst)
					{
						if (pe->proto != poly->portproto) continue;
						netnumber = pe->exportproto->network->temp1;
						break;
					}
				}
				if (netnumber < 0)
					netnumber = sim_irsimnewnetnumber(sim_irsimbuildnetname("unattached"));
				xformpoly(poly, localtrans);
				mergestorepolygon(netnumber, poly->tech, poly);
			}
		}
	}
	for(ai = facet->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		tot = arcpolys(ai, 0);
		for(i=0; i<tot; i++)
		{
			shapearcpoly(ai, i, poly);
			fun = layerfunction(poly->tech, poly->layer);
			if ((fun&LFTYPE) != LFDIFF) continue;
			xformpoly(poly, trans);
			if (sim_clipagainstgates(poly, ai, trans) != 0) continue;
			mergestorepolygon(ai->network->temp1, poly->tech, poly);
		}
	}
	return(backannotate);
}

/*
 * Routine to clip polygon "poly" on arc "ai" given a transformation for this facet
 * in "trans".
 * Returns nonzero if the polygon is clipped to oblivion.
 */
INTSML sim_clipagainstgates(POLYGON *poly, ARCINST *ai, XARRAY trans)
{
	REGISTER INTBIG e, i, tot, lfun, ret;
	INTBIG plx, phx, ply, phy, lx, hx, ly, hy;
	XARRAY singletrans, localtrans;
	REGISTER NODEINST *ni;
	static POLYGON *gatepoly = NOPOLYGON;

	/* get polygon */
	if (gatepoly == NOPOLYGON) gatepoly = allocstaticpolygon(4, sim_tool->cluster);

	if (isbox(poly, &plx, &phx, &ply, &phy) == 0) return(0);

	/* clip this active against gate of connecting transistors */
	for(e=0; e<2; e++)
	{
		ni = ai->end[e].nodeinst;
		if (isfet(ni->geom) == 0) continue;
		makerot(ni, singletrans);
		transmult(singletrans, trans, localtrans);
		tot = nodepolys(ni, 0, 0);
		for(i=0; i<tot; i++)
		{
			shapenodepoly(ni, i, gatepoly);
			lfun = layerfunction(gatepoly->tech, gatepoly->layer);
			if (layerispoly(lfun) == 0) continue;

			/* limitation: connecting transistors must be manhattan */
			xformpoly(gatepoly, localtrans);
			if (isbox(gatepoly, &lx, &hx, &ly, &hy) != 0)
			{
				ret = sim_cropbox(&plx, &phx, &ply, &phy,
					lx, hx, ly, hy);
				if (ret > 0) return(1);
			}
		}
	}
	makerectpoly(plx, phx, ply, phy, poly);
	return(0);
}

/*
 * routine to crop the box in the reference parameters (lx-hx, ly-hy)
 * against the box in (bx-ux, by-uy).  If the box is cropped into oblivion,
 * the routine returns 1.  Otherwise the box is cropped and zero is returned
 */
INTSML sim_cropbox(INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy, INTBIG bx, INTBIG ux, INTBIG by,
	INTBIG uy)
{
	/* if the two boxes don't touch, just return */
	if (bx >= *hx || by >= *hy || ux <= *lx || uy <= *ly) return(0);

	/* if the box to be cropped is within the other, mark it all covered */
	if (bx <= *lx && ux >= *hx && by <= *ly && uy >= *hy) return(1);

	/* reduce (lx-hx,ly-hy) by (bx-ux,by-uy) */
	if (bx <= *lx && ux >= *hx)
	{
		/* it covers in X...crop in Y */
		if (*hy < uy && *hy > by) *hy = by;
		if (*ly < uy && *ly > by) *ly = uy;
	}
	if (by <= *ly && uy >= *hy)
	{
		/* it covers in Y...crop in X */
		if (*hx < ux && *hx > bx) *hx = bx;
		if (*lx < ux && *lx > bx) *lx = ux;
	}
	return(0);
}

void sim_irsimgetallgeometry(INTSML layer, TECHNOLOGY *tech, INTBIG *x, INTBIG *y, INTSML count)
{
	REGISTER IRSIMTRANSISTOR *it;
	POLYGON fakepoly;
	REGISTER INTBIG found, lx, hx, ly, hy, i;
	INTBIG perimeter, a, p;
	REGISTER char *name;
	float area;

	/* get bounds of this geometry */
	lx = hx = x[0];
	ly = hy = y[0];
	for(i=1; i<count; i++)
	{
		if (x[i] < lx) lx = x[i];
		if (x[i] > hx) hx = x[i];
		if (y[i] < ly) ly = y[i];
		if (y[i] > hy) hy = y[i];
	}
	found = 0;
	for(it = sim_firstirsimtransistor; it != NOIRSIMTRANSISTOR; it = it->nextirsimtransistor)
		it->sourcefound = it->drainfound = 0;
	for(it = sim_firstirsimtransistor; it != NOIRSIMTRANSISTOR; it = it->nextirsimtransistor)
	{
		/* see if the source or drain of this transistor is in the polygon */
		if (it->sarea == 0.0)
		{
			if (it->sourcex >= lx && it->sourcex <= hx &&
				it->sourcey >= ly && it->sourcey <= hy)
			{
				fakepoly.count = count;
				fakepoly.xv = x;
				fakepoly.yv = y;
				fakepoly.style = FILLED;
				if (isinside(it->sourcex, it->sourcey, &fakepoly) != 0)
				{
					it->sourcefound = 1;
					found++;
				}
			}
		}
		if (it->darea == 0.0)
		{
			if (it->drainx >= lx && it->drainx <= hx &&
				it->drainy >= ly && it->drainy <= hy)
			{
				fakepoly.count = count;
				fakepoly.xv = x;
				fakepoly.yv = y;
				fakepoly.style = FILLED;
				if (isinside(it->drainx, it->drainy, &fakepoly) != 0)
				{
					it->drainfound = 1;
					found++;
				}
			}
		}
	}
	sim_irsimgetareaperimeter(x, y, count, &area, &perimeter);
	if (found != 0)
	{
		area /= found;   perimeter /= found;
		for(it = sim_firstirsimtransistor; it != NOIRSIMTRANSISTOR; it = it->nextirsimtransistor)
		{
			if (it->sourcefound != 0)
			{
				it->sarea = area;
				it->sperimeter = perimeter;
			}
			if (it->drainfound != 0)
			{
				it->darea = area;
				it->dperimeter = perimeter;
			}
		}
	} else
	{
		name = sim_irsimnetnames[layer];
		if (namesame(name, "vdd") != 0 && namesame(name, "gnd") != 0)
		{
			a = (INTBIG)(area / sim_irsimlambda);
			a = (a + sim_irsimlambda/2) / sim_irsimlambda * sim_irsimlambda;
			p = (perimeter + sim_irsimlambda/2) / sim_irsimlambda * sim_irsimlambda;
			ttyputerr("Warning: unattached diffusion on node %s has Area %s, Perimeter %s",
				name, latoa(a), latoa(p));
		}
	}
}

void sim_irsimgetareaperimeter(INTBIG *x, INTBIG *y, INTSML count, float *a, INTBIG *p)
{
	REGISTER INTBIG per, seglen, i,  lastx, lasty, thisx, thisy;
	float area, side1, side2;

	/* compute the perimeter */
	per = 0;
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			lastx = x[count-1];   lasty = y[count-1];
		} else
		{
			lastx = x[i-1];   lasty = y[i-1];
		}
		seglen = computedistance(lastx, lasty, x[i], y[i]);
		per += seglen;
	}
	*p = per;

	/* compute the area */
	area = 0.0;
	lastx = x[0];
	lasty = y[0];
	for (i=1; i<count; i++)
	{
		thisx = x[i];
		thisy = y[i];

		/* triangulate around the polygon */
		side1 = (float)(thisx - lastx);
		side2 = (float)(lasty + thisy);
		area += (side1 * side2) / 2.0f;
		lastx = thisx;
		lasty = thisy;
	}
	side1 = (float)(x[0] - lastx);
	side2 = (float)(y[0] + lasty);
	area += (side1 * side2) / 2.0f;
	area = (float)fabs(area);
	*a = area;
}

char *sim_irsimbuildnetname(char *name)
{
	REGISTER INTBIG i;
	NODEINST **nilist;
	INTSML depth;
	REGISTER VARIABLE *var;

	gettraversalpath(&nilist, &depth);
	(void)initinfstr();
	for(i=0; i<depth; i++)
	{
		var = getvalkey((INTBIG)nilist[i], VNODEINST, VSTRING, el_node_name);
		if (var != NOVARIABLE)
		{
			(void)addstringtoinfstr((char *)var->addr);
		} else
		{
			(void)addstringtoinfstr(nilist[i]->proto->cell->cellname);
		}
		(void)addtoinfstr('/');
	}
	(void)addstringtoinfstr(name);
	return(returninfstr());
}

IRSIMTRANSISTOR *sim_allocirsimtransistor(void)
{
	REGISTER IRSIMTRANSISTOR *it;

	if (sim_irsimtransistorfree != NOIRSIMTRANSISTOR)
	{
		it = sim_irsimtransistorfree;
		sim_irsimtransistorfree = it->nextirsimtransistor;
	} else
	{
		it = (IRSIMTRANSISTOR *)emalloc(sizeof (IRSIMTRANSISTOR), sim_tool->cluster);
		if (it == 0) return(NOIRSIMTRANSISTOR);
	}
	return(it);
}

void sim_freeirsimtransistor(IRSIMTRANSISTOR *it)
{
	it->nextirsimtransistor = sim_irsimtransistorfree;
	sim_irsimtransistorfree = it;
}

INTBIG sim_irsimnewnetnumber(char *name)
{
	REGISTER INTBIG newtotal, i, *newsignals;
	REGISTER char **newnames;

	if (sim_irsimnetnumber >= sim_irsimnettotal)
	{
		newtotal = sim_irsimnetnumber * 2;
		if (newtotal <= 0) newtotal = 10;
		newnames = (char **)emalloc(newtotal * (sizeof (char *)), sim_tool->cluster);
		if (newnames == 0) return(-1);
		newsignals = (INTBIG *)emalloc(newtotal * SIZEOFINTBIG, sim_tool->cluster);
		if (newsignals == 0) return(-1);
		for(i=0; i<sim_irsimnetnumber; i++)
		{
			newnames[i] = sim_irsimnetnames[i];
			newsignals[i] = sim_irsimnetsignals[i];
		}
		if (sim_irsimnettotal > 0)
		{
			efree((char *)sim_irsimnetnames);
			efree((char *)sim_irsimnetsignals);
		}
		sim_irsimnetnames = newnames;
		sim_irsimnetsignals = newsignals;
		sim_irsimnettotal = newtotal;
	}
	i = sim_irsimnetnumber++;
	(void)allocstring(&sim_irsimnetnames[i], name, sim_tool->cluster);
	return(i);
}

#endif  /* SIMTOOL - at top */
