/*
 * Electric(tm) VLSI Design System
 *
 * File: graphpccode.cpp
 * Interface for Win32 (Win9x/NT) computers
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "graphpcstdafx.h"
#include "graphpc.h"
#include "graphpcdoc.h"
#include "graphpcview.h"
#include "graphpcmainframe.h"
#include "graphpcchildframe.h"
#include "graphpcmsgview.h"
#include "graphpcdialog.h"

#include "global.h"
#include "database.h"
#include "egraphics.h"
#include "usr.h"
#include "eio.h"
#include "edialogs.h"
#if LANGTCL
#  include "dblang.h"
#endif

#include <io.h>
#include <signal.h>
#include <sys/stat.h>
#include <direct.h>
#include <sys/timeb.h>

/****** windows and control ******/
extern CElectricApp theApp;
static WINDOWFRAME *gra_cureditwindowframe;
static WINDOWFRAME *gra_palettewindowframe = NOWINDOWFRAME;
static INTSML       gra_windowframeindex = 0;
static INTSML       gra_newwindowoffset = 0;
static char        *gra_initialdirectory;
static INTSML       gra_creatingwindow = 0;

void   gra_activateframe(CChildFrame *frame, BOOL bActivate);
INTSML gra_buildoffscreenbuffer(WINDOWFRAME *wf, INTSML wid, INTSML hei, CDC **hdc,
		HBITMAP *bitmap, BITMAPINFO **bminfo, char **databuffer, char ***rowstart);
INTSML gra_buildwindow(WINDOWFRAME*, INTSML, RECTAREA*);
void   gra_floatpalette(void);
int    gra_closeframe(CChildFrame *frame);
int    gra_closeworld(void);
void   gra_timerticked(void);
void   gra_freewindowframe(WINDOWFRAME *wf);
RECT  *gra_geteditorwindowlocation(void);
void   gra_removewindowextent(RECT *r, CWnd *wnd);
void   gra_getdevices(void);
HWND   gra_getwindow(WINDOWPART *win);
#if LANGTCL
INTBIG gra_initializetcl(void);
#endif
INTSML gra_makeeditwindow(WINDOWFRAME*);
void   gra_movedwindow(CChildFrame *frame, int x, int y);
void   gra_redrawdisplay(WINDOWFRAME*);
void   gra_reloadmap(void);
INTSML gra_remakeeditwindow(WINDOWFRAME*);
void   gra_repaint(CChildFrame *frame, CPaintDC *dc);
void   gra_resize(CChildFrame *frame, int cx, int cy);
void   gra_resizemain(int cx, int cy);
void   gra_setrect(WINDOWFRAME*, INTSML, INTSML, INTSML, INTSML);
void   gra_copyhighlightedtoclipboard(void);

/****** the messages window ******/
#define MAXTYPEDLINE        256				/* max chars on input line */

	   CChildFrame *gra_messageswindow;		/* the messages window */
static CEdit       *gra_editCtrl;
static INTSML       gra_messagescurrent;	/* nonzero if messages window is current */
static INTSML       gra_messagesleft;		/* left bound of messages window screen */
static INTSML       gra_messagesright;		/* right bound of messages window screen */
static INTSML       gra_messagestop;		/* top bound of messages window screen */
static INTSML       gra_messagesbottom;		/* bottom bound of messages window screen */
static CDC         *gra_texthdc;			/* device context for text buffer */
static HBITMAP      gra_textbitmap;			/* bitmap for text buffer */
static BITMAPINFO  *gra_textbitmapinfo;		/* bitmap information structure */
static char        *gra_textdatabuffer;		/* data in text buffer */
static char       **gra_textrowstart;		/* row starts for text buffer */
static INTSML       gra_textbufwid, gra_textbufhei;	/* size of text buffer */
static INTSML       gra_textbufinited = 0;	/* nonzero if text buffer is initialized */

INTSML gra_makemessageswindow(void);
INTSML gra_messagesnotvisible(void);
void   gra_tomessagesbottom(void);

/****** the status bar ******/
#define MAXSTATUSLINES        1

STATUSFIELD *gra_statusfields[100];
char *gra_statusfieldtext[100];
INTSML gra_indicatorcount = 0;
static UINT gra_indicators[] = {ID_INDICATOR_ELECTRIC01, ID_INDICATOR_ELECTRIC02,
	ID_INDICATOR_ELECTRIC03, ID_INDICATOR_ELECTRIC04, ID_INDICATOR_ELECTRIC05,
	ID_INDICATOR_ELECTRIC06, ID_INDICATOR_ELECTRIC07, ID_INDICATOR_ELECTRIC08,
	ID_INDICATOR_ELECTRIC09, ID_INDICATOR_ELECTRIC10, ID_INDICATOR_ELECTRIC11,
	ID_INDICATOR_ELECTRIC12, ID_INDICATOR_ELECTRIC13, ID_INDICATOR_ELECTRIC14,
	ID_INDICATOR_ELECTRIC15, ID_INDICATOR_ELECTRIC16, ID_INDICATOR_ELECTRIC17};

void   gra_redrawstatusindicators(void);

/****** the dialogs ******/
#define PROGRESSOFFSET         100
#define MAXSCROLLMULTISELECT  1000
#define MAXDIALOGS               3   /* maximum nested dialogs */

typedef struct
{
	CElectricDialog *window;
	DIALOG          *itemdesc;
	int              defaultbutton;
	POINT            firstpoint;
	INTBIG           redrawitem;
	void           (*redrawroutine)(RECTAREA*);
} TDIALOG;

TDIALOG          gra_dialogs[MAXDIALOGS];
TDIALOG         *gra_curdialog;
INTBIG           gra_curdialogindex = -1;
int              gra_dialogdbnx, gra_dialogdbny;
int              gra_dialogdbux, gra_dialogdbuy;
CBrush          *gra_dialogoffbrush = 0;

int              gra_dialoghit;
int              gra_dialoghitchar;
int              gra_dialogeditline;

void   gra_diaredrawitem(void);
INTSML gra_dodialogisinsideuserdrawn(int x, int y);
int    gra_dodialoglistkey(UINT nKey, CListBox* pListBox, UINT nIndex);
void   gra_dodialogtextchange(int nID);
HICON  gra_makeicon(char *data);

/****** events ******/
#define CHARREAD           0177				/* character that was read */
#define ISBUTTON           0200				/* set if button pushed (or released) */
#define BUTTONUP           0400				/* set if button was released */
#define SHIFTDOWN         01000				/* set if shift key was held down */
#define COMMANDDOWN       02000				/* set if command key was held down */
#define ALTDOWN           04000				/* set if alt key was held down */
#define CONTROLDOWN      010000				/* set if control key was held down */
#define DOUBLECLICK      020000				/* set if this is second click */
#define MOTION           040000				/* set if mouse motion detected */
#define WINDOWSIZE      0100000				/* set if window grown */
#define WINDOWMOVE      0200000				/* set if window moved */
#define MENUEVENT       0400000				/* set if menu entry selected (values in cursor) */
#define FILEREPLY      01000000				/* set if file selected in standard file dialog */
#define DIAITEMCLICK   02000000				/* set if item clicked in dialog */
#define DIASCROLLSEL   04000000				/* set if scroll item selected in dialog */
#define DIAEDITTEXT   010000000				/* set if edit text changed in dialog */
#define DIAPOPUPSEL   020000000				/* set if popup item selected in dialog */
#define DIASETCONTROL 040000000				/* set if control changed in dialog */
#define DIAUSERMOUSE 0100000000				/* set if mouse moved in user area of dialog */
#define DIAENDDIALOG 0200000000				/* set when dialog terminates */
#define WHICHBUTTON 03400000000				/* which button was pushed */
#define ISLEFT                0				/*   the left button */
#define ISMIDDLE     0400000000				/*   the middle button */
#define ISRIGHT     01000000000				/*   the right button */
#define ISWHLFWD    01400000000				/*   forward on the mouse wheel */
#define ISWHLBKW    02000000000				/*   backward on the mouse wheel */
#define NOEVENT              -1				/* set if nothing happened */

struct
{
	INTBIG kind;
	INTSML x;
	INTSML y;
	INTSML frameindex;
} gra_action;

typedef struct
{
	INTSML  message;
	INTSML  what;
	UINTBIG when;
	INTSML  where;
	INTSML  modifiers;
} EventRecord;

static INTBIG       gra_inputstate;			/* current state of device input */
static INTBIG       gra_cursorx, gra_cursory;	/* current position of mouse */
static INTSML       gra_logrecordcount = 0;	/* count for session flushing */
static INTBIG       gra_playbackposition;	/* location in playback file */
static INTBIG       gra_playbacklength;		/* size in playback file */
static INTBIG       gra_lastloggedaction = NOEVENT;
static INTSML       gra_lastloggedx, gra_lastloggedy;
static INTSML       gra_lastloggedindex;
static char        *gra_logfile, *gra_logfilesave;

typedef struct
{
	INTBIG                cursorx, cursory;	/* current position of mouse */
	INTBIG                inputstate;		/* current state of device input */
} MYEVENTQUEUE;

#define EVENTQUEUESIZE	100

static MYEVENTQUEUE  gra_eventqueue[EVENTQUEUESIZE];
static MYEVENTQUEUE *gra_eventqueuehead;	/* points to next event in queue */
static MYEVENTQUEUE *gra_eventqueuetail;	/* points to first free event in queue */

void   gra_addeventtoqueue(INTBIG state, INTBIG x, INTBIG y);
void   gra_buttonaction(int state, UINT nFlags, CPoint point, CWnd *frm);
void   gra_itemclicked(int nID);
void   gra_itemdoubleclicked(int nID);
void   gra_keyaction(UINT nChar, UINT nFlags, UINT nRepCnt);
INTSML gra_loggetnextaction(char *message);
void   gra_logreadaction(void);
char   gra_logreadbyte(void);
INTBIG gra_logreadlong(void);
INTSML gra_logreadshort(void);
void   gra_logreadstring(char *msg);
void   gra_logwriteaction(INTBIG inputstate, INTSML cursorx, INTSML cursory, void *extradata);
void   gra_logwritebyte(char data);
void   gra_logwritelong(INTBIG data);
void   gra_logwriteshort(INTSML data);
void   gra_logwritestring(char *msg);
INTSML gra_makebutton(INTBIG);
void   gra_mouseaction(UINT nFlags, CPoint point, CWnd *frm);
int    gra_setpropercursor(CWnd *wnd, int x, int y);
void   gra_mousewheelaction(UINT nFlags, short zDelta, CPoint point, CWnd *frm);
void   gra_nextevent(void);
void   gra_onint(void);

/****** pulldown menus ******/
#define IDR_MENU           3000				/* base ID for menus */

static CMenu       *gra_hMenu;				/* System Menu */
	   CMenu      **gra_pulldownmenus;		/* list of Windows pulldown menus */
static POPUPMENU  **gra_pulldowns;			/* list of Electric pulldown menus */
	   INTSML       gra_pulldownmenucount;	/* number of pulldown menus */
static int          gra_menures[] = {ID_ELECTRIC_MENU01, ID_ELECTRIC_MENU02,
	ID_ELECTRIC_MENU03, ID_ELECTRIC_MENU04, ID_ELECTRIC_MENU05, ID_ELECTRIC_MENU06,
	ID_ELECTRIC_MENU07, ID_ELECTRIC_MENU08, ID_ELECTRIC_MENU09, ID_ELECTRIC_MENU10,
	ID_ELECTRIC_MENU11, ID_ELECTRIC_MENU12, ID_ELECTRIC_MENU13, ID_ELECTRIC_MENU14,
	ID_ELECTRIC_MENU15, ID_ELECTRIC_MENU16, ID_ELECTRIC_MENU17, ID_ELECTRIC_MENU18,
	ID_ELECTRIC_MENU19, ID_ELECTRIC_MENU20, ID_ELECTRIC_MENU21, ID_ELECTRIC_MENU22,
	ID_ELECTRIC_MENU23, ID_ELECTRIC_MENU24, ID_ELECTRIC_MENU25, ID_ELECTRIC_MENU26,
	ID_ELECTRIC_MENU27, ID_ELECTRIC_MENU28, ID_ELECTRIC_MENU29};

CMenu *gra_makepdmenu(POPUPMENU *);
void   gra_nativemenudoone(INTSML low, INTSML high);
INTSML gra_pulldownindex(POPUPMENU *);

/****** mouse buttons ******/
#define BUTTONS              45				/* cannot exceed NUMBUTS in "usr.h" */
#define REALBUTS              5				/* actual number of buttons */

struct
{
	char  *name;			/* button name */
	INTSML unique;			/* number of letters that make it unique */
} gra_buttonname[BUTTONS] =
{
	{"LEFT" ,1},  {"MIDDLE",  1}, {"RIGHT",  1}, {"FORWARD",   1},{"BACKWARD",   1},	/* 0: unshifted */
	{"SLEFT",2},  {"SMIDDLE", 2}, {"SRIGHT", 2}, {"SFORWARD",  2},{"SBACKWARD",  2},	/* 5: shift held down */
	{"CLEFT",2},  {"CMIDDLE", 2}, {"CRIGHT", 2}, {"CFORWARD",  2},{"CBACKWARD",  2},	/* 10: control held down */
	{"ALEFT",2},  {"AMIDDLE", 2}, {"ARIGHT", 2}, {"AFORWARD",  2},{"ABACKWARD",  2},	/* 15: alt held down */
	{"SCLEFT",3}, {"SCMIDDLE",3}, {"SCRIGHT",3}, {"SCFORWARD", 3},{"SCBACKWARD", 3},	/* 20: control/shift held down */
	{"SALEFT",3}, {"SAMIDDLE",3}, {"SARIGHT",3}, {"SAFORWARD", 3},{"SABACKWARD", 3},	/* 25: shift/alt held down */
	{"CALEFT",3}, {"CAMIDDLE",3}, {"CARIGHT",3}, {"CAFORWARD", 3},{"CABACKWARD", 3},	/* 30: control/alt held down */
	{"SCALEFT",4},{"SCAMIDDLE",4},{"SCARIGHT",4},{"SCAFORWARD",4},{"SCABACKWARD",4},	/* 35: shift/control/alt held down */
	{"DLEFT",2},  {"DMIDDLE", 2}, {"DRIGHT", 2}, {"DFORWARD",  2},{"DBACKWARD",  2}		/* 40: double-click */
};

/****** cursors ******/
static HCURSOR      gra_normalCurs;			/* the default cursor */
static HCURSOR      gra_wantttyCurs;		/* a "use the TTY" cursor */
static HCURSOR      gra_penCurs;			/* a "draw with pen" cursor */
static HCURSOR      gra_menuCurs;			/* a menu selection cursor */
static HCURSOR      gra_handCurs;			/* a hand dragging cursor */
static HCURSOR      gra_techCurs;			/* a technology edit cursor */
static HCURSOR      gra_ibeamCurs;			/* a text edit cursor */
static HCURSOR      gra_waitCurs;			/* an hourglass cursor */
static HCURSOR      gra_lrCurs;				/* a left/right pointing cursor */
static HCURSOR      gra_udCurs;				/* an up/down pointing cursor */

void   gra_setdefaultcursor(void);

/****** rectangle saving ******/
#define NOSAVEDBOX ((SAVEDBOX *)-1)
typedef struct Isavedbox
{
	HBITMAP           hBox;
	BITMAPINFO       *bminfo;
	CDC              *hMemDC;
	char             *data;
	char            **rowstart;
	WINDOWPART       *win;
	INTSML            lx, hx, ly, hy;
	struct Isavedbox *nextsavedbox;
} SAVEDBOX;

SAVEDBOX *gra_firstsavedbox = NOSAVEDBOX;

/****** polygon decomposition ******/
#define NOPOLYSEG ((POLYSEG *)-1)

typedef struct Isegment
{
	INTBIG fx,fy, tx,ty, direction, increment;
	struct Isegment *nextedge;
	struct Isegment *nextactive;
} POLYSEG;

static POLYSEG     *gra_polysegs;
static INTSML       gra_polysegcount = 0;

/****** curve drawing ******/
#define MODM(x) ((x<1) ? x+8 : x)
#define MODP(x) ((x>8) ? x-8 : x)
#define ODD(x)  (x&1)

static INTSML       gra_arcocttable[] = {0,0,0,0,0,0,0,0,0};
static INTBIG       gra_arccenterx, gra_arccentery, gra_arcradius, gra_curvemaxx, gra_curvemaxy;
static INTBIG       gra_arclx, gra_archx, gra_arcly, gra_archy;
static INTSML       gra_curvecol, gra_curvemask, gra_curvestyle, gra_arcfirst;
static WINDOWFRAME *gra_arcframe;			/* window frame for arc drawing */

void   gra_drawdiscrow(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc);

/****** fonts ******/
#define MAXCACHEDFONTS 200
#define FONTHASHSIZE   211		/* must be prime */

static CFont *gra_fontcache[MAXCACHEDFONTS];
INTBIG       gra_numfaces = 0;
char       **gra_facelist;

typedef struct
{
	CFont   *font;
	INTBIG   face;
	INTBIG   italic;
	INTBIG   bold;
	INTBIG   underline;
	INTBIG   size;
} FONTHASH;

FONTHASH gra_fonthash[FONTHASHSIZE];

CFont *gra_createtextfont(INTBIG, char*, INTBIG, INTBIG, INTBIG);
CFont *gra_gettextfont(WINDOWPART*, TECHNOLOGY*, UINTBIG*);
int APIENTRY MyEnumFaces(LPLOGFONT lpLogFont, LPTEXTMETRIC lpTEXTMETRICs, DWORD fFontType, LPVOID lpData);

/****** miscellaneous ******/
#define PALETTEWIDTH        136				/* width of palette */
#define MAXPATHLEN          256				/* max chars in file path */
#define SBARWIDTH            15				/* width of scroll bars */

static INTSML       gra_screenleft,			/* left bound of any editor window screen */
                    gra_screenright,		/* right bound of any editor window screen */
                    gra_screentop,			/* top bound of any editor window screen */
                    gra_screenbottom;		/* bottom bound of any editor window screen */
static char         gra_localstring[256];	/* local string */
static void        *gra_fileliststringarray = 0;
static time_t       gra_timebasesec;
static INTBIG       gra_timebasems;

INTSML gra_addfiletolist(char *file);
void   gra_drawline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask);
void   gra_drawpatline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col,
		INTSML mask, INTSML pattern);
void   gra_drawthickline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask);
void   gra_drawthickpoint(WINDOWFRAME *wf, INTSML x, INTSML y, INTSML mask, INTSML col);
void   gra_initfilelist(void);

/******************** INITIALIZATION ********************/

/*
 * routines to establish the default display
 */
void graphicsoptions(char *name, INTBIG *argc, char **argv)
{
	us_erasech = BACKSPACEKEY;
	us_killch = 025;
}

/*
 * routine to initialize the display device
 */
INTSML initgraphics(INTSML messages)
{
	char username[256];
	UINTBIG size;
	INTBIG i;

	gra_inputstate = NOEVENT;
	gra_pulldownmenucount = 0;	/* number of pulldown menus */
	gra_hMenu = 0;

	/* setup globals that describe location of windows */
	gra_getdevices();

	/* remember the initial directory and the log file locations */
	(void)allocstring(&gra_initialdirectory, currentdirectory(), db_cluster);
	size = 256;
	GetUserName(username, &size);
	(void)initinfstr();
	(void)addstringtoinfstr(gra_initialdirectory);
	(void)addstringtoinfstr("electric_");
	(void)addstringtoinfstr(username);
	(void)addstringtoinfstr(".log");
	(void)allocstring(&gra_logfile, returninfstr(), db_cluster);
	(void)initinfstr();
	(void)addstringtoinfstr(gra_initialdirectory);
	(void)addstringtoinfstr("electriclast_");
	(void)addstringtoinfstr(username);
	(void)addstringtoinfstr(".log");
	(void)allocstring(&gra_logfilesave, returninfstr(), db_cluster);

	/* create the scrolling messages window */
	if (messages == 0) gra_messageswindow = 0; else
	{
	   if (gra_makemessageswindow() != 0) error(_("Cannot create messages window"));
	}

	/* get cursors */
	gra_normalCurs = theApp.LoadStandardCursor(IDC_ARROW);
	gra_waitCurs = theApp.LoadStandardCursor(IDC_WAIT);
	gra_ibeamCurs = theApp.LoadStandardCursor(IDC_IBEAM);
	gra_wantttyCurs = theApp.LoadCursor(IDC_CURSORTTY);
	if (gra_wantttyCurs == 0) gra_wantttyCurs = gra_normalCurs;
	gra_penCurs = theApp.LoadCursor(IDC_CURSORPEN);
	if (gra_penCurs == 0) gra_penCurs = gra_normalCurs;
	gra_handCurs = theApp.LoadCursor(IDC_CURSORHAND);
	if (gra_handCurs == 0) gra_handCurs = gra_normalCurs;
	gra_menuCurs = theApp.LoadCursor(IDC_CURSORMENU);
	if (gra_menuCurs == 0) gra_menuCurs = gra_normalCurs;
	gra_techCurs = theApp.LoadCursor(IDC_CURSORTECH);
	if (gra_techCurs == 0) gra_techCurs = gra_normalCurs;
	gra_lrCurs = theApp.LoadStandardCursor(IDC_SIZEWE);
	if (gra_lrCurs == 0) gra_lrCurs = gra_normalCurs;
	gra_udCurs = theApp.LoadStandardCursor(IDC_SIZENS);
	if (gra_udCurs == 0) gra_udCurs = gra_normalCurs;

	/* initialize the cursor */
	SetCursor(0);
	us_normalcursor = NORMALCURSOR;
	us_cursorstate = us_normalcursor;

	/* initialize font cache */
	for(i=0; i<MAXCACHEDFONTS; i++) gra_fontcache[i] = 0;
	for(i=0; i<FONTHASHSIZE; i++) gra_fonthash[i].font = 0;

	gra_eventqueuehead = gra_eventqueuetail = gra_eventqueue;
	el_firstwindowframe = el_curwindowframe = NOWINDOWFRAME;
	gra_cureditwindowframe = NOWINDOWFRAME;

	return(0);
}

INTSML gra_makemessageswindow(void)
{
	RECT rmsg;
	CCreateContext context;

	/* create a context to tell this window to be an EditView */
	context.m_pNewViewClass = RUNTIME_CLASS(CElectricMsgView);
	context.m_pCurrentDoc = NULL;
	context.m_pNewDocTemplate = NULL;
	context.m_pLastView = NULL;
	context.m_pCurrentFrame = NULL;

	/* set messages window location */
	rmsg.left = gra_messagesleft;
	rmsg.right = gra_messagesright;
	rmsg.top = gra_messagestop;
	rmsg.bottom = gra_messagesbottom;

	/* create the window */
	gra_messageswindow = new CChildFrame();
	gra_messageswindow->Create(NULL, _("Electric Messages"),
		WS_CHILD|WS_OVERLAPPEDWINDOW|WS_VISIBLE, rmsg, NULL, &context);
	gra_messageswindow->InitialUpdateFrame(NULL, TRUE);
	CEditView *msgView = (CEditView *)gra_messageswindow->GetActiveView();
	gra_editCtrl = &msgView->GetEditCtrl();

	gra_messagescurrent = 0;
	return(0);
}

/*
 * Routine to examine the display devices available and to setup globals that describe
 * the editing windows and messages window extents.  On exit, the globals "gra_screenleft",
 * "gra_screenright", "gra_screentop", and "gra_screenbottom" will describe the area
 * for the editing windows and the variables "gra_messagesleft", "gra_messagesright",
 * "gra_messagestop", and "gra_messagesbottom" will describe the messages window.
 */
void gra_getdevices(void)
{
	RECT rDesktop;

	/* obtain the current device */
	theApp.m_pMainWnd->GetClientRect(&rDesktop);
	rDesktop.bottom -= 22;		/* status bar height? */

	gra_screenleft   = (INTSML)0;
	gra_screenright  = (INTSML)(rDesktop.right-rDesktop.left);
	gra_screentop    = (INTSML)0;
	gra_screenbottom = (INTSML)(rDesktop.bottom-rDesktop.top);

	gra_messagesleft   = gra_screenleft + PALETTEWIDTH;
	gra_messagesright  = gra_screenright - PALETTEWIDTH;
	gra_messagestop    = gra_screentop - 2 + (gra_screenbottom-gra_screentop) * 4 / 5;
	gra_messagesbottom = gra_screenbottom;
}

/******************** TERMINATION ********************/

void termgraphics(void)
{
	REGISTER WINDOWFRAME *wf;
	REGISTER INTBIG i;

	while (el_firstwindowframe != NOWINDOWFRAME)
	{
		wf = el_firstwindowframe;
		el_firstwindowframe = el_firstwindowframe->nextwindowframe;
		delete (CChildFrame *)wf->wndframe;
		gra_freewindowframe(wf);
	}
	if (gra_messageswindow != 0)
		delete gra_messageswindow;

	if (gra_textbufinited != 0)
	{
		efree((char *)gra_textrowstart);
		efree((char *)gra_textbitmapinfo);
		DeleteObject(gra_textbitmap);
		delete gra_texthdc;
	}

	if (gra_pulldownmenucount != 0)
	{
		efree((char *)gra_pulldownmenus);
		efree((char *)gra_pulldowns);
	}

	if (gra_polysegcount > 0) efree((char *)gra_polysegs);

	if (gra_fileliststringarray != 0)
		killstringarray(gra_fileliststringarray);
	efree((char *)gra_initialdirectory);
	efree((char *)gra_logfile);
	efree((char *)gra_logfilesave);

	/* free cached fonts */
	for(i=0; i<MAXCACHEDFONTS; i++)
	{
		if (gra_fontcache[i] != 0) delete gra_fontcache[i];
		gra_fontcache[i] = 0;
	}
	for(i=0; i<FONTHASHSIZE; i++)
	{
		if (gra_fonthash[i].font != 0) delete gra_fonthash[i].font;
		gra_fonthash[i].font = 0;
	}
	for(i=0; i<gra_numfaces; i++) efree((char *)gra_facelist[i]);
	if (gra_numfaces > 0) efree((char *)gra_facelist);
}

void exitprogram(void)
{
	exit(0);
}

/******************** WINDOW CONTROL ********************/

WINDOWFRAME *newwindowframe(INTSML floating, RECTAREA *r)
{
	WINDOWFRAME *wf, *oldlisthead;

	/* allocate one */
	wf = (WINDOWFRAME *)emalloc((sizeof (WINDOWFRAME)), us_tool->cluster);
	if (wf == 0) return(NOWINDOWFRAME);
	wf->numvar = 0;
	wf->rowstart = 0;
	wf->windindex = gra_windowframeindex++;

	/* insert window-frame in linked list */
	oldlisthead = el_firstwindowframe;
	wf->nextwindowframe = el_firstwindowframe;
	el_firstwindowframe = wf;
	el_curwindowframe = wf;
	if (floating != 0) gra_palettewindowframe = wf; else
		gra_cureditwindowframe = wf;

	/* load an editor window into this frame */
	wf->pColorPalette = 0;
	if (gra_buildwindow(wf, floating, r) != 0)
	{
		efree((char *)wf);
		el_firstwindowframe = oldlisthead;
		return(NOWINDOWFRAME);
	}
	return(wf);
}

void killwindowframe(WINDOWFRAME *owf)
{
	WINDOWFRAME *wf, *lastwf;

	/* kill the actual window */
	((CChildFrame *)owf->wndframe)->DestroyWindow();

	/* find this frame in the list */
	lastwf = NOWINDOWFRAME;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf == owf) break;
		lastwf = wf;
	}
	if (wf == NOWINDOWFRAME) return;
	if (lastwf == NOWINDOWFRAME) el_firstwindowframe = owf->nextwindowframe; else
		lastwf->nextwindowframe = owf->nextwindowframe;
	if (el_curwindowframe == owf) el_curwindowframe = NOWINDOWFRAME;
	if (gra_cureditwindowframe == owf) gra_cureditwindowframe = NOWINDOWFRAME;

	gra_freewindowframe(owf);
}

void gra_freewindowframe(WINDOWFRAME *wf)
{
	if (wf->rowstart != 0)
	{
		efree((char *)wf->rowstart);
		efree((char *)wf->bminfo);
		DeleteObject(wf->hBitmap);
		delete ((CDC *)wf->hDCOff);
		efree((char *)wf->pColorPalette);
	}
	efree((char *)wf);
}

WINDOWFRAME *getwindowframe(INTSML canfloat)
{
	if (el_curwindowframe != NOWINDOWFRAME)
	{
		if (canfloat != 0 || el_curwindowframe->floating == 0)
			return(el_curwindowframe);
	}
	if (gra_cureditwindowframe != NOWINDOWFRAME)
		return(gra_cureditwindowframe);
	return(NOWINDOWFRAME);
}

/*
 * routine to return size of window frame "wf" in "wid" and "hei"
 */
void getwindowframesize(WINDOWFRAME *wf, INTSML *wid, INTSML *hei)
{
	*wid = wf->swid;
	*hei = wf->shei;
}

/*
 * Routine to get the extent of the messages window.
 */
void getmessagesframeinfo(INTSML *top, INTSML *left, INTSML *bottom, INTSML *right)
{
	RECT fr;
	CWnd *parent;

	parent = gra_messageswindow->GetParent();
	gra_messageswindow->GetWindowRect(&fr);
	parent->ScreenToClient(&fr);
	*top = (INTSML)fr.top;
	*left = (INTSML)fr.left;
	*bottom = (INTSML)fr.bottom;
	*right = (INTSML)fr.right;
}

/*
 * Routine to set the size and position of the messages window.
 */
void setmessagesframeinfo(INTSML top, INTSML left, INTSML bottom, INTSML right)
{
	(void)gra_messageswindow->SetWindowPos(0, left, top, right-left, bottom-top,
		SWP_NOZORDER);
}

HWND gra_getwindow(WINDOWPART *win)
{
	return(win->frame->realwindow);
}

void sizewindowframe(WINDOWFRAME *wf, INTSML wid, INTSML hei)
{
	RECT fr, cr;
	int framewid, framehei;
	WINDOWPLACEMENT wp;

	/* if window frame already this size, stop now */
	if (wid == wf->swid && hei == wf->shei) return;

	/* resize window if needed */
	((CChildFrame *)wf->wndframe)->GetClientRect(&cr);
	if (wid != cr.right-cr.left || hei != cr.bottom-cr.top)
	{
		((CChildFrame *)wf->wndframe)->GetWindowPlacement(&wp);
		fr = wp.rcNormalPosition;

		framewid = (fr.right - fr.left) - (cr.right - cr.left);
		framehei = (fr.bottom - fr.top) - (cr.bottom - cr.top);

		/* determine new window size */
		wid += framewid;
		hei += framehei;

		/* resize the window */
		if (((CChildFrame *)wf->wndframe)->SetWindowPos(0, 0, 0, wid, hei, SWP_NOMOVE|SWP_NOZORDER) == 0)
		{
			/* failed to resize window */
			return;
		}
	}

	/* rebuild the offscreen windows */
	if (gra_remakeeditwindow(wf) != 0)
	{
//		fr.left += 1;   fr.right -= 2;   fr.bottom -= 2;
//		SetWindowPos(wf->realwindow, NULL, 0, 0, fr.right-fr.left, fr.bottom-fr.top, SWP_NOZORDER | SWP_NOMOVE);
		return;
	}
	gra_reloadmap();
}

void movewindowframe(WINDOWFRAME *wf, INTSML left, INTSML top)
{
	RECT fr;
	WINDOWPLACEMENT wp;

	((CChildFrame *)wf->wndframe)->GetWindowPlacement(&wp);
	fr = wp.rcNormalPosition;

	/* determine new window location */
	if (left == fr.left && top == fr.top) return;
	((CChildFrame *)wf->wndframe)->SetWindowPos(0, left, top, 0, 0, SWP_NOSIZE|SWP_NOZORDER);
}

/*
 * Routine to close the messages window if it is in front.  Returns nonzero if the
 * window was closed.
 */
INTSML closefrontmostmessages(void)
{
	if (gra_messagescurrent != 0 && gra_messageswindow != 0)
	{
		gra_messageswindow->DestroyWindow();
		gra_messagescurrent = 0;
		gra_messageswindow = 0;
		return(1);
	}
	return(0);
}

/*
 * Routine to bring window "win" to the front.
 */
void bringwindowtofront(WINDOWFRAME *fwf)
{
	REGISTER WINDOWFRAME *wf;

	/* validate the window frame */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if (wf == fwf) break;
	if (wf == NOWINDOWFRAME)
	{
		ttyputmsg("Warning: attempt to manipulate deleted window");
		return;
	}

	/* bring it to the top */
	((CChildFrame *)fwf->wndframe)->BringWindowToTop();
}

/*
 * Routine to organize the windows according to "how" (0: tile horizontally,
 * 1: tile vertically, 2: cascade).
 */
void adjustwindowframe(INTSML how)
{
	RECT r;
	REGISTER INTBIG children, ret;
	REGISTER WINDOWFRAME *wf;
	HWND *kids;
	CWnd *parent, *compmenu;

	/* figure out how many windows need to be rearranged */
	children = 0;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if (wf->floating == 0) children++;
	if (children <= 0) return;

	/* make a list of windows to rearrange */
	kids = (HWND *)emalloc(children * (sizeof HWND), el_tempcluster);
	if (kids == 0) return;
	children = 0;
	compmenu = NULL;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->floating == 0)
		{
			kids[children++] = wf->realwindow;
			parent = ((CChildFrame *)wf->wndframe)->GetParent();
		} else
		{
			compmenu = (CChildFrame *)wf->wndframe;
		}
	}

	/* determine area for windows */
	parent->GetClientRect(&r);
	if (compmenu != NULL)
	{
		/* remove component menu from area of tiling */
		gra_removewindowextent(&r, compmenu);
	}
	if (gra_messageswindow != 0)
	{
		/* remove messages menu from area of tiling */
		gra_removewindowextent(&r, gra_messageswindow);
	}

	/* rearrange the windows */
	switch (how)
	{
		case 0:		/* tile horizontally */
			ret = TileWindows(parent->m_hWnd, MDITILE_HORIZONTAL, &r, children, kids);
			break;
		case 1:		/* tile vertically */
			ret = TileWindows(parent->m_hWnd, MDITILE_VERTICAL, &r, children, kids);
			break;
		case 2:		/* cascade */
			ret = CascadeWindows(parent->m_hWnd, 0, &r, children, kids);
			break;
	}
	if (ret == 0)
	{
		ret = GetLastError();
		ttyputerr(_("Window rearrangement failed (error %ld)"), ret);
	}
	efree((char *)kids);
}

/*
 * Helper routine to remove the location of window "wnd" from the rectangle "r".
 */
void gra_removewindowextent(RECT *r, CWnd *wnd)
{
	RECT er;
	CWnd *parent;

	parent = wnd->GetParent();
	wnd->GetWindowRect(&er);
	parent->ScreenToClient(&er);
	if (er.right-er.left > er.bottom-er.top)
	{
		/* horizontal occluding window */
		if (er.bottom - r->top < r->bottom - er.top)
		{
			/* occluding window on top */
			r->top = er.bottom;
		} else
		{
			/* occluding window on bottom */
			r->bottom = er.top;
		}
	} else
	{
		/* vertical occluding window */
		if (er.right - r->left < r->right - er.left)
		{
			/* occluding window on left */
			r->left = er.right;
		} else
		{
			/* occluding window on right */
			r->right = er.left;
		}
	}
}

/*
 * routine to return the important pieces of information in placing the floating
 * palette with the fixed menu.  The maximum screen size is placed in "wid" and
 * "hei", and the amount of space that is being left for this palette on the
 * side of the screen is placed in "palettewidth".
 */
void getpaletteparameters(INTSML *wid, INTSML *hei, INTSML *palettewidth)
{
	RECT r;
	CMainFrame *wnd;

	wnd = (CMainFrame *)AfxGetMainWnd();
	wnd->GetClientRect(&r);
	*hei = r.bottom - r.top - 50;
	*wid = (INTSML)(r.right - r.left);
	*palettewidth = PALETTEWIDTH;
}

/*
 * Routine called when the component menu has moved to a different location
 * on the screen (left/right/top/bottom).  Resets local state of the position.
 */
void resetpaletteparameters(void)
{
}

RECT *gra_geteditorwindowlocation(void)
{
	static RECT redit;

	redit.top = gra_newwindowoffset + gra_screentop;
	redit.bottom = redit.top + (gra_screenbottom-gra_screentop) * 4 / 5 - 1;
	redit.left = gra_newwindowoffset + gra_screenleft + PALETTEWIDTH;
	redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH) * 4 / 5;

	/* is the editing window visible on this display? */
	if ((redit.bottom > gra_screenbottom) || (redit.right > gra_screenright))
	{
		gra_newwindowoffset = 0;
		redit.top = gra_screentop;
		redit.bottom = redit.top + (gra_screenbottom-gra_screentop) * 3 / 5;
		redit.left = gra_screenleft + PALETTEWIDTH;
		redit.right = redit.left + (gra_screenright-gra_screenleft-PALETTEWIDTH) * 4 / 5;
	}
	gra_newwindowoffset += 30;
	return(&redit);
}

/*
 * routine to build a new window frame
 */
INTSML gra_buildwindow(WINDOWFRAME *wf, INTSML floating, RECTAREA *r)
{
	RECT redit;
	REGISTER INTBIG i;
	REGISTER VARIABLE *varred, *vargreen, *varblue;
	CChildFrame *tempFrame;

	wf->floating = floating;
	wf->wndframe = 0;
	if (r != 0)
	{
		redit.left = r->left;   redit.right = r->right;
		redit.top = r->top;     redit.bottom = r->bottom;
	}
	if (floating == 0)
	{
		/* get editor window location */
		if (r == 0) redit = *gra_geteditorwindowlocation();

		/* create the editing window */
		tempFrame = new CChildFrame();
		tempFrame->Create(NULL, "Electric",
			WS_CHILD|WS_OVERLAPPEDWINDOW|WS_VISIBLE, redit, NULL, NULL);
		wf->wndframe = tempFrame;
		wf->realwindow = tempFrame->m_hWnd;
	} else
	{
		/* default palette window location */
		if (r == 0)
		{
			redit.top = gra_screentop + 2;
			redit.bottom = (gra_screentop+gra_screenbottom)/2;
			redit.left = gra_screenleft + 1;
			redit.right = redit.left + PALETTEWIDTH/2;
		}

		/* create the floating window */
#if 1	/* not a proper floating window, but it works */
		gra_creatingwindow = 1;
		tempFrame = new CChildFrame();
		tempFrame->m_wndFloating = 1;
		tempFrame->Create(NULL, "",
			WS_CHILD|WS_VISIBLE|WS_CAPTION, redit, NULL, NULL);
		gra_creatingwindow = 0;
		wf->wndframe = tempFrame;
		wf->realwindow = tempFrame->m_hWnd;
#else
		CMainFrame *wnd;
		wnd = (CMainFrame *)AfxGetMainWnd();
		wf->wndframe = wnd->m_wndComponentMenu;
		wf->realwindow = wnd->m_wndComponentMenu->m_hWnd;
		/* why doesn't this work? */
//		::SetWindowPos(tempFrame->m_hWnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE|SWP_NOSIZE);
//		tempFrame->ModifyStyleEx(0, WS_EX_TOPMOST);
//		tempFrame->SetWindowPos(&CWnd::wndTopMost, 0, 0, 0, 0, SWP_NOMOVE|SWP_NOSIZE);
#endif
	}
	if (wf->realwindow == 0) return(1);

	wf->hDC = (CDC *)((CChildFrame *)wf->wndframe)->GetDC();
	((CChildFrame *)wf->wndframe)->ReleaseDC((CDC *)wf->hDC);
	((CChildFrame *)wf->wndframe)->ShowWindow(SW_SHOW);
	wf->hPalette = 0;

	/* load any map the first time to establish the map segment length */
	el_maplength = 256;

	/* create a color palette for the buffer */
	varred   = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_green);
	varblue  = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE)
	{
		wf->pColorPalette = (LOGPALETTE *)emalloc(256 * (sizeof (PALETTEENTRY)) + 2 * (sizeof (WORD)),
			db_cluster);
		if (wf->pColorPalette == 0) return(1);
		wf->pColorPalette->palVersion = 0x300;
		wf->pColorPalette->palNumEntries = 256;
		for(i=0; i<256; i++)
		{
			wf->pColorPalette->palPalEntry[i].peRed   = (unsigned char)((INTBIG *)varred->addr)[i];
			wf->pColorPalette->palPalEntry[i].peGreen = (unsigned char)((INTBIG *)vargreen->addr)[i];
			wf->pColorPalette->palPalEntry[i].peBlue  = (unsigned char)((INTBIG *)varblue->addr)[i];
			wf->pColorPalette->palPalEntry[i].peFlags = 0;
		}
		wf->pColorPalette->palPalEntry[255].peRed   = (unsigned char)(255-((INTBIG *)varred->addr)[255]);
		wf->pColorPalette->palPalEntry[255].peGreen = (unsigned char)(255-((INTBIG *)vargreen->addr)[255]);
		wf->pColorPalette->palPalEntry[255].peBlue  = (unsigned char)(255-((INTBIG *)varblue->addr)[255]);
		wf->hPalette = (CPalette *)new CPalette();
		((CPalette *)wf->hPalette)->CreatePalette(wf->pColorPalette);
	}

	/* build the offscreen buffer for the window */
	if (gra_makeeditwindow(wf) != 0)
	{
/*		DestroyWindow(wf->realwindow); */
		return(1);
	}

	return(0);
}

/*
 * Routine to redraw editing window "wf" due to a drag or grow
 */
void gra_redrawdisplay(WINDOWFRAME *wf)
{
	us_beginchanges();
	us_drawmenu(-1, wf);
	us_endchanges(NOWINDOWPART);
	us_state |= HIGHLIGHTSET;
	us_showallhighlight();
}

/*
 *  Routine to allocate the offscreen buffer that corresponds with the window
 *  "wf->realwindow".  The buffer is 8-bits deep and is stored in "wf->window".
 *  Returns nonzero if memory cannot be allocated.
 */
INTSML gra_makeeditwindow(WINDOWFRAME *wf)
{
	RECT r;

	/* get information about the window that was just created */
	((CChildFrame *)wf->wndframe)->GetClientRect(&r);

	/* set frame characteristics */
	wf->swid = (INTSML)(r.right - r.left);
	wf->shei = (INTSML)(r.bottom - r.top);
	wf->revy = wf->shei - 1;
	wf->offscreendirty = 0;
	wf->screenleft   = 0;
	wf->screenright  = wf->swid;
	wf->screentop    = 0;
	wf->screenbottom = wf->shei;

	/* allocate the main offscreen buffer */
	if (gra_buildoffscreenbuffer(wf, wf->swid, wf->shei, (CDC **)&wf->hDCOff, &wf->hBitmap,
		&wf->bminfo, &wf->data, &wf->rowstart) != 0) return(1);

	/* attach the color palette to drawing window section */
	if (wf->hPalette != 0)
	{
		(void)((CDC *)wf->hDCOff)->SelectPalette((CPalette *)wf->hPalette, TRUE);
		(void)((CDC *)wf->hDCOff)->RealizePalette();
	}

	return(0);
}

/*
 * Routine to build an offscreen buffer that is "wid" by "hei" and based on window frame "wf".
 * The device context is placed in "hdc", the bitmap in "bitmap", the actual offscreen memory in
 * "databuffer" and the row start array in "rowstart".  Returns nonzero on error.
 */
INTSML gra_buildoffscreenbuffer(WINDOWFRAME *wf, INTSML wid, INTSML hei, CDC **hdc,
	HBITMAP *bitmap, BITMAPINFO **bmiInfo, char **databuffer, char ***rowstart)
{
	BITMAPINFOHEADER bmiHeader;
	REGISTER VARIABLE *varred, *vargreen, *varblue;
	RGBQUAD bmiColors[256];
	REGISTER INTSML i, j, retval;
	REGISTER INTBIG size;
	REGISTER char *ptr;

	/* make the info structure */
	*bmiInfo = (BITMAPINFO *)emalloc(sizeof(bmiHeader) + sizeof(bmiColors), db_cluster);
	if (*bmiInfo == 0) return(1);

	/* grab the device context */
	wf->hDC = (CDC *)((CChildFrame *)wf->wndframe)->GetDC();

	/* setup the header block */
	bmiHeader.biSize = (DWORD)sizeof(bmiHeader);
	bmiHeader.biWidth = (LONG)wid;
	bmiHeader.biHeight = (LONG)-hei;
	bmiHeader.biPlanes = 1;
	bmiHeader.biBitCount = 8;
	bmiHeader.biCompression = (DWORD)BI_RGB;
	bmiHeader.biSizeImage = 0;
	bmiHeader.biXPelsPerMeter = 0;
	bmiHeader.biYPelsPerMeter = 0;
	bmiHeader.biClrUsed = 256;
	bmiHeader.biClrImportant = 0;

	/* get color map information */
	varred = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred != NOVARIABLE && vargreen != NOVARIABLE && varblue != NOVARIABLE)
	{
		for(i=0; i<255; i++)
		{
			bmiColors[i].rgbRed = (unsigned char)((INTBIG *)varred->addr)[i];
			bmiColors[i].rgbGreen = (unsigned char)((INTBIG *)vargreen->addr)[i];
			bmiColors[i].rgbBlue = (unsigned char)((INTBIG *)varblue->addr)[i];
			bmiColors[i].rgbReserved = 0;
		}
		bmiColors[255].rgbRed = (unsigned char)(255-((INTBIG *)varred->addr)[255]);
		bmiColors[255].rgbGreen = (unsigned char)(255-((INTBIG *)vargreen->addr)[255]);
		bmiColors[255].rgbBlue = (unsigned char)(255-((INTBIG *)varblue->addr)[255]);
	}

	/* create a device context for this offscreen buffer */
	retval = 1;
	*hdc = new CDC();
	BOOL result = (*hdc)->CreateCompatibleDC((CDC *)wf->hDC);
	if (result)
	{
		/* prepare structure describing offscreen buffer */
		ptr = (char *)*bmiInfo;
		memcpy(ptr, &bmiHeader, sizeof(bmiHeader));
		ptr += sizeof(bmiHeader);
		memcpy(ptr, &bmiColors, sizeof(bmiColors));

		/* allocate offscreen buffer */
		*bitmap = CreateDIBSection((*hdc)->m_hDC, *bmiInfo, DIB_RGB_COLORS, (void **)databuffer, NULL, 0);
		if (*bitmap != 0)
		{
			(*hdc)->SelectObject(CBitmap::FromHandle(*bitmap));

			/* setup row pointers to offscreen buffer */
			*rowstart = (char **)emalloc(hei * (sizeof (char *)), db_cluster);
			if (*rowstart == 0) return(1);
			size = (wid+3)/4*4;
			for(j=0; j<hei; j++) (*rowstart)[j] = (*databuffer) + (size * j);
			retval = 0;
		}
	}

	/* release the device context */
	((CChildFrame *)wf->wndframe)->ReleaseDC((CDC *)wf->hDC);
	return(retval);
}

/*
 * Routine to rebuild the offscreen buffer when its size or depth has changed.
 * Returns nonzero if the window cannot be rebuilt.
 */
INTSML gra_remakeeditwindow(WINDOWFRAME *wf)
{
	/* free memory associated with the frame */
	if (wf->rowstart != 0)
	{
		efree((char *)wf->rowstart);
		efree((char *)wf->bminfo);
		DeleteObject(wf->hBitmap);
		delete (CDC *)wf->hDCOff;
	}
	wf->rowstart = 0;

	return(gra_makeeditwindow(wf));
}

/******************** MISCELLANEOUS EXTERNAL ROUTINES ********************/

/*
 * return nonzero if the capabilities in "want" are present
 */
INTSML graphicshas(INTSML want)
{
	/* only 1 status area, shared by each frame */
	if ((want&CANSTATUSPERFRAME) != 0) return(0);

	/* cannot run subprocesses */
	if ((want&CANRUNPROCESS) != 0) return(0);

	return(1);
}

/*
 * internal routine to beep the status display
 */
void ttybeep(void)
{
	if (us_tool->toolstate & TERMBEEP) MessageBeep(MB_ICONASTERISK);
}

extern "C" {
DIALOGITEM db_severeerrordialogitems[] =
{
 /*  1 */ {0, {80,8,104,72}, BUTTON, N_("Exit")},
 /*  2 */ {0, {80,96,104,160}, BUTTON, N_("Save")},
 /*  3 */ {0, {80,184,104,256}, BUTTON, N_("Continue")},
 /*  4 */ {0, {8,8,72,256}, MESSAGE, ""}
};
DIALOG db_severeerrordialog = {{50,75,163,341}, N_("Fatal Error"), 0, 4, db_severeerrordialogitems};
};

#define DSER_EXIT      1		/* Exit (button) */
#define DSER_SAVE      2		/* Save (button) */
#define DSER_CONTINUE  3		/* Continue (button) */
#define DSER_ERRMSG    4		/* Error message (stat text) */

void error(char *s, ...)
{
	va_list ap;
	char line[500];
	REGISTER INTBIG itemHit, *curstate;
	REGISTER INTBIG retval;
	REGISTER LIBRARY *lib;

	/* disable any session logging */
	if (us_logplay != NULL)
	{
		xclose(us_logplay);
		us_logplay = NULL;
	}

	/* build the error message */
	var_start(ap, s);
	evsnprintf(line, 500, s, ap);
	va_end(ap);

	/* display the severe error dialog box */
	if (DiaInitDialog(&db_severeerrordialog) != 0) return;

	/* load the message */
	DiaSetText(DSER_ERRMSG, line);

	/* loop until done */
	for(;;)
	{
		itemHit = DiaNextHit();
		if (itemHit == DSER_EXIT) exitprogram();
		if (itemHit == DSER_SAVE)
		{
			/* save libraries: make sure that backups are kept */
			curstate = io_getstatebits();
			if ((curstate[0]&BINOUTBACKUP) == BINOUTNOBACK)
			{
				curstate[0] |= BINOUTONEBACK;
				io_setstatebits(curstate);
			}

			/* save each modified library */
			for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			{
				if ((lib->userbits&HIDDENLIBRARY) != 0) continue;
				if ((lib->userbits&(LIBCHANGEDMAJOR|LIBCHANGEDMINOR)) == 0) continue;

				/* save the library in binary format */
				makeoptionstemporary(lib);
				retval = asktool(io_tool, "write", (INTBIG)lib, (INTBIG)"binary");
				restoreoptionstate(lib);
				if (retval != 0) return;
			}
		}
		if (itemHit == DSER_CONTINUE) break;
	}
	DiaDoneDialog();
}

/*
 * Routine to get the environment variable "name" and return its value.
 */
char *egetenv(char *name)
{
	return(0);
}

/*
 * Routine to fork a new process.  Returns the child process number if this is the
 * parent thread.  Returns 0 if this is the child thread.
 * Returns 1 if forking is not possible (process 1 is INIT on UNIX and can't possibly
 * be assigned to a normal process).
 */
INTBIG efork(void)
{
	return(1);
}

/*
 * Routine to run the string "command" in a shell.
 * Returns nonzero if the command cannot be run.
 */
INTBIG esystem(char *command)
{
	return(1);
}

/*
 * Routine to execute the program "program" with the arguments "args"
 */
void eexec(char *program, char *args[])
{
}

/*
 * routine to send signal "signal" to process "process".
 */
INTBIG ekill(INTBIG process)
{
	return(1);
}

/*
 * routine to wait for the completion of child process "process"
 */
void ewait(INTBIG process)
{
}

/*
 * Routine to determine the list of printers and return it.
 * The list terminates with a zero.
 */
char **eprinterlist(void)
{
	static char *lplist[1];

	lplist[0] = 0;
	return(lplist);
}

/*
 * Routine to establish the library directories from the environment.
 * The registry entry for this is stored in:
 *   HKEY_LOCAL_MACHINE\Software\Static Free Software\Electric\ELECTRIC_LIBDIR
 * If this directory is valid (i.e. has the file "cadrc" in it) then it is
 * used.  If the directory is not valid, the registry entry is removed.
 */
void setupenvironment(void)
{
	HKEY hKey;
	DWORD size;
	unsigned char tmp[256];
	char kVal[256], testfile[256];
	FILE *io;

	/* GetModuleFileName(0, testfile, 256); */

	/* default library location */
	(void)allocstring(&el_libdir, LIBDIR, db_cluster);

	/* see if a registry key overrides the location */
	(void)sprintf(kVal, "Software\\Static Free Software\\Electric");
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, kVal, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
	{
		if (RegQueryValueEx(hKey, "ELECTRIC_LIBDIR", NULL, NULL, tmp, &size) == ERROR_SUCCESS)
		{
			if (tmp[0] != 0)
			{
				strcpy(testfile, (char *)tmp);
				strcat(testfile, "cadrc");
				io = fopen(testfile, "r");
				if (io != 0)
				{
					/* valid "cadrc" file in directory, use it */
					fclose(io);
					(void)reallocstring(&el_libdir, (char *)tmp, db_cluster);
				} else
				{
					/* key points to bogus directory: delete it */
					RegCloseKey(hKey);
					if (RegCreateKey(HKEY_LOCAL_MACHINE, kVal, &hKey) == ERROR_SUCCESS)
					{
						(void)RegSetValueEx(hKey, "ELECTRIC_LIBDIR", 0, REG_SZ,
							(unsigned char *)"", 1);
					}
				}
			}
		}
		RegCloseKey(hKey);
	}

	/* set machine name */
	nextchangequiet();
	(void)setvalkey((INTBIG)us_tool, VTOOL, makekey("USER_machine"),
		(INTBIG)"Windows", VSTRING|VDONTSAVE);
}

void setlibdir(char *libdir)
{
	char *pp, abspath[256];
	HKEY hKey;
	INTBIG err;
	DWORD size;
	unsigned char tmp[256];
	char kVal[256];
	INTSML doupdate;

	/* make sure UNIX '/' isn't in use */
	for(pp = libdir; *pp != 0; pp++)
		if (*pp == '/') *pp = DIRSEP;

	(void)initinfstr();
	libdir = _fullpath(abspath, libdir, 256);
	(void)addstringtoinfstr(libdir);
	if (libdir[strlen(libdir)-1] != DIRSEP) (void)addtoinfstr(DIRSEP);
	(void)reallocstring(&el_libdir, returninfstr(), db_cluster);

	/* presume that registry must be updated */
	doupdate = 1;

	/* see what the current registry key says */
	(void)sprintf(kVal, "Software\\Static Free Software\\Electric");
	if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, kVal, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
	{
		if (RegQueryValueEx(hKey, "ELECTRIC_LIBDIR", NULL, NULL, tmp, &size) == ERROR_SUCCESS)
		{
			/* if the current registry entry is correct, do not update */
			if (strcmp(el_libdir, (char *)tmp) == 0) doupdate = 0;
		}
		RegCloseKey(hKey);
	}

	/* if the key must be updated, do it now */
	if (doupdate != 0)
	{
		if (RegCreateKey(HKEY_LOCAL_MACHINE, kVal, &hKey) == ERROR_SUCCESS)
		{
			err = RegSetValueEx(hKey, "ELECTRIC_LIBDIR", 0, REG_SZ,
				(unsigned char *)el_libdir, strlen(el_libdir)+1);
			if (err != 0) ttyputerr(_("Error %d setting key"), err);
			RegCloseKey(hKey);
		}
	}
}

/******************** MESSAGES WINDOW ROUTINES ********************/

/*
 * Routine to put the string "s" into the messages window.
 * Pops up the messages window if "important" is nonzero.
 */
void putmessagesstring(char *s, INTSML important)
{
	REGISTER INTBIG line, pos1, pos2;

	/* ensure window exists, is noniconic, and is on top */
	if (important != 0)
	{
		if (gra_messageswindow == 0)
		   (void)gra_makemessageswindow();
		if (gra_messageswindow->IsIconic())
			gra_messageswindow->ShowWindow(SW_RESTORE);
		if (gra_messagesnotvisible())
			gra_messageswindow->BringWindowToTop();
	}

	/* get next line and load it up */
	if (gra_messageswindow == 0) return;

	/* if messages buffer is running out of space, clear from the top */
	line = gra_editCtrl->GetLineCount();
	pos1 = gra_editCtrl->LineIndex(line-1);
	pos2 = gra_editCtrl->GetLimitText();
	if (pos2-pos1 < 2000)
	{
		gra_editCtrl->SetSel(0, 2000);
		gra_editCtrl->ReplaceSel("...");
	}

	gra_tomessagesbottom();
	gra_editCtrl->ReplaceSel(s);
	gra_editCtrl->ReplaceSel("\r\n");
}

void gra_tomessagesbottom(void)
{
	int pos;

	pos = gra_editCtrl->GetLimitText();
	gra_editCtrl->SetSel(pos, pos);
}

/*
 * Routine to return the name of the key that ends a session from the messages window.
 */
char *getmessageseofkey(void)
{
	return(_("ESC"));
}

/*
 * Routine to get a string from the scrolling messages window.  Returns zero if end-of-file
 * (^D) is typed.
 */
char *getmessagesstring(char *prompt)
{
	int line, pos, startpos, startposonline, charcount, pos2;
	char cTmpCh, *pt, ch[2];
	static char typedline[MAXTYPEDLINE];

	/* ensure window exists, is noniconic, and is on top */
	if (gra_messageswindow == 0)
	   (void)gra_makemessageswindow();
	if (gra_messageswindow->IsIconic())
		gra_messageswindow->ShowWindow(SW_RESTORE);
	if (gra_messagesnotvisible())
		gra_messageswindow->BringWindowToTop();
	flushscreen();

	/* advance to next line and allocate maximum number of characters */
	gra_messageswindow->ActivateFrame();
	gra_tomessagesbottom();
	gra_editCtrl->ReplaceSel(prompt);
	gra_editCtrl->GetSel(startpos, pos2);
	line = gra_editCtrl->GetLineCount() - 1;
	startposonline = gra_editCtrl->GetLine(line, typedline, MAXTYPEDLINE);

	for(;;)
	{
		gra_nextevent();
		if (gra_inputstate == NOEVENT) continue;
		cTmpCh = gra_inputstate & CHARREAD;
		if (cTmpCh != 0)
		{
			if (cTmpCh == CTRLDKEY || cTmpCh == ESCKEY)
			{
				/* ESC (eof) */
				gra_editCtrl->GetSel(pos, pos2);
				if (startpos == pos)
				{
					gra_editCtrl->ReplaceSel("\r\n");
					gra_inputstate = NOEVENT;
					return(NULL);
				}
				cTmpCh = 0x0D;
			}
			if (cTmpCh == BACKSPACEKEY)
			{
				/* backspace */
				gra_editCtrl->GetSel(pos, pos2);
				if (pos <= startpos) continue;
				gra_editCtrl->SetSel(pos-1, pos);
				ch[0] = 0;
				gra_editCtrl->ReplaceSel(ch);
				continue;
			}
			if (cTmpCh == 0x0A || cTmpCh == 0x0D)
			{
				/* end of line */
				gra_editCtrl->ReplaceSel("\r\n");
				charcount = gra_editCtrl->GetLine(line, typedline, MAXTYPEDLINE);
				typedline[charcount] = 0;
				pt = &typedline[startposonline];
				gra_inputstate = NOEVENT;
				return(pt);
			}
			gra_tomessagesbottom();
			ch[0] = cTmpCh;   ch[1] = 0;
			gra_editCtrl->ReplaceSel(ch);
		}
		gra_inputstate = NOEVENT;
	}
}

/*
 * routine to select fonts in the messages window
 */
void setmessagesfont(void)
{
	CFont *fnt;
	CFontDialog dlg;
	LOGFONT lf;

	if (dlg.DoModal() == IDOK)
	{
		/* Retrieve the dialog data */
		dlg.GetCurrentFont(&lf);
		fnt = new CFont();
		fnt->CreateFontIndirect(&lf);
		gra_editCtrl->SetFont(fnt);
	}
}

/*
 * routine to cut text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML cutfrommessages(void)
{
	if (gra_messageswindow != 0)
	{
		if (!gra_messageswindow->IsIconic())
		{
			CMDIFrameWnd *parent = gra_messageswindow->GetMDIFrame();
			CWnd *wnd = parent->MDIGetActive();
			if (gra_messageswindow == wnd)
			{
				/* do the cut */
				gra_editCtrl->Cut();
				return(1);
			}
		}
	}
	if (el_curwindowpart == NOWINDOWPART) return(0);
	if ((el_curwindowpart->state&WINDOWTYPE) == DISPWINDOW)
		gra_copyhighlightedtoclipboard();
	return(0);
}

/*
 * routine to copy text from the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML copyfrommessages(void)
{
	if (gra_messageswindow != 0)
	{
		if (!gra_messageswindow->IsIconic())
		{
			CMDIFrameWnd *parent = gra_messageswindow->GetMDIFrame();
			CWnd *wnd = parent->MDIGetActive();
			if (gra_messageswindow == wnd)
			{
				/* do the copy */
				gra_editCtrl->Copy();
				return(1);
			}
		}
	}
	if (el_curwindowpart == NOWINDOWPART) return(0);
	if ((el_curwindowpart->state&WINDOWTYPE) == DISPWINDOW)
		gra_copyhighlightedtoclipboard();
	return(0);
}

/*
 * routine to copy the highlighted graphics to the clipboard.
 */
void gra_copyhighlightedtoclipboard(void)
{
	REGISTER INTBIG wid, hei, *dest, x, y, r, g, b, index, hsize, dsize, rowbytes,
		*redmap, *greenmap, *bluemap;
	INTBIG lx, hx, ly, hy;
	REGISTER char *source;
	REGISTER unsigned char *data, *store;
	BITMAPINFOHEADER bmiHeader;
	REGISTER WINDOWFRAME *wf;
	REGISTER VARIABLE *varred, *vargreen, *varblue;
	HGLOBAL h;

	/* get information about the area to be copied */
	wf = el_curwindowpart->frame;
	if (us_getareabounds(&lx, &hx, &ly, &hy) == NONODEPROTO) return;
	(void)us_makescreen(&lx, &ly, &hx, &hy, el_curwindowpart);
	wid = hx - lx + 1;
	hei = hy - ly + 1;
	rowbytes = wid*4;
	dsize = rowbytes * hei;

	/* get colormap information */
	varred   = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_green);
	varblue  = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return;
	redmap = (INTBIG *)varred->addr;
	greenmap = (INTBIG *)vargreen->addr;
	bluemap = (INTBIG *)varblue->addr;

	/* build a buffer to hold the image */
	bmiHeader.biSize = (DWORD)sizeof(bmiHeader);
	bmiHeader.biWidth = wid;
	bmiHeader.biHeight = hei;
	bmiHeader.biPlanes = 1;
	bmiHeader.biBitCount = 32;
	bmiHeader.biCompression = (DWORD)BI_RGB;
	bmiHeader.biSizeImage = 0;
	bmiHeader.biXPelsPerMeter = 0;
	bmiHeader.biYPelsPerMeter = 0;
	bmiHeader.biClrUsed = 0;
	bmiHeader.biClrImportant = 0;
	hsize = sizeof (bmiHeader);
	h = GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE, hsize + dsize);
	if (h == 0) return;
	store = *((unsigned char **)h);
	memcpy(store, &bmiHeader, hsize);
	data = &store[hsize];

	/* load the image data */
	for(y=0; y<hei; y++)
	{
		dest = (INTBIG *)(&data[y * rowbytes]);
		source = &wf->rowstart[wf->revy-(y+ly)][lx];
		for(x=0; x<wid; x++)
		{
			index = ((*source++) & ~HIGHLIT) & 0xFF;
			r = redmap[index];
			g = greenmap[index];
			b = bluemap[index];
			*dest++ = b | (g << 8) | (r << 16);
		}
	}

	/* put the image into the clipboard */
	if (OpenClipboard(0) == 0) return;
	EmptyClipboard();
	SetClipboardData(CF_DIB, h);
	CloseClipboard();
}

/*
 * routine to paste text to the messages window if it is current.  Returns nonzero
 * if sucessful.
 */
INTSML pastetomessages(void)
{
	if (gra_messageswindow == 0) return(0);
	if (gra_messageswindow->IsIconic()) return(0);
	CMDIFrameWnd *parent = gra_messageswindow->GetMDIFrame();
	CWnd *wnd = parent->MDIGetActive();
	if (gra_messageswindow != wnd) return(0);

	/* do the copy */
	gra_editCtrl->Paste();
	return(1);
}

/*
 * routine to get the contents of the system cut buffer
 */
char *getcutbuffer(void)
{
	char *ptr;

	if (OpenClipboard(NULL) == 0) return("");
	ptr = (char *)GetClipboardData(CF_TEXT);
	(void)initinfstr();
	if (ptr != 0)
		(void)addstringtoinfstr(ptr);
	CloseClipboard();
	return(returninfstr());
}

/*
 * routine to set the contents of the system cut buffer to "msg"
 */
void setcutbuffer(char *msg)
{
	HGLOBAL hnd;

	if (OpenClipboard(NULL) == 0) return;

	/* Remove the current Clipboard contents */
	if (EmptyClipboard() == 0)
	{
		CloseClipboard();
		return;
	}

	hnd = GlobalAlloc(GHND, strlen(msg)+1);
	strcpy(*((char **)hnd), msg);
	(void)SetClipboardData(CF_TEXT, hnd);

	CloseClipboard();
}

/*
 * Routine to return nonzero if the messages window is obscured by
 * an editor window.  Should ignore edit windows below this!!!
 */
INTSML gra_messagesnotvisible(void)
{
	RECT mr, r;
	CWnd *wnd;

	gra_messageswindow->GetWindowRect(&mr);
	mr.top += 2;   mr.bottom -= 2;
	mr.left += 2;  mr.right -= 2;

	/* look for "previous" windows that may obscure this one */
	wnd = gra_messageswindow;
	for(;;)
	{
		wnd = wnd->GetNextWindow(GW_HWNDPREV);
		if (wnd == 0) break;
		wnd->GetWindowRect(&r);
		if (r.left < mr.right && r.right > mr.left &&
			r.top < mr.bottom && r.bottom > mr.top) return(1);
	}
	return(0);
}

/******************** STATUS BAR ROUTINES ********************/

/*
 * Routine to return the number of status lines on the display.
 */
INTSML ttynumstatuslines(void)
{
	return(MAXSTATUSLINES);
}

/*
 * Routine to free status field object "sf".
 */
void ttyfreestatusfield(STATUSFIELD *sf)
{
	INTSML i, j;

	for(i=0; i<gra_indicatorcount; i++)
	{
		if (gra_statusfields[i] != sf) continue;

		/* remove the field */
		efree(gra_statusfieldtext[i]);
		for(j=i+1; j<gra_indicatorcount; j++)
		{
			gra_statusfields[j-1] = gra_statusfields[j];
			gra_statusfieldtext[j-1] = gra_statusfieldtext[j];
		}
		gra_indicatorcount--;
		gra_redrawstatusindicators();
		break;
	}

	efree(sf->label);
	efree((char *)sf);
}

/*
 * Routine to display "message" in the status "field" of window "wf" (uses all windows
 * if "wf" is zero).  If "cancrop" is zero, field cannot be cropped and should be
 * replaced with "*" if there isn't room.
 */
void ttysetstatusfield(WINDOWFRAME *wf, STATUSFIELD *field, char *message, INTSML cancrop)
{
	INTSML len, i, j, pos, newpos;
	CMainFrame *wnd;

	/* figure out which indicator this is */
	if (field == 0) return;

	/* construct the status line */
	gra_localstring[0] = 0;
	(void)strcat(gra_localstring, field->label);
	(void)strcat(gra_localstring, message);
	len = strlen(gra_localstring);
	while (len > 0 && gra_localstring[len-1] == ' ') gra_localstring[--len] = 0;

	/* if this is a title bar setting, do it now */
	if (field->line == 0)
	{
		if (wf == NOWINDOWFRAME) return;
		if (*gra_localstring == 0) strcpy(gra_localstring, _("*** NO FACET ***"));
		((CChildFrame *)wf->wndframe)->SetWindowText(gra_localstring);
		return;
	}

	/* ignore null fields */
	if (*field->label == 0) return;

	/* see if this indicator is in the list */
	wnd = (CMainFrame *)AfxGetMainWnd();
	for(i=0; i<gra_indicatorcount; i++)
	{
		if (gra_statusfields[i]->line != field->line ||
			gra_statusfields[i]->startper != field->startper ||
			gra_statusfields[i]->endper != field->endper) continue;

		/* load the string */
		(void)reallocstring(&gra_statusfieldtext[i], gra_localstring, db_cluster);
		wnd->m_wndStatusBar.SetPaneText(i+1, gra_localstring);
		return;
	}

	/* not found: find where this field fits in the order */
	newpos = (field->line-1) * 100 + field->startper;
	for(i=0; i<gra_indicatorcount; i++)
	{
		pos = (gra_statusfields[i]->line-1) * 100 + gra_statusfields[i]->startper;
		if (newpos < pos) break;
	}
	for(j=gra_indicatorcount; j>i; j--)
	{
		gra_statusfieldtext[j] = gra_statusfieldtext[j-1];
		gra_statusfields[j] = gra_statusfields[j-1];
	}
	gra_statusfields[i] = field;
	(void)allocstring(&gra_statusfieldtext[i], gra_localstring, db_cluster);
	gra_indicatorcount++;

	/* reload all indicators */
	gra_redrawstatusindicators();
}

void gra_redrawstatusindicators(void)
{
	INTSML i, wid, screenwid, totalwid;
	RECT r;
	CMainFrame *wnd;

	wnd = (CMainFrame *)AfxGetMainWnd();
	wnd->GetClientRect(&r);
	screenwid = r.right - r.left - gra_indicatorcount * 5;
	wnd->m_wndStatusBar.SetIndicators(gra_indicators, gra_indicatorcount+1);
	totalwid = 0;
	for(i=0; i<gra_indicatorcount; i++)
		totalwid += gra_statusfields[i]->endper - gra_statusfields[i]->startper;
	for(i=0; i<=gra_indicatorcount; i++)
	{
		if (i == 0)
		{
			wnd->m_wndStatusBar.SetPaneInfo(i, gra_indicators[i], SBPS_NOBORDERS, 0);
			wnd->m_wndStatusBar.SetPaneText(i, "");
		} else
		{
			wid = gra_statusfields[i-1]->endper - gra_statusfields[i-1]->startper;
			wid = wid * screenwid / totalwid;
			wnd->m_wndStatusBar.SetPaneInfo(i, gra_indicators[i], SBPS_NORMAL, wid);
			wnd->m_wndStatusBar.SetPaneText(i, gra_statusfieldtext[i-1]);
		}
	}
}

/******************** GRAPHICS CONTROL ROUTINES ********************/

void flushscreen(void)
{
	WINDOWFRAME *wf;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->wndframe == 0) continue;

		/* if screen has not changed, stop now */
		if (wf->offscreendirty == 0) continue;
		wf->offscreendirty = 0;

		/* refresh screen */
		wf->hDC = (CDC *)((CChildFrame *)wf->wndframe)->GetDC();
		((CDC *)wf->hDC)->BitBlt(wf->copyrect.left, wf->copyrect.top, wf->copyrect.right-wf->copyrect.left,
			   wf->copyrect.bottom-wf->copyrect.top, (CDC *)wf->hDCOff, wf->copyrect.left, wf->copyrect.top, SRCCOPY);
		((CChildFrame *)wf->wndframe)->ReleaseDC((CDC *)wf->hDC);
	}
}

/*
 * Routine to accumulate the rectangle of change to the offscreen PixMap
 */
void gra_setrect(WINDOWFRAME *wf, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	if (wf->offscreendirty == 0)
	{
		wf->copyrect.left = lx;
		wf->copyrect.right = hx;
		wf->copyrect.top = ly;
		wf->copyrect.bottom = hy;
		wf->offscreendirty = 1;
	} else
	{
		if (lx < wf->copyrect.left)   wf->copyrect.left = lx;
		if (hx > wf->copyrect.right)  wf->copyrect.right = hx;
		if (ly < wf->copyrect.top)    wf->copyrect.top = ly;
		if (hy > wf->copyrect.bottom) wf->copyrect.bottom = hy;
	}
}

void gra_reloadmap(void)
{
	REGISTER VARIABLE *varred, *vargreen, *varblue;

	varred = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_red);
	vargreen = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_green);
	varblue = getvalkey((INTBIG)us_tool, VTOOL, VINTEGER|VISARRAY, us_colormap_blue);
	if (varred == NOVARIABLE || vargreen == NOVARIABLE || varblue == NOVARIABLE) return;
	colormapload((INTBIG *)varred->addr, (INTBIG *)vargreen->addr, (INTBIG *)varblue->addr, 0, 255);
}

void colormapload(INTBIG *red, INTBIG *green, INTBIG *blue, INTSML low, INTSML high)
{
	INTSML i;
	REGISTER WINDOWFRAME *wf;
	RGBQUAD bmiColors[256];

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		/* do not touch the color map when in B&W mode */
		if (wf->pColorPalette == 0) continue;

		i = GetDIBColorTable(((CDC *)wf->hDCOff)->m_hDC, 0, 256, bmiColors);
		for(i=low; i<=high; i++)
		{
			bmiColors[i].rgbRed = (unsigned char)red[i-low];
			bmiColors[i].rgbGreen = (unsigned char)green[i-low];
			bmiColors[i].rgbBlue = (unsigned char)blue[i-low];
			bmiColors[i].rgbReserved = 0;
			if (i == 255)
			{
				bmiColors[i].rgbRed = (unsigned char)(255-red[i-low]);
				bmiColors[i].rgbGreen = (unsigned char)(255-green[i-low]);
				bmiColors[i].rgbBlue = (unsigned char)(255-blue[i-low]);
			}

			wf->pColorPalette->palPalEntry[i].peRed = bmiColors[i].rgbRed;
			wf->pColorPalette->palPalEntry[i].peGreen = bmiColors[i].rgbGreen;
			wf->pColorPalette->palPalEntry[i].peBlue = bmiColors[i].rgbBlue;
			wf->pColorPalette->palPalEntry[i].peFlags = 0;
		}
		i = SetDIBColorTable(((CDC *)wf->hDCOff)->m_hDC, 0, 256, bmiColors);

		if (wf->hPalette != 0)
			delete (CPalette *)wf->hPalette;

		wf->hPalette = (CPalette *)new CPalette();
		((CPalette *)wf->hPalette)->CreatePalette(wf->pColorPalette);

		/* graphics window */
		((CDC *)wf->hDCOff)->SelectPalette((CPalette *)wf->hPalette, TRUE);
		((CDC *)wf->hDCOff)->RealizePalette();
		gra_setrect(wf, 0, wf->swid-1, 0, wf->shei-1);
	}
}

/*
 * helper routine to set the cursor shape to "state"
 */
void setdefaultcursortype(INTSML state)
{
	if (us_cursorstate == state) return;

	switch (state)
	{
		case NORMALCURSOR:  SetCursor(gra_normalCurs);   break;
		case WANTTTYCURSOR: SetCursor(gra_wantttyCurs);  break;
		case PENCURSOR:     SetCursor(gra_penCurs);      break;
		case NULLCURSOR:    SetCursor(gra_normalCurs);   break;
		case MENUCURSOR:    SetCursor(gra_menuCurs);     break;
		case HANDCURSOR:    SetCursor(gra_handCurs);     break;
		case TECHCURSOR:    SetCursor(gra_techCurs);     break;
		case IBEAMCURSOR:   SetCursor(gra_ibeamCurs);    break;
		case WAITCURSOR:    SetCursor(gra_waitCurs);     break;
		case LRCURSOR:      SetCursor(gra_lrCurs);       break;
		case UDCURSOR:      SetCursor(gra_udCurs);       break;
	}
	ShowCursor(TRUE);
	us_cursorstate = state;
}

/*
 * Routine to change the default cursor (to indicate modes).
 */
void setnormalcursor(INTSML curs)
{
	us_normalcursor = curs;
	setdefaultcursortype(us_normalcursor);
}

/******************** GRAPHICS LINES ********************/

void screendrawline(WINDOWPART *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2, GRAPHICS *desc, INTSML texture)
{
	REGISTER INTSML col, mask, lx, hx, ly, hy;
	REGISTER WINDOWFRAME *wf;

	/* make sure the image buffer has not moved */
	wf = win->frame;

	/* get line type parameters */
	col = desc->col;      mask = ~desc->bits;
	y1 = wf->revy - y1;   y2 = wf->revy - y2;

	switch (texture)
	{
		case 0: gra_drawline(wf, x1, y1, x2, y2, col, mask);            break;
		case 1: gra_drawpatline(wf, x1, y1, x2, y2, col, mask, 0x88);   break;
		case 2: gra_drawpatline(wf, x1, y1, x2, y2, col, mask, 0xE7);   break;
		case 3: gra_drawthickline(wf, x1, y1, x2, y2, col, mask);       break;
	}
		
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(wf, lx, (INTSML)(hx+1), ly, (INTSML)(hy+1));
}

void screeninvertline(WINDOWPART *win, INTSML x1, INTSML y1, INTSML x2, INTSML y2)
{
	REGISTER INTSML lx, hx, ly, hy, dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;
	REGISTER WINDOWFRAME *wf;

	/* get line type parameters */
	wf = win->frame;
	y1 = wf->revy - y1;   y2 = wf->revy - y2;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		wf->rowstart[y][x] = ~wf->rowstart[y][x];

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;   d += incr2;
			}
			wf->rowstart[y][x] = ~wf->rowstart[y][x];
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		wf->rowstart[y][x] = ~wf->rowstart[y][x];

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;   d += incr2;
			}
			wf->rowstart[y][x] = ~wf->rowstart[y][x];
		}
	}
	if (x1 < x2) { lx = x1;   hx = x2; } else { lx = x2;   hx = x1; }
	if (y1 < y2) { ly = y1;   hy = y2; } else { ly = y2;   hy = y1; }
	gra_setrect(wf, lx, (INTSML)(hx+1), ly, (INTSML)(hy+1));
}

void gra_drawpatline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col,
	INTSML mask, INTSML pattern)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr, i;

	/* initialize counter for line style */
	i = 0;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;
				d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;
				d += incr2;
			}
			if (i == 7) i = 0; else i++;
			if ((pattern & (1 << i)) == 0) continue;
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	}
}

void gra_drawline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;
				d += incr2;
			}
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;
				d += incr2;
			}
			wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
		}
	}
}

void gra_drawthickline(WINDOWFRAME *wf, INTSML x1, INTSML y1, INTSML x2, INTSML y2, INTSML col, INTSML mask)
{
	INTSML dx, dy, d, incr1, incr2, x, y, xend, yend, yincr, xincr;

	/* initialize the Bresenham algorithm */
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (dx > dy)
	{
		/* initialize for lines that increment along X */
		incr1 = 2 * dy;
		d = incr2 = 2 * (dy - dx);
		if (x1 > x2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (yend < y) yincr = -1; else yincr = 1;
		gra_drawthickpoint(wf, x, y, mask, col);

		/* draw line that increments along X */
		while (x < xend)
		{
			x++;
			if (d < 0) d += incr1; else
			{
				y += yincr;
				d += incr2;
			}
			gra_drawthickpoint(wf, x, y, mask, col);
		}
	} else
	{
		/* initialize for lines that increment along Y */
		incr1 = 2 * dx;
		d = incr2 = 2 * (dx - dy);
		if (y1 > y2)
		{
			x = x2;   y = y2;   xend = x1;   yend = y1;
		} else
		{
			x = x1;   y = y1;   xend = x2;   yend = y2;
		}
		if (xend < x) xincr = -1; else xincr = 1;
		gra_drawthickpoint(wf, x, y, mask, col);

		/* draw line that increments along X */
		while (y < yend)
		{
			y++;
			if (d < 0) d += incr1; else
			{
				x += xincr;
				d += incr2;
			}
			gra_drawthickpoint(wf, x, y, mask, col);
		}
	}
}

void gra_drawthickpoint(WINDOWFRAME *wf, INTSML x, INTSML y, INTSML mask, INTSML col)
{
	wf->rowstart[y][x] = (wf->rowstart[y][x]&mask) | col;
	if (x > 0)
		wf->rowstart[y][x-1] = (wf->rowstart[y][x-1]&mask) | col;
	if (x < wf->swid-1)
		wf->rowstart[y][x+1] = (wf->rowstart[y][x+1]&mask) | col;
	if (y > 0)
		wf->rowstart[y-1][x] = (wf->rowstart[y-1][x]&mask) | col;
	if (y < wf->shei-1)
		wf->rowstart[y+1][x] = (wf->rowstart[y+1][x]&mask) | col;
}

/******************** GRAPHICS POLYGONS ********************/

void screendrawpolygon(WINDOWPART *win, INTBIG *x, INTBIG *y, INTSML count, GRAPHICS *desc)
{
	REGISTER INTBIG i, j, k, l, ycur, yrev, wrap, lx, hx, ly, hy;
	REGISTER char *row;
	REGISTER INTSML col, mask, style, pat;
	REGISTER POLYSEG *a, *active, *edge, *lastedge, *left, *edgelist;
	REGISTER WINDOWFRAME *wf;

	/* get parameters */
	wf = win->frame;
	col = desc->col;   mask = ~desc->bits;
	style = desc->colstyle & NATURE;

	/* set redraw area */
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			lx = hx = x[i];
			ly = hy = y[i];
		} else
		{
			lx = mini(lx, x[i]);
			hx = maxi(hx, x[i]);
			ly = mini(ly, y[i]);
			hy = maxi(hy, y[i]);
		}
	}
	gra_setrect(wf, (INTSML)lx, (INTSML)(hx + 1), (INTSML)(wf->revy-hy),
		(INTSML)(wf->revy-ly + 1));

	/* make sure there is room in internal structures */
	if (count > gra_polysegcount)
	{
		if (gra_polysegcount > 0) efree((char *)gra_polysegs);
		gra_polysegcount = 0;
		gra_polysegs = (POLYSEG *)emalloc(count * (sizeof (POLYSEG)), us_tool->cluster);
		if (gra_polysegs == 0) return;
		gra_polysegcount = count;
	}

	/* fill in internal structures */
	edgelist = NOPOLYSEG;
	for(i=0; i<count; i++)
	{
		if (i == 0)
		{
			gra_polysegs[i].fx = x[count-1];
			gra_polysegs[i].fy = y[count-1];
		} else
		{
			gra_polysegs[i].fx = x[i-1];
			gra_polysegs[i].fy = y[i-1];
		}
		gra_polysegs[i].tx = x[i];   gra_polysegs[i].ty = y[i];

		/* draw the edge lines to make the polygon clean */
		if ((desc->colstyle&(NATURE|OUTLINEPAT)) != PATTERNED)
			gra_drawline(wf, (INTSML)gra_polysegs[i].fx, (INTSML)(wf->revy - gra_polysegs[i].fy),
				(INTSML)gra_polysegs[i].tx, (INTSML)(wf->revy - gra_polysegs[i].ty), col, mask);

		/* compute the direction of this edge */
		j = gra_polysegs[i].ty - gra_polysegs[i].fy;
		if (j > 0) gra_polysegs[i].direction = 1; else
			if (j < 0) gra_polysegs[i].direction = -1; else
				gra_polysegs[i].direction = 0;

		/* compute the X increment of this edge */
		if (j == 0) gra_polysegs[i].increment = 0; else
		{
			gra_polysegs[i].increment = gra_polysegs[i].tx - gra_polysegs[i].fx;
			if (gra_polysegs[i].increment != 0) gra_polysegs[i].increment =
				(gra_polysegs[i].increment * 65536 - j + 1) / j;
		}
		gra_polysegs[i].tx <<= 16;   gra_polysegs[i].fx <<= 16;

		/* make sure "from" is above "to" */
		if (gra_polysegs[i].fy > gra_polysegs[i].ty)
		{
			j = gra_polysegs[i].tx;
			gra_polysegs[i].tx = gra_polysegs[i].fx;
			gra_polysegs[i].fx = j;
			j = gra_polysegs[i].ty;
			gra_polysegs[i].ty = gra_polysegs[i].fy;
			gra_polysegs[i].fy = j;
		}

		/* insert this edge into the edgelist, sorted by ascending "fy" */
		if (edgelist == NOPOLYSEG)
		{
			edgelist = &gra_polysegs[i];
			gra_polysegs[i].nextedge = NOPOLYSEG;
		} else
		{
			/* insert by ascending "fy" */
			if (edgelist->fy > gra_polysegs[i].fy)
			{
				gra_polysegs[i].nextedge = edgelist;
				edgelist = &gra_polysegs[i];
			} else for(a = edgelist; a != NOPOLYSEG; a = a->nextedge)
			{
				if (a->nextedge == NOPOLYSEG ||
					a->nextedge->fy > gra_polysegs[i].fy)
				{
					/* insert after this */
					gra_polysegs[i].nextedge = a->nextedge;
					a->nextedge = &gra_polysegs[i];
					break;
				}
			}
		}
	}

	/* scan polygon and render */
	active = NOPOLYSEG;
	while (active != NOPOLYSEG || edgelist != NOPOLYSEG)
	{
		if (active == NOPOLYSEG)
		{
			active = edgelist;
			active->nextactive = NOPOLYSEG;
			edgelist = edgelist->nextedge;
			ycur = active->fy;
		}

		/* introduce edges from edge list into active list */
		while (edgelist != NOPOLYSEG && edgelist->fy <= ycur)
		{
			/* insert "edgelist" into active list, sorted by "fx" coordinate */
			if (active->fx > edgelist->fx ||
				(active->fx == edgelist->fx && active->increment > edgelist->increment))
			{
				edgelist->nextactive = active;
				active = edgelist;
				edgelist = edgelist->nextedge;
			} else for(a = active; a != NOPOLYSEG; a = a->nextactive)
			{
				if (a->nextactive == NOPOLYSEG ||
					a->nextactive->fx > edgelist->fx ||
						(a->nextactive->fx == edgelist->fx &&
							a->nextactive->increment > edgelist->increment))
				{
					/* insert after this */
					edgelist->nextactive = a->nextactive;
					a->nextactive = edgelist;
					edgelist = edgelist->nextedge;
					break;
				}
			}
		}

		/* generate regions to be filled in on current scan line */
		wrap = 0;
		left = active;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			wrap = wrap + edge->direction;
			if (wrap == 0)
			{
				j = (left->fx + 32768) >> 16;
				k = (edge->fx + 32768) >> 16;
				yrev = wf->revy - ycur;
				row = wf->rowstart[yrev];
				if (style == PATTERNED)
				{
					/* patterned fill */
					pat = desc->raster[yrev&7];
					if (pat != 0)
					{
						for(l=j; l<=k; l++)
						{
							if ((pat & (1 << (15-(l&15)))) != 0)
								row[l] = (row[l] & mask) | col;
						}
					}
				} else
				{
					/* solid fill */
					for(l=j; l<=k; l++)
						row[l] = (row[l] & mask) | col;
				}
				left = edge->nextactive;
			}
		}
		ycur++;

		/* update edges in active list */
		lastedge = NOPOLYSEG;
		for(edge = active; edge != NOPOLYSEG; edge = edge->nextactive)
		{
			if (ycur >= edge->ty)
			{
				if (lastedge == NOPOLYSEG) active = edge->nextactive;
					else lastedge->nextactive = edge->nextactive;
			} else
			{
				edge->fx += edge->increment;
				lastedge = edge;
			}
		}
	}
}

/******************** GRAPHICS BOXES ********************/

void screendrawbox(WINDOWPART *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy, GRAPHICS *desc)
{
	REGISTER INTSML col, mask, style, x, y, pat, top, bottom, left, right;
	REGISTER char *thisrow;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;

	/* get graphics parameters */
	col = desc->col;   mask = ~desc->bits;
	style = desc->colstyle & NATURE;
	left = lowx;                  right = highx + 1;
	bottom = wf->revy-lowy + 1;   top = wf->revy-highy;

	if (style == PATTERNED)
	{
		/* special case the patterned fill */
		for(y=top; y<bottom; y++)
		{
			pat = desc->raster[y&7];
			if (pat == 0) continue;
			thisrow = wf->rowstart[y];
			for(x=left; x<right; x++)
			{
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & mask) | col;
			}
		}
	} else
	{
		for(y=top; y<bottom; y++)
		{
			thisrow = wf->rowstart[y];
			for(x=left; x<right; x++)
				thisrow[x] = (thisrow[x] & mask) | col;
		}
	}

	gra_setrect(wf, left, right, top, bottom);
}

/*
 * routine to invert the bits in the box from (lowx, lowy) to (highx, highy)
 */
void screeninvertbox(WINDOWPART *win, INTSML lowx, INTSML highx, INTSML lowy, INTSML highy)
{
	REGISTER INTSML top, bottom, left, right, x, y;
	REGISTER char *thisrow;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;
	left = lowx;
	right = highx + 1;
	bottom = wf->revy - lowy + 1;
	top = wf->revy - highy;

	for(y=top; y<bottom; y++)
	{
		thisrow = wf->rowstart[y];
		for(x=left; x<right; x++) thisrow[x] = ~thisrow[x];
	}

	gra_setrect(wf, left, right, top, bottom);
}

/*
 * routine to move bits on the display starting with the area at
 * (sx,sy) and ending at (dx,dy).  The size of the area to be
 * moved is "wid" by "hei".
 */
void screenmovebox(WINDOWPART *win, INTSML sx, INTSML sy, INTSML wid, INTSML hei, INTSML dx, INTSML dy)
{
	REGISTER INTSML totop, tobottom, toleft, toright, fromleft, fromtop;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;

	/* setup source rectangle */
	fromleft = sx;
	fromtop = wf->revy + 1 - sy - hei;

	/* setup destination rectangle */
	toleft = dx;
	toright = toleft + wid;
	totop = wf->revy + 1 - dy - hei;
	tobottom = totop + hei;

	((CDC *)wf->hDCOff)->BitBlt(toleft, totop, wid, hei, (CDC *)wf->hDCOff,
		fromleft, fromtop, SRCCOPY);

	gra_setrect(wf, toleft, toright, totop, tobottom);
}

/*
 * routine to save the contents of the box from "lx" to "hx" in X and from
 * "ly" to "hy" in Y.  A code is returned that identifies this box for
 * overwriting and restoring.  The routine returns -1 if there is a error.
 */
INTBIG screensavebox(WINDOWPART *win, INTSML lx, INTSML hx, INTSML ly, INTSML hy)
{
	SAVEDBOX *box;
	REGISTER INTBIG xsize, ysize;
	REGISTER INTSML i, x, y;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *source, *dest;

	wf = win->frame;
	i = ly;
	ly = wf->revy-hy;
	hy = wf->revy-i;
	xsize = hx-lx+1;
	ysize = hy-ly+1;
	box = (SAVEDBOX *)emalloc((sizeof (SAVEDBOX)), us_tool->cluster);
	if (box == 0) return(-1);

	if (gra_buildoffscreenbuffer(wf, (INTSML)xsize, (INTSML)ysize, &box->hMemDC,
		&box->hBox, &box->bminfo, &box->data, &box->rowstart) != 0) return(-1);

	box->win = win;
	box->nextsavedbox = gra_firstsavedbox;
	gra_firstsavedbox = box;
	box->lx = lx;           box->hx = hx;
	box->ly = ly;           box->hy = hy;

	/* move the bits */
	for(y=0; y<ysize; y++)
	{
		source = &wf->rowstart[ly+y][lx];
		dest = box->rowstart[y];
		for(x=0; x<xsize; x++) *dest++ = *source++;
	}

	return((INTBIG)box);
}

/*
 * routine to shift the saved box "code" so that it is restored in a different location,
 * offset by (dx,dy)
 */
void screenmovesavedbox(INTBIG code, INTSML dx, INTSML dy)
{
	REGISTER SAVEDBOX *box;

	if (code == -1) return;
	box = (SAVEDBOX *)code;
	box->lx += dx;       box->hx += dx;
	box->ly -= dy;       box->hy -= dy;
}

/*
 * routine to restore saved box "code" to the screen.  "destroy" is:
 *  0   restore box, do not free memory
 *  1   restore box, free memory
 * -1   free memory
 * Returns nonzero if there is an error.
 */
INTSML screenrestorebox(INTBIG code, INTSML destroy)
{
	REGISTER SAVEDBOX *box, *lbox, *tbox;
	REGISTER INTSML x, y, xsize, ysize;
	REGISTER char *source, *dest;
	REGISTER WINDOWFRAME *wf;

	/* get the box */
	if (code == -1) return(1);
	box = (SAVEDBOX *)code;

	/* move the bits */
	if (destroy >= 0)
	{
		wf = box->win->frame;
		xsize = box->hx - box->lx + 1;
		ysize = box->hy - box->ly + 1;
		for(y=0; y<ysize; y++)
		{
			dest = &wf->rowstart[box->ly+y][box->lx];
			source = box->rowstart[y];
			for(x=0; x<xsize; x++) *dest++ = *source++;
		}
		efree((char *)box->rowstart);
		efree((char *)box->bminfo);
		DeleteObject(box->hBox);
		delete box->hMemDC;
		gra_setrect(wf, box->lx, (INTSML)(box->hx+1), box->ly, (INTSML)(box->hy+1));
	}

	/* destroy this box's memory if requested */
	if (destroy != 0)
	{
		lbox = NOSAVEDBOX;
		for(tbox = gra_firstsavedbox; tbox != NOSAVEDBOX; tbox = tbox->nextsavedbox)
		{
			if (tbox == box)
				break;
			lbox = tbox;
		}
		if (lbox == NOSAVEDBOX)
			gra_firstsavedbox = box->nextsavedbox;
		else
			lbox->nextsavedbox = box->nextsavedbox;
		efree((char *)box);
	}
	return(0);
}

/******************** GRAPHICS TEXT ********************/

/*
 * Routine to return the number of typefaces available on the system
 * and to return their names in the array "list".
 */
INTBIG screengetfacelist(char ***list)
{
	CDC *curdc;
	CChildFrame *child;

	if (gra_numfaces == 0)
	{
		if (el_curwindowframe != NOWINDOWFRAME)
			child = (CChildFrame *)el_curwindowframe->wndframe; else
				child = gra_messageswindow;
		curdc = (CDC *)child->GetDC();
		EnumFonts(curdc->m_hDC, NULL, (FONTENUMPROC)MyEnumFaces, (LPARAM)NULL);
		if (gra_numfaces > VTMAXFACE)
		{
			ttyputerr(_("Warning: found %ld fonts, but can only keep track of %d"), gra_numfaces,
				VTMAXFACE);
			gra_numfaces = VTMAXFACE;
		}
		esort(&gra_facelist[1], gra_numfaces-1, sizeof (char *), sort_stringascending);
	}
	*list = gra_facelist;
	return(gra_numfaces);
}

/*
 * Routine to return the default typeface used on the screen.
 */
char *screengetdefaultfacename(void)
{
	return("Arial");
}

/*
 * Callback routine to get the number of fonts
 */
int APIENTRY MyEnumFaces(LPLOGFONT lpLogFont, LPTEXTMETRIC lpTEXTMETRICs, DWORD fFontType, LPVOID lpData)
{
	INTBIG newfacetotal, i;
	char **newfacelist;

	if (gra_numfaces == 0) newfacetotal = 2; else
		newfacetotal = gra_numfaces + 1;
	newfacelist = (char **)emalloc(newfacetotal * (sizeof (char *)), us_tool->cluster);
	if (newfacelist == 0) return(0);
	for(i=0; i<gra_numfaces; i++)
		newfacelist[i] = gra_facelist[i];
	if (gra_numfaces > 0) efree((char *)gra_facelist);
	gra_facelist = newfacelist;
	if (gra_numfaces == 0)
		(void)allocstring(&gra_facelist[gra_numfaces++], "DEFAULT FACE", us_tool->cluster);
	(void)allocstring(&gra_facelist[gra_numfaces++], lpLogFont->lfFaceName, us_tool->cluster);
	return(1);
}

void screensettextinfo(WINDOWPART *win, TECHNOLOGY *tech, UINTBIG *descript)
{
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;
	wf->hFont = 0;
	wf->hFont = gra_gettextfont(win, tech, descript);
	if (wf->hFont == 0) return;
	((CDC *)wf->hDCOff)->SelectObject((CFont *)wf->hFont);
}

CFont *gra_gettextfont(WINDOWPART *win, TECHNOLOGY *tech, UINTBIG *descript)
{
	static CFont *txteditor = 0, *txtmenu = 0;
	CFont *theFont;
	REGISTER INTBIG font, size, italic, bold, underline, face, i;
	REGISTER UINTBIG hash;
	REGISTER char *facename, **list;

	font = TDGETSIZE(descript);
	if (font == TXTEDITOR)
	{
		if (txteditor == 0) txteditor = gra_createtextfont(12, "Lucida Console", 0, 0, 0);
		return(txteditor);
	}
	if (font == TXTMENU)
	{
		if (txtmenu == 0)   txtmenu = gra_createtextfont(13, "MS Sans Serif", 0, 0, 0);
		return(txtmenu);
	}
	size = truefontsize(font, win, tech);
	if (size < 4) size = 4;
	face = TDGETFACE(descript);
	italic = TDGETITALIC(descript);
	bold = TDGETBOLD(descript);
	underline = TDGETUNDERLINE(descript);
	if (face == 0 && italic == 0 && bold == 0 && underline == 0)
	{
		if (size >= MAXCACHEDFONTS) size = MAXCACHEDFONTS-1;
		if (gra_fontcache[size] == 0)
			gra_fontcache[size] = gra_createtextfont(size, "Arial", 0, 0, 0);
		return(gra_fontcache[size]);
	} else
	{
		hash = size + 3*italic + 5*bold + 7*underline + 11*face;
		hash %= FONTHASHSIZE;
		for(i=0; i<FONTHASHSIZE; i++)
		{
			if (gra_fonthash[hash].font == 0) break;
			if (gra_fonthash[hash].face == face && gra_fonthash[hash].size == size &&
				gra_fonthash[hash].italic == italic && gra_fonthash[hash].bold == bold &&
				gra_fonthash[hash].underline == underline)
					return(gra_fonthash[hash].font);
			hash++;
			if (hash >= FONTHASHSIZE) hash = 0;
		}
		facename = "Arial";
		if (face > 0)
		{
			if (gra_numfaces == 0) (void)screengetfacelist(&list);
			if (face < gra_numfaces)
				facename = gra_facelist[face];
		}
		theFont = gra_createtextfont(size, facename, italic, bold, underline);
		if (gra_fonthash[hash].font == 0)
		{
			gra_fonthash[hash].font = theFont;
			gra_fonthash[hash].face = face;
			gra_fonthash[hash].size = size;
			gra_fonthash[hash].italic = italic;
			gra_fonthash[hash].bold = bold;
			gra_fonthash[hash].underline = underline;
		}
		return(theFont);
	}
}

CFont *gra_createtextfont(INTBIG fontSize, char *face, INTBIG italic, INTBIG bold,
	INTBIG underline)
{
	LOGFONT lf;
	CFont *hf;

	lf.lfHeight = -fontSize;
	strcpy(lf.lfFaceName, face);
	lf.lfWidth = 0;
	lf.lfEscapement = 0;
	lf.lfOrientation = 0;
	if (bold != 0) lf.lfWeight = FW_BOLD; else
		lf.lfWeight = FW_NORMAL;
	if (italic != 0) lf.lfItalic = 1; else
		lf.lfItalic = 0;
	if (underline != 0) lf.lfUnderline = 1; else
		lf.lfUnderline = 0;
	lf.lfStrikeOut = 0;
	lf.lfCharSet = DEFAULT_CHARSET;
	lf.lfOutPrecision = OUT_STROKE_PRECIS;
	lf.lfClipPrecision = CLIP_STROKE_PRECIS;
	lf.lfQuality = DEFAULT_QUALITY;
	lf.lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;

	hf = new CFont();
	hf->CreateFontIndirect(&lf);
	return(hf);
}

void screengettextsize(WINDOWPART *win, char *str, INTSML *x, INTSML *y)
{
	INTSML len;
	CSize textSize;
	REGISTER WINDOWFRAME *wf;

	wf = win->frame;
	len = strlen(str);
	textSize = ((CDC *)wf->hDCOff)->GetTextExtent((LPCTSTR)str, len);
	*x = (INTSML)textSize.cx;
	*y = (INTSML)textSize.cy+1;
}

void screendrawtext(WINDOWPART *win, INTSML atx, INTSML aty, char *s, GRAPHICS *desc)
{
	REGISTER INTSML i, col, top, bottom, left, right, mask, j, pos, dpos,
		startx, endx, lx, hx, ly, hy;
	INTSML wid, hei;
	char **rowstart;
	CSize textSize;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *ptr, *dest;

	/* copy the string and correct it */
	gettextbits(win, s, &wid, &hei, &rowstart);
	if (wid == 0 || hei == 0) return;

	/* get parameters */
	wf = win->frame;
	col = desc->col;   mask = ~desc->bits;
	aty = wf->revy - aty;
	lx = win->uselx;   hx = win->usehx;
	ly = wf->revy - win->usehy;
	hy = wf->revy - win->usely;

	/* text is too high - rescale it? */
	if (aty-hei <= 0) return;

	/* copy text buffer to the main offscreen buffer */
	if (atx < lx) startx = lx-atx; else startx = 0;
	if (wid + atx >= hx) endx = hx - atx; else endx = wid;
	pos = aty - hei;
	for(i=0; i<hei; i++)
	{
		if (pos+i < ly || pos+i > hy) continue;
		ptr = rowstart[i];
		dest = wf->rowstart[pos+i];
		for(j=startx; j<endx; j++)
		{
			if (ptr[j] == 0) continue;
			dpos = atx + j;
			dest[dpos] = (dest[dpos] & mask) | col;
		}
	}

	/* set redraw area */
	left = atx;     right = left + wid;
	bottom = aty;   top = bottom - hei;
	gra_setrect(wf, left, right, top, bottom);
}

/*
 * Routine to convert the string "msg" (to be drawn into window "win") into an
 * array of pixels.  The size of the array is returned in "wid" and "hei", and
 * the pixels are returned in an array of character vectors "rowstart".
 * The routine returns nonzero if the operation cannot be done.
 */
INTSML gettextbits(WINDOWPART *win, char *msg, INTSML *wid, INTSML *hei, char ***rowstart)
{
	REGISTER INTSML i, len, j;
	CSize textSize;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *ptr;

	/* copy the string and correct it */
	len = strlen(msg);
	if (len == 0)
	{
		*wid = *hei = 0;
		return(0);
	}
	for(i=0; i<len; i++)
	{
		gra_localstring[i] = msg[i] & 0177;
		if (gra_localstring[i] < ' ') gra_localstring[i] = ' ';
	}
	gra_localstring[i] = 0;

	/* determine size of string */
	wf = win->frame;
	textSize = ((CDC *)wf->hDCOff)->GetTextExtent((LPCTSTR)gra_localstring, len);
	*wid = (INTSML)textSize.cx;
	*hei = (INTSML)textSize.cy;

	/* see if text buffer needs to be (re)initialized */
	if (gra_textbufinited != 0)
	{
		if (*wid > gra_textbufwid || *hei > gra_textbufhei)
		{
			efree((char *)gra_textrowstart);
			efree((char *)gra_textbitmapinfo);
			DeleteObject(gra_textbitmap);
			delete gra_texthdc;
			gra_textbufinited = 0;
		}
	}

	/* allocate text buffer if needed */
	if (gra_textbufinited == 0)
	{
		if (gra_buildoffscreenbuffer(wf, *wid, *hei, &gra_texthdc, &gra_textbitmap,
			&gra_textbitmapinfo, &gra_textdatabuffer, &gra_textrowstart) != 0) return(1);
		gra_texthdc->SetTextColor(PALETTEINDEX(1));
		gra_texthdc->SetROP2(R2_COPYPEN);
		gra_texthdc->SetBkMode(TRANSPARENT);
		gra_texthdc->SetTextAlign(TA_TOP<<8);

		/* remember information about text buffer */
		gra_textbufwid = *wid;   gra_textbufhei = *hei;
		gra_textbufinited = 1;
	}

	/* clear the text buffer */
	for(i=0; i < *hei; i++)
	{
		ptr = gra_textrowstart[i];
		for(j=0; j < *wid; j++) *ptr++ = 0;
	}

	/* write to the text buffer */
	gra_texthdc->SelectObject((CFont *)wf->hFont);
	gra_texthdc->TextOut(0, 0, (LPCTSTR)gra_localstring, len);

	*rowstart = gra_textrowstart;
	return(0);
}

/******************** CIRCLES ********************/

void screendrawcircle(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	REGISTER INTSML col, mask, top, bottom, left, right;
	REGISTER INTBIG x, y, d, maxx, maxy, thisx, thisy;
	REGISTER WINDOWFRAME *wf;
	REGISTER char *thisrow;

	/* get parameters */
	wf = win->frame;
	col = desc->col;   mask = ~desc->bits;
	aty = wf->revy - aty;

	/* set redraw area */
	left = (INTSML)(atx - radius);
	right = (INTSML)(atx + radius + 1);
	top = (INTSML)(aty - radius);
	bottom = (INTSML)(aty + radius + 1);
	gra_setrect(wf, left, right, top, bottom);

	maxx = wf->swid;
	maxy = wf->shei;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	if (left >= 0 && right < maxx && top >= 0 && bottom < maxy)
	{
		/* no clip version is faster */
		while (x <= y)
		{
			thisrow = wf->rowstart[aty + y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = wf->rowstart[aty - y];
			thisrow[atx + x] = (thisrow[atx + x] & mask) | col;
			thisrow[atx - x] = (thisrow[atx - x] & mask) | col;

			thisrow = wf->rowstart[aty + x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			thisrow = wf->rowstart[aty - x];
			thisrow[atx + y] = (thisrow[atx + y] & mask) | col;
			thisrow[atx - y] = (thisrow[atx - y] & mask) | col;

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	} else
	{
		/* clip version */
		while (x <= y)
		{
			thisy = aty + y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - y;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - x;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty + x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			thisy = aty - x;
			if (thisy >= 0 && thisy < maxy)
			{
				thisrow = wf->rowstart[thisy];
				thisx = atx + y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
				thisx = atx - y;
				if (thisx >= 0 && thisx < maxx) thisrow[thisx] = (thisrow[thisx] & mask) | col;
			}

			if (d < 0) d += 4*x + 6; else
			{
				d += 4 * (x-y) + 10;
				y--;
			}
			x++;
		}
	}
}

/*
 * routine to draw a filled-in circle of radius "radius"
 */
void gra_drawdiscrow(WINDOWFRAME *wf, INTBIG thisy, INTBIG startx, INTBIG endx, GRAPHICS *desc)
{
	REGISTER char *thisrow;
	REGISTER INTBIG x;
	REGISTER INTSML pat;

	if (thisy < 0 || thisy >= gra_curvemaxy) return;
	thisrow = wf->rowstart[thisy];
	if (startx < 0) startx = 0;
	if (endx >= gra_curvemaxx) endx = gra_curvemaxy - 1;
	if (gra_curvestyle == PATTERNED)
	{
		pat = desc->raster[thisy&7];
		if (pat != 0)
		{
			for(x=startx; x<=endx; x++)
				if ((pat & (1 << (15-(x&15)))) != 0)
					thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
		}
	} else
	{
		for(x=startx; x<=endx; x++)
			thisrow[x] = (thisrow[x] & gra_curvemask) | gra_curvecol;
	}
}

/*
 * routine to draw a filled-in circle of radius "radius"
 */
void screendrawdisc(WINDOWPART *win, INTBIG atx, INTBIG aty, INTBIG radius, GRAPHICS *desc)
{
	REGISTER WINDOWFRAME *wf;
	REGISTER INTBIG x, y, d;
	REGISTER INTSML top, bottom, left, right;

	/* get parameters */
	wf = win->frame;
	gra_curvestyle = desc->colstyle & NATURE;
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;

	/* set redraw area */
	aty = wf->revy - aty;
	left = (INTSML)(atx - radius);
	right = (INTSML)(atx + radius + 1);
	top = (INTSML)(aty - radius);
	bottom = (INTSML)(aty + radius + 1);
	gra_setrect(wf, left, right, top, bottom);

	gra_curvemaxx = wf->swid;
	gra_curvemaxy = wf->shei;
	x = 0;   y = radius;
	d = 3 - 2 * radius;
	while (x <= y)
	{
		gra_drawdiscrow(wf, aty+y, atx-x, atx+x, desc);
		gra_drawdiscrow(wf, aty-y, atx-x, atx+x, desc);
		gra_drawdiscrow(wf, aty+x, atx-y, atx+y, desc);
		gra_drawdiscrow(wf, aty-x, atx-y, atx+y, desc);

		if (d < 0) d += 4*x + 6; else
		{
			d += 4 * (x-y) + 10;
			y--;
		}
		x++;
	}
}

/******************** ARC DRAWING ********************/

INTSML gra_arcfindoctant(INTBIG x, INTBIG y)
{
	if (x > 0)
		if (y >= 0)
			if (y >= x)	 return 7;
			else         return 8;
		else
			if (x >= -y) return 1;
			else         return 2;
	else
		if (y > 0)
			if (y > -x)  return 6;
			else         return 5;
		else
			if (y > x)   return 4;
			else         return 3;
}

void gra_arcxformoctant(INTBIG x, INTBIG y, INTBIG oct, INTBIG *ox, INTBIG *oy)
{
	switch (oct)
	{
		case 1 : *ox = -y;   *oy = x;   break;
		case 2 : *ox = x;    *oy = -y;  break;
		case 3 : *ox = -x;   *oy = -y;  break;
		case 4 : *ox = -y;   *oy = -x;  break;
		case 5 : *ox = y;    *oy = -x;  break;
		case 6 : *ox = -x;   *oy = y;   break;
		case 7 : *ox = x;    *oy = y;   break;
		case 8 : *ox = y;    *oy = x;   break;
	}
}

void gra_arcdopixel(INTBIG x, INTBIG y)
{
	if (x < 0 || x >= gra_curvemaxx || y < 0 || y >= gra_curvemaxy) return;
	if (gra_arcfirst != 0)
	{
		gra_arcfirst = 0;
		gra_arclx = gra_archx = x;
		gra_arcly = gra_archy = y;
	} else
	{
		if (x < gra_arclx) gra_arclx = x;
		if (x > gra_archx) gra_archx = x;
		if (y < gra_arcly) gra_arcly = y;
		if (y > gra_archy) gra_archy = y;
	}
	gra_arcframe->rowstart[y][x] = (gra_arcframe->rowstart[y][x] & gra_curvemask) | gra_curvecol;
}

void gra_arcoutxform(INTBIG x, INTBIG y)
{
	if (gra_arcocttable[1]) gra_arcdopixel( y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[2]) gra_arcdopixel( x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[3]) gra_arcdopixel(-x + gra_arccenterx, -y + gra_arccentery);
	if (gra_arcocttable[4]) gra_arcdopixel(-y + gra_arccenterx, -x + gra_arccentery);
	if (gra_arcocttable[5]) gra_arcdopixel(-y + gra_arccenterx,  x + gra_arccentery);
	if (gra_arcocttable[6]) gra_arcdopixel(-x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[7]) gra_arcdopixel( x + gra_arccenterx,  y + gra_arccentery);
	if (gra_arcocttable[8]) gra_arcdopixel( y + gra_arccenterx,  x + gra_arccentery);
}

void gra_arcbrescw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < x1 && y > y1)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
	}

	/* get to the end */
	for ( ; x < x1; x++) gra_arcoutxform(x, y);
	for ( ; y > y1; y--) gra_arcoutxform(x, y);
   gra_arcoutxform(x1, y1);
}

void gra_arcbresmidcw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 - 2 * y + 4 * x;
	while (x < y)
	{
		gra_arcoutxform(x, y);
		if (d < 0) d += 4*x+6; else
		{
			d += 4*(x-y)+10;
			y--;
		}
		x++;
   }
   if (x == y) gra_arcoutxform(x, y);
}

void gra_arcbresmidccw(INTBIG x, INTBIG y)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y - 4 * x;
	while (x > 0)
	{
		gra_arcoutxform(x, y);
		if (d > 0) d += 6-4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
   }
   gra_arcoutxform(0, gra_arcradius);
}

void gra_arcbresccw(INTBIG x, INTBIG y, INTBIG x1, INTBIG y1)
{
	REGISTER INTBIG d;

	d = 3 + 2 * y + 4 * x;
	while(x > x1 && y < y1)
	{
		/* not always correct */
		gra_arcoutxform(x, y);
		if (d > 0) d += 6 - 4*x; else
		{
			d += 4*(y-x)+10;
			y++;
		}
		x--;
	}

	/* get to the end */
	for ( ; x > x1; x--) gra_arcoutxform(x, y);
	for ( ; y < y1; y++) gra_arcoutxform(x, y);
	gra_arcoutxform(x1, y1);
}

/*
 * draws an arc centered at (centerx, centery), clockwise,
 * passing by (x1,y1) and (x2,y2)
 */
void screendrawcirclearc(WINDOWPART *win, INTBIG centerx, INTBIG centery, INTBIG p1_x, INTBIG p1_y,
	INTBIG p2_x, INTBIG p2_y, GRAPHICS *desc)
{
	REGISTER INTBIG alternate, pa_x, pa_y, pb_x, pb_y, i, diff;
	INTBIG x, y;
	REGISTER INTSML start_oct, end_oct;

	/* ignore tiny arcs */
	if (p1_x == p2_x && p1_y == p2_y) return;

	/* get parameters */
	gra_arcframe = win->frame;
	gra_curvecol = desc->col;   gra_curvemask = ~desc->bits;
	gra_curvemaxx = gra_arcframe->swid;
	gra_curvemaxy = gra_arcframe->shei;
	p1_y = gra_arcframe->revy - p1_y;
	p2_y = gra_arcframe->revy - p2_y;
	gra_arccentery = gra_arcframe->revy - centery;
	gra_arccenterx = centerx;
	pa_x = p2_x - gra_arccenterx;
	pa_y = p2_y - gra_arccentery;
	pb_x = p1_x - gra_arccenterx;
	pb_y = p1_y - gra_arccentery;
	gra_arcradius = computedistance(gra_arccenterx, gra_arccentery, p2_x, p2_y);
	alternate = computedistance(gra_arccenterx, gra_arccentery, p1_x, p1_y);
	start_oct = gra_arcfindoctant(pa_x, pa_y);
	end_oct   = gra_arcfindoctant(pb_x, pb_y);
	gra_arcfirst = 1;

	/* move the point */
	if (gra_arcradius != alternate)
	{
		diff = gra_arcradius-alternate;
		switch (end_oct)
		{
			case 6:
			case 7: /*  y >  x */ pb_y += diff;  break;
			case 8: /*  x >  y */
			case 1: /*  x > -y */ pb_x += diff;  break;
			case 2: /* -y >  x */
			case 3: /* -y > -x */ pb_y -= diff;  break;
			case 4: /* -y < -x */
			case 5: /*  y < -x */ pb_x -= diff;  break;
		}
	}

	for(i=1; i<9; i++) gra_arcocttable[i] = 0;

	if (start_oct == end_oct)
	{
		INTBIG x1, y1, x2, y2;

		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x1, &y1);
		gra_arcxformoctant(pb_x, pb_y, start_oct, &x2 ,&y2);

		if (ODD(start_oct)) gra_arcbrescw(x1, y1, x2, y2);
		else				gra_arcbresccw(x1 ,y1, x2, y2);
		gra_arcocttable[start_oct] = 0;
	} else
	{
		gra_arcocttable[start_oct] = 1;
		gra_arcxformoctant(pa_x, pa_y, start_oct, &x, &y);
		if (ODD(start_oct)) gra_arcbresmidcw(x, y);
		else				gra_arcbresmidccw(x, y);
		gra_arcocttable[start_oct] = 0;

		gra_arcocttable[end_oct] = 1;
		gra_arcxformoctant(pb_x, pb_y, end_oct, &x, &y);
		if (ODD(end_oct)) gra_arcbresmidccw(x, y);
		else			  gra_arcbresmidcw(x, y);
		gra_arcocttable[end_oct] = 0;

		if (MODP(start_oct+1) != end_oct)
		{
			if (MODP(start_oct+1) == MODM(end_oct-1))
				gra_arcocttable[MODP(start_oct+1)] = 1; else
					for(i = MODP(start_oct+1); i != end_oct; i = MODP(i+1))
						gra_arcocttable[i] = 1;
			gra_arcbresmidcw(0, gra_arcradius);
		}
	}

	/* set redraw area */
	if (gra_arcfirst == 0)
		gra_setrect(gra_arcframe, (INTSML)(gra_arclx), (INTSML)(gra_archx+1), (INTSML)(gra_arcly),
			(INTSML)(gra_archy+1));
}

/******************** GRID CONTROL ********************/

/*
 * grid drawing routine
 */
void screendrawgrid(WINDOWPART *win, POLYGON *obj)
{
	REGISTER INTBIG i, j, xnum, xden, ynum, yden, x0,y0, x1,y1, x2,y2, x3,y3,
		x4,y4, x5,y5, x10, y10, y10mod, xspacing, yspacing, y1base, x1base;
	REGISTER INTSML x, y, fatdots;
	REGISTER WINDOWFRAME *wf;
	REGISTER VARIABLE *var;

	wf = win->frame;
	x0 = obj->xv[0];   y0 = obj->yv[0];		/* screen space grid spacing */
	x1 = obj->xv[1];   y1 = obj->yv[1];		/* screen space grid start */
	x2 = obj->xv[2];   y2 = obj->yv[2];		/* display space low */
	x3 = obj->xv[3];   y3 = obj->yv[3];		/* display space high */
	x4 = obj->xv[4];   y4 = obj->yv[4];		/* screen space low */
	x5 = obj->xv[5];   y5 = obj->yv[5];		/* screen space high */

	var = getvalkey((INTBIG)us_tool, VTOOL, -1, us_gridboldspacing);
	if (var == NOVARIABLE) xspacing = yspacing = 10; else
	{
		if ((var->type&VISARRAY) == 0)
			xspacing = yspacing = var->addr; else
		{
			xspacing = ((INTBIG *)var->addr)[0];
			yspacing = ((INTBIG *)var->addr)[1];
		}
	}

	xnum = x3 - x2;
	xden = x5 - x4;
	ynum = y3 - y2;
	yden = y5 - y4;
	x10 = x0*xspacing;       y10 = y0*yspacing;
	y1base = y1 - (y1 / y0 * y0);
	x1base = x1 - (x1 / x0 * x0);

	/* adjust grid placement according to scale */
	fatdots = 0;
	if (muldiv(x0, xnum, xden) < 5 || muldiv(y0, ynum, yden) < 5)
	{
		x1 = x1base - (x1base - x1) / x10 * x10;   x0 = x10;
		y1 = y1base - (y1base - y1) / y10 * y10;   y0 = y10;
	} else if (muldiv(x0, xnum, xden) > 75 && muldiv(y0, ynum, yden) > 75)
	{
		fatdots = 1;
	}

	/* draw the grid to the offscreen buffer */
	for(i = y1; i < y5; i += y0)
	{
		y = (INTSML)(muldiv(i-y4, ynum, yden) + y2);
		if (y < y2 || y > y3) continue;
		y = wf->revy - y;
		y10mod = (i-y1base) % y10;
		for(j = x1; j < x5; j += x0)
		{
			x = (INTSML)(muldiv(j-x4, xnum, xden) + x2);
			if (x >= x2 && x <= x3) wf->rowstart[y][x] |= GRID;

			/* special case every "spacing" grid points in each direction */
			if (fatdots != 0 || ((j-x1base)%x10) == 0 && y10mod == 0)
			{
				if (x > x2) wf->rowstart[y][x-1] |= GRID;
				if (x < x3) wf->rowstart[y][x+1] |= GRID;
				if (y > y2) wf->rowstart[y-1][x] |= GRID;
				if (y < y3) wf->rowstart[y+1][x] |= GRID;
				if (fatdots != 0 && ((j-x1base)%x10) == 0 && y10mod == 0)
				{
					if (x-1 > x2) wf->rowstart[y][x-2] |= GRID;
					if (x+1 < x3) wf->rowstart[y][x+2] |= GRID;
					if (y-1 > y2) wf->rowstart[y-2][x] |= GRID;
					if (y+1 < y3) wf->rowstart[y+2][x] |= GRID;
					if (x > x2 && y > y2) wf->rowstart[y-1][x-1] |= GRID;
					if (x > x2 && y < y3) wf->rowstart[y+1][x-1] |= GRID;
					if (x < x3 && y > y2) wf->rowstart[y-1][x+1] |= GRID;
					if (x < x3 && y < y3) wf->rowstart[y+1][x+1] |= GRID;
				}
			}
		}
	}

	/* copy it back to the screen */
	gra_setrect(wf, (INTSML)x2, (INTSML)x3, (INTSML)(wf->revy-y3), (INTSML)(wf->revy-y2));
}

/******************** MOUSE CONTROL ********************/

/*
 * routine to return the number of buttons on the mouse
 */
INTSML buttoncount(void)
{
	return((INTSML)mini(BUTTONS, NUMBUTS));
}

/*
 * routine to tell whether button "but" is a double-click
 */
INTSML doublebutton(INTSML b)
{
	if (b >= BUTTONS - REALBUTS) return(1);
	return(0);
}

/*
 * routine to tell whether button "but" has the "shift" key held
 */
INTSML shiftbutton(INTSML b)
{
	b = b / REALBUTS;

	/* this "switch" statement is taken from the array "gra_buttonname" */
	switch (b)
	{
		case 0: return(0);	/* no shift keys */
		case 1: return(1);	/* shift */
		case 2: return(0);	/* control */
		case 3: return(0);	/* alt */
		case 4: return(1);	/* shift-control */
		case 5: return(1);	/* shift-alt */
		case 6: return(0);	/* control-alt */
		case 7: return(1);	/* shift-control-alt */
		case 8: return(0);	/* double-click */
	}
	return(0);
}

/*
 * routine to tell whether button "but" is a "mouse wheel" button
 */
INTSML wheelbutton(INTSML b)
{
	b = b % REALBUTS;
	if (b == 3 || b == 4) return(1);
	return(0);
}

/*
 * routine to return the name of button "b" (from 0 to "buttoncount()").
 * The number of letters unique to the button is placed in "important".
 */
char *buttonname(INTSML b, INTSML *important)
{
	*important = gra_buttonname[b].unique;
	return(gra_buttonname[b].name);
}

/*
 * routine to convert from "gra_inputstate" (the typical input parameter)
 * to button numbers (the table "gra_buttonname")
 */
INTSML gra_makebutton(INTBIG state)
{
	REGISTER INTSML base;

	switch (state&WHICHBUTTON)
	{
		case ISLEFT:    base = 0;   break;
		case ISMIDDLE:  base = 1;   break;
		case ISRIGHT:   base = 2;   break;
		case ISWHLFWD:  base = 3;   break;
		case ISWHLBKW:  base = 4;   break;
	}

	if ((state&DOUBLECLICK) != 0) return(base+REALBUTS*8);
	switch (state & (SHIFTDOWN|CONTROLDOWN|ALTDOWN))
	{
		case SHIFTDOWN                    : return(base+REALBUTS*1);
		case           CONTROLDOWN        : return(base+REALBUTS*2);
		case                       ALTDOWN: return(base+REALBUTS*3);
		case SHIFTDOWN|CONTROLDOWN        : return(base+REALBUTS*4);
		case SHIFTDOWN|            ALTDOWN: return(base+REALBUTS*5);
		case           CONTROLDOWN|ALTDOWN: return(base+REALBUTS*6);
		case SHIFTDOWN|CONTROLDOWN|ALTDOWN: return(base+REALBUTS*7);
	}
	return(base);
}

/*
 * routine to wait for a button push and return its index (0 based) in "*but".
 * The coordinates of the cursor are placed in "*x" and "*y".  If there is no
 * button push, the value of "*but" is negative.
 */
void waitforbutton(INTSML *x, INTSML *y, INTSML *but)
{
	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = (INTSML)gra_cursorx;
		*y = (INTSML)gra_cursory;
		gra_inputstate = NOEVENT;
		if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(us_normalcursor);
		return;
	}
	gra_nextevent();
	if (gra_inputstate != NOEVENT && (gra_inputstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
	{
		*but = gra_makebutton(gra_inputstate);
		*x = (INTSML)gra_cursorx;
		*y = (INTSML)gra_cursory;
		gra_inputstate = NOEVENT;
		if (us_cursorstate != IBEAMCURSOR) setdefaultcursortype(us_normalcursor);
		return;
	}
	*but = -1;
}

/*
 * routine to track the cursor until a button is released, calling "whileup" for
 * each co-ordinate when the mouse moves before the first button push, calling
 * "whendown" once when the button goes down, calling "eachdown" for each
 * co-ordinate when the mouse moves after the button is pushed, calling
 * "eachchar" for each key that is typed at any time, and calling "done" once
 * when done.  The "whendown" and "done" routines are called with no parameters;
 * "whileup" and "eachdown" are called with the X and Y coordinates of the
 * cursor; and "eachchar" is called with the X, Y, and character value that was
 * typed.  The "whileup", "eachdown", and "eachchar" routines return nonzero to
 * abort tracking.
 * If "waitforpush" is nonzero then the routine will wait for a button to
 * actually be pushed before tracking (otherwise it will begin tracking
 * immediately).  The value of "purpose" determines what the cursor will look
 * like during dragging: 0 for normal (the standard cursor), 1 for drawing (a pen),
 * 2 for dragging (a hand), 3 for popup menu selection (a horizontal arrow), 4 for
 * hierarchical popup menu selection (arrow, stays at end).
 */
void trackcursor(INTSML waitforpush, INTSML (*whileup)(INTBIG, INTBIG),
	void (*whendown)(void), INTSML (*eachdown)(INTBIG, INTBIG),
	INTSML (*eachchar)(INTBIG, INTBIG, INTSML), void (*done)(void), INTSML purpose)
{
	REGISTER INTSML keepon, oldcursor;
	INTBIG action;

	/* change the cursor to an appropriate icon */
	oldcursor = us_normalcursor;
	switch (purpose)
	{
		case TRACKDRAWING:
			setnormalcursor(PENCURSOR);
			break;
		case TRACKDRAGGING:
			setnormalcursor(HANDCURSOR);
			break;
		case TRACKSELECTING:
		case TRACKHSELECTING:
			setnormalcursor(MENUCURSOR);
			break;
	}

	/* now wait for a button to go down, if requested */
	keepon = 0;
	if (waitforpush != 0)
	{
		while (keepon == 0)
		{
			gra_nextevent();
			if (gra_inputstate == NOEVENT) continue;
			action = gra_inputstate;
			gra_inputstate = NOEVENT;

			/* if button just went down, stop this loop */
			if ((action&ISBUTTON) != 0 && (action&BUTTONUP) == 0) break;
			if ((action&MOTION) != 0)
			{
				keepon = (*whileup)(gra_cursorx, gra_cursory);
			} else if ((action&CHARREAD) != 0)
			{
				keepon = (*eachchar)(gra_cursorx, gra_cursory, (INTSML)(action&CHARREAD));
			}
			if (el_pleasestop != 0) keepon = 1;
		}
	}

	/* button is now down, real tracking begins */
	if (keepon == 0)
	{
		(*whendown)();
		keepon = (*eachdown)(gra_cursorx, gra_cursory);
	}

	/* now track while the button is down */
	while (keepon == 0)
	{
		gra_nextevent();

		/* for each motion, report the coordinates */
		if (gra_inputstate == NOEVENT) continue;
		action = gra_inputstate;
		gra_inputstate = NOEVENT;
		if ((action&ISBUTTON) != 0 && (action&BUTTONUP) != 0) break;

		if ((action&MOTION) != 0)
		{
			keepon = (*eachdown)(gra_cursorx, gra_cursory);
		} else if ((action&CHARREAD) != 0)
		{
			keepon = (*eachchar)(gra_cursorx, gra_cursory, (INTSML)(action&CHARREAD));
		}
		if (el_pleasestop != 0) keepon = 1;
	}

	/* inform the user that all is done */
	(*done)();

	/* restore the state of the world */
	setnormalcursor(oldcursor);
}

/*
 * routine to read the current co-ordinates of the tablet and return them
 * in "*x" and "*y".
 */
void readtablet(INTSML *x, INTSML *y)
{
	*x = (INTSML)gra_cursorx;
	*y = (INTSML)gra_cursory;
}

/*
 * routine to turn off the cursor tracking if it is on
 */
void stoptablet(void)
{
	if (us_cursorstate != IBEAMCURSOR)
		setdefaultcursortype(us_normalcursor);
}

/******************** KEYBOARD CONTROL ********************/

/*
 * routine to get the next character from the keyboard
 */
INTSML getnxtchar(void)
{
	REGISTER INTSML i;

	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0)
	{
		i = gra_inputstate & CHARREAD;
		if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
		gra_inputstate = NOEVENT;
		return(i);
	}
	if (us_cursorstate != IBEAMCURSOR)
		setdefaultcursortype(WANTTTYCURSOR);
	for(;;)
	{
		gra_nextevent();
		if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0)
			break;
	}
	i = gra_inputstate & CHARREAD;
	if ((gra_inputstate&COMMANDDOWN) != 0) i |= 0200;
	gra_inputstate = NOEVENT;

	if (us_cursorstate != IBEAMCURSOR)
		setdefaultcursortype(us_normalcursor);
	return(i);
}

void checkforinterrupt(void)
{
	MSG msg;

	while ( ::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
	{
		if (!theApp.PumpMessage())
		{
			::PostQuitMessage(0);
			break;
		}
	}
	setdefaultcursortype(WAITCURSOR);
}

/*
 * routine to tell whether data is waiting at the terminal.  Returns nonzero
 * if data is ready.
 */
INTSML ttydataready(void)
{
	/* see if something is already pending */
	if (gra_inputstate != NOEVENT)
	{
		if ((gra_inputstate&CHARREAD) != 0) return(1);
	}

	/* wait for something and analyze it */
	gra_nextevent();
	if (gra_inputstate != NOEVENT && (gra_inputstate&CHARREAD) != 0) return(1);
	return(0);
}

/****************************** FILES ******************************/

/*
 * Routine to prompt for multiple files of type "filetype", giving the
 * message "msg".  Returns a string that contains all of the file names,
 * separated by the NONFILECH (a character that cannot be in a file name).
 */
#define MAXMULTIFILECHARS 4000
char *multifileselectin(char *msg, INTBIG filetype)
{
	char fs[MAXMULTIFILECHARS], prompt[256], *extension, *winfilter, *shortname,
		*longname, *pt;
	CFileDialog *fileDlg;
	REGISTER INTSML which;
	REGISTER INTBIG ret;
	INTBIG mactype, binary;

	/* build filter string */
	describefiletype(filetype, &extension, &winfilter, &mactype, &binary, &shortname, &longname);
	strcpy(fs, longname);
	strcat(fs, " (");
	strcat(fs, winfilter);
	strcat(fs, ")|");
	strcat(fs, winfilter);
	strcat(fs, _("|All Files (*.*)|*.*||"));

	fileDlg = new CFileDialog(TRUE, NULL, NULL, NULL, fs, NULL);
	if (fileDlg == NULL) return(0);
	fs[0] = 0;
	fileDlg->m_ofn.lpstrFile = fs;
	fileDlg->m_ofn.nMaxFile = MAXMULTIFILECHARS;
	fileDlg->m_ofn.Flags |= OFN_HIDEREADONLY | OFN_FILEMUSTEXIST | OFN_ALLOWMULTISELECT |
		OFN_EXPLORER;
	sprintf(prompt, _("%s Selection"), msg);
	fileDlg->m_ofn.lpstrTitle = prompt;
	pt = 0;
	ret = fileDlg->DoModal();
	if (ret == IDOK)
	{
		pt = fs;
		(void)initinfstr();
		which = 0;
		while (*pt != 0)
		{
			if (which == 0)
			{
				strcpy(gra_localstring, pt);
			} else
			{
				if (which > 1) (void)addtoinfstr(NONFILECH);
				(void)addstringtoinfstr(gra_localstring);
				(void)addtoinfstr(DIRSEP);
				(void)addstringtoinfstr(pt);
			}
			which++;
			while (*pt != 0) pt++;
			pt++;
		}
		if (which == 1) (void)addstringtoinfstr(gra_localstring);
		pt = returninfstr();
	}
	delete fileDlg;
	return(pt);
}

/*
 * Routine to display a standard file prompt dialog and return the selected file.
 * The prompt message is in "msg" and the kind of file is in "filetype".  The default
 * output file name is in "defofile" (only used if "filetype" is negative).
 */
char *fileselect(char *msg, INTBIG filetype, char *defofile)
{
	REGISTER INTSML i;
	char ofile[256], fs[256], prompt[256], *extension, *winfilter, *shortname,
		*longname;
	CFileDialog *fileDlg;
	INTBIG mactype, binary;

	/* build filter string */
	describefiletype(filetype, &extension, &winfilter, &mactype, &binary, &shortname, &longname);
	strcpy(fs, longname);
	strcat(fs, " (");
	strcat(fs, winfilter);
	strcat(fs, ")|");
	strcat(fs, winfilter);
	strcat(fs, "|");
	strcat(fs, _("All Files"));
	strcat(fs, " (*.*)|*.*||");

	if (us_logplay != NULL)
	{
		if (gra_loggetnextaction(gra_localstring) != 0) return(0);
		if (gra_action.kind != FILEREPLY) gra_localstring[0] = 0;
	} else
	{
		gra_localstring[0] = 0;
		if ((filetype & FILETYPEWRITE) == 0)
		{
			/* open file dialog */
			fileDlg = new CFileDialog(TRUE, NULL, NULL, NULL, fs, NULL);
			if (fileDlg == NULL) gra_localstring[0] = 0; else
			{
				fileDlg->m_ofn.Flags |= OFN_HIDEREADONLY | OFN_FILEMUSTEXIST;
				sprintf(prompt, _("%s Selection"), msg);
				fileDlg->m_ofn.lpstrTitle = prompt;
				if (fileDlg->DoModal() == IDOK)
					strcpy(gra_localstring, fileDlg->GetPathName());
				delete fileDlg;
			}
		} else
		{
			/* save file dialog */
			for(i = strlen(defofile)-1; i > 0; i--)
				if (defofile[i] == ':' || defofile[i] == '/' || defofile[i] == '\\') break;
			if (i > 0) i++;
			(void)strcpy(ofile, &defofile[i]);
			fileDlg = new CFileDialog(FALSE, extension, ofile, NULL, fs, NULL);
			if (fileDlg != NULL)
			{
				fileDlg->m_ofn.Flags |= OFN_OVERWRITEPROMPT;
				sprintf(prompt, _("%s Creation"), msg);
				fileDlg->m_ofn.lpstrTitle = prompt;
				if (fileDlg->DoModal() == IDOK)
					strcpy(gra_localstring, fileDlg->GetPathName());
				delete fileDlg;
			}
		}
	}

	/* log this result */
	gra_logwriteaction(FILEREPLY, 0, 0, gra_localstring);

	if (gra_localstring[0] == 0) return((char *)NULL); else
		return(gra_localstring);
}

/*
 * Helper routine to initialize the list of files in a directory.
 */
void gra_initfilelist(void)
{
	if (gra_fileliststringarray == 0)
	{
		gra_fileliststringarray = newstringarray(db_cluster);
		if (gra_fileliststringarray == 0) return;
	}
	clearstrings(gra_fileliststringarray);
}

/*
 * Helper routine to add "file" to the list of files in a directory.
 * Returns nonzero on error.
 */
INTSML gra_addfiletolist(char *file)
{
	addtostringarray(gra_fileliststringarray, file);
	return(0);
}

/*
 * Routine to search for all of the files/directories in directory "directory" and
 * return them in the array of strings "filelist".  Returns the number of files found.
 */
INTSML filesindirectory(char *directory, char ***filelist)
{
	HANDLE handle;
	WIN32_FIND_DATA data;
	BOOL found;
	char *dir;
	INTBIG len;

	/* search for all files in directory */
	(void)initinfstr();
	(void)addstringtoinfstr(directory);
	(void)addstringtoinfstr("*.*");
	dir = returninfstr();

	handle = FindFirstFile(dir, &data);
	if (handle == INVALID_HANDLE_VALUE) return(0);

	gra_initfilelist();
	for (found = 1; found; found = FindNextFile(handle, &data))
	{
		if (gra_addfiletolist(data.cFileName) != 0) return(0);
	}
	FindClose(handle);

	*filelist = getstringarray(gra_fileliststringarray, &len);
	return((INTSML)len);
}

/* routine to convert a path name with "~" to a real path */
char *truepath(char *line)
{
	/* only have tilde parsing on UNIX */
	return(line);
}

/*
 * Routine to return the full path to file "file".
 */
char *fullfilename(char *file)
{
	static char fullfile[MAXPATHLEN];

	/* is this a hack?  Full path defined when it starts with "LETTER:" */
	if (isalpha(file[0]) != 0 && file[1] == ':') return(file);

	/* build the proper path */
	strcpy(fullfile, currentdirectory());
	strcat(fullfile, file);
	return(fullfile);
}

/*
 * Routine to turn the file name "file" into a unique one by changing the
 * "XXX" part to a number that makes the file unique.
 */
void emaketemp(char *file)
{
	mktemp(file);
}

/*
 * routine to rename file "file" to "newfile"
 * returns nonzero on error
 */
INTBIG erename(char *file, char *newfile)
{
	return(rename(file, newfile));
}

/*
 * routine to delete file "file"
 */
INTBIG eunlink(char *file)
{
	return(_unlink(file));
}

/*
 * Routine to return information about the file or directory "name":
 *  0: does not exist
 *  1: is a file
 *  2: is a directory
 */
INTSML fileexistence(char *name)
{
	struct stat buf;

	if (stat(name, &buf) < 0) return(0);
	if ((buf.st_mode & S_IFMT) == S_IFDIR) return(2);
	return(1);
}

/*
 * Routine to create a directory.
 * Returns nonzero on error.
 */
INTSML createdirectory(char *dirname)
{
	if (mkdir(dirname) == 0) return(0);
	return(1);
}

/*
 * Routine to return the current directory name
 */
char *currentdirectory(void)
{
	static char line[MAXPATHLEN+1];
	REGISTER INTBIG len;

	getcwd(line, MAXPATHLEN);
	len = strlen(line);
	if (line[len-1] != DIRSEP)
	{
		line[len++] = DIRSEP;
		line[len] = 0;
	}
	return(line);
}

/*
 * Routine to return the home directory (returns 0 if it doesn't exist)
 */
char *hashomedir(void)
{
	return(0);
}

/*
 * Routine to return the path to the "options" library.
 */
char *optionsfilepath(void)
{
	char username[256];
	UINTBIG size;

	(void)initinfstr();
	(void)addstringtoinfstr(el_libdir);
	(void)addstringtoinfstr("electricoptions_");
	size = 256;
	GetUserName(username, &size);
	(void)addstringtoinfstr(username);
	(void)addstringtoinfstr(".elib");
	return(returninfstr());
}

UINTBIG filedate(char *filename)
{
	struct stat buf;

	stat(filename, &buf);
	buf.st_mtime -= machinetimeoffset();
	return(buf.st_mtime);
}

/*
 * Routine to lock a resource called "lockfilename" by creating such a file
 * if it doesn't exist.  Returns nonzero if successful, zero if unable to
 * lock the file.
 */
INTSML lockfile(char *lockfilename)
{
	int fd;

	fd = creat(lockfilename, 0);
	if (fd == -1) return(0);
	if (close(fd) == -1) return(0);
	return(1);
}

/*
 * Routine to unlock a resource called "lockfilename" by deleting such a file.
 */
void unlockfile(char *lockfilename)
{
	INTBIG attrs;
	attrs = GetFileAttributes(lockfilename);
	if ((attrs & FILE_ATTRIBUTE_READONLY) != 0)
	{
		if (SetFileAttributes(lockfilename, attrs & ~FILE_ATTRIBUTE_READONLY) == 0)
		{
			ttyputerr(_("Error removing readonly bit on %s (error %d)"),
				lockfilename, GetLastError());
		}
	}

	if (DeleteFile(lockfilename) == 0)
		ttyputerr(_("Error unlocking %s (error %d)"), lockfilename, GetLastError());
}

/*
 * Routine to show file "filename" in a browser window.
 * Returns nonzero if the operation cannot be done.
 */
INTSML browsefile(char *filename)
{
	CMainFrame *wnd;
	INTBIG err;

	wnd = (CMainFrame *)AfxGetMainWnd();
	err = (INTBIG)ShellExecute(wnd->m_hWnd, "open", filename, "",
		"C:\\Temp\\", SW_SHOWNORMAL);
	if (err > 32) return(0);
	ttyputmsg(_("Could not browse %s (error %ld)"), filename, err);
	return(0);
}

/****************************** CHANNELS ******************************/

/*
 * routine to create a pipe connection between the channels in "channels"
 */
INTBIG epipe(int channels[2])
{
	return(0);
}

/*
 * Routine to set channel "channel" into an appropriate mode for single-character
 * interaction (i.e. break mode).
 */
void setinteractivemode(int channel)
{
}

/*
 * Routine to replace channel "channel" with a pointer to file "file".
 * Returns a pointer to the original channel.
 */
INTBIG channelreplacewithfile(int channel, char *file)
{
	return(0);
}

/*
 * Routine to replace channel "channel" with new channel "newchannel".
 * Returns a pointer to the original channel.
 */
INTBIG channelreplacewithchannel(int channel, int newchannel)
{
	return(0);
}

/*
 * Routine to restore channel "channel" to the pointer that was returned
 * by "channelreplacewithfile" or "channelreplacewithchannel" (and is in "saved").
 */
void channelrestore(int channel, INTBIG saved)
{
}

/*
 * Routine to read "count" bytes from channel "channel" into "addr".
 * Returns the number of bytes read.
 */
INTBIG eread(int channel, char *addr, INTBIG count)
{
	return(0);
}

/*
 * Routine to write "count" bytes to channel "channel" from "addr".
 * Returns the number of bytes written.
 */
INTBIG ewrite(int channel, char *addr, INTBIG count)
{
	return(0);
}

/*
 * routine to close a channel in "channel"
 */
INTBIG eclose(int channel)
{
	return(0);
}

/*************************** TIME ROUTINES ***************************/

UINTBIG machinetimeoffset(void)
{
	return(0);
}

/* returns the current time in 60ths of a second */
UINTBIG ticktime(void)
{
	UINTBIG msTime;

	/* convert milliseconds to ticks */
	msTime = GetTickCount();
	return(msTime*6/100);
}

/* returns the double-click interval in 60ths of a second */
INTBIG doubleclicktime(void)
{
	INTBIG dct;

	dct = GetDoubleClickTime();
	dct = dct * 60 / 1000;
	return(dct);
}

/*
 * Routine to wait "ticks" sixtieths of a second and then return.
 */
void gotosleep(INTBIG ticks)
{
	Sleep(ticks * 100 / 6);
}

/*
 * Routine to start counting time.
 */
void starttimer(void)
{
	struct timeb timeptr;

	ftime(&timeptr);
	gra_timebasesec = timeptr.time;
	gra_timebasems = timeptr.millitm;
}

/*
 * Routine to stop counting time and return the number of elapsed seconds
 * since the last call to "starttimer()".
 */
float endtimer(void)
{
	float seconds;
	INTBIG milliseconds;
	struct timeb timeptr;

	ftime(&timeptr);
	milliseconds = (timeptr.time - gra_timebasesec) * 1000 + (timeptr.millitm-gra_timebasems);
	seconds = ((float)milliseconds) / 1000.0f;
	return(seconds);
}

/*************************** EVENT ROUTINES ***************************/

#define INITIALTIMERDELAY 8

INTBIG gra_lasteventstate = 0;
INTBIG gra_lasteventx;
INTBIG gra_lasteventy;
INTBIG gra_lasteventtime;
INTBIG gra_cureventtime = 0;

void gra_timerticked(void)
{
	gra_cureventtime++;
	if (gra_cureventtime > gra_lasteventtime+INITIALTIMERDELAY)
	{
		/* see if the last event was a mouse button going down */
		if ((gra_lasteventstate&(ISBUTTON|BUTTONUP)) == ISBUTTON)
		{
			/* turn it into a null motion */
			gra_addeventtoqueue((gra_lasteventstate & ~ISBUTTON) | MOTION,
				gra_lasteventx, gra_lasteventy);
			gra_lasteventtime -= INITIALTIMERDELAY;
			return;
		}

		/* see if the last event was a motion with button down */
		if ((gra_lasteventstate&(MOTION|BUTTONUP)) == MOTION)
		{
			/* repeat the last event */
			gra_addeventtoqueue(gra_lasteventstate, gra_lasteventx, gra_lasteventy);
			gra_lasteventtime -= INITIALTIMERDELAY;
			return;
		}
	}
}

void gra_addeventtoqueue(INTBIG state, INTBIG x, INTBIG y)
{
	MYEVENTQUEUE *next, *prev;

	if (us_logplay != NULL) return;

	next = gra_eventqueuetail + 1;
	if (next >= &gra_eventqueue[EVENTQUEUESIZE])
		next = gra_eventqueue;
	if (next == gra_eventqueuehead)
	{
		/* queue is full: see if last event was repeated */
		if (gra_eventqueuetail == gra_eventqueue)
			prev = &gra_eventqueue[EVENTQUEUESIZE-1]; else
				prev = gra_eventqueuetail - 1;
		if (prev->inputstate == state)
		{
			prev->cursorx = x;
			prev->cursory = y;
			return;
		}
		MessageBeep(MB_ICONASTERISK);
		return;
	}

	gra_lasteventstate = gra_eventqueuetail->inputstate = state;
	gra_lasteventx = gra_eventqueuetail->cursorx = x;
	gra_lasteventy = gra_eventqueuetail->cursory = y;
	gra_lasteventtime = gra_cureventtime;
	gra_eventqueuetail = next;
}

/*
 * Routine to get the next Electric input action and set the global "gra_inputstate"
 * accordingly.
 */
void gra_nextevent(void)
{
	MSG msg;
	REGISTER INTBIG windowindex, i, j, trueItem, saveslice;
	REGISTER INTSML x, y;
	POPUPMENU *pm;
	REGISTER POPUPMENUITEM *mi;
	extern INTBIG el_inslice;

	flushscreen();

	/* process events */
	while ( ::PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE))
	{
		if (!theApp.PumpMessage())
		{
			::PostQuitMessage(0);
			break;
		}
	}

	if (us_logplay != NULL)
	{
		/* playing back log file: get event from disk */
		gra_logreadaction();
	} else
	{
		/* normal interaction: get next event */
		gra_inputstate = NOEVENT;
		while (gra_eventqueuehead != gra_eventqueuetail)
		{
			gra_inputstate = gra_eventqueuehead->inputstate;
			gra_cursorx = gra_eventqueuehead->cursorx;
			gra_cursory = gra_eventqueuehead->cursory;
			us_state &= ~GOTXY;
			gra_eventqueuehead++;
			if (gra_eventqueuehead >= &gra_eventqueue[EVENTQUEUESIZE])
				gra_eventqueuehead = gra_eventqueue;

			/* stop if this is the last event in the queue */
			if (gra_eventqueuehead == gra_eventqueuetail) break;

			/* stop if this and the next event are not motion */
			if ((gra_inputstate&MOTION) == 0) break;
			if ((gra_eventqueuehead->inputstate&MOTION) == 0) break;
		}
	}

	/* record valid events */
	if (gra_inputstate != NOEVENT)
	{
		if (gra_inputstate == WINDOWSIZE || gra_inputstate == WINDOWMOVE)
		{
			windowindex = gra_cursorx;
			x = (INTSML)(gra_cursory >> 16);
			y = (INTSML)gra_cursory;
			gra_logwriteaction(gra_inputstate, x, y, (void *)windowindex);
			gra_inputstate = NOEVENT;
		} else
		{
			windowindex = -1;
			if (el_curwindowframe != NOWINDOWFRAME) windowindex = el_curwindowframe->windindex;
			gra_logwriteaction(gra_inputstate, (INTSML)gra_cursorx, (INTSML)gra_cursory,
				(void *)windowindex);
		}

		/* handle menu events */
		if (gra_inputstate == MENUEVENT)
		{
			gra_inputstate = NOEVENT;
			i = gra_cursorx;
			if (i >= 0 && i < gra_pulldownmenucount)
			{
				pm = gra_pulldowns[i];
				for(trueItem=j=0; j < pm->total; j++)
				{
					mi = &pm->list[j];
					if (mi->response->active < 0 && *mi->attribute == 0) continue;
					trueItem++;
					if (trueItem != gra_cursory) continue;
					us_state |= DIDINPUT;
					us_state &= ~GOTXY;
					setdefaultcursortype(us_normalcursor);

					/* special case for "system" commands": don't erase */
					us_forceeditchanges();
					saveslice = el_inslice;
					el_inslice = 1;
					us_execute(mi->response, us_tool->toolstate&ECHOBIND, 1, 1);
					el_inslice = saveslice;
					db_setcurrenttool(us_tool);
					setactivity(mi->attribute);
					break;
				}
			}
		}
	}
}

void gra_buttonaction(int state, UINT nFlags, CPoint point, CWnd *frm)
{
	WINDOWFRAME *wf;
	INTBIG event;
	INTBIG x, y, item;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if ((CChildFrame *)wf->wndframe == frm) break;
	if (wf == NOWINDOWFRAME)
	{
		if (gra_curdialogindex < 0 ||
			frm != gra_dialogs[gra_curdialogindex].window) return;
		item = gra_dodialogisinsideuserdrawn(point.x, point.y);
		if (item == 0) return;
		if (state != 1)
		{
			gra_itemclicked(item-1+ID_DIALOGITEM_0);
			return;
		}
	}

	/* set appropriate button */
	if ((nFlags & MK_LBUTTON) != 0) event = ISBUTTON|ISLEFT;
	if ((nFlags & MK_MBUTTON) != 0) event = ISBUTTON|ISMIDDLE;
	if ((nFlags & MK_RBUTTON) != 0) event = ISBUTTON|ISRIGHT;

	/* add in extras */
	if ((nFlags & MK_SHIFT) != 0) event |= SHIFTDOWN;
	if ((nFlags & MK_CONTROL) != 0) event |= CONTROLDOWN;
	if ((GetKeyState(VK_MENU)&0x8000) != 0) event |= ALTDOWN;
	if (state == 2 && (event&(SHIFTDOWN|CONTROLDOWN|ALTDOWN)) == 0)
		event |= DOUBLECLICK;
	if (state == 1)
	{
		event |= BUTTONUP;
	} else
	{
		if (us_logplay != NULL)
		{
			ttyputerr(_("Session playback aborted by mouse click"));
			xclose(us_logplay);
			us_logplay = NULL;
			return;
		}
	}

	x = point.x;
	if (wf == NOWINDOWFRAME) y = point.y; else
		y = wf->revy - point.y;
	us_state |= DIDINPUT;
	gra_addeventtoqueue(event, x, y);
}

void gra_mouseaction(UINT nFlags, CPoint point, CWnd *frm)
{
	WINDOWFRAME *wf;
	INTBIG event, x, y, inmenu;
	REGISTER VARIABLE *var;
	char *str;
	static INTSML overrodestatus = 0;
	COMMANDBINDING commandbinding;

	/* find the appropriate window */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if ((CChildFrame *)wf->wndframe == frm) break;
	inmenu = 0;
	if (wf != NOWINDOWFRAME)
	{
		/* report the menu if over one */
		if ((nFlags & (MK_LBUTTON|MK_MBUTTON|MK_RBUTTON)) == 0 && wf->floating != 0)
		{
			gra_cursorx = (INTSML)point.x;
			gra_cursory = (INTSML)(wf->revy - point.y);
			us_state &= ~GOTXY;
			if (us_menuxsz > 0 && us_menuysz > 0)
			{
				x = (gra_cursorx-us_menulx) / us_menuxsz;
				y = (gra_cursory-us_menuly) / us_menuysz;
				if (x >= 0 && y >= 0 && x < us_menux && y < us_menuy)
				{
					var = getvalkey((INTBIG)us_tool, VTOOL, VSTRING|VISARRAY, us_binding_menu);
					if (var != NOVARIABLE)
					{
						if (us_menupos <= 1) str = ((char **)var->addr)[y * us_menux + x]; else
							str = ((char **)var->addr)[x * us_menuy + y];
						us_parsebinding(str, &commandbinding);
						if (*commandbinding.command != 0)
						{
							if (commandbinding.nodeglyph != NONODEPROTO)
							{
								ttysetstatusfield(NOWINDOWFRAME, us_statusarc, describearcproto(us_curarcproto), 1);
								ttysetstatusfield(NOWINDOWFRAME, us_statusnode, us_describemenunode(&commandbinding), 1);
								overrodestatus = 1;
								inmenu = 1;
							}
							if (commandbinding.arcglyph != NOARCPROTO)
							{
								ttysetstatusfield(NOWINDOWFRAME, us_statusarc, describearcproto(commandbinding.arcglyph), 1);
								if (us_curnodeproto == NONODEPROTO) str = ""; else
									str = describenodeproto(us_curnodeproto);
								ttysetstatusfield(NOWINDOWFRAME, us_statusnode, str, 1);
								overrodestatus = 1;
								inmenu = 1;
							}
						}
						us_freebindingparse(&commandbinding);
					}
				}
			}
		}
	}
	if (inmenu == 0 && overrodestatus != 0)
	{
		ttysetstatusfield(NOWINDOWFRAME, us_statusarc, describearcproto(us_curarcproto), 1);
		if (us_curnodeproto == NONODEPROTO) str = ""; else
			str = describenodeproto(us_curnodeproto);
		ttysetstatusfield(NOWINDOWFRAME, us_statusnode, str, 1);
		overrodestatus = 0;
	}

	if (gra_inputstate != NOEVENT) return;

	if (wf == NOWINDOWFRAME)
	{
		if (gra_curdialogindex < 0 ||
			frm != gra_dialogs[gra_curdialogindex].window) return;
		if (gra_dodialogisinsideuserdrawn(point.x, point.y) == 0) return;
	}

	x = (INTSML)point.x;
	if (wf == NOWINDOWFRAME) y = (INTSML)point.y; else
		y = (INTSML)(wf->revy - point.y);
	event = MOTION;

	if ((nFlags & (MK_LBUTTON|MK_MBUTTON|MK_RBUTTON)) == 0)
		event |= BUTTONUP;
	gra_addeventtoqueue(event, x, y);
}

int gra_setpropercursor(CWnd *frm, int x, int y)
{
	WINDOWFRAME *wf;
	INTBIG lx, hx, ly, hy;
	REGISTER WINDOWPART *w;
	REGISTER EDITOR *e;

	/* find the appropriate window */
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if ((CChildFrame *)wf->wndframe == frm) break;
	if (wf != NOWINDOWFRAME)
	{
		/* if the window is known, set the cursor appropriately */
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if (w->frame != wf) continue;
			y = wf->revy - y;

			/* see if the cursor is over a window partition separator */
			us_gettruewindowbounds(w, &lx, &hx, &ly, &hy);
			if (x >= lx-1 && x <= lx+1 && y > ly+1 && y < hy-1 && us_hasotherwindowpart(lx-10, y, w) != 0)
			{
				setdefaultcursortype(LRCURSOR);
				return(1);
			} else if (x >= hx-1 && x <= hx+1 && y > ly+1 && y < hy-1 && us_hasotherwindowpart(hx+10, y, w) != 0)
			{
				setdefaultcursortype(LRCURSOR);
				return(1);
			} else if (y >= ly-1 && y <= ly+1 && x > lx+1 && x < hx-1 && us_hasotherwindowpart(x, ly-10, w) != 0)
			{
				setdefaultcursortype(UDCURSOR);
				return(1);
			} else if (y >= hy-1 && y <= hy+1 && x > lx+1 && x < hx-1 && us_hasotherwindowpart(x, hy+10, w) != 0)
			{
				setdefaultcursortype(UDCURSOR);
				return(1);
			}
			if (x < w->uselx || x > w->usehx || y < w->usely || y > w->usehy) continue;
			if ((w->state&WINDOWTYPE) == POPTEXTWINDOW ||
				(w->state&WINDOWTYPE) == TEXTWINDOW)
			{
				e = w->editor;
				if ((e->state&EDITORTYPE) == PACEDITOR)
				{
					if (x <= w->usehx - SBARWIDTH && y >= w->usely + SBARWIDTH && y < e->revy)
					{
						setdefaultcursortype(IBEAMCURSOR);
						return(1);
					}
				}
			}
			break;
		}
	}
	setdefaultcursortype(us_normalcursor);
	return(0);
}

void gra_mousewheelaction(UINT nFlags, short zDelta, CPoint point, CWnd *frm)
{
	WINDOWFRAME *wf;
	INTBIG event;
	INTBIG x, y;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
		if ((CChildFrame *)wf->wndframe == frm) break;
	if (wf == NOWINDOWFRAME) return;

	/* set appropriate "button" */
	if (zDelta > 0) event = ISBUTTON|ISWHLFWD; else
		event = ISBUTTON|ISWHLBKW;

	/* add in extras */
	if ((nFlags & MK_SHIFT) != 0) event |= SHIFTDOWN;
	if ((nFlags & MK_CONTROL) != 0) event |= CONTROLDOWN;
	if ((GetKeyState(VK_MENU)&0x8000) != 0) event |= ALTDOWN;

	x = point.x;
	if (wf == NOWINDOWFRAME) y = point.y; else
		y = wf->revy - point.y;
	us_state |= DIDINPUT;
	gra_addeventtoqueue(event, x, y);
}

void gra_keyaction(UINT nChar, UINT nFlags, UINT nRepCnt)
{
	POINT pt, p2;
	CWnd *wnd;
	INTBIG event, x, y;

	/* determine corner of window */
	if (el_curwindowframe == NOWINDOWFRAME) wnd = AfxGetMainWnd(); else
		wnd = (CWnd *)el_curwindowframe->wndframe;
	p2.x = p2.y = 0;
	wnd->MapWindowPoints(0, &p2, 1);

	/* determine current cursor coordinates */
	GetCursorPos(&pt);
	x = pt.x - p2.x;
	y = pt.y - p2.y;
	if (el_curwindowframe != NOWINDOWFRAME)
		y = el_curwindowframe->revy - y;

	/* queue the event */
	if (nChar == 015) nChar = 012;
	event = nChar & CHARREAD;
	if ((nChar&0200) != 0) event |= COMMANDDOWN;
	us_state |= DIDINPUT;
	gra_addeventtoqueue(event, x, y);
}

void gra_setdefaultcursor(void)
{
	int curstate;

	curstate = us_cursorstate;
	us_cursorstate++;
	setdefaultcursortype(curstate);
}

extern "C" { extern INTSML db_broadcasting; }

void gra_activateframe(CChildFrame *frame, BOOL bActivate)
{
	REGISTER WINDOWFRAME *wf;
	REGISTER WINDOWPART *w;

	if (!bActivate)
	{
		if (frame == gra_messageswindow) gra_messagescurrent = 0; else
			el_curwindowframe = NOWINDOWFRAME;
		return;
	}

	if (frame == gra_messageswindow) gra_messagescurrent = 1; else
	{
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
 			if ((CChildFrame *)wf->wndframe == frame) break;
		if (wf != NOWINDOWFRAME)
		{
			el_curwindowframe = wf;
			gra_cureditwindowframe = wf;

			/* see if the change of window frame invalidates the current window */
			if (el_curwindowpart == NOWINDOWPART || el_curwindowpart->frame != wf)
			{
				/* must choose new window (if not broadcasting) */
				if (db_broadcasting == 0)
				{
					for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
					{
						if (w->frame == wf)
						{
							(void)setvalkey((INTBIG)us_tool, VTOOL, us_current_window,
								(INTBIG)w, VWINDOWPART|VDONTSAVE);
							(void)setval((INTBIG)el_curlib, VLIBRARY, "curnodeproto",
								(INTBIG)w->curnodeproto, VNODEPROTO);
							break;
						}
					}
				}
			}
		}
	}

	/* if another windows was activated, make sure the palette is on top */
	gra_floatpalette();
}

/*
 * Routine to ensure that the floating component menu is on top
 */
void gra_floatpalette(void)
{
	REGISTER WINDOWFRAME *wf;

	if (gra_creatingwindow != 0) return;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (wf->floating == 0) continue;
		((CChildFrame *)wf->wndframe)->SetWindowPos(&CWnd::wndTop, 0, 0, 0, 0,
			SWP_NOMOVE|SWP_NOSIZE|SWP_NOACTIVATE);
		break;
	}
}

int gra_closeframe(CChildFrame *frame)
{
	WINDOWFRAME *wf;
	char *par[2];

	if (frame == gra_messageswindow)
	{
		gra_messagescurrent = 0;
		gra_messageswindow = 0;
		return(1);
	}

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (frame == (CChildFrame *)wf->wndframe) break;
	}
	if (wf == NOWINDOWFRAME) return(0);

//	((CChildFrame *)wf->wndframe)->SetActiveWindow();
	par[0] = "delete";
	us_window(1, par);
	return(0);
}

int gra_closeworld(void)
{
	if (us_preventloss(NOLIBRARY, "quit", 1)) return(1);  /* keep working */
	bringdown();
	return(0);
}

void gra_repaint(CChildFrame *frame, CPaintDC *dc)
{
	WINDOWFRAME *wf;

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (frame == (CChildFrame *)wf->wndframe)
		{
			dc->BitBlt(0, 0, wf->swid, wf->shei, (CDC *)wf->hDCOff, 0, 0, SRCCOPY);
			break;
		}
	}
	gra_floatpalette();
}

void gra_resize(CChildFrame *frame, int cx, int cy)
{
	WINDOWFRAME *wf;
	RECT r;
	WINDOWPLACEMENT wp;

	if (frame->IsIconic()) return;

	if (frame == gra_messageswindow)
	{
		gra_messageswindow->GetWindowPlacement(&wp);
		r = wp.rcNormalPosition;
		gra_messagesleft = (INTSML)r.left;
		gra_messagesright = (INTSML)r.right;
		gra_messagestop = (INTSML)r.top;
		gra_messagesbottom = (INTSML)r.bottom;
		return;
	}

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (frame == (CChildFrame *)wf->wndframe)
		{
			sizewindowframe(wf, cx, cy);
			gra_redrawdisplay(wf);
			gra_addeventtoqueue(WINDOWSIZE, wf->windindex, ((cx & 0xFFFF) << 16) | (cy & 0xFFFF));
			break;
		}
	}
	gra_floatpalette();
}

void gra_movedwindow(CChildFrame *frame, int x, int y)
{
	RECT fr, cr, r;
	int framewid, framehei;
	WINDOWPLACEMENT wp;
	WINDOWFRAME *wf;

	if (frame == gra_messageswindow)
	{
		gra_messageswindow->GetWindowPlacement(&wp);
		r = wp.rcNormalPosition;
		gra_messagesleft = (INTSML)r.left;
		gra_messagesright = (INTSML)r.right;
		gra_messagestop = (INTSML)r.top;
		gra_messagesbottom = (INTSML)r.bottom;
		return;
	}

	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		if (frame == (CChildFrame *)wf->wndframe)
		{
			((CChildFrame *)wf->wndframe)->GetWindowPlacement(&wp);
			fr = wp.rcNormalPosition;
			((CChildFrame *)wf->wndframe)->GetClientRect(&cr);
			framewid = (fr.right - fr.left) - (cr.right - cr.left);
			framehei = (fr.bottom - fr.top) - (cr.bottom - cr.top);
			x -= framewid/2;   y -= (framehei-framewid/2);
			gra_addeventtoqueue(WINDOWMOVE, wf->windindex, ((x & 0xFFFF) << 16) | (y & 0xFFFF));
			break;
		}
	}
	gra_floatpalette();
}

void gra_resizemain(int cx, int cy)
{
	gra_redrawstatusindicators();
	if (gra_palettewindowframe != NOWINDOWFRAME)
		us_drawmenu(-1, gra_palettewindowframe);
}

/* handle interrupts */
void gra_onint(void)
{
	el_pleasestop = 1;
	ttyputerr(_("Interrupted..."));
}

/*************************** SESSION LOGGING ROUTINES ***************************/

extern "C" {
/* Session Playback */
DIALOGITEM gra_sesplaydialogitems[] =
{
 /*  1 */ {0, {100,132,124,212}, BUTTON, N_("Yes")},
 /*  2 */ {0, {100,8,124,88}, BUTTON, N_("No")},
 /*  3 */ {0, {8,8,24,232}, MESSAGE, N_("Electric has found a session log file")},
 /*  4 */ {0, {24,8,40,232}, MESSAGE, N_("which may be from a recent crash.")},
 /*  5 */ {0, {52,8,68,232}, MESSAGE, N_("Do you wish to replay this session")},
 /*  6 */ {0, {68,8,84,232}, MESSAGE, N_("and reconstruct the lost work?")}
};
DIALOG gra_sesplaydialog = {{75,75,208,316}, N_("Replay Log?"), 0, 6, gra_sesplaydialogitems};
};

/* special items for the session playback dialog: */
#define DSPL_YES    1		/* Yes (button) */
#define DSPL_NO     2		/* No (button) */

/*
 * routine to create a session logging file
 */
void logstartrecord(void)
{
	unsigned char count;
	REGISTER INTBIG itemhit;
	REGISTER LIBRARY *lib;
	REGISTER WINDOWFRAME *wf;
	REGISTER WINDOWPART *w;
	RECT fr;
	WINDOWPLACEMENT wp;
	REGISTER VARIABLE *var;

	/* if there is already a log file, it may be from a previous crash */
	if (fileexistence(gra_logfile) == 1)
	{
		DiaInitDialog(&gra_sesplaydialog);
		for(;;)
		{
			itemhit = DiaNextHit();
			if (itemhit == DSPL_YES || itemhit == DSPL_NO) break;
		}
		DiaDoneDialog();
		if (itemhit == DSPL_YES)
		{
			if (fileexistence(gra_logfilesave) == 1)
				eunlink(gra_logfilesave);
			rename(gra_logfile, gra_logfilesave);
			(void)logplayback(gra_logfilesave);
		}
	}

	us_logrecord = xcreate(gra_logfile, us_filetypelog, 0, 0);
	if (us_logrecord == 0) return;

	/* log current libraries */
	count = 0;
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		if ((lib->userbits&HIDDENLIBRARY) == 0) count++;
	gra_logwritebyte((char)count);
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		if ((lib->userbits&HIDDENLIBRARY) != 0) continue;
		if ((lib->userbits&READFROMDISK) != 0) gra_logwritebyte(1); else
			gra_logwritebyte(0);
		gra_logwritestring(lib->libname);
		gra_logwritestring(lib->libfile);
	}

	/* log current windows */
	gra_logwriteshort(gra_newwindowoffset);
	count = 0;
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe) count++;
	gra_logwritebyte((char)count);
	for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
	{
		gra_logwriteshort((char)wf->windindex);
		gra_logwritebyte((char)wf->floating);
		((CChildFrame *)wf->wndframe)->GetWindowPlacement(&wp);
		fr = wp.rcNormalPosition;
		gra_logwriteshort((INTSML)fr.left);
		gra_logwriteshort((INTSML)fr.top);
		gra_logwriteshort((INTSML)(fr.right-fr.left));
		gra_logwriteshort((INTSML)(fr.bottom-fr.top));
		count = 0;
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
			if (w->frame == wf) count++;
		gra_logwritebyte((char)count);
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if (w->frame != wf) continue;
			if (w == el_curwindowpart) gra_logwritebyte(1); else
				gra_logwritebyte(0);
			gra_logwriteshort(w->uselx);
			gra_logwriteshort(w->usehx);
			gra_logwriteshort(w->usely);
			gra_logwriteshort(w->usehy);
			gra_logwritelong(w->screenlx);
			gra_logwritelong(w->screenhx);
			gra_logwritelong(w->screenly);
			gra_logwritelong(w->screenhy);
			gra_logwritelong(w->state);
			gra_logwritestring(describenodeproto(w->curnodeproto));
			gra_logwritestring(w->location);
		}
	}
	gra_logwriteshort(gra_windowframeindex);

	/* log current technology (macros store this in %H) */
	var = getval((INTBIG)us_tool, VTOOL, VSTRING, "USER_local_caph");
	if (var != NOVARIABLE)
	{
		/* technology name found in local variable */
		gra_logwritestring((char *)var->addr);
	} else
	{
		/* just write the current technology name */
		gra_logwritestring(el_curtech->techname);
	}
}

/*
 * routine to begin playback of session logging file "file".  The routine
 * returns nonzero if there is an error.
 */
INTSML logplayback(char *file)
{
	char *filename, tempstring[300];
	unsigned char count, wcount, i, j, floating, cur, fromdisk;
	REGISTER WINDOWFRAME *wf;
	RECTAREA r;
	REGISTER FILE *saveio;
	REGISTER WINDOWPART *w, *nextw;
	REGISTER INTSML wid, hei, uselx, usehx, usely, usehy, sindex;
	REGISTER INTBIG screenlx, screenhx, screenly, screenhy, state;
	REGISTER LIBRARY *lib, *firstlib;

	us_logplay = xopen(file, us_filetypelog, "", &filename);
	if (us_logplay == NULL) return(1);
	ttyputmsg(_("Playing log file..."));
	ttyputmsg(_("   Move mouse continuously to advance playback"));
	ttyputmsg(_("   Click mouse to abort playback"));
	gra_playbacklength = filesize(us_logplay);
	gra_playbackposition = 0;

	/* get current libraries */
	count = (unsigned char)gra_logreadbyte();
	firstlib = NOLIBRARY;
	for(i=0; i<count; i++)
	{
		fromdisk = (unsigned char)gra_logreadbyte();
		gra_logreadstring(tempstring);
		gra_logreadstring(gra_localstring);
		lib = getlibrary(tempstring);
		if (lib == NOLIBRARY)
		{
			/* read library file "gra_localstring" */
			lib = newlibrary(tempstring, gra_localstring);
			if (lib == NOLIBRARY) continue;
		}
		if (fromdisk != 0)
		{
			saveio = us_logplay;
			us_logplay = 0;
			(void)asktool(io_tool, "read", (INTBIG)lib, (INTBIG)"binary", 0);
			us_logplay = saveio;
		}
		if (firstlib == NOLIBRARY) firstlib = lib;
	}
	selectlibrary(firstlib);

	/* delete all existing windows */
	for(w = el_topwindowpart; w != NOWINDOWPART; w = nextw)
	{
		nextw = w->nextwindowpart;
		db_retractwindowpart(w);
	}
	el_curwindowpart = NOWINDOWPART;

	/* get current windows */
	gra_newwindowoffset = gra_logreadshort();
	count = (unsigned char)gra_logreadbyte();
	for(i=0; i<count; i++)
	{
		sindex = gra_logreadshort();
		floating = gra_logreadbyte();
		r.left = gra_logreadshort();
		r.top = gra_logreadshort();
		wid = gra_logreadshort();
		hei = gra_logreadshort();
		r.right = r.left + wid;
		r.bottom = r.top + hei;
		if (floating != 0)
		{
			/* get the floating window frame */
			for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
				if (wf->floating != 0) break;
			if (wf == NOWINDOWFRAME) wf = newwindowframe(1, &r);
		} else
		{
			/* create a new window frame */
			wf = newwindowframe(0, &r);
		}
		wf->windindex = sindex;

		wcount = (unsigned char)gra_logreadbyte();
		for(j=0; j<wcount; j++)
		{
			cur = gra_logreadbyte();
			uselx = gra_logreadshort();
			usehx = gra_logreadshort();
			usely = gra_logreadshort();
			usehy = gra_logreadshort();
			screenlx = gra_logreadlong();
			screenhx = gra_logreadlong();
			screenly = gra_logreadlong();
			screenhy = gra_logreadlong();
			state = gra_logreadlong();
			gra_logreadstring(gra_localstring);
			gra_logreadstring(tempstring);
			w = newwindowpart(tempstring, NOWINDOWPART);
			w->buttonhandler = DEFAULTBUTTONHANDLER;
			w->charhandler = DEFAULTCHARHANDLER;
			w->changehandler = DEFAULTCHANGEHANDLER;
			w->termhandler = DEFAULTTERMHANDLER;
			w->redisphandler = DEFAULTREDISPHANDLER;
			w->uselx = uselx;   w->usehx = usehx;
			w->usely = usely;   w->usehy = usehy;
			w->screenlx = screenlx;   w->screenhx = screenhx;
			w->screenly = screenly;   w->screenhy = screenhy;
			computewindowscale(w);
			w->state = state;
			w->curnodeproto = getnodeproto(gra_localstring);
			w->frame = wf;
			if (cur != 0) el_curwindowpart = w;
			us_redisplay(w);
		}
	}
	gra_windowframeindex = gra_logreadshort();

	/* switch to proper technology */
	gra_logreadstring(gra_localstring);
	us_ensurepropertechnology(NONODEPROTO, gra_localstring, 1);

	return(0);
}

/*
 * routine to terminate session logging
 */
void logfinishrecord(void)
{
	if (us_logrecord != NULL)
	{
		xclose(us_logrecord);
		if (fileexistence(gra_logfilesave) == 1)
			eunlink(gra_logfilesave);
		rename(gra_logfile, gra_logfilesave);
	}
	us_logrecord = NULL;
}

/*
 * Routine to log an event (if logging) of type "inputstate".
 * The event has parameters (cursorx,cursory) and "extradata", depending on "inputstate":
 *   WINDOWSIZE:    window "extradata" is now "cursorx" x "cursory"
 *   WINDOWMOVE:    window "extradata" is now at (cursorx, cursory)
 *   MENUEVENT:     selected menu "cursorx", item "cursory"
 *   FILEREPLY:     file selected by standard-file dialog is in "extradata"
 *   DIAITEMCLICK:  dialog item "cursorx" clicked
 *   DIASCROLLSEL:  dialog scroll item "cursorx" set to "cursory" entries in "extradata"
 *   DIAEDITTEXT:   dialog edit item "cursorx" addec char "cursory" and changed to "extradata"
 *   DIAPOPUPSEL:   dialog popup item "cursorx" set to entry "cursory"
 *   DIASETCONTROL: dialog control item "cursorx" changed to "cursory"
 *   DIAENDDIALOG:  dialog terminated
 *   all others:    cursor in (cursorx,cursory) and window index in "extradata"
 */
void gra_logwriteaction(INTBIG inputstate, INTSML cursorx, INTSML cursory, void *extradata)
{
	REGISTER char *filename;

	if (us_logrecord == NULL) return;

	/* load the event structure */
	gra_action.kind = inputstate;
	gra_action.x = cursorx;
	gra_action.y = cursory;
	gra_action.frameindex = (INTSML)extradata;

	/* ignore redundant cursor motion */
	if (inputstate == MOTION || inputstate == (MOTION|BUTTONUP))
	{
		if (inputstate == gra_lastloggedaction && cursorx == gra_lastloggedx &&
			cursory == gra_lastloggedy && gra_action.frameindex == gra_lastloggedindex)
				return;
	}

	if (xfwrite((char *)&gra_action, sizeof (gra_action), 1, us_logrecord) == 0)
	{
		ttyputerr(_("Error writing session log file: recording disabled"));
		logfinishrecord();
		return;
	}

	/* further annotate special types of change */
	if (inputstate == FILEREPLY || inputstate == DIAEDITTEXT)
	{
		filename = (char *)extradata;
		gra_logwritestring(filename);
	} else if (inputstate == DIASCROLLSEL)
	{
		(void)xfwrite((char *)extradata, sizeof(int), cursory, us_logrecord);
	}

	/* flush the log file every so often */
	gra_logrecordcount++;
	if (gra_logrecordcount >= us_logflushfreq)
	{
		gra_logrecordcount = 0;
		xflushbuf(us_logrecord);
	}
}

INTSML gra_loggetnextaction(char *message)
{
	REGISTER INTBIG amtread;

	if (gra_playbackposition + 100 > gra_playbacklength) amtread = 0; else
	{
		amtread = xfread((char *)&gra_action, sizeof (gra_action), 1, us_logplay);
		gra_playbackposition += amtread;
		if (stopping(STOPREASONPLAYBACK)) amtread = 0;
	}
	if (amtread == 0)
	{
		/* stop playback */
		ttyputmsg(_("End of session playback file"));
		xclose(us_logplay);
		us_logplay = NULL;
		return(1);
	}
	if (gra_action.kind == FILEREPLY || gra_action.kind == DIAEDITTEXT)
	{
		if (message == 0)
		{
			ttyputerr(_("Inconsistent log file"));
			return(1);
		}
		gra_logreadstring(message);
	}
	return(0);
}

void gra_logreadaction(void)
{
	REGISTER WINDOWFRAME *wf;
	REGISTER INTBIG i;
	int selintlist[MAXSCROLLMULTISELECT];
	INTBIG sellist[MAXSCROLLMULTISELECT];

	if (gra_loggetnextaction(gra_localstring) != 0) return;

	/* turn it into an event */
	gra_inputstate = NOEVENT;
	if (gra_action.kind == WINDOWSIZE)
	{
		/* window resized */
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
			if (wf->windindex == gra_action.frameindex) break;
		if (wf != NOWINDOWFRAME)
			sizewindowframe(wf, (INTSML)gra_action.x, (INTSML)gra_action.y);
	} else if (gra_action.kind == WINDOWMOVE)
	{
		/* window moved */
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
			if (wf->windindex == gra_action.frameindex) break;
		if (wf != NOWINDOWFRAME)
			movewindowframe(wf, (INTSML)gra_action.x, (INTSML)gra_action.y);
	} else if (gra_action.kind == DIAITEMCLICK)
	{
		/* dialog item clicked */
		gra_dialoghit = gra_action.x;
	} else if (gra_action.kind == DIASCROLLSEL)
	{
		/* dialog scroll item selected */
		(void)xfread((char *)selintlist, sizeof(int), gra_action.y, us_logplay);
		if (gra_action.y == 1)
		{
			DiaSelectLine(gra_action.x, selintlist[0]);
		} else
		{
			/* select multiple lines */
			for(i=0; i<gra_action.y; i++)
				sellist[i] = selintlist[i];
			DiaSelectLines(gra_action.x, gra_action.y, sellist);
		}
	} else if (gra_action.kind == DIAPOPUPSEL)
	{
		/* dialog popup item selected */
		DiaSetPopupEntry(gra_action.x, gra_action.y);
	} else if (gra_action.kind == DIASETCONTROL)
	{
		/* dialog control item changed */
		DiaSetControl(gra_action.x, gra_action.y);
	} else if (gra_action.kind == DIAEDITTEXT)
	{
		/* dialog edit item got text change */
		gra_dialoghit = gra_action.x;
		gra_dialoghitchar = gra_action.y;
		DiaSetText(gra_dialoghit, gra_localstring);
	} else
	{
		/* standard click/key event */
		for(wf = el_firstwindowframe; wf != NOWINDOWFRAME; wf = wf->nextwindowframe)
			if (wf->windindex == gra_action.frameindex) break;
		if (wf != NOWINDOWFRAME && wf != el_curwindowframe)
			((CChildFrame *)wf->wndframe)->ActivateFrame();
		gra_inputstate = gra_action.kind;
		gra_cursorx = gra_action.x;
		gra_cursory = gra_action.y;
		us_state &= ~GOTXY;
	}
}

/*
 * Routine to write "msg" to the log file.
 */
void gra_logwritestring(char *msg)
{
	INTBIG leng;

	leng = strlen(msg);
	gra_logwritebyte((char)leng);
	(void)xfwrite(msg, 1, leng, us_logrecord);
}

/*
 * Routine to read a string from the log file into "msg".
 */
void gra_logreadstring(char *msg)
{
	char len;
	INTBIG amtread;

	len = gra_logreadbyte();
	amtread = xfread(msg, 1, len, us_logplay);
	gra_playbackposition += amtread;
	msg[len] = 0;
}

/*
 * Routine to write a short integer "data" to the log file.
 */
void gra_logwriteshort(INTSML data)
{
	(void)xfwrite((char *)&data, SIZEOFINTSML, 1, us_logrecord);
}

/*
 * Routine to read a short integer from the log file.
 */
INTSML gra_logreadshort(void)
{
	INTBIG amtread;
	INTSML data;

	amtread = xfread((char *)&data, SIZEOFINTSML, 1, us_logplay);
	gra_playbackposition += amtread;
	return(data);
}

/*
 * Routine to write a long integer "data" to the log file.
 */
void gra_logwritelong(INTBIG data)
{
	(void)xfwrite((char *)&data, SIZEOFINTBIG, 1, us_logrecord);
}

/*
 * Routine to read a long integer from the log file.
 */
INTBIG gra_logreadlong(void)
{
	INTBIG amtread;
	INTBIG data;

	amtread = xfread((char *)&data, SIZEOFINTBIG, 1, us_logplay);
	gra_playbackposition += amtread;
	return(data);
}

/*
 * Routine to write a single byte "data" to the log file.
 */
void gra_logwritebyte(char data)
{
	(void)xfwrite((char *)&data, 1, 1, us_logrecord);
}

/*
 * Routine to read a single byte from the log file.
 */
char gra_logreadbyte(void)
{
	INTBIG amtread;
	char data;

	amtread = xfread(&data, 1, 1, us_logplay);
	gra_playbackposition += amtread;
	return(data);
}

/****************************** MENUS ******************************/

void getacceleratorstrings(char **acceleratorstring, char **acceleratorprefix)
{
	*acceleratorstring = _("Ctrl");
	*acceleratorprefix = _("Ctrl-");
}

char *getinterruptkey(void)
{
	return(_("Windows-C"));
}

INTSML nativepopupmenu(POPUPMENU **menu, INTSML header, INTSML left, INTSML top)
{
	CMainFrame *wnd;
	INTBIG j, k, pindex, submenus;
	REGISTER POPUPMENUITEM *mi, *submi;
	REGISTER POPUPMENU *themenu, **subpmlist;
	REGISTER USERCOM *uc;
	HMENU popmenu, subpopmenu, *submenulist;
	POINT p, p2;
	UINT flags;

	themenu = *menu;
	wnd = (CMainFrame *)AfxGetMainWnd();
	flags = TPM_NONOTIFY|TPM_RETURNCMD|TPM_LEFTALIGN|TPM_LEFTBUTTON;
	if (left < 0 || top < 0)
	{
		p2.x = p2.y = 0;
		wnd->MapWindowPoints(0, &p2, 1);
		GetCursorPos(&p);
		left = (INTSML)(p.x - p2.x);
		top = (INTSML)(p.y - p2.y);
		flags = TPM_NONOTIFY|TPM_RETURNCMD|TPM_CENTERALIGN|TPM_VCENTERALIGN|TPM_LEFTBUTTON;
	}
	popmenu = CreatePopupMenu();
	if (popmenu == 0) return(-1);
	if (header != 0)
	{
		if (AppendMenu(popmenu, MF_STRING, 0, themenu->header) == 0) return(-1);
		if (AppendMenu(popmenu, MF_SEPARATOR, 0, 0) == 0) return(-1);
	}

	/* count the number of submenus */
	submenus = 0;
	for(j=0; j < themenu->total; j++)
	{
		mi = &themenu->list[j];
		if (*mi->attribute != 0 && mi->response != NOUSERCOM &&
			mi->response->menu != NOPOPUPMENU) submenus++;
	}
	if (submenus > 0)
	{
		submenulist = (HMENU *)emalloc(submenus * (sizeof (HMENU)), us_tool->cluster);
		if (submenulist == 0) return(-1);
		subpmlist = (POPUPMENU **)emalloc(submenus * (sizeof (POPUPMENU *)), us_tool->cluster);
		if (subpmlist == 0) return(-1);
	}

	/* load the menus */
	submenus = 0;
	for(j=0; j < themenu->total; j++)
	{
		mi = &themenu->list[j];
		mi->changed = 0;
		if (*mi->attribute == 0)
		{
			if (AppendMenu(popmenu, MF_SEPARATOR, 0, 0) == 0) return(-1);
		} else
		{
			uc = mi->response;
			if (uc != NOUSERCOM && uc->menu != NOPOPUPMENU)
			{
				subpopmenu = CreatePopupMenu();
				if (subpopmenu == 0) return(-1);
				submenulist[submenus] = subpopmenu;
				subpmlist[submenus] = uc->menu;
				submenus++;

				for(k=0; k < uc->menu->total; k++)
				{
					submi = &uc->menu->list[k];
					if (*submi->attribute == 0)
					{
						if (AppendMenu(subpopmenu, MF_SEPARATOR, 0, 0) == 0) return(-1);
					} else
					{
						if (AppendMenu(subpopmenu, MF_STRING, (submenus << 16) | (k+1), submi->attribute) == 0) return(-1);
					}
				}
				if (InsertMenu(popmenu, j+2, MF_POPUP | MF_BYPOSITION,
					(DWORD)subpopmenu, mi->attribute) == 0)
						return(0);
			} else
			{
				if (AppendMenu(popmenu, MF_STRING, j+1, mi->attribute) == 0) return(-1);
			}
		}
	}
	pindex = TrackPopupMenu(popmenu, flags, left, top, 0, wnd->m_hWnd, 0) - 1;
	j = pindex >> 16;
	if (j != 0) *menu = subpmlist[j-1];
 	DestroyMenu(popmenu);
	for(j=0; j<submenus; j++)
		DestroyMenu(submenulist[j]);
	if (submenus > 0)
	{
		efree((char *)submenulist);
		efree((char *)subpmlist);
	}
	return(pindex & 0xFFFF);
}

/*
 * routine to handle the menu
 */
void gra_nativemenudoone(INTSML item, INTSML menuindex)
{
	gra_addeventtoqueue(MENUEVENT, menuindex, item);
}

/*
 * Routine to establish the "count" pulldown menu names in "par" as the pulldown menu bar
 */
INTSML nativemenuload(INTSML count, char *par[])
{
	REGISTER INTSML i, menuindex;
	REGISTER POPUPMENU *pm;
	POPUPMENU *pulls[25];

	if (gra_hMenu == 0)
	{
		gra_hMenu = new CMenu();
		gra_hMenu->CreateMenu();
	}

	for(i=0; i<count; i++)
	{
		pm = us_getpopupmenu(par[i]);
		if (pm == NOPOPUPMENU) continue;
		pulls[i] = pm;

		menuindex = gra_pulldownindex(pm);
		if (menuindex < 0) continue;
		if (gra_hMenu->InsertMenu(menuindex, MF_ENABLED | MF_POPUP | MF_BYPOSITION,
			(DWORD)gra_pulldownmenus[menuindex]->m_hMenu, pm->header) == 0)
				return(1);
	}
	CWnd* parent = AfxGetMainWnd();
	CMenu* lastMenu = parent->GetMenu();
	if (parent->SetMenu(gra_hMenu) == 0) return(1);
	gra_hMenu->Detach();
	parent->DrawMenuBar();
	return(0);
}

/*
 * Routine to create a pulldown menu from popup menu "pm".  Returns an index to
 * the table of pulldown menus (-1 on error).
 */
INTSML gra_pulldownindex(POPUPMENU *pm)
{
	REGISTER INTSML i, pindex;
	CMenu **newpulldownmenus;
	POPUPMENU **newpulldowns;

	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm) return(i);

	/* allocate new space with one more */
	newpulldownmenus = (CMenu **)emalloc((gra_pulldownmenucount+1) *
		(sizeof (CMenu *)), us_tool->cluster);
	if (newpulldownmenus == 0) return(-1);
	newpulldowns = (POPUPMENU **)emalloc((gra_pulldownmenucount+1) *
		(sizeof (POPUPMENU *)), us_tool->cluster);
	if (newpulldowns == 0) return(-1);

	/* copy former arrays then delete them */
	for(i=0; i<gra_pulldownmenucount; i++)
	{
		newpulldownmenus[i] = gra_pulldownmenus[i];
		newpulldowns[i] = gra_pulldowns[i];
	}
	if (gra_pulldownmenucount != 0)
	{
		efree((char *)gra_pulldownmenus);
		efree((char *)gra_pulldowns);
	}

	gra_pulldownmenus = newpulldownmenus;
	gra_pulldowns = newpulldowns;

	pindex = gra_pulldownmenucount++;
	gra_pulldownmenus[pindex] = gra_makepdmenu(pm);
	if (gra_pulldownmenus[pindex] == 0)
		return(-1);
	gra_pulldowns[pindex] = pm;
	return(pindex);
}

CMenu *gra_makepdmenu(POPUPMENU *pm)
{
	CMenu *hPrevMenu, *hMenu;
	REGISTER USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;
	REGISTER INTSML j, submenuindex, len, idIndex;
	REGISTER INTBIG flags;
	REGISTER char *pt, slash;
	char keychr, myline[100];

	hMenu = new CMenu();
	if (hMenu->CreatePopupMenu() == 0) return(0);

	idIndex = 0;
	for(j=0; j < pm->total; j++)
	{
		mi = &pm->list[j];
		uc = mi->response;
		if (uc->active < 0)
		{
			if (*mi->attribute == 0)
			{
				if (hMenu->AppendMenu(MF_SEPARATOR) == 0) return(0);
			} else
			{
				if (hMenu->AppendMenu(MF_STRING | MF_DISABLED, gra_menures[idIndex++], mi->attribute) == 0) return(0);
			}
			continue;
		}

		if (uc->menu != NOPOPUPMENU)
		{
			hPrevMenu = hMenu;
			idIndex++;
			submenuindex = gra_pulldownindex(uc->menu);
			if (hPrevMenu->InsertMenu(-1, MF_POPUP | MF_BYPOSITION,
				(DWORD)gra_pulldownmenus[submenuindex]->m_hMenu, mi->attribute) == 0)
					return(0);
			continue;
		}
		flags = 0;
		if (mi->attribute[0] == '>')
		{
			flags = MF_CHECKED;
			strcpy(myline, &mi->attribute[1]);
		} else
		{
			strcpy(myline, mi->attribute);
		}
		len = strlen(myline);
		if (myline[len-1] == '<') myline[len-1] = 0;
		for(pt = myline; *pt != 0; pt++) if (*pt == '/' || *pt == '\\') break;
		if (*pt != 0)
		{
			slash = *pt;
			*pt++ = 0;
			keychr = *pt & 0xFF;
			len = strlen(myline);
			if (keychr >= FUNCTIONF1 && keychr <= FUNCTIONF12)
			{
				sprintf(&myline[len], "\tF%d", keychr-FUNCTIONF1+1);
			} else if (slash == '/')
			{
				sprintf(&myline[len], "\tCtrl-%c", keychr);
			} else
			{
				sprintf(&myline[len], "\t%c", keychr);
			}
		}
		if (hMenu->AppendMenu(MF_STRING|flags, gra_menures[idIndex++], myline) == 0) return(0);
	}
	return(hMenu);
}

/* routine to redraw entry "pindex" of popupmenu "pm" because it changed */
void nativemenurename(POPUPMENU *pm, INTSML pindex)
{
	INTSML i, j, len, key, idIndex;
	REGISTER INTBIG flags;
	char line[100], *pt, slash;
	USERCOM *uc;
	REGISTER POPUPMENUITEM *mi;

	for(i=0; i<gra_pulldownmenucount; i++)
		if (gra_pulldowns[i] == pm)
	{
		idIndex = 0;
		for(j=0; j < pm->total; j++)
		{
			mi = &pm->list[j];
			uc = mi->response;
			if (j == pindex) break;
			if (uc->active < 0 && *mi->attribute == 0) continue;
			idIndex++;
		}

		if (uc->active < 0)
		{
			if (*pm->list[i].attribute == 0)
			{
				gra_pulldownmenus[i]->ModifyMenu(pindex, MF_BYPOSITION|MF_SEPARATOR);
			} else
			{
				gra_pulldownmenus[i]->EnableMenuItem(pindex, MF_BYPOSITION|MF_GRAYED);
			}
		} else
		{
			gra_pulldownmenus[i]->EnableMenuItem(pindex, MF_BYPOSITION|MF_ENABLED);
		}

		/* copy and examine the menu string */
		flags = 0;
		if (pm->list[pindex].attribute[0] == '>')
		{
			flags = MF_CHECKED;
			(void)strcpy(line, &pm->list[pindex].attribute[1]);
		} else
		{
			(void)strcpy(line, pm->list[pindex].attribute);
		}
		len = strlen(line);
		if (line[len-1] == '<') line[len-1] = 0;
		for(pt = line; *pt != 0; pt++) if (*pt == '/' || *pt == '\\') break;

		/* handle single-key equivalents */
		if (*pt != 0)
		{
			slash = *pt;
			*pt++ = 0;
			key = *pt & 0xFF;
			len = strlen(line);
			if (key >= FUNCTIONF1 && key <= FUNCTIONF12)
			{
				sprintf(&line[len], "\tF%d", key-FUNCTIONF1+1);
			} else if (slash == '/')
			{
				sprintf(&line[len], "\tCtrl-%c", key);
			} else
			{
				sprintf(&line[len], "\t%c", key);
			}
		}
		gra_pulldownmenus[i]->ModifyMenu(pindex, MF_BYPOSITION|flags, gra_menures[idIndex], line);
		break;
	}
}

/****************************** DIALOGS ******************************/

/*
 * Routine to initialize a dialog described by "dialog".
 * Returns nonzero if dialog cannot be initialized.
 */
INTSML DiaInitDialog(DIALOG *dialog)
{
	DLGTEMPLATE *dtheader;
	WORD *dtmenu, *dtclass, *dttitle;
	int headerlength, menulength, classlength, titlelength, itemslength,
		totallen;
	INTBIG dbu, dbuL, dbuH;
	DLGITEMTEMPLATE **dtitems;
	RECT r;
	char *descblock, *title, *pt, *msg;
	int *itemlengths, i, j, len, itemtype, style, itemclass;
	WORD output[100];
	CProgressCtrl *prog;
	CListBox *list;
	CStatic *stat;
	CWnd *wnd;

	/* be sure the dialog is translated */
	DiaTranslate(dialog);

	/* get the current dialog structure */
	gra_curdialogindex++;
	gra_curdialog = &gra_dialogs[gra_curdialogindex];

	/*
	 * for small fonts, dbuH = 16, dbuL = 8,
	 * for large fonts, dbuH = 20, dbuL = 10
	 */
	dbu = GetDialogBaseUnits();
	dbuL = dbu & 0xFFFF;
	dbuH = (dbu >> 16) & 0xFFFF;
	gra_dialogdbnx = 2 * (46-dbuL)-1;   gra_dialogdbux = 14 * dbuL;
	gra_dialogdbny = 131 * 8;           gra_dialogdbuy = 105 * dbuH;

	gra_curdialog->window = new CElectricDialog();
	gra_curdialog->itemdesc = dialog;
	gra_curdialog->redrawroutine = 0;
	gra_dialogeditline = -1;

	/* determine item lengths */
	itemlengths = (int *)emalloc(dialog->items * (sizeof (int)), us_tool->cluster);
	if (itemlengths == 0) return(1);
	dtitems = (DLGITEMTEMPLATE **)emalloc(dialog->items * (sizeof (DLGITEMTEMPLATE *)), us_tool->cluster);
	if (dtitems == 0) return(1);
	itemslength = 0;
	for(i=0; i<dialog->items; i++)
	{
		itemlengths[i] = (sizeof DLGITEMTEMPLATE) + 3 * (sizeof (WORD));
		itemtype = dialog->list[i].type;
		switch (itemtype&ITEMTYPE)
		{
			case BUTTON:
			case DEFBUTTON:
			case CHECK:
			case RADIO:
			case EDITTEXT:
			case MESSAGE:
				itemlengths[i] += (strlen(dialog->list[i].msg) + 1) * (sizeof (WORD));
				break;
			default:
				itemlengths[i] += 1 * (sizeof (WORD));
				break;
		}
		itemlengths[i] = (itemlengths[i] + 3) & ~3;
		itemslength += itemlengths[i];
	}

	/* allocate space for entire dialog template */
	headerlength = sizeof (DLGTEMPLATE);
	menulength = sizeof (WORD);
	classlength = sizeof (WORD);
	if (dialog->movable != 0) title = dialog->movable; else
		title = "";
	titlelength = (strlen(title) + 1) * (sizeof (WORD));
	i = headerlength + menulength + classlength + titlelength;
	totallen = (i + 3) & ~3;
	descblock = (char *)emalloc(totallen + itemslength, us_tool->cluster);
	if (descblock == 0) return(1);
	pt = descblock;
	dtheader = (DLGTEMPLATE *)pt;  pt += headerlength;
	dtmenu = (WORD *)pt;           pt += menulength;
	dtclass = (WORD *)pt;          pt += classlength;
	dttitle = (WORD *)pt;          pt += titlelength;
	for(i=0; i<dialog->items; i++)
	{
		pt = (char *)(((DWORD)pt + 3) & ~3);
		dtitems[i] = (DLGITEMTEMPLATE *)pt;
		pt += itemlengths[i];
	}

	/* load dialog template header */
	dtheader->style = WS_VISIBLE | WS_DLGFRAME | WS_POPUP /* | DS_ABSALIGN */;
	if (dialog->movable != 0) dtheader->style |= WS_CAPTION | DS_MODALFRAME;
	dtheader->dwExtendedStyle = 0;
	dtheader->cdit = (unsigned short)dialog->items;
	dtheader->x = dialog->windowRect.left /* * gra_dialogdbnx / gra_dialogdbux */ ;
	dtheader->y = dialog->windowRect.top /* * gra_dialogdbny / gra_dialogdbuy */ ;
	dtheader->cx = (dialog->windowRect.right - dialog->windowRect.left) * gra_dialogdbnx / gra_dialogdbux;
	dtheader->cy = (dialog->windowRect.bottom - dialog->windowRect.top) * gra_dialogdbny / gra_dialogdbuy;

	/* no menu or class in this dialog */
	dtmenu[0] = 0;
	dtclass[0] = 0;

	/* load the dialog title */
	len = MultiByteToWideChar(CP_ACP, 0, title, strlen(title), output, 100);
	for(j=0; j<len; j++) *dttitle++ = output[j];
	*dttitle++ = 0;

	/* find the default button */
	gra_curdialog->defaultbutton = 1;
	for(i=0; i<dialog->items; i++)
	{
		itemtype = dialog->list[i].type;
		if ((itemtype&ITEMTYPE) == DEFBUTTON) gra_curdialog->defaultbutton = i+1;
	}

	/* load the items */
	for(i=0; i<dialog->items; i++)
	{
		dtitems[i]->x = dialog->list[i].r.left * gra_dialogdbnx / gra_dialogdbux;
		dtitems[i]->y = dialog->list[i].r.top * gra_dialogdbny / gra_dialogdbuy;
		dtitems[i]->cx = (dialog->list[i].r.right - dialog->list[i].r.left) * gra_dialogdbnx / gra_dialogdbux;
		dtitems[i]->cy = (dialog->list[i].r.bottom - dialog->list[i].r.top) * gra_dialogdbny / gra_dialogdbuy;
		dtitems[i]->dwExtendedStyle = 0;
		dtitems[i]->id = i+ID_DIALOGITEM_0;
		itemtype = dialog->list[i].type;
		switch (itemtype&ITEMTYPE)
		{
			case BUTTON:
			case DEFBUTTON:
				itemclass = 0x80;
				style = WS_VISIBLE | WS_CHILD | BS_PUSHBUTTON | WS_TABSTOP;
				if (i+1 == gra_curdialog->defaultbutton) style |= BS_DEFPUSHBUTTON;
				break;
			case CHECK:
				itemclass = 0x80;
				style = WS_VISIBLE | WS_CHILD | BS_CHECKBOX | WS_TABSTOP;
				break;
			case RADIO:
				itemclass = 0x80;
				style = WS_VISIBLE | WS_CHILD | BS_RADIOBUTTON | WS_TABSTOP;
				break;
			case EDITTEXT:
				itemclass = 0x81;
				style = WS_CHILD | WS_VISIBLE | WS_BORDER | ES_LEFT | WS_TABSTOP;
				if (dtitems[i]->cy > 10) style |= ES_MULTILINE;
				dtitems[i]->y--;
				dtitems[i]->cy += 2;
				break;
			case MESSAGE:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE | SS_LEFT;
				if ((itemtype&INACTIVE) == 0) style |= SS_NOTIFY;
				dtitems[i]->y--;
				dtitems[i]->cy += 2;
				break;
			case PROGRESS:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE;
				dtitems[i]->id += PROGRESSOFFSET;
				break;
			case USERDRAWN:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE;
				dtitems[i]->x += 1000;
				dtitems[i]->y += 1000;
				break;
			case POPUP:
				itemclass = 0x85;
				dtitems[i]->cy *= 8;
				dtitems[i]->y -= 2;
				style = WS_CHILD | WS_VISIBLE | WS_VSCROLL | CBS_DROPDOWNLIST | WS_TABSTOP;
				break;
			case ICON:
				itemclass = 0x82;
				style = WS_CHILD | WS_VISIBLE;
				dtitems[i]->x += 1000;
				dtitems[i]->y += 1000;
				if ((itemtype&INACTIVE) == 0) style |= SS_NOTIFY;
				break;
			case SCROLL:
			case SCROLLMULTI:
				itemclass = 0x83;
				style = WS_BORDER | WS_CHILD | WS_VISIBLE | LBS_USETABSTOPS |
					WS_VSCROLL | WS_HSCROLL | LBS_WANTKEYBOARDINPUT | LBS_SORT | WS_TABSTOP;
				if ((itemtype&ITEMTYPE) == SCROLLMULTI)
					style |= LBS_EXTENDEDSEL;
				if ((itemtype&INACTIVE) == 0) style |= LBS_NOTIFY;
				break;
			default:
				itemclass = 0x82;
				style = WS_CHILD | SS_LEFT;
				break;
		}
		dtitems[i]->style = style;
		pt = ((char *)dtitems[i]) + (sizeof (DLGITEMTEMPLATE));
		dtclass = (WORD *)pt;
		*dtclass++ = 0xFFFF;
		*dtclass++ = itemclass;
		switch (itemtype&ITEMTYPE)
		{
			case BUTTON:
			case DEFBUTTON:
			case CHECK:
			case RADIO:
			case EDITTEXT:
			case MESSAGE:
				msg = dialog->list[i].msg;
				len = MultiByteToWideChar(CP_ACP, 0, msg, strlen(msg), output, 100);
				for(j=0; j<len; j++) *dtclass++ = output[j];
				*dtclass++ = 0;
				break;
			default:
				*dtclass++ = 0;
				break;
		}
		*dtclass++ = 0;
	}

	/* create the dialog */
	if (gra_curdialog->window->CreateIndirect(dtheader) == 0)
	{
		return(1);
	}

	/* finish initialization */
	wnd = 0;
	for(i=0; i<dialog->items; i++)
	{
		itemtype = dialog->list[i].type;
		switch (itemtype&ITEMTYPE)
		{
			case EDITTEXT:
				if (wnd == 0)
					wnd = gra_curdialog->window->GetDlgItem(i+ID_DIALOGITEM_0);
				allocstring((char **)&dialog->list[i].data, dialog->list[i].msg, el_tempcluster);
				break;
			case PROGRESS:
				stat = (CStatic *)gra_curdialog->window->GetDlgItem(i+ID_DIALOGITEM_0+PROGRESSOFFSET);
				stat->GetWindowRect(&r);
				gra_curdialog->window->ScreenToClient(&r);
				prog = new CProgressCtrl();
				prog->Create(WS_CHILD | WS_VISIBLE, r, gra_curdialog->window, i+ID_DIALOGITEM_0);
				break;
			case SCROLL:
			case SCROLLMULTI:
				list = (CListBox *)gra_curdialog->window->GetDlgItem(i+ID_DIALOGITEM_0);
				if (list == 0) break;
				list->SetTabStops(1);
				break;
		}
	}
	if (wnd == 0)
		wnd = gra_curdialog->window->GetDlgItem(gra_curdialog->defaultbutton-1+ID_DIALOGITEM_0);
	gra_curdialog->window->GotoDlgCtrl(wnd);
	gra_curdialog->window->SetDefID(gra_curdialog->defaultbutton-1+ID_DIALOGITEM_0);

	/* determine location of dialog in screen coordinates */
	gra_curdialog->window->GetWindowRect(&r);
	gra_curdialog->firstpoint.x = r.left;
	gra_curdialog->firstpoint.y = r.top;

	efree((char *)itemlengths);
	efree((char *)dtitems);
	efree((char *)descblock);

	/* disable everything but this */
	if (gra_curdialogindex > 0)
	{
		/* previous dialog already present: disable it */
		gra_dialogs[gra_curdialogindex-1].window->EnableWindow(FALSE);
	} else
	{
		/* no previous dialog: disable main window */
		AfxGetApp()->m_pMainWnd->EnableWindow(FALSE);
	}
	gra_curdialog->window->EnableWindow(TRUE);

	gra_dialoghit = -1;
	return(0);
}

/*
 * Routine to handle actions and return the next item hit.
 */
INTBIG DiaNextHit(void)
{
	INTBIG item;

	for(;;)
	{
		flushscreen();
		gra_nextevent();
		if (gra_dialoghit != -1)
		{
			item = gra_dialoghit;
			gra_dialoghit = -1;
			return(item);
		}
	}
}

/*
 * Routine to parse the next input event and return the next character typed.
 * If the routine returns -1, nothing has happened.  If the
 * routine returns -2, an item has been hit (and is in "itemHit").
 */
INTSML DiaGetNextCharacter(INTBIG *itemHit)
{
	INTSML chr;

	gra_nextevent();
	if (gra_dialoghit == -1) return(-1);
	*itemHit = gra_dialoghit;
	if (gra_dialoghitchar == 0) return(-2);
	chr = gra_dialoghitchar;
	gra_dialoghitchar = 0;
	return(chr);
}

void DiaDoneDialog(void)
{
	REGISTER INTBIG dx, dy, i, itemtype;
	RECT rect;

	/* if playing back, search for end-dialog marker */
	while (us_logplay != NULL)
	{
		gra_logreadaction();
		if (gra_inputstate == DIAENDDIALOG) break;
		gra_inputstate = NOEVENT;
	}
	gra_inputstate = NOEVENT;
	gra_logwriteaction(DIAENDDIALOG, 0, 0, 0);

	/* free memory used by edit text fields */
	for(i=0; i<gra_curdialog->itemdesc->items; i++)
	{
		itemtype = gra_curdialog->itemdesc->list[i].type;
		if ((itemtype&ITEMTYPE) == EDITTEXT)
			efree((char *)gra_curdialog->itemdesc->list[i].data);
	}

	/* update dialog location if it was moved */
	gra_curdialog->window->GetWindowRect(&rect);
	dx = rect.left - gra_curdialog->firstpoint.x;
	dy = rect.top - gra_curdialog->firstpoint.y;
	dx = (dx * gra_dialogdbnx) / gra_dialogdbux;
	dy = (dy * gra_dialogdbny) / gra_dialogdbuy;

	gra_curdialog->itemdesc->windowRect.left += (INTSML)dx;
	gra_curdialog->itemdesc->windowRect.right += (INTSML)dx;
	gra_curdialog->itemdesc->windowRect.top += (INTSML)dy;
	gra_curdialog->itemdesc->windowRect.bottom += (INTSML)dy;

	gra_curdialog->window->EndDialog(0);

	gra_curdialogindex--;
	if (gra_curdialogindex >= 0)
		gra_curdialog = &gra_dialogs[gra_curdialogindex];

	/* reenable previous dialog or world */
	if (gra_curdialogindex >= 0)
	{
		/* previous dialog: reenable it */
		AfxGetApp()->m_pMainWnd->EnableWindow(FALSE);
		gra_curdialog->window->EnableWindow(TRUE);
	} else
	{
		/* no other dialogs: enable world */
		AfxGetApp()->m_pMainWnd->EnableWindow(TRUE);
	}
	gra_dialogeditline = -1;
}

/*
 * Routine to set the text in item "item" to "msg"
 */
void DiaSetText(INTBIG item, char *msg)
{
	INTBIG type, highlight, len;
	REGISTER char *pt;
	CWnd *wnd;
	CEdit *edit;

	highlight = 0;
	if (item < 0)
	{
		item = -item;
		highlight = 1;
	}
	item--;
	for(pt = msg; *pt != 0; pt++) if (strncmp(pt, "(c)", 3) == 0)
	{
		(void)strcpy(pt, "");		/* "copyright" character */
		(void)strcpy(&pt[1], &pt[3]);
		break;
	}
	wnd = gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (wnd == 0) return;
	wnd->SetWindowText(msg);
	type = gra_curdialog->itemdesc->list[item].type;
	if ((type&ITEMTYPE) == EDITTEXT)
	{
		edit = (CEdit *)wnd;
		len = strlen(msg);
		if (highlight != 0) edit->SetSel(0, len); else
			edit->SetSel(len, len);
		reallocstring((char **)&gra_curdialog->itemdesc->list[item].data, msg, el_tempcluster);
	}
}

/*
 * Routine to return the text in item "item"
 */
char *DiaGetText(INTBIG item)
{
	static int bufnum = 0;
	static char line[10][300];
	CWnd *wnd;

	item--;
	wnd = gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (wnd == 0) return("");
	bufnum++;
	if (bufnum >= 10) bufnum = 0;
	wnd->GetWindowText(line[bufnum], 300);
	return(line[bufnum]);
}

/*
 * Routine to set the value in item "item" to "value"
 */
void DiaSetControl(INTBIG item, INTBIG value)
{
	CButton *but;

	item--;
	but = (CButton *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (but == 0) return;
	but->SetCheck(value);
	gra_logwriteaction(DIASETCONTROL, item + 1, (INTSML)value, 0);
}

/*
 * Routine to return the value in item "item"
 */
INTBIG DiaGetControl(INTBIG item)
{
	CButton *but;

	item--;
	but = (CButton *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (but == 0) return(0);
	return(but->GetCheck());
}

/*
 * Routine to check item "item" to make sure that there is
 * text in it.  If so, it returns nonzero.  Otherwise it beeps and returns zero.
 */
INTSML DiaValidEntry(INTBIG item)
{
	char *msg;

	msg = DiaGetText(item);
	while (*msg == ' ') msg++;
	if (*msg != 0) return(1);
	ttybeep();
	return(0);
}

/*
 * Routine to dim item "item"
 */
void DiaDimItem(INTBIG item)
{
	CWnd *wnd, *focus;

	item--;
	wnd = gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	focus = CWnd::GetFocus();
	if (focus == wnd)
		gra_curdialog->window->NextDlgCtrl();
	wnd->EnableWindow(0);
}

/*
 * Routine to un-dim item "item"
 */
void DiaUnDimItem(INTBIG item)
{
	CWnd *wnd;

	item--;
	wnd = gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	wnd->EnableWindow(1);
}

/*
 * Routine to change item "item" to be a message rather
 * than editable text
 */
void DiaNoEditControl(INTBIG item)
{
	DiaDimItem(item);
}

/*
 * Routine to change item "item" to be editable text rather
 * than a message
 */
void DiaEditControl(INTBIG item)
{
	DiaUnDimItem(item);
}

void DiaOpaqueEdit(INTBIG item)
{
	CEdit *edit;

	item--;
	edit = (CEdit *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	edit->ModifyStyle(0, ES_PASSWORD);
	edit->SetPasswordChar('*');
}

/*
 * Routine to cause item "item" to report character hits
 */
void DiaCharacterEdit(INTBIG item)
{
	item--;
	gra_dialogeditline = item;
}

/*
 * Routine to cause item "item" to be the default button
 */
void DiaDefaultButton(INTBIG item)
{
	gra_curdialog->defaultbutton = item;
	gra_curdialog->window->SetDefID(gra_curdialog->defaultbutton-1+ID_DIALOGITEM_0);
}

/*
 * Routine to change item "item" into a popup with "count" entries
 * in "names".
 */
void DiaSetPopup(INTBIG item, INTBIG count, char **names)
{
	INTBIG i;
	CComboBox *cb;

	item--;
	cb = (CComboBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (cb == 0) return;
	cb->ResetContent();
	for(i=0; i<count; i++) cb->AddString(names[i]);
	cb->SetCurSel(0);
}

/*
 * Routine to change popup item "item" so that the current entry is "entry".
 */
void DiaSetPopupEntry(INTBIG item, INTBIG entry)
{
	CComboBox *cb;

	item--;
	cb = (CComboBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (cb == 0) return;
	cb->SetCurSel(entry);
}

/*
 * Routine to return the current item in popup menu item "item".
 */
INTBIG DiaGetPopupEntry(INTBIG item)
{
	CComboBox *cb;

	item--;
	cb = (CComboBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (cb == 0) return(0);
	return(cb->GetCurSel());
}

void DiaInitTextDialog(INTBIG item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTBIG sortpos, INTBIG flags)
{
	long add, remove;
	static CFont *fnt = 0;
	LOGFONT lf;
	CListBox *list;

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;

	add = remove = 0;
	if ((flags&SCHORIZBAR) == 0) remove |= WS_HSCROLL; else
	{
		add |= WS_HSCROLL;
		list->SetHorizontalExtent(1000);
	}
	list->ModifyStyle(remove, add, SWP_NOSIZE);
	gra_curdialog->itemdesc->list[item].data = flags;
	if ((flags&SCFIXEDWIDTH) != 0)
	{
		if (fnt == 0)
		{
			lf.lfHeight = -10;
			strcpy(lf.lfFaceName, "Lucida Console");
			lf.lfWidth = 0;
			lf.lfEscapement = 0;
			lf.lfOrientation = 0;
			lf.lfWeight = FW_NORMAL;
			lf.lfItalic = 0;
			lf.lfUnderline = 0;
			lf.lfStrikeOut = 0;
			lf.lfCharSet = 0;
			lf.lfOutPrecision = OUT_STROKE_PRECIS;
			lf.lfClipPrecision = CLIP_STROKE_PRECIS;
			lf.lfQuality = 1;
			lf.lfPitchAndFamily = FF_DONTCARE | FF_SWISS;

			fnt = new CFont();
			fnt->CreateFontIndirect(&lf);
		}
		list->SetFont(fnt);
	}
	DiaLoadTextDialog(item+1, toplist, nextinlist, donelist, sortpos);
}

void DiaLoadTextDialog(INTBIG item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
	void (*donelist)(void), INTBIG sortpos)
{
	char *next, line[256];
	INTBIG i;
	CListBox *list;

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;

	/* clear the list */
	list->ResetContent();

	/* load the list */
	line[0] = 0;
	next = line;
	(void)(*toplist)(&next);
	for(i=0; ; i++)
	{
		next = (*nextinlist)();
		if (next == 0) break;
		if (sortpos < 0) list->InsertString(-1, next); else
			list->AddString(next);
	}
	(*donelist)();
	if (i > 0) list->SetCurSel(0);
}

/*
 * Routine to stuff line "line" at the end of the edit buffer.
 */
void DiaStuffLine(INTBIG item, char *line)
{
	CListBox *list;

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	list->InsertString(-1, line);
}

/*
 * Routine to select line "line" of scroll item "item".
 */
void DiaSelectLine(INTBIG item, INTBIG line)
{
	CListBox *list;
	INTBIG numitems, botitem, topitem, visibleitems, type;
	CPoint pt;
	BOOL outside;

	item--;
	type = gra_curdialog->itemdesc->list[item].type;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	pt.x = 0;   pt.y = 0;
	topitem = list->ItemFromPoint(pt, outside); 
	pt.x = 0;   pt.y = 9999;
	botitem = list->ItemFromPoint(pt, outside);
	visibleitems = botitem - topitem;
	numitems = list->GetCount();
	if (line < topitem || line >= botitem)
	{
		topitem = line - visibleitems/2;
		list->SetTopIndex(topitem);
	}
	if ((type&ITEMTYPE) == SCROLL)
	{
		list->SetCurSel(line);
	} else
	{
		list->SetSel(-1, FALSE);
		list->SetSel(line, TRUE);
	}
}

/*
 * Routine to select "count" lines in "lines" of scroll item "item".
 */
void DiaSelectLines(INTBIG item, INTBIG count, INTBIG *lines)
{
	CListBox *list;
	INTBIG numitems, botitem, topitem, visibleitems, i, low, high;
	CPoint pt;
	BOOL outside;

	if (count <= 0) return;
	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	pt.x = 0;   pt.y = 0;
	topitem = list->ItemFromPoint(pt, outside); 
	pt.x = 0;   pt.y = 9999;
	botitem = list->ItemFromPoint(pt, outside);
	visibleitems = botitem - topitem;
	numitems = list->GetCount();
	low = high = lines[0];
	for(i=1; i<count; i++)
	{
		if (lines[i] < low) low = lines[i];
		if (lines[i] > high) high = lines[i];
	}
	if (high < topitem || low >= botitem)
	{
		topitem = low;
		list->SetTopIndex(topitem);
	}
	list->SetSel(-1, FALSE);
	for(i=0; i<count; i++)
		list->SetSel(lines[i], TRUE);
}

/*
 * Returns the currently selected line in the scroll list "item".
 */
INTBIG DiaGetCurLine(INTBIG item)
{
	INTBIG line;
	CListBox *list;

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return(-1);
	line = list->GetCurSel();
	if (line == LB_ERR) return(-1);
	return(line);
}

/*
 * Returns the currently selected lines in the scroll list "item".  The returned
 * array is terminated with -1.
 */
INTBIG *DiaGetCurLines(INTBIG item)
{
	REGISTER INTBIG total, i;
	CListBox *list;
	static INTBIG selitemlist[MAXSCROLLMULTISELECT];
	int selintlist[MAXSCROLLMULTISELECT];

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) total = 0; else
	{
		total = list->GetSelItems(MAXSCROLLMULTISELECT-1, selintlist);
		if (total == LB_ERR) total = 0;
		for(i=0; i<total; i++) selitemlist[i] = selintlist[i];
	}
	selitemlist[total] = -1;
	return(selitemlist);
}

char *DiaGetScrollLine(INTBIG item, INTBIG line)
{
	static char text[300];
	CListBox *list;

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return("");
	if (list->GetText(line, text) == -1) return("");
	return(text);
}

void DiaSetScrollLine(INTBIG item, INTBIG line, char *msg)
{
	CListBox *list;

	item--;
	list = (CListBox *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	if (list == 0) return;
	list->DeleteString(line);
	list->InsertString(line, msg);
	list->SetCurSel(line);
}

void DiaItemRect(INTBIG item, RECTAREA *rect)
{
	item--;
	if (item < 0 || item >= gra_curdialog->itemdesc->items) return;
	*rect = gra_curdialog->itemdesc->list[item].r;
}

void DiaPercent(INTBIG item, INTBIG percent)
{
	CProgressCtrl *prog;

	item--;
	prog = (CProgressCtrl *)gra_curdialog->window->GetDlgItem(item+ID_DIALOGITEM_0);
	prog->SetPos(percent);
}

void DiaRedispRoutine(INTBIG item, void (*routine)(RECTAREA*))
{
	gra_curdialog->redrawroutine = routine;
	gra_curdialog->redrawitem = item;
}

void DiaFrameRect(INTBIG item, RECTAREA *ur)
{
	RECT r;
	CDC *dc;

	r.left = ur->left;
	r.right = ur->right-1;
	r.top = ur->top;
	r.bottom = ur->bottom-1;

	if (gra_dialogoffbrush == 0) gra_dialogoffbrush = new CBrush((COLORREF)0xFFFFFF);
	dc = gra_curdialog->window->GetDC();
	dc->FillRect(&r, gra_dialogoffbrush);
	dc->MoveTo(r.left, r.top);
	dc->LineTo(r.right, r.top);
	dc->LineTo(r.right, r.bottom);
	dc->LineTo(r.left, r.bottom);
	dc->LineTo(r.left, r.top);
	gra_curdialog->window->ReleaseDC(dc);
}

void DiaDrawLine(INTBIG item, INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty, INTBIG mode)
{
	CDC *dc;

	dc = gra_curdialog->window->GetDC();
	switch (mode)
	{
		case DLMODEON:     dc->SetROP2(R2_BLACK);   break;
		case DLMODEOFF:    dc->SetROP2(R2_WHITE);   break;
		case DLMODEINVERT: dc->SetROP2(R2_NOT);     break;
	}
	dc->MoveTo(fx, fy);
	dc->LineTo(tx, ty);
	dc->SetROP2(R2_COPYPEN);
	gra_curdialog->window->ReleaseDC(dc);
}

void DiaFillPoly(INTBIG item, INTBIG *x, INTBIG *y, INTBIG count, INTBIG r, INTBIG g, INTBIG b)
{
	CBrush *polybrush;
	CPen *polypen;
	COLORREF brushcolor;
	POINT points[50];
	INTBIG i;
	CDC *dc;

	for(i=0; i<count; i++)
	{
		points[i].x = x[i];
		points[i].y = y[i];
	}

	brushcolor = (COLORREF)((b << 16) | (g << 8) | r);
	polybrush = new CBrush(brushcolor);
	polypen = new CPen(PS_SOLID, 0, brushcolor);
	dc = gra_curdialog->window->GetDC();
	dc->SelectObject(polybrush);
	dc->SelectObject(polypen);
	dc->Polygon(points, count);
	delete polybrush;
	delete polypen;
	gra_curdialog->window->ReleaseDC(dc);
}

void DiaDrawRect(INTBIG item, RECTAREA *ur, INTBIG r, INTBIG g, INTBIG b)
{
	RECT rr;
	CDC *dc;
	CBrush *brush;
	COLORREF color;

	rr.left = ur->left;
	rr.right = ur->right;
	rr.top = ur->top;
	rr.bottom = ur->bottom;

	color = (b << 16) | (g << 8) | r;
	brush = new CBrush(color);
	dc = gra_curdialog->window->GetDC();
	dc->FillRect(&rr, brush);
	gra_curdialog->window->ReleaseDC(dc);
	delete brush;
}

void DiaPutText(INTBIG item, char *msg, INTBIG x, INTBIG y)
{
	CFont *font;
	CDC *dc;
	UINTBIG descript[TEXTDESCRIPTSIZE];

	TDCLEAR(descript);
	TDSETSIZE(descript, TXTSETPOINTS(8));
	font = gra_gettextfont(NOWINDOWPART, NOTECHNOLOGY, descript);
	dc = gra_curdialog->window->GetDC();
	dc->SelectObject(font);
	dc->SetBkMode(TRANSPARENT);
	dc->TextOut(x, y, (LPCTSTR)msg, strlen(msg));
	gra_curdialog->window->ReleaseDC(dc);
}

void DiaGetTextInfo(char *msg, INTBIG *wid, INTBIG *hei)
{
	CFont *font;
	CSize textSize;
	CDC *dc;
	UINTBIG descript[TEXTDESCRIPTSIZE];

	TDCLEAR(descript);
	TDSETSIZE(descript, TXTSETPOINTS(8));
	font = gra_gettextfont(NOWINDOWPART, NOTECHNOLOGY, descript);
	dc = gra_curdialog->window->GetDC();
	dc->SelectObject(font);
	textSize = dc->GetTextExtent((LPCTSTR)msg, strlen(msg));
	*wid = textSize.cx;
	*hei = textSize.cy+1;
	gra_curdialog->window->ReleaseDC(dc);
}

void DiaGetMouse(INTSML *x, INTSML *y)
{
	POINT p, p2;

	if (us_logplay != NULL)
	{
		if (gra_loggetnextaction(0) == 0)
		{
			*x = (INTSML)gra_action.x;
			*y = (INTSML)gra_action.y;
		}
	}
	if (us_logplay == NULL)
	{
		p2.x = p2.y = 0;
		gra_curdialog->window->MapWindowPoints(0, &p2, 1);
		GetCursorPos(&p);
		*x = (INTSML)(p.x - p2.x);   *y = (INTSML)(p.y - p2.y);
	}
	gra_logwriteaction(DIAUSERMOUSE, *x, *y, 0);
}

INTSML DiaNullDlogList(char **c) { return(0); }

char *DiaNullDlogItem(void) { return(0); }

void DiaNullDlogDone(void) {}

/************************* DIALOG SUPPORT *************************/

/*
 * Routine to redraw user items and divider lines
 */
void gra_diaredrawitem(void)
{
	RECTAREA ra;
	INTBIG i, itemtype, x, y;
	HICON map;
	CDC *dc;

	if (gra_curdialog->redrawroutine != 0)
	{
		DiaItemRect(gra_curdialog->redrawitem, &ra);
		(*gra_curdialog->redrawroutine)(&ra);
	}

	/* redraw any special items */
	for(i=0; i<gra_curdialog->itemdesc->items; i++)
	{
		itemtype = gra_curdialog->itemdesc->list[i].type;
		if ((itemtype&ITEMTYPE) == DIVIDELINE)
		{
			DiaDrawRect(i+1, &gra_curdialog->itemdesc->list[i].r, 0, 0, 0);
		}
		if ((itemtype&ITEMTYPE) == ICON)
		{
			x = gra_curdialog->itemdesc->list[i].r.left;
			y = gra_curdialog->itemdesc->list[i].r.top;
			map = gra_makeicon(gra_curdialog->itemdesc->list[i].msg);
			dc = gra_curdialog->window->GetDC();
			dc->DrawIcon(x, y, map);
			gra_curdialog->window->ReleaseDC(dc);
			DestroyIcon(map);
		}
	}
}

/*
 * Routine called when an item is clicked
 */
void gra_itemclicked(int nID)
{
	int itemtype, line, count;
	CListBox *list;
	CComboBox *cb;
	int selintlist[MAXSCROLLMULTISELECT];

	gra_dialoghitchar = 0;
	if (nID == 1)
	{
		gra_dialoghit = gra_curdialog->defaultbutton;
		gra_logwriteaction(DIAITEMCLICK, gra_dialoghit, 0, 0);
		return;
	}
	if (nID == 2)
	{
		gra_dialoghit = 2;
		gra_logwriteaction(DIAITEMCLICK, gra_dialoghit, 0, 0);
		return;
	}

	/* handle scroll areas */
	itemtype = gra_curdialog->itemdesc->list[nID-ID_DIALOGITEM_0].type;
	if ((itemtype&ITEMTYPE) == SCROLL || (itemtype&ITEMTYPE) == SCROLLMULTI)
	{
		list = (CListBox *)gra_curdialog->window->GetDlgItem(nID);

		/* log the selection */
		if ((itemtype&ITEMTYPE) == SCROLLMULTI)
		{
			count = list->GetSelItems(MAXSCROLLMULTISELECT-1, selintlist);
			if (count != LB_ERR) gra_logwriteaction(DIASCROLLSEL, nID - ID_DIALOGITEM_0 + 1, count, selintlist);
		} else
		{
			line = list->GetCurSel();
			selintlist[0] = line;
			if (line != LB_ERR) gra_logwriteaction(DIASCROLLSEL, nID - ID_DIALOGITEM_0 + 1, 1, selintlist);
		}

		/* if no mouse selection allowed, deselect the list (leaves an outline, but oh well) */
		if ((gra_curdialog->itemdesc->list[nID-ID_DIALOGITEM_0].data&SCSELMOUSE) == 0)
		{
			if (list == 0) return;
			list->SetCurSel(-1);
			return;
		}

		/* ignore clicks in scroll areas that do not want hits reported */
		if ((gra_curdialog->itemdesc->list[nID-ID_DIALOGITEM_0].data&SCREPORT) == 0)
			return;
	}
	if ((itemtype&ITEMTYPE) == POPUP)
	{
		cb = (CComboBox *)gra_curdialog->window->GetDlgItem(nID);
		gra_logwriteaction(DIAPOPUPSEL, nID - ID_DIALOGITEM_0 + 1, cb->GetCurSel(), 0);
	}

	gra_dialoghit = nID - ID_DIALOGITEM_0 + 1;
	gra_logwriteaction(DIAITEMCLICK, gra_dialoghit, 0, 0);
}

/*
 * Routine called when an item is double-clicked
 */
void gra_itemdoubleclicked(int nID)
{
	int itemtype;

	if (gra_curdialog->window == 0) return;
	itemtype = gra_curdialog->itemdesc->list[nID].type;
	if ((itemtype&ITEMTYPE) != SCROLL && (itemtype&ITEMTYPE) != SCROLLMULTI) return;
	if ((gra_curdialog->itemdesc->list[nID].data&SCDOUBLEQUIT) == 0) return;

	gra_dialoghit = gra_curdialog->defaultbutton;
	gra_dialoghitchar = 0;
	gra_logwriteaction(DIAITEMCLICK, gra_dialoghit, 0, 0);
}

/*
 * Called when a key is typed to a list box.  Returns nonzero to accept the
 * keystroke, zero to ignore typed keys in the list box.
 */
int gra_dodialoglistkey(UINT nKey, CListBox* pListBox, UINT nIndex)
{
	CListBox *list;
	INTBIG i, itemtype, line, count;
	int selintlist[MAXSCROLLMULTISELECT];

	/* always allow arrow keys */
	if (nKey == VK_DOWN || nKey == VK_UP) return(1);

	for(i=0; i<gra_curdialog->itemdesc->items; i++)
	{
		itemtype = gra_curdialog->itemdesc->list[i].type;
		if ((itemtype&ITEMTYPE) != SCROLL && (itemtype&ITEMTYPE) != SCROLLMULTI) continue;
		list = (CListBox *)gra_curdialog->window->GetDlgItem(i+ID_DIALOGITEM_0);
		if (list != pListBox) continue;
		if ((gra_curdialog->itemdesc->list[i].data&SCSELKEY) == 0) return(0);
		if ((itemtype&ITEMTYPE) == SCROLLMULTI)
		{
			count = list->GetSelItems(MAXSCROLLMULTISELECT-1, selintlist);
			if (count != LB_ERR) gra_logwriteaction(DIASCROLLSEL, i + 1, (int)count, selintlist);
		} else
		{
			line = list->GetCurSel();
			selintlist[0] = line;
			if (line != LB_ERR) gra_logwriteaction(DIASCROLLSEL, i + 1, 1, selintlist);
		}
		break;
	}

	return(1);
}

/*
 * Routine called when a character is typed
 */
void gra_dodialogtextchange(int nID)
{
	char line[300], *lastline;
	CWnd *wnd;
	CEdit *edit;
	int len;

	/* get current contents of the edit field */
	gra_dialoghit = nID + 1;
	wnd = gra_curdialog->window->GetDlgItem(nID+ID_DIALOGITEM_0);
	if (wnd == 0) return;
	wnd->GetWindowText(line, 300);

	/* if this field is being character-edited, undo the change */
	if (nID == gra_dialogeditline)
	{
		/* get the previous contents of the edit field */
		lastline = (char *)gra_curdialog->itemdesc->list[nID].data;

		/* determine which character was typed (hack!!!) */
		if (strlen(line) > strlen(lastline))
		{
			gra_dialoghitchar = line[strlen(line)-1];
		} else if (strlen(line) < strlen(lastline))
		{
			gra_dialoghitchar = BACKSPACEKEY;
		}

		/* reset to previous state */
		wnd->SetWindowText(lastline);
		edit = (CEdit *)wnd;
		len = strlen(lastline);
		edit->SetSel(len, len);
	}
	gra_logwriteaction(DIAEDITTEXT, gra_dialoghit, gra_dialoghitchar, line);
}

/*
 * Routine to return nonzero if point (x,y) is inside a user-drawn item
 */
INTSML gra_dodialogisinsideuserdrawn(int x, int y)
{
	int i, itemtype;

	if (gra_curdialog->window == 0) return(0);
	for(i=0; i<gra_curdialog->itemdesc->items; i++)
	{
		itemtype = gra_curdialog->itemdesc->list[i].type;
		if ((itemtype&ITEMTYPE) != USERDRAWN &&
			(itemtype&ITEMTYPE) != ICON) continue;
		if (x < gra_curdialog->itemdesc->list[i].r.left) continue;
		if (x > gra_curdialog->itemdesc->list[i].r.right) continue;
		if (y < gra_curdialog->itemdesc->list[i].r.top) continue;
		if (y > gra_curdialog->itemdesc->list[i].r.bottom) continue;
		return(i+1);
	}
	return(0);
}

/*
 * Routine to make an icon from data
 */
HICON gra_makeicon(char *data)
{
	unsigned char zero[128];
	int i;

	for(i=0; i<128; i++)
	{
		zero[i] = 0;
		data[i] = ~data[i];
	}
	HICON icon = CreateIcon(0, 32, 32, 1, 1, (unsigned char *)data, zero);
	for(i=0; i<128; i++) data[i] = ~data[i];
	return(icon);
}

/****************************** TCL SUPPORT ******************************/

#if LANGTCL
INTBIG gra_initializetcl(void)
{
	INTBIG err;
	char *newArgv[2];

	/* set the program name/path */
	newArgv[0] = "Electric";
	newArgv[1] = NULL;
	(void)Tcl_FindExecutable(newArgv[0]);

	tcl_interp = Tcl_CreateInterp();
	if (tcl_interp == 0) error(_("from Tcl_CreateInterp"));

	/* tell Electric the TCL interpreter handle */
	el_tclinterpreter(tcl_interp);

	/* Make command-line arguments available in the Tcl variables "argc" and "argv" */
	Tcl_SetVar(tcl_interp, "argv", "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(tcl_interp, "argc", "0", TCL_GLOBAL_ONLY);
	Tcl_SetVar(tcl_interp, "argv0", "electric", TCL_GLOBAL_ONLY);

	/* Set the "tcl_interactive" variable */
	Tcl_SetVar(tcl_interp, "tcl_interactive", "1", TCL_GLOBAL_ONLY);

	/* initialize the interpreter */
	err = Tcl_Init(tcl_interp);
	if (err != TCL_OK) error(_("(from Tcl_Init) %s"), tcl_interp->result);

	return(err);
}
#endif

/****************************** PRINTING ******************************/

extern "C"
{
	void gra_printwindow(void);
}

/* call this routine to print the current window */
void gra_printwindow(void)
{
	INTBIG pagewid, pagehei, pagehpixperinch, pagevpixperinch,
		marginx, marginy, slx, shx, sly, shy, *curstate,
		ulx, uhx, uly, uhy, width, height, centerx, centery, prod1, prod2;
	INTSML i;
	REGISTER char *ptr;
	HCURSOR hCursorBusy, hCursorOld;
	static DOCINFO di = {sizeof(DOCINFO), "Electric", NULL};
	RGBQUAD bmiColors[256];
	BITMAPINFO *bmiInfo;
	CDC printDC, *pDCPrint;
	HDC hPrnDC;
	WINDOWPART *win;
	NODEPROTO *np;
	WINDOWFRAME *wf;
	DEVMODE *dm;

	/* get facet to plot */
	win = el_curwindowpart;
	if (win == NOWINDOWPART || win->curnodeproto == NONODEPROTO)
	{
		ttyputerr(_("No current facet to plot"));
		return;
	}
	wf = win->frame;
	np = win->curnodeproto;

	/* determine area to plot (in database units) */
	if ((win->state&WINDOWTYPE) == DISPWINDOW)
	{
		/* display window: determine bounds */
		if (io_getareatoprint(np, &slx, &shx, &sly, &shy, 0) != 0) return;
		(void)us_makescreen(&slx, &sly, &shx, &shy, win);
	} else
	{
		/* nondisplay window: simply print it */
		slx = win->uselx;   shx = win->usehx;
		sly = win->usely;   shy = win->usehy;
	}

	/* get printing control bits */
	curstate = io_getstatebits();

	/* create a print structure */
	CPrintDialog printDlg(FALSE);

	/* set requested orientation */
	if (printDlg.GetDefaults() == 0) return;
	dm = printDlg.GetDevMode();
	if ((curstate[0]&PSROTATE) != 0) dm->dmOrientation = DMORIENT_LANDSCAPE; else
		dm->dmOrientation = DMORIENT_PORTRAIT;
	printDlg.m_pd.Flags &= ~PD_RETURNDEFAULT;

	/* show the dialog */
	if (printDlg.DoModal() != IDOK) return;

	/* set cursor to busy */
	hCursorBusy = LoadCursor(NULL, IDC_WAIT);
	hCursorOld = SetCursor(hCursorBusy);

	/* collect size information about the Printer DC */
	hPrnDC = printDlg.m_pd.hDC;   /* GetPrinterDC(); */
	if (hPrnDC == 0)
	{
		ttyputerr(_("Printing error: could not start print job"));
		return;
	}
	pDCPrint = printDC.FromHandle(hPrnDC);
	pagewid = pDCPrint->GetDeviceCaps(HORZRES);
	pagehei = pDCPrint->GetDeviceCaps(VERTRES);
	pagehpixperinch = pDCPrint->GetDeviceCaps(LOGPIXELSX);
	pagevpixperinch = pDCPrint->GetDeviceCaps(LOGPIXELSY);
	marginx = pagehpixperinch / 2;
	marginy = pagevpixperinch / 2;

	/* make the plot have square pixels */
	ulx = marginx;
	uly = marginy;
	uhx = pagewid-marginx;
	uhy = pagehei-marginy;
	prod1 = (shx - slx) * (uhy - uly);
	prod2 = (shy - sly) * (uhx - ulx);
	if (prod1 != prod2)
	{
		/* adjust the scale */
		if (prod1 > prod2)
		{
			/* make it fill the width of the screen */
			height = muldiv(uhx - ulx, shy - sly, shx - slx);
			centery = (uly + uhy) / 2;
			uly = centery - height/2;
			uhy = uly + height;
		} else
		{
			/* make it fill the height of the screen */
			width = muldiv(uhy - uly, shx - slx, shy - sly);
			centerx = (ulx + uhx) / 2;
			ulx = centerx - width/2;
			uhx = ulx + width;
		}
	}

	/* adjust the background color to white */
	i = GetDIBColorTable(((CDC *)wf->hDCOff)->m_hDC, 0, 256, bmiColors);
	bmiColors[0].rgbRed = 0xFF;
	bmiColors[0].rgbGreen = 0xFF;
	bmiColors[0].rgbBlue = 0xFF;
	bmiColors[0].rgbReserved = 0;

	bmiInfo = (BITMAPINFO *)emalloc(sizeof(BITMAPINFOHEADER) + sizeof(bmiColors), db_cluster);
	if (bmiInfo == 0) return;
	ptr = (char *)bmiInfo;
	memcpy(ptr, wf->bminfo, sizeof(BITMAPINFOHEADER));
	ptr += sizeof(BITMAPINFOHEADER);
	memcpy(ptr, &bmiColors, sizeof(bmiColors));

	/* start printing */
	pDCPrint->StartDoc(&di);
	pDCPrint->StartPage();

	/* print the window */
	StretchDIBits(hPrnDC, ulx, uly, uhx-ulx, uhy-uly,
		slx, sly, shx-slx+1, shy-sly+1,
		wf->data, bmiInfo, DIB_RGB_COLORS, SRCCOPY);

	/* finish printing */
	pDCPrint->EndPage();
	pDCPrint->EndDoc();
	pDCPrint->Detach();
	DeleteDC(hPrnDC);

	/* restore original cursor */
	SetCursor(hCursorOld);

	efree((char *)bmiInfo);
}
