/*
 * Electric(tm) VLSI Design System
 *
 * File: usrhigh.c
 * User interface aid: object highlighting
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "egraphics.h"
#include "usr.h"
#include "tech.h"
#include "tecgen.h"

#define DOTRADIUS 4				/* size of dots in nodes when highlighting networks */

extern GRAPHICS us_box, us_arbit, us_hbox;

/* working memory for "us_gethighlighted()" */
static GEOM   **us_gethighlist;
static INTSML   us_gethighlistsize = 0;
static INTBIG   us_selectedtexttotal = 0;	/* size of selected text item list */
static char   **us_selectedtextlist;		/* the selected text items */

/* working memory for "us_getallhighlighted()" */
static INTBIG  *us_getallhighlist;
static INTSML   us_getallhighlistsize = 0;

/* prototypes for local routines */
void us_makecurrentobject(void);
void us_highlightverbose(GEOM*, PORTPROTO*, INTSML, NODEPROTO*, POLYGON*);

/*
 * Routine to free all memory associated with this module.
 */
void us_freehighmemory(void)
{
	REGISTER INTBIG i;

	if (us_gethighlistsize != 0) efree((char *)us_gethighlist);
	if (us_getallhighlistsize != 0) efree((char *)us_getallhighlist);
	for(i=0; i<us_selectedtexttotal; i++)
		if (us_selectedtextlist[i] != 0) efree((char *)us_selectedtextlist[i]);
	if (us_selectedtexttotal > 0) efree((char *)us_selectedtextlist);
}

/******************** MENU AND WINDOW HIGHLIGHTING ********************/

/*
 * routine to highlight a menu entry
 */
void us_highlightmenu(INTBIG x, INTBIG y, INTSML color)
{
	REGISTER INTBIG xl, yl;
	INTSML swid, shei, pwid;
	WINDOWPART ww;
	static POLYGON *poly = NOPOLYGON;

	if ((us_aid->aidstate&MENUON) == 0) return;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(5, us_aid->cluster);

	/* set drawing bounds to include menu */
	if (us_menuframe == NOWINDOWFRAME)
	{
		getpaletteparameters(&swid, &shei, &pwid);
		ww.frame = getwindowframe(1);
	} else
	{
		getwindowframesize(us_menuframe, &swid, &shei);
		ww.frame = us_menuframe;
	}
	ww.screenlx = ww.screenly = ww.uselx = ww.usely = 0;
	ww.screenhx = ww.usehx = swid-1;
	ww.screenhy = ww.usehy = shei-1;
	ww.state = DISPWINDOW;
	computewindowscale(&ww);

	xl = x * us_menuxsz + us_menulx;
	yl = y * us_menuysz + us_menuly;
	maketruerectpoly(xl, xl+us_menuxsz, yl, yl+us_menuysz, poly);
	poly->desc = &us_box;
	poly->style = CLOSEDRECT;
	us_box.col = color;
	(void)us_showpoly(poly, &ww);
}

/*
 * routine to highlight window "w" and set current window to this.  If
 * "force" is nonzero, highlight this window even if it is already the
 * current window.
 */
void us_highlightwindow(WINDOWPART *w, INTSML force)
{
	if (w == el_curwindowpart && force == 0) return;

	if (el_curwindowpart != NOWINDOWPART) us_drawwindow(el_curwindowpart, el_colwinbor);
	if (w != NOWINDOWPART) us_drawwindow(w, el_colhwinbor);

	/* set new window */
	el_curwindowpart = w;
}

/******************** OBJECT HIGHLIGHTING ********************/

/*
 * routine to add "thishigh" to the list of objects that are highlighted.
 * returns nonzero if the highlight is already there
 */
INTSML us_addhighlight(HIGHLIGHT *thishigh)
{
	REGISTER char *str, **list;
	REGISTER INTSML i, len, rewritefirst;
	char *onelist[1];
	REGISTER VARIABLE *var;
	HIGHLIGHT oldhigh;

	/* see if anything is currently highlighted */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		/* nothing highlighted: add this one line */
		onelist[0] = us_makehighlightstring(thishigh);
		(void)setvalkey((INTBIG)us_aid, VAID, us_highlighted, (INTBIG)onelist,
			VSTRING|VISARRAY|(1<<VLENGTHSH)|VDONTSAVE);

#if SIMAID
		/* if current window is simulating, show equivalent elsewhere */
		if (el_curwindowpart != NOWINDOWPART && (el_curwindowpart->state&WINDOWSIMULATING) != 0)
		{
			onelist[0] = "show-equivalent";
			tellaid(net_aid, 1, onelist);
		}
#endif

		/* set the current node/arc */
		us_makecurrentobject();
		return(0);
	}

	/* see if it is already highlighted */
	len = (INTSML)getlength(var);
	for(i=0; i<len; i++)
	{
		if (us_makehighlight(((char **)var->addr)[i], &oldhigh) != 0) continue;
		if ((oldhigh.status&HIGHTYPE) != (thishigh->status&HIGHTYPE)) continue;
		if (oldhigh.facet != thishigh->facet) continue;
		if ((oldhigh.status&HIGHTYPE) == HIGHLINE || (oldhigh.status&HIGHTYPE) == HIGHBBOX)
		{
			if (oldhigh.stalx != thishigh->stalx || oldhigh.stahx != thishigh->stahx ||
				oldhigh.staly != thishigh->staly || oldhigh.stahy != thishigh->stahy) continue;
		} else
		{
			if (oldhigh.fromgeom != thishigh->fromgeom) continue;
			if ((oldhigh.status&HIGHTYPE) == HIGHTEXT && oldhigh.fromvar != thishigh->fromvar)
				continue;
		}
		return(1);
	}

	/* process tangent snap points */
	rewritefirst = 0;
	if (len == 1 && (oldhigh.status&HIGHSNAP) != 0 && (thishigh->status&HIGHSNAP) != 0 &&
		((oldhigh.status&HIGHSNAPTAN) != 0 || (thishigh->status&HIGHSNAPTAN) != 0))
	{
		us_adjusttangentsnappoints(&oldhigh, thishigh);
		if ((oldhigh.status&HIGHSNAPTAN) != 0) rewritefirst = 1;
	} else if (len == 1 && (oldhigh.status&HIGHSNAP) != 0 && (thishigh->status&HIGHSNAP) != 0 &&
		((oldhigh.status&HIGHSNAPPERP) != 0 || (thishigh->status&HIGHSNAPPERP) != 0))
	{
		us_adjustperpendicularsnappoints(&oldhigh, thishigh);
		if ((oldhigh.status&HIGHSNAPPERP) != 0) rewritefirst = 1;
	}

	/* convert the highlight module to a string */
	str = us_makehighlightstring(thishigh);

	/* things are highlighted: build a new list */
	list = (char **)emalloc(((len+1) * (sizeof (char *))), el_tempcluster);
	if (list == 0) return(0);
	for(i=0; i<len; i++)
	{
		if (i == 0 && rewritefirst != 0)
		{
			(void)allocstring(&list[i], us_makehighlightstring(&oldhigh), el_tempcluster);
		} else
		{
			(void)allocstring(&list[i], ((char **)var->addr)[i], el_tempcluster);
		}
	}
	(void)allocstring(&list[len], str, el_tempcluster);

	/* save the new list */
	(void)setvalkey((INTBIG)us_aid, VAID, us_highlighted, (INTBIG)list,
		VSTRING|VISARRAY|((len+1)<<VLENGTHSH)|VDONTSAVE);

	/* clean up */
	for(i=0; i<=len; i++) efree(list[i]);
	efree((char *)list);

	/* set the current node/arc */
	us_makecurrentobject();
	return(0);
}

/*
 * routine to delete "thishigh" from the list of objects that are highlighted.
 * returns nonzero if the highlight is not there
 */
INTSML us_delhighlight(HIGHLIGHT *thishigh)
{
	REGISTER char **list;
	REGISTER INTSML i, len, j, thish;
	REGISTER VARIABLE *var;
	HIGHLIGHT oldhigh;

	/* see if anything is currently highlighted */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(1);

	/* see if it is highlighted */
	len = (INTSML)getlength(var);
	for(thish=0; thish<len; thish++)
	{
		if (us_makehighlight(((char **)var->addr)[thish], &oldhigh) != 0) continue;
		if ((oldhigh.status&HIGHTYPE) != (thishigh->status&HIGHTYPE)) continue;
		if (oldhigh.facet != thishigh->facet) continue;
		if ((oldhigh.status&HIGHTYPE) == HIGHLINE || (oldhigh.status&HIGHTYPE) == HIGHBBOX)
		{
			if (oldhigh.stalx != thishigh->stalx || oldhigh.stahx != thishigh->stahx ||
				oldhigh.staly != thishigh->staly || oldhigh.stahy != thishigh->stahy) continue;
		} else
		{
			if (oldhigh.fromgeom != thishigh->fromgeom) continue;
			if ((oldhigh.status&HIGHTYPE) == HIGHTEXT && oldhigh.fromvar != thishigh->fromvar)
				continue;
		}
		break;
	}
	if (thish >= len) return(1);

	/* removing highlight: build a new list */
	if (len <= 1)
	{
		(void)delvalkey((INTBIG)us_aid, VAID, us_highlighted);
		if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
		return(0);
	}
	list = (char **)emalloc(((len-1) * (sizeof (char *))), el_tempcluster);
	if (list == 0) return(0);
	for(j=i=0; i<len; i++)
	{
		if (i == thish) continue;
		(void)allocstring(&list[j], ((char **)var->addr)[i], el_tempcluster);
		j++;
	}

	/* save the new list */
	(void)setvalkey((INTBIG)us_aid, VAID, us_highlighted, (INTBIG)list,
		VSTRING|VISARRAY|((len-1)<<VLENGTHSH)|VDONTSAVE);
	us_makecurrentobject();

	/* clean up */
	for(i=0; i<len-1; i++) efree(list[i]);
	efree((char *)list);
	return(0);
}

void us_clearhighlightcount(void)
{
	if (getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted) != NOVARIABLE)
		(void)delvalkey((INTBIG)us_aid, VAID, us_highlighted);
	if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
}

/*
 * routine to add highlight module "newhigh" to the display.  If "findpoint" is nonzero,
 * find the closest vertex/line to the cursor.  If "extrainfo" is nonzero, highlight
 * with extra information.  If "findmore" is nonzero, add this highlight rather
 * than replacing.  If "nobox" is nonzero, do not draw the basic box.
 */
void us_setfind(HIGHLIGHT *newhigh, INTBIG findpoint, INTBIG extrainfo, INTBIG findmore, INTBIG nobox)
{
	REGISTER NODEINST *ni;
	REGISTER VARIABLE *var;
	REGISTER INTBIG x, y, d, bestdist, bestpoint, i, total;
	INTBIG xp, yp, xc, yc;
	XARRAY trans;

	/* set to highlight only one one object */
	if (findmore == 0) us_clearhighlightcount();

	/* find the vertex/line if this is a nodeinst and it was requested */
	if (findpoint != 0 && (newhigh->status&HIGHTYPE) != HIGHTEXT &&
		newhigh->fromgeom->entrytype == OBJNODEINST)
	{
		ni = newhigh->fromgeom->entryaddr.ni;
		var = gettrace(ni);
	} else var = NOVARIABLE;
	if (var == NOVARIABLE) bestpoint = 0; else
	{
		(void)getxy(&xc, &yc);
		makerot(ni, trans);
		total = getlength(var) / 2;
		x = (ni->highx + ni->lowx) / 2;   y = (ni->highy + ni->lowy) / 2;
		for(i=0; i<total; i++)
		{
			xform(((INTBIG *)var->addr)[i*2]+x, ((INTBIG *)var->addr)[i*2+1]+y, &xp, &yp, trans);
			d = abs(xp-xc) + abs(yp-yc);
			if (i == 0) bestdist = d + 1;
			if (d > bestdist) continue;
			bestdist = d;   bestpoint = i;
		}
		bestpoint++;
	}

	/* load the highlight structure */
	newhigh->status |= extrainfo;
	newhigh->frompoint = (INTSML)bestpoint;
	if (nobox != 0) newhigh->status |= HIGHNOBOX;

	/* show the highlighting */
	(void)us_addhighlight(newhigh);
}

/*
 * routine to demand a specified type of highlighted object (OBJNODEINST or
 * OBJARCINST, according to "type") and return its address.  If "canbetext"
 * is nonzero, the highlighted object can be a text object on the requested
 * type.  If the specified object is not currently highlighted, an error is
 * printed and the routine returns -1.
 */
UINTBIG us_getobject(INTSML type, INTSML canbetext)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	REGISTER INTSML len;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		us_abortcommand(_("Nothing is highlighted"));
		return((UINTBIG)-1);
	}
	len = (INTSML)getlength(var);
	if (len > 1)
	{
		us_abortcommand(_("Too much is highlighted"));
		return((UINTBIG)-1);
	}

	/* get the highlighted object */
	if (us_makehighlight(((char **)var->addr)[0], &high) != 0)
	{
		us_abortcommand(_("Highlight unintelligible"));
		return((UINTBIG)-1);
	}

	if ((high.status&HIGHTYPE) == HIGHTEXT && canbetext != 0) high.status = HIGHFROM;
	if ((high.status&HIGHTYPE) != HIGHFROM || high.fromgeom->entrytype != type)
	{
		if (type == OBJARCINST) us_abortcommand(_("Select an arc")); else
			us_abortcommand(_("Select a node"));
		return((UINTBIG)-1);
	}

	/* set the current position */
	return((UINTBIG)high.fromgeom->entryaddr.blind);
}

/*
 * routine to return the one object currently highlighted.  If no objects
 * are highlighted or many are, an error is printed and the routine returns
 * NOHIGHLIGHT.
 */
HIGHLIGHT *us_getonehighlight(void)
{
	static HIGHLIGHT high;
	REGISTER VARIABLE *var;
	REGISTER INTSML len;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		us_abortcommand(_("Nothing is highlighted"));
		return(NOHIGHLIGHT);
	}
	len = (INTSML)getlength(var);
	if (len > 1)
	{
		us_abortcommand(_("Too much is highlighted"));
		return(NOHIGHLIGHT);
	}

	/* get the highlighted object */
	if (us_makehighlight(((char **)var->addr)[0], &high) != 0)
	{
		us_abortcommand(_("Highlight unintelligible"));
		return(NOHIGHLIGHT);
	}
	return(&high);
}

/*
 * routine to get the two nodes that are selected and their ports.
 * If there are not two nodes selected, the routine returns nonzero.
 */
INTSML us_gettwoobjects(GEOM **firstgeom, PORTPROTO **firstport, GEOM **secondgeom,
	PORTPROTO **secondport)
{
	REGISTER VARIABLE *var;
	HIGHLIGHT newhigh;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(1);
	if (getlength(var) != 2) return(1);

	/* with 2 highlight objects, get them */
	(void)us_makehighlight(((char **)var->addr)[0], &newhigh);
	if ((newhigh.status&HIGHFROM) == 0) return(1);
	*firstgeom = newhigh.fromgeom;
	*firstport = newhigh.fromport;

	(void)us_makehighlight(((char **)var->addr)[1], &newhigh);
	if ((newhigh.status&HIGHFROM) == 0) return(1);
	*secondgeom = newhigh.fromgeom;
	*secondport = newhigh.fromport;
	return(0);
}

/*
 * routine to get a list of all objects that are highlighted, build an array
 * of them, and return the array.
 * If "type" is OBJNODEINST, only the highlighted nodes are returned.
 * If "type" is OBJARCINST, only the highlighted arcs are returned.
 * if "type" is OBJNODEINST|OBJARCINST, all highlighted objects are returned.
 * The list is terminated with the value NOGEOM.
 * If "textcount" and "textinfo" are nonzero, they are set to the number of
 * text objects that are highlighted (and the highlight descriptors for each).
 */
GEOM **us_gethighlighted(INTSML type, INTBIG *textcount, char ***textinfo)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	REGISTER INTBIG total, len, i, j, k, newcount, numtext;
	REGISTER INTBIG search;
	REGISTER GEOM *geom;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER char **newtext;
	static GEOM *nulllist[1];
	static POLYGON *poly = NOPOLYGON;
	XARRAY trans;

	if (textcount != 0) *textcount = 0;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		nulllist[0] = NOGEOM;
		return(nulllist);
	}

	/* first search to see how many objects there are */
	len = (INTSML)getlength(var);
	numtext = 0;
	total = 0;
	for(i=0; i<len; i++)
	{
		(void)us_makehighlight(((char **)var->addr)[i], &high);
		if ((high.status&HIGHTYPE) == HIGHFROM)
			if ((high.fromgeom->entrytype & type) != 0) total++;
		if ((high.status&HIGHTYPE) == HIGHBBOX)
		{
			search = initsearch(high.stalx, high.stahx, high.staly, high.stahy,
				high.facet);
			while ((geom = nextobject(search)) != NOGEOM)
				if ((geom->entrytype & type) != 0) total++;
		}

		/* special case: include in node list if text tied to node and no text requested */
		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			if (us_nodemoveswithtext(&high) != 0 && textcount == 0)
				total++;
		}
	}

	/* get memory for array of these objects */
	if (total >= us_gethighlistsize)
	{
		if (us_gethighlistsize != 0) efree((char *)us_gethighlist);
		us_gethighlist = (GEOM **)emalloc(((total+1) * (sizeof (GEOM *))),
			us_aid->cluster);
		if (us_gethighlist == 0)
		{
			ttyputnomemory();
			nulllist[0] = NOGEOM;
			return(nulllist);
		}
		us_gethighlistsize = total + 1;
	}

	/* now place the objects in the list */
	total = 0;
	for(i=0; i<len; i++)
	{
		(void)us_makehighlight(((char **)var->addr)[i], &high);
		if ((high.status&HIGHTYPE) == HIGHFROM)
			if ((high.fromgeom->entrytype & type) != 0)
		{
			for(j=0; j<total; j++) if (us_gethighlist[j] == high.fromgeom) break;
			if (j >= total) us_gethighlist[total++] = high.fromgeom;
		}
		if ((high.status&HIGHTYPE) == HIGHBBOX)
		{
			search = initsearch(high.stalx, high.stahx, high.staly, high.stahy, high.facet);
			for(;;)
			{
				geom = nextobject(search);
				if (geom == NOGEOM) break;

				/* make sure the object has the desired type */
				if ((geom->entrytype & type) == 0) continue;

				/* make sure the object isn't already in the list */
				for(j=0; j<total; j++) if (us_gethighlist[j] == geom) break;
				if (j < total) continue;

				/* check carefully for "edge select" primitives */
				if (geom->entrytype == OBJNODEINST &&
					geom->entryaddr.ni->proto->primindex != 0 &&
					(geom->entryaddr.ni->proto->userbits&NEDGESELECT) != 0)
				{
					ni = geom->entryaddr.ni;
					k = nodepolys(ni, 0, NOWINDOWPART);
					if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);
					makerot(ni, trans);
					for(j=0; j<k; j++)
					{
						shapenodepoly(ni, j, poly);
						if ((poly->desc->colstyle&INVISIBLE) == 0)
						{
							xformpoly(poly, trans);
							if (polyinrect(poly, high.stalx, high.stahx,
								high.staly, high.stahy) == 0) break;
						}
					}
					if (j < k) continue;
				} else if (geom->entrytype == OBJARCINST &&
					(geom->entryaddr.ai->proto->userbits&AEDGESELECT) != 0)
				{
					ai = geom->entryaddr.ai;
					k = arcpolys(ai, NOWINDOWPART);
					if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);
					for(j=0; j<k; j++)
					{
						shapearcpoly(ai, j, poly);
						if ((poly->desc->colstyle&INVISIBLE) == 0)
						{
							if (polyinrect(poly, high.stalx, high.stahx,
								high.staly, high.stahy) == 0) break;
						}
					}
					if (j < k) continue;
				}

				/* add this object to the list */
				us_gethighlist[total++] = geom;
			}
		}
		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			if (us_nodemoveswithtext(&high) != 0 && textcount == 0)
			{
				/* include in node list if text tied to node and no text requested */
				us_gethighlist[total++] = high.fromgeom;
			} else
			{
				if (textcount != 0)
				{
					if (numtext >= us_selectedtexttotal)
					{
						newcount = numtext + 5;
						newtext = (char **)emalloc(newcount * (sizeof (char *)),
							us_aid->cluster);
						if (newtext == 0)
						{
							nulllist[0] = NOGEOM;
							return(nulllist);
						}
						for(j=0; j<us_selectedtexttotal; j++)
							newtext[j] = us_selectedtextlist[j];
						for( ; j<newcount; j++) newtext[j] = 0;
						if (us_selectedtexttotal > 0) efree((char *)us_selectedtextlist);
						us_selectedtextlist = newtext;
						us_selectedtexttotal = newcount;
					}
					if (us_selectedtextlist[numtext] == 0)
					{
						allocstring(&us_selectedtextlist[numtext],
							((char **)var->addr)[i], us_aid->cluster);
					} else
					{
						reallocstring(&us_selectedtextlist[numtext],
							((char **)var->addr)[i], us_aid->cluster);
					}
					numtext++;
				}
			}
		}
	}
	if (textcount != 0 && textinfo != 0)
	{
		*textcount = numtext;
		*textinfo = us_selectedtextlist;
	}
	us_gethighlist[total] = NOGEOM;
	return(us_gethighlist);
}

/*
 * routine to get a list of all objects that are highlighted (including nodes, arcs,
 * and ports), build an array that alternates address and type, and return the array.
 * The list is terminated with the type 0.
 */
INTBIG *us_getallhighlighted(void)
{
	HIGHLIGHT high;
	REGISTER VARIABLE *var;
	REGISTER INTBIG total, len, i, j, k;
	REGISTER INTBIG search;
	REGISTER GEOM *geom;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	static INTBIG nulllist[2];
	static POLYGON *poly = NOPOLYGON;
	XARRAY trans;

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE)
	{
		nulllist[0] = nulllist[1] = 0;
		return(nulllist);
	}

	/* first search to see how many objects there are */
	len = (INTSML)getlength(var);
	total = 0;
	for(i=0; i<len; i++)
	{
		(void)us_makehighlight(((char **)var->addr)[i], &high);
		if ((high.status&HIGHTYPE) == HIGHFROM) total++;
		if ((high.status&HIGHTYPE) == HIGHBBOX)
		{
			search = initsearch(high.stalx, high.stahx, high.staly, high.stahy, high.facet);
			while ((geom = nextobject(search)) != NOGEOM) total++;
		}

		/* special case: accept ports and invisible node if it holds highlighted text */
		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			if (high.fromvar != NOVARIABLE && high.fromgeom->entrytype == OBJNODEINST &&
				high.fromgeom->entryaddr.ni->proto == gen_invispinprim) total++;
			if (high.fromport != NOPORTPROTO) total++;
		}
	}

	/* get memory for array of these objects */
	if (total*2 >= us_getallhighlistsize)
	{
		if (us_getallhighlistsize != 0) efree((char *)us_getallhighlist);
		us_getallhighlist = (INTBIG *)emalloc((total+1)*2 * SIZEOFINTBIG, us_aid->cluster);
		if (us_getallhighlist == 0)
		{
			ttyputnomemory();
			nulllist[0] = nulllist[1] = 0;
			return(nulllist);
		}
		us_getallhighlistsize = (total+1)*2;
	}

	/* now place the objects in the list */
	total = 0;
	for(i=0; i<len; i++)
	{
		(void)us_makehighlight(((char **)var->addr)[i], &high);
		if ((high.status&HIGHTYPE) == HIGHFROM)
		{
			for(j=0; j<total; j += 2)
				if (us_getallhighlist[j] == (INTBIG)high.fromgeom->entryaddr.blind) break;
			if (j >= total)
			{
				us_getallhighlist[total++] = (INTBIG)high.fromgeom->entryaddr.blind;
				switch (high.fromgeom->entrytype)
				{
					case OBJNODEINST: us_getallhighlist[total++] = VNODEINST;   break;
					case OBJARCINST:  us_getallhighlist[total++] = VARCINST;    break;
				}
			}
		}
		if ((high.status&HIGHTYPE) == HIGHBBOX)
		{
			search = initsearch(high.stalx, high.stahx, high.staly, high.stahy, high.facet);
			for(;;)
			{
				geom = nextobject(search);
				if (geom == NOGEOM) break;

				/* make sure it isn't already in the list */
				for(j=0; j<total; j++)
					if (us_getallhighlist[j] == (INTBIG)geom->entryaddr.blind) break;
				if (j < total) continue;

				/* check carefully for "edge select" primitives */
				if (geom->entrytype == OBJNODEINST && geom->entryaddr.ni->proto->primindex != 0 &&
					(geom->entryaddr.ni->proto->userbits&NEDGESELECT) != 0)
				{
					ni = geom->entryaddr.ni;
					k = nodepolys(ni, 0, NOWINDOWPART);
					if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);
					makerot(ni, trans);
					for(j=0; j<k; j++)
					{
						shapenodepoly(ni, j, poly);
						if ((poly->desc->colstyle&INVISIBLE) == 0)
						{
							xformpoly(poly, trans);
							if (polyinrect(poly, high.stalx, high.stahx, high.staly, high.stahy) == 0) break;
						}
					}
					if (j < k) continue;
				} else if (geom->entrytype == OBJARCINST &&
					(geom->entryaddr.ai->proto->userbits&AEDGESELECT) != 0)
				{
					ai = geom->entryaddr.ai;
					k = arcpolys(ai, NOWINDOWPART);
					if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);
					for(j=0; j<k; j++)
					{
						shapearcpoly(ai, j, poly);
						if ((poly->desc->colstyle&INVISIBLE) == 0)
						{
							if (polyinrect(poly, high.stalx, high.stahx, high.staly, high.stahy) == 0) break;
						}
					}
					if (j < k) continue;
				}

				/* add this object to the list */
				us_getallhighlist[total++] = (INTBIG)geom->entryaddr.blind;
				switch (geom->entrytype)
				{
					case OBJNODEINST: us_getallhighlist[total++] = VNODEINST;   break;
					case OBJARCINST:  us_getallhighlist[total++] = VARCINST;    break;
				}
			}
		}
		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			if (high.fromvar != NOVARIABLE && high.fromgeom->entrytype == OBJNODEINST &&
				high.fromgeom->entryaddr.ni->proto == gen_invispinprim)
			{
				us_getallhighlist[total++] = (INTBIG)high.fromgeom->entryaddr.blind;
				us_getallhighlist[total++] = VNODEINST;
			}
			if (high.fromport != NOPORTPROTO)
			{
				us_getallhighlist[total++] = (INTBIG)high.fromport;
				us_getallhighlist[total++] = VPORTPROTO;
			}
		}
	}
	us_getallhighlist[total++] = 0;
	us_getallhighlist[total] = 0;
	return(us_getallhighlist);
}

/*
 * routine to get the boundary of an area delimited by the highlight.
 * The bounds are put in the reference variables "lx", "hx", "ly", and "hy".
 * If there is no highlighted area, the routine returns with NONODEPROTO
 * Otherwise the routine returns the facet that contains the area.
 */
NODEPROTO *us_getareabounds(INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy)
{
	HIGHLIGHT high;
	REGISTER NODEINST *ni;
	REGISTER INTSML len, pointlen, i;
	INTBIG tlx, thx, tly, thy, xc, yc;
	XARRAY trans;
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEPROTO *thisfacet;
	REGISTER TECHNOLOGY *tech;
	REGISTER VARIABLE *var;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);

	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var == NOVARIABLE) return(NONODEPROTO);

	(void)us_makehighlight(((char **)var->addr)[0], &high);
	thisfacet = high.facet;
	len = (INTSML)getlength(var);
	for(i=0; i<len; i++)
	{
		(void)us_makehighlight(((char **)var->addr)[i], &high);
		if (high.facet != thisfacet)
		{
			us_abortcommand(_("Highlighted areas must be in one facet"));
			return(NONODEPROTO);
		}
		if ((high.status&HIGHTYPE) == HIGHTEXT)
		{
			if (high.fromvar != NOVARIABLE)
			{
				pointlen = 1;
				ni = NONODEINST;
				if (high.fromport != NOPORTPROTO) ni = high.fromport->subnodeinst; else
					if (high.fromgeom != NOGEOM && high.fromgeom->entrytype == OBJNODEINST)
				{
					ni = high.fromgeom->entryaddr.ni;
					if ((high.fromvar->type&VISARRAY) != 0)
						pointlen = (INTSML)getlength(high.fromvar);
				}
				if (pointlen > 1)
				{
					makedisparrayvarpoly(high.fromgeom, el_curwindowpart, high.fromvar, poly);
				} else
				{
					if (ni != NONODEINST) tech = ni->proto->tech; else
					{
						if (high.fromgeom != NOGEOM) tech = high.fromgeom->entryaddr.ai->proto->tech; else
							tech = high.facet->tech;
					}
					us_maketextpoly(describevariable(high.fromvar, -1, -1), el_curwindowpart,
						(high.fromgeom->lowx + high.fromgeom->highx) / 2,
							(high.fromgeom->lowy + high.fromgeom->highy) / 2,
								ni, tech, high.fromvar->textdescript, poly);
				}
				if (ni != NONODEINST)
				{
					makerot(ni, trans);
					xformpoly(poly, trans);
				}
			} else if (high.fromport != NOPORTPROTO)
			{
				portposition(high.fromport->subnodeinst, high.fromport->subportproto, &xc, &yc);
				us_maketextpoly(us_displayedportname(high.fromport, (us_useroptions&EXPORTLABELS) >> EXPORTLABELSSH),
					el_curwindowpart, xc, yc, high.fromport->subnodeinst,
						high.fromport->subnodeinst->proto->tech, high.fromport->textdescript, poly);
			} else if (high.fromgeom->entrytype == OBJNODEINST)
			{
				ni = high.fromgeom->entryaddr.ni;
				us_maketextpoly(describenodeproto(ni->proto), el_curwindowpart,
					(ni->lowx + ni->highx) / 2, (ni->lowy + ni->highy) / 2,
						ni, ni->proto->tech, ni->textdescript, poly);
			}
			getbbox(poly, &tlx, &thx, &tly, &thy);
		}
		if ((high.status&HIGHTYPE) == HIGHBBOX || (high.status&HIGHTYPE) == HIGHLINE)
		{
			tlx = high.stalx;   thx = high.stahx;
			tly = high.staly;   thy = high.stahy;
		}
		if ((high.status&HIGHTYPE) == HIGHFROM)
		{
			if (high.fromgeom->entrytype == OBJNODEINST)
			{
				us_getnodebounds(high.fromgeom->entryaddr.ni, &tlx, &thx, &tly, &thy);
			} else
			{
				us_getarcbounds(high.fromgeom->entryaddr.ai, &tlx, &thx, &tly, &thy);
			}
		}
		if (i == 0)
		{
			*lx = tlx;   *hx = thx;   *ly = tly;   *hy = thy;
		} else
		{
			*lx = mini(*lx, tlx);   *hx = maxi(*hx, thx);
			*ly = mini(*ly, tly);   *hy = maxi(*hy, thy);
		}
	}
	return(thisfacet);
}

/*
 * Routine to move the "numtexts" text objects described (as highlight strings)
 * in the array "textlist", by "odx" and "ody".
 */
void us_moveselectedtext(INTBIG numtexts, char **textlist, INTBIG odx, INTBIG ody)
{
	REGISTER INTBIG i, lambda, descript;
	INTBIG dx, dy, addr, type;
	REGISTER NODEINST *ni;
	XARRAY trans;
	HIGHLIGHT high;

	for(i=0; i<numtexts; i++)
	{
		dx = odx;   dy = ody;
		(void)us_makehighlight(textlist[i], &high);
		if ((high.status&HIGHTYPE) != HIGHTEXT) continue;

		/* get object and movement amount */
		us_gethighaddrtype(&high, &addr, &type);

		/* undraw the text */
		if (type == VPORTPROTO)
		{
			ni = ((PORTPROTO *)addr)->subnodeinst;
			startobjectchange((INTBIG)ni, VNODEINST);
			if (us_nodemoveswithtext(&high) != 0)
			{
				modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
				endobjectchange((INTBIG)ni, VNODEINST);
				continue;
			}
			if (ni->transpose != 0)
				makeangle(ni->rotation, ni->transpose, trans); else
					makeangle((INTSML)((3600-ni->rotation)%3600), 0, trans);
			xform(dx, dy, &dx, &dy, trans);
		} else
		{
			startobjectchange(addr, type);
			if (type == VNODEINST)
			{
				ni = (NODEINST *)addr;
				if (us_nodemoveswithtext(&high) != 0)
				{
					modifynodeinst(ni, dx, dy, dx, dy, 0, 0);
					endobjectchange((INTBIG)ni, VNODEINST);
					continue;
				}
				if (ni->transpose != 0)
					makeangle(ni->rotation, ni->transpose, trans); else
						makeangle((INTSML)((3600-ni->rotation)%3600), 0, trans);
				xform(dx, dy, &dx, &dy, trans);
			}
		}
		if (type == VNODEPROTO && high.fromvar != NOVARIABLE)
			us_undrawfacetvariable(high.fromvar, (NODEPROTO *)addr);

		/* set the new descriptor on the text */
		lambda = el_curlib->lambda[us_hightech(&high)->techindex];
		dx = dx*4/lambda;   dy = dy*4/lambda;
		descript = us_gethighdescript(&high);
		if ((descript&VTXOFFNEG) != 0) dx -= (descript&VTXOFF) >> VTXOFFSH; else
			dx += (descript&VTXOFF) >> VTXOFFSH;
		if ((descript&VTYOFFNEG) != 0) dy -= (descript&VTYOFF) >> VTYOFFSH; else
			dy += (descript&VTYOFF) >> VTYOFFSH;
		descript = us_setdescriptoffset(descript, dx, dy);
		us_modifytextdescript(&high, descript);

		/* redisplay the text */
		if (type == VNODEPROTO && high.fromvar != NOVARIABLE)
			us_drawfacetvariable(high.fromvar, (NODEPROTO *)addr);
		if (type == VPORTPROTO)
		{
			endobjectchange((INTBIG)ni, VNODEINST);
		} else
		{
			endobjectchange(addr, type);
		}
		(void)us_addhighlight(&high);

		/* modify all higher-level nodes if port moved */
		if (high.fromvar == NOVARIABLE && high.fromport != NOPORTPROTO)
		{
			for(ni = high.fromport->parent->firstinst; ni != NONODEINST; ni = ni->nextinst)
			{
				if ((ni->userbits&NEXPAND) != 0 &&
					(high.fromport->userbits&PORTDRAWN) == 0) continue;
				startobjectchange((INTBIG)ni, VNODEINST);
				endobjectchange((INTBIG)ni, VNODEINST);
			}
		}
	}
}

/*
 * Routine to find a single selected snap point and return it in (x, y).
 * Returns zero if there is not a single snap point.
 */
INTSML us_getonesnappoint(INTBIG *x, INTBIG *y)
{
	REGISTER INTSML snapcount;
	REGISTER VARIABLE *var;
	REGISTER INTBIG len, i;
	HIGHLIGHT thishigh;

	snapcount = 0;
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE)
	{
		len = getlength(var);
		for(i=0; i<len; i++)
		{
			if (us_makehighlight(((char **)var->addr)[i], &thishigh) == 0)
			{
				if ((thishigh.status&HIGHSNAP) != 0)
				{
					us_getsnappoint(&thishigh, x, y);
					snapcount++;
				}
			}
		}
	}
	if (snapcount == 1) return(1);
	return(0);
}

/*
 * routine to draw the highlighting for module "high".  The highlighting is
 * drawn on if "on" is HIGHLIT, off if "on" is ALLOFF.
 */
void us_sethighlight(HIGHLIGHT *high, INTSML on)
{
	REGISTER INTSML len;
	REGISTER INTBIG xw, yw, descript;
	INTBIG xc, yc, lx, hx, ly, hy, addr, type;
	INTSML tsx, tsy, style;
	REGISTER WINDOWPART *w;
	REGISTER TECHNOLOGY *tech;
	static POLYGON *poly = NOPOLYGON;
	REGISTER char *str;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(5, us_aid->cluster);

	/* handle drawing bounding boxes */
	if ((high->status&HIGHTYPE) == HIGHBBOX)
	{
		maketruerectpoly(high->stalx, high->stahx, high->staly, high->stahy, poly);
		poly->desc = &us_hbox;
		poly->style = CLOSEDRECT;
		us_hbox.col = on;
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto == high->facet) (void)us_showpoly(poly, w);
		}
		return;
	}

	/* handle drawing lines */
	if ((high->status&HIGHTYPE) == HIGHLINE)
	{
		poly->xv[0] = high->stalx;   poly->yv[0] = high->staly;
		poly->xv[1] = high->stahx;   poly->yv[1] = high->stahy;
		poly->count = 2;
		poly->desc = &us_hbox;
		poly->style = VECTORS;
		us_hbox.col = on;
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto == high->facet) (void)us_showpoly(poly, w);
		}
		return;
	}

	/* handle text outlines */
	if ((high->status&HIGHTYPE) == HIGHTEXT)
	{
		descript = us_gethighdescript(high);
		str = us_gethighstring(high);

		/* determine center of text */
		us_gethightextcenter(high, &xc, &yc, &style);

		/* initialize polygon */
		poly->layer = -1;
		poly->desc = &us_hbox;
		us_hbox.col = on;

		/* determine number of lines of text */
		len = 1;
		if (high->fromvar != NOVARIABLE && (high->fromvar->type&VISARRAY) != 0)
			len = (INTSML)getlength(high->fromvar);

		/* display in every appropriate window */
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto != high->facet) continue;

			/* determine size of text */
			if (len > 1)
			{
				makedisparrayvarpoly(high->fromgeom, w, high->fromvar, poly);
				getbbox(poly, &lx, &hx, &ly, &hy);
				xw = hx - lx;
				yw = hy - ly;
			} else
			{
				/* determine size from text */
				tech = us_hightech(high);
				screensettextsize(w, truefontsize((descript&VTSIZE)>>VTSIZESH, w, tech));
				screengettextsize(w, str, &tsx, &tsy);
				xw = muldiv(tsx, w->screenhx-w->screenlx, w->usehx-w->uselx);
				yw = muldiv(tsy, w->screenhy-w->screenly, w->usehy-w->usely);
			}

			/* construct polygon */
			us_gethighaddrtype(high, &addr, &type);
			(void)us_getobjectinfo(addr, type, &lx, &hx, &ly, &hy);
			us_buildtexthighpoly(lx, hx, ly, hy, xc, yc, xw, yw, style, poly);
			(void)us_showpoly(poly, w);
		}
		return;
	}

	/* quit if nothing to draw */
	if ((high->status&HIGHFROM) == 0 || high->fromgeom == NOGEOM) return;

	/* highlight "from" object */
	us_highlighteverywhere(high->fromgeom, high->fromport, high->frompoint,
		(INTSML)(high->status&HIGHEXTRA), on, (INTSML)(high->status&HIGHNOBOX));

	/* draw the snap point if set */
	if ((high->status&HIGHSNAP) != 0)
	{
		us_getsnappoint(high, &poly->xv[0], &poly->yv[0]);
		poly->count = 1;
		poly->desc = &us_hbox;
		poly->style = BIGCROSS;
		us_hbox.col = on;
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto == high->facet) (void)us_showpoly(poly, w);
		}
	}
}

/*
 * routine to build a polygon in "poly" that describes the highlighting of
 * a text object whose grab point is (xc,yc), size is (xw,yw), graphics
 * style is in "style" (either TEXTCENT, TEXTBOT, ...) and whose attached
 * object is "geom".
 */
void us_buildtexthighpoly(INTBIG lx, INTBIG hx, INTBIG ly, INTBIG hy,
	INTBIG xc, INTBIG yc, INTBIG xw, INTBIG yw, INTSML style, POLYGON *poly)
{
	REGISTER INTBIG xsh, ysh;

	/* compute appropriate highlight */
	switch (style)
	{
		case TEXTCENT:
		case TEXTBOX:
			poly->xv[0] = xc - xw/2;   poly->yv[0] = yc - yw/2;
			poly->xv[1] = xc + xw/2;   poly->yv[1] = yc + yw/2;
			poly->xv[2] = xc - xw/2;   poly->yv[2] = yc + yw/2;
			poly->xv[3] = xc + xw/2;   poly->yv[3] = yc - yw/2;
			poly->style = VECTORS;     poly->count = 4;
			if (style == TEXTCENT) break;
			if (poly->limit < 12) (void)extendpolygon(poly, 12);
			xsh = (hx - lx) / 6;
			ysh = (hy - ly) / 6;
			poly->xv[4] = lx + xsh;    poly->yv[4] = ly;
			poly->xv[5] = hx - xsh;    poly->yv[5] = ly;
			poly->xv[6] = lx + xsh;    poly->yv[6] = hy;
			poly->xv[7] = hx - xsh;    poly->yv[7] = hy;
			poly->xv[8] = lx;          poly->yv[8] = ly + ysh;
			poly->xv[9] = lx;          poly->yv[9] = hy - ysh;
			poly->xv[10] = hx;         poly->yv[10] = ly + ysh;
			poly->xv[11] = hx;         poly->yv[11] = hy - ysh;
			poly->count = 12;
			break;
		case TEXTBOT:
			poly->xv[0] = xc - xw/2;   poly->yv[0] = yc + yw;
			poly->xv[1] = xc - xw/2;   poly->yv[1] = yc;
			poly->xv[2] = xc + xw/2;   poly->yv[2] = yc;
			poly->xv[3] = xc + xw/2;   poly->yv[3] = yc + yw;
			poly->style = OPENED;      poly->count = 4;
			break;
		case TEXTTOP:
			poly->xv[0] = xc - xw/2;   poly->yv[0] = yc - yw;
			poly->xv[1] = xc - xw/2;   poly->yv[1] = yc;
			poly->xv[2] = xc + xw/2;   poly->yv[2] = yc;
			poly->xv[3] = xc + xw/2;   poly->yv[3] = yc - yw;
			poly->style = OPENED;      poly->count = 4;
			break;
		case TEXTLEFT:
			poly->xv[0] = xc + xw;     poly->yv[0] = yc + yw/2;
			poly->xv[1] = xc;          poly->yv[1] = yc + yw/2;
			poly->xv[2] = xc;          poly->yv[2] = yc - yw/2;
			poly->xv[3] = xc + xw;     poly->yv[3] = yc - yw/2;
			poly->style = OPENED;      poly->count = 4;
			break;
		case TEXTRIGHT:
			poly->xv[0] = xc - xw;     poly->yv[0] = yc + yw/2;
			poly->xv[1] = xc;          poly->yv[1] = yc + yw/2;
			poly->xv[2] = xc;          poly->yv[2] = yc - yw/2;
			poly->xv[3] = xc - xw;     poly->yv[3] = yc - yw/2;
			poly->style = OPENED;      poly->count = 4;
			break;
		case TEXTTOPLEFT:
			poly->xv[0] = xc + xw;     poly->yv[0] = yc;
			poly->xv[1] = xc;          poly->yv[1] = yc;
			poly->xv[2] = xc;          poly->yv[2] = yc - yw;
			poly->style = OPENED;      poly->count = 3;
			break;
		case TEXTBOTLEFT:
			poly->xv[0] = xc;          poly->yv[0] = yc + yw;
			poly->xv[1] = xc;          poly->yv[1] = yc;
			poly->xv[2] = xc + xw;     poly->yv[2] = yc;
			poly->style = OPENED;      poly->count = 3;
			break;
		case TEXTTOPRIGHT:
			poly->xv[0] = xc - xw;     poly->yv[0] = yc;
			poly->xv[1] = xc;          poly->yv[1] = yc;
			poly->xv[2] = xc;          poly->yv[2] = yc - yw;
			poly->style = OPENED;      poly->count = 3;
			break;
		case TEXTBOTRIGHT:
			poly->xv[0] = xc - xw;     poly->yv[0] = yc;
			poly->xv[1] = xc;          poly->yv[1] = yc;
			poly->xv[2] = xc;          poly->yv[2] = yc + yw;
			poly->style = OPENED;      poly->count = 3;
			break;
	}
}

/*
 * general routine to change the highlighting of an object.  Every displayed
 * occurrence of geometry module "look" will have its highlighting drawn.
 * The highlighting will be in color "color".
 * If "port" is not NOPORTPROTO and the geometry module points to
 * a nodeinst, that port will also be highlighted.  If "point" is nonzero,
 * then that point/line of the trace will be highlighted.  If "verbose" is
 * nonzero, the object will be highlighted with extra information.  If
 * "nobbox" is nonzero, don't draw the basic box around the object.
 */
void us_highlighteverywhere(GEOM *look, PORTPROTO *port, INTSML point, INTSML verbose, INTSML color,
	INTSML nobbox)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER NODEPROTO *par;
	REGISTER INTBIG i, x, y;
	REGISTER WINDOWPART *w;
	REGISTER VARIABLE *var;
	static POLYGON *poly = NOPOLYGON;
	XARRAY trans;
	INTBIG lx, hx, ly, hy;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);

	/* sensibility check to prevent null highlighting */
	if (look->entrytype == OBJNODEINST)
	{
		if (port == NOPORTPROTO && verbose == 0 && point == 0) nobbox = 0;
	} else
	{
		point = 0;
		if (verbose == 0) nobbox = 0;
	}

	/* determine parent, compute polygon */
	par = geomparent(look);
	if (look->entrytype == OBJNODEINST)
	{
		ni = look->entryaddr.ni;
		makerot(ni, trans);
		nodesizeoffset(ni, &lx, &ly, &hx, &hy);
		makerectpoly(ni->lowx+lx, ni->highx-hx, ni->lowy+ly, ni->highy-hy, poly);
		poly->style = FILLED;
		xformpoly(poly, trans);
	} else
	{
		ai = look->entryaddr.ai;
		i = ai->width - arcwidthoffset(ai);
		if (curvedarcoutline(ai, poly, CLOSED, i) != 0)
			makearcpoly(ai->length, i, ai, poly, FILLED);
	}

	/* figure out the line type for this highlighting */
	poly->desc = &us_hbox;
	poly->desc->col = color;
	poly->style = OPENEDO1;

	/* if extra information is being un-drawn, show it now */
	if (verbose != 0 && color == ALLOFF)
		us_highlightverbose(look, port, color, par, poly);

	/* add the final point to close the polygon */
	if (poly->limit < poly->count+1) (void)extendpolygon(poly, (INTSML)(poly->count+1));
	poly->xv[poly->count] = poly->xv[0];
	poly->yv[poly->count] = poly->yv[0];
	poly->count++;

	/* if all points are the same, draw as a cross */
	for(i=1; i<poly->count; i++)
		if (poly->xv[i-1] != poly->xv[i] || poly->yv[i-1] != poly->yv[i]) break;
	if (i >= poly->count) poly->style = CROSS;

	/* loop through windows and draw the highlighting */
	if (nobbox == 0)
	{
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto == par) (void)us_showpoly(poly, w);
		}
	}

	/* remove that point to make the polygon centered */
	poly->count--;

	/* if extra information is being drawn, show it now */
	if (verbose != 0 && color != ALLOFF)
		us_highlightverbose(look, port, color, par, poly);

	/* draw the port prototype if requested */
	if (look->entrytype == OBJNODEINST && port != NOPORTPROTO)
	{
		/* compute the port bounds */
		shapeportpoly(ni, port, poly, 0);

		/* compute graphics for the port outline */
		poly->desc = &us_arbit;
		us_arbit.bits = LAYERH;   us_arbit.col = color;

		/* see if the polygon is a single point */
		for(i=1; i<poly->count; i++)
			if (poly->xv[i] != poly->xv[i-1] || poly->yv[i] != poly->yv[i-1]) break;
		if (i < poly->count)
		{
			/* not a single point, draw its outline */
			if (poly->style == FILLEDRECT) poly->style = CLOSEDRECT; else
				if (poly->style == FILLED) poly->style = CLOSED; else
					if (poly->style == DISC) poly->style = CIRCLE;
		} else
		{
			/* single point port: make it a cross */
			poly->count = 1;
			poly->style = CROSS;
		}

		/* draw the port */
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto == par) (void)us_showpoly(poly, w);
		}
	}

	/* draw the point or line on the trace if requested */
	if (point != 0)
	{
		var = gettrace(ni);
		if (var == NOVARIABLE) return;
		x = (ni->highx + ni->lowx) / 2;   y = (ni->highy + ni->lowy) / 2;

		xform(((INTBIG *)var->addr)[(point-1)*2]+x, ((INTBIG *)var->addr)[(point-1)*2+1]+y,
			&poly->xv[0], &poly->yv[0], trans);
		poly->desc = &us_arbit;
		us_arbit.bits = LAYERH;   us_arbit.col = color;
		poly->count = 1;
		poly->style = CROSS;
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if ((w->state&WINDOWTYPE) != DISPWINDOW &&
				(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
			if (w->curnodeproto == par) (void)us_showpoly(poly, w);
		}

		if (point*2 < getlength(var))
		{
			xform(((INTBIG *)var->addr)[point*2]+x, ((INTBIG *)var->addr)[point*2+1]+y,
				&poly->xv[1], &poly->yv[1], trans);
			poly->desc = &us_arbit;
			us_arbit.bits = LAYERH;   us_arbit.col = color;
			poly->count = 2;
			poly->style = OPENED;
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
			{
				if ((w->state&WINDOWTYPE) != DISPWINDOW &&
					(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
				if (w->curnodeproto == par) (void)us_showpoly(poly, w);
			}
		}
	}
}

/*
 * Routine to draw extra verbose highlighting on object "look" (if it is a node, then
 * "pp" is the port on that node).  The color is "color", the parent facet is "par", and
 * if it is an arc, the polygon for that arc is "arcpoly".
 */
void us_highlightverbose(GEOM *look, PORTPROTO *pp, INTSML color, NODEPROTO *par, POLYGON *arcpoly)
{
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEINST *ni, *hni;
	REGISTER ARCINST *ai, *oai;
	REGISTER WINDOWPART *w;
	REGISTER PORTARCINST *pi;
	REGISTER INTSML i, thisend;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);
	poly->desc = &us_hbox;
	poly->desc->col = color;

	if (look->entrytype == OBJNODEINST)
	{
		/* handle verbose highlighting of nodes */
		if (pp == NOPORTPROTO) return;
		if (pp->network == NONETWORK) return;

		hni = look->entryaddr.ni;
		for(ni = par->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			ni->userbits &= ~HIGHLIGHTNVER;
		for(ai = par->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			ai->userbits &= ~HIGHLIGHTAVER;

		/* determine which arcs should be highlighted */
		for(pi = hni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
		{
			if (pi->proto->network != pp->network) continue;
			ai = pi->conarcinst;
			ai->userbits |= HIGHLIGHTAVER;
			if (ai->network != NONETWORK)
			{
				for(oai = par->firstarcinst; oai != NOARCINST; oai = oai->nextarcinst)
				{
					if (oai->network != ai->network) continue;
					oai->userbits |= HIGHLIGHTAVER;
					oai->end[0].nodeinst->userbits |= HIGHLIGHTNVER;
					oai->end[1].nodeinst->userbits |= HIGHLIGHTNVER;
				}
			}
		}

		/* draw lines along all of the arcs on the network */
		for(ai = par->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			if ((ai->userbits&HIGHLIGHTAVER) == 0) continue;
			poly->xv[0] = ai->end[0].xpos;
			poly->yv[0] = ai->end[0].ypos;
			poly->xv[1] = ai->end[1].xpos;
			poly->yv[1] = ai->end[1].ypos;
			poly->count = 2;
			poly->style = OPENEDT2;
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
			{
				if ((w->state&WINDOWTYPE) != DISPWINDOW &&
					(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
				if (w->curnodeproto == par) (void)us_showpoly(poly, w);
			}
		}

		/* draw dots in all connected nodes */
		for(ni = par->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			if (ni == hni) continue;
			if ((ni->userbits&HIGHLIGHTNVER) == 0) continue;
			poly->xv[0] = (ni->lowx + ni->highx) / 2;
			poly->yv[0] = (ni->lowy + ni->highy) / 2;
			poly->xv[1] = poly->xv[0] + el_curlib->lambda[el_curtech->techindex]/4;
			poly->yv[1] = poly->yv[0];
			poly->count = 2;
			poly->style = DISC;
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
			{
				if ((w->state&WINDOWTYPE) != DISPWINDOW &&
					(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
				if (w->curnodeproto != par) continue;
				poly->xv[1] = poly->xv[0] + (INTBIG)(DOTRADIUS / w->scalex);
				(void)us_showpoly(poly, w);
			}

			/* connect the center dots to the input arcs */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = pi->nextportarcinst)
			{
				ai = pi->conarcinst;
				if ((ai->userbits&HIGHLIGHTAVER) == 0) continue;
				if (ai->end[0].portarcinst == pi) thisend = 0; else thisend = 1;
				if (ai->end[thisend].xpos != poly->xv[0] ||
					ai->end[thisend].ypos != poly->yv[0])
				{
					poly->xv[1] = ai->end[thisend].xpos;
					poly->yv[1] = ai->end[thisend].ypos;
					poly->count = 2;
					poly->style = OPENEDT1;
					for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
					{
						if ((w->state&WINDOWTYPE) != DISPWINDOW &&
							(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
						if (w->curnodeproto == par) (void)us_showpoly(poly, w);
					}
				}
			}
		}
		return;
	}

	/* handle verbose highlighting of arcs */
	ai = look->entryaddr.ai;

	/* get a description of the constraints on the arc */
	poly->string = (char *)(*(el_curconstraint->request))("describearc", (INTBIG)ai);

	/* quit now if there is no verbose text to print */
	if (poly->string == 0 || *poly->string == 0) return;

	/* determine the location of this text */
	if (arcpoly->count != 4)
	{
		/* presume curved arc outline and determine center four points */
		i = arcpoly->count/2;
		poly->xv[0] = (arcpoly->xv[i/2] + arcpoly->xv[arcpoly->count-i/2-1] +
			arcpoly->xv[(i-1)/2] + arcpoly->xv[arcpoly->count-(i-1)/2-1]) / 4;
		poly->yv[0] = (arcpoly->yv[i/2] + arcpoly->yv[arcpoly->count-i/2-1] +
			arcpoly->yv[(i-1)/2] + arcpoly->yv[arcpoly->count-(i-1)/2-1]) / 4;
	} else
	{
		/* use straight arc: simply get center */
		poly->xv[0] = (ai->end[0].xpos + ai->end[1].xpos) / 2;
		poly->yv[0] = (ai->end[0].ypos + ai->end[1].ypos) / 2;
	}
	poly->style = TEXTCENT;
	if (ai->width == 0)
	{
		if (ai->end[0].ypos == ai->end[1].ypos) poly->style = TEXTBOT; else
			if (ai->end[0].xpos == ai->end[1].xpos) poly->style = TEXTLEFT; else
				if ((ai->end[0].xpos-ai->end[1].xpos) * (ai->end[0].ypos-ai->end[1].ypos) > 0)
					poly->style = TEXTBOTRIGHT; else
						poly->style = TEXTBOTLEFT;
	}
	poly->count = 1;
	poly->font = TXT1L;
	poly->tech = ai->parent->tech;
	for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
	{
		if ((w->state&WINDOWTYPE) != DISPWINDOW &&
			(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
		if (w->curnodeproto == par) (void)us_showpoly(poly, w);
	}
}

/*
 * routine to push the currently highlighted configuration onto a stack
 */
void us_pushhighlight(void)
{
	REGISTER VARIABLE *var;
	REGISTER INTBIG len, i;
	REGISTER char **list;
	char *one[1];

	/* get what is highlighted */
	(void)initinfstr();
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);
	if (var != NOVARIABLE)
	{
		len = getlength(var);
		for(i=0; i<len; i++)
		{
			if (i != 0) (void)addtoinfstr('\n');
			(void)addstringtoinfstr(((char **)var->addr)[i]);
		}
	}

	/* now add this to the stack variable */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlightstack);
	if (var == NOVARIABLE)
	{
		one[0] = returninfstr();
		(void)setvalkey((INTBIG)us_aid, VAID, us_highlightstack, (INTBIG)one,
			VSTRING|VISARRAY|(1<<VLENGTHSH)|VDONTSAVE);
		return;
	}

	len = getlength(var);
	list = (char **)emalloc(((len+1) * (sizeof (char *))), el_tempcluster);
	if (list == 0) return;
	for(i=0; i<len; i++)
		(void)allocstring(&list[i], ((char **)var->addr)[i], el_tempcluster);
	list[len] = returninfstr();
	(void)setvalkey((INTBIG)us_aid, VAID, us_highlightstack, (INTBIG)list,
		VSTRING|VISARRAY|((len+1)<<VLENGTHSH)|VDONTSAVE);
	for(i=0; i<len; i++) efree(list[i]);
	efree((char *)list);
}

/*
 * routine to pop the stack onto the currently highlighted configuration.
 * Returns nonzero if the stack is empty.
 */
INTSML us_pophighlight(INTSML clearsnap)
{
	REGISTER VARIABLE *var;
	REGISTER char **vaddr;
	REGISTER INTSML len, i;
	REGISTER char **list;

	/* get the stack variable */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlightstack);
	if (var == NOVARIABLE) return(1);
	len = (INTSML)getlength(var);
	vaddr = (char **)var->addr;

	/* set the configuration */
	us_setmultiplehighlight(vaddr[len-1], clearsnap);

	/* shorten the stack */
	if (len == 1) (void)delvalkey((INTBIG)us_aid, VAID, us_highlightstack); else
	{
		len--;
		list = (char **)emalloc((len * (sizeof (char *))), el_tempcluster);
		if (list == 0) return(0);
		for(i=0; i<len; i++)
			(void)allocstring(&list[i], vaddr[i], el_tempcluster);
		(void)setvalkey((INTBIG)us_aid, VAID, us_highlightstack, (INTBIG)list,
			VSTRING|VISARRAY|(len<<VLENGTHSH)|VDONTSAVE);
		for(i=0; i<len; i++) efree(list[i]);
		efree((char *)list);
	}
	return(0);
}

/*
 * routine to highlight the multiple highlight information in "str".
 */
void us_setmultiplehighlight(char *str, INTSML clearsnap)
{
	char *pp;
	REGISTER char *line, **list;
	REGISTER INTSML total, i;
	HIGHLIGHT high;

	/* count the number of highlights */
	pp = str;
	for(total=0; ; total++)
	{
		line = getkeyword(&pp, "\n");
		if (line == NOSTRING) break;
		if (*line == 0) break;
		(void)tonextchar(&pp);
	}
	if (total == 0) return;

	/* make space for all of the highlight strings */
	list = (char **)emalloc((total * (sizeof (char *))), el_tempcluster);
	if (list == 0) return;

	/* fill the list */
	pp = str;
	for(total=0; ; )
	{
		line = getkeyword(&pp, "\n");
		if (line == NOSTRING) break;
		if (*line == 0) break;

		if (clearsnap != 0)
		{
			(void)initinfstr();
			(void)addstringtoinfstr(line);
			if (us_makehighlight(returninfstr(), &high) == 0)
			{
				high.status &= ~(HIGHSNAP | HIGHSNAPTAN | HIGHSNAPPERP);
				line = us_makehighlightstring(&high);
				(void)allocstring(&list[total++], line, el_tempcluster);
			}
		} else
		{
			(void)allocstring(&list[total++], line, el_tempcluster);
		}
		if (tonextchar(&pp) != '\n') break;
	}

	/* set the multiple list */
	if (total == 0) return;
	(void)setvalkey((INTBIG)us_aid, VAID, us_highlighted, (INTBIG)list,
		VSTRING|VISARRAY|(total<<VLENGTHSH)|VDONTSAVE);

	for(i=0; i<total; i++) efree(list[i]);
	efree((char *)list);
	us_makecurrentobject();
}

/*
 * routine to set the current prototype from the current selection
 */
void us_makecurrentobject(void)
{
	REGISTER NODEPROTO *np, *onp;
	REGISTER GEOM *from;
	HIGHLIGHT high;
	REGISTER VARIABLE *var;

	/* see what is highlighted */
	var = getvalkey((INTBIG)us_aid, VAID, VSTRING|VISARRAY, us_highlighted);

	/* there must be exactly 1 object highlighted */
	if (var == NOVARIABLE || getlength(var) != 1)
	{
		if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
		return;
	}

	/* get information about the highlighted object */
	if (us_makehighlight(((char **)var->addr)[0], &high) != 0) return;
	if ((high.status&HIGHTYPE) != HIGHFROM && (high.status&HIGHTYPE) != HIGHTEXT)
	{
		if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
		return;
	}
	from = high.fromgeom;
	if (from == NOGEOM)
	{
		us_setnodeproto(NONODEPROTO);
		return;
	}

	/* if an arc is selected, show it */
	if (from->entrytype == OBJARCINST)
	{
		us_setarcproto(from->entryaddr.ai->proto, 1);
		if ((us_state&NONPERSISTENTCURNODE) != 0) us_setnodeproto(NONODEPROTO);
		return;
	}

	/* a node is selected */
	np = from->entryaddr.ni->proto;

	/* for icon facets, see if the contents should be used */
	if (np->cellview == el_iconview)
	{
		onp = contentsview(np);
		if (onp != NONODEPROTO) np = onp;
	}
	us_setnodeproto(np);
}

/*
 * routine to convert the string in "str" into the highlight module "high".
 * returns nonzero on error
 */
INTSML us_makehighlight(char *str, HIGHLIGHT *high)
{
	char *pt;
	REGISTER char *keyword;

	pt = str;

	/* look for the "FACET=" keyword */
	keyword = getkeyword(&pt, "=");
	if (keyword == NOSTRING) return(1);
	if (namesame(keyword, "FACET") != 0) return(1);
	if (tonextchar(&pt) != '=') return(1);

	/* get the facet name */
	keyword = getkeyword(&pt, " \t");
	if (keyword == NOSTRING) return(1);
	high->facet = getnodeproto(keyword);
	if (high->facet == NONODEPROTO) return(1);

	/* get the type of highlight */
	keyword = getkeyword(&pt, "=");
	if (keyword == NOSTRING) return(1);
	if (tonextchar(&pt) != '=') return(1);
	if (namesame(keyword, "FROM") == 0)
	{
		/* parse the three values: geom/port/point */
		high->status = HIGHFROM;
		high->fromvar = NOVARIABLE;
		high->fromvarnoeval = NOVARIABLE;
		keyword = getkeyword(&pt, ";");
		if (keyword == NOSTRING) return(1);
		high->fromgeom = (GEOM *)myatoi(keyword);
		if (tonextchar(&pt) != ';') return(1);
		keyword = getkeyword(&pt, ";");
		if (keyword == NOSTRING) return(1);
		high->fromport = (PORTPROTO *)myatoi(keyword);
		if (tonextchar(&pt) != ';') return(1);
		keyword = getkeyword(&pt, ";");
		if (keyword == NOSTRING) return(1);
		high->frompoint = (INTSML)myatoi(keyword);
	} else if (namesame(keyword, "AREA") == 0 || namesame(keyword, "LINE") == 0)
	{
		if (namesame(keyword, "AREA") == 0) high->status = HIGHBBOX; else
			high->status = HIGHLINE;
		keyword = getkeyword(&pt, ",");
		if (keyword == NOSTRING) return(1);
		high->stalx = myatoi(keyword);
		if (tonextchar(&pt) != ',') return(1);
		keyword = getkeyword(&pt, ",");
		if (keyword == NOSTRING) return(1);
		high->stahx = myatoi(keyword);
		if (tonextchar(&pt) != ',') return(1);
		keyword = getkeyword(&pt, ",");
		if (keyword == NOSTRING) return(1);
		high->staly = myatoi(keyword);
		if (tonextchar(&pt) != ',') return(1);
		keyword = getkeyword(&pt, ",");
		if (keyword == NOSTRING) return(1);
		high->stahy = myatoi(keyword);
	} else if (namesame(keyword, "TEXT") == 0)
	{
		/* parse the three values: geom/port/var */
		high->status = HIGHTEXT;
		high->frompoint = 0;
		keyword = getkeyword(&pt, ";");
		if (keyword == NOSTRING) return(1);
		high->fromgeom = (GEOM *)myatoi(keyword);
		if (tonextchar(&pt) != ';') return(1);
		keyword = getkeyword(&pt, ";");
		if (keyword == NOSTRING) return(1);
		high->fromport = (PORTPROTO *)myatoi(keyword);
		if (tonextchar(&pt) != ';') return(1);
		keyword = getkeyword(&pt, ";");
		if (keyword == NOSTRING) return(1);
		if (*keyword == '-') high->fromvarnoeval = high->fromvar = NOVARIABLE; else
		{
			if (high->fromport != NOPORTPROTO)
			{
				high->fromvarnoeval = getvalnoeval((INTBIG)high->fromport, VPORTPROTO, -1, keyword);
			} else if (high->fromgeom != NOGEOM)
			{
				if (high->fromgeom->entrytype == OBJNODEINST)
				{
					high->fromvarnoeval = getvalnoeval((INTBIG)high->fromgeom->entryaddr.ni,
						VNODEINST, -1, keyword);
				} else
				{
					high->fromvarnoeval = getvalnoeval((INTBIG)high->fromgeom->entryaddr.ai,
						VARCINST, -1, keyword);
				}
			} else
			{
				high->fromvarnoeval = getvalnoeval((INTBIG)high->facet, VNODEPROTO, -1, keyword);
			}
			high->fromvar = evalvar(high->fromvarnoeval);
		}
	} else return(1);

	/* parse any additional keywords */
	for(;;)
	{
		if (tonextchar(&pt) == 0) break;
		keyword = getkeyword(&pt, ";,=");
		if (keyword == NOSTRING) return(1);
		if ((namesame(keyword, "SNAP") == 0 || namesame(keyword, "SNAPTAN") == 0 ||
			namesame(keyword, "SNAPPERP") == 0) && tonextchar(&pt) == '=')
		{
			high->status |= HIGHSNAP;
			if (namesame(keyword, "SNAPTAN") == 0) high->status |= HIGHSNAPTAN;
			if (namesame(keyword, "SNAPPERP") == 0) high->status |= HIGHSNAPPERP;
			keyword = getkeyword(&pt, ",");
			if (keyword == NOSTRING) return(1);
			high->snapx = myatoi(keyword);
			if (tonextchar(&pt) != ',') return(1);
			keyword = getkeyword(&pt, ";,");
			if (keyword == NOSTRING) return(1);
			high->snapy = myatoi(keyword);
			continue;
		}
		if (namesame(keyword, "EXTRA") == 0)
		{
			high->status |= HIGHEXTRA;
			continue;
		}
		if (namesame(keyword, "NOBBOX") == 0)
		{
			high->status |= HIGHNOBOX;
			continue;
		}
		return(1);
	}
	return(0);
}

/*
 * routine to convert the highlight in "high" into a string which is returned.
 */
char *us_makehighlightstring(HIGHLIGHT *high)
{

	(void)initinfstr();
	(void)formatinfstr("FACET=%s", describenodeproto(high->facet));
	switch (high->status&HIGHTYPE)
	{
		case HIGHFROM:
			(void)formatinfstr(" FROM=0%lo;", (UINTBIG)high->fromgeom);
			if (high->fromport == NOPORTPROTO) (void)addstringtoinfstr("-1;"); else
				(void)formatinfstr("0%lo;", (UINTBIG)high->fromport);
			(void)formatinfstr("%d", high->frompoint);
			if ((high->status&HIGHEXTRA) != 0) (void)addstringtoinfstr(";EXTRA");
			if ((high->status&HIGHNOBOX) != 0) (void)addstringtoinfstr(";NOBBOX");
			break;
		case HIGHTEXT:
			(void)formatinfstr(" TEXT=0%lo;", (UINTBIG)high->fromgeom);
			if (high->fromport == NOPORTPROTO) (void)addstringtoinfstr("-1;"); else
				(void)formatinfstr("0%lo;", (UINTBIG)high->fromport);
			if (high->fromvar == NOVARIABLE) (void)addstringtoinfstr("-"); else
				(void)addstringtoinfstr(makename(high->fromvar->key));
			break;
		case HIGHBBOX:
			(void)formatinfstr(" AREA=%ld,%ld,%ld,%ld", high->stalx, high->stahx, high->staly, high->stahy);
			break;
		case HIGHLINE:
			(void)formatinfstr(" LINE=%ld,%ld,%ld,%ld", high->stalx, high->stahx, high->staly, high->stahy);
			break;
	}
	if ((high->status&HIGHSNAP) != 0)
	{
		if ((high->status&HIGHSNAPTAN) != 0) (void)addstringtoinfstr(";SNAPTAN="); else
			if ((high->status&HIGHSNAPPERP) != 0) (void)addstringtoinfstr(";SNAPPERP="); else
				(void)addstringtoinfstr(";SNAP=");
		(void)formatinfstr("%ld,%ld", high->snapx, high->snapy);
	}
	return(returninfstr());
}

/*
 * routine to determine the center of the highlighted text described by
 * "high".  The coordinates are placed in (xc, yc), and the appropriate
 * polygon style is placed in "style".
 */
void us_gethightextcenter(HIGHLIGHT *high, INTBIG *xc, INTBIG *yc, INTSML *style)
{
	static POLYGON *poly = NOPOLYGON;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER NODEPROTO *facet;
	XARRAY trans;
	INTBIG newxc, newyc;
	REGISTER INTSML saverot, savetrn;
	REGISTER INTBIG descript, lambda;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(1, us_aid->cluster);

	poly->count = 1;
	poly->style = FILLED;
	if (high->fromvar != NOVARIABLE)
	{
		descript = high->fromvar->textdescript;
		if (high->fromport != NOPORTPROTO)
		{
			ni = high->fromport->subnodeinst;
			saverot = ni->rotation;   savetrn = ni->transpose;
			ni->rotation = ni->transpose = 0;
			portposition(ni, high->fromport->subportproto,
				&poly->xv[0], &poly->yv[0]);
			ni->rotation = saverot;   ni->transpose = savetrn;
			adjustdisoffset((INTBIG)high->fromport, VPORTPROTO, high->facet->tech,
				poly, descript);
			makerot(high->fromport->subnodeinst, trans);
			xformpoly(poly, trans);
		} else if (high->fromgeom != NOGEOM)
		{
			if (high->fromgeom->entrytype == OBJNODEINST)
			{
				ni = high->fromgeom->entryaddr.ni;
				poly->xv[0] = (ni->lowx + ni->highx) / 2;
				poly->yv[0] = (ni->lowy + ni->highy) / 2;
				adjustdisoffset((INTBIG)ni, VNODEINST, ni->proto->tech, poly, descript);
				makerot(ni, trans);
				xformpoly(poly, trans);
			} else
			{
				ai = high->fromgeom->entryaddr.ai;
				poly->xv[0] = (ai->end[0].xpos + ai->end[1].xpos) / 2;
				poly->yv[0] = (ai->end[0].ypos + ai->end[1].ypos) / 2;
				adjustdisoffset((INTBIG)ai, VARCINST, ai->proto->tech, poly, descript);
			}
		} else
		{
			poly->xv[0] = (high->facet->lowx + high->facet->highx) / 2;
			poly->yv[0] = (high->facet->lowy + high->facet->highy) / 2;
			adjustdisoffset((INTBIG)high->facet, VNODEPROTO, high->facet->tech,
				poly, descript);
		}
		*style = poly->style;
	} else if (high->fromport != NOPORTPROTO)
	{
		descript = high->fromport->textdescript;
		ni = high->fromgeom->entryaddr.ni;
		portposition(high->fromport->subnodeinst, high->fromport->subportproto,
			&poly->xv[0], &poly->yv[0]);
		makeangle(ni->rotation, ni->transpose, trans);
		facet = ni->parent;
		lambda = facet->cell->lib->lambda[facet->tech->techindex];
		newxc = (descript&VTXOFF)>>VTXOFFSH;
		if ((descript&VTXOFFNEG) != 0) newxc = -newxc;
		newxc = newxc * lambda / 4;
		newyc = (descript&VTYOFF)>>VTYOFFSH;
		if ((descript&VTYOFFNEG) != 0) newyc = -newyc;
		newyc = newyc * lambda / 4;
		xform(newxc, newyc, &newxc, &newyc, trans);
		poly->xv[0] += newxc;   poly->yv[0] += newyc;
		switch (descript&VTPOSITION)
		{
			case VTPOSCENT:      *style = TEXTCENT;      break;
			case VTPOSBOXED:     *style = TEXTBOX;       break;
			case VTPOSUP:        *style = TEXTBOT;       break;
			case VTPOSDOWN:      *style = TEXTTOP;       break;
			case VTPOSLEFT:      *style = TEXTRIGHT;     break;
			case VTPOSRIGHT:     *style = TEXTLEFT;      break;
			case VTPOSUPLEFT:    *style = TEXTBOTRIGHT;  break;
			case VTPOSUPRIGHT:   *style = TEXTBOTLEFT;   break;
			case VTPOSDOWNLEFT:  *style = TEXTTOPRIGHT;  break;
			case VTPOSDOWNRIGHT: *style = TEXTTOPLEFT;   break;
		}
		*style = rotatelabel(*style, trans);
	} else if (high->fromgeom->entrytype == OBJNODEINST)
	{
		ni = high->fromgeom->entryaddr.ni;
		descript = ni->textdescript;
		poly->xv[0] = (ni->lowx + ni->highx) / 2;
		poly->yv[0] = (ni->lowy + ni->highy) / 2;
		adjustdisoffset((INTBIG)ni, VNODEINST, ni->proto->tech, poly, descript);
		makerot(ni, trans);
		xformpoly(poly, trans);
		*style = poly->style;
	}
	getcenter(poly, xc, yc);
}

/*
 * Routine to determine whether highlight "high" resides on a node, and return
 * that node if so (NONODEINST if not).
 */
NODEINST *us_gethighnodeinst(HIGHLIGHT *high)
{
	if (high->fromport != NOPORTPROTO) return(high->fromport->subnodeinst);
	if (high->fromgeom != NOGEOM && high->fromgeom->entrytype == OBJNODEINST)
		return(high->fromgeom->entryaddr.ni);
	return(NONODEINST);
}

/*
 * Routine to return the text descriptor on highlight "high".
 */
INTBIG us_gethighdescript(HIGHLIGHT *high)
{
	if (high->fromvar != NOVARIABLE)
		return(high->fromvar->textdescript);
	if (high->fromport != NOPORTPROTO)
		return(high->fromport->textdescript);
	if (high->fromgeom->entrytype == OBJNODEINST)
		return(high->fromgeom->entryaddr.ni->textdescript);
	return(0);
}

/*
 * Routine to return the text on highlight "high".
 */
char *us_gethighstring(HIGHLIGHT *high)
{
	REGISTER char *str;
	REGISTER VARIABLE *var;

	var = high->fromvar;
	if (var != NOVARIABLE)
	{
		switch (var->textdescript&VTDISPLAYPART)
		{
			case VTDISPLAYNAME:
				str = describedisplayedvariable(var, -1, -1);
				break;
			case VTDISPLAYVALUE:
			case VTDISPLAYNAMEVALUE:
				if (high->fromvarnoeval != NOVARIABLE) var = evalvar(high->fromvarnoeval);
				str = describedisplayedvariable(var, -1, -1);
				break;
		}
		return(str);
	}
	if (high->fromport != NOPORTPROTO)
		return(us_displayedportname(high->fromport, (us_useroptions&EXPORTLABELS) >> EXPORTLABELSSH));
	if (high->fromgeom->entrytype == OBJNODEINST)
		return(describenodeproto(high->fromgeom->entryaddr.ni->proto));
	return("");
}

/*
 * Routine to examine highlight "high" and return the address and type
 * of the object to which it refers.
 */
void us_gethighaddrtype(HIGHLIGHT *high, INTBIG *addr, INTBIG *type)
{
	if ((high->status&HIGHTYPE) == HIGHTEXT)
	{
		if (high->fromvar != NOVARIABLE)
		{
			if (high->fromport != NOPORTPROTO)
			{
				*addr = (INTBIG)high->fromport;
				*type = VPORTPROTO;
				return;
			}
			if (high->fromgeom == NOGEOM)
			{
				*addr = (INTBIG)high->facet;
				*type = VNODEPROTO;
				return;
			}

			*addr = (INTBIG)high->fromgeom->entryaddr.blind;
			if (high->fromgeom->entrytype == OBJNODEINST)
			{
				*type = VNODEINST;
			} else
			{
				*type = VARCINST;
			}
			return;
		}
		if (high->fromport != NOPORTPROTO)
		{
			*addr = (INTBIG)high->fromport;
			*type = VPORTPROTO;
			return;
		}
		*addr = (INTBIG)high->fromgeom->entryaddr.blind;
		if (high->fromgeom->entrytype == OBJNODEINST) *type = VNODEINST; else
			*type = VARCINST;
		return;
	}
}

/*
 * Routine to return nonzero if highlight "high" is text on a node that must move together.
 * This only happens for "nonlayout text" (a variable on an invisible node) or
 * for exports on an invisible pin.
 */
INTSML us_nodemoveswithtext(HIGHLIGHT *high)
{
	REGISTER NODEINST *ni;

	if (high->status != HIGHTEXT) return(0);
	if (high->fromgeom == NOGEOM) return(0);
	if (high->fromgeom->entrytype == OBJARCINST) return(0);
	ni = high->fromgeom->entryaddr.ni;
	if (ni == NONODEINST) return(0);
	if (ni->proto != gen_invispinprim) return(0);
	if (high->fromvar == NOVARIABLE)
	{
		if (high->fromport != NOPORTPROTO) return(1);
	} else
	{
		if (high->fromport == NOPORTPROTO) return(1);
	}
	return(0);
}

/*
 * Routine to figure out the technology to use for highlight "high".
 */
TECHNOLOGY *us_hightech(HIGHLIGHT *high)
{
	if (high->fromvar != NOVARIABLE)
	{
		if (high->fromport != NOPORTPROTO)
			return(high->fromport->subnodeinst->parent->tech);
		if (high->fromgeom == NOGEOM) return(high->facet->tech);
		if (high->fromgeom->entrytype == OBJNODEINST)
			return(high->fromgeom->entryaddr.ni->parent->tech);
		return(high->fromgeom->entryaddr.ai->parent->tech);
	}
	if (high->fromport != NOPORTPROTO)
		return(high->fromgeom->entryaddr.ni->parent->tech);
	if (high->fromgeom->entrytype == OBJNODEINST)
		return(high->fromgeom->entryaddr.ni->parent->tech);
	if (high->fromgeom->entrytype == OBJARCINST)
		return(high->fromgeom->entryaddr.ai->parent->tech);
	return(NOTECHNOLOGY);
}

TECHNOLOGY *us_getobjectinfo(INTBIG addr, INTBIG type, INTBIG *lx, INTBIG *hx, INTBIG *ly, INTBIG *hy)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER PORTPROTO *pp;
	REGISTER NODEPROTO *np;

	switch (type)
	{
		case VNODEINST:
			ni = (NODEINST *)addr;
			*lx = ni->geom->lowx;   *hx = ni->geom->highx;
			*ly = ni->geom->lowy;   *hy = ni->geom->highy;
			return(ni->parent->tech);
		case VARCINST:
			ai = (ARCINST *)addr;
			*lx = ai->geom->lowx;   *hx = ai->geom->highx;
			*ly = ai->geom->lowy;   *hy = ai->geom->highy;
			return(ai->parent->tech);
		case VPORTPROTO:
			pp = (PORTPROTO *)addr;
			ni = pp->subnodeinst;
			*lx = ni->geom->lowx;   *hx = ni->geom->highx;
			*ly = ni->geom->lowy;   *hy = ni->geom->highy;
			return(pp->parent->tech);
		case VNODEPROTO:
			np = (NODEPROTO *)addr;
			*lx = np->lowx;   *hx = np->highx;
			*ly = np->lowy;   *hy = np->highy;
			return(np->tech);
	}
	return(NOTECHNOLOGY);
}

void us_drawfacetvariable(VARIABLE *var, NODEPROTO *np)
{
	REGISTER INTBIG i, displaytotal;
	REGISTER VARIABLE *thisvar;
	REGISTER WINDOWPART *w;
	static POLYGON *poly = NOPOLYGON;

	/* get polygon */
	if (poly == NOPOLYGON) poly = allocstaticpolygon(4, us_aid->cluster);

	displaytotal = tech_displayablefacetvars(np, NOWINDOWPART);
	for(i = 0; i < displaytotal; i++)
	{
		thisvar = tech_filldisplayablefacetvar(np, poly, NOWINDOWPART, 0);
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if (w->curnodeproto != np) continue;
			(*us_displayroutine)(poly, w);
		}
	}
}

void us_undrawfacetvariable(VARIABLE *var, NODEPROTO *np)
{
	INTBIG x, y;
	INTSML tsx, tsy;
	REGISTER INTSML font;
	REGISTER INTBIG tf, objwid, objhei, sea, slx, shx, sly, shy, nudge;
	INTBIG lx, hx, ly, hy;
	REGISTER TECHNOLOGY *tech;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER GEOM *geom;
	REGISTER char *str;
	REGISTER WINDOWPART *w;
	extern GRAPHICS us_ebox;

	if ((var->type&VDISPLAY) == 0) return;
	tech = np->tech;
	font = (INTSML)((var->textdescript & VTSIZE) >> VTSIZESH);
	str = describevariable(var, -1, -1);
	for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
	{
		if ((w->state&WINDOWTYPE) != DISPWINDOW) continue;
		if (w->curnodeproto != np) continue;
		tf = truefontsize(font, w, tech);
		screensettextsize(w, tf);

		getdisparrayvarlinepos((INTBIG)np, VNODEPROTO, tech,
			w, var, 0, &x, &y, 1);
		screengettextsize(w, str, &tsx, &tsy);
		objwid = roundfloat((float)tsx / w->scalex);
		objhei = roundfloat((float)tsy / w->scaley);
		slx = x;   shx = slx + objwid;
		sly = y;   shy = sly + objhei;

		/* erase that area */
		lx = slx;   hx = shx;
		ly = sly;   hy = shy;
		if (us_makescreen(&lx, &ly, &hx, &hy, w) != 0) continue;
		screendrawbox(w, (INTSML)lx, (INTSML)hx, (INTSML)ly, (INTSML)hy, &us_ebox);

		/* now redraw everything that touches this box */
		nudge = (INTBIG)(1.0f/w->scalex);
		slx -= nudge;
		shx += nudge;
		nudge = (INTBIG)(1.0f/w->scaley);
		sly -= nudge;
		shy += nudge;
		sea = initsearch(slx, shx, sly, shy, np);
		for(;;)
		{
			geom = nextobject(sea);
			if (geom == NOGEOM) break;
			if (geom->entrytype == OBJNODEINST)
			{
				ni = geom->entryaddr.ni;
				(void)us_drawfacet(ni, LAYERA, el_matid, 3, w);
			} else
			{
				ai = geom->entryaddr.ai;
				(void)us_drawarcinst(ai, LAYERA, el_matid, 3, w);
			}
		}
	}
}
