// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

namespace sipdriver {
using namespace ost;
using namespace std;

Session::Session(timeslot_t ts) :
BayonneSession(&Driver::sip, ts),
TimerPort()
{
#ifndef	WIN32
	if(getppid() > 1)
		logevents = &cout;
#endif
	iface = IF_INET;
	bridge = BR_GATE;
	cid = 0;
	did = 0;
	rtp = NULL;
	dtmf_payload = Driver::sip.dtmf_negotiate;
	data_payload = Driver::sip.data_negotiate;
	dtmf_inband = Driver::sip.dtmf_inband;

	update_pos = false;

	buffer = new unsigned char[Driver::sip.info.framesize];
	lbuffer = new Sample[Driver::sip.info.framecount];
	peer_buffer = NULL;
	peer_codec = NULL;

	if(Driver::peer_encoding == Audio::pcm16Mono)
	{
		peer_buffer = new Sample[Driver::sip.info.framecount];
		peer_codec = AudioCodec::getCodec(Driver::sip.info);
	} 

	codec = AudioCodec::getCodec(Driver::sip.info);
}

Session::~Session()
{
}

void Session::sendDTMFInfo(char digit, unsigned duration)
{
	osip_message_t *message = NULL;
	char dtmf_body[1000];

        snprintf(dtmf_body, sizeof(dtmf_body) - 1,
                "Signal=%c\r\nDuration=%d\r\n", digit, duration);  

	eXosip_lock();
	if(eXosip_call_build_info(did, &message) != 0)
	{
		eXosip_unlock();
		return;
	}

	osip_message_set_body(message, dtmf_body, strlen(dtmf_body));
	osip_message_set_content_type(message, "application/dtmf-relay");
	eXosip_call_send_request(did, message);
	eXosip_unlock();
}

timeout_t Session::getRemaining(void)
{
	return TimerPort::getTimer();
}

void Session::startTimer(timeout_t timer)
{
	TimerPort::setTimer(timer);
	msgport->update();
}

void Session::stopTimer(void)
{
	TimerPort::endTimer();
	msgport->update();
}

tpport_t Session::getLocalPort(void)
{
	Driver *d = (Driver *)(driver);

	return d->rtp_port + (4 * timeslot);
}	

void Session::startRTP(void)
{
	if(rtp)
	{
		slog.error("%s: rtp already started", logname);
		return;
	}

	rtp = new RTPStream(this);

	if(!rtp->addDestination(remote_address, remote_port))
		slog.error("%s: destination not available");

	rtp->start();
}		

void Session::stopRTP(void)
{
	if(!rtp)
		return;

	rtp->setSource(NULL);
	rtp->setSink(NULL);
	rtp->setTone(NULL);
	delete rtp;
	rtp = NULL;
}

void Session::makeIdle(void)
{
	update_pos = false;
	if(offhook)
		sipHangup();
	BayonneSession::makeIdle();
	dtmf_inband = Driver::sip.dtmf_inband;
	dtmf_payload = Driver::sip.dtmf_negotiate;
	data_payload = Driver::sip.data_negotiate;
}

bool Session::enterSeize(Event *event)
{
	char buf[128];
	char cref[64];
	const char *cp, *p, *to, *from = NULL;
	const char *caller;
	char *sp;
	const char *svr;
	osip_message_t *invite = NULL;
	unsigned len;
	Name *scr = ScriptInterp::getName();
	Registry *reg = NULL;
	ScriptImage *img = getImage();
	char rbuf[16];
	char sdp[512];
	const char *rtpmap;
	const char *route;
	char cbuf[65];
	const char *lp = driver->getLast("localip");

	switch(event->id)
	{
	case ENTER_STATE:
		cp = getSymbol("session.dialed");
		if(!cp)
		{
			event->id = DIAL_FAILED;
			return false;
		}
		route = Driver::sip.getLast("outbound");
		if(!strchr(cp, '@'))
		{
			snprintf(buf, sizeof(buf), "uri.%s", cp);
			reg = (Registry *)img->getPointer(buf);
			if(reg && !strnicmp(reg->type, "ext", 3))
			{
				snprintf(buf, sizeof(buf), "sip:%s@%s",
					reg->userid, reg->proxy);
				if(!route)
					route = reg->proxy;
				sp = strrchr(buf, ':');
				if(sp && !stricmp(sp, ":5060"))
					*sp = 0;
				setConst("session.uri_remote", buf);
				goto dialing;
			}
		}
		if(!strchr(cp, '@'))
			snprintf(buf, sizeof(buf), "sip:%s@%s", cp, route);
		else 
		{
			if(!strnicmp(cp, "sip:", 4))
				cp += 4;
			snprintf(buf, sizeof(buf), "sip:%s", cp);	
		}
		if(sp && stricmp(sp, ":5060"))
			sp = 0;
		setConst("session.uri_remote", buf);
		cp = getSymbol("session.uri_remote");
		cp = strchr(cp, '@');
		if(cp)
		{
			snprintf(buf, sizeof(buf), "sip.%s", ++cp);
			reg = (Registry *)img->getPointer(buf);
		}
dialing:
		if(reg)
		{
			if(!route)
				route = reg->proxy;
			snprintf(rbuf, sizeof(rbuf), "sipreg.%d", reg->regid);
			setConst("session.registry", rbuf);
			setConst("session.uri_local", reg->contact);
			setConst("session.ip_local", lp);
			if(reg->address)
			{
				InetAddress host(reg->address);
				snprintf(cbuf, sizeof(cbuf), "%s",
					inet_ntoa(host.getAddress()));
				setConst("session.ip_public", cbuf);
			}
			else
				setConst("session.ip_public", lp);
		}
		else
		{
			snprintf(buf, sizeof(buf), "sip:anon@%s",
				Driver::sip.getLast("interface"));
			setConst("session.uri_local", buf);
			setConst("session.ip_local", lp);
			setConst("session.ip_public", lp);
		}
			
		to = getSymbol("session.uri_remote");
		from = getSymbol("session.uri_local");

		if(!route)
		{
			route = strchr(to, '@');
			if(route)
				++route;
		}

		if(!strnicmp(route, "sip:", 4))
			route += 4;

		snprintf(buf, sizeof(buf), "<sip:%s;lr>", route);

		slog.debug("invite %s from %s using %s", to, from, route);
		eXosip_lock();
		if(eXosip_call_build_initial_invite(&invite, 
			(char *)to, (char *)from, buf, "Bayonne Call"))
		{
			eXosip_unlock();
failure:
			slog.error("sip: invite invalid");
			event->id = DIAL_FAILED;
			return false;
		}
		eXosip_unlock();
		cp = Driver::sip.getLast("localip");
		data_payload = Driver::sip.data_negotiate;
		dtmf_payload = Driver::sip.dtmf_negotiate;
		switch(Driver::sip.info.encoding)
		{
		case mulawAudio:
			data_payload = 0;
			rtpmap = "a=rtpmap:0 PCMU/8000/1";
			break;
		case alawAudio:
			data_payload = 8;
			rtpmap = "a=rtpmap:8 PCMA/8000/1";
			break;
		case gsmVoice:
			data_payload = 3;
			rtpmap = "a=rtpmap:3 GSM/8000/1";
			break;
		}			
		if(dtmf_payload)
                        snprintf(sdp, sizeof(sdp),
                                "v=0\r\n"
                                "o=bayonne 0 0 IN IP4 %s\r\n"
                                "s=call\r\n"
                                "c=IN IP4 %s\r\n"
                                "t=0 0\r\n"
                                "m=audio %d RTP/AVP %d %d\r\n"
                                "%s\r\n"
				"a=rtpmap:%d telephone-events/8000\r\n",
				cp, cp, 
				getLocalPort(), data_payload, dtmf_payload,
				rtpmap, dtmf_payload);

		else
			snprintf(sdp, sizeof(sdp),
				"v=0\r\n"
				"o=bayonne 0 0 IN IP4 %s\r\n"
				"s=call\r\n"
				"c=IN IP4 %s\r\n"
				"t=0 0\r\n"
				"m=audio %d RTP/AVP %d\r\n"
				"%s\r\n",
				cp, cp, 
				getLocalPort(), data_payload, 
				rtpmap);

		setSymbol("session.rtpmap", rtpmap);

		eXosip_lock();
		osip_message_set_body(invite, sdp, strlen(sdp));
		osip_message_set_content_type(invite, "application/sdp");
		
		offhook = true;
		setSid();
		snprintf(cref, sizeof(cref), "%s@%s",
			var_sid, inet_ntoa(local_address.getAddress()));
		setConst("session.callref", cref);
		cid = eXosip_call_send_initial_invite(invite);
		if(cid > 0)
			eXosip_call_set_reference(cid, cref);
		eXosip_unlock();
		if(cid <= 0)
			goto failure;
		startTimer(atol(Driver::sip.getLast("invite")));
		return true;
	case CALL_CONNECTED:
		startRTP();
		setRunning();
		return true;
	case CALL_ANSWERED:
		setState(STATE_PICKUP);
		return true;
	case CALL_RINGING:
		stopTimer();
		return true;
	case TIMER_EXPIRED:
		offhook = false;
		return false;
	}
	return false;
}

bool Session::enterHangup(Event *event)
{
	switch(event->id)
	{
	case TIMER_EXPIRED:
		sipHangup();
	}

	update_pos = false;
	return false;
}

bool Session::enterRecord(Event *event)
{
	switch(event->id)
	{
	case ENTER_STATE:
		audio.record(state.audio.list[0], state.audio.mode, state.audio.note);
		update_pos = true;
		if(!audio.isOpen())
		{
			slog.error("%s: audio file access error", logname);
                        error("no-files");
                        setRunning();
			return true;
		}
		rtp->setSink(&audio, state.audio.total);
		return false;
	}
	return false;
}

bool Session::enterPlay(Event *event)
{
	switch(event->id)
	{
	case AUDIO_IDLE:
		if(!Driver::sip.audio_timer)
			return false;
		startTimer(Driver::sip.audio_timer);
		return true;
	case ENTER_STATE:
		if(state.audio.mode == Audio::modeReadAny)
			update_pos = false;
		else
			update_pos = true;
		audio.play(state.audio.list, state.audio.mode);
		if(!audio.isOpen())
		{
                	slog.error("%s: audio file access error", logname);
                	error("no-files");
			setRunning();
			return true;
		}
		rtp->setSource(&audio);		
		return false;
	}
	return false;
}

bool Session::enterPickup(Event *event)
{
	if(event->id == ENTER_STATE)
	{
		offhook = true;
		startRTP();
		startTimer(driver->getPickupTimer());
		return true;
	}
	else if(event->id == CALL_ACCEPTED)
	{
		startTimer(Driver::sip.accept_timer);
		return true;
	}

	return false;
}

bool Session::enterTone(Event *event)
{
	if(event->id == ENTER_STATE && audio.tone)
		rtp->setTone(audio.tone);

	return false;
}

bool Session::enterXfer(Event *event)
{
	osip_message_t *refer;
	int rtn;

	switch(event->id)
	{
	case ENTER_STATE:
		eXosip_lock();
		eXosip_call_build_refer(did, (char *)state.url.ref, &refer);
		rtn = eXosip_call_send_request(did, refer);
		eXosip_unlock();
		if(!rtn)
		{
			setState(STATE_HANGUP);
			return true;
		}
		event->errmsg = "transfer-invalid";
		event->id = ERROR_STATE;
		return false;
	}	
	return false;
}

void Session::clrAudio(void)
{
	if(rtp)
	{
		rtp->setSource(NULL);
		rtp->setTone(NULL);
	}

	if(audio.isOpen() && update_pos)
	{
		audio.getPosition(audio.var_position, 12);
		update_pos = false;
	}
	audio.cleanup();
}

void Session::sipHangup(void)
{
	eXosip_lock();
	eXosip_call_terminate(cid, did);
	eXosip_unlock();
	stopRTP();
	offhook = false;
}

timeout_t Session::getToneFraming(void)
{
	return Driver::sip.info.framing;
}

const char *Session::checkAudio(bool live)
{
	audio.libext = ".au";

	switch(Driver::sip.info.encoding)
	{
	case alawAudio:
		audio.libext = ".al";
		break;
	case gsmVoice:
		if(!audio.extension)
			audio.extension = ".gsm";
		audio.libext = ".gsm";
		break;
	}

	if(!audio.extension)
		audio.extension = ".au";

	if(audio.encoding == unknownEncoding)
		audio.encoding = Driver::sip.info.encoding;
	
	if(!live)
	{
		if(!audio.framing)
			audio.framing = 10;
		return NULL;
	}

	audio.framing = Driver::sip.info.framing;
	if(audio.encoding != Driver::sip.info.encoding)
		return "unsupported audio format";

	return NULL;
}

void Session::postAudio(Encoded encoded)
{
	if(!peer)
		return;

	if(Driver::peer_encoding == pcm16Mono)
	{
		peer_codec->decode(peer_buffer, encoded, Driver::sip.info.framecount);
		encoded = (Encoded)peer_buffer;
	}

	peer->peerAudio(encoded);
}
	
bool Session::peerAudio(Encoded encoded)
{
	if(Driver::peer_encoding == pcm16Mono)
	{
		codec->encode((Linear)encoded, buffer, Driver::sip.info.framecount);
		encoded = buffer;
	}

	enter();
	if(!peer || !rtp || rtp->source)
	{
		leave();
		return false;
	}

	rtp->peerAudio(encoded);
	leave();
	return true;
}
	
		
} // namespace
