/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 *
 */
/*
 *	Lots of modifications (new guts, more or less..) by
 *	Matti Aarnio <mea@nic.funet.fi>  (copyright) 1992-1995
 */

#include "hostenv.h"
#include <stdio.h>
#include <ctype.h>
#include <syslog.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "mail.h"
#include "scheduler.h"
#include NDIR_H

#include "prototypes.h"

static void unvertex __((struct vertex *));
static struct vertex *findvertex __((long, long, int*));
static int ctlowner __((struct ctlfile *));
static void vtxupdate __((struct vertex *, int));
static void expaux __((struct vertex *, int, char *));

extern char *strchr();
extern char *strrchr();
extern int cistrcmp();
extern int ermdir();
extern int rmdir();
extern int eunlink();
extern struct spblk *sp_fhead(), *sp_fnext();
extern time_t time();

/*
 * Parse a diagnostic string from a transport agent, and do whatever is
 * necessary to update our model of the world to reflect the new reality.
 *
 *	#hungry
 *	# other debug comment
 *	18453/3527\tIETF-NOTARY\tok Quick SMTP connection!
 *	18453/3527\tIETF-NOTARY\tdeferred Unable to contact host!
 *	18453/3527\tIETF-NOTARY\terror Unknown user name "fred".
 *	18453/3527\tIETF-NOTARY\tretryat [+]NNNN circuit unavailable.
 */

/* dispatch table for diagnostic types */

static int u_ok       __((struct vertex *, long, long, long, char*, char*));
static int u_deferred __((struct vertex *, long, long, long, char*, char*));
static int u_error    __((struct vertex *, long, long, long, char*, char*));
static int u_retryat  __((struct vertex *, long, long, long, char*, char*));

static struct diagcodes {
	char	*name;
	int	(*fcn)();
} diags[] = {
		{	"ok",		u_ok		},
		{	"deferred",	u_deferred	},
		{	"error",	u_error		},
		{	"retryat",	u_retryat	},
		{	NULL,		NULL		}
};

void
update(fd, diagnostic)
	int fd;
	char	*diagnostic;
{
	register char	*cp;
	char	*type, *message, *notary;
	long	offset;
	long	inum;
	int	index;
	struct vertex *vp;
	struct diagcodes *dcp;
	struct procinfo *proc = &cpids[fd];

	if (*diagnostic == 0) {
	  /* Lone newline.. */
	  if (proc->tofd >= 0 &&
	      proc->hungry == 0) {
	    proc->hungry = 1;
	    time(&proc->hungertime);
	    ++hungry_childs;
	  }
	  /* Pick next, and feed it! */
	  pick_next_vertex(proc);
	  if (proc->hungry)
	    feed_child(proc);
	  return;
	}
	if (*diagnostic == '#' /* DEBUG comment.. */) {
	  if (strncmp(diagnostic,"#hungry",7)==0) {
	    if (proc->tofd >= 0 &&
		proc->hungry == 0) {
	      proc->hungry = 1;
	      time(&proc->hungertime);
	      ++hungry_childs;
	    }
	    /* Pick next, and feed it! */
	    pick_next_vertex(proc);
	    if (proc->hungry)
	      feed_child(proc);
	    return;
	  }
	  fprintf(stderr, "%s DBGdiag: %s\n", timestring(), diagnostic);
	  return;
	}
	inum = atol(diagnostic);
	if ((cp = strchr(diagnostic, '/')) == NULL) {
	  fprintf(stderr, "%s Misformed diagnostic: %s\n",
		  timestring(), diagnostic);
	  return;
	}
	offset = atol(++cp);
	if ((cp = strchr(cp, '\t')) == NULL) {
	  fprintf(stderr, "%s Misformed diagnostic: %s\n",
		  timestring(), diagnostic);
	  return;
	}
	notary = ++cp;
	if ((cp = strchr(cp, '\t')) == NULL) {
	  fprintf(stderr, "%s Misformed diagnostic: %s\n",
		  timestring(), diagnostic);
	  return;
	}
	*cp = 0; /* Trailing TAB after notary string, NULL it */
	type = ++cp;
	while (*cp != '\0' && isascii(*cp) && !isspace(*cp))
	  ++cp;
	if (*cp == '\0') {
	  message = NULL;
	} else {
	  *cp++ = '\0';
	  message = cp;
	}
	if (verbose)
	  printf("diagnostic: %ld/%ld\t%s\t%s\n",inum,offset,notary,type);

	if ((vp = findvertex(inum, offset, &index)) == NULL)
	  return;
	vp->attempts += 1;

	if (vp->notary != NULL)
	  free(vp->notary);
	vp->notary = NULL;
	if (*notary)
	  vp->notary = strsave(notary);

	/* Select function by the type name: ok/error/retryat/deferred */

	for (dcp = &diags[0]; dcp->name != NULL; ++dcp) {
	  if (cistrcmp(dcp->name, type) == 0) {
	    (dcp->fcn)(vp, index, inum, offset, notary, message);
	    break;
	  }
	}
	if (dcp->name == NULL)
	  fprintf(stderr, "%s Unknown diagnostic type ignored: %s\n",
		  timestring(), type);
}


/*
 * Deallocate a control file descriptor.
 */

void
unctlfile(cfp)
	struct ctlfile *cfp;
{
	struct spblk *spl;
	char	path[MAXPATHLEN+1];

	reporterrs(cfp);
#ifdef	LOG_INFO
	syslog(LOG_INFO, "%s: done", cfp->mid);
#endif	/* LOG_INFO */
	eunlink(cfp->mid);
	if (verbose)
	  printf("%s: unlink %s", cfp->logident, cfp->mid);

	sprintf(path, "../%s/%s", QUEUEDIR, cfp->mid);
	eunlink(path);
	if (verbose)
	  printf("   and %s/\n", path);

	if (cfp->vfp != NULL) {
	  fseek(cfp->vfp, 0, 2);
	  fprintf(cfp->vfp,
		  "scheduler done processing %s\n", cfp->mid);
	  fclose(cfp->vfp);
	}
	spl = sp_lookup(cfp->id, spt_mesh[L_CTLFILE]);
	if (spl != NULL)
	  sp_delete(spl, spt_mesh[L_CTLFILE]);
	if (cfp->contents != NULL)
	  free(cfp->contents);
	free((char *)cfp);
}

static void unvertex(vp)
	register struct vertex *vp;
{
	int	i, remove;

	if (vp->ngroup > 0)
	  return;

	if (vp->proc && vp->proc->vertex == vp)
	  /* Pick next, but don't feed it (yet)! */
	  pick_next_vertex(vp->proc);

	for (i = 0; i < SIZE_L; ++i) {
	  if (vp->next[i] != NULL)
	    vp->next[i]->prev[i] = vp->prev[i];
	  if (vp->prev[i] != NULL)
	    vp->prev[i]->next[i] = vp->next[i];
	  if (i == L_CTLFILE)
	    continue;
	  remove = 0;
	  if (vp->orig[i]->link == vp)
	    if ((vp->orig[i]->link = vp->next[i]) == NULL)
	      remove = 1;
	  if (vp->orig[i]->lastlink == vp)
	    if ((vp->orig[i]->lastlink = vp->prev[i]) == NULL)
	      remove = 1;
	  if (remove) {
	    vp->orig[i]->linkcnt -= 1;
	    unweb(i, vp->orig[i]);
	    vp->orig[i] = NULL;
	  }
	}

	if (vp->cfp->head == vp)
	  if ((vp->cfp->head = vp->next[L_CTLFILE]) == NULL) 
	    unctlfile(vp->cfp);

	web_disentangle(vp); /* does also unthread() */

	if (vp->message != NULL)
	  free(vp->message);
	if (vp->notary != NULL)
	  free(vp->notary);
	free((char *)vp);
}

static struct vertex *findvertex(inum, offset, idx)
	long	inum;
	long	offset;
	int	*idx;
{
	struct spblk *spl;
	struct ctlfile *cfp;
	struct vertex *vp;
	int	i;

	/* It is NOT POSSIBLE to cache cfp, based on the inum */
	spl = sp_lookup((u_long)inum, spt_mesh[L_CTLFILE]);
	if (spl == NULL || (cfp = (struct ctlfile *)spl->data) == NULL) {
	  fprintf(stderr, "%s: cannot find control file for %ld!\n",
		  progname, inum);
	  return NULL;
	}
	for (i = 0; i < cfp->nlines; ++i) {
	  if (cfp->offset[i] == offset) {
	    *idx = i;
	    break;
	  }
	}
	if (i >= cfp->nlines) {
	  fprintf(stderr,
		  "%s: unknown address offset %ld in control file %ld!\n",
		  progname, offset, inum);
	  return NULL;
	}
	for (vp = cfp->head; vp != NULL; vp = vp->next[L_CTLFILE])
	  for (i = 0; i < vp->ngroup; ++i)
	    if (vp->index[i] == *idx)
	      return vp;
	fprintf(stderr,
		"%s: multiple processing of address at %ld in control file %ld!\n",
		progname, offset, inum);
	return NULL;
}

/*
 * To implement the CF_OBSOLETES command, we need to map a message-id to
 * a control file structure, unlink the unprocessed addresses from the web,
 * and physically get rid of the message.  We also need to implement some
 * form of security here; same file ownership and error return address might
 * be a good approximation.
 */

static int ctlowner(cfp)
	struct ctlfile *cfp;
{
	char *path;
	struct stat stbuf;
	static int nope = -9;

	if (cfp->mid == NULL)
	  abort();
	path = emalloc(5+strlen(cfp->mid)+sizeof QUEUEDIR);
	sprintf(path, "../%s/%s", QUEUEDIR, cfp->mid);
	if (stat(path, &stbuf) < 0)
	  return --nope;
	free(path);
	return stbuf.st_uid;
}

void
deletemsg(msgid, curcfp)
	char *msgid;
	struct ctlfile *curcfp;
{
	struct ctlfile *cfp = NULL;
	struct spblk *spl;

	/* map message id to ctl structure */
	for (spl = sp_fhead(spt_mesh[L_CTLFILE]); spl != NULL ;
	     spl = sp_fnext(spl)) {
	  cfp = (struct ctlfile *)spl->data;
	  /* XX: message-id comparison is a Hard Problem. Approximate. */
	  if (strcmp(cfp->logident, msgid) == 0)
	    break;
	}
	if (spl == NULL)
		return;
	/* security checks */
	/* XX: address comparison is also a Hard Problem... sigh */
	if ((cfp->erroraddr == NULL && curcfp->erroraddr != NULL)
	    || (cfp->erroraddr != NULL && curcfp->erroraddr == NULL)
	    || (cfp->erroraddr != curcfp->erroraddr
		&& strcmp(cfp->erroraddr, curcfp->erroraddr) != 0))
		return;
	if (ctlowner(cfp) != ctlowner(curcfp))
		return;
	/*
	 * It might be useful to return a report about what happened, but
	 * for mailing lists this is dangerous.  Let's not, until we can
	 * test for some 'return-receipt-requested' flag.
	 */

#ifdef	LOG_INFO
	syslog(LOG_INFO, "%s: obsoleted by %s", cfp->mid, curcfp->mid);
#endif	/* LOG_INFO */

	/*
	 * unvertex() will do unctlfile() on the last vertex, hence
	 * this strange way of doing the unlink.
	 */
	while (cfp->head->next[L_CTLFILE] != NULL) {
	  cfp->head->ngroup = 0;
	  unvertex(cfp->head);
	}
	cfp->head->ngroup = 0;
	unvertex(cfp->head);
}


/* Lifted from BIND res/res_debug.c */
/*
 * Return a mnemonic for a time to live
 */
char *
saytime(value, buf, shortform)
	u_long value;
	char *buf;
	int shortform;
{
	int secs, mins, hours, fields = 0;
	register char *p;

	p = buf;

	secs = value % 60;
	value /= 60;
	mins = value % 60;
	value /= 60;
	hours = value % 24;
	value /= 24;

#define	PLURALIZE(x)	x, (x == 1) ? "" : "s"
	while (*p) ++p;
	if (value) {
	  if (shortform)
	    sprintf(p, "%ldd", value);
	  else
	    sprintf(p, "%ld day%s", PLURALIZE(value));
	  ++fields;
	  while (*++p);
	}
	if (hours) {
	  if (shortform)
	    sprintf(p, "%dh", hours);
	  else {
	    if (value && p != buf)
	      *p++ = ' ';
	    sprintf(p, "%d hour%s", PLURALIZE(hours));
	  }
	  ++fields;
	  while (*++p);
	}
	if (mins && fields < 2) {
	  if (shortform)
	    sprintf(p, "%dm", mins);
	  else {
	    if ((hours || value) && p != buf)
	      *p++ = ' ';
	    sprintf(p, "%d min%s", PLURALIZE(mins));
	  }
	  while (*++p);
	}
	if (secs && fields < 2) {
	  if (shortform)
	    sprintf(p, "%ds", secs);
	  else {
	    if ((mins || hours || value) && p != buf)
	      *p++ = ' ';
	    sprintf(p, "%d sec%s", PLURALIZE(secs));
	  }
	  while (*++p);
	}
	*p = '\0';
	return buf;
}

/*
 * vtxupdate() -- delete the vertex matching our (sub)index
 *
 */

static void vtxupdate(vp, index)
	struct vertex *vp;
	int index;
{
	int i;

	for (i = 0; i < vp->ngroup; ++i)
	  if (vp->index[i] == index) {
	    for (++i; i < vp->ngroup; ++i)
	      vp->index[i-1] = vp->index[i];
	    vp->ngroup -= 1;
	    if (vp->ngroup <= 0)
	      unvertex(vp);
	    break;
	  }
}

static void expaux(vp, index, buf)
	struct vertex *vp;
	int index;
	char *buf;
{
	int i;

	/* Report expiry */
	for (i = 0 ; i < vp->ngroup; ++i)
	  if (vp->index[i] == index)
	    msgerror(vp, vp->cfp->offset[index], buf);

	/* Log something into the scheduler log */
	fprintf(stderr, "%s %s: %s/%s from %s %s\n", timestring(), progname,
		vp->orig[L_CHANNEL]->name, vp->orig[L_HOST]->name,
		vp->cfp->erroraddr == NULL ? "?" : vp->cfp->erroraddr, buf);

	/* Delete this vertex from scheduling datasets */
	vtxupdate(vp, index);
}

void
expire(vp, index)
	struct vertex *vp;
	int index;
{
	int i;
	char *emsg, buf[BUFSIZ];
	static char *fmt = "\r%s, problem was:\r%s";

	if (vp->notary == NULL) {
	  sprintf(buf,"%s\001%s\001%s",
		  "**ADDR**", /* XX: recipient address! */
		  "failed",
		  "500 (Expired after");
	  saytime((u_long)(vp->ce.expiry - vp->cfp->ctime), buf, 0);
	  strcat(buf,")");
	  vp->notary = strsave(buf);
	}

	strcpy(buf, "expired after ");
	saytime((u_long)(vp->ce.expiry - vp->cfp->ctime), buf, 0);

	if (vp->message != NULL && *(vp->message) != '\0') {
	  emsg = emalloc(strlen(buf) + strlen(vp->message) + strlen(fmt));
	  sprintf(emsg, fmt, buf, vp->message);
	} else
	  emsg = buf;

	if (index < 0) {
	  for (i = 0 ; i < vp->ngroup; ++i)
	    expaux(vp, vp->index[i], emsg);
	} else
	  expaux(vp, index, emsg);

	if (emsg != buf)
	  free(emsg);
}

/*ARGSUSED*/
static int u_ok(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	/* printf("%s: %d/%d/%s/ok %s\n", vp->cfp->logident, inum, offset,
	   notary, message ? message : "-"); */
	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: ok %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  message == NULL ? "(sent)" : message);
	}

	/* XX: Save/process info regarding delivery receipts! */

	/* Delete this vertex from scheduling datasets */
	vtxupdate(vp, index);
	return 1;
}

static int u_deferred(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	/* printf("%s: %d/%d/%s/deferred %s\n", vp->cfp->logident,
	   inum, offset, notary, message ? message : "-"); */
	if (message != NULL) {
	  if (vp->message != NULL)
	    free(vp->message);
	  /* fprintf(stderr, "add message '%s' to node %s/%s\n",
	     message, vp->orig[L_CHANNEL]->name,
	     vp->orig[L_HOST]->name); */
	  vp->message = strsave(message);
	}

	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: deferred %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  message == NULL ? "(unknown)" : message);
	}
	/*
	 * Even though we may get several of these per web entry,
	 * the heuristic in reschedule() to ignore the request if
	 * the time is already in the future should help out.
	 */
	reschedule(vp, -1, index);
	return 1;
}

static int u_error(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	if (message == NULL)
	  message = "(unknown)";
	fprintf(stderr,
		"%s %s: %ld/%ld/%s/error %s\n", timestring(),
		vp->ce.command, inum, offset, notary, message);

	msgerror(vp, offset, message);

	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: error %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  message);
	}
	/* Delete this vertex from scheduling datasets */
	vtxupdate(vp, index);
	return 1;
}

/*
 * specify relative (w/ leading +) or absolute (w/o leading +) retry time.
 */

static int u_retryat(vp, index, inum, offset, notary, message)
	struct vertex *vp;
	long	index, inum, offset;
	char	*notary;
	char	*message;
{
	time_t	retrytime;
	char	*cp;

	if (*message == '+') {
	  time(&retrytime);
	  ++message;
	} else
	  retrytime = 0;
	for (cp = message; *cp != '\0' && isdigit(*cp); ++cp)
	  continue;
	if (*cp != '\0')
	  *cp++ = '\0';
	retrytime += atoi(message);
	message = cp;
	if (*message == '\0')
	  message = NULL;
	
	if (message != NULL) {
	  if (vp->message != NULL)
	    free(vp->message);
	  /* fprintf(stderr, "add message '%s' to node %s/%s\n",
	     message, vp->orig[L_CHANNEL]->name,
	     vp->orig[L_HOST]->name); */
	  vp->message = strsave(message);
	}

	if (vp->cfp->vfp != NULL && vp->cfp->contents != NULL) {
	  fseek(vp->cfp->vfp, 0, 2);
	  fprintf(vp->cfp->vfp, "%s: retryat %d %s\n",
		  vp->cfp->contents + offset + 2 + _CFTAG_RCPTPIDSIZE,
		  (int)retrytime, message == NULL ? "(unknown)" : message);
	}
	/*
	 * Even though we may get several of these per web entry,
	 * the heuristic in reschedule() to ignore the request if
	 * the time is already in the future should help out.
	 */
	reschedule(vp, -(int)retrytime, index);
	return 1;
}
