--
-- nssample.sql - A sample meta data table.
--
-- This SQL script shows how you would manually create a database
-- table on the server for use by NaviServer.   In this case we're
-- creating a "meta data" table which will allow NaviPress users
-- to describe local and remote URL's on the net.
--
--
-- $Id: nssample.sql,v 1.2 1995/06/16 18:20:41 jimbo Exp $
--


begin transaction;

--
-- First, we create the "whats_hot" meta data table for describing URLs.
-- A meta data table is a table with a column called "ns_url" of type "text".
-- To use a meta data table, a user in NaviPress finds and interesting
-- page (local or remote) and presses the "Describe" button and selects
-- a NaviServer.  The NaviServer responds with a list of known meta
-- data tables.  Following a meta data table link results in an
-- Entry of Update form for the meta data table with the URL filled in.
--
create table whats_hot (
	ns_url		text not null unique,
	why		text not null,
	how		integer default 5
);

--
-- Add an index to speed up searches on the whats_hot table based
-- on the URL, e.g., "how hot is the http://foo.com/bar.htm URL?"
--
create index whats_hot_index_by_url on whats_hot using btree(ns_url);


--
-- Add an index to speed up searches on the whats_hot table based
-- on the hotness, e.g., "which URL's are hotter than 5?"
--
create index whats_hot_index_by_how on whats_hot using btree(how);


--
-- Once the table is created, we must add information about it into
-- the ns_tables table for it to be accessed by NaviServer.
-- Specifing table_ismeta = 't' makes the table show up on the
-- meta data table picker.  Specifing table_issearchable = 't'
-- makes the table show up on the "Searchable Tables" page
-- when the "Search" NaviPress button is pressed.
--
insert into ns_tables (
	table_name, table_description, table_ismeta, table_issearchable
) values (
	'whats_hot', 'Sample URL Meta Data Table', 't', 't'
);


--
-- Add column meta data about the "why" column so the automatically
-- generated search, entry, and update forms use a large TEXTAREA
-- instead of the default INPUT TYPE=TEXT for describing why the
-- URL is hot.
--
insert into ns_columns (
	column_table, column_name, column_description,
	column_htmltag_type, column_htmltag_data
) values (
	'whats_hot', 'why', 'Why the URL is Hot!',
	'TEXTAREA', 'ROWS=6 COLS=60'
);


--
-- Add column meta data about the "how" column so the automatically
-- generated search, entry, and update forms use a SELECT widget
-- which only allow selecting a hotness from 1 to 10.
--
insert into ns_columns (
	column_table, column_name, column_description,
	column_htmltag_type, column_htmltag_data
) values (
	'whats_hot', 'how', 'How hot on a scale from 1 to 10!',
	'SELECT', '1 2 3 4 5 6 7 8 9 10'
);


--
-- Add column meta data about the "ns_url" column.
--
insert into ns_columns (
	column_table, column_name, column_description
) values (
	'whats_hot', 'ns_url', 'URL that''s hot!'
);



--
-- By default, the URL's to modify the whats_hot table would be
-- restricted to the "nsadmin" super user or any member of the
-- "system" group.  These ns_permissions table inserts will also
-- allow any member of the "users" group (the default NaviServer
-- user group) full access.  However, because permission_user is
-- still "nsadmin", only "nsadmin" can later change these permissions
-- through the NaviServer.
--
insert into ns_permissions (
	permission_method, permission_url, permission_user, permission_group,
	permission_user_ok, permission_group_ok, permission_world_ok
) values (
	'GET', '/NS/SearchQBF/whats_hot', 'nsadmin', 'users', 't', 't', 'f'
);
insert into ns_permissions (
	permission_method, permission_url, permission_user, permission_group,
	permission_user_ok, permission_group_ok, permission_world_ok
) values (
	'POST', '/NS/SearchQBF/whats_hot', 'nsadmin', 'users', 't', 't', 'f'
);
insert into ns_permissions (
	permission_method, permission_url, permission_user, permission_group,
	permission_user_ok, permission_group_ok, permission_world_ok
) values (
	'POST', '/NS/InsertRow/whats_hot', 'nsadmin', 'users', 't', 't', 'f'
);
insert into ns_permissions (
	permission_method, permission_url, permission_user, permission_group,
	permission_user_ok, permission_group_ok, permission_world_ok
) values (
	'POST', '/NS/UpdateRow/whats_hot', 'nsadmin', 'users', 't', 't', 'f'
);
insert into ns_permissions (
	permission_method, permission_url, permission_user, permission_group,
	permission_user_ok, permission_group_ok, permission_world_ok
) values (
	'POST', '/NS/DeleteRow/whats_hot', 'nsadmin', 'users', 't', 't', 'f'
);


--
-- Grant access on this table the "nsadmin", the Illustra user the
-- NaviServer normally uses to connect to the database.
--
grant all on table whats_hot to nsadmin;


--
-- Finally, fire the ns_tables and ns_permissions alerts so running
-- NaviServers connected to this database will flush their caches.
--
alert ns_tables_alert;
alert ns_permissions_alert;

end transaction;
