package winsite.wizards;

import javax.swing.*;
import javax.swing.border.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;

/**
 * a wizard to ease the insertion of ordered and unordered lists
 * into HTML documents.
 */
public class ListWizardUI extends JDialog implements ActionListener {
	
	private static JTextField txtItems;
	private JLabel lblItems;
	private static JComboBox cmbOrderedStyle, cmbUnorderedStyle;
	private static JRadioButton rdoOrdered, rdoUnordered;
	private JButton btnInsert, btnCancel;
	private JFrame parentFrame;
	private static String listTag = "";
	
	/**
	 * the constructor of the class
	 */
	public ListWizardUI(JFrame parent) {

		super(parent, "List Wizard", true);
		parentFrame = parent;
						
		setResizable(false);

		JPanel pnlPrompt = new JPanel();
		JLabel prompt = new JLabel("Please enter the relevant information");
		prompt.setForeground(Color.black);
		pnlPrompt.add(prompt);

		JPanel pnlItems = new JPanel();
		pnlItems.setLayout(new BoxLayout(pnlItems,BoxLayout.X_AXIS));
		pnlItems.add(Box.createRigidArea(new Dimension(40,0)));
		lblItems = new JLabel("Items (1-100) *:");
		lblItems.setDisplayedMnemonic('i');
		pnlItems.add(lblItems);
		pnlItems.add(Box.createRigidArea(new Dimension(10,0)));
		txtItems = new JTextField(15);
		pnlItems.add(txtItems);
		pnlItems.add(Box.createRigidArea(new Dimension(70,0)));
		
		JPanel pnlOrderedList = new JPanel();
		Border brdEtched = BorderFactory.createEtchedBorder();
		pnlOrderedList.setLayout(new BoxLayout(pnlOrderedList,
				BoxLayout.Y_AXIS));
		pnlOrderedList.add(Box.createRigidArea(new Dimension(0,10)));
		rdoOrdered = new JRadioButton("Ordered List");
		rdoOrdered.setMnemonic('o');
		rdoOrdered.setActionCommand("ordered");
		rdoOrdered.setSelected(true);
		pnlOrderedList.add(rdoOrdered);
		pnlOrderedList.add(Box.createRigidArea(new Dimension(0,10)));
		pnlOrderedList.setBorder(new TitledBorder(brdEtched));
		String[] orderedItems = { 
				"default",
				"1,2,3",
				"a,b,c",
				"A,B,C",
				"i,ii,iii"};
		cmbOrderedStyle = new JComboBox(orderedItems);	
		pnlOrderedList.add(cmbOrderedStyle);
		pnlOrderedList.add(Box.createRigidArea(new Dimension(0,10)));
		
		JPanel pnlUnorderedList = new JPanel();
		pnlUnorderedList.setLayout(new BoxLayout(pnlUnorderedList,
				BoxLayout.Y_AXIS));
		pnlUnorderedList.add(Box.createRigidArea(new Dimension(0,10)));
		rdoUnordered = new JRadioButton("Unordered List");
		rdoUnordered.setMnemonic('u');
		pnlUnorderedList.add(rdoUnordered);
		pnlUnorderedList.add(Box.createRigidArea(new Dimension(0,10)));
		pnlUnorderedList.setBorder(new TitledBorder(brdEtched));
		
		String[] unorderedItems = { 
				"default",
				"circle",
				"disk",
				"square" };
		cmbUnorderedStyle = new JComboBox(unorderedItems);	
		cmbUnorderedStyle.setEnabled(false);
		pnlUnorderedList.add(cmbUnorderedStyle);		
		pnlUnorderedList.add(Box.createRigidArea(new Dimension(0,10)));
		
		// add listeners to activate the relevant combobox
		RadioListener rdoListener = new RadioListener();
		rdoOrdered.addActionListener(rdoListener);
		rdoUnordered.addActionListener(rdoListener);	
		
		// add the radio buttons to a group to ensure only one being set
		ButtonGroup bgroup = new ButtonGroup();
		bgroup.add(rdoOrdered);
		bgroup.add(rdoUnordered);
		
		JPanel pnlStyles = new JPanel();
		pnlStyles.setLayout(new BoxLayout(pnlStyles,BoxLayout.X_AXIS));
		pnlStyles.add(Box.createRigidArea(new Dimension(10,0)));	
		pnlStyles.add(pnlOrderedList);
		pnlStyles.add(Box.createRigidArea(new Dimension(15,0)));	
		pnlStyles.add(pnlUnorderedList);
		pnlStyles.add(Box.createRigidArea(new Dimension(10,0)));	
		
		JPanel pnlButtons = new JPanel();
		
		btnInsert = new JButton("Insert Tags");
		btnInsert.setMnemonic('i');
		btnInsert.setActionCommand("insert");
		btnInsert.addActionListener(this);
		pnlButtons.add(btnInsert);
		btnCancel = new JButton("Cancel");
		btnCancel.setMnemonic('c');
		btnCancel.setActionCommand("cancel");
		btnCancel.addActionListener(this);
		pnlButtons.add(btnCancel);
		
		JPanel pnlCenter = new JPanel();
		pnlCenter.setLayout(new BoxLayout(pnlCenter,BoxLayout.Y_AXIS));
		pnlCenter.setBorder(BorderFactory.createRaisedBevelBorder());
		pnlCenter.add(Box.createRigidArea(new Dimension(0,10)));
		pnlCenter.add(pnlItems);
		pnlCenter.add(Box.createRigidArea(new Dimension(0,10)));
		pnlCenter.add(pnlStyles);
		pnlCenter.add(Box.createRigidArea(new Dimension(0,10)));
		pnlCenter.add(pnlButtons);
		getContentPane().add(pnlCenter);
		
		setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
		pack();
		Dimension dim = this.getPreferredSize();
		Dimension screenSize = 	Toolkit.getDefaultToolkit().
				getScreenSize();
		setLocation((screenSize.width - dim.width)/2,
				(screenSize.height - dim.height)/2);
		SwingUtilities.invokeLater(new Runnable() {
			public void run() {
				txtItems.requestFocus();
			}
		});	
	}
	
	/**
	 * listen for changes in the radio buttons
	 */
	class RadioListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			String action = e.getActionCommand();

			if (action.equals("ordered")) {
				cmbOrderedStyle.setEnabled(true);
				cmbUnorderedStyle.setEnabled(false);
			} else {
				cmbUnorderedStyle.setEnabled(true);
				cmbOrderedStyle.setEnabled(false);
			}
		}
	}	

	/**
	 * the action performed method
	 */
	public void actionPerformed(ActionEvent e) {
		String action = e.getActionCommand();
		String filename;

		if (action.equals("insert")) {
			recordSettings(); 
			if (getListTag().length() > 0) {
				setVisible(false);
			} else {
				// no URL entered - warn user
				JOptionPane.showMessageDialog(parentFrame,
						"You must enter a number between 1 and 100\n" +
						"for the number of list items.",
						"Error",
						JOptionPane.ERROR_MESSAGE);	
			}
		} else if (action.equals("cancel")) {
			setListTag("");
			setVisible(false);
		}
	}	

	/**
	 * record the settings
	 */
	public void recordSettings() {
		String strListTag = "";
		String style;
		int numItems = 0;
		
		try {
			numItems = Integer.parseInt(txtItems.getText().trim());
		} catch (NumberFormatException nfe) {
			setListTag("");
			return;
		}	
		
		if ((numItems <= 0) || (numItems > 100)) {
			setListTag("");
			return;
		}
		
		if (rdoOrdered.isSelected()) {
			// an ordered list is being used - determine style
			strListTag += "\n<OL";
			style = cmbOrderedStyle.getSelectedItem().toString();
			if (!style.equals("default")) {
				strListTag += " TYPE=\"" + style.charAt(0) + "\"";
			}
		}
		
		if (rdoUnordered.isSelected()) {
			// an unordered list is being used - determine style
			strListTag += "\n<UL";
			style = cmbUnorderedStyle.getSelectedItem().toString();
			if (!style.equals("default")) {
				strListTag += " TYPE=\"" + style + "\"";
			}
		}
		
		strListTag += ">\n";
		
		for (int i = 0; i < numItems; i++) {
			strListTag += "\t<LI>\nList item " + (i + 1) + "\n";
		}	
		
		if (rdoOrdered.isSelected()) {
			strListTag += "</OL>\n";
		} else {
			strListTag += "</UL>\n";
		}
		
		setListTag(strListTag);
	}
	
	/**
	 * obtain the list tags
	 */
	public String getListTag() {
		return listTag;
	}
	
	/**
	 * set the list tags
	 */
	public void setListTag(String listTag) {
		this.listTag = listTag;
	}
	
	/**
	 * reinitialise the textfields so they are back
	 * to default when the wizard is displayed next
	 */	
	public static void reInit() {
		cmbOrderedStyle.setSelectedIndex(0);
		cmbOrderedStyle.setEnabled(true);
		cmbUnorderedStyle.setSelectedIndex(0);
		cmbUnorderedStyle.setEnabled(false);
		rdoOrdered.setSelected(true);
		txtItems.setText("");
	}
}