#!perl

# t/manifest-write.t #

#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
#
#   Copyright © 2015 Van de Bugger
#
#   This file is part of perl-Dist-Zilla-Plugin-Manifest-Write.
#
#   perl-Dist-Zilla-Plugin-Manifest-Write is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by the Free Software
#   Foundation, either version 3 of the License, or (at your option) any later version.
#
#   perl-Dist-Zilla-Plugin-Manifest-Write is distributed in the hope that it will be useful, but
#   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
#   PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License along with
#   perl-Dist-Zilla-Plugin-Manifest-Write. If not, see <http://www.gnu.org/licenses/>.
#
#   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

use strict;
use warnings;
use lib 't';

use Test::Routine::Util;
use Test::More;

plan tests => 12;

my $role     = 'ManifestWriteTester';
my $aborting = "Aborting...\n";

#   No source providers are specified, files will be recognized as "3rd party". However, default
#   metainfo providers work: `MANIFEST` and `META.yml` should be recognized properly.
run_tests( 'No source providers', $role, {
    expected => {
        manifest => [
            'MANIFEST     #  metainfo file built by Manifest::Write',
            'META.yml     #  metainfo file built by MetaYAML',
            'dist.ini     # 3rd party file added by GatherDir',
            'lib/Dummy.pm # 3rd party file added by GatherDir',
            # There is no lines "Dummy file added by GatherDir"
        ],
    },
} );

#   Source provider is specified, files will should be recognized as "Dummy" (it is name of
#   distibution).
run_tests( 'Source provider specified', $role, {
    options => {
        source_providers => 'GatherDir',
        #                    ^^^^^^^^^
    },
    expected => {
        manifest => [
            'MANIFEST     #  metainfo file built by Manifest::Write',
            'META.yml     #  metainfo file built by MetaYAML',
            'dist.ini     #     Dummy file added by GatherDir',
            #                   ^^^^^               ^^^^^^^^^
            'lib/Dummy.pm #     Dummy file added by GatherDir',
            #                   ^^^^^               ^^^^^^^^^
        ],
    },
} );

run_tests( 'Multiple source providers in the same line', $role, {
    options => { source_providers => 'GatherDir GenerateFile' },
    #                                 ^^^^^^^^^^^^^^^^^^^^^^
    extra_plugins => [
        [ 'GenerateFile', { filename => 'README', content => 'Yep.', } ],
    ],
    expected => {
        manifest => [
            'MANIFEST     #  metainfo file built by Manifest::Write',
            'META.yml     #  metainfo file built by MetaYAML',
            'README       #     Dummy file built by GenerateFile',
            #                   ^^^^^               ^^^^^^^^^^^^
            'dist.ini     #     Dummy file added by GatherDir',
            #                   ^^^^^               ^^^^^^^^^
            'lib/Dummy.pm #     Dummy file added by GatherDir',
            #                   ^^^^^               ^^^^^^^^^
        ],
    },
} );

#   Specifying any metainfo provider kills all default metainfo providers.
run_tests( 'Explicit metainfo_providers kills default metaproviders', $role, {
    options => { source_providers => 'GatherDir', metainfo_providers => '' },
    #                                                                   ^^
    expected => {
        manifest => [
            'MANIFEST     # 3rd party file built by Manifest::Write',
            #               ^^^^^^^^^^^^^^
            'META.yml     # 3rd party file built by MetaYAML',
            #               ^^^^^^^^^^^^^^
            'dist.ini     #     Dummy file added by GatherDir',
            'lib/Dummy.pm #     Dummy file added by GatherDir',
        ],
    },
} );

run_tests( 'Multiple source providers in different lines', $role, {
    options => { source_providers => [ 'GatherDir', 'GenerateFile' ] },
    #                                ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
    extra_plugins => [
        [ 'GenerateFile', { filename => 'README', content => 'Yep.', } ],
    ],
    expected => {
        manifest => [
            'MANIFEST     #  metainfo file built by Manifest::Write',
            'META.yml     #  metainfo file built by MetaYAML',
            'README       #     Dummy file built by GenerateFile',
            #                   ^^^^^               ^^^^^^^^^^^^
            'dist.ini     #     Dummy file added by GatherDir',
            #                   ^^^^^               ^^^^^^^^^
            'lib/Dummy.pm #     Dummy file added by GatherDir',
            #                   ^^^^^               ^^^^^^^^^
        ],
    },
} );

#   Adding `PkgVersion` — it modifies Perl module.
run_tests( 'Munger in action', $role, {
    options => { source_providers => 'GatherDir' },
    extra_plugins => [
        'PkgVersion',
        #^^^^^^^^^^         # `PkgVersion` is a file munger.
    ],
    expected => {
        manifest => [
            'MANIFEST     #  metainfo file built by Manifest::Write',
            'META.yml     #  metainfo file built by MetaYAML',
            'dist.ini     #     Dummy file added by GatherDir',
            'lib/Dummy.pm #     Dummy file added by GatherDir and modified by PkgVersion',
            #                                                 ^^^^^^^^^^^^^^^^^^^^^^^^^^
        ],
    },
} );

run_tests( 'Plugin in custom location', $role, {
    options => { source_providers => 'GatherDir =CustomPlugin' },
    extra_plugins => [
        '=CustomPlugin',
    ],
    expected => {
        manifest => [
            'GeneratedFile.txt      #     Dummy file built by =CustomPlugin',
            'MANIFEST               #  metainfo file built by Manifest::Write',
            'META.yml               #  metainfo file built by MetaYAML',
            'dist.ini               #     Dummy file added by GatherDir',
            'lib/Dummy.pm           #     Dummy file added by GatherDir',
            'lib/GeneratedModule.pm #     Dummy file built by =CustomPlugin',
        ],
    },
} );

run_tests( 'Not a FileInjector', $role, {
    options => { source_providers => 'FinderCode' },
    expected => {
        exception => $aborting,
        messages => [
            'FinderCode is not a FileInjector',
        ],
    },
} );

run_tests( 'Not a loaded plugin', $role, {
    options => { source_providers => 'MetaJSON' },
    expected => {
        exception => $aborting,
        messages => [
            'MetaJSON is not a loaded plugin',
        ],
    },
} );

run_tests( 'No strict', $role, {
    options => {
        source_providers => [ 'MetaJSON', 'FinderCode' ],
        strict => -1,
    },
    expected => {
        messages => [],
    },
} );

run_tests( 'strict = 0', $role, {
    options => {
        source_providers => [ 'MetaJSON' ],
        strict => 0,
    },
    expected => {
        # No exception expected.
        messages => [
            'MetaJSON is not a loaded plugin',
        ],
    },
} );

run_tests( 'strict = -1', $role, {
    options => {
        source_providers => [ 'MetaJSON' ],
        strict => -1,
    },
    expected => {
        # No exception expected.
        messages => [
            # And no messages.
        ],
    },
} );

done_testing;

exit( 0 );

# end of file #
