package Pod::Weaver::Section::Changes;

use 5.010;
use Moose;
with 'Pod::Weaver::Role::Section';

use Log::Any '$log';

use Config::INI::Reader;
use CPAN::Changes;
use Moose::Autobox;
use Pod::Elemental;
use Pod::Elemental::Element::Nested;

our $VERSION = '0.01'; # VERSION

sub weave_section {
    $log->trace("-> ".__PACKAGE__."::weave_section()");
    my ($self, $document, $input) = @_;

    my $filename = $input->{filename} || 'file';

    # try to find main package name, from dist.ini. there should be a better
    # way.
    my $ini = Config::INI::Reader->read_file('dist.ini');
    my $main_package = $ini->{_}{name};
    $main_package =~ s/-/::/g;

    # guess package name from filename
    my $package;
    if ($filename =~ m!^lib/(.+)\.(?:pm|pod)$!) {
        $package = $1;
        $package =~ s!/!::!g;
    } else {
        $log->debugf("skipped file %s (not a Perl module)", $filename);
        return;
    }

    if ($package ne $main_package) {
        $log->debugf("skipped file %s (not main module)", $filename);
        return;
    }

    my $changes;
    for my $f (qw/Changes CHANGES ChangeLog CHANGELOG/) {
        if (-f $f) {
            $changes = CPAN::Changes->load($f);
            last;
        }
    }

    unless ($changes) {
        $self->log(["skipped adding CHANGES section to %s (no valid Changes file)", $filename]);
        $log->infof("skipped adding CHANGES section to %s (no valid Changes file)", $filename );
        return;
    }

    my @content;
    for my $rel (reverse $changes->releases) {
        my @rel_changes;
        my $rchanges = $rel->changes;
        for my $cgrp (sort keys %$rchanges) {
            push @rel_changes, Pod::Elemental::Element::Pod5::Command->new({
                command => 'over',
                content => '4',
            });
            for my $c (@{ $rchanges->{$cgrp} }) {
                push @rel_changes, Pod::Elemental::Element::Nested->new({
                    command => 'item',
                    content => '*',
                    children => [Pod::Elemental::Element::Pod5::Ordinary->new({
                        content => ($cgrp ? "[$cgrp] " : "") . $c,
                    })]
                });
            }
            push @rel_changes, Pod::Elemental::Element::Pod5::Command->new({
                command => 'back',
                content => '',
            });
        }

        push @content, Pod::Elemental::Element::Nested->new({
            command => 'head2',
            content => "Version " . $rel->version . " (". $rel->date . ")",
            children => \@rel_changes,
        });
    }

    $document->children->push(
        Pod::Elemental::Element::Nested->new({
            command  => 'head1',
            content  => 'CHANGES',
            children => \@content,
        }),
    );

    $self->log(["added CHANGES section to %s", $filename]);
    $log->infof("added CHANGES section to %s", $filename );

    $log->trace("<- ".__PACKAGE__."::weave_section()");
}

1;
# ABSTRACT: Add a CHANGES POD section


__END__
=pod

=head1 NAME

Pod::Weaver::Section::Changes - Add a CHANGES POD section

=head1 VERSION

version 0.01

=head1 SYNOPSIS

In your C<weaver.ini>:

 [Changes]

=head1 DESCRIPTION

Inserting information in distribution's C<Changes> to main module's POD can be a
convenience for your module users.

Changes is parsed using L<CPAN::Changes> and markup in text entries are
currently assumed to be POD too.

=for Pod::Coverage weave_section

=head1 SEE ALSO

L<Pod::Weaver>

L<CPAN::Changes>

=head1 AUTHOR

Steven Haryanto <stevenharyanto@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2012 by Steven Haryanto.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut

