use Test::More;

BEGIN {
    eval { require XML::Twig };
    if ($@) {
        plan 'skip_all' => 'XML::Twig not installed';
    }
    else {
        Test::More->import('no_plan');
    }
}
use Bio::Phylo::IO 'parse';
use Bio::Phylo::Util::CONSTANT qw':objecttypes looks_like_object';

my $proj = parse(
    '-format' => 'tolweb',
    '-handle' => \*DATA,
    '-as_project' => 1,
);
ok( looks_like_object $proj, _PROJECT_ );

my ($forest) = @{ $proj->get_forests };
ok( looks_like_object $forest, _FOREST_ );

my ($tree) = @{ $forest->get_entities };
ok( looks_like_object $tree, _TREE_ );

{
    my $unnamed = 0;
    $tree->visit(sub{
        my $node = shift;
        $unnamed++ unless $node->get_name;
    });
    ok( $unnamed == 20 );
}

{
    my $hasdesc = 0;
    $tree->visit(sub{
       my $node = shift;
       my ($meta) = @{ $node->get_meta('dc:description') };
       $hasdesc++ if $meta;
    });
    ok( $hasdesc == 1 );
}

__DATA__
<?xml version="1.0" standalone="yes"?>

<TREE>
  <NODE EXTINCT="0" ID="194" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="200" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="85">
    <NAME><![CDATA[Bembidion]]></NAME>
    <DESCRIPTION><![CDATA[A large genus with over 1200 species]]></DESCRIPTION>
    <AUTHORITY><![CDATA[Latreille]]></AUTHORITY>
    <NAMECOMMENT></NAMECOMMENT>
    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
    <AUTHDATE>1802</AUTHDATE>
    <NODES>
      <NODE EXTINCT="0" ID="133816" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
        <NAME><![CDATA[Bembidion (Odontium) Complex]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="367" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133816" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion (Bracteon)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bedel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1879</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1939" ITALICIZENAME="1">
                <NAME><![CDATA[Stylobracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1939" ITALICIZENAME="1">
                <NAME><![CDATA[Litoreobracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1929" ITALICIZENAME="1">
                <NAME><![CDATA[Parabracteon]]></NAME>
                <AUTHORITY><![CDATA[Notman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1939" ITALICIZENAME="1">
                <NAME><![CDATA[Foveobracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1914" ITALICIZENAME="1">
                <NAME><![CDATA[Chrysobracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1939" ITALICIZENAME="1">
                <NAME><![CDATA[Argyrobracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1939" ITALICIZENAME="1">
                <NAME><![CDATA[Conicibracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="1" ISPREFERRED="0" SEQUENCE="7" DATE="null" ITALICIZENAME="0">
                <NAME><![CDATA[Bembidion litorale group]]></NAME>
                <AUTHORITY></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
            <NODES>
              <NODE EXTINCT="0" ID="133793" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="378" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion balli]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1962</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="379" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion foveum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1844</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1939" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion beringi]]></NAME>
                        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1932" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion bryanti]]></NAME>
                        <AUTHORITY><![CDATA[Carr]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1951" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion grahami]]></NAME>
                        <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="133794" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
                <NAME></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="133795" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                    <NAME></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>null</AUTHDATE>
                    <NODES>
                      <NODE EXTINCT="0" ID="394" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                        <NAME><![CDATA[Bembidion punctatostriatum]]></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>1823</AUTHDATE>
                        <OTHERNAMES>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1830" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion sigillare]]></NAME>
                            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion stigmaticum]]></NAME>
                            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                        </OTHERNAMES>
                      </NODE>
                      <NODE EXTINCT="0" ID="133796" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                        <NAME></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>null</AUTHDATE>
                        <NODES>
                          <NODE EXTINCT="0" ID="393" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                            <NAME><![CDATA[Bembidion lapponicum]]></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY><![CDATA[Zetterstedt]]></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>1828</AUTHDATE>
                            <OTHERNAMES>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1880" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion jenisseense]]></NAME>
                                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1844" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion latiusculum]]></NAME>
                                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1916" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion pugetanum]]></NAME>
                                <AUTHORITY><![CDATA[Fall]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                            </OTHERNAMES>
                          </NODE>
                          <NODE EXTINCT="0" ID="133797" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                            <NAME></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>null</AUTHDATE>
                            <NODES>
                              <NODE EXTINCT="0" ID="392" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                                <NAME><![CDATA[Bembidion velox]]></NAME>
                                <DESCRIPTION></DESCRIPTION>
                                <AUTHORITY><![CDATA[Linnaeus]]></AUTHORITY>
                                <NAMECOMMENT></NAMECOMMENT>
                                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                <AUTHDATE>1761</AUTHDATE>
                                <OTHERNAMES>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion evertsi]]></NAME>
                                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1887" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion guentheri]]></NAME>
                                    <AUTHORITY><![CDATA[Seidlitz]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1904" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion bimaculatum]]></NAME>
                                    <AUTHORITY><![CDATA[Uyttenboogaart]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1911" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion nigrescens]]></NAME>
                                    <AUTHORITY><![CDATA[Kuhnt]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1797" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion impressum]]></NAME>
                                    <AUTHORITY><![CDATA[Panzer]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1899" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion semicyaneum]]></NAME>
                                    <AUTHORITY><![CDATA[Meier]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1928" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion moestum]]></NAME>
                                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1798" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion striatum]]></NAME>
                                    <AUTHORITY><![CDATA[Paykull]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                </OTHERNAMES>
                              </NODE>
                              <NODE EXTINCT="0" ID="133798" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                                <NAME></NAME>
                                <DESCRIPTION></DESCRIPTION>
                                <AUTHORITY></AUTHORITY>
                                <NAMECOMMENT></NAMECOMMENT>
                                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                <AUTHDATE>null</AUTHDATE>
                                <NODES>
                                  <NODE EXTINCT="0" ID="391" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                                    <NAME><![CDATA[Bembidion stenoderum]]></NAME>
                                    <DESCRIPTION></DESCRIPTION>
                                    <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                                    <NAMECOMMENT></NAMECOMMENT>
                                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                    <AUTHDATE>1873</AUTHDATE>
                                    <OTHERNAMES>
                                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1965" ITALICIZENAME="1">
                                        <NAME><![CDATA[Bembidion uenoshiba]]></NAME>
                                        <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                                        <COMMENTS></COMMENTS>
                                      </OTHERNAME>
                                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1984" ITALICIZENAME="1">
                                        <NAME><![CDATA[Bembidion mukdense]]></NAME>
                                        <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
                                        <COMMENTS></COMMENTS>
                                      </OTHERNAME>
                                    </OTHERNAMES>
                                  </NODE>
                                  <NODE EXTINCT="0" ID="390" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                                    <NAME><![CDATA[Bembidion carinula]]></NAME>
                                    <DESCRIPTION></DESCRIPTION>
                                    <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                                    <NAMECOMMENT></NAMECOMMENT>
                                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                    <AUTHDATE>1868</AUTHDATE>
                                    <OTHERNAMES>
                                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1929" ITALICIZENAME="1">
                                        <NAME><![CDATA[Bembidion tuberculatum]]></NAME>
                                        <AUTHORITY><![CDATA[Notman]]></AUTHORITY>
                                        <COMMENTS></COMMENTS>
                                      </OTHERNAME>
                                    </OTHERNAMES>
                                  </NODE>
                                </NODES>
                              </NODE>
                            </NODES>
                          </NODE>
                        </NODES>
                      </NODE>
                    </NODES>
                  </NODE>
                  <NODE EXTINCT="0" ID="133799" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                    <NAME></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>null</AUTHDATE>
                    <NODES>
                      <NODE EXTINCT="0" ID="133800" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                        <NAME></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>null</AUTHDATE>
                        <NODES>
                          <NODE EXTINCT="0" ID="384" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                            <NAME><![CDATA[Bembidion argenteolum]]></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY><![CDATA[Ahrens]]></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>1812</AUTHDATE>
                            <OTHERNAMES>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion glabriusculum]]></NAME>
                                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1926" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion trifoveolatipennis]]></NAME>
                                <AUTHORITY><![CDATA[Emden]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1908" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion virens]]></NAME>
                                <AUTHORITY><![CDATA[Schilsky]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1833" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion azureum]]></NAME>
                                <AUTHORITY><![CDATA[Gebler]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1843" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion chalybaeum]]></NAME>
                                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1899" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion amethystinum]]></NAME>
                                <AUTHORITY><![CDATA[Meier]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                            </OTHERNAMES>
                          </NODE>
                          <NODE EXTINCT="0" ID="385" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                            <NAME><![CDATA[Bembidion semenovi]]></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>1965</AUTHDATE>
                          </NODE>
                        </NODES>
                      </NODE>
                      <NODE EXTINCT="0" ID="382" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                        <NAME><![CDATA[Bembidion alaskense]]></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>1962</AUTHDATE>
                        <OTHERNAMES>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1965" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion colvillense]]></NAME>
                            <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                        </OTHERNAMES>
                      </NODE>
                    </NODES>
                  </NODE>
                  <NODE EXTINCT="0" ID="133801" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
                    <NAME></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>null</AUTHDATE>
                    <NODES>
                      <NODE EXTINCT="0" ID="133802" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
                        <NAME></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>null</AUTHDATE>
                        <NODES>
                          <NODE EXTINCT="0" ID="133803" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="367" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                            <NAME></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>null</AUTHDATE>
                            <NODES>
                              <NODE EXTINCT="0" ID="403" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                                <NAME><![CDATA[Bembidion inaequale]]></NAME>
                                <DESCRIPTION></DESCRIPTION>
                                <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                                <NAMECOMMENT></NAMECOMMENT>
                                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                <AUTHDATE>1823</AUTHDATE>
                                <OTHERNAMES>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1848" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion lacustre]]></NAME>
                                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion arenarium]]></NAME>
                                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion opaciceps]]></NAME>
                                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                </OTHERNAMES>
                              </NODE>
                              <NODE EXTINCT="0" ID="404" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                                <NAME><![CDATA[Bembidion levettei]]></NAME>
                                <DESCRIPTION></DESCRIPTION>
                                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                                <NAMECOMMENT></NAMECOMMENT>
                                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                <AUTHDATE>1918</AUTHDATE>
                                <OTHERNAMES>
                                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                                    <NAME><![CDATA[Bembidion carrianum]]></NAME>
                                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                                    <COMMENTS></COMMENTS>
                                  </OTHERNAME>
                                </OTHERNAMES>
                              </NODE>
                            </NODES>
                          </NODE>
                          <NODE EXTINCT="0" ID="401" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                            <NAME><![CDATA[Bembidion conicolle]]></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>1844</AUTHDATE>
                            <OTHERNAMES>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1939" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion baikaloussuricum]]></NAME>
                                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1850" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion conicicolle]]></NAME>
                                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                            </OTHERNAMES>
                          </NODE>
                          <NODE EXTINCT="0" ID="400" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                            <NAME><![CDATA[Bembidion litorale]]></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY><![CDATA[Olivier]]></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>1790</AUTHDATE>
                            <OTHERNAMES>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1915" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion melanotinctum]]></NAME>
                                <AUTHORITY><![CDATA[Wagner, H.]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1877" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion evanescens]]></NAME>
                                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1824" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion elegans]]></NAME>
                                <AUTHORITY><![CDATA[Germar]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1794" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion paludosum]]></NAME>
                                <AUTHORITY><![CDATA[Panzer]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1928" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion krynickii]]></NAME>
                                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1832" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion coeruleum]]></NAME>
                                <AUTHORITY><![CDATA[Krynicki]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1877" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion palmosum]]></NAME>
                                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1912" ITALICIZENAME="1">
                                <NAME><![CDATA[Bembidion nigrans]]></NAME>
                                <AUTHORITY><![CDATA[Barthe]]></AUTHORITY>
                                <COMMENTS></COMMENTS>
                              </OTHERNAME>
                            </OTHERNAMES>
                          </NODE>
                        </NODES>
                      </NODE>
                      <NODE EXTINCT="0" ID="398" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                        <NAME><![CDATA[Bembidion zephyrum]]></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY><![CDATA[Fall]]></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>1910</AUTHDATE>
                        <OTHERNAMES>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1993" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion zephyrium]]></NAME>
                            <AUTHORITY><![CDATA[Bousquet &amp; Larochelle]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion marginosum]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1924" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion tristiculum]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                        </OTHERNAMES>
                      </NODE>
                      <NODE EXTINCT="0" ID="396" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                        <NAME><![CDATA[Bembidion hesperium]]></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY><![CDATA[Fall]]></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>1910</AUTHDATE>
                        <OTHERNAMES>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1910" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion hesperum]]></NAME>
                            <AUTHORITY><![CDATA[Fall]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion binarium]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                        </OTHERNAMES>
                      </NODE>
                      <NODE EXTINCT="0" ID="397" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="367" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                        <NAME><![CDATA[Bembidion lorquinii]]></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>1868</AUTHDATE>
                        <OTHERNAMES>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion tacomae]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                        </OTHERNAMES>
                      </NODE>
                    </NODES>
                  </NODE>
                </NODES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="369" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133816" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion (Ochthedromus)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="511" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="369" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion americanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion illini]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="512" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="369" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bifossulatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nuperum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion ferreum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion regestum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion sufflatum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cheyennense]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="368" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133816" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="22">
            <NAME><![CDATA[Bembidion (Odontium)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Ocys]]></NAME>
                <AUTHORITY><![CDATA[Gistel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1939" ITALICIZENAME="1">
                <NAME><![CDATA[Cylindrobracteon]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
            <NODES>
              <NODE EXTINCT="0" ID="533" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion aeneipes]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1883</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="532" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion aenulum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1901</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131252" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion arizonae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1963</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="516" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bowditchii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1878</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="527" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion carinatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="536" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion chloropus]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1883</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="529" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion confusum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1897</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion marquettense]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nitidulum]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion aeneorubrum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="526" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion coxendix]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1823</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion unicum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion snowi]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1927" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion weesi]]></NAME>
                    <AUTHORITY><![CDATA[Hatch &amp; Ortenburger]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion venator]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion vigilans]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="530" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion durangoense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1891</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1963" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion gilae]]></NAME>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="517" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion foraminosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1825</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1812" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion bipunctatum]]></NAME>
                    <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="534" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fusiforme]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1914</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="522" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion gebieni]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1928</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="523" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion japonicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1961</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="520" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion narzikulovi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Kryzhanovskij]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1973</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="521" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion persimile]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Morawitz, A.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1862</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1961" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion kaszabi]]></NAME>
                    <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="525" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion robusticolle]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1897</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="528" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sculpturatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1859</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="524" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion soederbomi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1965</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="518" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion striatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1792</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1899" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion hamburgense]]></NAME>
                    <AUTHORITY><![CDATA[Meier]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1888" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nigrescens]]></NAME>
                    <AUTHORITY><![CDATA[Schilsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion maurum]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1812" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion orichalcium]]></NAME>
                    <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1922" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nigricans]]></NAME>
                    <AUTHORITY><![CDATA[Everts]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1932" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tschalikovi]]></NAME>
                    <AUTHORITY><![CDATA[Samko]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="535" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion subfusum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Darlington]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1959</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="537" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion suensoni]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1984</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="519" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="368" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion suturale]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1850</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="370" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133816" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion (Pseudoperyphus)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1950</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1955" ITALICIZENAME="1">
                <NAME><![CDATA[Bracteomimus]]></NAME>
                <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
            <NODES>
              <NODE EXTINCT="0" ID="132629" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
                <NAME><![CDATA[Bembidion chalceum subgroup]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="543" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion chalceum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1831</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="105270" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion rothfelsi]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Maddison]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>2008</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="105271" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion bellorum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Maddison]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>2008</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="545" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion antiquum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1831</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1848" ITALICIZENAME="1">
                        <NAME><![CDATA[Ochthedromus dilatatus]]></NAME>
                        <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="105272" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion louisella]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Maddison]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>2008</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="132630" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
                <NAME><![CDATA[Bembidion honestum subgroup]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="547" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion rufotinctum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1868</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1897" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion blanchardi]]></NAME>
                        <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="546" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion integrum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1918</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="544" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion honestum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1823</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1868" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion platyderum]]></NAME>
                        <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1848" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion basale]]></NAME>
                        <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="105274" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="370" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion arenobilis]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Maddison]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>2008</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23678" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
        <NAME><![CDATA[Bembidion (Ananotaphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1931</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131534" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23678" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ateradustum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Liebherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1888" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dubium]]></NAME>
                <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion blackburni]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131535" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23678" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion errans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1888</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131536" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23678" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion proprium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1888</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131537" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23678" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion rotundicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131538" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23678" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion rotundicolle rotundicolle]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1874</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131539" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23678" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion rotundicolle eustictum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1878</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1893" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion rotundicolle clevedonense]]></NAME>
                    <AUTHORITY><![CDATA[Broun]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1910" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion rotundicolle waikatoense]]></NAME>
                    <AUTHORITY><![CDATA[Broun]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23719" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion incisum]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1921</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1931" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Andrewesa)]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="133822" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
        <NAME><![CDATA[Bembidion (Antiperyphus) Complex]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="23713" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133822" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="10">
            <NAME><![CDATA[Bembidion (Antiperyphus)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="58071" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion eburneonigrum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58072" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion engelhardti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1910</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58073" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion germainianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion germaini]]></NAME>
                    <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="58074" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion hirtipes]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58075" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion parvum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58076" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion philippii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58077" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ringueleti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58078" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion rufoplagiatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58079" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tucumanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58080" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23713" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion uniforme]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1928</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion unicolor]]></NAME>
                    <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23715" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133822" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion (Antiperyphanes)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="58085" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23715" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion chilense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1849</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58089" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23715" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion loscondesi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nivale]]></NAME>
                    <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="58088" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23715" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion maculiferum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1868</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion maculatum]]></NAME>
                    <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="58087" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23715" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion negreanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion kaszabi]]></NAME>
                    <AUTHORITY><![CDATA[N?gre]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="58086" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23715" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion spinolai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1849</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion delamarei]]></NAME>
                    <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23714" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="133822" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion (Chilioperyphus)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="58083" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23714" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cassinense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Roig-Ju&#241;ent]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2001</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58081" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23714" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion mendocinum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1910</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="58082" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23714" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion orregoi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23712" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="133822" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mandibulare]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1849" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angustius]]></NAME>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="1" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion (Plocamoperyphus)]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132575" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Apteromimus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1877</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132576" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132575" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion platyderoides]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1877</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132577" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132575" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion wollastoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1972</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="357" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="31">
        <NAME><![CDATA[Bembidion (Asioperyphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Vysoky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1986</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1986" ITALICIZENAME="1">
            <NAME><![CDATA[Chinoperyphus]]></NAME>
            <AUTHORITY><![CDATA[Vysoky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="132113" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion algidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132114" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion altestriatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1934</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1966" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bajani]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132115" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion amaurum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1883</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1951" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion osakaense]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132116" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion amnicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1900</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132117" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bimaculatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132118" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion chloreum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1873</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132119" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion collutum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1873</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132120" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion collutum collutum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1873</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132121" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion collutum nakanoshimense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Ueno]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1955</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132122" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion collutum semiluitum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1883</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132123" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion exornatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1930</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132124" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gotoense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1973</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132125" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion infuscatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1901" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion postae]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion transbaicalicum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132126" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kazakhstanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kryzhanovskij]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1979</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132127" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion ladas]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132128" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ladas ladas]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132129" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ladas lehense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1985</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132130" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lenae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion intermedium]]></NAME>
                <AUTHORITY><![CDATA[Poppius]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132131" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lindbergi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132132" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lunatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1812</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1920" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion koltzei]]></NAME>
                <AUTHORITY><![CDATA[Meyer, P.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ustum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pallidum]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132133" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion macropterum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1880</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132134" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion obenbergeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132135" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ocylum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132136" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ovale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rugicolle]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132137" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion pamiricola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1930</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132138" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion pamiricola pamiricola]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132139" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion pamiricola beybienkoi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Kryzhanovskij]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1979</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132140" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion peleum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1984" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nanpingense]]></NAME>
                <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132141" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion postremum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1830</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132142" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion renoanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132143" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sajanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Shilenkov in Kryzhanovskij &amp; al.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1995</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion conforme]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132144" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sapporense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1951</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132145" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semiferrugineum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132146" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion semilunium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1914</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132147" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion semilunium semilunium]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1914</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1951" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion semilunium yanoi]]></NAME>
                    <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1965" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion semilunium janoi]]></NAME>
                    <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132148" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion semilunium muchei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1961</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132149" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion semilunium serorum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1934</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132150" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion smirnovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kryzhanovskij]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1979</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132151" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sordidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132152" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion umiatense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1963</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132153" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="357" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ustum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Quensel in Sch&#246;nherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1806</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23692" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Aureoplataphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1943</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131781" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23692" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aureofuscum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1883</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131782" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23692" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jaechi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23672" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="14">
        <NAME><![CDATA[Bembidion (Bembidion)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Latreille]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1802</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1856" ITALICIZENAME="1">
            <NAME><![CDATA[Taractus]]></NAME>
            <AUTHORITY><![CDATA[Gistel]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1806" ITALICIZENAME="1">
            <NAME><![CDATA[Tachys]]></NAME>
            <AUTHORITY><![CDATA[Sch&#246;nherr]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1821" ITALICIZENAME="1">
            <NAME><![CDATA[Lopha]]></NAME>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1868" ITALICIZENAME="1">
            <NAME><![CDATA[Bembicidium]]></NAME>
            <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1847" ITALICIZENAME="1">
            <NAME><![CDATA[Bembecidium]]></NAME>
            <AUTHORITY><![CDATA[Agassiz]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1810" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidium]]></NAME>
            <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="23774" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion adductum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion reliquum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion relictum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23767" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion crassicorne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1872</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23766" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion humerale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1825</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131588" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion humerale humerale]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1825</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1827" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion humerale bellum]]></NAME>
                    <AUTHORITY><![CDATA[Sahlberg, C.R.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1827" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion humerale pulchrum]]></NAME>
                    <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131589" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion humerale petribulirschi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2000</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23773" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jucundum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Horn, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1895</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23769" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mutatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion axillare]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion vegetum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23772" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion oregonense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1953</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131590" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pacificum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23765" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion paediscum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1883</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1964" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aimaki]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1958" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion anomalum]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1955" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion szekessyi]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23771" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pedicellatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1857</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fastidiosum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion strigulosum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23770" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion praecinctum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1879</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion veridicum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1950" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion alutaceum]]></NAME>
                <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23763" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="6">
            <NAME><![CDATA[Bembidion quadrimaculatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Linnaeus]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1761</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131591" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23763" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadrimaculatum quadrimaculatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Linnaeus]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1761</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1944" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum roubali]]></NAME>
                    <AUTHORITY><![CDATA[Kult]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1827" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum coarctatum]]></NAME>
                    <AUTHORITY><![CDATA[Sahlberg, C.R.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1926" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum arragonense]]></NAME>
                    <AUTHORITY><![CDATA[Wagner, H.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1827" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum formosum]]></NAME>
                    <AUTHORITY><![CDATA[Sahlberg, C.R.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum sibiricum]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1796" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum pulchellum]]></NAME>
                    <AUTHORITY><![CDATA[Panzer]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1775" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum quadriguttatum]]></NAME>
                    <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1792" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum subglobosum]]></NAME>
                    <AUTHORITY><![CDATA[Rossi, P.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1821" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum quadriguttatum]]></NAME>
                    <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131592" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23763" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadrimaculatum caporiaccoi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131593" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23763" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadrimaculatum cardiaderum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1874</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131594" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23763" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadrimaculatum dubitans]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum tenax]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum pugetanum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum crurale]]></NAME>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum caseyi]]></NAME>
                    <AUTHORITY><![CDATA[Leng]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum gregale]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadrimaculatum sapphicum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131595" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23763" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadrimaculatum mandli]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1932</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131596" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23763" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadrimaculatum oppositum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1823</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23764" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion quadripustulatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1821</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131597" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadripustulatum quadripustulatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1821</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1866" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadripustulatum antiquorum]]></NAME>
                    <AUTHORITY><![CDATA[Crotch]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1956" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadripustulatum conjunctum]]></NAME>
                    <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1795" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadripustulatum quadriguttatum]]></NAME>
                    <AUTHORITY><![CDATA[Olivier]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1900" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadripustulatum albomaculatum]]></NAME>
                    <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1866" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quadripustulatum olivieri]]></NAME>
                    <AUTHORITY><![CDATA[Crotch]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131598" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadripustulatum afghanistanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131599" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadripustulatum hassani]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1933</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131600" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadripustulatum iraniculum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1963</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131601" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quadripustulatum trimaculatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1927</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23768" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion sciakyi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1999</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131602" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sciakyi sciakyi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1999</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131603" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sciakyi luguense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1999</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131604" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sciakyi rinaldi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1999</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131605" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23672" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion teres]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1881</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23697" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="58">
        <NAME><![CDATA[Bembidion (Bembidionetolitzkya)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Strand, E.]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1929</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1910" ITALICIZENAME="1">
            <NAME><![CDATA[Daniela]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131800" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion abchasicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1989</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131801" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ascendens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bokori]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1902" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion axillare]]></NAME>
                <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion unicolor]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1847" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion planum]]></NAME>
                <AUTHORITY><![CDATA[Schilling]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1902" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion egregium]]></NAME>
                <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion efasciatum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131802" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion astrabadense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mannerheim in Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion leander]]></NAME>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1973" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion chotjai]]></NAME>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1957" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion horioni]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1988" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion richterianum]]></NAME>
                <AUTHORITY><![CDATA[M?ller-Motzfeld]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131803" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion atrocaeruleum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1828</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1835" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cyanescens]]></NAME>
                <AUTHORITY><![CDATA[Wesmael]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131804" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion atrox]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131805" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bactrianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131806" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion beloborodovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1990</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131807" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion brunoi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonavita]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2001</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131808" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion bugnioni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131809" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bugnioni bugnioni]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131810" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bugnioni cassolai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bonavita &amp; Vigna Taglianti]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1993</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131811" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bugnioni sclanoi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Magrini]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1996</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131812" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion catharinae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131813" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cimmerium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131814" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion coeruleum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1821</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1944" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion scelio]]></NAME>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1846" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion distinctum]]></NAME>
                <AUTHORITY><![CDATA[Lucas]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131815" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion complanatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Heer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131816" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion concoeruleum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131817" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion conforme]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1812" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion luridum]]></NAME>
                <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131818" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cyaneum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1846</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1938" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion chevsuricum]]></NAME>
                <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131819" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion depressicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Landin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131820" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion depressum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1832</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131821" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion eleonorae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonavita &amp; Vigna Taglianti]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1993</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131822" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fasciolatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1812</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angusticolle]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131823" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion ferghanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld &amp; Kryzhanovskij]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1983</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131824" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ferghanicum ferghanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld &amp; Kryzhanovskij]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1983</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131825" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ferghanicum almasyi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1990</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131826" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion geniculatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Heer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131827" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion geniculatum geniculatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Heer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1837</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1947" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum cavillatum]]></NAME>
                    <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1928" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum anthracinum]]></NAME>
                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1958" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum totacyaneum]]></NAME>
                    <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum affine]]></NAME>
                    <AUTHORITY><![CDATA[Redtenbacher, L.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum nigrum]]></NAME>
                    <AUTHORITY><![CDATA[Kolbe, W.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1902" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum redtenbacheri]]></NAME>
                    <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131828" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion geniculatum ayderense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Korge]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1971</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1943" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion mamillatum]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131829" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion geniculatum claudejeannei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Marggi in L&#246;bl &amp; Smetana]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2003</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1974" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion geniculatum nevadense]]></NAME>
                    <AUTHORITY><![CDATA[Jeanne]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131830" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion geniculatum dilutipes]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sainte-Claire Deville]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131831" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gerdi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1995</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131832" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gotschii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1846</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131833" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gredosanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeanne]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1974</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131834" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion grosclaudei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Normand]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1940</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131835" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hamanense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131836" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion josephi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonavita]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2001</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1947" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion olympicum]]></NAME>
                <AUTHORITY><![CDATA[Meschnigg]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131837" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion justinae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Meschnigg]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1947</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131838" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kartalinicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1938</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131839" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion leonense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeanne &amp; M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131840" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion leonhardi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1909</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131841" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion livens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1930</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131842" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion longipes]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1898" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion liguricum]]></NAME>
                <AUTHORITY><![CDATA[Daniel, K. &amp; Daniel, J.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131843" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion marggii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schmidt, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2004</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131844" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion meschniggi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131845" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion milosfassatii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schmidt, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2004</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131846" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mingrelicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1994</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131847" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion motzfeldi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1994</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131848" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion orinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131849" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion paracomplanatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Nitzu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1995</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131850" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion peliopterum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1890" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion macrophthalmum]]></NAME>
                <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131851" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion penninum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131852" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion piceocyaneum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131853" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion piceocyaneum piceocyaneum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1874</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131854" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion piceocyaneum zaisanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld &amp; Kryzhanovskij]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1983</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131855" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pseudascendens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Manderbach &amp; M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2004</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131856" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pseudocyaneum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1994</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131857" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion relictum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1976" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion araxidis]]></NAME>
                <AUTHORITY><![CDATA[Iablokoff-Khnzorian]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1971" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion elburziacum]]></NAME>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1978" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion keilbachi]]></NAME>
                <AUTHORITY><![CDATA[M?ller-Motzfeld]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131858" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rhodopense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131859" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rionicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1983</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131860" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion satanas]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131861" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion satanas satanas]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion satanas aquilum]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131862" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion satanas mandlianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131863" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion scandens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Landin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131864" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tabellatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1854</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131865" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tibiale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1812</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nigrescens]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1954" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion baudoni]]></NAME>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion viridiaeneum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion virescens]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1839" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cnemerythrum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131866" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion toledanoi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schmidt, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2004</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131867" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion trabzonicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1994</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131868" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion transcaucasicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1938</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131869" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion varicolor]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1803</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131870" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion varicolor varicolor]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1803</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion varicolor erichsoni]]></NAME>
                    <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1801" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion varicolor tricolor]]></NAME>
                    <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131871" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion varicolor agurense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1938</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131872" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23697" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vodozi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sainte-Claire Deville]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1906</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132490" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion panda]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>2000</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="2000" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Bembidromus)]]></NAME>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23689" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
        <NAME><![CDATA[Bembidion (Blepharoplataphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1920</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131776" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23689" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion davaai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1968</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131777" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23689" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hastii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sahlberg, C.R.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1827</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ventricosum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cupripenne]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cupreum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion litigiosum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131778" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23689" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hiogoense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1873</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131779" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23689" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion virens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1827</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1827" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pfeiffii]]></NAME>
                <AUTHORITY><![CDATA[Sahlberg, C.R.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23671" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="8">
        <NAME><![CDATA[Bembidion (Cyclolopha)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1918</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="23744" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion championi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1882</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23740" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cyclodes]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1884</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aztecanum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23742" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dannieae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Perrault]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23743" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion erwini]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Perrault]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23746" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion georgettae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Perrault]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23739" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion poculare]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1884</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1919" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dilaticolle]]></NAME>
                <AUTHORITY><![CDATA[Notman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23741" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sphaeroderum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1882</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion minax]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion occultum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131494" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23671" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sphaeruliferum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1891</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion furtivum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23653" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="10">
        <NAME><![CDATA[Bembidion (Diplocampa)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Bedel]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1896</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
            <NAME><![CDATA[Paralopha]]></NAME>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131520" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion assimile]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1810</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion spencii]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1889" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion productum]]></NAME>
                <AUTHORITY><![CDATA[Schilsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1838" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dubium]]></NAME>
                <AUTHORITY><![CDATA[Heer]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1912" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fasciatum]]></NAME>
                <AUTHORITY><![CDATA[Petri]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1812" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion guttula]]></NAME>
                <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion doris]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion viridanum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion castanopterum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1835" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion haemorrhoidale]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131521" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bisulcatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131522" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion blandulum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion elbursiacum]]></NAME>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131523" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion clarkii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dawson]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131524" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion clarkii clarkii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dawson]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1849</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1970" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion clarkii aquitanum]]></NAME>
                    <AUTHORITY><![CDATA[Aubry]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131525" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion clarkii magistrettii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1947</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131526" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fumigatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1812</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion variolosum]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion stictum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1846" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dejeani]]></NAME>
                <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131527" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hesperum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Crotch]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1866</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131528" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion loeffleri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1963</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131529" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion longipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1846</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131530" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion skoraszewskyi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Korge]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1971</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131531" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion transparens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gebler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1829</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131532" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion transparens transparens]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Gebler]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1829</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion transparens fortestriatum]]></NAME>
                    <AUTHORITY><![CDATA[Mannerheim]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion transparens edmontonense]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1875" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion transparens contaminatum]]></NAME>
                    <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1848" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion transparens trepidum]]></NAME>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1848" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion transparens sulcatum]]></NAME>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131533" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23653" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion transparens prostratum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1844</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="113128" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="7">
        <NAME><![CDATA[Bembidion (Ecuadion)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>2002</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="133885" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion whymperi group]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131634" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion whymperi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131608" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion azuayi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="133886" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion jimburae group]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132515" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion mucubaji]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Perrault]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1991</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131622" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion jimburae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133858" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion wardi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133857" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion walterrossii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="133887" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion camposi group]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="133844" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion boyaca]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133843" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atillense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131611" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion camposi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133851" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion moreti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133854" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion pichincha]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="133842" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion andersoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133888" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion georgeballi group]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="133850" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion guaramacal]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133849" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion georgeballi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133848" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cosangaense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133855" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion pierrei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="133889" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="6">
            <NAME><![CDATA[Bembidion chimborazonum group]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="133890" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion agonoides]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Vigna Tagliani &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131614" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion chimborazonum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1891</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133891" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131625" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion paulinae]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>2002</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="133853" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion paulinae bolvianum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>2008</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="132356" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion rogersi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1878</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132527" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sanctaemarthae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Darlington]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1934</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="2002" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion giselae]]></NAME>
                    <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132530" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion subapterum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Darlington]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1934</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="133892" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="30">
            <NAME><![CDATA[Bembidion fulvocinctum - andinum group]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131607" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion andinum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1891</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131606" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion achipungi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133841" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion allegroi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131609" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bibliani]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131610" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bolivari]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133845" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion callacalla]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131612" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion carreli]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131613" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion chilesi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133846" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cocuyanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133847" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion colombianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131615" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cotopaxi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131616" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion davidsoni]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132504" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion espejoense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Perrault]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1991</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131617" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fulvocinctum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1891</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion fulvotinctum]]></NAME>
                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131619" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion guamani]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131620" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion humboldti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131621" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion illuchi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131623" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion mathani]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131624" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion onorei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133852" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion oxapampa]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131626" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion perraulti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131627" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion pyxidum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="133856" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quebrada]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2008</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131628" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion rawlinsi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131629" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saragurense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131630" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sirinae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131631" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sulfurarium]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131632" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion vignai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131633" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion villagomesi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131635" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="113128" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion youngi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Moret &amp; Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23668" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="21">
        <NAME><![CDATA[Bembidion (Emphanes)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1850</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
            <NAME><![CDATA[Omala]]></NAME>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131454" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion articulatoides]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1932</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131455" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion axillare]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131456" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion axillare axillare]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1844</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131457" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion axillare euxinum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1904</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131458" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion axillare occiduum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Marggi &amp; Huber, C.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2001</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1933" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion axillare punicum]]></NAME>
                    <AUTHORITY><![CDATA[Normand]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion axillare rivulare]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131459" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion azurescens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1877</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131460" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion azurescens azurescens]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1877</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1888" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion azurescens triste]]></NAME>
                    <AUTHORITY><![CDATA[Schilsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1877" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion azurescens chlorizans]]></NAME>
                    <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131461" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion azurescens kurangense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1973</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131462" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bulgani]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1968</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131463" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion diligens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion vilescans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion parabile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131464" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion eupages]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1934</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131465" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gobiense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1964</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131466" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion iliense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Iablokoff-Khnzorian]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1970</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131467" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion inconspicuum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1864</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131468" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion karounense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131469" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lais]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bedel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1900</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131470" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion latiplaga]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131471" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion latiplaga latiplaga]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1850</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1882" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion latiplaga quadrispilotum]]></NAME>
                    <AUTHORITY><![CDATA[Schaufuss, L.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131472" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion latiplaga flavibase]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1956</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1956" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion latiplaga ablutum]]></NAME>
                    <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131473" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion latiplaga mateui]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1953</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131474" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion minimum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1792</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1821" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion minutum]]></NAME>
                <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1888" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bicolor]]></NAME>
                <AUTHORITY><![CDATA[Schilsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1835" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nanum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1827" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pusillum]]></NAME>
                <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131475" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion motschulskyi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angusticolle]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131476" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion netolitzkyanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1940</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131477" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion normannum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131478" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion normannum normannum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131479" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion normannum apfelbecki]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1986</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1904" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion normannum orientale]]></NAME>
                    <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131480" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion normannum mediterraneum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1928</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1891" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion normannum meridionale]]></NAME>
                    <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131481" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion quadriplagiatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131482" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion tenellum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erichson]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131483" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tenellum tenellum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Erichson]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1837</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tenellum atratum]]></NAME>
                    <AUTHORITY><![CDATA[Hornung]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1910" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tenellum pseudotenellum]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1812" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tenellum doris]]></NAME>
                    <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1845" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tenellum maeoticum]]></NAME>
                    <AUTHORITY><![CDATA[Kolenati]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131484" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tenellum buchariplaga]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131485" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tenellum pseudoplaga]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131486" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tenellum telemus]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Ragusa]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1892</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131487" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion transversum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131488" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vile]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131489" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23668" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion zagrosense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1972</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132578" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion megalops]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1877</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1877" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Endosomatium)]]></NAME>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23700" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="12">
        <NAME><![CDATA[Bembidion (Euperyphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1941</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131894" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion alikhelicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1989</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131895" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion combustum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1832</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131896" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion combustum combustum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1832</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1832" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion combustum lividipenne]]></NAME>
                    <AUTHORITY><![CDATA[M?n?tri?s]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1832" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion combustum testaceipenne]]></NAME>
                    <AUTHORITY><![CDATA[M?n?tri?s]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131897" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion combustum lugubriveste]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131898" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion eques]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1825</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1863" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bulsanense]]></NAME>
                <AUTHORITY><![CDATA[Gredler]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131899" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="8">
            <NAME><![CDATA[Bembidion fluviatile]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131900" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile fluviatile]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1947" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion fluviatile testaceoides]]></NAME>
                    <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131901" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile acragas]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1956</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131902" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile amplum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1908</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131903" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile basale]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1844</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131904" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile exhibitum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1911</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1846" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion fluviatile fasciatum]]></NAME>
                    <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131905" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile pyrrium]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1956</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131906" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile richteri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1961</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131907" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion fluviatile unctulum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1941</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131908" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fulvipes]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1827</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion distinctum]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion viride]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1825" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion picipes]]></NAME>
                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion coeruleum]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131909" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion giganteum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1900</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131910" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion giganteum giganteum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1900</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131911" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion giganteum reinigi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1934</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131912" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nobile]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Rottenberg]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1870</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1907" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion adusticauda]]></NAME>
                <AUTHORITY><![CDATA[Costa, G.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131913" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion parallelipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131914" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion parallelipenne parallelipenne]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1850</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131915" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion parallelipenne exisonum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1938</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131916" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion parallelipenne pseudoripicola]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Iablokoff-Khnzorian]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1963</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131917" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion regale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131918" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ripicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dufour, L.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1820</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1937" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion suzannae]]></NAME>
                <AUTHORITY><![CDATA[Puel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1937" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion juliettae]]></NAME>
                <AUTHORITY><![CDATA[Puel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131919" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion scapulare]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131920" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scapulare scapulare]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131921" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scapulare lomnickii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1916</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131922" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scapulare oblongum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131923" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scapulare tergluense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131924" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion testaceum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1812</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131925" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion testaceum testaceum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1812</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum falcozi]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum obscurum]]></NAME>
                    <AUTHORITY><![CDATA[Redtenbacher, L.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1877" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum obscurum]]></NAME>
                    <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum obsoletum]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1877" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum nigrescens]]></NAME>
                    <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum nigricolle]]></NAME>
                    <AUTHORITY><![CDATA[Redtenbacher, L.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion testaceum neglectum]]></NAME>
                    <AUTHORITY><![CDATA[Dawson]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131926" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23700" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion testaceum brevius]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="348" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="12">
        <NAME><![CDATA[Bembidion (Eupetedromus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1911</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="372" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aratum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion definitum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1882" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion scintillans]]></NAME>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1884" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion vinnulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23733" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dentelloides]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="423" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dentellum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Thunberg]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1787</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1845" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion meridionale]]></NAME>
                <AUTHORITY><![CDATA[Kolenati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1825" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion undulatum]]></NAME>
                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1806" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion flammulatum]]></NAME>
                <AUTHORITY><![CDATA[Clairville]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1812" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ustulatum]]></NAME>
                <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="427" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion graciliforme]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="426" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion immaturum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1954</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="425" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion incrematum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1860</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fortunatum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion mobile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1878" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion arcuatum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1919" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lengi]]></NAME>
                <AUTHORITY><![CDATA[Notman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion oblectans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion gulosum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nubiferum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion semotum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23734" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ruthenum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Tschitscherine]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1895</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23735" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sibiricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tschitscherini]]></NAME>
                <AUTHORITY><![CDATA[Jacobson]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1895" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion amurense]]></NAME>
                <AUTHORITY><![CDATA[Tschitscherine]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1845" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fuscovariegatum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1972" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion inouyei]]></NAME>
                <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23736" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion starkii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1860</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="424" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tinctum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Zetterstedt]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1828</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nigripes]]></NAME>
                <AUTHORITY><![CDATA[Mannerheim]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="373" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion scintillans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1882</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1884" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion vinnulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion definitum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="482" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="348" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion variegatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1823</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1893" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion postfasciatum]]></NAME>
                <AUTHORITY><![CDATA[Hamilton]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23667" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="11">
        <NAME><![CDATA[Bembidion (Furcacampa)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1931</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="52639" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion affine]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1823</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion thespis]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fallax]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion decipiens]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132493" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion barrense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="52640" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion egens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion demissum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="52641" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion flavopictum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion flavopictum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="52642" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fuchsii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blaisdell]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="52643" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion impotens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion virgatulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion gratuitum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion frugale]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aestivum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion indigens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pictum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pullulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion imbelle]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="52644" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mimus]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pellax]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rotundiceps]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="52645" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nogalesium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="52646" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion timidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sordidulum]]></NAME>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion oregonum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="52647" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion triviale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion temperans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="52648" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23667" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion versicolor]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion terracense]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tolerans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion wisconsinium]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1837" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion variegatum]]></NAME>
                <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132631" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Gnatholymnaeum)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132618" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132631" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion spurcum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1881</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1903" ITALICIZENAME="1">
                <NAME><![CDATA[Nesolymnaeum]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Nesolimnaeum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132619" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="132631" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion blackburni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23690" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="11">
        <NAME><![CDATA[Bembidion (Hirmoplataphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1963</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1942" ITALICIZENAME="1">
            <NAME><![CDATA[Hirmoplataphus]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="23759" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion alpineanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23760" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion avidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23751" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion concolor]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion longulum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion subaeneum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131780" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion friebi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1914</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23747" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hirmocaelum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion punctatostriatum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1880" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion parvicolle]]></NAME>
                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23755" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion humboldtense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blaisdell]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1920" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion humboldtiense]]></NAME>
                <AUTHORITY><![CDATA[Leng]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1953" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion chetcoens]]></NAME>
                <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23748" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nigrum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1823</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion facile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion morosum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23753" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion quadrulum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1861</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion viridinigrum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tritum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion callidum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tartareum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aegrotum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion porrectum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23754" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion recticolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1863</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion oblatum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion umbraticola]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tetragonoderum]]></NAME>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pertinax]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23752" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion salebratum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1929" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion mackinacense]]></NAME>
                <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion consessor]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion purpurascens]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion inopinum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23762" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23690" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion subaerarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23691" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
        <NAME><![CDATA[Bembidion (Hydriomicrus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1918</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="23749" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23691" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion brevistriatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23761" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23691" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion californicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion innocuum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23757" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23691" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion marinianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="23758" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23691" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion quadratulum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Notman]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1919</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1919" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion proximum]]></NAME>
                <AUTHORITY><![CDATA[Notman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="23750" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23691" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semistriatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Haldeman]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1843</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="133820" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="194" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
        <NAME><![CDATA[Bembidion (Hydrium) Complex]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="133818" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="194" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131250" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="9">
                <NAME><![CDATA[Bembidion (Eurytrachelus)]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1850</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                    <NAME><![CDATA[Trachyplatys]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1837" ITALICIZENAME="1">
                    <NAME><![CDATA[Eudromus]]></NAME>
                    <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1891" ITALICIZENAME="1">
                    <NAME><![CDATA[Pogonidium]]></NAME>
                    <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1844" ITALICIZENAME="1">
                    <NAME><![CDATA[Platytrachelus]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
                <NODES>
                  <NODE EXTINCT="0" ID="23575" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion chalcodes]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1935</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="441" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion interventor]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1963</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="431" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion laevibase]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1902</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="433" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion laticolle]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1812</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion inserticeps]]></NAME>
                        <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="439" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion nitidum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1837</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion edolatum]]></NAME>
                        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="443" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion obliquulum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1859</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion josephineum]]></NAME>
                        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1859" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion aptum]]></NAME>
                        <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="435" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion pogonoides]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1883</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="23576" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion rufotibiellum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Fairmaire]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1888</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="437" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131250" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion vitiosum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1868</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion sibiricum]]></NAME>
                        <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="340" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME><![CDATA[Bembidion (Hydrium)]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1848</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="444" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="340" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion levigatum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1823</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1830" ITALICIZENAME="1">
                        <NAME><![CDATA[Hydrium laevigatum]]></NAME>
                        <AUTHORITY><![CDATA[Say]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                        <NAME><![CDATA[Hydrium delawarense]]></NAME>
                        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="131249" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="340" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion nipponicum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Habu &amp; Ueno]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1955</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23649" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="11">
            <NAME><![CDATA[Bembidion (Chlorodium)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1864</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131271" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME><![CDATA[Bembidion almum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1900</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131272" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion almum almum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1900</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="131273" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion almum protalmum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1934</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="131274" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion difforme]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1844</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1868" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion posterius]]></NAME>
                    <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1844" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion laticolle]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1905" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion hammarstroemi]]></NAME>
                    <AUTHORITY><![CDATA[Poppius]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131275" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion elbursicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1939</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131276" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion leucolenum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1873</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131277" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion loricatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1922</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131278" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME><![CDATA[Bembidion luridicorne]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1874</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131279" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion luridicorne luridicorne]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1874</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="131280" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion luridicorne lamprinum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1895</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="131281" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion mundatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1920</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131282" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion pygmaeum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1792</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1942" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion dellabeffai]]></NAME>
                    <AUTHORITY><![CDATA[Burlini]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion bilunulatum]]></NAME>
                    <AUTHORITY><![CDATA[Bielz]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1871" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion serotinum]]></NAME>
                    <AUTHORITY><![CDATA[Hochhuth]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion chalcopterum]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1796" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion orichalceum]]></NAME>
                    <AUTHORITY><![CDATA[Panzer]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1922" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion coelesticolor]]></NAME>
                    <AUTHORITY><![CDATA[Roubal]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1817" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion fornicatum]]></NAME>
                    <AUTHORITY><![CDATA[Beck]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1798" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion orichalcium]]></NAME>
                    <AUTHORITY><![CDATA[Illiger]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1936" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion caeruleoviride]]></NAME>
                    <AUTHORITY><![CDATA[Roubal]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131283" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion quailaicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1984</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131284" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
                <NAME><![CDATA[Bembidion splendidum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1825</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131285" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion splendidum splendidum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1825</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion splendidum venustulum]]></NAME>
                        <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1855" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion splendidum luridipes]]></NAME>
                        <AUTHORITY><![CDATA[Reiche &amp; Saulcy]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1826" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion splendidum metallicum]]></NAME>
                        <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="131286" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion splendidum luteipes]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1844</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion splendidum colchicum]]></NAME>
                        <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="131287" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion splendidum pincum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1957</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="131288" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23649" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion wittmeri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1979</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23644" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="10">
            <NAME><![CDATA[Bembidion (Neja)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1864</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131257" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ambiguum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1838" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion bifoveolatum]]></NAME>
                    <AUTHORITY><![CDATA[Rambur]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1835" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion variabile]]></NAME>
                    <AUTHORITY><![CDATA[Waltl]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1846" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion mauritanicum]]></NAME>
                    <AUTHORITY><![CDATA[Lucas]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1955" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion canonicum]]></NAME>
                    <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131258" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cirtense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1914</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1846" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion pulchellum]]></NAME>
                    <AUTHORITY><![CDATA[Lucas]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131259" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion gansuense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1965</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131260" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME><![CDATA[Bembidion leucoscelis]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1850</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131261" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion leucoscelis leucoscelis]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1850</AUTHDATE>
                    <OTHERNAMES>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion leucoscelis palpale]]></NAME>
                        <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1908" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion leucoscelis lamprinulum]]></NAME>
                        <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                      <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1851" ITALICIZENAME="1">
                        <NAME><![CDATA[Bembidion leucoscelis curtulum]]></NAME>
                        <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                        <COMMENTS></COMMENTS>
                      </OTHERNAME>
                    </OTHERNAMES>
                  </NODE>
                  <NODE EXTINCT="0" ID="131262" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion leucoscelis tripolitanum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1937</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="131263" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion nigricorne]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1827</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1911" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion wahnschaffei]]></NAME>
                    <AUTHORITY><![CDATA[Heinemann]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131264" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion palumboi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Ragusa]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1887</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131265" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME><![CDATA[Bembidion rugicolle]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Reiche &amp; Saulcy]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1855</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131266" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion rugicolle rugicolle]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Reiche &amp; Saulcy]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1855</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="131267" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion rugicolle sporadicum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1903</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="131268" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion striaticeps]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131269" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion submutatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1911</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131270" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23644" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion torosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Marggi &amp; Huber, C.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1999</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1968" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion anatolicum]]></NAME>
                    <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23650" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="7">
            <NAME><![CDATA[Bembidion (Metallina)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1821" ITALICIZENAME="1">
                <NAME><![CDATA[Leja]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
            <NODES>
              <NODE EXTINCT="0" ID="131289" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion bodenheimeri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131290" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dyschirinum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1861</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1950" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion keechelus]]></NAME>
                    <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion aleneanum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atrolucens]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1924" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion speculinum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion agitabile]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion perturbatum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131291" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME><![CDATA[Bembidion elevatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1844</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="131292" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion elevatum elevatum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1844</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="131293" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion elevatum lamprosimile]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1939</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="131294" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion lampros]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Herbst]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1784</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1910" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nigroaeneum]]></NAME>
                    <AUTHORITY><![CDATA[Gerhardt]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1802" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion pulchellum]]></NAME>
                    <AUTHORITY><![CDATA[Marsham]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1994" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion boreonense]]></NAME>
                    <AUTHORITY><![CDATA[Avon]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1802" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion acutum]]></NAME>
                    <AUTHORITY><![CDATA[Marsham]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1792" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion celere]]></NAME>
                    <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1956" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion phygas]]></NAME>
                    <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1792" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion triste]]></NAME>
                    <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1790" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion rufipes]]></NAME>
                    <AUTHORITY><![CDATA[Paykull]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1906" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion lampron]]></NAME>
                    <AUTHORITY><![CDATA[Jacobson]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion lithuanicum]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="10" DATE="1798" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion pygmaeum]]></NAME>
                    <AUTHORITY><![CDATA[Paykull]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="11" DATE="1837" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion felixianum]]></NAME>
                    <AUTHORITY><![CDATA[Heer]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="12" DATE="1964" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion pseudolampros]]></NAME>
                    <AUTHORITY><![CDATA[Schweiger]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="13" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion annulosum]]></NAME>
                    <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="14" DATE="1790" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion velocipes]]></NAME>
                    <AUTHORITY><![CDATA[Rossi, P.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131295" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ovoideum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Marggi &amp; Huber, C.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1999</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1972" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion colasi]]></NAME>
                    <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131296" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion properans]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1828</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion plumbeum]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1913" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cyaneotinctum]]></NAME>
                    <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1871" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion quatuordecimstriatum]]></NAME>
                    <AUTHORITY><![CDATA[Thomson, C.G.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1908" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion coeruleitinctum]]></NAME>
                    <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1924" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion virens]]></NAME>
                    <AUTHORITY><![CDATA[Kolbe, W.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1828" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion chalceum]]></NAME>
                    <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1928" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion silesiacum]]></NAME>
                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1837" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion velox]]></NAME>
                    <AUTHORITY><![CDATA[Erichson]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1828" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion orichalcium]]></NAME>
                    <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1938" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion coeruleipenne]]></NAME>
                    <AUTHORITY><![CDATA[Saunders]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131297" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23650" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion veneriatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Normand]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1946</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132589" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="9">
        <NAME><![CDATA[Bembidion (Hypsipezum)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1917</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1917" ITALICIZENAME="1">
            <NAME><![CDATA[Acropezum]]></NAME>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="132590" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aberdarense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1939</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132591" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion buxtoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1953</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132592" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion celisi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132593" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion (Hypsipezum) edwardsi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1953</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132594" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion kenyense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1917</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132595" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion kenyense kenyense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1917</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132596" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion kenyense eborense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1917</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132597" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mackinderi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1917</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132598" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mus]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1931</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132599" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ruwenzoricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132600" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132589" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sjoestedti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1926</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23654" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Hoquedela)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1988</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131253" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23654" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion csikii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1932</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131254" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23654" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion kirschenhoferi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131255" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23654" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion kirschenhoferi kirschenhoferi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1988</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131256" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23654" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion kirschenhoferi annapurnae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld &amp; Schmidt, J.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2001</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132484" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
        <NAME><![CDATA[Bembidion (Josefia)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>2000</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132485" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132484" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion belousovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132486" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132484" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion luisae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132487" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132484" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pieroi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132488" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132484" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion shugela]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132489" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132484" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion taguense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23696" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion tigrinum]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1879</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Leuchydrium)]]></NAME>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23695" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
        <NAME><![CDATA[Bembidion (Liocosmius)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1918</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="52261" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23695" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion festivum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="52260" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23695" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion horni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="52259" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="23695" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mundum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion hilare]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bifasciatum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="343" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Lionepha)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1918</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="133804" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="133805" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="133806" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                    <NAME></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>null</AUTHDATE>
                    <NODES>
                      <NODE EXTINCT="0" ID="133807" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                        <NAME></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>null</AUTHDATE>
                        <NODES>
                          <NODE EXTINCT="0" ID="131688" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                            <NAME><![CDATA[Bembidion lindrothellum]]></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>1981</AUTHDATE>
                          </NODE>
                          <NODE EXTINCT="0" ID="133808" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                            <NAME></NAME>
                            <DESCRIPTION></DESCRIPTION>
                            <AUTHORITY></AUTHORITY>
                            <NAMECOMMENT></NAMECOMMENT>
                            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                            <AUTHDATE>null</AUTHDATE>
                            <NODES>
                              <NODE EXTINCT="0" ID="464" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                                <NAME><![CDATA[Bembidion lummi]]></NAME>
                                <DESCRIPTION></DESCRIPTION>
                                <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
                                <NAMECOMMENT></NAMECOMMENT>
                                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                <AUTHDATE>1981</AUTHDATE>
                              </NODE>
                              <NODE EXTINCT="0" ID="465" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                                <NAME><![CDATA[Bembidion chintimini]]></NAME>
                                <DESCRIPTION></DESCRIPTION>
                                <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
                                <NAMECOMMENT></NAMECOMMENT>
                                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                                <AUTHDATE>1981</AUTHDATE>
                              </NODE>
                            </NODES>
                          </NODE>
                        </NODES>
                      </NODE>
                      <NODE EXTINCT="0" ID="460" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                        <NAME><![CDATA[Bembidion erasum]]></NAME>
                        <DESCRIPTION></DESCRIPTION>
                        <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                        <NAMECOMMENT></NAMECOMMENT>
                        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                        <AUTHDATE>1859</AUTHDATE>
                        <OTHERNAMES>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion lubricum]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion probatum]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                            <NAME><![CDATA[Bembidion lascivum]]></NAME>
                            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                            <COMMENTS></COMMENTS>
                          </OTHERNAME>
                        </OTHERNAMES>
                      </NODE>
                    </NODES>
                  </NODE>
                  <NODE EXTINCT="0" ID="458" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion disjunctum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1963</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="456" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion castum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion brumale]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion vacivum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion nescium]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion serenum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="133809" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>null</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="133810" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
                <NAME></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>null</AUTHDATE>
                <NODES>
                  <NODE EXTINCT="0" ID="454" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion sequoiae]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1963</AUTHDATE>
                  </NODE>
                  <NODE EXTINCT="0" ID="453" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                    <NAME><![CDATA[Bembidion pseudoerasum]]></NAME>
                    <DESCRIPTION></DESCRIPTION>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <NAMECOMMENT></NAMECOMMENT>
                    <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                    <AUTHDATE>1963</AUTHDATE>
                  </NODE>
                </NODES>
              </NODE>
              <NODE EXTINCT="0" ID="451" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="343" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion osculans]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion speculum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23717" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion quadriimpressum]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1860</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1963" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Limnaeoperyphus)]]></NAME>
            <AUTHORITY><![CDATA[Nakane]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23718" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
        <NAME><![CDATA[Bembidion (Lymnaeum)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1828</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
            <NAME><![CDATA[Lymneops]]></NAME>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1847" ITALICIZENAME="1">
            <NAME><![CDATA[Limnaeum]]></NAME>
            <AUTHORITY><![CDATA[Agassiz]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1928" ITALICIZENAME="1">
            <NAME><![CDATA[Limneops]]></NAME>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="132427" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23718" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion abeillei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bedel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1879</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132428" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23718" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion eichleri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Marggi; Wrase &amp; Huber, C.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132429" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23718" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion laticeps]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1858</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angusticeps]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132430" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23718" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nigropiceum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Marsham]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1802</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sulcatum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1897" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion puritanum]]></NAME>
                <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1846" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sulcatulum]]></NAME>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132431" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23718" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion utahense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[VanDyke]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1925</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="131235" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="11">
        <NAME><![CDATA[Bembidion (Microserrullula)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1921</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1910" ITALICIZENAME="1">
            <NAME><![CDATA[Serrula]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131236" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aegyptiacum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Microserrullula senegalensis]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131237" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion apicale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1832</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131238" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion discordans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131239" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion icterodes]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131240" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion luniferum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131241" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ochsi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131242" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pogonopsis]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131243" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion quadricolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1933" ITALICIZENAME="1">
                <NAME><![CDATA[Microserrullula cyaneomicans]]></NAME>
                <AUTHORITY><![CDATA[Maran]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Microserrullula gymischchanense]]></NAME>
                <AUTHORITY><![CDATA[Eichler]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131244" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion splendens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131245" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion xanthacrum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1860" ITALICIZENAME="1">
                <NAME><![CDATA[Microserrullula luridipennis]]></NAME>
                <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1892" ITALICIZENAME="1">
                <NAME><![CDATA[Microserrullula xanthotela]]></NAME>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131246" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="131235" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion xanthoxanthum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1939</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="22941" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="12">
        <NAME><![CDATA[Bembidion (Microsinocys)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1998</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="22945" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion barkamense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22954" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion daxuense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22946" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion facchinii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22952" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion herbertfranzi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22951" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jani]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22949" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion luhuoense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22944" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion qinghaicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22947" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rebeccae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22950" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion schillhammeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131587" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion schuelkei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22953" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion turnai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="22948" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="22941" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion wraseanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1998</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23673" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion punctatellum]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1844</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Necpericompsus)]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="133879" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
        <NAME><![CDATA[Bembidion (Neobembidion)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Bosquet]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>2006</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131368" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="133879" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion constricticolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131387" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133879" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nudipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1963</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133880" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133879" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nitidicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bousquet]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2006</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23669" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Neoemphanes)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1978</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131490" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23669" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion satoi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morita]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1993</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131491" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23669" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion shimoyamai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1978</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23683" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="24">
        <NAME><![CDATA[Bembidion (Nepha)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1864</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131636" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion assalemi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1968</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131637" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion callosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[K&#252;ster]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1847</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131638" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion callosum callosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[K&#252;ster]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1847</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1943" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion callosum pseudolaterale]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion callosum quadriplagiatum]]></NAME>
                    <AUTHORITY><![CDATA[K?ster]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion callosum laterale]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1858" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion callosum semipunctatum]]></NAME>
                    <AUTHORITY><![CDATA[Gra?lls]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1968" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion callosum demontei]]></NAME>
                    <AUTHORITY><![CDATA[Ravizza]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1898" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion callosum strasseri]]></NAME>
                    <AUTHORITY><![CDATA[Fuente]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131639" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion callosum paralaterale]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1953</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131640" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion callosum subconnexum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1953</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131641" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cantalicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fauvel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1882</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131642" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion caucasicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angusticolle]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1929" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ebneri]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1891" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion turcicum]]></NAME>
                <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1846" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion substriatum]]></NAME>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion laevissimum]]></NAME>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131643" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion deliae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1973</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131644" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fortunatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1871</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1839" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion concolor]]></NAME>
                <AUTHORITY><![CDATA[Brull?]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131645" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion genei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[K&#252;ster]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1847</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1847" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion speculare]]></NAME>
                <AUTHORITY><![CDATA[K?ster]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131646" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion glabrum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131647" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion grisvardi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[DeWailly]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1949</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131648" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hauserianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131649" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ibericum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Piochard de la Br&#251;lerie]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1867</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131650" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion illigeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1914</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1798" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion quadriguttatum]]></NAME>
                <AUTHORITY><![CDATA[Illiger]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion kuesteri]]></NAME>
                <AUTHORITY><![CDATA[Jacobson]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131651" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion laevipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion yasujense]]></NAME>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131652" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion loefflerianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1963</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1963" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion loeffleri]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131653" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion menetriesii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kolenati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1845</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131654" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pinkeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131655" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion retipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131656" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rufimacula]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131657" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="8">
            <NAME><![CDATA[Bembidion schmidti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1854</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131658" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti schmidti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1854</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131659" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti alluaudi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1925</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131660" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti jeannelianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Casale &amp; Vigna Taglianti]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1993</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1951" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion schmidti jeanneli]]></NAME>
                    <AUTHORITY><![CDATA[DeWailly]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131661" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti mequignoni]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Colas]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1939</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1960" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion schmidti azoricum]]></NAME>
                    <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131662" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti moses]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1936</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131663" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti pseudocallosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Meyer, P.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1949</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131664" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti subcallosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1864</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1839" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion schmidti quadriguttatum]]></NAME>
                    <AUTHORITY><![CDATA[Brull?]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131665" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion schmidti tibesticum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Bruneau de Mir&#233;]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1990</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131666" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion seriatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131667" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion tetragrammum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1846</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131668" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tetragrammum tetragrammum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1846</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tatragrammum brevinotum]]></NAME>
                    <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tatragrammum tetrastigma]]></NAME>
                    <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131669" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tetragrammum cycladicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1940</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131670" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tetrasemum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1846</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131671" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion turcicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1932" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion staneki]]></NAME>
                <AUTHORITY><![CDATA[Maran]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1851" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion versicolor]]></NAME>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131672" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion vseteckai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Maran]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1936</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131673" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion vseteckai vseteckai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Maran]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1936</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1972" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion vseteckai naviauxi]]></NAME>
                    <AUTHORITY><![CDATA[Jeanne]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131674" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion vseteckai adanae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1953</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131675" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23683" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion vseteckai dissimile]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="132632" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="21">
        <NAME><![CDATA[Bembidion (Nesocidium)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="133835" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion advena]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132611" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ignicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1903" ITALICIZENAME="1">
                <NAME><![CDATA[Nesocidium laeticulum]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1903" ITALICIZENAME="1">
                <NAME><![CDATA[Nesocidium lahainense]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1903" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion molokaiense]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="133836" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pacificum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133837" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion teres]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Blackburn]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1881</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132603" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion admirandum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132601" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion munroi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1904" ITALICIZENAME="1">
                <NAME><![CDATA[Atelidium munronis]]></NAME>
                <AUTHORITY><![CDATA[Seidlitz]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1903" ITALICIZENAME="1">
                <NAME><![CDATA[Atelidium]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132613" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rude]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133811" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion waialeale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Liebherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132614" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion smaragdinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132609" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fulgens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132607" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion auratum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Perkins]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1917</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132617" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kauaiensis]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132616" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion coecus]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132612" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion perkinsi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132610" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion koebelei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132608" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion corticarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133813" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gagneorum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Liebherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133812" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion paratomarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Liebherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132606" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion atomarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1903" ITALICIZENAME="1">
                <NAME><![CDATA[Nesocidium scydmaenoides]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="133814" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kamakou]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Liebherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133815" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="132632" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion haleakalae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Liebherr]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2008</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23681" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion ainu]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Habu &amp; Baba]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1968</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1968" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Nipponobembidion)]]></NAME>
            <AUTHORITY><![CDATA[Habu &amp; Baba]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23662" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion ephippium]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Marsham]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1802</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1920" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Notaphemphanes)]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion pallidipenne]]></NAME>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23660" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
        <NAME><![CDATA[Bembidion (Notaphocampa)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1914</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1931" ITALICIZENAME="1">
            <NAME><![CDATA[Notaphomimus]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131417" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ejusmodi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Landin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131418" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion foveolatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sobrinum]]></NAME>
                <AUTHORITY><![CDATA[Boheman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1926" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion diversum]]></NAME>
                <AUTHORITY><![CDATA[P?ringuey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131419" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion niloticum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131420" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion niloticum niloticum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1845" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion niloticum hamatum]]></NAME>
                    <AUTHORITY><![CDATA[Kolenati]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1936" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion niloticum immaculatum]]></NAME>
                    <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion niloticum terminale]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1868" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion niloticum terminans]]></NAME>
                    <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1844" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion niloticum apicale]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131421" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion niloticum batesi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1875</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131422" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion opulentum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Nietner]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1858</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131423" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion opulentum opulentum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Nietner]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1858</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1886" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion opulentum europs]]></NAME>
                    <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131424" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion opulentum riverinae]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sloane]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1895</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion opulentum caledonicum]]></NAME>
                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1882" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion opulentum hamiferum]]></NAME>
                    <AUTHORITY><![CDATA[Fauvel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131425" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion opulentum luzonicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1931</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131426" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23660" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tropicale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bruneau de Mir&#233;]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1952</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="133861" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="17">
        <NAME><![CDATA[Bembidion (Nothocys)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131443" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion anthracinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1906</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131444" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bolsoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion coerulescens]]></NAME>
                <AUTHORITY><![CDATA[N?gre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131437" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cekalovici]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131438" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cekalovicianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aereum]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion olivieri]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131439" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cupreostriatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1906</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131445" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion delamarei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131446" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion grossepunctatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1906</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131440" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hornense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131447" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jeannelicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nitidum]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131448" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion marginatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sylvaticum]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion silvaticum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131449" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nigritum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fraudulentum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fallax]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion araucanum]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131441" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion obliteratum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131450" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion paralongulum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion longulum]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131451" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sanandresi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion marcidum]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131452" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion silvicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131442" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion solieri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion valdivianum]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1910" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dubium]]></NAME>
                <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion jensenhaarupi]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1849" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion incertum]]></NAME>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1849" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion convexiusculum]]></NAME>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1874" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion varicolor]]></NAME>
                <AUTHORITY><![CDATA[Reed]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131453" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="133861" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion topali]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[N&#232;gre]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1973</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="347" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="86">
        <NAME><![CDATA[Bembidion (Notaphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1821</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
            <NAME><![CDATA[Notaphidius]]></NAME>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1962" ITALICIZENAME="1">
            <NAME><![CDATA[Austronotaphus]]></NAME>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131352" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion abnormale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1965</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131353" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion acticola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1884</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion assensum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion argutum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="489" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aeneicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131354" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion antarcticum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fairmaire]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1889</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion riparium]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion magellanicum]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1853" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion luridum]]></NAME>
                <AUTHORITY><![CDATA[Blanchard]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1907" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fuegianum]]></NAME>
                <AUTHORITY><![CDATA[Kolbe, H.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion patagonicum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1907" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion riparum]]></NAME>
                <AUTHORITY><![CDATA[Kolbe, H.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="470" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion approximatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1859" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion laterimaculatum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion simulator]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1902" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion suspectum]]></NAME>
                <AUTHORITY><![CDATA[Blaisdell]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1859" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion flammulipenne]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1953" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion stevensoni]]></NAME>
                <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cernens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion haustum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131355" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aricense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132492" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion armuelles]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131356" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aubei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="498" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion auxiliator]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131357" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion basiplagiatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131358" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bicolor]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1970</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1970" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion proximum]]></NAME>
                <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131359" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bonariense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Boheman]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1858</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1843" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion laticolle]]></NAME>
                <AUTHORITY><![CDATA[Brull?]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion montevideanum]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131360" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion brullei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion gameani]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1838" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion variegatum]]></NAME>
                <AUTHORITY><![CDATA[Brull?]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131361" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion callens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131362" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion calverti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1906</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="492" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion castor]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1963</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131363" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cayambense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1891</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131364" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion chaudoirianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1883" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion chaudoiri]]></NAME>
                <AUTHORITY><![CDATA[Berg, C.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131365" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cognatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="469" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion coloradense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion albertanum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion prociduum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion imperitum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion amplipenne]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131366" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion commissum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erichson]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1847</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131367" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion consimile]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="472" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion conspersum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1851" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tesselatum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion xanthostictum]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131369" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion constrictum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion festinans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion civile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1884" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion vernula]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="473" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion consuetum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion augurale]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131370" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion contractum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1823</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131371" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion convergens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Berg, C.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1883</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1910" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion paravarium]]></NAME>
                <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="486" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cordatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion placabile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132497" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cortes]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="495" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion debiliceps]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131372" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dejectum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1884</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1951" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tencenti]]></NAME>
                <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1950" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aberti]]></NAME>
                <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fidele]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="471" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dorsale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1823</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="474" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion evidens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131373" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fischeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion porteri]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131374" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion flohri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1897" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion henshawi]]></NAME>
                <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131375" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gabrielum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1970</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131376" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion galapagoense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Waterhouse, G.R.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1845</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="494" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion graphicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lassulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131377" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hageni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sexpunctatum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="475" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion idoneum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion obsequens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion efficiens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion continens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131378" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion igorot]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Darlington]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131379" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion inconstans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fallaciosum]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="485" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion indistinctum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion barbarae]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion franciscanum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion caudex]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion formale]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion consentaneum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion devinctum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion curiosum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion expositum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion reconditum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion extricatum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="10" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion derisor]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="11" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ornatellum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="487" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion insulatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion caliginosum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="491" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion intermedium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="476" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jacobianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion procax]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131380" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jacobseni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1907" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion magellanicum]]></NAME>
                <AUTHORITY><![CDATA[Kolbe, H.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="null" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion kolbei]]></NAME>
                <AUTHORITY><![CDATA[Bruch, C.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131381" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kuscheli]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="490" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion latebricola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131382" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lecontei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion grandicolle]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131383" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lorenzi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion stenoderum]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131384" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion luculentum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion prosperum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131385" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mirasoi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1911" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion mirasonis]]></NAME>
                <AUTHORITY><![CDATA[Seidlitz]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131386" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mormon]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="496" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nigripes]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion concitatum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion retractum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1853" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion quadraticolle]]></NAME>
                <AUTHORITY><![CDATA[Mannerheim]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="477" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nubiculosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion platyderum]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion amnicum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1869" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pardale]]></NAME>
                <AUTHORITY><![CDATA[Zimmermann]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion laticolle]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion daphnis]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131388" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion oberthueri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1901</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="468" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion obliquum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1825</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1875" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion immaculatum]]></NAME>
                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fasciatum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1915" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion freymuthi]]></NAME>
                <AUTHORITY><![CDATA[Wagner, H.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131389" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion obscuromaculatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1859</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131390" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion obtusangulum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1863</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cornix]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131391" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion obtusidens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fall]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="500" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion operosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="497" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion patruele]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion monstratum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion mediocre]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fenisex]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion imitator]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion editum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion subexiguum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1843" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion posticum]]></NAME>
                <AUTHORITY><![CDATA[Haldeman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion marcidum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1857" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fraternum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion plectile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="10" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion exclusum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="11" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion contristans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131392" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pedestre]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131393" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pilatei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="501" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pimanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131394" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion placitum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131395" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion posticale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1906</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131396" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion posticale posticale]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion posticale punctigerum]]></NAME>
                    <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131397" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion posticale platense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[N&#232;gre]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1973</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="488" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rapidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sociale]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion negligens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion docile]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fugitans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131398" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion regismontium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131399" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion scudderi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion spissicorne]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="478" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semifasciatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Say]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1830</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="502" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semiopacum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="493" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semipunctatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Donovan]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1806</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion accuratum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1845" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion alternans]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1860" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion adustum]]></NAME>
                <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion elegantulum]]></NAME>
                <AUTHORITY><![CDATA[Sahlberg, R.F.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion provoanum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1812" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sturmii]]></NAME>
                <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1854" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rupestre]]></NAME>
                <AUTHORITY><![CDATA[Dawson]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1943" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion atrocyaneum]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131400" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion serpentinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Landin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131401" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion servillei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1849</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion uliginosum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1906" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lacustre]]></NAME>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="479" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sparsum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1882</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="480" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion spretum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1841" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fastidiosum]]></NAME>
                <AUTHORITY><![CDATA[Lafert?-S?nect?re]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131402" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ugartei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion deplanatum]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="484" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion umbratum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1848</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1859" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion variolosum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131403" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion unifasciatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1970</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="467" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion varium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Olivier]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1795</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1916" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nigrocyaneum]]></NAME>
                <AUTHORITY><![CDATA[Wagner, H.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1864" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion marginicolle]]></NAME>
                <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1902" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rumelicum]]></NAME>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nebulosum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion basale]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion apicale]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1913" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion mendacissimum]]></NAME>
                <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bifasciatum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1877" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bifasciatum]]></NAME>
                <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1761" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ustulatum]]></NAME>
                <AUTHORITY><![CDATA[Linnaeus]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="10" DATE="1812" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion flammulatum]]></NAME>
                <AUTHORITY><![CDATA[Duftschmid]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="11" DATE="1947" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dyrrhachii]]></NAME>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="12" DATE="1896" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion amoenulum]]></NAME>
                <AUTHORITY><![CDATA[P&#233;ringuey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="13" DATE="1827" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion majus]]></NAME>
                <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="14" DATE="1895" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion heptapotamicum]]></NAME>
                <AUTHORITY><![CDATA[Tschitscherine]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="15" DATE="1929" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion fulgifer]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="16" DATE="1847" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion infuscatum]]></NAME>
                <AUTHORITY><![CDATA[Schilling]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="17" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tenebrosum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="483" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion versutum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131404" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion viridicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LaFert&#233;-S&#233;nect&#232;re]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1841</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion particeps]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion hamiferum]]></NAME>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion chevrolati]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1857" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion apicale]]></NAME>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131405" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vividum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1884</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="481" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="347" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vulpecula]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23684" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Nothonepha)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1962</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="59631" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23684" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lonae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion baptisatum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1849" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion elegans]]></NAME>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="59632" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23684" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pallideguttula]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hensen-Haarup]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23716" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Notoperyphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1970</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132425" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23716" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion angelieri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1970</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132426" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23716" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bonniardae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1970" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cekalovici]]></NAME>
                <AUTHORITY><![CDATA[Bonniard de Saludo]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="364" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="39">
        <NAME><![CDATA[Bembidion (Ocydromus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Clairville]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1806</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1947" ITALICIZENAME="1">
            <NAME><![CDATA[Perigonium]]></NAME>
            <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1942" ITALICIZENAME="1">
            <NAME><![CDATA[Synechoperyphus]]></NAME>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1926" ITALICIZENAME="1">
            <NAME><![CDATA[Protoperyphus]]></NAME>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
        <NODES>
          <NODE EXTINCT="0" ID="131927" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion adowanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1876</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131928" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion aetolicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1901</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131929" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion aetolicum aetolicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1901</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131930" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion aetolicum taygetanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Maran]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131931" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion africanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1876</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131932" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion atlanticum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1854</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131933" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atlanticum atlanticum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1854</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1930" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum phobon]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1839" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum decorum]]></NAME>
                    <AUTHORITY><![CDATA[Brull?]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131934" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atlanticum galbenum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum khyber]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131935" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atlanticum leveillei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sainte-Claire Deville]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1914</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131936" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atlanticum megaspilum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Walker]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1871</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1930" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum sinemacula]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1904" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum serdicanum]]></NAME>
                    <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1876" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum jordanense]]></NAME>
                    <AUTHORITY><![CDATA[Piochard de la Br?lerie]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131937" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atlanticum palaciosi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1932</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1934" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion atlanticum hutchinsoni]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131938" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion baehri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131939" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cnemidotum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1883</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion chujoianum]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131940" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion decorum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Panzer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1799</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131941" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion decorum decorum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Panzer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1799</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1936" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum ticinense]]></NAME>
                    <AUTHORITY><![CDATA[Meyer, P.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1943" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum luridum]]></NAME>
                    <AUTHORITY><![CDATA[Suffrian]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1941" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum nigrotinctum]]></NAME>
                    <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1908" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum munganasti]]></NAME>
                    <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1828" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum agile]]></NAME>
                    <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131942" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion decorum bodemeyeri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Daniel, K. &amp; Daniel, J.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1909" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum eskilos]]></NAME>
                    <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131943" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion decorum caraffai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sainte-Claire Deville]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131944" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion decorum subconvexum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Daniel, K. &amp; Daniel, J.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1938" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum festinum]]></NAME>
                    <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1986" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum balkariense]]></NAME>
                    <AUTHORITY><![CDATA[M?ller-Motzfeld]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1963" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum schuberti]]></NAME>
                    <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1919" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum analogicum]]></NAME>
                    <AUTHORITY><![CDATA[Meyer, P.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum lopatini]]></NAME>
                    <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion decorum kurnakovi]]></NAME>
                    <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131945" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion derelictum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1926</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131946" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dolorosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1910" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion interstictum]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1860" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angusticlle]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131947" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion drago]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1958</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131948" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion echarouxi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131949" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion echigonum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu &amp; Baba]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1957</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131950" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion hasurada]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131951" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion hasurada hasurada]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion hasurada surdum]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131952" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion hasurada pagmanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1959</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131953" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hesperidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1867</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131954" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hiekei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1986</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131955" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion ispartanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1930</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131956" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ispartanum ispartanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131957" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ispartanum nairicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Iablokoff-Khnzorian]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1976</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131958" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion leleupi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1954</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131959" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion maddisoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131960" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion melanocerum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1876</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131961" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion meruanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131962" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion merum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1958" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion kinfushanum]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131963" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion modestum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1801</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1792" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cursor]]></NAME>
                <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1957" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pernigrum]]></NAME>
                <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion perplexum]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1949" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion balthasari]]></NAME>
                <AUTHORITY><![CDATA[Smetana]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131964" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion moritai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131965" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion muellermotzfeldi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131966" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion multipunctatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1850</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="null" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion [NN11]]]></NAME>
                <AUTHORITY></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1953" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion muscicola]]></NAME>
                <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131967" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion negrei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1958</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131968" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion ortsi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1938</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131969" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ortsi ortsi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1938</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131970" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ortsi grabense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1956</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131971" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion roberti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131972" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion (Ocydromus) rude]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1947</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131973" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="8">
            <NAME><![CDATA[Bembidion saxatile]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1827</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131974" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile saxatile]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1827</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1887" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion saxatile vectense]]></NAME>
                    <AUTHORITY><![CDATA[Fowler]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131975" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile bimorsum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131976" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile caesareum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1914</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131977" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile devillei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion saxatile peyerimhoffi]]></NAME>
                    <AUTHORITY><![CDATA[Sainte-Claire Deville]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131978" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile flavipalpe]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1930" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion saxatile staurophor]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131979" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile fuscomaculatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1844</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131980" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile kuruschicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131981" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion saxatile morsum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1930" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion saxatile apterum]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131982" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion schoenmanni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2000</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131983" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion scopulinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion obliquelunatum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion gelidum]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1967" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion mongolicum]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1844" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion thermarum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bellulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131984" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semilotum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1911</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131985" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="6">
            <NAME><![CDATA[Bembidion siculum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131986" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion siculum siculum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1909" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion siculum fauveli]]></NAME>
                    <AUTHORITY><![CDATA[Breit]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1857" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion siculum praeustum]]></NAME>
                    <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1972" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion siculum coiffaiti]]></NAME>
                    <AUTHORITY><![CDATA[Jeanne]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131987" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion siculum breiti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1919" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion siculum clermonti]]></NAME>
                    <AUTHORITY><![CDATA[Puel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131988" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion siculum certans]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131989" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion siculum durudense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Marggi &amp; Huber, C.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1999</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion siculum zagrosense]]></NAME>
                    <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131990" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion siculum smyrnense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1904</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1909" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion siculum rebellum]]></NAME>
                    <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131991" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion siculum winkleri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131992" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tolbonuri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131993" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion transsylvanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bielz]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1875" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cardionotum]]></NAME>
                <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131994" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion uenoianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morita]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1996</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131995" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion yunnanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1965" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion niedli]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131996" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="364" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion zolotarewi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1965" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion matalabai]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23701" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="46">
        <NAME><![CDATA[Bembidion (Ocyturanes)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1986</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131997" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion antennarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1972</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131998" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion argaeicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1905</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131999" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion babaulti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132000" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion baghlanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1989</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132001" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion balcanicum balcanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1899</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1899" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion balcanicum]]></NAME>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1910" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion balcanicum basirufum]]></NAME>
                <AUTHORITY><![CDATA[Holdhaus]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1929" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion balcanicum apicirufum]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132002" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion circassicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1890</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1983" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rousi]]></NAME>
                <AUTHORITY><![CDATA[Vysoky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132003" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion culminicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Piochard de la Br&#251;lerie]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1876</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132004" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion culminicola culminicola]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Piochard de la Br&#251;lerie]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1876</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132005" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion culminicola davatchii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1971</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132006" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dieckmanni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1957</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132007" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion dudichi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132008" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dudichi dudichi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1928</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1911" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion dudichi africanum]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1968" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion dudichi flavipenne]]></NAME>
                    <AUTHORITY><![CDATA[Jeanne]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132009" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dudichi tunetanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1947</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132010" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dyscheres]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132011" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion eucheres]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132012" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion eucheres eucheres]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132013" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion eucheres darvasicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1984</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132014" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion eucheres michailovi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1986</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132015" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fulvipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132016" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gudenzii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Neri]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1981</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132017" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion heinzi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Korge]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1971</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132018" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hissaricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1986" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion wrasei]]></NAME>
                <AUTHORITY><![CDATA[M?ller-Motzfeld]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132019" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hoberlandtianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132020" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion iphigenia]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1931</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132021" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion iranicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132022" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kabakovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132023" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion karokhense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Marggi in L&#246;bl &amp; Smetana]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2003</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1959" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion bifasciatum]]></NAME>
                <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132024" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion khanakense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132025" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kiritshenkoi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132026" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kuhitangi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov &amp; Belousov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1991</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132027" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kurdistanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1920</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132028" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ledouxianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1989</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132029" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lobanovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132030" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lysander]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132031" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion manfredschmidi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1985</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132032" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion martachemai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Toribio]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132033" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion marginipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132034" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion menander]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132035" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mirzayani]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1973</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132036" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion notatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132037" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion parsorum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1934</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132038" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pindicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1901</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132039" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion praeustum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132040" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion praeustum praeustum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132041" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion praeustum fauveli]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1891</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132042" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion praeustum viridifluum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1929</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132043" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion reiseri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132044" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion reiseri reiseri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132045" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion reiseri vranense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132046" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion schermanni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1985</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132047" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion signatipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1883" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion parnassium]]></NAME>
                <AUTHORITY><![CDATA[Miller, L.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132048" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion stolfai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132049" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion subcylindricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1892</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132050" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion subcylindricum subcylindricum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1892</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132051" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion subcylindricum kuliabense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1931</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132052" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion subcylindricum kyros]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1931</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132053" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion tauricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132054" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tauricum tauricum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132055" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tauricum frivaldszkyi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1928</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tauricum planipenne]]></NAME>
                    <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132056" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion tauricum weiratheri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132057" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion viduum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1910</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132058" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion waziristanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1932</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132059" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion waziristanum waziristanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1932</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132060" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion waziristanum murreense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1985</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132061" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion xanthochiton]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132062" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23701" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion xestum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23698" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="8">
        <NAME><![CDATA[Bembidion (Omoperyphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1931</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131875" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion caricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1907</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1858" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion orientale]]></NAME>
                <AUTHORITY><![CDATA[Peyron]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131876" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion hypocrita]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131877" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion hypocrita hypocrita]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1856" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion hypocrita distans]]></NAME>
                    <AUTHORITY><![CDATA[Rosenhauer]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131878" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion hypocrita illyricum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion hypocrita fastidiosum]]></NAME>
                    <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131879" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion hypocrita normandi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1947</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131880" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion israelita]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Ravizza]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1971</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131881" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kempi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131882" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion manicatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131883" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion semibraccatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1911</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131884" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion steinbuehleri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1891</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131885" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion steinbuehleri steinbuehleri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Ganglbauer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1891</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1847" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion steinbuehleri maritimum]]></NAME>
                    <AUTHORITY><![CDATA[K?ster]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1869" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion steinbuehleri saphyreum]]></NAME>
                    <AUTHORITY><![CDATA[Gautier des Cottes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131886" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion steinbuehleri gridellii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1947</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="131887" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23698" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion strictum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23659" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="11">
        <NAME><![CDATA[Bembidion (Omotaphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1914</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131406" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aethiopicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Raffray]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1885</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131407" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion alsium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Coquerel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1866</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131408" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion altipeta]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1917</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion montanellum]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion alticola]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131409" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kilimanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1908</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131410" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kivuanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1951</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131411" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mellissii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wollaston]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1869</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131412" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mixtum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1863</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1898" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion picturatum]]></NAME>
                <AUTHORITY><![CDATA[Fairmaire]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1876" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion madagascariense]]></NAME>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion tumidum]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1955" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion stricticolle]]></NAME>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1848" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion variegatum]]></NAME>
                <AUTHORITY><![CDATA[Boheman]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="131413" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion scotti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1931</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131414" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion simienense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Basilewsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1957</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131415" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sperans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1931</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131416" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23659" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion tropicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1876</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1900" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion abyssinicum]]></NAME>
                <AUTHORITY><![CDATA[Canaviello]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="133838" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="194" ITALICIZENAME="0" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="0" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
        <NAME><![CDATA[Bembidion (Pacmophena) Complex]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>null</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="23663" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion (Notholopha)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131427" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23663" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion atrum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion dimidiatum]]></NAME>
                    <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1928" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion germaini]]></NAME>
                    <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131428" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23663" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion epistomale]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131429" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23663" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion punctigerum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1849</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion andinum]]></NAME>
                    <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131430" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23663" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sexfoveatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="23664" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="6">
            <NAME><![CDATA[Bembidion (Pacmophena)]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131431" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23664" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion melanopodum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1849</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131432" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23664" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion penai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion kuscheli]]></NAME>
                    <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131433" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23664" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion renei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2002</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1962" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion tenuestriatum]]></NAME>
                    <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131434" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23664" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion rugosellum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131435" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23664" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scitulum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Erichson]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1834</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1849" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion fabricii]]></NAME>
                    <AUTHORITY><![CDATA[Solier]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131436" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23664" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion stricticolle]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Germain]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1906</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23709" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="16">
        <NAME><![CDATA[Bembidion (Pamirium)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1920</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132408" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion badakshanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132409" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bucephalum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1920</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132410" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ghilarovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132411" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion glasunovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132412" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion himalayanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132413" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jacobsoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132414" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kryzhanovskii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132415" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kurram]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132416" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mesasiaticum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132417" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mikitjukovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1997</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132418" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion petrimagni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1920</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132419" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion platypterum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1933" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rimskyikorsakovi]]></NAME>
                <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132420" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion punctulipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132421" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion roborowskii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132422" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rufum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1996</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132423" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23709" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion zlotini]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1996</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23694" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion heyrovskyi]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1932</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1932" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Parataphus)]]></NAME>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="23674" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
        <NAME><![CDATA[Bembidion chinense]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1901</AUTHDATE>
        <OTHERNAMES>
          <OTHERNAME ISIMPORTANT="0" ISPREFERRED="1" SEQUENCE="0" DATE="1901" ITALICIZENAME="1">
            <NAME><![CDATA[Bembidion (Pekinium)]]></NAME>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <COMMENTS></COMMENTS>
          </OTHERNAME>
        </OTHERNAMES>
      </NODE>
      <NODE EXTINCT="0" ID="355" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="32">
        <NAME><![CDATA[Bembidion (Peryphanes)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1941</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132063" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion adygorum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1996</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132064" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion brunnicorne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1969" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cauracum]]></NAME>
                <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion incertum]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132065" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion castaneipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132066" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion dalmatinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132067" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dalmatinum dalmatinum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1902" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion dalmatinum variabile]]></NAME>
                    <AUTHORITY><![CDATA[M?ller, G.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132068" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dalmatinum haupti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1908</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion dalmatinum biguttatum]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132069" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dalmatinum hybridum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Apfelbeck]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1904</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132070" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dalmatinum levantinum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Vsetecka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1941</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132071" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion dalmatinum rufoguttatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132072" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion deletum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1821</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132073" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion deletum deletum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Audinet-Serville]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1821</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum alpinum]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1976" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum reticulatum]]></NAME>
                    <AUTHORITY><![CDATA[Iablokoff-Khnzorian]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1846" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum heerii]]></NAME>
                    <AUTHORITY><![CDATA[Kelch]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1825" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum brunnipes]]></NAME>
                    <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1802" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum nitidulum]]></NAME>
                    <AUTHORITY><![CDATA[Marsham]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1810" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum rufipes]]></NAME>
                    <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132074" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion deletum schulerianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1986</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1956" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion deletum ovalipenne]]></NAME>
                    <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132075" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dostali]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132076" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion florentinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K. &amp; Daniel, J.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1898</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132077" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fraxator]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1832</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1908" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pseudonitidulum]]></NAME>
                <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132078" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fuliginosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1914</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132079" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gautieri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1921</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1866" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion luridipes]]></NAME>
                <AUTHORITY><![CDATA[Gautier des Cottes]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132080" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion grandipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1862</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132081" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion grandipenne grandipenne]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Schaum]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1862</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132082" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion grandipenne bulgardagense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1990</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132083" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion grandipenne freyi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1937</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132084" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hayachinense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Nakane]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1979</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132085" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hikosanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu &amp; Ueno]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1963" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion thermoides]]></NAME>
                <AUTHORITY><![CDATA[Nakane]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132086" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion imereticum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1996</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132087" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion italicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132088" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kulzeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132089" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lacrimans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132090" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lacunarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Zimmermann]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1869</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1884" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion militare]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion histricum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132091" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lafertei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132092" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion latinum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1911</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rufipes]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1943" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ficuzzense]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132093" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lirykense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1908</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1989" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion zechneticum]]></NAME>
                <AUTHORITY><![CDATA[M?ller-Motzfeld]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132094" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lulinense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1973</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132095" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion maroccanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1940" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion scorteccii]]></NAME>
                <AUTHORITY><![CDATA[Schatzmayr]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132096" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion milleri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132097" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion milleri milleri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion milleri violaceum]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1912" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion milleri brunneum]]></NAME>
                    <AUTHORITY><![CDATA[Petri]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1852" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion milleri lateritium]]></NAME>
                    <AUTHORITY><![CDATA[Miller, L.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132098" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion milleri carpathicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132099" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion milleri kulti]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1942</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132100" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion milleri pseudocarpathicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati in Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132101" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion milleri vogesiacum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1962</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132102" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion morvanianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1986</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion zagrosense]]></NAME>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132103" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion olegleonidovici]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1990</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132104" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pallidicorne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1921</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132105" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion phryganobium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Sokolov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1996</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132106" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion platynoides]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion optatum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion merens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion concolor]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion insopitans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sedulum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132107" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion stephensii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Crotch]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1866</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1957" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion angustum]]></NAME>
                <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1952" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion ferlani]]></NAME>
                <AUTHORITY><![CDATA[Quezel &amp; Verdier]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1835" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion affine]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1949" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion glandonense]]></NAME>
                <AUTHORITY><![CDATA[Ochs]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1897" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion canadense]]></NAME>
                <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1908" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion marthae]]></NAME>
                <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion heterocerum]]></NAME>
                <AUTHORITY><![CDATA[Thomson, C.G.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132108" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion texanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1868</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion inquietum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cogitans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132109" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion weiratherianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1932</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132110" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion weiratherianum weiratherianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1932</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132111" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion weiratherianum cilicicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1947</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132112" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="355" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion weiratherianum syriacum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1953</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23699" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
        <NAME><![CDATA[Bembidion (Peryphiolus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1941</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="131888" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23699" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion caligatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jeanne &amp; M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131889" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23699" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lucifugum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Neri &amp; Pavesi]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1989</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="131890" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23699" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion monticola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1825</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="131891" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23699" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion monticola monticola]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1825</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1825" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion monticola monticulum]]></NAME>
                    <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1831" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion monticola fuscicorne]]></NAME>
                    <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="131892" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23699" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion monticola makolskii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1929</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="131893" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23699" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion monticola ponticum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Korge]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1964</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23705" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="55">
        <NAME><![CDATA[Bembidion (Peryphodes)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1918</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132313" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ephippiger]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132314" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion salinarium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="null" ITALICIZENAME="0">
                <NAME><![CDATA[Bembidion bryanti-group]]></NAME>
                <AUTHORITY><![CDATA[-]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="null" ITALICIZENAME="0">
                <NAME><![CDATA[Bembidion (Ocydromus incert.)]]></NAME>
                <AUTHORITY></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132315" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bakeri]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132316" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bryanti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1921</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1933" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion salamander]]></NAME>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132317" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dammermani]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132318" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hammondianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132319" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion linauense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132320" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion storkianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1988</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132321" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion aegrum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132322" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion amurense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1859</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1940" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sigma]]></NAME>
                <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132323" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion angulicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132324" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bandotaro]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morita]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1991</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132325" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion chakrata]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132326" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion charon]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1926</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1973" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion matsumurai]]></NAME>
                <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1935" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion psilax]]></NAME>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132327" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion chiriqui]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132328" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion citulum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132329" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ciudadense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1891</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132330" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion daviacum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1970</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132331" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion diabola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132332" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dimidiatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1832</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1836" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lucidum]]></NAME>
                <AUTHORITY><![CDATA[Faldermann]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132333" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion edwardsi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132334" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ehikoense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132335" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion eutherum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132336" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion farsense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Marggi &amp; Huber, C.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1999</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1972" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion depressum]]></NAME>
                <AUTHORITY><![CDATA[Morvan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132337" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion fassatii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1951</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132338" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion grapii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1827</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1851" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aereum]]></NAME>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1838" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion grapei]]></NAME>
                <AUTHORITY><![CDATA[Zetterstedt]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nitens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1837" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion picipes]]></NAME>
                <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1827" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion brunnipes]]></NAME>
                <AUTHORITY><![CDATA[Sahlberg, C.R.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1831" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sahlbergii]]></NAME>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion scrutatum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion seductum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1860" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion planicolle]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1900" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion islandicum]]></NAME>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132339" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hirtipalposum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Landin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132340" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hoberlandti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1951</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132341" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion jeanneli]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Alluaud]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1939</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132342" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kamikochii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1965</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132343" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion klapperichi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1953</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132344" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion lavernae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132345" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion macrogonum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1891</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132346" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion mexicanum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion badiipenne]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion retectum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1928" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sallei]]></NAME>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion vafrum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1879" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion stabile]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1857" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lugubre]]></NAME>
                <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion canonicum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1882" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion sallaei]]></NAME>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132347" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion ohkurai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Morita]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1992</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132348" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion parepum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="66460" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pernotum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132349" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion phaedrum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1932" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion purkynei]]></NAME>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132350" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pseudolucillum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1938</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132351" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion quetzal]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Erwin]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1982</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132352" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion radians]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1922</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132353" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion radians radians]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1922</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132354" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion radians puetzi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2000</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132355" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion radians shaanxianum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Toledano]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>2000</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132357" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sanatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1883</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132358" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion satellites]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1884</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132359" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion shunichii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1973</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132360" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sjoelanderi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1965</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132361" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion spectans]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132362" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sterbai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1965</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="374" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion submaculatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1878</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="133863" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion torosiense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1961</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132363" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion trajectum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1939</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132364" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion umeyai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1959</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132365" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vaillanti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1956</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132366" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vernale]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1882</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132367" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion vulcanium]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Darlington]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1934</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132368" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23705" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion wutaishanense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23707" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
        <NAME><![CDATA[Bembidion (Peryphophila)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1939</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132369" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23707" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion clarum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1923</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132370" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23707" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion endymion]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132371" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23707" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion eurydice]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1926</AUTHDATE>
          </NODE>
        </NODES>
      </NODE>
      <NODE EXTINCT="0" ID="23704" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="1" ANCESTORWITHPAGE="194" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="0" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="65">
        <NAME><![CDATA[Bembidion (Peryphus)]]></NAME>
        <DESCRIPTION></DESCRIPTION>
        <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
        <NAMECOMMENT></NAMECOMMENT>
        <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
        <AUTHDATE>1821</AUTHDATE>
        <NODES>
          <NODE EXTINCT="0" ID="132175" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion abbreviatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132176" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion abbreviatum abbreviatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1874</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132177" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion abbreviatum longicrum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1956</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132178" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion abbreviatum pulpani]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1955</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1957" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion abbreviatum krejcareki]]></NAME>
                    <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132179" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion abbreviatum uvidum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion abbreviatum chitral]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132180" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion actuosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion urgens]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion debilicolle]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132181" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion alaf]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Britton]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1948</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132182" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion alticola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fiori, A.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1903</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132183" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion andreae]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fabricius]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1787</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1852" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lusitanicum]]></NAME>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion occidentale]]></NAME>
                <AUTHORITY><![CDATA[M?ller, G.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132184" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion asiaeminoris]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132185" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion asiaeminoris asiaeminoris]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1990" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion sevanense]]></NAME>
                    <AUTHORITY><![CDATA[Belousov in Belousov &amp; M?ller-Motzfeld]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132186" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sevanense asiorum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ler-Motzfeld in Belousov &amp; M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1990</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132187" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bracculatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1889</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132188" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion brittoni]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132189" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion bruxellense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Wesmael]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1835</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1835" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion elegans]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1948" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion kratochvili]]></NAME>
                <AUTHORITY><![CDATA[Pulpan]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1919" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion dixoni]]></NAME>
                <AUTHORITY><![CDATA[Everts]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1919" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion extinctum]]></NAME>
                <AUTHORITY><![CDATA[Everts]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1767" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion rupestre]]></NAME>
                <AUTHORITY><![CDATA[Linnaeus]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1827" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion femoratum]]></NAME>
                <AUTHORITY><![CDATA[Gyllenhal]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132190" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion captivorum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1943</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132191" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion consanguineum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Hayward]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1897</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132192" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion cordicolle]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1994" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion veldkampi]]></NAME>
                <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132193" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="16">
            <NAME><![CDATA[Bembidion cruciatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132194" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum cruciatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1831</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1953" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum wollastoni]]></NAME>
                    <AUTHORITY><![CDATA[Lindberg]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum monostigma]]></NAME>
                    <AUTHORITY><![CDATA[M?ller, G.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1980" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum proximum]]></NAME>
                    <AUTHORITY><![CDATA[Aubry]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1871" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum concinnum]]></NAME>
                    <AUTHORITY><![CDATA[Thomson, C.G.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132195" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum albanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132196" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum anglicanum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Sharp]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1869</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132197" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum atlantis]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1944</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1955" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum bisseptemstriatum]]></NAME>
                    <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1944" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum subflavescens]]></NAME>
                    <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132198" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum baenningeri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1926</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132199" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum bualei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1877" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum virescens]]></NAME>
                    <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1958" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum veselyi]]></NAME>
                    <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1877" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum coerulescens]]></NAME>
                    <AUTHORITY><![CDATA[DallaTorre]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132200" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum dromioides]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132201" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum espinassei]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Puel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1937</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132202" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum hummleri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132203" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum liberum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1937</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132204" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum polonicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1930</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1934" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum dissolutum]]></NAME>
                    <AUTHORITY><![CDATA[Hell?n]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132205" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum quadriflammeum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1889</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132206" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum subtypicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Schuler]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1958</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132207" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum tyrrhenicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jeannel]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1941</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132208" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum vorax]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132209" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion cruciatum xanthomum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1850</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1846" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum basale]]></NAME>
                    <AUTHORITY><![CDATA[Chaudoir]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion cruciatum chaudoirii]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132210" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dauricum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1844</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1930" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lysholmi]]></NAME>
                <AUTHORITY><![CDATA[Munster]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1920" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion pseudoproperans]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132211" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion dilutipenne]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132212" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="5">
            <NAME><![CDATA[Bembidion distinguendum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1852</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132213" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion distinguendum distinguendum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jacquelin du Val]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1852</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132214" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion distinguendum arrisi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1937</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132215" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion distinguendum lindrothi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1938" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion distinguendum nescium]]></NAME>
                    <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132216" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion distinguendum meyeri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132217" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion distinguendum rupestroides]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1943</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132218" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion femoratum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Sturm]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1825</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1785" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion quadrifer]]></NAME>
                <AUTHORITY><![CDATA[Geoffroy in Fourcroy]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion caucasicola]]></NAME>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1828" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion concinnum]]></NAME>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1777" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nigrothoracicum]]></NAME>
                <AUTHORITY><![CDATA[Goeze]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1993" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion daugavense]]></NAME>
                <AUTHORITY><![CDATA[Barsevskis]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1850" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion caucasicum]]></NAME>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1937" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion lemovicense]]></NAME>
                <AUTHORITY><![CDATA[Puel]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132219" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion franzi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1957</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1957" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion unicoloratum]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1957" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion volans]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1957" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion quattuormacula]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132220" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion gersdorfi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1957</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132221" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="3">
            <NAME><![CDATA[Bembidion gilgit]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132222" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion gilgit gilgit]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132223" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion gilgit nuristanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132224" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion gilgit peratrum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1956</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132225" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion havelkai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1955</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132226" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion hispanicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Dejean]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1831</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132227" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion incognitum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#252;ller, G.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1931</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132228" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="6">
            <NAME><![CDATA[Bembidion insidiosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1874</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132229" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion insidiosum insidiosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Solsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1874</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1958" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion insidiosum paulmeyeri]]></NAME>
                    <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1878" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion insidiosum pamirense]]></NAME>
                    <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132230" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion insidiosum betegara]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132231" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion insidiosum clops]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion insidiosum parkeri]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132232" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion insidiosum holdhausi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1952</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132233" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion insidiosum ixion]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1934</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132234" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion insidiosum luntaka]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1935" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion insidiosum atalanta]]></NAME>
                    <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132235" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion jedlickai]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1945</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132236" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion jedlickai jedlickai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1945</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132237" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion jedlickai turkestanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132238" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion klapperichianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1957</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132239" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion koikei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Habu &amp; Baba]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1960</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132240" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion kuznetsovi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lafer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>2002</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132241" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion ladakense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132242" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ladakense ladakense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1924</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132243" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ladakense baumi]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1985</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132244" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ladakense hazara]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132245" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion ladakense naltarense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[M&#252;ller-Motzfeld]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1988</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132246" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion languens]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132247" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion maritimum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1839</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132248" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion maritimum maritimum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Stephens]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1839</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1879" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion maritimum dorsuarium]]></NAME>
                    <AUTHORITY><![CDATA[Bedel]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132249" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion maritimum lusitanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Putzeys]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1845</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132250" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion marussii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[DeMonte]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1956</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132251" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion massaicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Antoine]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1962</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132252" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion misellum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Harold]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1877</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1883" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion nikkoense]]></NAME>
                <AUTHORITY><![CDATA[Bates]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132253" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion morawitzi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1928</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1862" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion cognatum]]></NAME>
                <AUTHORITY><![CDATA[Morawitz, A.]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1868" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion consentaneum]]></NAME>
                <AUTHORITY><![CDATA[Gemminger &amp; Harold]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132254" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion nevadense]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Ulke]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1875</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1918" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion viaticum]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132255" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="6">
            <NAME><![CDATA[Bembidion obscurellum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1845</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132256" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion obscurellum obscurellum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1845</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1855" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum fuscicrus]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum petulans]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1850" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum maritimum]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum parowanum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1853" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum lucidum]]></NAME>
                    <AUTHORITY><![CDATA[Mannerheim]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum caducum]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1863" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum mixtum]]></NAME>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum albidipenne]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1910" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum cribrulum]]></NAME>
                    <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1875" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum repandum]]></NAME>
                    <AUTHORITY><![CDATA[Sahlberg, J.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132257" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion obscurellum corporaali]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1935</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132258" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion obscurellum fumipenne]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132259" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion obscurellum insperatum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1938</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132260" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion obscurellum thibeticum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1957</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132261" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion obscurellum turanicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Csiki]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1928</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1835" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion obscurellum pictum]]></NAME>
                    <AUTHORITY><![CDATA[Faldermann]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132262" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion olemartini]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132263" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion persicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[M&#233;n&#233;tri&#233;s]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1832</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132264" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion petrosum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Gebler]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1833</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132265" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion petrosum petrosum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Gebler]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1833</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1980" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum carlhlindrothi]]></NAME>
                    <AUTHORITY><![CDATA[Kangas]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="1" DATE="1859" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum subinflatum]]></NAME>
                    <AUTHORITY><![CDATA[Motschulsky]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="2" DATE="1924" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum exiguiceps]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="3" DATE="1848" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum lucidum]]></NAME>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="4" DATE="1893" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum wagneri]]></NAME>
                    <AUTHORITY><![CDATA[Tschitscherine]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="5" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum castalium]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="6" DATE="1950" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum wenatchee]]></NAME>
                    <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="7" DATE="1910" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum siebkei]]></NAME>
                    <AUTHORITY><![CDATA[Sparre-Schneider]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="8" DATE="1848" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum substrictum]]></NAME>
                    <AUTHORITY><![CDATA[LeConte]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="9" DATE="1918" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion petrosum lepusculus]]></NAME>
                    <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132266" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion petrosum attuense]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1963</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132267" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion plagiatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Zimmermann]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1869</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132268" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="4">
            <NAME><![CDATA[Bembidion poppii]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1914</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132269" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion poppii poppii]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1914</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1906" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion poppii submaculatum]]></NAME>
                    <AUTHORITY><![CDATA[Poppius]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132270" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion poppii eugenes]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Jedlicka]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1933</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132271" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion poppii pohlai]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1984</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132272" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion poppii schalleri]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Lindroth]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1963</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132273" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion pseudosiebkei]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirschenhofer]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1984</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132274" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion psilodorum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1933</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132275" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion reichardti]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Lutshnik]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1930</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132276" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rickmersi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Reitter]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1898</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132277" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rubidum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Andrewes]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1924</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132278" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="1" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion rupicola]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Kirby]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1837</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1924" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion aversans]]></NAME>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
            </OTHERNAMES>
          </NODE>
          <NODE EXTINCT="0" ID="132279" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion satelles]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132280" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion scottustulatum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1937</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132281" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion scythicum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1902</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132282" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scythicum scythicum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1902" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion scythicum transiens]]></NAME>
                    <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
              <NODE EXTINCT="0" ID="132283" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion scythicum imitator]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Daniel, K.]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1902</AUTHDATE>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132284" CONFIDENCE="0" PHYLESIS="0" LEAF="0" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="2">
            <NAME><![CDATA[Bembidion sejunctum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1918</AUTHDATE>
            <NODES>
              <NODE EXTINCT="0" ID="132285" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sejunctum sejunctum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Casey]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1918</AUTHDATE>
              </NODE>
              <NODE EXTINCT="0" ID="132286" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
                <NAME><![CDATA[Bembidion sejunctum semiaureum]]></NAME>
                <DESCRIPTION></DESCRIPTION>
                <AUTHORITY><![CDATA[Fall]]></AUTHORITY>
                <NAMECOMMENT></NAMECOMMENT>
                <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
                <AUTHDATE>1922</AUTHDATE>
                <OTHERNAMES>
                  <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1950" ITALICIZENAME="1">
                    <NAME><![CDATA[Bembidion sejunctum fenderi]]></NAME>
                    <AUTHORITY><![CDATA[Hatch]]></AUTHORITY>
                    <COMMENTS></COMMENTS>
                  </OTHERNAME>
                </OTHERNAMES>
              </NODE>
            </NODES>
          </NODE>
          <NODE EXTINCT="0" ID="132287" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sillemi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1935</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132288" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion sogdianum]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Belousov &amp; Mikhailov]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1990</AUTHDATE>
          </NODE>
          <NODE EXTINCT="0" ID="132289" CONFIDENCE="0" PHYLESIS="0" LEAF="1" HASPAGE="0" ANCESTORWITHPAGE="23704" ITALICIZENAME="1" INCOMPLETESUBGROUPS="0" SHOWAUTHORITY="1" SHOWAUTHORITYCONTAINING="1" IS_NEW_COMBINATION="0" COMBINATION_DATE="null" CHILDCOUNT="0">
            <NAME><![CDATA[Bembidion solskyi]]></NAME>
            <DESCRIPTION></DESCRIPTION>
            <AUTHORITY><![CDATA[Netolitzky]]></AUTHORITY>
            <NAMECOMMENT></NAMECOMMENT>
            <COMBINATION_AUTHOR></COMBINATION_AUTHOR>
            <AUTHDATE>1934</AUTHDATE>
            <OTHERNAMES>
              <OTHERNAME ISIMPORTANT="0" ISPREFERRED="0" SEQUENCE="0" DATE="1949" ITALICIZENAME="1">
                <NAME><![CDATA[Bembidion obscuricorne]]></NAME>
                <AUTHORITY><![CDATA[Fassati]]></AUTHORITY>
                <COMMENTS></COMMENTS>
              </OTHERNAME>
  