=pod

=head1 NAME

Date::Manip::Range - Parses and holds a date range

=head1 SYNOPSIS

  use Date::Manip::Range;
  my $object = Date::Manip::Range->new();
  $object->method();

=head1 DESCRIPTION

B<Date::Manip::Range> parses and holds a date range. The range is defined by
a start and end point. The module accepts ranges as a single string, start
and end dates, or a list of dates.

=head3 Single string

The string has two dates separated by a range operator. Some examples...

  my $range = Date::Manip::Range->new( {range => 'today - tommorrow'} );
  my $range = Date::Manip::Range->new( {range => 'Jan 21 through Feb 3'} );
  my $range = Date::Manip::Range->new( {range => '2015-01-29 to 2015-02-03'} );
  my $range = Date::Manip::Range->new( {range => 'from Jan 21 to Feb 3'} );
  my $range = Date::Manip::Range->new( {range => 'between Jan 21 and Feb 3'} );

B<Date::Manip::Range> recognizes the following range operators...

=over

=item through

=item thru

=item to

=item -

=item ...

=item ..

=item between/and

=item and

=item from/through

=item from/thru

=item from/to

=back

B<Date::Manip::Range> splits the string on the operator, extracting the start
and end points. It creates L<Date::Manip> objects from those two points. The 
dates can be anything parsable by L<Date::Manip>.

=head3 Start and end dates

Pass two values and B<Date::Manip::Range> assumes they are the start and end
points of the range. These values can be strings parsable by L<Date::Manip>
or actual Date::Manip objects. 

  my $range = Date::Manip::Range->parse( 'today', 'tommorrow' );
  my $range = Date::Manip::Range->parse( '2015-02-03', '2015-01-29' );
  
  my $day1 = Date::Manip::Date->new( '2015-01-29' );
  my $day2 = Date::Manip::Date->new( '2015-02-03' );
  my $range = Date::Manip::Range->new( $day1, $day2 );

=head3 List of dates

With three or more dates, B<Date::Manip::Range> assumes that the earliest
and latest values are the start and end. It discards everything else. 

Like the options above, dates can be strings, L<Date::Manip> objects, or any
combination of the two.

=head2 Important Facts

=over

=item Date strings can be anything parsable by L<Date::Manip>.

=item Dates can be in any order. The earliest becomes the start.

=item Range operators are case insensetive.

=back

=cut

package Date::Manip::Range;

use 5.14.0;
use warnings;

use Date::Manip;
use Moose;
use String::Util qw/define hascontent nocontent trim/;


our $VERSION = '1.00';


=head1 METHODS & ATTRIBUTES

=head3 new

Like every other L<Moose> object, you can pass default values for any
attributes. C<new> ensures that the L</start> date always comes before the
L</end> date. It swaps the two if necessary.

In addition, C<new> also accepts a B<parse> attribute. B<parse> is a date range
in the form of a string. C<new> calls the L</parse> method, passing it this
string. L</parse> sets the L</start> and L</end> dates appropriately.

Check the L</error> attribute for problems.

=cut

sub BUILD {
	my ($self, $attributes) = @_;

	my $range = $attributes->{parse};
	if (defined $range) {
		if    (ref( $range ) eq 'ARRAY') { $self->parse( @$range ); }
		elsif (hascontent( $range )    ) { $self->parse(  $range ); }
	}

	if ($self->is_valid && $self->start->cmp( $self->end ) > 0) {
		my $swap = $self->start;
		$self->_start( $self->end );
		$self->_end( $swap );
	}
}


=head3 parse

This method takes any acceptable input for a range, parses the values, and 
configures the C<Date::Manip::Range> object. C<parse> returns B<true> on 
success or B<false> if any of the dates are invalid. L</error> tells you which
date.

=cut

sub parse {
	my $self = shift;
	
	if (scalar( @_ ) == 1) {
		# Split the string into pieces around the operator.
		my $string = shift;
		$string = $2 if $string =~ m/^\s*(between|from)\s(.*)$/i;

		my ($first, $second) = ('', '');
		if ($string =~ m/^(.*)\s(-|and|through|thru|to)\s(.*)$/i) {
			$first  = define( $1 );
			$second = define( $3 );
		} elsif ($string =~ m/^(.*)(\.\.\.)(.*)$/i) {
			$first  = define( $1 );
			$second = define( $3 );
		} elsif ($string =~ m/^(.*)(\.\.)(.*)$/i) {
			$first  = define( $1 );
			$second = define( $3 );
		}

		# Create the date objects.
		my $date1 = $self->_date_object( trim( $first ), 'The first date' );
		return 0 if $self->error ne '';

		my $date2 = $self->_date_object( trim( $second ), 'The second date' );
		return 0 if $self->error ne '';
		
		($date1, $date2) = sort { $a->cmp( $b ) } ($date1, $date2);
		$self->_start( $date1 );
		$self->_end( $date2 );
	} elsif (scalar( @_ ) > 1) {
		# Seed the search with the first two dates from the parameters.
		my $earliest = $self->_date_object( shift, 'Index 0' );
		return 0 if $self->error ne '';

		my $latest = $self->_date_object( shift, 'Index 1' );
		return 0 if $self->error ne '';

		($earliest, $latest) = sort { $a->cmp( $b ) } ($earliest, $latest);

		# Find any dates that extend the range.
		my $index = 2;
		foreach my $date (@_) {
			my $current = $self->_date_object( $date, "Index $index" );
			return 0 if $self->error ne '';
			
			if ($current->cmp( $earliest ) < 0) {
				$earliest = $current;
			} elsif ($current->cmp( $latest ) > 0) {
				$latest = $current;
			}
		} continue { $index++; }
		
		$self->_start( $earliest );
		$self->_end( $latest );
	} else { 
		$self->_error( 'Start and end date required' );
		return 0;
	}

	return 1;
}


=head3 includes

This method tells you if a given date falls within the range. A B<true> value
means that the date is inside of the range. B<false> says that the date falls
outside of the range.

The date can be a string or L<Date::Manip> object. Strings accept any valid
input for L<Date::Manip::Date>. If the date is invalid, C<inside> returns an 
error message that Perl evaluates as B<false>.

Note that C<inside> does not tell you if the date comes before or after the 
range. That didn't seem relevant.

=cut

sub includes {
	my ($self, $check) = @_;

	my $date = $self->_date_object( $check );
	return $self->error if $self->error ne '';
	
	my $after_start = 0;
	if ($self->include_start) {
		$after_start = 1 if $date->cmp( $self->start ) >= 0;
	} else {
		$after_start = 1 if $date->cmp( $self->start ) > 0;
	}

	my $before_end = 0;
	if ($self->include_end) {
		$before_end = 1 if $date->cmp( $self->end ) <= 0;
	} else {
		$before_end = 1 if $date->cmp( $self->end ) < 0;
	}
	
	return ($after_start && $before_end ? 1 : 0);
}


=head3 include_start / include_end

These attributes mark inclusive or exclusive ranges. By default, a range 
includes dates that fall on the start or end. For example...

  $range->new( {date => '2015-01-15 to 2015-01-31'} );
  # returns true because the start is included
  $range->includes( '2015-01-15' );
  # retruns true because it is between the start and end
  $range->includes( '2015-01-20' );
  # retruns true because the end is included
  $range->includes( '2015-01-31' );

For exclusive ranges, set one or both of these values to B<false>.

  $range->new( {date => '2015-01-15 to 2015-01-31'} );
  $range->include_start( 0 );
  # returns false because the start is excluded
  $range->includes( '2015-01-15' );
  # retruns true because it is between the start and end
  $range->includes( '2015-01-20' );
  # retruns true because the end is included
  $range->includes( '2015-01-31' );

=cut

has 'include_start' => (
	default => 1,
	is      => 'rw',
	isa     => 'Bool',
);

has 'include_end' => (
	default => 1,
	is      => 'rw',
	isa     => 'Bool',
);


=head3 is_valid

This method tells you if the object holds a valid date range. Use this after
calling L</new>. If anything failed (invalid dates), C<is_valid> returns 
B<false>. 

=cut

sub is_valid {
	my $self = shift;

	return 0 if !defined( $self->start );
	return 0 if !defined( $self->end );
	return 0 if $self->error ne '';
	return 1;
}


=head3 error

Returns the last date parsing error. C<Date::Manip::Range> sets this attributes
when any method encounters an invalid date. An empty string indicates no 
problem. You should check this value after calling methods such as L</parse> 
or L</includes>.

=cut

has 'error' => (
	default => '',
	isa     => 'Str',
	reader  => 'error',
	writer  => '_error',
);


=head3 start / end

The L<Date::Manip::Date> objects representing the end points of the range. Note 
that you cannot change these values. Use L</parse> to do that.

=cut

has 'start' => (
	isa    => 'Date::Manip::Date',
	reader => 'start',
	writer => '_start'
);

has 'end' => (
	isa    => 'Date::Manip::Date',
	reader => 'end',
	writer => '_end'
);

=head3 operator

The last range operator used when parsing a string. When strinifying a range, 
B<Date::Manip::Range> re-uses this same operator. The default operator is B<to>.

=cut

has 'operator' => (
	default => 'to',
	is      => 'rw',
	isa     => 'Str',
);


#------------------------------------------------------------------------------
# Internal methods and attributes...

# This method turns an function parameter into a Date::Manip::Date object.
# It processes strings or Date::Manip::Date objects. It sets $self->_error to
# a message if the date is invalid. $self->_error is the empty string if the
# conversion was a success (aka the object is valid). It both cases, it returns
# a new Date::Manip::Date object.
sub _date_object {
	my ($self, $value, $description) = @_;

	my $message = hascontent( $description ) 
		? "$description is an invalid date"
		: 'Invalid date'
	;
	
	$value = $value->input() if ref( $value ) eq 'Date::Manip::Date';
	
	my $date = Date::Manip::Date->new();
	if    (nocontent( $value )   ) { $self->_error( $message ); }
	elsif ($date->parse( $value )) { $self->_error( $message ); }
	else                           { $self->_error( ''       ); }

	return $date;
}

#------------------------------------------------------------------------------


=head1 BUGS/CAVEATS/etc

B<Date::Manip::Range> only supports English range operators. Translations 
welcome.

=head1 AUTHOR

Robert Wohlfarth <rbwohlfarth@gmail.com>

=head1 SEE ALSO

L<Date::Manip>

=head1 LICENSE

Copyright (c) 2015  Robert Wohlfarth

This program is free software; you can redistribute it and/or modify it under
the same terms as Perl itself. This software comes with NO WARRANTY of any
kind.

=cut

no Moose;
__PACKAGE__->meta->make_immutable;
