package Statocles::Deploy::File;
# ABSTRACT: Deploy a site to a folder on the filesystem
$Statocles::Deploy::File::VERSION = '0.059';
use Statocles::Base 'Class';
with 'Statocles::Deploy';


has path => (
    is => 'ro',
    isa => Path,
    coerce => Path->coercion,
    default => sub { Path::Tiny->new( '.' ) },
);


sub deploy {
    my ( $self, $from_store, %options ) = @_;

    die sprintf 'Deploy directory "%s" does not exist (did you forget to make it?)',
        $self->path
            if !$self->path->is_dir;

    if ( $options{ clean } ) {
        $_->remove_tree for $self->path->children;
    }

    my @files;
    my $iter = $from_store->find_files( include_documents => 1 );
    while ( my $path = $iter->() ) {
        # Git versions before 1.7.4.1 require a relative path to 'git add'
        push @files, $path->relative( "/" )->stringify;

        # XXX Implement a friendlier way to copy files from Stores
        my $in_fh = $from_store->open_file( $path );
        my $out_fh = $self->path->child( $path )->touchpath->openw_raw;
        while ( my $line = <$in_fh> ) {
            $out_fh->print( $line );
        }
    }

    return @files;
}

1;

__END__

=pod

=head1 NAME

Statocles::Deploy::File - Deploy a site to a folder on the filesystem

=head1 VERSION

version 0.059

=head1 DESCRIPTION

This class allows a site to be deployed to a folder on the filesystem.

This class consumes L<Statocles::Deploy|Statocles::Deploy>.

=head1 ATTRIBUTES

=head2 path

The path to deploy to.

=head1 METHODS

=head2 deploy

    my @paths = $deploy->deploy( $from_store, %options );

Deploy the site, copying from the given L<from_store|Statocles::Store>.
Returns the paths that were deployed.

Possible options are:

=over 4

=item clean

Remove all the current contents of the deploy directory before copying the
new content.

=back

=head1 SEE ALSO

=over 4

=item L<Statocles::Deploy>

=back

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
