#!/usr/bin/perl

# Copyright 2001-2014, Paul Johnson (paul@pjcj.net)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

require 5.006001;

use strict;
use warnings;

use Cwd;
use ExtUtils::MakeMaker;
use File::Copy;

$| = 1;

my $Version = "1.09";
my $Author  = 'paul@pjcj.net';
my @perlbug = ("perlbug", "-a", $Author,
                          "-s", "Installation of Devel::Cover $Version");
my $Perlbug = join " ", map { / / ? "'$_'" : $_ } @perlbug;
my $base    = getcwd;
my %inc     = map { -d $_ ? (($_ eq "." ? $_ : Cwd::abs_path($_)) => 1) : () }
                  @INC;
my @inc     = sort keys %inc;

open I, ">lib/Devel/Cover/Inc.pm"
    or die "Cannot open lib/Devel/Cover/Inc.pm: $!";
print I <<"EOI";
# Copyright 2001-2014, Paul Johnson (paul\@pjcj.net)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

# This file was automatically generated by Makefile.PL.

package Devel::Cover::Inc;

use strict;
use warnings;

our \$VERSION      = "$Version";
our \$Perl_version = '$]';
our \$Base         = '$base';
our \@Inc          = qw( @inc );
chomp (our \$Perl  = <<'EOV');  # Careful with \\\\ in the path
$^X
EOV

if (\$Perl_version ne \$])
{
    print STDERR <<"EOM";

This version of Devel::Cover was built with Perl version \$Perl_version.
It is now being run with Perl version \$].
Attempting to make adjustments, but you may find that some of your modules do
not have coverage data collected.  You may need to alter the +-inc, +-ignore
and +-select options.

EOM
    eval "use Cwd";
    my \%inc = map { -d \$_ ? ((\$_ eq "." ? \$_ : Cwd::abs_path(\$_)) => 1) : () }
                   \@INC;
    \@Inc = sort keys \%inc;
}

# TODO - check for threadedness, 64bits etc. ?

1
EOI
close I or die "Cannot close lib/Devel/Cover/Inc.pm: $!";

print "Writing tests ........ ";

for my $d (qw( t t/e2e ))
{
    unless (mkdir $d)
    {
        die "Cannot mkdir $d: $!" unless -d $d;
    }
}

my @tests;
opendir D, "tests" or die "Cannot opendir tests: $!";
for my $t (readdir D)
{
    next unless -f "tests/$t";
    next if $t =~ /\.(pm|pl|org|bak|uncoverable)$/;
    next if $t =~ /~$/;

    push @tests, $t;

    if ($t =~ /\.t/)
    {
        copy("tests/$t", "t/e2e/$t")
            or die "Cannot copy tests/$t to t/e2e/$t: $!";
        next
    }

    open T, ">t/e2e/a$t.t" or die "Cannot open t/e2e/a$t.t: $!";
    print T <<EOT;
#!$^X

# Copyright 2002-2014, Paul Johnson (paul\@pjcj.net)

# This software is free.  It is licensed under the same terms as Perl itself.

# The latest version of this software should be available from my homepage:
# http://www.pjcj.net

use strict;
use warnings;

use lib "$base/lib";
use lib "$base/blib/lib";
use lib "$base/blib/arch";
use lib "$base/t";

use Devel::Cover::Test;

my \$test = Devel::Cover::Test->new("$t");
\$test->run_test;
no warnings;
\$test  # for create_gold
EOT
    close T or die "Cannot close t/e2e/a$t.t: $!";
}
closedir D or die "Cannot closedir tests: $!";

s/^/tests\// for @tests;
push @tests, grep !/e2e/, glob "t/*/*.t";

if ($ENV{DEVEL_COVER_NO_TESTS})
{
    # don't run tests under p5cover
    print "removing all tests with DEVEL_COVER_NO_TESTS\n";
    system "rm -rf t/*";  # TODO portability
    @tests = ();
}

print "done\n\n";

my %checked;

sub check
{
    my ($module, $text, $version) = @_;

    printf "checking for %-18s %-16s .... ",
           $module, $version ? "version $version" : "";

    {
        local $SIG{__WARN__} = sub {};
        eval "use $module";
    }
    (my $mod = $module) =~ s/::/\//g;
    if (my $m = $INC{"$mod.pm"})
    {
        my $v = eval { no warnings; eval "\$${module}::VERSION" };
        printf "%-8s $m\n", $v;
        if ($version && $v < $version)
        {
            print "\n\n\n$text\n" unless $checked{$text}++;
            print "\n";
        }
    }
    else
    {
        print "not found";
        print "\n\n\n$text\n" unless $checked{$text}++;
        print "\n";
    }
};

my $d = <<EOM;
The coverage database can be stored in either Storable format or in JSON.  To
use the Storable format, the Storable module is required.  This has been a
core module since perl-5.8.0.  To store the database in JSON format JSON::PP
can be used.  This has been a core module since perl-5.14.0.  If you wish, you
may install the JSON module which will be used in preference.  This may be
configured to make database operations faster.  If both formats are available,
JSON will be preferred by default.  The JSON format should be portable between
all systems, whereas Storable format may not be.
EOM

check "Storable", $d;
check "JSON",     $d;
check "JSON::PP", $d;

check "Digest::MD5", <<EOM;
Digest::MD5 is required to check whether covered files have changed.  You can
download Digest::MD5 from CPAN.
EOM

check "Test::More", <<EOM;
Test::More is required to run the Devel::Cover tests.  You can download
Test::More from CPAN.
EOM

check "Test::Warn", <<EOM;
Test::Warn is required to run some of the Devel::Cover tests.  You can
download Test::Warn from CPAN.
EOM

check "Template", <<EOM, "2.00";
Template 2.00 is required to run the some HTML backends to cover and
for cpancover.  Unless you have specific requirements this should not be
a problem, but you will not be able to use these reports until you
install the Template Toolkit, available from CPAN.  In the meantime you
may continue to use the rest of Devel::Cover.
EOM

my $m = <<EOM;
One of PPI::HTML 1.07 or Perl::Tidy 20060719 is required to add syntax
highlighting to some HTML backends to cover and for cpancover.  Unless
you have specific requirements this should not be a problem, but you will
not be able to use syntax highlighting in these reports until you install
PPI::HTML or Perl::Tidy, available from the CPAN.  In the meantime you
may continue to use the rest of Devel::Cover.
EOM

check "PPI::HTML",  $m, "1.07";
check "Perl::Tidy", $m, "20060719";

print "checking for Pod::Coverage      version 0.06     .... ";
my $e = <<EOM;
Pod::Coverage 0.06 is required to do pod coverage.  This will tell
you how well you have documented your modules.  Pod coverage will be
unavailable until you install this module, available from CPAN.  In the
meantime, you may continue to use the rest of Devel::Cover.
EOM

eval "use Pod::Coverage";
if (my $m = $INC{"Pod/Coverage.pm"})
{
    my $v = eval { no warnings; $Pod::Coverage::VERSION };
    print $v < 0.06 ? "$v\n\n\n$e\n\n" : "$v     $m\n";
    print "checking for Pod::Coverage::CountParents         .... ";
    $e = <<EOM;
    Pod::Coverage::CountParents.pm is used for Pod coverage if it is available.
    We will fall back to using Pod::Coverage.pm.  If you want to use
    Pod::Coverage::CountParents.pm, just install it from CPAN.
EOM
    eval "use Pod::Coverage::CountParents";
    if (my $m = $INC{"Pod/Coverage/CountParents.pm"})
    {
        my $v = eval { no warnings; $Pod::Coverage::CountParents::VERSION };
        $v ||= "    ";
        print "$v     $m\n";
    }
    else
    {
        print "not found\n\n\n$e\n\n";
    }
}
else
{
    print "not found\n\n$e\n";
}

check "Test::Differences", <<EOM;
Test::Differences is used to display output from failed tests.
Hopefully there won't be any failed tests, but if there are you will get
output that may not be a model of clarity.  If you do get test failures
and you fancy helping me by debugging them, then you might like to
consider installing Test::Differences.  You can download
Test::Differences from CPAN.
EOM

check "Browser::Open", <<EOM;
Browser::Open is used to launch a web browser when the -launch flag is
specified with HTML report formats. You can download Browser::Open from CPAN.
EOM

check "CGI", <<EOM;
CGI is used to run the Html_minimal report.  It used to be a part of the perl
core, but as of perl 5.20 it will need to be downloaded from CPAN.  If you
would like to use the Html_minimal report, please install CGI.  Otherwise,
other reports will be available as usual.
EOM

my $latest_tested = "5.018002";
print <<EOM if $] > $latest_tested;

Devel::Cover $Version has not been tested with perl $].
Testing will take place against expected output from perl $latest_tested.
You may well find failing tests.

EOM

if ($] < 5.008)
{
    print <<EOM;

Devel::Cover $Version is not fully functional on perl $].  It should
mostly work, but there are some constructs for which coverage will not
be collected, and you may well encounter bugs which have been fixed in
subsequent versions of perl.  Perl versions 5.8.8 and above should work
better.

EOM

    print <<EOM if $^O eq "MSWin32";
And things are even worse under Windows.  You may well find random bugs
of various severities.

EOM
}
elsif ($] < 5.008002)
{
    print <<EOM;

Devel::Cover $Version mostly works on perl $], but you may encounter
strange behaviours (bugs) which have been fixed in subsequent versions
of perl.  This is particularly true if you are running a threaded perl.
Perl versions 5.8.2 and above should work better.

EOM
}

print "\n" x 3;

$ExtUtils::MakeMaker::Verbose = 0;

my $opts = {
    NAME             => "Devel::Cover",
    VERSION          => $Version,
    AUTHOR           => 'Paul Johnson <paul@pjcj.net>',
    ABSTRACT_FROM    => "lib/Devel/Cover.pm",
    DIR              => [],
    EXE_FILES        => [ map "bin/$_", qw( cover gcov2perl cpancover ) ],
    PERL_MALLOC_OK   => 1,
    PREREQ_PM        => {
                            Storable      => 0,
                            "Digest::MD5" => 0,
                            $ENV{DEVEL_COVER_NO_TESTS}
                                ? ()
                                : ( "Test::More"  => 0, "Test::Warn"  => 0 )
                        },
    TYPEMAPS         => [ "utils/typemap" ],
    clean            => {
                            FILES => "t/e2e/* cover_db* t/e2e/*cover_db " .
                                     "README *.gcov *.out"
                        },
    dist             => { COMPRESS => "gzip --best --force" },
    test             => {
                            TESTS => $ENV{DEVEL_COVER_NO_TESTS}
                                         ? ""
                                         : "t/*/*.t"
                        },
    realclean        => $] < 5.008008 ?
                        {
                            FILES  => "lib/Devel/Cover/Inc.pm",
                            POSTOP => "\$(RM_RF) cover_db t/e2e"
                        } :
                        { FILES => "lib/Devel/Cover/Inc.pm cover_db t/e2e" },
};
# use Data::Dumper; print Dumper $opts;
WriteMakefile(%$opts);

print "\n";
print <<EOM if 0;

--------------------------------------------------------------------------------

I like to have some idea of the distribution of this software.  To that end I
would be very grateful for mail from you.  This will be used only to satisfy my
curiosity and to help me make decisions which may involve changes to the
module.

If you can run perlbug you can send me a success report with "make ok".
Failure reports with "make nok" are also appreciated.

If you have any questions or comments, mailing list details are available in the
README file, or send mail to me directly.

EOM

sub MY::postamble
{
    my %tests;
    @tests{@tests} = map { (my $t = $_) =~ s/\W/_/g; "cover_db_$t" } @tests;
    my @reports =
      qw(compilation html_basic html_minimal html html_subtle sort text2 text
         vim);

    qq[
tags : pure_all
\t ctags --recurse --exclude=blib --exclude=Devel-Cover-* .

README : lib/Devel/Cover.pm
\t TERMCAP= COLUMNS=80 pod2text -s lib/Devel/Cover.pm | \\
    \$(PERL) -n \\
    -e 'print if (/NAME/         ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/SYNOPSIS/     ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    -e 'print if (/DESCRIPTION/  ... /^[A-Z ]+\$\$/) =~ /^\\d+\$\$/;' \\
    > README

show_version :
\t \@echo \$(VERSION)

ppm : ppd pure_all
\t tar cf Devel-Cover.tar blib
\t gzip --best --force Devel-Cover.tar
\t \$(PERL) -pi.bak \\
    -e 's/(OS NAME=")[^"]*/\$\$1MSWin32/;' \\
    -e 's/(ARCHITECTURE NAME=")[^"]*/\$\$1MSWin32-x86-multi-thread/;' \\
    -e 's/(CODEBASE HREF=")[^"]*/\$\$1Devel-Cover.tar.gz/;' \\
    Devel-Cover.ppd

TAINT =
TAINT = -T
COVER_OPTIONS =
PERL5OPT =

_run : pure_all
\t \$(PERL) \$(TAINT) -Iblib/lib -Iblib/arch -MDevel::Cover=-merge,0,`\$(PERL) -e '\$\$l = qx|grep __COVER__ \$\$ARGV[0]|; \$\$l =~ /__COVER__\\s+criteria\\s+(.*)/; (\$\$c = \$\$1 || "all") =~ s/\\s+/,/g; \$\$p = "\$\$1," if \$\$l =~ /__COVER__\\s+test_parameters\\s+(.*)/; print "\$\$p-coverage,\$\$c"' tests/\$(TEST)`,\$(COVER_OPTIONS) tests/\$(TEST)

COVER_PARAMETERS = \$(PERL) -e '\$\$l = qx|grep __COVER__ \$\$ARGV[0]|; \$\$u = "-uncoverable_file \$\$1" if \$\$l =~ /__COVER__\\s+uncoverable_file\\s+(.*)/; (\$\$p) = \$\$l =~ /__COVER__\\s+cover_parameters\\s+(.*)/; print "\$\$u \$\$p"' tests/\$(TEST)

html : _run
\t \$(PERL) -Mblib bin/cover `\$(COVER_PARAMETERS)` -report html

basic : _run
\t \$(PERL) -Mblib bin/cover `\$(COVER_PARAMETERS)` -report html_basic

out : _run
\t \$(PERL) -Mblib bin/cover `\$(COVER_PARAMETERS)` -report text > \$(TEST).out

text : out
\t \$(VISUAL) \$(TEST).out

wrun : pure_all
\t \$(PERL) \$(TAINT) -Iblib/lib -Iblib/arch -MDevel::Cover=-ignore,blib,-merge,0 tests/\$(TEST)

prove : pure_all
\t \$(PERL) -Iutils -MDevel::Cover::BuildUtils=prove_command -le '\$\$c = prove_command and print \$\$c and system \$\$c'

t : pure_all
\t \$(PERL) -Mblib bin/cover -delete
\t exec make test HARNESS_OPTIONS=j`\$(PERL) -Iutils -MDevel::Cover::BuildUtils=nice_cpus -e 'print nice_cpus'`:c HARNESS_TIMER=1

DB = cover_db

dump :
\t \$(PERL) -Mblib bin/cover -dump_db \$(DB)

FONT = "Inconsolata 10"
GEOM = 260x85+0+0

diff : out
\t \$(PERL) utils/makeh strip_criterion 'time' \$(TEST).out
\t \$(PERL) utils/makeh strip_criterion ' pod' \$(TEST).out
\t gold="`\$(PERL) -Mblib -MDevel::Cover::Test -e '\$\$t = Devel::Cover::Test->new(qq(\$(TEST))); print join qq(.), \$\$t->cover_gold'`" && gvim -geom \$(GEOM) -d -font \$(FONT) "\$\$gold" \$(TEST).out

gold : pure_all
\t \$(PERL) utils/create_gold \$(TEST)

all_test :
\t exec \$(PERL) utils/all_versions make t

all_gold :
\t \$(PERL) utils/create_all_gold \$(TEST)

_delete_db : pure_all
\t rm -rf cover_db

_self_cover_tests : @{[sort values %tests]}
\t DEVEL_COVER_SELF=1 \$(PERL) -Mblib -MDevel::Cover bin/cover -silent -write cover_db @{[sort values %tests]}

self_cover : _self_cover_reports
\t \$(PERL) -Mblib bin/cover -report html_basic -launch
\t \$(PERL) -Mblib bin/cover -report vim

ok :
\t \@$Perlbug -okay  || echo "Please send your report manually to $Author"

nok :
\t \@$Perlbug -nokay || echo "Please send your report manually to $Author"
    ] . "\n" .

    join "\n",
         map("$tests{$_} : _delete_db\n" .
             "\t \@echo Running $tests{$_}\n" .
             "\t \@rm -rf $tests{$_}\n" .
             "\t \@DEVEL_COVER_SELF=1 \$(PERL) -Mblib -MDevel::Cover=-db,$tests{$_},-silent,1,-coverage,all,-ignore,tests/,-coverage,pod-also_private-xx $_\n",
             sort keys %tests),

         "_self_cover_reports : @{[map qq(report_$_), @reports]}\n",

         map("report_$_ : _self_cover_tests\n" .
             "\t \@echo Generating $_ report\n" .
             "\t \@DEVEL_COVER_SELF=1 \$(PERL) -Mblib -MDevel::Cover bin/cover -silent -report $_ > /dev/null\n",
             @reports)
}
