/*  You may distribute under the terms of either the GNU General Public License
 *  or the Artistic License (the same terms as Perl itself)
 *
 *  (C) Paul Evans, 2023 -- leonerd@leonerd.org.uk
 */
#define PERL_NO_GET_CONTEXT

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include <tree_sitter/api.h>

#include <dlfcn.h>

typedef TSLanguage  *Text__Treesitter__Language;
typedef TSNode      *Text__Treesitter__Node;
typedef TSParser    *Text__Treesitter__Parser;
typedef TSTree      *Text__Treesitter__Tree;

MODULE = Text::Treesitter  PACKAGE = Text::Treesitter::Language

Text::Treesitter::Language load(const char *path, const char *name)
  CODE:
  {
    /* Maybe perl has some wrappings of these things we can use */
    void *langmod = dlopen(path, RTLD_LAZY|RTLD_LOCAL);
    if(!langmod)
      croak("Unable to load tree-sitter language from %s - %s",
        path, dlerror());

    SV *symnamesv = newSVpvf("tree_sitter_%s", name);
    SAVEFREESV(symnamesv);

    void *(*langfunc)(void) = dlsym(langmod, SvPVbyte_nolen(symnamesv));
    if(!langfunc)
      croak("Unable to use tree-sitter language library %s - no symbol named '%s'",
        path, SvPVbyte_nolen(symnamesv));

    RETVAL = (*langfunc)();
  }
  OUTPUT:
    RETVAL

MODULE = Text::Treesitter  PACKAGE = Text::Treesitter::Node

void DESTROY(Text::Treesitter::Node self)
  CODE:
    Safefree(self);

const char *type(Text::Treesitter::Node self)
  CODE:
    RETVAL = ts_node_type(*self);
  OUTPUT:
    RETVAL

U32 start_byte(Text::Treesitter::Node self)
  ALIAS:
    start_byte = 0
    end_byte   = 1
  CODE:
    RETVAL = ix == 0 ? ts_node_start_byte(*self) : ts_node_end_byte(*self);
  OUTPUT:
    RETVAL

void start_point(Text::Treesitter::Node self)
  ALIAS:
    start_point = 0
    end_point   = 1
  PPCODE:
  {
    TSPoint point = ix == 0 ? ts_node_start_point(*self) : ts_node_end_point(*self);
    EXTEND(SP, 2);
    mPUSHu(point.row);
    mPUSHu(point.column);
    XSRETURN(2);
  }

bool is_named(Text::Treesitter::Node self)
  ALIAS:
    is_null     = 0
    is_named    = 1
    is_missing  = 2
    is_extra    = 3
    has_changes = 4
    has_error   = 5
  CODE:
    switch(ix) {
      case 0: RETVAL = ts_node_is_null(*self);     break;
      case 1: RETVAL = ts_node_is_named(*self);    break;
      case 2: RETVAL = ts_node_is_missing(*self);  break;
      case 3: RETVAL = ts_node_is_extra(*self);    break;
      case 4: RETVAL = ts_node_has_changes(*self); break;
      case 5: RETVAL = ts_node_has_error(*self);   break;
    }
  OUTPUT:
    RETVAL

Text::Treesitter::Node parent(Text::Treesitter::Node self)
  CODE:
    Newx(RETVAL, 1, TSNode);
    *RETVAL = ts_node_parent(*self);
  OUTPUT:
    RETVAL

U32 child_count(Text::Treesitter::Node self)
  CODE:
    RETVAL = ts_node_child_count(*self);
  OUTPUT:
    RETVAL

void child_nodes(Text::Treesitter::Node self)
  ALIAS:
    child_nodes = 0
    field_names_with_child_nodes = 1
  PPCODE:
  {
    U32 nodecount = ts_node_child_count(*self);
    U32 retcount = nodecount * (ix + 1);

    EXTEND(SP, retcount);
    for(U32 i = 0; i < nodecount; i++) {
      if(ix) {
        const char *field_name = ts_node_field_name_for_child(*self, i);
        if(field_name)
          mPUSHp(field_name, strlen(field_name));
        else
          PUSHs(&PL_sv_undef);
      }

      SV *childsv = newSV(0);
      TSNode *child;
      Newx(child, 1, TSNode);
      *child = ts_node_child(*self, i);
      sv_setref_pv(childsv, "Text::Treesitter::Node", child);
      mPUSHs(childsv);
    }
    XSRETURN(retcount);
  }

MODULE = Text::Treesitter  PACKAGE = Text::Treesitter::Parser  PREFIX = ts_parser_

Text::Treesitter::Parser new(SV *cls)
  CODE:
    RETVAL = ts_parser_new();
  OUTPUT:
    RETVAL

void DESTROY(Text::Treesitter::Parser self)
  CODE:
    ts_parser_delete(self);

bool ts_parser_set_language(Text::Treesitter::Parser self, Text::Treesitter::Language lang)

Text::Treesitter::Tree parse_string(Text::Treesitter::Parser self, SV *str)
  CODE:
    SvGETMAGIC(str);

    /* TODO: upgrade if required */
    STRLEN len;
    char *pv = SvPVbyte(str, len);

    RETVAL = ts_parser_parse_string(self, NULL, pv, len);
  OUTPUT:
    RETVAL

MODULE = Text::Treesitter  PACKAGE = Text::Treesitter::Tree

void DESTROY(Text::Treesitter::Tree self)
  CODE:
    ts_tree_delete(self);

void print_dot_graph_stdout(Text::Treesitter::Tree self)
  ALIAS:
    print_dot_graph_stdout = 1
    print_dot_graph_stderr = 2
  CODE:
    ts_tree_print_dot_graph(self, ix == 1 ? stdout : stderr);

Text::Treesitter::Node root_node(Text::Treesitter::Tree self)
  CODE:
    Newx(RETVAL, 1, TSNode);
    *RETVAL = ts_tree_root_node(self);
  OUTPUT:
    RETVAL

MODULE = Text::Treesitter  PACKAGE = Text::Treesitter
