package Complete::Env;

our $DATE = '2015-09-17'; # DATE
our $VERSION = '0.37'; # VERSION

use 5.010001;
use strict;
use warnings;

use Complete::Setting;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(
                       complete_env
               );

our %SPEC;

$SPEC{':package'} = {
    v => 1.1,
    summary => 'Completion routines related to environment variables',
};

$SPEC{complete_env} = {
    v => 1.1,
    summary => 'Complete from environment variables',
    description => <<'_',

On Windows, environment variable names are all converted to uppercase. You can
use case-insensitive option (`ci`) to match against original casing.

_
    args => {
        word     => { schema=>[str=>{default=>''}], pos=>0, req=>1 },
        ci       => { schema=>['bool'] },
        fuzzy    => { schema=>['int*', min=>0] },
        map_case => { schema=>['bool'] },
    },
    result_naked => 1,
    result => {
        schema => 'array',
    },
};
sub complete_env {
    require Complete::Util;

    my %args  = @_;
    my $word     = $args{word} // "";
    my $ci       = $args{ci} // $Complete::Setting::OPT_CI;
    my $fuzzy    = $args{fuzzy} // $Complete::Setting::OPT_FUZZY;
    my $map_case = $args{map_case} // $Complete::Setting::OPT_MAP_CASE;
    if ($word =~ /^\$/) {
        Complete::Util::complete_array_elem(
            word=>$word, array=>[map {"\$$_"} keys %ENV],
            ci=>$ci, fuzzy=>$fuzzy, map_case=>$map_case);
    } else {
        Complete::Util::complete_array_elem(
            word=>$word, array=>[keys %ENV],
            ci=>$ci, fuzzy=>$fuzzy, map_case=>$map_case);
    }
}
1;
# ABSTRACT: Completion routines related to environment variables

__END__

=pod

=encoding UTF-8

=head1 NAME

Complete::Env - Completion routines related to environment variables

=head1 VERSION

This document describes version 0.37 of Complete::Env (from Perl distribution Complete-Env), released on 2015-09-17.

=head1 DESCRIPTION

=head1 FUNCTIONS


=head2 complete_env(%args) -> array

Complete from environment variables.

On Windows, environment variable names are all converted to uppercase. You can
use case-insensitive option (C<ci>) to match against original casing.

Arguments ('*' denotes required arguments):

=over 4

=item * B<ci> => I<bool>

=item * B<fuzzy> => I<int>

=item * B<map_case> => I<bool>

=item * B<word>* => I<str> (default: "")

=back

Return value:  (array)

=head1 SEE ALSO

L<Complete>

Other C<Complete::*> modules.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Complete-Env>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Complete-Env>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Complete-Env>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
