package Bencher::Scenario::Serializers;

our $DATE = '2016-01-05'; # DATE
our $VERSION = '0.06'; # VERSION

use 5.010001;
use strict;
use utf8;
use warnings;

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },
    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 bytes long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-bytes-long Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-byte long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers

=head1 VERSION

This document describes version 0.06 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2016-01-05.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 BENCHMARKED MODULES

L<JSON::PP> 2.27300

L<JSON::XS> 3.01

L<Cpanel::JSON::XS> 3.0210

L<JSON::MaybeXS> 1.003005

L<JSON::Decode::Regexp> 0.03

L<JSON::Decode::Marpa> 0.02

L<Pegex::JSON> 0.27

L<JSON::Create> 0.14

L<JSON::Parse> 0.39

L<YAML::Old> 1.07

L<YAML::Syck> 1.29

L<YAML::XS> 0.59

L<Storable> 2.53

L<Sereal> 3.014

L<Data::MessagePack> 0.49

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code)

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::XS::encode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code)

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code)

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code)

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code)

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code)

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * Pegex::JSON (perl_code)

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * JSON::Create::create_json (perl_code)

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Parse::parse_json (perl_code)

Function call template:

 JSON::Parse::parse_json(<data>)



=item * YAML::Old::Dump (perl_code)

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Load (perl_code)

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Syck::Dump (perl_code)

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Load (perl_code)

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::XS::Dump (perl_code)

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Load (perl_code)

Function call template:

 YAML::XS::Load(<data>)



=item * Storable::freeze (perl_code)

Function call template:

 Storable::freeze(<data>)



=item * Storable::thaw (perl_code)

Function call template:

 Storable::thaw(<data>)



=item * Sereal::encode_sereal (perl_code)

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code)

Function call template:

 Sereal::decode_sereal(<data>)



=item * Data::MessagePack::pack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::unpack (perl_code)

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.22.0 >>, CPU: I<< Intel(R) Core(TM) i5-2400 CPU @ 3.10GHz (4 cores) >>, OS: I<< GNU/Linux Debian version 8.0 >>, OS kernel: I<< Linux version 3.16.0-4-amd64 >>.

Benchmark with default option:

 +-----+---------------------------------------------------------------------------------+------------+-------------+---------+---------+
 | seq | name                                                                            | rate       | time        | errors  | samples |
 +-----+---------------------------------------------------------------------------------+------------+-------------+---------+---------+
 | 97  | {dataset=>"json:hash_int_1000",participant=>"JSON::Decode::Regexp::from_json"}  | 6.4        | 1.6e+02ms   | 0.00092 | 20      |
 | 114 | {dataset=>"json:hash_int_1000",participant=>"Pegex::JSON"}                      | 15.6       | 64ms        | 5.2e-05 | 20      |
 | 104 | {dataset=>"json:hash_int_1000",participant=>"JSON::Decode::Marpa::from_json"}   | 18         | 55.5ms      | 0.00013 | 21      |
 | 148 | {dataset=>"hash_int_1000",participant=>"YAML::Old::Dump"}                       | 23.5       | 42.6ms      | 5.2e-05 | 20      |
 | 110 | {dataset=>"json:array_int_1000",participant=>"Pegex::JSON"}                     | 27.6       | 36.3ms      | 2e-05   | 24      |
 | 143 | {dataset=>"array_int_1000",participant=>"YAML::Old::Dump"}                      | 30.5       | 32.8ms      | 5.8e-05 | 21      |
 | 93  | {dataset=>"json:array_int_1000",participant=>"JSON::Decode::Regexp::from_json"} | 36         | 28ms        | 0.00014 | 20      |
 | 100 | {dataset=>"json:array_int_1000",participant=>"JSON::Decode::Marpa::from_json"}  | 40.8       | 24.5ms      | 2.1e-05 | 21      |
 | 21  | {dataset=>"json:hash_int_1000",participant=>"JSON::PP::decode_json"}            | 88.5       | 11.3ms      | 9.2e-06 | 20      |
 | 101 | {dataset=>"json:array_str1k_10",participant=>"JSON::Decode::Marpa::from_json"}  | 93.9       | 10.6ms      | 1.8e-05 | 20      |
 | 18  | {dataset=>"json:array_str1k_10",participant=>"JSON::PP::decode_json"}           | 132        | 7.56ms      | 8.1e-06 | 20      |
 | 113 | {dataset=>"json:hash_int_100",participant=>"Pegex::JSON"}                       | 149        | 6.72ms      | 5.4e-06 | 20      |
 | 103 | {dataset=>"json:hash_int_100",participant=>"JSON::Decode::Marpa::from_json"}    | 167        | 5.98ms      | 8.3e-06 | 20      |
 | 17  | {dataset=>"json:array_int_1000",participant=>"JSON::PP::decode_json"}           | 176        | 5.67ms      | 5.6e-06 | 20      |
 | 11  | {dataset=>"hash_int_1000",participant=>"JSON::PP::encode_json"}                 | 225        | 4.44ms      | 2.2e-06 | 20      |
 | 147 | {dataset=>"hash_int_100",participant=>"YAML::Old::Dump"}                        | 231        | 4.33ms      | 5.7e-06 | 21      |
 | 109 | {dataset=>"json:array_int_100",participant=>"Pegex::JSON"}                      | 255        | 3.92ms      | 4.7e-06 | 20      |
 | 142 | {dataset=>"array_int_100",participant=>"YAML::Old::Dump"}                       | 302        | 3.31ms      | 3.4e-06 | 22      |
 | 96  | {dataset=>"json:hash_int_100",participant=>"JSON::Decode::Regexp::from_json"}   | 309        | 3.23ms      | 2.4e-06 | 21      |
 | 99  | {dataset=>"json:array_int_100",participant=>"JSON::Decode::Marpa::from_json"}   | 330        | 3.03ms      | 3.8e-06 | 20      |
 | 6   | {dataset=>"array_int_1000",participant=>"JSON::PP::encode_json"}                | 594        | 1.68ms      | 1.8e-06 | 20      |
 | 111 | {dataset=>"json:array_str1k_10",participant=>"Pegex::JSON"}                     | 6.1e+02    | 1.6ms       | 6.2e-06 | 20      |
 | 160 | {dataset=>"hash_int_1000",participant=>"YAML::Syck::Dump"}                      | 655        | 1.53ms      | 1.4e-06 | 24      |
 | 172 | {dataset=>"hash_int_1000",participant=>"YAML::XS::Dump"}                        | 671        | 1.49ms      | 1.1e-06 | 20      |
 | 102 | {dataset=>"json:hash_int_10",participant=>"JSON::Decode::Marpa::from_json"}     | 762        | 1.31ms      | 9.6e-07 | 20      |
 | 145 | {dataset=>"array_ustr1k_10",participant=>"YAML::Old::Dump"}                     | 848        | 1.18ms      | 1.3e-06 | 20      |
 | 92  | {dataset=>"json:array_int_100",participant=>"JSON::Decode::Regexp::from_json"}  | 870        | 1.15ms      | 1.8e-06 | 20      |
 | 112 | {dataset=>"json:hash_int_10",participant=>"Pegex::JSON"}                        | 919        | 1.09ms      | 1.3e-06 | 21      |
 | 20  | {dataset=>"json:hash_int_100",participant=>"JSON::PP::decode_json"}             | 961        | 1.04ms      | 6.4e-07 | 20      |
 | 98  | {dataset=>"json:array_int_10",participant=>"JSON::Decode::Marpa::from_json"}    | 985        | 1.02ms      | 4.8e-07 | 20      |
 | 108 | {dataset=>"json:array_int_10",participant=>"Pegex::JSON"}                       | 1.24e+03   | 0.808ms     | 9.1e-07 | 20      |
 | 14  | {dataset=>"json:str1k",participant=>"JSON::PP::decode_json"}                    | 1.32e+03   | 0.759ms     | 1.1e-06 | 20      |
 | 155 | {dataset=>"array_int_1000",participant=>"YAML::Syck::Dump"}                     | 1.59e+03   | 0.627ms     | 6.9e-07 | 20      |
 | 167 | {dataset=>"array_int_1000",participant=>"YAML::XS::Dump"}                       | 1.7e+03    | 0.6ms       | 3.1e-06 | 20      |
 | 8   | {dataset=>"array_ustr1k_10",participant=>"JSON::PP::encode_json"}               | 1889       | 0.5294ms    | 5.3e-08 | 20      |
 | 16  | {dataset=>"json:array_int_100",participant=>"JSON::PP::decode_json"}            | 1.92e+03   | 0.521ms     | 2.5e-07 | 22      |
 | 144 | {dataset=>"array_str1k_10",participant=>"YAML::Old::Dump"}                      | 1.93e+03   | 0.519ms     | 6.4e-07 | 20      |
 | 146 | {dataset=>"hash_int_10",participant=>"YAML::Old::Dump"}                         | 1.94e+03   | 0.517ms     | 6.9e-07 | 20      |
 | 107 | {dataset=>"json:str1k",participant=>"Pegex::JSON"}                              | 1.98e+03   | 0.504ms     | 6.4e-07 | 20      |
 | 105 | {dataset=>"json:null",participant=>"Pegex::JSON"}                               | 2.18e+03   | 0.458ms     | 8.1e-07 | 22      |
 | 10  | {dataset=>"hash_int_100",participant=>"JSON::PP::encode_json"}                  | 2.21e+03   | 0.452ms     | 4.3e-07 | 20      |
 | 106 | {dataset=>"json:num",participant=>"Pegex::JSON"}                                | 2.3e+03    | 0.435ms     | 4.3e-07 | 20      |
 | 141 | {dataset=>"array_int_10",participant=>"YAML::Old::Dump"}                        | 2.46e+03   | 0.407ms     | 1.1e-06 | 21      |
 | 52  | {dataset=>"array_ustr1k_10",participant=>"Cpanel::JSON::XS::encode_json"}       | 2.7e+03    | 0.371ms     | 4.1e-07 | 22      |
 | 74  | {dataset=>"array_ustr1k_10",participant=>"JSON::MaybeXS::encode_json"}          | 2.7e+03    | 0.37ms      | 6.4e-07 | 20      |
 | 30  | {dataset=>"array_ustr1k_10",participant=>"JSON::XS::encode_json"}               | 2.74e+03   | 0.365ms     | 2.1e-07 | 20      |
 | 169 | {dataset=>"array_ustr1k_10",participant=>"YAML::XS::Dump"}                      | 3.05e+03   | 0.328ms     | 1.6e-07 | 20      |
 | 94  | {dataset=>"json:array_str1k_10",participant=>"JSON::Decode::Regexp::from_json"} | 3.44e+03   | 0.291ms     | 4.3e-07 | 20      |
 | 192 | {dataset=>"hash_int_1000",participant=>"Sereal::encode_sereal"}                 | 3.52e+03   | 0.284ms     | 8.8e-07 | 21      |
 | 95  | {dataset=>"json:hash_int_10",participant=>"JSON::Decode::Regexp::from_json"}    | 3.6e+03    | 0.28ms      | 1.2e-06 | 20      |
 | 204 | {dataset=>"hash_int_1000",participant=>"Data::MessagePack::pack"}               | 4e+03      | 0.25ms      | 2.1e-07 | 20      |
 | 180 | {dataset=>"hash_int_1000",participant=>"Storable::freeze"}                      | 4.07e+03   | 0.245ms     | 2.1e-07 | 21      |
 | 55  | {dataset=>"hash_int_1000",participant=>"Cpanel::JSON::XS::encode_json"}         | 4.48e+03   | 0.223ms     | 2.5e-07 | 22      |
 | 77  | {dataset=>"hash_int_1000",participant=>"JSON::MaybeXS::encode_json"}            | 4.52e+03   | 0.221ms     | 2.1e-07 | 20      |
 | 33  | {dataset=>"hash_int_1000",participant=>"JSON::XS::encode_json"}                 | 4.54e+03   | 0.22ms      | 4.3e-07 | 20      |
 | 126 | {dataset=>"hash_int_1000",participant=>"JSON::Create::create_json"}             | 4.65e+03   | 0.215ms     | 2.1e-07 | 20      |
 | 5   | {dataset=>"array_int_100",participant=>"JSON::PP::encode_json"}                 | 5.68e+03   | 0.176ms     | 2.1e-07 | 20      |
 | 91  | {dataset=>"json:array_int_10",participant=>"JSON::Decode::Regexp::from_json"}   | 5.75e+03   | 0.174ms     | 2.1e-07 | 20      |
 | 140 | {dataset=>"str1k #3",participant=>"YAML::Old::Dump"}                            | 6.63e+03   | 0.151ms     | 2.1e-07 | 21      |
 | 159 | {dataset=>"hash_int_100",participant=>"YAML::Syck::Dump"}                       | 6.73e+03   | 0.149ms     | 2e-07   | 22      |
 | 87  | {dataset=>"json:hash_int_1000",participant=>"JSON::MaybeXS::decode_json"}       | 6.93e+03   | 0.144ms     | 2.1e-07 | 20      |
 | 43  | {dataset=>"json:hash_int_1000",participant=>"JSON::XS::decode_json"}            | 6.94e+03   | 0.144ms     | 2e-07   | 22      |
 | 65  | {dataset=>"json:hash_int_1000",participant=>"Cpanel::JSON::XS::decode_json"}    | 7e+03      | 0.14ms      | 4.8e-07 | 20      |
 | 136 | {dataset=>"json:hash_int_1000",participant=>"JSON::Parse::parse_json"}          | 7e+03      | 0.143ms     | 1.6e-07 | 20      |
 | 171 | {dataset=>"hash_int_100",participant=>"YAML::XS::Dump"}                         | 7.01e+03   | 0.143ms     | 1.6e-07 | 20      |
 | 157 | {dataset=>"array_ustr1k_10",participant=>"YAML::Syck::Dump"}                    | 7.13e+03   | 0.14ms      | 2.1e-07 | 20      |
 | 168 | {dataset=>"array_str1k_10",participant=>"YAML::XS::Dump"}                       | 7.3e+03    | 0.14ms      | 4.8e-07 | 20      |
 | 19  | {dataset=>"json:hash_int_10",participant=>"JSON::PP::decode_json"}              | 9195.23    | 0.108752ms  | 4.6e-11 | 20      |
 | 69  | {dataset=>"str1k #3",participant=>"JSON::MaybeXS::encode_json"}                 | 9.95e+03   | 0.101ms     | 1.9e-07 | 20      |
 | 47  | {dataset=>"str1k #3",participant=>"Cpanel::JSON::XS::encode_json"}              | 1e+04      | 0.0996ms    | 1.1e-07 | 20      |
 | 25  | {dataset=>"str1k #3",participant=>"JSON::XS::encode_json"}                      | 1.01e+04   | 0.0991ms    | 1.2e-07 | 25      |
 | 7   | {dataset=>"array_str1k_10",participant=>"JSON::PP::encode_json"}                | 1e+04      | 0.096ms     | 4.5e-07 | 20      |
 | 90  | {dataset=>"json:str1k",participant=>"JSON::Decode::Regexp::from_json"}          | 1.1e+04    | 0.09ms      | 3.5e-07 | 20      |
 | 139 | {dataset=>"str1k",participant=>"YAML::Old::Dump"}                               | 1.19e+04   | 0.0842ms    | 1.1e-07 | 20      |
 | 89  | {dataset=>"json:num",participant=>"JSON::Decode::Regexp::from_json"}            | 1.23e+04   | 0.0815ms    | 1.2e-07 | 23      |
 | 138 | {dataset=>"num",participant=>"YAML::Old::Dump"}                                 | 1.24e+04   | 0.0809ms    | 8e-08   | 20      |
 | 175 | {dataset=>"array_int_1000",participant=>"Storable::freeze"}                     | 1.37e+04   | 0.0732ms    | 8e-08   | 20      |
 | 123 | {dataset=>"array_ustr1k_10",participant=>"JSON::Create::create_json"}           | 1.43e+04   | 0.0701ms    | 1.6e-07 | 26      |
 | 137 | {dataset=>"undef",participant=>"YAML::Old::Dump"}                               | 1.43e+04   | 0.07ms      | 1.1e-07 | 20      |
 | 88  | {dataset=>"json:null",participant=>"JSON::Decode::Regexp::from_json"}           | 1.5e+04    | 0.067ms     | 6.5e-07 | 21      |
 | 154 | {dataset=>"array_int_100",participant=>"YAML::Syck::Dump"}                      | 1.6e+04    | 0.063ms     | 2.3e-07 | 21      |
 | 166 | {dataset=>"array_int_100",participant=>"YAML::XS::Dump"}                        | 1.62e+04   | 0.0618ms    | 2.7e-08 | 20      |
 | 15  | {dataset=>"json:array_int_10",participant=>"JSON::PP::decode_json"}             | 1.74e+04   | 0.0576ms    | 2.7e-08 | 20      |
 | 72  | {dataset=>"array_int_1000",participant=>"JSON::MaybeXS::encode_json"}           | 1.81e+04   | 0.0553ms    | 2.7e-08 | 20      |
 | 3   | {dataset=>"str1k #3",participant=>"JSON::PP::encode_json"}                      | 1.83e+04   | 0.0547ms    | 1.1e-07 | 20      |
 | 50  | {dataset=>"array_int_1000",participant=>"Cpanel::JSON::XS::encode_json"}        | 18496.3    | 0.0540649ms | 0       | 20      |
 | 9   | {dataset=>"hash_int_10",participant=>"JSON::PP::encode_json"}                   | 1.89e+04   | 0.053ms     | 1.1e-07 | 20      |
 | 28  | {dataset=>"array_int_1000",participant=>"JSON::XS::encode_json"}                | 1.9e+04    | 0.0527ms    | 2.7e-08 | 20      |
 | 156 | {dataset=>"array_str1k_10",participant=>"YAML::Syck::Dump"}                     | 1.96e+04   | 0.0511ms    | 6.5e-08 | 21      |
 | 121 | {dataset=>"array_int_1000",participant=>"JSON::Create::create_json"}            | 2.18e+04   | 0.0458ms    | 5.2e-08 | 21      |
 | 199 | {dataset=>"array_int_1000",participant=>"Data::MessagePack::pack"}              | 2.65e+04   | 0.0378ms    | 5.3e-08 | 20      |
 | 187 | {dataset=>"array_int_1000",participant=>"Sereal::encode_sereal"}                | 2.7e+04    | 0.037ms     | 2.1e-07 | 21      |
 | 164 | {dataset=>"str1k #3",participant=>"YAML::XS::Dump"}                             | 2.84e+04   | 0.0352ms    | 1.3e-08 | 20      |
 | 61  | {dataset=>"json:array_int_1000",participant=>"Cpanel::JSON::XS::decode_json"}   | 2.93e+04   | 0.0341ms    | 1.3e-08 | 20      |
 | 132 | {dataset=>"json:array_int_1000",participant=>"JSON::Parse::parse_json"}         | 2.98e+04   | 0.0336ms    | 5.3e-08 | 20      |
 | 83  | {dataset=>"json:array_int_1000",participant=>"JSON::MaybeXS::decode_json"}      | 2.98e+04   | 0.0336ms    | 5.3e-08 | 20      |
 | 39  | {dataset=>"json:array_int_1000",participant=>"JSON::XS::decode_json"}           | 3.05e+04   | 0.0328ms    | 6.4e-08 | 22      |
 | 133 | {dataset=>"json:array_str1k_10",participant=>"JSON::Parse::parse_json"}         | 3.59e+04   | 0.0279ms    | 1.3e-08 | 21      |
 | 179 | {dataset=>"hash_int_100",participant=>"Storable::freeze"}                       | 3.9e+04    | 0.0257ms    | 4.9e-08 | 24      |
 | 122 | {dataset=>"array_str1k_10",participant=>"JSON::Create::create_json"}            | 4.024e+04  | 0.02485ms   | 6.4e-09 | 22      |
 | 191 | {dataset=>"hash_int_100",participant=>"Sereal::encode_sereal"}                  | 4.1e+04    | 0.024ms     | 8.7e-08 | 20      |
 | 73  | {dataset=>"array_str1k_10",participant=>"JSON::MaybeXS::encode_json"}           | 4.16e+04   | 0.024ms     | 2.7e-08 | 20      |
 | 51  | {dataset=>"array_str1k_10",participant=>"Cpanel::JSON::XS::encode_json"}        | 4.174e+04  | 0.02396ms   | 6.7e-09 | 20      |
 | 4   | {dataset=>"array_int_10",participant=>"JSON::PP::encode_json"}                  | 4.27e+04   | 0.0234ms    | 2e-08   | 20      |
 | 29  | {dataset=>"array_str1k_10",participant=>"JSON::XS::encode_json"}                | 4.29e+04   | 0.02331ms   | 6.7e-09 | 20      |
 | 203 | {dataset=>"hash_int_100",participant=>"Data::MessagePack::pack"}                | 4.43e+04   | 0.0226ms    | 2.7e-08 | 20      |
 | 62  | {dataset=>"json:array_str1k_10",participant=>"Cpanel::JSON::XS::decode_json"}   | 5.114e+04  | 0.01956ms   | 5.6e-09 | 28      |
 | 84  | {dataset=>"json:array_str1k_10",participant=>"JSON::MaybeXS::decode_json"}      | 5.14e+04   | 0.0195ms    | 6.5e-09 | 21      |
 | 158 | {dataset=>"hash_int_10",participant=>"YAML::Syck::Dump"}                        | 5.19e+04   | 0.0193ms    | 5.1e-08 | 22      |
 | 32  | {dataset=>"hash_int_100",participant=>"JSON::XS::encode_json"}                  | 5.27e+04   | 0.019ms     | 2.7e-08 | 20      |
 | 152 | {dataset=>"str1k #3",participant=>"YAML::Syck::Dump"}                           | 53852.9    | 0.0185691ms | 1.2e-11 | 25      |
 | 54  | {dataset=>"hash_int_100",participant=>"Cpanel::JSON::XS::encode_json"}          | 5.4e+04    | 0.018ms     | 6e-08   | 20      |
 | 76  | {dataset=>"hash_int_100",participant=>"JSON::MaybeXS::encode_json"}             | 5.43e+04   | 0.0184ms    | 3.3e-08 | 21      |
 | 125 | {dataset=>"hash_int_100",participant=>"JSON::Create::create_json"}              | 5.83e+04   | 0.0172ms    | 2.7e-08 | 20      |
 | 170 | {dataset=>"hash_int_10",participant=>"YAML::XS::Dump"}                          | 5.9e+04    | 0.017ms     | 2.7e-08 | 20      |
 | 163 | {dataset=>"str1k",participant=>"YAML::XS::Dump"}                                | 62628.5    | 0.0159672ms | 1e-11   | 20      |
 | 40  | {dataset=>"json:array_str1k_10",participant=>"JSON::XS::decode_json"}           | 6.6e+04    | 0.015ms     | 7.8e-08 | 21      |
 | 13  | {dataset=>"json:num",participant=>"JSON::PP::decode_json"}                      | 7.2e+04    | 0.014ms     | 5.1e-08 | 22      |
 | 153 | {dataset=>"array_int_10",participant=>"YAML::Syck::Dump"}                       | 8.39e+04   | 0.0119ms    | 1e-08   | 20      |
 | 2   | {dataset=>"str1k",participant=>"JSON::PP::encode_json"}                         | 8.93e+04   | 0.0112ms    | 1.3e-08 | 20      |
 | 135 | {dataset=>"json:hash_int_100",participant=>"JSON::Parse::parse_json"}           | 8.98e+04   | 0.0111ms    | 1.3e-08 | 20      |
 | 64  | {dataset=>"json:hash_int_100",participant=>"Cpanel::JSON::XS::decode_json"}     | 9e+04      | 0.0111ms    | 1e-08   | 20      |
 | 42  | {dataset=>"json:hash_int_100",participant=>"JSON::XS::decode_json"}             | 9.1e+04    | 0.011ms     | 2.4e-08 | 32      |
 | 86  | {dataset=>"json:hash_int_100",participant=>"JSON::MaybeXS::decode_json"}        | 9.11e+04   | 0.011ms     | 1.5e-08 | 26      |
 | 151 | {dataset=>"str1k",participant=>"YAML::Syck::Dump"}                              | 1.02e+05   | 0.00983ms   | 3.3e-09 | 20      |
 | 174 | {dataset=>"array_int_100",participant=>"Storable::freeze"}                      | 1.05e+05   | 0.00951ms   | 3.3e-09 | 20      |
 | 165 | {dataset=>"array_int_10",participant=>"YAML::XS::Dump"}                         | 1.06e+05   | 0.00946ms   | 1.3e-08 | 20      |
 | 177 | {dataset=>"array_ustr1k_10",participant=>"Storable::freeze"}                    | 1.22e+05   | 0.00821ms   | 1e-08   | 20      |
 | 12  | {dataset=>"json:null",participant=>"JSON::PP::decode_json"}                     | 1.33e+05   | 0.00752ms   | 2.9e-09 | 26      |
 | 150 | {dataset=>"num",participant=>"YAML::Syck::Dump"}                                | 1.35e+05   | 0.00741ms   | 3e-09   | 24      |
 | 178 | {dataset=>"hash_int_10",participant=>"Storable::freeze"}                        | 1.39e+05   | 0.0072ms    | 1e-08   | 20      |
 | 118 | {dataset=>"str1k #3",participant=>"JSON::Create::create_json"}                  | 1.43e+05   | 0.00698ms   | 1.3e-08 | 20      |
 | 176 | {dataset=>"array_str1k_10",participant=>"Storable::freeze"}                     | 1.6e+05    | 0.00624ms   | 1.6e-08 | 23      |
 | 149 | {dataset=>"undef",participant=>"YAML::Syck::Dump"}                              | 1.734e+05  | 0.005768ms  | 1.6e-09 | 21      |
 | 71  | {dataset=>"array_int_100",participant=>"JSON::MaybeXS::encode_json"}            | 1.77e+05   | 0.00566ms   | 8.1e-09 | 21      |
 | 49  | {dataset=>"array_int_100",participant=>"Cpanel::JSON::XS::encode_json"}         | 1.78e+05   | 0.00562ms   | 5e-09   | 20      |
 | 27  | {dataset=>"array_int_100",participant=>"JSON::XS::encode_json"}                 | 1.79e+05   | 0.0056ms    | 6e-09   | 25      |
 | 201 | {dataset=>"array_ustr1k_10",participant=>"Data::MessagePack::pack"}             | 1.79e+05   | 0.00558ms   | 6.7e-09 | 20      |
 | 189 | {dataset=>"array_ustr1k_10",participant=>"Sereal::encode_sereal"}               | 1.846e+05  | 0.005417ms  | 1.7e-09 | 20      |
 | 1   | {dataset=>"num",participant=>"JSON::PP::encode_json"}                           | 1.89e+05   | 0.00528ms   | 6.7e-09 | 20      |
 | 162 | {dataset=>"num",participant=>"YAML::XS::Dump"}                                  | 2.08e+05   | 0.00481ms   | 5e-09   | 20      |
 | 200 | {dataset=>"array_str1k_10",participant=>"Data::MessagePack::pack"}              | 2.32e+05   | 0.0043ms    | 1.7e-09 | 20      |
 | 173 | {dataset=>"array_int_10",participant=>"Storable::freeze"}                       | 2.34e+05   | 0.00428ms   | 6.7e-09 | 20      |
 | 198 | {dataset=>"array_int_100",participant=>"Data::MessagePack::pack"}               | 2.3529e+05 | 0.0042501ms | 4.6e-11 | 21      |
 | 120 | {dataset=>"array_int_100",participant=>"JSON::Create::create_json"}             | 2.36e+05   | 0.00424ms   | 6.7e-09 | 20      |
 | 186 | {dataset=>"array_int_100",participant=>"Sereal::encode_sereal"}                 | 2.45e+05   | 0.00409ms   | 7.5e-09 | 25      |
 | 188 | {dataset=>"array_str1k_10",participant=>"Sereal::encode_sereal"}                | 2.48e+05   | 0.00403ms   | 5e-09   | 20      |
 | 131 | {dataset=>"json:array_int_100",participant=>"JSON::Parse::parse_json"}          | 2.51e+05   | 0.00399ms   | 6.7e-09 | 20      |
 | 60  | {dataset=>"json:array_int_100",participant=>"Cpanel::JSON::XS::decode_json"}    | 2.5839e+05 | 0.0038701ms | 4.6e-11 | 20      |
 | 38  | {dataset=>"json:array_int_100",participant=>"JSON::XS::decode_json"}            | 2.62e+05   | 0.00381ms   | 6.7e-09 | 20      |
 | 82  | {dataset=>"json:array_int_100",participant=>"JSON::MaybeXS::decode_json"}       | 2.66e+05   | 0.00376ms   | 6.7e-09 | 20      |
 | 0   | {dataset=>"undef",participant=>"JSON::PP::encode_json"}                         | 2.99e+05   | 0.00334ms   | 1.7e-09 | 20      |
 | 129 | {dataset=>"json:str1k",participant=>"JSON::Parse::parse_json"}                  | 3.51e+05   | 0.00285ms   | 3e-09   | 24      |
 | 117 | {dataset=>"str1k",participant=>"JSON::Create::create_json"}                     | 3.8159e+05 | 0.0026206ms | 1.2e-11 | 20      |
 | 190 | {dataset=>"hash_int_10",participant=>"Sereal::encode_sereal"}                   | 4.03e+05   | 0.00248ms   | 6.8e-09 | 24      |
 | 46  | {dataset=>"str1k",participant=>"Cpanel::JSON::XS::encode_json"}                 | 4.06e+05   | 0.00246ms   | 3.3e-09 | 20      |
 | 68  | {dataset=>"str1k",participant=>"JSON::MaybeXS::encode_json"}                    | 4.06e+05   | 0.00246ms   | 3.3e-09 | 20      |
 | 202 | {dataset=>"hash_int_10",participant=>"Data::MessagePack::pack"}                 | 4.24e+05   | 0.00236ms   | 2.4e-09 | 21      |
 | 161 | {dataset=>"undef",participant=>"YAML::XS::Dump"}                                | 4.26e+05   | 0.00235ms   | 3.3e-09 | 20      |
 | 24  | {dataset=>"str1k",participant=>"JSON::XS::encode_json"}                         | 4.4e+05    | 0.00227ms   | 3.3e-09 | 20      |
 | 75  | {dataset=>"hash_int_10",participant=>"JSON::MaybeXS::encode_json"}              | 4.78e+05   | 0.00209ms   | 3.3e-09 | 20      |
 | 53  | {dataset=>"hash_int_10",participant=>"Cpanel::JSON::XS::encode_json"}           | 4.83e+05   | 0.00207ms   | 4.2e-09 | 20      |
 | 31  | {dataset=>"hash_int_10",participant=>"JSON::XS::encode_json"}                   | 4.9e+05    | 0.002ms     | 7.3e-09 | 21      |
 | 58  | {dataset=>"json:str1k",participant=>"Cpanel::JSON::XS::decode_json"}            | 5.08e+05   | 0.00197ms   | 7.1e-10 | 28      |
 | 124 | {dataset=>"hash_int_10",participant=>"JSON::Create::create_json"}               | 5.12e+05   | 0.00195ms   | 8.3e-10 | 20      |
 | 80  | {dataset=>"json:str1k",participant=>"JSON::MaybeXS::decode_json"}               | 5.2605e+05 | 0.001901ms  | 4.2e-11 | 20      |
 | 36  | {dataset=>"json:str1k",participant=>"JSON::XS::decode_json"}                    | 6.58e+05   | 0.00152ms   | 6.8e-10 | 30      |
 | 134 | {dataset=>"json:hash_int_10",participant=>"JSON::Parse::parse_json"}            | 8.14e+05   | 0.00123ms   | 1.7e-09 | 20      |
 | 63  | {dataset=>"json:hash_int_10",participant=>"Cpanel::JSON::XS::decode_json"}      | 8.33e+05   | 0.0012ms    | 4.2e-10 | 20      |
 | 85  | {dataset=>"json:hash_int_10",participant=>"JSON::MaybeXS::decode_json"}         | 8.441e+05  | 0.001185ms  | 4.5e-11 | 26      |
 | 41  | {dataset=>"json:hash_int_10",participant=>"JSON::XS::decode_json"}              | 8.628e+05  | 0.001159ms  | 4.6e-11 | 20      |
 | 35  | {dataset=>"json:num",participant=>"JSON::XS::decode_json"}                      | 9.23e+05   | 0.00108ms   | 2.1e-09 | 20      |
 | 57  | {dataset=>"json:num",participant=>"Cpanel::JSON::XS::decode_json"}              | 9.25e+05   | 0.00108ms   | 4.2e-10 | 20      |
 | 79  | {dataset=>"json:num",participant=>"JSON::MaybeXS::decode_json"}                 | 9.29e+05   | 0.00108ms   | 2.1e-09 | 20      |
 | 128 | {dataset=>"json:num",participant=>"JSON::Parse::parse_json"}                    | 9.77e+05   | 0.00102ms   | 2.1e-09 | 20      |
 | 185 | {dataset=>"array_int_10",participant=>"Sereal::encode_sereal"}                  | 1.05e+06   | 0.000956ms  | 1.7e-09 | 20      |
 | 48  | {dataset=>"array_int_10",participant=>"Cpanel::JSON::XS::encode_json"}          | 1.24e+06   | 0.000808ms  | 1.7e-09 | 20      |
 | 70  | {dataset=>"array_int_10",participant=>"JSON::MaybeXS::encode_json"}             | 1.26e+06   | 0.000793ms  | 3.8e-10 | 24      |
 | 26  | {dataset=>"array_int_10",participant=>"JSON::XS::encode_json"}                  | 1.28e+06   | 0.000781ms  | 1.7e-09 | 20      |
 | 67  | {dataset=>"num",participant=>"JSON::MaybeXS::encode_json"}                      | 1.29e+06   | 0.000774ms  | 1.3e-09 | 20      |
 | 45  | {dataset=>"num",participant=>"Cpanel::JSON::XS::encode_json"}                   | 1.299e+06  | 0.0007698ms | 4.7e-11 | 20      |
 | 23  | {dataset=>"num",participant=>"JSON::XS::encode_json"}                           | 1.32e+06   | 0.000759ms  | 1.7e-09 | 20      |
 | 197 | {dataset=>"array_int_10",participant=>"Data::MessagePack::pack"}                | 1.32e+06   | 0.000758ms  | 4.2e-10 | 20      |
 | 59  | {dataset=>"json:array_int_10",participant=>"Cpanel::JSON::XS::decode_json"}     | 1.42e+06   | 0.000702ms  | 6.3e-10 | 20      |
 | 130 | {dataset=>"json:array_int_10",participant=>"JSON::Parse::parse_json"}           | 1.43e+06   | 0.000701ms  | 4e-10   | 22      |
 | 81  | {dataset=>"json:array_int_10",participant=>"JSON::MaybeXS::decode_json"}        | 1.439e+06  | 0.0006948ms | 6.9e-11 | 20      |
 | 119 | {dataset=>"array_int_10",participant=>"JSON::Create::create_json"}              | 1.49e+06   | 0.000671ms  | 8.3e-10 | 20      |
 | 37  | {dataset=>"json:array_int_10",participant=>"JSON::XS::decode_json"}             | 1.53e+06   | 0.000656ms  | 2.1e-10 | 20      |
 | 184 | {dataset=>"str1k #3",participant=>"Sereal::encode_sereal"}                      | 1.65e+06   | 0.000604ms  | 1.9e-09 | 20      |
 | 183 | {dataset=>"str1k",participant=>"Sereal::encode_sereal"}                         | 1.81e+06   | 0.000553ms  | 2.1e-10 | 20      |
 | 116 | {dataset=>"num",participant=>"JSON::Create::create_json"}                       | 2.03e+06   | 0.000492ms  | 2.1e-10 | 21      |
 | 182 | {dataset=>"num",participant=>"Sereal::encode_sereal"}                           | 2.17e+06   | 0.000461ms  | 2.1e-10 | 21      |
 | 196 | {dataset=>"str1k #3",participant=>"Data::MessagePack::pack"}                    | 2.39e+06   | 0.000418ms  | 8e-10   | 34      |
 | 181 | {dataset=>"undef",participant=>"Sereal::encode_sereal"}                         | 2.533e+06  | 0.0003947ms | 4.6e-11 | 20      |
 | 195 | {dataset=>"str1k",participant=>"Data::MessagePack::pack"}                       | 2.862e+06  | 0.0003494ms | 4.6e-11 | 20      |
 | 194 | {dataset=>"num",participant=>"Data::MessagePack::pack"}                         | 4.43e+06   | 0.000226ms  | 2e-10   | 23      |
 | 66  | {dataset=>"undef",participant=>"JSON::MaybeXS::encode_json"}                    | 4.888e+06  | 0.0002046ms | 4.5e-11 | 23      |
 | 44  | {dataset=>"undef",participant=>"Cpanel::JSON::XS::encode_json"}                 | 4.909e+06  | 0.0002037ms | 4.6e-11 | 20      |
 | 22  | {dataset=>"undef",participant=>"JSON::XS::encode_json"}                         | 5.021e+06  | 0.0001992ms | 4.7e-11 | 20      |
 | 193 | {dataset=>"undef",participant=>"Data::MessagePack::pack"}                       | 5.09e+06   | 0.000196ms  | 1.4e-10 | 22      |
 | 115 | {dataset=>"undef",participant=>"JSON::Create::create_json"}                     | 5.57e+06   | 0.0001795ms | 4.6e-11 | 20      |
 | 56  | {dataset=>"json:null",participant=>"Cpanel::JSON::XS::decode_json"}             | 6.94e+06   | 0.000144ms  | 4.6e-11 | 20      |
 | 34  | {dataset=>"json:null",participant=>"JSON::XS::decode_json"}                     | 7.01e+06   | 0.000143ms  | 4.6e-11 | 31      |
 | 78  | {dataset=>"json:null",participant=>"JSON::MaybeXS::decode_json"}                | 7.05e+06   | 0.000142ms  | 1.1e-10 | 22      |
 | 127 | {dataset=>"json:null",participant=>"JSON::Parse::parse_json"}                   | 7.651e+06  | 0.0001307ms | 1.1e-11 | 23      |
 +-----+---------------------------------------------------------------------------------+------------+-------------+---------+---------+


Benchmark module startup overhead:

 +-----+----------------------+-----------+-------------------+---------+---------+
 | seq | name                 | time      | mod_overhead_time | errors  | samples |
 +-----+----------------------+-----------+-------------------+---------+---------+
 | 6   | JSON::Decode::Marpa  | 1.3e+02ms | 1.2e+02ms         | 0.00053 | 20      |
 | 12  | YAML::XS             | 54.1ms    | 47.2ms            | 0.00017 | 20      |
 | 1   | JSON::PP             | 27ms      | 20.1ms            | 8.1e-05 | 20      |
 | 7   | Pegex::JSON          | 24ms      | 17ms              | 8.3e-05 | 20      |
 | 4   | JSON::MaybeXS        | 18.9ms    | 12ms              | 5.5e-05 | 20      |
 | 13  | Storable             | 16.9ms    | 10ms              | 3e-05   | 20      |
 | 14  | Sereal               | 16.8ms    | 9.91ms            | 3.4e-05 | 20      |
 | 10  | YAML::Old            | 16ms      | 9.5ms             | 9e-05   | 21      |
 | 2   | JSON::XS             | 14.1ms    | 7.17ms            | 2.2e-05 | 22      |
 | 3   | Cpanel::JSON::XS     | 13.7ms    | 6.8ms             | 3.9e-05 | 20      |
 | 11  | YAML::Syck           | 14ms      | 6.7ms             | 4.8e-05 | 20      |
 | 9   | JSON::Parse          | 13.5ms    | 6.63ms            | 2.6e-05 | 20      |
 | 8   | JSON::Create         | 11ms      | 4.2ms             | 4.9e-05 | 20      |
 | 15  | Data::MessagePack    | 10ms      | 3.6ms             | 4e-05   | 20      |
 | 5   | JSON::Decode::Regexp | 10ms      | 3.6ms             | 4.1e-05 | 20      |
 | 0   | perl -e1 (baseline)  | 6.9ms     | 0ms               | 1.7e-05 | 20      |
 +-----+----------------------+-----------+-------------------+---------+---------+

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
